/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 C O M B I N E L O C I (NEWTON RAPHSON)  R O U T I N E S 

 combines loci by estimating a gamma shape parameter
 

 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: combine.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"
#include "parameter.h"
#include "world.h"


#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

/* structure for passing derivatives between functions */
typedef struct derive_fmt
  {
    double l0[GAMMA_INTERVALS];
    double l1000[GAMMA_INTERVALS];
    double l0100[GAMMA_INTERVALS];
    double l0010[GAMMA_INTERVALS];
    double l0001[GAMMA_INTERVALS];
    double l2000[GAMMA_INTERVALS];
    double l1100[GAMMA_INTERVALS];
    double l1010[GAMMA_INTERVALS];
    double l1001[GAMMA_INTERVALS];
    double l0200[GAMMA_INTERVALS];
    double l0110[GAMMA_INTERVALS];
    double l0101[GAMMA_INTERVALS];
    double l0020[GAMMA_INTERVALS];
    double l0011[GAMMA_INTERVALS];
    double l0002[GAMMA_INTERVALS];
    double tba[GAMMA_INTERVALS];
  }
derive_fmt;



/* prototypes ------------------------------------------- */
long combine_loci (world_fmt * world);
void create_nr (nr_fmt * nr, long numpop, long G);
void destroy_nr (nr_fmt * nr);
double calc_loci_like (nr_fmt * nr, timearchive_fmt * atl, long loci, boolean gamma);

void calc_gamma (nr_fmt * nr);	/* calculates gamma values a specified probabilities */

/* private functions prototypes-------------------------- */
/*derivative calculations */
long dgamma_nr_driver (timearchive_fmt * tyme, long loci,
		       world_fmt * world, char **plane);
void dgamma_derivatives (nr_fmt * nr,
       timearchive_fmt * tyme, double *param, long loci, boolean boolgamma);
void simple_loci_derivatives (nr_fmt * nr,
			  timearchive_fmt * tyme, double *param, long loci);
void dgamma_parts (derive_fmt * derive,
		   nr_fmt * nr, timearchive_fmt * tyme,
		   double tb1, double tb2, double m1, double m2,
		   double a);
void dgamma_first (double integral, derive_fmt * derive,
		   nr_fmt * nr, timearchive_fmt * tyme,
		   double tb1, double tb2, double r1, double r2,
		   double a);
void dgamma_secon (double integral, derive_fmt * derive,
		   nr_fmt * nr, timearchive_fmt * tyme,
		   double tb1, double tb2, double r1, double r2,
		   double a);
void dgamma_sum (double *intergral, double tba[],
		 double l0[], nr_fmt * nr, timearchive_fmt * atl);
/* likelihood calculations with gamma variation */
double calc_like2 (nr_fmt * nr, double t, double *param0, tarchive_fmt * tyme, long G);

/* Newton-Raphson stuff and other specific helper functions */
void calc_loci_param (nr_fmt * nr, double *param, double lamda);
void create_apg0 (nr_fmt * nr, timearchive_fmt * tyme);
void reset_parts (derive_fmt * derive, nr_fmt * nr);
void init_part (double *x, double value, long n);
void copy_and_clear_ddd (nr_fmt * nr, double *d, double **dd);
void add_back_ddd (nr_fmt * nr, double *d, double **dd);
/* printing sthings */
void print_menu_finalestimate (option_fmt * options, char text[]);

/* public functions---------------------------------------------- */
long
combine_loci (world_fmt * world)
{
  long trials;
  trials = dgamma_nr_driver (world->atl, world->data->loci,
			     world, world->plane[world->loci]);
  if (world->options->plot)
    create_loci_plot (world, world->plane[world->loci], world->atl, world->loci);
  return trials;
}

long
dgamma_nr_driver (timearchive_fmt * tyme, long loci,
		  world_fmt * world, char **plane)
{

  boolean notinverse, tttt, savety_belt = FALSE, do_newton = TRUE;
  char *strllike, kind[20];
  long pop, locus, trials = -1, panic, elem = 5, Gmax = 1;
  double **idd, *nld, lam1 = 0.0, lam2 = 0.0, lamda = 1., nld2;
  double llike = -DBL_MAX, normd = DBL_MAX /*, maxllike */ ;
  nr_fmt *nr;
  double lamdasign = 1;

  print_menu_finalestimate (world->options, "Newton-Raphson method");
  nr = (nr_fmt *) calloc (1, sizeof (nr_fmt) * 1);
  nr->numpop = world->numpop;
  if (world->options->gamma)
    nr->numpop2 = world->numpop * 2 + 1;
  else
    nr->numpop2 = world->numpop * 2;
  for (locus = 1; locus < loci + 1; locus++)
    {
      if (Gmax < tyme[locus].T)
	{
	  Gmax = tyme[locus].T;
	}
    }
  nr->partsize = (nr->numpop2 + nr->numpop2 * nr->numpop2);
  /* initialize local stuff: mallocing */
  strllike = (char *) calloc (1, sizeof (char) * 128);
  nr->gammaI = GAMMA_INTERVALS;
  nr->gamma = (double *) calloc (1, nr->gammaI * sizeof (double));
  nr->parts = (double *) calloc (1, nr->partsize * sizeof (double));
  nr->d = (double *) malloc (nr->numpop2 * sizeof (double));
  nr->param = (double *) malloc ((1 + nr->numpop2) * sizeof (double));
  nr->oparam = (double *) malloc ((1 + nr->numpop2) * sizeof (double));
  nr->datalike = (double *) malloc (Gmax * sizeof (double));
  nr->apg0 = (double *) malloc (Gmax * sizeof (double));
  nr->apg = (double *) malloc (Gmax * sizeof (double));
  nld = (double *) malloc ((1 + NTRIALS) * sizeof (double));
  nr->skiploci = world->data->skiploci;
  nr->dd = world->cov[loci];
  idd = (double **) calloc (1, sizeof (double *) * nr->numpop2);
  idd[0] = (double *) calloc (1, sizeof (double) * nr->numpop2 * nr->numpop2);
  for (pop = 1; pop < nr->numpop2; pop++)
    {
      idd[pop] = idd[0] + pop * nr->numpop2;
    }
  world->param0 = (double *) realloc (world->param0, sizeof (double) * nr->numpop2);
  memset (world->param0, 0, sizeof (double) * nr->numpop2);
  for (locus = 1; locus < loci + 1; locus++)
    {
      for (pop = 0; pop < nr->numpop2; pop++)
	{
	  if (!world->data->skiploci[locus - 1])
	    world->param0[pop] += world->atl[locus].param[pop];
	}
    }
  for (pop = 0; pop < nr->numpop2; pop++)
    {
      world->param0[pop] /= (loci - world->skipped);
    }
  if (!world->options->gamma)
    {
      elem = nr->numpop2;
      nr->gammaI = 1;
      nr->gamma[0] = 1.;
    }
  else
    {
      elem = nr->numpop2;
      world->param0[nr->numpop2 - 1] = 1.0;
    }
  memcpy (nr->param, world->param0, sizeof (double) * nr->numpop2);
  nr->ollike = -DBL_MAX;
  /*  maxllike= -DBL_MAX; */
  while (trials++ < NTRIALS)
    {
#ifdef MAC
      eventloop ();
#endif
      reset_nr (nr);
      if (world->options->gamma)
	calc_gamma (nr);
      calc_loci_like (nr, tyme, loci, world->options->gamma);
      dgamma_derivatives (nr, tyme, world->param0, loci, world->options->gamma);
      normd = norm (nr->d, nr->numpop2);
      if (normd == 0.0)
	break;
      notinverse = is_singular (nr->dd, elem /*number rows in matrix */ );
      if (!notinverse)
	{
	  for (pop = 0; pop < nr->numpop2; pop++)
	    memcpy (idd[pop], nr->dd[pop], sizeof (double) * nr->numpop2);
	  invert_matrix (idd, elem);
	  tttt = nrcheck (nr->dd, idd, nr->d, elem, &lam1, &lam2, do_newton);
	  if (tttt)
	    {
	      strcpy (kind, "NEWTON: ");
	      lamda = 1.;
	      nld[trials] = 1.0;
	    }
	  else
	    notinverse = TRUE;
	}
      if (notinverse)
	{
	  if (lam2 > 0)
	    {
	      lamda = lam1 / lam2;
	      if (lamda >= 1.0)
		nld[trials] = normd;
	      else
		nld[trials] = normd * (lamda);
	    }
	  else
	    {
	      if (trials == 0)
		{
		  lamda = 1.0;
		  nld[0] = normd;
		}
	      else
		{
		  nld2 = normd;
		  if (nld2 == 0.0)
		    {
		      nld[trials] = 0.0;
		      lamda = 0.0;
		      fprintf (stderr, "norm(d) is 0.0 we should stop!");
		    }
		  else
		    {
		      lamda = vector_max (nld, trials) / nld2;
		      if (lamda >= 1.0)
			nld[trials] = normd;
		      else
			nld[trials] = nld2 * (lamda);
		    }
		}
	    }
	  strcpy (kind, "SLOW:   ");
	}
      if (world->options->gamma)
	calc_loci_param (nr, world->param0, lamda);
      else
	calc_param (nr, world->param0, lamda);
      nr->ollike = nr->llike;
      calc_loci_like (nr, tyme, loci, world->options->gamma);
      panic = 0;
      memcpy (nr->oparam, nr->param, nr->numpop2 * sizeof (double));
      if (nr->ollike >= nr->llike)
	{
	  lamda = lamdasign * lamda;
	  while (((nr->llike - nr->ollike) < 0.) && (panic++ < 20))
	    {
	      memcpy (nr->oparam, world->param0, nr->numpop2 * sizeof (double));
	      lamda /= 2.;
	      if (world->options->gamma)
		calc_loci_param (nr, world->param0, lamda);
	      else
		calc_param (nr, world->param0, lamda);
	      calc_loci_like (nr, tyme, loci, world->options->gamma);
	    }
	  if (panic > 20)
	    {
	      memcpy (world->param0, nr->param, nr->numpop2 * sizeof (double));
	      if (!savety_belt)
		{
		  fprintf (stdout, "Halfing limit reached! Disable Newton-steps and use only gradient\n");
		  ;
		  do_newton = FALSE;
		  if (world->options->gamma)
		    calc_loci_param (nr, world->param0, lamda);
		  else
		    calc_param (nr, world->param0, lamda);
		  calc_loci_like (nr, tyme, loci, world->options->gamma);
		  savety_belt = !savety_belt;
		  panic = 0;
		}
	      else
		{
		  fprintf (stdout, "Halfing limit reached! Rescue failed!\n");
		  break;
		}
	    }
	  memcpy (world->param0, nr->param, nr->numpop2 * sizeof (double));
	}
      else
	{
	  savety_belt = FALSE;
	  while (nr->llike - nr->ollike > EPSILON && panic++ < 10)
	    {
	      memcpy (nr->oparam, nr->param, nr->numpop2 * sizeof (double));
	      lamda *= 2.;
	      if (world->options->gamma)
		calc_loci_param (nr, world->param0, lamda);
	      else
		calc_param (nr, world->param0, lamda);
	      llike = nr->ollike;
	      nr->ollike = nr->llike;
	      nr->oPGC = nr->PGC;
	      memcpy (world->param0, nr->param, nr->numpop2 * sizeof (double));
	      calc_loci_like (nr, tyme, loci, world->options->gamma);
	    }
	  if (panic > 0)
	    {
	      memcpy (world->param0, nr->oparam, nr->numpop2 * sizeof (double));
	      memcpy (nr->param, nr->oparam, nr->numpop2 * sizeof (double));
	      nr->llike = nr->ollike;
	      nr->ollike = llike;
	      nr->PGC = nr->oPGC;
	    }
	  else
	    {
	      if (world->options->gamma)
		calc_loci_param (nr, world->param0, lamda);
	      else
		calc_param (nr, world->param0, lamda);
	      memcpy (world->param0, nr->param, nr->numpop2 * sizeof (double));
	    }
	}
      if (!((((normd > LOCI_NORM) && (trials < NTRIALS))) || trials == 0))
	{
	  break;
	}
    }
  world->param_like = nr->llike;
  tyme[loci + 1].param = (double *) realloc (tyme[loci + 1].param,
					     sizeof (double) * elem);
  tyme[loci + 1].param_like = nr->llike;
  tyme[loci + 1].normd = normd;
  tyme[loci + 1].trials = trials;
  memcpy (tyme[loci + 1].param, world->param0, sizeof (double) * elem);
  if (!world->options->simulation)
    calc_cov (nr->dd, nr->d, world->param0, elem);
  free (strllike);
  free_nr (nr);
  free (nld);
  free (idd[0]);
  free (idd);
  if (world->options->progress)
    {
      fprintf (stdout, "           Newton-Raphson cycles needed: %li\n", trials);
      fprintf (stdout, "           (norm(first derivatives: %f)\n", normd);
    }
  return trials;
}

/* calculates the likelihood over all loci for the new parameter set */
double
calc_loci_like (nr_fmt * nr, timearchive_fmt * atl, long loci,
		boolean boolgamma)
{
  static double oldalpha, oldtheta;
  long /*lazy, */ locus, i, maxloc = 0;
  double *temp, *param0;
  double ll, theta, result, logres = 0, tmax;
  temp = (double *) malloc (sizeof (double) * nr->gammaI);
  param0 = (double *) malloc (sizeof (double) * nr->numpop * 2);

  for (locus = 1; locus < loci + 1; locus++)
    {
      if (nr->skiploci[locus - 1])
	continue;
      result = 0.;
      tmax = -DBL_MAX;
      memcpy (param0, atl[locus].param0, sizeof (double) * nr->numpop * 2);
      if (!boolgamma)
	{
	  create_apg0 (nr, &atl[locus]);
	  logres += calc_like (nr, atl[locus].tl, atl[locus].T);
	}
      else
	{
	  if (oldalpha != nr->param[4] || oldtheta != nr->param[0])
	    {
	      calc_gamma (nr);
	      oldalpha = nr->param[4];
	      oldtheta = nr->param[0];
	    }
/*       lazy=0; */
	  for (i = nr->gammaI - 1; i >= 0; i--)
	    {
	      theta = nr->gamma[i];
	      ll = calc_like2 (nr, theta, param0,
			       atl[locus].tl, atl[locus].T);
	      temp[i] = -theta / (nr->param[0] * nr->param[4]) +
		(1. / nr->param[4] - 1.) * log (theta) + ll;

	      if (temp[i] > tmax)
		{
		  maxloc = i;
		  tmax = temp[i];
		}
	      /*      else {
	         lazy++;
	         if ((lazy>10) && (temp[i]< tmax-40.)){
	         lowerbound=i;
	         break;
	         }
	         } */
	    }
	  for (i = maxloc; i < nr->gammaI; i++)
	    {
	      temp[i] -= tmax;
	      if (temp[i] > -40)
		result += exp (temp[i]);
	      else
		{
		  break;
		}
	    }
	  for (i = maxloc - 1; i >= 0; i--)
	    {
	      temp[i] -= tmax;
	      if (temp[i] > -40)
		result += exp (temp[i]);
	      else
		{
		  break;
		}
	    }
	  logres += tmax + log (result);
	  logres = logres - (1. / nr->param[4] * (log (nr->param[0]) +
			  log (nr->param[4])) + lgamma (1. / nr->param[4]));
	}
    }
  free (param0);
  free (temp);
  nr->llike = logres;
  return logres;
}

void
create_nr (nr_fmt * nr, long numpop, long G)
{
  long pop;
  nr->numpop = numpop;
  nr->numpop2 = numpop * 2;
  nr->partsize = (nr->numpop2 + nr->numpop2 * nr->numpop2);
  nr->parts = (double *) calloc (1, nr->partsize * sizeof (double));
  nr->d = (double *) malloc (nr->numpop2 * sizeof (double));
  nr->dd = (double **) calloc (1, sizeof (double *) * nr->numpop2);
  nr->dd[0] = (double *) calloc (1, sizeof (double) * nr->numpop2 * nr->numpop2);
  for (pop = 1; pop < nr->numpop2; pop++)
    {
      nr->dd[pop] = nr->dd[0] + pop * nr->numpop2;
    }
  nr->param = (double *) malloc ((1 + nr->numpop2) * sizeof (double));
  nr->oparam = (double *) malloc ((1 + nr->numpop2) * sizeof (double));
  nr->datalike = (double *) malloc (G * sizeof (double));
  nr->apg0 = (double *) malloc (G * sizeof (double));
  nr->apg = (double *) malloc (G * sizeof (double));
  nr->gammaI = GAMMA_INTERVALS;
  nr->gamma = (double *) malloc (nr->gammaI * sizeof (double));
}

void
destroy_nr (nr_fmt * nr)
{
  free (nr->parts);
  free (nr->d);
  free (nr->param);
  free (nr->oparam);
  free (nr->datalike);
  free (nr->apg0);
  free (nr->apg);
  free (nr->gamma);
}

/* calculation of theta values following a gamma distribution for
   given probability values */
void
calc_gamma (nr_fmt * nr)
{
  long i, panic;
  double low, mid, high, xlow, xhigh, tmp, freq = 0, x = 10, alpha = 1. / nr->param[4],
    elements = (double) nr->gammaI, theta = nr->param[0];
  freq = -(0.5 / elements);	/*so we have midpoints instead of endpoints */
  for (i = 0; i < elements; i++)
    {
      low = 0;
      mid = /*exp(-lgamma(alpha)) */ incompletegamma (10., alpha);
      high = 1.;
      freq += 1. / (elements);
      if (freq < mid)
	{
	  high = mid;
	  xlow = 0;
	  xhigh = 10.;
	  x = 5.;
	}
      else
	{
	  low = mid;
	  xhigh = 1e10;
	  xlow = 10.;
	  x = 1e5;
	}
      panic = 0;
      while (panic++ < 1000 && fabs (low - high) > 0.0001 && x > SMALLEST_THETA)
	{
	  mid = /*exp(-lgamma(alpha)) */ incompletegamma (x, alpha);
	  if (freq < mid)
	    {
	      high = mid;
	      tmp = x;
	      x = (x + xlow) / 2.;
	      xhigh = tmp;
	    }
	  else
	    {
	      low = mid;
	      tmp = x;
	      x = (x + xhigh) / 2.;
	      xlow = tmp;
	    }
	}
      nr->gamma[i] = x * theta / alpha;
      if (x >= 10e10)
	{
	  nr->gammaI = i + 1;
	  return;
	}
    }
}

/* private functions---------------------------------------------- */
/* derivatives */
void
dgamma_derivatives (nr_fmt * nr,
	timearchive_fmt * tyme, double *param, long loci, boolean boolgamma)
{
  long locus;
  derive_fmt derive;
  double integral;
  double tb1 = nr->param[0];
  double tb2 = nr->param[1];
  double r1 = nr->param[2];
  double r2 = nr->param[3];
  double invalpha = 0, alpha = 0;
  if (boolgamma)
    {
      invalpha = nr->param[4];
      alpha = 1. / invalpha;
    }
  for (locus = 1; locus < loci + 1; locus++)
    {
      if (nr->skiploci[locus - 1])
	continue;
      reset_parts (&derive, nr);
      if (!boolgamma)
	{
	  simple_loci_derivatives (nr, tyme, param, locus);
	}
      else
	{
	  dgamma_parts (&derive, nr, &tyme[locus], tb1,
			tb2, r1, r2, alpha);
	  dgamma_sum (&integral, derive.tba, derive.l0, nr, &tyme[locus]);
	  dgamma_first (integral, &derive,
			nr, &tyme[locus], tb1, tb2, r1, r2, invalpha);
	  dgamma_secon (integral, &derive,
			nr, &tyme[locus], tb1, tb2, r1, r2, invalpha);
	}
    }
  derivatives_to_logderivatives (nr);
}

void
simple_loci_derivatives (nr_fmt * nr,
			 timearchive_fmt * tyme, double *param, long locus)
{
  long g;
  static double *d, **dd;
  d = (double *) calloc (1, sizeof (double) * nr->numpop2);
  dd = (double **) calloc (1, sizeof (double *) * nr->numpop2);
  dd[0] = (double *) calloc (1, sizeof (double) *
			     nr->numpop2 * nr->numpop2);
  for (g = 1; g < nr->numpop2; g++)
    {
      dd[g] = dd[0] + g * nr->numpop2;
    }
  copy_and_clear_ddd (nr, d, dd);
  nr->PGC = 0;
  nr->apg_max = -DBL_MAX;
  for (g = 0; g < tyme[locus].T; g++)
    {
      nr->apg0[g] = probG (tyme[locus].param0, &tyme[locus].tl[g], nr->numpop);
    }
  for (g = 0; g < tyme[locus].T; g++)
    {
      nr->apg[g] = probG (nr->param, &tyme[locus].tl[g], nr->numpop) - nr->apg0[g];
      if (nr->apg[g] > nr->apg_max)
	nr->apg_max = nr->apg[g];
    }
  for (g = 0; g < tyme[locus].T; g++)
    {
      nr->apg[g] -= nr->apg_max;
      nr->PGC += tyme[locus].tl[g].copies * exp (nr->apg[g]);
    }
  derivatives (1, nr, tyme[locus].tl, tyme[locus].T,
	       param, 0);
  add_back_ddd (nr, d, dd);
  free (dd[0]);
  free (dd);
  free (d);
}

void
dgamma_parts (derive_fmt * derive,
	      nr_fmt * nr, timearchive_fmt * tyme,
	      double tb1, double tb2, double m1, double m2,
	      double a)
{
  long g, gg, gsum = 0;
  double x1000, x0100, x0010, x0001, x1100, x1010, x1001, x2000, x0200;
  double x0020, x0002, t, /*st, */ k1, k2, kk1, kk2, l1, l2, p1, p2;
  double logt1, logt2, logr1, logr2, el0;
  double l0max = -DBL_MAX, tbamax = -DBL_MAX;
  double stb1 = tb1 * tb1, stb2 = tb2 * tb2;
  double tb10 = tyme->param0[0], tb20 = tyme->param0[1];
  double m10 = tyme->param0[2], m20 = tyme->param0[3];
  double logt, lgsum, logpt1, logpt2, logpr1, logpr2;
  double **l0;
  l0 = (double **) calloc (1, sizeof (double *) * nr->gammaI);
  for (gg = 0; gg < tyme->T; gg++)
    {
      gsum += tyme->tl[gg].copies;
    }
  lgsum = log ((double) gsum);
  logpr1 = log (m1 * tb1 / (m10));
  logpr2 = log (m2 * tb1 / (m20));
  logpt1 = log (tb10);
  logpt2 = log (tb20 * tb1 / (tb2));
  for (g = 0; g < nr->gammaI; g++)
    {
      l0[g] = (double *) calloc (1, sizeof (double) * gsum);
      t = nr->gamma[g];
/*      st = t * t; */
      logt = log (t);
      logr1 = logpr1 - logt;
      logr2 = logpr2 - logt;
      logt1 = logpt1 - logt;
      logt2 = logpt2 - logt;
      for (gg = 0; gg < tyme->T; gg++)
	{
	  kk1 = tyme->tl[gg].kt[0];
	  kk2 = tyme->tl[gg].kt[1];
	  k1 = tyme->tl[gg].km[0];
	  k2 = tyme->tl[gg].km[1];
	  l1 = tyme->tl[gg].l[0];
	  l2 = tyme->tl[gg].l[1];
	  p1 = tyme->tl[gg].p[0];
	  p2 = tyme->tl[gg].p[1];

	  l0[g][gg] = k1 * m10 + k2 * m20 - kk1 / t - (k1 * m1 * tb1) / t
	    - (k2 * m2 * tb1) / t + kk1 / tb10 - (kk2 * tb1) / (t * tb2)
	    + kk2 / tb20 + l1 * logr1 + l2 * logr2
	    + p1 * logt1 + p2 * logt2;
	  if (l0[g][gg] > l0max)
	    l0max = l0[g][gg];
	}
      derive->tba[g] = (a - 1) * logt - (t * a / tb1);
      if (derive->tba[g] > tbamax)
	tbamax = derive->tba[g];
    }
  for (g = 0; g < nr->gammaI; g++)
    {
      for (gg = 0; gg < tyme->T; gg++)
	{
	  l0[g][gg] -= l0max;
	  derive->l0[g] += exp (l0[g][gg]) * tyme->tl[gg].copies;
	}
      if (derive->l0[g] == 0.0)
	derive->l0[g] = -DBL_MAX;
      else
	derive->l0[g] = log (derive->l0[g]) - lgsum;
      derive->tba[g] -= tbamax;
    }

  for (g = 0; g < nr->gammaI; g++)
    {
      t = nr->gamma[g];

      for (gg = 0; gg < tyme->T; gg++)
	{
	  el0 = exp (l0[g][gg] - lgsum + log ((double) tyme->tl[gg].copies));
	  kk1 = tyme->tl[gg].kt[0];
	  kk2 = tyme->tl[gg].kt[1];
	  k1 = tyme->tl[gg].km[0];
	  k2 = tyme->tl[gg].km[1];
	  l1 = tyme->tl[gg].l[0];
	  l2 = tyme->tl[gg].l[1];
	  p2 = tyme->tl[gg].p[1];

	  x1000 = (-kk2 / tb2 - k1 * m1 - k2 * m2) / t + (l1 + l2 + p2) / tb1;
	  x0100 = kk2 * tb1 / (stb2 * t) - p2 / tb2;
	  x0010 = l1 / m1 - k1 * tb1 / t;
	  x0001 = l2 / m2 - k2 * tb1 / t;

	  x2000 = -(l1 + l2 + p2) / stb1;
	  x0200 = p2 / stb2 - 2. * kk2 * tb1 / (t * stb2 * tb2);
	  x0020 = -l1 / (m1 * m1);
	  x0002 = -l2 / (m2 * m2);

	  x1100 = kk2 / (t * stb2);
	  x1010 = -k1 / t;
	  x1001 = -k2 / t;


	  derive->l1000[g] += el0 * x1000;
	  derive->l0100[g] += el0 * x0100;
	  derive->l0010[g] += el0 * x0010;
	  derive->l0001[g] += el0 * x0001;

	  derive->l2000[g] += el0 * (x1000 * x1000 + x2000);
	  derive->l0200[g] += el0 * (x0100 * x0100 + x0200);
	  derive->l0020[g] += el0 * (x0010 * x0010 + x0020);
	  derive->l0002[g] += el0 * (x0001 * x0001 + x0002);

	  derive->l1100[g] += el0 * (x1000 * x0100 + x1100);
	  derive->l1010[g] += el0 * (x1000 * x0010 + x1010);
	  derive->l1001[g] += el0 * (x1000 * x0001 + x1001);
	  derive->l0110[g] += el0 * x0100 * x0010;
	  derive->l0101[g] += el0 * x0100 * x0001;
	  derive->l0011[g] += el0 * x0010 * x0001;
	}
      free (l0[g]);
    }
  free (l0);
}

void
dgamma_sum (double *integral, double tba[],
	    double l0[], nr_fmt * nr, timearchive_fmt * atl)
{
  long g;

  *integral = 0.0;
  for (g = 0; g < nr->gammaI; g++)
    {
      *integral += exp (tba[g] + l0[g]);
    }
  if (*integral == 0.0)
    *integral = DBL_EPSILON;
}

/* first derivatives */
void
dgamma_first (double integral, derive_fmt * derive,
	      nr_fmt * nr, timearchive_fmt * tyme,
	      double tb1, double tb2, double r1, double r2,
	      double ia)
{
  long g;
  double t, logt, pt, el0;

  for (g = 0; g < nr->gammaI; g++)
    {
      el0 = exp (derive->l0[g]);
      pt = exp (derive->tba[g]);
      t = nr->gamma[g];
      logt = log (t);
      /*tb1--------------------------------------------- */
      nr->parts[0] += pt * (t * el0 / (ia * tb1 * tb1) + derive->l1000[g]);
      /*tb2--------------------------------------------- */
      nr->parts[1] += pt * derive->l0100[g];
      /*r1 and r2--------------------------------------------- */
      nr->parts[2] += pt * derive->l0010[g];
      nr->parts[3] += pt * derive->l0001[g];
      /*a-------------------------------------------------- */
      nr->parts[4] += pt * el0 * (t - tb1 * logt) / (ia * ia * tb1);

    }

  for (g = 0; g < 5; g++)
    {
      nr->parts[g] /= integral;
    }
}

void
dgamma_secon (double integral, derive_fmt * derive,
	      nr_fmt * nr, timearchive_fmt * tyme,
	      double tb1, double tb2, double r1, double r2,
	      double ia)
{
  long g;
  double t, st, logt, el0, pt, twoat, stb1 = tb1 * tb1;
  double a = 1. / ia;
  double sa = a * a;
  for (g = 0; g < nr->gammaI; g++)
    {
      el0 = exp (derive->l0[g]);
      pt = exp (derive->tba[g]);
      t = nr->gamma[g];
      st = t * t;
      logt = log (t);
      twoat = 2. * a * t;
      /*tb1,tb1--------------------------------------------- */
      nr->parts[5] += (pt * (sa * st * el0 - twoat * tb1 * el0 + twoat * stb1 * derive->l1000[g] + stb1 * stb1 *
			     derive->l2000[g])) / (stb1 * stb1);
/*    ((el0/stb1*(st*sa - twoat *tb1) + twoat * derive->l1000[g])/stb1 
   + derive->l2000[g]); */
      /*tb2,tb2--------------------------------------------- */
      nr->parts[6] += pt * derive->l0200[g];
      /*r1, r1--------------------------------------------- */
      nr->parts[7] += pt * derive->l0020[g];
      /*r2, r2--------------------------------------------- */
      nr->parts[8] += pt * derive->l0002[g];
      /*a,a--------------------------------------------- */
      nr->parts[9] += pt * el0 * (-t + tb1 * logt) *
	(-t + 2. * ia * tb1 + tb1 * logt) / (ia * ia * ia * ia * stb1);
      /*tb1,tb2--------------------------------------------- */
      nr->parts[10] += pt * (a * t * derive->l0100[g] / stb1 + derive->l1100[g]);
      /*tb1,r1--------------------------------------------- */
      nr->parts[11] += pt * (a * t * derive->l0010[g] / stb1 + derive->l1010[g]);
      /* tb1,r2-------------------------- */
      nr->parts[12] += pt * (a * t * derive->l0001[g] / stb1 + derive->l1001[g]);
      /* tb1,a-------------------------- */
      nr->parts[13] += pt * (st * el0 - ia * t * tb1 * el0 - t * tb1 * el0 * logt +
			     ia * t * stb1 * derive->l1000[g] -
			     ia * stb1 * tb1 * logt * derive->l1000[g]) / (ia * ia * ia * stb1 * tb1);
      /* tb2,r1-------------------------- */
      nr->parts[14] += pt * derive->l0110[g];
      /* tb2,r2-------------------------- */
      nr->parts[15] += pt * derive->l0101[g];
      /* tb2,a-------------------------- */
      nr->parts[16] += pt * (t - tb1 * logt) * derive->l0100[g] / (ia * ia * tb1);
      /* r1,r2-------------------------- */
      nr->parts[17] += pt * derive->l0011[g];
      /* r1,a-------------------------- */
      nr->parts[18] += pt * (t - tb1 * logt) * derive->l0010[g] / (ia * ia * tb1);
      /* r2,a-------------------------- */
      nr->parts[19] += pt * (t - tb1 * logt) * derive->l0001[g] / (ia * ia * tb1);
    }
  nr->dd[0][0] -= 1. / (ia * stb1) - nr->parts[0] * nr->parts[0] + nr->parts[5] / integral;
  nr->dd[1][1] -= -(nr->parts[1] * nr->parts[1]) + nr->parts[6] / integral;

  nr->dd[2][2] -= -(nr->parts[2] * nr->parts[2]) + nr->parts[7] / integral;
  nr->dd[3][3] -= -(nr->parts[3] * nr->parts[3]) + nr->parts[8] / integral;
  nr->dd[4][4] -= (3. - 2. * log (ia * tb1) - 2. * polygamma (0, a) - polygamma (1, a) / ia) / (ia * ia * ia)
    - nr->parts[4] * nr->parts[4] + nr->parts[9] / integral;
  nr->dd[1][0] -= -(nr->parts[0] * nr->parts[1]) + nr->parts[10] / integral;
  nr->dd[2][0] -= -(nr->parts[0] * nr->parts[2]) + nr->parts[11] / integral;
  nr->dd[3][0] -= -(nr->parts[0] * nr->parts[3]) + nr->parts[12] / integral;
  nr->dd[4][0] -= 1. / (ia * ia * tb1) - (nr->parts[0] * nr->parts[4]) + nr->parts[13] / integral;
  nr->dd[2][1] -= -(nr->parts[1] * nr->parts[2]) + nr->parts[14] / integral;
  nr->dd[3][1] -= -(nr->parts[1] * nr->parts[3]) + nr->parts[15] / integral;
  nr->dd[4][1] -= -(nr->parts[1] * nr->parts[4]) + nr->parts[16] / integral;
  nr->dd[3][2] -= -(nr->parts[2] * nr->parts[3]) + nr->parts[17] / integral;
  nr->dd[4][2] -= -(nr->parts[2] * nr->parts[4]) + nr->parts[18] / integral;
  nr->dd[4][3] -= -(nr->parts[3] * nr->parts[4]) + nr->parts[19] / integral;
  nr->d[0] -= (-1. / (ia * tb1) + nr->parts[0]);
  nr->d[1] -= nr->parts[1];
  nr->d[2] -= nr->parts[2];
  nr->d[3] -= nr->parts[3];
  nr->d[4] -= (-1 + log (tb1 * ia) + polygamma (0, a)) / (ia * ia) + nr->parts[4];
}


/* likelihood calculations */
double
calc_like2 (nr_fmt * nr, double t, double *param0, tarchive_fmt * tyme, long G)
{
  long g;
  double k1, k2, kk1, kk2, l1, l2, p1, p2, apg_max = -DBL_MAX, gsum = 0.,
    logt1, logt2, logr1, logr2, m10, m20, /*r1, r2, r10, r20, */ tb1 = nr->param[0],
    tb2 = nr->param[1], m1 = nr->param[2], m2 = nr->param[3], tb10 = param0[0],
    tb20 = param0[1];
  nr->PGC = 0;
/*    r1 = m1 * tb1; */
/*    r2 = m2 * tb2; */
  m10 = param0[2];
  m20 = param0[3];
/*    r10 = m10 * tb10; */
/*    r20 = m20 * tb20; */
  logr1 = log (m1 * tb1 / (m10 * t));
  logr2 = log (m2 * tb1 / (m20 * t));
  logt2 = log (tb20 * tb1 / (tb2 * t));
  logt1 = log (tb10 / t);

  for (g = 0; g < G; g++)
    {
      kk1 = tyme[g].kt[0];
      kk2 = tyme[g].kt[1];
      k1 = tyme[g].km[0];
      k2 = tyme[g].km[1];
      l1 = tyme[g].l[0];
      l2 = tyme[g].l[1];
      p1 = tyme[g].p[0];
      p2 = tyme[g].p[1];
      nr->apg[g] = k1 * m10 + k2 * m20 - kk1 / t - (k1 * m1 * tb1) / t
	- (k2 * m2 * tb1) / t + kk1 / tb10 - (kk2 * tb1) / (t * tb2)
	+ kk2 / tb20 + l1 * logr1 + l2 * logr2 + p1 * logt1 + p2 * logt2;
      if (nr->apg[g] > apg_max)
	apg_max = nr->apg[g];
    }
  for (g = 0; g < G; g++)
    {
      gsum += tyme[g].copies;
      nr->apg[g] -= apg_max;
      nr->PGC += tyme[g].copies * exp (nr->apg[g]);
    }
  return apg_max + log (nr->PGC) - log (gsum);
}


/* Newton-Raphson stuff and other specific helper functions */
void
calc_loci_param (nr_fmt * nr, double *param, double lamda)
{
  long i;
  for (i = 0; i < nr->numpop2; i++)
    nr->param[i] = param[i] * exp ((MAX (-100, MIN (-lamda * nr->d[i], 100))));
  param_all_adjust (nr, param, 1);
}


void
reset_parts (derive_fmt * derive, nr_fmt * nr)
{
  init_part (derive->l0, 0., GAMMA_INTERVALS);
  init_part (derive->l1000, 0., GAMMA_INTERVALS);
  init_part (derive->l0100, 0., GAMMA_INTERVALS);
  init_part (derive->l0010, 0., GAMMA_INTERVALS);
  init_part (derive->l0001, 0., GAMMA_INTERVALS);
  init_part (derive->l2000, 0., GAMMA_INTERVALS);
  init_part (derive->l1100, 0., GAMMA_INTERVALS);
  init_part (derive->l1010, 0., GAMMA_INTERVALS);
  init_part (derive->l1001, 0., GAMMA_INTERVALS);
  init_part (derive->l0200, 0., GAMMA_INTERVALS);
  init_part (derive->l0110, 0., GAMMA_INTERVALS);
  init_part (derive->l0101, 0., GAMMA_INTERVALS);
  init_part (derive->l0020, 0., GAMMA_INTERVALS);
  init_part (derive->l0011, 0., GAMMA_INTERVALS);
  init_part (derive->l0002, 0., GAMMA_INTERVALS);
  init_part (nr->parts, 0., nr->partsize);

}

void
init_part (double *x, double value, long n)
{
  long i;
  if (value == 0)
    {
      memset (x, 0, sizeof (double) * n);
    }
  else
    {
      for (i = 0; i < n; i++)
	{
	  x[i] = value;
	}
    }
}

void
copy_and_clear_ddd (nr_fmt * nr, double *d, double **dd)
{
  long pop;
  memcpy (d, nr->d, sizeof (double) * nr->numpop2);
  memset (nr->d, 0, sizeof (double) * nr->numpop2);
  for (pop = 0; pop < nr->numpop2; pop++)
    {
      memcpy (dd[pop], nr->dd[pop], sizeof (double) * nr->numpop2);
      memset (nr->dd[pop], 0, sizeof (double) * nr->numpop2);
    }
}
void
add_back_ddd (nr_fmt * nr, double *d, double **dd)
{
  long pop, p;
  for (pop = 0; pop < nr->numpop2; pop++)
    {
      nr->d[pop] += d[pop];
      for (p = 0; p < nr->numpop2; p++)
	{
	  nr->dd[pop][p] += dd[pop][p];
	}
    }
}

void
create_apg0 (nr_fmt * nr, timearchive_fmt * tyme)
{
  long g;
  /* Prob(G|Param0) */
  for (g = 0; g < tyme->T; g++)
    {
      nr->apg0[g] = probG (tyme->param0, &tyme->tl[g], nr->numpop);
    }
}

void
print_menu_finalestimate (option_fmt * options, char text[])
{
  char nowstr[LINESIZE];
  if (options->progress)
    {
      get_time (nowstr, "%H:%M:%S");
      fprintf (stdout, "%s   Final parameter estimation over all loci\n           using %s\n", nowstr, text);
    }
}












/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 D A T A   R O U T I N E S 

 creates data structures,
 read data (Electrophoretic loci, sequences, microsats),
 feeds data into tree (?),
 prints data,
 destroys data.
 
 
 Theta(1)=4 N(1)mu, Theta(2)=4 N(2)mu,
 M(1) = m(1)/mu, and M(2)= m(2)/mu
                                                                                                               
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: data.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/
#include <string.h>

#include "migration.h"
#include "data.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif
/* prototypes ----------------------------------------- */
void create_data (data_fmt ** data);
void get_data (FILE * infile, data_fmt * data, option_fmt * options);
void print_data (world_fmt * world, data_fmt * data, option_fmt * options);
void print_data_summary (FILE * file, world_fmt * world);
void free_datapart (data_fmt * data, option_fmt * options, long locus);
/*private functions */
void init_data_structure1 (data_fmt ** data, option_fmt * options);
void read_header (FILE * infile, data_fmt * data, option_fmt * options);
void read_sites (data_fmt * data);
void init_data_structure2 (data_fmt ** data, option_fmt * options, long pop);
void init_data_structure3 (data_fmt * data);
void read_popheader (FILE * infile, data_fmt * data, long pop);
void read_indname (FILE * file, data_fmt * data, long pop, long ind,
		   long nmlength);
void read_popdata (FILE * file, data_fmt * data, long pop,
		   option_fmt * options);
void read_microalleles (FILE * infile, data_fmt * data, long pop, long ind);
void read_alleles (FILE * infile, data_fmt * data, long pop, long ind);
long read_ind_seq (FILE * infile, data_fmt * data, option_fmt * options,
		   long locus, long pop, long ind, long baseread);
void finish_read_seq (FILE * infile, data_fmt * data,
		      option_fmt * options, long pop, long baseread);
void print_alleledata (world_fmt * world, data_fmt * data, option_fmt * options);
void print_seqdata (world_fmt * world, data_fmt * data);
void print_header (FILE * outfile, long pop, world_fmt * world);
void create_alleles (data_fmt * data);
void addAllele (data_fmt * data, char s[], long locus, long *z);
void set_numind (data_fmt * data);
void print_seq_pop (long locus, long pop, world_fmt * world);
void print_seq_ind (long locus, long pop, long ind, world_fmt * world);
void print_locus_head (long locus, world_fmt * world);
/*=====================================================*/
void
create_data (data_fmt ** data)
{
  (*data) = (data_fmt *) calloc (1, sizeof (data_fmt));
}

void
init_data (data_fmt * data)
{

}

void
destroy_data (data_fmt * data)
{
  free (data);
}



void
get_data (FILE * infile, data_fmt * data, option_fmt * options)
{
  long pop;
  read_header (infile, data, options);
  init_data_structure1 (&data, options);
  if (options->datatype == 's')
    read_sites (data);
  else
    data->seq->fracchange = 1.0;
  for (pop = 0; pop < data->numpop; pop++)
    {
      read_popheader (data->infile, data, pop);
      init_data_structure2 (&data, options, pop);
      read_popdata (data->infile, data, pop, options);
    }
  init_data_structure3 (data);
  set_numind (data);		/* replace this, if loci can have different number of ind */
  switch (options->datatype)
    {
    case 'a':
      create_alleles (data);
      break;
    case 'b':
    case 'm':
      create_alleles (data);
      for (pop = 0; pop < data->loci; pop++)
	data->maxalleles[pop] = options->micro_stepnum;
      break;
    }
}

/* private functions ========================================== */

void
init_data_structure1 (data_fmt ** data, option_fmt * options)
{
  long pop;
  if ((*data)->yy == NULL)
    {
      (*data)->yy = (char *****) malloc (sizeof (char ****) * (*data)->numpop);
      (*data)->seq = (seqmodel_fmt *) malloc (sizeof (seqmodel_fmt));
      (*data)->popnames = (char **) malloc (sizeof (char *) * (*data)->numpop);
      (*data)->indnames = (char ***) malloc (sizeof (char **) * (*data)->numpop);
      (*data)->numind = (long **) malloc (sizeof (long *) * (*data)->numpop);
      for (pop = 0; pop < (*data)->numpop; pop++)
	{
	  (*data)->popnames[pop] = (char *) malloc (sizeof (char) *
						    options->popnmlength);
	  (*data)->numind[pop] = (long *) malloc (sizeof (long) * (*data)->loci);
	}
/*     if(options->datatype=='s') */
      (*data)->seq->sites = (long *) calloc (1, sizeof (long) * (*data)->loci);
/*     else */
/*       (*data)->seq->sites = (long *) calloc(1,sizeof(long)); */
    }
  else
    {
      fprintf (stderr,
	       "data->yy is obviously initialized, and should not be!?\n");
      exit (EXIT_FAILURE);
    }
}


void
init_data_structure2 (data_fmt ** data, option_fmt * options, long pop)
{
  long ind, locus;
  (*data)->yy[pop] = (char ****) malloc (sizeof (char ***) * (*data)->numind[pop][FLOC]);
  (*data)->indnames[pop] = (char **) calloc (1, sizeof (char *) * (*data)->numind[pop][FLOC]);
  for (ind = 0; ind < (*data)->numind[pop][FLOC]; ind++)
    {
      (*data)->indnames[pop][ind] = (char *) calloc (1, sizeof (char) * (1 + options->nmlength));
      (*data)->yy[pop][ind] = (char ***) malloc (sizeof (char **) * (*data)->loci);
      for (locus = 0; locus < (*data)->loci; locus++)
	{
	  if (options->datatype != 's')
	    {
	      (*data)->yy[pop][ind][locus] = (char **) calloc (1, sizeof (char *) * 2);
	      (*data)->yy[pop][ind][locus][0] =
		(char *) calloc (1, sizeof (char) * options->allelenmlength);
	      (*data)->yy[pop][ind][locus][1] =
		(char *) calloc (1, sizeof (char) * options->allelenmlength);
	    }
	  else
	    {
	      (*data)->yy[pop][ind][locus] = (char **) calloc (1, sizeof (char *));
	      (*data)->yy[pop][ind][locus][0] =
		(char *) calloc (1, sizeof (char) * (*data)->seq->sites[locus]);
	    }
	}
    }
}


void
free_datapart (data_fmt * data, option_fmt * options, long locus)
{
  long ind, pop, genomes;
  if (options->datatype == 's')
    genomes = 1;
  else
    genomes = 2;
  for (pop = 0; pop < data->numpop; pop++)
    {
      for (ind = 0; ind < data->numind[pop][locus] / genomes; ind++)
	{
	  if (options->datatype != 's')
	    {
	      free (data->yy[pop][ind][locus][0]);
	      free (data->yy[pop][ind][locus][1]);
	      free (data->yy[pop][ind][locus]);
	    }
	  else
	    {
	      free (data->yy[pop][ind][locus][0]);
	      free (data->yy[pop][ind][locus]);
	    }
	}
    }
}


void
init_data_structure3 (data_fmt * data)
{
  long locus, pop, maxi;
  data->allele = (allele_fmt **)
    calloc (1, sizeof (allele_fmt *) * data->loci);
  for (locus = 0; locus < data->loci; locus++)
    {
      maxi = 0;
      for (pop = 0; pop < data->numpop; pop++)
	maxi += data->numind[pop][FLOC] * 2;
      data->allele[locus] = (allele_fmt *)
	calloc (1, sizeof (allele_fmt) * maxi);
    }
  data->maxalleles = (long *) calloc (1, sizeof (long) * data->loci);
  data->skiploci = (boolean *) calloc (1, sizeof (boolean) * data->loci);
}


void
read_header (FILE * infile, data_fmt * data, option_fmt * options)
{
  char input[LINESIZE], *p;
  fgets (input, sizeof (input), infile);
  if ((p = (char *) strchr (input, '\n')) != NULL)
    *p = '\0';
  switch (lowercase (input[0]))
    {
    case 'a':
      sscanf (input, "%1s%ld%ld%[^\n]", &options->datatype,
	      &(data->numpop), &(data->loci), options->title);
      break;
    case 'b':
    case 'm':
      sscanf (input, "%1s%ld%ld%1s%[^\n]", &options->datatype,
	      &(data->numpop), &(data->loci), &data->dlm, options->title);
      break;
    case 's':
      sscanf (input, "%1s%ld%ld%[^\n]", &options->datatype,
	      &(data->numpop), &(data->loci), options->title);
      break;
    default:
      switch (options->datatype)
	{
	case 'a':
	  sscanf (input, "%ld%ld%[^\n]", &(data->numpop),
		  &(data->loci), options->title);
	  break;
	case 'b':
	case 'm':
	  sscanf (input, "%ld%ld%1s%[^\n]", &(data->numpop),
		  &(data->loci), &(data->dlm), options->title);
	  break;
	case 's':
	  sscanf (input, "%ld%ld%[^\n]", &(data->numpop),
		  &(data->loci), options->title);
	  break;
	}
    }
  options->datatype = lowercase (options->datatype);
}

void
read_sites (data_fmt * data)
{
  long locus;
  char input[LINESIZE], *p, *a;
  fgets (input, sizeof (input), data->infile);
  if ((p = (char *) strchr (input, '\n')) != NULL)
    *p = '\0';
  for (locus = 0; locus < data->loci; locus++)
    {
      while (isspace ((int) *input))
	(*input)++;
      if (locus == 0)
	a = strtok (input, " ");
      else
	a = strtok (NULL, " ");
      data->seq->sites[locus] = atoi (a);
    }
}

void
read_popheader (FILE * infile, data_fmt * data, long pop)
{
  char input[LINESIZE], *p;
  fgets (input, sizeof (input), infile);
  if ((p = (char *) strchr (input, '\n')) != NULL)
    *p = '\0';
  sscanf (input, "%ld%[^\n]", &(data->numind[pop][FLOC]), data->popnames[pop]);
  translate (data->popnames[pop], ' ', '_');

}

void
read_indname (FILE * file, data_fmt * data, long pop, long ind, long nmlength)
{
  long i = 0;
  while (i < nmlength)
    data->indnames[pop][ind][i++] = getc (file);
  data->indnames[pop][ind][nmlength] = '\0';
}

void
read_popdata (FILE * infile, data_fmt * data, long pop, option_fmt * options)
{
  long ind, baseread = 0;
  long locus = 0;
  for (ind = 0; ind < data->numind[pop][FLOC]; ind++)
    {
      read_indname (infile, data, pop, ind, options->nmlength);
      switch (options->datatype)
	{
	case 'a':
	  read_alleles (infile, data, pop, ind);
	  break;
	case 'b':
	case 'm':
	  if (data->dlm == '\0')
	    read_alleles (infile, data, pop, ind);
	  else
	    read_microalleles (infile, data, pop, ind);
	  break;
	case 's':
	  baseread = read_ind_seq (infile, data, options, locus, pop, ind, 0);
	  break;
	default:
	  fprintf (stderr, "Wrong datatype, only the types a, m, s");
	  fprintf (stderr, " (electrophoretic alleles, \nmicrosatellite data, ");
	  fprintf (stderr, "sequence data) are allowed.\n");
	  exit (EXIT_FAILURE);
	  break;
	}
    }
  if (options->datatype != 's')
    return;
  else
    {
      finish_read_seq (infile, data, options, pop, baseread);
    }
}

void
read_microalleles (FILE * infile, data_fmt * data, long pop, long ind)
{
  char *input, *isave, dlm[2], ddlm[2], *p, *a, *a1, *a2;
  long locus, i;
  input = (char *) calloc (1, sizeof (char) * (LINESIZE + 1));
  isave = input;
  a = (char *) calloc (1, sizeof (char) * LINESIZE);
  a1 = (char *) calloc (1, sizeof (char) * LINESIZE);
  a2 = (char *) calloc (1, sizeof (char) * LINESIZE);
  dlm[0] = data->dlm, dlm[1] = '\0';
  ddlm[0] = ' ', ddlm[1] = '\0';
  fgets (input, LINESIZE, infile);
  if ((p = (char *) strchr (input, '\n')) != NULL)
    *p = '\0';
  for (locus = 0; locus < data->loci; locus++)
    {
      while (isspace ((int) *input))
	input++;
      if (input[0] == '\0')
	fgets (input, LINESIZE, infile);
      i = 0;
      while (input[i] != ' ' && input[i] != dlm[0])
	{
	  a1[i] = input[i];
	  i++;
	}
      a1[i] = '\0';
      input += i;
      i = 0;
      if (input[i] == dlm[0])
	{
	  input++;
	  while (input[i] != ' ' && input[i] != '\0')
	    {
	      a2[i] = input[i];
	      i++;
	    }
	  a2[i] = '\0';
	  if (a2[0] == '\0')
	    {
	      strcpy (a2, a1);
	    }
	  input += i;
	}
      else
	{
	  strcpy (a2, a1);
	}
      strcpy (data->yy[pop][ind][locus][0], a1);
      strcpy (data->yy[pop][ind][locus][1], a2);
    }
  free (a);
  free (a1);
  free (a2);
  free (isave);
}

void
read_alleles (FILE * infile, data_fmt * data, long pop, long ind)
{
  char *input, *p, *a;
  long locus, track = 0;
  a = (char *) calloc (1, sizeof (char) * LINESIZE);

  input = (char *) calloc (1, sizeof (char) * LINESIZE);
  fgets (input, LINESIZE, infile);
  if ((p = (char *) strchr (input, '\n')) != NULL)
    *p = '\0';
  for (locus = 0; locus < data->loci; locus++)
    {
      while (isspace ((int) *input))
	{
	  input++;
	  track++;
	}
      if (sscanf (input, "%s", a) == 1)
	{
	  input += strlen (a);
	  track += strlen (a);
	}

      data->yy[pop][ind][locus][0][0] = a[0];
      data->yy[pop][ind][locus][0][1] = '\0';
      if (a[1] == '\0')
	{
	  data->yy[pop][ind][locus][1][0] = a[0];
	  data->yy[pop][ind][locus][1][1] = '\0';
	}
      else
	{
	  data->yy[pop][ind][locus][1][0] = a[1];
	  data->yy[pop][ind][locus][1][1] = '\0';
	}
    }
  free (a);
  input -= track;
  free (input);
}

long
read_ind_seq (FILE * infile, data_fmt * data, option_fmt * options,
	      long locus, long pop, long ind, long baseread)
{
  long j;
  char charstate;
  j = (options->interleaved) ? baseread : 0;
  charstate = getc (infile);
  ungetc ((int) charstate, infile);
  while (j < data->seq->sites[locus] && !(options->interleaved && charstate == '\n'))
    {
      charstate = getc (infile);
      if (charstate == '\n')
	{
	  if (options->interleaved)
	    return j;
	  else
	    charstate = ' ';
	}
      if (charstate == ' ' || (charstate >= '0' && charstate <= '9'))
	continue;
      charstate = uppercase (charstate);
      if ((strchr ("ABCDGHKMNRSTUVWXY?O-", (int) charstate)) == NULL)
	{
	  printf ("ERROR: BAD BASE: %c AT POSITION %5ld OF INDIVIDUUM %3li in POPULATION %ld\n",
		  charstate, j, ind, pop);
	  exit (EXIT_FAILURE);
	}
      data->yy[pop][ind][locus][0][j++] = charstate;
    }
  charstate = getc (infile);	/* swallow the \n */
  return j;
}


void
finish_read_seq (FILE * infile, data_fmt * data,
		 option_fmt * options, long pop, long baseread)
{

  long ind, baseread2 = 0, locus = 0;
  if (options->interleaved)
    {
      while (baseread < data->seq->sites[0])
	{
	  for (ind = 0; ind < data->numind[pop][FLOC]; ind++)
	    {
	      baseread2 = read_ind_seq (infile, data, options, locus, pop, ind, baseread);
	    }
	  baseread = baseread2;
	}
    }
  for (locus = 1; locus < data->loci; locus++)
    {
      baseread = 0;
      for (ind = 0; ind < data->numind[pop][FLOC]; ind++)
	{
	  read_indname (infile, data, pop, ind, options->nmlength);
	  baseread = read_ind_seq (infile, data, options, locus, pop, ind, 0);
	}
      if (options->interleaved)
	{
	  while (baseread < data->seq->sites[locus])
	    {
	      for (ind = 0; ind < data->numind[pop][FLOC]; ind++)
		{
		  baseread2 = read_ind_seq (infile, data, options, locus, pop, ind, baseread);
		}
	      baseread = baseread2;
	    }
	}
    }
}

void
print_data_summary (FILE * file, world_fmt * world)
{
  long total = 0;
  long pop;
  fprintf (file, "\n\nSummary of data:\n");
  fprintf (file, "---------------\n");
  fprintf (file, "Datatype:                               %20s\n",
	   world->options->datatype == 'a' ?
	   "Allelic data" : (world->options->datatype == 's' ?
			     "Sequence data" : "Microsatellite data"));

  fprintf (file, "Number of loci:                         %20li\n\n", world->data->loci);

  fprintf (file, "Population                                       Individuals\n");
  fprintf (file, "------------------------------------------------------------\n");
  for (pop = 0; pop < world->numpop; pop++)
    {
      fprintf (file, "%3li %-50.50s%6li\n", pop + 1, world->data->popnames[pop], world->data->numind[pop][FLOC]);
      total += world->data->numind[pop][FLOC];
    }
  fprintf (file, "Total of all populations                              %6li\n\n", total);

}

void
print_data (world_fmt * world, data_fmt * data, option_fmt * options)
{
  if (options->printdata)
    {
      switch (options->datatype)
	{
	case 'a':
	case 'b':
	case 'm':
	  print_alleledata (world, data, options);
	  break;
	case 's':
	  print_seqdata (world, data);
	  break;
	}
    }
}

void
print_alleledata (world_fmt * world, data_fmt * data, option_fmt * options)
{
  long i, pop, ind, locus, mult80;
  for (pop = 0; pop < data->numpop; pop++)
    {
      print_header (world->outfile, pop, world);
      for (ind = 0; ind < data->numind[pop][FLOC]; ind++)
	{
	  fprintf (world->outfile, "%-*.*s ",
		   (int) options->nmlength, (int) options->nmlength,
		   data->indnames[pop][ind]);
	  mult80 = options->nmlength;
	  for (locus = 0; locus < data->loci; locus++)
	    {
	      mult80 += 1 + strlen (data->yy[pop][ind][locus][0]) + strlen (data->yy[pop][ind][locus][1]);
	      if (mult80 >= 80)
		{
		  mult80 = 0;
		  fprintf (world->outfile, "\n");
		  for (i = 0; i < options->nmlength; i++)
		    fputc (' ', world->outfile);
		}
	      fprintf (world->outfile, " %s.%-s",
		       data->yy[pop][ind][locus][0],
		       data->yy[pop][ind][locus][1]);
	    }
	  fprintf (world->outfile, "\n");
	}
      fprintf (world->outfile, "\n");
    }
  fprintf (world->outfile, "\n\n");
}

void
print_seqdata (world_fmt * world, data_fmt * data)
{
  long pop, locus;
  for (pop = 0; pop < data->numpop; pop++)
    {
      print_header (world->outfile, pop, world);
      for (locus = 0; locus < data->loci; locus++)
	{
	  print_locus_head (locus, world);
	  print_seq_pop (locus, pop, world);
	}
    }
}

void
print_header (FILE * outfile, long pop, world_fmt * world)
{
  long i;
  long locus, mult80 = 80;
  char input[LINESIZE];
  fprintf (outfile, "\n%-s", world->data->popnames[pop]);
  for (i = 0; i < (long) (80 - strlen (world->data->popnames[pop])); i++)
    fputc ('-', outfile);
  fprintf (outfile, "\n\n");
  if (world->options->datatype != 's')
    {
      fprintf (outfile, "%-s  ", (world->data->loci == 1 ? "locus" : "loci "));
      for (i = 0; i < (world->options->nmlength - 6); i++)
	fputc (' ', outfile);
      for (locus = 0; locus < world->data->loci; locus++)
	{
	  if (locus * 4 + world->options->nmlength > mult80)
	    {
	      mult80 += 80;
	      fprintf (outfile, "\n");
	      for (i = 0; i < world->options->nmlength; i++)
		fputc (' ', outfile);
	    }
	  fprintf (outfile, "  %2li", locus + 1);
	}
      fprintf (outfile, "\n%-s\n", strncpy (input, "indiv.", world->options->nmlength));
    }
}



void
create_alleles (data_fmt * data)
{
  long locus, pop, ind;
  long z;
  char a1[DEFAULT_ALLELENMLENGTH];
  char a2[DEFAULT_ALLELENMLENGTH];
  for (locus = 0; locus < data->loci; locus++)
    {
      z = 0;
      for (pop = 0; pop < data->numpop; pop++)
	{
	  for (ind = 0; ind < data->numind[pop][locus]; ind++)
	    {
	      strcpy (a1, data->yy[pop][ind][locus][0]);
	      strcpy (a2, data->yy[pop][ind][locus][1]);
	      if (!strcmp (a1, a2))
		{
		  addAllele (data, a1, locus, &z);
		}
	      else
		{
		  addAllele (data, a1, locus, &z);
		  addAllele (data, a2, locus, &z);
		}
	    }
	}
      data->maxalleles[locus] = z + 1;
      /* + 1: for all the unencountered alleles */
    }
}

void
addAllele (data_fmt * data, char s[], long locus, long *z)
{
  long found = 0;
  while ((data->allele[locus][found++][0] != '\0')
	 && (strcmp (s, data->allele[locus][found - 1])));
  if (found > (*z))
    {
      strcpy (data->allele[locus][*z], s);
      (*z)++;
    }
}

void
set_numind (data_fmt * data)
{
  long locus, pop;
  for (locus = 1; locus < data->loci; locus++)
    {
      for (pop = 0; pop < data->numpop; pop++)
	{
	  data->numind[pop][locus] = data->numind[pop][FLOC];
	}
    }
}


void
print_seq_pop (long locus, long pop, world_fmt * world)
{
  long ind;
  for (ind = 0; ind < world->data->numind[pop][locus]; ind++)
    {
      print_seq_ind (locus, pop, ind, world);
    }
}

void
print_seq_ind (long locus, long pop, long ind, world_fmt * world)
{
  long site;
  char blank[2] = " ";
  fprintf (world->outfile, "%-*.*s", (int) world->options->nmlength,
	   (int) world->options->nmlength,
	   world->data->indnames[pop][ind]);
  fprintf (world->outfile, " %c",
	   world->data->yy[pop][ind][locus][0][0]);
  for (site = 1; site < world->data->seq->sites[locus]; site++)
    {
      if ((site) % 60 == 0)
	{
	  fprintf (world->outfile, "\n%-*.*s %c", (int) world->options->nmlength,
		   (int) world->options->nmlength,
		   blank,
		   world->data->yy[pop][ind][locus][0][site]);
	}
      else
	{
	  if ((site) % 10 == 0)
	    {
	      fprintf (world->outfile, " ");
	    }
	  fprintf (world->outfile, "%c",
		   world->data->yy[pop][ind][locus][0][site]);
	}
    }
  fprintf (world->outfile, "\n");
}


void
print_locus_head (long locus, world_fmt * world)
{
  char *head;
  head = (char *) calloc (1, sizeof (char) * MAX (10, world->options->nmlength));
  sprintf (head, "Locus %li", locus);
  fprintf (world->outfile, "%-*.*s --------10 --------20 --------30",
	   (int) world->options->nmlength,
	   (int) world->options->nmlength,
	   head);
  fprintf (world->outfile, " --------40 --------50 --------60\n");

  free (head);
}

/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 F S T   R O U T I N E S 

 calculates FST

 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: fst.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"
#include "tools.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

/* prototypes ------------------------------------------- */
void fst_type (char type);
void calc_fst (world_fmt * world);
/* private functions */
void frequencies (double ***f, double **f2, char *****data, long numpop, long **numind, long loci);
void calc_fw (double ***f, long numpop, long locus, double *fw);
void calc_fb (double ***f, long numpop, long locus, double *fb);
void calc_seq_fw (data_fmt * data, long numpop, long locus, double *fw);
void calc_seq_fb (data_fmt * data, long numpop, long locus, double *fb);
void solveNm_vartheta (double *fw, double *fb, long numpop, double *params);
void solveNm_varm (double *fw, double *fb, long numpop, double *params);

/* global variable SOLVENM points to function solveNm_varxx() */
static void (*solveNm) (double *, double *, long, double *);

/*=======================================================*/
void
fst_type (char type)
{
  if (type == 'T')
    solveNm = (void (*)(double *, double *, long, double *)) solveNm_vartheta;
  else
    solveNm = (void (*)(double *, double *, long, double *)) solveNm_varm;
}

void
calc_fst (world_fmt * world)
{
  long pop, locus;
  long connections = world->numpop * (world->numpop - 1) / 2;
  double ***fstfreq = NULL, **fstfreq2 = NULL, *fw, *fb, *sumfb, *sumfw;
  fw = (double *) calloc (1, sizeof (double) * world->numpop);
  fb = (double *) calloc (1, sizeof (double) * connections);
  sumfw = (double *) calloc (1, sizeof (double) * world->numpop);
  sumfb = (double *) calloc (1, sizeof (double) * connections);
  if (world->options->datatype != 's')
    {
      fstfreq = (double ***) calloc (1, sizeof (double **) * world->numpop);
      fstfreq2 = (double **) calloc (1, sizeof (double *) * world->loci);
      for (locus = 0; locus < world->loci; locus++)
	fstfreq2[locus] = (double *) calloc (1, sizeof (double) * 255);
      for (pop = 0; pop < world->numpop; pop++)
	{
	  fstfreq[pop] = (double **) calloc (1, sizeof (double *) * world->loci);
	  for (locus = 0; locus < world->loci; locus++)
	    fstfreq[pop][locus] = (double *) calloc (1, sizeof (double) * 255);
	}
      frequencies (fstfreq, fstfreq2, world->data->yy, world->numpop,
		   world->data->numind, world->loci);
    }
  for (locus = 0; locus < world->loci; locus++)
    {
      if (world->options->datatype == 's')
	{
	  calc_seq_fw (world->data, world->numpop, locus, fw);
	  calc_seq_fb (world->data, world->numpop, locus, fb);
	}
      else
	{
	  calc_fw (fstfreq, world->numpop, locus, fw);
	  calc_fb (fstfreq, world->numpop, locus, fb);
	}

      (*solveNm) (fw, fb, world->numpop, world->fstparam[locus]);

      for (pop = 0; pop < world->numpop; pop++)
	{
	  sumfw[pop] += fw[pop];
	}
      for (pop = 0; pop < connections; pop++)
	{
	  sumfb[pop] += fb[pop];
	}
    }
  for (pop = 0; pop < world->numpop; pop++)
    {
      sumfw[pop] /= world->loci;
    }
  for (pop = 0; pop < connections; pop++)
    {
      sumfb[pop] /= world->loci;
    }

  (*solveNm) (sumfw, sumfb, world->numpop, world->fstparam[world->loci]);
  if (world->options->datatype != 's')
    {
      for (pop = 0; pop < world->numpop; pop++)
	{
	  for (locus = 0; locus < world->loci; locus++)
	    free (fstfreq[pop][locus]);
	  free (fstfreq[pop]);
	}
      for (locus = 0; locus < world->loci; locus++)
	free (fstfreq2[locus]);
      free (fstfreq);
      free (fstfreq2);
    }
  free (fw);
  free (fb);
  free (sumfw);
  free (sumfb);
}


/*=======================================================*/

void
frequencies (double ***f, double **f2, char *****data, long numpop,
	     long **numind, long loci)
{
  long **buckets, *buckets2;
  long *total, pop, locus, ind, a;
  buckets = (long **) malloc (sizeof (long *) * numpop);
  buckets2 = (long *) calloc (1, sizeof (long) * 255);
  total = (long *) calloc (1, sizeof (long) * numpop);
  for (pop = 0; pop < numpop; pop++)
    {
      buckets[pop] = (long *) calloc (1, sizeof (long) * 255);
      for (locus = 0; locus < loci; locus++)
	{
	  memset (buckets[pop], 0, sizeof (long) * 255);
	  total[pop] = 0;
	  for (ind = 0; ind < numind[pop][FLOC]; ind++)
	    {
	      if (data[pop][ind][locus][0][0] != '?')
		{
		  buckets[pop][data[pop][ind][locus][0][0] - '!'] += 1;
		  buckets2[data[pop][ind][locus][0][0] - '!'] += 1;
		  total[pop] += 1;
		}
	      if (data[pop][ind][locus][1][0] != '?')
		{
		  buckets[pop][data[pop][ind][locus][1][0] - '!'] += 1;
		  buckets2[data[pop][ind][locus][1][0] - '!'] += 1;
		  total[pop] += 1;
		}
	      for (a = 0; a < 255; a++)
		{
		  if (total[pop] > 0)
		    f[pop][locus][a] = (double) buckets[pop][a] / (double) total[pop];
		  if (total[0] + total[1] > 0)
		    f2[locus][a] = (double) buckets2[a] / ((double) total[0] + total[1]);
		}
	    }
	}
      free (buckets[pop]);
    }
  free (total);
  free (buckets);
  free (buckets2);
}



void
calc_fw (double ***f, long numpop, long locus, double *fw)
{
  long pop, i;
  for (pop = 0; pop < numpop; pop++)
    {
      fw[pop] = 0;
      for (i = 0; i < 255; i++)
	{
	  fw[pop] += f[pop][locus][i] * f[pop][locus][i];
	}
    }
}
void
calc_fb (double ***f, long numpop, long locus, double *fb)
{
  long i, p1, p2, zz = 0;
  for (p1 = 0; p1 < numpop; p1++)
    {
      for (p2 = p1 + 1; p2 < numpop; p2++)
	{
	  fb[zz] = 0.0;
	  for (i = 0; i < 255; i++)
	    {
	      fb[zz] += f[p1][locus][i] * f[p2][locus][i];
	    }
	  zz++;
	}
    }
}

void
calc_seq_fw (data_fmt * data, long numpop, long locus, double *fw)
{
  long pop, i, k, j;
  double nn;
  double diff;
  for (pop = 0; pop < numpop; pop++)
    {
      fw[pop] = 0;
      nn = data->seq->sites[locus] * (data->numind[pop][locus] *
		  data->numind[pop][locus] - data->numind[pop][locus]) / 2.;
      for (i = 0; i < data->numind[pop][locus]; i++)
	{
	  for (k = i + 1; k < data->numind[pop][locus]; k++)
	    {
	      diff = 0.;
	      for (j = 0; j < data->seq->sites[locus]; j++)
		{
		  diff +=
		    (data->yy[pop][i][locus][0][j] !=
		     data->yy[pop][k][locus][0][j]);
		}
	      if (nn > 0)
		fw[pop] += diff / nn;
	    }
	}
      fw[pop] = 1. - fw[pop];
    }
}

void
calc_seq_fb (data_fmt * data, long numpop, long locus, double *fb)
{
  long i, k, j;
  double nn, temp;
  double diff;
  long p1, p2, zz = 0;
  for (p1 = 0; p1 < numpop; p1++)
    {
      for (p2 = p1 + 1; p2 < numpop; p2++)
	{
	  temp = 0;
	  nn = data->seq->sites[locus] * data->numind[p1][locus] *
	    data->numind[p2][locus];
	  for (i = 0; i < data->numind[p1][locus]; i++)
	    {
	      for (k = 0; k < data->numind[p2][locus]; k++)
		{
		  diff = 0.;
		  for (j = 0; j < data->seq->sites[locus]; j++)
		    {
		      diff +=
			(data->yy[p1][i][locus][0][j] !=
			 data->yy[p2][k][locus][0][j]);
		    }
		  if (nn > 0)
		    temp += diff / nn;
		}
	    }
	  fb[zz++] = 1. - temp;
	}
    }
}

void
solveNm_varm (double *fw, double *fb, long numpop, double *params)
{
  long i, p1 /*,p2 */ ;
  /*Version 2.0  double sumfw = sum(fw,numpop);
     double sumfb = sum(fb,numpop*(numpop-1)/2); */
  double first = (2. - fw[0] - fw[1]) / (2.*fb[0] + fw[0] + fw[1]);
  long offset2 = numpop + numpop * (numpop - 1);
  long offset = numpop;
  long numfb = 0;

  for (p1 = 0; p1 < numpop; p1++)
    {
      numfb += p1;
      params[p1] = first;
      params[offset2 + p1] = fw[p1];
    }
  params[offset] = (2.*fb[0]-fw[0]-2. * fb[0]*fw[0]+fw[1])/
    ((fb[0]-fw[0])*(-2. + fw[0] + fw[1]));
  params[offset + 1] = (2.*fb[0]-fw[1]-2. * fb[0]*fw[1]+fw[0])/
    ((fb[0]-fw[1])*(-2. + fw[0] + fw[1]));
    
  for (i = 0; i < offset2; i++)
    {
      if (params[i] < 0.)
	params[i] = -999;
    }
  for (i = 0; i < numfb; i++)
    {
      params[offset2 + numpop + i] = fb[i];
    }
}

void
solveNm_vartheta (double *fw, double *fb, long numpop, double *params)
{
  long i;
  double nom;
  double denom;
  /* Version 2.0  double sumfw = sum(fw,numpop);
     double sumfb = sum(fb,numpop*(numpop-1)/2); */

  long offset2 = numpop + numpop * (numpop - 1);
  long numfb = 0;
  nom = (-2. * fb[0] + fw[0] + fw[1]);
  for (i = 0; i < numpop; i++)
    {
      numfb += i;
    }
  denom = -2.*fb[0] * fb[0] + fw[0] * fw[1];
  for (i = 0; i < numpop; i++)
    {
      params[offset2 + i] = fw[i];
      params[i] = (nom * (1.-fw[i]))/(denom + fw[i] * fw[i]);
    }
  if(nom==0.0)
    params[numpop] = -999.;
  else
    params[numpop] = 2. * fb[0] / nom;
  for (i = 1; i < numpop * (numpop - 1); i++)
    {
      params[numpop + i] = params[numpop];
    }
  for (i = 0; i < offset2; i++)
    {
      if (params[i] < 0.)
	params[i] = -999;
    }
  for (i = 0; i < numfb; i++)
    {
      params[offset2 + numpop + i] = fb[i];
    }
}










/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 M A I N   P R O G R A M 

 drives the whole program MIGRATE.

 for options of the program: README and parmfile.distrib, 
                             options.c
 
 Theta(1)=4 N(1)mu, Theta(2)=4 N(2)mu,
 M(1) = m(1)/mu, and M(2)= m(2)/mu
                                                                                                               
 Peter Beerli 1997, Seattle
 beerli@genetics.washington.edu
 $Id: main.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/


#include "migration.h"
#include "world.h"
#include "data.h"
#include "options.h"
#include "mcmc.h"
#include "parameter.h"
#include "combine.h"
#include "menu.h"
#include "sequence.h"
#include "tree.h"


#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

char appl[8];
longer seed;

/*--------------------------------------------
   main program 


*/
int 
main (int argc, char **argv)
{
  long locus, i, j, accepted, G = 1, increment, outfilepos, treefilepos;
  long step, steps, oldsteps, chain, chains, runs;
  char type;
  char *this_string;
  data_fmt *data;
  world_fmt *world;
  option_fmt *options;
  this_string = (char *) calloc (1, sizeof (char) * 256);
  strcpy (appl, "Migrate");
  /* try to catch and beautify some error messages -------------- */
  signalhandling (ON);
  /* create main data structures -------------------------------- */
  create_data (&data);
  create_world (&world);
  create_options (&options);
  /* parmfile and menu ------------------------------------------ */
  init_options (options);
  get_options (options);
  print_menu_title (options);
  get_menu (options);
  /* initialization and data reading phase ---------------------- */
  init_files (world, data, options);
  get_data (data->infile, data, options);
  init_world (world, data, options);
  calc_simple_param (world);
  print_menu_options (options, world);
  outfilepos = print_title (world);
  print_options (world->outfile, options, world);
  print_data_summary (world->outfile, world);
  print_data (world, data, options);
  /* loop over all loci ---------------------------------------- */
  for (locus = 0; locus < data->loci; locus++)
    {
      world->locus = locus;
      set_param (world, data, options, locus);
      print_menu_locus (options, locus);
      type = 's';
      runs = 1;
      buildtree (world, locus);
      if (options->datatype == 's')
	{
	  print_seqfreqs (world);
	  print_tbl (world, locus);
	  print_weights (world, locus);
	}
      free_datapart (data, options, locus);
      if (data->skiploci[locus])
	{			/* skip loci with no tips */
	  cleanup_world (world, locus);
	  continue;
	}
      create_treetimelist (world, &world->treetimes, locus);
      fix_times (world);
      first_smooth (world, locus);
      world->likelihood[0] = treelikelihood (world);
      world->allikemax = -DBL_MAX;	/* for best tree option */
      /* short and long chains ------------------------------------- */
      set_bounds (&increment, &oldsteps, &chains, options, type);
      while (runs-- >= 0)
	{
	  print_menu_chain (type, FIRSTCHAIN, oldsteps, world);
	  if (world->options->treeprint == ALL)
	    print_tree (world, 0, &treefilepos);
	  for (chain = 0; chain < chains ||
	       (type == 'l' && chain >= chains
		&& world->param_like > options->lcepsilon); chain++)
	    {
	      memset (world->likelihood + 1, 0, sizeof (double) * (world->atl[0].allocT - 1));
	      burnin_chain (world);	/*removes start conditions */
	      G = 1;
	      accepted = 0;
	      steps = oldsteps;
	      world->maxdatallike = world->likelihood[0];
	      if ((type == 'l') && (chain == chains - 1))
		{
		  world->in_last_chain = TRUE;
		}
	      /* steps for each chain ----------------------------------- */
	      for (step = 0; step < steps; step++)
		{
		  j = 0;
		  /* hop over INCREMENT trees, and sample only last */
		  for (i = 0; i < increment; i++)
		    {
		      j += metropolize (world, G - 1);

		      if (world->likelihood[G - 1] > world->maxdatallike)
			{
			  world->maxdatallike = world->likelihood[G - 1];
			}
		    }
		  /* store condensed tree information for estimation */
		  if (step == 0)
		    {
		      copy_time (world, world->treetimes,
				 FIRSTSTEP, 0, world->numpop);
		      accepted += j;
		    }
		  else
		    {
		      copy_time (world, world->treetimes, G - 1,
				 G - 1 + (long) (j > 0), world->numpop);
		      G += (long) (j > 0);
		      accepted += j;
		    }
		  if (step >= oldsteps - 1 && options->movingsteps)
		    {
		      if (G < options->acceptfreq * oldsteps)
			{
			  steps++;
			}
		    }
		}
	      /*estimate locus-parameters */
	      decide_plot (options, chain, chains, type);
	      memcpy (world->param00, world->param0, sizeof (double) * world->numpop2);
	      estimateParameter (&world->atl[0], G, world, world->cov[locus], chain,
				 type, world->plane[locus]);
	      world->likelihood[0] = world->likelihood[G - 1];
	      world->treetimes[0].copies = 0;
	      if (options->progress)
		{
		  print_menu_coalnodes (world, G);
		  print_menu_accratio (accepted, steps * increment);
		}
	    }
	  /* switch to LONG chains
	     and reset stopping criteria if we have set
	     OPTIONS->LCEPSILON */
	  if (type == 's')
	    {
	      type = 'l';
	      create_treetimelist (world, &world->treetimes, locus);
	      set_bounds (&increment, &oldsteps, &chains, options, type);
	    }
	  if (runs < 0)
	    {
	      if (type == 'l' && world->param_like > LONGCHAINEPSILON)
		{
		  runs++;
		  continue;
		}
	      copy_atl (world, &world->atl[0],
			&world->atl[locus + 1], G - 1);
	    }
	}
      cleanup_world (world, locus);
    }
  /* multiple loci combination ---------------------------------- */
  if (world->loci - world->skipped > 1)
    {
      combine_loci (world);
    }
  /* printing of data ------------------------------------------- */
  if (options->simulation)
    print_simresults (world);
  else
    {
      print_list (world);
    }
  /* closing all files ------------------------------------------ */
  print_finish (world, outfilepos);
#ifdef MAC
  fixmacfile (options->outfilename);
  if (options->plotmethod == PLOTALL)
    fixmacfile (options->mathfilename);
  if (options->treeprint)
    fixmacfile (options->treefilename);
#endif
  exit_files (world, data, options);
  return 0;
}



/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effective population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 M C M C   R O U T I N E S 

 Markov Monte Carlo stuff: treechange, acceptance
 

 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: mcmc1.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"
#include "random.h"
#include "tree.h"
#include "mcmc2.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

#define MIGRATION_AIR (boolean) 1
#define MIGRATION_IN_TREE (boolean) 0
#define NO_MIGR_NODES 0
#define WITH_MIGR_NODES 1
#define MIGRATION_LIMIT 1000
/* prototypes ------------------------------------------- */
long metropolize (world_fmt * world, long g);
/* private functions */
void new_localtimelist (timelist_fmt ** ntl, timelist_fmt * otl);
void new_proposal (proposal_fmt ** proposal, timelist_fmt * tl, world_fmt * world);
void chooseOrigin (proposal_fmt * proposal);
void construct_localtimelist (timelist_fmt * timevector, proposal_fmt * proposal);
void traverseAllNodes (node * theNode, node *** nodelist, long *node_elem, long *oldnode_elem, int include_migration);
void add_locallineages (timelist_fmt * timevector, proposal_fmt * proposal);
void chooseTarget (proposal_fmt * proposal, timelist_fmt * timevector,
		   node ** bordernodes, long *bordernum);
void findbordernodes (node * theNode, proposal_fmt * proposal, long pop,
		      node ** bordernodes, long *bordernum,
		      vtlist ** tyme, long gte);
void free_proposal (proposal_fmt * proposal);
void free_timevector (timelist_fmt * timevector);
long remove_doublettes (timelist_fmt * timevector, node ** ptr);
long xor (node ** ptrl1, node ** ptrl2);
long rmigrcount (proposal_fmt * proposal);

int migrate (proposal_fmt * proposal, node * up, long *old_migr_table_counter, boolean air);
int pre_population (proposal_fmt * proposal, vtlist * ltime, long gte, long *slider);
/* boolean same_pop(node * up, double tyme, long pop); */
boolean acceptlike (world_fmt * world, proposal_fmt * proposal, long g, timelist_fmt * tyme);
double eventtime (proposal_fmt * proposal, long pop, vtlist * tentry, char *event, long g);
node *showsister (node * theNode);
void count_migrations (node * p, long *count);
long interior_pop0 (timelist_fmt * timevector, proposal_fmt * proposal);

/*=======================================================*/
long 
metropolize (world_fmt * world, long g)
{				/*return 1 if tree was accepted 0 otherwise */
  static long treefilepos;	/* write position in the treefile */
  boolean coalesced;
  boolean test;
  char event;
  long slider;
  long bordernum;
  long actualpop = -99, zz;
  double endtime, nexttime, age;

  proposal_fmt *proposal;	/*scratchpad  in which all involved nodes
				   are recorded and help arrays, e.g. migration arrays,
				   are stored */
  timelist_fmt *timevector;	/* local timelist */
  vtlist *tentry;		/*pointer into timeslice */

  /* ---------------------------------------------------------
     initialize local timelist and construct residual timelist 
     find the start node and snip of the branch and node below */
#ifdef MAC
  eventloop ();
#endif
  new_localtimelist (&timevector, &world->treetimes[0]);
  new_proposal (&proposal, &world->treetimes[0], world);
  chooseOrigin (proposal);
  construct_localtimelist (timevector, proposal);
  tentry = &(*timevector).tl[0];
  if (proposal->origin->tyme == 0.0)
    {
      age = 0.0;
    }
  else
    {
      age = proposal->origin->tyme;
      zz = 0;
      while (tentry->age < age && zz < (*timevector).T)
	{
	  tentry = &(*timevector).tl[zz];
	  zz++;
	}
    }
  nexttime = tentry->age;
  if ((*timevector).T > 1)
    endtime = (*timevector).tl[(*timevector).T - 2].age;
  else
    endtime = 0.0;
  proposal->time = age;
  coalesced = FALSE;
  /*------------------------------------
    main loop: sliding down the tree  */
  slider = 0;
  while (nexttime <= endtime)
    {
      actualpop = (proposal->migr_table_counter > 0) ?
	proposal->migr_table[proposal->migr_table_counter - 1].from :
	proposal->origin->pop;
      proposal->time = age + eventtime (proposal, actualpop, tentry, &event, g);
      if (proposal->time < nexttime)
	{
	  if (event == 'm')
	    {
	      if (!migrate (proposal, proposal->origin,
			  &proposal->old_migr_table_counter, MIGRATION_AIR))
		{
		  free_proposal (proposal);
		  free_timevector (timevector);
		  return 0;
		}
	      age = proposal->time;
	      continue;
	    }
	  else
	    {			/*coalesce */
	      if (event != 'c')	/* saveguard */
		exit (-1);
	      chooseTarget (proposal, timevector, proposal->bordernodes, &bordernum);
	      if (proposal->target != NULL)
		{
		  pretendcoalesce1p (proposal);
		  coalesced = TRUE;
		  break;
		}
	      else
		{
		  fprintf (stderr, "\n\n\n###########################################\n");
		  fprintf (stderr, "Target=NULL, proposal->time=%f\n", proposal->time);
		  fprintf (stderr, "###########################################\n\n\n\n\n");
		  return 0;
		}
	    }
	}			/*end if proposal->time < nextime */
      age = nexttime;
      tentry = &(*timevector).tl[(tentry->slice) + 1];	/*next entry in timelist */
      nexttime = tentry->age;
    }
  if (!coalesced)
    {
      if (!pre_population (proposal, (*timevector).tl, (*timevector).T - 1, &slider))
	{
	  free_proposal (proposal);
	  free_timevector (timevector);
	  return 0;
	}
      pretendcoalesce1p (proposal);
    }
  test = acceptlike (world, proposal, g, timevector);
  if (test)
    {
      if (proposal->time > world->root->tyme)
	{			/*saveguard */
	  world->root->tyme += proposal->time;
	  fprintf (stdout, "ROOT moved from %f to %f because population joining moves to %f\n",
		   world->root->tyme - proposal->time, world->root->tyme, proposal->time);
	}
      coalesce1p (proposal);
      world->likelihood[g] = treelikelihood (world);	/*recalculate the p->x */
      if (fabs (world->likelihood[g] - proposal->likelihood) > EPSILON)
	fprintf (stderr, "STRANGE: proposed and new likelihood differ: %f != %f\n",
		 proposal->likelihood, world->likelihood[g]);
      construct_tymelist (world, &world->treetimes[0]);		/* create a new timelist */
      if (world->options->treeprint != NONE)
	print_tree (world, g, &treefilepos);
      world->migration_counts = 0;	/* report the number of migration on the tree */
      count_migrations (world->root->next->back, &world->migration_counts);
      free_proposal (proposal);
      free_timevector (timevector);
      return 1;			/* new tree accepted */
    }

  free_proposal (proposal);
  free_timevector (timevector);
  return 0;			/* not accepted */

}



/*=======================================================*/
void 
new_localtimelist (timelist_fmt ** ntl, timelist_fmt * otl)
{
  (*ntl) = (timelist_fmt *) calloc (1, sizeof (timelist_fmt) * 1);
  (*ntl)[0].tl = (vtlist *) malloc ((*otl).allocT * sizeof (vtlist));
  (*ntl)[0].allocT = (*otl).allocT;
  (*ntl)[0].T = (*otl).T;
  memcpy ((*ntl)[0].tl, (*otl).tl, (*otl).allocT * sizeof (vtlist));
}

void 
new_proposal (proposal_fmt ** proposal, timelist_fmt * tl, world_fmt * world)
{
  long j;
  long mal = world->data->maxalleles[world->locus];
  (*proposal) = (proposal_fmt *) calloc (1, sizeof (proposal_fmt));
  (*proposal)->listsize = ((*tl).allocT + (*tl).T + 5);
  (*proposal)->aboveorigin = (node **) calloc (1, sizeof (node *) * (*proposal)->listsize);
  (*proposal)->bordernodes = (node **) calloc (1, sizeof (node *) * (*proposal)->listsize);
  (*proposal)->world = world;
  (*proposal)->datatype = world->options->datatype;
  (*proposal)->sumtips = world->sumtips;
  (*proposal)->numpop = world->numpop;
  (*proposal)->endsite = world->data->seq->endsite;
  (*proposal)->fracchange = world->data->seq->fracchange;
  (*proposal)->param0 = world->param0;
  (*proposal)->root = world->root;
  (*proposal)->migration_model = world->options->migration_model;
  (*proposal)->line_f = (node **) calloc (1, sizeof (node *) * (*proposal)->sumtips);
  (*proposal)->line_t = (node **) calloc (1, sizeof (node *) * (*proposal)->sumtips);
  if ((*proposal)->datatype == 's')
    {
      (*proposal)->xf.s = (phenotype) malloc (world->data->seq->endsite * sizeof (ratelike *));
      (*proposal)->xt.s = (phenotype) malloc (world->data->seq->endsite * sizeof (ratelike *));
      for (j = 0; j < (*proposal)->endsite; j++)
	{
	  (*proposal)->xf.s[j] = (ratelike) malloc (world->options->rcategs * sizeof (sitelike));
	  (*proposal)->xt.s[j] = (ratelike) malloc (world->options->rcategs * sizeof (sitelike));
	}
    }
  else
    {
      (*proposal)->xf.a = (double *) calloc (1, sizeof (double) * mal);
      (*proposal)->xt.a = (double *) calloc (1, sizeof (double) * mal);
    }
  (*proposal)->old_migr_table_counter = 4 * (*proposal)->sumtips /* 100 */ ;
  (*proposal)->old_migr_table_counter2 = 4 * (*proposal)->sumtips /* 100 */ ;
  (*proposal)->migr_table = (migr_table_fmt *) calloc (1,
	     sizeof (migr_table_fmt) * (*proposal)->old_migr_table_counter);
  (*proposal)->migr_table2 = (migr_table_fmt *) calloc (1,
	    sizeof (migr_table_fmt) * (*proposal)->old_migr_table_counter2);
  (*proposal)->migr_table_counter = 0;
  (*proposal)->migr_table_counter2 = 0;
}

void 
chooseOrigin (proposal_fmt * proposal)
{
  long elem = 0, oldelem = (proposal->sumtips * 2);
  node *tmp, **goal;
  goal = (node **) calloc (1, sizeof (node *) * oldelem);


  traverseAllNodes (crawlback (proposal->root->next), &goal, &elem, &oldelem,
		    NO_MIGR_NODES);
  tmp = goal[RANDINT (0, elem - 2)];
  proposal->origin = tmp;
  if (proposal->origin != showtop (crawlback (proposal->root->next)))
    {
      proposal->oback = showtop (crawlback (proposal->origin));
      proposal->osister = showsister (proposal->origin);
      if (proposal->oback != showtop (crawlback (proposal->root->next)))
	{
	  proposal->ocousin = showsister (proposal->oback);
	}
      else
	{
	  proposal->ocousin = NULL;
	}
    }
  if (proposal->origin == NULL)
    error ("Designation of origin for branch removal failed");
  free (goal);
}

void 
construct_localtimelist (timelist_fmt * timevector, proposal_fmt * proposal)
{
  long z = 0;
/*     size_t num = 0; */
  long oz = proposal->listsize;
  proposal->mig_removed = FALSE;
  traverseAllNodes (crawlback (proposal->origin)->back, &proposal->aboveorigin, &z, &oz, WITH_MIGR_NODES);
  proposal->aboveorigin[z++] = proposal->oback;
  /*    num = (size_t) z; */
  z = remove_doublettes (timevector, proposal->aboveorigin);
  qsort ((void *) (*timevector).tl, (*timevector).T, sizeof (vtlist), agecmp);
  (*timevector).T -= z;
  if ((*timevector).tl[(*timevector).T - 1].eventnode->type != 'r')
    {
      fprintf (stderr, "autsch, root not at the end of timelist\n");
    }
  timeslices (&timevector);
  add_locallineages (timevector, proposal);
}

/*---------------------------------------------------------------------------- 
finds all nodes in a tree starting at the root node and crawling up 
to the tips in a recursive fashion, writing nodeptrs in the nodelist vector
the flag include_migration is 1 if we want to touch the migration nodes too,
otherwise =0 -> jump over the migration nodes. for convenience we define the 
the macros NO_MIGR_NODES=0 and WITH_MIGR_NODES=1 in the treesetup.h file
PB 1995
 */
void 
traverseAllNodes (node * theNode, node *** nodelist, long *node_elem, long *oldnode_elem, int include_migration)
{
  long elem;
  if (include_migration == NO_MIGR_NODES)
    {
      if (theNode->type != 't')
	{
	  if (crawlback (theNode->next) != NULL)
	    traverseAllNodes (crawlback (theNode->next),
			  nodelist, node_elem, oldnode_elem, NO_MIGR_NODES);
	  if (theNode->type != 'm' && crawlback (theNode->next->next) != NULL)
	    traverseAllNodes (crawlback (theNode->next->next),
			  nodelist, node_elem, oldnode_elem, NO_MIGR_NODES);
	  if ((*node_elem) == (*oldnode_elem - 1))
	    {
	      elem = *oldnode_elem = ((*oldnode_elem) + (*oldnode_elem) / 2);
	      (*nodelist) = (node **) realloc ((*nodelist), sizeof (node *) * elem);
	      memset ((*nodelist) + (*oldnode_elem), 0, sizeof (node *) * (elem - (*oldnode_elem)));
	      *oldnode_elem = elem;
	    }
	  (*nodelist)[(*node_elem)++] = theNode;
	  if (theNode->type == 'm')
	    {
	      error ("Migration node encountered?! and died!");
	    }
	}
      else
	{
	  if ((*node_elem) == (*oldnode_elem - 1))
	    {
	      elem = *oldnode_elem = ((*oldnode_elem) + (*oldnode_elem) / 2);
	      (*nodelist) = (node **) realloc ((*nodelist), sizeof (node *) * elem);
	      memset ((*nodelist) + (*oldnode_elem), 0, sizeof (node *) * (elem - (*oldnode_elem)));
	      *oldnode_elem = elem;
	    }
	  (*nodelist)[(*node_elem)] = theNode;
	  (*node_elem) += 1;
	}
    }
  else
    {
      if (theNode->type != 't')
	{
	  if (theNode->next->back != NULL)
	    traverseAllNodes (theNode->next->back,
			nodelist, node_elem, oldnode_elem, WITH_MIGR_NODES);
	  if (theNode->type != 'm' && theNode->next->next->back != NULL)
	    traverseAllNodes (theNode->next->next->back,
			nodelist, node_elem, oldnode_elem, WITH_MIGR_NODES);
	  if ((*node_elem) == (*oldnode_elem - 1))
	    {
	      elem = *oldnode_elem = ((*oldnode_elem) + (*oldnode_elem) / 2);
	      (*nodelist) = (node **) realloc ((*nodelist), sizeof (node *) * elem);
	      memset ((*nodelist) + (*oldnode_elem), 0, sizeof (node *) * (elem - (*oldnode_elem)));
	      *oldnode_elem = elem;
	    }
	  (*nodelist)[(*node_elem)++] = theNode;
	}
      else
	{
	  if ((*node_elem) == (*oldnode_elem - 1))
	    {
	      elem = *oldnode_elem = ((*oldnode_elem) + (*oldnode_elem) / 2);
	      (*nodelist) = (node **) realloc ((*nodelist), sizeof (node *) * elem);
	      memset ((*nodelist) + (*oldnode_elem), 0, sizeof (node *) * (elem - (*oldnode_elem)));
	      *oldnode_elem = elem;
	    }
	  (*nodelist)[(*node_elem)++] = theNode;
	}
    }
}

/* prepares to remove elements of a timelist by setting eventnode to NULL
   and age to Infinity, so a simple sort will push them over the edge */
long 
remove_doublettes (timelist_fmt * timevector, node ** ptr)
{
  long i = 0, j = 0, slot = 0;
  /* assumes that there is an NULL element at the end */
  for (i = 0; i < (*timevector).T; j = 0, i++)
    {
      while (((*timevector).tl[i].eventnode != ptr[j]) && (ptr[j] != NULL))
	j++;
      if (ptr[j] != NULL)
	{
	  slot++;
	  (*timevector).tl[i].age = DBL_MAX;
	}
    }
  return slot;
}

void 
add_locallineages (timelist_fmt * timevector, proposal_fmt * proposal)
{
  long pop = -99, numpop = proposal->world->numpop;
  if (timevector->T <= 0)
    error ("Help: timelist contains 0 elements");
  for (pop = 0; pop < numpop; pop++)
    timevector->tl[timevector->T - 1].lineages[pop] = 0;
  if (timevector->T == 1)
    timevector->tl[0].lineages[proposal->osister->actualpop] += 1;
  else
    timevector->tl[timevector->T - 1].lineages[timevector->tl[timevector->T - 2].from] += 1;
  add_partlineages (numpop, &timevector);

}


/* replaces nodepointers in list 1 with NULL if they are present in list 2
   returns the first NULL slot in the array.
 */
long 
xor (node ** ptrl1, node ** ptrl2)
{
  long i = 0, j = 0, slot = -1;
  /* assumes that there is an NULL element at the end */
  for (i = 0; ptrl1[i] != NULL; j = 0, i++)
    {
      while ((ptrl1[i] != ptrl2[j]) && (ptrl2[j] != NULL))
	j++;
      if (ptrl2[j] != NULL)
	{
	  if (slot == -1)
	    slot = i;
	  ptrl1[i] = NULL;
	}
    }
  return slot;
}


int 
migrate (proposal_fmt * proposal, node * up, long *old_migr_table_counter, boolean air)
{
  migr_table_fmt *array;
  long i;
  if (air)
    {
      array = proposal->migr_table;
      i = proposal->migr_table_counter;
    }
  else
    {
      array = proposal->migr_table2;
      i = proposal->migr_table_counter2;
    }
  if (i > MIGRATION_LIMIT)
    {
      fprintf (stdout, "migration limit reached\n");
      return 0;
    }
  switch (proposal->migration_model)
    {
    case STEPSTONE:
    case ISLAND:
    case MATRIX:
      array[i].from = RANDINT (0, proposal->numpop - 1);
      if (i > 0)
	{
	  while (array[i].from == array[i - 1].from)
	    array[i].from = RANDINT (0, proposal->numpop - 1);
	  array[i].to = array[i - 1].from;
	}
      else
	{
	  while (array[i].from == up->pop)
	    array[i].from = RANDINT (0, proposal->numpop - 1);
	  array[i].to = up->pop;
	}
      break;
    default:
      break;
    }
  array[i++].time = proposal->time;
  if (i > (*old_migr_table_counter) - 5)
    {
      (*old_migr_table_counter) += 10;
      if (air)
	{
	  proposal->migr_table = (migr_table_fmt *) realloc (proposal->migr_table,
		       sizeof (migr_table_fmt) * (*old_migr_table_counter));
	  array = proposal->migr_table;
	}
      else
	{
	  proposal->migr_table2 = (migr_table_fmt *) realloc (proposal->migr_table2,
		       sizeof (migr_table_fmt) * (*old_migr_table_counter));
	  array = proposal->migr_table;
	}
    }
  if (air)
    {
      proposal->migr_table_counter = i;
    }
  else
    {
      proposal->migr_table_counter2 = i;
    }
  return 1;
}

void 
chooseTarget (proposal_fmt * proposal, timelist_fmt * timevector,
	      node ** bordernodes, long *bordernum)
{
  long actualpop = -99;
  node *rb = crawlback (proposal->root->next);
  *bordernum = 0;
  proposal->target = NULL;
  proposal->realtarget = NULL;
  if (proposal->migr_table_counter == 0)
    actualpop = proposal->origin->pop;
  else
    actualpop = proposal->migr_table[proposal->migr_table_counter - 1].from;
  if (rb->tyme < proposal->time)
    {
      error ("why is rb->tyme smaller than proopsal-time in chooseTarget??\n");
    }
  findbordernodes (rb, proposal, actualpop, bordernodes, bordernum,
		   &(*timevector).tl, (*timevector).T);
  if (*bordernum > 0)
    {
      proposal->target = bordernodes[RANDINT (0, (*bordernum) - 1)];
      if (proposal->target != rb)
	{
	  proposal->tsister = showsister (proposal->target);
	  proposal->realtsister = crawlback (proposal->tsister)->back;
	}
      else
	proposal->tsister = NULL;
      proposal->realtarget = proposal->target;
      if (proposal->target->type == 'm')
	proposal->target = crawlback (showtop (proposal->target)->next);
    }
  else
    {
      proposal->target = NULL;
      proposal->tsister = NULL;
      proposal->realtsister = NULL;
      proposal->realtarget = NULL;
    }
}

void 
findbordernodes (node * theNode, proposal_fmt * proposal, long pop,
		 node ** bordernodes, long *bordernum,
		 vtlist ** tyme, long gte)
{
  node *tmp, *back;
  if (theNode == proposal->oback)
    {
      tmp = showtop (crawlback (proposal->osister)->back);
      back = showtop (proposal->oback->back);
    }
  else
    {
      tmp = showtop (theNode);
      back = showtop (theNode->back);
    }
  if (pop == tmp->pop && pop == back->actualpop &&
      tmp->tyme < proposal->time &&
      back->tyme > proposal->time)
    {
      bordernodes[(*bordernum)++] = tmp;
    }
  else
    {
      if (back->tyme < proposal->time)
	return;
      if (!tmp->tip)
	{
	  if (tmp->next->back != NULL)
	    findbordernodes (tmp->next->back, proposal, pop, bordernodes,
			     bordernum, tyme, gte);
	  if (tmp->type != 'm' && tmp->next->next->back != NULL)
	    findbordernodes (tmp->next->next->back, proposal, pop, bordernodes,
			     bordernum, tyme, gte);
	}
    }
}

/*
   boolean
   same_pop(node * up, double tyme, long pop)
   {
   node *oldnn = showtop(up->back);
   node *nn = up;
   while (nn->tyme < tyme) {
   oldnn = nn;
   nn = showtop(nn->back);
   }
   if (oldnn->pop == pop && nn->actualpop == pop)
   return TRUE;
   else
   return FALSE;
   }
 */


/* -----------------------------------------------------------------------
   simulates two lineages at once, if we are extending below the last node */
int 
pre_population (proposal_fmt * proposal, vtlist * ltime, long gte, long *slider)
{
  boolean coalesced = FALSE;
  boolean choice = FALSE;
  long pop1 = -99, pop2 = -98;
  double age1, denom, rr, r0, r1, horizon, mm, mm2;
  if (gte > 0)
    proposal->realtarget = ltime[gte - 1].eventnode;
  else
    proposal->realtarget = ltime[gte].eventnode->next->back;
  if (proposal->realtarget == proposal->oback)
    {
      proposal->realtarget = crawlback (proposal->osister)->back;
    }
  if (proposal->realtarget->type == 'm')
    {
      proposal->target = crawlback (proposal->realtarget->next);
      if (proposal->target == proposal->oback)
	{
	  proposal->target = proposal->osister;
	}
    }
  else
    {
      proposal->target = proposal->realtarget;
    }
  proposal->tsister = NULL;
  pop2 = proposal->realtarget->pop;
  pop1 = proposal->migr_table_counter > 0 ?
    proposal->migr_table[proposal->migr_table_counter - 1].from : proposal->origin->pop;
  age1 = MAX (proposal->realtarget->tyme, proposal->migr_table_counter > 0 ?
	      proposal->migr_table[proposal->migr_table_counter - 1].time :
	      proposal->origin->tyme);
  horizon = MAX (proposal->oback->tyme, age1);
  while (age1 < horizon)
    {
      mm = proposal->param0[proposal->numpop + pop1];
      if (pop1 == pop2)
	{
	  denom = mm + (2. / proposal->param0[pop1]);
	  proposal->time = age1 - log (RANDUM ()) / denom;
	  age1 = proposal->time;
	  if (age1 < horizon)
	    {
	      rr = RANDUM ();
	      r0 = (2. / proposal->param0[pop1]) / denom;
	      if (rr < r0)
		{
		  return 1;
		}
	    }
	}
      else
	{
	  denom = mm;
	  proposal->time = age1 - log (RANDUM ()) / denom;
	  age1 = proposal->time;
	}
      if (age1 < horizon)
	{
	  if (!migrate (proposal, proposal->origin,
			&proposal->old_migr_table_counter, MIGRATION_AIR))
	    {
	      return 0;
	    }
	  pop1 = proposal->migr_table_counter > 0 ?
	    proposal->migr_table[proposal->migr_table_counter - 1].from :
	    proposal->origin->pop;
	}
    }
  age1 = horizon;
  while (!coalesced)
    {
      mm = proposal->param0[pop1 + proposal->numpop];
      mm2 = proposal->param0[pop2 + proposal->numpop];
      denom = mm + mm2;
      if (pop1 == pop2)
	{
	  denom = denom + (2. / proposal->param0[pop1]);
	  proposal->time = age1 - log (RANDUM ()) / denom;
	  age1 = proposal->time;
	  rr = RANDUM ();
	  r0 = ((2. / proposal->param0[pop1]) / denom);
	  r1 = r0 + mm / denom;
	  if (rr < r0)
	    {
	      return 1;
	    }
	  else
	    {
	      if (rr < r1)
		{
		  choice = TRUE;
		}
	      else
		{
		  choice = FALSE;
		}
	    }
	}
      else
	{			/*pop1 not equal pop2 */
	  proposal->time = age1 - log (RANDUM ()) / denom;
	  age1 = proposal->time;
	  if (RANDUM () < (mm / denom))
	    {
	      choice = TRUE;
	    }
	  else
	    {
	      choice = FALSE;
	    }
	}
      if (choice)
	{
	  if (!migrate (proposal, proposal->origin,
			&proposal->old_migr_table_counter, MIGRATION_AIR))
	    {
	      return 0;		/* migration limit reached */
	    }
	  pop1 = proposal->migr_table_counter > 0 ?
	    proposal->migr_table[proposal->migr_table_counter - 1].from :
	    proposal->origin->pop;
	}
      else
	{
	  if (!migrate (proposal, proposal->realtarget,
		     &proposal->old_migr_table_counter2, MIGRATION_IN_TREE))
	    {
	      return 0;		/* migration limit reached */
	    }
	  pop2 = proposal->migr_table_counter2 > 0 ?
	    proposal->migr_table2[proposal->migr_table_counter2 - 1].from :
	    proposal->realtarget->pop;
	}
    }
  error ("Reached the end of function without coalescing");
  return -1;			/*makes the compiler happy */
}

void 
free_proposal (proposal_fmt * proposal)
{
  long j;
  free (proposal->aboveorigin);
  free (proposal->bordernodes);
  free (proposal->line_f);
  free (proposal->line_t);
  if (proposal->datatype == 's')
    {
      for (j = 0; j < proposal->endsite; j++)
	{
	  free (proposal->xt.s[j]);
	  free (proposal->xf.s[j]);
	}
      free (proposal->xf.s);
      free (proposal->xt.s);
    }
  else
    {
      free (proposal->xf.a);
      free (proposal->xt.a);
    }
  free (proposal->migr_table);
  free (proposal->migr_table2);
  free (proposal);
}

void 
free_timevector (timelist_fmt * timevector)
{
  free (timevector[0].tl);
  free (timevector);
}

/*----------------------------------------------------------*
 * rejection/acceptance of the new tree according to the likelihood
 * and an acceptance ratio which is higher the better the
 * likelihood values are (-> Metropolis)
 */
boolean
acceptlike (world_fmt * world, proposal_fmt * proposal, long g, timelist_fmt * tyme)
{
  static boolean report = TRUE, oldg = -1;
  double rr, expo;
  long rm;
  long rmc = rmigrcount (proposal);
  if ((rm = proposal->migr_table_counter + proposal->migr_table_counter2 +
       world->migration_counts - rmc) > MIGRATION_LIMIT)
    {
      if (report || g != oldg)
	{
	  fprintf (stderr, "migration limit (%i) exceeded: %li\n", MIGRATION_LIMIT, rm);
	  report = FALSE;
	  oldg = g;
	}
      return FALSE;
    }
  proposal->likelihood = pseudotreelikelihood (world, proposal);
  if (world->likelihood[g] < proposal->likelihood)
    {
      return TRUE;
    }
  else
    {
      expo = exp (proposal->likelihood - world->likelihood[g]);
      rr = RANDUM ();
      if (rr < expo)
	{
	  return TRUE;
	}
    }
  return FALSE;
}

long 
rmigrcount (proposal_fmt * proposal)
{
  node *p;
  long count = 0;
  for (p = proposal->origin; p != proposal->oback; p = showtop (p->back))
    {
      if (p->type == 'm')
	count++;
    }
  return count;
}

double 
eventtime (proposal_fmt * proposal, long pop, vtlist * tentry, char *event, long g)
{
  static boolean mig_force = TRUE;
  double interval, lines, denom, mm = 0.0;
  lines = 2.0 * tentry->lineages[pop];
  /*    for(i=0;i<proposal->numpop;i++){
     if (i!=pop)
     mm +=  *//*tentry->lineages[i] *//*  proposal->param0[i+proposal->numpop];
     } */
  mm = proposal->param0[pop + proposal->numpop];
  denom = mm + (lines / proposal->param0[pop]);
  interval = -(log (RANDUM ())) / denom;
  if (lines > 0)
    {
      if (RANDUM () < mm / denom)
	{
	  /*      if(proposal->world->atl[0].tl[g].p[pop]==0){
	     if(RANDUM()> 0.90){
	     *event = 'c';
	     return interval;
	     }
	     } */
	  *event = 'm';
	  return interval;
	}
      else
	{
	  /* if migration parameter x is 0 
	     then we will insert a migration in population x
	     with probability so that 4Nm=0.1*theta
	   */
	  if ((proposal->param0[proposal->numpop + pop] <= SMALLEST_MIGRATION))
	    {
	      if (RANDUM () < 0.1)
		{
		  if (mig_force)
		    {
		      fprintf (stderr, "Migration forced\n");
		      mig_force = FALSE;
		    }
		  *event = 'm';
		  return interval;
		}
	    }
	  *event = 'c';
	  return interval;
	}
    }
  else
    {
      *event = 'm';
      return interval;
    }
}

/*--------------------------------------------------------*
 * showsister() 
 * find the sisternode, by going down the branch and up on 
 * the other side again, neglecting the migration nodes.
 */
node *
showsister (node * theNode)
{
  node *tmp = crawlback (theNode);

  if (tmp->next->top)
    {
      return crawlback (tmp->next->next);
    }
  else
    {
      if (tmp->next->next->top)
	{
	  return crawlback (tmp->next);
	}
      else
	{
	  error ("error in treestructure, cannot find sisternode\n");
	}
    }
  return NULL;
}

void 
count_migrations (node * p, long *count)
{
  if (p->type != 't')
    {
      if (p->type == 'm')
	{
	  *count += 1;
	  count_migrations (p->next->back, count);
	}
      else
	{
	  count_migrations (p->next->back, count);
	  count_migrations (p->next->next->back, count);
	}
    }
}
/*
   long interior_pop0(timelist_fmt *timevector, proposal_fmt* proposal)
   {
   long sum=0,i;
   for(i=0;i<timevector->T;i++){
   if((timevector[0].tl[i].from==0) && (timevector[0].tl[i].from==timevector[0].tl[i].to))
   sum += (timevector[0].tl[i].from==0);
   }
   return sum;
   }
 */











/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 M C M C 2   R O U T I N E S 

 Tree changing routines
 

 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: mcmc2.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"
#include "tree.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

#define comment(a) fprintf(stderr,a)


/* prototypes ------------------------------------------- */
/* changes the the tree to the proposed tree by rebuilding parts
   of the tree and also inserts new migration events. */
void coalesce1p (proposal_fmt * proposal);
/* and its pretend version */
void pretendcoalesce1p (proposal_fmt * proposal);

/* private functions */
/* family of tree manipulation routines: dependent on the position of
   target and origin in the tree different routines are used.
   oback: there is no change in the tree topology.
   ocousin: target is the cousin-node, the branch is inserted in its
   former sister branch.
   rbcoa: target is the bottommost node, the ripped branch will be 
   bring the new bottommost node.
   stancoa: all other cases. */
void coat_oback (proposal_fmt * proposal);
void coat_ocousin (proposal_fmt * proposal);
void coat_obbcoa (proposal_fmt * proposal);
void coat_rbcoa (proposal_fmt * proposal);
void coat_stancoa (proposal_fmt * proposal);
/* pretend versions of the above */
void target_oback (proposal_fmt * proposal);
void target_obbcoa (proposal_fmt * proposal);
void target_ocousin (proposal_fmt * proposal);
void target_rbcoa (proposal_fmt * proposal);
void target_stancoa (proposal_fmt * proposal);
/* subroutines in target_stancoa */
void t_s_upper (proposal_fmt * proposal, node * connect, node * obb);
node *t_s_obbncon (proposal_fmt * proposal, node * obb);
void t_s_tncon (proposal_fmt * proposal, node * obb);
void t_s_tcon (proposal_fmt * proposal, node * upper);
long erase_migr_nodes (node * up);
node *findcrossing (node ** ptrl1, node ** ptrl2);
void connectnodes (node * mother, node * brother, node * sister);
void gotoroot (node * origin, node ** ptrlist);
void adjust (node * theNode, double time, long level);
void localevaluate (node * mother);
void copy_x (proposal_fmt * proposal, xarray_fmt xx1, xarray_fmt xx2);
void fix_root_pop (node * p);
void pseudoevaluate (proposal_fmt * proposal, xarray_fmt x, double *lx, node * mother, node * newdaughter, double v);
node *crawl_down (node * theNode, double time);


/*=========================================================*/

void 
coalesce1p (proposal_fmt * proposal)
{
  node *obb = NULL;
  erase_migr_nodes (proposal->origin);
  if (proposal->migr_table_counter2 > 0 || proposal->mig_removed)
    erase_migr_nodes (proposal->realtarget);
  if (proposal->target == proposal->ocousin)
    {
      coat_ocousin (proposal);
      if (proposal->oback->tyme > showtop (proposal->oback->back)->tyme && crawlback (proposal->oback)->type != 'r')
	error ("problem with time encountered");	/* was >= ** */
    }
  else
    {
      if (((proposal->target == proposal->oback) || (proposal->target == proposal->osister)
	   || (proposal->target == proposal->origin)))
	{
	  obb = proposal->oback->back;
	  coat_oback (proposal);
	  if (proposal->oback->tyme > showtop (proposal->oback->back)->tyme)
	    {			/* was >= ** */
	      if (proposal->oback->back->type == 'r')
		{
		  adjust (proposal->root, proposal->oback->tyme + 10000, 0);
		  comment ("Root time adjusted: was to small");
		}
	      else
		error ("problem with time encountered");
	    }
	}
      else
	{
	  if (crawlback (proposal->target)->type == 'r')
	    {
	      coat_rbcoa (proposal);
	      if (proposal->oback->tyme > showtop (proposal->oback->back)->tyme)
		{		/* was >= ** */
		  if (proposal->oback->back->type == 'r')
		    {
		      adjust (proposal->root, proposal->oback->tyme + 10000, 0);
		      comment ("Root time adjusted: was to small");
		    }
		  else
		    error ("problem with time encountered");
		}
	    }
	  else
	    {
	      obb = showtop (crawlback (proposal->oback));
	      if ((obb != NULL) && (obb == proposal->target))
		{
		  coat_obbcoa (proposal);
		  if (proposal->oback->tyme > showtop (proposal->oback->back)->tyme)	/* was >= ** */
		    error ("problem with time encountered");
		}
	      else
		{
		  coat_stancoa (proposal);
		  if (proposal->oback->tyme > showtop (proposal->oback->back)->tyme)	/* was >= ** */
		    error ("problem with time encountered");
		}
	    }
	}
    }
  set_pop (proposal->oback, proposal->origin->actualpop, proposal->origin->actualpop);
/*   free_migr_node(proposal->origin, crawlback(proposal->origin)); */
  if (proposal->migr_table_counter > 0)
    {
      set_pop (proposal->oback, proposal->migr_table[proposal->migr_table_counter - 1].from,
	       proposal->migr_table[proposal->migr_table_counter - 1].from);
      insert_migr_node (proposal->world, proposal->origin, crawlback (proposal->origin), proposal->migr_table, &(proposal->migr_table_counter));
    }
  proposal->migr_table_counter = 0;
  if (proposal->migr_table_counter2)
    {
      insert_migr_node (proposal->world, proposal->realtarget, crawlback (proposal->target),
		   proposal->migr_table2, &(proposal->migr_table_counter2));
      proposal->migr_table_counter2 = 0;
    }
  /*    if (proposal->oback == crawlback(proposal->root->next)) */
  fix_root_pop (crawlback (proposal->root->next));
}



void 
pretendcoalesce1p (proposal_fmt * proposal)
{
  if (proposal->target == proposal->ocousin)
    {
      target_ocousin (proposal);
    }
  else
    {
      if (((proposal->target == proposal->oback) || (proposal->target == proposal->osister) ||
	   (proposal->target == proposal->origin)))
	{
	  target_oback (proposal);
	}
      else
	{
	  if (crawlback (proposal->target)->type == 'r')
	    {
	      target_rbcoa (proposal);
	    }
	  else
	    {
	      if ((showtop (crawlback (proposal->oback)) != NULL) &&
		(showtop (crawlback (proposal->oback)) == proposal->target))
		{
		  target_obbcoa (proposal);
		}
	      else
		{
		  target_stancoa (proposal);
		}
	    }
	}
    }
}

/*================================================================*/

void 
coat_oback (proposal_fmt * proposal)
{
  node *obb, *obm;
  obb = showtop (crawlback (proposal->oback));
  connectnodes (showtop (proposal->oback->back), proposal->osister, NULL);
  adjust_time (proposal->oback, proposal->time);
  obm = showtop (crawl_down (proposal->osister, proposal->time)->back);
  connectnodes (proposal->oback, proposal->origin, proposal->osister);
  proposal->oback->back = NULL;
  if (obb->type == 'r')
    connectnodes (obb, proposal->oback, NULL);
  else
    {
      if (obm->type == 'm')
	connectnodes (obm, proposal->oback, NULL);
      connectnodes (obb, proposal->oback, proposal->ocousin);
    }
  adjust (obb, obb->tyme, 2);
  set_dirty (proposal->oback);
  localevaluate (proposal->oback);
}

void 
coat_ocousin (proposal_fmt * proposal)
{
  node *obb, *rcback;
  obb = showtop (crawlback (proposal->oback));
  connectnodes (showtop (proposal->oback->back), proposal->osister, NULL);
  rcback = showtop (crawl_down (proposal->ocousin, proposal->time)->back);
  adjust_time (proposal->oback, proposal->time);
  proposal->oback->back = NULL;
  connectnodes (rcback, proposal->oback, NULL);
  connectnodes (proposal->oback, proposal->ocousin, proposal->origin);
  connectnodes (obb, proposal->oback, proposal->osister);
  adjust (obb, obb->tyme, 2);
  set_dirty (proposal->oback);
  set_dirty (obb);
  if (crawlback (obb)->type != 'r')
    localevaluate (showtop (crawlback (obb)));
}

void 
coat_rbcoa (proposal_fmt * proposal)
{
  node *obb, *root;
  if (proposal->ocousin != NULL)
    {
      obb = showtop (crawlback (proposal->oback));
      root = proposal->root;
      connectnodes (showtop (proposal->oback->back), proposal->osister, NULL);
      connectnodes (obb, proposal->ocousin, proposal->osister);
      adjust (obb, obb->tyme, 2);
      adjust_time (proposal->oback, proposal->time);
      proposal->oback->back = NULL;
      connectnodes (proposal->oback, proposal->origin, proposal->target);
      connectnodes (root, proposal->oback, NULL);
      adjust (proposal->oback, proposal->time, 2);
      set_dirty (obb);
      set_dirty (proposal->oback);
      localevaluate (showtop (crawlback (obb)));
    }
  else
    {
      error ("error in coat_rbcoa\n");
    }
}

void 
coat_obbcoa (proposal_fmt * proposal)
{
  node /**obb,*/  * proposalack;
  if (proposal->ocousin != NULL)
    {
      /*   obb = showtop(crawlback(proposal->oback)); */
      connectnodes (showtop (proposal->oback->back), proposal->osister, NULL);
      proposalack = showtop (crawlback (proposal->target));
      adjust_time (proposal->oback, proposal->time);
      connectnodes (proposal->target, proposal->ocousin, proposal->osister);
      adjust (proposal->target, proposal->target->tyme, 1);
      proposal->oback->back = NULL;
      adjust_time (proposal->oback, proposal->time);
      if (showtop (proposal->realtarget->back)->type == 'm')
	{
	  connectnodes (showtop (proposal->realtarget->back), proposal->oback, NULL);
	}
      connectnodes (proposal->oback, proposal->origin, proposal->target);
      connectnodes (proposalack, proposal->tsister, proposal->oback);
      adjust (proposalack, proposalack->tyme, 2);
      set_dirty (proposal->target);
      set_dirty (proposal->oback);
      localevaluate (proposalack);
    }
  else
    {
      error ("error in coat_obbcoa\n");
    }
}

void 
coat_stancoa (proposal_fmt * proposal)
{
  node *obb, *proposalack;
  if (proposal->ocousin != NULL)
    {
      obb = showtop (crawlback (proposal->oback));
      proposalack = showtop (crawlback (proposal->target));
      connectnodes (showtop (proposal->oback->back), proposal->osister, NULL);
      proposal->oback->back = NULL;
      connectnodes (obb, proposal->osister, proposal->ocousin);
      adjust_time (proposal->oback, proposal->time);
      if (showtop (proposal->realtarget->back)->type == 'm')
	connectnodes (showtop (proposal->realtarget->back), proposal->oback, NULL);
      connectnodes (proposal->oback, proposal->origin, proposal->target);
      connectnodes (proposalack, proposal->oback, proposal->tsister);
      adjust (proposalack, proposalack->tyme, 3);
      adjust (obb, obb->tyme, 3);
      set_dirty (obb);
      set_dirty (proposal->oback);
      localevaluate (proposalack);
      localevaluate (obb);
    }
  else
    {
      adjust_time (proposal->oback, proposal->time);
      obb = showtop (crawlback (proposal->oback));
      connectnodes (showtop (proposal->oback->back), proposal->osister, NULL);
      proposalack = showtop (crawlback (proposal->target));
      adjust_time (proposal->oback, proposal->time);
      proposal->oback->back = NULL;
      if (showtop (proposal->realtarget->back)->type == 'm')
	connectnodes (showtop (proposal->realtarget->back), proposal->oback, NULL);
      connectnodes (proposal->oback, proposal->origin, proposal->target);
      connectnodes (proposalack, proposal->oback, proposal->tsister);
      /* either proposalack is osister or a descendent of her
         other case are disallowed by time constraints */
      connectnodes (obb, proposal->osister, NULL);
      adjust (proposalack, proposalack->tyme, 3);
      adjust (obb, obb->tyme, 3);
      set_dirty (proposal->oback);
      localevaluate (proposalack);

    }
}


void 
target_oback (proposal_fmt * proposal)
{
  node *obb = showtop (crawlback (proposal->oback));
  copy_x (proposal, proposal->xf, proposal->origin->x);
  proposal->mf = proposal->origin->lxmax;
  proposal->v = (proposal->time - proposal->origin->tyme);
  proposal->vs = (proposal->time - proposal->osister->tyme);
  pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v, proposal->osister->x, proposal->osister->lxmax, proposal->vs);
  pseudoevaluate (proposal, proposal->xf, &proposal->mf, obb, proposal->oback, fabs (proposal->time - obb->tyme));
}

void 
target_ocousin (proposal_fmt * proposal)
{
  node *obb;
  obb = showtop (crawlback (proposal->oback));
  copy_x (proposal, proposal->xf, proposal->origin->x);
  proposal->mf = proposal->origin->lxmax;
  copy_x (proposal, proposal->xt, proposal->ocousin->x);
  proposal->mt = proposal->ocousin->lxmax;
  proposal->v = (proposal->time - proposal->origin->tyme);
  proposal->vs = fabs (proposal->ocousin->tyme - proposal->time);
  pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v,
		proposal->xt, proposal->mt, proposal->vs);
  proposal->v = fabs (proposal->time - obb->tyme);
  proposal->vs = obb->tyme - proposal->osister->tyme;
  pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v, proposal->osister->x, proposal->osister->lxmax, proposal->vs);
  pseudoevaluate (proposal, proposal->xf, &proposal->mf, showtop (crawlback (obb)), obb,
		  (showtop (crawlback (obb))->tyme - obb->tyme));
}

void 
target_rbcoa (proposal_fmt * proposal)
{
  node *obb;
  if (proposal->ocousin != NULL)
    {
      obb = showtop (crawlback (proposal->oback));
      proposal->vs = obb->tyme - proposal->osister->tyme;
      proposal->v = obb->tyme - proposal->ocousin->tyme;
      copy_x (proposal, proposal->xt, proposal->osister->x);
      proposal->mt = proposal->osister->lxmax;
      pseudonuview (proposal, proposal->xt, &proposal->mt, proposal->vs, proposal->ocousin->x, proposal->ocousin->lxmax, proposal->v);
      pseudoevaluate (proposal, proposal->xt, &proposal->mt, showtop (crawlback (obb)), obb,
		      showtop (crawlback (obb))->tyme - obb->tyme);
      proposal->v = proposal->time - proposal->origin->tyme;
      proposal->vs = proposal->time - proposal->target->tyme;
      copy_x (proposal, proposal->xf, proposal->origin->x);
      proposal->mf = proposal->origin->lxmax;
      pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v, proposal->xt, proposal->mt, proposal->vs);
    }
  else
    {
      error ("error in target_rbcoa\n");
    }
}

void 
target_obbcoa (proposal_fmt * proposal)
{
  node *obb, *obbb;
  if (proposal->ocousin != NULL)
    {
      obb = showtop (crawlback (proposal->oback));
      proposal->vs = obb->tyme - proposal->osister->tyme;
      proposal->v = obb->tyme - proposal->ocousin->tyme;
      copy_x (proposal, proposal->xt, proposal->osister->x);
      proposal->mt = proposal->osister->lxmax;
      pseudonuview (proposal, proposal->xt, &proposal->mt, proposal->vs, proposal->ocousin->x, proposal->ocousin->lxmax, proposal->v);
      proposal->v = proposal->time - proposal->origin->tyme;
      proposal->vs = fabs (obb->tyme - proposal->time);
      copy_x (proposal, proposal->xf, proposal->origin->x);
      proposal->mf = proposal->origin->lxmax;
      pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v, proposal->xt, proposal->mt, proposal->vs);
      proposal->v = fabs (showtop (crawlback (obb))->tyme - proposal->time);
      obbb = showtop (crawlback (obb));
      pseudoevaluate (proposal, proposal->xf, &proposal->mf, obbb, obb, proposal->v);
    }
  else
    {
      error ("error in target_obbcoa\n");
    }
}

void 
target_stancoa (proposal_fmt * proposal)
{
  node *obb, *nn, *oldnn, *d1, *d2, *proposalack, *upper = NULL;	/* o R */
  node **double_line;
  double_line = (node **) calloc (1, sizeof (node *) * 2);
  if (proposal->ocousin != NULL)
    {
      obb = showtop (crawlback (proposal->oback));
      gotoroot (proposal->target, proposal->line_t);
      double_line[0] = proposal->osister;	/*findcrossing needs a last NULL element */
      if (findcrossing (double_line, proposal->line_t) != NULL)
	{
	  t_s_upper (proposal, proposal->osister, obb);
	}
      else
	{
	  double_line[0] = proposal->ocousin;	/*findcrossing needs a last NULL element */
	  if (findcrossing (double_line, proposal->line_t) != NULL)
	    {
	      t_s_upper (proposal, proposal->ocousin, obb);
	    }
	  else
	    {
	      gotoroot (obb, proposal->line_f);
	      proposal->connect = findcrossing (proposal->line_f, proposal->line_t);
	      proposal->vs = obb->tyme - proposal->osister->tyme;
	      proposal->v = obb->tyme - proposal->ocousin->tyme;
	      copy_x (proposal, proposal->xt, proposal->osister->x);
	      proposal->mt = proposal->osister->lxmax;
	      /*       if (obb!=proposal->connect) { */
	      upper = t_s_obbncon (proposal, obb);
	      /*} */
	      proposal->v = proposal->time - proposal->origin->tyme;
	      copy_x (proposal, proposal->xf, proposal->origin->x);
	      proposal->mf = proposal->origin->lxmax;
	      if (proposal->target != proposal->connect)
		{
		  t_s_tncon (proposal, obb);
		}
	      else
		{
		  t_s_tcon (proposal, upper);
		}
	    }
	}
    }
  else
    {
      proposal->v = proposal->time - proposal->origin->tyme;
      proposal->vs = proposal->time - proposal->target->tyme;
      copy_x (proposal, proposal->xf, proposal->origin->x);
      proposal->mf = proposal->origin->lxmax;
      copy_x (proposal, proposal->xt, proposal->target->x);
      proposal->mt = proposal->target->lxmax;
      pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v, proposal->xt, proposal->mt, proposal->vs);
      proposal->v = fabs (((proposalack = showtop (crawlback (proposal->target)))->tyme
			   - proposal->time));
      if (proposalack != proposal->oback)
	{
	  children (proposalack, &d1, &d2);
	  if (d1 == proposal->target)
	    d1 = d2;
	  pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v, d1->x, d1->lxmax, d1->v);
	  oldnn = proposalack;
	  nn = showtop (crawlback (oldnn));
	  while (nn != proposal->oback)
	    {
	      children (nn, &d1, &d2);
	      if (d1 == oldnn)
		d1 = d2;
	      pseudonuview (proposal, proposal->xf, &proposal->mf, oldnn->v, d1->x, d1->lxmax, d1->v);
	      oldnn = nn;
	      nn = showtop (crawlback (nn));
	    }
	}
    }
  free (double_line);
}

void 
t_s_upper (proposal_fmt * proposal, node * connect, node * obb)
{
  node *nn, *oldnn, *d1, *d2;
  proposal->v = proposal->time - proposal->origin->tyme;
  proposal->vs = proposal->time - proposal->target->tyme;
  copy_x (proposal, proposal->xf, proposal->origin->x);
  proposal->mf = proposal->origin->lxmax;
  pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v, proposal->target->x, proposal->target->lxmax, proposal->vs);
  nn = showtop (crawlback (proposal->target));
  oldnn = proposal->target;
  proposal->vs = fabs (nn->tyme - proposal->time);
  while (nn != connect)
    {
      children (nn, &d1, &d2);
      if (d1 == oldnn)
	d1 = d2;
      pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->vs,
		    d1->x, d1->lxmax, d1->v);
      proposal->vs = nn->v;
      oldnn = nn;
      nn = showtop (crawlback (nn));
    }
  children (nn, &d1, &d2);
  if (d1 == oldnn)
    d1 = d2;
  pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->vs,
		d1->x, d1->lxmax, d1->v);
  proposal->v = obb->tyme - proposal->osister->tyme;
  proposal->vs = obb->tyme - proposal->ocousin->tyme;
  if (connect == proposal->ocousin)
    pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->vs, proposal->osister->x, proposal->osister->lxmax, proposal->v);
  else
    pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v, proposal->ocousin->x, proposal->ocousin->lxmax, proposal->vs);
  pseudoevaluate (proposal, proposal->xf, &proposal->mf, showtop (crawlback (obb)), obb, obb->v);
}

node *
t_s_obbncon (proposal_fmt * proposal, node * obb)
{

  node *nn, *oldnn, *d1, *d2;
/*     fprintf(stdout,"t_s_obbncon\n"); */
  pseudonuview (proposal, proposal->xt, &proposal->mt, proposal->vs, proposal->ocousin->x, proposal->ocousin->lxmax, proposal->v);
  nn = showtop (crawlback (obb));
  oldnn = obb;
  proposal->vs = fabs (nn->tyme - obb->tyme);
  while (nn != proposal->connect)
    {
      children (nn, &d1, &d2);
      if (d1 == oldnn)
	d1 = d2;
      pseudonuview (proposal, proposal->xt, &proposal->mt, proposal->vs,
		    d1->x, d1->lxmax, d1->v);
      proposal->vs = nn->v;
      oldnn = nn;
      nn = showtop (crawlback (nn));
    }
  return oldnn;
}

void 
t_s_tncon (proposal_fmt * proposal, node * obb)
{
  node *nn, *oldnn, *d1, *d2;
  pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v,
		proposal->target->x, proposal->target->lxmax,
		proposal->time - proposal->target->tyme);
  nn = showtop (crawlback (proposal->target));
  oldnn = proposal->target;
  proposal->v = fabs (nn->tyme - proposal->time);
  while (nn != proposal->connect)
    {
      children (nn, &d1, &d2);
      if (d1 == oldnn)
	d1 = d2;
      pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v, d1->x, d1->lxmax, d1->v);
      proposal->v = nn->v;
      oldnn = nn;
      nn = showtop (crawlback (nn));
    }
  if (obb != proposal->connect)
    pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v, proposal->xt, proposal->mt, proposal->vs);
  proposal->v = proposal->connect->v;
  pseudoevaluate (proposal, proposal->xf, &proposal->mf, showtop (crawlback (proposal->connect)), proposal->connect, proposal->v);
}

void 
t_s_tcon (proposal_fmt * proposal, node * upper)
{
  node *obb, *d1, *d2;
  children (proposal->target, &d1, &d2);
  if (d1 == upper)
    d1 = d2;
  proposal->vs = fabs (proposal->target->tyme - showtop (upper)->tyme);
  pseudonuview (proposal, proposal->xt, &proposal->mt, proposal->vs, d1->x, d1->lxmax, d1->v);	/* eval target */
  proposal->vs = proposal->time - proposal->target->tyme;
  pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v, proposal->xt, proposal->mt, proposal->vs);	/*eval oback */
  obb = showtop (crawlback (proposal->target));
  proposal->v = fabs (obb->tyme - proposal->time);
  if (obb->type == 'r')
    return;
  else
    {
      children (obb, &d1, &d2);
      if (d1 == proposal->target)
	d1 = d2;
      pseudonuview (proposal, proposal->xf, &proposal->mf, proposal->v, d1->x, d1->lxmax, d1->v);
      pseudoevaluate (proposal, proposal->xf, &proposal->mf, showtop (crawlback (obb)), obb, obb->v);
    }
}


/*-----------------------------------------------------------------
erase all migration nodes between proposal->origin and proposal->oback 
*/
long 
erase_migr_nodes (node * up)
{
  long deleted = 0;
  node *theNode, *down, *oldNode;
  down = crawlback (up);
  theNode = up->back;

  while (theNode->type == 'm')
    {

      oldNode = theNode;
      theNode = theNode->next->back;
      if (oldNode->type == 'm')
	{
	  free (oldNode->next);
	  free (oldNode);
	  deleted++;
	}
    }


  down->back = up;
  up->back = down;
  /*  adjust_migr_above_nodes(up, down->actualpop, down); */
  return deleted;
}

/*-------------------------------------------------------
Connects and adjusts three nodes, the first is the 
mother node and the next two are child nodes, if one
of the child nodes is NULL it just connects mother with
the not-NULL child.
*/
void 
connectnodes (node * mother, node * brother, node * sister)
{
  node *tmp;

  if ((sister != NULL) && (brother != NULL))
    {
      if ((mother->id == brother->id) ||
	  (mother->id == sister->id) ||
	  (sister->id == brother->id))
	{
	  error ("connectnodes() conflict");
	}
      tmp = crawl_down (brother, mother->tyme);
      mother->next->back = tmp;
      tmp->back = mother->next;
      tmp = crawl_down (sister, mother->tyme);
      mother->next->next->back = tmp;
      tmp->back = mother->next->next;
    }
  else
    {
      if (sister == NULL)
	{
	  tmp = crawl_down (brother, mother->tyme);
	  mother->next->back = tmp;
	  tmp->back = mother->next;
	}
      else
	{

	  if (brother == NULL)
	    {
	      if (mother->type == 'm')
		{
		  tmp = crawl_down (sister, mother->tyme);
		  mother->next->back = tmp;
		  tmp->back = mother->next;
		}
	      else
		{
		  tmp = crawl_down (sister, mother->tyme);
		  mother->next->next->back = tmp;
		  tmp->back = mother->next->next;
		}
	    }
	  else
	    {
	      error ("Single, lonely rootnode detected in connectenodes()\n");
	    }
	}
    }
}

void 
gotoroot (node * origin, node ** ptrlist)
{
  node *theNode;
  long i = 0;

  for (theNode = origin;
       (theNode != NULL) &&
       (crawlback (theNode)->type != 'r');
       theNode = showtop (crawlback (theNode)))
    {
      ptrlist[i++] = theNode;
    }
  ptrlist[i] = theNode;		/* adds root->back to the list */
}

void 
adjust (node * theNode, double tyme, long level)
{
  if (level < 0 || theNode == NULL)
    return;

  theNode->tyme = tyme;
  if (theNode->type == 'r')
    {
      theNode->v = 1;
      theNode->length = DBL_MAX;
      if (theNode->next->back != NULL)
	adjust (crawlback (theNode->next), crawlback (theNode->next)->tyme, level);
      if (theNode->next->next->back != NULL)
	adjust (crawlback (theNode->next->next), crawlback (theNode->next->next)->tyme, level);
    }
  else if (theNode->type == 't')
    {
      theNode->tyme = 0.0;
      theNode->length = lengthof (theNode);
      ltov (theNode);
      return;
    }
  else if ((theNode->type != 't'))
    {
      if (theNode->type != 'm')
	{
	  theNode->length = lengthof (theNode);
	  ltov (theNode);
	  if (theNode->next->back != NULL)
	    {
	      theNode->next->tyme = tyme;
	      theNode->next->v = theNode->v;
	      theNode->next->length = theNode->length;
	      adjust (crawlback (theNode->next), crawlback (theNode->next)->tyme, level - 1);
	    }
	  if (theNode->next->next->back != NULL)
	    {
	      theNode->next->next->tyme = tyme;
	      theNode->next->next->v = theNode->v;
	      theNode->next->next->length = theNode->length;
	      adjust (crawlback (theNode->next->next), crawlback (theNode->next->next)->tyme, level - 1);
	    }
	}
      else
	adjust (crawlback (theNode->next), crawlback (theNode->next)->tyme, level);
    }
}


/* calculates x-array down the tree assuming that only one line 
   is affected by the change of the sub-likelihoods above that line
   BUT does NOT calculate the tree-likelihood as evaluate() does.
   THIS CHANGES THE TREE
 */
void 
localevaluate (node * mother)
{
  node *nn = NULL;

  if (mother->type != 'r')
    {
      set_dirty (mother);
      for (nn = mother; crawlback (nn)->type != 'r';
	   nn = showtop (crawlback (nn)))
	{
	  set_dirty (nn);
	}
    }
}

void 
copy_x (proposal_fmt * proposal, xarray_fmt xx1, xarray_fmt xx2)
{
  long i, j;
  switch (proposal->datatype)
    {
    case 'a':
    case 'b':
    case 'm':
      memcpy (xx1.a, xx2.a, sizeof (double)
	      * proposal->world->data->maxalleles[proposal->world->locus]);
      break;
    case 's':
      for (i = 0; i < proposal->endsite; i++)
	{
	  for (j = 0; j < proposal->world->options->rcategs; j++)
	    {
	      memcpy (xx1.s[i][j], xx2.s[i][j], sizeof (sitelike));
	    }
	}
      break;
    }
}

void 
fix_root_pop (node * p)
{
  if (crawlback (p) != p->back)
    {
      erase_migr_nodes (p);
    }
  if (p->back->actualpop != p->actualpop)
    {
      p->back->pop = p->back->next->pop = p->back->next->next->pop = p->pop;
      p->back->actualpop = p->back->next->actualpop = p->actualpop;
      p->back->next->next->actualpop = p->actualpop;
    }
}


/* transfers an x-array down the tree assuming that only one line 
   is affected by the change of the sub-likelihoods above that line
   BUT does NOT calculate the tree-likelihood as evaluate() does.
   DOES NOT CHANGE THE TREE
 */
void 
pseudoevaluate (proposal_fmt * proposal, xarray_fmt x, double *lx, node * mother, node * newdaughter, double v)
{
  node *nn = NULL, *d1 = NULL, *d2 = NULL, *oldnn = NULL;

  if (mother->type != 'r')
    {
      children (mother, &d1, &d2);
      if (d1 == newdaughter)
	d1 = d2;
      pseudonuview (proposal, x, lx, v, d1->x, d1->lxmax, d1->v);
      oldnn = mother;
      nn = showtop (crawlback (mother));
      while (nn->type != 'r')
	{
	  children (nn, &d1, &d2);
	  if (d1 == oldnn)
	    d1 = d2;
	  pseudonuview (proposal, x, lx, oldnn->v, d1->x, d1->lxmax, d1->v);
	  oldnn = nn;
	  nn = showtop (crawlback (nn));
	}
    }
}

node *
findcrossing (node ** ptrl1, node ** ptrl2)
{
  long i = 0, j = 0;

  /* assumes that there is an NULL element at the end */
  for (i = 0; ptrl1[i] != NULL; j = 0, i++)
    {
      while ((ptrl1[i] != ptrl2[j]) && (ptrl2[j] != NULL))
	j++;
      if (ptrl2[j] != NULL)
	{
	  break;
	}
    }
  return ptrl1[i];
}

node *
crawl_down (node * theNode, double tyme)
{
  node *otmp, *tmp = theNode->back;

  otmp = theNode;
  if (tmp == NULL)
    return otmp;
  while ((tmp->type == 'm') && showtop (tmp)->tyme < tyme)
    {
      otmp = tmp->next;
      tmp = tmp->next->back;
      if (tmp == NULL)
	return otmp;
    }
  return otmp;
}
/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 M E N U   R O U T I N E S 

 presents the menu and its submenus.                                                                                                               
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: menu.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"
#include "sequence.h"
#include "fst.h"
#include "options.h"
#include "menu.h"
#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif
/* prototypes ------------------------------------------- */
void print_menu_title (option_fmt * options);
void print_menu_accratio (long a, long b);
long print_title (world_fmt * world);
void get_menu (option_fmt * options);
/* private functions */
void menuData (option_fmt * options, char datatype[]);
void menuInput (option_fmt * options);
void menuParameters (option_fmt * options);
void menuStrategy (option_fmt * options);
void menuSequences (option_fmt * options);


void 
print_menu_title (option_fmt * options)
{
  char nowstr[LINESIZE];
  if (options->menu || options->progress)
    {
      fprintf (stdout, "  =============================================\n");
      fprintf (stdout, "  MIGRATION RATE AND POPULATION SIZE ESTIMATION\n");
      fprintf (stdout, "  using Markov Chain Monte Carlo simulation\n");
      fprintf (stdout, "  2-population  version\n");
      fprintf (stdout, "  =============================================\n");
      fprintf (stdout, "  Version %s\n", VERSION);
      get_time (nowstr, "  %c");
      if (nowstr[0] != '\0')
	fprintf (stdout, "  Program started at %s\n", nowstr);
      fprintf (stdout, "\n\n");
    }
}

void 
print_menu_accratio (long a, long b)
{
  fprintf (stdout, "           Acceptance-ratio = %li/%li (%f)\n", a, b, (double) a / (double) b);
}

long 
print_title (world_fmt * world)
{
  char nowstr[LINESIZE];
  long len, i, filepos = -1;
  if (!world->options->simulation)
    {
      if (world->options->title[0] == '\0')
	{
	  fprintf (world->outfile, "  ====================================================\n");
	  fprintf (world->outfile, "     MIGRATION RATE AND POPULATION SIZE ESTIMATION\n");
	  fprintf (world->outfile, "       using Markov Chain Monte Carlo simulation\n");
	  fprintf (world->outfile, "  ====================================================\n");
	  fprintf (world->outfile, "  Version %s\n", VERSION);
	  get_time (nowstr, "%c");
	  if (nowstr[0] != '\0')
	    {
	      fprintf (world->outfile, "  Program started at %s\n", nowstr);
	      filepos = ftell (world->outfile);
	      fprintf (world->outfile, "                     %*s \n", (int) strlen (nowstr), " ");
	    }
	  fprintf (world->outfile, "\n\n");
	}
      else
	{
	  len = MAX (strlen (world->options->title), 45);
	  fprintf (world->outfile, "  ");
	  for (i = 0; i < len; i++)
	    fprintf (world->outfile, "=");
	  fprintf (world->outfile, "\n  %s\n  ", world->options->title);
	  for (i = 0; i < len; i++)
	    fprintf (world->outfile, "=");
	  fprintf (world->outfile, "\n  MIGRATION RATE AND POPULATION SIZE ESTIMATION\n");
	  fprintf (world->outfile, "  using Markov Chain Monte Carlo simulation\n  ");
	  fprintf (stdout, "  2-population  version\n");
	  for (i = 0; i < len; i++)
	    fprintf (world->outfile, "=");
	  fprintf (world->outfile, "\n  Version %s\n", VERSION);
	  get_time (nowstr, "%c");
	  if (nowstr[0] != '\0')
	    {
	      fprintf (world->outfile, "\n  Program started at %s\n", nowstr);
	      filepos = ftell (world->outfile);
	      fprintf (world->outfile, "                                                   \n");
	    }
	  fprintf (world->outfile, "\n\n");
	}
    }
  return filepos;
}


void 
get_menu (option_fmt * options)
{
  char input[LINESIZE];
  char datatype[30];
  if (options->menu)
    {
      switch (options->datatype)
	{
	case 'a':
	  strcpy (datatype, "infinite allele model");
	  break;
	case 'm':
	  strcpy (datatype, "microsatellite model");
	  break;
	case 'b':
	  strcpy (datatype, "microsatellite model [Brownian motion]");
	  break;
	case 's':
	  strcpy (datatype, "sequence model");
	  break;
	default:
	  options->datatype = 'a';
	  strcpy (datatype, "infinite allele model");
	  break;
	}
      do
	{
	  printf ("  Settings for this run:\n");
	  printf ("  D       Data type (currently set: %s)\n", datatype);
	  printf ("  I       Input/Output formats\n");
	  printf ("  P       Start values for the Parameters\n");
	  printf ("  S       Search strategy\n");
	  printf ("\n\n");
	  printf ("  Are the settings correct?\n");
	  printf ("  (Type Y or the letter for one to change)\n");
	  fgets (input, LINESIZE, stdin);
	  switch (uppercase (input[0]))
	    {
	    case 'D':
	      menuData (options, datatype);
	      break;
	    case 'I':
	      menuInput (options);
	      break;
	    case 'P':
	      menuParameters (options);
	      break;
	    case 'S':
	      menuStrategy (options);
	      break;
	    default:
	      break;
	    }
	}
      while (uppercase (input[0]) != 'Y');
      if (options->usertree && options->datatype != 's')
	options->usertree = FALSE;
    }
}

/* private functions--------------------------------------------------- */
void 
menuData (option_fmt * options, char datatype[])
{
  static boolean didchangecat, didchangercat;
  static boolean ttr;
  long z = 0;
  char input[LINESIZE];
  char starttree[LINESIZE];

  /* in case we have  already read a parmfile */
  if (options->categs > 1)
    didchangecat = TRUE;
  else
    didchangecat = FALSE;
  if (options->rcategs > 1)
    didchangercat = TRUE;
  else
    didchangercat = FALSE;

  do
    {
      switch (options->datatype)
	{
	case 'a':
	  strcpy (datatype, "infinite allele model");
	  break;
	case 'm':
	  strcpy (datatype, "microsatellite model [Ladder model]");
	  break;
	case 'b':
	  strcpy (datatype, "microsatellite model [Brownian motion model]");
	  break;
	case 's':
	  strcpy (datatype, "sequence model");
	  break;
	default:
	  options->datatype = 'a';
	  strcpy (datatype, "infinite allele model");
	  break;
	}
      if (options->usertree && options->datatype == 's')
	sprintf (starttree, "is supplied in %s", options->utreefilename);
      else
	strcpy (starttree, "is estimated using a UPGMA topology");
      printf ("  DATATYPE AND DATA SPECIFIC OPTIONS\n\n");
      if (options->datatype == 's')
	printf ("  0       Start genealogy %s\n", starttree);
      printf ("  1       Datatype is currently a %s\n", datatype);
      switch (options->datatype)
	{
	case 'a':
	  printf ("          (using a k-allele model which simulates an infinite allele model)\n");
	  break;
	case 'm':
	  printf ("          (using a simple allele ladder model)\n");
	  printf ("  2       Size of the step matrix:      %li\n", options->micro_stepnum);
	  printf ("  3       Threshold value:              %li\n\n\n\n", options->micro_threshold);
	  break;
	case 's':
	  printf ("  2       Transition/transversion ratio:");
	  z = 0;
	  while (options->ttratio[z] > 0.0)
	    printf ("%8.4f ", options->ttratio[z++]);
	  printf ("\n");
	  printf ("  3       Use empirical base frequencies?  %s\n",
		  (options->freqsfrom ? "Yes" : "No"));
	  printf ("  4       One category of sites?");
	  if (options->categs == ONECATEG)
	    printf ("  One category\n");
	  else
	    printf ("  More than one  category of sites\n");
	  printf ("  5       One region of substitution rates?");
	  if (options->rcategs == 1)
	    printf ("  Yes\n");
	  else
	    {
	      printf ("  %ld categories of regions\n", options->rcategs);
	      printf ("  6       Rates at adjacent sites correlated?");
	      if (!options->autocorr)
		printf ("  No, they are independent\n");
	      else
		printf ("  Yes, mean block length =%6.1f\n", 1.0 / options->lambda);
	    }
	  printf ("  7       Sites weighted?  %s\n",
		  (options->weights ? "Yes" : "No"));
	  printf ("  8       Input sequences interleaved?  %s\n",
		  (options->interleaved ? "Yes" : "No, sequential"));
	  break;
	}
      printf ("\n\n");
      printf ("  Are the settings correct?\n");
      printf ("  (Type Y or the number of the entry to change)\n");
      fgets (input, LINESIZE, stdin);
      switch (uppercase (input[0]))
	{
	case '0':
	  if (options->datatype == 's')
	    options->usertree = !options->usertree;
	  break;
	case '1':
	  do
	    {
	      printf ("  (a)llele model\n");
	      printf ("  (m)icrosatellite model [Ladder model; exact but slow]\n");
	      printf ("  (b)rownian microsatellite model [Brownian motion model]\n");
	      printf ("  (s)equence model\n");
	      fgets (input, LINESIZE, stdin);
	    }
	  while (strchr ("ambs", (int) (lowercase (input[0]))) == NULL);
	  options->datatype = input[0];
	  if (options->datatype != 's')
	    options->usertree = FALSE;
	  break;
	default:
	  break;
	}
      if (options->datatype == 'm')
	{
	  switch (uppercase (input[0]))
	    {
	    case '2':		/*micro-stepmax */
	      printf ("  What is the maximal size of the stepmatrix?\n");
	      printf ("  [the bigger the longer the run and the more accurate the estimate]\n");
	      fscanf (stdin, "%ld%*[^\n]", &options->micro_stepnum);
	      break;
	    case '3':		/*micro-threshold */
	      printf ("  What is the threshold value?\n");
	      printf ("  E.g. if your allele is 24 and the threshold is 10\n");
	      printf ("  there is some probability that the allele 24 can\n");
	      printf ("  change to allele 14 (or 38), but there is a probability\n");
	      printf ("  of 0.0 (ZERO) to go to 13 (39),\n");
	      printf ("  if you choose this too small, than the program will fail\n");
	      printf ("  [the bigger the longer the run and the more\n accurate the estimate]\n");
	      fscanf (stdin, "%ld%*[^\n]", &options->micro_threshold);
	      break;
	    }
	}
      if (options->datatype == 's')
	{
	  switch (uppercase (input[0]))
	    {
	    case '2':
	      ttr = !ttr;
	      if (ttr)
		initratio (options);
	      break;
	    case '3':
	      options->freqsfrom = !options->freqsfrom;
	      if (!options->freqsfrom)
		{
		  initfreqs (&options->freqa, &options->freqc,
			     &options->freqg, &options->freqt);
		}
	      break;
	    case '4':
	      if (options->categs == ONECATEG)
		{
		  options->categs = MANYCATEGS;
		  didchangecat = TRUE;
		}
	      else
		{
		  options->categs = ONECATEG;
		  didchangecat = FALSE;
		}
	      /*printf("\n  Sitewise user-assigned categories:\n\n");
	         initcatn(&options->categs);
	         options->rate = (double *) realloc(options->rate,
	         options->categs * sizeof(double));
	         didchangecat = TRUE;
	         initcategs(options->categs, options->rate); */
	      break;
	    case '5':
	      if (options->rcategs == 1)
		{
		  options->autocorr = FALSE;
		  didchangercat = FALSE;
		}
	      else
		{
		  printf ("\n  Regional rates:\n");
		  initcatn (&options->rcategs);
		  options->probcat = (double *) realloc (options->probcat,
					options->rcategs * sizeof (double));
		  options->rrate = (double *) realloc (options->rrate,
					options->rcategs * sizeof (double));
		  didchangercat = TRUE;
		  initcategs (options->rcategs, options->rrate);
		  initprobcat (options->rcategs, &options->probsum, options->probcat);
		}
	      break;
	    case '6':
	      options->autocorr = !options->autocorr;
	      if (options->autocorr)
		initlambda (options);
	      break;
	    case '7':
	      options->weights = !options->weights;
	      break;
	    case '8':
	      options->interleaved = !options->interleaved;
	      break;
	    default:
	      break;
	    }
	  if (!didchangercat)
	    {
	      options->probcat = (double *) realloc (options->probcat, sizeof (double) * 2);
	      options->rrate = (double *) realloc (options->rrate, sizeof (double) * 2);
	      options->rrate[0] = 1.0;
	      options->probcat[0] = 1.0;

	    }
	  if (!didchangecat)
	    {
	      options->rate = (double *) realloc (options->rate, sizeof (double) * 2);
	      options->rate[0] = 1.0;
	    }

	}
    }
  while (uppercase (input[0]) != 'Y');
}

void 
menuInput (option_fmt * options)
{
  char input[LINESIZE];
  char *stringstep, *string2, *string3;
  char *treestr[] =
  {"None", "All", "Best", "Last chain"};

  stringstep = (char *) malloc (sizeof (char) * 128);
  string2 = (char *) malloc (sizeof (char) * 128);
  string3 = (char *) malloc (sizeof (char) * 128);

  do
    {
      printf ("  INPUT/OUTPUT FORMATS\n\n");
      printf ("  1       Echo the data at start of run?         %s\n",
	      options->printdata ? "Yes" : "No");
      printf ("  2       Print indications of progress of run?  %s\n",
	      options->progress ? "Yes" : "No");
      if (options->plot)
	{
	  switch (options->plotmethod)
	    {
	    case PLOTALL:
	      strcpy (string2, "Yes, to outfile and mathfile");
	      break;
	    default:
	      strcpy (string2, "Yes, to outfile");
	      break;
	    }
	}
      else
	{
	  strcpy (string2, "No");
	}
      printf ("  3       Plot likelihood surface?               %s\n",
	      string2);
      switch (options->treeprint)
	{
	case ALL:
	  printf ("  4       Print genealogies?                     %s\n", treestr[1]);
	  break;
	case BEST:
	  printf ("  4       Print genealogies?                     %s\n", treestr[2]);
	  break;
	case LASTCHAIN:
	  printf ("  4       Print genealogies?                     %s\n", treestr[3]);
	  break;
	case NONE:
	default:
	  printf ("  4       Print genealogies?                     %s\n", treestr[0]);
	  break;
	}
      switch (options->autoseed)
	{
	case AUTO:
	  sprintf (stringstep, "Yes");
	  break;
	case NOAUTO:
	  sprintf (stringstep, "No, use seedfile");
	  break;
	case NOAUTOSELF:
	  sprintf (stringstep, "No, seed=%li ", options->inseed);
	  break;
	default:
	  options->autoseed = AUTO;
	  sprintf (stringstep, "Yes");
	  break;
	}
      printf ("  5       Use automatic seed for randomisation?  %s\n",
	      stringstep);

      printf ("  6       Datafile name is %s\n",
	      options->infilename);

      printf ("  7       Outputfile name is %s\n",
	      options->outfilename);

      printf ("  8       Plot coordinates are saved in %s\n",
	      options->mathfilename);

      printf ("  T       Title of the analysis is\n");
      printf ("          %s\n", options->title);

      printf ("\n\n  Are the settings correct?\n");
      printf ("  (type Y to go back to the main menu or a number for the entry to change)\n");
      fgets (input, LINESIZE, stdin);
      switch (uppercase (input[0]))
	{
	case '1':
	  options->printdata = !options->printdata;
	  break;
	case '2':
	  options->progress = !options->progress;
	  break;
	case '3':
	  options->plot = !options->plot;
	  if (options->plot)
	    {
	      do
		{
		  printf ("  Plot Likelihood surface:\n");
		  printf ("  (B)oth to outfile and mathfile, (O)utfile only, (N)o plot\n");
		  fgets (input, LINESIZE, stdin);
		}
	      while (strchr ("BON", (int) uppercase (input[0])) == NULL);
	      switch (uppercase (input[0]))
		{
		case 'B':
		  options->plotmethod = PLOTALL;
		  break;
		case 'O':
		  options->plotmethod = PLOTOUTFILE;
		  break;
		case 'N':
		  options->plot = FALSE;
		  break;
		}
	    }
	  break;
	case '4':
	  do
	    {
	      printf ("  Print genealogies:\n");
	      printf ("  (N)one, (A)all [!], (B)est, (L)ast chain\n");
	      fgets (input, LINESIZE, stdin);
	    }
	  while (strchr ("NABL", (int) uppercase (input[0])) == NULL);
	  switch (uppercase (input[0]))
	    {
	    case 'N':
	      options->treeprint = NONE;
	      break;
	    case 'A':
	      options->treeprint = ALL;
	      break;
	    case 'B':
	      options->treeprint = BEST;
	      break;
	    case 'L':
	      options->treeprint = LASTCHAIN;
	      break;
	    default:
	      options->treeprint = NONE;
	      break;
	    }
	  break;
	case '5':
	  do
	    {
	      printf ("  (A)utomatic or (S)eedfile or (O)wn\n");
	      printf ("  Start value for Random-generator seed\n");
	      fgets (input, LINESIZE, stdin);
	      switch (uppercase (input[0]))
		{
		case 'A':
		  options->autoseed = AUTO;
		  options->inseed = time (0);
		  break;
		case 'S':
		  options->seedfile = fopen (SEEDFILE, "r");
		  if (options->seedfile)
		    {
		      options->autoseed = NOAUTO;
		      fscanf (options->seedfile, "%ld%*[^\n]", &options->inseed);
		      fclose (options->seedfile);
		    }
		  else
		    printf ("\n\n  There is no seedfile present\n");
		  break;
		case 'O':
		  options->autoseed = NOAUTOSELF;
		  printf ("  Random number seed (best values are x/4 +1)?\n");
		  scanf ("%ld%*[^\n]", &options->inseed);
		  break;
		}
	    }
	  while (options->autoseed < AUTO || options->autoseed > NOAUTOSELF);
	  break;
	case '6':
	  printf ("  What is the datafile name?\n");
	  fgets (input, LINESIZE, stdin);
	  input[strlen(input)-1] = '\0';
	  if (input[0] == '\0')
	    strcpy (options->infilename, "infile");
	  else
	    strcpy (options->infilename, input);
	  break;
	case '7':
	  printf ("  What is the output filename?\n");
	  fgets (input, LINESIZE, stdin);
	  input[strlen(input)-1] = '\0';	  
	  if (input[0] == '\0')
	    strcpy (options->outfilename, "outfile");
	  else
	    strcpy (options->outfilename, input);
	  break;
	case '8':
	  printf ("  What is the plot coordinate filename?\n");
	  fgets (input, LINESIZE, stdin);
	  input[strlen(input)-1] = '\0';
	  if (input[0] == '\0')
	    strcpy (options->mathfilename, "mathfile");
	  else
	    strcpy (options->mathfilename, input);
	  break;
	case 'T':
	  printf ("  Enter a title? [max 80 Characters]\n");
	  fgets (input, LINESIZE, stdin);
	  if (input[0] == '\0')
	    options->title[0] = '\0';
	  else
	    strncpy (options->title, input, 80);
	  break;
	default:
	  break;
	}
    }
  while (uppercase (input[0]) != 'Y');
  free (stringstep);
  free (string2);
  free (string3);
}


void 
menuParameters (option_fmt * options)
{
  char input[LINESIZE];

  do
    {
      printf ("  START VALUES FOR PARAMETERS\n\n");
      printf ("  1       Use a simple estimate of theta as start?");
      switch (options->thetaguess)
	{
	case FST:
	  printf (" Estimate with FST (Fw/Fb) measure\n");
	  break;
	default:
	  printf (" No, initial theta = {%6.4f, %6.4f}\n",
		  options->param0[0], options->param0[1]);
	}
      printf ("  2       Use a simple estimate of migration rate as start?");
      switch (options->migrguess)
	{
	case FST:
	  printf (" Estimate with FST (Fw/Fb) measure\n");
	  break;
	default:
	  printf (" No, initial 4Nm = {%6.4f, %6.4f}\n",
		  options->param0[2], options->param0[3]);
	}
      printf ("  3       Mutation rate is constant? %s\n",
	      !options->gamma ? "Yes" : "No, varying");
      printf ("\n\n\n  FST-CALCULATION\n\n");
      printf ("  4       %s\n", options->fsttype == 'M' ?
	      "Variable Theta, M is the same for all populations" :
	      "Variable M, Theta is the same for all populations");

      printf ("\n\n  Are the settings correct?\n");
      printf ("  (Type Y to go back to the main menu or a number for an entry to change)\n");
      fgets (input, LINESIZE, stdin);
      switch (input[0])
	{
	case '1':
	  printf ("  Which method? (F)st or (O)wn value\n");
	  fgets (input, LINESIZE, stdin);
	  if (uppercase (input[0]) == 'F')
	    options->thetaguess = FST;
	  else
	    {
	      options->thetaguess = OWN;
	      do
		{
		  printf ("  Initial Theta estimates?\nGive both start values separated by blanks\n");
		  fgets (input, LINESIZE, stdin);
		  sscanf (input, "%lf %lf", &options->param0[0], &options->param0[1]);
		}
	      while (options->param0[0] <= 0.0 || options->param0[1] <= 0.0);
	    }
	  break;
	case '2':
	  printf ("  (F)ST or (O)wn value\n");
	  fgets (input, LINESIZE, stdin);
	  if (uppercase (input[0]) == 'F')
	    options->migrguess = FST;
	  else
	    {
	      if (uppercase (input[0]) == 'O')
		options->migrguess = OWN;
	      do
		{
		  printf ("  Initial 4Nm estimates?\nGive both values separated by blanks\n");
		  fgets (input, LINESIZE, stdin);
		  sscanf (input, "%lf %lf", &options->param0[2], &options->param0[3]);
		}
	      while (options->param0[2] < 0.0 || options->param0[3] < 0.0);
	    }
	  break;
	case '3':
	  options->gamma = !options->gamma;
	  break;
	case '4':
	  if (options->fsttype == 'M')
	    options->fsttype = 'T';
	  else
	    options->fsttype = 'M';
	  fst_type (options->fsttype);
	  break;
	default:
	  break;
	}
    }
  while (uppercase (input[0]) != 'Y');
}


void 
menuStrategy (option_fmt * options)
{
  char input[LINESIZE];
  do
    {
      printf ("  SEARCH STRATEGY\n\n");
      printf ("  1       Number of short chains to run?           %3ld\n",
	      options->schains);
      if (options->schains > 0)
	{
	  printf ("  2       Short sampling increment?             %6ld\n",
		  options->sincrement);
	  printf ("  3       Number of steps along short chains?   %6ld\n",
		  options->ssteps);
	}
      printf ("  4       Number of long chains to run?            %3ld\n",
	      options->lchains);
      if (options->lchains > 0)
	{
	  printf ("  5       Long sampling increment?              %6ld\n",
		  options->lincrement);
	  printf ("  6       Number of steps along long chains?    %6ld\n",
		  options->lsteps);
	}
      printf ("\n  ---------------------------------------------------------\n");
      printf ("  Obscure options (consult the documentation on these)\n\n");
      if (options->movingsteps)
	printf ("  7       Sample a fraction of %2.2f new genealogies? Yes\n",
		options->acceptfreq);
      else
	printf ("  7       Sample at least a fraction of new genealogies? No\n");
      printf ("  8       Number of genealogies to discard at \n");
      printf ("          the beginning of each chain? [Burn-in]%6d\n",
	      options->burn_in);
      printf ("  9       Epsilon of parameter likelihood\n");
      printf ("          [please read the manual for this!] %9.5f\n",
	      options->lcepsilon);

      printf ("\n\n  Are the settings correct?\n");
      printf ("  (Type Y to go back to the main menu or a number for an entry to change)\n");
      fgets (input, LINESIZE, stdin);
      switch (input[0])
	{
	case '1':
	  do
	    {
	      printf ("  How many Short Chains?\n");
	      fgets (input, LINESIZE, stdin);
	      options->schains = atoi (input);
	      if (options->schains < 0)
		printf ("  Must be non-negative\n");
	    }
	  while (options->schains < 0);
	  break;
	case '2':
	  do
	    {
	      printf ("  How many trees to skip?\n");
	      fgets (input, LINESIZE, stdin);
	      options->sincrement = atoi (input);
	      if (options->sincrement <= 0)
		printf ("  Must be positive\n");
	    }
	  while (options->sincrement <= 0);
	  break;
	case '3':
	  do
	    {
	      printf ("  How many trees to sample?\n");
	      fgets (input, LINESIZE, stdin);
	      options->ssteps = atoi (input);
	      if (options->ssteps <= 0)
		printf ("  Must be a positive integer\n");
	    }
	  while (options->ssteps <= 0);
	  break;
	case '4':
	  do
	    {
	      printf ("  How many Long Chains?\n");
	      fgets (input, LINESIZE, stdin);
	      options->lchains = atoi (input);
	      if (options->lchains < 0)
		printf ("  Must be non-negative\n");
	    }
	  while (options->lchains < 0);
	  break;
	case '5':
	  do
	    {
	      printf ("  How many trees to skip?\n");
	      fgets (input, LINESIZE, stdin);
	      options->lincrement = atoi (input);
	      if (options->lincrement <= 0)
		printf ("  Must be positive\n");
	    }
	  while (options->lincrement <= 0);
	  break;
	case '6':
	  do
	    {
	      printf ("  How many trees to sample?\n");
	      fgets (input, LINESIZE, stdin);
	      options->lsteps = atoi (input);
	      if (options->lsteps <= 0)
		printf ("  Must be a positive integer\n");
	    }
	  while (options->lsteps <= 0);
	  break;

	case '7':
	  options->movingsteps = !options->movingsteps;
	  if (options->movingsteps)
	    {
	      do
		{
		  printf ("  How big should the fraction of new genealogies\n");
		  printf ("  of the originally proposed number of samples be?\n");
		  fgets (input, LINESIZE, stdin);
		  options->acceptfreq = atof (input);
		  if (options->acceptfreq < 0)
		    printf ("  Range should be between 0 - 1, and not %f\n", options->acceptfreq);
		}
	      while (options->acceptfreq < 0);
	    }
	  break;
	case '8':
	  do
	    {
	      printf ("  How many genealogies to discard?\n");
	      fgets (input, LINESIZE, stdin);
	      options->burn_in = atoi (input);
	      if (options->burn_in <= 0)
		printf ("  Must be a positive integer or zero (0)\n");
	    }
	  while (options->burn_in < 0);
	  break;
	case '9':
	  do
	    {
	      printf ("  Parameter likelihood epsilon?\n");
	      fgets (input, LINESIZE, stdin);
	      options->lcepsilon = atof (input);
	      if (options->lcepsilon <= 0)
		printf ("  Must be a positive value, be warned: too small values will run the program forever\n");
	    }
	  while (options->lcepsilon <= 0);
	  break;
	default:
	  break;
	}
    }
  while (uppercase (input[0]) != 'Y');
}
/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 O P T I O N S   R O U T I N E S 

 creates options structures,
 reads options from parmfile if present
 
 prints options,
 and finally helps to destroy itself.
                                                                                                               
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: options.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/
#include <stdio.h>
#include <time.h>
#include "migration.h"
#include "fst.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

/* parmfile parameter specifications and keywords */
#define LINESIZE 1024
#define NUMBOOL 11
#define BOOLTOKENS {"menu","interleaved","printdata","-reserved-",\
                          "moving-steps","freqs-from-data","usertree", \
                          "autocorrelation", "simulation","plot", "weights"}
#define NUMNUMBER 30
#define NUMBERTOKENS {"ttratio","short-chains",\
 "short-steps","short-inc","long-chains",\
 "long-steps", "long-inc", "theta", \
 "nmlength","seed","migration","mutation",\
 "datatype", "categories","rates","prob-rates", \
 "micro-max", "micro-threshold", "delimiter","burn-in",\
 "infile", "outfile", "mathfile", "title", \
 "long-chain-epsilon","print-tree","progress","l-ratio",\
 "fst-type","profile"};



/* prototypes ------------------------------------------- */
void create_options (option_fmt ** options);
void init_options (option_fmt * options);
void get_options (option_fmt * options);
void set_param (world_fmt * world, data_fmt * data, option_fmt * options, long locus);
void set_profile_options (option_fmt * options);
void print_menu_options (option_fmt * options, world_fmt * world);
void print_options (FILE * file, option_fmt * options, world_fmt * world);
void decide_plot (option_fmt * options, long chain, long chains, char type);
void destroy_options (option_fmt * options);

/* private functions */
boolean booleancheck (option_fmt * options, char *var, char *value);
boolean numbercheck (option_fmt * options, char *var, char *value);
void reset_oneline (option_fmt * options, long position);
void read_theta (option_fmt * options);
void read_mig (option_fmt * options);
char skip_space (option_fmt * options);
/*======================================================*/
void 
create_options (option_fmt ** options)
{
  (*options) = (option_fmt *) calloc (1, sizeof (option_fmt));
}

void 
init_options (option_fmt * options)
{
  long i;
  /* General options --------------------------------------- */
  options->nmlength = DEFAULT_NMLENGTH;
  options->popnmlength = DEFAULT_POPNMLENGTH;
  options->allelenmlength = DEFAULT_ALLELENMLENGTH;

  /* input/output options ---------------------------------- */
  options->menu = TRUE;
  options->progress = TRUE;
  options->verbose = FALSE;
  options->printdata = FALSE;
  options->usertree = FALSE;
  options->treeprint = NONE;
  options->printfst = TRUE;
  options->fsttype = THETAVARIABLE;
  fst_type (options->fsttype);
  options->plot = TRUE;
  options->plotmethod = PLOTALL;	/* outfile and mathematica file */
  options->simulation = FALSE;
  options->movingsteps = FALSE;
  options->acceptfreq = 0.1;
  strcpy (options->infilename, INFILE);
  strcpy (options->outfilename, OUTFILE);
  strcpy (options->mathfilename, MATHFILE);
  strcpy (options->treefilename, TREEFILE);
  strcpy (options->utreefilename, UTREEFILE);
  strcpy (options->catfilename, CATFILE);
  strcpy (options->weightfilename, WEIGHTFILE);
  strcpy (options->title, "\0");
  options->lratio = (lratio_fmt *) calloc (1, sizeof (lratio_fmt));
  options->lratio->data = (lr_data_fmt *) calloc (1, sizeof (lr_data_fmt) * 1);
  options->lratio->data[0].value = (char *) calloc (1, sizeof (char) * LINESIZE);
  options->lratio->alloccounter = 1;
  options->profile = ALL;
  options->qdprofile = FALSE;
  options->printprofsummary = TRUE;
  options->printprofile = TRUE;
  /* data options ------------------------------------------ */
  options->datatype = 's';
  options->migration_model = MATRIX;
  options->thetag = (double *) calloc (1, sizeof (double) * NUMPOP);
  options->mg = (double *) calloc (1, sizeof (double) * NUMPOP);
  options->param0 = (double *) calloc (1, sizeof (double) * NUMPOP * 2);
  options->gamma = FALSE;
  /* EP data */
  options->dlm = '\0';
  /* microsat data */
  options->micro_threshold = MICRO_THRESHOLD;
  options->micro_stepnum = MAX_MICROSTEPNUM;
  /*sequence data */
  options->interleaved = FALSE;
  options->ttratio = (double *) calloc (1, sizeof (double) * 2);
  options->ttratio[0] = 2.0;
  options->freqsfrom = TRUE;
  options->categs = ONECATEG;
  options->rate = (double *) calloc (1, sizeof (double));
  options->rate[0] = 1.0;
  options->rcategs = 1;
  options->rrate = (double *) calloc (1, sizeof (double));
  options->probcat = (double *) calloc (1, sizeof (double));
  options->autocorr = FALSE;
  options->rrate[0] = 1.0;
  options->probcat[0] = 1.0;

  options->probsum = 0.0;

  options->lambda = 1.0;
  options->weights = FALSE;
  /* random number options --------------------------------- */
  options->autoseed = AUTO;

  /* mcmc options ------------------------------------------ */
  options->thetaguess = FST;
  options->migrguess = FST;
  for (i = 0; i < NUMPOP; i++)
    {
      options->thetag[i] = 1.0;
      options->mg[i] = 1.0;
    }
  options->param0[0] = 1.0;
  options->param0[1] = 1.0;
  options->param0[2] = 1.0;
  options->param0[3] = 1.0;
  options->numthetag = options->nummg = 0;
  options->schains = 10;
  options->sincrement = 20;
  options->ssteps = 100;
  options->lchains = 3;
  options->lincrement = 20;
  options->lsteps = 1000;
  options->burn_in = TWOHUNDRED;
  options->lcepsilon = LONGCHAINEPSILON;
}

void 
get_options (option_fmt * options)
{
  long counter = 0;
  long position = 0;
  char varvalue[LINESIZE];
  char parmvar[LINESIZE];
  char input[LINESIZE];
  char *p;

  options->parmfile = fopen (PARMFILE, "r");
  if (options->parmfile)
    {
      counter = 0;
      position = ftell (options->parmfile);
      while (fgets (input, LINESIZE, options->parmfile) != NULL)
	{
	  counter++;
	  if ((input[0] == '#') || isspace ((int) input[0]))
	    continue;
	  else
	    {
	      if (!isalpha ((int) input[0]))
		{
		  fprintf (stderr, "the parmfile contains an error on the line %li\n",
			   counter);
		  exit (EXIT_FAILURE);
		}
	    }
	  if ((p = strchr (input, '#')) != NULL)
	    *p = '\n';
	  if (!strncmp (input, "end", 3))
	    break;
	  strcpy (parmvar, strtok (input, "="));
	  /* for version 2.0 (n-population)
	     if(!strncmp(parmvar,"theta",5)){
	     reset_oneline(options,position);
	     read_theta(options);
	     position=ftell(options->parmfile);
	     continue;
	     }
	     if(!strncmp(parmvar,"migration",5)){
	     reset_oneline(options,position);
	     read_mig(options);
	     position=ftell(options->parmfile);
	     continue;
	     } */
	  strcpy (varvalue, strtok (NULL, "\n"));
	  if (!booleancheck (options, parmvar, varvalue))
	    {
	      if (!numbercheck (options, parmvar, varvalue))
		fprintf (stderr, "Inappropiate entry in parmfile: %s ignored\n", input);
	    }
	  position = ftell (options->parmfile);
	}
    }
}

void 
print_menu_options (option_fmt * options, world_fmt * world)
{
  if (options->numpop > world->numpop)
    error ("Inconsistency between your Menu/Parmfile and your datafile\n \
Check the number of populations!\n");
  if (options->progress)
    {
      print_options (stdout, options, world);
    }
}

void 
print_options (FILE * file, option_fmt * options, world_fmt * world)
{
  /*for Version 2.0 long i,j, tt; */
  char mytext[LINESIZE];
  char seedgen[LINESIZE], spacer[LINESIZE];
  char paramtgen[LINESIZE], parammgen[LINESIZE];
  switch ((short) options->autoseed)
    {
    case AUTO:
      strcpy (seedgen, "with internal timer");
      strcpy (spacer, "");
      break;
    case NOAUTOSELF:
      strcpy (seedgen, "from parmfile");
      strcpy (spacer, "      ");
      break;
    case NOAUTO:
      strcpy (seedgen, "from seedfile");
      strcpy (spacer, "      ");
      break;
    default:
      strcpy (seedgen, "ERROR");
      strcpy (spacer, " ");
      break;
    }
  switch (options->thetaguess)
    {
    case OWN:
      strcpy (paramtgen, "from guessed values");
      break;
    case FST:
      strcpy (paramtgen, "from FST-calculation");
      break;
    default:
      strcpy (paramtgen, "ERROR");
      break;
    }
  switch (options->migrguess)
    {
    case OWN:
      strcpy (parammgen, "from guessed values");
      break;
    case FST:
      strcpy (parammgen, "from FST-calculation");
      break;
    default:
      strcpy (parammgen, "ERROR");
      break;
    }
  fprintf (file, "Options in use:\n");
  fprintf (file, "---------------\n");
  fprintf (file, "Datatype:%-44.44s\n",
	   options->datatype == 'a' ?
	   "Allelic data" :
	   (options->datatype == 's' ?
	    "Sequence data" :
	    (options->datatype == 'm' ? "Microsatellite data" :
	     "Microsatellite data [Brownian motion model]")));
  fprintf (file, "Random number seed (%s)%s%20li\n",
	   seedgen, spacer, options->saveseed);
  fprintf (file, "Start parameters:\n   Theta values were generated ");
  fprintf (file, " %s\n", paramtgen);
  if (options->thetaguess == OWN)
    fprintf (file, "   Theta = {%.5f,%.5f}\n",
	     options->param0[0], options->param0[1]);
  fprintf (file, "   M values were generated %s\n", parammgen);
  if (options->migrguess == OWN)
    fprintf (file, "   4Nm   = {%.5f,%.5f}\n",
	     options->param0[2], options->param0[3]);
  /* for version 2.0
     if (options->thetaguess == OWN){
     fprintf(file, "   Theta = ");
     for(i=0;i<options->numthetag;i++){
     fprintf(file,"%.5f,",options->thetag[i]);
     }
     fprintf(file,"\n");
     }
     fprintf(file, "   M values were generated %s\n", parammgen);
     if (options->migrguess == OWN){
     tt=0;
     fprintf(file, "   4Nm-matrix: ");
     if(options->nummg==1){
     fprintf(file,"%5.2f [all are the same]\n",options->mg[tt++]);
     }
     else {
     for(i=0;i<world->numpop;i++){
     for(j=0;j<world->numpop;j++){
     if(i!=j)
     fprintf(file,"%5.2f ",options->mg[tt++]); 
     else
     fprintf(file,"----- ");
     if(j>10)
     fprintf(file,"\n                ");
     }
     fprintf(file,"\n               ");
     }
     fprintf(file,"\n");
     }
     }
     end for version 2.0  */
  fprintf (file, "Gamma-distributed mutation rate %s\n",
	   options->gamma ? "is used" : "is not used");
  fprintf (file, "Markov chain parameters:\n");
  fprintf (file, "   Short chains (short-chains):         %20li\n", options->schains);
  fprintf (file, "      Trees sampled (short-inc*steps):  %20li\n",
	   options->sincrement * options->ssteps);
  fprintf (file, "      Trees recorded (short-steps):     %20li\n",
	   options->ssteps);
  fprintf (file, "   Long chains (long-chains):           %20li\n",
	   options->lchains);
  fprintf (file, "      Trees sampled (long-inc*steps):   %20li\n",
	   options->lincrement * options->lsteps);
  fprintf (file, "      Trees recorded (long-steps):      %20li\n",
	   options->lsteps);
  if (options->movingsteps)
    {
      fprintf (file, "   Percentage of new genealogies:       %20.2f\n",
	       (double) options->acceptfreq);
    }
  if (options->burn_in > 0)
    {
      fprintf (file, "   Number of discard trees per chain:   %20li\n",
	       (long) options->burn_in);
    }
  if (options->lcepsilon < LONGCHAINEPSILON)
    {
      fprintf (file, "   parameter-likelihood epsilon:        %20.5f\n",
	       options->lcepsilon);
    }
  fprintf (file, "Print options:\n");
  fprintf (file, "   Data file: %46.46s\n",
	   options->infilename);
  fprintf (file, "   Output file: %44.44s\n",
	   options->outfilename);
  fprintf (file, "   Print data: %45.45s\n",
	   options->printdata ? "Yes" : "No");
  switch (options->treeprint)
    {
    case NONE:
      fprintf (file, "   Print genealogies: %38.38s\n", "No");
      break;
    case ALL:
      fprintf (file, "   Print genealogies: %38.38s\n", "Yes, all");
      break;
    case LASTCHAIN:
      fprintf (file, "   Print genealogies: %38.38s\n", "Yes, only those in last chain");
      break;
    case BEST:
      fprintf (file, "   Print genealogies: %38.38s\n", "Yes, only the best");
      break;
    }
  if (options->plot)
    {
      switch (options->plotmethod)
	{
	case PLOTALL:
	  sprintf (mytext, "Yes, to outfile and %s", options->mathfilename);
	  break;
	default:
	  strcpy (mytext, "Yes, to outfile");
	  break;
	}
    }
  else
    {
      strcpy (mytext, "No");
    }
  fprintf (file, "   Plot data: %46.46s\n", mytext);
  /* Version 2.0   switch(options->profile){
     case NONE:  strcpy(mytext,"No"); break;
     case ALL:  strcpy(mytext,"Yes, tables and summary"); break;
     case TABLES:  strcpy(mytext,"Yes, tables"); break;
     case SUMMARY:  strcpy(mytext,"Yes, summary"); break;
     }     
     fprintf(file, "   Profile likelihood:%.36s\n\n\n\n", mytext);
   */
}

void 
set_param (world_fmt * world, data_fmt * data, option_fmt * options, long locus)
{
  long i /*Version 2.0 , ii  , j */ ;
  if (strchr ("mb", (int) options->datatype))
    {
      if (options->thetaguess != OWN)
	{
	  options->thetaguess = OWN;
	  for (i = 0; i < world->numpop; i++)
	    {
	      world->param0[i] = 1.0;
	    }
	}
      if (options->migrguess != OWN)
	{
	  options->migrguess = OWN;
	  for (i = world->numpop; i < world->numpop2; i++)
	    {
	      world->param0[i] = 1.0;
	    }
	}
    }
  switch (options->thetaguess)
    {
    case OWN:
      if (world->numpop < options->numpop)
	{
	  fprintf (stderr, "There is a conflict between your menu/parmfile\n");
	  fprintf (stderr, "and your datafile: number of populations\n");
	  fprintf (stderr, "are not the same\n");
	  exit (-1);
	}
      for (i = 0; i < world->numpop; i++)
	{
	  /* Version 2.0 if(i<options->numthetag-1)
	     ii=i;
	     else
	     ii=options->numthetag-1;            
	     if(options->thetag[ii]==0.0)
	   */
	  if (options->param0[i] == 0.0)
	    world->param0[i] = SMALLEST_THETA;
	  else
	    {
	      /*Version 2.0 world->param0[i] = options->thetag[ii]; */
	      world->param0[i] = options->param0[i];
	    }
	}
      break;

    case FST:
    default:
      for (i = 0; i < world->numpop; i++)
	{
	  if (world->fstparam[locus][i] > 0)
	    {
	      if (world->fstparam[locus][i] > 100)
		world->param0[i] = 1.0;
	      else
		world->param0[i] = world->fstparam[locus][i];
	    }
	  else
	    world->param0[i] = 1.0;
	}
      break;
    }
  switch (options->migrguess)
    {
    case OWN:
      for (i = world->numpop; i < 2 * world->numpop; i++)
	{
	  if (options->param0[i] == 0.0)
	    world->param0[i] = SMALLEST_MIGRATION;
	  else
	    world->param0[i] = options->param0[i] / world->param0[i - world->numpop];
	}
      break;
      /* Version 2.0
         for (i = 0; i < world->numpop; i++) {
         for (j = 0; j < world->numpop-1; j++) {
         if((world->numpop-1)*i+j < options->nummg)
         ii=(world->numpop-1)*i+j;
         else
         ii=options->nummg-1;        
         world->param0[world->numpop+(world->numpop-1)*i+j] = 
         options->mg[ii]/world->param0[i];
         }
         }
         break;
       */
    case FST:
    default:
      for (i = world->numpop; i < world->numpop2; i++)
	{
	  if (world->fstparam[locus][i] > 0)
	    {
	      if (world->fstparam[locus][i] > 100)
		world->param0[i] = 1.0;
	      else
		world->param0[i] = world->fstparam[locus][i];
	    }
	  else
	    world->param0[i] = 1.0;
	}
      break;
    }
}




void 
destroy_options (option_fmt * options)
{
  free (options->param0);
  free (options->thetag);
  free (options->mg);
  free (options->ttratio);
  free (options);
}

void 
decide_plot (option_fmt * options, long chain, long chains, char type)
{
  if (options->plot && (chain >= chains - 1) && (type == 'l'))
    options->plotnow = TRUE;
  else
    options->plotnow = FALSE;
}

/*private functions============================================= */

long 
boolcheck (char ch)
{
  char c = uppercase (ch);
  if ((c == 'F') || (c == 'N'))
    return 0;
  else if ((c == 'T') || (c == 'Y'))
    return 1;
  else
    return -1;
}				/* boolcheck */

boolean 
booleancheck (option_fmt * options, char *var, char *value)
{
  long i, check;
  char *booltokens[NUMBOOL] = BOOLTOKENS;
  char *tmp;
  check = boolcheck (value[0]);
  if (check == -1)
    return FALSE;
  i = 0;
  while (i < NUMBOOL && strcmp (var, booltokens[i]))
    i++;
  switch ((short) i)
    {
    case 0:
      options->menu = (boolean) (check);
      break;
    case 1:
      options->interleaved = (boolean) (check);
      break;
    case 2:
      options->printdata = (boolean) (check);
      break;
    case 3:
      /*reserved */
      break;
    case 4:
      options->movingsteps = (boolean) (check);
      if (options->movingsteps)
	{
	  strtok (value, ":");
	  options->acceptfreq = atof ((char *) strtok (NULL, " ,\n"));
	}
      break;
    case 5:
      options->freqsfrom = (boolean) (check);
      if (!options->freqsfrom)
	{
	  strtok (value, ":");
	  options->freqa = atof ((char *) strtok (NULL, " ,"));
	  options->freqc = atof ((char *) strtok (NULL, " ,"));
	  options->freqg = atof ((char *) strtok (NULL, " ,"));
	  options->freqt = atof ((char *) strtok (NULL, " ;\n"));
	}
      break;
    case 6:
      options->usertree = (boolean) (check);
      break;
    case 7:
      {				/* autocorrelation=<YES:value | NO> */
	options->autocorr = (boolean) (check);
	if (options->autocorr)
	  {
	    strtok (value, ":");
	    options->lambda = 1.0 / atof ((char *) strtok (NULL, " ;\n"));
	  }
	break;
      }
    case 8:
      options->simulation = (boolean) check;
      break;
    case 9:
      options->plot = (boolean) check;
      if (options->plot)
	{
	  strtok (value, ":");
	  if (toupper (value[0]) == 'Y')
	    options->plotmethod = PLOTALL;
	  else
	    {
	      tmp = strtok (NULL, ";");
	      switch (lowercase (tmp[0]))
		{
		case 'o':
		  options->plotmethod = PLOTOUTFILE;
		  break;
		case 'b':
		  options->plotmethod = PLOTALL;
		  break;
		default:
		  options->plotmethod = PLOTALL;
		  break;
		}
	    }
	}
      break;
    case 10:
      options->weights = (boolean) check;
      break;
    default:
      return FALSE;
    }
  return TRUE;
}				/* booleancheck */

boolean 
numbercheck (option_fmt * options, char *var, char *value)
{
  long i = 0, z;

  char *tmp, *temp;

  char *numbertokens[NUMNUMBER] = NUMBERTOKENS;
  tmp = (char *) malloc (sizeof (char) * LINESIZE);
  while (i < NUMNUMBER && strcmp (var, numbertokens[i]))
    i++;
  switch ((short) i)
    {
    case 0:
      z = 0;
      temp = strtok (value, " ,;\n");
      while (temp != NULL)
	{
	  options->ttratio[z++] = atof (temp);
	  options->ttratio = (double *) realloc (options->ttratio, sizeof (double) * (z + 1));
	  options->ttratio[z] = 0.0;
	  temp = strtok (NULL, " ,;\n");
	}
      break;
    case 1:
      options->schains = atoi (value);
      break;
    case 2:
      options->ssteps = atoi (value);
      break;
    case 3:
      options->sincrement = atoi (value);
      break;
    case 4:
      options->lchains = atoi (value);
      break;
    case 5:
      options->lsteps = atoi (value);
      break;
    case 6:
      options->lincrement = atoi (value);
      break;
    case 7:
      break;			/* theta: already handled in read_theta() */
    case 8:
      options->nmlength = atoi (value);
      break;
    case 9:			/* "seed" */
      switch (value[0])
	{
	case 'A':
	case 'a':
	case '0':
	  options->autoseed = AUTO;
	  options->inseed = (long) time (0);
	  break;
	case 'S':
	case 's':
	case '1':
	  options->autoseed = NOAUTO;
	  options->seedfile = fopen ("seedfile", "r");
	  if (options->seedfile == NULL)
	    {
	      fprintf (stderr, "Error: cannot find seedfile\n");
	      exit (EXIT_FAILURE);
	    }
	  fscanf (options->seedfile, "%ld%*[^\n]", &options->inseed);
	  fclose (options->seedfile);
	  break;
	case 'O':
	case 'o':
	case '2':
	  options->autoseed = NOAUTOSELF;
	  strtok (value, ":");
	  options->inseed = atoi ((char *) strtok (NULL, ";"));
	  break;
	default:
	  fprintf (stderr, "failure to read seed method, should be\nseed=auto or seed=seedfile or seed=own:value\n");
	  break;
	}
      break;
    case 10:
      break;			/*"migration" fake: this is already handled in read_migrate */
    case 11:			/*mutation */
      switch (value[0])
	{
	case 'A':		/*automatic */
	case 'a':
	case 'G':
	case 'g':
	  options->gamma = TRUE;
	  break;
	case 'N':		/*none, all loci have same mu */
	case 'n':
	  options->gamma = FALSE;
	  break;
	default:
	  break;
	}
      break;
    case 12:			/*datatype */
      switch (value[0])
	{
	case 'a':
	case 'A':
	  options->datatype = 'a';
	  break;
	case 'm':
	case 'M':
	  options->datatype = 'm';
	  break;
	case 'b':
	case 'B':
	  options->datatype = 'b';
	  break;
	case 's':
	case 'S':
	  options->datatype = 's';
	  break;
	default:
	  options->datatype = 'a';
	  break;
	}
      break;
    case 13:
      if ((toupper (value[0]) == 'Y') || (toupper (value[0] == 'T')))
	options->categs = MANYCATEGS;
      else
	options->categs = atoi (value);		/* categories */
      /* needs to read auxilliary file catfile */
      break;
    case 14:
      strncpy (tmp, value, strcspn (value, ":"));
      if (atoi (tmp) > 1)
	{			/* rate categories */
	  options->rcategs = atoi (tmp);
	  options->rrate = (double *) realloc (options->rrate, sizeof (double)
					       * (options->rcategs + 1));
	  temp = strtok (value, " :");
	  temp = strtok (NULL, " ,;\n");
	  z = 0;
	  while (temp != NULL)
	    {
	      if (z > options->rcategs)
		error ("check parmfile rates, missing rate\n");
	      options->rrate[z++] = atof (temp);
	      temp = strtok (NULL, " ,;\n");
	    }
	}
      break;
    case 15:
      strncpy (tmp, value, strcspn (value, ":"));
      if (atoi (tmp) > 1)
	{			/* probabilities for each rate category */
	  options->rcategs = atoi (tmp);
	  options->probcat = (double *) realloc (options->probcat, sizeof (double)
						 * (options->rcategs + 1));
	  temp = strtok (value, " :");
	  temp = strtok (NULL, " ,;\n");
	  z = 0;
	  while (temp != NULL)
	    {
	      if (z > options->rcategs)
		error ("check parmfile prob-rates, missing rate probability\n");
	      options->probcat[z++] = atof (temp);
	      temp = strtok (NULL, " ,;\n");
	    }
	}
      break;
    case 16:			/*micro-stepmax */
      options->micro_stepnum = atoi (value);
      break;
    case 17:			/*micro-threshold */
      options->micro_threshold = atoi (value);
      break;
    case 18:			/*delimiter */
      options->dlm = value[0];
      break;
    case 19:			/*burn-in */
      options->burn_in = atoi (value);
      break;
    case 20:			/*infilename */
      strcpy (options->infilename, value);
      break;
    case 21:			/*outfilename */
      strcpy (options->outfilename, value);
      break;
    case 22:			/*mathfilename */
      strcpy (options->mathfilename, value);
      break;
    case 23:			/*infilename */
      strncpy (options->title, value, 80);
      break;
    case 24:			/*long-chain-epsilon */
      options->lcepsilon = atof (value);
      break;
    case 25:			/* print tree options */
      switch (uppercase (value[0]))
	{
	case 'N':
	  options->treeprint = NONE;
	  break;
	case 'A':
	  options->treeprint = ALL;
	  break;
	case 'B':
	  options->treeprint = BEST;
	  break;
	case 'L':
	  options->treeprint = LASTCHAIN;
	  break;
	default:
	  options->treeprint = NONE;
	  break;
	}
      break;
    case 26:			/* progress: No, Yes, Verbose */
      switch (uppercase (value[0]))
	{
	case 'F':
	case 'N':
	  options->progress = FALSE;
	  options->verbose = FALSE;
	  break;
	case 'T':
	case 'Y':
	  options->progress = TRUE;
	  options->verbose = FALSE;
	  break;
	case 'V':
	  options->progress = TRUE;
	  options->verbose = TRUE;
	  break;
	}
      break;
    case 27:			/* lr-ratio: <none | mean | locus>:val1,val2,val3,val4,val5 */
      switch (uppercase (value[0]))
	{
	case 'M':
	  options->lratio->data[options->lratio->counter].type = MEAN;
	  break;
	case 'L':
	  options->lratio->data[options->lratio->counter].type = LOCUS;
	  break;
	case 'N':
	default:
	  free (tmp);
	  return FALSE;
	}
      temp = strtok (value, ":");
      temp = strtok (NULL, "\n");
      if (temp != NULL)
	strcpy (options->lratio->data[options->lratio->counter].value, temp);
      if (options->lratio->counter + 1 == options->lratio->alloccounter)
	{
	  options->lratio->alloccounter += 2;
	  options->lratio->data = (lr_data_fmt *) realloc (options->lratio->data, sizeof (lr_data_fmt) * options->lratio->alloccounter);
	  for (i = options->lratio->counter + 1; i < options->lratio->alloccounter; i++)
	    {
	      options->lratio->data[i].elem = 0;
	      options->lratio->data[i].value = (char *) calloc (1, sizeof (char) * LINESIZE);
	    }
	}
      options->lratio->counter++;
      break;
    case 28:			/* fst-type: <Theta | Migration> */
      switch (uppercase (value[0]))
	{
	case 'T':
	  options->fsttype = 'T';
	  break;
	case 'M':
	default:
	  options->fsttype = 'M';
	  break;
	}
      fst_type (options->fsttype);
      break;
    case 29:			/*profile=<NONE | ALL | TABLES | SUMMARY> */
      switch (uppercase (value[0]))
	{
	case 'S':
	  options->profile = SUMMARY;
	  break;
	case 'A':
	  options->profile = ALL;
	  break;
	case 'N':
	  options->profile = NONE;
	  break;
	case 'T':
	  options->profile = TABLES;
	  break;
	default:		/*A */
	  options->profile = ALL;
	  break;
	}
      set_profile_options (options);
      break;
    default:
      free (tmp);
      return FALSE;

    }
  free (tmp);
  return TRUE;
}				/* numbercheck */

void 
reset_oneline (option_fmt * options, long position)
{
  fseek (options->parmfile, position, SEEK_SET);
}


void 
read_theta (option_fmt * options)
{
  char parmvar[LINESIZE];
  char tmp[LINESIZE];
  char varvalue[LINESIZE];
  char ch;

  long i = 0;

  while ((ch = getc (options->parmfile)) != '=')
    {
      parmvar[i++] = ch;
    }
  i = 0;
  ch = getc (options->parmfile);
  while (!isspace ((int) ch) && ch != ':' && ch != '{')
    {
      varvalue[i++] = ch;
      ch = getc (options->parmfile);
    }
  switch (varvalue[0])
    {
    case 'F':
    case 'f':
    case '_':
      options->thetaguess = FST;
      break;
    case 'O':
    case 'o':
    case '0':
      options->thetaguess = OWN;
      ch = skip_space (options);
      if (ch == '\0')
	return;
      if (ch == '{')
	{

	  while (ch != '}')
	    {
	      i = 0;
	      ch = skip_space (options);
	      if (ch == '\0')
		return;
	      while (ch != ' ' && ch != ',' && ch != '}')
		{
		  tmp[i++] = ch;
		  ch = getc (options->parmfile);
		}
	      if (ch == '}')
		break;
	      tmp[i] = '\0';
	      options->thetag[options->numthetag] = atof (tmp);
	      options->numthetag += 1;
	      options->thetag = realloc (options->thetag,
				sizeof (double) * (1 + options->numthetag));

	    }
	}
      else
	{
	  i = 0;
	  tmp[i++] = ch;
	  while (!isspace (ch))
	    {
	      tmp[i++] = ch;
	      ch = getc (options->parmfile);
	    }
	  tmp[i] = '\0';
	  options->thetag[options->numthetag] = atof (tmp);
	  options->numthetag += 1;
	  options->thetag = realloc (options->thetag,
				sizeof (double) * (1 + options->numthetag));
	}
      options->numpop = options->numthetag;
      if (options->numthetag == 0)
	{
	  fprintf (stderr, "You forgot to add your guess value:\n");
	  fprintf (stderr, "Theta=Own:{pop1,pop2, ...}\n");
	  fprintf (stderr, "or Theta=Own:guess_pop (same value for all)");
	}
      break;
    default:
      fprintf (stderr, "Failure to read start theta method, should be\ntheta=FST or theta=Own:x.x\n or theta=Own:{x.x, x.x , x.x, .....}");
      exit (-1);
    }

}


void 
read_mig (option_fmt * options)
{
  char parmvar[LINESIZE];
  char tmp[LINESIZE];
  char varvalue[LINESIZE];
  char ch;
  long test = 0;
  long i = 0;
  /* 1st example:  1.0 (n-island model)
     2nd example: {1.0} (migration matrix model, all the same start values  
     3rd example: the dashes on the diagonal are NECESSARY, {} are facultativ
     -  1.0 0.1
     1.0  -  2.0
     0.9 1.2  -
     to specify real 0.0 enter a -1, because those have to be treated
     specifically. 0.0 in the table will be change to SMALLES_MIGRATION
   */

  while ((ch = getc (options->parmfile)) != '=')
    {
      parmvar[i++] = ch;
    }
  i = 0;
  ch = getc (options->parmfile);
  while (!isspace ((int) ch) && ch != ':' && ch != '{')
    {
      varvalue[i++] = ch;
      ch = getc (options->parmfile);
    }
  switch (varvalue[0])
    {
    case 'F':
    case 'f':
    case '_':
      options->migrguess = FST;
    case 'O':
    case 'o':
    case '0':
      options->migrguess = OWN;
      ch = skip_space (options);
      if (ch == '\0')
	return;
      if (ch == '{')
	{
	  options->migration_model = MATRIX;
	  while (ch != '}')
	    {
	      ch = skip_space (options);
	      if ((ch == '\0') || (ch == '}'))
		return;
	      i = 0;
	      while (ch != ' ' && ch != ',' && ch != '}')
		{
		  tmp[i++] = ch;
		  ch = getc (options->parmfile);
		}
	      tmp[i] = '\0';
	      if (strcmp (tmp, "-"))
		{
		  options->mg[options->nummg] = atof (tmp);
		  options->nummg += 1;
		  options->mg = realloc (options->mg,
				    sizeof (double) * (1 + options->nummg));
		}
	      else
		{
		  test++;
		}
	    }
	  options->numpop = test;
	}
      else
	{
	  options->migration_model = ISLAND;
	  i = 0;
	  options->numpop = 1;
	  tmp[i++] = ch;
	  while (!isspace (ch))
	    {
	      tmp[i++] = ch;
	      ch = getc (options->parmfile);
	    }
	  options->mg[options->nummg] = atof (tmp);
	  options->nummg += 1;
	}
      if (options->nummg == 0)
	{
	  fprintf (stderr, "You forgot to add your guess value, use either:\n");
	  fprintf (stderr, "migration=FST\n");
	  fprintf (stderr, "migration=Own:{migration matrix, diagonal is -}\n");
	  fprintf (stderr, "migration=Own:{migration value}, all matrix elements have the same value\n");
	  /*      fprintf(stderr,"or migration=Own:value (n-island model)"); */
	  exit (-1);
	}
      break;
    default:
      fprintf (stderr, "Failure to read start migration method\n");
      exit (-1);
    }

}

char 
skip_space (option_fmt * options)
{
  char ch = getc (options->parmfile);
  while (isspace ((int) ch) || ch == ',')
    {
      ch = getc (options->parmfile);
    }
  if (isalpha (ch))
    {
      ungetc (ch, options->parmfile);
      ch = '\0';
    }
  return ch;
}

void 
set_profile_options (option_fmt * options)
{
  switch (options->profile)
    {
    case NONE:
      options->printprofsummary = options->printprofile = FALSE;
      break;
    case ALL:
      options->printprofsummary = options->printprofile = TRUE;
      break;
    case TABLES:
      options->printprofsummary = FALSE;
      options->printprofile = TRUE;
      break;
    case SUMMARY:
      options->printprofsummary = TRUE;
      options->printprofile = FALSE;
      break;
    }
}
/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 P A R A M E T E R E S T I M A T I O N   R O U T I N E S 

 estimates parameter for each locus
 using a Newton-Rapshon maximization
 

 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: parameter.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"
#include "world.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

/* prototypes ------------------------------------------- */
void estimateParameter (timearchive_fmt * tyme, long G,
       world_fmt * world, double **dd, long chain, char type, char **plane);
double probG (double *param, tarchive_fmt * tl, long numpop);
double calc_like (nr_fmt * nr, tarchive_fmt * tyme, long G);
void derivatives_to_logderivatives (nr_fmt * nr);
void calc_cov (double **dd, double *d, double *param, long n);
/* private functions */
void derivatives (long trials, nr_fmt * nr, tarchive_fmt * tl, long G,
		  double *param, boolean forloci);
void solveParameters (tarchive_fmt * tyme, long G,
       world_fmt * world, double **dd, long chain, char type, char **plane);
void reset_nr (nr_fmt * nr);
void free_nr (nr_fmt * nr);
boolean is_singular (double **dd, long n);
void param_adjust (double *value, double oldval, double min, double max);
void param_all_adjust (nr_fmt * nr, double *param, long gamma_param);
/* calculate and adjust the new parameterset */
void calc_param (nr_fmt * nr, double *param, double lamda);

/* finds the biggest value in the vector */
double vector_max (double *v, long size);
/* calculate the norm sqrt(sum(v*v)) */
double norm (double *d, long size);
void print_contribution (nr_fmt * nr, tarchive_fmt * tyme, long G);


/*=======================================================*/

void 
estimateParameter (timearchive_fmt * tyme, long G,
	world_fmt * world, double **dd, long chain, char type, char **plane)
{
  switch (world->numpop)
    {				/* estimate new theta and m values */
    case 1:
      error ("The 1 world case with migration is not yet implemented!\n");
      break;
    case 2:
      solveParameters (tyme[0].tl, G, world, dd, chain, type, plane);
      break;
    default:
      error ("Multiworld estimators are not implemented yet! But will come!\n");
      break;
    }
}

/*=======================================================*/


/* parameter calculation with a specified
   interval sampling, not only the changed trees
   are sampled.

   1       Prob(G|Param)
   L(Param) = -  Sum[---------------, G]
   G       Prob(G|Param0)

   The parameter are found with a damped Newton-Raphson procedure.
 */
void 
solveParameters (tarchive_fmt * tyme, long G,
	world_fmt * world, double **dd, long chain, char type, char **plane)
{

  boolean notinverse;
  char *strllike, kind[20];
  long pop = -99, g, trials = -1, savety_belt;
  double **idd, *nld, lam1, lam2, lamda = 1., nld2, llike = -DBL_MAX, normd;
  nr_fmt *nr;

  nr = (nr_fmt *) calloc (1, sizeof (nr_fmt) * 1);
  nr->numpop = world->numpop;
  nr->numpop2 = world->numpop * 2;
  nr->partsize = (4 + nr->numpop2 * nr->numpop2);

  strllike = (char *) calloc (1, sizeof (char) * 128);
  nr->parts = (double *) calloc (1, nr->partsize * sizeof (double));
  nr->d = (double *) malloc (nr->numpop2 * sizeof (double));
  nr->param = (double *) malloc (nr->numpop2 * sizeof (double));
  nr->oparam = (double *) malloc (nr->numpop2 * sizeof (double));
  nr->datalike = (double *) malloc (G * sizeof (double));
  nr->apg0 = (double *) malloc (G * sizeof (double));
  nr->apg = (double *) malloc (G * sizeof (double));
  nld = (double *) malloc ((1 + NTRIALS) * sizeof (double));
  nr->dd = dd;
  idd = (double **) calloc (1, sizeof (double *) * nr->numpop2);
  idd[0] = (double *) calloc (1, sizeof (double) * nr->numpop2 * nr->numpop2);
  for (pop = 1; pop < nr->numpop2; pop++)
    {
      idd[pop] = idd[0] + pop * nr->numpop2;
    }
  memcpy (nr->param, world->param0, sizeof (double) * nr->numpop2);
  memcpy (nr->datalike, world->likelihood, sizeof (double) * G);

  /* Prob(G|Param0) */
  for (g = 0; g < G; g++)
    {
      nr->apg0[g] = probG (world->param0, &tyme[g], nr->numpop);
    }

  /* Newton Raphson loop */
  while (trials++ < NTRIALS)
    {
      reset_nr (nr);
      if (trials == 0)
	calc_like (nr, tyme, G);

      derivatives (trials, nr, tyme, G, world->param0, 0);
      derivatives_to_logderivatives (nr);
      normd = norm (nr->d, nr->numpop2);
      if (normd == 0.0)
	break;
      notinverse = is_singular (nr->dd, nr->numpop2);
      if (!notinverse)
	{
	  for (pop = 0; pop < nr->numpop2; pop++)
	    memcpy (idd[pop], nr->dd[pop], sizeof (double) * nr->numpop2);
	  invert_matrix (idd, 4);
	  if (nrcheck (nr->dd, idd, nr->d, 4, &lam1, &lam2, TRUE))
	    {
	      /* nrcheck calculates as a sideeffect 
	         the change in the newton case: 
	         d is already idd * gradient */
	      strcpy (kind, "NEWTON: ");
	      lamda = 1.;
	      nld[trials] = 1.0;
	    }
	  else
	    notinverse = TRUE;
	}
      else
	{
	  lam1 = lam2 = 1.0;
	}
      if (notinverse)
	{
	  if (lam2 > 0)
	    {
	      lamda = lam1 / lam2;
	      if (lamda >= 1.0)
		nld[trials] = normd;
	      else
		nld[trials] = normd * (lamda);
	    }
	  else
	    {
	      if (trials == 0)
		{
		  lamda = 1.0;
		  nld[0] = normd;
		}
	      else
		{
		  nld2 = normd;
		  if (nld2 == 0.0)
		    {
		      nld[trials] = 0.0;
		      lamda = 0.0;
		      fprintf (stderr, "norm(d) is 0.0 we should stop!");
		    }
		  else
		    {
		      lamda = vector_max (nld, trials) / nld2;
		      if (lamda >= 1.0)
			nld[trials] = normd;
		      else
			nld[trials] = nld2 * (lamda);
		    }
		}
	    }
	  strcpy (kind, "SLOW:   ");
	}
      calc_param (nr, world->param0, lamda);
      nr->ollike = nr->llike;
      calc_like (nr, tyme, G);
      savety_belt = 0;
      memcpy (nr->oparam, nr->param, nr->numpop2 * sizeof (double));
      if (nr->ollike > nr->llike)
	{			/* halfing if the new likelihood is worse than the old */
	  while (nr->llike - nr->ollike < -EPSILON && savety_belt++ < 50)
	    {
	      memcpy (nr->oparam, world->param0, nr->numpop2 * sizeof (double));
	      lamda /= 2.;
	      calc_param (nr, world->param0, lamda);
	      calc_like (nr, tyme, G);
	    }
	  if (savety_belt > 50)
	    {
	      fprintf (stderr, "halfing limit reached!\n");
	    }
	  memcpy (world->param0, nr->param, nr->numpop2 * sizeof (double));

	}
      else
	{			/* doubling if the new likelihood is better than the old */
	  while (nr->llike - nr->ollike > EPSILON && savety_belt++ < 50)
	    {
	      memcpy (nr->oparam, nr->param, nr->numpop2 * sizeof (double));
	      lamda *= 2.;
	      calc_param (nr, world->param0, lamda);
	      llike = nr->ollike;
	      nr->ollike = nr->llike;
	      nr->oPGC = nr->PGC;
	      calc_like (nr, tyme, G);
	    }
	  if (savety_belt > 0)
	    {
	      memcpy (world->param0, nr->oparam, nr->numpop2 * sizeof (double));
	      memcpy (nr->param, nr->oparam, nr->numpop2 * sizeof (double));
	      nr->llike = nr->ollike;
	      nr->ollike = llike;
	      nr->PGC = nr->oPGC;
	    }
	  else
	    {
	      calc_param (nr, world->param0, lamda);
	      memcpy (world->param0, nr->param, nr->numpop2 * sizeof (double));
	    }
	}
      if (!((((normd > 0.001) && (trials < NTRIALS))) || trials == 0))
	{
	  break;
	}
    }
  memcpy (world->param0, nr->param, nr->numpop2 * sizeof (double));
  llike = nr->llike;
  world->param_like = nr->llike;
  if (world->options->progress)
    {
      print_menu_chain (type, chain, G, world);
      if (world->options->verbose)
	{
	  print_contribution (nr, tyme, G);
	  fprintf (stdout, "           Maximization steps needed:   %li\n", trials);
	}
    }
  if (world->param_like < world->options->lcepsilon &&
      world->options->plotnow && !world->options->simulation)
    create_locus_plot (world, plane, tyme, nr, G);
  if (!world->options->simulation)
    calc_cov (nr->dd, nr->d, world->param0, 4);
  free (strllike);
  free_nr (nr);
  free (nld);
  free (idd[0]);
  free (idd);
}

/* calculates P(G | theta1,theta2,...,m1, m2,...)
   AND RETURNS a LOG(results)
 */

double 
probG (double *param, tarchive_fmt * tl, long numpop)
{
  long i;
  double result = 0;
  for (i = 0; i < numpop; i++)
    {
      result += tl->p[i] * (LOG2 - log (param[i])) + tl->l[i] * log (param[i + numpop]) - tl->km[i] * param[i + numpop] - tl->kt[i] / param[i];
    }
  return result;
}

boolean 
is_singular (double **dd, long n)
{
  long i, j;
  double temp;
  boolean singular = FALSE;
  for (i = 0; i < n; i++)
    {
      temp = 0.0;
      for (j = 0; j < n; j++)
	{
	  temp += dd[i][j];
	}
      if (temp == 0.0)
	{
	  singular = TRUE;
	  break;
	}
    }
  for (i = 0; i < n; i++)
    {
      temp = 0.0;
      for (j = 0; j < n; j++)
	{
	  temp += dd[i][j];
	}
      if (temp == 0.0)
	{
	  singular = TRUE;
	  break;
	}
    }
  return singular;
}


void 
param_all_adjust (nr_fmt * nr, double *param, long gamma_param)
{
  double ff, f = 1., denom;
  long i;
  boolean overboard = FALSE;

  double minima[5] =
  {SMALLEST_THETA, SMALLEST_THETA, SMALLEST_MIGRATION, SMALLEST_MIGRATION, SMALLEST_GAMMA};
  double maxima[5] =
  {BIGGEST_THETA, BIGGEST_THETA, BIGGEST_MIGRATION, BIGGEST_MIGRATION, BIGGEST_GAMMA};
  for (i = 0; i < 4 + gamma_param; i++)
    {
      if (nr->param[i] < minima[i] || nr->param[i] > maxima[i])
	{
	  overboard = TRUE;
	  break;
	}
    }
  if (overboard)
    {
      for (i = 0; i < 4 + gamma_param; i++)
	{
	  denom = nr->param[i] - param[i];
	  if (denom != 0)
	    {
	      ff = MIN (1., fabs ((minima[i] - param[i]) / denom));
	      ff = MIN (ff, fabs ((maxima[i] - param[i]) / denom));
	    }
	  else
	    ff = 1.;
	  if (ff < f)
	    f = ff;
	}
      if (f < 1.)
	{
	  for (i = 0; i < 4 + gamma_param; i++)
	    {
	      nr->param[i] = param[i] + f * (nr->param[i] - param[i]);
	    }
	}
    }
}


void 
calc_param (nr_fmt * nr, double *param, double lamda)
{
  long i;
  for (i = 0; i < nr->numpop2; i++)
    {
      nr->param[i] = param[i] * exp ((MAX (-100, MIN (-lamda * nr->d[i], 100))));
    }
  param_all_adjust (nr, param, 0);
}


double 
calc_like (nr_fmt * nr, tarchive_fmt * atl, long G)
{
  int g;
  double gsum = 0;
  nr->PGC = 0.0;
  nr->apg_max = -DBL_MAX;
  for (g = 0; g < G; g++)
    {
      nr->apg[g] = probG (nr->param, &atl[g], nr->numpop) - nr->apg0[g];
      if (nr->apg[g] > nr->apg_max)
	nr->apg_max = nr->apg[g];
    }
  for (g = 0; g < G; g++)
    {
      gsum += atl[g].copies;
      nr->apg[g] -= nr->apg_max;
      nr->PGC += atl[g].copies * exp (nr->apg[g]);
    }
  nr->llike = nr->apg_max + log (nr->PGC) - log (gsum);
  return nr->llike;
}

double 
norm (double *d, long size)
{
  int i;
  double sum = 0.;
  for (i = 0; i < (int) size; i++)
    {
      sum += d[i] * d[i];
    }
  return sqrt (sum);
}

double 
vector_max (double *v, long size)
{
  double maxval = -DBL_MAX;
  while (--size >= 0)
    if (v[size] > maxval)
      maxval = v[size];
  return maxval;
}

void 
reset_nr (nr_fmt * nr)
{
  long pop;
  memset (nr->d, 0, sizeof (double) * nr->numpop2);
  for (pop = 0; pop < nr->numpop2; pop++)
    memset (nr->dd[pop], 0, sizeof (double) * nr->numpop2);
  memset (nr->parts, 0, sizeof (double) * 2 * nr->numpop2);
}

void 
derivatives (long trials, nr_fmt * nr, tarchive_fmt * tl, long G,
	     double *param, boolean forloci)
{

  long g, j, i;
  double tsq1, tsq2, /*ttr1, ttr2, */ expapg, *thetas, *m;


  thetas = param;
  m = param + nr->numpop;
  tsq1 = param[0] * param[0];
  tsq2 = param[1] * param[1];
/*   ttr1 = tsq1 * param[0]; */
/*   ttr2 = tsq2 * param[1]; */

  for (g = 0; g < G; g++)
    {
      if (nr->apg[g] > -100)
	{
	  nr->parts[0] = (-tl[g].p[0] + tl[g].kt[0] / thetas[0]) / (thetas[0]);
	  nr->parts[1] = (-tl[g].p[1] + tl[g].kt[1] / thetas[1]) / (thetas[1]);
	  nr->parts[2] = tl[g].l[0] / m[0] - tl[g].km[0];
	  nr->parts[3] = tl[g].l[1] / m[1] - tl[g].km[1];
	  /* 2nd derivatives for x_i, x_i */
	  nr->parts[4] = (tl[g].p[0] - 2. * tl[g].kt[0] / thetas[0]) / (tsq1);
	  nr->parts[5] = (tl[g].p[1] - 2. * tl[g].kt[1] / thetas[1]) / (tsq2);
	  nr->parts[6] = -tl[g].l[0] / (m[0] * m[0]);
	  nr->parts[7] = -tl[g].l[1] / (m[1] * m[1]);
	  expapg = tl[g].copies * exp (nr->apg[g]);
	  for (i = 0; i < 4; i++)
	    {
	      nr->d[i] += expapg * nr->parts[i];
	      nr->dd[i][i] += expapg * (nr->parts[i] * nr->parts[i] + nr->parts[4 + i]);
	      for (j = 0; j < i; j++)
		nr->dd[i][j] += expapg * (nr->parts[i] * nr->parts[j]);
	    }
	}
    }
  for (i = 0; i < 4; i++)
    nr->d[i] /= nr->PGC;
  for (i = 0; i < 4; i++)
    {
      for (j = 0; j < i; j++)
	{
	  nr->dd[i][j] = -(nr->dd[i][j]) / nr->PGC + nr->d[i] * nr->d[j];
	}
      nr->dd[i][i] = -(nr->dd[i][i]) / nr->PGC + nr->d[i] * nr->d[i];
    }
  for (i = 0; i < 4; i++)
    nr->d[i] = -nr->d[i];
}



void 
calc_cov (double **dd, double *d, double *param, long n)
{
  long i, j;
  for (i = 0; i < n; i++)
    {
      for (j = 0; j < i; j++)
	{
	  dd[i][j] /= (param[i] * param[j]);
	  dd[j][i] = dd[i][j];
	}
      dd[i][i] = (dd[i][i] - param[i] * d[i]) / (param[i] * param[i]);
    }
  if (!is_singular (dd, n))
    invert_matrix (dd, n);
}


void 
free_nr (nr_fmt * nr)
{
  free (nr->parts);
  free (nr->param);
  free (nr->oparam);
  free (nr->datalike);
  free (nr->apg0);
  free (nr->apg);
  free (nr->d);
  free (nr);

}

/* change of variables: from parameters to log(parameters) */
void 
derivatives_to_logderivatives (nr_fmt * nr)
{
  long i, j;
  for (i = 0; i < nr->numpop2; i++)
    {
      for (j = 0; j < i; j++)
	{
	  nr->dd[i][j] = nr->param[i] * nr->param[j] * nr->dd[i][j];
	  nr->dd[j][i] = nr->dd[i][j];
	}
      nr->dd[i][i] = nr->param[i] * nr->d[i] + nr->param[i] *
	nr->param[i] * nr->dd[i][i];
    }
  for (i = 0; i < nr->numpop2; i++)
    {
      nr->d[i] = nr->param[i] * nr->d[i];
    }
}

void 
print_contribution (nr_fmt * nr, tarchive_fmt * tyme, long G)
{
  long g;
  long contribution[11];
  for (g = 0; g < 11; g++)
    contribution[g] = 0;
  for (g = 0; g < G; g++)
    {
      if (nr->apg[g] > -20)
	{
	  contribution[9 - (long) (fabs (nr->apg[g]) / 2)] += tyme[g].copies;
	}
      contribution[10] += tyme[g].copies;
    }
  fprintf (stdout, "           log(P(g|Param))  -20 to ");
  for (g = -18; g <= 0; g += 2)
    {
      fprintf (stdout, "%4li ", g);
    }
  fprintf (stdout, "  All\n");
  fprintf (stdout, "           Counts                  ");
  for (g = 0; g < 10; g++)
    {
      fprintf (stdout, "%4li ", contribution[g]);
    }
  fprintf (stdout, "%5li\n", contribution[10]);
}
/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 R A N D O M   G E N E R A T O R   R O U T I N E S 

 creates options structures,
 reads options from parmfile if present
 
 prints options,
 and finally helps to destroy itself.
                                                                                                               
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: random.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/


#include "migration.h"

/*extern gettimeofday(); */
/* prototypes ----------------------------------------- */
void getseed (option_fmt * options);

#ifdef HIGHBITS
double mk_randum (void);
#define RANDUM() mk_randum()
#define RANDINT(a,b) (long)((a)+(mk_randum()* (((b)-(a))+1.)))
#else
double jf_randum (void);
#define RANDUM() jf_randum()
#define RANDINT(a,b) (long)((a)+(jf_randum()* (((b)-(a))+1.)))
#endif


void 
getseed (option_fmt * options)
{
  long i, timeseed;

  switch (options->autoseed)
    {
    case AUTO:
      timeseed = (long) time (NULL);
      options->inseed = labs(4 * timeseed + 1);
      break;
    case NOAUTO:
      break;
    case NOAUTOSELF:
      break;
    default:
      error ("Error: Seed value not defined");
      break;
    }
  options->saveseed = options->inseed;
  for (i = 1; i <= 1000; i++)	/* clear the random numbergenerator */
    RANDUM ();
#ifdef HIGHBITS
  for (i = 0; i <= 2; i++)
    seed[i] = 0;
  i = 0;
  do
    {
      seed[i] = options->inseed & 2047;
      options->inseed /= 2048;
      i++;
    }
  while (options->inseed != 0);
#else
  for (i = 0; i <= 5; i++)
    seed[i] = 0;
  i = 0;
  do
    {
      seed[i] = options->inseed & 63;
      options->inseed /= 64;
      i++;
    }
  while (options->inseed != 0);
#endif
}

#ifdef HIGHBITS
double 
mk_randum (void)
/* Mary's version--faster but needs 32 bits.  Loops have been unrolled
   for speed. */
{
  longer newseed;

  newseed[0] = 1549 * seed[0];
  newseed[1] = newseed[0] / 2048;
  newseed[0] &= 2047;
  newseed[2] = newseed[1] / 2048;
  newseed[1] &= 2047;
  newseed[1] += 1549 * seed[1] + 812 * seed[0];
  newseed[2] += newseed[1] / 2048;
  newseed[1] &= 2047;
  newseed[2] += 1549 * seed[2] + 812 * seed[1];

  memcpy (seed, newseed, sizeof (longer));
  seed[2] &= 1023;
  return (((seed[0] / 2048.0 + seed[1]) / 2048.0 + seed[2]) / 1024.0);
}				/* randum */
#else
double 
jf_randum (void)
{				/* randum -- slow but machine independent */
  /* random number generator -- slow but machine independent */
  long i, j, k, sum;
  longer mult, newseed;
  double x;

  mult[0] = 13;
  mult[1] = 24;
  mult[2] = 22;
  mult[3] = 6;
  for (i = 0; i <= 5; i++)
    newseed[i] = 0;
  for (i = 0; i <= 5; i++)
    {
      sum = newseed[i];
      k = i;
      if (i > 3)
	k = 3;
      for (j = 0; j <= k; j++)
	sum += mult[j] * seed[i - j];
      newseed[i] = sum;
      for (j = i; j <= 4; j++)
	{
	  newseed[j + 1] += newseed[j] / 64;
	  newseed[j] &= 63;
	}
    }
  memcpy (seed, newseed, sizeof (longer));
  seed[5] &= 3;
  x = 0.0;
  for (i = 0; i <= 5; i++)
    x = x / 64.0 + seed[i];
  x /= 4.0;
  return x;
}				/* randum */
#endif
/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 S E Q U E N C E S   R O U T I N E S 

 used in menu.c

 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: sequence.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif
/* prototypes ------------------------------------------- */
void make_sequences (world_fmt * world, long locus);
void init_sequences (world_fmt * world, long locus);
void init_sequences2 (seqmodel_fmt * seq, option_fmt * options, long locus);
void initratio (option_fmt * options);
void initfreqs (double *freqa, double *freqc, double *freqg, double *freqt);
void initcatn (long *categs);
void initcategs (long categs, double *rate);
void initprobcat (long categs, double *probsum, double *probcat);
void init_tbl (world_fmt * world, long locus);
void print_weights (world_fmt * world, long locus);
void print_tbl (world_fmt * world, long locus);
double treelike_seq (world_fmt * world, long locus);
/*private functions */
void getbasefreqs (option_fmt * options, seqmodel_fmt * seq, long locus);
void empiricalfreqs (world_fmt * world, seqmodel_fmt * seq, long locus);
void makeweights (world_fmt * world, long locus);
void makevalues_seq (world_fmt * world, long locus);
void sitecombine2 (world_fmt * world, long sites, long locus);
void sitesort2 (world_fmt * world, long sites, long locus);
void sitescrunch2 (world_fmt * world, long sites,
		   long i, long j, long locus);
void inputoptions (world_fmt * world, long locus);
void inputweights (world_fmt * world, long chars);
void inputcategs (long a, long b, world_fmt * world);
void initlambda (option_fmt * options);
void printweights (world_fmt * world, short inc, long chars, short *weight, char *letters);
void print_seqfreqs (world_fmt * world);

const short C = 1;
const short G = 2;
const short T = 3;


/* allocation things */
void
init_sequences (world_fmt * world, long locus)
{
  long sites = world->data->seq->sites[locus];
  if (locus == 0)
    {
      world->data->seq->alias = (long *) calloc (1, sizeof (long) * sites);
      world->data->seq->ally = (long *) calloc (1, sizeof (long) * sites);
      world->data->seq->aliasweight = (long *) calloc (1, sizeof (long) * sites);
      world->data->seq->location = (long *) calloc (1, sizeof (long) * sites);
      world->data->seq->category = (long *) calloc (1, sizeof (long) * sites);
      world->data->seq->weight = (short *) calloc (1, sizeof (short) * sites);
    }
  else
    {
      world->data->seq->alias = (long *) realloc (world->data->seq->alias, sizeof (long) * sites);
      world->data->seq->ally = (long *) realloc (world->data->seq->ally, sizeof (long) * sites);
      world->data->seq->aliasweight = (long *) realloc (world->data->seq->aliasweight, sizeof (long) * sites);
      world->data->seq->location = (long *) realloc (world->data->seq->location, sizeof (long) * sites);
      world->data->seq->category = (long *) realloc (world->data->seq->category, sizeof (long) * sites);
      world->data->seq->weight = (short *) realloc (world->data->seq->weight, sizeof (short) * sites);
    }
  inputoptions (world, locus);
  if (!world->options->freqsfrom)
    getbasefreqs (world->options, world->data->seq, locus);
  makeweights (world, locus);

}



/* menu material ----------------------------------------- */
void
initratio (option_fmt * options)
{
  long z = 0;
  char *tmp;
  char input[LINESIZE];
  printf ("Transition/transversion ratio?\nEnter a value for each locus, spaced by blanks or commas\n");
  fgets (input, LINESIZE, stdin);
  tmp = strtok (input, " ,\n");
  while (tmp != NULL)
    {
      options->ttratio[z++] = atof (tmp);
      tmp = strtok (NULL, " ,;\n");
      options->ttratio = (double *) realloc (options->ttratio, sizeof (double) * (z + 1));
      options->ttratio[z] = 0.0;
    }
}

void
initfreqs (double *freqa, double *freqc, double *freqg, double *freqt)
{
  char input[LINESIZE];
  int scanned;
  printf ("Base frequencies for A, C, G, T/U (use blanks to separate)?\n");
  for (;;)
    {
      fgets (input, LINESIZE, stdin);
      scanned = sscanf (input, "%lf%lf%lf%lf%*[^\n]", freqa, freqc, freqg, freqt);
      if (scanned == 4)
	break;
      else
	printf ("Please enter exactly 4 values.\n");
    };
}


void
initcatn (long *categs)
{				/* initialize category number */

  do
    {
      printf ("Number of categories (1-%d)?\n", MAXCATEGS);
      scanf ("%ld%*[^\n]", categs);
      getchar ();
    }
  while (*categs > MAXCATEGS || *categs < 1);
}


void
initcategs (long categs, double *rate)
{				/* initialize category rates */
  long i;
  char input[LINESIZE];
  char rest[LINESIZE];
  int scanned;
  boolean done;

  for (;;)
    {
      printf ("Rate for each category? (use a space to separate)\n");
      fgets (input, LINESIZE, stdin);
      done = TRUE;
      for (i = 0; i < categs; i++)
	{
	  scanned = sscanf (input, "%lf %[^\n]", &rate[i], rest);
	  if ((scanned < 2 && i < (categs - 1)) ||
	      (scanned < 1 && i == (categs - 1)))
	    {
	      printf ("Please enter exactly %ld values.\n", categs);
	      done = FALSE;
	      break;
	    }
	  strcpy (input, rest);
	}
      if (done)
	break;
    }
}


void
initprobcat (long categs, double *probsum, double *probcat)
{
  long i;
  boolean done;
  char input[LINESIZE];
  char rest[LINESIZE];
  int scanned;

  do
    {
      printf ("Probability for each category?");
      printf (" (use a space to separate)\n");
      fgets (input, LINESIZE, stdin);
      done = TRUE;
      for (i = 0; i < categs; i++)
	{
	  scanned = sscanf (input, "%lf %[^\n]", &probcat[i], rest);
	  if ((scanned < 2 && i < (categs - 1)) ||
	      (scanned < 1 && i == (categs - 1)))
	    {
	      done = FALSE;
	      printf ("Please enter exactly %ld values.\n", categs);
	      break;
	    }
	  strcpy (input, rest);
	}
      if (!done)
	continue;
      *probsum = 0.0;
      for (i = 0; i < categs; i++)
	*probsum += probcat[i];
      if (fabs (1.0 - (*probsum)) > 0.001)
	{
	  done = FALSE;
	  printf ("Probabilities must add up to");
	  printf (" 1.0, plus or minus 0.001.\n");
	}
    }
  while (!done);
}


/*data read material ===================================== */

void
make_sequences (world_fmt * world, long locus)
{
  makevalues_seq (world, locus);
  if (world->options->freqsfrom)
    {
      empiricalfreqs (world, world->data->seq, locus);
      getbasefreqs (world->options, world->data->seq, locus);
    }
}

/* private functions================================== */
void
getbasefreqs (option_fmt * options, seqmodel_fmt * seq, long locus)
{
  long l;
  double aa, bb;
  if(locus==0)
    seq->ttratio = options->ttratio[0];
  else
    {
      for (l=1;l<=locus;l++)
	{
	  if(options->ttratio[l]==0.0)
	    seq->ttratio = options->ttratio[l-1];
	    break;
	}
      if(l>locus)
	seq->ttratio = options->ttratio[locus];
    }
  seq->freqa = options->freqa;
  seq->freqc = options->freqc;
  seq->freqg = options->freqg;
  seq->freqt = options->freqt;
  seq->freqr = seq->freqa + seq->freqg;
  seq->freqy = seq->freqc + seq->freqt;
  seq->freqar = seq->freqa / seq->freqr;
  seq->freqcy = seq->freqc / seq->freqy;
  seq->freqgr = seq->freqg / seq->freqr;
  seq->freqty = seq->freqt / seq->freqy;
  aa = seq->ttratio * (seq->freqr) * (seq->freqy)
    - seq->freqa * seq->freqg - seq->freqc * seq->freqt;
  bb = seq->freqa * (seq->freqgr) + seq->freqc * (seq->freqty);
  seq->xi = aa / (aa + bb);
  seq->xv = 1.0 - seq->xi;
  if (seq->xi <= 0.0)
    {
      printf ("\n WARNING: This transition/transversion ratio\n");
      printf (" is impossible with these base frequencies!\n");
      seq->xi = 0.0;
      seq->xv = 1.0;
      seq->ttratio = (seq->freqa * seq->freqg
		      + seq->freqc * seq->freqt)
	/ ((seq->freqr) * (seq->freqy));
      printf (" Transition/transversion parameter reset\n");
      printf ("  so transition/transversion ratio is %10.6f\n\n", (seq->ttratio));
    }
  seq->fracchange = (seq->xi) * (2 * seq->freqa * (seq->freqgr)
				 + 2 * seq->freqc * (seq->freqty)) +
    (seq->xv) * (1.0 - seq->freqa * seq->freqa
		 - seq->freqc * seq->freqc - seq->freqg
		 * seq->freqg
		 - seq->freqt * seq->freqt);
}

/*===================================================*/

void
makeweights (world_fmt * world, long locus)
{
  /* make up weights vector to avoid duplicate computations */
  long i;

  for (i = 1; i <= world->data->seq->sites[locus]; i++)
    {
      world->data->seq->alias[i - 1] = i;
      world->data->seq->ally[i - 1] = 0;
      world->data->seq->aliasweight[i - 1] = world->data->seq->weight[i - 1];
      world->data->seq->location[i - 1] = 0;
    }
  sitesort2 (world, world->data->seq->sites[locus], locus);
  sitecombine2 (world, world->data->seq->sites[locus], locus);
  sitescrunch2 (world, world->data->seq->sites[locus], 1, 2, locus);
  for (i = 1; i <= world->data->seq->sites[locus]; i++)
    {
      if (world->data->seq->aliasweight[i - 1] > 0)
	world->data->seq->endsite = i;
    }
  for (i = 1; i <= world->data->seq->endsite; i++)
    {
      world->data->seq->location[world->data->seq->alias[i - 1] - 1] = i;
      world->data->seq->ally[world->data->seq->alias[i - 1] - 1] = world->data->seq->alias[i - 1];
    }
  init_sequences2 (world->data->seq, world->options, locus);
}				/* makeweights */

void
init_sequences2 (seqmodel_fmt * seq, option_fmt * options, long locus)
{
  /*long i;
     seq->term = (double **) malloc(seq->endsite * sizeof(double *));
     for (i = 0; i <seq->endsite; i++)
     seq->term[i] = (double *) malloc(options->rcategs * sizeof(double));
     seq->slopeterm = (double **) malloc(seq->endsite * sizeof(double *));
     for (i = 0; i <seq->endsite; i++)
     seq->slopeterm[i] = (double *) malloc(options->rcategs * sizeof(double));
     seq->curveterm = (double **) malloc(seq->endsite * sizeof(double *));
     for (i = 0; i <seq->endsite; i++)
     seq->curveterm[i] = (double *) malloc(options->rcategs * sizeof(double));
     seq->mp = (val *) malloc(seq->sites[locus] * sizeof(val)); */
  if (locus == 0)
    seq->contribution = (contribarr *) malloc (seq->endsite * sizeof (contribarr));
  else
    seq->contribution = (contribarr *) realloc (seq->contribution, seq->endsite * sizeof (contribarr));

}


void
makevalues_seq (world_fmt * world, long locus)
{
  long i, ii, j, k, l, pop;
  long b;
  node **treenode = world->nodep;
  for (k = 0; k < world->data->seq->endsite; k++)
    {
      j = world->data->seq->alias[k] - 1;
      i = -1;
      for (pop = 0; pop < world->numpop; pop++)
	{
	  for (ii = 0; ii < world->data->numind[pop][locus]; ii++)
	    {
	      i++;
	      if (!world->options->usertree)
		strcpy (treenode[i]->nayme, world->data->indnames[pop][ii]);
	      for (l = 0; l < world->options->rcategs; l++)
		{		/*?????????????????was categs need to ask joe?????????????? */
		  for (b = 0; b < 4; b++)
		    treenode[i]->x.s[k][l][b] = 0.0;
		  switch (world->data->yy[pop][ii][locus][0][j])
		    {

		    case 'A':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      break;

		    case 'C':
		      treenode[i]->x.s[k][l][C] = 1.0;
		      break;

		    case 'G':
		      treenode[i]->x.s[k][l][G] = 1.0;
		      break;

		    case 'T':
		      treenode[i]->x.s[k][l][T] = 1.0;
		      break;

		    case 'U':
		      treenode[i]->x.s[k][l][T] = 1.0;
		      break;

		    case 'M':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      treenode[i]->x.s[k][l][C] = 1.0;
		      break;

		    case 'R':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      treenode[i]->x.s[k][l][G] = 1.0;
		      break;

		    case 'W':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      treenode[i]->x.s[k][l][T] = 1.0;
		      break;

		    case 'S':
		      treenode[i]->x.s[k][l][C] = 1.0;
		      treenode[i]->x.s[k][l][G] = 1.0;
		      break;

		    case 'Y':
		      treenode[i]->x.s[k][l][C] = 1.0;
		      treenode[i]->x.s[k][l][T] = 1.0;
		      break;

		    case 'K':
		      treenode[i]->x.s[k][l][G] = 1.0;
		      treenode[i]->x.s[k][l][T] = 1.0;
		      break;

		    case 'B':
		      treenode[i]->x.s[k][l][C] = 1.0;
		      treenode[i]->x.s[k][l][G] = 1.0;
		      treenode[i]->x.s[k][l][T] = 1.0;
		      break;

		    case 'D':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      treenode[i]->x.s[k][l][G] = 1.0;
		      treenode[i]->x.s[k][l][T] = 1.0;
		      break;

		    case 'H':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      treenode[i]->x.s[k][l][C] = 1.0;
		      treenode[i]->x.s[k][l][T] = 1.0;
		      break;

		    case 'V':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      treenode[i]->x.s[k][l][C] = 1.0;
		      treenode[i]->x.s[k][l][G] = 1.0;
		      break;

		    case 'N':
		      for (b = 0; b < 4; b++)
			treenode[i]->x.s[k][l][b] = 1.0;
		      break;

		    case 'X':
		      for (b = 0; b < 4; b++)
			treenode[i]->x.s[k][l][b] = 1.0;
		      break;

		    case '?':
		      for (b = 0; b < 4; b++)
			treenode[i]->x.s[k][l][b] = 1.0;
		      break;

		    case 'O':
		      for (b = 0; b < 4; b++)
			treenode[i]->x.s[k][l][b] = 1.0;
		      break;

		    case '-':
		      for (b = 0; b < 4; b++)
			treenode[i]->x.s[k][l][b] = 1.0;
		      break;
		    }
		}
	    }
	}
    }
}

void
sitesort2 (world_fmt * world, long sites, long locus)
{
  long gap, i, j, jj, jg, k, kk, itemp, pop;
  boolean flip, tied, samewt;
  seqmodel_fmt *seq;
  long *tempsum;
  tempsum = (long *) calloc (1, sizeof (long) * world->numpop);
  tempsum[0] = world->data->numind[0][FLOC];
  for (i = 1; i < world->numpop; i++)
    {
      tempsum[i] = tempsum[i - 1] + world->data->numind[i][FLOC];
    }
  seq = world->data->seq;
  gap = sites / 2;
  while (gap > 0)
    {
      for (i = gap + 1; i <= sites; i++)
	{
	  j = i - gap;
	  flip = TRUE;
	  while (j > 0 && flip)
	    {
	      jj = seq->alias[j - 1];
	      jg = seq->alias[j + gap - 1];
	      samewt = ((seq->weight[jj - 1] != 0) && (seq->weight[jg - 1] != 0))
		|| ((seq->weight[jj - 1] == 0) && (seq->weight[jg - 1] == 0));
	      tied = samewt && (seq->category[jj - 1] == seq->category[jg - 1]);
	      flip = ((!samewt) && (seq->weight[jj - 1] == 0))
		|| (samewt && (seq->category[jj - 1] > seq->category[jg - 1]));
	      k = 0;
	      pop = 0;
	      kk = -1;
	      while (k < world->sumtips && tied)
		{
		  if (k == tempsum[pop])
		    {
		      kk = 0;
		      pop++;
		    }
		  else
		    {
		      kk++;
		    }
		  flip = (world->data->yy[pop][kk][locus][0][jj - 1] >
			  world->data->yy[pop][kk][locus][0][jg - 1]);
		  tied = (tied && world->data->yy[pop][kk][locus][0][jj - 1]
			  == world->data->yy[pop][kk][locus][0][jg - 1]);
		  k++;
		}
	      if (!flip)
		break;
	      itemp = seq->alias[j - 1];
	      seq->alias[j - 1] = seq->alias[j + gap - 1];
	      seq->alias[j + gap - 1] = itemp;
	      itemp = seq->aliasweight[j - 1];
	      seq->aliasweight[j - 1] = seq->aliasweight[j + gap - 1];
	      seq->aliasweight[j + gap - 1] = itemp;
	      j -= gap;
	    }
	}
      gap /= 2;
    }
  free (tempsum);
}				/* sitesort2 */


void
sitecombine2 (world_fmt * world, long sites, long locus)
{
  long i, j, k, kk, pop;
  boolean tied, samewt;
  seqmodel_fmt *seq;
  long *tempsum;
  tempsum = (long *) calloc (1, sizeof (long) * world->numpop);
  tempsum[0] = world->data->numind[0][FLOC];
  for (i = 1; i < world->numpop; i++)
    {
      tempsum[i] = tempsum[i - 1] + world->data->numind[i][FLOC];
    }

  seq = world->data->seq;
  i = 1;
  while (i < sites)
    {
      j = i + 1;
      tied = TRUE;
      while (j <= sites && tied)
	{
	  samewt = ((seq->aliasweight[i - 1] != 0) && (seq->aliasweight[j - 1] != 0))
	    || ((seq->aliasweight[i - 1] == 0) && (seq->aliasweight[j - 1] == 0));
	  tied = samewt
	    && (seq->category[seq->alias[i - 1] - 1] == seq->category[seq->alias[j - 1] - 1]);
	  k = 0;
	  pop = 0;
	  kk = -1;
	  while (k < world->sumtips && tied)
	    {
	      if (k == tempsum[pop])
		{
		  kk = 0;
		  pop++;
		}
	      else
		{
		  kk++;
		}
	      tied = (tied &&
		   world->data->yy[pop][kk][locus][0][seq->alias[i - 1] - 1]
	      == world->data->yy[pop][kk][locus][0][seq->alias[j - 1] - 1]);
	      k++;
	    }
	  if (!tied)
	    break;
	  seq->aliasweight[i - 1] += seq->aliasweight[j - 1];
	  seq->aliasweight[j - 1] = 0;
	  seq->ally[seq->alias[j - 1] - 1] = seq->alias[i - 1];
	  j++;
	}
      i = j;
    }
}				/* sitecombine2 */


void
sitescrunch2 (world_fmt * world, long sites,
	      long i, long j, long locus)
{
  /* move so positively weighted sites come first */
  /* used by dnainvar, dnaml, dnamlk, & restml */
  long itemp;
  boolean done, found;
  seqmodel_fmt *seq;
  seq = world->data->seq;
  done = FALSE;
  while (!done)
    {
      found = FALSE;
      if (seq->aliasweight[i - 1] > 0)
	i++;
      else
	{
	  if (j <= i)
	    j = i + 1;
	  if (j <= sites)
	    {
	      found = FALSE;
	      do
		{
		  found = (seq->aliasweight[j - 1] > 0);
		  j++;
		}
	      while (!(found || j > sites));
	      if (found)
		{
		  j--;
		  itemp = seq->alias[i - 1];
		  seq->alias[i - 1] = seq->alias[j - 1];
		  seq->alias[j - 1] = itemp;
		  itemp = seq->aliasweight[i - 1];
		  seq->aliasweight[i - 1] = seq->aliasweight[j - 1];
		  seq->aliasweight[j - 1] = itemp;
		}
	      else
		done = TRUE;
	    }
	  else
	    done = TRUE;
	}
      done = (done || i >= sites);
    }
}				/* sitescrunch2 */

void
inputoptions (world_fmt * world, long locus)
{
  long i;
  long sites = world->data->seq->sites[locus];
  for (i = 0; i < sites; i++)
    world->data->seq->category[i] = 1;
  for (i = 0; i < sites; i++)
    world->data->seq->weight[i] = 1;
  if (world->options->weights)
    inputweights (world, sites);
  world->data->seq->weightsum = 0;
  for (i = 0; i < sites; i++)
    world->data->seq->weightsum += world->data->seq->weight[i];
  if (world->options->categs > 1)
    {
      inputcategs (0, sites, world);
    }
}				/* inputoptions */

void
inputweights (world_fmt * world, long chars)
{
  /* input the character weights, 0-9 and A-Z for weights 0 - 35 */
  char ch;
  long i;

  for (i = 0; i < chars; i++)
    {
      do
	{
	  ch = getc (world->data->weightfile);
	  if (ch == '\n')
	    ch = ' ';
	}
      while (ch == ' ');
      world->data->seq->weight[i] = 1;
      if (isdigit ((int) ch))
	world->data->seq->weight[i] = ch - '0';
      else if (isalpha ((int) ch))
	{
	  ch = uppercase (ch);
	  world->data->seq->weight[i] = ch - 'A' + 10;
	}
      else
	{
	  printf ("BAD WEIGHT CHARACTER: %c\n", ch);
	  exit (EXIT_FAILURE);
	}
    }
}				/* inputweights */

void
inputcategs (long a, long b, world_fmt * world)
{
  /* input the categories, 1-9 */
  char ch;
  long i;
  option_fmt *options = world->options;

  ch = getc (world->data->catfile);
  while (ch == '#')
    {
      while (ch != '\n')
	ch = getc (world->data->catfile);
      ch = getc (world->data->catfile);
    }
  ungetc (ch, world->data->catfile);
  fscanf (world->data->catfile, "%ld", &options->categs);
  options->rate = (double *) realloc (options->rate, sizeof (double)
				      * options->categs);
  for (i = 0; i < options->categs; i++)
    {
      fscanf (world->data->catfile, "%lf", &options->rate[i]);
    }

  for (i = a; i < b; i++)
    {
      do
	{
	  ch = getc (world->data->catfile);
	  if (ch == '\n')
	    ch = ' ';
	}
      while (ch == ' ');
      if ((ch >= '1') && (ch <= ('0' + world->options->categs)))
	world->data->seq->category[i] = ch - '0';
      else
	{
	  printf ("BAD CATEGORY CHARACTER: %c -- CATEGORIES ARE CURRENTLY 1-%ld\n",
		  ch, world->options->categs);
	  exit (EXIT_FAILURE);
	}
    }
}				/* inputcategs */




void
empiricalfreqs (world_fmt * world, seqmodel_fmt * seq, long locus)
{
  /* Get empirical base frequencies from the data */
  /* used in dnaml & dnamlk */
  long i, j, k;
  double sum, suma, sumc, sumg, sumt, w;

  world->options->freqa = 0.25;
  world->options->freqc = 0.25;
  world->options->freqg = 0.25;
  world->options->freqt = 0.25;
  for (k = 1; k <= 8; k++)
    {
      suma = 0.0;
      sumc = 0.0;
      sumg = 0.0;
      sumt = 0.0;
      for (i = 0; i < world->sumtips; i++)
	{
	  for (j = 0; j < seq->endsite; j++)
	    {
	      w = seq->aliasweight[j];
	      sum = (world->options->freqa) * world->nodep[i]->x.s[j][0][0];
	      sum += (world->options->freqc) * world->nodep[i]->x.s[j][0][C];
	      sum += (world->options->freqg) * world->nodep[i]->x.s[j][0][G];
	      sum += (world->options->freqt) * world->nodep[i]->x.s[j][0][T];
	      suma += w * (world->options->freqa) * world->nodep[i]->x.s[j][0][0] / sum;
	      sumc += w * (world->options->freqc) * world->nodep[i]->x.s[j][0][C] / sum;
	      sumg += w * (world->options->freqg) * world->nodep[i]->x.s[j][0][G] / sum;
	      sumt += w * (world->options->freqt) * world->nodep[i]->x.s[j][0][T] / sum;
	    }
	}
      sum = suma + sumc + sumg + sumt;
      world->options->freqa = suma / sum;
      world->options->freqc = sumc / sum;
      world->options->freqg = sumg / sum;
      world->options->freqt = sumt / sum;
    }
}				/* empiricalfreqs */


void
initlambda (option_fmt * options)
{
  do
    {
      printf ("Mean block length of sites having the same rate (greater than 1)?\n");
      scanf ("%lf%*[^\n]", &options->lambda);
      getchar ();
    }
  while (options->lambda <= 1.0);
  options->lambda = 1.0 / options->lambda;
}



void
init_tbl (world_fmt * world, long locus)
{
  /* Define a lookup table. Precompute values and print them out in tables */
  long i, j;
  double sumrates;
  long categs = world->options->categs;
  long rcategs = world->options->rcategs;
  world->data->seq->tbl = (valrec ***) malloc (rcategs * sizeof (valrec **));
  for (i = 0; i < rcategs; i++)
    {
      world->data->seq->tbl[i] = (valrec **) malloc (categs * sizeof (valrec *));
      for (j = 0; j < categs; j++)
	world->data->seq->tbl[i][j] = (valrec *) malloc (sizeof (valrec));
    }
  for (i = 0; i < rcategs; i++)
    {
      for (j = 0; j < categs; j++)
	{
	  world->data->seq->tbl[i][j]->rat =
	    world->options->rrate[i] * world->options->rate[j];
	  world->data->seq->tbl[i][j]->ratxi =
	    world->data->seq->tbl[i][j]->rat * world->data->seq->xi;
	  world->data->seq->tbl[i][j]->ratxv =
	    world->data->seq->tbl[i][j]->rat * world->data->seq->xv;
	}
    }
  sumrates = 0.0;
  for (i = 0; i < world->data->seq->endsite; i++)
    {
      for (j = 0; j < rcategs; j++)
	sumrates += world->data->seq->aliasweight[i]
	  * world->options->probcat[j]
	  * world->data->seq->tbl[j][world->data->seq->category[world->data->seq->alias[i] - 1] - 1]->rat;
    }
  sumrates /= (double) world->data->seq->sites[locus];
  for (i = 0; i < rcategs; i++)
    for (j = 0; j < categs; j++)
      {
	world->data->seq->tbl[i][j]->rat /= sumrates;
	world->data->seq->tbl[i][j]->ratxi /= sumrates;
	world->data->seq->tbl[i][j]->ratxv /= sumrates;
      }
}				/* inittable */

void
print_weights (world_fmt * world, long locus)
{
  if (world->options->weights)
    {
      if (world->options->printdata)
	{
	  printweights (world, 0, world->data->seq->sites[locus],
			world->data->seq->weight, "Sites");
	}
    }
}

void
print_tbl (world_fmt * world, long locus)
{
  long i;

  option_fmt *opt;
  FILE *outfile;
  outfile = world->outfile;

  opt = world->options;
  if (opt->rcategs > 1)
    {
      fprintf (outfile, "Region type     Rate of change    Probability\n");
      fprintf (outfile, "---------------------------------------------\n");
      for (i = 0; i < opt->rcategs; i++)
	fprintf (outfile, "%9ld%16.3f%17.3f\n", i + 1, opt->rrate[i], opt->probcat[i]);
      putc ('\n', outfile);
      if (opt->autocorr)
	fprintf (outfile,
	"Expected length of a patch of sites having the same rate = %8.3f\n",
		 1. / opt->lambda);
      putc ('\n', outfile);
    }
  if (opt->categs > 1)
    {
      fprintf (outfile, "Site category   Rate of change\n");
      fprintf (outfile, "------------------------------\n");
      for (i = 0; i < opt->categs; i++)
	fprintf (outfile, "%9ld%16.3f\n", i + 1, opt->rate[i]);
    }
  if ((opt->rcategs > 1) || (opt->categs > 1))
    fprintf (outfile, "\n");
}


void
printweights (world_fmt * world, short inc, long chars, short *weight, char *letters)
{
  /* print out the weights of sites */
  long i, j;
  FILE *filename;
  filename = world->outfile;
  fprintf (filename, "\n    %s are weighted as follows:\n", letters);
  for (i = 0; i < chars; i++)
    {
      if (i % 60 == 0)
	{
	  putc ('\n', filename);
	  for (j = 1; j <= world->options->nmlength + 3; j++)
	    putc (' ', filename);
	}
      fprintf (filename, "%hd", weight[i + inc]);
      if ((i + 1) % 10 == 0 && (i + 1) % 60 != 0)
	putc (' ', filename);
    }
  fprintf (filename, "\n\n");
}				/* printweights */



void
print_seqfreqs (world_fmt * world)
{
  option_fmt *options = world->options;
  if (world->locus == 0)
    {
      if (options->freqsfrom)
	fprintf (world->outfile, "Empirical ");
      fprintf (world->outfile, "Base Frequencies\n");
      fprintf (world->outfile, "------------------------------------------------------------\n");
      fprintf (world->outfile, "Locus     Nucleotide                        Transition/\n");
      fprintf (world->outfile, "          ------------------------------  Transversion ratio\n");
      fprintf (world->outfile, "          A       C       G       T(U)\n");
      fprintf (world->outfile, "------------------------------------------------------------\n");
    }
  /*  fprintf(world->outfile,  "1234      0.245  0.333  0.345  0.234 */
  fprintf (world->outfile, "%4li      %6.4f  %6.4f  %6.4f  %6.4f    %10.5f\n",
	   world->locus + 1, options->freqa, options->freqc,
	   options->freqg, options->freqt, world->data->seq->ttratio);
}


double
treelike_seq (world_fmt * world, long locus)
{
  contribarr tterm;
  double sum, sum2, sumc, sumterm, lterm;
  long i, j, k, lai;
  double termtest;
  node *p;
  sitelike x1;
  option_fmt *opt;
  seqmodel_fmt *seq;
  opt = world->options;
  seq = world->data->seq;
  p = crawlback (world->root->next);
  sum = 0.0;

  /*  y = p->v; */
/*   lz = -y; */
  for (i = 0; i < seq->endsite; i++)
    {
      termtest = 0.0;
      k = seq->category[seq->alias[i] - 1] - 1;
      for (j = 0; j < opt->rcategs; j++)
	{
	  memcpy (x1, p->x.s[i][j], sizeof (sitelike));
	  tterm[j] = seq->freqa * x1[0] + seq->freqc * x1[1] +
	    seq->freqg * x1[2] + seq->freqt * x1[3];
	  termtest += tterm[j];
	}
      if (termtest == 0.0)
	{
	  error ("Encountered tree incompatible with data\n");
	}
      sumterm = 0.0;
      for (j = 0; j < opt->rcategs; j++)
	sumterm += opt->probcat[j] * tterm[j];
      lterm = log (sumterm);
      for (j = 0; j < opt->rcategs; j++)
	seq->clai[j] = tterm[j] / sumterm;
      memcpy (seq->contribution[i], seq->clai, sizeof (contribarr));
      sum += seq->aliasweight[i] * lterm;
    }
  for (j = 0; j < opt->rcategs; j++)
    seq->like[j] = 1.0;
  for (i = 0; i < seq->sites[locus]; i++)
    {
      sumc = 0.0;
      for (k = 0; k < opt->rcategs; k++)
	sumc += opt->probcat[k] * seq->like[k];
      sumc *= opt->lambda;
      if ((seq->ally[i] > 0) && (seq->location[seq->ally[i] - 1] > 0))
	{
	  lai = seq->location[seq->ally[i] - 1];
	  memcpy (seq->clai, seq->contribution[lai - 1], sizeof (contribarr));
	  for (j = 0; j < opt->rcategs; j++)
	    seq->nulike[j] = ((1.0 - opt->lambda) * seq->like[j] + sumc) * seq->clai[j];
	}
      else
	{
	  for (j = 0; j < opt->rcategs; j++)
	    seq->nulike[j] = ((1.0 - opt->lambda) * seq->like[j] + sumc);
	}
      memcpy (seq->like, seq->nulike, sizeof (contribarr));
    }
  sum2 = 0.0;
  for (i = 0; i < opt->rcategs; i++)
    sum2 += opt->probcat[i] * seq->like[i];
  sum += log (sum2);
  return sum;
}				/* evaluate */
/* -----------------------------------------------------     
   sighandler.c                         
   handels to following signals:                
   SIGIOT         Input  Output problems             
   SIGIRAP        Over  Underflow, 0 divide          
   SIGFPE         Floating point exceptions         
   SIGBUS         Bus error                 
   SIGSEGV        Segmentation fault                
   SIGXCPU        CPU time limit exceeded           
   SIGXFSZ        File size limit exceeded          
   SIGILL         Illegal instruction                       
   SIGUSR1        User signal 1                             
   if most of those exception are encountered the system     
   tries to exit gracefully, but with some it dies          
   anyway, but tries to say why in a way which is      
   for humans better understandable  (I hope)                       
   -----------------------------------------------------     
   part of the lamarc package                   

   P. Beerli                            
   ----------------------------------------------------- */
#include <stdio.h>
#ifndef __MWERKS__
#include <sys/types.h>
#endif
#ifdef __NEXT__
#include <libc.h>
#endif
#include <signal.h>
#include <stdarg.h>
#include "migration.h"
#include "sighandler.h"
#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif
#ifdef WATCOM
#define __MWERKS__
#endif
#undef debug
void 
signalhandling (long switcher)
{
  if (switcher == ON)
    {
#ifndef __MWERKS__
      signal (SIGIOT, signalhandler);
      signal (SIGTRAP, signalhandler);
      signal (SIGBUS, signalhandler);
      signal (SIGUSR1, signalhandler);
#ifndef SYSTEM_V
      signal (SIGXCPU, signalhandler);
      signal (SIGXFSZ, signalhandler);
#endif
#endif
      signal (SIGFPE, signalhandler);
      signal (SIGSEGV, signalhandler);
      signal (SIGILL, signalhandler);
    }
  else
    {
#ifndef __MWERKS__
      signal (SIGIOT, SIG_DFL);
      signal (SIGTRAP, SIG_DFL);
      signal (SIGBUS, SIG_DFL);
      signal (SIGUSR1, SIG_DFL);
#ifndef SYSTEM_V
      signal (SIGXCPU, SIG_DFL);
      signal (SIGXFSZ, SIG_DFL);
#endif
#endif
      signal (SIGFPE, SIG_DFL);
      signal (SIGSEGV, SIG_DFL);
      signal (SIGILL, SIG_DFL);
    }
}

void 
signalhandler (int sig)
{
  fputc ('\040', stderr);
  switch (sig)
    {
#ifndef __MWERKS__
    case SIGIOT:
      fprintf (stderr, "\nInput/Output error!\n");
      fprintf (stderr, "Most likely a fatal error in the infile or parmfile\n\n");
      exit (sig);
      break;
    case SIGTRAP:
      fprintf (stderr, "Trace trap\n");
      fprintf (stderr, "Please report this bug!\n   (Peter Beerli, beerli@genetics.washington.edu)\n");
      fprintf (stderr, "there was an an overflow/underflow/0 divide problem\n");
      fprintf (stderr, "or some other problems.\n");
      fprintf (stderr, "But check the correctedness of your data first.\n");
      exit (sig);
      break;
    case SIGBUS:
      fprintf (stderr, "Bus error\n");
      fprintf (stderr, "Please report this bug!\n(Peter Beerli, beerli@genetics.washington.edu)\n");
      fprintf (stderr, "there was a bus error, this results in an non recoverable crash\n");
      fprintf (stderr, "But check the correctedness of your data first.\n");
      exit (sig);
      break;
    case SIGXCPU:
      fprintf (stderr, "This program has a time limit?\n");
      fprintf (stderr, "We didn' program a time limit, so it is your or,\n");
      fprintf (stderr, "your system administrator's problem, to correct this.\n");
      exit (sig);
      break;
    case SIGXFSZ:
      fprintf (stderr, "This program has a file size limit?\n");
      fprintf (stderr, "We didn't program a file size limit, so it is your or,\n");
      fprintf (stderr, "your system administrator's problem, to correct this.\n");
      exit (sig);
      break;
    case SIGUSR1:
      fprintf (stderr, "\nUser signal received and ignored\n");
      fprintf (stderr, "\n\n");
      break;

#endif
    case SIGFPE:
      fprintf (stderr, "Floating point exception\n");
      fprintf (stderr, "Please report this bug!\n   (Peter Beerli, beerli@genetics.washington.edu)\n");
      fprintf (stderr, "There was an integer/floating point problem\n");
      fprintf (stderr, "Often this is a division by zero. If you dataset is moderately sized\n");
      fprintf (stderr, "this is most likely an error in you data (infile).\n");
      exit (sig);
      break;
    case SIGSEGV:
      fprintf (stderr, "Segmentation fault\n");
      fprintf (stderr, "Please report this bug!\n(Peter Beerli, beerli@genetics.washington.edu)\n");
      fprintf (stderr, "there was a segmentation fault, this results in an non recoverable crash\n");
      fprintf (stderr, "But check the correctedness of your data first.\n");
      exit (sig);
      break;
    case SIGILL:
      fprintf (stderr, "\nIllegal instruction!\n");
      fprintf (stderr, "this is maybe a programming error\n(Peter Beerli, beerli@genetics.washington.edu),\n but check the intput file\n");
      exit (sig);
      break;
    default:
      break;
    }
#ifndef __MWERKS__
  signal (SIGIOT, signalhandler);
  signal (SIGTRAP, signalhandler);
  signal (SIGUSR1, signalhandler);
  signal (SIGBUS, signalhandler);
#ifndef SYSTEM_V
  signal (SIGXCPU, signalhandler);
  signal (SIGXFSZ, signalhandler);
#endif
#endif
  signal (SIGFPE, signalhandler);
  signal (SIGSEGV, signalhandler);
  signal (SIGILL, signalhandler);

}

void 
sig_error (char string[], char filename[], long line)
{
  fprintf (stdout, "\nIn file %s on line %li\n%s\n", filename, line, string);
  exit (-11);
}


#ifdef NEXTAPP
void 
malloc_error_found (int value)
{
  switch (value)
    {
    case 0:
      fprintf (stderr, "vm_allocate failed\n");
      break;
    case 1:
      fprintf (stderr, "vm_deallocate failed\n");
      break;
    case 2:
      fprintf (stderr, "vm_copy failed\n");
      break;
    case 3:
      fprintf (stderr, "I tried to reallocate or free space which was already freed\n");
      break;
    case 4:
      fprintf (stderr, "Internal memory verification in heap failed\n");
      break;
    case 5:
      fprintf (stderr, "I tried to reallocate or free space which was never allocated\n");
      break;
    default:
      fprintf (stderr, "huh, what malloc error was this?\n");
      break;
    }
  errno = ENOMEM;
}
#endif


/* save memory routines, this routines are used
   with defines in migration.h, DO NOT move this part somewhere else */
#ifdef LAMARC_MALLOC
#undef malloc
#undef realloc
#undef calloc
#undef free

void *
LAMARC_malloc (const long size, const char file[], const long line)
{
  void *x;
  x = malloc (size);
  if (x == NULL)
    {
      fprintf (stderr, "ERROR in memory allocation (malloc(%li)) in file %s on line %li\n",
	       size, file, line);
      exit (-1);
    }
  return x;
}


void *
LAMARC_calloc (const long repeats, const long size, const char file[], const long line)
{
  void *x;
  x = calloc (repeats, size);
  if (x == NULL)
    {
      fprintf (stderr, "ERROR in memory allocation (calloc(%li,%li)) in file %s on line %li\n",
	       repeats, size, file, line);
      exit (-1);
    }
  return x;
}

void *
LAMARC_realloc (void *ptr, const long size, const char file[], const long line)
{
  void *x;
  x = realloc (ptr, size);
  if (x == NULL)
    {
      fprintf (stderr, "ERROR in memory allocation (realloc(ptr,%li)) in file %s on lin %li\n",
	       size, file, line);
      exit (-1);
    }
  return x;
}


#endif /*LAMARC_MALLOC */




/* ----------------------------------------------------- */
/* sort.c                       */
/* comparison routeins for various qsorts and bsearch's */
/* ----------------------------------------------------- */
/* part of the lamarc package               */
/*                              */
/* P. Beerli                        */
/* ----------------------------------------------------- */

/* include files */
#include "migration.h"
#include "sort.h"
/* private functions */

/* public functions */
int 
charcmp (const void *v1, const void *v2)
{
  if (*(char *) v1 < *(char *) v2)
    {
      return -1;
    }
  else
    {
      if (*(char *) v1 > *(char *) v2)
	{
	  return 1;
	}
      else
	return 0;
    }
}

int 
stringcmp (const void *v1, const void *v2)
{
  if (strcmp ((char *) v1, (char *) v2) < 0)
    {
      return -1;
    }
  else
    {
      if (strcmp ((char *) v1, (char *) v2) > 1)
	return 1;
      else
	return 0;
    }
}

int 
numcmp (const void *v1, const void *v2)
{
  if (*(double *) v1 < *(double *) v2)
    {
      return -1;
    }
  else
    {
      if (*(double *) v1 > *(double *) v2)
	{
	  return 1;
	}
      else
	return 0;
    }
}
int 
longcmp (const void *v1, const void *v2)
{
  if (*(long *) v1 < *(long *) v2)
    {
      return -1;
    }
  else
    {
      if (*(long *) v1 > *(long *) v2)
	{
	  return 1;
	}
      else
	return 0;
    }
}

int 
intcmp (const void *v1, const void *v2)
{
  if (*(int *) v1 < *(int *) v2)
    {
      return -1;
    }
  else
    {
      if (*(int *) v1 > *(int *) v2)
	{
	  return 1;
	}
      else
	return 0;
    }
}


int 
agecmp (const void *x, const void *y)
{
  if (((vtlist *) x)->age < ((vtlist *) y)->age)
    {
      return -1;
    }
  else
    {
      if (((vtlist *) x)->age == ((vtlist *) y)->age)
	{
	  return 0;
	}
      else
	return 1;
    }
}

int 
delcmp (const void *x, const void *y)
{
  if ((*((node **) x))->id < (*((node **) y))->id)
    {
      return -1;
    }
  else
    {
      if ((*((node **) x))->id == (*((node **) y))->id)
	{
	  return 0;
	}
      else
	return 1;
    }
}

int 
migr_time_cmp (const void *x, const void *y)
{
  if (((migr_table_fmt *) x)->time < ((migr_table_fmt *) y)->time)
    {
      return -1;
    }
  else
    {
      if (((migr_table_fmt *) x)->time == ((migr_table_fmt *) y)->time)
	{
	  return 0;
	}
      else
	return 1;
    }
}


int 
searchagecmp (const void *x, const void *y)
{
  double xx = (double) *((double *) x);
  double age = (double) ((vtlist *) y)->age;

  if (xx < age)
    {
      return -1;
    }
  else
    {
      if (xx == age)
	{
	  return 0;
	}
      else
	return 1;
    }

}
/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 H E L P E R     R O U T I N E S 
 
 some math stuff and 
 string and file manipulation routines
 

 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: tools.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/
#include "migration.h"
#include "random.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

/* prototypes ------------------------------------------- */
double lengthof (node * p);
node *crawlback (const node * theNode);
/*node *crawl(node * theNode); */
node *showtop (node * theNode);
void adjust_time (node * theNode, double tyme);
void insert_migr_node (world_fmt * world, node * up, node * down,
		     migr_table_fmt * migr_table, long *migr_table_counter);
void children (node * mother, node ** brother, node ** sister);
/* math tools */
double incompletegamma (double x, double alpha);
double polygamma (long n, double z);
void invert_matrix (double **a, long nsize);
boolean nrcheck (double **m, double **tm, double *v, long nrows, double *r1, double *r2, boolean do_newton);
double rannor (double mean, double sd);
char lowercase (char c);
char uppercase (char c);
double sum (double *vector, long n);
/*filemanipulation */
void init_files (world_fmt * world, data_fmt * data, option_fmt * options);
void exit_files (world_fmt * world, data_fmt * data, option_fmt * options);
void openfile (FILE ** fp, char *filename, char *mode, const char *appl, char *perm);
/* string manipulation */
void translate (char *text, char from, char to);
/* time reporting */
void get_time (char *nowstr, char ts[]);
/*printing aid */
void print_llike (double llike, char *strllike);

/* private functions */
double alnorm (double x, int up);
void lu_decomp (double **m, long *indeks, long nrows);
void lu_substitution (double **m, long *indeks, double *v, long nrows);
double d1mach (long i);
long i1mach (long i);
int dpsifn (double *x, long *n, long kode, long m, double *ans, long *nz, long *ierr);



/*FILEMANIPULATION======================================================= */
void 
init_files (world_fmt * world, data_fmt * data, option_fmt * options)
{
  openfile (&data->infile, options->infilename, "r+", appl, NULL);
  openfile (&world->outfile, options->outfilename, "w+", appl, NULL);
  if (options->usertree)
    openfile (&data->utreefile, options->utreefilename, "r+", appl, NULL);
  if (options->weights)
    openfile (&data->weightfile, options->weightfilename, "r+", appl, NULL);
  if (options->categs > 1)
    openfile (&data->catfile, options->catfilename, "r+", appl, NULL);
  if (options->treeprint > 0)
    openfile (&world->treefile, options->treefilename, "w+", appl, NULL);
  if (options->plot)
    {
      switch (options->plotmethod)
	{
	case 0:
	  openfile (&world->mathfile, options->mathfilename, "w+", appl, NULL);
	  break;
	default:		/*e.g. 0 this create just the plots in outfile */
	  break;
	}
    }
}

void 
exit_files (world_fmt * world, data_fmt * data, option_fmt * options)
{
  FClose (data->infile);
  FClose (world->outfile);

  if (options->weights)
    FClose (data->weightfile);
  if (options->categs > 1)
    FClose (data->catfile);
  if (options->treeprint)
    FClose (world->treefile);
  if (options->plot && options->plotmethod == PLOTALL)
    FClose (world->mathfile);
}

/* string manipulation ================================== */
/* Converts any character from to character to in string text */
void 
translate (char *text, char from, char to)
{
  int i, j, gap = 0;
  while (text[gap] == from)
    gap++;
  for (i = gap, j = 0; text[i] != '\0'; i++)
    {
      if (text[i] != from)
	{
	  text[j++] = text[i];
	}
      else
	{
	  if (text[i - 1] != from)
	    {
	      text[j++] = to;
	    }
	}
    }
  text[j] = '\0';
}


/*===============================================
 timer utility
 
 ts = "%c" -> time + full date (see man strftime)
      = "%H:%M:%S" -> time hours:minutes:seconds */
void 
get_time (char *nowstr, char ts[])
{
#ifdef NOTIME_FUNC
  switch(strlen(ts)){
  case 2: strcpy(nowstr," ");
    break;
  case 3: strcpy(nowstr,"  ");
    break;
  case 8: strcpy(nowstr,"        ");
    break;
  default:
    strcpy(nowstr," ");
    break;
  }
#else
  time_t nowbin;
  struct tm *nowstruct;
  if (time (&nowbin) != (time_t) - 1)
    {
      nowstruct = localtime (&nowbin);
      strftime (nowstr, LINESIZE, ts, nowstruct);
    }
#endif
}
/*===============================================
 printer utility
 */
void 
print_llike (double llike, char *strllike)
{
  if (fabs (llike) > 10e20)
    {
      sprintf (strllike, "%cInfinity ", llike < 0 ? '-' : ' ');
    }
  else
    sprintf (strllike, "%-10.5f", llike);
}

void 
openfile (FILE ** fp, char *filename, char *mode, const char *application, char *perm)
{
  int trials=0;
  FILE *of;
  char file[LINESIZE];
  strcpy (file, filename);
  while (trials++ < 10)
    {
      of = fopen (file, mode);
      if (of)
	break;
      else
	{
	  switch (*mode)
	    {
	    case 'r':
	      printf ("%s:  can't read %s\n", application, file);
	      file[0] = '\0';
	      while (file[0] == '\0')
		{
		  printf ("Please enter a new filename for reading>");
		  fgets (file, LINESIZE, stdin);
		}
	      break;
	    case 'w':
	      printf ("%s: can't write %s\n", application, file);
	      file[0] = '\0';
	      while (file[0] == '\0')
		{
		  printf ("Please enter a new filename for writing>");
		  fgets (file, LINESIZE, stdin);
		}
	      break;
	    }
	}
      file[strlen(file)-1] = '\0';
    }
  if(trials>=10)
    {
      printf("You cannot find your file either, so I stop\n\n");
      exit(0);
    }
  *fp = of;
  if (perm != NULL)
    strcpy (perm, file);
  strcpy (filename, file);

}



/*=======================================================*/




/*--------------------------------
creates the length value in a node
*/
double 
lengthof (node * p)
{
  if (p->type == 'm')
    fprintf (stderr, "a migration node was feed into lengthof");
  return fabs (p->tyme - crawlback (p)->tyme);
}				/* length */


/*------------------------------------------------
Find the next non-migration node starting
with the theNode, returns to backnode which is not 
a migration, does NOT return always a top-node!
*/
node *
crawlback (const node * theNode)
{
  node *tmp = theNode->back;

  while (tmp->type == 'm')
    {
      tmp = tmp->next->back;
    }
  return tmp;
}

/*--------------------------------------------
returns the last migration node in a branch or 
the node if there is no migration node

node *crawl(node * theNode)
{
   node *otmp, *tmp = theNode->back;

   otmp = theNode;
   if (tmp == NULL)
	  return otmp;
   while (tmp->type == 'm') {
	  otmp = tmp->next;
	  tmp = tmp->next->back;
	  if (tmp == NULL)
		 return otmp;
   }
   return otmp;
}
*/


node *
showtop (node * theNode)
{
  if (theNode == NULL)
    return NULL;
  else
    {
      if (theNode->top)
	{
	  return theNode;
	}
      else
	{
	  if (theNode->next->top)
	    {
	      return theNode->next;
	    }
	  else
	    {
	      return theNode->next->next;
	    }
	}
    }

}

/* adjust the time in a node to time */
void 
adjust_time (node * theNode, double tyme)
{
  switch (theNode->type)
    {
    case 'm':
      theNode->tyme = theNode->next->tyme = tyme;
      break;
    case 'i':
      theNode->tyme = theNode->next->tyme = theNode->next->next->tyme = tyme;
      break;
    case 'r':
    case 't':
      break;
    default:
      fprintf (stderr, "this node cannot exist");
      exit (EXIT_FAILURE);
      break;

    }
}

void 
insert_migr_node (world_fmt * world, node * up, node * down,
		  migr_table_fmt * migr_table, long *migr_table_counter)
{
  long i, panic;
  node *tmp, *tmp2, *oldNode, *oldNode2, *theNode;
  if (!up->top)
    fprintf (stderr, "up has to be a top-node");
  theNode = showtop (up)->back;
  if (*migr_table_counter > 0 && up->tyme > migr_table[0].time)
    fprintf (stderr, "insert_migr_node: the first migration node has a wrong time for up");
  if (migr_table[(*migr_table_counter) - 1].from != down->actualpop)
    {
      fprintf (stderr, "this should never happen -> wrong choice of nodes\n");
      (*migr_table_counter)--;
    }
  if (((*migr_table_counter) > 0) && (migr_table[(*migr_table_counter) - 1].from != down->actualpop))
    fprintf (stderr, "problem catched in inser_migr_table");
  for (i = 0; i < (*migr_table_counter); i++)
    {
      tmp = (node *) calloc (1, sizeof (node));
      tmp2 = (node *) calloc (1, sizeof (node));
      oldNode = up;
      theNode = up->back;
      panic = 0;
      while (theNode->tyme < migr_table[i].time && panic++ < 10000)
	{
	  if (theNode->tip && theNode->type != 'r')
	    {
	      oldNode = theNode;
	      theNode = theNode->back;
	    }
	  else
	    {
	      oldNode = theNode->next;
	      theNode = theNode->next->back;
	    }
	}
      tmp->back = oldNode;
      oldNode->back = tmp;
      tmp->number = -999;
      tmp->nayme = NULL;
      tmp->tip = 0;
      tmp->top = 0;
      tmp->dirty = TRUE;
      tmp->id = world->unique_id++;
      tmp->tyme = migr_table[i].time;
      tmp->type = 'm';
      tmp->actualpop = migr_table[i].to;
      tmp->pop = migr_table[i].from;
      tmp2->tyme = migr_table[i].time;
      tmp2->type = 'm';
      tmp2->id = world->unique_id++;
      tmp2->actualpop = migr_table[i].to;
      tmp2->pop = migr_table[i].from;
      tmp->next = tmp2;
      tmp2->next = tmp;
      tmp2->top = 1;

      oldNode2 = down;
      theNode = down->back;
      while (theNode->tyme > migr_table[i].time)
	{
	  oldNode2 = theNode->next;
	  theNode = theNode->next->back;
	}
      tmp2->back = oldNode2;
      oldNode2->back = tmp2;
    }
}


void 
children (node * mother, node ** brother, node ** sister)
{
  node *m;

  m = showtop (mother);

  if (m->type == 't')
    {
      error ("this is a tip, so there are no more child nodes\n");
    }
  else
    {
      (*brother) = crawlback (m->next);
      (*sister) = crawlback (m->next->next);
    }
}

#ifndef HAVE_LGAMMA
double 
lgamma (double z)
{
  const double a[9] =
  {.9999999999995183, 676.5203681218835,
   -1259.139216722289, 771.3234287757674, -176.6150291498386,
   12.50734324009056, -.1385710331296526, 9.934937113930748e-6,
   1.659470187408462e-7};
  const double lnsqrt2pi = .9189385332046727;
  double result;

  long j;
  double tmp;


  /*       Uses Lanczos-type approximation to ln(gamma) for z > 0. */
  /*       Reference: */
  /*            Lanczos, C. 'A precision approximation of the gamma */
  /*                    function', J. SIAM Numer. Anal., B, 1, 86-96, 1964. */
  /*       Accuracy: About 14 significant digits except for small regions */
  /*                 in the vicinity of 1 and 2. */
  /*       Programmer: Alan Miller */
  /*                   CSIRO Division of Mathematics & Statistics */
  /*       Latest revision - 17 April 1988 */
  /* translated and modified into C by Peter Beerli 1997 */


  if (z <= 0.)
    {
      return DBL_MAX;		/*this will kill the receiving calculation */
    }
  result = 0.;
  tmp = z + 7.;
  for (j = 9; j >= 2; --j)
    {
      result += a[j - 1] / tmp;
      tmp += -1.;
    }
  result += a[0];
  result = log (result) + lnsqrt2pi - (z + 6.5) + (z - .5) *
    log (z + 6.5);
  return result;
}				/* lgamma */
#endif

/* ALGORITHM AS239  APPL. STATIST. (1988) VOL. 37, NO. 3 
   Computation of the Incomplete Gamma Integral 
   Auxiliary functions required: lgamma() = logarithm of the gamma 
   function, and alnorm() = algorithm AS66 */
double 
incompletegamma (double x, double alpha)
{
  double gama, d_1, d_2, d_3;
  static double a, b, c, an, rn;
  static double pn1, pn2, pn3, pn4, pn5, pn6, arg;

  gama = 0.;
  /*  Check that we have valid values for X and P */
  if (alpha <= 0. || x < 0.)
    {
      exit (-2);
    }
  if (x == 0.)
    {
      return gama;
    }

  /*  Use a normal approximation if P > PLIMIT */
  if (alpha > 1e3)
    {
      pn1 = sqrt (alpha) * 3. * (pow (x / alpha, (1. / 3.)) + 1. / (alpha * 9.) - 1.);
      gama = alnorm (pn1, FALSE);
      return gama;
    }

  /*  If X is extremely large compared to P then set GAMMAD = 1 */
  if (x > 1e8)
    {
      gama = 1.;
      return gama;
    }

  if (x <= 1. || x < alpha)
    {
      /*  Use Pearson's series expansion. */
      /*  (Note that P is not large enough to force overflow in lgamma()). */
      arg = alpha * log (x) - x - lgamma (alpha + 1.);
      c = 1.;
      gama = 1.;
      a = alpha;
      while (c > 1e-14)
	{
	  a += 1.;
	  c = c * x / a;
	  gama += c;
	}
      arg += log (gama);
      gama = 0.;
      if (arg >= -88.)
	{
	  gama = exp (arg);
	}

    }
  else
    {
      /*  Use a continued fraction expansion */
      arg = alpha * log (x) - x - lgamma (alpha);
      a = 1. - alpha;
      b = a + x + 1.;
      c = 0.;
      pn1 = 1.;
      pn2 = x;
      pn3 = x + 1.;
      pn4 = x * b;
      gama = pn3 / pn4;
      for (;;)
	{
	  a += 1.;
	  b += 2.;
	  c += 1.;
	  an = a * c;
	  pn5 = b * pn3 - an * pn1;
	  pn6 = b * pn4 - an * pn2;
	  if (fabs (pn6) > 0.)
	    {
	      rn = pn5 / pn6;
	      /* Computing MIN */
	      d_2 = 1e-14, d_3 = rn * 1e-14;
	      if ((d_1 = gama - rn, fabs (d_1)) <= MIN (d_2, d_3))
		{
		  arg += log (gama);
		  gama = 1.;
		  if (arg >= -88.)
		    {
		      gama = 1. - exp (arg);
		    }
		  return gama;
		}
	      gama = rn;
	    }
	  pn1 = pn3;
	  pn2 = pn4;
	  pn3 = pn5;
	  pn4 = pn6;
	  if (fabs (pn5) >= 1e37)
	    {
	      /*  Re-scale terms in continued fraction if terms are large */
	      pn1 /= 1e37;
	      pn2 /= 1e37;
	      pn3 /= 1e37;
	      pn4 /= 1e37;
	    }
	}
    }
  /*fake */
  return gama;
}				/* incompletegamma() */


/* calculation is replaced by the correct function in 
   polygamma.c (which is a translation of a fortran program by amos

   driver for the polygamma calculation */
double 
polygamma (long n, double z)
{
  double ans;
  long nz, ierr;
  dpsifn (&z, &n, 1, 1, &ans, &nz, &ierr);
  if (n == 0)
    return -ans;
  else
    return ans;
}

/*-------------------------------------------------------*/
/* nrcheck subroutine (used in damped newton raphson proc */
/* syntax: nrcheck(matrix,inversematrix,ncols=nrows,returnval1,returnval2) */
/* mai 95 PB                                             */
boolean 
nrcheck (double **m, double **tm, double *v, long nrows, double *r1, double *r2, boolean do_newton)
{
  long i, j, k;
  double *tmp, *tmp2, tmp3 = 0.0, tmp4 = 0.0;
  tmp = (double *) calloc (1, sizeof (double) * nrows);
  tmp2 = (double *) calloc (1, sizeof (double) * nrows);
  /*first evaluate r1 */
  (*r1) = (*r2) = 0.0;
  for (i = 0; i < nrows; i++)
    {
      (*r1) += v[i] * v[i];
    }
  /*                                       T    */
  for (j = 0; j < nrows; j++)
    {				/* g . G */
      for (k = 0; k < nrows; k++)
	{
	  tmp[j] += v[k] * m[j][k];
	  tmp2[j] += v[k] * tm[j][k];
	}
    }
  /*                                       T        */
  for (i = 0; i < nrows; i++)
    {				/* g . G . g */
      (*r2) += tmp[i] * v[i];
      tmp3 += tmp2[i] * v[i];
    }
  tmp4 = log (fabs ((*r1)));
  tmp4 = tmp4 + tmp4 - log (fabs ((*r2)));
  tmp4 = ((*r2) < 0 ? -1 : 1) * exp (tmp4);
  free (tmp);
/*    fprintf(stderr,"g(G~1)g=%f  > Max(gg^2 / gGg = %f , 0) %s\n",tmp3,tmp4, (tmp3 > MAX(tmp4,0) ? "YES" : "NO")); */
  if (do_newton && (tmp3 > (tmp4 > 0 ? tmp4 : 0)))
    {
      memcpy (v, tmp2, sizeof (double) * nrows);
      free (tmp2);
      return TRUE;
    }
  free (tmp2);
  return FALSE;
}


/*-------------------------------------------------------*/
/* Matrix inversion subroutine                           */
/* The passed matrix will be replaced by its inverse!!!!! */
/* Gauss-Jordan reduction -- invert matrix a in place,   */
/* overwriting previous contents of a.  On exit, matrix a */
/* contains the inverse.                                 */
void 
invert_matrix (double **a, long nsize)
{
  long i, j;
  long *indeks;
  double *column, **result;
  indeks = (long *) malloc (sizeof (long) * nsize);
  column = (double *) malloc (sizeof (double) * nsize);
  result = (double **) malloc (sizeof (double *) * nsize);
  for (i = 0; i < nsize; i++)
    {
      result[i] = (double *) malloc (sizeof (double) * nsize);
    }
  lu_decomp (a, indeks, nsize);
  for (j = 0; j < nsize; j++)
    {
      memset (column, 0, sizeof (double) * nsize);
      column[j] = 1.0;
      lu_substitution (a, indeks, column, nsize);
      for (i = 0; i < nsize; i++)
	result[i][j] = column[i];
    }
  for (i = 0; i < nsize; i++)
    {
      memcpy (a[i], result[i], sizeof (double) * nsize);
      free (result[i]);
    }
  free (result);
  free (column);
  free (indeks);
}

/*=======================================================*/

/*-------------------------------------------------------*/
/* LU decomposition                                      */
/* after Dahlquist et al. 1974 and Press et al. 1988     */
/* the method's uses Crout's procedure and the pivoting  */
/* described in Press et al.                             */
/* Syntax: lu_decomp(matrix, indeks, nrows)               */
/* matrix will be destroyed and filled with the two      */
/* triangular matrices, indeks is the index vector for the */
/* pivoting and the row change in case of 0 pivot values */
/* nrows is the number of rows and columns in matrix     */
/* april 95 PB                                           */
void 
lu_decomp (double **m, long *indeks, long nrows)
{
  long i, j, k, p, kmax = -1;
  double *max_row_vals, big, sum, pivot, bigt;
  max_row_vals = (double *) calloc (1, sizeof (double) * nrows);
  for (i = 0; i < nrows; i++)
    {
      big = 0.0;
      for (j = 0; j < nrows; j++)
	{
	  if ((bigt = fabs (m[i][j])) > big)
	    big = bigt;
	}
      max_row_vals[i] = 1.0 / big;
      if (big == 0.0)
	{
	  error ("Singular matrix detected in lu_decomp\n");
	}
    }
  for (i = 0; i < nrows; i++)
    {
      for (k = 0; k < i; k++)
	{			/* upper half of matrix */
	  sum = m[k][i];
	  for (p = 0; p < k; p++)
	    sum -= m[k][p] * m[p][i];
	  m[k][i] = sum;
	}
      big = 0.0;
      for (k = i; k < nrows; k++)
	{			/* lower half of matrix */
	  sum = m[k][i];
	  for (p = 0; p < i; p++)
	    sum -= m[k][p] * m[p][i];
	  m[k][i] = sum;
	  pivot = fabs (sum) /**max_row_vals[k]*/ ;
	  /*  fprintf(stdout,"i=%li,pivot=%f,big=%f\n",i,pivot,big); */
	  if (pivot >= big)
	    {
	      big = pivot;
	      kmax = k;
	    }
	}
      if (i != kmax)
	{
	  for (p = 0; p < nrows; p++)
	    {
	      pivot = m[kmax][p];
	      m[kmax][p] = m[i][p];
	      m[i][p] = pivot;
	    }
	  max_row_vals[kmax] = max_row_vals[i];
	}
      indeks[i] = kmax;
      if (m[i][i] == 0.0)
	m[i][i] = SMALL_VALUE;
      if (i != nrows - 1)
	{
	  pivot = 1. / m[i][i];
	  for (k = i + 1; k < nrows; k++)
	    m[k][i] *= pivot;
	}
    }
  free (max_row_vals);
}				/* end of lu_decomp */

/*-------------------------------------------------------*/
/* LU substitution                                       */
/* after Dahlquist et al. 1974 and Press et al. 1988     */
/* needs first the evaluation LU decomposition           */
/* Syntax: lu_substition(matrix, indeks, vector, nrows)   */
/* matrix = LU decomposed matrix, indeks = order of matrix */
/* vector = value vevtor, nrows = number of rows/columns */
/* april 95 PB                                           */
void 
lu_substitution (double **m, long *indeks, double *v, long nrows)
{
  long i, j;
  double sum;
  for (i = 0; i < nrows; i++)
    {
      sum = v[indeks[i]];
      v[indeks[i]] = v[i];
      for (j = 0; j < i; j++)
	sum -= m[i][j] * v[j];
      v[i] = sum;
    }
  for (i = nrows - 1; i >= 0; i--)
    {
      sum = v[i];
      for (j = i + 1; j < nrows; j++)
	sum -= m[i][j] * v[j];
      v[i] = sum / m[i][i];
    }
}


/* Algorithm AS66 Applied Statistics (1973) vol22 no.3
   Evaluates the tail area of the standardised normal curve
   from x to infinity if upper is .true. or
   from minus infinity to x if upper is .false. */
double 
alnorm (double x, int up)
{
  /* Initialized data */
  /* *** machine dependent constants ????????????? */
  static double zero = 0.;
  static double a1 = 5.75885480458;
  static double a2 = 2.62433121679;
  static double a3 = 5.92885724438;
  static double b1 = -29.8213557807;
  static double b2 = 48.6959930692;
  static double c1 = -3.8052e-8;
  static double c2 = 3.98064794e-4;
  static double c3 = -.151679116635;
  static double c4 = 4.8385912808;
  static double c5 = .742380924027;
  static double one = 1.;
  static double c6 = 3.99019417011;
  static double d1 = 1.00000615302;
  static double d2 = 1.98615381364;
  static double d3 = 5.29330324926;
  static double d4 = -15.1508972451;
  static double d5 = 30.789933034;
  static double half = .5;
  static double ltone = 7.;
  static double utzero = 18.66;
  static double con = 1.28;
  static double p = .398942280444;
  static double q = .39990348504;
  static double r = .398942280385;

  static double y, result;

  if (x < zero)
    {
      up = !up;
      x = -x;
    }
  if (x <= ltone || (up && x <= utzero))
    {
      y = half * x * x;
      if (x > con)
	{
	  result = r * exp (-y) / (x + c1 + d1 / (x + c2 + d2 / (x + c3 + d3 / (x +
				     c4 + d4 / (x + c5 + d5 / (x + c6))))));
	  return ((!up) ? one - result : result);
	}
      result = half - x * (p - q * y / (y + a1 + b1 / (y + a2 + b2 / (y + a3))));
      return ((!up) ? one - result : result);
    }
  else
    {
      return ((!up) ? 1.0 : 0.);
    }
  /*fake */ return -99;
}				/* alnorm */

/* dpsifn.c -- translated by f2c (version 19950808).
   and hand-patched by Peter Beerli Seattle, 1996
   SUBROUTINE DPSIFN (X, N, KODE, M, ANS, NZ, IERR)

   C***BEGIN PROLOGUE  DPSIFN
   C***PURPOSE  Compute derivatives of the Psi function.
   C***LIBRARY   SLATEC
   C***CATEGORY  C7C
   C***TYPE      DOUBLE PRECISION (PSIFN-S, DPSIFN-D)
   C***KEYWORDS  DERIVATIVES OF THE GAMMA FUNCTION, POLYGAMMA FUNCTION,
   C             PSI FUNCTION
   C***AUTHOR  Amos, D. E., (SNLA)
   C***DESCRIPTION
   C
   C         The following definitions are used in DPSIFN:
   C
   C      Definition 1
   C         PSI(X) = d/dx (ln(GAMMA(X)), the first derivative of
   C                  the log GAMMA function.
   C      Definition 2
   C                     K   K
   C         PSI(K,X) = d /dx (PSI(X)), the K-th derivative of PSI(X).
   C   ___________________________________________________________________
   C      DPSIFN computes a sequence of SCALED derivatives of
   C      the PSI function; i.e. for fixed X and M it computes
   C      the M-member sequence
   C
   C                    ((-1)**(K+1)/GAMMA(K+1))*PSI(K,X)
   C                       for K = N,...,N+M-1
   C
   C      where PSI(K,X) is as defined above.   For KODE=1, DPSIFN returns
   C      the scaled derivatives as described.  KODE=2 is operative only
   C      when K=0 and in that case DPSIFN returns -PSI(X) + LN(X).  That
   C      is, the logarithmic behavior for large X is removed when KODE=2
   C      and K=0.  When sums or differences of PSI functions are computed
   C      the logarithmic terms can be combined analytically and computed
   C      separately to help retain significant digits.
   C
   C         Note that CALL DPSIFN(X,0,1,1,ANS) results in
   C                   ANS = -PSI(X)
   C
   C     Input      X is DOUBLE PRECISION
   C           X      - Argument, X .gt. 0.0D0
   C           N      - First member of the sequence, 0 .le. N .le. 100
   C                    N=0 gives ANS(1) = -PSI(X)       for KODE=1
   C                                       -PSI(X)+LN(X) for KODE=2
   C           KODE   - Selection parameter
   C                    KODE=1 returns scaled derivatives of the PSI
   C                    function.
   C                    KODE=2 returns scaled derivatives of the PSI
   C                    function EXCEPT when N=0. In this case,
   C                    ANS(1) = -PSI(X) + LN(X) is returned.
   C           M      - Number of members of the sequence, M.ge.1
   C
   C    Output     ANS is DOUBLE PRECISION
   C           ANS    - A vector of length at least M whose first M
   C                    components contain the sequence of derivatives
   C                    scaled according to KODE.
   C           NZ     - Underflow flag
   C                    NZ.eq.0, A normal return
   C                    NZ.ne.0, Underflow, last NZ components of ANS are
   C                             set to zero, ANS(M-K+1)=0.0, K=1,...,NZ
   C           IERR   - Error flag
   C                    IERR=0, A normal return, computation completed
   C                    IERR=1, Input error,     no computation
   C                    IERR=2, Overflow,        X too small or N+M-1 too
   C                            large or both
   C                    IERR=3, Error,           N too large. Dimensioned
   C                            array TRMR(NMAX) is not large enough for N
   C
   C         The nominal computational accuracy is the maximum of unit
   C         roundoff (=D1MACH(4)) and 1.0D-18 since critical constants
   C         are given to only 18 digits.
   C
   C         PSIFN is the single precision version of DPSIFN.
   C
   C *Long Description:
   C
   C         The basic method of evaluation is the asymptotic expansion
   C         for large X.ge.XMIN followed by backward recursion on a two
   C         term recursion relation
   C
   C                  W(X+1) + X**(-N-1) = W(X).
   C
   C         This is supplemented by a series
   C
   C                  SUM( (X+K)**(-N-1) , K=0,1,2,... )
   C
   C         which converges rapidly for large N. Both XMIN and the
   C         number of terms of the series are calculated from the unit
   C         roundoff of the machine environment.
   C
   C***REFERENCES  Handbook of Mathematical Functions, National Bureau
   C                 of Standards Applied Mathematics Series 55, edited
   C                 by M. Abramowitz and I. A. Stegun, equations 6.3.5,
   C                 6.3.18, 6.4.6, 6.4.9 and 6.4.10, pp.258-260, 1964.
   C               D. E. Amos, A portable Fortran subroutine for
   C                 derivatives of the Psi function, Algorithm 610, ACM
   C                 Transactions on Mathematical Software 9, 4 (1983),
   C                 pp. 494-502.
   C***ROUTINES CALLED  D1MACH, I1MACH
   C***REVISION HISTORY  (YYMMDD)
   C   820601  DATE WRITTEN
   C   890531  Changed all specific intrinsics to generic.  (WRB)
   C   890911  Removed unnecessary intrinsics.  (WRB)
   C   891006  Cosmetic changes to prologue.  (WRB)
   C   891006  REVISION DATE from Version 3.2
   C   891214  Prologue converted to Version 4.0 format.  (BAB)
   C   920501  Reformatted the REFERENCES section.  (WRB)
   C***END PROLOGUE  DPSIFN


 */

static long fifteen = 15;
static long sixteen = 16;
static long five = 5;
static long four = 4;
static long fourteen = 14;

double 
d1mach (long i)
{
  switch (i)
    {
    case 1:
      return DBL_MIN;
    case 2:
      return DBL_MAX;
    case 3:
      return DBL_EPSILON / FLT_RADIX;
    case 4:
      return DBL_EPSILON;
    case 5:
      return log10 (FLT_RADIX);
    }
  fprintf (stderr, "invalid argument: d1mach(%ld)\n", i);
  exit (1);
  return 0;			/* for compilers that complain of missing return values */
}

long 
i1mach (long i)
{
  switch (i)
    {
    case 1:
      return 5;			/* standard input */
    case 2:
      return 6;			/* standard output */
    case 3:
      return 7;			/* standard punch */
    case 4:
      return 0;			/* standard error */
    case 5:
      return 32;		/* bits per integer */
    case 6:
      return 1;			/* Fortran 77 value */
    case 7:
      return 2;			/* base for integers */
    case 8:
      return 31;		/* digits of integer base */
    case 9:
      return LONG_MAX;
    case 10:
      return FLT_RADIX;
    case 11:
      return FLT_MANT_DIG;
    case 12:
      return FLT_MIN_EXP;
    case 13:
      return FLT_MAX_EXP;
    case 14:
      return DBL_MANT_DIG;
    case 15:
      return DBL_MIN_EXP;
    case 16:
      return DBL_MAX_EXP;
    }
  fprintf (stderr, "invalid argument: i1mach(%ld)\n", i);
  exit (1);
  return 0;			/* for compilers that complain of missing return values */
}

int 
dpsifn (double *x, long *n, long kode, long m, double *ans, long *nz, long *ierr)
{
  /* Initialized data */

  static long nmax = 100;
  static double b[22] =
  {1., -.5, .166666666666666667,
   -.0333333333333333333, .0238095238095238095, -.0333333333333333333,
   .0757575757575757576, -.253113553113553114, 1.16666666666666667,
   -7.09215686274509804, 54.9711779448621554, -529.124242424242424,
   6192.1231884057971, -86580.2531135531136, 1425517.16666666667,
   -27298231.067816092, 601580873.900642368, -15116315767.0921569,
   429614643061.166667, -13711655205088.3328, 488332318973593.167,
   -19296579341940068.1};

  /* System generated locals */
  long i1, i2;
  double d1, d2;


  /* Local variables */
  static double elim, xinc, xmin, tols, xdmy, yint, trmr[100], rxsq;
  static long i__, j, k;
  static double s, t, slope, xdmln, wdtol;
  static double t1, t2;
  static long fn;
  static double ta;
  static long mm, nn, np;
  static double fx, tk;
  static long mx, nx;
  static double xm, tt, xq, den, arg, fln, r1m4, r1m5, eps, rln, tol, xln,
    trm[22], tss, tst;

  /* Parameter adjustments */
  --ans;

  /* Function Body */
/* ----------------------------------------------------------------------- */
/*             BERNOULLI NUMBERS */
/* ----------------------------------------------------------------------- */

/* ***FIRST EXECUTABLE STATEMENT  DPSIFN */
  *ierr = 0;
  *nz = 0;
  if (*x <= 0.)
    {
      *ierr = 1;
    }
  if (*n < 0)
    {
      *ierr = 1;
    }
  if (kode < 1 || kode > 2)
    {
      *ierr = 1;
    }
  if (m < 1)
    {
      *ierr = 1;
    }
  if (*ierr != 0)
    {
      return 0;
    }
  mm = m;
/* Computing MIN */
  i1 = -fifteen, i2 = sixteen;
  nx = MIN (-i1mach (fifteen), i1mach (sixteen));
  r1m5 = d1mach (five);
  r1m4 = d1mach (four) * .5;
  wdtol = MAX (r1m4, 5e-19);
/* ----------------------------------------------------------------------- */
/*     ELIM = APPROXIMATE EXPONENTIAL OVER AND UNDERFLOW LIMIT */
/* ----------------------------------------------------------------------- */
  elim = (nx * r1m5 - 3.) * 2.302;
  xln = log (*x);
L41:
  nn = *n + mm - 1;
  fn = nn;
  t = (fn + 1) * xln;
/* ----------------------------------------------------------------------- */
/*     OVERFLOW AND UNDERFLOW TEST FOR SMALL AND LARGE X */
/* ----------------------------------------------------------------------- */
  if (fabs (t) > elim)
    {
      goto L290;
    }
  if (*x < wdtol)
    {
      goto L260;
    }
/* ----------------------------------------------------------------------- */
/*     COMPUTE XMIN AND THE NUMBER OF TERMS OF THE SERIES, FLN+1 */
/* ----------------------------------------------------------------------- */
  rln = r1m5 * i1mach (fourteen);
  rln = MIN (rln, 18.06);
  fln = MAX (rln, 3.) - 3.;
  yint = fln * .4 + 3.5;
  slope = fln * (fln * 6.038e-4 + .008677) + .21;
  xm = yint + slope * fn;
  mx = (long) xm + 1;
  xmin = (double) mx;
  if (*n == 0)
    {
      goto L50;
    }
  xm = rln * -2.302 - MIN (0., xln);
  arg = xm / *n;
  arg = MIN (0., arg);
  eps = exp (arg);
  xm = 1. - eps;
  if (fabs (arg) < .001)
    {
      xm = -arg;
    }
  fln = *x * xm / eps;
  xm = xmin - *x;
  if (xm > 7. && fln < 15.)
    {
      goto L200;
    }
L50:
  xdmy = *x;
  xdmln = xln;
  xinc = 0.;
  if (*x >= xmin)
    {
      goto L60;
    }
  nx = (long) (*x);
  xinc = xmin - nx;
  xdmy = *x + xinc;
  xdmln = log (xdmy);
L60:
/* ----------------------------------------------------------------------- */
/*     GENERATE W(N+MM-1,X) BY THE ASYMPTOTIC EXPANSION */
/* ----------------------------------------------------------------------- */
  t = fn * xdmln;
  t1 = xdmln + xdmln;
  t2 = t + xdmln;
/* Computing MAX */
  d1 = fabs (t), d2 = fabs (t1), d1 = MAX (d1, d2), d2 = fabs (t2);
  tk = MAX (d1, d2);
  if (tk > elim)
    {
      goto L380;
    }
  tss = exp (-t);
  tt = .5 / xdmy;
  t1 = tt;
  tst = wdtol * tt;
  if (nn != 0)
    {
      t1 = tt + 1. / fn;
    }
  rxsq = 1. / (xdmy * xdmy);
  ta = rxsq * .5;
  t = (fn + 1) * ta;
  s = t * b[2];
  if (fabs (s) < tst)
    {
      goto L80;
    }
  tk = 2.;
  for (k = 4; k <= 22; ++k)
    {
      t = t * ((tk + fn + 1) / (tk + 1.)) * ((tk + fn) / (tk + 2.)) * rxsq;
      trm[k - 1] = t * b[k - 1];
      if ((d1 = trm[k - 1], fabs (d1)) < tst)
	{
	  goto L80;
	}
      s += trm[k - 1];
      tk += 2.;
/* L70: */
    }
L80:
  s = (s + t1) * tss;
  if (xinc == 0.)
    {
      goto L100;
    }
/* ----------------------------------------------------------------------- */
/*     BACKWARD RECUR FROM XDMY TO X */
/* ----------------------------------------------------------------------- */
  nx = (long) xinc;
  np = nn + 1;
  if (nx > nmax)
    {
      goto L390;
    }
  if (nn == 0)
    {
      goto L160;
    }
  xm = xinc - 1.;
  fx = *x + xm;
/* ----------------------------------------------------------------------- */
/*     THIS LOOP SHOULD NOT BE CHANGED. FX IS ACCURATE WHEN X IS SMALL */
/* ----------------------------------------------------------------------- */
  i1 = nx;
  for (i__ = 1; i__ <= i1; ++i__)
    {
      i2 = -np;
      trmr[i__ - 1] = pow (fx, i2);
      s += trmr[i__ - 1];
      xm += -1.;
      fx = *x + xm;
/* L90: */
    }
L100:
  ans[mm] = s;
  if (fn == 0)
    {
      goto L180;
    }
/* ----------------------------------------------------------------------- */
/*     GENERATE LOWER DERIVATIVES, J.LT.N+MM-1 */
/* ----------------------------------------------------------------------- */
  if (mm == 1)
    {
      return 0;
    }
  i1 = mm;
  for (j = 2; j <= i1; ++j)
    {
      --fn;
      tss *= xdmy;
      t1 = tt;
      if (fn != 0)
	{
	  t1 = tt + 1. / fn;
	}
      t = (fn + 1) * ta;
      s = t * b[2];
      if (fabs (s) < tst)
	{
	  goto L120;
	}
      tk = (double) (fn + 4);
      for (k = 4; k <= 22; ++k)
	{
	  trm[k - 1] = trm[k - 1] * (fn + 1) / tk;
	  if ((d1 = trm[k - 1], fabs (d1)) < tst)
	    {
	      goto L120;
	    }
	  s += trm[k - 1];
	  tk += 2.;
/* L110: */
	}
    L120:
      s = (s + t1) * tss;
      if (xinc == 0.)
	{
	  goto L140;
	}
      if (fn == 0)
	{
	  goto L160;
	}
      xm = xinc - 1.;
      fx = *x + xm;
      i2 = nx;
      for (i__ = 1; i__ <= i2; ++i__)
	{
	  trmr[i__ - 1] *= fx;
	  s += trmr[i__ - 1];
	  xm += -1.;
	  fx = *x + xm;
/* L130: */
	}
    L140:
      mx = mm - j + 1;
      ans[mx] = s;
      if (fn == 0)
	{
	  goto L180;
	}
/* L150: */
    }
  return 0;
/* ----------------------------------------------------------------------- */
/*     RECURSION FOR N = 0 */
/* ----------------------------------------------------------------------- */
L160:
  i1 = nx;
  for (i__ = 1; i__ <= i1; ++i__)
    {
      s += 1. / (*x + nx - i__);
/* L170: */
    }
L180:
  if (kode == 2)
    {
      goto L190;
    }
  ans[1] = s - xdmln;
  return 0;
L190:
  if (xdmy == *x)
    {
      return 0;
    }
  xq = xdmy / *x;
  ans[1] = s - log (xq);
  return 0;
/* ----------------------------------------------------------------------- */
/*     COMPUTE BY SERIES (X+K)**(-(N+1)) , K=0,1,2,... */
/* ----------------------------------------------------------------------- */
L200:
  nn = (long) fln + 1;
  np = *n + 1;
  t1 = (*n + 1) * xln;
  t = exp (-t1);
  s = t;
  den = *x;
  i1 = nn;
  for (i__ = 1; i__ <= i1; ++i__)
    {
      den += 1.;
      i2 = -np;
      trm[i__ - 1] = pow (den, i2);
      s += trm[i__ - 1];
/* L210: */
    }
  ans[1] = s;
  if (*n != 0)
    {
      goto L220;
    }
  if (kode == 2)
    {
      ans[1] = s + xln;
    }
L220:
  if (mm == 1)
    {
      return 0;
    }
/* ----------------------------------------------------------------------- */
/*     GENERATE HIGHER DERIVATIVES, J.GT.N */
/* ----------------------------------------------------------------------- */
  tol = wdtol / 5.;
  i1 = mm;
  for (j = 2; j <= i1; ++j)
    {
      t /= *x;
      s = t;
      tols = t * tol;
      den = *x;
      i2 = nn;
      for (i__ = 1; i__ <= i2; ++i__)
	{
	  den += 1.;
	  trm[i__ - 1] /= den;
	  s += trm[i__ - 1];
	  if (trm[i__ - 1] < tols)
	    {
	      goto L240;
	    }
/* L230: */
	}
    L240:
      ans[j] = s;
/* L250: */
    }
  return 0;
/* ----------------------------------------------------------------------- */
/*     SMALL X.LT.UNIT ROUND OFF */
/* ----------------------------------------------------------------------- */
L260:
  i1 = -(*n) - 1;
  ans[1] = pow (*x, i1);
  if (mm == 1)
    {
      goto L280;
    }
  k = 1;
  i1 = mm;
  for (i__ = 2; i__ <= i1; ++i__)
    {
      ans[k + 1] = ans[k] / *x;
      ++k;
/* L270: */
    }
L280:
  if (*n != 0)
    {
      return 0;
    }
  if (kode == 2)
    {
      ans[1] += xln;
    }
  return 0;
L290:
  if (t > 0.)
    {
      goto L380;
    }
  *nz = 0;
  *ierr = 2;
  return 0;
L380:
  ++(*nz);
  ans[mm] = 0.;
  --mm;
  if (mm == 0)
    {
      return 0;
    }
  goto L41;
L390:
  *nz = 0;
  *ierr = 3;
  return 0;
}				/* dpsifn_ */




double 
rannor (double mean, double sd)
{
  double r1, r2;
  r1 = RANDUM ();
  r2 = RANDUM ();
  return sd * sqrt (-2. * log (r1)) * cos (TWOPI * r2) + mean;
}


char 
lowercase (char c)
{
  return (char) tolower ((int) c);
}

char 
uppercase (char c)
{
  return (char) toupper ((int) c);
}

double 
sum (double *vector, long n)
{
  long i;
  double summ = 0.0;
  for (i = 0; i < n; i++)
    summ += vector[i];
  return summ;
}
/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 T R E E B U I L D I N G   R O U T I N E S 

 
 

 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: tree.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"
#include "random.h"
#include "sequence.h"
#include "world.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

#define NOTIPS 0
#define WITHTIPS 1

/* prototypes ------------------------------------------- */
void buildtree (world_fmt * world, long locus);
void create_treetimelist (world_fmt * world, timelist_fmt ** ltl, long locus);
void fix_times (world_fmt * world);
void first_smooth (world_fmt * world, long locus);
void set_dirty (node * p);
void construct_tymelist (world_fmt * world, timelist_fmt * timevector);
void timeslices (timelist_fmt ** timevector);
void add_partlineages (long numpop, timelist_fmt ** timevector);
double treelikelihood (world_fmt * world);
double pseudotreelikelihood (world_fmt * world, proposal_fmt * proposal);
void set_pop (node * theNode, long pop, long actualpop);
void pseudonuview (proposal_fmt * proposal, xarray_fmt xx1, double *lx1, double v1, xarray_fmt xx2, double lx2, double v2);
void ltov (node * p);
void treeout (FILE * treefile, node * joint, node * p, long s);
void print_tree (world_fmt * world, long g, long *filepos);
/* private functions------------------------------------- */
void treesetup (world_fmt * world, long locus);
/* allocations of nodes */
void allocatetips (world_fmt * world, long pop, long locus);
void allocateinterior (world_fmt * world, long locus);
void allocatepoproot (world_fmt * world, long locus);
void allocate_tip (world_fmt * world, node ** p, long pop, long locus, long a);
void alloc_seqx (world_fmt * world, node * theNode);
/* first tree material (upgma, distance) */
void set_tree (world_fmt * world, long locus);
void distance_allele (char *data, long tips, double **m);
void distance_micro (char **data, long tips, double **m);
void distance_sequence (data_fmt * data, long locus, long tips,
			long sites, double **m);

void makevalues (world_fmt * world, long locus);
void make_alleles (world_fmt * world, long locus);
void make_microsatellites (world_fmt * world, long locus);
void make_microbrownian (world_fmt * world, long locus);
void upgma (world_fmt * world, double **x, long tips, node ** nodep);
void set_top (world_fmt * world, node * p, long pop, long locus);
void set_v (node * p);
void fitch (world_fmt * world, long locus, node ** topnodes, long topmax);
void set_fitchpop1 (double *x, long *n, long pop);
void set_fitchpop2 (double *x1, long *n1, double *x2, long *n2);
void reduce_fitchpop (node * theNode, node *** tienodes);
void set_x (world_fmt * world, long locus, node * theNode);
void set_migration (world_fmt * world, node ** tienodes);
boolean notNULL (node ** nodelist, long n);
boolean is_same_x (node * p1, node * p2);
void set_pop (node * theNode, long pop, long actualpop);
void children (node * mother, node ** brother, node ** sister);
short findAllele (data_fmt * data, char s[], long locus);
void free_treetimes (world_fmt * world, long size);
void traverseNodes (node * theNode, timelist_fmt ** timevector, long *slice);
void increase_timelist (timelist_fmt ** timevector);
void add_lineages (long numpop, timelist_fmt ** timevector);
void smooth (const node * root, node * p, world_fmt * world, const long locus);
void which_nuview (char datatype);
void nuview_allele (node * mother, world_fmt * world, const long locus);
void nuview_micro (node * mother, world_fmt * world, const long locus);
void nuview_brownian (node * mother, world_fmt * world, const long locus);
void nuview_sequence (node * mother, world_fmt * world, const long locus);
void adjustroot (node * r);
double pseudo_tl_seq (phenotype xx1, phenotype xx2, double v1, double v2,
		      proposal_fmt * proposal, world_fmt * world);
void pseudonu_allele (proposal_fmt * proposal, double **xx1, double *lx1,
		      double v1, double *xx2, double lx2, double v2);
void pseudonu_micro (proposal_fmt * proposal, double **xx1, double *lx1,
		     double v1, double *xx2, double lx2, double v2);
void pseudonu_brownian (proposal_fmt * proposal, double **xx1, double *lx1,
			double v1, double *xx2, double lx2, double v2);
void pseudonu_seq (proposal_fmt * proposal, phenotype xxx1, double v1, phenotype xxx2, double v2);
void calculate_steps (world_fmt * world);
double logfac (long n);
double prob_micro (double t, long diff, world_fmt * world);

void treereader (world_fmt * world);
void length_to_times (node * p);
void treeread (FILE * file, node ** pp, node * q);
char processlength (FILE * file, node ** p);
node *allocate_nodelet (long num, char type);
void find_tips (node * p, node ** nodelist, long *z);
node *add_migration (node * p, long from, long to, double utime);
node *create_interior_node (node ** q);
node *create_root_node (node ** q);
node *create_tip_node (FILE * file, node ** q, char *ch);
char processbracket (FILE * file, node ** p);
void set_tree_pop (node * p, long *pop);
void allocate_x (node * p, world_fmt * world, char datatype, boolean withtips);
long find_firstpop (node * p);

/* global variables used for fitch() */
static long tienodenum = 100;
static long zzz = 0;
/* global variable NUVIEW points to function nuview_datatype() */
static void (*nuview) (node *, world_fmt *, long);



/*=======================================================*/
/* 
   Creates a start-genealogy using a coalescence approach

   - set NUVIEW according to datatype
   (this should go somewhere else perhaps)
   - initializes tree structure
   - fills tree with data
   - set_tree(): upgma-tree, 
   adjust for times, 
   fitch for migration events
 */
void 
buildtree (world_fmt * world, long locus)
{
  long pop;
  long genomes = (world->options->datatype == 's') ? 1 : 2;
  world->sumtips = 0;
  world->migration_counts = 0;
  for (pop = 0; pop < world->numpop; pop++)
    {
      world->sumtips += world->data->numind[pop][locus] * genomes;
    }
  which_nuview (world->options->datatype);
  switch (world->options->datatype)
    {
    case 's':
      init_sequences (world, locus);
      break;
    case 'b':
      world->data->freq = -10000000000000.;
      break;
    case 'm':
/*      world->data->freq = 1. / world->options->micro_stepnum; */
      break;
    case 'a':
      world->data->freq = 1. / MAXALLELES;
      world->data->freqlast = 1. - (world->data->maxalleles[locus] - 1) / MAXALLELES;
    }

  if (world->options->usertree)
    {
      treereader (world);
      makevalues (world, locus);
      if (world->options->datatype == 's')
	{
	  init_tbl (world, locus);
	  print_seqfreqs (world);
	  print_tbl (world, locus);
	  print_weights (world, locus);
	}
    }
  else
    {
      treesetup (world, locus);	/* setup of all population trees  */
      allocatepoproot (world, locus);
      makevalues (world, locus);
      if (world->data->skiploci[locus])
	return;
      if (world->options->datatype == 's')
	{
	  init_tbl (world, locus);
	  print_tbl (world, locus);
	  print_weights (world, locus);
	}
      set_tree (world, locus);
    }
  if (world->options->datatype == 'b')
    world->data->maxalleles[locus] = XBROWN_SIZE;
}

/* 
   creates the timelist which represents all time intervals
   on a tree. The timelist is an array of pointers and not a 
   linked list.

   - allocates memory using an arbitrary value
   this will be later adjusted if a longer list is needed
   - construct
 */
void 
create_treetimelist (world_fmt * world, timelist_fmt ** ltl, long locus)
{
  if ((*ltl) == NULL)
    {
      (*ltl) = (timelist_fmt *) calloc (1, sizeof (timelist_fmt));
      (*ltl)[0].tl = (vtlist *) calloc (1, TIMELIST_GUESS * sizeof (vtlist));
      (*ltl)[0].allocT = TIMELIST_GUESS;
    }
  (*ltl)[0].copies = 0;
  construct_tymelist (world, &(*ltl)[0]);
}


/* 
   start first pass trhough the tree to calculate 
   the tree-likleihood
 */
void 
first_smooth (world_fmt * world, long locus)
{
  smooth (world->root->next, crawlback (world->root->next), world, locus);
}

/* 
   Marks a node, so that TREELIKELIHOOD() 
   will recalulated values in node
 */
void 
set_dirty (node * p)
{
  p->dirty = TRUE;
}


/* 
   timlist constructor parts
 */
void 
timeslices (timelist_fmt ** timevector)
{
  long z;
  for (z = 0; z < (*timevector)->T; z++)
    {
      (*timevector)->tl[z].from = (*timevector)->tl[z].eventnode->pop;
      (*timevector)->tl[z].to = (*timevector)->tl[z].eventnode->actualpop;
      (*timevector)->tl[z].slice = z;
    }
}

void 
add_partlineages (long numpop, timelist_fmt ** timevector)
{
  long i, pop;
  for (i = (*timevector)->T - 2; i >= 0; i--)
    {
      for (pop = 0; pop < numpop; pop++)
	(*timevector)->tl[i].lineages[pop] = (*timevector)->tl[i + 1].lineages[pop];
      if ((*timevector)->tl[i].from == (*timevector)->tl[i].to)
	{
	  (*timevector)->tl[i].lineages[(*timevector)->tl[i].from] += 1;
	}
      else
	{
	  (*timevector)->tl[i].lineages[(*timevector)->tl[i].from] -= 1;
	  if ((*timevector)->tl[i].lineages[(*timevector)->tl[i].from] < 0)
	    {
	      error ("Error in add_partlineages");
	    }
	  (*timevector)->tl[i].lineages[(*timevector)->tl[i].to] += 1;
	}
    }
}

/*
   calculates the tree-likelihood according to datatype a, m, b, s
 */
double 
treelikelihood (world_fmt * world)
{
  long a;
  double term = 0.0;
  node *nn = crawlback (world->root->next);
  set_dirty (nn);
  smooth (world->root->next, crawlback (world->root->next), world, world->locus);
  adjustroot (world->root);
  switch (world->options->datatype)
    {
    case 's':
      return treelike_seq (world, world->locus);
    case 'a':
      for (a = 0; a < world->data->maxalleles[world->locus] - 1; a++)
	{
	  term += (world->data->freq * nn->x.a[a]);
	}
      term += (world->data->freqlast * nn->x.a[a]);
      break;
    case 'm':
      for (a = 0; a < world->data->maxalleles[world->locus]; a++)
	{
	  term += nn->x.a[a];
	}
      break;
    case 'b':
      return nn->x.a[2];

    }
  if (term == 0.0)
    return -DBL_MAX;
  else
    return log (term) + nn->lxmax;
}

/* 
   calculates tree-likelihood using only arrays
   DOES NOT CHANGE ARRAYS IN THE TREE
 */
double 
pseudotreelikelihood (world_fmt * world, proposal_fmt * proposal)
{
  long a, locus = world->locus;
  /* freq is not different between pop */
  double term = 0.0;
  switch (world->options->datatype)
    {
    case 's':
      return pseudo_tl_seq (proposal->xf.s, proposal->xt.s,
			    proposal->v, proposal->vs, proposal, world);

    case 'a':
      for (a = 0; a < world->data->maxalleles[locus] - 1; a++)
	{
	  term += (world->data->freq * proposal->xf.a[a]);
	}
      term += (world->data->freqlast * proposal->xf.a[a]);
      if (term == 0.0)
	{
	  error ("pseudotreelikelihood(): likelihood = -Infinity\n");
	}
      return (log (term) + proposal->mf);

    case 'b':
      return proposal->xf.a[2];

    case 'm':
      for (a = 0; a < world->data->maxalleles[locus]; a++)
	{
	  term += proposal->xf.a[a];
	}
      if (term == 0.0)
	{
	  return -DBL_MAX;
	}
      return (log (term) + proposal->mf);

    }
  return -DBL_MAX;
}


/* 
   Calculates the sub-likelihoods but does not change the arrays in
   the tree, it uses the passed arrays and overwrites the xx1 array
   DOES NOT CHANGE THE TREE
 */
void 
pseudonuview (proposal_fmt * proposal, xarray_fmt xx1, double *lx1, double v1, xarray_fmt xx2, double lx2, double v2)
{
  switch (proposal->datatype)
    {
    case 'a':
      pseudonu_allele (proposal, &xx1.a, lx1, v1, xx2.a, lx2, v2);
      break;
    case 'b':
      pseudonu_brownian (proposal, &xx1.a, lx1, v1, xx2.a, lx2, v2);
      break;
    case 'm':
      pseudonu_micro (proposal, &xx1.a, lx1, v1, xx2.a, lx2, v2);
      break;
    case 's':
      pseudonu_seq (proposal, xx1.s, v1, xx2.s, v2);
      break;
    }
}
void 
pseudonu_allele (proposal_fmt * proposal, double **xx1, double *lx1,
		 double v1, double *xx2, double lx2, double v2)
{
  long a, aa, locus = proposal->world->locus;	/* allele counters */
  long mal = proposal->world->data->maxalleles[locus];	/* maxalleles */
  double freq = proposal->world->data->freq;
  double freqlast = proposal->world->data->freqlast;
  double w1 = 0.0, w2 = 0.0;	/* time variables */
  double pija1, pija2;		/* summary of probabilities */
  double x3m = -DBL_MAX;

#ifdef __GNU__
  double xx3[proposal->world->data->maxalleles[locus]];
#else
  double *xx3;			/* likelihoodarray, overwrites xx1 */
  xx3 = (double *) calloc (1, sizeof (double) * mal);
#endif
  v1 = 1 - exp (-v1);
  v2 = 1 - exp (-v2);
  if (v1 >= 1.)
    {
      w1 = 0.0;
      v1 = 1.0;
    }
  else
    {
      w1 = 1.0 - (v1);
    }
  if (v2 >= 1.)
    {
      w2 = 0.0;
      v2 = 1.0;
    }
  else
    {
      w2 = 1.0 - v2;
    }
  for (aa = 0; aa < mal; aa++)
    {
      pija1 = pija2 = 0.0;
      for (a = 0; a < mal - 1; a++)
	{
	  pija1 += ((aa == a) * w1 + v1 * freq) * (*xx1)[a];
	  pija2 += ((aa == a) * w2 + v2 * freq) * xx2[a];
	}
      pija1 += ((aa == a) * w1 + v1 * freqlast) * (*xx1)[a];
      pija2 += ((aa == a) * w2 + v2 * freqlast) * xx2[a];
      xx3[aa] = pija1 * pija2;
      if (xx3[aa] > x3m)
	x3m = xx3[aa];
    }

  for (aa = 0; aa < mal; aa++)
    {
      xx3[aa] /= x3m;
    }
  *lx1 += log (x3m) + lx2;
  memcpy (*xx1, xx3, sizeof (double) * mal);
#ifndef __GNU__
  free (xx3);
#endif
}

void 
pseudonu_micro (proposal_fmt * proposal, double **xx1, double *lx1,
		double v1, double *xx2, double lx2, double v2)
{
  long a, s, diff, locus = proposal->world->locus;	/* allele counters */
  long smax = proposal->world->data->maxalleles[locus];
  long margin = proposal->world->options->micro_threshold;
  double pija1s, pija2s, vv1, vv2;
  double x3m = -DBL_MAX;
  world_fmt *world = proposal->world;
#ifdef __GNU__
  double xx3[proposal->world->data->maxalleles[locus]];
#else
  double *xx3;
  xx3 = (double *) calloc (1, sizeof (double) * smax);
#endif
  vv1 = v1;
  vv2 = v2;
  for (s = 0; s < smax; s++)
    {
      pija1s = pija2s = 0.0;
      for (a = MAX (0, s - margin); a < s + margin && a < smax; a++)
	{
	  diff = labs (s - a);
	  if ((*xx1)[a] > 0)
	    {
	      pija1s += prob_micro (vv1, diff, world) * (*xx1)[a];
	    }
	  if (xx2[a] > 0)
	    {
	      pija2s += prob_micro (vv2, diff, world) * xx2[a];
	    }
	}
      xx3[s] = pija1s * pija2s;
      if (xx3[s] > x3m)
	x3m = xx3[s];
    }
  if (x3m == 0.0)
    {
      *lx1 = -DBL_MAX;
    }
  else
    {
      for (s = 0; s < smax; s++)
	{
	  xx3[s] /= x3m;
	}
      *lx1 += log (x3m) + lx2;
    }
  memcpy (*xx1, xx3, sizeof (double) * smax);
#ifndef __GNU__
  free (xx3);
#endif
}

void 
pseudonu_brownian (proposal_fmt * proposal, double **xx1, double *lx1,
		   double v1, double *xx2, double lx2, double v2)
{
  double vtot, c12;
  double mean1, mean2, mean, vv1, vv2, f1, f2, diff;
  mean1 = (*xx1)[0];
  mean2 = xx2[0];

  vv1 = v1 + (*xx1)[1];
  vv2 = v2 + xx2[1];
  vtot = vv1 + vv2;
  if (vtot > 0.0)
    f1 = vv2 / vtot;
  else
    f1 = 0.5;
  f2 = 1.0 - f1;
  mean = f1 * mean1 + f2 * mean2;
  vtot = vv1 + vv2;
  diff = mean1 - mean2;
  c12 = diff * diff / vtot;
  (*xx1)[2] = (*xx1)[2] + xx2[2] + MIN (0, -0.5 * (log (vtot) + c12) + LOG2PIHALF);
  (*xx1)[1] = vv1 * f1;
  (*xx1)[0] = mean;
}

/*
   adjust the variables POP and ACTUALPOP in interior nodes
 */
void 
set_pop (node * theNode, long pop, long actualpop)
{
  switch (theNode->type)
    {
    case 'm':
      theNode->pop = theNode->next->pop = pop;
      theNode->actualpop = theNode->next->actualpop = actualpop;
      break;
    case 'i':
    case 'r':
      theNode->pop = theNode->next->pop = theNode->next->next->pop = pop;
      theNode->actualpop = theNode->next->actualpop = actualpop;
      theNode->next->next->actualpop = actualpop;
      break;
    case 't':
      if (theNode->pop != pop)
	{
	  fprintf (stderr, "do not try to change the tips!");
	  exit (EXIT_FAILURE);
	}
      break;
    default:
      fprintf (stderr, "this node cannot exist, and we stop here!");
      exit (EXIT_FAILURE);
      break;
    }
}



/* =======================================================
   local functions 
 */

void 
treesetup (world_fmt * world, long locus)
{
  long nodenum = 0, pop, numpop = world->numpop;
  long genomes = world->options->datatype == 's' ? 1 : 2;
  for (pop = 0; pop < numpop; pop++)
    {
      nodenum += world->data->numind[pop][locus] * genomes * 2;
    }
  world->nodep = (node **) malloc (nodenum * sizeof (node *));
  for (pop = 0; pop < numpop; pop++)
    {
      allocatetips (world, pop, locus);
    }
  allocateinterior (world, locus);
}

void 
allocatetips (world_fmt * world, long pop, long locus)
{
  long a, mini = 0, maxi = 0, genomes;
  genomes = (world->options->datatype == 's') ? 1 : 2;
  for (a = 0; a < pop; a++)
    mini += world->data->numind[a][locus] * genomes;
  maxi = mini + world->data->numind[a][locus] * genomes;
  for (a = mini; a < maxi; a++)
    {
      allocate_tip (world, &world->nodep[a], pop, locus, a);
    }
}
void 
allocateinterior (world_fmt * world, long locus)
{
  node *p;
  long i;
  long mini = 0, maxi = 0;
  long numpop = world->numpop;
  long genomes = (world->options->datatype == 's') ? 1 : 2;
  for (i = 0; i < numpop; i++)
    {
      mini += world->data->numind[i][locus] * genomes;
      maxi += world->data->numind[i][locus] * genomes * 2;
    }
  for (i = mini; i < maxi; i++)
    {
      p = allocate_nodelet (3, 'i');
      p->top = TRUE;
      world->nodep[i] = p;
    }
}

node *
allocate_nodelet (long num, char type)
{
  static long unique_id = 0;
  boolean isfirst = TRUE;
  long j, temp;
  node *p, *q = NULL, *pfirst = NULL;
  temp = unique_id;
  for (j = 0; j < num; j++)
    {
      p = (node *) malloc (sizeof (node));
      p->tip = FALSE;
      p->number = temp;
      p->pop = -1;
      p->actualpop = -1;
      p->type = type;
      p->id = unique_id++;
      p->top = FALSE;
      p->dirty = TRUE;
      p->next = q;
      p->x.s = NULL;
      p->x.a = NULL;
      p->lxmax = 0.0;
      p->back = NULL;
      p->nayme = NULL;
      p->v = 0.0;
      p->tyme = 0.0;
      p->length = 0.0;
      if (isfirst)
	{
	  isfirst = FALSE;
	  pfirst = p;
	}

      q = p;
    }
  pfirst->next = q;
  return q;
}

void 
allocatepoproot (world_fmt * world, long locus)
{
  long i;
  node *p, *q, *qq;
  long nodenum = 0;		/*arbitrarily to the first */
  long genomes = (world->options->datatype == 's') ? 1 : 2;
  q = NULL;
  p = allocate_nodelet (3, 'i');
  p->top = TRUE;
  for (i = 0; i < world->numpop; i++)
    nodenum += world->data->numind[i][locus] * genomes;
  p->x.a = (double *) calloc (1, nodenum * sizeof (double));
  p->top = TRUE;
  qq = p;
  q = NULL;
  p = allocate_nodelet (3, 'r');
  p->top = TRUE;
  p->next->back = qq;
  qq->back = p->next;
  world->root = p;
}


void 
allocate_tip (world_fmt * world, node ** p, long pop, long locus, long a)
{
  (*p) = allocate_nodelet (1, 't');
  (*p)->tip = TRUE;
  (*p)->top = TRUE;
  (*p)->pop = (*p)->actualpop = pop;
  if (world->options->datatype == 's')
    {
      (*p)->nayme = (char *) calloc (1, sizeof (char) * (world->options->nmlength + 1));
      alloc_seqx (world, (*p));
    }
  else
    {
      (*p)->x.a = (double *) calloc (1, world->data->maxalleles[locus] * sizeof (double));
      (*p)->nayme = (char *) calloc (1, sizeof (char) * (DEFAULT_ALLELENMLENGTH + 1));
    }
}

void 
makevalues (world_fmt * world, long locus)
{
  switch (world->options->datatype)
    {
    case 'a':
      make_alleles (world, locus);
      break;
    case 'b':
      make_microbrownian (world, locus);
      break;
    case 'm':
      make_microsatellites (world, locus);
      break;
    case 's':
      make_sequences (world, locus);
      break;
    default:
      fprintf (stderr, "Oh yes, it would be nice if there were more\n");
      fprintf (stderr, "possible datatypes than just an\n");
      fprintf (stderr, "allele model, microsatellite model or sequence model.\n");
      fprintf (stderr, "But there are currently no others, so the programs stops\n\n");
      exit (EXIT_FAILURE);
      break;
    }
}

/*
   creates the branchlength and adds migraiton nodes to the 
   start tree
   - creates rough genetic distance for upgma
   - upgma
   - find branches where we need to insert migrations
   - insert migrations
   - adjust time of all nodes using the coalescent with migration
 */
void 
set_tree (world_fmt * world, long locus)
{
  long pop, tips = world->sumtips;

  double **distm;
  char *adata, **mdata;
  node **topnodes;
  topnodes = (node **) calloc (1, sizeof (node *) * tips);
  distm = (double **) calloc (1, sizeof (double *) * tips);
  distm[0] = (double *) calloc (1, sizeof (double) * tips * tips);
  for (pop = 0; pop < tips; pop++)
    distm[pop] = distm[0] + pop * tips;
  /* create a crude distance matrix according to the datatype */
  switch (world->options->datatype)
    {
    case 'a':
      adata = (char *) calloc (1, sizeof (char) * (tips + 1));
      for (pop = 0; pop < tips; pop++)
	{
	  adata[pop] = world->nodep[pop]->nayme[0];
	}
      distance_allele (adata, tips, distm);
      free (adata);
      break;
    case 'b':
    case 'm':
      mdata = (char **) calloc (1, sizeof (char *) * (tips + 1));
      for (pop = 0; pop < tips; pop++)
	{
	  mdata[pop] = (char *) calloc (1, sizeof (char) * world->options->allelenmlength);
	  strcpy (mdata[pop], world->nodep[pop]->nayme);
	}
      distance_micro (mdata, tips, distm);
      for (pop = 0; pop < tips; pop++)
	{
	  free (mdata[pop]);
	}
      free (mdata);
      break;
    case 's':
      distance_sequence (world->data, locus, tips,
			 world->data->seq->sites[locus], distm);
      break;
    }
  upgma (world, distm, tips, world->nodep);
  free (distm[0]);
  free (distm);
  world->root->tyme = world->root->next->tyme = world->root->next->next->tyme =
    world->root->next->back->tyme + 10000.;
  /* orient the tree up-down, set the length and v */
  set_top (world, world->root->next->back, pop, locus);
  set_v (world->root->next->back);
  /* insert migration nodes into the tree using 
     the Slatkin and Maddison approach (Fitch parsimony) */
  memcpy (topnodes, world->nodep, sizeof (node *) * tips);
  zzz = 0;
  allocate_x (world->root, world, 'a', NOTIPS);
  fitch (world, locus, topnodes, tips);
  free (topnodes);
}				/* set_tree */

void 
distance_allele (char *data, long tips, double **m)
{
  long i, j;
  for (i = 0; i < tips; i++)
    {
      for (j = 0; j < i; j++)
	{
	  if (data[i] != data[j])
	    m[i][j] = m[j][i] = fabs (rannor (1., 0.5));
	  else
	    m[i][j] = m[j][i] = fabs (rannor (0., 0.5));
	}
    }
}
void 
distance_micro (char **data, long tips, double **m)
{
  long i, j;
  for (i = 0; i < tips; i++)
    {
      for (j = 0; j < i; j++)
	{
	  m[i][j] = m[j][i] = pow (atof (data[i]) - atof (data[j]), 2.);
	  m[i][j] = m[j][i] = fabs (rannor (m[i][j], 0.1));
	}
    }
}

/* calculate  pairwise distances using sequence similarity */
void 
distance_sequence (data_fmt * data, long locus, long tips, long sites, double **m)
{
  long i = 0, j, z, pop, ind;
  char **dat;
  dat = (char **) malloc (sizeof (char *) * tips);
  for (pop = 0; pop < data->numpop; pop++)
    {
      for (ind = 0; ind < data->numind[pop][locus]; ind++)
	{
	  dat[i++] = data->yy[pop][ind][locus][0];
	}
    }
  if (i != tips)
    {
      error ("Mistake ind distance_sequence() tips is not equal sum(i)\n");
    }
  for (i = 0; i < tips; i++)
    {
      for (j = i + 1; j < tips; j++)
	{
	  for (z = 0; z < sites; z++)
	    {
	      if (dat[i][z] != dat[j][z])
		m[i][j] = m[j][i] += fabs (rannor (1.0, 0.1));
	    }
	}
    }
  free (dat);
}



void 
fix_times (world_fmt * world)
{
  long k, k1, k2;
  double age = 0;
  if (!world->options->usertree)
    {
      k1 = world->treetimes[0].tl[0].lineages[0];
      k2 = world->treetimes[0].tl[0].lineages[1];
      age = 1. / ((world->param0[2 + 0] * (k1) + world->param0[2 + 1] * (k2)) +
		  (k1 * (k1 - 1) / world->param0[0]) + (k2 * (k2 - 1) / world->param0[1]));
      world->treetimes[0].tl[0].age = age;
      adjust_time (world->treetimes[0].tl[0].eventnode, age);
      for (k = 1; k < world->treetimes[0].T - 1; k++)
	{
	  k1 = world->treetimes[0].tl[k].lineages[0];
	  k2 = world->treetimes[0].tl[k].lineages[1];
	  age += 1. /
	    ((world->param0[2 + 0] * (k1) + world->param0[2 + 1] * (k2)) +
	     (k1 * (k1 - 1) / world->param0[0]) + (k2 * (k2 - 1) / world->param0[1]));
	  world->treetimes[0].tl[k].age = age;
	  adjust_time (world->treetimes[0].tl[k].eventnode, age);
	}
      world->treetimes[0].tl[k].age = age + 10000.;	/* this is the root */
      adjust_time (world->treetimes[0].tl[k].eventnode, age);
    }
  set_v (world->root->next->back);
}

/* creates a UPGMA tree:
   x     = distance matrix which will be destroyed through
   the process, 
   tips  = # of sequences/alleles,
   nodep = treenodes have to be allocated for ALL nodes

   This code is stripped neighbor-joining code out of 
   phylip v3.6. Only the upgma option is present.
 */
void 
upgma (world_fmt * world, double **x, long tips, node ** nodep)
{
  long nc, nextnode, mini = -900, minj = -900, i, j, ia, ja, zz = 1;
  double total, tmin, bi, bj, /* ti, tj, */ da;
  double *av;
  long *oc;
  node **cluster;

  /* First initialization */
  nextnode = tips;
  av = (double *) calloc (1, tips * sizeof (double));
/*    at = (double *) calloc(1, tips * sizeof(double)); */
  oc = (long *) malloc (tips * 10 * sizeof (long));
  cluster = (node **) calloc (1, tips * sizeof (node *));
  for (i = 0; i < tips; i++)
    oc[i] = 1;
  for (i = 0; i < tips; i++)
    cluster[i] = nodep[i];
  /* Enter the main cycle */
  for (nc = 0; nc < tips - 1; nc++)
    {
      tmin = 99999.0;
      /* Compute sij and minimize */
      for (ja = 1; ja < tips; ja++)
	{
	  if (cluster[ja] != NULL)
	    {
	      for (ia = 0; ia < ja; ia++)
		{
		  if (cluster[ia] != NULL)
		    {
		      total = x[ia][ja];
		      if (total < tmin)
			{
			  tmin = total;
			  mini = ia;
			  minj = ja;
			}
		    }
		}
	    }
	}			/* compute lengths and print */
      bi = x[mini][minj] / 2.0 - av[mini];
      bj = x[mini][minj] / 2.0 - av[minj];
      av[mini] += bi;
      /*    ti = (bi<1) ? log(1.- bi) : 0.0 - at[mini]; */
      /*    tj = (bj<1) ? log(1.- bj) : 0.0 - at[minj]; */
      /*    at[mini] -= ti; */
      nodep[nextnode]->next->back = cluster[mini];
      cluster[mini]->back = nodep[nextnode]->next;
      nodep[nextnode]->next->next->back = cluster[minj];
      cluster[minj]->back = nodep[nextnode]->next->next;
      cluster[mini]->back->v = cluster[mini]->v = bi;
      cluster[minj]->back->v = cluster[minj]->v = bj;
      /*    cluster[mini]->back->tyme = cluster[mini]->tyme = ti; */
      /*    cluster[minj]->back->tyme = cluster[minj]->tyme = tj; */
      cluster[mini] = nodep[nextnode];
      adjust_time (nodep[nextnode], (double) zz++);

      cluster[minj] = NULL;
      nextnode++;
      /* re-initialization */
      for (j = 0; j < tips; j++)
	{
	  if (cluster[j] != NULL)
	    {
	      da = (x[mini][j] * oc[mini] + x[minj][j] * oc[minj]) / (oc[mini] + oc[minj]);
	      x[mini][j] = x[j][mini] = da;
	    }
	}
      for (j = 0; j < tips; j++)
	{
	  x[minj][j] = x[j][minj] = 0.0;
	}
      oc[mini] += oc[minj];
    }
  /* the last cycle */
  for (i = 0; i < tips; i++)
    {
      if (cluster[i] != NULL)
	break;
    }
  world->root->next->back = cluster[i];
  cluster[i]->back = world->root->next;
  free (av);
  free (oc);
  free (cluster);
}

void 
set_top (world_fmt * world, node * p, long pop, long locus)
{


  if (p->tip)
    {
      p->top = TRUE;
      p->tyme = 0.0;
      return;
    }
  p->top = TRUE;
  p->next->top = FALSE;
  if (p->type != 'm')
    {
      p->next->next->top = FALSE;
    }
  set_top (world, p->next->back, pop, locus);
  if (p->type != 'm')
    {
      set_top (world, p->next->next->back, pop, locus);
    }
  if (p->id == crawlback (world->root->next)->id)
    {
      p->back->top = FALSE;
      p->back->tyme = ROOTLENGTH;
    }
}				/* set_top */

void 
set_v (node * p)
{
  if (p->tip)
    {
      p->v = p->length = lengthof (p);
      return;
    }
  ltov (p);
  set_v (crawlback (p->next));
  set_v (crawlback (p->next->next));
}				/* set_v */

void 
ltov (node * p)
{
  p->v = lengthof (p);
}				/* ltov */

void 
fitch (world_fmt * world, long locus, node ** topnodes, long topmax)
{
  long i, j;
  node *tmp;
  node **tienodes;
  tienodes = (node **) calloc (1, sizeof (node *) * tienodenum);
  while (notNULL (topnodes, topmax))
    {
      for (i = 0; i < topmax; i++)
	{
	  if (topnodes[i] != NULL)
	    {
	      for (j = 0; j < topmax; j++)
		{
		  if (topnodes[j] != NULL && topnodes[i] != topnodes[j])
		    {
		      if ((tmp = showtop (topnodes[i]->back)) == showtop (topnodes[j]->back))
			{
			  if (tmp->pop == -1)
			    tmp->pop = 0;
			  if (topnodes[i]->type == 't')
			    set_fitchpop1 (tmp->x.a, &tmp->pop, topnodes[i]->pop);
			  else
			    set_fitchpop2 (tmp->x.a, &tmp->pop, topnodes[i]->x.a, &topnodes[i]->pop);
			  if (topnodes[j]->type == 't')
			    set_fitchpop1 (tmp->x.a, &tmp->pop, topnodes[j]->pop);
			  else
			    set_fitchpop2 (tmp->x.a, &tmp->pop, topnodes[j]->x.a, &topnodes[j]->pop);
			  reduce_fitchpop (tmp, &tienodes);
			  topnodes[i] = tmp;
			  topnodes[j] = NULL;
			}
		    }
		}
	    }
	}
    }
  set_x (world, locus, world->root->next->back);
  set_migration (world, tienodes);
  free (tienodes);
}

boolean 
notNULL (node ** nodelist, long n)
{
  long i, j = 0;
  for (i = 0; i < n; i++)
    if (nodelist[i] != NULL)
      j++;
  return (j > 1);
}
void 
set_fitchpop1 (double *x, long *n, long pop)
{
  x[(*n)++] = pop;
}


void 
set_fitchpop2 (double *x1, long *n1, double *x2, long *n2)
{
  long j;
  for (j = 0; j < (*n2); j++)
    x1[(*n1)++] = x2[j];
}
void 
reduce_fitchpop (node * theNode, node *** tienodes)
{
  long i, z, *group;
  double *value;
  group = (long *) calloc (1, sizeof (long) * (theNode->pop + 1));
  value = (double *) calloc (1, sizeof (double) * (theNode->pop + 1));
  qsort (theNode->x.a, theNode->pop, sizeof (double), numcmp);
  z = 0;
  group[0] = 1;
  value[0] = theNode->x.a[0];
  group[0] = 1;
  for (i = 1; i < theNode->pop; i++)
    {
      if (theNode->x.a[i] == theNode->x.a[i - 1])
	group[z] += 1;
      else
	{
	  z++;
	  value[z] = theNode->x.a[i];
	  group[z] = 1;
	}
    }
  switch ((short) z)
    {
    case 0:			/* only one group found */
      theNode->pop = 1;
      break;
    case 1:			/*two groups */
      if (group[0] == group[1])
	{
	  theNode->x.a[0] = value[0];
	  theNode->x.a[1] = value[1];
	  theNode->pop = 2;
	  (*tienodes)[zzz++] = theNode;
	  if (zzz > tienodenum - 1)
	    {
	      (*tienodes) = (node **) realloc ((*tienodes), sizeof (node *) * 2 * tienodenum);
	      memset ((*tienodes) + tienodenum, 0, sizeof (node *) * tienodenum);
	      tienodenum *= 2;
	    }
	}
      else
	{
	  if (group[0] > group[1])
	    {
	      theNode->x.a[0] = value[0];
	      theNode->pop = 1;
	    }
	  else
	    {
	      theNode->x.a[0] = value[1];
	      theNode->pop = 1;
	    }
	}
      break;
    default:
      error ("not yet implemented, this has to done for popsizes>2\n");
      break;
    }
  free (group);
  free (value);
}


void 
set_x (world_fmt * world, long locus, node * theNode)
{
  if (theNode != NULL)
    {
      if (theNode->type != 't')
	{
	  if (theNode->x.a != NULL)
	    set_pop (theNode, (long) theNode->x.a[0], (long) theNode->x.a[0]);
	  if (theNode->x.a != NULL)
	    {
	      switch (world->options->datatype)
		{
		case 'a':
		case 'm':
		  theNode->x.a = (double *) realloc (theNode->x.a, sizeof (double)
					  * world->data->maxalleles[locus]);
		  break;
		case 'b':
		  theNode->x.a = (double *) realloc (theNode->x.a,
					     sizeof (double) * XBROWN_SIZE);
		  break;
		case 's':
		  free (theNode->x.a);
		  alloc_seqx (world, theNode);
		  break;
		}
	      set_x (world, locus, theNode->next->back);
	      set_x (world, locus, theNode->next->next->back);
	    }
	}
    }
}

void 
set_migration (world_fmt * world, node ** tienodes)
{
  long i, migr_table_counter;
  node *d1, *d2, *d3, *bottom = world->root->next->back;
  migr_table_fmt *migr_table;
  migr_table = (migr_table_fmt *) calloc (1, sizeof (migr_table_fmt) * 2);

  i = zzz - 1;
  while (i >= 0 && tienodes[i] != bottom)
    i--;
  if (i >= 0)
    {
      tienodes[i] = NULL;
      children (bottom, &d1, &d2);
      if (RANDUM () < 0.5)
	{
	  set_pop (world->root, d1->pop, d1->pop);
	  set_pop (d2, d2->pop, d2->pop);
	  set_pop (bottom, d1->pop, d1->pop);
	  d3 = d2;
	}
      else
	{
	  set_pop (world->root, d2->pop, d2->pop);
	  set_pop (d1, d1->pop, d1->pop);
	  set_pop (bottom, d2->pop, d2->pop);
	  d3 = d1;
	}
      if (d1->pop != d2->pop)
	{
	  migr_table[0].from = bottom->pop;
	  migr_table[0].to = d3->pop;
	  migr_table[0].time = d3->tyme + RANDUM () * (bottom->tyme - d3->tyme);
	  migr_table_counter = 1;
	  insert_migr_node (world, d3, d3->back, migr_table, &migr_table_counter);
	}
      else
	{
	  set_pop (world->root, world->root->next->back->pop,
		   world->root->next->back->pop);
	}
    }
  else
    {
      set_pop (world->root, world->root->next->back->pop,
	       world->root->next->back->pop);
    }
  for (i = zzz - 1; i >= 0; i--)
    {
      if (tienodes[i] != NULL)
	{
	  migr_table_counter = 0;
	  children (tienodes[i], &d1, &d2);
	  if (d1->pop == d2->pop && d1->pop != showtop (tienodes[i]->back)->actualpop)
	    {
	      migr_table[0].from = showtop (tienodes[i]->back)->actualpop;
	      migr_table[0].to = tienodes[i]->pop;
	      migr_table[0].time = tienodes[i]->tyme +
		RANDUM () * (showtop (tienodes[i]->back)->tyme - tienodes[i]->tyme);
	      migr_table_counter = 1;
	      insert_migr_node (world, tienodes[i], tienodes[i]->back,
				migr_table, &migr_table_counter);
	    }
	  else
	    {
	      set_pop (tienodes[i], showtop (tienodes[i]->back)->pop,
		       showtop (tienodes[i]->back)->pop);
	      migr_table[0].from = tienodes[i]->pop;
	      if (d2->pop == tienodes[i]->pop)
		d2 = d1;
	      migr_table[0].to = d2->pop;
	      migr_table[0].time = d2->tyme + RANDUM () * (tienodes[i]->tyme - d2->tyme);
	      migr_table_counter = 1;
	      insert_migr_node (world, d2, d2->back, migr_table, &migr_table_counter);
	    }
	}
    }
  free (migr_table);
}



boolean 
is_same_x (node * p1, node * p2)
{
  long i, j, count = 0;
  if (p1->type == 't' || p2->type == 't')
    return FALSE;
  for (i = 0; i < p1->pop; i++)
    {
      for (j = 0; j < p2->pop; j++)
	{
	  if (p1->x.a[i] == p2->x.a[j])
	    count++;
	}
    }
  if (count < MAX (p1->pop, p2->pop))
    return FALSE;
  else
    return TRUE;
}



void 
alloc_seqx (world_fmt * world, node * theNode)
{
  short j;
  long endsite = world->data->seq->endsite;
  theNode->x.s = (phenotype) malloc (endsite * sizeof (ratelike *));
  for (j = 0; j < endsite; j++)
    theNode->x.s[j] =
      (ratelike) malloc (world->options->rcategs * sizeof (sitelike));
}

void 
make_alleles (world_fmt * world, long locus)
{
  long pop, ind, tmp = 0;
  char a1[DEFAULT_ALLELENMLENGTH];
  char a2[DEFAULT_ALLELENMLENGTH];
  node **nodelist = world->nodep;
  long *unknowns, unknownsum = 0, tips = 0;
  unknowns = (long *) calloc (1, sizeof (long) * world->numpop);
  for (pop = 0; pop < world->numpop; pop++)
    {
      tips += 2 * world->data->numind[pop][locus];
      for (ind = 0; ind < world->data->numind[pop][locus]; ind++)
	{
	  strcpy (a1, world->data->yy[pop][ind][locus][0]);
	  strcpy (a2, world->data->yy[pop][ind][locus][1]);
	  strcpy (nodelist[tmp + ind]->nayme, a1);
	  strcpy (nodelist[tmp + ind + world->data->numind[pop][locus]]->nayme, a2);
	  nodelist[tmp + ind]->x.a[findAllele (world->data, a1, locus)] = 1.0;
	  nodelist[tmp + ind + world->data->numind[pop][locus]]->x.a[findAllele (world->data, a2, locus)] = 1.0;
	  if (!strcmp (a1, "?"))
	    {
	      unknowns[pop] += 1;
	      nodelist[tmp + ind]->id = MAXLONG;
	    }
	  if (!strcmp (a2, "?"))
	    {
	      unknowns[pop] += 1;
	      nodelist[tmp + ind + world->data->numind[pop][locus]]->id = MAXLONG;
	    }
	}
      tmp += 2 * world->data->numind[pop][locus];
      unknownsum += unknowns[pop];
    }
  for (pop = 0; pop < world->numpop; pop++)
    world->data->numind[pop][locus] *= 2;
  if (unknownsum > 0)
    {
      qsort (world->nodep, 2 * tips, sizeof (node *), delcmp);
      for (pop = 0; pop < world->numpop; pop++)
	{
	  world->data->numind[pop][locus] -= unknowns[pop];
	  world->sumtips -= unknowns[pop];
	  if (world->data->numind[pop][locus] < 2)
	    {
	      world->data->skiploci[locus] = TRUE;
	    }
	}
      if (world->data->skiploci[locus])
	world->skipped += 1;
    }
  free (unknowns);
}

void 
make_microsatellites (world_fmt * world, long locus)
{
  long pop, ind, tmp = 0, tips = 0, unknownsum = 0;
  long *unknowns;
  long genomes = world->options->datatype == 's' ? 1 : 2;
  node **nodelist = world->nodep;
  char a1[DEFAULT_ALLELENMLENGTH];
  char a2[DEFAULT_ALLELENMLENGTH];
  unknowns = (long *) calloc (1, sizeof (long) * world->numpop);
  calculate_steps (world);
  for (pop = 0; pop < world->numpop; pop++)
    {
      tips += world->data->numind[pop][locus] * genomes;
      for (ind = 0; ind < world->data->numind[pop][locus]; ind++)
	{
	  strcpy (a1, world->data->yy[pop][ind][locus][0]);
	  strcpy (a2, world->data->yy[pop][ind][locus][1]);
	  strcpy (nodelist[tmp + ind]->nayme, a1);
	  strcpy (nodelist[tmp + ind + world->data->numind[pop][locus]]->nayme, a2);
	  nodelist[tmp + ind]->x.a[atoi (a1)] = 1.0;
	  nodelist[tmp + ind + world->data->numind[pop][locus]]->x.a[atoi (a2)] = 1.0;
	  if (!strcmp (a1, "?"))
	    {
	      unknowns[pop] += 1;
	      nodelist[tmp + ind]->id = MAXLONG;
	    }
	  if (!strcmp (a2, "?"))
	    {
	      unknowns[pop] += 1;
	      nodelist[tmp + ind + world->data->numind[pop][locus]]->id = MAXLONG;
	    }
	}
      tmp += 2 * world->data->numind[pop][locus];
      unknownsum += unknowns[pop];
    }
  for (pop = 0; pop < world->numpop; pop++)
    world->data->numind[pop][locus] *= genomes;
  if (unknownsum > 0)
    {
      qsort (world->nodep, 2 * tips, sizeof (node *), delcmp);
      for (pop = 0; pop < world->numpop; pop++)
	{
	  world->data->numind[pop][locus] -= unknowns[pop];
	  world->sumtips -= unknowns[pop];
	  if (world->data->numind[pop][locus] < 2)
	    {
	      world->data->skiploci[locus] = TRUE;
	    }
	}
      if (world->data->skiploci[locus])
	world->skipped += 1;
    }
  free (unknowns);
}

void 
make_microbrownian (world_fmt * world, long locus)
{
  long pop, ind, tmp = 0, tips = 0;
  long genomes = world->options->datatype == 's' ? 1 : 2;
  char a1[DEFAULT_ALLELENMLENGTH];
  char a2[DEFAULT_ALLELENMLENGTH];
  node **nodelist = world->nodep;
  long *unknowns, unknownsum = 0;
  unknowns = (long *) calloc (1, sizeof (long) * world->numpop);
  for (pop = 0; pop < world->numpop; pop++)
    {
      tips += world->data->numind[pop][locus] * genomes;
      for (ind = 0; ind < world->data->numind[pop][locus]; ind++)
	{
	  strcpy (a1, world->data->yy[pop][ind][locus][0]);
	  strcpy (a2, world->data->yy[pop][ind][locus][1]);
	  strcpy (nodelist[tmp + ind]->nayme, a1);
	  strcpy (nodelist[tmp + ind + world->data->numind[pop][locus]]->nayme, a2);
	  nodelist[tmp + ind]->x.a[0] = atof (a1);
	  nodelist[tmp + ind + world->data->numind[pop][locus]]->x.a[0] = atof (a2);
	  if (!strcmp (a1, "?"))
	    {
	      unknowns[pop] += 1;
	      nodelist[tmp + ind]->id = MAXLONG;
	    }
	  if (!strcmp (a2, "?"))
	    {
	      unknowns[pop] += 1;
	      nodelist[tmp + ind + world->data->numind[pop][locus]]->id = MAXLONG;
	    }
	}
      tmp += 2 * world->data->numind[pop][locus];
      unknownsum += unknowns[pop];
    }
  for (pop = 0; pop < world->numpop; pop++)
    world->data->numind[pop][locus] *= genomes;
  if (unknownsum > 0)
    {
      qsort (world->nodep, 2 * tips, sizeof (node *), delcmp);
      for (pop = 0; pop < world->numpop; pop++)
	{
	  world->data->numind[pop][locus] -= unknowns[pop];
	  world->sumtips -= unknowns[pop];
	  if (world->data->numind[pop][locus] < 2)
	    {
	      world->data->skiploci[locus] = TRUE;
	    }
	}
      if (world->data->skiploci[locus])
	world->skipped += 1;
    }
  free (unknowns);
}


short 
findAllele (data_fmt * data, char s[], long locus)
{
  short found = 0;
  while ((strcmp (s, data->allele[locus][found])
	  && data->allele[locus][found][0] != '\0'))
    found++;
  return found;
}

/*---------------------------------------------
free_treetimes frees the ptr_array of timeslist
*/
void 
free_treetimes (world_fmt * world, long size)
{
  while (size >= 0)
    free (world->treetimes[size--].tl);
}

void 
construct_tymelist (world_fmt * world, timelist_fmt * timevector)
{
  long z = 0;
  /*   timevector->tl[0].age = world->root->tyme;
     timevector->tl[0].eventnode = world->root; */
  traverseNodes (world->root, &timevector, &z);
  timevector->T = (size_t) z;
  qsort ((void *) timevector->tl, timevector->T, sizeof (vtlist), agecmp);
  if ((*timevector).tl[(*timevector).T - 1].eventnode->type != 'r')
    {
      z = 0;
      while ((*timevector).tl[z].eventnode->type != 'r')
	z++;
      (*timevector).tl[z].eventnode->tyme = (*timevector).tl[z].eventnode->next->tyme =
	(*timevector).tl[z].eventnode->next->next->tyme = (*timevector).tl[(*timevector).T - 1].eventnode->tyme + 10000.;
      (*timevector).tl[z].age = (*timevector).tl[z].eventnode->tyme;
      qsort ((void *) timevector->tl, timevector->T, sizeof (vtlist), agecmp);
      printf ("construct_tymelist root moved\n");
    }
  timeslices (&timevector);
  add_lineages (world->numpop, &timevector);
}

/* traverse the tree and writes node-information into the real timelist 
   also takes care that the size of timelist is increased accordingly */
void 
traverseNodes (node * theNode, timelist_fmt ** timevector, long *slice)
{
  if (theNode->type != 't')
    {
      if (theNode->next->back != NULL)
	traverseNodes (theNode->next->back, timevector, slice);
      if (theNode->type != 'm' && theNode->next->next->back != NULL)
	traverseNodes (theNode->next->next->back, timevector, slice);

      if (theNode->top)
	{
	  /* Here we are on the save side if we increase the timelist so
	     never a fence-write can happen */
	  if (*slice >= (*timevector)->allocT)
	    {
	      increase_timelist (timevector);
	    }
	  /*(*timevector)->tl[*slice].pop = theNode->actualpop; */
	  (*timevector)->tl[*slice].age = theNode->tyme;
	  (*timevector)->tl[*slice].eventnode = theNode;
	  (*slice) += 1;
	}
    }
}

void 
increase_timelist (timelist_fmt ** timevector)
{
  (*timevector)->oldT = (*timevector)->allocT;
  (*timevector)->allocT += (*timevector)->allocT / 4;	/*increase timelist by 25% */
  (*timevector)->tl = (vtlist *) realloc (
		(*timevector)->tl, sizeof (vtlist) * (*timevector)->allocT);
  memset ((*timevector)->tl + (*timevector)->oldT, 0,
	  ((*timevector)->allocT - (*timevector)->oldT) * sizeof (vtlist));
}

void 
add_lineages (long numpop, timelist_fmt ** timevector)
{
  long pop;
  for (pop = 0; pop < numpop; pop++)
    (*timevector)->tl[(*timevector)->T - 1].lineages[pop] = 0;
  (*timevector)->tl[(*timevector)->T - 1].lineages[(*timevector)->tl[(*timevector)->T - 2].from] += 1;
  add_partlineages (numpop, timevector);
}

void 
smooth (const node * root, node * p, world_fmt * world, const long locus)
{
  static long panic;
  /* only changed lineages are considered */
  if (!p->dirty)
    return;

  if (p == (crawlback (root)))
    panic = 0;
  if (p->type == 'm')
    {
      fprintf (stderr, "I found a migration node in smooth.\n");
      exit (EXIT_FAILURE);
    }
  if (panic++ > 1000)
    {
      fprintf (stderr, "smooth out of control\n");
      exit (EXIT_FAILURE);
    }
  if (p->type == 'i')
    {
      smooth (root, crawlback (p->next), world, locus);
      smooth (root, crawlback (p->next->next), world, locus);
      (*nuview) (p, world, locus);
      p->dirty = FALSE;
/*       fprintf(stdout,"*%li\n",p->id); */
    }
}				/* smooth */

void 
which_nuview (char datatype)
{
  switch (datatype)
    {
    case 'a':
      nuview = (void (*)(node *, world_fmt *, long)) nuview_allele;
      break;
    case 'b':
      nuview = (void (*)(node *, world_fmt *, long)) nuview_brownian;
      break;
    case 'm':
      nuview = (void (*)(node *, world_fmt *, long)) nuview_micro;
      break;
    case 's':
      nuview = (void (*)(node *, world_fmt *, long)) nuview_sequence;
      break;
    }
}

void 
nuview_allele (node * mother, world_fmt * world, const long locus)
{
  node *d1 = NULL, *d2 = NULL;
  long a, aa;
  long mal = world->data->maxalleles[locus];
  double w1, w2, v1, v2;
  double pija1, pija2, lx1, lx2;
  double test = 0.0;
  double freq = world->data->freq;
  double freql = world->data->freqlast;
  double x3m = -DBL_MAX;
  double *xx1, *xx2;
#ifdef __GNU__
  double xx3[world->data->maxalleles[locus]];
#else
  double *xx3;
  xx3 = (double *) calloc (1, sizeof (double) * mal);
#endif

  children (mother, &d1, &d2);
  xx1 = d1->x.a;
  xx2 = d2->x.a;
  lx1 = d1->lxmax;
  lx2 = d2->lxmax;
  v1 = 1 - exp (-d1->v);
  v2 = 1 - exp (-d2->v);
  if (v1 >= 1.)
    {
      w1 = 0.0;
      v1 = 1.0;
    }
  else
    {
      w1 = 1.0 - v1;
    }
  if (v2 >= 1.)
    {
      w2 = 0.0;
      v2 = 1.0;
    }
  else
    {
      w2 = 1.0 - v2;
    }
  for (aa = 0; aa < mal; aa++)
    {
      pija1 = pija2 = 0.0;
      for (a = 0; a < mal - 1; a++)
	{
	  pija1 += ((aa == a) * w1 + v1 * freq) * xx1[a];
	  pija2 += ((aa == a) * w2 + v2 * freq) * xx2[a];
	}
      pija1 += ((aa == a) * w1 + v1 * freql) * xx1[a];
      pija2 += ((aa == a) * w2 + v2 * freql) * xx2[a];
      xx3[aa] = pija1 * pija2;
      test += xx3[aa];
      if (xx3[aa] > x3m)
	x3m = xx3[aa];

    }
  if (test <= 0.0)
    {
      fprintf (stderr, "xx3 is 0 or garbage!");
      exit (EXIT_FAILURE);
    }
  for (aa = 0; aa < mal; aa++)
    {
      xx3[aa] /= x3m;
    }
  mother->lxmax = log (x3m) + lx2 + lx1;
  memcpy (mother->x.a, xx3, sizeof (double) * mal);
#ifndef __GNU__
  free (xx3);
#endif
}

void 
nuview_brownian (node * mother, world_fmt * world, const long locus)
{
  node *d1 = NULL, *d2 = NULL;
  double xx1, xx2, c12, diff;
  double mean1, mean2, mean, v1, v2, vtot, f1, f2;
  children (mother, &d1, &d2);
  mean1 = d1->x.a[0];
  mean2 = d2->x.a[0];
  xx1 = d1->x.a[2];
  xx2 = d2->x.a[2];

  v1 = d1->v + d1->x.a[1];	/* di->v == length of branch time(n1) - time(n2) */
  v2 = d2->v + d2->x.a[1];	/* x.a[1] contains the deltav */
  vtot = v1 + v2;
  if (vtot > 0.0)
    f1 = v2 / vtot;
  else
    f1 = 0.5;
  f2 = 1.0 - f1;
  mean = f1 * mean1 + f2 * mean2;
  diff = mean1 - mean2;
  c12 = diff * diff / vtot;
  mother->x.a[2] = xx1 + xx2 + MIN (0.0, -0.5 * (log (vtot) + c12) + LOG2PIHALF);
  /*  printf("L=%f , L1=%f, L2=%f, log(vtot=%f)=%f, c12=%f\n",mother->x.a[2], xx1, xx2,vtot,log(vtot),c12); */
  mother->x.a[1] = v1 * f1;
  mother->x.a[0] = mean;

}


void 
nuview_micro (node * mother, world_fmt * world, const long locus)
{
  node *d1 = NULL, *d2 = NULL;
  long s, a, diff;
  long margin = world->options->micro_threshold;
  double vv1, vv2, lx1, lx2;
  double x3m = -DBL_MAX;
  double pija1s, pija2s;
  double *xx1 = NULL, *xx2 = NULL;

  long smax = world->data->maxalleles[locus];
#ifdef __GNU__
  double xx3[world->data->maxalleles[locus]];
#else
  double *xx3;
  xx3 = (double *) calloc (1, sizeof (double) * smax);
#endif
  children (mother, &d1, &d2);
  vv1 = d1->v;
  vv2 = d2->v;
  xx1 = d1->x.a;
  xx2 = d2->x.a;
  lx1 = d1->lxmax;
  lx2 = d2->lxmax;
  for (s = 0; s < smax; s++)
    {
      pija1s = pija2s = 0.0;
      for (a = MAX (0, s - margin); a < s + margin && a < smax; a++)
	{
	  diff = labs (s - a);
	  if (xx1[a] > 0)
	    {
	      pija1s += prob_micro (vv1, diff, world) * xx1[a];
	    }
	  if (xx2[a] > 0)
	    {
	      pija2s += prob_micro (vv2, diff, world) * xx2[a];
	    }
	}
      xx3[s] = pija1s * pija2s;
      if (xx3[s] > x3m)
	x3m = xx3[s];
    }
  if (x3m == 0.0)
    {
      mother->lxmax = -DBL_MAX;
    }
  else
    {
      for (s = 0; s < smax; s++)
	{
	  xx3[s] /= x3m;
	}
      mother->lxmax = log (x3m) + lx1 + lx2;
    }
  memcpy (mother->x.a, xx3, sizeof (double) * smax);
#ifndef __GNU__
  free (xx3);
#endif
}

void 
calculate_steps (world_fmt * world)
{
  long k, diff;
  const long stepnum = world->options->micro_threshold;
  double **steps = world->options->steps;

  for (diff = 0; diff < stepnum; diff++)
    {
      for (k = diff; k < stepnum; k += 2)
	{
	  steps[diff][k - diff] = LOG2 * k +
	    logfac ((k - diff) / 2) + logfac ((k + diff) / 2);
	}
    }
}

double 
logfac (long n)
{
  /* log(n!) values were calculate with Mathematica 
     with a precision of 30 digits */
  switch (n)
    {
    case 0:
      return 0.;
    case 1:
      return 0.;
    case 2:
      return 0.693147180559945309417232121458;
    case 3:
      return 1.791759469228055000812477358381;
    case 4:
      return 3.1780538303479456196469416013;
    case 5:
      return 4.78749174278204599424770093452;
    case 6:
      return 6.5792512120101009950601782929;
    case 7:
      return 8.52516136106541430016553103635;
    case 8:
      return 10.60460290274525022841722740072;
    case 9:
      return 12.80182748008146961120771787457;
    case 10:
      return 15.10441257307551529522570932925;
    case 11:
      return 17.50230784587388583928765290722;
    case 12:
      return 19.98721449566188614951736238706;
    default:
      return lgamma (n + 1.);
    }
}



double 
prob_micro (double t, long diff, world_fmt * world)
{
  double **steps = world->options->steps;
  long stepnum = world->options->micro_threshold;
  long k;
  double sum = 0.0, oldsum = 0.0;
  double logt = log (t);
  if (diff >= stepnum)
    return sum;
  for (k = diff; k < diff + stepnum; k += 2)
    {
      sum += exp (-t + logt * k - steps[diff][k - diff]);
      if (oldsum - sum < DBL_EPSILON)
	break;
      oldsum = sum;
    }
  return sum;
}


void 
nuview_sequence (node * mother, world_fmt * world, const long locus)
{
  short i, j, k;
  double lw1, lw2, yy1, yy2, ww1zz1, vv1zz1, ww2zz2, vv2zz2, vzsumr1, vzsumr2,
    vzsumy1, vzsumy2, sum1, sum2, sumr1, sumr2, sumy1, sumy2, ww1, ww2,
    zz1, zz2;
  node *q, *r;
  sitelike xx1, xx2, xx3;
  short rcategs = world->options->rcategs;
  short categs = world->options->categs;
  tbl_fmt tbl = world->data->seq->tbl;
  seqmodel_fmt *seq;
  seq = world->data->seq;
  q = crawlback (mother->next);
  r = crawlback (mother->next->next);
  lw1 = -q->v / seq->fracchange;
  if (rcategs == 1 && categs == 1)
    {
      ww1 = exp (tbl[0][0]->ratxi * lw1);
      zz1 = exp (tbl[0][0]->ratxv * lw1);
      ww1zz1 = ww1 * zz1;
      vv1zz1 = (1.0 - ww1) * zz1;
      lw2 = -r->v / seq->fracchange;
      ww2 = exp (tbl[0][0]->ratxi * lw2);
      zz2 = exp (tbl[0][0]->ratxv * lw2);
      ww2zz2 = ww2 * zz2;
      vv2zz2 = (1.0 - ww2) * zz2;
      yy1 = 1.0 - zz1;
      yy2 = 1.0 - zz2;
      for (i = 0; i < seq->endsite; i++)
	{
	  memcpy (xx1, q->x.s[i][0], sizeof (sitelike));
	  memcpy (xx2, r->x.s[i][0], sizeof (sitelike));
	  sum1 = yy1 * (seq->freqa * xx1[0] + seq->freqc * xx1[1] +
			seq->freqg * xx1[2] + seq->freqt
			* xx1[3]);
	  sum2 = yy2 * (seq->freqa * xx2[0] + seq->freqc * xx2[1] +
			seq->freqg * xx2[2] + seq->freqt
			* xx2[3]);
	  sumr1 = seq->freqar * xx1[0] + seq->freqgr * xx1[2];
	  sumr2 = seq->freqar * xx2[0] + seq->freqgr * xx2[2];
	  sumy1 = seq->freqcy * xx1[1]
	    + seq->freqty * xx1[3];
	  sumy2 = seq->freqcy * xx2[1]
	    + seq->freqty * xx2[3];
	  vzsumr1 = vv1zz1 * sumr1;
	  vzsumr2 = vv2zz2 * sumr2;
	  vzsumy1 = vv1zz1 * sumy1;
	  vzsumy2 = vv2zz2 * sumy2;
	  xx3[0] = (sum1 + ww1zz1 * xx1[0] + vzsumr1) *
	    (sum2 + ww2zz2 * xx2[0] + vzsumr2);
	  xx3[1] =
	    (sum1 + ww1zz1 * xx1[1] + vzsumy1) *
	    (sum2 + ww2zz2 * xx2[1] + vzsumy2);
	  xx3[2] =
	    (sum1 + ww1zz1 * xx1[2] + vzsumr1) *
	    (sum2 + ww2zz2 * xx2[2] + vzsumr2);
	  xx3[3] =
	    (sum1 + ww1zz1 * xx1[3] + vzsumy1) *
	    (sum2 + ww2zz2 * xx2[3] + vzsumy2);
	  memcpy (mother->x.s[i][0], xx3, sizeof (sitelike));
	}
    }
  else
    {
      for (i = 0; i < rcategs; i++)
	for (j = 0; j < categs; j++)
	  {
	    tbl[i][j]->ww1 = exp (tbl[i][j]->ratxi * lw1);
	    tbl[i][j]->zz1 = exp (tbl[i][j]->ratxv * lw1);
	    tbl[i][j]->ww1zz1 = tbl[i][j]->ww1 * tbl[i][j]->zz1;
	    tbl[i][j]->vv1zz1 = (1.0 - tbl[i][j]->ww1) * tbl[i][j]->zz1;
	  }
      lw2 = -r->v / seq->fracchange;
      for (i = 0; i < rcategs; i++)
	for (j = 0; j < categs; j++)
	  {
	    tbl[i][j]->ww2 = exp (tbl[i][j]->ratxi * lw2);
	    tbl[i][j]->zz2 = exp (tbl[i][j]->ratxv * lw2);
	    tbl[i][j]->ww2zz2 = tbl[i][j]->ww2 * tbl[i][j]->zz2;
	    tbl[i][j]->vv2zz2 = (1.0 - tbl[i][j]->ww2) * tbl[i][j]->zz2;
	  }
      for (i = 0; i < seq->endsite; i++)
	{
	  k = seq->category[seq->alias[i] - 1] - 1;
	  for (j = 0; j < rcategs; j++)
	    {
	      ww1zz1 = tbl[j][k]->ww1zz1;
	      vv1zz1 = tbl[j][k]->vv1zz1;
	      yy1 = 1.0 - tbl[j][k]->zz1;
	      ww2zz2 = tbl[j][k]->ww2zz2;
	      vv2zz2 = tbl[j][k]->vv2zz2;
	      yy2 = 1.0 - tbl[j][k]->zz2;
	      memcpy (xx1, q->x.s[i][j], sizeof (sitelike));
	      memcpy (xx2, r->x.s[i][j], sizeof (sitelike));
	      sum1 = yy1 * (seq->freqa * xx1[0] + seq->freqc * xx1[1] +
			    seq->freqg * xx1[2] + seq->freqt
			    * xx1[3]);
	      sum2 = yy2 * (seq->freqa * xx2[0] + seq->freqc * xx2[1] +
			    seq->freqg * xx2[2] + seq->freqt
			    * xx2[3]);
	      sumr1 = seq->freqar * xx1[0] + seq->freqgr * xx1[2];
	      sumr2 = seq->freqar * xx2[0] + seq->freqgr * xx2[2];
	      sumy1 = seq->freqcy * xx1[1]
		+ seq->freqty * xx1[3];
	      sumy2 = seq->freqcy * xx2[1]
		+ seq->freqty * xx2[3];
	      vzsumr1 = vv1zz1 * sumr1;
	      vzsumr2 = vv2zz2 * sumr2;
	      vzsumy1 = vv1zz1 * sumy1;
	      vzsumy2 = vv2zz2 * sumy2;
	      xx3[0] = (sum1 + ww1zz1 * xx1[0] + vzsumr1) *
		(sum2 + ww2zz2 * xx2[0] + vzsumr2);
	      xx3[1] =
		(sum1 + ww1zz1 * xx1[1] + vzsumy1) *
		(sum2 + ww2zz2 * xx2[1] + vzsumy2);
	      xx3[2] =
		(sum1 + ww1zz1 * xx1[2] + vzsumr1) *
		(sum2 + ww2zz2 * xx2[2] + vzsumr2);
	      xx3[3] =
		(sum1 + ww1zz1 * xx1[3] + vzsumy1) *
		(sum2 + ww2zz2 * xx2[3] + vzsumy2);
	      memcpy (mother->x.s[i][j], xx3, sizeof (sitelike));
	    }
	}
    }
}				/* nuview */

void 
adjustroot (node * r)
{
  r->next->tyme = r->tyme;
  r->next->length = r->length;
  r->next->v = r->v;
  r->next->next->tyme = r->tyme;
  r->next->next->length = r->length;
  r->next->next->v = r->v;
}

void 
pseudonu_seq (proposal_fmt * proposal, phenotype xxx1, double v1, phenotype xxx2, double v2)
{
  short i, j, k;
  double lw1, lw2, yy1, yy2, ww1zz1, vv1zz1, ww2zz2, vv2zz2, vzsumr1, vzsumr2,
    vzsumy1, vzsumy2, sum1, sum2, sumr1, sumr2, sumy1, sumy2, ww1, ww2,
    zz1, zz2;
  sitelike xx1, xx2, xx3;
  short rcategs = proposal->world->options->rcategs;
  short categs = proposal->world->options->categs;
  tbl_fmt tbl = proposal->world->data->seq->tbl;
  seqmodel_fmt *seq;
  seq = proposal->world->data->seq;
  lw1 = -v1 / seq->fracchange;
  if (rcategs == 1 && categs == 1)
    {
      ww1 = exp (tbl[0][0]->ratxi * lw1);
      zz1 = exp (tbl[0][0]->ratxv * lw1);
      ww1zz1 = ww1 * zz1;
      vv1zz1 = (1.0 - ww1) * zz1;
      lw2 = -v2 / seq->fracchange;
      ww2 = exp (tbl[0][0]->ratxi * lw2);
      zz2 = exp (tbl[0][0]->ratxv * lw2);
      ww2zz2 = ww2 * zz2;
      vv2zz2 = (1.0 - ww2) * zz2;
      yy1 = 1.0 - zz1;
      yy2 = 1.0 - zz2;
      for (i = 0; i < seq->endsite; i++)
	{
	  memcpy (xx1, xxx1[i][0], sizeof (sitelike));
	  memcpy (xx2, xxx2[i][0], sizeof (sitelike));
	  sum1 = yy1 * (seq->freqa * xx1[0] + seq->freqc * xx1[1] +
			seq->freqg * xx1[2] + seq->freqt
			* xx1[3]);
	  sum2 = yy2 * (seq->freqa * xx2[0] + seq->freqc * xx2[1] +
			seq->freqg * xx2[2] + seq->freqt
			* xx2[3]);
	  sumr1 = seq->freqar * xx1[0] + seq->freqgr * xx1[2];
	  sumr2 = seq->freqar * xx2[0] + seq->freqgr * xx2[2];
	  sumy1 = seq->freqcy * xx1[1]
	    + seq->freqty * xx1[3];
	  sumy2 = seq->freqcy * xx2[1]
	    + seq->freqty * xx2[3];
	  vzsumr1 = vv1zz1 * sumr1;
	  vzsumr2 = vv2zz2 * sumr2;
	  vzsumy1 = vv1zz1 * sumy1;
	  vzsumy2 = vv2zz2 * sumy2;
	  xx3[0] = (sum1 + ww1zz1 * xx1[0] + vzsumr1) *
	    (sum2 + ww2zz2 * xx2[0] + vzsumr2);
	  xx3[1] =
	    (sum1 + ww1zz1 * xx1[1] + vzsumy1) *
	    (sum2 + ww2zz2 * xx2[1] + vzsumy2);
	  xx3[2] =
	    (sum1 + ww1zz1 * xx1[2] + vzsumr1) *
	    (sum2 + ww2zz2 * xx2[2] + vzsumr2);
	  xx3[3] =
	    (sum1 + ww1zz1 * xx1[3] + vzsumy1) *
	    (sum2 + ww2zz2 * xx2[3] + vzsumy2);
	  memcpy (xxx1[i][0], xx3, sizeof (sitelike));
	}
    }
  else
    {
      for (i = 0; i < rcategs; i++)
	for (j = 0; j < categs; j++)
	  {
	    tbl[i][j]->ww1 = exp (tbl[i][j]->ratxi * lw1);
	    tbl[i][j]->zz1 = exp (tbl[i][j]->ratxv * lw1);
	    tbl[i][j]->ww1zz1 = tbl[i][j]->ww1 * tbl[i][j]->zz1;
	    tbl[i][j]->vv1zz1 = (1.0 - tbl[i][j]->ww1) * tbl[i][j]->zz1;
	  }
      lw2 = -v2 / seq->fracchange;
      for (i = 0; i < rcategs; i++)
	for (j = 0; j < categs; j++)
	  {
	    tbl[i][j]->ww2 = exp (tbl[i][j]->ratxi * lw2);
	    tbl[i][j]->zz2 = exp (tbl[i][j]->ratxv * lw2);
	    tbl[i][j]->ww2zz2 = tbl[i][j]->ww2 * tbl[i][j]->zz2;
	    tbl[i][j]->vv2zz2 = (1.0 - tbl[i][j]->ww2) * tbl[i][j]->zz2;
	  }
      for (i = 0; i < seq->endsite; i++)
	{
	  k = seq->category[seq->alias[i] - 1] - 1;
	  for (j = 0; j < rcategs; j++)
	    {
	      ww1zz1 = tbl[j][k]->ww1zz1;
	      vv1zz1 = tbl[j][k]->vv1zz1;
	      yy1 = 1.0 - tbl[j][k]->zz1;
	      ww2zz2 = tbl[j][k]->ww2zz2;
	      vv2zz2 = tbl[j][k]->vv2zz2;
	      yy2 = 1.0 - tbl[j][k]->zz2;
	      memcpy (xx1, xxx1[i][j], sizeof (sitelike));
	      memcpy (xx2, xxx2[i][j], sizeof (sitelike));
	      sum1 = yy1 * (seq->freqa * xx1[0] + seq->freqc * xx1[1] +
			    seq->freqg * xx1[2] + seq->freqt
			    * xx1[3]);
	      sum2 = yy2 * (seq->freqa * xx2[0] + seq->freqc * xx2[1] +
			    seq->freqg * xx2[2] + seq->freqt
			    * xx2[3]);
	      sumr1 = seq->freqar * xx1[0] + seq->freqgr * xx1[2];
	      sumr2 = seq->freqar * xx2[0] + seq->freqgr * xx2[2];
	      sumy1 = seq->freqcy * xx1[1]
		+ seq->freqty * xx1[3];
	      sumy2 = seq->freqcy * xx2[1]
		+ seq->freqty * xx2[3];
	      vzsumr1 = vv1zz1 * sumr1;
	      vzsumr2 = vv2zz2 * sumr2;
	      vzsumy1 = vv1zz1 * sumy1;
	      vzsumy2 = vv2zz2 * sumy2;
	      xx3[0] = (sum1 + ww1zz1 * xx1[0] + vzsumr1) *
		(sum2 + ww2zz2 * xx2[0] + vzsumr2);
	      xx3[1] =
		(sum1 + ww1zz1 * xx1[1] + vzsumy1) *
		(sum2 + ww2zz2 * xx2[1] + vzsumy2);
	      xx3[2] =
		(sum1 + ww1zz1 * xx1[2] + vzsumr1) *
		(sum2 + ww2zz2 * xx2[2] + vzsumr2);
	      xx3[3] =
		(sum1 + ww1zz1 * xx1[3] + vzsumy1) *
		(sum2 + ww2zz2 * xx2[3] + vzsumy2);
	      memcpy (xxx1[i][j], xx3, sizeof (sitelike));
	    }
	}
    }
}				/* pseudonu_seq */

double 
pseudo_tl_seq (phenotype xx1, phenotype xx2, double v1, double v2,
	       proposal_fmt * proposal, world_fmt * world)
{
  contribarr tterm;
  double sum, sum2, sumc, sumterm, lterm;
  short i, j, k, lai;
  sitelike x1;
  option_fmt *opt;
  seqmodel_fmt *seq;
  opt = world->options;
  seq = world->data->seq;
  sum = 0.0;
  /*   y = v1 / seq->fracchange;
     lz = -y; */
  if (opt->rcategs == 1 && opt->categs == 1)
    {
      for (i = 0; i < seq->endsite; i++)
	{
	  memcpy (x1, xx1[i][0], sizeof (sitelike));
	  tterm[0] = seq->freqa * x1[0] + seq->freqc * x1[1] +
	    seq->freqg * x1[2] + seq->freqt * x1[3];
	  if (tterm[0] == 0.0)
	    {
	      fprintf (stderr, "Encountered tree incompatible with data\n");
	      exit (EXIT_FAILURE);
	    }
	  lterm = log (tterm[0]);
	  sum += seq->aliasweight[i] * lterm;
	}
      seq->like[0] = 1.0;
      for (i = 0; i < seq->sites[world->locus]; i++)
	{
	  sumc = opt->lambda * seq->like[0];
	  seq->nulike[0] = ((1.0 - opt->lambda) * seq->like[0] + sumc);
	  memcpy (seq->like, seq->nulike, sizeof (contribarr));
	}
      sum += log (seq->like[0]);
      return sum;
    }
  else
    {
      for (i = 0; i < seq->endsite; i++)
	{
	  k = seq->category[seq->alias[i] - 1] - 1;
	  for (j = 0; j < opt->rcategs; j++)
	    {
	      memcpy (x1, xx1[i][j], sizeof (sitelike));
	      tterm[j] = seq->freqa * x1[0] + seq->freqc * x1[1] +
		seq->freqg * x1[2] + seq->freqt * x1[3];
	      if (tterm[j] == 0.0)
		{
		  fprintf (stderr, "Encountered tree incompatible with data\n");
		  exit (EXIT_FAILURE);
		}
	    }
	  sumterm = 0.0;
	  for (j = 0; j < opt->rcategs; j++)
	    sumterm += opt->probcat[j] * tterm[j];
	  lterm = log (sumterm);
	  for (j = 0; j < opt->rcategs; j++)
	    seq->clai[j] = tterm[j] / sumterm;
	  memcpy (seq->contribution[i], seq->clai, sizeof (contribarr));
	  sum += seq->aliasweight[i] * lterm;
	}
      for (j = 0; j < opt->rcategs; j++)
	seq->like[j] = 1.0;
      for (i = 0; i < seq->sites[world->locus]; i++)
	{
	  sumc = 0.0;
	  for (k = 0; k < opt->rcategs; k++)
	    sumc += opt->probcat[k] * seq->like[k];
	  sumc *= opt->lambda;
	  if ((seq->ally[i] > 0) && (seq->location[seq->ally[i] - 1] > 0))
	    {
	      lai = seq->location[seq->ally[i] - 1];
	      memcpy (seq->clai, seq->contribution[lai - 1], sizeof (contribarr));
	      for (j = 0; j < opt->rcategs; j++)
		seq->nulike[j] = ((1.0 - opt->lambda) * seq->like[j] + sumc) * seq->clai[j];
	    }
	  else
	    {
	      for (j = 0; j < opt->rcategs; j++)
		seq->nulike[j] = ((1.0 - opt->lambda) * seq->like[j] + sumc);
	    }
	  memcpy (seq->like, seq->nulike, sizeof (contribarr));
	}
      sum2 = 0.0;
      for (i = 0; i < opt->rcategs; i++)
	sum2 += opt->probcat[i] * seq->like[i];
      sum += log (sum2);
      return sum;
    }
}


void
treeout (FILE * file, node * joint, node * p, long s)
{
  /* write out file with representation of final tree */
  static long col = 0;
  long w;
  double x;
  char migstring[30];
  if (p->type == 't')
    {
      translate (p->nayme, ' ', '_');
      fprintf (file, "%s", p->nayme);
      col += strlen (p->nayme);
    }
  else
    {
      putc ('(', file);
      col++;
      treeout (file, joint, crawlback (p->next), s);
      putc (',', file);
      col++;
      if (col > 80)
	{
	  putc ('\n', file);
	  col = 0;
	}
      treeout (file, joint, crawlback (p->next->next), s);
      putc (')', file);
      col++;
    }
  if (p == joint)
    {
      x = 0.0;
    }
  else
    {
      x = crawlback (p)->tyme - p->tyme;
    }
  if (x > 0.0)
    {
      w = (long) (0.4343 * log (x));
    }
  else
    {
      if (x == 0.0)
	w = 0;
      else
	w = (long) (0.4343 * log (-x)) + 1;
    }
  if (w < 0)
    w = 0;
  fprintf (file, ":%*.10f", (int) (w + 7), x /*< 10000. ? x : 10000. */ );
  col += w + 8;
  if (col > 80)
    {
      putc ('\n', file);
      col = 0;
    }
  if (p != joint)
    {
      p = showtop (p->back);
      while (p->type == 'm')
	{
	  sprintf (migstring, " [&M %li %li:%g]",
	     p->pop, p->actualpop, p->tyme - showtop (p->next->back)->tyme);
	  fprintf (file, "%s", migstring);
	  col += strlen (migstring) + 1;
	  if (col > 80)
	    {
	      putc ('\n', file);
	      col = 0;
	    }
	  p = showtop (p->back);
	}
    }
  else
    {
      fprintf (file, ";\n");
      col = 0;
    }
}				/* treeout */


void 
print_tree (world_fmt * world, long g, long *filepos)
{
  switch (world->options->treeprint)
    {
    case BEST:
      if (world->likelihood[g] > world->allikemax)
	{
	  if (world->allikemax == -DBL_MAX)
	    {
	      *filepos = ftell (world->treefile);
	    }
	  else
	    {
	      fseek (world->treefile, *filepos, SEEK_SET);
	    }
	  world->allikemax = world->likelihood[g];
	  fprintf (world->treefile, "[Comment: Locus %li, best log likelihood = %f]\n",
		   world->locus + 1, world->likelihood[g]);
	  treeout (world->treefile, crawlback (world->root->next),
		   crawlback (world->root->next), 0);
	}
      break;
    case ALL:
      fprintf (world->treefile, "[Comment: Locus %li, log likelihood = %f]\n",
	       world->locus + 1, world->likelihood[g]);
      treeout (world->treefile, crawlback (world->root->next),
	       crawlback (world->root->next), 0);
      break;
    case LASTCHAIN:
      if (world->in_last_chain)
	{
	  fprintf (world->treefile, "[Comment: Locus %li, log likelihood = %f]\n",
		   world->locus + 1, world->likelihood[g]);
	  treeout (world->treefile, crawlback (world->root->next),
		   crawlback (world->root->next), 0);
	}
      break;
    case NONE:
      break;
    default:
      break;
    }

}


void 
treereader (world_fmt * world)
{
  /* read a migration tree from the usertree and set up nodes and pointers */

  node **nodelist;
  char *nayme;
  char *temp, *temp2;
  long pop, w, zz, z = 0, zzz = 0;
  world->nodep = (node **) calloc (1, world->sumtips * sizeof (node *));
  temp = (char *) malloc (LINESIZE * sizeof (char));
  temp2 = (char *) malloc (LINESIZE * sizeof (char));
  treeread (world->data->utreefile, &(world->root), NULL);
  /*fscanf(world->data->utreefile, "%*[^\n]"); */
  /*  getc(world->data->utreefile); */
  length_to_times (world->root->next->back);
  nodelist = (node **) calloc (1, sizeof (node *) * (world->sumtips + 1));
  pop = find_firstpop (world->root);
  set_tree_pop (world->root, &pop);
  allocate_x (world->root, world, world->options->datatype, WITHTIPS);
  find_tips (world->root, nodelist, &z);
  for (pop = 0; pop < world->numpop; pop++)
    {
      for (w = 0; w < world->data->numind[pop][world->locus]; w++)
	{
	  strcpy (temp2, world->data->indnames[pop][w]);
	  temp2[strcspn (temp2, " ")] = '\0';
	  sprintf (temp, "%li%s", pop, temp2);
	  for (zz = 0; zz < z; zz++)
	    {
	      nayme = nodelist[zz]->nayme;
	      if (!strcmp (temp, nayme) || !strcmp (temp2, nayme))
		{
		  world->nodep[zzz++] = nodelist[zz];
		  break;;
		}
	    }
	}
    }
  free (nodelist);
  free (temp);
  free (temp2);
}


char
processlength (FILE * file, node ** p)
{
  char ch;
  long digit, ordzero;
  double valyew, divisor;
  boolean pointread, minusread;

  ordzero = '0';
  pointread = FALSE;
  minusread = FALSE;
  valyew = 0.0;
  divisor = 1.0;
  ch = getc (file);
  digit = ch - ordzero;
  while (((unsigned long) digit <= 9) | (ch == '.') || (ch == '-'))
    {
      if (ch == '.')
	pointread = TRUE;
      else if (ch == '-')
	minusread = TRUE;
      else
	{
	  valyew = valyew * 10.0 + digit;
	  if (pointread)
	    divisor *= 10.0;
	}
      ch = getc (file);
      digit = ch - ordzero;
    }
  if (!minusread)
    (*p)->length = valyew / divisor;
  else
    (*p)->length = 0.0;
  return ch;
}

void
treeread (FILE * file, node ** pp, node * q)
{
  node *p;
  char ch = getc (file);
  while (ch != ';')
    {
      switch (ch)
	{
	case '(':
	  p = create_interior_node (&q);
	  q = p->next;
	  ch = getc (file);
	  break;
	case ',':
	  q = q->next;
	  if (q->top)
	    {
	      fprintf (stderr, "Multifurcation handling not yet installed");
	      exit (-1);
	    }
	  ch = getc (file);
	  break;
	case ')':
	  p = showtop (q);
	  q = p->back;
	  ch = getc (file);
	  break;
	case ' ':
	case '\n':
	case '\t':
	  ch = getc (file);
	  break;
	case ':':
	  ch = processlength (file, &p);
	  break;
	case '[':
	  ch = processbracket (file, &p);
	  q->back = p;
	  p->back = q;
	  break;
	default:
	  p = create_tip_node (file, &q, &ch);
	  break;
	}
    }
  p->length = 10000.;
  (*pp) = showtop (p->back);
  fscanf (file, "%*[^\n]");
  getc (file);
}

void 
length_to_times (node * p)
{
  node *q;
  if (!(p)->tip)
    {
      length_to_times ((p)->next->back);
      if ((p)->type == 'i')
	length_to_times ((p)->next->next->back);
    }
  q = showtop ((p)->back);
  q->tyme = q->next->tyme = q->next->next->tyme = (p)->tyme + (p)->length;
}

void 
find_tips (node * p, node ** nodelist, long *z)
{
  if (p->type == 't')
    {
      nodelist[(*z)++] = p;
    }
  else
    {
      if (p->next->back != NULL)
	find_tips (crawlback (p->next), nodelist, z);
      if (p->next->next->back != NULL)
	find_tips (crawlback (p->next->next), nodelist, z);
    }
}

long 
find_firstpop (node * p)
{
  static boolean found = FALSE;
  static long pop = -1;
  if (p->type == 'm')
    {
      found = TRUE;
      pop = p->pop;
    }
  else
    {
      if (p->next->back != NULL)
	{
	  find_firstpop (p->next->back);
	  if (found)
	    return pop;
	}
      if (p->next->next->back != NULL)
	find_firstpop (p->next->next->back);
    }
  return pop;
}

/* touches only coalescent nodes! migration nodes are already set */
void 
set_tree_pop (node * p, long *pop)
{
  if (p->type != 'r')
    {
      (*pop) = (showtop (p->back)->actualpop != *pop) ? showtop (p->back)->actualpop : *pop;
    }
  p->actualpop = p->pop = *pop;
  if (!p->tip)
    {
      if (p->next->back != NULL)
	{
	  set_tree_pop (crawlback (p->next), pop);
	}
      if (p->type != 'm' && p->next->next->back != NULL)
	{
	  set_tree_pop (crawlback (p->next->next), pop);
	}
    }
}


node *
create_interior_node (node ** q)
{
  node *p;
  p = allocate_nodelet (3, 'i');
  p->top = TRUE;
  p->back = *q;
  if ((*q) == NULL)
    create_root_node (&p);
  else
    (*q)->back = p;
  return p;
}

node *
create_root_node (node ** q)
{
  node *p;
  p = allocate_nodelet (3, 'r');
  p->top = TRUE;
  p->next->back = *q;
  (*q)->back = p->next;
  return p;
}


node *
create_tip_node (FILE * file, node ** q, char *ch)
{
  node *p;
  char c;
  char *nayme;
  long nl;
  long i = 1;
  nayme = (char *) calloc (1, sizeof (char) * 200);
  nayme[0] = (*ch);
  while (strchr ("[):;,\t\n", (int) (c = getc (file))) == NULL)
    nayme[i++] = c;
  nayme[i] = '\0';
  p = allocate_nodelet (1, 't');
  nl = strlen (nayme);
  p->nayme = (char *) calloc (1, sizeof (char) * (nl + 1));
  p->top = TRUE;
  p->tip = TRUE;
  strcpy (p->nayme, nayme);
  p->back = *q;
  (*q)->back = p;
  free (nayme);
  (*ch) = c;
  return p;
}

char 
processbracket (FILE * file, node ** p)
{
  long pop1, pop2;
  double utime;
  char c;
  c = getc (file);
  if (c == '&')
    {
      c = getc (file);
      switch (c)
	{
	case 'M':
	  fscanf (file, "%li %li:%lf", &pop1, &pop2, &utime);
	  c = getc (file);
	  (*p) = add_migration (*p, pop1, pop2, utime);
	  break;
	default:
	  while (c != ']')
	    c = getc (file);
	  break;
	}
    }
  else
    {
      while (c != ']')
	c = getc (file);
    }
  c = getc (file);
  return c;
}


node *
add_migration (node * p, long from, long to, double utime)
{
  node *tmp;
  tmp = allocate_nodelet (2, 'm');
  tmp->top = TRUE;
  tmp->next->back = p;
  p->back = tmp->next;
  tmp->length = p->length - utime;
  p->length = utime;
  tmp->pop = tmp->next->pop = from;
  tmp->actualpop = tmp->next->actualpop = to;
  return tmp;
}

void 
allocate_x (node * p, world_fmt * world, char datatype, boolean withtips)
{
  if (p->type != 't')
    {
      if (p->next->back != NULL)
	allocate_x (crawlback (p->next), world, datatype, withtips);
      if (p->next->next->back != NULL)
	allocate_x (crawlback (p->next->next), world, datatype, withtips);
      if (datatype == 's')
	{
	  alloc_seqx (world, p);
	}
      else
	{
	  if (world->options->datatype == 's')	/* this setup is used in fitch() */
	    p->x.a = (double *) calloc (1, world->sumtips * sizeof (double));
	  else
	    p->x.a = (double *) calloc (1, MAX (world->sumtips, world->data->maxalleles[world->locus]) * sizeof (double));
	}
    }
  else
    {
      if (withtips)
	{
	  if (datatype == 's')
	    {
	      alloc_seqx (world, p);
	    }
	  else
	    {
	      p->x.a = (double *) calloc (1, world->data->maxalleles[world->locus] * sizeof (double));
	    }
	}
    }
}
/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 W O R L D   R O U T I N E S 

 creates tree structures,
 reads tree [has to be done
 
 prints results,
 and finally helps to destroy itself.
                                                                                                               
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: world.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"
#include "mcmc.h"
#include "world.h"
#include "fst.h"
#include "random.h"
#include "parameter.h"
#include "combine.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

#define PLANESIZE 36
#define PLANEBIGTICKS 6
#define PLANEBIGTICKVALUES {-3, -2, -1, 0, 1, 2}
#define PLANETICKS   {0.001, 0.0013895, 0.0019307, 0.0026827, 0.00372759, \
                     0.00517947, 0.00719686, 0.01, 0.013895, 0.019307, \
                     0.026827, 0.0372759, 0.0517947, 0.0719686, 0.1, \
                     0.13895, 0.19307, 0.26827, 0.372759, 0.517947, \
                     0.719686, 1., 1.3895, 1.9307, 2.6827, 3.72759, \
                     5.17947, 7.19686, 10., 13.895, 19.307, 26.827, 37.2759, \
                     51.7947, 71.9686, 100.}
#define CONTOURLEVELS 8
#define CONTOURS_LOCUS {0.0,-3.35670/2., -9.48773/2., -13.2767/2.,\
                        0.0,-3.35670/2., -9.48773/2., -13.2767/2.}
#define CONTOURS_LOCI  {0.0,-4.35146/2., -11.0705/2., -15.0863/2.,\
                        0.0,-4.35146/2., -11.0705/2., -15.0863/2.}

/* prototypes ------------------------------------------- */
void create_world (world_fmt ** world);
void init_world (world_fmt * world, data_fmt * data, option_fmt * options);
void calc_simple_param (world_fmt * world);
void print_menu_locus (option_fmt * options, long locus);
void print_menu_chain (char type, long chain, long steps, world_fmt * world);
void set_bounds (long *increment, long *steps, long *chain,
		 const option_fmt * options, const char type);
void burnin_chain (world_fmt * world);
void copy_atl (world_fmt * world, timearchive_fmt * old, timearchive_fmt * new, long steps);
void print_simresults (world_fmt * world);
void print_list (world_fmt * world);
void plot_surface (world_fmt * world, char ***plane, long x);
void print_alpha_curve (world_fmt * world, timearchive_fmt * atl);
void create_loci_plot (world_fmt * world, char **plane, timearchive_fmt * atl, long loci);
void create_locus_plot (world_fmt * world, char **plane, tarchive_fmt * tl, nr_fmt * nr, long G);
void cleanup_world (world_fmt * world, long locus);
void apocalypse_now (world_fmt * world);
/* private functions */
void create_timearchive (timearchive_fmt ** atl, long loci, long samples, long numpop);
void create_plotplane (world_fmt * world);
void create_cov (world_fmt * world);
void print_menu_equilib (option_fmt * options);
void print_finish (world_fmt * world, long filepos);
void copy_time (world_fmt * world, timelist_fmt * ltl, long from, long to, long np);
void archive_timelist (tarchive_fmt * atl, vtlist * tl, long T, long np);
void copy_timelist (tarchive_fmt * from, tarchive_fmt * to, long np);
long whichp (long from, long to, long pop);
long whichl (long from, long to, long pop);
/* long calc_T(timearchive_fmt * old, long steps); */
void print_results (world_fmt * world);
void print_fst (world_fmt * world, double **fstparam);
void prepare_print_nu (double nu, char *str);
void prepare_print_nm (double nm, double nmu, char *strllike);
void print_menu_coalnodes (world_fmt * world, long G);
void print_menu_createplot (void);
void calc_loci_plane (world_fmt * world, nr_fmt * nr, timearchive_fmt * atl, double **pl, long loci, double contours[]);
void calc_locus_plane (world_fmt * world, nr_fmt * nr, tarchive_fmt * tl, long G,
		       double **pl, double contours[]);
void fill_plotplane (char **plane, double **pl, double contours[], plotmax_fmt * planemax);
void print_mathematica (world_fmt * world, double **plane, long x, long y);
void print_cov (world_fmt * world, long numpop, long loci, double ***cov);
void print_cov_table (FILE * outfile, long locus, world_fmt * world, double *corr, long addvar);
void free_seqx (node * p, long sites);
void free_x (node * p);
void free_tree (node * p);
void free_nodelet (node * p, long num);
void increase_timearchive (world_fmt * world, long locus, long sample, long numpop);
void test_locus_like (double *param0, double *param1, long df, long locus, world_fmt * world,
		      boolean withhead, char *this_string);
void test_loci_like (double *param0, double *param1, long df, long loci, world_fmt * world,
		     boolean withhead, char *this_string);

double chisquare (long df, double alpha);
long set_test_param (double *param, char *strp, world_fmt * world, long lrline, long locus);
void print_CV (world_fmt * world);
/*======================================================*/
void 
create_world (world_fmt ** world)
{
  (*world) = (world_fmt *) calloc (1, sizeof (world_fmt));
  (*world)->param_like = DBL_MAX;
}

void 
init_world (world_fmt * world, data_fmt * data, option_fmt * options)
{
  long locus;
  world->options = options;
  world->data = data;
  world->loci = data->loci;
  world->skipped = 0;
  world->numpop = data->numpop;
  world->numpop2 = 2 * world->numpop;
  getseed (options);
  create_timearchive (&(world->atl), data->loci,
		      SAMPLETREE_GUESS, data->numpop);
  create_plotplane (world);
  create_cov (world);
  world->likelihood = (double *) calloc (1, sizeof (double) * SAMPLETREE_GUESS);
  world->lineages = (long *) calloc (1, sizeof (long) * world->numpop);
  world->param0 = (double *) calloc (1, sizeof (double) * world->numpop2);
  world->param00 = (double *) calloc (1, sizeof (double) * world->numpop2);
  switch (options->datatype)
    {
    case 's':
    case 'a':
      world->fstparam = (double **) calloc (1, sizeof (double *) * (world->loci + 1));
      for (locus = 0; locus < world->loci + 1; locus++)
	world->fstparam[locus] = (double *) calloc (1,
				      sizeof (double) * world->numpop2 * 2);
      break;
    case 'm':
      world->options->steps = (double **)
	calloc (1, sizeof (double *) * world->options->micro_stepnum);
      for (locus = 0; locus < world->options->micro_stepnum; locus++)
	world->options->steps[locus] = (double *) calloc (1,
			   sizeof (double) * world->options->micro_stepnum);
      break;
    }
  /* tree and treetimes are not yet allocated */
}

void 
calc_simple_param (world_fmt * world)
{
  switch (world->options->datatype)
    {
    case 'a':
      calc_fst (world);
      break;
    case 's':
      calc_fst (world);
      break;
    default:
      break;
    }
}

void 
set_bounds (long *increment, long *steps, long *chains,
	    const option_fmt * options, const char type)
{
  switch (type)
    {
    case 's':
      *increment = options->sincrement;
      *steps = options->ssteps;
      *chains = options->schains;
      break;
    case 'l':
      *increment = options->lincrement;
      *steps = options->lsteps;
      *chains = options->lchains;
      break;
    default:
      fprintf (stderr, "this is either a short nor a long chain??\n");
      exit (EXIT_FAILURE);
      break;
    }
}

void 
burnin_chain (world_fmt * world)
{
  long step;
  if (world->options->burn_in == 0)
    return;
  print_menu_equilib (world->options);
  for (step = 0; step < world->options->burn_in; step++)
    {
      metropolize (world, 0);
    }
}

void 
copy_atl (world_fmt * world, timearchive_fmt * old, timearchive_fmt * new, long steps)
{
  long j;
  long numpop = world->numpop;

  increase_timearchive (world, world->locus + 1, steps, numpop);

  new->T = /*calc_T(old, steps); */ old->T;
  new->param_like = world->param_like;
  new->sumtips = world->sumtips;
  new->numpop = world->data->numpop;
  memcpy (new->param, world->param0, sizeof (double) * 2 * numpop);
  memcpy (new->param0, world->param00, sizeof (double) * 2 * numpop);
  memcpy (new->likelihood, world->likelihood, sizeof (double) * steps);
  for (j = 0; j < new->T; j++)
    {
      new->tl[j].copies = old->tl[j].copies;
      copy_timelist (&old->tl[j], &new->tl[j], numpop);
    }
}

void 
print_simresults (world_fmt * world)
{
  long l = 1;
  double ft1 = 0., ft2 = 0., fm1 = 0., fm2 = 0., lt1 = 0., lt2 = 0., lm1 = 0.,
    lm2 = 0.;
  switch (world->options->datatype)
    {
    case 's':
      fprintf (world->outfile,
	       " %3li %li %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f\n",
	       l, world->atl[l].trials, world->atl[l].param[0],
	       world->atl[l].param[1], world->atl[l].param[2],
	       world->atl[l].param[3], world->options->gamma ?
	       world->atl[l].param[4] : 999999., world->atl[l].param_like);
      break;
    case 'm':
      fprintf (world->outfile,
	       " %3li %li %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f\n",
	       l, world->atl[l].trials, world->atl[l].param[0],
	       world->atl[l].param[1], world->atl[l].param[2],
	       world->atl[l].param[3], world->options->gamma ?
	       world->atl[l].param[4] : 999999., world->atl[l].param_like);
      break;
    case 'a':
      for (l = 0; l < world->loci + 1; l++)
	{
	  if (world->fstparam[l][0] != -999.)
	    {
	      ft1 += world->fstparam[l][0];
	      lt1++;
	    }
	  if (world->fstparam[l][1] != -999.)
	    {
	      ft2 += world->fstparam[l][1];
	      lt2++;
	    }
	  if (world->fstparam[l][2] != -999.)
	    {
	      fm1 += world->fstparam[l][2];
	      lm1++;
	    }
	  if (world->fstparam[l][3] != -999.)
	    {
	      fm2 += world->fstparam[l][3];
	      lm2++;
	    }
	}
      l = world->loci + 1;
      if (lt1 == 0.)
	lt1 = 1.;
      if (lt2 == 0.)
	lt2 = 1.;
      if (lm1 == 0.)
	lm1 = 1.;
      if (lm2 == 0.)
	lm2 = 1.;
      fprintf (world->outfile,
	       " %3li %li %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f\n",
	       l - 1, world->atl[l].trials, world->atl[l].param[0],
	       world->atl[l].param[1], world->atl[l].param[2],
	       world->atl[l].param[3], world->options->gamma ?
	       world->atl[l].param[4] : 999999., world->atl[l].param_like,
	       ft1 / lt1, ft2 / lt2, fm1 / lm1, fm2 / lm2);
      break;
    }
}

void 
print_list (world_fmt * world)
{

  /*  long i,locus, df; */
  double *param0;
  param0 = (double *) calloc (1, sizeof (double) * (world->numpop2 + 1));

  print_results (world);
  if ((world->options->datatype == 'a') || (world->options->datatype == 's'))
    print_fst (world, world->fstparam);
  if (world->options->plot)
    plot_surface (world, world->plane, PLANESIZE + 2);
  print_alpha_curve (world, world->atl);
#ifdef TESTING
  /* usage of this section is discouraged, if you understand what's going on here
     and want to use it please contact me (beerli@genetics.washington.edu) */
  if (world->options->lratio->counter > 0)
    {
      if (world->loci - world->skipped > 1 && world->options->lratio->data[0].type == MEAN)
	{
	  df = set_test_param (param0, world->options->lratio->data[0].value, world, 0, -1);
	  test_loci_like (param0,
			  world->atl[world->loci + 1].param, df,
			  world->loci, world, HEADER, NULL);
	}
      else
	{
	  for (locus = 0; locus < world->loci; locus++)
	    {
	      df = set_test_param (param0, world->options->lratio->data[0].value, world, 0, locus);
	      test_locus_like (param0,
			       world->atl[locus + 1].param, df,
			       locus, world, HEADER, NULL);
	    }
	}
    }
  for (i = 1; i < world->options->lratio->counter; i++)
    {
      if (world->loci - world->skipped > 1 && world->options->lratio->data[i].type == MEAN)
	{
	  df = set_test_param (param0, world->options->lratio->data[i].value, world, i, -1);
	  test_loci_like (param0,
			  world->atl[world->loci + 1].param, df,
			  world->loci, world, FALSE, NULL);
	}
      else
	{
	  for (locus = 0; locus < world->loci; locus++)
	    {
	      df = set_test_param (param0, world->options->lratio->data[i].value, world, i, locus);
	      test_locus_like (param0,
			       world->atl[locus + 1].param, df,
			       locus, world, FALSE, NULL);
	    }
	}
    }
#endif
  print_cov (world, world->numpop, world->loci, world->cov);
#ifdef TESTING
  print_CV (world);
#endif
}

void 
plot_surface (world_fmt * world, char ***plane, long x)
{
  long i, locus;
  long loci = world->loci;
  FILE *outfile = world->outfile;
  if (world->options->progress)
    fprintf (stdout, "           Plotting the likelihood surfaces\n");
  PAGEFEED;
  fprintf (outfile, "Log-Likelihood surface of 2 populations\n");
  fprintf (outfile, "---------------------------------------\n\n");
  fprintf (outfile, "Legend:\n");
  fprintf (outfile, "   X = Maximum likelihood\n");
  fprintf (outfile, "   * = in approximative 50%% confidence limit\n");
  fprintf (outfile, "   + = in approximative 95%% confidence limit\n");
  fprintf (outfile, "   - = in approximative 99%% confidence limit\n");
  for (locus = 0; locus < loci; locus++)
    {
      if (world->data->skiploci[locus])
	{
	  continue;
	}
      fprintf (outfile, "\n\nLocus %li\nx-axis= 4Nm [effective population size * migration rate],\n", locus + 1);
      fprintf (outfile, " y-axis = Theta,\nunits = log10\nMaximum log likelihood on plot\n");
      fprintf (outfile, "  Population 1: 4Nm=%f, Theta=%f, log likelihood=%f\n",
	       world->plotmax[locus].x1, world->plotmax[locus].y1,
	       world->plotmax[locus].l1);
      fprintf (outfile, "  Population 2: 4Nm=%f, Theta=%f, log likelihood=%f\n",
	       world->plotmax[locus].x2, world->plotmax[locus].y2,
	       world->plotmax[locus].l2);
      fprintf (outfile, "\n            Population 1                         Population 2\n\n");
      for (i = x; i >= 0; i--)
	fprintf (outfile, "%s\n", plane[locus][i]);
      fprintf (outfile, "%s\n", plane[locus][x]);
      PAGEFEED;
    }
  if (loci - world->skipped > 1)
    {
      fprintf (outfile, "\nOver all loci\n\n");
      locus = loci;
      fprintf (outfile, "x-axis= Nm [effective population size * migration rate],\n");
      fprintf (outfile, "y-axis = Theta,\nunits = log10\nMaximum log likelihood on plot\n");
      fprintf (outfile, "  Population 1: 4Nm=%f, Theta=%f, log likelihood=%f\n",
	       world->plotmax[locus].x1, world->plotmax[locus].y1,
	       world->plotmax[locus].l1);
      fprintf (outfile, "  Population 2: 4Nm=%f, Theta=%f, log likelihood=%f\n",
	       world->plotmax[locus].x2, world->plotmax[locus].y2,
	       world->plotmax[locus].l2);
      fprintf (outfile, "\n            Population 1                         Population 2\n\n");
      for (i = x; i >= 0; i--)
	fprintf (outfile, "%s\n", plane[locus][i]);
      PAGEFEED;
    }
}

void 
print_alpha_curve (world_fmt * world, timearchive_fmt * atl)
{
  const double alphas[19] =
  {0.0001, 0.0002, 0.0005,
   0.001, 0.002, 0.005,
   0.01, 0.02, 0.05,
   0.1, 0.2, 0.5,
   1., 2., 5.,
   10., 20., 50., 100.};
  double contours[CONTOURLEVELS] = CONTOURS_LOCI;
  boolean once = FALSE;
  long g, a, gmax = 1, loci = world->loci;
  double likes[21];
  char confidence[21];
  nr_fmt *nr;
  FILE *outfile = world->outfile;
  if (world->options->gamma && loci - world->skipped > 1)
    {
      nr = (nr_fmt *) calloc (1, sizeof (nr_fmt));
      for (g = 0; g < CONTOURLEVELS; g++)
	{
	  contours[g] += atl[loci + 1].param_like;
	}
      for (g = 1; g < loci + 1; g++)
	{
	  if (gmax < atl[g].T)
	    gmax = atl[g].T;
	}
      create_nr (nr, atl[loci].numpop, gmax);
      nr->skiploci = world->data->skiploci;
      memcpy (nr->param, atl[loci + 1].param, sizeof (double) * 5);
      if (world->options->progress)
	fprintf (stdout, "           Printing Inv(alpha) x Log(likelihood)\n");
      gmax = 0;
      calc_gamma (nr);
      likes[20] = calc_loci_like (nr, atl, loci, TRUE);
      for (a = 0; a < 19; a++)
	{
	  nr->param[4] = alphas[a];
	  likes[a] = calc_loci_like (nr, atl, loci, TRUE);
	  if (atl[loci + 1].param[4] < alphas[a] && !once)
	    {
	      gmax = a - 1;
	      once = TRUE;
	    }
	}
      for (a = 0; a < 19; a++)
	{
	  if (likes[a] < contours[1])
	    {
	      if (likes[a] < contours[2])
		{
		  if (likes[a] < contours[3])
		    {
		      confidence[a] = ' ';
		    }
		  else
		    {
		      confidence[a] = '-';
		    }
		}
	      else
		{
		  confidence[a] = '+';
		}
	    }
	  else
	    {
	      if (likes[a] < contours[0])
		confidence[a] = '*';
	      else
		{
		  confidence[a] = 'X';
		}
	    }
	}
      fprintf (outfile, "Log-Likleihood curve for the shape ");
      fprintf (outfile, "parameter Inv(alpha) [square(CV(mu))]\n");
      fprintf (outfile, "-----------------------------------");
      fprintf (outfile, "--------------------------\n\n");
      fprintf (outfile, "Legend:\n");
      fprintf (outfile, "   X = Maximum likelihood\n");
      fprintf (outfile, "   * = in approximative 50%% confidence limit\n");
      fprintf (outfile, "   + = in approximative 95%% confidence limit\n\n");
      fprintf (outfile, "   - = in approximative 99%% confidence limit\n\n");
      fprintf (outfile, "Inv(Alpha)     Log(Likelihood)   Confidence limit\n");
      fprintf (outfile, "-------------------------------------------------\n");
      if (gmax < 0)
	{
	  fprintf (outfile, "%10.5g     % 20.5g            X\n", atl[loci + 1].param[4],
		   likes[20]);
	}
      for (a = 0; a < 19; a++)
	{
	  fprintf (outfile, "% 10.5f     % 20.5g            %c\n", alphas[a],
		   likes[a], confidence[a]);
	  if (gmax == a)
	    fprintf (outfile, "% 10.5f     % 20.5g            X\n", atl[loci + 1].param[4],
		     likes[20]);
	}
      if (gmax >= 20)
	{
	  fprintf (outfile, "% 10.5g     % 20.5g            X\n", atl[loci + 1].param[4],
		   likes[20]);
	}
      destroy_nr (nr);
    }
}

void 
create_loci_plot (world_fmt * world, char **plane, timearchive_fmt * atl, long loci)
{
  long intervals = PLANESIZE;
  long i, g = 1;
  nr_fmt *nr;
  double **pl;
  double contours[CONTOURLEVELS] = CONTOURS_LOCI;

  if (world->options->progress)
    print_menu_createplot ();
  nr = (nr_fmt *) calloc (1, sizeof (nr_fmt));
  for (i = 1; i < loci + 1; i++)
    {
      if (g < atl[i].T)
	g = atl[i].T;
    }
  create_nr (nr, atl[loci].numpop, g);
  nr->skiploci = world->data->skiploci;
  pl = (double **) calloc (1, sizeof (double *) * intervals);
  for (i = 0; i < intervals; i++)
    {
      pl[i] = (double *) calloc (1, sizeof (double) * 2 * intervals);
    }
  calc_loci_plane (world, nr, atl, pl, loci, contours);
  fill_plotplane (plane, pl, contours, &world->plotmax[world->loci]);
  destroy_nr (nr);
  print_mathematica (world, pl, intervals, intervals);
  for (i = 0; i < intervals; i++)
    {
      free (pl[i]);
    }
  free (pl);
}


void 
create_locus_plot (world_fmt * world, char **plane, tarchive_fmt * tl, nr_fmt * nr, long G)
{
  long intervals = PLANESIZE;
  long i;
  double **pl;
  double contours[CONTOURLEVELS] = CONTOURS_LOCI;
  if (world->options->verbose)
    print_menu_createplot ();
  pl = (double **) calloc (1, sizeof (double *) * intervals);
  for (i = 0; i < intervals; i++)
    {
      pl[i] = (double *) calloc (1, sizeof (double) * 2 * intervals);
    }
  calc_locus_plane (world, nr, tl, G, pl, contours);
  fill_plotplane (plane, pl, contours, &world->plotmax[world->locus]);
  print_mathematica (world, pl, intervals, intervals);
  for (i = 0; i < intervals; i++)
    {
      free (pl[i]);
    }
  free (pl);
}

void 
cleanup_world (world_fmt * world, long locus)
{
  if (locus >= 0)
    {
      if (world->options->datatype == 's')
	free_seqx (world->root, world->data->seq->endsite);
      else
	free_x (world->root);
      free_tree (world->root);
      free (world->nodep);
    }
}


void 
apocalypse_now (world_fmt * world)
{
  free (world);
  raise (SIGILL);
}

/* printing stuff =========================================== */
void 
print_menu_locus (option_fmt * options, long locus)
{
  if (options->progress)
    fprintf (stdout, "Locus %-3li:\n", locus + 1);
}

void 
print_menu_chain (char type, long chain, long steps, world_fmt * world)
{
  char strllike[LINESIZE];
  char nowstr[LINESIZE];
#ifdef _GNU_
  double likes[steps];
#else
  double *likes;
#endif
  if (world->options->progress)
    {
      print_llike (world->param_like, strllike);
      get_time (nowstr, "%H:%M:%S");
      if (chain == FIRSTCHAIN)
	{
	  fprintf (stdout, "%s   Start conditions: theta={%5.5f,%5.5f}, M={%5.5f,%5.5f},\n",
		   nowstr,
		   world->param0[0], world->param0[1],
		   world->param0[2], world->param0[3]);
	  fprintf (stdout, "           Start-tree-log(L)=%f\n", world->likelihood[0]);
	}
      else
	{
	  fprintf (stdout, "%s   %*s chain %3li: lnL=%s,\n           theta={%5.5f,%5.5f}, M={%5.5f,%5.5f}\n",
		   nowstr, type == 's' ? 5 : 4, type == 's' ? "Short" : "Long", chain + 1, strllike,
		   world->param0[0], world->param0[1],
		   world->param0[2], world->param0[3]);
	  if (world->options->verbose)
	    {
#ifndef _GNU_
	      likes = (double *) malloc (sizeof (double) * steps);
#endif
	      memcpy (likes, world->likelihood, steps * sizeof (double));
	      qsort ((void *) likes, steps, sizeof (double), numcmp);
	      fprintf (stdout, "           Sampled tree-log(L)={%f .. %f}, best in group =%f\n", likes[0], likes[steps - 1], world->maxdatallike);
#ifndef _GNU_
	      free (likes);
	    }
#endif
	}
    }
}

void 
copy_time (world_fmt * world, timelist_fmt * ltl, long from, long to, long np)
{
  long T;
  tarchive_fmt *atl = world->atl[0].tl;

  if (from == -1)
    {
      ltl[0].copies = 1;
      T = ltl[0].T - 1;
      atl = world->atl[0].tl;
      atl[0].copies = ltl[0].copies;
      increase_timearchive (world, 0, 1, np);
      archive_timelist (&(atl[0]), ltl[0].tl, T, np);
      return;
    }
  if (from == to)
    {
      ltl[0].copies += 1;
      atl[from].copies = ltl[0].copies;
    }
  else
    {
      T = ltl[0].T - 1;
      increase_timearchive (world, 0, to, np);
      atl = world->atl[0].tl;
      atl[to].copies = ltl[0].copies = 1;
      world->likelihood[to] = world->likelihood[from];
      archive_timelist (&atl[to], ltl[0].tl, T, np);

    }
}


/*private functions========================================== */
/* ---------------------------------------------------------
   creates memory for archive of timelists for each locus
   the "locus 0" is reserved for for the current locus, this
   fake locus is use to speed up the NR-estimation for all chains
   whereas the loci 1 .. n are used to save the results of the last
   chain for the combined estimate at the end of the program */
void 
create_timearchive (timearchive_fmt ** atl, long loci, long samples, long numpop)
{
  long i, j;
  (*atl) = (timearchive_fmt *) calloc (1, sizeof (timearchive_fmt) * (2 + loci));
  for (i = 0; i < loci + 2; i++)
    {
      (*atl)[i].param = (double *) calloc (1, numpop * 2 * sizeof (double));
      (*atl)[i].param0 = (double *) calloc (1, numpop * 2 * sizeof (double));
      (*atl)[i].likelihood = (double *) calloc (1, (1 + samples) * sizeof (double));
      (*atl)[i].tl = (tarchive_fmt *) calloc (1, samples * sizeof (tarchive_fmt));
      (*atl)[i].T = (*atl)[i].allocT = samples;
      for (j = 0; j < samples; j++)
	{
	  (*atl)[i].tl[j].km = (double *) calloc (1, sizeof (double) * numpop);
	  (*atl)[i].tl[j].kt = (double *) calloc (1, sizeof (double) * numpop);
	  (*atl)[i].tl[j].p = (long *) calloc (1, sizeof (long) * numpop);
	  (*atl)[i].tl[j].l = (long *) calloc (1, sizeof (long) * numpop);
	}
    }
}

void 
increase_timearchive (world_fmt * world, long locus, long sample, long numpop)
{
  long i = locus, j, oldT = 0, size;
  if (sample >= world->atl[i].allocT)
    {
      oldT = world->atl[i].allocT;
      world->atl[i].allocT = MAX (sample + 1, world->atl[i].allocT + world->atl[i].allocT / 4.);
      size = world->atl[i].allocT;
      world->atl[i].tl = (tarchive_fmt *) realloc (world->atl[i].tl,
					      size * sizeof (tarchive_fmt));
      if (i == 0)
	world->likelihood = (double *) realloc (
			   world->likelihood, sizeof (double) * (1 + size));
      world->atl[i].likelihood = (double *) realloc (
		    world->atl[i].likelihood, sizeof (double) * (1 + size));
      for (j = oldT; j < world->atl[i].allocT; j++)
	{
	  world->atl[i].tl[j].km = (double *) calloc (1, sizeof (double) * numpop);
	  world->atl[i].tl[j].kt = (double *) calloc (1, sizeof (double) * numpop);
	  world->atl[i].tl[j].p = (long *) calloc (1, sizeof (long) * numpop);
	  world->atl[i].tl[j].l = (long *) calloc (1, sizeof (long) * numpop);
	}
      world->atl[i].T = sample;
    }
  else
    {
      world->atl[i].T = sample;
    }
}

void 
create_plotplane (world_fmt * world)
{
  short locus, i;
  world->plane = (char ***) calloc (1, sizeof (char **) * (world->loci + 1));
  world->plotmax = (plotmax_fmt *) calloc (1, sizeof (plotmax_fmt) * (world->loci + 1));
  for (locus = 0; locus < world->loci + 1; locus++)
    {
      world->plane[locus] =
	(char **) calloc (1, sizeof (char *) * (PLANESIZE + 3));
      for (i = 0; i < PLANESIZE + 3; i++)
	{
	  world->plane[locus][i] =
	    (char *) calloc (1, sizeof (char) * (PLANESIZE + PLANESIZE + 20));
	}
    }
}

void 
create_cov (world_fmt * world)
{
  short locus, i;
  world->cov = (double ***) calloc (1, sizeof (double **) * (world->loci + 1));
  for (locus = 0; locus < world->loci + 1; locus++)
    {
      world->cov[locus] = (double **) calloc (1, sizeof (double *)
					      * (world->numpop * 2 + 1));
      for (i = 0; i < world->numpop * 2 + 1; i++)
	{
	  world->cov[locus][i] = (double *) calloc (1, sizeof (double) * (world->numpop * 2 + 1));
	}
    }
}

void 
print_menu_equilib (option_fmt * options)
{
  char nowstr[LINESIZE];
  if (options->progress)
    {
      get_time (nowstr, "%H:%M:%S");
      fprintf (stdout, "%s   Equilibrate tree (first %i trees are not used)\n", nowstr,
	       options->burn_in);
    }
}

void 
print_finish (world_fmt * world, long filepos)
{
  char nowstr[LINESIZE];
  if (world->options->progress)
    {
      get_time (nowstr, "%H:%M:%S");
      fprintf (stdout, "%s   Program finished\n", nowstr);
    }
  get_time (nowstr, "%c");
  if (nowstr[0] != '\0')
    {
      if (filepos > 0)
	fseek (world->outfile, filepos, SEEK_SET);
      fprintf (world->outfile, "         finished at %s\n", nowstr);
    }

}

void 
archive_timelist (tarchive_fmt * atl, vtlist * tl, long T, long np)
{
  long j, i;
  double t;
  double line1 /*, line2 */ ;
  for (i = 0; i < np; i++)
    {
      line1 = tl[0].lineages[i];
      /* line2=0.;
         for(z=0; z< np; z++){
         if(i!=z)
         line2 +=  tl[0].lineages[z];
         }
       */
      atl->km[i] = line1 * tl[0].age;
      atl->kt[i] = line1 * (line1 - 1) * tl[0].age;
      atl->p[i] = whichp (tl[0].from, tl[0].to, i);
      atl->l[i] = whichl (tl[0].from, tl[0].to, i);
    }
  for (j = 1; j < T; j++)
    {
      t = tl[j].age - tl[j - 1].age;
      for (i = 0; i < np; i++)
	{
	  line1 = tl[j].lineages[i];
	  /*      line2=0.;
	     for(z=0; z< np; z++){
	     if(i!=z)
	     line2 +=  tl[j].lineages[z];
	     } */
	  atl->km[i] += line1 * t;
	  atl->kt[i] += line1 * (line1 - 1) * t;
	  atl->p[i] += whichp (tl[j].from, tl[j].to, i);
	  atl->l[i] += whichl (tl[j].from, tl[j].to, i);
	}
    }
}


void 
copy_timelist (tarchive_fmt * from, tarchive_fmt * to, long np)
{
  memcpy (to->km, from->km, sizeof (double) * np);
  memcpy (to->kt, from->kt, sizeof (double) * np);
  memcpy (to->p, from->p, sizeof (long) * np);
  memcpy (to->l, from->l, sizeof (long) * np);
}

long 
whichp (long from, long to, long pop)
{
  if (from == to)
    {
      if (from == pop)
	return 1;
    }
  return 0;
}

long 
whichl (long from, long to, long pop)
{
  if (from != to)
    {
      if (to == pop)
	return 1;
    }
  return 0;
}

/*
   long calc_T(timearchive_fmt * old, long steps)
   {
   long i, sum = 0;;

   for (i = 0; i < steps; i++) {
   sum += old->tl[i].copies;
   if (sum == steps)
   return i + 1;
   }
   return steps;
   }
 */
void 
print_results (world_fmt * world)
{
  long l, skipped = 0;
  long loci = world->loci;
  FILE *outfile;
  option_fmt *opt;
  char sch[10], lch[10], cva[50];
  opt = world->options;
  outfile = world->outfile;
  if (opt->schains == 1)
    strcpy (sch, "chain");
  else
    strcpy (sch, "chains");
  if (opt->lchains == 1)
    strcpy (lch, "chain");
  else
    strcpy (lch, "chains");
  fprintf (outfile, "\n\n");
  PAGEFEED;
  fprintf (outfile, "================================================");
  fprintf (outfile, "================================\n");
  fprintf (outfile, "MCMC-estimation\n");
  fprintf (outfile, "------------------------------------------------");
  fprintf (outfile, "--------------------------------\n");
  fprintf (outfile, "There were %li short %s (%li used trees out of ",
	   opt->schains, sch, opt->ssteps);
  fprintf (outfile, "sampled %li)\n", opt->sincrement * opt->ssteps);
  fprintf (outfile, "and %li long %s (%li used trees out of sampled %li),\n",
	   opt->lchains, lch, opt->lsteps, opt->lincrement * opt->lsteps);
  fprintf (outfile, "the final estimated parameter are:\n");
  fprintf (outfile, "Locus            Theta [4N(mu)]     ");
  fprintf (outfile, "           4Nm             ");
  fprintf (outfile, "Log(Likelihood)\n");
  fprintf (outfile, "         -------------------------- ");
  fprintf (outfile, "-------------------------- \n");
  fprintf (outfile, "             Pop 1        Pop 2          Pop 1       Pop 2                        \n");
  fprintf (outfile, "----------------------------------------------------------------");
  fprintf (outfile, "----------------\n");
  for (l = 1; l < loci + 1; l++)
    {
      if (world->data->skiploci[l - 1])
	{
	  skipped++;
	  continue;
	}
      fprintf (outfile, " % 3li     % 12.6f % 12.6f  % 12.6f % 12.6f   % 12.5g\n",
	       l, world->atl[l].param[0], world->atl[l].param[1],
	       world->atl[l].param[2] * world->atl[l].param[0],
	       world->atl[l].param[3] * world->atl[l].param[1], world->atl[l].param_like);
    }
  if (loci - skipped > 1)
    {
      if (!opt->gamma)
	{
	  fprintf (outfile, " All       % 10.6f   % 10.6f    % 10.6f   % 10.6f     % 10.5g\n",
		   world->atl[l].param[0], world->atl[l].param[1],
		   world->atl[l].param[2] * world->atl[l].param[0],
		   world->atl[l].param[3] * world->atl[l].param[1], world->atl[l].param_like);
	}
      else
	{
	  if (world->atl[l].param[4] < 10e-9)
	    strcpy (cva, "0");
	  else
	    sprintf (cva, "%f", sqrt (world->atl[l].param[4]));
	  fprintf (outfile, " All       % 10.6f   % 10.6f    % 10.6f   % 10.6f     % 10.5g\n",
		   world->atl[l].param[0], world->atl[l].param[1],
		   world->atl[l].param[2] * world->atl[l].param[0],
		   world->atl[l].param[3] * world->atl[l].param[1], world->atl[l].param_like);
	  fprintf (outfile, "With shape parameter Inv(alpha)=%g ([CV(mu)]^2; CV(mu)=%s,alpha=%g)\n",
		   world->atl[l].param[4], cva, 1. / world->atl[l].param[4]);
	  fprintf (outfile, "[Newton-Raphson needed %li cycles of maximal %i,\n  Norm(first derivatives)=%f (normal stopping criteria is < %f)]\n\n\n",
	     world->atl[l].trials, NTRIALS, world->atl[l].normd, LOCI_NORM);
	}
    }
  fprintf (outfile, "================================================");
  fprintf (outfile, "================================\n");
}

void 
print_fst (world_fmt * world, double **fstparam)
{
  long l, skipped = 0;
  long numpop = world->numpop;
  long loci = world->loci;
  FILE *outfile = world->outfile;
  char str1[LINESIZE], str2[LINESIZE], str3[LINESIZE], str4[LINESIZE];
  if (loci < 40)
    {
      PAGEFEED;
    }
  fprintf (outfile, "\n\n");
  fprintf (outfile, "----------------------------------------------------------------\n");
  fprintf (outfile, "\"Fst\"-calculation: Fw/Fb estimator(*) using %li islands\n",
	   numpop);
  fprintf (outfile, "%s\n", world->options->fsttype == 'M' ?
	   "Migration rates are variable, Theta_1 = Theta2" :
	   "Thetas are variable, M_1 = M_2\n");
  fprintf (outfile, "----------------------------------------------------------------\n");
  fprintf (outfile, " Locus                       Population 1\n");
  fprintf (outfile, "         -------------------------------------------------------\n");
  fprintf (outfile, "           Theta           4Nm            Fw             Fb\n");
  fprintf (outfile, "-----------------------------------  ---------------------------\n");
  for (l = 0; l < loci; l++)
    {
      if (world->data->skiploci[l])
	{
	  skipped++;
	  continue;
	}
      prepare_print_nu (fstparam[l][0], str1);
      prepare_print_nm (fstparam[l][0], fstparam[l][2], str3);
      fprintf (outfile, " % 3li %12.12s  %12.12s  % 12.6g  % 12.6g\n",
	       l + 1, str1, str3,
	       fstparam[l][numpop * 2], fstparam[l][numpop * 2 + 2]);
    }
  if (world->loci - skipped > 1)
    {
      prepare_print_nu (fstparam[l][0], str1);
      prepare_print_nm (fstparam[l][0], fstparam[l][2], str3);
      fprintf (outfile, "  All%12.12s  %12.12s  % 12.6g  % 12.6g\n",
	       str1, str3,
	       fstparam[l][numpop * 2], fstparam[l][numpop * 2 + 2]);
    }
  fprintf (outfile, "----------------------------------------------------------------\n");
  fprintf (outfile, " Locus                     Population 2\n");
  fprintf (outfile, "         -------------------------------------------------------\n");
  fprintf (outfile, "           Theta         4Nm              Fw             Fb\n");
  fprintf (outfile, "-----------------------------------  ---------------------------\n");
  for (l = 0; l < loci; l++)
    {
      if (world->data->skiploci[l])
	continue;
      prepare_print_nu (fstparam[l][1], str2);
      prepare_print_nm (fstparam[l][1], fstparam[l][3], str4);
      fprintf (outfile, " % 3li %12.12s  %12.12s  % 12.6g  % 12.6g\n",
	       l + 1, str2, str4,
	       fstparam[l][numpop * 2 + 1], fstparam[l][numpop * 2 + 2]);
    }
  if (world->loci - skipped > 1)
    {
      prepare_print_nu (fstparam[l][1], str2);
      prepare_print_nm (fstparam[l][1], fstparam[l][3], str4);
      fprintf (outfile, "  All%12.12s  %12.12s  % 12.6g  % 12.6g\n",
	       str2, str4,
	       fstparam[l][numpop * 2 + 1], fstparam[l][numpop * 2 + 2]);
    }
  fprintf (outfile, "----------------------------------------------------------------\n");
  fprintf (outfile, "(*) based on Maynard Smith (1970), American Naturalist 104:231-237\n");
  fprintf (outfile, "and Nei & Feldman (1972), Theoretical Population Biology 3:460-465\n");
  fprintf (outfile, "Negative values show a violation of the assumptions and these\n");
  fprintf (outfile, "estimates (Theta AND 4Nm for both populations) should be discarded.\n");
  fprintf (outfile, "(-) can not be estimated\n");
  fprintf (outfile, "(0/0 or x/0) Divisions by zero\n");
}

void 
prepare_print_nu (double nu, char *str)
{
  if (nu <= -999)
    sprintf (str, "-");
  else
    sprintf (str, "% 12.6f", nu);
}

void 
prepare_print_nm (double nm, double nmu, char *strllike)
{
  if ((fabs (nmu) > 10e-20) && (fabs (nm) > 10e-20))
    sprintf (strllike, "% 10.5f", nm * nmu);
  else
    {
      if ((fabs (nmu) < 10e-20) && (fabs (nm) < 10e-20))
	{
	  sprintf (strllike, "0/0");
	}
      else
	{
	  if ((fabs (nmu) < 10e-20) && (fabs (nm) > 10e-20))
	    sprintf (strllike, "% 10.5f/0", nm);
	  else
	    {
	      if ((fabs (nmu) > 10e-20) && (fabs (nm) < 10e-20))
		sprintf (strllike, "0");
	    }
	}
    }
}

void 
print_menu_coalnodes (world_fmt * world, long G)
{
  long g, pop, minp = world->sumtips, maxp = 0;
  char ss[10];
  long **contribution;
  long nodenum = world->sumtips;
  tarchive_fmt *tl = world->atl[0].tl;
  if (world->options->verbose)
    {
      contribution = (long **) calloc (1, sizeof (long *) * nodenum);
      contribution[0] = (long *) calloc (1, sizeof (long) * nodenum * world->numpop);
      for (pop = 1; pop < world->numpop; pop++)
	{
	  contribution[pop] = contribution[0] + pop * nodenum;
	}
      for (g = 0; g < G; g++)
	{
	  for (pop = 0; pop < world->numpop; pop++)
	    {
	      contribution[pop][(long) tl[g].p[pop]] += tl[g].copies;
	    }
	}
      for (g = 0; g < nodenum; g++)
	{
	  for (pop = 0; pop < world->numpop; pop++)
	    {
	      if (maxp < g && contribution[pop][g] > 0)
		maxp = g;
	      if (minp > g && contribution[pop][g] > 0)
		minp = g;
	    }
	}
      fprintf (stdout, "           Coalescent nodes: ");
      for (g = minp; g < maxp + 1; g++)
	{
	  fprintf (stdout, "%2li ", g);
	}
      fprintf (stdout, "\n");

      for (pop = 0; pop < world->numpop; pop++)
	{
	  fprintf (stdout, "             population %3li: ", pop);
	  for (g = minp; g < maxp + 1; g++)
	    {
	      if (contribution[pop][g] == 0)
		{
		  strcpy (ss, "-");
		}
	      else
		{
		  if (contribution[pop][g] >= 100)
		    strcpy (ss, "*");
		  else
		    sprintf (ss, "%-6li", contribution[pop][g]);
		}
	      fprintf (stdout, "%2.2s ", ss);
	    }
	  fprintf (stdout, "\n");
	}
      free (contribution[0]);
      free (contribution);
    }
}


void 
print_menu_createplot (void)
{
  char nowstr[LINESIZE];
  get_time (nowstr, "%H:%M:%S");
  fprintf (stdout, "%s   Creating data for plots\n", nowstr);
}


void 
calc_loci_plane (world_fmt * world, nr_fmt * nr, timearchive_fmt * atl, double **pl, long loci, double contours[])
{
  long i, j;
  double max1 = -DBL_MAX;
  double max2 = -DBL_MAX;
  double values[PLANESIZE] = PLANETICKS;
  long intervals = PLANESIZE;
  if (world->options->gamma)
    {
      nr->param[4] = world->atl[loci + 1].param[4];
    }
  for (i = 0; i < intervals; i++)
    {
#ifdef MAC
      eventloop ();
#endif
      nr->param[0] = values[i];
      if (world->options->gamma)
	calc_gamma (nr);
      for (j = 0; j < intervals; j++)
	{
	  nr->param[0] = values[i];
	  nr->param[1] = world->atl[loci + 1].param[1];
	  nr->param[2] = values[j] / values[i];
	  nr->param[3] = world->atl[loci + 1].param[3];
	  calc_loci_like (nr, atl, loci, world->options->gamma);
	  pl[i][j] = nr->llike;
	  if (max1 < nr->llike)
	    max1 = nr->llike;
	}
    }
  nr->param[0] = world->atl[loci + 1].param[0];
  if (world->options->gamma)
    calc_gamma (nr);
  for (i = 0; i < intervals; i++)
    {
      for (j = 0; j < intervals; j++)
	{
	  nr->param[0] = world->atl[loci + 1].param[0];
	  nr->param[1] = values[i];
	  nr->param[2] = world->atl[loci + 1].param[2];
	  nr->param[3] = values[j] / values[i];
	  calc_loci_like (nr, atl, loci, world->options->gamma);
	  pl[i][j + intervals] = nr->llike;
	  if (max2 < nr->llike)
	    max2 = nr->llike;
	}
    }
  contours[0] += max1;
  contours[1] += max1;
  contours[2] += max1;
  contours[3] += max1;
  contours[4] += max2;
  contours[5] += max2;
  contours[6] += max2;
  contours[7] += max2;
}

void 
calc_locus_plane (world_fmt * world, nr_fmt * nr, tarchive_fmt * tl, long G,
		  double **pl, double contours[])
{
  long intervals = PLANESIZE;

  long i, j;
  double max1 = -DBL_MAX;
  double max2 = -DBL_MAX;
  double values[PLANESIZE] = PLANETICKS;
#ifdef _GNU_
  double param[nr->numpop2];
#else
  double *param;
  param = (double *) calloc (1, sizeof (double) * nr->numpop2);
#endif
  memcpy (param, nr->param, sizeof (double) * nr->numpop2);

  for (i = 0; i < intervals; i++)
    {
#ifdef MAC
      eventloop ();
#endif
      for (j = 0; j < intervals; j++)
	{
	  nr->param[0] = values[i];
	  nr->param[1] = param[1];
	  nr->param[2] = values[j] / values[i];
	  nr->param[3] = param[3];
	  calc_like (nr, tl, G);
	  pl[i][j] = nr->llike;
	  if (max1 < nr->llike)
	    max1 = nr->llike;
	}
    }
  for (i = 0; i < intervals; i++)
    {
      for (j = 0; j < intervals; j++)
	{
	  nr->param[0] = param[0];
	  nr->param[1] = values[i];
	  nr->param[2] = param[2];
	  nr->param[3] = values[j] / values[i];
	  calc_like (nr, tl, G);
	  pl[i][j + intervals] = nr->llike;
	  if (max2 < nr->llike)
	    max2 = nr->llike;
	}
    }
  contours[0] += max1;
  contours[1] += max1;
  contours[2] += max1;
  contours[3] += max1;
  contours[4] += max2;
  contours[5] += max2;
  contours[6] += max2;
  contours[7] += max2;
#ifndef _GNU_
  free (param);
#endif
}

void 
fill_plotplane (char **plane, double **pl, double contours[], plotmax_fmt * plotmax)
{
  long i, j, z, zz = 0;

  char line[100];
  long myval[PLANEBIGTICKS] = PLANEBIGTICKVALUES;
  double values[PLANESIZE] = PLANETICKS;
  long intervals = PLANESIZE;
  for (i = 0; i < intervals; i++)
    {
      if (i % 7)
	line[i] = '-';
      else
	line[i] = '+';
    }
  line[i] = '\0';
  sprintf (plane[0], "     +%s+    +%s+   ", line, line);
  for (i = 0; i < intervals; i++)
    {
      memset (plane[i + 1], ' ', sizeof (char) * (intervals + intervals + 20));
      plane[i + 1][intervals + intervals + 19] = '\0';
      if (!((i) % 7))
	{
	  sprintf (plane[i + 1], "  %2.0li +", myval[zz++]);
	  if (myval[zz - 1] == 0)
	    plane[i + 1][3] = '0';
	}
      else
	plane[i + 1][5] = '|';
      for (j = 0; j < intervals; j++)
	{
	  if (pl[i][j] < contours[1])
	    {
	      if (pl[i][j] < contours[2])
		{
		  if (pl[i][j] < contours[3])
		    {
		      plane[i + 1][j + 6] = ' ';
		    }
		  else
		    {
		      plane[i + 1][j + 6] = '-';
		    }
		}
	      else
		{
		  plane[i + 1][j + 6] = '+';
		}
	    }
	  else
	    {
	      if (pl[i][j] < (contours[0] - EPSILON))
		{
		  plane[i + 1][j + 6] = '*';
		}
	      else
		{
		  plane[i + 1][j + 6] = 'X';
		  plotmax->l1 = pl[i][j];
		  plotmax->x1 = values[j];
		  plotmax->y1 = values[i];
		}
	    }
	}
      if ((i) % 7)
	{
	  plane[i + 1][j + 6] = '|';
	  plane[i + 1][j + 11] = '|';
	}
      else
	{
	  plane[i + 1][j + 6] = '+';
	  plane[i + 1][j + 11] = '+';
	}
      for (j = intervals + 7 + 5 /*- 1*/ ; j < intervals + 7 + 5 + intervals /*- 2*/ ; j++)
	{
	  z = j - 7 - 5 /*+ 1 */ ;
	  if (pl[i][z] < contours[5])
	    {
	      if (pl[i][z] < contours[6])
		{
		  if (pl[i][z] < contours[7])
		    {
		      plane[i + 1][j] = ' ';
		    }
		  else
		    {
		      plane[i + 1][j] = '-';
		    }
		}
	      else
		{
		  plane[i + 1][j] = '+';
		}
	    }
	  else
	    {
	      if (pl[i][z] < (contours[4] - EPSILON))
		{
		  plane[i + 1][j] = '*';
		}
	      else
		{
		  plane[i + 1][j] = 'X';
		  plotmax->l2 = pl[i][z];
		  plotmax->x2 = values[z - intervals];
		  plotmax->y2 = values[i];
		}
	    }
	}
      if ((i) % 7)
	{
	  plane[i + 1][j] = '|';
	  plane[i + 1][j + 1] = '\0';
	}
      else
	{
	  plane[i + 1][j] = '+';
	  plane[i + 1][j + 1] = '\0';
	}
    }
  sprintf (plane[intervals + 1], "     +%s+    +%s+", line, line);
  sprintf (plane[intervals + 2], "     -3     -2     -1      0      1      2     -3     -2     -1      0      1      2");
}

void 
print_mathematica (world_fmt * world, double **plane, long x, long y)
{
  long i, j;
  static long number = 1;
  FILE *mathfile = world->mathfile;
  if (world->options->plot == 1 && world->options->plotmethod == PLOTALL)
    {
      fprintf (mathfile, "\n\nlocus%-li={{\n", number++);
      for (i = 0; i < x - 1; i++)
	{
	  fprintf (mathfile, "{");
	  for (j = 0; j < y - 1; j++)
	    {
	      fprintf (mathfile, "%20.20g,", plane[i][j]);
	    }
	  fprintf (mathfile, "%20.20g},\n", plane[i][j]);
	}
      fprintf (mathfile, "{");
      for (j = 0; j < y - 1; j++)
	{
	  fprintf (mathfile, "%20.20g,", plane[i][j]);
	}
      fprintf (mathfile, "%20.20g}},{\n", plane[i][j]);
      for (i = 0; i < x - 1; i++)
	{
	  fprintf (mathfile, "{");
	  for (j = y; j < 2 * y - 1; j++)
	    {
	      fprintf (mathfile, "%20.20g,", plane[i][j]);
	    }
	  fprintf (mathfile, "%20.20g},\n", plane[i][j]);
	}
      fprintf (mathfile, "{");
      for (j = y; j < 2 * y - 1; j++)
	{
	  fprintf (mathfile, "%20.20g,", plane[i][j]);
	}
      fprintf (mathfile, "%20.20g}}}\n", plane[i][j]);
    }
}


void 
print_cov (world_fmt * world, long numpop, long loci, double ***cov)
{
  FILE *outfile = world->outfile;
  long locus, skipped = 0;
#ifdef _GNU_
  double corr[2 * numpop + 1];
#else
  double *corr;
  corr = (double *) calloc (1, sizeof (double) * (2 * numpop + 1));
#endif
  fprintf (outfile, "\n\n");
  fprintf (outfile, "--------------------------------------------------------------------\n");
  fprintf (outfile, "MCMC estimation:\n");
  fprintf (outfile, "Covariance matrix(*)                         Correlation matrix\n");
  fprintf (outfile, "-------------------------------------------  -----------------------\n");
  for (locus = 0; locus < world->loci; locus++)
    {
      if (world->data->skiploci[locus])
	{
	  skipped++;
	  continue;
	}
      fprintf (outfile, "Locus %li:\n", locus + 1);
      print_cov_table (outfile, locus, world, corr, 0);
      fprintf (outfile, "\n");
    }
  fprintf (outfile, "\n\n");
  if (world->loci - skipped > 1)
    {
      if (world->options->gamma)
	{
	  fprintf (outfile, "Over all loci\n");
	  fprintf (outfile, "------------------------------------------------------  ");
	  fprintf (outfile, "----------------------------\n");
	  print_cov_table (outfile, locus, world, corr, 1);
	}
      else
	{
	  fprintf (outfile, "Over all loci\n");
	  fprintf (outfile, "-------------------------------------------  -----------------------\n");
	  print_cov_table (outfile, locus, world, corr, 0);
	}
    }
#ifndef _GNU_
  free (corr);
#endif
}

void 
print_cov_table (FILE * outfile, long locus, world_fmt * world, double *corr, long addvar)
{
  long i, j;
  double denom, temp1, temp2;
  for (i = 0; i < world->numpop * 2 + addvar; i++)
    {
      for (j = 0; j < world->numpop * 2 + addvar; j++)
	{
	  temp1 = fabs (world->cov[locus][i][i]);
	  if (temp1 < DBL_EPSILON)
	    temp1 = DBL_EPSILON;
	  temp2 = fabs (world->cov[locus][j][j]);
	  if (temp2 < DBL_EPSILON)
	    temp2 = DBL_EPSILON;
	  denom = 0.5 * (log (temp1) + log (temp2));
	  if ((temp1 = exp (denom)) == 0.0)
	    corr[j] = 9.99;
	  else
	    corr[j] = world->cov[locus][i][j] / exp (denom);
	}
      for (j = 0; j < world->numpop * 2 + addvar; j++)
	{
#if TESTING
	  fprintf (outfile, "% 20.10f ", world->cov[locus][i][j]);	/*was 10.4 */
#else
	  fprintf (outfile, "% 10.4f ", world->cov[locus][i][j]);
#endif
	}
      for (j = 0; j < world->numpop * 2 + addvar; j++)
	{
	  if (corr[j] < 9.)
	    fprintf (outfile, "% 4.2f ", corr[j]);
	  else
	    fprintf (outfile, "  --  ");
	}
      fprintf (outfile, "\n");
    }
}


void 
free_tree (node * p)
{
  if (!p->tip)
    {
      if (p->next->back != NULL)
	{
	  free_tree (p->next->back);
	}
      if (p->type != 'm' && p->next->next->back != NULL)
	{
	  free_tree (p->next->next->back);
	}
    }
  else
    {
      free (p->nayme);
    }
  switch (p->type)
    {
    case 'm':
      free_nodelet (p, 2);
      break;
    case 't':
      free_nodelet (p, 1);
      break;
    default:
      free_nodelet (p, 3);
      break;
    }
}

void 
free_nodelet (node * p, long num)
{
  long i;
  node *q;
  switch ((short) num)
    {
    case 3:
      free (p->next->next);
    case 2:
      free (p->next);
    case 1:
      free (p);
      break;
    default:
      for (i = 0; i < num; i++)
	{
	  q = p->next;
	  free (p);
	  p = q;
	}
    }
}

void 
free_seqx (node * p, long sites)
{
  long j;
  if (!p->tip)
    {
      if (p->next->back != NULL)
	{
	  free_seqx (crawlback (p->next), sites);
	}
      if (p->next->next->back != NULL)
	{
	  free_seqx (crawlback (p->next->next), sites);
	}
    }
  if (p->type == 'r')
    return;
  for (j = 0; j < sites; j++)
    free (p->x.s[j]);
}

void 
free_x (node * p)
{
  if (!p->tip)
    {
      if (p->next->back != NULL)
	{
	  free_x (crawlback (p->next));
	}
      if (p->next->next->back != NULL)
	{
	  free_x (crawlback (p->next->next));
	}
    }
  if (p->type == 'r')
    return;
  free (p->x.a);
}


void 
test_locus_like (double *param0, double *param1, long df, long locus, world_fmt * world,
		 boolean withhead, char *this_string)
{
  char c;
  char *teststat, temp[256];
  double like1, like0, testval, chi05, chi01;
  nr_fmt *nr;
  long i, g, part1len = 1, part2len = 1;
  long elem;
  nr = (nr_fmt *) calloc (1, sizeof (nr_fmt));
  create_nr (nr, world->atl[locus + 1].numpop, world->atl[locus + 1].T);
  nr->skiploci = world->data->skiploci;
  elem = nr->numpop2 = world->numpop2;
  nr->numpop = world->numpop;
  memcpy (nr->param, param0, sizeof (double) * elem);
  /* Prob(G|Param0) */
  for (g = 0; g < world->atl[locus + 1].T; g++)
    {
      nr->apg0[g] = probG (world->param0, &world->atl[locus + 1].tl[g], nr->numpop);
    }
  like0 = calc_like (nr, world->atl[locus + 1].tl, world->atl[locus + 1].T);
  memcpy (nr->param, param1, sizeof (double) * elem);
  like1 = calc_like (nr, world->atl[locus + 1].tl, world->atl[locus + 1].T);
  testval = -2. * (like0 - like1);
  chi05 = chisquare (df, 0.05);
  chi01 = chisquare (df, 0.01);
  if (this_string != NULL)
    {
      c = this_string[0];
      for (i = 1; i < 1000 && c != '\0'; i++)
	c = this_string[i];
      part2len = part2len = i;
      teststat = this_string;
    }
  else
    {
      teststat = (char *) calloc (1, sizeof (char) * 256);
      sprintf (teststat, "Locus %li: H0:{%f", locus + 1, param0[0]);
      for (i = 1; i < elem; i++)
	{
	  sprintf (temp, ",%f", param0[i]);
	  strcat (teststat, temp);
	}
      sprintf (temp, "}=");
      strcat (teststat, temp);
      part1len = strlen (teststat);
      sprintf (temp, "\n       {%f", param1[0]);
      strcat (teststat, temp);
      for (i = 1; i < elem; i++)
	{
	  sprintf (temp, ",%f", param1[i]);
	  strcat (teststat, temp);
	}
      sprintf (temp, "}");
      strcat (teststat, temp);
      part2len = strlen (teststat);
    }
  if (withhead)
    {
      fprintf (world->outfile, "Likelihood ratio test with alpha=0.05 and 0.01\n");
      part1len = MAX (part1len, part2len - part1len);
      fprintf (world->outfile, "%*.*s %-10.10s %-3.3s %-4.4s %-4.4s\n",
	       (int) part1len, (int) part1len, " ", "Test value", "DF", "0.05", "0.01");
      for (i = 0; i < part2len + 35; i++)
	fputc ('-', world->outfile);
      fprintf (world->outfile, "\n");
    }
  fprintf (world->outfile, "%s %10.6f %3li %4.2f %4.2f\n", teststat,
	   testval, elem, chi05, chi01);
  if (this_string == NULL)
    free (teststat);
  free_nr (nr);
}


void 
test_loci_like (double *param0, double *param1, long df, long loci, world_fmt * world,
		boolean withhead, char *this_string)
{
  char c;
  char *teststat, temp[256];
  double like1, like0, testval, chi05, chi01;
  nr_fmt *nr;
  long i, g = world->atl[1].T, part1len = 1, part2len = 1;
  long elem;
  nr = (nr_fmt *) calloc (1, sizeof (nr_fmt));
  for (i = 1; i < loci + 1; i++)
    {
      if (g < world->atl[i].T)
	g = world->atl[i].T;
    }
  create_nr (nr, world->atl[loci].numpop, g);
  elem = world->options->gamma ? nr->numpop2 + 1 : nr->numpop2;
  nr->skiploci = world->data->skiploci;
  memcpy (nr->param, param0, sizeof (double) * elem);
  if (world->options->gamma)
    calc_gamma (nr);
  like0 = calc_loci_like (nr, world->atl, world->loci, world->options->gamma);
  memcpy (nr->param, param1, sizeof (double) * elem);
  if (world->options->gamma)
    calc_gamma (nr);
  like1 = calc_loci_like (nr, world->atl, world->loci, world->options->gamma);
  testval = -2 * (like0 - like1);
  chi05 = chisquare (df, 0.05);
  chi01 = chisquare (df, 0.01);
  if (this_string != NULL)
    {
      c = this_string[0];
      for (i = 1; i < 1000 && c != '\0'; i++)
	c = this_string[i];
      part2len = part2len = i;
      teststat = this_string;
    }
  else
    {
      teststat = (char *) calloc (1, sizeof (char) * 256);
      sprintf (teststat, "All   :{%4.4f", param0[0]);
      for (i = 1; i < elem; i++)
	{
	  sprintf (temp, ",%4.4f", param0[i]);
	  strcat (teststat, temp);
	}
      sprintf (temp, "}=");
      strcat (teststat, temp);
      part1len = strlen (teststat);
      sprintf (temp, "\n       {%4.4f", param1[0]);
      strcat (teststat, temp);
      for (i = 1; i < elem; i++)
	{
	  sprintf (temp, ",%4.4f", param1[i]);
	  strcat (teststat, temp);
	}
      sprintf (temp, "}");
      strcat (teststat, temp);
      part2len = strlen (teststat);
    }
  if (withhead)
    {
      fprintf (world->outfile, "\n\n\nLikelihood ratio test with alpha=0.05 and 0.01\n");
      part1len = MAX (part1len, part2len - part1len);
      fprintf (world->outfile, "%*.*s %-10.10s %-3.3s %-4.4s %-4.4s\n",
	       (int) part1len, (int) part1len, " ", "Test value", "DF", "0.05", "0.01");
      for (i = 0; i < part1len + 35; i++)
	fputc ('-', world->outfile);
      fprintf (world->outfile, "\n");
    }
  fprintf (world->outfile, "%s %10.2f %3li %4.2f %4.2f\n", teststat,
	   testval, elem, chi05, chi01);
  if (this_string == NULL)
    free (teststat);
  free_nr (nr);
}


double 
chisquare (long df, double alpha)
{
  const double table05[] =
  {3.84146, 5.99147, 7.81473, 9.48773, 11.0705, 12.5916};
  const double table01[] =
  {6.63490, 9.21034, 11.3449, 13.2767, 15.0863, 16.8119};

  if (alpha == 0.05)
    return table05[df - 1];
  if (alpha == 0.01)
    return table01[df - 1];
  error ("Chi-distribution for any probability alpha is not implemented");
  return -1;
}


long 
set_test_param (double *param, char *strp, world_fmt * world, long lrline, long locus)
{
  long i = 0, z = 0, zz = 0, zzz = 0, df = 0;
  char *tmp, *ss;
  double *meanparam, mean;
  ss = (char *) calloc (1, sizeof (char) * LINESIZE);
  tmp = (char *) calloc (1, sizeof (char) * LINESIZE);
  strcpy (ss, strp);
  if (world->loci - world->skipped > 1 && world->options->lratio->data[lrline].type == MEAN)
    meanparam = world->atl[world->loci + 1].param;
  else
    meanparam = world->atl[locus + 1].param;
  while (ss[zzz] != '\0')
    {
      tmp[i] = ss[zzz++];
      if (!(tmp[i] == ',' || tmp[i] == '\0' || tmp[i] == '\n' || tmp[i] == ';'))
	{
	  if (tmp[i] != ' ')
	    i++;
	}
      else
	{
	  tmp[i] = '\0';
	  i = 0;
	  switch (tmp[0])
	    {
	    case '*':
	      param[z] = meanparam[z];
	      z++;
	      break;
	    case 't':
	      zz = atol (tmp) - 1;
	      df++;
	      if (zz < 0)
		{
		  mean = 0.0;
		  for (zz = 0; zz < world->numpop; zz++)
		    mean += meanparam[zz];
		  mean /= world->numpop;
		  param[z] = mean;
		}
	      else
		{
		  param[z] = meanparam[zz];
		}
	      z++;
	      break;
	    case 'm':
	      zz = atol (tmp) - 1;
	      df++;
	      if (zz < 0)
		{
		  mean = 0.0;
		  for (zz = world->numpop; zz < 2 * world->numpop; zz++)
		    mean += meanparam[zz];
		  mean /= world->numpop;
		  param[z] = mean;
		}
	      else
		{
		  param[z] = meanparam[zz];
		}
	      z++;
	      break;
	    default:
	      df++;
	      param[z] = MAX (atof (tmp), SMALLEST_THETA);
	      if (z >= world->numpop)
		param[z] /= param[z - world->numpop];
	      z++;
	      break;
	    }
	}
    }
  return df;
}


void 
print_CV (world_fmt * world)
{
  long i;
  long elem;
  char temp[100];
  fprintf (world->outfile, "\n\nVariance and coefficient of variance\n");
  if (world->loci - world->skipped > 1)
    {
      elem = 2 * world->numpop + (world->options->gamma ? 1 : 0);
      fprintf (world->outfile, "PARAM@ ");
      for (i = 0; i < elem; i++)
	{
	  fprintf (world->outfile, "%20.20g ", world->atl[world->loci + 1].param[i]);
	}
      fprintf (world->outfile, "\nVAR@   ");
      for (i = 0; i < elem; i++)
	{
	  fprintf (world->outfile, "%20.20g ", world->cov[world->loci][i][i]);
	}
      fprintf (world->outfile, "\nCV@    ");
      for (i = 0; i < elem; i++)
	{
	  strcpy (temp, "-");
	  if (world->cov[world->loci][i][i] >= 0)
	    {
	      sprintf (temp, "%20.20g", (sqrt (world->cov[world->loci][i][i])) /
		       world->atl[world->loci + 1].param[i]);
	    }
	  fprintf (world->outfile, "%-s ", temp);
	}
      fprintf (world->outfile, "\n");
    }
  else
    {				/* one locus */
      elem = 2 * world->numpop;
      fprintf (world->outfile, "PARAM@ ");
      for (i = 0; i < elem; i++)
	{
	  fprintf (world->outfile, "%20.20g ", world->atl[1].param[i]);
	}
      fprintf (world->outfile, "\nVAR@   ");
      for (i = 0; i < elem; i++)
	{
	  fprintf (world->outfile, "%20.20g ", world->cov[0][i][i]);
	}
      fprintf (world->outfile, "\nCV@    ");
      for (i = 0; i < elem; i++)
	{
	  strcpy (temp, "-");
	  if (world->cov[0][i][i] >= 0)
	    {
	      sprintf (temp, "%20.20g", (sqrt (world->cov[0][i][i])) /
		       world->atl[1].param[i]);
	    }
	  fprintf (world->outfile, "%-s ", temp);
	}
      fprintf (world->outfile, "\n");
    }
}
