


/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effective population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 M C M C   R O U T I N E S 

 Markov Monte Carlo stuff: treechange, acceptance
 

 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: mcmc1.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"
#include "random.h"
#include "tree.h"
#include "mcmc2.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

#define MIGRATION_AIR (boolean) 1
#define MIGRATION_IN_TREE (boolean) 0
#define NO_MIGR_NODES 0
#define WITH_MIGR_NODES 1
#define MIGRATION_LIMIT 1000
/* prototypes ------------------------------------------- */
long metropolize (world_fmt * world, long g);
/* private functions */
void new_localtimelist (timelist_fmt ** ntl, timelist_fmt * otl);
void new_proposal (proposal_fmt ** proposal, timelist_fmt * tl, world_fmt * world);
void chooseOrigin (proposal_fmt * proposal);
void construct_localtimelist (timelist_fmt * timevector, proposal_fmt * proposal);
void traverseAllNodes (node * theNode, node *** nodelist, long *node_elem, long *oldnode_elem, int include_migration);
void add_locallineages (timelist_fmt * timevector, proposal_fmt * proposal);
void chooseTarget (proposal_fmt * proposal, timelist_fmt * timevector,
		   node ** bordernodes, long *bordernum);
void findbordernodes (node * theNode, proposal_fmt * proposal, long pop,
		      node ** bordernodes, long *bordernum,
		      vtlist ** tyme, long gte);
void free_proposal (proposal_fmt * proposal);
void free_timevector (timelist_fmt * timevector);
long remove_doublettes (timelist_fmt * timevector, node ** ptr);
long xor (node ** ptrl1, node ** ptrl2);
long rmigrcount (proposal_fmt * proposal);

int migrate (proposal_fmt * proposal, node * up, long *old_migr_table_counter, boolean air);
int pre_population (proposal_fmt * proposal, vtlist * ltime, long gte, long *slider);
/* boolean same_pop(node * up, double tyme, long pop); */
boolean acceptlike (world_fmt * world, proposal_fmt * proposal, long g, timelist_fmt * tyme);
double eventtime (proposal_fmt * proposal, long pop, vtlist * tentry, char *event, long g);
node *showsister (node * theNode);
void count_migrations (node * p, long *count);
long interior_pop0 (timelist_fmt * timevector, proposal_fmt * proposal);

/*=======================================================*/
long 
metropolize (world_fmt * world, long g)
{				/*return 1 if tree was accepted 0 otherwise */
  static long treefilepos;	/* write position in the treefile */
  boolean coalesced;
  boolean test;
  char event;
  long slider;
  long bordernum;
  long actualpop = -99, zz;
  double endtime, nexttime, age;

  proposal_fmt *proposal;	/*scratchpad  in which all involved nodes
				   are recorded and help arrays, e.g. migration arrays,
				   are stored */
  timelist_fmt *timevector;	/* local timelist */
  vtlist *tentry;		/*pointer into timeslice */

  /* ---------------------------------------------------------
     initialize local timelist and construct residual timelist 
     find the start node and snip of the branch and node below */
#ifdef MAC
  eventloop ();
#endif
  new_localtimelist (&timevector, &world->treetimes[0]);
  new_proposal (&proposal, &world->treetimes[0], world);
  chooseOrigin (proposal);
  construct_localtimelist (timevector, proposal);
  tentry = &(*timevector).tl[0];
  if (proposal->origin->tyme == 0.0)
    {
      age = 0.0;
    }
  else
    {
      age = proposal->origin->tyme;
      zz = 0;
      while (tentry->age < age && zz < (*timevector).T)
	{
	  tentry = &(*timevector).tl[zz];
	  zz++;
	}
    }
  nexttime = tentry->age;
  if ((*timevector).T > 1)
    endtime = (*timevector).tl[(*timevector).T - 2].age;
  else
    endtime = 0.0;
  proposal->time = age;
  coalesced = FALSE;
  /*------------------------------------
    main loop: sliding down the tree  */
  slider = 0;
  while (nexttime <= endtime)
    {
      actualpop = (proposal->migr_table_counter > 0) ?
	proposal->migr_table[proposal->migr_table_counter - 1].from :
	proposal->origin->pop;
      proposal->time = age + eventtime (proposal, actualpop, tentry, &event, g);
      if (proposal->time < nexttime)
	{
	  if (event == 'm')
	    {
	      if (!migrate (proposal, proposal->origin,
			  &proposal->old_migr_table_counter, MIGRATION_AIR))
		{
		  free_proposal (proposal);
		  free_timevector (timevector);
		  return 0;
		}
	      age = proposal->time;
	      continue;
	    }
	  else
	    {			/*coalesce */
	      if (event != 'c')	/* saveguard */
		exit (-1);
	      chooseTarget (proposal, timevector, proposal->bordernodes, &bordernum);
	      if (proposal->target != NULL)
		{
		  pretendcoalesce1p (proposal);
		  coalesced = TRUE;
		  break;
		}
	      else
		{
		  fprintf (stderr, "\n\n\n###########################################\n");
		  fprintf (stderr, "Target=NULL, proposal->time=%f\n", proposal->time);
		  fprintf (stderr, "###########################################\n\n\n\n\n");
		  return 0;
		}
	    }
	}			/*end if proposal->time < nextime */
      age = nexttime;
      tentry = &(*timevector).tl[(tentry->slice) + 1];	/*next entry in timelist */
      nexttime = tentry->age;
    }
  if (!coalesced)
    {
      if (!pre_population (proposal, (*timevector).tl, (*timevector).T - 1, &slider))
	{
	  free_proposal (proposal);
	  free_timevector (timevector);
	  return 0;
	}
      pretendcoalesce1p (proposal);
    }
  test = acceptlike (world, proposal, g, timevector);
  if (test)
    {
      if (proposal->time > world->root->tyme)
	{			/*saveguard */
	  world->root->tyme += proposal->time;
	  fprintf (stdout, "ROOT moved from %f to %f because population joining moves to %f\n",
		   world->root->tyme - proposal->time, world->root->tyme, proposal->time);
	}
      coalesce1p (proposal);
      world->likelihood[g] = treelikelihood (world);	/*recalculate the p->x */
      if (fabs (world->likelihood[g] - proposal->likelihood) > EPSILON)
	fprintf (stderr, "STRANGE: proposed and new likelihood differ: %f != %f\n",
		 proposal->likelihood, world->likelihood[g]);
      construct_tymelist (world, &world->treetimes[0]);		/* create a new timelist */
      if (world->options->treeprint != NONE)
	print_tree (world, g, &treefilepos);
      world->migration_counts = 0;	/* report the number of migration on the tree */
      count_migrations (world->root->next->back, &world->migration_counts);
      free_proposal (proposal);
      free_timevector (timevector);
      return 1;			/* new tree accepted */
    }

  free_proposal (proposal);
  free_timevector (timevector);
  return 0;			/* not accepted */

}



/*=======================================================*/
void 
new_localtimelist (timelist_fmt ** ntl, timelist_fmt * otl)
{
  (*ntl) = (timelist_fmt *) calloc (1, sizeof (timelist_fmt) * 1);
  (*ntl)[0].tl = (vtlist *) malloc ((*otl).allocT * sizeof (vtlist));
  (*ntl)[0].allocT = (*otl).allocT;
  (*ntl)[0].T = (*otl).T;
  memcpy ((*ntl)[0].tl, (*otl).tl, (*otl).allocT * sizeof (vtlist));
}

void 
new_proposal (proposal_fmt ** proposal, timelist_fmt * tl, world_fmt * world)
{
  long j;
  long mal = world->data->maxalleles[world->locus];
  (*proposal) = (proposal_fmt *) calloc (1, sizeof (proposal_fmt));
  (*proposal)->listsize = ((*tl).allocT + (*tl).T + 5);
  (*proposal)->aboveorigin = (node **) calloc (1, sizeof (node *) * (*proposal)->listsize);
  (*proposal)->bordernodes = (node **) calloc (1, sizeof (node *) * (*proposal)->listsize);
  (*proposal)->world = world;
  (*proposal)->datatype = world->options->datatype;
  (*proposal)->sumtips = world->sumtips;
  (*proposal)->numpop = world->numpop;
  (*proposal)->endsite = world->data->seq->endsite;
  (*proposal)->fracchange = world->data->seq->fracchange;
  (*proposal)->param0 = world->param0;
  (*proposal)->root = world->root;
  (*proposal)->migration_model = world->options->migration_model;
  (*proposal)->line_f = (node **) calloc (1, sizeof (node *) * (*proposal)->sumtips);
  (*proposal)->line_t = (node **) calloc (1, sizeof (node *) * (*proposal)->sumtips);
  if ((*proposal)->datatype == 's')
    {
      (*proposal)->xf.s = (phenotype) malloc (world->data->seq->endsite * sizeof (ratelike *));
      (*proposal)->xt.s = (phenotype) malloc (world->data->seq->endsite * sizeof (ratelike *));
      for (j = 0; j < (*proposal)->endsite; j++)
	{
	  (*proposal)->xf.s[j] = (ratelike) malloc (world->options->rcategs * sizeof (sitelike));
	  (*proposal)->xt.s[j] = (ratelike) malloc (world->options->rcategs * sizeof (sitelike));
	}
    }
  else
    {
      (*proposal)->xf.a = (double *) calloc (1, sizeof (double) * mal);
      (*proposal)->xt.a = (double *) calloc (1, sizeof (double) * mal);
    }
  (*proposal)->old_migr_table_counter = 4 * (*proposal)->sumtips /* 100 */ ;
  (*proposal)->old_migr_table_counter2 = 4 * (*proposal)->sumtips /* 100 */ ;
  (*proposal)->migr_table = (migr_table_fmt *) calloc (1,
	     sizeof (migr_table_fmt) * (*proposal)->old_migr_table_counter);
  (*proposal)->migr_table2 = (migr_table_fmt *) calloc (1,
	    sizeof (migr_table_fmt) * (*proposal)->old_migr_table_counter2);
  (*proposal)->migr_table_counter = 0;
  (*proposal)->migr_table_counter2 = 0;
}

void 
chooseOrigin (proposal_fmt * proposal)
{
  long elem = 0, oldelem = (proposal->sumtips * 2);
  node *tmp, **goal;
  goal = (node **) calloc (1, sizeof (node *) * oldelem);


  traverseAllNodes (crawlback (proposal->root->next), &goal, &elem, &oldelem,
		    NO_MIGR_NODES);
  tmp = goal[RANDINT (0, elem - 2)];
  proposal->origin = tmp;
  if (proposal->origin != showtop (crawlback (proposal->root->next)))
    {
      proposal->oback = showtop (crawlback (proposal->origin));
      proposal->osister = showsister (proposal->origin);
      if (proposal->oback != showtop (crawlback (proposal->root->next)))
	{
	  proposal->ocousin = showsister (proposal->oback);
	}
      else
	{
	  proposal->ocousin = NULL;
	}
    }
  if (proposal->origin == NULL)
    error ("Designation of origin for branch removal failed");
  free (goal);
}

void 
construct_localtimelist (timelist_fmt * timevector, proposal_fmt * proposal)
{
  long z = 0;
/*     size_t num = 0; */
  long oz = proposal->listsize;
  proposal->mig_removed = FALSE;
  traverseAllNodes (crawlback (proposal->origin)->back, &proposal->aboveorigin, &z, &oz, WITH_MIGR_NODES);
  proposal->aboveorigin[z++] = proposal->oback;
  /*    num = (size_t) z; */
  z = remove_doublettes (timevector, proposal->aboveorigin);
  qsort ((void *) (*timevector).tl, (*timevector).T, sizeof (vtlist), agecmp);
  (*timevector).T -= z;
  if ((*timevector).tl[(*timevector).T - 1].eventnode->type != 'r')
    {
      fprintf (stderr, "autsch, root not at the end of timelist\n");
    }
  timeslices (&timevector);
  add_locallineages (timevector, proposal);
}

/*---------------------------------------------------------------------------- 
finds all nodes in a tree starting at the root node and crawling up 
to the tips in a recursive fashion, writing nodeptrs in the nodelist vector
the flag include_migration is 1 if we want to touch the migration nodes too,
otherwise =0 -> jump over the migration nodes. for convenience we define the 
the macros NO_MIGR_NODES=0 and WITH_MIGR_NODES=1 in the treesetup.h file
PB 1995
 */
void 
traverseAllNodes (node * theNode, node *** nodelist, long *node_elem, long *oldnode_elem, int include_migration)
{
  long elem;
  if (include_migration == NO_MIGR_NODES)
    {
      if (theNode->type != 't')
	{
	  if (crawlback (theNode->next) != NULL)
	    traverseAllNodes (crawlback (theNode->next),
			  nodelist, node_elem, oldnode_elem, NO_MIGR_NODES);
	  if (theNode->type != 'm' && crawlback (theNode->next->next) != NULL)
	    traverseAllNodes (crawlback (theNode->next->next),
			  nodelist, node_elem, oldnode_elem, NO_MIGR_NODES);
	  if ((*node_elem) == (*oldnode_elem - 1))
	    {
	      elem = *oldnode_elem = ((*oldnode_elem) + (*oldnode_elem) / 2);
	      (*nodelist) = (node **) realloc ((*nodelist), sizeof (node *) * elem);
	      memset ((*nodelist) + (*oldnode_elem), 0, sizeof (node *) * (elem - (*oldnode_elem)));
	      *oldnode_elem = elem;
	    }
	  (*nodelist)[(*node_elem)++] = theNode;
	  if (theNode->type == 'm')
	    {
	      error ("Migration node encountered?! and died!");
	    }
	}
      else
	{
	  if ((*node_elem) == (*oldnode_elem - 1))
	    {
	      elem = *oldnode_elem = ((*oldnode_elem) + (*oldnode_elem) / 2);
	      (*nodelist) = (node **) realloc ((*nodelist), sizeof (node *) * elem);
	      memset ((*nodelist) + (*oldnode_elem), 0, sizeof (node *) * (elem - (*oldnode_elem)));
	      *oldnode_elem = elem;
	    }
	  (*nodelist)[(*node_elem)] = theNode;
	  (*node_elem) += 1;
	}
    }
  else
    {
      if (theNode->type != 't')
	{
	  if (theNode->next->back != NULL)
	    traverseAllNodes (theNode->next->back,
			nodelist, node_elem, oldnode_elem, WITH_MIGR_NODES);
	  if (theNode->type != 'm' && theNode->next->next->back != NULL)
	    traverseAllNodes (theNode->next->next->back,
			nodelist, node_elem, oldnode_elem, WITH_MIGR_NODES);
	  if ((*node_elem) == (*oldnode_elem - 1))
	    {
	      elem = *oldnode_elem = ((*oldnode_elem) + (*oldnode_elem) / 2);
	      (*nodelist) = (node **) realloc ((*nodelist), sizeof (node *) * elem);
	      memset ((*nodelist) + (*oldnode_elem), 0, sizeof (node *) * (elem - (*oldnode_elem)));
	      *oldnode_elem = elem;
	    }
	  (*nodelist)[(*node_elem)++] = theNode;
	}
      else
	{
	  if ((*node_elem) == (*oldnode_elem - 1))
	    {
	      elem = *oldnode_elem = ((*oldnode_elem) + (*oldnode_elem) / 2);
	      (*nodelist) = (node **) realloc ((*nodelist), sizeof (node *) * elem);
	      memset ((*nodelist) + (*oldnode_elem), 0, sizeof (node *) * (elem - (*oldnode_elem)));
	      *oldnode_elem = elem;
	    }
	  (*nodelist)[(*node_elem)++] = theNode;
	}
    }
}

/* prepares to remove elements of a timelist by setting eventnode to NULL
   and age to Infinity, so a simple sort will push them over the edge */
long 
remove_doublettes (timelist_fmt * timevector, node ** ptr)
{
  long i = 0, j = 0, slot = 0;
  /* assumes that there is an NULL element at the end */
  for (i = 0; i < (*timevector).T; j = 0, i++)
    {
      while (((*timevector).tl[i].eventnode != ptr[j]) && (ptr[j] != NULL))
	j++;
      if (ptr[j] != NULL)
	{
	  slot++;
	  (*timevector).tl[i].age = DBL_MAX;
	}
    }
  return slot;
}

void 
add_locallineages (timelist_fmt * timevector, proposal_fmt * proposal)
{
  long pop = -99, numpop = proposal->world->numpop;
  if (timevector->T <= 0)
    error ("Help: timelist contains 0 elements");
  for (pop = 0; pop < numpop; pop++)
    timevector->tl[timevector->T - 1].lineages[pop] = 0;
  if (timevector->T == 1)
    timevector->tl[0].lineages[proposal->osister->actualpop] += 1;
  else
    timevector->tl[timevector->T - 1].lineages[timevector->tl[timevector->T - 2].from] += 1;
  add_partlineages (numpop, &timevector);

}


/* replaces nodepointers in list 1 with NULL if they are present in list 2
   returns the first NULL slot in the array.
 */
long 
xor (node ** ptrl1, node ** ptrl2)
{
  long i = 0, j = 0, slot = -1;
  /* assumes that there is an NULL element at the end */
  for (i = 0; ptrl1[i] != NULL; j = 0, i++)
    {
      while ((ptrl1[i] != ptrl2[j]) && (ptrl2[j] != NULL))
	j++;
      if (ptrl2[j] != NULL)
	{
	  if (slot == -1)
	    slot = i;
	  ptrl1[i] = NULL;
	}
    }
  return slot;
}


int 
migrate (proposal_fmt * proposal, node * up, long *old_migr_table_counter, boolean air)
{
  migr_table_fmt *array;
  long i;
  if (air)
    {
      array = proposal->migr_table;
      i = proposal->migr_table_counter;
    }
  else
    {
      array = proposal->migr_table2;
      i = proposal->migr_table_counter2;
    }
  if (i > MIGRATION_LIMIT)
    {
      fprintf (stdout, "migration limit reached\n");
      return 0;
    }
  switch (proposal->migration_model)
    {
    case STEPSTONE:
    case ISLAND:
    case MATRIX:
      array[i].from = RANDINT (0, proposal->numpop - 1);
      if (i > 0)
	{
	  while (array[i].from == array[i - 1].from)
	    array[i].from = RANDINT (0, proposal->numpop - 1);
	  array[i].to = array[i - 1].from;
	}
      else
	{
	  while (array[i].from == up->pop)
	    array[i].from = RANDINT (0, proposal->numpop - 1);
	  array[i].to = up->pop;
	}
      break;
    default:
      break;
    }
  array[i++].time = proposal->time;
  if (i > (*old_migr_table_counter) - 5)
    {
      (*old_migr_table_counter) += 10;
      if (air)
	{
	  proposal->migr_table = (migr_table_fmt *) realloc (proposal->migr_table,
		       sizeof (migr_table_fmt) * (*old_migr_table_counter));
	  array = proposal->migr_table;
	}
      else
	{
	  proposal->migr_table2 = (migr_table_fmt *) realloc (proposal->migr_table2,
		       sizeof (migr_table_fmt) * (*old_migr_table_counter));
	  array = proposal->migr_table;
	}
    }
  if (air)
    {
      proposal->migr_table_counter = i;
    }
  else
    {
      proposal->migr_table_counter2 = i;
    }
  return 1;
}

void 
chooseTarget (proposal_fmt * proposal, timelist_fmt * timevector,
	      node ** bordernodes, long *bordernum)
{
  long actualpop = -99;
  node *rb = crawlback (proposal->root->next);
  *bordernum = 0;
  proposal->target = NULL;
  proposal->realtarget = NULL;
  if (proposal->migr_table_counter == 0)
    actualpop = proposal->origin->pop;
  else
    actualpop = proposal->migr_table[proposal->migr_table_counter - 1].from;
  if (rb->tyme < proposal->time)
    {
      error ("why is rb->tyme smaller than proopsal-time in chooseTarget??\n");
    }
  findbordernodes (rb, proposal, actualpop, bordernodes, bordernum,
		   &(*timevector).tl, (*timevector).T);
  if (*bordernum > 0)
    {
      proposal->target = bordernodes[RANDINT (0, (*bordernum) - 1)];
      if (proposal->target != rb)
	{
	  proposal->tsister = showsister (proposal->target);
	  proposal->realtsister = crawlback (proposal->tsister)->back;
	}
      else
	proposal->tsister = NULL;
      proposal->realtarget = proposal->target;
      if (proposal->target->type == 'm')
	proposal->target = crawlback (showtop (proposal->target)->next);
    }
  else
    {
      proposal->target = NULL;
      proposal->tsister = NULL;
      proposal->realtsister = NULL;
      proposal->realtarget = NULL;
    }
}

void 
findbordernodes (node * theNode, proposal_fmt * proposal, long pop,
		 node ** bordernodes, long *bordernum,
		 vtlist ** tyme, long gte)
{
  node *tmp, *back;
  if (theNode == proposal->oback)
    {
      tmp = showtop (crawlback (proposal->osister)->back);
      back = showtop (proposal->oback->back);
    }
  else
    {
      tmp = showtop (theNode);
      back = showtop (theNode->back);
    }
  if (pop == tmp->pop && pop == back->actualpop &&
      tmp->tyme < proposal->time &&
      back->tyme > proposal->time)
    {
      bordernodes[(*bordernum)++] = tmp;
    }
  else
    {
      if (back->tyme < proposal->time)
	return;
      if (!tmp->tip)
	{
	  if (tmp->next->back != NULL)
	    findbordernodes (tmp->next->back, proposal, pop, bordernodes,
			     bordernum, tyme, gte);
	  if (tmp->type != 'm' && tmp->next->next->back != NULL)
	    findbordernodes (tmp->next->next->back, proposal, pop, bordernodes,
			     bordernum, tyme, gte);
	}
    }
}

/*
   boolean
   same_pop(node * up, double tyme, long pop)
   {
   node *oldnn = showtop(up->back);
   node *nn = up;
   while (nn->tyme < tyme) {
   oldnn = nn;
   nn = showtop(nn->back);
   }
   if (oldnn->pop == pop && nn->actualpop == pop)
   return TRUE;
   else
   return FALSE;
   }
 */


/* -----------------------------------------------------------------------
   simulates two lineages at once, if we are extending below the last node */
int 
pre_population (proposal_fmt * proposal, vtlist * ltime, long gte, long *slider)
{
  boolean coalesced = FALSE;
  boolean choice = FALSE;
  long pop1 = -99, pop2 = -98;
  double age1, denom, rr, r0, r1, horizon, mm, mm2;
  if (gte > 0)
    proposal->realtarget = ltime[gte - 1].eventnode;
  else
    proposal->realtarget = ltime[gte].eventnode->next->back;
  if (proposal->realtarget == proposal->oback)
    {
      proposal->realtarget = crawlback (proposal->osister)->back;
    }
  if (proposal->realtarget->type == 'm')
    {
      proposal->target = crawlback (proposal->realtarget->next);
      if (proposal->target == proposal->oback)
	{
	  proposal->target = proposal->osister;
	}
    }
  else
    {
      proposal->target = proposal->realtarget;
    }
  proposal->tsister = NULL;
  pop2 = proposal->realtarget->pop;
  pop1 = proposal->migr_table_counter > 0 ?
    proposal->migr_table[proposal->migr_table_counter - 1].from : proposal->origin->pop;
  age1 = MAX (proposal->realtarget->tyme, proposal->migr_table_counter > 0 ?
	      proposal->migr_table[proposal->migr_table_counter - 1].time :
	      proposal->origin->tyme);
  horizon = MAX (proposal->oback->tyme, age1);
  while (age1 < horizon)
    {
      mm = proposal->param0[proposal->numpop + pop1];
      if (pop1 == pop2)
	{
	  denom = mm + (2. / proposal->param0[pop1]);
	  proposal->time = age1 - log (RANDUM ()) / denom;
	  age1 = proposal->time;
	  if (age1 < horizon)
	    {
	      rr = RANDUM ();
	      r0 = (2. / proposal->param0[pop1]) / denom;
	      if (rr < r0)
		{
		  return 1;
		}
	    }
	}
      else
	{
	  denom = mm;
	  proposal->time = age1 - log (RANDUM ()) / denom;
	  age1 = proposal->time;
	}
      if (age1 < horizon)
	{
	  if (!migrate (proposal, proposal->origin,
			&proposal->old_migr_table_counter, MIGRATION_AIR))
	    {
	      return 0;
	    }
	  pop1 = proposal->migr_table_counter > 0 ?
	    proposal->migr_table[proposal->migr_table_counter - 1].from :
	    proposal->origin->pop;
	}
    }
  age1 = horizon;
  while (!coalesced)
    {
      mm = proposal->param0[pop1 + proposal->numpop];
      mm2 = proposal->param0[pop2 + proposal->numpop];
      denom = mm + mm2;
      if (pop1 == pop2)
	{
	  denom = denom + (2. / proposal->param0[pop1]);
	  proposal->time = age1 - log (RANDUM ()) / denom;
	  age1 = proposal->time;
	  rr = RANDUM ();
	  r0 = ((2. / proposal->param0[pop1]) / denom);
	  r1 = r0 + mm / denom;
	  if (rr < r0)
	    {
	      return 1;
	    }
	  else
	    {
	      if (rr < r1)
		{
		  choice = TRUE;
		}
	      else
		{
		  choice = FALSE;
		}
	    }
	}
      else
	{			/*pop1 not equal pop2 */
	  proposal->time = age1 - log (RANDUM ()) / denom;
	  age1 = proposal->time;
	  if (RANDUM () < (mm / denom))
	    {
	      choice = TRUE;
	    }
	  else
	    {
	      choice = FALSE;
	    }
	}
      if (choice)
	{
	  if (!migrate (proposal, proposal->origin,
			&proposal->old_migr_table_counter, MIGRATION_AIR))
	    {
	      return 0;		/* migration limit reached */
	    }
	  pop1 = proposal->migr_table_counter > 0 ?
	    proposal->migr_table[proposal->migr_table_counter - 1].from :
	    proposal->origin->pop;
	}
      else
	{
	  if (!migrate (proposal, proposal->realtarget,
		     &proposal->old_migr_table_counter2, MIGRATION_IN_TREE))
	    {
	      return 0;		/* migration limit reached */
	    }
	  pop2 = proposal->migr_table_counter2 > 0 ?
	    proposal->migr_table2[proposal->migr_table_counter2 - 1].from :
	    proposal->realtarget->pop;
	}
    }
  error ("Reached the end of function without coalescing");
  return -1;			/*makes the compiler happy */
}

void 
free_proposal (proposal_fmt * proposal)
{
  long j;
  free (proposal->aboveorigin);
  free (proposal->bordernodes);
  free (proposal->line_f);
  free (proposal->line_t);
  if (proposal->datatype == 's')
    {
      for (j = 0; j < proposal->endsite; j++)
	{
	  free (proposal->xt.s[j]);
	  free (proposal->xf.s[j]);
	}
      free (proposal->xf.s);
      free (proposal->xt.s);
    }
  else
    {
      free (proposal->xf.a);
      free (proposal->xt.a);
    }
  free (proposal->migr_table);
  free (proposal->migr_table2);
  free (proposal);
}

void 
free_timevector (timelist_fmt * timevector)
{
  free (timevector[0].tl);
  free (timevector);
}

/*----------------------------------------------------------*
 * rejection/acceptance of the new tree according to the likelihood
 * and an acceptance ratio which is higher the better the
 * likelihood values are (-> Metropolis)
 */
boolean
acceptlike (world_fmt * world, proposal_fmt * proposal, long g, timelist_fmt * tyme)
{
  static boolean report = TRUE, oldg = -1;
  double rr, expo;
  long rm;
  long rmc = rmigrcount (proposal);
  if ((rm = proposal->migr_table_counter + proposal->migr_table_counter2 +
       world->migration_counts - rmc) > MIGRATION_LIMIT)
    {
      if (report || g != oldg)
	{
	  fprintf (stderr, "migration limit (%i) exceeded: %li\n", MIGRATION_LIMIT, rm);
	  report = FALSE;
	  oldg = g;
	}
      return FALSE;
    }
  proposal->likelihood = pseudotreelikelihood (world, proposal);
  if (world->likelihood[g] < proposal->likelihood)
    {
      return TRUE;
    }
  else
    {
      expo = exp (proposal->likelihood - world->likelihood[g]);
      rr = RANDUM ();
      if (rr < expo)
	{
	  return TRUE;
	}
    }
  return FALSE;
}

long 
rmigrcount (proposal_fmt * proposal)
{
  node *p;
  long count = 0;
  for (p = proposal->origin; p != proposal->oback; p = showtop (p->back))
    {
      if (p->type == 'm')
	count++;
    }
  return count;
}

double 
eventtime (proposal_fmt * proposal, long pop, vtlist * tentry, char *event, long g)
{
  static boolean mig_force = TRUE;
  double interval, lines, denom, mm = 0.0;
  lines = 2.0 * tentry->lineages[pop];
  /*    for(i=0;i<proposal->numpop;i++){
     if (i!=pop)
     mm +=  *//*tentry->lineages[i] *//*  proposal->param0[i+proposal->numpop];
     } */
  mm = proposal->param0[pop + proposal->numpop];
  denom = mm + (lines / proposal->param0[pop]);
  interval = -(log (RANDUM ())) / denom;
  if (lines > 0)
    {
      if (RANDUM () < mm / denom)
	{
	  /*      if(proposal->world->atl[0].tl[g].p[pop]==0){
	     if(RANDUM()> 0.90){
	     *event = 'c';
	     return interval;
	     }
	     } */
	  *event = 'm';
	  return interval;
	}
      else
	{
	  /* if migration parameter x is 0 
	     then we will insert a migration in population x
	     with probability so that 4Nm=0.1*theta
	   */
	  if ((proposal->param0[proposal->numpop + pop] <= SMALLEST_MIGRATION))
	    {
	      if (RANDUM () < 0.1)
		{
		  if (mig_force)
		    {
		      fprintf (stderr, "Migration forced\n");
		      mig_force = FALSE;
		    }
		  *event = 'm';
		  return interval;
		}
	    }
	  *event = 'c';
	  return interval;
	}
    }
  else
    {
      *event = 'm';
      return interval;
    }
}

/*--------------------------------------------------------*
 * showsister() 
 * find the sisternode, by going down the branch and up on 
 * the other side again, neglecting the migration nodes.
 */
node *
showsister (node * theNode)
{
  node *tmp = crawlback (theNode);

  if (tmp->next->top)
    {
      return crawlback (tmp->next->next);
    }
  else
    {
      if (tmp->next->next->top)
	{
	  return crawlback (tmp->next);
	}
      else
	{
	  error ("error in treestructure, cannot find sisternode\n");
	}
    }
  return NULL;
}

void 
count_migrations (node * p, long *count)
{
  if (p->type != 't')
    {
      if (p->type == 'm')
	{
	  *count += 1;
	  count_migrations (p->next->back, count);
	}
      else
	{
	  count_migrations (p->next->back, count);
	  count_migrations (p->next->next->back, count);
	}
    }
}
/*
   long interior_pop0(timelist_fmt *timevector, proposal_fmt* proposal)
   {
   long sum=0,i;
   for(i=0;i<timevector->T;i++){
   if((timevector[0].tl[i].from==0) && (timevector[0].tl[i].from==timevector[0].tl[i].to))
   sum += (timevector[0].tl[i].from==0);
   }
   return sum;
   }
 */
