/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 M E N U   R O U T I N E S 

 presents the menu and its submenus.                                                                                                               
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: menu.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"
#include "sequence.h"
#include "fst.h"
#include "options.h"
#include "menu.h"
#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif
/* prototypes ------------------------------------------- */
void print_menu_title (option_fmt * options);
void print_menu_accratio (long a, long b);
long print_title (world_fmt * world);
void get_menu (option_fmt * options);
/* private functions */
void menuData (option_fmt * options, char datatype[]);
void menuInput (option_fmt * options);
void menuParameters (option_fmt * options);
void menuStrategy (option_fmt * options);
void menuSequences (option_fmt * options);


void 
print_menu_title (option_fmt * options)
{
  char nowstr[LINESIZE];
  if (options->menu || options->progress)
    {
      fprintf (stdout, "  =============================================\n");
      fprintf (stdout, "  MIGRATION RATE AND POPULATION SIZE ESTIMATION\n");
      fprintf (stdout, "  using Markov Chain Monte Carlo simulation\n");
      fprintf (stdout, "  2-population  version\n");
      fprintf (stdout, "  =============================================\n");
      fprintf (stdout, "  Version %s\n", VERSION);
      get_time (nowstr, "  %c");
      if (nowstr[0] != '\0')
	fprintf (stdout, "  Program started at %s\n", nowstr);
      fprintf (stdout, "\n\n");
    }
}

void 
print_menu_accratio (long a, long b)
{
  fprintf (stdout, "           Acceptance-ratio = %li/%li (%f)\n", a, b, (double) a / (double) b);
}

long 
print_title (world_fmt * world)
{
  char nowstr[LINESIZE];
  long len, i, filepos = -1;
  if (!world->options->simulation)
    {
      if (world->options->title[0] == '\0')
	{
	  fprintf (world->outfile, "  ====================================================\n");
	  fprintf (world->outfile, "     MIGRATION RATE AND POPULATION SIZE ESTIMATION\n");
	  fprintf (world->outfile, "       using Markov Chain Monte Carlo simulation\n");
	  fprintf (world->outfile, "  ====================================================\n");
	  fprintf (world->outfile, "  Version %s\n", VERSION);
	  get_time (nowstr, "%c");
	  if (nowstr[0] != '\0')
	    {
	      fprintf (world->outfile, "  Program started at %s\n", nowstr);
	      filepos = ftell (world->outfile);
	      fprintf (world->outfile, "                     %*s \n", (int) strlen (nowstr), " ");
	    }
	  fprintf (world->outfile, "\n\n");
	}
      else
	{
	  len = MAX (strlen (world->options->title), 45);
	  fprintf (world->outfile, "  ");
	  for (i = 0; i < len; i++)
	    fprintf (world->outfile, "=");
	  fprintf (world->outfile, "\n  %s\n  ", world->options->title);
	  for (i = 0; i < len; i++)
	    fprintf (world->outfile, "=");
	  fprintf (world->outfile, "\n  MIGRATION RATE AND POPULATION SIZE ESTIMATION\n");
	  fprintf (world->outfile, "  using Markov Chain Monte Carlo simulation\n  ");
	  fprintf (stdout, "  2-population  version\n");
	  for (i = 0; i < len; i++)
	    fprintf (world->outfile, "=");
	  fprintf (world->outfile, "\n  Version %s\n", VERSION);
	  get_time (nowstr, "%c");
	  if (nowstr[0] != '\0')
	    {
	      fprintf (world->outfile, "\n  Program started at %s\n", nowstr);
	      filepos = ftell (world->outfile);
	      fprintf (world->outfile, "                                                   \n");
	    }
	  fprintf (world->outfile, "\n\n");
	}
    }
  return filepos;
}


void 
get_menu (option_fmt * options)
{
  char input[LINESIZE];
  char datatype[30];
  if (options->menu)
    {
      switch (options->datatype)
	{
	case 'a':
	  strcpy (datatype, "infinite allele model");
	  break;
	case 'm':
	  strcpy (datatype, "microsatellite model");
	  break;
	case 'b':
	  strcpy (datatype, "microsatellite model [Brownian motion]");
	  break;
	case 's':
	  strcpy (datatype, "sequence model");
	  break;
	default:
	  options->datatype = 'a';
	  strcpy (datatype, "infinite allele model");
	  break;
	}
      do
	{
	  printf ("  Settings for this run:\n");
	  printf ("  D       Data type (currently set: %s)\n", datatype);
	  printf ("  I       Input/Output formats\n");
	  printf ("  P       Start values for the Parameters\n");
	  printf ("  S       Search strategy\n");
	  printf ("\n\n");
	  printf ("  Are the settings correct?\n");
	  printf ("  (Type Y or the letter for one to change)\n");
	  fgets (input, LINESIZE, stdin);
	  switch (uppercase (input[0]))
	    {
	    case 'D':
	      menuData (options, datatype);
	      break;
	    case 'I':
	      menuInput (options);
	      break;
	    case 'P':
	      menuParameters (options);
	      break;
	    case 'S':
	      menuStrategy (options);
	      break;
	    default:
	      break;
	    }
	}
      while (uppercase (input[0]) != 'Y');
      if (options->usertree && options->datatype != 's')
	options->usertree = FALSE;
    }
}

/* private functions--------------------------------------------------- */
void 
menuData (option_fmt * options, char datatype[])
{
  static boolean didchangecat, didchangercat;
  static boolean ttr;
  long z = 0;
  char input[LINESIZE];
  char starttree[LINESIZE];

  /* in case we have  already read a parmfile */
  if (options->categs > 1)
    didchangecat = TRUE;
  else
    didchangecat = FALSE;
  if (options->rcategs > 1)
    didchangercat = TRUE;
  else
    didchangercat = FALSE;

  do
    {
      switch (options->datatype)
	{
	case 'a':
	  strcpy (datatype, "infinite allele model");
	  break;
	case 'm':
	  strcpy (datatype, "microsatellite model [Ladder model]");
	  break;
	case 'b':
	  strcpy (datatype, "microsatellite model [Brownian motion model]");
	  break;
	case 's':
	  strcpy (datatype, "sequence model");
	  break;
	default:
	  options->datatype = 'a';
	  strcpy (datatype, "infinite allele model");
	  break;
	}
      if (options->usertree && options->datatype == 's')
	sprintf (starttree, "is supplied in %s", options->utreefilename);
      else
	strcpy (starttree, "is estimated using a UPGMA topology");
      printf ("  DATATYPE AND DATA SPECIFIC OPTIONS\n\n");
      if (options->datatype == 's')
	printf ("  0       Start genealogy %s\n", starttree);
      printf ("  1       Datatype is currently a %s\n", datatype);
      switch (options->datatype)
	{
	case 'a':
	  printf ("          (using a k-allele model which simulates an infinite allele model)\n");
	  break;
	case 'm':
	  printf ("          (using a simple allele ladder model)\n");
	  printf ("  2       Size of the step matrix:      %li\n", options->micro_stepnum);
	  printf ("  3       Threshold value:              %li\n\n\n\n", options->micro_threshold);
	  break;
	case 's':
	  printf ("  2       Transition/transversion ratio:");
	  z = 0;
	  while (options->ttratio[z] > 0.0)
	    printf ("%8.4f ", options->ttratio[z++]);
	  printf ("\n");
	  printf ("  3       Use empirical base frequencies?  %s\n",
		  (options->freqsfrom ? "Yes" : "No"));
	  printf ("  4       One category of sites?");
	  if (options->categs == ONECATEG)
	    printf ("  One category\n");
	  else
	    printf ("  More than one  category of sites\n");
	  printf ("  5       One region of substitution rates?");
	  if (options->rcategs == 1)
	    printf ("  Yes\n");
	  else
	    {
	      printf ("  %ld categories of regions\n", options->rcategs);
	      printf ("  6       Rates at adjacent sites correlated?");
	      if (!options->autocorr)
		printf ("  No, they are independent\n");
	      else
		printf ("  Yes, mean block length =%6.1f\n", 1.0 / options->lambda);
	    }
	  printf ("  7       Sites weighted?  %s\n",
		  (options->weights ? "Yes" : "No"));
	  printf ("  8       Input sequences interleaved?  %s\n",
		  (options->interleaved ? "Yes" : "No, sequential"));
	  break;
	}
      printf ("\n\n");
      printf ("  Are the settings correct?\n");
      printf ("  (Type Y or the number of the entry to change)\n");
      fgets (input, LINESIZE, stdin);
      switch (uppercase (input[0]))
	{
	case '0':
	  if (options->datatype == 's')
	    options->usertree = !options->usertree;
	  break;
	case '1':
	  do
	    {
	      printf ("  (a)llele model\n");
	      printf ("  (m)icrosatellite model [Ladder model; exact but slow]\n");
	      printf ("  (b)rownian microsatellite model [Brownian motion model]\n");
	      printf ("  (s)equence model\n");
	      fgets (input, LINESIZE, stdin);
	    }
	  while (strchr ("ambs", (int) (lowercase (input[0]))) == NULL);
	  options->datatype = input[0];
	  if (options->datatype != 's')
	    options->usertree = FALSE;
	  break;
	default:
	  break;
	}
      if (options->datatype == 'm')
	{
	  switch (uppercase (input[0]))
	    {
	    case '2':		/*micro-stepmax */
	      printf ("  What is the maximal size of the stepmatrix?\n");
	      printf ("  [the bigger the longer the run and the more accurate the estimate]\n");
	      fscanf (stdin, "%ld%*[^\n]", &options->micro_stepnum);
	      break;
	    case '3':		/*micro-threshold */
	      printf ("  What is the threshold value?\n");
	      printf ("  E.g. if your allele is 24 and the threshold is 10\n");
	      printf ("  there is some probability that the allele 24 can\n");
	      printf ("  change to allele 14 (or 38), but there is a probability\n");
	      printf ("  of 0.0 (ZERO) to go to 13 (39),\n");
	      printf ("  if you choose this too small, than the program will fail\n");
	      printf ("  [the bigger the longer the run and the more\n accurate the estimate]\n");
	      fscanf (stdin, "%ld%*[^\n]", &options->micro_threshold);
	      break;
	    }
	}
      if (options->datatype == 's')
	{
	  switch (uppercase (input[0]))
	    {
	    case '2':
	      ttr = !ttr;
	      if (ttr)
		initratio (options);
	      break;
	    case '3':
	      options->freqsfrom = !options->freqsfrom;
	      if (!options->freqsfrom)
		{
		  initfreqs (&options->freqa, &options->freqc,
			     &options->freqg, &options->freqt);
		}
	      break;
	    case '4':
	      if (options->categs == ONECATEG)
		{
		  options->categs = MANYCATEGS;
		  didchangecat = TRUE;
		}
	      else
		{
		  options->categs = ONECATEG;
		  didchangecat = FALSE;
		}
	      /*printf("\n  Sitewise user-assigned categories:\n\n");
	         initcatn(&options->categs);
	         options->rate = (double *) realloc(options->rate,
	         options->categs * sizeof(double));
	         didchangecat = TRUE;
	         initcategs(options->categs, options->rate); */
	      break;
	    case '5':
	      if (options->rcategs == 1)
		{
		  options->autocorr = FALSE;
		  didchangercat = FALSE;
		}
	      else
		{
		  printf ("\n  Regional rates:\n");
		  initcatn (&options->rcategs);
		  options->probcat = (double *) realloc (options->probcat,
					options->rcategs * sizeof (double));
		  options->rrate = (double *) realloc (options->rrate,
					options->rcategs * sizeof (double));
		  didchangercat = TRUE;
		  initcategs (options->rcategs, options->rrate);
		  initprobcat (options->rcategs, &options->probsum, options->probcat);
		}
	      break;
	    case '6':
	      options->autocorr = !options->autocorr;
	      if (options->autocorr)
		initlambda (options);
	      break;
	    case '7':
	      options->weights = !options->weights;
	      break;
	    case '8':
	      options->interleaved = !options->interleaved;
	      break;
	    default:
	      break;
	    }
	  if (!didchangercat)
	    {
	      options->probcat = (double *) realloc (options->probcat, sizeof (double) * 2);
	      options->rrate = (double *) realloc (options->rrate, sizeof (double) * 2);
	      options->rrate[0] = 1.0;
	      options->probcat[0] = 1.0;

	    }
	  if (!didchangecat)
	    {
	      options->rate = (double *) realloc (options->rate, sizeof (double) * 2);
	      options->rate[0] = 1.0;
	    }

	}
    }
  while (uppercase (input[0]) != 'Y');
}

void 
menuInput (option_fmt * options)
{
  char input[LINESIZE];
  char *stringstep, *string2, *string3;
  char *treestr[] =
  {"None", "All", "Best", "Last chain"};

  stringstep = (char *) malloc (sizeof (char) * 128);
  string2 = (char *) malloc (sizeof (char) * 128);
  string3 = (char *) malloc (sizeof (char) * 128);

  do
    {
      printf ("  INPUT/OUTPUT FORMATS\n\n");
      printf ("  1       Echo the data at start of run?         %s\n",
	      options->printdata ? "Yes" : "No");
      printf ("  2       Print indications of progress of run?  %s\n",
	      options->progress ? "Yes" : "No");
      if (options->plot)
	{
	  switch (options->plotmethod)
	    {
	    case PLOTALL:
	      strcpy (string2, "Yes, to outfile and mathfile");
	      break;
	    default:
	      strcpy (string2, "Yes, to outfile");
	      break;
	    }
	}
      else
	{
	  strcpy (string2, "No");
	}
      printf ("  3       Plot likelihood surface?               %s\n",
	      string2);
      switch (options->treeprint)
	{
	case ALL:
	  printf ("  4       Print genealogies?                     %s\n", treestr[1]);
	  break;
	case BEST:
	  printf ("  4       Print genealogies?                     %s\n", treestr[2]);
	  break;
	case LASTCHAIN:
	  printf ("  4       Print genealogies?                     %s\n", treestr[3]);
	  break;
	case NONE:
	default:
	  printf ("  4       Print genealogies?                     %s\n", treestr[0]);
	  break;
	}
      switch (options->autoseed)
	{
	case AUTO:
	  sprintf (stringstep, "Yes");
	  break;
	case NOAUTO:
	  sprintf (stringstep, "No, use seedfile");
	  break;
	case NOAUTOSELF:
	  sprintf (stringstep, "No, seed=%li ", options->inseed);
	  break;
	default:
	  options->autoseed = AUTO;
	  sprintf (stringstep, "Yes");
	  break;
	}
      printf ("  5       Use automatic seed for randomisation?  %s\n",
	      stringstep);

      printf ("  6       Datafile name is %s\n",
	      options->infilename);

      printf ("  7       Outputfile name is %s\n",
	      options->outfilename);

      printf ("  8       Plot coordinates are saved in %s\n",
	      options->mathfilename);

      printf ("  T       Title of the analysis is\n");
      printf ("          %s\n", options->title);

      printf ("\n\n  Are the settings correct?\n");
      printf ("  (type Y to go back to the main menu or a number for the entry to change)\n");
      fgets (input, LINESIZE, stdin);
      switch (uppercase (input[0]))
	{
	case '1':
	  options->printdata = !options->printdata;
	  break;
	case '2':
	  options->progress = !options->progress;
	  break;
	case '3':
	  options->plot = !options->plot;
	  if (options->plot)
	    {
	      do
		{
		  printf ("  Plot Likelihood surface:\n");
		  printf ("  (B)oth to outfile and mathfile, (O)utfile only, (N)o plot\n");
		  fgets (input, LINESIZE, stdin);
		}
	      while (strchr ("BON", (int) uppercase (input[0])) == NULL);
	      switch (uppercase (input[0]))
		{
		case 'B':
		  options->plotmethod = PLOTALL;
		  break;
		case 'O':
		  options->plotmethod = PLOTOUTFILE;
		  break;
		case 'N':
		  options->plot = FALSE;
		  break;
		}
	    }
	  break;
	case '4':
	  do
	    {
	      printf ("  Print genealogies:\n");
	      printf ("  (N)one, (A)all [!], (B)est, (L)ast chain\n");
	      fgets (input, LINESIZE, stdin);
	    }
	  while (strchr ("NABL", (int) uppercase (input[0])) == NULL);
	  switch (uppercase (input[0]))
	    {
	    case 'N':
	      options->treeprint = NONE;
	      break;
	    case 'A':
	      options->treeprint = ALL;
	      break;
	    case 'B':
	      options->treeprint = BEST;
	      break;
	    case 'L':
	      options->treeprint = LASTCHAIN;
	      break;
	    default:
	      options->treeprint = NONE;
	      break;
	    }
	  break;
	case '5':
	  do
	    {
	      printf ("  (A)utomatic or (S)eedfile or (O)wn\n");
	      printf ("  Start value for Random-generator seed\n");
	      fgets (input, LINESIZE, stdin);
	      switch (uppercase (input[0]))
		{
		case 'A':
		  options->autoseed = AUTO;
		  options->inseed = time (0);
		  break;
		case 'S':
		  options->seedfile = fopen (SEEDFILE, "r");
		  if (options->seedfile)
		    {
		      options->autoseed = NOAUTO;
		      fscanf (options->seedfile, "%ld%*[^\n]", &options->inseed);
		      fclose (options->seedfile);
		    }
		  else
		    printf ("\n\n  There is no seedfile present\n");
		  break;
		case 'O':
		  options->autoseed = NOAUTOSELF;
		  printf ("  Random number seed (best values are x/4 +1)?\n");
		  scanf ("%ld%*[^\n]", &options->inseed);
		  break;
		}
	    }
	  while (options->autoseed < AUTO || options->autoseed > NOAUTOSELF);
	  break;
	case '6':
	  printf ("  What is the datafile name?\n");
	  fgets (input, LINESIZE, stdin);
	  input[strlen(input)-1] = '\0';
	  if (input[0] == '\0')
	    strcpy (options->infilename, "infile");
	  else
	    strcpy (options->infilename, input);
	  break;
	case '7':
	  printf ("  What is the output filename?\n");
	  fgets (input, LINESIZE, stdin);
	  input[strlen(input)-1] = '\0';	  
	  if (input[0] == '\0')
	    strcpy (options->outfilename, "outfile");
	  else
	    strcpy (options->outfilename, input);
	  break;
	case '8':
	  printf ("  What is the plot coordinate filename?\n");
	  fgets (input, LINESIZE, stdin);
	  input[strlen(input)-1] = '\0';
	  if (input[0] == '\0')
	    strcpy (options->mathfilename, "mathfile");
	  else
	    strcpy (options->mathfilename, input);
	  break;
	case 'T':
	  printf ("  Enter a title? [max 80 Characters]\n");
	  fgets (input, LINESIZE, stdin);
	  if (input[0] == '\0')
	    options->title[0] = '\0';
	  else
	    strncpy (options->title, input, 80);
	  break;
	default:
	  break;
	}
    }
  while (uppercase (input[0]) != 'Y');
  free (stringstep);
  free (string2);
  free (string3);
}


void 
menuParameters (option_fmt * options)
{
  char input[LINESIZE];

  do
    {
      printf ("  START VALUES FOR PARAMETERS\n\n");
      printf ("  1       Use a simple estimate of theta as start?");
      switch (options->thetaguess)
	{
	case FST:
	  printf (" Estimate with FST (Fw/Fb) measure\n");
	  break;
	default:
	  printf (" No, initial theta = {%6.4f, %6.4f}\n",
		  options->param0[0], options->param0[1]);
	}
      printf ("  2       Use a simple estimate of migration rate as start?");
      switch (options->migrguess)
	{
	case FST:
	  printf (" Estimate with FST (Fw/Fb) measure\n");
	  break;
	default:
	  printf (" No, initial 4Nm = {%6.4f, %6.4f}\n",
		  options->param0[2], options->param0[3]);
	}
      printf ("  3       Mutation rate is constant? %s\n",
	      !options->gamma ? "Yes" : "No, varying");
      printf ("\n\n\n  FST-CALCULATION\n\n");
      printf ("  4       %s\n", options->fsttype == 'M' ?
	      "Variable Theta, M is the same for all populations" :
	      "Variable M, Theta is the same for all populations");

      printf ("\n\n  Are the settings correct?\n");
      printf ("  (Type Y to go back to the main menu or a number for an entry to change)\n");
      fgets (input, LINESIZE, stdin);
      switch (input[0])
	{
	case '1':
	  printf ("  Which method? (F)st or (O)wn value\n");
	  fgets (input, LINESIZE, stdin);
	  if (uppercase (input[0]) == 'F')
	    options->thetaguess = FST;
	  else
	    {
	      options->thetaguess = OWN;
	      do
		{
		  printf ("  Initial Theta estimates?\nGive both start values separated by blanks\n");
		  fgets (input, LINESIZE, stdin);
		  sscanf (input, "%lf %lf", &options->param0[0], &options->param0[1]);
		}
	      while (options->param0[0] <= 0.0 || options->param0[1] <= 0.0);
	    }
	  break;
	case '2':
	  printf ("  (F)ST or (O)wn value\n");
	  fgets (input, LINESIZE, stdin);
	  if (uppercase (input[0]) == 'F')
	    options->migrguess = FST;
	  else
	    {
	      if (uppercase (input[0]) == 'O')
		options->migrguess = OWN;
	      do
		{
		  printf ("  Initial 4Nm estimates?\nGive both values separated by blanks\n");
		  fgets (input, LINESIZE, stdin);
		  sscanf (input, "%lf %lf", &options->param0[2], &options->param0[3]);
		}
	      while (options->param0[2] < 0.0 || options->param0[3] < 0.0);
	    }
	  break;
	case '3':
	  options->gamma = !options->gamma;
	  break;
	case '4':
	  if (options->fsttype == 'M')
	    options->fsttype = 'T';
	  else
	    options->fsttype = 'M';
	  fst_type (options->fsttype);
	  break;
	default:
	  break;
	}
    }
  while (uppercase (input[0]) != 'Y');
}


void 
menuStrategy (option_fmt * options)
{
  char input[LINESIZE];
  do
    {
      printf ("  SEARCH STRATEGY\n\n");
      printf ("  1       Number of short chains to run?           %3ld\n",
	      options->schains);
      if (options->schains > 0)
	{
	  printf ("  2       Short sampling increment?             %6ld\n",
		  options->sincrement);
	  printf ("  3       Number of steps along short chains?   %6ld\n",
		  options->ssteps);
	}
      printf ("  4       Number of long chains to run?            %3ld\n",
	      options->lchains);
      if (options->lchains > 0)
	{
	  printf ("  5       Long sampling increment?              %6ld\n",
		  options->lincrement);
	  printf ("  6       Number of steps along long chains?    %6ld\n",
		  options->lsteps);
	}
      printf ("\n  ---------------------------------------------------------\n");
      printf ("  Obscure options (consult the documentation on these)\n\n");
      if (options->movingsteps)
	printf ("  7       Sample a fraction of %2.2f new genealogies? Yes\n",
		options->acceptfreq);
      else
	printf ("  7       Sample at least a fraction of new genealogies? No\n");
      printf ("  8       Number of genealogies to discard at \n");
      printf ("          the beginning of each chain? [Burn-in]%6d\n",
	      options->burn_in);
      printf ("  9       Epsilon of parameter likelihood\n");
      printf ("          [please read the manual for this!] %9.5f\n",
	      options->lcepsilon);

      printf ("\n\n  Are the settings correct?\n");
      printf ("  (Type Y to go back to the main menu or a number for an entry to change)\n");
      fgets (input, LINESIZE, stdin);
      switch (input[0])
	{
	case '1':
	  do
	    {
	      printf ("  How many Short Chains?\n");
	      fgets (input, LINESIZE, stdin);
	      options->schains = atoi (input);
	      if (options->schains < 0)
		printf ("  Must be non-negative\n");
	    }
	  while (options->schains < 0);
	  break;
	case '2':
	  do
	    {
	      printf ("  How many trees to skip?\n");
	      fgets (input, LINESIZE, stdin);
	      options->sincrement = atoi (input);
	      if (options->sincrement <= 0)
		printf ("  Must be positive\n");
	    }
	  while (options->sincrement <= 0);
	  break;
	case '3':
	  do
	    {
	      printf ("  How many trees to sample?\n");
	      fgets (input, LINESIZE, stdin);
	      options->ssteps = atoi (input);
	      if (options->ssteps <= 0)
		printf ("  Must be a positive integer\n");
	    }
	  while (options->ssteps <= 0);
	  break;
	case '4':
	  do
	    {
	      printf ("  How many Long Chains?\n");
	      fgets (input, LINESIZE, stdin);
	      options->lchains = atoi (input);
	      if (options->lchains < 0)
		printf ("  Must be non-negative\n");
	    }
	  while (options->lchains < 0);
	  break;
	case '5':
	  do
	    {
	      printf ("  How many trees to skip?\n");
	      fgets (input, LINESIZE, stdin);
	      options->lincrement = atoi (input);
	      if (options->lincrement <= 0)
		printf ("  Must be positive\n");
	    }
	  while (options->lincrement <= 0);
	  break;
	case '6':
	  do
	    {
	      printf ("  How many trees to sample?\n");
	      fgets (input, LINESIZE, stdin);
	      options->lsteps = atoi (input);
	      if (options->lsteps <= 0)
		printf ("  Must be a positive integer\n");
	    }
	  while (options->lsteps <= 0);
	  break;

	case '7':
	  options->movingsteps = !options->movingsteps;
	  if (options->movingsteps)
	    {
	      do
		{
		  printf ("  How big should the fraction of new genealogies\n");
		  printf ("  of the originally proposed number of samples be?\n");
		  fgets (input, LINESIZE, stdin);
		  options->acceptfreq = atof (input);
		  if (options->acceptfreq < 0)
		    printf ("  Range should be between 0 - 1, and not %f\n", options->acceptfreq);
		}
	      while (options->acceptfreq < 0);
	    }
	  break;
	case '8':
	  do
	    {
	      printf ("  How many genealogies to discard?\n");
	      fgets (input, LINESIZE, stdin);
	      options->burn_in = atoi (input);
	      if (options->burn_in <= 0)
		printf ("  Must be a positive integer or zero (0)\n");
	    }
	  while (options->burn_in < 0);
	  break;
	case '9':
	  do
	    {
	      printf ("  Parameter likelihood epsilon?\n");
	      fgets (input, LINESIZE, stdin);
	      options->lcepsilon = atof (input);
	      if (options->lcepsilon <= 0)
		printf ("  Must be a positive value, be warned: too small values will run the program forever\n");
	    }
	  while (options->lcepsilon <= 0);
	  break;
	default:
	  break;
	}
    }
  while (uppercase (input[0]) != 'Y');
}
