/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 O P T I O N S   R O U T I N E S 

 creates options structures,
 reads options from parmfile if present
 
 prints options,
 and finally helps to destroy itself.
                                                                                                               
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: options.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/
#include <stdio.h>
#include <time.h>
#include "migration.h"
#include "fst.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

/* parmfile parameter specifications and keywords */
#define LINESIZE 1024
#define NUMBOOL 11
#define BOOLTOKENS {"menu","interleaved","printdata","-reserved-",\
                          "moving-steps","freqs-from-data","usertree", \
                          "autocorrelation", "simulation","plot", "weights"}
#define NUMNUMBER 30
#define NUMBERTOKENS {"ttratio","short-chains",\
 "short-steps","short-inc","long-chains",\
 "long-steps", "long-inc", "theta", \
 "nmlength","seed","migration","mutation",\
 "datatype", "categories","rates","prob-rates", \
 "micro-max", "micro-threshold", "delimiter","burn-in",\
 "infile", "outfile", "mathfile", "title", \
 "long-chain-epsilon","print-tree","progress","l-ratio",\
 "fst-type","profile"};



/* prototypes ------------------------------------------- */
void create_options (option_fmt ** options);
void init_options (option_fmt * options);
void get_options (option_fmt * options);
void set_param (world_fmt * world, data_fmt * data, option_fmt * options, long locus);
void set_profile_options (option_fmt * options);
void print_menu_options (option_fmt * options, world_fmt * world);
void print_options (FILE * file, option_fmt * options, world_fmt * world);
void decide_plot (option_fmt * options, long chain, long chains, char type);
void destroy_options (option_fmt * options);

/* private functions */
boolean booleancheck (option_fmt * options, char *var, char *value);
boolean numbercheck (option_fmt * options, char *var, char *value);
void reset_oneline (option_fmt * options, long position);
void read_theta (option_fmt * options);
void read_mig (option_fmt * options);
char skip_space (option_fmt * options);
/*======================================================*/
void 
create_options (option_fmt ** options)
{
  (*options) = (option_fmt *) calloc (1, sizeof (option_fmt));
}

void 
init_options (option_fmt * options)
{
  long i;
  /* General options --------------------------------------- */
  options->nmlength = DEFAULT_NMLENGTH;
  options->popnmlength = DEFAULT_POPNMLENGTH;
  options->allelenmlength = DEFAULT_ALLELENMLENGTH;

  /* input/output options ---------------------------------- */
  options->menu = TRUE;
  options->progress = TRUE;
  options->verbose = FALSE;
  options->printdata = FALSE;
  options->usertree = FALSE;
  options->treeprint = NONE;
  options->printfst = TRUE;
  options->fsttype = THETAVARIABLE;
  fst_type (options->fsttype);
  options->plot = TRUE;
  options->plotmethod = PLOTALL;	/* outfile and mathematica file */
  options->simulation = FALSE;
  options->movingsteps = FALSE;
  options->acceptfreq = 0.1;
  strcpy (options->infilename, INFILE);
  strcpy (options->outfilename, OUTFILE);
  strcpy (options->mathfilename, MATHFILE);
  strcpy (options->treefilename, TREEFILE);
  strcpy (options->utreefilename, UTREEFILE);
  strcpy (options->catfilename, CATFILE);
  strcpy (options->weightfilename, WEIGHTFILE);
  strcpy (options->title, "\0");
  options->lratio = (lratio_fmt *) calloc (1, sizeof (lratio_fmt));
  options->lratio->data = (lr_data_fmt *) calloc (1, sizeof (lr_data_fmt) * 1);
  options->lratio->data[0].value = (char *) calloc (1, sizeof (char) * LINESIZE);
  options->lratio->alloccounter = 1;
  options->profile = ALL;
  options->qdprofile = FALSE;
  options->printprofsummary = TRUE;
  options->printprofile = TRUE;
  /* data options ------------------------------------------ */
  options->datatype = 's';
  options->migration_model = MATRIX;
  options->thetag = (double *) calloc (1, sizeof (double) * NUMPOP);
  options->mg = (double *) calloc (1, sizeof (double) * NUMPOP);
  options->param0 = (double *) calloc (1, sizeof (double) * NUMPOP * 2);
  options->gamma = FALSE;
  /* EP data */
  options->dlm = '\0';
  /* microsat data */
  options->micro_threshold = MICRO_THRESHOLD;
  options->micro_stepnum = MAX_MICROSTEPNUM;
  /*sequence data */
  options->interleaved = FALSE;
  options->ttratio = (double *) calloc (1, sizeof (double) * 2);
  options->ttratio[0] = 2.0;
  options->freqsfrom = TRUE;
  options->categs = ONECATEG;
  options->rate = (double *) calloc (1, sizeof (double));
  options->rate[0] = 1.0;
  options->rcategs = 1;
  options->rrate = (double *) calloc (1, sizeof (double));
  options->probcat = (double *) calloc (1, sizeof (double));
  options->autocorr = FALSE;
  options->rrate[0] = 1.0;
  options->probcat[0] = 1.0;

  options->probsum = 0.0;

  options->lambda = 1.0;
  options->weights = FALSE;
  /* random number options --------------------------------- */
  options->autoseed = AUTO;

  /* mcmc options ------------------------------------------ */
  options->thetaguess = FST;
  options->migrguess = FST;
  for (i = 0; i < NUMPOP; i++)
    {
      options->thetag[i] = 1.0;
      options->mg[i] = 1.0;
    }
  options->param0[0] = 1.0;
  options->param0[1] = 1.0;
  options->param0[2] = 1.0;
  options->param0[3] = 1.0;
  options->numthetag = options->nummg = 0;
  options->schains = 10;
  options->sincrement = 20;
  options->ssteps = 100;
  options->lchains = 3;
  options->lincrement = 20;
  options->lsteps = 1000;
  options->burn_in = TWOHUNDRED;
  options->lcepsilon = LONGCHAINEPSILON;
}

void 
get_options (option_fmt * options)
{
  long counter = 0;
  long position = 0;
  char varvalue[LINESIZE];
  char parmvar[LINESIZE];
  char input[LINESIZE];
  char *p;

  options->parmfile = fopen (PARMFILE, "r");
  if (options->parmfile)
    {
      counter = 0;
      position = ftell (options->parmfile);
      while (fgets (input, LINESIZE, options->parmfile) != NULL)
	{
	  counter++;
	  if ((input[0] == '#') || isspace ((int) input[0]))
	    continue;
	  else
	    {
	      if (!isalpha ((int) input[0]))
		{
		  fprintf (stderr, "the parmfile contains an error on the line %li\n",
			   counter);
		  exit (EXIT_FAILURE);
		}
	    }
	  if ((p = strchr (input, '#')) != NULL)
	    *p = '\n';
	  if (!strncmp (input, "end", 3))
	    break;
	  strcpy (parmvar, strtok (input, "="));
	  /* for version 2.0 (n-population)
	     if(!strncmp(parmvar,"theta",5)){
	     reset_oneline(options,position);
	     read_theta(options);
	     position=ftell(options->parmfile);
	     continue;
	     }
	     if(!strncmp(parmvar,"migration",5)){
	     reset_oneline(options,position);
	     read_mig(options);
	     position=ftell(options->parmfile);
	     continue;
	     } */
	  strcpy (varvalue, strtok (NULL, "\n"));
	  if (!booleancheck (options, parmvar, varvalue))
	    {
	      if (!numbercheck (options, parmvar, varvalue))
		fprintf (stderr, "Inappropiate entry in parmfile: %s ignored\n", input);
	    }
	  position = ftell (options->parmfile);
	}
    }
}

void 
print_menu_options (option_fmt * options, world_fmt * world)
{
  if (options->numpop > world->numpop)
    error ("Inconsistency between your Menu/Parmfile and your datafile\n \
Check the number of populations!\n");
  if (options->progress)
    {
      print_options (stdout, options, world);
    }
}

void 
print_options (FILE * file, option_fmt * options, world_fmt * world)
{
  /*for Version 2.0 long i,j, tt; */
  char mytext[LINESIZE];
  char seedgen[LINESIZE], spacer[LINESIZE];
  char paramtgen[LINESIZE], parammgen[LINESIZE];
  switch ((short) options->autoseed)
    {
    case AUTO:
      strcpy (seedgen, "with internal timer");
      strcpy (spacer, "");
      break;
    case NOAUTOSELF:
      strcpy (seedgen, "from parmfile");
      strcpy (spacer, "      ");
      break;
    case NOAUTO:
      strcpy (seedgen, "from seedfile");
      strcpy (spacer, "      ");
      break;
    default:
      strcpy (seedgen, "ERROR");
      strcpy (spacer, " ");
      break;
    }
  switch (options->thetaguess)
    {
    case OWN:
      strcpy (paramtgen, "from guessed values");
      break;
    case FST:
      strcpy (paramtgen, "from FST-calculation");
      break;
    default:
      strcpy (paramtgen, "ERROR");
      break;
    }
  switch (options->migrguess)
    {
    case OWN:
      strcpy (parammgen, "from guessed values");
      break;
    case FST:
      strcpy (parammgen, "from FST-calculation");
      break;
    default:
      strcpy (parammgen, "ERROR");
      break;
    }
  fprintf (file, "Options in use:\n");
  fprintf (file, "---------------\n");
  fprintf (file, "Datatype:%-44.44s\n",
	   options->datatype == 'a' ?
	   "Allelic data" :
	   (options->datatype == 's' ?
	    "Sequence data" :
	    (options->datatype == 'm' ? "Microsatellite data" :
	     "Microsatellite data [Brownian motion model]")));
  fprintf (file, "Random number seed (%s)%s%20li\n",
	   seedgen, spacer, options->saveseed);
  fprintf (file, "Start parameters:\n   Theta values were generated ");
  fprintf (file, " %s\n", paramtgen);
  if (options->thetaguess == OWN)
    fprintf (file, "   Theta = {%.5f,%.5f}\n",
	     options->param0[0], options->param0[1]);
  fprintf (file, "   M values were generated %s\n", parammgen);
  if (options->migrguess == OWN)
    fprintf (file, "   4Nm   = {%.5f,%.5f}\n",
	     options->param0[2], options->param0[3]);
  /* for version 2.0
     if (options->thetaguess == OWN){
     fprintf(file, "   Theta = ");
     for(i=0;i<options->numthetag;i++){
     fprintf(file,"%.5f,",options->thetag[i]);
     }
     fprintf(file,"\n");
     }
     fprintf(file, "   M values were generated %s\n", parammgen);
     if (options->migrguess == OWN){
     tt=0;
     fprintf(file, "   4Nm-matrix: ");
     if(options->nummg==1){
     fprintf(file,"%5.2f [all are the same]\n",options->mg[tt++]);
     }
     else {
     for(i=0;i<world->numpop;i++){
     for(j=0;j<world->numpop;j++){
     if(i!=j)
     fprintf(file,"%5.2f ",options->mg[tt++]); 
     else
     fprintf(file,"----- ");
     if(j>10)
     fprintf(file,"\n                ");
     }
     fprintf(file,"\n               ");
     }
     fprintf(file,"\n");
     }
     }
     end for version 2.0  */
  fprintf (file, "Gamma-distributed mutation rate %s\n",
	   options->gamma ? "is used" : "is not used");
  fprintf (file, "Markov chain parameters:\n");
  fprintf (file, "   Short chains (short-chains):         %20li\n", options->schains);
  fprintf (file, "      Trees sampled (short-inc*steps):  %20li\n",
	   options->sincrement * options->ssteps);
  fprintf (file, "      Trees recorded (short-steps):     %20li\n",
	   options->ssteps);
  fprintf (file, "   Long chains (long-chains):           %20li\n",
	   options->lchains);
  fprintf (file, "      Trees sampled (long-inc*steps):   %20li\n",
	   options->lincrement * options->lsteps);
  fprintf (file, "      Trees recorded (long-steps):      %20li\n",
	   options->lsteps);
  if (options->movingsteps)
    {
      fprintf (file, "   Percentage of new genealogies:       %20.2f\n",
	       (double) options->acceptfreq);
    }
  if (options->burn_in > 0)
    {
      fprintf (file, "   Number of discard trees per chain:   %20li\n",
	       (long) options->burn_in);
    }
  if (options->lcepsilon < LONGCHAINEPSILON)
    {
      fprintf (file, "   parameter-likelihood epsilon:        %20.5f\n",
	       options->lcepsilon);
    }
  fprintf (file, "Print options:\n");
  fprintf (file, "   Data file: %46.46s\n",
	   options->infilename);
  fprintf (file, "   Output file: %44.44s\n",
	   options->outfilename);
  fprintf (file, "   Print data: %45.45s\n",
	   options->printdata ? "Yes" : "No");
  switch (options->treeprint)
    {
    case NONE:
      fprintf (file, "   Print genealogies: %38.38s\n", "No");
      break;
    case ALL:
      fprintf (file, "   Print genealogies: %38.38s\n", "Yes, all");
      break;
    case LASTCHAIN:
      fprintf (file, "   Print genealogies: %38.38s\n", "Yes, only those in last chain");
      break;
    case BEST:
      fprintf (file, "   Print genealogies: %38.38s\n", "Yes, only the best");
      break;
    }
  if (options->plot)
    {
      switch (options->plotmethod)
	{
	case PLOTALL:
	  sprintf (mytext, "Yes, to outfile and %s", options->mathfilename);
	  break;
	default:
	  strcpy (mytext, "Yes, to outfile");
	  break;
	}
    }
  else
    {
      strcpy (mytext, "No");
    }
  fprintf (file, "   Plot data: %46.46s\n", mytext);
  /* Version 2.0   switch(options->profile){
     case NONE:  strcpy(mytext,"No"); break;
     case ALL:  strcpy(mytext,"Yes, tables and summary"); break;
     case TABLES:  strcpy(mytext,"Yes, tables"); break;
     case SUMMARY:  strcpy(mytext,"Yes, summary"); break;
     }     
     fprintf(file, "   Profile likelihood:%.36s\n\n\n\n", mytext);
   */
}

void 
set_param (world_fmt * world, data_fmt * data, option_fmt * options, long locus)
{
  long i /*Version 2.0 , ii  , j */ ;
  if (strchr ("mb", (int) options->datatype))
    {
      if (options->thetaguess != OWN)
	{
	  options->thetaguess = OWN;
	  for (i = 0; i < world->numpop; i++)
	    {
	      world->param0[i] = 1.0;
	    }
	}
      if (options->migrguess != OWN)
	{
	  options->migrguess = OWN;
	  for (i = world->numpop; i < world->numpop2; i++)
	    {
	      world->param0[i] = 1.0;
	    }
	}
    }
  switch (options->thetaguess)
    {
    case OWN:
      if (world->numpop < options->numpop)
	{
	  fprintf (stderr, "There is a conflict between your menu/parmfile\n");
	  fprintf (stderr, "and your datafile: number of populations\n");
	  fprintf (stderr, "are not the same\n");
	  exit (-1);
	}
      for (i = 0; i < world->numpop; i++)
	{
	  /* Version 2.0 if(i<options->numthetag-1)
	     ii=i;
	     else
	     ii=options->numthetag-1;            
	     if(options->thetag[ii]==0.0)
	   */
	  if (options->param0[i] == 0.0)
	    world->param0[i] = SMALLEST_THETA;
	  else
	    {
	      /*Version 2.0 world->param0[i] = options->thetag[ii]; */
	      world->param0[i] = options->param0[i];
	    }
	}
      break;

    case FST:
    default:
      for (i = 0; i < world->numpop; i++)
	{
	  if (world->fstparam[locus][i] > 0)
	    {
	      if (world->fstparam[locus][i] > 100)
		world->param0[i] = 1.0;
	      else
		world->param0[i] = world->fstparam[locus][i];
	    }
	  else
	    world->param0[i] = 1.0;
	}
      break;
    }
  switch (options->migrguess)
    {
    case OWN:
      for (i = world->numpop; i < 2 * world->numpop; i++)
	{
	  if (options->param0[i] == 0.0)
	    world->param0[i] = SMALLEST_MIGRATION;
	  else
	    world->param0[i] = options->param0[i] / world->param0[i - world->numpop];
	}
      break;
      /* Version 2.0
         for (i = 0; i < world->numpop; i++) {
         for (j = 0; j < world->numpop-1; j++) {
         if((world->numpop-1)*i+j < options->nummg)
         ii=(world->numpop-1)*i+j;
         else
         ii=options->nummg-1;        
         world->param0[world->numpop+(world->numpop-1)*i+j] = 
         options->mg[ii]/world->param0[i];
         }
         }
         break;
       */
    case FST:
    default:
      for (i = world->numpop; i < world->numpop2; i++)
	{
	  if (world->fstparam[locus][i] > 0)
	    {
	      if (world->fstparam[locus][i] > 100)
		world->param0[i] = 1.0;
	      else
		world->param0[i] = world->fstparam[locus][i];
	    }
	  else
	    world->param0[i] = 1.0;
	}
      break;
    }
}




void 
destroy_options (option_fmt * options)
{
  free (options->param0);
  free (options->thetag);
  free (options->mg);
  free (options->ttratio);
  free (options);
}

void 
decide_plot (option_fmt * options, long chain, long chains, char type)
{
  if (options->plot && (chain >= chains - 1) && (type == 'l'))
    options->plotnow = TRUE;
  else
    options->plotnow = FALSE;
}

/*private functions============================================= */

long 
boolcheck (char ch)
{
  char c = uppercase (ch);
  if ((c == 'F') || (c == 'N'))
    return 0;
  else if ((c == 'T') || (c == 'Y'))
    return 1;
  else
    return -1;
}				/* boolcheck */

boolean 
booleancheck (option_fmt * options, char *var, char *value)
{
  long i, check;
  char *booltokens[NUMBOOL] = BOOLTOKENS;
  char *tmp;
  check = boolcheck (value[0]);
  if (check == -1)
    return FALSE;
  i = 0;
  while (i < NUMBOOL && strcmp (var, booltokens[i]))
    i++;
  switch ((short) i)
    {
    case 0:
      options->menu = (boolean) (check);
      break;
    case 1:
      options->interleaved = (boolean) (check);
      break;
    case 2:
      options->printdata = (boolean) (check);
      break;
    case 3:
      /*reserved */
      break;
    case 4:
      options->movingsteps = (boolean) (check);
      if (options->movingsteps)
	{
	  strtok (value, ":");
	  options->acceptfreq = atof ((char *) strtok (NULL, " ,\n"));
	}
      break;
    case 5:
      options->freqsfrom = (boolean) (check);
      if (!options->freqsfrom)
	{
	  strtok (value, ":");
	  options->freqa = atof ((char *) strtok (NULL, " ,"));
	  options->freqc = atof ((char *) strtok (NULL, " ,"));
	  options->freqg = atof ((char *) strtok (NULL, " ,"));
	  options->freqt = atof ((char *) strtok (NULL, " ;\n"));
	}
      break;
    case 6:
      options->usertree = (boolean) (check);
      break;
    case 7:
      {				/* autocorrelation=<YES:value | NO> */
	options->autocorr = (boolean) (check);
	if (options->autocorr)
	  {
	    strtok (value, ":");
	    options->lambda = 1.0 / atof ((char *) strtok (NULL, " ;\n"));
	  }
	break;
      }
    case 8:
      options->simulation = (boolean) check;
      break;
    case 9:
      options->plot = (boolean) check;
      if (options->plot)
	{
	  strtok (value, ":");
	  if (toupper (value[0]) == 'Y')
	    options->plotmethod = PLOTALL;
	  else
	    {
	      tmp = strtok (NULL, ";");
	      switch (lowercase (tmp[0]))
		{
		case 'o':
		  options->plotmethod = PLOTOUTFILE;
		  break;
		case 'b':
		  options->plotmethod = PLOTALL;
		  break;
		default:
		  options->plotmethod = PLOTALL;
		  break;
		}
	    }
	}
      break;
    case 10:
      options->weights = (boolean) check;
      break;
    default:
      return FALSE;
    }
  return TRUE;
}				/* booleancheck */

boolean 
numbercheck (option_fmt * options, char *var, char *value)
{
  long i = 0, z;

  char *tmp, *temp;

  char *numbertokens[NUMNUMBER] = NUMBERTOKENS;
  tmp = (char *) malloc (sizeof (char) * LINESIZE);
  while (i < NUMNUMBER && strcmp (var, numbertokens[i]))
    i++;
  switch ((short) i)
    {
    case 0:
      z = 0;
      temp = strtok (value, " ,;\n");
      while (temp != NULL)
	{
	  options->ttratio[z++] = atof (temp);
	  options->ttratio = (double *) realloc (options->ttratio, sizeof (double) * (z + 1));
	  options->ttratio[z] = 0.0;
	  temp = strtok (NULL, " ,;\n");
	}
      break;
    case 1:
      options->schains = atoi (value);
      break;
    case 2:
      options->ssteps = atoi (value);
      break;
    case 3:
      options->sincrement = atoi (value);
      break;
    case 4:
      options->lchains = atoi (value);
      break;
    case 5:
      options->lsteps = atoi (value);
      break;
    case 6:
      options->lincrement = atoi (value);
      break;
    case 7:
      break;			/* theta: already handled in read_theta() */
    case 8:
      options->nmlength = atoi (value);
      break;
    case 9:			/* "seed" */
      switch (value[0])
	{
	case 'A':
	case 'a':
	case '0':
	  options->autoseed = AUTO;
	  options->inseed = (long) time (0);
	  break;
	case 'S':
	case 's':
	case '1':
	  options->autoseed = NOAUTO;
	  options->seedfile = fopen ("seedfile", "r");
	  if (options->seedfile == NULL)
	    {
	      fprintf (stderr, "Error: cannot find seedfile\n");
	      exit (EXIT_FAILURE);
	    }
	  fscanf (options->seedfile, "%ld%*[^\n]", &options->inseed);
	  fclose (options->seedfile);
	  break;
	case 'O':
	case 'o':
	case '2':
	  options->autoseed = NOAUTOSELF;
	  strtok (value, ":");
	  options->inseed = atoi ((char *) strtok (NULL, ";"));
	  break;
	default:
	  fprintf (stderr, "failure to read seed method, should be\nseed=auto or seed=seedfile or seed=own:value\n");
	  break;
	}
      break;
    case 10:
      break;			/*"migration" fake: this is already handled in read_migrate */
    case 11:			/*mutation */
      switch (value[0])
	{
	case 'A':		/*automatic */
	case 'a':
	case 'G':
	case 'g':
	  options->gamma = TRUE;
	  break;
	case 'N':		/*none, all loci have same mu */
	case 'n':
	  options->gamma = FALSE;
	  break;
	default:
	  break;
	}
      break;
    case 12:			/*datatype */
      switch (value[0])
	{
	case 'a':
	case 'A':
	  options->datatype = 'a';
	  break;
	case 'm':
	case 'M':
	  options->datatype = 'm';
	  break;
	case 'b':
	case 'B':
	  options->datatype = 'b';
	  break;
	case 's':
	case 'S':
	  options->datatype = 's';
	  break;
	default:
	  options->datatype = 'a';
	  break;
	}
      break;
    case 13:
      if ((toupper (value[0]) == 'Y') || (toupper (value[0] == 'T')))
	options->categs = MANYCATEGS;
      else
	options->categs = atoi (value);		/* categories */
      /* needs to read auxilliary file catfile */
      break;
    case 14:
      strncpy (tmp, value, strcspn (value, ":"));
      if (atoi (tmp) > 1)
	{			/* rate categories */
	  options->rcategs = atoi (tmp);
	  options->rrate = (double *) realloc (options->rrate, sizeof (double)
					       * (options->rcategs + 1));
	  temp = strtok (value, " :");
	  temp = strtok (NULL, " ,;\n");
	  z = 0;
	  while (temp != NULL)
	    {
	      if (z > options->rcategs)
		error ("check parmfile rates, missing rate\n");
	      options->rrate[z++] = atof (temp);
	      temp = strtok (NULL, " ,;\n");
	    }
	}
      break;
    case 15:
      strncpy (tmp, value, strcspn (value, ":"));
      if (atoi (tmp) > 1)
	{			/* probabilities for each rate category */
	  options->rcategs = atoi (tmp);
	  options->probcat = (double *) realloc (options->probcat, sizeof (double)
						 * (options->rcategs + 1));
	  temp = strtok (value, " :");
	  temp = strtok (NULL, " ,;\n");
	  z = 0;
	  while (temp != NULL)
	    {
	      if (z > options->rcategs)
		error ("check parmfile prob-rates, missing rate probability\n");
	      options->probcat[z++] = atof (temp);
	      temp = strtok (NULL, " ,;\n");
	    }
	}
      break;
    case 16:			/*micro-stepmax */
      options->micro_stepnum = atoi (value);
      break;
    case 17:			/*micro-threshold */
      options->micro_threshold = atoi (value);
      break;
    case 18:			/*delimiter */
      options->dlm = value[0];
      break;
    case 19:			/*burn-in */
      options->burn_in = atoi (value);
      break;
    case 20:			/*infilename */
      strcpy (options->infilename, value);
      break;
    case 21:			/*outfilename */
      strcpy (options->outfilename, value);
      break;
    case 22:			/*mathfilename */
      strcpy (options->mathfilename, value);
      break;
    case 23:			/*infilename */
      strncpy (options->title, value, 80);
      break;
    case 24:			/*long-chain-epsilon */
      options->lcepsilon = atof (value);
      break;
    case 25:			/* print tree options */
      switch (uppercase (value[0]))
	{
	case 'N':
	  options->treeprint = NONE;
	  break;
	case 'A':
	  options->treeprint = ALL;
	  break;
	case 'B':
	  options->treeprint = BEST;
	  break;
	case 'L':
	  options->treeprint = LASTCHAIN;
	  break;
	default:
	  options->treeprint = NONE;
	  break;
	}
      break;
    case 26:			/* progress: No, Yes, Verbose */
      switch (uppercase (value[0]))
	{
	case 'F':
	case 'N':
	  options->progress = FALSE;
	  options->verbose = FALSE;
	  break;
	case 'T':
	case 'Y':
	  options->progress = TRUE;
	  options->verbose = FALSE;
	  break;
	case 'V':
	  options->progress = TRUE;
	  options->verbose = TRUE;
	  break;
	}
      break;
    case 27:			/* lr-ratio: <none | mean | locus>:val1,val2,val3,val4,val5 */
      switch (uppercase (value[0]))
	{
	case 'M':
	  options->lratio->data[options->lratio->counter].type = MEAN;
	  break;
	case 'L':
	  options->lratio->data[options->lratio->counter].type = LOCUS;
	  break;
	case 'N':
	default:
	  free (tmp);
	  return FALSE;
	}
      temp = strtok (value, ":");
      temp = strtok (NULL, "\n");
      if (temp != NULL)
	strcpy (options->lratio->data[options->lratio->counter].value, temp);
      if (options->lratio->counter + 1 == options->lratio->alloccounter)
	{
	  options->lratio->alloccounter += 2;
	  options->lratio->data = (lr_data_fmt *) realloc (options->lratio->data, sizeof (lr_data_fmt) * options->lratio->alloccounter);
	  for (i = options->lratio->counter + 1; i < options->lratio->alloccounter; i++)
	    {
	      options->lratio->data[i].elem = 0;
	      options->lratio->data[i].value = (char *) calloc (1, sizeof (char) * LINESIZE);
	    }
	}
      options->lratio->counter++;
      break;
    case 28:			/* fst-type: <Theta | Migration> */
      switch (uppercase (value[0]))
	{
	case 'T':
	  options->fsttype = 'T';
	  break;
	case 'M':
	default:
	  options->fsttype = 'M';
	  break;
	}
      fst_type (options->fsttype);
      break;
    case 29:			/*profile=<NONE | ALL | TABLES | SUMMARY> */
      switch (uppercase (value[0]))
	{
	case 'S':
	  options->profile = SUMMARY;
	  break;
	case 'A':
	  options->profile = ALL;
	  break;
	case 'N':
	  options->profile = NONE;
	  break;
	case 'T':
	  options->profile = TABLES;
	  break;
	default:		/*A */
	  options->profile = ALL;
	  break;
	}
      set_profile_options (options);
      break;
    default:
      free (tmp);
      return FALSE;

    }
  free (tmp);
  return TRUE;
}				/* numbercheck */

void 
reset_oneline (option_fmt * options, long position)
{
  fseek (options->parmfile, position, SEEK_SET);
}


void 
read_theta (option_fmt * options)
{
  char parmvar[LINESIZE];
  char tmp[LINESIZE];
  char varvalue[LINESIZE];
  char ch;

  long i = 0;

  while ((ch = getc (options->parmfile)) != '=')
    {
      parmvar[i++] = ch;
    }
  i = 0;
  ch = getc (options->parmfile);
  while (!isspace ((int) ch) && ch != ':' && ch != '{')
    {
      varvalue[i++] = ch;
      ch = getc (options->parmfile);
    }
  switch (varvalue[0])
    {
    case 'F':
    case 'f':
    case '_':
      options->thetaguess = FST;
      break;
    case 'O':
    case 'o':
    case '0':
      options->thetaguess = OWN;
      ch = skip_space (options);
      if (ch == '\0')
	return;
      if (ch == '{')
	{

	  while (ch != '}')
	    {
	      i = 0;
	      ch = skip_space (options);
	      if (ch == '\0')
		return;
	      while (ch != ' ' && ch != ',' && ch != '}')
		{
		  tmp[i++] = ch;
		  ch = getc (options->parmfile);
		}
	      if (ch == '}')
		break;
	      tmp[i] = '\0';
	      options->thetag[options->numthetag] = atof (tmp);
	      options->numthetag += 1;
	      options->thetag = realloc (options->thetag,
				sizeof (double) * (1 + options->numthetag));

	    }
	}
      else
	{
	  i = 0;
	  tmp[i++] = ch;
	  while (!isspace (ch))
	    {
	      tmp[i++] = ch;
	      ch = getc (options->parmfile);
	    }
	  tmp[i] = '\0';
	  options->thetag[options->numthetag] = atof (tmp);
	  options->numthetag += 1;
	  options->thetag = realloc (options->thetag,
				sizeof (double) * (1 + options->numthetag));
	}
      options->numpop = options->numthetag;
      if (options->numthetag == 0)
	{
	  fprintf (stderr, "You forgot to add your guess value:\n");
	  fprintf (stderr, "Theta=Own:{pop1,pop2, ...}\n");
	  fprintf (stderr, "or Theta=Own:guess_pop (same value for all)");
	}
      break;
    default:
      fprintf (stderr, "Failure to read start theta method, should be\ntheta=FST or theta=Own:x.x\n or theta=Own:{x.x, x.x , x.x, .....}");
      exit (-1);
    }

}


void 
read_mig (option_fmt * options)
{
  char parmvar[LINESIZE];
  char tmp[LINESIZE];
  char varvalue[LINESIZE];
  char ch;
  long test = 0;
  long i = 0;
  /* 1st example:  1.0 (n-island model)
     2nd example: {1.0} (migration matrix model, all the same start values  
     3rd example: the dashes on the diagonal are NECESSARY, {} are facultativ
     -  1.0 0.1
     1.0  -  2.0
     0.9 1.2  -
     to specify real 0.0 enter a -1, because those have to be treated
     specifically. 0.0 in the table will be change to SMALLES_MIGRATION
   */

  while ((ch = getc (options->parmfile)) != '=')
    {
      parmvar[i++] = ch;
    }
  i = 0;
  ch = getc (options->parmfile);
  while (!isspace ((int) ch) && ch != ':' && ch != '{')
    {
      varvalue[i++] = ch;
      ch = getc (options->parmfile);
    }
  switch (varvalue[0])
    {
    case 'F':
    case 'f':
    case '_':
      options->migrguess = FST;
    case 'O':
    case 'o':
    case '0':
      options->migrguess = OWN;
      ch = skip_space (options);
      if (ch == '\0')
	return;
      if (ch == '{')
	{
	  options->migration_model = MATRIX;
	  while (ch != '}')
	    {
	      ch = skip_space (options);
	      if ((ch == '\0') || (ch == '}'))
		return;
	      i = 0;
	      while (ch != ' ' && ch != ',' && ch != '}')
		{
		  tmp[i++] = ch;
		  ch = getc (options->parmfile);
		}
	      tmp[i] = '\0';
	      if (strcmp (tmp, "-"))
		{
		  options->mg[options->nummg] = atof (tmp);
		  options->nummg += 1;
		  options->mg = realloc (options->mg,
				    sizeof (double) * (1 + options->nummg));
		}
	      else
		{
		  test++;
		}
	    }
	  options->numpop = test;
	}
      else
	{
	  options->migration_model = ISLAND;
	  i = 0;
	  options->numpop = 1;
	  tmp[i++] = ch;
	  while (!isspace (ch))
	    {
	      tmp[i++] = ch;
	      ch = getc (options->parmfile);
	    }
	  options->mg[options->nummg] = atof (tmp);
	  options->nummg += 1;
	}
      if (options->nummg == 0)
	{
	  fprintf (stderr, "You forgot to add your guess value, use either:\n");
	  fprintf (stderr, "migration=FST\n");
	  fprintf (stderr, "migration=Own:{migration matrix, diagonal is -}\n");
	  fprintf (stderr, "migration=Own:{migration value}, all matrix elements have the same value\n");
	  /*      fprintf(stderr,"or migration=Own:value (n-island model)"); */
	  exit (-1);
	}
      break;
    default:
      fprintf (stderr, "Failure to read start migration method\n");
      exit (-1);
    }

}

char 
skip_space (option_fmt * options)
{
  char ch = getc (options->parmfile);
  while (isspace ((int) ch) || ch == ',')
    {
      ch = getc (options->parmfile);
    }
  if (isalpha (ch))
    {
      ungetc (ch, options->parmfile);
      ch = '\0';
    }
  return ch;
}

void 
set_profile_options (option_fmt * options)
{
  switch (options->profile)
    {
    case NONE:
      options->printprofsummary = options->printprofile = FALSE;
      break;
    case ALL:
      options->printprofsummary = options->printprofile = TRUE;
      break;
    case TABLES:
      options->printprofsummary = FALSE;
      options->printprofile = TRUE;
      break;
    case SUMMARY:
      options->printprofsummary = TRUE;
      options->printprofile = FALSE;
      break;
    }
}
