/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 W O R L D   R O U T I N E S 

 creates tree structures,
 reads tree [has to be done
 
 prints results,
 and finally helps to destroy itself.
                                                                                                               
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: world.c,v 1.1.1.1 1998/06/06 06:09:51 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"
#include "mcmc.h"
#include "world.h"
#include "fst.h"
#include "random.h"
#include "parameter.h"
#include "combine.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

#define PLANESIZE 36
#define PLANEBIGTICKS 6
#define PLANEBIGTICKVALUES {-3, -2, -1, 0, 1, 2}
#define PLANETICKS   {0.001, 0.0013895, 0.0019307, 0.0026827, 0.00372759, \
                     0.00517947, 0.00719686, 0.01, 0.013895, 0.019307, \
                     0.026827, 0.0372759, 0.0517947, 0.0719686, 0.1, \
                     0.13895, 0.19307, 0.26827, 0.372759, 0.517947, \
                     0.719686, 1., 1.3895, 1.9307, 2.6827, 3.72759, \
                     5.17947, 7.19686, 10., 13.895, 19.307, 26.827, 37.2759, \
                     51.7947, 71.9686, 100.}
#define CONTOURLEVELS 8
#define CONTOURS_LOCUS {0.0,-3.35670/2., -9.48773/2., -13.2767/2.,\
                        0.0,-3.35670/2., -9.48773/2., -13.2767/2.}
#define CONTOURS_LOCI  {0.0,-4.35146/2., -11.0705/2., -15.0863/2.,\
                        0.0,-4.35146/2., -11.0705/2., -15.0863/2.}

/* prototypes ------------------------------------------- */
void create_world (world_fmt ** world);
void init_world (world_fmt * world, data_fmt * data, option_fmt * options);
void calc_simple_param (world_fmt * world);
void print_menu_locus (option_fmt * options, long locus);
void print_menu_chain (char type, long chain, long steps, world_fmt * world);
void set_bounds (long *increment, long *steps, long *chain,
		 const option_fmt * options, const char type);
void burnin_chain (world_fmt * world);
void copy_atl (world_fmt * world, timearchive_fmt * old, timearchive_fmt * new, long steps);
void print_simresults (world_fmt * world);
void print_list (world_fmt * world);
void plot_surface (world_fmt * world, char ***plane, long x);
void print_alpha_curve (world_fmt * world, timearchive_fmt * atl);
void create_loci_plot (world_fmt * world, char **plane, timearchive_fmt * atl, long loci);
void create_locus_plot (world_fmt * world, char **plane, tarchive_fmt * tl, nr_fmt * nr, long G);
void cleanup_world (world_fmt * world, long locus);
void apocalypse_now (world_fmt * world);
/* private functions */
void create_timearchive (timearchive_fmt ** atl, long loci, long samples, long numpop);
void create_plotplane (world_fmt * world);
void create_cov (world_fmt * world);
void print_menu_equilib (option_fmt * options);
void print_finish (world_fmt * world, long filepos);
void copy_time (world_fmt * world, timelist_fmt * ltl, long from, long to, long np);
void archive_timelist (tarchive_fmt * atl, vtlist * tl, long T, long np);
void copy_timelist (tarchive_fmt * from, tarchive_fmt * to, long np);
long whichp (long from, long to, long pop);
long whichl (long from, long to, long pop);
/* long calc_T(timearchive_fmt * old, long steps); */
void print_results (world_fmt * world);
void print_fst (world_fmt * world, double **fstparam);
void prepare_print_nu (double nu, char *str);
void prepare_print_nm (double nm, double nmu, char *strllike);
void print_menu_coalnodes (world_fmt * world, long G);
void print_menu_createplot (void);
void calc_loci_plane (world_fmt * world, nr_fmt * nr, timearchive_fmt * atl, double **pl, long loci, double contours[]);
void calc_locus_plane (world_fmt * world, nr_fmt * nr, tarchive_fmt * tl, long G,
		       double **pl, double contours[]);
void fill_plotplane (char **plane, double **pl, double contours[], plotmax_fmt * planemax);
void print_mathematica (world_fmt * world, double **plane, long x, long y);
void print_cov (world_fmt * world, long numpop, long loci, double ***cov);
void print_cov_table (FILE * outfile, long locus, world_fmt * world, double *corr, long addvar);
void free_seqx (node * p, long sites);
void free_x (node * p);
void free_tree (node * p);
void free_nodelet (node * p, long num);
void increase_timearchive (world_fmt * world, long locus, long sample, long numpop);
void test_locus_like (double *param0, double *param1, long df, long locus, world_fmt * world,
		      boolean withhead, char *this_string);
void test_loci_like (double *param0, double *param1, long df, long loci, world_fmt * world,
		     boolean withhead, char *this_string);

double chisquare (long df, double alpha);
long set_test_param (double *param, char *strp, world_fmt * world, long lrline, long locus);
void print_CV (world_fmt * world);
/*======================================================*/
void 
create_world (world_fmt ** world)
{
  (*world) = (world_fmt *) calloc (1, sizeof (world_fmt));
  (*world)->param_like = DBL_MAX;
}

void 
init_world (world_fmt * world, data_fmt * data, option_fmt * options)
{
  long locus;
  world->options = options;
  world->data = data;
  world->loci = data->loci;
  world->skipped = 0;
  world->numpop = data->numpop;
  world->numpop2 = 2 * world->numpop;
  getseed (options);
  create_timearchive (&(world->atl), data->loci,
		      SAMPLETREE_GUESS, data->numpop);
  create_plotplane (world);
  create_cov (world);
  world->likelihood = (double *) calloc (1, sizeof (double) * SAMPLETREE_GUESS);
  world->lineages = (long *) calloc (1, sizeof (long) * world->numpop);
  world->param0 = (double *) calloc (1, sizeof (double) * world->numpop2);
  world->param00 = (double *) calloc (1, sizeof (double) * world->numpop2);
  switch (options->datatype)
    {
    case 's':
    case 'a':
      world->fstparam = (double **) calloc (1, sizeof (double *) * (world->loci + 1));
      for (locus = 0; locus < world->loci + 1; locus++)
	world->fstparam[locus] = (double *) calloc (1,
				      sizeof (double) * world->numpop2 * 2);
      break;
    case 'm':
      world->options->steps = (double **)
	calloc (1, sizeof (double *) * world->options->micro_stepnum);
      for (locus = 0; locus < world->options->micro_stepnum; locus++)
	world->options->steps[locus] = (double *) calloc (1,
			   sizeof (double) * world->options->micro_stepnum);
      break;
    }
  /* tree and treetimes are not yet allocated */
}

void 
calc_simple_param (world_fmt * world)
{
  switch (world->options->datatype)
    {
    case 'a':
      calc_fst (world);
      break;
    case 's':
      calc_fst (world);
      break;
    default:
      break;
    }
}

void 
set_bounds (long *increment, long *steps, long *chains,
	    const option_fmt * options, const char type)
{
  switch (type)
    {
    case 's':
      *increment = options->sincrement;
      *steps = options->ssteps;
      *chains = options->schains;
      break;
    case 'l':
      *increment = options->lincrement;
      *steps = options->lsteps;
      *chains = options->lchains;
      break;
    default:
      fprintf (stderr, "this is either a short nor a long chain??\n");
      exit (EXIT_FAILURE);
      break;
    }
}

void 
burnin_chain (world_fmt * world)
{
  long step;
  if (world->options->burn_in == 0)
    return;
  print_menu_equilib (world->options);
  for (step = 0; step < world->options->burn_in; step++)
    {
      metropolize (world, 0);
    }
}

void 
copy_atl (world_fmt * world, timearchive_fmt * old, timearchive_fmt * new, long steps)
{
  long j;
  long numpop = world->numpop;

  increase_timearchive (world, world->locus + 1, steps, numpop);

  new->T = /*calc_T(old, steps); */ old->T;
  new->param_like = world->param_like;
  new->sumtips = world->sumtips;
  new->numpop = world->data->numpop;
  memcpy (new->param, world->param0, sizeof (double) * 2 * numpop);
  memcpy (new->param0, world->param00, sizeof (double) * 2 * numpop);
  memcpy (new->likelihood, world->likelihood, sizeof (double) * steps);
  for (j = 0; j < new->T; j++)
    {
      new->tl[j].copies = old->tl[j].copies;
      copy_timelist (&old->tl[j], &new->tl[j], numpop);
    }
}

void 
print_simresults (world_fmt * world)
{
  long l = 1;
  double ft1 = 0., ft2 = 0., fm1 = 0., fm2 = 0., lt1 = 0., lt2 = 0., lm1 = 0.,
    lm2 = 0.;
  switch (world->options->datatype)
    {
    case 's':
      fprintf (world->outfile,
	       " %3li %li %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f\n",
	       l, world->atl[l].trials, world->atl[l].param[0],
	       world->atl[l].param[1], world->atl[l].param[2],
	       world->atl[l].param[3], world->options->gamma ?
	       world->atl[l].param[4] : 999999., world->atl[l].param_like);
      break;
    case 'm':
      fprintf (world->outfile,
	       " %3li %li %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f\n",
	       l, world->atl[l].trials, world->atl[l].param[0],
	       world->atl[l].param[1], world->atl[l].param[2],
	       world->atl[l].param[3], world->options->gamma ?
	       world->atl[l].param[4] : 999999., world->atl[l].param_like);
      break;
    case 'a':
      for (l = 0; l < world->loci + 1; l++)
	{
	  if (world->fstparam[l][0] != -999.)
	    {
	      ft1 += world->fstparam[l][0];
	      lt1++;
	    }
	  if (world->fstparam[l][1] != -999.)
	    {
	      ft2 += world->fstparam[l][1];
	      lt2++;
	    }
	  if (world->fstparam[l][2] != -999.)
	    {
	      fm1 += world->fstparam[l][2];
	      lm1++;
	    }
	  if (world->fstparam[l][3] != -999.)
	    {
	      fm2 += world->fstparam[l][3];
	      lm2++;
	    }
	}
      l = world->loci + 1;
      if (lt1 == 0.)
	lt1 = 1.;
      if (lt2 == 0.)
	lt2 = 1.;
      if (lm1 == 0.)
	lm1 = 1.;
      if (lm2 == 0.)
	lm2 = 1.;
      fprintf (world->outfile,
	       " %3li %li %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f\n",
	       l - 1, world->atl[l].trials, world->atl[l].param[0],
	       world->atl[l].param[1], world->atl[l].param[2],
	       world->atl[l].param[3], world->options->gamma ?
	       world->atl[l].param[4] : 999999., world->atl[l].param_like,
	       ft1 / lt1, ft2 / lt2, fm1 / lm1, fm2 / lm2);
      break;
    }
}

void 
print_list (world_fmt * world)
{

  /*  long i,locus, df; */
  double *param0;
  param0 = (double *) calloc (1, sizeof (double) * (world->numpop2 + 1));

  print_results (world);
  if ((world->options->datatype == 'a') || (world->options->datatype == 's'))
    print_fst (world, world->fstparam);
  if (world->options->plot)
    plot_surface (world, world->plane, PLANESIZE + 2);
  print_alpha_curve (world, world->atl);
#ifdef TESTING
  /* usage of this section is discouraged, if you understand what's going on here
     and want to use it please contact me (beerli@genetics.washington.edu) */
  if (world->options->lratio->counter > 0)
    {
      if (world->loci - world->skipped > 1 && world->options->lratio->data[0].type == MEAN)
	{
	  df = set_test_param (param0, world->options->lratio->data[0].value, world, 0, -1);
	  test_loci_like (param0,
			  world->atl[world->loci + 1].param, df,
			  world->loci, world, HEADER, NULL);
	}
      else
	{
	  for (locus = 0; locus < world->loci; locus++)
	    {
	      df = set_test_param (param0, world->options->lratio->data[0].value, world, 0, locus);
	      test_locus_like (param0,
			       world->atl[locus + 1].param, df,
			       locus, world, HEADER, NULL);
	    }
	}
    }
  for (i = 1; i < world->options->lratio->counter; i++)
    {
      if (world->loci - world->skipped > 1 && world->options->lratio->data[i].type == MEAN)
	{
	  df = set_test_param (param0, world->options->lratio->data[i].value, world, i, -1);
	  test_loci_like (param0,
			  world->atl[world->loci + 1].param, df,
			  world->loci, world, FALSE, NULL);
	}
      else
	{
	  for (locus = 0; locus < world->loci; locus++)
	    {
	      df = set_test_param (param0, world->options->lratio->data[i].value, world, i, locus);
	      test_locus_like (param0,
			       world->atl[locus + 1].param, df,
			       locus, world, FALSE, NULL);
	    }
	}
    }
#endif
  print_cov (world, world->numpop, world->loci, world->cov);
#ifdef TESTING
  print_CV (world);
#endif
}

void 
plot_surface (world_fmt * world, char ***plane, long x)
{
  long i, locus;
  long loci = world->loci;
  FILE *outfile = world->outfile;
  if (world->options->progress)
    fprintf (stdout, "           Plotting the likelihood surfaces\n");
  PAGEFEED;
  fprintf (outfile, "Log-Likelihood surface of 2 populations\n");
  fprintf (outfile, "---------------------------------------\n\n");
  fprintf (outfile, "Legend:\n");
  fprintf (outfile, "   X = Maximum likelihood\n");
  fprintf (outfile, "   * = in approximative 50%% confidence limit\n");
  fprintf (outfile, "   + = in approximative 95%% confidence limit\n");
  fprintf (outfile, "   - = in approximative 99%% confidence limit\n");
  for (locus = 0; locus < loci; locus++)
    {
      if (world->data->skiploci[locus])
	{
	  continue;
	}
      fprintf (outfile, "\n\nLocus %li\nx-axis= 4Nm [effective population size * migration rate],\n", locus + 1);
      fprintf (outfile, " y-axis = Theta,\nunits = log10\nMaximum log likelihood on plot\n");
      fprintf (outfile, "  Population 1: 4Nm=%f, Theta=%f, log likelihood=%f\n",
	       world->plotmax[locus].x1, world->plotmax[locus].y1,
	       world->plotmax[locus].l1);
      fprintf (outfile, "  Population 2: 4Nm=%f, Theta=%f, log likelihood=%f\n",
	       world->plotmax[locus].x2, world->plotmax[locus].y2,
	       world->plotmax[locus].l2);
      fprintf (outfile, "\n            Population 1                         Population 2\n\n");
      for (i = x; i >= 0; i--)
	fprintf (outfile, "%s\n", plane[locus][i]);
      fprintf (outfile, "%s\n", plane[locus][x]);
      PAGEFEED;
    }
  if (loci - world->skipped > 1)
    {
      fprintf (outfile, "\nOver all loci\n\n");
      locus = loci;
      fprintf (outfile, "x-axis= Nm [effective population size * migration rate],\n");
      fprintf (outfile, "y-axis = Theta,\nunits = log10\nMaximum log likelihood on plot\n");
      fprintf (outfile, "  Population 1: 4Nm=%f, Theta=%f, log likelihood=%f\n",
	       world->plotmax[locus].x1, world->plotmax[locus].y1,
	       world->plotmax[locus].l1);
      fprintf (outfile, "  Population 2: 4Nm=%f, Theta=%f, log likelihood=%f\n",
	       world->plotmax[locus].x2, world->plotmax[locus].y2,
	       world->plotmax[locus].l2);
      fprintf (outfile, "\n            Population 1                         Population 2\n\n");
      for (i = x; i >= 0; i--)
	fprintf (outfile, "%s\n", plane[locus][i]);
      PAGEFEED;
    }
}

void 
print_alpha_curve (world_fmt * world, timearchive_fmt * atl)
{
  const double alphas[19] =
  {0.0001, 0.0002, 0.0005,
   0.001, 0.002, 0.005,
   0.01, 0.02, 0.05,
   0.1, 0.2, 0.5,
   1., 2., 5.,
   10., 20., 50., 100.};
  double contours[CONTOURLEVELS] = CONTOURS_LOCI;
  boolean once = FALSE;
  long g, a, gmax = 1, loci = world->loci;
  double likes[21];
  char confidence[21];
  nr_fmt *nr;
  FILE *outfile = world->outfile;
  if (world->options->gamma && loci - world->skipped > 1)
    {
      nr = (nr_fmt *) calloc (1, sizeof (nr_fmt));
      for (g = 0; g < CONTOURLEVELS; g++)
	{
	  contours[g] += atl[loci + 1].param_like;
	}
      for (g = 1; g < loci + 1; g++)
	{
	  if (gmax < atl[g].T)
	    gmax = atl[g].T;
	}
      create_nr (nr, atl[loci].numpop, gmax);
      nr->skiploci = world->data->skiploci;
      memcpy (nr->param, atl[loci + 1].param, sizeof (double) * 5);
      if (world->options->progress)
	fprintf (stdout, "           Printing Inv(alpha) x Log(likelihood)\n");
      gmax = 0;
      calc_gamma (nr);
      likes[20] = calc_loci_like (nr, atl, loci, TRUE);
      for (a = 0; a < 19; a++)
	{
	  nr->param[4] = alphas[a];
	  likes[a] = calc_loci_like (nr, atl, loci, TRUE);
	  if (atl[loci + 1].param[4] < alphas[a] && !once)
	    {
	      gmax = a - 1;
	      once = TRUE;
	    }
	}
      for (a = 0; a < 19; a++)
	{
	  if (likes[a] < contours[1])
	    {
	      if (likes[a] < contours[2])
		{
		  if (likes[a] < contours[3])
		    {
		      confidence[a] = ' ';
		    }
		  else
		    {
		      confidence[a] = '-';
		    }
		}
	      else
		{
		  confidence[a] = '+';
		}
	    }
	  else
	    {
	      if (likes[a] < contours[0])
		confidence[a] = '*';
	      else
		{
		  confidence[a] = 'X';
		}
	    }
	}
      fprintf (outfile, "Log-Likleihood curve for the shape ");
      fprintf (outfile, "parameter Inv(alpha) [square(CV(mu))]\n");
      fprintf (outfile, "-----------------------------------");
      fprintf (outfile, "--------------------------\n\n");
      fprintf (outfile, "Legend:\n");
      fprintf (outfile, "   X = Maximum likelihood\n");
      fprintf (outfile, "   * = in approximative 50%% confidence limit\n");
      fprintf (outfile, "   + = in approximative 95%% confidence limit\n\n");
      fprintf (outfile, "   - = in approximative 99%% confidence limit\n\n");
      fprintf (outfile, "Inv(Alpha)     Log(Likelihood)   Confidence limit\n");
      fprintf (outfile, "-------------------------------------------------\n");
      if (gmax < 0)
	{
	  fprintf (outfile, "%10.5g     % 20.5g            X\n", atl[loci + 1].param[4],
		   likes[20]);
	}
      for (a = 0; a < 19; a++)
	{
	  fprintf (outfile, "% 10.5f     % 20.5g            %c\n", alphas[a],
		   likes[a], confidence[a]);
	  if (gmax == a)
	    fprintf (outfile, "% 10.5f     % 20.5g            X\n", atl[loci + 1].param[4],
		     likes[20]);
	}
      if (gmax >= 20)
	{
	  fprintf (outfile, "% 10.5g     % 20.5g            X\n", atl[loci + 1].param[4],
		   likes[20]);
	}
      destroy_nr (nr);
    }
}

void 
create_loci_plot (world_fmt * world, char **plane, timearchive_fmt * atl, long loci)
{
  long intervals = PLANESIZE;
  long i, g = 1;
  nr_fmt *nr;
  double **pl;
  double contours[CONTOURLEVELS] = CONTOURS_LOCI;

  if (world->options->progress)
    print_menu_createplot ();
  nr = (nr_fmt *) calloc (1, sizeof (nr_fmt));
  for (i = 1; i < loci + 1; i++)
    {
      if (g < atl[i].T)
	g = atl[i].T;
    }
  create_nr (nr, atl[loci].numpop, g);
  nr->skiploci = world->data->skiploci;
  pl = (double **) calloc (1, sizeof (double *) * intervals);
  for (i = 0; i < intervals; i++)
    {
      pl[i] = (double *) calloc (1, sizeof (double) * 2 * intervals);
    }
  calc_loci_plane (world, nr, atl, pl, loci, contours);
  fill_plotplane (plane, pl, contours, &world->plotmax[world->loci]);
  destroy_nr (nr);
  print_mathematica (world, pl, intervals, intervals);
  for (i = 0; i < intervals; i++)
    {
      free (pl[i]);
    }
  free (pl);
}


void 
create_locus_plot (world_fmt * world, char **plane, tarchive_fmt * tl, nr_fmt * nr, long G)
{
  long intervals = PLANESIZE;
  long i;
  double **pl;
  double contours[CONTOURLEVELS] = CONTOURS_LOCI;
  if (world->options->verbose)
    print_menu_createplot ();
  pl = (double **) calloc (1, sizeof (double *) * intervals);
  for (i = 0; i < intervals; i++)
    {
      pl[i] = (double *) calloc (1, sizeof (double) * 2 * intervals);
    }
  calc_locus_plane (world, nr, tl, G, pl, contours);
  fill_plotplane (plane, pl, contours, &world->plotmax[world->locus]);
  print_mathematica (world, pl, intervals, intervals);
  for (i = 0; i < intervals; i++)
    {
      free (pl[i]);
    }
  free (pl);
}

void 
cleanup_world (world_fmt * world, long locus)
{
  if (locus >= 0)
    {
      if (world->options->datatype == 's')
	free_seqx (world->root, world->data->seq->endsite);
      else
	free_x (world->root);
      free_tree (world->root);
      free (world->nodep);
    }
}


void 
apocalypse_now (world_fmt * world)
{
  free (world);
  raise (SIGILL);
}

/* printing stuff =========================================== */
void 
print_menu_locus (option_fmt * options, long locus)
{
  if (options->progress)
    fprintf (stdout, "Locus %-3li:\n", locus + 1);
}

void 
print_menu_chain (char type, long chain, long steps, world_fmt * world)
{
  char strllike[LINESIZE];
  char nowstr[LINESIZE];
#ifdef _GNU_
  double likes[steps];
#else
  double *likes;
#endif
  if (world->options->progress)
    {
      print_llike (world->param_like, strllike);
      get_time (nowstr, "%H:%M:%S");
      if (chain == FIRSTCHAIN)
	{
	  fprintf (stdout, "%s   Start conditions: theta={%5.5f,%5.5f}, M={%5.5f,%5.5f},\n",
		   nowstr,
		   world->param0[0], world->param0[1],
		   world->param0[2], world->param0[3]);
	  fprintf (stdout, "           Start-tree-log(L)=%f\n", world->likelihood[0]);
	}
      else
	{
	  fprintf (stdout, "%s   %*s chain %3li: lnL=%s,\n           theta={%5.5f,%5.5f}, M={%5.5f,%5.5f}\n",
		   nowstr, type == 's' ? 5 : 4, type == 's' ? "Short" : "Long", chain + 1, strllike,
		   world->param0[0], world->param0[1],
		   world->param0[2], world->param0[3]);
	  if (world->options->verbose)
	    {
#ifndef _GNU_
	      likes = (double *) malloc (sizeof (double) * steps);
#endif
	      memcpy (likes, world->likelihood, steps * sizeof (double));
	      qsort ((void *) likes, steps, sizeof (double), numcmp);
	      fprintf (stdout, "           Sampled tree-log(L)={%f .. %f}, best in group =%f\n", likes[0], likes[steps - 1], world->maxdatallike);
#ifndef _GNU_
	      free (likes);
	    }
#endif
	}
    }
}

void 
copy_time (world_fmt * world, timelist_fmt * ltl, long from, long to, long np)
{
  long T;
  tarchive_fmt *atl = world->atl[0].tl;

  if (from == -1)
    {
      ltl[0].copies = 1;
      T = ltl[0].T - 1;
      atl = world->atl[0].tl;
      atl[0].copies = ltl[0].copies;
      increase_timearchive (world, 0, 1, np);
      archive_timelist (&(atl[0]), ltl[0].tl, T, np);
      return;
    }
  if (from == to)
    {
      ltl[0].copies += 1;
      atl[from].copies = ltl[0].copies;
    }
  else
    {
      T = ltl[0].T - 1;
      increase_timearchive (world, 0, to, np);
      atl = world->atl[0].tl;
      atl[to].copies = ltl[0].copies = 1;
      world->likelihood[to] = world->likelihood[from];
      archive_timelist (&atl[to], ltl[0].tl, T, np);

    }
}


/*private functions========================================== */
/* ---------------------------------------------------------
   creates memory for archive of timelists for each locus
   the "locus 0" is reserved for for the current locus, this
   fake locus is use to speed up the NR-estimation for all chains
   whereas the loci 1 .. n are used to save the results of the last
   chain for the combined estimate at the end of the program */
void 
create_timearchive (timearchive_fmt ** atl, long loci, long samples, long numpop)
{
  long i, j;
  (*atl) = (timearchive_fmt *) calloc (1, sizeof (timearchive_fmt) * (2 + loci));
  for (i = 0; i < loci + 2; i++)
    {
      (*atl)[i].param = (double *) calloc (1, numpop * 2 * sizeof (double));
      (*atl)[i].param0 = (double *) calloc (1, numpop * 2 * sizeof (double));
      (*atl)[i].likelihood = (double *) calloc (1, (1 + samples) * sizeof (double));
      (*atl)[i].tl = (tarchive_fmt *) calloc (1, samples * sizeof (tarchive_fmt));
      (*atl)[i].T = (*atl)[i].allocT = samples;
      for (j = 0; j < samples; j++)
	{
	  (*atl)[i].tl[j].km = (double *) calloc (1, sizeof (double) * numpop);
	  (*atl)[i].tl[j].kt = (double *) calloc (1, sizeof (double) * numpop);
	  (*atl)[i].tl[j].p = (long *) calloc (1, sizeof (long) * numpop);
	  (*atl)[i].tl[j].l = (long *) calloc (1, sizeof (long) * numpop);
	}
    }
}

void 
increase_timearchive (world_fmt * world, long locus, long sample, long numpop)
{
  long i = locus, j, oldT = 0, size;
  if (sample >= world->atl[i].allocT)
    {
      oldT = world->atl[i].allocT;
      world->atl[i].allocT = MAX (sample + 1, world->atl[i].allocT + world->atl[i].allocT / 4.);
      size = world->atl[i].allocT;
      world->atl[i].tl = (tarchive_fmt *) realloc (world->atl[i].tl,
					      size * sizeof (tarchive_fmt));
      if (i == 0)
	world->likelihood = (double *) realloc (
			   world->likelihood, sizeof (double) * (1 + size));
      world->atl[i].likelihood = (double *) realloc (
		    world->atl[i].likelihood, sizeof (double) * (1 + size));
      for (j = oldT; j < world->atl[i].allocT; j++)
	{
	  world->atl[i].tl[j].km = (double *) calloc (1, sizeof (double) * numpop);
	  world->atl[i].tl[j].kt = (double *) calloc (1, sizeof (double) * numpop);
	  world->atl[i].tl[j].p = (long *) calloc (1, sizeof (long) * numpop);
	  world->atl[i].tl[j].l = (long *) calloc (1, sizeof (long) * numpop);
	}
      world->atl[i].T = sample;
    }
  else
    {
      world->atl[i].T = sample;
    }
}

void 
create_plotplane (world_fmt * world)
{
  short locus, i;
  world->plane = (char ***) calloc (1, sizeof (char **) * (world->loci + 1));
  world->plotmax = (plotmax_fmt *) calloc (1, sizeof (plotmax_fmt) * (world->loci + 1));
  for (locus = 0; locus < world->loci + 1; locus++)
    {
      world->plane[locus] =
	(char **) calloc (1, sizeof (char *) * (PLANESIZE + 3));
      for (i = 0; i < PLANESIZE + 3; i++)
	{
	  world->plane[locus][i] =
	    (char *) calloc (1, sizeof (char) * (PLANESIZE + PLANESIZE + 20));
	}
    }
}

void 
create_cov (world_fmt * world)
{
  short locus, i;
  world->cov = (double ***) calloc (1, sizeof (double **) * (world->loci + 1));
  for (locus = 0; locus < world->loci + 1; locus++)
    {
      world->cov[locus] = (double **) calloc (1, sizeof (double *)
					      * (world->numpop * 2 + 1));
      for (i = 0; i < world->numpop * 2 + 1; i++)
	{
	  world->cov[locus][i] = (double *) calloc (1, sizeof (double) * (world->numpop * 2 + 1));
	}
    }
}

void 
print_menu_equilib (option_fmt * options)
{
  char nowstr[LINESIZE];
  if (options->progress)
    {
      get_time (nowstr, "%H:%M:%S");
      fprintf (stdout, "%s   Equilibrate tree (first %i trees are not used)\n", nowstr,
	       options->burn_in);
    }
}

void 
print_finish (world_fmt * world, long filepos)
{
  char nowstr[LINESIZE];
  if (world->options->progress)
    {
      get_time (nowstr, "%H:%M:%S");
      fprintf (stdout, "%s   Program finished\n", nowstr);
    }
  get_time (nowstr, "%c");
  if (nowstr[0] != '\0')
    {
      if (filepos > 0)
	fseek (world->outfile, filepos, SEEK_SET);
      fprintf (world->outfile, "         finished at %s\n", nowstr);
    }

}

void 
archive_timelist (tarchive_fmt * atl, vtlist * tl, long T, long np)
{
  long j, i;
  double t;
  double line1 /*, line2 */ ;
  for (i = 0; i < np; i++)
    {
      line1 = tl[0].lineages[i];
      /* line2=0.;
         for(z=0; z< np; z++){
         if(i!=z)
         line2 +=  tl[0].lineages[z];
         }
       */
      atl->km[i] = line1 * tl[0].age;
      atl->kt[i] = line1 * (line1 - 1) * tl[0].age;
      atl->p[i] = whichp (tl[0].from, tl[0].to, i);
      atl->l[i] = whichl (tl[0].from, tl[0].to, i);
    }
  for (j = 1; j < T; j++)
    {
      t = tl[j].age - tl[j - 1].age;
      for (i = 0; i < np; i++)
	{
	  line1 = tl[j].lineages[i];
	  /*      line2=0.;
	     for(z=0; z< np; z++){
	     if(i!=z)
	     line2 +=  tl[j].lineages[z];
	     } */
	  atl->km[i] += line1 * t;
	  atl->kt[i] += line1 * (line1 - 1) * t;
	  atl->p[i] += whichp (tl[j].from, tl[j].to, i);
	  atl->l[i] += whichl (tl[j].from, tl[j].to, i);
	}
    }
}


void 
copy_timelist (tarchive_fmt * from, tarchive_fmt * to, long np)
{
  memcpy (to->km, from->km, sizeof (double) * np);
  memcpy (to->kt, from->kt, sizeof (double) * np);
  memcpy (to->p, from->p, sizeof (long) * np);
  memcpy (to->l, from->l, sizeof (long) * np);
}

long 
whichp (long from, long to, long pop)
{
  if (from == to)
    {
      if (from == pop)
	return 1;
    }
  return 0;
}

long 
whichl (long from, long to, long pop)
{
  if (from != to)
    {
      if (to == pop)
	return 1;
    }
  return 0;
}

/*
   long calc_T(timearchive_fmt * old, long steps)
   {
   long i, sum = 0;;

   for (i = 0; i < steps; i++) {
   sum += old->tl[i].copies;
   if (sum == steps)
   return i + 1;
   }
   return steps;
   }
 */
void 
print_results (world_fmt * world)
{
  long l, skipped = 0;
  long loci = world->loci;
  FILE *outfile;
  option_fmt *opt;
  char sch[10], lch[10], cva[50];
  opt = world->options;
  outfile = world->outfile;
  if (opt->schains == 1)
    strcpy (sch, "chain");
  else
    strcpy (sch, "chains");
  if (opt->lchains == 1)
    strcpy (lch, "chain");
  else
    strcpy (lch, "chains");
  fprintf (outfile, "\n\n");
  PAGEFEED;
  fprintf (outfile, "================================================");
  fprintf (outfile, "================================\n");
  fprintf (outfile, "MCMC-estimation\n");
  fprintf (outfile, "------------------------------------------------");
  fprintf (outfile, "--------------------------------\n");
  fprintf (outfile, "There were %li short %s (%li used trees out of ",
	   opt->schains, sch, opt->ssteps);
  fprintf (outfile, "sampled %li)\n", opt->sincrement * opt->ssteps);
  fprintf (outfile, "and %li long %s (%li used trees out of sampled %li),\n",
	   opt->lchains, lch, opt->lsteps, opt->lincrement * opt->lsteps);
  fprintf (outfile, "the final estimated parameter are:\n");
  fprintf (outfile, "Locus            Theta [4N(mu)]     ");
  fprintf (outfile, "           4Nm             ");
  fprintf (outfile, "Log(Likelihood)\n");
  fprintf (outfile, "         -------------------------- ");
  fprintf (outfile, "-------------------------- \n");
  fprintf (outfile, "             Pop 1        Pop 2          Pop 1       Pop 2                        \n");
  fprintf (outfile, "----------------------------------------------------------------");
  fprintf (outfile, "----------------\n");
  for (l = 1; l < loci + 1; l++)
    {
      if (world->data->skiploci[l - 1])
	{
	  skipped++;
	  continue;
	}
      fprintf (outfile, " % 3li     % 12.6f % 12.6f  % 12.6f % 12.6f   % 12.5g\n",
	       l, world->atl[l].param[0], world->atl[l].param[1],
	       world->atl[l].param[2] * world->atl[l].param[0],
	       world->atl[l].param[3] * world->atl[l].param[1], world->atl[l].param_like);
    }
  if (loci - skipped > 1)
    {
      if (!opt->gamma)
	{
	  fprintf (outfile, " All       % 10.6f   % 10.6f    % 10.6f   % 10.6f     % 10.5g\n",
		   world->atl[l].param[0], world->atl[l].param[1],
		   world->atl[l].param[2] * world->atl[l].param[0],
		   world->atl[l].param[3] * world->atl[l].param[1], world->atl[l].param_like);
	}
      else
	{
	  if (world->atl[l].param[4] < 10e-9)
	    strcpy (cva, "0");
	  else
	    sprintf (cva, "%f", sqrt (world->atl[l].param[4]));
	  fprintf (outfile, " All       % 10.6f   % 10.6f    % 10.6f   % 10.6f     % 10.5g\n",
		   world->atl[l].param[0], world->atl[l].param[1],
		   world->atl[l].param[2] * world->atl[l].param[0],
		   world->atl[l].param[3] * world->atl[l].param[1], world->atl[l].param_like);
	  fprintf (outfile, "With shape parameter Inv(alpha)=%g ([CV(mu)]^2; CV(mu)=%s,alpha=%g)\n",
		   world->atl[l].param[4], cva, 1. / world->atl[l].param[4]);
	  fprintf (outfile, "[Newton-Raphson needed %li cycles of maximal %i,\n  Norm(first derivatives)=%f (normal stopping criteria is < %f)]\n\n\n",
	     world->atl[l].trials, NTRIALS, world->atl[l].normd, LOCI_NORM);
	}
    }
  fprintf (outfile, "================================================");
  fprintf (outfile, "================================\n");
}

void 
print_fst (world_fmt * world, double **fstparam)
{
  long l, skipped = 0;
  long numpop = world->numpop;
  long loci = world->loci;
  FILE *outfile = world->outfile;
  char str1[LINESIZE], str2[LINESIZE], str3[LINESIZE], str4[LINESIZE];
  if (loci < 40)
    {
      PAGEFEED;
    }
  fprintf (outfile, "\n\n");
  fprintf (outfile, "----------------------------------------------------------------\n");
  fprintf (outfile, "\"Fst\"-calculation: Fw/Fb estimator(*) using %li islands\n",
	   numpop);
  fprintf (outfile, "%s\n", world->options->fsttype == 'M' ?
	   "Migration rates are variable, Theta_1 = Theta2" :
	   "Thetas are variable, M_1 = M_2\n");
  fprintf (outfile, "----------------------------------------------------------------\n");
  fprintf (outfile, " Locus                       Population 1\n");
  fprintf (outfile, "         -------------------------------------------------------\n");
  fprintf (outfile, "           Theta           4Nm            Fw             Fb\n");
  fprintf (outfile, "-----------------------------------  ---------------------------\n");
  for (l = 0; l < loci; l++)
    {
      if (world->data->skiploci[l])
	{
	  skipped++;
	  continue;
	}
      prepare_print_nu (fstparam[l][0], str1);
      prepare_print_nm (fstparam[l][0], fstparam[l][2], str3);
      fprintf (outfile, " % 3li %12.12s  %12.12s  % 12.6g  % 12.6g\n",
	       l + 1, str1, str3,
	       fstparam[l][numpop * 2], fstparam[l][numpop * 2 + 2]);
    }
  if (world->loci - skipped > 1)
    {
      prepare_print_nu (fstparam[l][0], str1);
      prepare_print_nm (fstparam[l][0], fstparam[l][2], str3);
      fprintf (outfile, "  All%12.12s  %12.12s  % 12.6g  % 12.6g\n",
	       str1, str3,
	       fstparam[l][numpop * 2], fstparam[l][numpop * 2 + 2]);
    }
  fprintf (outfile, "----------------------------------------------------------------\n");
  fprintf (outfile, " Locus                     Population 2\n");
  fprintf (outfile, "         -------------------------------------------------------\n");
  fprintf (outfile, "           Theta         4Nm              Fw             Fb\n");
  fprintf (outfile, "-----------------------------------  ---------------------------\n");
  for (l = 0; l < loci; l++)
    {
      if (world->data->skiploci[l])
	continue;
      prepare_print_nu (fstparam[l][1], str2);
      prepare_print_nm (fstparam[l][1], fstparam[l][3], str4);
      fprintf (outfile, " % 3li %12.12s  %12.12s  % 12.6g  % 12.6g\n",
	       l + 1, str2, str4,
	       fstparam[l][numpop * 2 + 1], fstparam[l][numpop * 2 + 2]);
    }
  if (world->loci - skipped > 1)
    {
      prepare_print_nu (fstparam[l][1], str2);
      prepare_print_nm (fstparam[l][1], fstparam[l][3], str4);
      fprintf (outfile, "  All%12.12s  %12.12s  % 12.6g  % 12.6g\n",
	       str2, str4,
	       fstparam[l][numpop * 2 + 1], fstparam[l][numpop * 2 + 2]);
    }
  fprintf (outfile, "----------------------------------------------------------------\n");
  fprintf (outfile, "(*) based on Maynard Smith (1970), American Naturalist 104:231-237\n");
  fprintf (outfile, "and Nei & Feldman (1972), Theoretical Population Biology 3:460-465\n");
  fprintf (outfile, "Negative values show a violation of the assumptions and these\n");
  fprintf (outfile, "estimates (Theta AND 4Nm for both populations) should be discarded.\n");
  fprintf (outfile, "(-) can not be estimated\n");
  fprintf (outfile, "(0/0 or x/0) Divisions by zero\n");
}

void 
prepare_print_nu (double nu, char *str)
{
  if (nu <= -999)
    sprintf (str, "-");
  else
    sprintf (str, "% 12.6f", nu);
}

void 
prepare_print_nm (double nm, double nmu, char *strllike)
{
  if ((fabs (nmu) > 10e-20) && (fabs (nm) > 10e-20))
    sprintf (strllike, "% 10.5f", nm * nmu);
  else
    {
      if ((fabs (nmu) < 10e-20) && (fabs (nm) < 10e-20))
	{
	  sprintf (strllike, "0/0");
	}
      else
	{
	  if ((fabs (nmu) < 10e-20) && (fabs (nm) > 10e-20))
	    sprintf (strllike, "% 10.5f/0", nm);
	  else
	    {
	      if ((fabs (nmu) > 10e-20) && (fabs (nm) < 10e-20))
		sprintf (strllike, "0");
	    }
	}
    }
}

void 
print_menu_coalnodes (world_fmt * world, long G)
{
  long g, pop, minp = world->sumtips, maxp = 0;
  char ss[10];
  long **contribution;
  long nodenum = world->sumtips;
  tarchive_fmt *tl = world->atl[0].tl;
  if (world->options->verbose)
    {
      contribution = (long **) calloc (1, sizeof (long *) * nodenum);
      contribution[0] = (long *) calloc (1, sizeof (long) * nodenum * world->numpop);
      for (pop = 1; pop < world->numpop; pop++)
	{
	  contribution[pop] = contribution[0] + pop * nodenum;
	}
      for (g = 0; g < G; g++)
	{
	  for (pop = 0; pop < world->numpop; pop++)
	    {
	      contribution[pop][(long) tl[g].p[pop]] += tl[g].copies;
	    }
	}
      for (g = 0; g < nodenum; g++)
	{
	  for (pop = 0; pop < world->numpop; pop++)
	    {
	      if (maxp < g && contribution[pop][g] > 0)
		maxp = g;
	      if (minp > g && contribution[pop][g] > 0)
		minp = g;
	    }
	}
      fprintf (stdout, "           Coalescent nodes: ");
      for (g = minp; g < maxp + 1; g++)
	{
	  fprintf (stdout, "%2li ", g);
	}
      fprintf (stdout, "\n");

      for (pop = 0; pop < world->numpop; pop++)
	{
	  fprintf (stdout, "             population %3li: ", pop);
	  for (g = minp; g < maxp + 1; g++)
	    {
	      if (contribution[pop][g] == 0)
		{
		  strcpy (ss, "-");
		}
	      else
		{
		  if (contribution[pop][g] >= 100)
		    strcpy (ss, "*");
		  else
		    sprintf (ss, "%-6li", contribution[pop][g]);
		}
	      fprintf (stdout, "%2.2s ", ss);
	    }
	  fprintf (stdout, "\n");
	}
      free (contribution[0]);
      free (contribution);
    }
}


void 
print_menu_createplot (void)
{
  char nowstr[LINESIZE];
  get_time (nowstr, "%H:%M:%S");
  fprintf (stdout, "%s   Creating data for plots\n", nowstr);
}


void 
calc_loci_plane (world_fmt * world, nr_fmt * nr, timearchive_fmt * atl, double **pl, long loci, double contours[])
{
  long i, j;
  double max1 = -DBL_MAX;
  double max2 = -DBL_MAX;
  double values[PLANESIZE] = PLANETICKS;
  long intervals = PLANESIZE;
  if (world->options->gamma)
    {
      nr->param[4] = world->atl[loci + 1].param[4];
    }
  for (i = 0; i < intervals; i++)
    {
#ifdef MAC
      eventloop ();
#endif
      nr->param[0] = values[i];
      if (world->options->gamma)
	calc_gamma (nr);
      for (j = 0; j < intervals; j++)
	{
	  nr->param[0] = values[i];
	  nr->param[1] = world->atl[loci + 1].param[1];
	  nr->param[2] = values[j] / values[i];
	  nr->param[3] = world->atl[loci + 1].param[3];
	  calc_loci_like (nr, atl, loci, world->options->gamma);
	  pl[i][j] = nr->llike;
	  if (max1 < nr->llike)
	    max1 = nr->llike;
	}
    }
  nr->param[0] = world->atl[loci + 1].param[0];
  if (world->options->gamma)
    calc_gamma (nr);
  for (i = 0; i < intervals; i++)
    {
      for (j = 0; j < intervals; j++)
	{
	  nr->param[0] = world->atl[loci + 1].param[0];
	  nr->param[1] = values[i];
	  nr->param[2] = world->atl[loci + 1].param[2];
	  nr->param[3] = values[j] / values[i];
	  calc_loci_like (nr, atl, loci, world->options->gamma);
	  pl[i][j + intervals] = nr->llike;
	  if (max2 < nr->llike)
	    max2 = nr->llike;
	}
    }
  contours[0] += max1;
  contours[1] += max1;
  contours[2] += max1;
  contours[3] += max1;
  contours[4] += max2;
  contours[5] += max2;
  contours[6] += max2;
  contours[7] += max2;
}

void 
calc_locus_plane (world_fmt * world, nr_fmt * nr, tarchive_fmt * tl, long G,
		  double **pl, double contours[])
{
  long intervals = PLANESIZE;

  long i, j;
  double max1 = -DBL_MAX;
  double max2 = -DBL_MAX;
  double values[PLANESIZE] = PLANETICKS;
#ifdef _GNU_
  double param[nr->numpop2];
#else
  double *param;
  param = (double *) calloc (1, sizeof (double) * nr->numpop2);
#endif
  memcpy (param, nr->param, sizeof (double) * nr->numpop2);

  for (i = 0; i < intervals; i++)
    {
#ifdef MAC
      eventloop ();
#endif
      for (j = 0; j < intervals; j++)
	{
	  nr->param[0] = values[i];
	  nr->param[1] = param[1];
	  nr->param[2] = values[j] / values[i];
	  nr->param[3] = param[3];
	  calc_like (nr, tl, G);
	  pl[i][j] = nr->llike;
	  if (max1 < nr->llike)
	    max1 = nr->llike;
	}
    }
  for (i = 0; i < intervals; i++)
    {
      for (j = 0; j < intervals; j++)
	{
	  nr->param[0] = param[0];
	  nr->param[1] = values[i];
	  nr->param[2] = param[2];
	  nr->param[3] = values[j] / values[i];
	  calc_like (nr, tl, G);
	  pl[i][j + intervals] = nr->llike;
	  if (max2 < nr->llike)
	    max2 = nr->llike;
	}
    }
  contours[0] += max1;
  contours[1] += max1;
  contours[2] += max1;
  contours[3] += max1;
  contours[4] += max2;
  contours[5] += max2;
  contours[6] += max2;
  contours[7] += max2;
#ifndef _GNU_
  free (param);
#endif
}

void 
fill_plotplane (char **plane, double **pl, double contours[], plotmax_fmt * plotmax)
{
  long i, j, z, zz = 0;

  char line[100];
  long myval[PLANEBIGTICKS] = PLANEBIGTICKVALUES;
  double values[PLANESIZE] = PLANETICKS;
  long intervals = PLANESIZE;
  for (i = 0; i < intervals; i++)
    {
      if (i % 7)
	line[i] = '-';
      else
	line[i] = '+';
    }
  line[i] = '\0';
  sprintf (plane[0], "     +%s+    +%s+   ", line, line);
  for (i = 0; i < intervals; i++)
    {
      memset (plane[i + 1], ' ', sizeof (char) * (intervals + intervals + 20));
      plane[i + 1][intervals + intervals + 19] = '\0';
      if (!((i) % 7))
	{
	  sprintf (plane[i + 1], "  %2.0li +", myval[zz++]);
	  if (myval[zz - 1] == 0)
	    plane[i + 1][3] = '0';
	}
      else
	plane[i + 1][5] = '|';
      for (j = 0; j < intervals; j++)
	{
	  if (pl[i][j] < contours[1])
	    {
	      if (pl[i][j] < contours[2])
		{
		  if (pl[i][j] < contours[3])
		    {
		      plane[i + 1][j + 6] = ' ';
		    }
		  else
		    {
		      plane[i + 1][j + 6] = '-';
		    }
		}
	      else
		{
		  plane[i + 1][j + 6] = '+';
		}
	    }
	  else
	    {
	      if (pl[i][j] < (contours[0] - EPSILON))
		{
		  plane[i + 1][j + 6] = '*';
		}
	      else
		{
		  plane[i + 1][j + 6] = 'X';
		  plotmax->l1 = pl[i][j];
		  plotmax->x1 = values[j];
		  plotmax->y1 = values[i];
		}
	    }
	}
      if ((i) % 7)
	{
	  plane[i + 1][j + 6] = '|';
	  plane[i + 1][j + 11] = '|';
	}
      else
	{
	  plane[i + 1][j + 6] = '+';
	  plane[i + 1][j + 11] = '+';
	}
      for (j = intervals + 7 + 5 /*- 1*/ ; j < intervals + 7 + 5 + intervals /*- 2*/ ; j++)
	{
	  z = j - 7 - 5 /*+ 1 */ ;
	  if (pl[i][z] < contours[5])
	    {
	      if (pl[i][z] < contours[6])
		{
		  if (pl[i][z] < contours[7])
		    {
		      plane[i + 1][j] = ' ';
		    }
		  else
		    {
		      plane[i + 1][j] = '-';
		    }
		}
	      else
		{
		  plane[i + 1][j] = '+';
		}
	    }
	  else
	    {
	      if (pl[i][z] < (contours[4] - EPSILON))
		{
		  plane[i + 1][j] = '*';
		}
	      else
		{
		  plane[i + 1][j] = 'X';
		  plotmax->l2 = pl[i][z];
		  plotmax->x2 = values[z - intervals];
		  plotmax->y2 = values[i];
		}
	    }
	}
      if ((i) % 7)
	{
	  plane[i + 1][j] = '|';
	  plane[i + 1][j + 1] = '\0';
	}
      else
	{
	  plane[i + 1][j] = '+';
	  plane[i + 1][j + 1] = '\0';
	}
    }
  sprintf (plane[intervals + 1], "     +%s+    +%s+", line, line);
  sprintf (plane[intervals + 2], "     -3     -2     -1      0      1      2     -3     -2     -1      0      1      2");
}

void 
print_mathematica (world_fmt * world, double **plane, long x, long y)
{
  long i, j;
  static long number = 1;
  FILE *mathfile = world->mathfile;
  if (world->options->plot == 1 && world->options->plotmethod == PLOTALL)
    {
      fprintf (mathfile, "\n\nlocus%-li={{\n", number++);
      for (i = 0; i < x - 1; i++)
	{
	  fprintf (mathfile, "{");
	  for (j = 0; j < y - 1; j++)
	    {
	      fprintf (mathfile, "%20.20g,", plane[i][j]);
	    }
	  fprintf (mathfile, "%20.20g},\n", plane[i][j]);
	}
      fprintf (mathfile, "{");
      for (j = 0; j < y - 1; j++)
	{
	  fprintf (mathfile, "%20.20g,", plane[i][j]);
	}
      fprintf (mathfile, "%20.20g}},{\n", plane[i][j]);
      for (i = 0; i < x - 1; i++)
	{
	  fprintf (mathfile, "{");
	  for (j = y; j < 2 * y - 1; j++)
	    {
	      fprintf (mathfile, "%20.20g,", plane[i][j]);
	    }
	  fprintf (mathfile, "%20.20g},\n", plane[i][j]);
	}
      fprintf (mathfile, "{");
      for (j = y; j < 2 * y - 1; j++)
	{
	  fprintf (mathfile, "%20.20g,", plane[i][j]);
	}
      fprintf (mathfile, "%20.20g}}}\n", plane[i][j]);
    }
}


void 
print_cov (world_fmt * world, long numpop, long loci, double ***cov)
{
  FILE *outfile = world->outfile;
  long locus, skipped = 0;
#ifdef _GNU_
  double corr[2 * numpop + 1];
#else
  double *corr;
  corr = (double *) calloc (1, sizeof (double) * (2 * numpop + 1));
#endif
  fprintf (outfile, "\n\n");
  fprintf (outfile, "--------------------------------------------------------------------\n");
  fprintf (outfile, "MCMC estimation:\n");
  fprintf (outfile, "Covariance matrix(*)                         Correlation matrix\n");
  fprintf (outfile, "-------------------------------------------  -----------------------\n");
  for (locus = 0; locus < world->loci; locus++)
    {
      if (world->data->skiploci[locus])
	{
	  skipped++;
	  continue;
	}
      fprintf (outfile, "Locus %li:\n", locus + 1);
      print_cov_table (outfile, locus, world, corr, 0);
      fprintf (outfile, "\n");
    }
  fprintf (outfile, "\n\n");
  if (world->loci - skipped > 1)
    {
      if (world->options->gamma)
	{
	  fprintf (outfile, "Over all loci\n");
	  fprintf (outfile, "------------------------------------------------------  ");
	  fprintf (outfile, "----------------------------\n");
	  print_cov_table (outfile, locus, world, corr, 1);
	}
      else
	{
	  fprintf (outfile, "Over all loci\n");
	  fprintf (outfile, "-------------------------------------------  -----------------------\n");
	  print_cov_table (outfile, locus, world, corr, 0);
	}
    }
#ifndef _GNU_
  free (corr);
#endif
}

void 
print_cov_table (FILE * outfile, long locus, world_fmt * world, double *corr, long addvar)
{
  long i, j;
  double denom, temp1, temp2;
  for (i = 0; i < world->numpop * 2 + addvar; i++)
    {
      for (j = 0; j < world->numpop * 2 + addvar; j++)
	{
	  temp1 = fabs (world->cov[locus][i][i]);
	  if (temp1 < DBL_EPSILON)
	    temp1 = DBL_EPSILON;
	  temp2 = fabs (world->cov[locus][j][j]);
	  if (temp2 < DBL_EPSILON)
	    temp2 = DBL_EPSILON;
	  denom = 0.5 * (log (temp1) + log (temp2));
	  if ((temp1 = exp (denom)) == 0.0)
	    corr[j] = 9.99;
	  else
	    corr[j] = world->cov[locus][i][j] / exp (denom);
	}
      for (j = 0; j < world->numpop * 2 + addvar; j++)
	{
#if TESTING
	  fprintf (outfile, "% 20.10f ", world->cov[locus][i][j]);	/*was 10.4 */
#else
	  fprintf (outfile, "% 10.4f ", world->cov[locus][i][j]);
#endif
	}
      for (j = 0; j < world->numpop * 2 + addvar; j++)
	{
	  if (corr[j] < 9.)
	    fprintf (outfile, "% 4.2f ", corr[j]);
	  else
	    fprintf (outfile, "  --  ");
	}
      fprintf (outfile, "\n");
    }
}


void 
free_tree (node * p)
{
  if (!p->tip)
    {
      if (p->next->back != NULL)
	{
	  free_tree (p->next->back);
	}
      if (p->type != 'm' && p->next->next->back != NULL)
	{
	  free_tree (p->next->next->back);
	}
    }
  else
    {
      free (p->nayme);
    }
  switch (p->type)
    {
    case 'm':
      free_nodelet (p, 2);
      break;
    case 't':
      free_nodelet (p, 1);
      break;
    default:
      free_nodelet (p, 3);
      break;
    }
}

void 
free_nodelet (node * p, long num)
{
  long i;
  node *q;
  switch ((short) num)
    {
    case 3:
      free (p->next->next);
    case 2:
      free (p->next);
    case 1:
      free (p);
      break;
    default:
      for (i = 0; i < num; i++)
	{
	  q = p->next;
	  free (p);
	  p = q;
	}
    }
}

void 
free_seqx (node * p, long sites)
{
  long j;
  if (!p->tip)
    {
      if (p->next->back != NULL)
	{
	  free_seqx (crawlback (p->next), sites);
	}
      if (p->next->next->back != NULL)
	{
	  free_seqx (crawlback (p->next->next), sites);
	}
    }
  if (p->type == 'r')
    return;
  for (j = 0; j < sites; j++)
    free (p->x.s[j]);
}

void 
free_x (node * p)
{
  if (!p->tip)
    {
      if (p->next->back != NULL)
	{
	  free_x (crawlback (p->next));
	}
      if (p->next->next->back != NULL)
	{
	  free_x (crawlback (p->next->next));
	}
    }
  if (p->type == 'r')
    return;
  free (p->x.a);
}


void 
test_locus_like (double *param0, double *param1, long df, long locus, world_fmt * world,
		 boolean withhead, char *this_string)
{
  char c;
  char *teststat, temp[256];
  double like1, like0, testval, chi05, chi01;
  nr_fmt *nr;
  long i, g, part1len = 1, part2len = 1;
  long elem;
  nr = (nr_fmt *) calloc (1, sizeof (nr_fmt));
  create_nr (nr, world->atl[locus + 1].numpop, world->atl[locus + 1].T);
  nr->skiploci = world->data->skiploci;
  elem = nr->numpop2 = world->numpop2;
  nr->numpop = world->numpop;
  memcpy (nr->param, param0, sizeof (double) * elem);
  /* Prob(G|Param0) */
  for (g = 0; g < world->atl[locus + 1].T; g++)
    {
      nr->apg0[g] = probG (world->param0, &world->atl[locus + 1].tl[g], nr->numpop);
    }
  like0 = calc_like (nr, world->atl[locus + 1].tl, world->atl[locus + 1].T);
  memcpy (nr->param, param1, sizeof (double) * elem);
  like1 = calc_like (nr, world->atl[locus + 1].tl, world->atl[locus + 1].T);
  testval = -2. * (like0 - like1);
  chi05 = chisquare (df, 0.05);
  chi01 = chisquare (df, 0.01);
  if (this_string != NULL)
    {
      c = this_string[0];
      for (i = 1; i < 1000 && c != '\0'; i++)
	c = this_string[i];
      part2len = part2len = i;
      teststat = this_string;
    }
  else
    {
      teststat = (char *) calloc (1, sizeof (char) * 256);
      sprintf (teststat, "Locus %li: H0:{%f", locus + 1, param0[0]);
      for (i = 1; i < elem; i++)
	{
	  sprintf (temp, ",%f", param0[i]);
	  strcat (teststat, temp);
	}
      sprintf (temp, "}=");
      strcat (teststat, temp);
      part1len = strlen (teststat);
      sprintf (temp, "\n       {%f", param1[0]);
      strcat (teststat, temp);
      for (i = 1; i < elem; i++)
	{
	  sprintf (temp, ",%f", param1[i]);
	  strcat (teststat, temp);
	}
      sprintf (temp, "}");
      strcat (teststat, temp);
      part2len = strlen (teststat);
    }
  if (withhead)
    {
      fprintf (world->outfile, "Likelihood ratio test with alpha=0.05 and 0.01\n");
      part1len = MAX (part1len, part2len - part1len);
      fprintf (world->outfile, "%*.*s %-10.10s %-3.3s %-4.4s %-4.4s\n",
	       (int) part1len, (int) part1len, " ", "Test value", "DF", "0.05", "0.01");
      for (i = 0; i < part2len + 35; i++)
	fputc ('-', world->outfile);
      fprintf (world->outfile, "\n");
    }
  fprintf (world->outfile, "%s %10.6f %3li %4.2f %4.2f\n", teststat,
	   testval, elem, chi05, chi01);
  if (this_string == NULL)
    free (teststat);
  free_nr (nr);
}


void 
test_loci_like (double *param0, double *param1, long df, long loci, world_fmt * world,
		boolean withhead, char *this_string)
{
  char c;
  char *teststat, temp[256];
  double like1, like0, testval, chi05, chi01;
  nr_fmt *nr;
  long i, g = world->atl[1].T, part1len = 1, part2len = 1;
  long elem;
  nr = (nr_fmt *) calloc (1, sizeof (nr_fmt));
  for (i = 1; i < loci + 1; i++)
    {
      if (g < world->atl[i].T)
	g = world->atl[i].T;
    }
  create_nr (nr, world->atl[loci].numpop, g);
  elem = world->options->gamma ? nr->numpop2 + 1 : nr->numpop2;
  nr->skiploci = world->data->skiploci;
  memcpy (nr->param, param0, sizeof (double) * elem);
  if (world->options->gamma)
    calc_gamma (nr);
  like0 = calc_loci_like (nr, world->atl, world->loci, world->options->gamma);
  memcpy (nr->param, param1, sizeof (double) * elem);
  if (world->options->gamma)
    calc_gamma (nr);
  like1 = calc_loci_like (nr, world->atl, world->loci, world->options->gamma);
  testval = -2 * (like0 - like1);
  chi05 = chisquare (df, 0.05);
  chi01 = chisquare (df, 0.01);
  if (this_string != NULL)
    {
      c = this_string[0];
      for (i = 1; i < 1000 && c != '\0'; i++)
	c = this_string[i];
      part2len = part2len = i;
      teststat = this_string;
    }
  else
    {
      teststat = (char *) calloc (1, sizeof (char) * 256);
      sprintf (teststat, "All   :{%4.4f", param0[0]);
      for (i = 1; i < elem; i++)
	{
	  sprintf (temp, ",%4.4f", param0[i]);
	  strcat (teststat, temp);
	}
      sprintf (temp, "}=");
      strcat (teststat, temp);
      part1len = strlen (teststat);
      sprintf (temp, "\n       {%4.4f", param1[0]);
      strcat (teststat, temp);
      for (i = 1; i < elem; i++)
	{
	  sprintf (temp, ",%4.4f", param1[i]);
	  strcat (teststat, temp);
	}
      sprintf (temp, "}");
      strcat (teststat, temp);
      part2len = strlen (teststat);
    }
  if (withhead)
    {
      fprintf (world->outfile, "\n\n\nLikelihood ratio test with alpha=0.05 and 0.01\n");
      part1len = MAX (part1len, part2len - part1len);
      fprintf (world->outfile, "%*.*s %-10.10s %-3.3s %-4.4s %-4.4s\n",
	       (int) part1len, (int) part1len, " ", "Test value", "DF", "0.05", "0.01");
      for (i = 0; i < part1len + 35; i++)
	fputc ('-', world->outfile);
      fprintf (world->outfile, "\n");
    }
  fprintf (world->outfile, "%s %10.2f %3li %4.2f %4.2f\n", teststat,
	   testval, elem, chi05, chi01);
  if (this_string == NULL)
    free (teststat);
  free_nr (nr);
}


double 
chisquare (long df, double alpha)
{
  const double table05[] =
  {3.84146, 5.99147, 7.81473, 9.48773, 11.0705, 12.5916};
  const double table01[] =
  {6.63490, 9.21034, 11.3449, 13.2767, 15.0863, 16.8119};

  if (alpha == 0.05)
    return table05[df - 1];
  if (alpha == 0.01)
    return table01[df - 1];
  error ("Chi-distribution for any probability alpha is not implemented");
  return -1;
}


long 
set_test_param (double *param, char *strp, world_fmt * world, long lrline, long locus)
{
  long i = 0, z = 0, zz = 0, zzz = 0, df = 0;
  char *tmp, *ss;
  double *meanparam, mean;
  ss = (char *) calloc (1, sizeof (char) * LINESIZE);
  tmp = (char *) calloc (1, sizeof (char) * LINESIZE);
  strcpy (ss, strp);
  if (world->loci - world->skipped > 1 && world->options->lratio->data[lrline].type == MEAN)
    meanparam = world->atl[world->loci + 1].param;
  else
    meanparam = world->atl[locus + 1].param;
  while (ss[zzz] != '\0')
    {
      tmp[i] = ss[zzz++];
      if (!(tmp[i] == ',' || tmp[i] == '\0' || tmp[i] == '\n' || tmp[i] == ';'))
	{
	  if (tmp[i] != ' ')
	    i++;
	}
      else
	{
	  tmp[i] = '\0';
	  i = 0;
	  switch (tmp[0])
	    {
	    case '*':
	      param[z] = meanparam[z];
	      z++;
	      break;
	    case 't':
	      zz = atol (tmp) - 1;
	      df++;
	      if (zz < 0)
		{
		  mean = 0.0;
		  for (zz = 0; zz < world->numpop; zz++)
		    mean += meanparam[zz];
		  mean /= world->numpop;
		  param[z] = mean;
		}
	      else
		{
		  param[z] = meanparam[zz];
		}
	      z++;
	      break;
	    case 'm':
	      zz = atol (tmp) - 1;
	      df++;
	      if (zz < 0)
		{
		  mean = 0.0;
		  for (zz = world->numpop; zz < 2 * world->numpop; zz++)
		    mean += meanparam[zz];
		  mean /= world->numpop;
		  param[z] = mean;
		}
	      else
		{
		  param[z] = meanparam[zz];
		}
	      z++;
	      break;
	    default:
	      df++;
	      param[z] = MAX (atof (tmp), SMALLEST_THETA);
	      if (z >= world->numpop)
		param[z] /= param[z - world->numpop];
	      z++;
	      break;
	    }
	}
    }
  return df;
}


void 
print_CV (world_fmt * world)
{
  long i;
  long elem;
  char temp[100];
  fprintf (world->outfile, "\n\nVariance and coefficient of variance\n");
  if (world->loci - world->skipped > 1)
    {
      elem = 2 * world->numpop + (world->options->gamma ? 1 : 0);
      fprintf (world->outfile, "PARAM@ ");
      for (i = 0; i < elem; i++)
	{
	  fprintf (world->outfile, "%20.20g ", world->atl[world->loci + 1].param[i]);
	}
      fprintf (world->outfile, "\nVAR@   ");
      for (i = 0; i < elem; i++)
	{
	  fprintf (world->outfile, "%20.20g ", world->cov[world->loci][i][i]);
	}
      fprintf (world->outfile, "\nCV@    ");
      for (i = 0; i < elem; i++)
	{
	  strcpy (temp, "-");
	  if (world->cov[world->loci][i][i] >= 0)
	    {
	      sprintf (temp, "%20.20g", (sqrt (world->cov[world->loci][i][i])) /
		       world->atl[world->loci + 1].param[i]);
	    }
	  fprintf (world->outfile, "%-s ", temp);
	}
      fprintf (world->outfile, "\n");
    }
  else
    {				/* one locus */
      elem = 2 * world->numpop;
      fprintf (world->outfile, "PARAM@ ");
      for (i = 0; i < elem; i++)
	{
	  fprintf (world->outfile, "%20.20g ", world->atl[1].param[i]);
	}
      fprintf (world->outfile, "\nVAR@   ");
      for (i = 0; i < elem; i++)
	{
	  fprintf (world->outfile, "%20.20g ", world->cov[0][i][i]);
	}
      fprintf (world->outfile, "\nCV@    ");
      for (i = 0; i < elem; i++)
	{
	  strcpy (temp, "-");
	  if (world->cov[0][i][i] >= 0)
	    {
	      sprintf (temp, "%20.20g", (sqrt (world->cov[0][i][i])) /
		       world->atl[1].param[i]);
	    }
	  fprintf (world->outfile, "%-s ", temp);
	}
      fprintf (world->outfile, "\n");
    }
}
