/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 P A R A M E T E R E S T I M A T I O N   R O U T I N E S 

 estimates parameter for each locus
 using a Broyden minimization
 

 Peter Beerli 1997, Seattle
 beerli@genetics.washington.edu
 $Id: broyden.c,v 1.15 1999/06/02 18:46:24 beerli Exp $
-------------------------------------------------------*/
extern double norm (double *d, long size);

#include "migration.h"
#include "world.h"
#include "random.h"
#include "combroyden.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif
/* prototypes ----------------------------------------- */
void broyden (world_fmt * world, double **covariance, long numg, long chain, char type, char ***plane);

double absmaxvec (double *v, long n);
void create_nr (nr_fmt * nr, world_fmt * world, long G, long profilenum);
void reset_hess (double **hess, long n);
void destroy_nr (nr_fmt * nr, world_fmt * world);
double calc_like (nr_fmt * nr, double *param, tarchive_fmt * atl, long G, long locus);
void calc_param (nr_fmt * nr, double *param, double lamda);
void param_all_adjust (nr_fmt * nr, boolean boolgamma);
void gradient (double *d, nr_fmt * nr);
double probG (double *param, double *lparam, tarchive_fmt * tl, nr_fmt * nr);
double logprob_noevent (world_fmt * world, long interval);
double sum_migprob (world_fmt * world, long pop, long interval);

void calc_cov (double **dd, double *d, double *param, long n);
boolean is_singular (double **dd, long n);
void print_contribution (nr_fmt * nr, tarchive_fmt * tyme, long G);

void calc_dv (double *dv, double **hess, double *gxv, long n);
double calc_line (nr_fmt * nr, double a, double b, double c, double (*psi) (double lamda, nr_fmt * nr));
void calc_hessian (double **hess, long n, double *delta, double *gamma);
double psi (double lamda, nr_fmt * nr);
void grad2loggrad (nr_fmt * nr, double *d, double PGC, long nn);
void log_param0 (double *param, double *lparam,long nn);
void create_apg0 (double *apg0, nr_fmt * nr, timearchive_fmt * tyme);

void force_sametheta (nr_fmt * nr);
void force_samemigration (nr_fmt * nr);
void calc_symmetric_d (nr_fmt * nr, long nn, long start, long stop);
void force_symmetric_d (long model, nr_fmt * nr, long nn);
void check_symmetric_d (nr_fmt * nr, long nn);
void check_matrix_arbitrary (nr_fmt * nr);

/* see Dahlquist p 443 */


void
broyden (world_fmt * world, double **covariance, long numg, long chain, char type, char ***plane)
{

  long i, nn, trials;
  double lamda = 1.0, normd, normd20 = 1e20;
  double newL, oldL, *xv, *xv0;
  double *gamma, *delta, *tvec, *dv, *gxv, *gxv0, **hess;
  nr_fmt *nr;
  nr = calloc (1, sizeof (nr_fmt));
  create_nr (nr, world, numg, 0);
  nn = nr->numpop2;
  tvec = (double *) calloc (1, sizeof (double) * nn);
  reset_hess (covariance, nn);
  delta = nr->delta;
  gamma = nr->gdelta;
  hess = nr->dd = covariance;
  dv = nr->dv;
  gxv = nr->d;
  gxv0 = nr->od;
  create_apg0 (nr->apg0[0], nr, &world->atl[0]);
  xv0 = nr->oparam;
  xv = nr->param;
  oldL = calc_like (nr, nr->param, nr->tl, nr->numg, 0);
  gradient (gxv, nr);
  grad2loggrad (nr, nr->d, nr->PGC, nn);
  memcpy (gxv0, gxv, sizeof (double) * nn);
  memcpy (xv0, xv, sizeof (double) * nn);
  memcpy (dv, gxv, sizeof (double) * nn);
  memset (delta, 0, sizeof (double) * nn);

  for (trials = 0; trials < NTRIALS; trials++)
    {
#ifdef MAC
      eventloop ();
#endif
      newL = -DBL_MAX;
      lamda = calc_line (nr, -1., 0.1, 1., psi);
      while (newL < oldL && fabs (lamda) > DBL_EPSILON)
	{
	  calc_loci_param (nr, nr->oparam, lamda,FALSE);
	  newL = calc_like (nr, nr->param, nr->tl, nr->numg, 0);
	  lamda /= 2.;
	}
      normd = norm (gxv, nr->partsize);
      if(world->options->verbose)
	fprintf(stdout,"<%li,%f>",trials,normd);
      oldL = newL;
      if ((trials + 1) % 20 == 0)
	{
	  if (fabs (normd - normd20) < EPSILON)
	    break;
	  normd20 = normd;
	}
      if (!((((normd > EPSILON) &&
      (fabs (lamda) > DBL_EPSILON) && (trials < NTRIALS))) || (trials < 3)))
	break;
/*-------------------------------------------*/
      memcpy (gxv0, gxv, sizeof (double) * nn);
      gradient (gxv, nr);
      grad2loggrad (nr, nr->d, nr->PGC, nn);
      for (i = 0; i < nn; i++)
	{
	  delta[i] = log (xv[i] / xv0[i]);	/*log(xv[i])-log(xv0[i]); */
	  gamma[i] = gxv[i] - gxv0[i];
	}
      memset (tvec, 0, sizeof (double) * nn);
      calc_hessian (hess, nn, delta, gamma);
      calc_dv (dv, hess, gxv, nn);
/*-------------------------------------------*/
      memcpy (xv0, xv, sizeof (double) * nn);
    }
  memcpy (world->param0, nr->param, sizeof (double) * nn);
  world->param_like = newL;
  if (world->options->progress)
    {
      print_menu_chain (type, chain, nr->numg, world);
      if (world->options->verbose)
	{
	  print_contribution (nr, nr->tl, nr->numg);
	  fprintf (stdout, "           Maximization steps needed:   %li\n", trials);
	  fprintf (stdout, "           Norm of first derivatives:   %f\n", normd);
	}
    }
  if (world->param_like < world->options->lcepsilon &&
      world->options->plotnow && !world->options->simulation)
    {
      create_locus_plot (world, plane, nr->tl, nr, nr->numg);
      //      calc_cov (hess, gxv, xv, nn);
    }
  free(tvec);
  destroy_nr (nr, world);
}

void
calc_dv (double *dv, double **hess, double *gxv, long n)
{
  long i, j;
  memset (dv, 0, sizeof (double) * n);
  for (i = 0; i < n; i++)
    {
      for (j = 0; j < n; j++)
	{
	  dv[i] += hess[i][j] * gxv[j];
	}
    }
}

#define PP 0.61803399
#define QQ 0.38196601
#define MOVE3(a,b,c,d) (a)=(b);(b)=(c);(c)=(d)

double
calc_line (nr_fmt * nr, double a, double b, double c,
	   double (*psi) (double lamda, nr_fmt * nr))
{
  /* a < b < c AND psia > psib < psic */

  double d, psib, psic;
  d = c;
  if (fabs (c - b) > fabs (b - a))
    {
      c = b + QQ * (c - b);
    }
  else
    {
      c = b;
      b = b - QQ * (b - a);
    }
  psib = (*psi) (b, nr);
  psic = (*psi) (c, nr);
  while (fabs (d - a) > EPSILON * (fabs (b) + fabs (c)))
    {
      if (psic < psib)
	{
	  MOVE3 (a, b, c, PP * b + QQ * d);
	  psib = psic;
	  psic = (*psi) (c, nr);
	}
      else
	{
	  MOVE3 (d, c, b, PP * c + QQ * a);
	  psic = psib;
	  psib = (*psi) (b, nr);
	}
    }
  if (psib < psic)
    {
      return b;
    }
  else
    {
      return c;
    }
}

double
psi (double lamda, nr_fmt * nr)
{
  static double *param;
  static boolean done = FALSE;

  double like;

  if (!done)
    {
      param = (double *) malloc (sizeof (double) * nr->numpop2);
      done = TRUE;
    }
  memcpy (param, nr->param, sizeof (double) * nr->numpop2);
  calc_loci_param (nr, nr->oparam, lamda, FALSE);
  like = calc_like (nr, nr->param, nr->tl, nr->numg, 0);
  memcpy (nr->param, param, sizeof (double) * nr->numpop2);
  return -like;
}


void
calc_hessian (double **hess, long n, double *delta, double *gamma)
{
  double **dd, *temp, t;
  long i, j, k;
  double dtg;
  temp = (double *) calloc (1, sizeof (double) * n);
  dd = (double **) calloc (1, sizeof (double *) * n);
  dd[0] = (double *) calloc (1, sizeof (double) * n * n);
  dtg = delta[0] * gamma[0];
  for (i = 1; i < n; i++)
    {
      dd[i] = dd[0] + n * i;
      dtg += delta[i] * gamma[i];
    }
  if (dtg != 0.0)
    dtg = 1. / dtg;
  else
    {
      reset_hess (hess, n);
      return;
    }
  for (i = 0; i < n; i++)
    {
      for (j = 0; j < n; j++)
	{
	  temp[i] += gamma[j] * hess[j][i];
	}
    }
  t = 0.0;
  for (i = 0; i < n; i++)
    t += temp[i] * gamma[i];
  t = (1.0 + t * dtg) * dtg;
  for (i = 0; i < n; i++)
    {
      for (j = 0; j < n; j++)
	{
	  for (k = 0; k < n; k++)
	    {
	      dd[i][j] += delta[i] * gamma[k] * hess[k][j];
	    }
	}
    }
  for (i = 0; i < n; i++)
    {
      temp[i] = 0.0;
      for (j = 0; j < n; j++)
	{
	  temp[i] += hess[i][j] * gamma[j];
	}
    }
  for (i = 0; i < n; i++)
    {
      for (j = 0; j < n; j++)
	{
	  dd[i][j] += temp[i] * delta[j];
	  dd[i][j] *= dtg;
	  hess[i][j] += delta[i] * delta[j] * t - dd[i][j];
	}
    }
  free (temp);
  free (dd[0]);
  free (dd);
}

double
absmaxvec (double *v, long n)
{
  long i;
  double max = fabs (v[0]);
  for (i = 1; i < n; i++)
    {
      if (max < fabs (v[i]))
	max = fabs (v[i]);
    }
  return max;
}

void
create_nr (nr_fmt * nr, world_fmt * world, long G, long profilenum)
{
  long i;
  nr->numpop = world->numpop;
  nr->numpop2 = world->numpop2;
  nr->skiploci = world->data->skiploci;
  nr->world = world;
  nr->profilenum = profilenum;
  if (world->options->gamma && world->locus >= world->loci)
    {
      nr->gammaI = GAMMA_INTERVALS;
      nr->gamma = (double *) malloc (nr->gammaI * sizeof (double));
      nr->partsize = world->numpop2 + 1;
    }
  else
    {
      if (!world->options->gamma && world->locus >= world->loci)
	{
	  nr->gammaI = 1;
	  nr->gamma = (double *) malloc (nr->gammaI * sizeof (double));
	  nr->partsize = world->numpop2;
	}
      else
	{
	  nr->gammaI = 3;	/* debug this would trap any looping */
	  nr->gamma = NULL;
	  nr->partsize = world->numpop2;
	}
    }
  nr->numg = G;
  nr->tl = world->atl[0].tl;
  nr->parts = (double *) calloc (1, nr->partsize * sizeof (double));
  nr->d = (double *) calloc (1, nr->partsize * sizeof (double));
  nr->od = (double *) calloc (1, nr->partsize * sizeof (double));
  nr->dv = (double *) calloc (1, nr->partsize * sizeof (double));
  nr->delta = (double *) malloc (nr->partsize * sizeof (double));
  nr->gdelta = (double *) calloc (1, nr->partsize * sizeof (double));
  nr->param = (double *) malloc (nr->partsize * sizeof (double));
  memcpy (nr->param, world->param0, nr->partsize * sizeof (double));
  nr->oparam = (double *) malloc (nr->partsize * sizeof (double));
  memcpy (nr->oparam, world->param0, nr->partsize * sizeof (double));
  nr->datalike = (double *) malloc (G * sizeof (double));
  nr->locilikes = (double *) malloc ((world->loci + 1) * sizeof (double));


  nr->apg0 = (double **) calloc (1,(world->loci + 1) * sizeof (double *));
  nr->apg0[0] = (double *) calloc (1,G * (world->loci + 1) * sizeof (double));
  for (i = 1; i < world->loci + 1; i++)
    {
      nr->apg0[i] = nr->apg0[0] + i * G;
    }
  nr->apg = (double *) calloc (1,G * sizeof (double));
}

void
reset_hess (double **hess, long n)
{
  long pop;
  memset (hess[0], 0, sizeof (double) * n * n);
  for (pop = 1; pop < n; pop++)
    {
      hess[pop] = hess[0] + pop * n;
      hess[pop][pop] = 1.0;
    }
  hess[0][0] = 1.0;
}

void
destroy_nr (nr_fmt * nr, world_fmt * world)
{
  free (nr->parts);
  free (nr->d);
  free (nr->od);
  free (nr->dv);
  free (nr->delta);
  free (nr->gdelta);
  free (nr->param);
  free (nr->oparam);
  free (nr->locilikes);
  free (nr->datalike);
  free (nr->apg0[0]);
  free (nr->apg0);
  free (nr->apg);

  if (nr->gamma != NULL)
    free (nr->gamma);
  free (nr);
}

double
calc_like (nr_fmt * nr, double *param, tarchive_fmt * atl, long G, long locus)
{
  static double *lparam;
  static boolean done = FALSE;

  int g;
  double gsum = 0;
  double *apg0;
  if (!done)
    {
      lparam = (double *) calloc (1, sizeof (double) * nr->numpop2);
      done=TRUE;
    }
  for (g = 0; g < nr->numpop2; g++)
    lparam[g] = log (param[g]);
  nr->PGC = 0.0;
  nr->apg_max = -DBL_MAX;
  apg0 = nr->apg0[locus];
  for (g = 0; g < G; g++)
    {
      nr->apg[g] = probG (param, lparam, &atl[g], nr) - apg0[g];
      if (nr->apg[g] > nr->apg_max)
	nr->apg_max = nr->apg[g];
    }
  for (g = 0; g < G; g++)
    {
      gsum += atl[g].copies;
      nr->apg[g] -= nr->apg_max;
      nr->PGC += atl[g].copies * exp (nr->apg[g]);
    }
  nr->llike = nr->apg_max + log (nr->PGC) - log (gsum);
  /*  free (lparam); */
  return nr->llike;
}

void
calc_param (nr_fmt * nr, double *param, double lamda)
{
  long i;
  double elem = nr->partsize;
  for (i = 0; i < elem; i++)
    {
      nr->param[i] = param[i] * exp ((MAX (-100, MIN (-lamda * nr->dv[i], 100))));
    }
  param_all_adjust (nr, FALSE);
}

void 
force_sametheta (nr_fmt * nr)
{
  long i;
  double sum = 0;
  for (i = 0; i < nr->numpop; i++)
    {
      sum += nr->param[i];
    }
  sum /= nr->numpop;
  for (i = 0; i < nr->numpop; i++)
    {
      nr->param[i] = sum;
    }
}

void 
force_samemigration (nr_fmt * nr)
{
  long i;
  double sum = 0;
  for (i = nr->numpop; i < nr->numpop2; i++)
    {
      sum += nr->param[i];
    }
  sum /= nr->numpop * (nr->numpop - 1);
  for (i = nr->numpop; i < nr->numpop2; i++)
    {
      nr->param[i] = sum;
    }
}

void
param_all_adjust (nr_fmt * nr, boolean boolgamma)
{
  const double minima[3] =
  {SMALLEST_THETA, SMALLEST_MIGRATION, 1. / BIGGEST_GAMMA};
  const double maxima[3] =
  {BIGGEST_THETA, BIGGEST_MIGRATION, 1. / SMALLEST_GAMMA};

  double ff, f = 1., denom;
  long i=0, ii, j, z=0;
  boolean overboard = FALSE;

  for (ii = 0; ii < nr->numpop2-nr->profilenum ; ii++)
    {
      i = (nr->profilenum > 0) ? nr->indeks[z++] : ii;
      if(i<nr->numpop)
	{
	  if (nr->param[i] < minima[0] || nr->param[i] > maxima[0])
	    {
	      overboard = TRUE;
	      break;
	    }
	}
      else
	{
	  if (nr->param[i] < minima[1] || nr->param[i] > maxima[1])
	    {
	      overboard = TRUE;
	      break;
	    }
	}
    }
  if (boolgamma)
    {
      if (nr->param[i] < minima[2] || nr->param[i] > maxima[2])
	{
	  overboard = TRUE;
	}
    }
  if (overboard)
    {
      for (i = 0; i < nr->partsize; i++)
	{
	  denom = nr->param[i] - nr->oparam[i];
	  if (fabs (denom) > 10e-10)
	    {
	      j = (i < nr->numpop ? 0 : (i < nr->numpop2 ? 1 : 2));
	      ff = MIN (1., fabs ((minima[j] - nr->oparam[i]) / denom));
	      ff = MIN (ff, fabs ((maxima[j] - nr->oparam[i]) / denom));
	    }
	  else
	    ff = 1.;
	  if (ff < f)
	    f = ff;
	}
      if (f < 1.)
	{
	  for (i = 0; i < nr->partsize; i++)
	    {
	      nr->param[i] = nr->oparam[i] + f * (nr->param[i] - nr->oparam[i]);
	      if (nr->param[i] <
		  MIN3 (SMALLEST_THETA, SMALLEST_MIGRATION, 1. / SMALLEST_GAMMA))
		{
		  if (nr->world->options->custm2[i] != '0')
		    nr->param[i] = (i < nr->numpop) ? SMALLEST_THETA :
		    ((i < nr->numpop2) ? SMALLEST_MIGRATION :
		     1. / SMALLEST_GAMMA);
		}
	    }
	}
    }
  switch (nr->world->options->migration_model)
    {
    case MATRIX:
      break;
    case MATRIX_ARBITRARY:
      check_matrix_arbitrary (nr);
      break;
    case MATRIX_SAMETHETA:
      force_sametheta (nr);
      break;
    case ISLAND:
      force_sametheta (nr);
      force_samemigration (nr);
      break;
    case ISLAND_VARTHETA:
      force_samemigration (nr);
      break;
    }
}

void 
check_matrix_arbitrary (nr_fmt * nr)
{

  static char *custm = NULL;
  static long zeron = 0;
  static long symn = 0;
  static long sym2n = 0;
  static long numpop = 0;
  static boolean done=FALSE;
  static twin_fmt * syms=NULL;
  static quad_fmt * sym2s=NULL;
  static long * zeros=NULL;
  static long z=0, i;
  static char *p;
  static double mm;

  if(!done) /* on system with working static this will execute once*/
    {
      done=TRUE;
      numpop = nr->numpop;
      custm = nr->world->options->custm2;
      zeron = nr->world->options->zeron;
      symn = nr->world->options->symn;
      sym2n = nr->world->options->sym2n;
      if(symn)
	syms = nr->world->options->symparam;
      if(sym2n)
	sym2s = nr->world->options->sym2param;
      if(zeron)
	zeros = nr->world->options->zeroparam;
      p = custm;
      if (*p == 'm')
	{
	  while (*p == 'm')
	    {
	      p++;
	      z++;
	    }
	}
    }

  if (z >= numpop)
    force_sametheta (nr);

  for (i = 0; i < zeron; i++)
    {
	nr->param[zeros[i]] = 0.;
    }

  for (i = 0; i < symn; i++)
    {
	mm = (nr->param[syms[i][0]] + nr->param[syms[i][1]])/2.;
	nr->param[syms[i][0]] = nr->param[syms[i][1]] = mm;
    }

  for (i = 0; i < sym2n; i++)
    {
	mm = (nr->param[sym2s[i][0]] * nr->param[sym2s[i][2]] +
              nr->param[sym2s[i][1]] * nr->param[sym2s[i][3]])/2.;
	nr->param[sym2s[i][0]] = mm / nr->param[sym2s[i][2]];
        nr->param[sym2s[i][1]] = mm / nr->param[sym2s[i][3]];
    }
}


void
gradient (double *d, nr_fmt * nr)
{
  long z;
  long g, i, ii, pop, offset, offset2;
  long nn = nr->partsize - nr->profilenum;
  double expapg, *thetas, *m;

  tarchive_fmt *tl = nr->tl;

  //  long model = nr->world->options->migration_model;
  memset (d, 0, sizeof (double) * nr->numpop2);
  thetas = nr->param;
  m = nr->param + nr->numpop;
  for (g = 0; g < nr->numg; g++)
    {
      if (nr->apg[g] > -100)
	{
	  for (pop = 0; pop < nr->numpop; pop++)
	    {
	      nr->parts[pop] = -tl[g].p[pop] / thetas[pop] + tl[g].kt[pop] / (thetas[pop] * thetas[pop]);
	      offset = nr->numpop + pop * (nr->numpop - 1);
	      offset2 = pop * nr->numpop;
	      z = 0;
	      for (i = offset; i < offset + nr->numpop - 1; i++)
		{
		  if (z == pop)
		    z++;
		  nr->parts[i] = (tl[g].l[offset2 + z] / nr->param[i]) - tl[g].km[pop];
		  z++;
		}
	    }
	  expapg = tl[g].copies * exp (nr->apg[g]);
	  z = 0;
	  for (i = 0; i < nn; i++)
	    {
	      ii = (nr->profilenum > 0) ? nr->indeks[z++] : i;
	      d[i] += expapg * nr->parts[ii];
	    }
	}
    }
  //  force_symmetric_d (model, nr, nn);
}

/* calculates the derivatives for the different
   migration models
 */
void 
force_symmetric_d (long model, nr_fmt * nr, long nn)
{
  switch (model)
    {
    case MATRIX:
      break;
    case MATRIX_ARBITRARY:
      check_symmetric_d (nr, nn);
      break;
    case MATRIX_SAMETHETA:
      calc_symmetric_d (nr, nn, 0, nr->numpop);
      break;
    case ISLAND:
      calc_symmetric_d (nr, nn, 0, nr->numpop);
      calc_symmetric_d (nr, nn, nr->numpop, nr->numpop2);
      break;
    case ISLAND_VARTHETA:
      calc_symmetric_d (nr, nn, nr->numpop, nr->numpop2);
      break;
    }
}

void 
calc_symmetric_d (nr_fmt * nr, long nn, long start, long stop)
{
  long i, ii;
  long z = 0;
  double dt = 0;
  for (i = 0; i < nn; i++)
    {
      ii = (nr->profilenum > 0) ? nr->indeks[z++] : i;
      if (ii >= start && ii < stop)
	dt += nr->d[i];
    }
  dt /= (stop - start);
  z = 0;
  for (i = 0; i < nn; i++)
    {
      ii = (nr->profilenum > 0) ? nr->indeks[z++] : i;
      if (ii >= start && ii < stop)
	nr->d[i] = dt;
    }
}

void 
check_symmetric_d (nr_fmt * nr, long nn)
{
  static char *custm = NULL;
  static long zeron = 0;
  static long symn = 0;
    static long sym2n = 0;
  static long mmn = 0;
  static long numpop = 0;
  static boolean done=FALSE;
  static twin_fmt * syms=NULL;
  static quad_fmt * sym2s=NULL;
  static long * zeros=NULL;
  static long * mms=NULL;
  static long z=0, i;
  static char *p;
  static double mm;
  double sum;
  double sign;
  
  if(!done)
    {
	done = TRUE;
      custm = nr->world->options->custm2;
      zeron = nr->world->options->zeron;
      symn = nr->world->options->symn;
      sym2n = nr->world->options->sym2n;
      mmn = nr->world->options->mmn;
      if(symn)
	syms = nr->world->options->symparam;
      if(sym2n)
	sym2s = nr->world->options->sym2param;
      if(zeron)
	zeros = nr->world->options->zeroparam;
      if(mmn)
        mms = nr->world->options->mmparam;
      p = custm;
      custm = nr->world->options->custm2;
      numpop = nr->numpop;
      p = custm;
      if (*p == 'm')
	{
	  while (*p == 'm')
	    {
	      p++;
	      z++;
	    }
	}
    }
/* if thetas = m then use only a mean nr->d for them*/
  if (z >= numpop)
    calc_symmetric_d (nr, nn, 0, nr->numpop);
/* do not zero anything, we are only recording the 
   non-zero nr->d  
  for (i = 0; i < zeron; i++)
    {
	nr->d[zeros[i]] = 0.;
    }
*/	
/* if there are symmetric migration rates M go here*/
  for(i = 0; i < symn; i++)
    {
      	mm = (nr->d[syms[i][0]] + nr->d[syms[i][1]])/2.;
	nr->d[syms[i][0]] = nr->d[syms[i][1]] = mm;
    }
/* if there are symmetric migration rates 4Nm do nothing but
   you need to do something when the parameters are changed, TEST*/
  for(i = 0; i < sym2n; i++)
    {
      	mm = (sqrt((nr->d[sym2s[i][0]]*nr->d[sym2s[i][0]]) +
                  (nr->d[sym2s[i][2]]*nr->d[sym2s[i][2]])) +
             sqrt((nr->d[sym2s[i][1]]*nr->d[sym2s[i][1]]) +
                  (nr->d[sym2s[i][3]]*nr->d[sym2s[i][3]])))/2.;
        mm *= mm;
        sign = nr->d[sym2s[i][0]] * nr->d[sym2s[i][2]]<0 ? -1 : 1;
	nr->d[sym2s[i][0]] = sqrt(mm - (nr->d[sym2s[i][2]]*nr->d[sym2s[i][2]]));
        sign = nr->d[sym2s[i][1]] * nr->d[sym2s[i][3]]<0 ? -1 : 1;
        nr->d[sym2s[i][1]] = sign * sqrt(mm - (nr->d[sym2s[i][3]]*nr->d[sym2s[i][3]]));
    }


/* if there is a mixture of zeros (0) and m than go here,
 e.g. stepping stone */
  if(mmn>0)
{
 sum=0;
 for(i=nr->numpop;i<nr->numpop+mmn;i++)
 {
 sum += nr->d[i];
 }
      sum /= mmn;
      for(i=nr->numpop;i<mmn+nr->numpop;i++)
	{
	    nr->d[i] = sum;
	}
    }
}


void
grad2loggrad (nr_fmt * nr, double *d, double PGC, long nn)
{
  long i, ii, z = 0;
  for (i = 0; i < nn; i++)
    {
      ii = (nr->profilenum > 0) ? nr->indeks[z++] : i;
      d[i] = -nr->param[ii] * d[i] / PGC;	/* to log derivatives */
      /* the division by PGC (the underflow protected uncorrected likelihood)
         is made here, because
         in multiple loci where I have to summ over loci before "taking logs"
         I use the derivatives of L with PGC=1 instead of Log(L) 
         where I use PGC = nr->PGC, the minus is for minimizing the function -L
         instead of maximizing L */
    }
}

double
probG (double *param, double *lparam, tarchive_fmt * tl, nr_fmt * nr)
{
  const long numpop=nr->numpop;
  const long numpopm=numpop-1;
  
  long i, j, z, offset, offset2, offsetup;
  double result = 0, sm;

  
  for (i = 0; i < numpop; i++)
    {
      if (lparam[i] <= -DBL_MAX)
	return DBL_MAX;
      result += tl->p[i] * (LOG2 - lparam[i]);
      offset = numpop + i * numpopm;
      offset2 = i * numpop;
      z = sm = 0.0;
      offsetup = offset + numpopm;
      for (j = offset; j < offsetup; j++,z++)
	{
	  if (z == i)
	    z++;
	  if (param[j] > 0.0)
	    {
	      result += tl->l[offset2 + z] * lparam[j];
	      sm += param[j];
	    }
	}
      result += -tl->km[i] * sm - tl->kt[i] / param[i];
    }
  return result;
}


boolean
is_singular (double **dd, long n)
{
  long i, j;
  double temp;
  boolean singular = FALSE;
  for (i = 0; i < n; i++)
    {
      temp = 0.0;
      for (j = 0; j < n; j++)
	{
	  temp += dd[i][j];
	}
      if (temp == 0.0)
	{
	  singular = TRUE;
	  break;
	}
    }
  for (i = 0; i < n; i++)
    {
      temp = 0.0;
      for (j = 0; j < n; j++)
	{
	  temp += dd[i][j];
	}
      if (temp == 0.0)
	{
	  singular = TRUE;
	  break;
	}
    }
  return singular;
}


void
calc_cov (double **dd, double *d, double *param, long n)
{
  long i, j;
  if (!is_singular (dd, n))
    invert_matrix (dd, n);
  else
    {
      reset_hess (dd, n);
      return;
    }
  for (i = 0; i < n; i++)
    {
      for (j = 0; j < i; j++)
	{
	  dd[i][j] /= (param[i] * param[j]);
	  dd[j][i] = dd[i][j];
	}
      dd[i][i] = (dd[i][i] - param[i] * d[i]) / (param[i] * param[i]);
    }
  if (!is_singular (dd, n))
    invert_matrix (dd, n);
  else
    reset_hess (dd, n);
}


void
print_contribution (nr_fmt * nr, tarchive_fmt * tyme, long G)
{
  long g,i,j;
  double temp=0,temp2=0,maxtemp;
  FILE *mixfile=NULL;
  long events=0;
  long contribution[11];
  for (g = 0; g < 11; g++)
    contribution[g] = 0;
  if(nr->world->options->mixplot)
    {
      mixfile = fopen("mixfile","a");
      fprintf(mixfile,"\n\n");
    }
  maxtemp = -DBL_MAX;
  for (g = 0; g < G; g++)
    {
      temp = nr->apg[g] + nr->apg0[nr->world->locus][g];
      temp2 = temp + nr->world->likelihood[g];
      if(temp2>maxtemp)
	maxtemp = temp2;
      if(nr->world->options->mixplot)
	{
	  events = 0;
	  for(i=0;i<nr->world->numpop;i++)
	    {
	      for(j=0;j<nr->world->numpop;j++)
		events += tyme[g].l[i*nr->world->numpop+j];
	    }
	  for(i=0;i<tyme[g].copies;i++)
	    fprintf(mixfile,"%li %f %f ",events, temp,nr->world->likelihood[g]);
	}
      temp2 -= maxtemp;
      if (temp2 > -20)
	    {
	  contribution[9 - (long) (fabs (temp2) / 2)] += tyme[g].copies;
	}
      contribution[10] += tyme[g].copies;
    }
  fprintf (stdout, "           log(P(g|Param) * P(data|g)\n");
  fprintf (stdout, "                            -20 to ");
  for (g = -18; g <= 0; g += 2)
    {
      fprintf (stdout, "%4li ", g);
    }
  fprintf (stdout, "  All\n");
  fprintf (stdout, "           Counts                  ");
  for (g = 0; g < 10; g++)
    {
      fprintf (stdout, "%4li ", contribution[g]);
    }
  fprintf (stdout, "%5li\n", contribution[10]);
  if(nr->world->options->mixplot)
    fclose(mixfile);
}

/* calculates log(parameters)*/
void 		  
log_param0 (double *param, double *lparam,long nn)
{
  long i;
  for(i=0;i<nn;i++)
    {
      if(param[i]>0)
	lparam[i] = log(param[i]);
      else
	lparam[i] = -DBL_MAX;
    }
}


void
create_apg0 (double *apg0, nr_fmt * nr, timearchive_fmt * tyme)
{
  long g;
  /* Prob(G|Param0) */
  for (g = 0; g < tyme->T; g++)
    {
      apg0[g] = probG (tyme->param0, tyme->lparam0, &tyme->tl[g], nr);
    }

}

double
logprob_noevent (world_fmt * world, long interval)
{
  long pop, k;
  double result = 0.0;
  for (pop = 0; pop < world->numpop; pop++)
    {
      k = world->treetimes[0].tl[interval].lineages[pop];
      result += (k * (k - 1) / world->param0[pop]) + sum_migprob (world, pop, interval);
    }
  return result;
}

double
sum_migprob (world_fmt * world, long pop, long interval)
{
  long i;
  double result = 0.0;
  long *lineages = world->treetimes[0].tl[interval].lineages;
  long offset = world->numpop + (world->numpop - 1) * pop;
  for (i = offset; i < offset + world->numpop - 1; i++)
    {
      result += world->param0[i];
    }
  return result * lineages[pop];
}













