/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 M A X I M I Z E R      R O U T I N E S

 -calculates single locus ML
 - mulitlocus ML with constant mutation rate
 - multilocus ML with gamma deviated mutation rate 

  using Broyden-Fletcher-Goldfarb-Shanno method
  using the cholesky decomposition method

 Peter Beerli 1996-1998, Seattle
 beerli@genetics.washington.edu
 $Id: combroyden2.c,v 1.14 1999/05/14 00:06:21 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"
#include "broyden.h"
#include "world.h"
#include "integrate.h"
#include "derivatives.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

/* prototypes ------------------------------------------- */
/* public function used in main.c */

long estimateParameter (timearchive_fmt * tyme, long G,
       world_fmt * world, double **dd, long chain, char type, 
		   char ***plane,long kind);


/* calculates likelihood */
double calc_loci_like (nr_fmt * nr, timearchive_fmt * atl,
		       long loci, boolean gamma);
/* first derivatives for ML with constant mutation rate */
void simple_loci_derivatives (nr_fmt * nr,
			      timearchive_fmt * tyme,
			      double *param, long loci);
/* progress-reporter */
void print_menu_finalestimate (boolean progress, long locus, long loci);
/* maximizer routine */
long broyden_driver (timearchive_fmt * tyme, long loci,
		     world_fmt * world, double **covariance, char ***plane,
		     long *profiles, double *value, long profilenum,
		     double *profparam, double *proflike, double *profnorm,
		     long kind);
/* start parameter for multilocus estimator, creates simple average */
void calc_meanparam (world_fmt * world, long n);
/* multilocus first derivatives with or without gamma distrib. mut. rate */
void combine_gradient (nr_fmt * nr,
		       timearchive_fmt * tyme,
		       long loci, boolean boolgamma);
/* this helps to use the simple single locus material */
void copy_and_clear_d (nr_fmt * nr, double *d);
void add_back_d (nr_fmt * nr, double *d);
/* rescales p35narameter with theta1 for gamma calculation */
void set_gamma_param (double *paramn, double *paramo, double theta, nr_fmt * nr);
/* calculates norm, this creates a stopping criteria */
double norm (double *d, long size);
/* used in multilocus  line search */
double psilo (double lamda, nr_fmt * nr);
/* used in integral for gamma distr. mutation rate */
double phi (double t, void *b);

/* calculates new multilocus parameter values */
void calc_loci_param (nr_fmt * nr, double *param, double lamda,boolean boolgamma);
/* used to reset the BFSG stuff */
void reset_work (double **work, long nnn);

/* linesearch of BFGS
   and helper functions for this linesearch */
double line_search (double *dv, double *gxv, double (*func) (double lamda, nr_fmt * nr),
		    nr_fmt * nr, double oldL);

void replace_with (int mode, double *a, double *b, double *c, double m,
		   double *la, double *lb, double *lc, double ll);
double psil (double (*func) (double lamda, nr_fmt * nr),
	     double a, double b, nr_fmt * nr, double *gxv, double *dv, double oldL, double *val1, double *val2);
double quadratic_lamda (double lamda,
			double a, double b, double c,
			double la, double lb, double lc);


void print_reset_big_param (FILE *file, double average, double maxtheta, long pop);



/* public functions---------------------------------------------- */
/* driver for maximizing single locus, multilocus functions      */
long
estimateParameter (timearchive_fmt * tyme, long G,
       world_fmt * world, double **dd, long chain, char type, 
		   char ***plane,long kind)
{
  long z=0,i=0, loci=0;
  char *p;
  long *profiles;
  double *values;
  long trials=0;
  double likes=0.,normd=0.;
  if(kind==MULTILOCUS)
    {
      loci = world->loci;
      print_menu_finalestimate (world->options->progress, 
				world->loci, world->loci);
    }
  if(strchr(world->options->custm,'0') || strchr(world->options->custm,'c'))
    {
      profiles = (long *) calloc(1,sizeof(long) * world->numpop2);
      values = (double *) calloc(1,sizeof(double) * world->numpop2);
      p = world->options->custm2;
      while(*p!='\0')
	{
	  if(*p=='0' || *p=='c')
	    {
	      profiles[z] = i;
	      z++;
	    }
	  p++;i++;
	}
      trials=broyden_driver (world->atl, loci,
			     world, world->cov[loci],
			     plane, profiles,
			     values, z,
			     world->param0, &likes, &normd,kind);
      free(profiles);
      free(values);
    }
  else
    trials=broyden_driver (world->atl, loci,
			   world, world->cov[loci],
			   plane, NULL,
			   NULL, 0,
			   NULL, NULL, &normd, kind);
  
  if (kind==SINGLELOCUS && world->options->progress)
    {
      print_menu_chain (type, chain, world->atl[0].T, world);
      fprintf (stdout, "           Maximization cycles needed: %li\n",trials);
      fprintf (stdout, "           Norm of first derivatives:  %f\n",normd); 
    }
  if (kind==MULTILOCUS && world->options->plotnow && world->loci>1)
    {
      create_loci_plot (world, world->plane[world->loci], world->atl,
			world->loci);
    }
  return trials;
}


/*---------------------------------------------------------------
  Maximizer using Broyden-Fletcher-Shanno-Goldfarb method
 
  TYME  is pointer to the timeliststructure (which is actually WORLD->ATL)
  LOCI  number of loci or if =0 this is the single locus estimate
  WORLD contains all necessary structures
  PLANE returns the contour plots
  PROFILES specifies which parameters to profile (PARAM[PROFILE[i]])
  VALUE the values of the profiled parameters
  PROFILENUM number of parameters to profile
  PROFPARAM returns the parameters at the ML of param[profile]
  PROFLIKE the log(likelihood) at the profile maximum
  PROFNORM the norm of the first derivatives of the profile-maximizer
 */
long
broyden_driver (timearchive_fmt * tyme, long loci,
		world_fmt* world, double **covariance, char ***plane,
		long *profiles, double *value, long profilenum,
		double *profparam, double *proflike, double *profnorm,
		long kind)
{
  boolean multilocus = FALSE;
  boolean boolgamma= (loci == 0) ? FALSE : world->options->gamma;

  long i, ii, z, locus, trials, pop;
  long Gmax = 1;
  nr_fmt *nr;
  long nn, nnn;
  double oldL, newL, lamda, normd, normd20 = 1e20, nsum, ssum;
  double *xv, *xv0, **hess, *delta, *dv, *gxv, *gxv0, *gamma;

  long *indeks;
  /*  double **work, **lower, *diagonal;*/
  nr = (nr_fmt *) calloc (1, sizeof (nr_fmt) * 1);

  if (loci == 0)
    {
      multilocus = FALSE;
      Gmax = tyme[loci].T;	/* if loci=0 -> single locus ML */
    }
  else
    {
      multilocus = TRUE;
      for (locus = 1; locus < loci + 1; locus++)
	{
	  if (Gmax < tyme[locus].T)
	    Gmax = tyme[locus].T;
	}
      world->locus = loci;
    }


  /* calculcate multilocus mean parameter */
  if (multilocus)
    {
      nnn = nn = boolgamma ? world->numpop2 + 1 : world->numpop2;
      calc_meanparam (world, world->numpop2);
      if (boolgamma)
	{
	  world->param0 = (double *) 
	    realloc(world->param0,sizeof(double)*nnn);
	  world->param0[world->numpop2] = 1. / START_ALPHA;
	}
    }
  else
    nnn = nn = world->numpop2;
  /* profiler setup-------------------------- 
     needs profiles setup (profile.c)
   */
  if (profilenum > 0)
    {
      nnn -= profilenum;
      for (i = 0; i < profilenum; i++)
	world->param0[profiles[i]] = value[i];
    }

  create_nr (nr, world, Gmax, profilenum);
  nr->profiles = profiles;

  indeks = (long *) calloc (1, sizeof (long) * nnn);
  nr->indeks = indeks;
  if (multilocus)
    {
      for (locus = 1; locus < loci + 1; locus++)
	{
	  create_apg0 (nr->apg0[locus], nr, &world->atl[locus]);
	}
    }
  else
    create_apg0 (nr->apg0[0], nr, &world->atl[0]);

  /* BEGIN allocation of BFGS stuff */
  /* ***********disabled section + no compile of cholesky.c
     there is in error in the more sophisticated
     cholesky decomposition of the A matrix
     I replaced this with the old code
     using  Inv(A) and this seems to work fine even
     many population 220998PB*/
  /* diagonal = (double *) calloc (1, sizeof (double) * nnn);
  lower = (double **) calloc (1, sizeof (double *) * nnn);
  lower[0] = (double *) calloc (1,
				sizeof (double) * nnn * nnn);
  work = (double **) calloc (1, sizeof (double *) * (nnn + 5));
  work[0] = (double *) calloc (1,
			       sizeof (double) * (nnn + 5) * (nnn + 1));
  diagonal[0] = 1.0;
  for (i = 1; i < nnn; i++)
    {
      lower[i] = lower[0] + i * (nnn);
      work[i] = work[0] + i * (nnn + 1);
      diagonal[i] = 1.0;
    }
  for (i = nnn; i < nnn + 5; i++)
    {
      work[i] = work[0] + i * (nnn + 1);
      }*/
  hess = nr->dd = covariance;
  reset_hess (hess, nnn);

  delta = nr->delta;
  gamma = nr->gdelta;
  dv = nr->dv;
  gxv = nr->d;
  gxv0 = nr->od;
  xv0 = nr->oparam;
  xv = nr->param;
  /*END BFGS allocation stuff */

  /* profiling: filling of index array */
  if ((profilenum > 0) && profilenum < nr->partsize)
    {
      for (i = 0, z = 0, ii = 0; i < nr->partsize; i++)
	{
	  if (i != profiles[z])
	      indeks[ii++] = i;
	  else
	      z++;
	}
    }

  /* maximization starts here */
  oldL = calc_loci_like (nr, tyme, loci, boolgamma);
  combine_gradient (nr, tyme, loci, boolgamma);
  memcpy (gxv0, gxv, sizeof (double) * nnn);
  memcpy (xv0, xv, sizeof (double) * nn);
  memcpy (dv, gxv, sizeof (double) * nnn);
  memset (delta, 0, sizeof (double) * nnn);
  for (trials = 0; trials < NTRIALS; trials++)
    {
#ifdef MAC
      eventloop ();
#endif
      newL = -DBL_MAX;
      lamda = calc_line (nr, -1., 0.1, 1., loci>0 ? psilo : psi);
      /*lamda = line_search (dv, gxv, psilo, nr, oldL); */
      calc_loci_param (nr, nr->oparam, lamda, boolgamma);
      newL = nr->llike;
      while (newL < oldL && fabs (lamda) > 10. * DBL_EPSILON)
	{
	  calc_loci_param (nr, nr->oparam, lamda, boolgamma);
	  newL = calc_loci_like (nr, tyme, loci, boolgamma);
	  lamda /= -2.;
	}
      normd = norm (gxv, nnn);
      /*if (world->options->verbose)
	{
	if(trials % 10 == 0)
	fprintf(stdout,"\n<%3li>",trials);
	fprintf (stdout, " %f", normd);
	}*/
      if ((trials + 1) % 20 == 0)
	{
	  if (fabs (normd - normd20) < EPSILON)
	    break;
	  normd20 = normd;
	}

      if (normd < LOCI_NORM * nnn)
	break;
      else if (oldL >= newL)
	lamda = 0;
      else
	oldL = newL;
/*-------------------------------------------*/
      if (/*normd>normd20 ||*/ fabs (lamda) <= DBL_EPSILON * 10.)
	{
	  /*	  normd20=normd;*/
	  reset_hess (hess, nnn);
	  /*reset_work (work, nnn);*/
	  for (i = 0; i < nnn; i++)
	    {
	      /*diagonal[i] = 1.;*/
	      dv[i] = gxv[i];
	    }
	  /* if (world->options->verbose)
	     printf ("Approx sec. derivative matrix reset\n");*/
	  continue;
	}
      memcpy (gxv0, gxv, sizeof (double) * nnn);
      combine_gradient (nr, tyme, loci, boolgamma);
      if ((profilenum > 0) && profilenum < nr->partsize)
	{
	  for (i = 0; i < nnn; i++)
	    {
	      ii = indeks[i];
	      delta[i] = log (xv[ii]) - log (xv0[ii]);
	      gamma[i] = gxv[i] - gxv0[i];
	    }
	}
      else
	{
	  for (i = 0; i < nnn; i++)
	    {
	      delta[i] = log (xv[i]) - log (xv0[i]);
	      gamma[i] = gxv[i] - gxv0[i];
	    }
	}

      calc_hessian (hess, nnn, delta, gamma);
      calc_dv (dv, hess, gxv, nnn);
      /*
      update_B (lower, diagonal, lamda, dv, gamma, delta, nnn, work);
      calc_direction (lower, diagonal, dv, gxv, hess, nnn);*/
      memcpy (xv0, xv, sizeof (double) * nn);
    }
  memcpy (world->param0, nr->param, sizeof (double) * nn);
  for(pop=0;pop<world->numpop;pop++)
    {
      if(world->param0[pop]>=BIGGEST_THETA)
	{
	  ssum=0; nsum=0;
	  for(i=0;i<world->numpop;i++)
	    {
	      if(world->param0[i]<BIGGEST_THETA)
		{
		  ssum += world->param0[i];
		  nsum++;
		}
	    }
	  world->param0[pop] = ssum/nsum;
	  print_reset_big_param(world->outfile,world->param0[pop],BIGGEST_THETA,pop);
	  print_reset_big_param(stdout,world->param0[pop],BIGGEST_THETA,pop);
	}
    }
  switch(kind)
    {
    case MULTILOCUS:
      tyme[loci + 1].param = (double *) realloc (tyme[loci + 1].param,
						 sizeof (double) * nn);
      tyme[loci + 1].param_like = nr->llike;
      tyme[loci + 1].normd = normd;
      tyme[loci + 1].trials = trials;
      memcpy (tyme[loci + 1].param, world->param0, 
	      sizeof (double) * nn);
      *profnorm = normd;
      break;
    case SINGLELOCUS:
      world->param_like = newL;
       if (world->param_like < world->options->lcepsilon &&
	   world->options->plotnow && !world->options->simulation)
	 {
	   create_locus_plot (world, plane, nr->tl, nr, nr->numg);      
	   //	   calc_cov (hess, gxv, xv, nn);
	 }
      *profnorm = normd;
       break;
    case PROFILE:
      memmove (profparam, world->param0, sizeof (double) * nr->partsize);
      *proflike = nr->llike;
      *profnorm = normd;
    }
  /*free(diagonal);
    free (work[0]);
    free (work);
    free(lower[0]);
    free(lower);
  */
  free (indeks);

  if(world->options->verbose)
    {
      if(kind==SINGLELOCUS)
	print_contribution (nr, nr->tl, nr->numg);
    }
  destroy_nr (nr, world);
  return trials;
}

/* calculates the likelihood over all loci for the new parameter set */
double
calc_loci_like (nr_fmt * nr, timearchive_fmt * atl, long loci,
		boolean boolgamma)
{
  double a = 0.0, b = 9999., d = 20.0, p = 0., eps = 10e-15, result;
  long m = 10000, inf = 2;
  helper_fmt helper;
  double denom = 0.0;
  double theta1;
  long locus;
  //  double *temp, 
  //double *param0;
  double /*ll, theta, */ invalpha, alpha, logres = 0;
  //param0 = (double *) malloc (sizeof (double) * nr->partsize);
  //memcpy (param0, nr->param, sizeof (double) * nr->partsize);
  if(loci!=0)
    {
      if (!boolgamma)
	{
	  /* mutation rate is constant log(L(loci)) = Sum[log(L(locus))) */
	  for (locus = 1; locus < loci + 1; locus++)
	    {
	      if (nr->skiploci[locus-1])
		continue;
	      logres += calc_like (nr, nr->param, atl[locus].tl, 
				   atl[locus].T, locus);
	    }
	}
      else
	{
	  /* mutation rate is Gamma deviated 
	     log(L(loci) = Sum[Integral[gammafunc[] exp[log[L(locus}]],
	                      0,INF]
	  */
	  invalpha = nr->param[nr->numpop2];
	  alpha = 1. / invalpha;
	  theta1 = nr->param[0];
	  denom = lgamma (alpha) + log (theta1 / alpha) * alpha;
	  for (locus = 1; locus < loci + 1; locus++)
	    {
	      if (nr->skiploci[locus-1])
		continue;
	      helper.nr = nr;
	      helper.atl = atl;
	      helper.locus = locus;
	      helper.weight = -denom;
	      helper.ll = -DBL_MAX;
	      a = 0.0;		//SMALLEST_THETA; 	      
	      b = 9999.;
	      d = 20.0;
	      p = 0.;
	      eps = 10e-10;
	      m = 10000;
	      inf = 2;
	      result=0.0;
	      inthp (&a, &b, &d, phi, &m, &p, &eps, &inf, 
		     (void *) (&helper), &result);
	      if (result > 0.0)
		{
		  nr->locilikes[locus] = log (result);
		  logres += nr->locilikes[locus];
		}
	      else /*underflow protection: more work, but often works*/
		{
		  helper.weight = -denom - helper.ll;
		  helper.ll = -DBL_MAX;
		  inthp (&a, &b, &d, phi, &m, &p, &eps, &inf, 
			 (void *) (&helper), &result);
		  if (result > 0)
		    {
		      nr->locilikes[locus] = log (result) - helper.weight 
			- denom;
		      logres += nr->locilikes[locus];
		    }
		  else
		    {
		      nr->locilikes[locus] = -DBL_MAX;
		      logres += -DBL_MAX;
		    }
		}
	    }
	}
    }
  else /*->singlelocus*/
    logres += calc_like (nr, nr->param, atl[0].tl, 
			 atl[0].T, 0);

  //  memcpy (nr->param, param0, sizeof (double) * nr->partsize);
  //free (param0);
  nr->llike = logres;
  return logres;
}

double
phi (double t, void *b)
{
  double ll, alpha, beta;
  long locus;
  double weight;
  helper_fmt *helper;
  nr_fmt nr;
  double *param;
  timearchive_fmt *atl;
  helper = (helper_fmt *) b;
  locus = (long) helper->locus;
  weight = (double) helper->weight;
  atl = (timearchive_fmt *) helper->atl;
  nr = *(nr_fmt *) helper->nr;
  param = (double *) calloc (1, sizeof (double) * nr.partsize);
  alpha = 1. / nr.param[nr.numpop2];
  beta = nr.param[0] / alpha;
  set_gamma_param (param, nr.param, t, &nr);
  /* likelihood calculation */
  ll = calc_like (&nr, param, atl[locus].tl, atl[locus].T, locus);
  if (ll > helper->ll)
    helper->ll = ll;
  ll = exp (-t / beta +
	    (alpha - 1.) * log (t) + ll + weight);
  free (param);
  return ll;
}


/* private functions---------------------------------------------- */
/* derivatives */
void
combine_gradient (nr_fmt * nr,
		  timearchive_fmt * tyme,
		  long loci, boolean boolgamma)
{
  long locus;
  long nnn = nr->partsize - nr->profilenum;
  memset (nr->d, 0, sizeof (double) * nnn);
  if(loci==0)
    {
      simple_loci_derivatives (nr, tyme, nr->param, 0);
      force_symmetric_d (nr->world->options->migration_model, nr, nnn);
      grad2loggrad (nr, nr->d, 1., nnn);
    }
  else
    {
      if (!boolgamma)
	{
	  for (locus = 1; locus < loci + 1; locus++)
	    {
	      if (nr->skiploci[locus-1])
		  continue;
	      simple_loci_derivatives (nr, tyme, nr->param, locus);
	    }
	  force_symmetric_d (nr->world->options->migration_model, nr, nnn);
	  grad2loggrad (nr, nr->d, 1., nnn);
	}
      else
	{
	  dt (nr, tyme, loci);
	  force_symmetric_d (nr->world->options->migration_model, nr, nnn);
	  nr->param[nr->numpop2] = 1. / nr->param[nr->numpop2];
	  grad2loggrad (nr, nr->d, 1., nnn);
	  nr->param[nr->numpop2] = 1. / nr->param[nr->numpop2];
	}
    }
}

void
simple_loci_derivatives (nr_fmt * nr,
			 timearchive_fmt * tyme, double *param, long locus)
{
  static double *lparam;
  static double *d;
  static boolean done=FALSE;
  
  long g;
  double *apg0;
  if(!done)
    {
      d = (double *) calloc (1, sizeof (double) * nr->numpop2);
      lparam = (double *) calloc (1, sizeof (double) * nr->numpop2);
      done = TRUE;
    }
  for (g = 0; g < nr->numpop2; g++)
    lparam[g] = log (nr->param[g]);
  copy_and_clear_d (nr, d);
  nr->PGC = 0;
  nr->apg_max = -DBL_MAX;
  apg0 = nr->apg0[locus];
  for (g = 0; g < tyme[locus].T; g++)
    {
      nr->apg[g] = probG (nr->param, lparam, &tyme[locus].tl[g], nr) - apg0[g];
      if (nr->apg[g] > nr->apg_max)
	nr->apg_max = nr->apg[g];
    }
  for (g = 0; g < tyme[locus].T; g++)
    {
      nr->apg[g] -= nr->apg_max;
      nr->PGC += tyme[locus].tl[g].copies * exp (nr->apg[g]);
    }
  nr->tl = tyme[locus].tl;
  nr->numg = tyme[locus].T;
  gradient (nr->d, nr);
  for (g = 0; g < nr->partsize - nr->profilenum; g++)
    nr->d[g] /= nr->PGC;
  add_back_d (nr, d);
}




void
set_gamma_param (double *paramn, double *paramo, double theta, nr_fmt * nr)
{
  long i;
  paramn[0] = theta;
  for (i = 1; i < nr->numpop; i++)
    {
      paramn[i] = paramo[i] * theta / paramo[0];
    }
  for (i = nr->numpop; i < nr->numpop2; i++)
    {
      paramn[i] = paramo[i] * paramo[0] / theta;
    }
  paramn[i] = paramo[i];
}


void
copy_and_clear_d (nr_fmt * nr, double *d)
{
  memcpy (d, nr->d, sizeof (double) * nr->numpop2);
  memset (nr->d, 0, sizeof (double) * nr->numpop2);
}

void
add_back_d (nr_fmt * nr, double *d)
{
  long pop;
  for (pop = 0; pop < nr->numpop2; pop++)
    {
      nr->d[pop] += d[pop];
    }
}


void
print_menu_finalestimate (boolean progress, long locus, long loci)
{
  static char nowstr[STRSIZE];
  if (progress)
    {
      get_time (nowstr, "%H:%M:%S");
      if (locus < loci)
	fprintf (stdout, "%s   Parameter estimation for locus %li\n",
		 nowstr, locus);
      else
	fprintf (stdout, "%s   Parameter estimation over all loci\n",
		 nowstr);
    }
}

void
print_reset_big_param (FILE *file, double average, double maxtheta, long pop)
{
  static char nowstr[STRSIZE];
      get_time (nowstr, "%H:%M:%S");

      fprintf (file, "%s   Theta for population %li was reset to the average\n",
	       nowstr, pop);
      fprintf (file, "          of %f, it exceed the maximum of %f \n",average,maxtheta);
}

void
calc_meanparam (world_fmt * world, long n)
{
  long pop, locus, loci = world->loci;


  memset (world->param0, 0, sizeof (double) * n);
  for (locus = 1; locus < loci + 1; locus++)
    {
      for (pop = 0; pop < n; pop++)
	{
	  if (!world->data->skiploci[locus-1])
	    world->param0[pop] += world->atl[locus].param[pop];
	}
    }
  for (pop = 0; pop < n; pop++)
    {
      world->param0[pop] /= (loci - world->skipped);
    }
}


double
norm (double *d, long size)
{
  int i;
  double sum = 0.;
  for (i = 0; i < (int) size; i++)
    {
      sum += d[i] * d[i];
    }
  return sqrt (sum);
}


double
psilo (double lamda, nr_fmt * nr)
{
  static double *param;
  static boolean done = FALSE;

  double like;

  if (!done)
    {
      param = (double *) malloc (sizeof (double) * nr->partsize);
      done = TRUE;
    }
  memcpy (param, nr->param, sizeof (double) * nr->partsize);
  calc_loci_param (nr, nr->oparam, lamda, nr->world->options->gamma);
  like = calc_loci_like (nr, nr->world->atl, nr->world->loci,
			 nr->world->options->gamma);
  memcpy (nr->param, param, sizeof (double) * nr->partsize);
  //  fprintf(stdout,".");fflush(stdout);
  return -like;
}

void
calc_loci_param (nr_fmt * nr, double *param, double lamda, boolean boolgamma)
{
  long i, ii, z = 0;
  double alpha;
  double elem = (boolgamma ? nr->numpop2 : nr->partsize);	/*see below why */

  for (i = 0, ii = 0; i < elem - nr->profilenum; i++)
    {
      ii = (nr->profilenum > 0) ? nr->indeks[z++] : i;
      nr->param[ii] = param[ii] * exp ((MAX (-100, MIN (-lamda * nr->dv[i], 100))));
    }
  if (boolgamma)
    {
      ii = (nr->profilenum > 0) ? nr->indeks[z] : i;
      alpha = 1. / param[nr->numpop2];
      alpha = alpha * exp ((MAX (-10, MIN (-lamda * nr->dv[i], 10))));
      nr->param[nr->numpop2] = 1. / alpha;
    }
  param_all_adjust (nr, boolgamma);
}


void 
reset_work (double **work, long nnn)
{
  long i, j;
  for (i = nnn; i < nnn + 5; i++)
    {
      for (j = 0; j < nnn + 1; j++)
	work[i][j] = 0.0;
    }
}



#define LAMBDA_ 1
#define M_      0

/* line_search returns LAMBDA (how far to jump in the search direction)
   needed in the the multilocus broyden-fletcher-shanno maximizer

   DV    search direction
   GXV   first derivatives of function to minimize
   FUNC  the function to minimize
   NR    all additional variables needed to calculate FUNC
   OLDL  -log(likelihood) of FUNC with LAMBDA=0 (previously calculated)

   needs functions: psil, replace_with, quadratic_lamda
 */
double
line_search (double *dv, double *gxv, double (*func) (double lamda, nr_fmt * nr), nr_fmt * nr, double oldL)
{
  long trials = 0;
  double lamda_, ql;
  double a, b, c;

  double ll, la, lb, lc;
  double m = 1.0;
  a = 0.0;
  b = 0.000001;
  c = 0.9;
  la = -oldL;
  lb = (*func) (b, nr);
  lc = (*func) (c, nr);
  while (trials++ < NTRIALS)
    {
      /*      printf ("*%3li> %f %f %f / %f %f %f\n", trials, a, b, c, la, lb, lc); */
      lamda_ = (c * c * (lb - la) + b * b * (la - lc) + a * a * (lc - lb)) / (2. * (b * la - c * la - a * lb + c * lb + a * lc - b * lc));
      if ((lamda_ <= 0.0) || (lamda_ >= m))
	{
	  if ((a == m || b == m || c == m))
	    return m;
	  else
	    {
	      ll = (*func) (m, nr);
//      replace_with(M_,&a,&b,&c,m,&la,&lb,&lc,ll);
	      replace_with (LAMBDA_, &a, &b, &c, m, &la, &lb, &lc, ll);
	      continue;
	    }
	}
      ll = (*func) (lamda_, nr);
      ql = quadratic_lamda (lamda_, a, b, c, la, lb, lc);
      if ((fabs (ll - MIN3 (la, lb, lc)) <= BIGEPSILON) || (fabs (ll - ql) <= BIGEPSILON))
	return lamda_;
      else
	{
	  /*  if(((a<b<c) || (c<b<a)) && (lb < MIN(la,lc)))
	     return lamda_;
	     if(((b<a<c) || (c<a<b)) && (la < MIN(lb,lc)))
	     return lamda_;
	     if(((a<c<b) || (b<c<a)) && (lc < MIN(la,lb)))
	     return lamda_; */
	  replace_with (LAMBDA_, &a, &b, &c, lamda_, &la, &lb, &lc, ll);
	  m = MAX3 (a, b, c);
	}
    }
  return lamda_;
}

void
replace_with (int mode, double *a, double *b, double *c, double m,
	      double *la, double *lb, double *lc, double ll)
{
  double ma, mb, mc;
  if (mode == LAMBDA_)
    {
      ma = *la;
      mb = *lb;
      mc = *lc;
    }
  else
    {
      ma = ll - *la;
      mb = ll - *lb;
      mc = ll - *lc;
    }
  if (ma > mb)
    {
      if (ma > mc)
	{
	  *a = m;
	  *la = ll;
	}
      else
	{
	  *c = m;
	  *lc = ll;
	}
    }
  else
    {
      if (mb > mc)
	{
	  *b = m;
	  *lb = ll;
	}
      else
	{
	  *c = m;
	  *lc = ll;
	}
    }
}

double
psil (double (*func) (double lamda, nr_fmt * nr),
      double a, double b, nr_fmt * nr, double *gxv, double *dv, double oldL, double *val1, double *val2)
{
  long i;
  long nn = nr->partsize;
  double val = 0.0;

  if (a == 0.0 && b == 0)
    {
      for (i = 0; i < nn; i++)
	{
	  val += dv[i] /* gxv[i] */ ;
	}
      *val1 = -oldL;
      *val2 = -oldL;
      return val;
    }
  else
    {
      if (a == 0)
	val = (((*val2) = (*func) (b, nr)) - ((*val1) = -oldL)) / b;
      else
	{
	  if (b == 0)
	    val = (((*val2) = -oldL) - ((*val1) = (*func) (a, nr))) / (-a);
	  else
	    val = (((*val2) = (*func) (b, nr)) - ((*val1) = (*func) (a, nr))) / (b - a);
	}
      return val;
    }
}

double
quadratic_lamda (double lamda,
		 double a, double b, double c,
		 double la, double lb, double lc)
{
  double alpha, beta, gamma;
  double aa, bb, cc;
  aa = a * a;
  bb = b * b;
  cc = c * c;
  alpha = ((c - b) * la + (a - c) * lb + (b - a) * lc) / ((b - a) * (c - a) * (c - b));
  beta = ((cc - bb) * la + (aa - cc) * lb + (bb - aa) * lc) / ((b - a) * (b - c) * (c - a));
  gamma = (b * cc * la - bb * c * la + aa * c * lb - a * cc * lb - aa * b * lc + a * bb * lc) / ((b - a) * (c - a) * (c - b));


  return alpha * lamda * lamda + beta * lamda + gamma;

}










