/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 P R O F I L E    L I K E L I H O O D    R O U T I N E S 

 Peter Beerli 1997, Seattle
 beerli@genetics.washington.edu
 $Id: profile.c,v 1.18 1999/06/02 18:46:27 beerli Exp $
-------------------------------------------------------*/

#include "migration.h"

#include "world.h"
#include "derivatives.h"
#include "tools.h"
#include "broyden.h"
#include "combroyden.h"


#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif
long find_profilelike (double testlike, long whichprob, double minparam, double maxparam,
		       double *param, double *llike,
 long which, world_fmt * world, boolean QD, boolean QD2,double *mlparam, double *normd);
void calc_profile_likelihood (char method, long which, double *likes, double **param,
			      world_fmt * world, long nn);
void print_profile_likelihood (long which, world_fmt * world);
void print_line (FILE * outfile, char c, long nn, long flag);
void prepare_header (char *var, long which, world_fmt * world);
void print_profile_title (world_fmt * world);
void print_profile_table (char method, long which, char *var, double *likes, double **param, world_fmt * world);
void print_profile_percentile (world_fmt * world);
void print_percentile_header (FILE * outfile, boolean first);
void allocate_profile_percentiles (world_fmt * world);
void destroy_profile_percentiles (world_fmt * world);
void print_menu_profile (long which, long nn);
double interpolate (double xl, double xc, double xh,
		    double low, double center, double high, double testlike);

long warp (long ii);

void prognose_profile_end (long starttime, long numpop2, long nn);


void profile_max_percentiles (long which, double *likes, double **param,
			      world_fmt * world, long nn);

void profile_max_spline (char method, long which, double *likes, double **param,
			 world_fmt * world, long nn);


void calc_spline (double **rawparam, double *rawlikes, long *index,
	      long which, double *param, double *likes, double *yy, long n);

double recalc_profilelike (double testparam,
	      double *splineparam, double *splinelike, double *splinederiv);
long set_df(long which, char *custm2, long numpop, long numpop2);


long  set_profile_param(double *param, long which, long *allwhich, 
			double xval, double *xvals, world_fmt *world);


#define GRIDSIZE 9
#define GRIDMIDDLE 5
#define GRID {0.01,0.05,0.10,0.25,0.99,0.95,0.90,0.75,0.50}
#define GRID2 {0.01,0.05,0.10,0.25,0.5,0.75,0.90,0.95,0.99}
#define INDEX {0,1,2,3,8,7,6,5,4}
#define DEVIATE {0.02,0.10,0.20, 0.5, 50.,10., 5., 2., 1.}
//#define DEVIATE {0.002,0.010,0.020, 0.05, 5000.,1000., 500., 200., 1.}
#define ABSOLUTE {1e-100,1e100}
boolean
print_profile_likelihood_driver (long which, world_fmt * world)
{
  static boolean onetheta=FALSE;
  static boolean onemig=FALSE;
  long i,j;
  switch(world->options->custm2[which])
    {
    case '0': 
      return 0;
    case 'm': 
      if(which<world->numpop && onetheta)
	return 0;
      if(which<world->numpop)
	{
	  print_profile_likelihood(which,world);
	  onetheta=TRUE;
	}
      if(which>=world->numpop && onemig)
	return 0;
      if(which>=world->numpop)
	{
	  print_profile_likelihood(which,world);
	  onemig=TRUE;
	}
      return 1;
    case 's':
      j = (which - world->numpop) % (world->numpop - 1);
      i = (which - world->numpop) / (world->numpop - 1);
      if(i<=j)
	{
	  print_profile_likelihood(which,world);
	  return 1;
	}
      return 0;
    default:
      print_profile_likelihood(which,world);
    }
  return 1;
}

void
print_profile_likelihood (long which, world_fmt * world)
{
  long i, ii;
  char var[LINESIZE];
  char method = world->options->profilemethod;
  double **param;
  double likes[GRIDSIZE];
  long elem = world->options->gamma ?
  world->numpop * world->numpop + 1 : world->numpop * world->numpop;
  param = (double **) calloc (1, sizeof (double *) * GRIDSIZE);
  param[0] = (double *) calloc (1, sizeof (double) * GRIDSIZE * elem);
  for (i = 1; i < GRIDSIZE; i++)
    {
      param[i] = param[0] + i * elem;
    }
  if (world->options->progress)
    {
      print_menu_profile (which, world->numpop2);
    }
  calc_profile_likelihood (method, which, likes, param, world, GRIDSIZE);

  prepare_header (var, which, world);
  if (world->options->printprofile)
    {
      print_profile_table (method, which, var, likes, param, world);
    }
  if (world->options->printprofsummary)
    {
      strcpy (world->quantiles[which].name, var);
      for (ii = 0; ii < GRIDSIZE; ii++)
	{
	  i = warp (ii);
	  world->quantiles[which].param[i] = param[i][which];
	}
    }
  free (param[0]);
  free (param);
}

void
print_profile_table (char method, long which, char *var, double *likes, double **param, world_fmt * world)
{
  const double probabilities[] = GRID;
  char star;
  char methodstring[LINESIZE];
  long i, ii, j, likei = 0;
  long elem = world->options->gamma ?
  world->numpop * world->numpop + 1 : world->numpop * world->numpop;

  FILE *outfile = world->outfile;
  char *temp, *var2, temp2[LINESIZE], likestr[LINESIZE], *paramstr;
  double likemax = -DBL_MAX;

  var2 = (char *) calloc (1, sizeof (char) * MAX (LINESIZE, elem / 10. * LINESIZE));
  paramstr = (char *) calloc (1, sizeof (char) * MAX (LINESIZE, elem / 10. * LINESIZE));
  temp = var2;
  for (j = 0; j < elem; j++)
    {
      prepare_header (temp2, j, world);
      if (which == j)
	star = '*';
      else
	star = ' ';
      if (((j + 1) % 5) == 0)
	{
	  sprintf (temp, "\n                           %c%.7s%c     ", star, temp2, star);
	  temp += 28;
	}
      else
	sprintf (temp, "%c%.7s%c    ", star, temp2, star);
      temp += 11;
    }
  switch (method)
    {
    case 'p':
      strcpy (methodstring, "Parameters are evaluated at percentiles\nusing bisection method (slow, but exact).");
      break;
    case 's':
      strcpy (methodstring, "Parameters are evaluated at percentiles\nusing cubic splines of profiled parameter\n(faster, but not so exact).");
      break;
    case 'd':
      strcpy (methodstring, "Parameters are evaluated at pre-defined values\n");
      break;
    case 'q':
      strcpy (methodstring, "Parameters are evaluated assuming complete independence\n");
      break;
    case 'f':
      strcpy (methodstring, "Parameters are evaluated assuming complete independence\n and then once maximized at the found profiled parameter value");
      break;
    }
  fprintf (outfile, "\n\nProfile likelihood for parameter %s\n", var);
  fprintf (outfile, "%s\n", methodstring);
  print_line (outfile, '-', 79, CONT);
  if (method == 'd')
    fprintf (outfile, "      Ln(L)     %7.7s     %s\n", var, var2);
  else
    fprintf (outfile, "Per.  Ln(L)     %7.7s     %s\n", var, var2);
  print_line (outfile, '-', 79,START);
  for (i = 0; i < GRIDSIZE; i++)
    {
      if (likemax < MIN (likes[i], DBL_MAX))
	{
	  likemax = likes[i];
	  likei = i;
	}
    }
  for (ii = 0; ii < GRIDSIZE; ii++)
    {
      i = warp (ii);
      if (likes[i] >= DBL_MAX - EPSILON)
	sprintf (likestr, "     -     ");
      else
	sprintf (likestr, "% 7.3f%c", likes[i], likei == i ? '*' : ' ');
      temp = paramstr;
      for (j = 0; j < elem; j++)
	{
	  if (((j + 1) % 5) == 0)
	    {
	      sprintf (temp, "\n                              ");
	      temp += 26;
	    }
	  if (param[i][j] <= 0)
	    sprintf (temp, "      -      ");
	  else
          {
              if(param[i][j]<0.000001)
                  sprintf (temp, "% 7.5e ", param[i][j]);
              else
                  sprintf (temp, "% 10.6f ", param[i][j]);
          }
          
	  temp += 11;
	}
      if (param[i][which] <= 0)
	{
	  if (method == 'd')
	    fprintf (outfile, "        -           -       %s\n", paramstr);
	  else
	    fprintf (outfile, "%3.2f     -           -       %s\n",
		     probabilities[i], paramstr);
	}
      else
	{
	  if (method == 'd')
	    fprintf (outfile, "    %8.8s % 10.6g %s\n",
		     likestr, param[i][which], paramstr);
	  else
	    fprintf (outfile, "%3.2f %8.8s % 10.6g %s\n",
		     probabilities[i], likestr, param[i][which], paramstr);
	}
    }
  print_line (outfile, '-', 79, STOP);
  fprintf (outfile, "- = not possible to evaluate, most likely value either 0.0 or Infinity\n");
  fprintf (outfile, "    in the parameter direction, the likelihood surface is so flat\n");
  fprintf (outfile, "    that the calculation of the percentile(s)  failed.\n");
  free (paramstr);
  free (var2);
}


void
print_profile_title (world_fmt * world)
{
  print_line (world->outfile, '=', 79,CONT);
  fprintf (world->outfile, "Profile likelihood tables\n");
  print_line (world->outfile, '=', 79, CONT);
}

long
warp (long ii)
{
  long i;
  if (ii < 4)
    i = ii;
  else
    {
      if (ii == 4)
	{
	  i = GRIDSIZE - 1;
	}
      else
	{
	  i = GRIDSIZE - ii + 3;
	}
    }
  return i;
}


void
print_profile_percentile (world_fmt * world)
{
  const double probabilities[] = GRID2;
  long i, j, jj;
  boolean first = TRUE;
  boolean last = FALSE;
  fprintf (world->outfile, "\n\n");
  print_line (world->outfile, '=', 79, CONT);
  fprintf (world->outfile, "Summary of profile likelihood percentiles of all parameters\n");
  print_line (world->outfile, '=', 79,CONT);
  fprintf (world->outfile, "\n");
  print_percentile_header (world->outfile, first);
  for (i = 0; i < world->numpop2; i++)
    {
      if(world->quantiles[i].name[0]=='\0')
	continue; /* this variable was not estimated, so return*/
      fprintf (world->outfile, "%-10.10s  ", world->quantiles[i].name);
      for (jj = 0; jj < GRIDSIZE - 1; jj++)
	{
	  j = warp (jj);
	  if (probabilities[j] >= 0.5)
	    break;
	  if (world->quantiles[i].param[j] < 0)
	    fprintf (world->outfile, "      -       ");
	  else
	    {
		fprintf (world->outfile, "%11.6g   ", 
			 world->quantiles[i].param[j]);
	    }
	}
      if (world->quantiles[i].param[warp (jj)] < 0)
	fprintf (world->outfile, "      -       \n");
      else
	fprintf (world->outfile, "%11.6g\n", world->quantiles[i].param[warp (jj)]);
    }
  fprintf (world->outfile, "\n\n");
  print_percentile_header (world->outfile, last);
  for (i = 0; i < world->numpop2; i++)
    {
      if(world->quantiles[i].name[0]=='\0')
	continue; /* this variable was not estimated, so return*/
      fprintf (world->outfile, "%-10.10s  ", world->quantiles[i].name);
      for (jj = 0; jj < GRIDSIZE - 1; jj++)
	{
	  j = warp (jj);
	  if (probabilities[j] < 0.5)
	    continue;
	  if (world->quantiles[i].param[j] < 0)
	    fprintf (world->outfile, "      -       ");
	  else
	    fprintf (world->outfile, "%11.6g   ", world->quantiles[i].param[j]);
	}
      if (world->quantiles[i].param[warp (jj)] < 0)
	fprintf (world->outfile, "      -       \n");
      else
	fprintf (world->outfile, "%11.6g\n", world->quantiles[i].param[warp (jj)]);
    }
  print_line (world->outfile, '-', 79,CONT);
  fprintf (world->outfile, "- = not possible to evaluate, most likely value either 0.0 or Infinity\n");
  fprintf (world->outfile, "    in the parameter direction, the likelihood surface is so flat\n");
  fprintf (world->outfile, "    that the percentiles cannot be calculated.\n");
}

void
print_percentile_header (FILE * outfile, boolean first)
{
  const double probabilities[] = GRID2;
  long i;
  if (first)
    fprintf (outfile, "Parameter                          Lower percentiles\n");
  else
    fprintf (outfile, "Parameter                          Upper percentiles\n");
  fprintf (outfile, "            ");
  print_line (outfile, '-', 67,CONT);
  fprintf (outfile, "            ");
  for (i = 0; i < GRIDSIZE - 1; i++)
    {
      if (first)
	{
	  if (probabilities[i] >= 0.5)
	    break;
	}
      else
	{
	  if (probabilities[i] < 0.5)
	    continue;
	}
      fprintf (outfile, "    %3.2f      ", probabilities[i]);
    }
  fprintf (outfile, "    %3.2f\n", probabilities[i]);
  print_line (outfile, '-', 79,CONT);
}


void
prepare_header (char *var, long which, world_fmt * world)
{
  long i, j, from, to;
  if (which < world->numpop)
    {
      sprintf (var, "Theta_%li", from = which + 1);
    }
  else
    {
      i = (which - world->numpop) / (long) (world->numpop - 1);
      j = which - world->numpop - i * (world->numpop - 1);
      to = i + 1;
      from = (i <= j) ? j + 2 : j + 1;
      if(world->options->profileparamtype==PLOT4NM)
          sprintf (var, "4Nm_%li%li", from, to);
      else
          sprintf (var, "  M_%li%li", from, to);
    }
}

void
print_line (FILE * outfile, char c, long nn, long flag)
{
  long i,start=0;
  switch(flag)
    {
    case START: 
      start=2; 
      fputc ('=', outfile);
      fputc ('-', outfile);
      fputc ('-', outfile);
      break;
    case STOP: 
      start=2; 
      fputc ('=', outfile);
      fputc ('=', outfile);
      fputc ('-', outfile);
      break;
    default:
      start=0;
    }
  for (i = start; i < nn; i++)
    {
      fputc (c, outfile);
    }
  fputc ('\n', outfile);
}



void
calc_profile_likelihood (char method, long which, double *likes, double **param,
			 world_fmt * world, long nn)
{
  long i,j;
  switch (method)
    {
    case 'p' /*percentiles */ :
    case 'q': /*quick and dirty: no correlation between parametes */
    case 'f': /*mixture of quick and precise*/
      profile_max_percentiles (which, likes, param, world, nn);
      break;
    case 's': /* spline*/
    case 'd' /*discrete */ :
      profile_max_spline (method, which, likes, param, world, nn);
      break;
    }
  if(world->options->profileparamtype==PLOT4NM)
  {
      for(i=0;i<GRIDSIZE;i++)
      {
          for(j=world->numpop;j<world->numpop2;j++)
          {
              param[i][j] *= param[i][(long)(j-world->numpop)/(world->numpop-1)];
          }
      }
  }
  else
    printf("           Using M (m/mu) instead of 4Nm\n");
}


void
prognose_profile_end (long starttime, long numpop2, long nn)
{
#ifndef NOTIME_FUNC
  static long sumtime = 0;
  static long tally = 0;
  static char nowstr[LINESIZE];
  unsigned long endtime;
  time_t nowbin;
  struct tm *nowstruct;

  if (tally % nn == 0)
    {
      tally++;
      time (&endtime);
      sumtime = (endtime - starttime);

      nowbin = starttime + sumtime / tally * (nn * numpop2);
      nowstruct = localtime (&nowbin);
      strftime (nowstr, LINESIZE, "%c", nowstruct);
      fprintf (stdout, "           Prognosed end of run: %s\n", nowstr);
    }
  else
    tally++;
#endif
}


void 
profile_max_percentiles (long which, double *likes, double **param,
			 world_fmt * world, long nn)
{
  const double probabilities[] = GRID;
  
  long i, trials = 0;
  long df;
  boolean QD=FALSE, QD2=FALSE;
  double prob, normd, maxlike, testlike, minparam = 0, maxparam = 0;
  double *mlparam;
  
  df = (world->options->df>0) ? world->options->df : 
    set_df(which,world->options->custm2,world->numpop,world->numpop2);
  /* QD assumes that there is NO correlation between parameters
     that is certainly wrong but sometimes close and much faster
  */
  if(world->options->profilemethod=='q')
    QD=TRUE;
  else
    {
      if(world->options->profilemethod=='f')
	{
	  QD=TRUE;
	  QD2=TRUE;
	}
      else
	QD=FALSE;
    }
  /* the minimum */
  minparam = which < world->numpop ? SMALLEST_THETA : SMALLEST_MIGRATION;
  /* the maximum */
  if (world->loci > 1)
    {
      mlparam = world->atl[world->loci + 1].param;
      maxlike = world->atl[world->loci + 1].param_like;
    }
  else
    {
      mlparam = world->atl[world->loci].param;
      maxlike = world->atl[world->loci].param_like;
    }
  memcpy (param[nn - 1], mlparam, sizeof (double) * world->numpop2);;
  likes[nn - 1] = maxlike;
  maxparam = param[nn - 1][which];

  for (i = 0; i <  nn - 1; i++)
    {
      if (probabilities[i] > 0.5)
	prob = 1 - probabilities[i];
      else
	prob = probabilities[i];
      testlike = maxlike - 0.5 * find_chi (df, prob);
      if (i > 0)
	{
	  minparam = param[i - 1][which];
	  if (minparam < 0)
	    minparam = which < world->numpop ? SMALLEST_THETA : SMALLEST_MIGRATION;
	}
#ifdef MAC
      eventloop ();
#endif
      trials = find_profilelike (testlike, i, minparam, maxparam, param[i], &likes[i], which,
				 world, QD, QD2, mlparam, &normd);
      prognose_profile_end (world->starttime, world->numpop2, nn);
      /*if (trials >= MAX_PROFILE_TRIALS)
	{
	  likes[i] = DBL_MAX;
	  for (jj = 0; jj < world->numpop2; jj++)
	    {
	      param[i][jj] = -1;
	    }
	    }*/
    }
}

void 
profile_max_spline (char method, long which, double *likes, double **param,
		    world_fmt * world, long nn)
{
  double *splineparam, *splinelike, *splinederiv;
  double deviate[] = DEVIATE;

  long index[] = INDEX;
  const double probabilities[] = GRID;
  long i, trials = 0;
  long panic, df, profilenum;
  long *allwhich;
  double prob, normd, maxlike, testlike, minparam = 0, maxparam = 0;
  double *mlparam, *allvals;
  double tmp, x, xx, xlow, xhigh, mid, low, high, value;
  /* set degree of freedom*/

  df = (world->options->df>0) ? world->options->df : 
    set_df(which,world->options->custm2,world->numpop,world->numpop2);  
  allwhich = (long *) calloc(1,sizeof(long)*(world->numpop2+1));
  allvals = (double *) calloc(1,sizeof(double)*(world->numpop2+1));
  /* the minimum */
  minparam = which < world->numpop ? SMALLEST_THETA : SMALLEST_MIGRATION;
  /* the maximum */
  if (world->loci > 1)
    {
      mlparam = world->atl[world->loci + 1].param;
      maxlike = world->atl[world->loci + 1].param_like;
    }
  else
    {
      mlparam = world->atl[world->loci].param;
      maxlike = world->atl[world->loci].param_like;
    }
  memcpy (param[nn - 1], mlparam, sizeof (double) * world->numpop2);
  likes[nn - 1] = maxlike;
  maxparam = param[nn - 1][which];

  for (i = 0; i < nn - 1; i++)
    {
      if (which >= world->numpop && mlparam[which] < EPSILON)
	value = deviate[i];
      else
	value = deviate[i] * mlparam[which];
      profilenum = set_profile_param(param[i],which, allwhich, value, allvals,world);
      broyden_driver (world->atl, world->loci,
		  world, world->cov[world->loci], world->plane[world->loci],
		      allwhich, allvals, profilenum,
		      param[i], &likes[i], &normd,PROFILE);
      trials = 1;
      if (method != 's')
	prognose_profile_end (world->starttime, world->numpop2, nn);
    }

  if (method == 's')
    {
      splineparam = (double *) calloc (1, sizeof (double) * (GRIDSIZE + 2));
      splinelike = (double *) calloc (1, sizeof (double) * (GRIDSIZE + 2));
      splinederiv = (double *) calloc (1, sizeof (double) * (GRIDSIZE + 2));

      calc_spline (param, likes, index, which, splineparam, splinelike, splinederiv, nn);
      for (i = 0; i < nn - 1; i++)
	{
	  if (probabilities[i] < 0.5)
	    {
	      xhigh = splineparam[(nn + 2) / 2];
	      high = splinelike[(nn + 2) / 2];
	      xlow = splineparam[0];
	      low = splinelike[0];
	      prob = probabilities[i];
	    }
	  else
	    {
	      xlow = splineparam[nn + 1];
	      low = splinelike[nn + 1];
	      xhigh = splineparam[(nn + 2) / 2];
	      high = splinelike[(nn + 2) / 2];
	      prob = 1. - probabilities[i];
	    }
	  testlike = maxlike - 0.5 * find_chi (df, prob);
	  x = (xhigh + xlow) / 2.;
	  panic = 0;
	  mid = testlike - 1000.;
	  while (panic++ < MAX_PROFILE_TRIALS && fabs (mid - testlike) > BIGEPSILON)
	    {
	      mid = recalc_profilelike (x, splineparam, splinelike, splinederiv);
	      if (mid < low || mid > high)
		break;
	      if (testlike < mid)
		{
		  high = mid;
		  tmp = x;
		  x = (xlow + tmp) / 2;
		  xhigh = tmp;
		}
	      else
		{
		  low = mid;
		  tmp = x;
		  x = (tmp + xhigh) / 2;
		  xlow = tmp;
		}
	    }
	  xx = exp (x);
	  profilenum = set_profile_param(param[i],which, allwhich, xx, allvals,world);
	  broyden_driver (world->atl, world->loci,
		  world, world->cov[world->loci], world->plane[world->loci],
			  allwhich, allvals, profilenum, param[i], &likes[i], &normd,PROFILE);
	  x = log (xx);
	  prognose_profile_end (world->starttime, world->numpop2, nn);
	}
      free (splineparam);
      free (splinelike);
      free (splinederiv);
    }
  free(allwhich);
  free(allvals);
}

/* calc_spline() calculates the spline function derivatives,
   and adds two additional
   points, these two points bracket the real data points:
   so that param[0]=0 and param[n+1] = 1000 * the last real param,
   likes[0] is set -DBL_MAX and likes[n+1] is set using a linear interpolation
   using the last two real data points.

   Attention the arrays param, likes, and yy need to have a size
   of  n+2 and not n
 */
void
calc_spline (double **rawparam, double *rawlikes, long *index,
	     long which, double *param, double *likes, double *yy, long n)
{
  long i;
  double diff, tmp1, tmp2, first0, firstn;
  double *temp;
  temp = (double *) calloc (1, sizeof (double) * (GRIDSIZE + 2));
  for (i = 0; i < n; i++)
    {
      param[i + 1] = log (rawparam[index[i]][which]);
      likes[i + 1] = rawlikes[index[i]];
    }
  param[i + 1] = 100. + param[i];
  likes[i + 1] = ((likes[i] - likes[i - 1]) /
		  (param[i] - param[i - 1]) * param[i + 1]) +
    (likes[i - 1] +
     param[i - 1] * (likes[i] + likes[i - 1]) /
     (param[i] - param[i - 1]));
  firstn = ((likes[n] - likes[n + 1]) /
	    (param[n] - param[n + 1]));
  param[0] = log (SMALLEST_THETA);
  i = 2;
  likes[0] = ((likes[i] - likes[i - 1]) /
	      (param[i] - param[i - 1]) * param[0]) +
    (likes[i - 1] +
     param[i - 1] * (likes[i] + likes[i - 1]) / (param[i] - param[i - 1]));
  first0 = ((likes[i-0] - likes[i - 1]) /
	    (param[i-0] - param[i - 1]));
  yy[0] = -0.5;
  temp[0] = 3. / (param[1] - param[0]) * ((likes[1] - likes[0]) /
					  (param[1] - param[0]) - first0);
  for (i = 1; i < n + 1; i++)
    {
        diff = (param[i] - param[i - 1]) / (param[i + 1] - param[i - 1]);
      tmp1 = diff * yy[i - 1] + 2.0;
      yy[i] = (diff - 1.0) / tmp1;
      tmp2 = (likes[i + 1] - likes[i]) /
	(param[i + 1] - param[i]) -
	(likes[i] - likes[i - 1]) / (param[i] - param[i - 1]);
          temp[i] = (6.0 * tmp2 /
		 (param[i + 1] - param[i - 1]) -
		 diff * temp[i - 1]) / tmp1;
    }
  yy[n + 1] = ((3. / (param[n + 1] - param[n]) *
	 (firstn - (likes[n + 1] - likes[n]) / (param[n + 1] - param[n]))) -
	       0.5 * temp[n]) / (0.5 * yy[n] + 1.);
  for (i = n; i >= 0; i--)
    {
      yy[i] = yy[i] * yy[i + 1] + temp[i];
    }
  free (temp);
}

/* recalc_profilelike uses the spline derivatives and returns a new 
   loglikelihood value,
   testparam is THE LOG(testparam) !
 */
double
recalc_profilelike (double testparam,
	       double *splineparam, double *splinelike, double *splinederiv)
{
  double tmp1, tmp2, diff;
  long i = 0, lower, upper;

  while (testparam >= splineparam[i])
    i++;
  lower = i - 1;
  upper = i;
  if (splinelike[upper] - splinelike[lower] == 0.0)
    return splineparam[lower];
  diff = splineparam[upper] - splineparam[lower];
  tmp1 = (splineparam[upper] - testparam) / diff;
  tmp2 = (testparam - splineparam[lower]) / diff;

  return (tmp1 * splinelike[lower] + tmp2 * splinelike[upper] +
	  ((tmp1 * tmp1 * tmp1 - tmp1) * splinederiv[lower] +
     (tmp2 * tmp2 * tmp2 - tmp2) * splinederiv[upper]) * diff * diff / 6.0);
}

long 
set_profile_param(double *param, long which, long *allwhich,
		  double xval, double *xvals, world_fmt *world)
{
  long z=0, i;
  long numpop = world->numpop;
  long numpop2 = world->numpop2;
  char *custm2 = world->options->custm2;
  switch(custm2[which])
    {
        case 'S':
        case '*': param[which] = xval; allwhich[0]=which, xvals[0]=xval; return 1;
    case 's': z=0;
      while(which != world->options->symparam[z][0] &&
	    which != world->options->symparam[z][1])
	z++;
      param[world->options->symparam[z][0]] = xval;
      param[world->options->symparam[z][1]] = xval;
      allwhich[0]=which;
      allwhich[1]=z;
      xvals[0]=xval;
      xvals[1]=xval; 
      return 2;
    case 'm':
      if(which<numpop)
	{
	  for(i=0;i<numpop;i++)
	    {
	      param[i] = xval;
	      xvals[i]=xval;
	      allwhich[i] = i;
	    }
	  return numpop;
	}
      else 
	{
	  for(i=numpop;i<numpop2;i++){
	    param[i] = xval;
	    xvals[i-numpop]=xval;
	    allwhich[i-numpop] = i;
	  }
	  return numpop2-numpop;
	}
    }
  return -1; /* never come here */
}

long
find_profilelike (double testlike, long whichprob, double minparam, double maxparam,
		  double *param, double *llike,
  long which, world_fmt * world, boolean QD, boolean QD2, double *mlparam, double *normd)
{
  const double probabilities[] = GRID;

  boolean test;
  nr_fmt *nr = NULL;
  long locus, Gmax = 0;
  double tmp;
  double xlow, low;
  double x, mid = -DBL_MAX, xx = 0;
  double xhigh, high;
double *allvals;
  long panic = 0;
  long profilenum, *allwhich;
  *normd = -1;
  allwhich = (long *) calloc(1,sizeof(long)*(world->numpop2+1));
  allvals = (double *) calloc(1,sizeof(double)*(world->numpop2+1));  
if (probabilities[whichprob] < 0.5)
    {
      xhigh = maxparam;
      xlow = whichprob > 0 ? minparam : SMALLEST_THETA / 10000.;
    }
  else
    {
      if (probabilities[whichprob] > 0.5)
	{
	  xhigh = maxparam;
	  if (maxparam < EPSILON)
	    xlow = probabilities[whichprob] >= 0.99 ? 1000000.0 : minparam;
	  else
	    xlow = probabilities[whichprob] >= 0.99 ? maxparam * 10000 : minparam;
	}
      else
	{
	  xlow = minparam;
	  xhigh = maxparam;
	}
    }
  if (QD)
    {
      nr = (nr_fmt *) calloc (1, sizeof (nr_fmt) * 1);
      for (locus = 1; locus < world->loci + 1; locus++)
	{
	  if (Gmax < world->atl[locus].T)
	    {
	      Gmax = world->atl[locus].T;
	    }
	}
      create_nr (nr, world, Gmax, which);
      memcpy (nr->param, mlparam, sizeof (double) * world->numpop2);
      profilenum = set_profile_param(nr->param,which, allwhich, xlow, allvals,world);
      //nr->param[which] = xlow;
      for (locus = 1; locus < world->loci + 1; locus++)
	{
	  create_apg0 (nr->apg0[locus], nr, &world->atl[locus]);
	}
      low = calc_loci_like (nr, world->atl, world->loci, world->options->gamma);
      profilenum = set_profile_param(nr->param,which, allwhich, xhigh, allvals,world);
      //nr->param[which] = xhigh;
      high = calc_loci_like (nr, world->atl, world->loci, world->options->gamma);
    }
  else
    {
      profilenum=set_profile_param(param,which, allwhich, xlow, allvals,world);
      broyden_driver (world->atl, world->loci,
		  world, world->cov[world->loci], world->plane[world->loci],
		      allwhich, allvals, profilenum, param, llike, normd,PROFILE);
      low = *llike;
      profilenum = set_profile_param(param,which, allwhich, xhigh, allvals,world);
      broyden_driver (world->atl, world->loci,
		  world, world->cov[world->loci], world->plane[world->loci],
		      allwhich, allvals, profilenum, param, llike, normd,PROFILE);
      high = *llike;
    }
  panic = 0;
  x = (xhigh + xlow) / 2.;
  while (panic++ < MAX_PROFILE_TRIALS && fabs (mid - testlike) > BIGEPSILON)
    {
      if (QD)
	{
	  //nr->param[which] = x;
	  profilenum = set_profile_param(nr->param,which, allwhich, x, allvals,world);
	  mid = calc_loci_like (nr, world->atl, world->loci, world->options->gamma);
	}
      else
	{
	  profilenum = set_profile_param(param,which, allwhich, x, allvals, world);
	  broyden_driver (world->atl, world->loci,
		  world, world->cov[world->loci], world->plane[world->loci],
			  allwhich, allvals, profilenum, param, llike, normd,PROFILE);
	  mid = *llike;
	}
      test = ((fabs (mid - low) < EPSILON) || (fabs (mid - high) < EPSILON));
      test = TRUE;
      if (!test)
	{
	  xx = exp (interpolate (log (xlow), log (x), log (xhigh), low, mid, high, testlike));
	}
      if ((testlike < low) || (testlike > high))
	return MAX_PROFILE_TRIALS;
      if (testlike < mid)
	{
	  high = mid;
	  tmp = x;
	  if (test)
	    x = (xlow + tmp) / 2;
	  else
	    x = xx;
	  xhigh = tmp;
	}
      else
	{
	  low = mid;
	  tmp = x;
	  if (test)
	    x = (tmp + xhigh) / 2;
	  xlow = tmp;
	}
    }
  if (QD)
    {
      memcpy (param, nr->param, sizeof (double) * nr->numpop2);
      destroy_nr (nr, world);
      *llike = mid;
      if(QD2)
	{
	  x = param[which];
	  profilenum = set_profile_param(param,which, allwhich, x, allvals, world);
	  broyden_driver (world->atl, world->loci,
			  world, world->cov[world->loci], 
			  world->plane[world->loci],
			  allwhich, allvals, profilenum, param, llike, 
			  normd,PROFILE);
	}
    }
  free(allwhich);
  return panic;
}


double
interpolate (double xl, double xc, double xh,
	     double low, double center, double high, double testlike)
{
  double xc2, xl2, xh2, a, b, c, x;

  xc2 = xc * xc;
  xl2 = xl * xl;
  xh2 = xh * xh;

  a = (-(high * xc) + low * xc + center * xh -
       low * xh - center * xl + high * xl) /
    ((-xc + xh) * (xh - xl) * (-xc + xl));

  b = (high * xc2 - low * xc2 -
       center * xh2 + low * xh2 +
       center * xl2 - high * xl2) /
    ((-xc + xh) * (xc - xl) * (-xh + xl));

  c = -((low * xc2 * xh - low * xc * xh2 -
	 high * xc2 * xl + center * xh2 * xl +
	 high * xc * xl2 - center * xh * xl2) /
	((-xc + xh) * (xc - xl) * (xh - xl)));


  x = (-b - sqrt (b * b - 4 * a * (c - testlike))) / (2 * a);

  return x;
}


void
allocate_profile_percentiles (world_fmt * world)
{
  long i;
  world->quantiles = (quantile_fmt *) calloc (1, sizeof (quantile_fmt) * world->numpop2);
  for (i = 0; i < world->numpop2; i++)
    {
      world->quantiles[i].name = (char *) calloc (1, sizeof (char) * 20);
      world->quantiles[i].param = (double *) calloc (1, sizeof (double) * GRIDSIZE);
    }
}

void
destroy_profile_percentiles (world_fmt * world)
{
  long i;
  for (i = 0; i < world->numpop2; i++)
    {
      free (world->quantiles[i].name);
      free (world->quantiles[i].param);
    }
  free (world->quantiles);
}


void
print_menu_profile (long which, long nn)
{
  char nowstr[LINESIZE];
  get_time (nowstr, "%H:%M:%S");
  fprintf (stdout, "%s   Calculating profile likelihoods for parameter %2li out of %2li\n",
	   nowstr, 1 + which, nn);
}


long set_df(long which, char *custm2, long numpop, long numpop2)
{
  char ch;
  long i;
  long zeros=0;
  long ms=0;
  long ss=0;

 for(i=0;i<numpop2;i++)
    {
      ch=custm2[i];
      switch(ch)
	{
	case '0':
	  zeros++; break;
	case 'm':
	  ms++;break;
	case 's':
	  ss++; break;
	}
    }
  if (numpop2-zeros==ms)
    return 2;
  else
    {
      if (custm2[0]=='m')
	return numpop2 - numpop + 1 - zeros - ss/2;
      else
	return numpop2 - zeros - ss/2;
    }
  return -1;
}













