/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 W O R L D   R O U T I N E S 

 creates tree structures
 
 prints results,
 and finally helps to destroy itself.
                                                                                                               
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: world.c,v 1.19 1999/06/02 18:46:29 beerli Exp $
-------------------------------------------------------*/
#define SICK_VALUE    -1
#include "migration.h"
#include "mcmc.h"

#include "fst.h"
#include "random.h"
#include "tools.h"
#include "broyden.h"
#include "combroyden.h"
#include "options.h"

#ifdef DMALLOC_FUNC_CHECK
#include "dmalloc.h"
#endif

#define PLANESIZE 36
#define PLANEBIGTICKS 6
//#define PLANEBIGTICKVALUES {-3, -2, -1, 0, 1, 2}
#define PLANEBIGTICKVALUES {1,2,3,4,5,6}
/*#define PLANETICKS   {0.001, 0.0013895, 0.0019307, 0.0026827, 0.00372759, \
                     0.00517947, 0.00719686, 0.01, 0.013895, 0.019307, \
                     0.026827, 0.0372759, 0.0517947, 0.0719686, 0.1, \
                     0.13895, 0.19307, 0.26827, 0.372759, 0.517947, \
                     0.719686, 1., 1.3895, 1.9307, 2.6827, 3.72759, \
                     5.17947, 7.19686, 10., 13.895, 19.307, 26.827, 37.2759, \
                     51.7947, 71.9686, 100.}
*/
#define CONTOURLEVELS 8
#define CONTOURS_LOCUS {0.0,-3.35670/2., -9.48773/2., -13.2767/2.,\
                        0.0,-3.35670/2., -9.48773/2., -13.2767/2.}
#define CONTOURS_LOCI  {0.0,-4.35146/2., -11.0705/2., -15.0863/2.,\
                        0.0,-4.35146/2., -11.0705/2., -15.0863/2.}

/* prototypes ------------------------------------------- */
void create_world (world_fmt ** world);
void init_world (world_fmt * world, data_fmt * data, option_fmt * options);
void calc_simple_param (world_fmt * world);
void print_menu_locus (option_fmt * options, long locus);
void print_menu_chain (char type, long chain, long steps, world_fmt * world);
void set_bounds (long *increment, long *steps, long *chain,
		 const option_fmt * options, const char type);
void heating_ratio(world_fmt *world);
double  heater_complex(long i, double heat, long maxdiff);
double heater_simple(long i, double heat, long maxdiff);
void burnin_chain (world_fmt * world);
void copy_atl (world_fmt * world, timearchive_fmt * old, timearchive_fmt * new, long steps);

void print_list (world_fmt * world);
void plot_surface (world_fmt * world, char ****plane, long x);
void plot_surface_header2(FILE *outfile, long locus);
void plot_surface2(FILE *outfile, long x, long locus, long numpop,
		   char ****plane, plotmax_fmt **plotmax, char **popnames,
		   long migrmodel);
void print_alpha_curve (world_fmt * world, timearchive_fmt * atl);
void create_loci_plot (world_fmt * world, char ***plane, timearchive_fmt * atl, long loci);
void create_locus_plot (world_fmt * world, char ***plane, tarchive_fmt * tl, nr_fmt * nr, long G);
void cleanup_world (world_fmt * world, long locus);
void apocalypse_now (world_fmt * world);
/* private functions */
void create_timearchive (timearchive_fmt ** atl, long loci, long samples, long numpop);
void create_plotplane (world_fmt * world);
void create_cov (world_fmt * world);
void print_menu_equilib (option_fmt * options);
void print_menu_heating (option_fmt * options);
void print_finish (world_fmt * world, long filepos);
void copy_time (world_fmt * world, timelist_fmt * ltl, long from, long to, long np);
void archive_timelist (tarchive_fmt * atl, vtlist * tl, long T, long np, world_fmt *world);
void copy_timelist (tarchive_fmt * from, tarchive_fmt * to, long np);
long whichp (long from, long to, long pop);
long whichl (long from, long to, long pop);
/* long calc_T(timearchive_fmt * old, long steps); */
void print_results (world_fmt * world);
void print_fst (world_fmt * world, double **fstparam);
void prepare_print_nu (double nu, char *str);
void prepare_print_nm (double nm, double nmu, char *strllike);
void print_menu_coalnodes (world_fmt * world, long G);
void print_menu_createplot (void);
void calc_loci_plane (world_fmt * world, nr_fmt * nr, timearchive_fmt * atl, double ***pl, long loci, double **contours);
void calc_locus_plane (world_fmt * world, nr_fmt * nr, tarchive_fmt * tl, long G,
		       double ***pl, double **contours);
void fill_plotplane (world_fmt * world, char **plane, double **pl, double *contours);
void print_mathematica (world_fmt * world, double ***plane, long x, long y);
void print_cov (world_fmt * world, long numpop, long loci, double ***cov);
void print_cov_table (FILE * outfile, long locus, world_fmt * world, double *corr, long addvar);
void free_seqx (node * p, long sites);
void free_x (node * p);
void free_tree (node * p);
void free_nodelet (node * p, long num);
void increase_timearchive (world_fmt * world, long locus, long sample, long numpop);
void test_locus_like (nr_fmt * nr, double *param0, double *param1, long df,
		 long locus, world_fmt * world, long *maxwhich, long maxnum,
		      boolean withhead, char *this_string);
void test_loci_like (nr_fmt * nr, double *param0, double *param1, long df,
		  long loci, world_fmt * world, long *maxwhich, long maxnum,
		     boolean withhead, char *this_string);


long set_test_param (double *param, char *strp, world_fmt * world, long lrline, long locus, long *maxwhich, long *maxnum);
void print_CV (world_fmt * world);
void print_param (FILE * file, double *param, long nn, char spacer[]);
void set_contours (double **contours, long df, long numpop);
void print_lratio_test (world_fmt * world);
void print_result_header (char *titletext, world_fmt * world);
void print_result_population (long pop, world_fmt * world);
void print_result_param (FILE * file, double *param, long numpop, long pop);
void print_result_fst (long pop, world_fmt * world);
void print_simulation (world_fmt * world);
void prognose_time (char *nowstr, world_fmt * world, long steps, char *spacer);

void
print_mighist(world_fmt *world);

/*======================================================*/
void
create_world (world_fmt ** world)
{
  (*world) = (world_fmt *) calloc (1, sizeof (world_fmt));
  (*world)->param_like = DBL_MAX;
}

void
init_world (world_fmt * world, data_fmt * data, option_fmt * options)
{
  long locus, i;
  long custmlen = 0;
  world->options = options;
  world->data = data;
  if (!options->readsum)
    {
      world->loci = data->loci;
      world->skipped = 0;
      world->numpop = data->numpop;
      world->numpop2 = world->numpop * world->numpop;
      if(world->numpop==1)
	options->plot = FALSE;
    }
  custmlen = strlen(options->custm);
  fillup_custm(custmlen,world,options);
	 /*  if ((custmlen = strlen (options->custm)) <= world->numpop2)
	     {
	     options->custm = (char *) realloc (options->custm, sizeof (char) *
	     (world->numpop2 + 1));
	     for (i = custmlen; i < world->numpop2; i++)
	     {
	     options->custm[i] = '*';
	     }
	     options->custm[i] = '\0';
	     }
	 */
  getseed (options);
  create_timearchive (&(world->atl), world->loci,
		      SAMPLETREE_GUESS, world->numpop);
  create_plotplane (world);
  create_cov (world);
  world->likelihood = (double *) calloc (1, sizeof (double) * SAMPLETREE_GUESS);
  world->lineages = (long *) calloc (1, sizeof (long) * world->numpop);
  world->param0 = (double *) calloc (1, sizeof (double) * world->numpop2);
  world->param00 = (double *) calloc (1, sizeof (double) * world->numpop2);
  world->heat = (double *) calloc (1, sizeof (double) * 1
				   /*realloc in heating_ratio()*/);
  switch (options->datatype)
  {
      case 's':
      case 'n':
      case 'u':
      case 'a':
          world->fstparam = (double **) calloc (1, sizeof (double *) * (world->loci + 1));
          for (locus = 0; locus < world->loci + 1; locus++)
              world->fstparam[locus] = (double *) calloc (1, sizeof (double) * world->numpop2 * 2);
          break;
      case 'm':
          world->options->steps = (double **)
              calloc (1, sizeof (double *) * world->options->micro_stepnum);
          for (locus = 0; locus < world->options->micro_stepnum; locus++)
              world->options->steps[locus] = (double *) calloc (1,
                                                                sizeof (double) * world->options->micro_stepnum);
              /* do not break here*/
      case 'b':
          world->fstparam = (double **) calloc (1, sizeof (double *) * (world->loci + 1));
          for (locus = 0; locus < world->loci + 1; locus++)
              world->fstparam[locus] = (double *) calloc (1, sizeof (double) * world->numpop2 * 2);
          
          break;
  }
      /* timer -- how long will the program run? */
  time (&world->starttime);
  world->treestotal = world->loci * (options->schains * options->sincrement * options->ssteps +
                                     options->lchains * options->lincrement * options->lsteps +
                                     (options->lchains + options->schains) * options->burn_in);
  
      /* migration histogram */
  if(world->options->mighist)
  {
      world->mighistloci = (mighistloci_fmt *) 
          calloc(1,sizeof(mighistloci_fmt) * world->loci);
      world->mighistlocinum=0;
      for(locus=0;locus<world->loci;locus++)
      {
          world->mighistloci[locus].mighist = (mighist_fmt *) 
		  calloc(1,sizeof(mighist_fmt) * world->options->lsteps);
		world->mighistloci[locus].mighistnum=0;
		for(i=0;i<world->options->lsteps;i++)
		  world->mighistloci[locus].mighist[i].migevents = (migevent_fmt *) calloc(1,sizeof(migevent_fmt)*1);
	    /* allocation will be in archive_timelist()*/
	      }
    }
  /* tree and treetimes are not yet allocated */
}


void
calc_simple_param (world_fmt * world)
{
  switch (world->options->datatype)
    {
    case 'a':    
    case 's':
    case 'n':
    case 'b':
    case 'm':
        case 'u':
            
      calc_fst (world);
      break;
    default:
      break;
    }
}

void
set_bounds (long *increment, long *steps, long *chains,
	    const option_fmt * options, const char type)
{
  switch (type)
    {
    case 's':
      *increment = options->sincrement;
      *steps = options->ssteps;
      *chains = options->schains;
      break;
    case 'l':
      *increment = options->lincrement;
      *steps = options->lsteps;
      *chains = options->lchains;
      break;
    default:
      error( "Wrong chain type\n");
      break;
    }
}

void 
heating_ratio(world_fmt *world)
{
  short heating;
  double (*heater) (long, double, long);
  long step, treeprint, j=0,i;
  double minheat,maxheat, startlike=0, bestlike=0;
  if (!world->options->heating)
    return;
  if(world->options->heating==1)
    {
      heater = (double (*) (long, double, long)) heater_simple;
      heating=1;
    }
  else
    {
      heater = (double (*) (long, double, long)) heater_complex;
      heating=2;
    }
  world->heat = (double *) realloc(world->heat,
				   sizeof(double)* world->increment);
  world->options->heating=0;
  print_menu_heating(world->options);
  treeprint = world->options->treeprint;
  world->options->treeprint = NONE;
  startlike = bestlike =world->likelihood[0];
  for (step = 0; step < MAX(world->options->burn_in,TWOHUNDRED); step++)
    {
      j += metropolize (world, 0);
      if(world->likelihood[0]>bestlike)
	bestlike = world->likelihood[0];
    }
  world->heatratio = (bestlike - startlike);
  fprintf(stdout,"           Acceptance ratio = %f\n", 
	  (double) j / MAX(world->options->burn_in,TWOHUNDRED));
  fprintf(stdout, "          Heating ratio    = %f\n",world->heatratio);

  if(world->increment % 2 == 0)
    {
      for(i=0;i<world->increment/2;i++)
	world->heat[i] = (*heater)(i,world->heatratio,world->increment/2);
      for(i=world->increment/2;i< world->increment;i++)
	world->heat[i] = (*heater)((world->increment-i),world->heatratio,
				world->increment/2);
    }
  else
    {
      for(i=0;i<world->increment/2;i++)
	world->heat[i] = (*heater)(i,world->heatratio,world->increment/2);
      for(i=world->increment/2;i< world->increment;i++)
	world->heat[i] = (*heater)((world->increment-i),world->heatratio,
				world->increment/2);
    }
  maxheat = world->heat[0];
  minheat = world->heat[world->increment/2];
  for(i=0;i<world->increment;i++)
    {
      if(world->options->heating==1)
	world->heat[i] = log(world->heat[i]);
      else
	world->heat[i] = log(world->heat[i] + (world->heat[i]-minheat)/
					    (maxheat-minheat) * 
(1.-maxheat));
    }
  world->options->heating=heating;
  world->options->treeprint = treeprint;
}

double 
heater_complex(long i, double heat, long maxdiff)
{
  //  return (1. - exp(1. * heat * i)/((1.+ heat) * exp(1. * heat * maxdiff)));
  return (1./(1. + (heat * i/maxdiff)));
}

double 
heater_simple(long i, double heat, long maxdiff)
{
  //  return (exp((-1. + MIN(2. * heat,0.5)) * i));
  return (1./(1. + (2. * i/maxdiff)));
}

void
burnin_chain (world_fmt * world)
{
  boolean heating;
  long step, treeprint;
  if (world->options->burn_in == 0)
    return;
  heating=world->options->heating;
  world->options->heating=FALSE;
  print_menu_equilib (world->options);
  treeprint = world->options->treeprint;
  world->options->treeprint = NONE;
  for (step = 0; step < world->options->burn_in; step++)
    {
      metropolize (world, 0);
    }
  world->options->heating=heating;
  world->options->treeprint = treeprint;
}

void
copy_atl (world_fmt * world, timearchive_fmt * old, timearchive_fmt * new, long steps)
{
  long j;
  long numpop = world->numpop;

  increase_timearchive (world, world->locus + 1, steps, numpop);

  new->T = /*calc_T(old, steps); */ old->T;
  new->param_like = world->param_like;
  new->sumtips = world->sumtips;
  new->numpop = world->numpop;
  memcpy (new->param, world->param0, sizeof (double) * world->numpop2);
  memcpy (new->param0, world->param00, sizeof (double) * world->numpop2);
  memcpy (new->lparam0, old->lparam0, sizeof (double) * world->numpop2);
  memcpy (new->likelihood, world->likelihood, sizeof (double) * steps);
  for (j = 0; j < new->T; j++)
    {
      new->tl[j].copies = old->tl[j].copies;
      copy_timelist (&old->tl[j], &new->tl[j], numpop);
    }
}

void
print_list (world_fmt * world)
{
  if (world->options->simulation)
    print_simulation (world);
  else
    {

      print_results (world);
      if (!world->options->readsum)
	{
	  if (world->options->printfst)
	    print_fst (world, world->fstparam);
	}
      if (world->options->plot)
	{
	  plot_surface (world, world->plane, world->options->plotintervals + 2);
	}
      if (world->options->gamma)
	print_alpha_curve (world, world->atl);
      if (world->options->lratio->counter > 0)
	{
	  print_lratio_test (world);
	}
      print_mighist(world);
      /*  print_cov(world, world->numpop, world->loci, world->cov); */
    }
}

void
print_lratio_test (world_fmt * world)
{
  long i, locus, Gmax=0;
  long df;
  nr_fmt *nr;
  double *param0;
  long *maxwhich;
  long maxnum = 0;
  param0 = (double *) calloc (1, sizeof (double) * (world->numpop2 + 1));
  maxwhich = (long *) calloc (1, sizeof (long) * (world->numpop2 + 1));


  if (world->options->progress)
    fprintf (stdout, "           Printing likelihood ratio tests\n");

  nr = (nr_fmt *) calloc (1, sizeof (nr_fmt));
  for (i = 1; i < world->loci + 1; i++)
    {
      if (Gmax < world->atl[i].T)
	Gmax = world->atl[i].T;
    }
  create_nr (nr, world, Gmax, 0);

  if (world->options->lratio->counter > 0)
    {
      PAGEFEEDWORLD;
      for (locus = 1; locus < world->loci + 1; locus++)
	{
	  create_apg0 (nr->apg0[locus], nr, &world->atl[locus]);
	}
      if (world->loci - world->skipped > 1 &&
	  world->options->lratio->data[0].type == MEAN)
	{
	  df = set_test_param (param0, world->options->lratio->data[0].value,
			       world, 0, -1, maxwhich, &maxnum);
	  test_loci_like (nr, param0,
			  world->atl[world->loci + 1].param, df,
			  world->loci, world, maxwhich, maxnum, HEADER,
			  world->options->lratio->data[0].value);
	}
      else
	{
	  for (locus = 0; locus < world->loci; locus++)
	    {
	      df = set_test_param (param0, world->options->lratio->data[0].value,
				   world, 0, locus, maxwhich, &maxnum);
	      test_locus_like (nr, param0,
			       world->atl[locus + 1].param, df,
			       locus + 1, world, maxwhich, maxnum, HEADER,
			       world->options->lratio->data[0].value);
	      if ((locus + 1) % 2 == 0)
		PAGEFEEDWORLD;
	    }
	}
    }
  for (i = 1; i < world->options->lratio->counter; i++)
    {
      if (world->loci - world->skipped > 1 &&
	  world->options->lratio->data[i].type == MEAN)
	{
	  df = set_test_param (param0, world->options->lratio->data[i].value,
			       world, i, -1, maxwhich, &maxnum);
	  test_loci_like (nr, param0,
			  world->atl[world->loci + 1].param, df,
			  world->loci, world, maxwhich, maxnum, FALSE,
			  world->options->lratio->data[i].value);
	  if (i % 2 == 0)
	    PAGEFEEDWORLD;

	}
      else
	{
	  for (locus = 0; locus < world->loci; locus++)
	    {
	      df = set_test_param (param0, world->options->lratio->data[i].value,
				   world, i, locus, maxwhich, &maxnum);
	      test_locus_like (nr, param0,
			       world->atl[locus + 1].param, df,
			       locus + 1, world, maxwhich, maxnum, FALSE,
			       world->options->lratio->data[i].value);
	      if ((locus + 1) % 2 == 0)
		PAGEFEEDWORLD;
	    }
	}
    }
  fprintf (world->outfile, "Remember: Tests based on a single locus may not be valid\n");
  free (param0);
  free (maxwhich);
  destroy_nr (nr, world);
}

void
set_ticks(double **ticks,double *plotrange, short type)
{
  long i;
  double diff;
  (*ticks)[0] = plotrange[0];
  (*ticks)[PLANEBIGTICKS-1] = plotrange[1];
  if(type==PLOTSCALELOG)
    {
      diff = (log10(plotrange[1])-log10(plotrange[0]))/(PLANEBIGTICKS-1.);
      for(i=1;i<PLANEBIGTICKS-1;i++)
	{
	  (*ticks)[i] = pow(10.,(log10((*ticks)[0]) + i * diff));
	}
    }
  else
    {
      diff = (plotrange[1]-plotrange[0])/(PLANEBIGTICKS-1.);
      for(i=1;i<PLANEBIGTICKS-1;i++)
	{
	  (*ticks)[i] = (*ticks)[0] + i * diff;
	}	
    }
}
void
plot_surface (world_fmt * world, char ****plane, long x)
{
  long locus;
  long loci = world->loci;
  FILE *outfile = world->outfile;
  double *ticks;
  double prangex[2];
  double prangey[2];
  ticks = (double *) calloc(1,sizeof(double)*PLANEBIGTICKS);
  prangex[0] = world->options->plotrange[0];
  prangex[1] = world->options->plotrange[1];
  prangey[0] = world->options->plotrange[2];
  prangey[1] = world->options->plotrange[3];
  if (world->options->progress)
    fprintf (stdout, "           Plotting the likelihood surfaces\n");
  PAGEFEED;
  fprintf (outfile, "Ln-Likelihood surfaces for each of the %3li populations\n", world->numpop);
  fprintf (outfile, "-------------------------------------------------------\n\n");
  fprintf (outfile, "Legend:\n");
  fprintf (outfile, "   X = Maximum likelihood\n");
  fprintf (outfile, "   * = in approximative 50%% confidence limit\n");
  fprintf (outfile, "   + = in approximative 95%% confidence limit\n");
  fprintf (outfile, "   - = in approximative 99%% confidence limit\n");
  set_ticks(&ticks,prangex,world->options->plotscale);
  fprintf (outfile, "X-tickmarks are (1) %f, (2) %f, (3) %f\n",
	   ticks[0],ticks[1],ticks[2]);
  fprintf (outfile, "                (4) %f, (5) %f, (6) %f\n",
	   ticks[3],ticks[4],ticks[5]);
  set_ticks(&ticks,prangey,world->options->plotscale);
  fprintf (outfile, "Y-tickmarks are (1) %f, (2) %f, (3) %f\n",
	   ticks[0],ticks[1],ticks[2]);
  fprintf (outfile, "                (4) %f, (5) %f, (6) %f\n",
	   ticks[3],ticks[4],ticks[5]);
  if (!world->options->readsum)
    {
      for (locus = 0; locus < loci; locus++)
	{
	  if (world->data->skiploci[locus])
	    {
	      continue;
	    }
	  fprintf (outfile, "\n\nLocus %li\n",locus+1);
	  plot_surface_header2(outfile,locus);
	  plot_surface2(outfile, PLANESIZE+2, locus, world->numpop, plane,world->plotmax,world->data->popnames,
			world->options->migration_model);
	}
    }
  if ((loci - world->skipped > 1) || (world->options->readsum))
    {
      fprintf (outfile, "\n%s\n\n", (world->options->readsum) ?
	       ((loci - world->skipped > 1) ? "Over all loci" :
		"Over locus 1") : "Over all loci");
      locus = loci;
      plot_surface_header2(outfile,locus);
      plot_surface2(outfile, PLANESIZE+2, locus, world->numpop, plane,world->plotmax,world->data->popnames,
		    world->options->migration_model);
    }  
}

void plot_surface_header2(FILE *outfile, long locus)
{
  fprintf(outfile, "    x-axis= 4Nm [effective population size * migration rate],\n");
  fprintf(outfile, "    y-axis = Theta,\n    units = see above\n");
}

void plot_surface2(FILE *outfile, long x, long locus, long numpop, 
		   char ****plane, plotmax_fmt **plotmax, char **popnames,
		   long migrmodel)
{
  long pop,i;
  if(migrmodel==ISLAND)
    {
      fprintf (outfile, "N-ISLAND MODEL: For each Population\n");
      fprintf (outfile, "  Immigration: 4Nm=%f, Theta=%f, log likelihood=%f\n",
	       plotmax[locus][0].x1, plotmax[locus][0].y1,
	       plotmax[locus][0].l1);
      fprintf (outfile, "\n                 Immigration   \n\n");
      for (i = x; i >= 0; i--)
	fprintf (outfile, "%46.46s\n", plane[locus][0][i]);
      fprintf (outfile, "%46.46s\n", plane[locus][0][x]);
      PAGEFEED;
    }
  else
    {
      for (pop = 0; pop < numpop; pop++)
	{
	  if(popnames==NULL)
	    fprintf (outfile, "Population %li\n", pop + 1);
	  else
	    fprintf (outfile, "Population %li: %s\n", 
		     pop + 1, popnames[pop]);
	  fprintf (outfile, "  Immigration: 4Nm=%f, Theta=%f, log likelihood=%f\n",
		   plotmax[locus][pop].x1, plotmax[locus][pop].y1,
		   plotmax[locus][pop].l1);
	  fprintf (outfile, "  Emmigration: 4Nm=%f, Theta=%f, log likelihood=%f\n",
		   plotmax[locus][pop].x2, plotmax[locus][pop].y2,
		   plotmax[locus][pop].l2);
	  fprintf (outfile, "\n                 Immigration                              Emmigration\n\n");
	  for (i = x; i >= 0; i--)
	    fprintf (outfile, "%s\n", plane[locus][pop][i]);
	  fprintf (outfile, "%s\n", plane[locus][pop][x]);
	  PAGEFEED;
	}
    }
}



void
print_alpha_curve (world_fmt * world, timearchive_fmt * atl)
{
  const double invalphas[19] =
  {0.0001, 0.0002, 0.0005,
   0.001, 0.002, 0.005,
   0.01, 0.02, 0.05,
   0.1, 0.2, 0.5,
   1., 2., 5.,
   10., 20., 50., 100.};
  boolean once = FALSE;
  long g, a, gmax = 1, loci = world->loci;
  double likes[21];
  char confidence[21];
  nr_fmt *nr;
  FILE *outfile = world->outfile;
  double **contours;

  if (world->options->gamma && loci - world->skipped > 1)
    {
      nr = (nr_fmt *) calloc (1, sizeof (nr_fmt));
      for (g = 1; g < loci + 1; g++)
	{
	  if (gmax < atl[g].T)
	    gmax = atl[g].T;
	}
      create_nr (nr, world, gmax, 0);
      nr->skiploci = world->data->skiploci;
      for (g = 1; g < world->loci + 1; g++)
	create_apg0 (nr->apg0[g], nr, &world->atl[g]);
      memcpy (nr->param, atl[loci + 1].param, sizeof (double) * world->numpop2);
      contours = (double **) calloc (1, sizeof (double *) * 1);
      contours[0] = (double *) calloc (1, sizeof (double) * 1 * CONTOURLEVELS);
      set_contours (contours, world->numpop2, 1);

      for (g = 0; g < CONTOURLEVELS; g++)
	{
	  contours[0][g] += atl[loci + 1].param_like;
	}
      if (world->options->progress)
	fprintf (stdout, "           Printing Inv(alpha) x Ln(likelihood)\n");
      gmax = 0;

      likes[20] = calc_loci_like (nr, atl, loci, TRUE);
      for (a = 0; a < 19; a++)
	{
	  nr->param[nr->numpop2] = invalphas[a];
	  likes[a] = calc_loci_like (nr, atl, loci, TRUE);
	  if (atl[loci + 1].param[nr->numpop2] < invalphas[a] && !once)
	    {
	      gmax = a - 1;
	      once = TRUE;
	    }
	}
      for (a = 0; a < 19; a++)
	{
	  if (likes[a] < contours[0][1])
	    {
	      if (likes[a] < contours[0][2])
		{
		  if (likes[a] < contours[0][3])
		    {
		      confidence[a] = ' ';
		    }
		  else
		    {
		      confidence[a] = '-';
		    }
		}
	      else
		{
		  confidence[a] = '+';
		}
	    }
	  else
	    {
	      if (likes[a] < contours[0][0])
		confidence[a] = '*';
	      else
		{
		  confidence[a] = 'X';
		}
	    }
	}
      fprintf (outfile, "Ln-Likleihood curve for the shape ");
      fprintf (outfile, "parameter Inv(alpha) [square(CV(mu))]\n");
      fprintf (outfile, "-----------------------------------");
      fprintf (outfile, "--------------------------\n\n");
      fprintf (outfile, "Legend:\n");
      fprintf (outfile, "   X = Maximum likelihood\n");
      fprintf (outfile, "   * = in approximative 50%% confidence limit\n");
      fprintf (outfile, "   + = in approximative 95%% confidence limit\n\n");
      fprintf (outfile, "   - = in approximative 99%% confidence limit\n\n");
      fprintf (outfile, "Inv(Alpha)      Ln(Likelihood)   Confidence limit\n");
      fprintf (outfile, "-------------------------------------------------\n");
      if (gmax < 0)
	{
	  fprintf (outfile, "%10.5g     % 20.5g            X\n", atl[loci + 1].param[nr->numpop2],
		   likes[20]);
	}
      for (a = 0; a < 19; a++)
	{
	  fprintf (outfile, "% 10.5f     % 20.5g            %c\n", invalphas[a],
		   likes[a], confidence[a]);
	  if (gmax == a)
	    fprintf (outfile, "% 10.5f     % 20.5g            X\n", atl[loci + 1].param[nr->numpop2],
		     likes[20]);
	}
      if (gmax >= 20)
	{
	  fprintf (outfile, "% 10.5g     % 20.5g            X\n", atl[loci + 1].param[nr->numpop2],
		   likes[20]);
	}
      destroy_nr (nr, world);
    }
}

void
create_loci_plot (world_fmt * world, char ***plane,
		  timearchive_fmt * atl, long loci)
{
  long intervals = world->options->plotintervals;
  long locus, pop, i, g = 1;
  nr_fmt *nr;
  double ***pl;
  double **contours;
  contours = (double **) calloc (1, sizeof (double *) * world->numpop);
  contours[0] = (double *) calloc (1, sizeof (double) * world->numpop * CONTOURLEVELS);
  for (pop = 1; pop < world->numpop; pop++)
    {
      contours[pop] = contours[0] + pop * CONTOURLEVELS;
    }
  set_contours (contours, world->numpop2, world->numpop);

  if(world->options->progress)
    {
      if (world->options->verbose)
	fprintf(stdout, "\n");
      print_menu_createplot ();
    }
  nr = (nr_fmt *) calloc (1, sizeof (nr_fmt));
  for (i = 1; i < loci + 1; i++)
    {
      if (g < atl[i].T)
	g = atl[i].T;
    }

  create_nr (nr, world, g, 0);
  for (locus = 1; locus < world->loci + 1; locus++)
    {
      create_apg0 (nr->apg0[locus], nr, &world->atl[locus]);
    }

  nr->skiploci = world->data->skiploci;
  pl = (double ***) calloc (1, sizeof (double **) * world->numpop);
  for (pop = 0; pop < world->numpop; pop++)
    {
      pl[pop] = (double **) calloc (1, sizeof (double *) * intervals);
      pl[pop][0] = (double *) calloc (1, sizeof (double) * 2 * intervals * intervals);
      for (i = 1; i < intervals; i++)
	{
	  pl[pop][i] = pl[pop][0] + i * 2 * intervals;
	}
    }
  calc_loci_plane (world, nr, atl, pl, loci, contours);
  for (pop = 0; pop < world->numpop; pop++)
    {
      fill_plotplane (world, plane[pop], pl[pop], contours[pop]);
    }
  destroy_nr (nr, world);
  print_mathematica (world, pl, intervals, intervals);
  for (i = 0; i < world->numpop; i++)
    {
      free (pl[i][0]);
      free (pl[i]);
    }
  free (pl);
  free (contours[0]);
  free (contours);
}


void
create_locus_plot (world_fmt * world, char ***plane, tarchive_fmt * tl, nr_fmt * nr, long G)
{
  long intervals = world->options->plotintervals;
  long i, pop;
  double ***pl;
  double **contours;
  if(world->options->progress)
    {
      if (world->options->verbose)
	fprintf(stdout,"\n");
      print_menu_createplot ();
    }
  pl = (double ***) calloc (1, sizeof (double **) * world->numpop);
  contours = (double **) calloc (1, sizeof (double *) * world->numpop);
  contours[0] = (double *) calloc (1, sizeof (double) * CONTOURLEVELS * world->numpop);
  for (pop = 0; pop < world->numpop; pop++)
    {
      pl[pop] = (double **) calloc (1, sizeof (double *) * intervals);
      pl[pop][0] = (double *) calloc (1, sizeof (double) * 2 * intervals * intervals);
      for (i = 1; i < intervals; i++)
	{
	  pl[pop][i] = pl[pop][0] + i * 2 * intervals;
	}
    }
  for (pop = 1; pop < world->numpop; pop++)
    {
      contours[pop] = contours[0] + pop * CONTOURLEVELS;
    }
  set_contours (contours, nr->numpop2, nr->numpop);
  calc_locus_plane (world, nr, tl, G, pl, contours);
  for (pop = 0; pop < world->numpop; pop++)
    {

      fill_plotplane (world, plane[pop], pl[pop], contours[pop]);

    }
  print_mathematica (world, pl, intervals, intervals);
  for (i = 0; i < world->numpop; i++)
    {
      free (pl[i][0]);
      free (pl[i]);
    }
  free (contours[0]);
  free (contours);
  free (pl);
}

void
set_contours (double **contours, long df, long numpop)
{
  long pop, i;
  contours[0][0] = contours[0][4] = 0.0;
  contours[0][1] = contours[0][5] = -0.5 * find_chi (df, 0.5);
  contours[0][2] = contours[0][6] = -0.5 * find_chi (df, 0.05);
  contours[0][3] = contours[0][7] = -0.5 * find_chi (df, 0.01);
  for (pop = 1; pop < numpop; pop++)
    {
      for (i = 0; i < CONTOURLEVELS; i++)
	{
	  contours[pop][i] = contours[0][i];
	}
    }
}

void
cleanup_world (world_fmt * world, long locus)
{
    if (locus >= 0)
    {
        if (strchr(SEQUENCETYPES, world->options->datatype))
            free_seqx (world->root, world->data->seq->endsite);
        else
            free_x (world->root);
        free_tree (world->root);
        free (world->nodep);
    }
}
    

void
apocalypse_now (world_fmt * world)
{
  free (world);
  raise (SIGILL);
}

/* printing stuff =========================================== */
void
print_menu_locus (option_fmt * options, long locus)
{
  if (options->progress)
    fprintf (stdout, "Locus %-3li:\n", locus + 1);
}

void
print_menu_chain (char type, long chain, long steps, world_fmt * world)
{
  long i,j,k,sum;
  char strllike[LINESIZE];
  char nowstr[LINESIZE];
  char spacer[] = "           ";
  double value;
  double mini,maxi;
  if (world->options->progress)
    {
      print_llike (world->param_like, strllike);
      get_time (nowstr, "%H:%M:%S");
      if (chain == FIRSTCHAIN)
	{
	  fprintf (stdout, "%s   Start conditions:\n", nowstr);
	  print_param (stdout, world->param0, world->numpop, spacer);
	  fprintf (stdout, "%sStart-tree-log(L)=%f\n",
		   spacer, world->likelihood[0]);
	  prognose_time (nowstr, world, 0, spacer);
	}
      else
	{
	  fprintf (stdout, "%s   %*s chain %3li:\n",
		   nowstr, type == 's' ? 5 : 4, 
		   type == 's' ? "Short" : "Long", chain + 1);
	  fprintf (stdout, "%sln L(Param)=ln (P(G|Param)/P(G|Param0))=%s,\n",
                           spacer, strllike);
	  fprintf (stdout, "%sStart-tree-ln(L)=ln P(D|G)=%f\n",
		   spacer, world->likelihood[0]);
	  print_param (stdout, world->param0, world->numpop, spacer);
	  if (world->options->verbose)
	    {
	      mini =  DBL_MAX;
	      maxi = -DBL_MAX;
	      for(i=0;i<steps;i++)
		{
		  value = world->likelihood[i];
		  if(mini>value)
		    mini = value;
		  if(maxi<value)
		    maxi=value;
		}
	      fprintf (stdout, "           Sampled tree-log(L)={%f .. %f},\n",
		       mini, maxi);
	      fprintf(stdout,  "           Best of all visited trees =%f\n",world->maxdatallike);
	      /* report migration events*/
	      fprintf(stdout,  "           Average migration events per genealogy (x-axis=to, y=from)\n           ");
	      for(i=0;i<world->numpop;i++)
		{
		  for(j=0;j<world->numpop;j++)
		    {
		      if(i!=j)
			{
			  sum=0;
			  for(k=0;k<world->atl[0].T;k++)
			    sum += world->atl[0].tl[k].l[i*world->numpop+j]*
			      world->atl[0].tl[k].copies;
			  fprintf(stdout,"%6.2f ",((double) sum)/((double)steps));
			}
		      else
			  fprintf(stdout,"------- ");
		    }
		  fprintf(stdout,"\n           ");
		}
	      fprintf(stdout,"\n");
	    }
	  prognose_time (nowstr, world, steps, spacer);
	}
    }
}

void 
prognose_time (char *nowstr, world_fmt * world, long steps, char *spacer)
{
#ifndef NOTIME_FUNC
  time_t nowbin;
  time_t proposetime;
  struct tm *nowstruct;
  long increment = (steps == world->options->sincrement) ?
  world->options->sincrement : world->options->lincrement;
  if (world->treesdone == 0)
    {
      strcpy (nowstr, "indeterminate");
      world->treesdone += steps * increment + world->options->burn_in;
    }
  else
    {
      if (steps != 0)
	world->treesdone += steps * increment + world->options->burn_in;
      proposetime = (long) ((double) (time (&nowbin) -
				      world->starttime)) /
	((double) world->treesdone) * world->treestotal;
      if (nowbin != (time_t) - 1)
	{
	  nowbin = world->starttime + proposetime;
	  nowstruct = localtime (&nowbin);
	  strftime (nowstr, LINESIZE, "%c", nowstruct);
	}
      else
	{
	  strcpy (nowstr, "indeterminate");
	}
    }
  fprintf (stdout, "%sPrognosed date/time to finish sampling is\n%s  %s\n",
	   spacer, spacer, nowstr);
#endif
}

void
copy_time (world_fmt * world, timelist_fmt * ltl, long from, long to, long np)
{
  long T;
  tarchive_fmt *atl = world->atl[0].tl;
  
  if (from == -1)
    {
      ltl[0].copies = 1;
      T = ltl[0].T - 1;
      atl = world->atl[0].tl;
      atl[0].copies = ltl[0].copies;
      increase_timearchive (world, 0, 1, np);
      archive_timelist (&(atl[0]), ltl[0].tl, T, np, world);

      if(world->in_last_chain && world->options->mighist)
	{
	  world->mighistloci[world->locus].mighist[world->mighistloci[world->locus].mighistnum].copies = 1;
	  world->mighistloci[world->locus].mighistnum++;
	}
      return;
    }
  if (from == to)
    {
      ltl[0].copies += 1;
      atl[from].copies = ltl[0].copies;
      if(world->in_last_chain && world->options->mighist)
	{
	  world->mighistloci[world->locus].mighist[world->mighistloci[world->locus].mighistnum].copies++;
	}
    }
  else
    {
      T = ltl[0].T - 1;
      increase_timearchive (world, 0, to + 1, np);
      atl = world->atl[0].tl;
      atl[to].copies = ltl[0].copies = 1;
      world->likelihood[to] = world->likelihood[from];
      archive_timelist (&atl[to], ltl[0].tl, T, np, world);

      if(world->in_last_chain && world->options->mighist)
	{
	  world->mighistloci[world->locus].mighist[world->mighistloci[world->locus].mighistnum].copies = 1;
	  world->mighistloci[world->locus].mighistnum++;
	}

    }
}


/*private functions========================================== */
/* ---------------------------------------------------------
   creates memory for archive of timelists for each locus
   the "locus 0" is reserved for for the current locus, this
   fake locus is use to speed up the NR-estimation for all chains
   whereas the loci 1 .. n are used to save the results of the last
   chain for the combined estimate at the end of the program */
void
create_timearchive (timearchive_fmt ** atl, long loci, long samples, long numpop)
{
  long i, j;
  long nn = numpop * numpop;
  long numpop2 = numpop + numpop * (numpop - 1);
  (*atl) = (timearchive_fmt *) calloc (1, sizeof (timearchive_fmt) * (2 + loci));
  for (i = 0; i < loci + 2; i++)
    {
      (*atl)[i].param = (double *) calloc (1, numpop2 * sizeof (double));
      (*atl)[i].param0 = (double *) calloc (1, numpop2 * sizeof (double));
      (*atl)[i].lparam0 = (double *) calloc (1, numpop2 * sizeof (double));
      (*atl)[i].likelihood = (double *) calloc (1, (1 + samples) * sizeof (double));
      (*atl)[i].tl = (tarchive_fmt *) calloc (1, samples * sizeof (tarchive_fmt));
      (*atl)[i].T = (*atl)[i].allocT = samples;
      for (j = 0; j < samples; j++)
	{
	  (*atl)[i].tl[j].km = (double *) calloc (1, sizeof (double) * numpop);
	  (*atl)[i].tl[j].kt = (double *) calloc (1, sizeof (double) * numpop);
	  (*atl)[i].tl[j].p = (long *) calloc (1, sizeof (long) * numpop);
	  (*atl)[i].tl[j].l = (long *) calloc (1, sizeof (long) * nn);
	}
    }
}

void
increase_timearchive (world_fmt * world, long locus, long sample, long numpop)
{
  long i = locus, j, oldT = 0, size;
  long nn = numpop * numpop;
  if (sample >= world->atl[i].allocT)
    {
      oldT = world->atl[i].allocT;
      world->atl[i].allocT = MAX (sample + 1, world->atl[i].allocT +
				  world->atl[i].allocT / 4.);
      size = world->atl[i].allocT;
      world->atl[i].tl = (tarchive_fmt *) realloc (world->atl[i].tl,
					      size * sizeof (tarchive_fmt));
      if (i == 0)
	world->likelihood =
	  (double *) realloc (world->likelihood, sizeof (double) * (1 + size));
      world->atl[i].likelihood =
	(double *) realloc (world->atl[i].likelihood, sizeof (double) * (1 + size));
      for (j = oldT; j < world->atl[i].allocT; j++)
	{
	  world->atl[i].tl[j].km = (double *) calloc (1, sizeof (double) * numpop);
	  world->atl[i].tl[j].kt = (double *) calloc (1, sizeof (double) * numpop);
	  world->atl[i].tl[j].p = (long *) calloc (1, sizeof (long) * numpop);
	  world->atl[i].tl[j].l = (long *) calloc (1, sizeof (long) * nn);
	}
      world->atl[i].T = sample;
    }
  else
    {
      world->atl[i].T = sample;
    }
}

void
create_plotplane (world_fmt * world)
{
  short pop, locus, i;
  long plotsize = world->options->plotintervals;
  world->plane = (char ****) calloc (1, sizeof (char ***) * (world->loci + 1));
  world->plotmax = (plotmax_fmt **) calloc (1, sizeof (plotmax_fmt *) * (world->loci + 1));
  world->plotmax[0] = (plotmax_fmt *) calloc (1, sizeof (plotmax_fmt) * (world->loci + 1) * world->numpop);
  for (pop = 1; pop < world->loci + 1; pop++)
    {
      world->plotmax[pop] = world->plotmax[0] + pop * (world->numpop);
    }
  for (locus = 0; locus < world->loci + 1; locus++)
    {
      world->plane[locus] =
	(char ***) calloc (1, sizeof (char **) * world->numpop);
      for (pop = 0; pop < world->numpop; pop++)
	{
	  world->plane[locus][pop] =
	    (char **) calloc (1, sizeof (char *) * (plotsize + 3));
	  for (i = 0; i < plotsize + 3; i++)
	    {
	      world->plane[locus][pop][i] =
		(char *) calloc (1, sizeof (char) * plotsize + plotsize + 20);
	    }
	}
    }
}

void
create_cov (world_fmt * world)
{
  short locus, i;
  world->cov = (double ***) calloc (1, sizeof (double **) * (world->loci + 1));
  for (locus = 0; locus < world->loci + 1; locus++)
    {
      world->cov[locus] = (double **) calloc (1, sizeof (double *)
					      * (world->numpop2 + 1));
      world->cov[locus][0] = (double *) calloc (1, sizeof (double)
			     * (world->numpop2 + 1) * (world->numpop2 + 1));
      for (i = 1; i < world->numpop2 + 1; i++)
	{
	  world->cov[locus][i] = world->cov[locus][0] + i * (world->numpop2 + 1);
	}
    }
}

void
print_menu_equilib (option_fmt * options)
{
  char nowstr[LINESIZE];
  if (options->progress)
    {
      get_time (nowstr, "%H:%M:%S");
      fprintf (stdout, "%s   Equilibrate tree (first %i trees are not used)\n", nowstr,
	       options->burn_in);
    }
}


void
print_menu_heating (option_fmt * options)
{
  char nowstr[LINESIZE];
  if (options->progress)
    {
      get_time (nowstr, "%H:%M:%S");
      fprintf (stdout, "%s   Running %i genealogies for heating-ratio\n", nowstr,
	       MAX(TWOHUNDRED,options->burn_in));
    }
}

void
print_finish (world_fmt * world, long filepos)
{
  char nowstr[LINESIZE];
  if (world->options->progress)
    {
      get_time (nowstr, "%H:%M:%S");
      fprintf (stdout, "%s   Program finished\n", nowstr);
    }
  get_time (nowstr, "%c");
  if (nowstr[0] != '\0')
    {
      if (filepos > 0)
	fseek (world->outfile, filepos, SEEK_SET);
      fprintf (world->outfile, "         finished at %s\n", nowstr);
    }

}

void
archive_timelist (tarchive_fmt * atl, vtlist * tl, long T, long np, world_fmt *world)
{
  long j, i;
  double t;
  double line;
  mighist_fmt *aa=NULL;
  for (i = 0; i < np; i++)
    {
      line = tl[0].lineages[i];
      atl->km[i] = line * tl[0].age;
      atl->kt[i] = line * (line - 1) * tl[0].age;
      atl->p[i] = whichp (tl[0].from, tl[0].to, i);
    }
  memset (atl->l, 0, sizeof (long) * np * np);
  atl->l[tl[0].to * np + tl[0].from] += (tl[0].from != tl[0].to);
  if(world->in_last_chain && world->options->mighist)
    {
      aa = &(world->mighistloci[world->locus].mighist[world->mighistloci[world->locus].mighistnum]);
      aa->migevents = (migevent_fmt *) realloc(aa->migevents,sizeof(migevent_fmt)*T);
      aa->migeventsize=0;
      if(tl[0].from!=tl[0].to)
	{
	  aa->migevents[0][0] = tl[0].age;
	  aa->migevents[0][1] = (double ) tl[0].from;
	  aa->migevents[0][2] = (double ) tl[0].to;
	  aa->migeventsize++;
	}
    }
  for (j = 1; j < T; j++)
    {
      t = tl[j].age - tl[j - 1].age;
      for (i = 0; i < np; i++)
	{
	  line = (double) tl[j].lineages[i];
	  atl->km[i] += line * t;
	  atl->kt[i] += line * (line - 1.) * t;
	  atl->p[i] += whichp (tl[j].from, tl[j].to, i);
	}
      atl->l[tl[j].to * np + tl[j].from] += (tl[j].from != tl[j].to);
      if(world->in_last_chain)
	{
	  if(world->options->mighist && (tl[j].from != tl[j].to))
	    {
	      aa->migevents[aa->migeventsize][0] = tl[j].age;
	      aa->migevents[aa->migeventsize][1] = (double ) tl[j].from;
	      aa->migevents[aa->migeventsize][2] = (double ) tl[j].to;
	      aa->migeventsize++;
	    }
	}
    }
  if(world->in_last_chain && world->options->mighist)
    {
      aa->migevents = (migevent_fmt *) realloc(aa->migevents,sizeof(migevent_fmt)*aa->migeventsize);
    }
}


void
copy_timelist (tarchive_fmt * from, tarchive_fmt * to, long np)
{
  memcpy (to->km, from->km, sizeof (double) * np);
  memcpy (to->kt, from->kt, sizeof (double) * np);
  memcpy (to->p, from->p, sizeof (long) * np);
  memcpy (to->l, from->l, sizeof (long) * np * np);
}

long
whichp (long from, long to, long pop)
{
  if (from == to)
    {
      if (from == pop)
	return 1;
    }
  return 0;
}

long
whichl (long from, long to, long pop)
{
  if (from != to)
    {
      if (to == pop)
	return 1;
    }
  return 0;
}

/*
   long calc_T(timearchive_fmt * old, long steps)
   {
   long i, sum = 0;;

   for (i = 0; i < steps; i++) {
   sum += old->tl[i].copies;
   if (sum == steps)
   return i + 1;
   }
   return steps;
   }
 */
void
print_results (world_fmt * world)
{
  long pop;
  FILE *outfile;
  option_fmt *opt;
  char sch[10], lch[10], cva[50];
  opt = world->options;
  outfile = world->outfile;
  if (opt->schains == 1)
    strcpy (sch, "chain");
  else
    strcpy (sch, "chains");
  if (opt->lchains == 1)
    strcpy (lch, "chain");
  else
    strcpy (lch, "chains");
  fprintf (outfile, "\n\n");
  PAGEFEED;
  print_result_header ("MCMC estimates", world);
  for (pop = 0; pop < world->numpop; pop++)
    {
      print_result_population (pop, world);
    }
  fprintf (outfile, "\nComments:\n There were %li short %s (%li used trees out of ",
	   opt->schains, sch, opt->ssteps);
  fprintf (outfile, "sampled %li)\n", opt->sincrement * opt->ssteps);
  fprintf (outfile, "and %li long %s (%li used trees out of sampled %li)\n",
	   opt->lchains, lch, opt->lsteps, opt->lincrement * opt->lsteps);
  if (opt->gamma)
    {
      if (world->atl[world->loci + 1].param[world->numpop2] < 10e-9)
	strcpy (cva, "0");
      else
	sprintf (cva, "%f", sqrt (world->atl[world->loci + 1].param[world->numpop2]));
      fprintf (outfile, "With shape parameter Inv(alpha)=%g ([CV(mu)]^2; CV(mu)=%s,alpha=%g)\n",
	       world->atl[world->loci + 1].param[world->numpop2], cva, 1. / world->atl[world->loci + 1].param[world->numpop2]);
      fprintf (outfile, "[Maximization needed %li cycles of maximal %i,\n  Norm(first derivatives)=%f (Normal stopping criteria is < %f)]\n\n\n",
	       world->atl[world->loci + 1].trials, NTRIALS, world->atl[world->loci + 1].normd, LOCI_NORM *
	       (world->numpop2 + world->options->gamma ? 1 : 0));
    }
}

void
print_fst (world_fmt * world, double **fstparam)
{
  long pop;
  long loci = world->loci;
  FILE *outfile = world->outfile;
  if (loci < 40)
    {
      PAGEFEED;
    }
  if (world->options->fsttype == 'T')
    print_result_header ("FST estimates (Thetas are variable)\n   [Only used as start values for MCMC run]", world);
  else
    print_result_header ("FST estimates (Migration rates are variable)\n  [Only used as start values for MCMC run]", world);
  for (pop = 0; pop < world->numpop; pop++)
    {
      print_result_fst (pop, world);
    }
  fprintf (world->outfile, "\nComments:\n");
  fprintf (world->outfile, "(-) can not be estimated\n");
  fprintf (world->outfile, "(0/0 or x/0) Divisions by zero\n");
  fprintf (world->outfile,
	   "Maynard Smith, J. 1970. Population size, polymorphism, and the rate of\n");
  fprintf (world->outfile,
	   "    non-Darwinian evolution. American Naturalist 104:231-237\n");
  fprintf (world->outfile,
  "Nei, M., and M. Feldman 1972. Identity of genes by descent within and\n");
  fprintf (world->outfile,
       "    between populations under mutation and migration pressures.\n");
  fprintf (world->outfile,
	   "    Theoretical Population Biology 3: 460-465\n");
}

void
prepare_print_nu (double nu, char *str)
{
  if (nu <= -999)
    sprintf (str, "-");
  else
    sprintf (str, "% 12.6f", nu);
}

void
prepare_print_nm (double nm, double nmu, char *strllike)
{
  if ((fabs (nmu) > 10e-20) && (fabs (nm) > 10e-20))
    sprintf (strllike, "% 10.5f", nm * nmu);
  else
    {
      if ((fabs (nmu) < 10e-20) && (fabs (nm) < 10e-20))
	{
	  sprintf (strllike, "0/0");
	}
      else
	{
	  if ((fabs (nmu) < 10e-20) && (fabs (nm) > 10e-20))
	    sprintf (strllike, "% 10.5f/0", nm);
	  else
	    {
	      if ((fabs (nmu) > 10e-20) && (fabs (nm) < 10e-20))
		sprintf (strllike, "0");
	    }
	}
    }
}

void
print_menu_coalnodes (world_fmt * world, long G)
{
  long g, pop, minp = world->sumtips, maxp = 0;
  char ss[10];
  long **contribution;
  long nodenum = world->sumtips;
  tarchive_fmt *tl = world->atl[0].tl;
  if (world->options->verbose)
    {
      contribution = (long **) calloc (1, sizeof (long *) * nodenum);
      contribution[0] = (long *) calloc (1, sizeof (long) * nodenum * world->numpop);
      for (pop = 1; pop < world->numpop; pop++)
	{
	  contribution[pop] = contribution[0] + pop * nodenum;
	}
      for (g = 0; g < G; g++)
	{
	  for (pop = 0; pop < world->numpop; pop++)
	    {
	      contribution[pop][(long) tl[g].p[pop]] += tl[g].copies;
	    }
	}
      for (g = 0; g < nodenum; g++)
	{
	  for (pop = 0; pop < world->numpop; pop++)
	    {
	      if (maxp < g && contribution[pop][g] > 0)
		maxp = g;
	      if (minp > g && contribution[pop][g] > 0)
		minp = g;
	    }
	}
      fprintf (stdout, "           Coalescent nodes: ");
      for (g = minp; g < maxp + 1; g++)
	{
	  fprintf (stdout, "%2li ", g);
	}
      fprintf (stdout, "\n");

      for (pop = 0; pop < world->numpop; pop++)
	{
	  fprintf (stdout, "             population %3li: ", pop);
	  for (g = minp; g < maxp + 1; g++)
	    {
	      if (contribution[pop][g] == 0)
		{
		  strcpy (ss, "-");
		}
	      else
		{
		  if (contribution[pop][g] >= 100)
		    strcpy (ss, "*");
		  else
		    sprintf (ss, "%-6li", contribution[pop][g]);
		}
	      fprintf (stdout, "%2.2s ", ss);
	    }
	  fprintf (stdout, "\n");
	}
      free (contribution[0]);
      free (contribution);
    }
}


void
print_menu_createplot (void)
{
  char nowstr[LINESIZE];
  get_time (nowstr, "%H:%M:%S");
  fprintf (stdout, "%s   Creating data for plots\n", nowstr);
}


void
calc_loci_plane (world_fmt * world, nr_fmt * nr, timearchive_fmt * atl, double ***pl, long loci, double **contours)
{
  long intervals = world->options->plotintervals;
  long i, ii,j, jj, k, kk, m, offset, pop;
  double max1 = -DBL_MAX;
  double max2 = -DBL_MAX;
  double val;
  double *xvalues = world->options->plotxvalues;
  double *yvalues = world->options->plotyvalues;
  plotmax_fmt *plotmax;
  double *param;
  param = (double *) calloc (1, sizeof (double) * nr->numpop2);
  if (world->options->gamma)
    {
      nr->param[nr->numpop2] = world->atl[loci + 1].param[nr->numpop2];
    }
  memcpy (param, nr->param, sizeof (double) * nr->numpop2);
  for (pop = 0; pop < nr->numpop; pop++) //over all populations=======
    {
      if(pop>0 && nr->world->options->migration_model==ISLAND)
	break;
      plotmax = &nr->world->plotmax[loci][pop];
      memcpy (nr->param, param, sizeof (double) * nr->numpop2);
      offset = nr->numpop + pop * (nr->numpop - 1);
      for (i = 0; i < intervals; i++) // grid over theta==============
	{
#ifdef MAC
            eventloop ();
#endif
	  nr->param[pop] = yvalues[i];
	  if(nr->world->options->migration_model==ISLAND)
	    {
	      for(ii=0;ii<nr->numpop;ii++)
		nr->param[ii]=yvalues[i];
	    }
	  for (j = 0; j < intervals; j++) // grid over 4Nm or M======
	    {                             // immigration (left plot)
	      if(nr->world->options->plotvar==PLOT4NM)
		val = xvalues[j] / yvalues[i];
	      else
		val = xvalues[j];
	      nr->param[offset] = val;
	      if(nr->world->options->migration_model==ISLAND)
		{
		  for(ii=nr->numpop;ii<nr->numpop2;ii++)
		    nr->param[ii]=val;
		}
	      calc_loci_like (nr, atl, loci, world->options->gamma);
	      pl[pop][i][j] = nr->llike;
	      for (k = offset + 1; k < offset + nr->numpop - 1; k++)
		{
		  nr->param[k - 1] = param[k - 1];
		  nr->param[k] = val;
		  if(nr->world->options->migration_model==ISLAND)
		    {
		      for(ii=nr->numpop;ii<nr->numpop2;ii++)
			nr->param[ii]=val;
		    }
		  calc_loci_like (nr, atl, loci, world->options->gamma);
		  pl[pop][i][j] += nr->llike;
		}
	      nr->param[k - 1] = param[k - 1];
	      if (max1 < pl[pop][i][j])
		{
		  max1 = pl[pop][i][j];
		  plotmax->l1 = pl[pop][i][j];
		  plotmax->x1 = xvalues[j];
		  plotmax->y1 = yvalues[i];
		}
	    }	  
	  memcpy (nr->param, param, sizeof (double) * nr->numpop2);
	  // emmigration (right plot) only if not n-island==============
	  if(nr->world->options->migration_model!=ISLAND)
	    {
                    // for (i = 0; i < intervals; i++) //over all thetas========
                    //	{
		  nr->param[pop] = yvalues[i];
                      //  if(nr->world->options->migration_model==ISLAND)
                      // {
                      // for(ii=0;ii<nr->numpop;ii++)
                      //		nr->param[ii]=yvalues[i];
                      // }
		  for (j = 0; j < intervals; j++) //over all 4Nm or M======
		    {
		      jj = j + intervals;
		      if(nr->world->options->plotvar==PLOT4NM)
			val = xvalues[j] / yvalues[i];
		      else
			val = xvalues[j];
		      k=pop;
                          // if(nr->world->options->migration_model==ISLAND)
                          //	{
			  //for(ii=nr->numpop;ii<nr->numpop2;ii++)
			  //  nr->param[ii]=val;
                          //}
		      for (m = 0; m < nr->numpop; m++)
                      {
                          if (k != m)
                          {
                              kk = nr->numpop + ((nr->numpop - 1) * m + ((k > m) ? k - 1 : k));
			      nr->param[kk] = val;
			      calc_loci_like (nr, atl, loci, world->options->gamma);
			      if (nr->llike > -DBL_MAX)
                                  pl[pop][i][jj] += nr->llike;
			      nr->param[kk] = param[kk];
                          }
                      }
		      if (max2 < pl[pop][i][jj])
                      {
			  max2 = pl[pop][i][jj];
			  plotmax->l2 = pl[pop][i][jj];
			  plotmax->x2 = xvalues[j];
			  plotmax->y2 = yvalues[i];
                      } 
		    }
            }
          
//        }
        }
      contours[pop][0] += max1;
      contours[pop][1] += max1;
      contours[pop][2] += max1;
      contours[pop][3] += max1;
      contours[pop][4] += max2;
      contours[pop][5] += max2;
      contours[pop][6] += max2;
      contours[pop][7] += max2;
      max1 = max2 = -DBL_MAX;
    }
  
  free (param);

}


void
calc_locus_plane (world_fmt * world, nr_fmt * nr, tarchive_fmt * tl, long G,
		  double ***pl, double **contours)
{
  long intervals = world->options->plotintervals;
  long locus;
  long i, ii,j, jj, k, kk, m, offset, pop;
  double max1 = -DBL_MAX;
  double max2 = -DBL_MAX;
  double val;
  double *xvalues = world->options->plotxvalues;
  double *yvalues = world->options->plotyvalues;
  plotmax_fmt *plotmax;
  double *param;
  param = (double *) calloc (1, sizeof (double) * nr->numpop2);
  if(world->options->datatype=='g')
locus=1;
  else
    locus=0;
  memcpy (param, nr->param, sizeof (double) * nr->numpop2);
  /* population pop -> row of2 pictures with left: immigration into pop
     right emmigration out of pop
     immigration is calculated as the sum of all loglikelihoods of
     m_{k,pop}, emigration is the sum of m_{pop,{k,m}} which is moving
     over {intervals}.
   */
  for (pop = 0; pop < nr->numpop; pop++)
    {
      if(nr->world->options->migration_model==ISLAND && pop>0)
          break;
      plotmax = &nr->world->plotmax[nr->world->locus][pop];
      memcpy (nr->param, param, sizeof (double) * nr->numpop2);
      offset = nr->numpop + pop * (nr->numpop - 1);
      for (i = 0; i < intervals; i++)
      {
	  nr->param[pop] = yvalues[i];
	  if(nr->world->options->migration_model==ISLAND)
          {
	      for(ii=0;ii<nr->numpop;ii++)
                  nr->param[ii]=yvalues[i];
          }
          for (j = 0; j < intervals; j++)
          {
	      if(nr->world->options->plotvar==PLOT4NM)
                  val = xvalues[j] / yvalues[i];
	      else
                  val = xvalues[j];
	      nr->param[offset] = val;
	      if(nr->world->options->migration_model==ISLAND)
              {
		  for(ii=nr->numpop;ii<nr->numpop2;ii++)
                      nr->param[ii]=val;
              }
	      calc_like (nr, nr->param, nr->tl, nr->numg, locus);
	      pl[pop][i][j] = nr->llike;
	      if(nr->world->options->migration_model!=ISLAND)
              {
                  for (k = offset + 1; k < offset + nr->numpop - 1; k++)
                  {
		      nr->param[k - 1] = param[k - 1];
		      nr->param[k] = val;
                          //if(nr->world->options->migration_model==ISLAND)
                          //{
                          //		  for(ii=nr->numpop;ii<nr->numpop2;ii++)
                          //		    nr->param[ii]=val;
                          //		}
		      calc_like (nr, nr->param, nr->tl, nr->numg, 0);
		      pl[pop][i][j] += nr->llike;
                  }
              }
	      else
                  k=offset+1;
	      nr->param[k - 1] = param[k - 1];
	      if (max1 < pl[pop][i][j])
              {
		  max1 = pl[pop][i][j];
		  plotmax->l1 = pl[pop][i][j];
		  plotmax->x1 = xvalues[j];
		  plotmax->y1 = yvalues[i];
		}
	    }
	}
      memcpy (nr->param, param, sizeof (double) * nr->numpop2);
      if(nr->world->options->migration_model!=ISLAND)
	{
	  for (i = 0; i < intervals; i++)
	    {
	      nr->param[pop] = yvalues[i];
	      if(nr->world->options->migration_model==ISLAND)
		{
		  for(ii=0;ii<nr->numpop;ii++)
		    nr->param[ii]=yvalues[i];
		}
	      for (j = 0; j < intervals; j++)
		{
		  jj = j + intervals;
		  if(nr->world->options->plotvar==PLOT4NM)
		    val = xvalues[j] / yvalues[i];
		  else
		    val = xvalues[j];
		  pl[pop][i][jj] = 0.;
		  k=pop;
		  if(nr->world->options->migration_model==ISLAND)
		    {
		      for(ii=nr->numpop;ii<nr->numpop2;ii++)
			nr->param[ii]=val;
		    }
		  for (m = 0; m < nr->numpop; m++)
		    {
		      if (k != m)
			{
			  kk = nr->numpop + ((nr->numpop - 1) * m + ((k > m) ? k - 1 : k));
			  nr->param[kk] = val;
			  calc_like (nr, nr->param, nr->tl, nr->numg, 0);
			  pl[pop][i][jj] += nr->llike;
			  nr->param[kk]=param[kk];
			}
		    }
		  if (max2 < pl[pop][i][jj])
		    {
		      max2 = pl[pop][i][jj];
		      plotmax->l2 = pl[pop][i][jj];
		      plotmax->x2 = xvalues[j];
		      plotmax->y2 = yvalues[i];
		    }
		}
	      
	    }
	}
      contours[pop][0] += max1;
      contours[pop][1] += max1;
      contours[pop][2] += max1;
      contours[pop][3] += max1;
      contours[pop][4] += max2;
      contours[pop][5] += max2;
      contours[pop][6] += max2;
      contours[pop][7] += max2;
      max1 = max2 = -DBL_MAX;
    }
  free (param);
}

void
fill_plotplane (world_fmt * world, char **plane, double **pl, double *contours)
{
  long i, j, ii, jj, z, zz = 0;

  char line[100];
  long myval[PLANEBIGTICKS] = PLANEBIGTICKVALUES;
  long intervals = MAX(world->options->plotintervals,PLANESIZE);
  double delta, deltahalf;

  delta = (double) intervals/PLANESIZE;
  deltahalf = delta/2.;

  for (i = 0; i < PLANESIZE; i++)
    {
      if (i % 7)
	line[i] = '-';
      else
	line[i] = '+';
    }
  line[i] = '\0';
  sprintf (plane[0], "     +%s+    +%s+   ", line, line);
  for (i = 0; i < PLANESIZE; i++)
    {
      memset (plane[i + 1], ' ', sizeof (char) * (2 * PLANESIZE + 20));
      plane[i + 1][2 * PLANESIZE + 19] = '\0';
      if (!((i) % 7))
	{
	  sprintf (plane[i + 1], "  %2.0li +", myval[zz++]);
	  if (myval[zz - 1] == 0)
	    plane[i + 1][3] = '0';
	}
      else
	plane[i + 1][5] = '|';
      ii= (long)(delta * i + deltahalf);
      for (j = 0; j < PLANESIZE; j++)
	{
	  jj= (long)(delta * j + deltahalf);
	  if (pl[ii][jj] < contours[1])
	    {
	      if (pl[ii][jj] < contours[2])
		{
		  if (pl[ii][jj] < contours[3])
		    {
		      plane[i+1][j+6] = ' ';
		    }
		  else
		    {
		      plane[i+1][j+6] = '-';
		    }
		}
	      else
		{
		  plane[i+1][j+6] = '+';
		}
	    }
	  else
	    {
	      if (pl[ii][jj] < (contours[0] - EPSILON))
		{
		  plane[i+1][j+6] = '*';
		}
	      else
		{
		  plane[i+1][j+6] = 'X';
		}
	    }
	}
      if ((i) % 7)
	{
	  plane[i+1][j+ 6] = '|';
	  plane[i+1][j+11] = '|';
	}
      else
	{
	  plane[i+1][j+ 6] = '+';
	  plane[i+1][j+11] = '+';
	}
      for (j = PLANESIZE+12; j < 2*PLANESIZE+12; j++)
	{
	  z = (long)(delta * (j-12) + deltahalf);
	  if (pl[ii][z] < contours[5])
	    {
	      if (pl[ii][z] < contours[6])
		{
		  if (pl[ii][z] < contours[7])
		    {
		      plane[i+1][j] = ' ';
		    }
		  else
		    {
		      plane[i+1][j] = '-';
		    }
		}
	      else
		{
		  plane[i+1][j] = '+';
		}
	    }
	  else
	    {
	      if (pl[ii][z] < (contours[4] - EPSILON))
		{
		  plane[i+1][j] = '*';
		}
	      else
		{
		  plane[i+1][j] = 'X';
		}
	    }
	}
      if ((i) % 7)
	{
	  plane[i+1][j] = '|';
	  plane[i+1][j+ 1] = '\0';
	}
      else
	{
	  plane[i+1][j] = '+';
	  plane[i+1][j+ 1] = '\0';
	}
    }
  sprintf (plane[PLANESIZE + 1], "     +%s+    +%s+", line, line);
  sprintf (plane[PLANESIZE + 2], "      1      2      3      4      5      6      1      2      3      4      5      6");
}

void
print_mathematica (world_fmt * world, double ***plane, long x, long y)
{
  long i, j, pop;
  FILE *mathfile = world->mathfile;
  if (world->options->plot == 1 && world->options->plotmethod == PLOTALL)
    {
      for (pop = 0; pop < world->numpop; pop++)
	{
	  for (i = 0; i < x; i++)
	    {
	      for (j = 0; j < y - 1; j++)
		{
		  fprintf (mathfile, "%20.20g ", plane[pop][i][j]);
		}
	      fprintf (mathfile, "%20.20g\n", plane[pop][i][j]);
	    }
	  if(world->options->migration_model==ISLAND)
	    break;
	  for (i = 0; i < x; i++)
	    {
	      for (j = y; j < 2 * y - 1; j++)
		{
		  fprintf (mathfile, "%20.20g ", plane[pop][i][j]);
		}
	      fprintf (mathfile, "%20.20g\n", plane[pop][i][j]);
	    }
	}
    }
  fflush(mathfile);
}

void
print_cov (world_fmt * world, long numpop, long loci, double ***cov)
{
  FILE *outfile = world->outfile;
  long locus, skipped = 0;
#ifdef __GNU__
  double corr[world->numpop2 + 1];
#else
  double *corr;
  corr = (double *) calloc (1, sizeof (double) * (world->numpop2 + 1));
#endif
  fprintf (outfile, "\n\n");
  fprintf (outfile, "--------------------------------------------------------------------\n");
  fprintf (outfile, "MCMC estimation:\n");
  fprintf (outfile, "Covariance matrix(*)                         Correlation matrix\n");
  fprintf (outfile, "-------------------------------------------  -----------------------\n");
  for (locus = 0; locus < world->loci; locus++)
    {
      if (world->data->skiploci[locus+1])
	{
	  skipped++;
	  continue;
	}
      fprintf (outfile, "Locus %li:\n", locus + 1);
      print_cov_table (outfile, locus, world, corr, 0);
      fprintf (outfile, "\n");
    }
  fprintf (outfile, "\n\n");
  if (world->loci - skipped > 1)
    {
      if (world->options->gamma)
	{
	  fprintf (outfile, "Over all loci\n");
	  fprintf (outfile, "------------------------------------------------------  ");
	  fprintf (outfile, "----------------------------\n");
	  print_cov_table (outfile, locus, world, corr, 1);
	}
      else
	{
	  fprintf (outfile, "Over all loci\n");
	  fprintf (outfile, "-------------------------------------------  -----------------------\n");
	  print_cov_table (outfile, locus, world, corr, 0);
	}
    }
#ifndef __GNU__
  free (corr);
#endif
}

void
print_cov_table (FILE * outfile, long locus, world_fmt * world, double *corr, long addvar)
{
  long i, j;
  double denom, temp1, temp2;
  for (i = 0; i < world->numpop2 + addvar; i++)
    {
      for (j = 0; j < world->numpop2 + addvar; j++)
	{
	  temp1 = fabs (world->cov[locus][i][i]);
	  if (temp1 < DBL_EPSILON)
	    temp1 = DBL_EPSILON;
	  temp2 = fabs (world->cov[locus][j][j]);
	  if (temp2 < DBL_EPSILON)
	    temp2 = DBL_EPSILON;
	  denom = 0.5 * (log (temp1) + log (temp2));
	  if ((temp1 = exp (denom)) == 0.0)
	    corr[j] = 9.99;
	  else
	    corr[j] = world->cov[locus][i][j] / exp (denom);
	}
      for (j = 0; j < world->numpop2 + addvar; j++)
	{
#if TESTING
	  fprintf (outfile, "% 20.10f ", world->cov[locus][i][j]);	/*was 10.4 */
#else
	  fprintf (outfile, "% 10.4f ", world->cov[locus][i][j]);
#endif
	}
      for (j = 0; j < world->numpop2 + addvar; j++)
	{
	  if (corr[j] < 9.)
	    fprintf (outfile, "% 4.2f ", corr[j]);
	  else
	    fprintf (outfile, "  --  ");
	}
      fprintf (outfile, "\n");
    }
}


void
free_tree (node * p)
{
  if (!p->tip)
    {
      if (p->next->back != NULL)
	{
	  free_tree (p->next->back);
	}
      if (p->type != 'm' && p->next->next->back != NULL)
	{
	  free_tree (p->next->next->back);
	}
    }
  else
    {
      free (p->nayme);
    }
  switch (p->type)
    {
    case 'm':
      free_nodelet (p, 2);
      break;
    case 't':
      free_nodelet (p, 1);
      break;
    default:
      free_nodelet (p, 3);
      break;
    }
}

void
free_nodelet (node * p, long num)
{
  long i;
  node *q;
  switch ((short) num)
    {
    case 3:
      free (p->next->next);
    case 2:
      free (p->next);
    case 1:
      free (p);
      break;
    default:
      for (i = 0; i < num; i++)
	{
	  q = p->next;
	  free (p);
	  p = q;
	}
    }
}

void
free_seqx (node * p, long sites)
{
  long j;
  if (!p->tip)
    {
      if (p->next->back != NULL)
	{
	  free_seqx (crawlback (p->next), sites);
	}
      if (p->next->next->back != NULL)
	{
	  free_seqx (crawlback (p->next->next), sites);
	}
    }
  if (p->type == 'r')
    return;
  for (j = 0; j < sites; j++)
    {
      free (p->x.s[j]);
    }
}

void
free_x (node * p)
{
  if (!p->tip)
    {
      if (p->next->back != NULL)
	{
	  free_x (crawlback (p->next));
	}
      if (p->next->next->back != NULL)
	{
	  free_x (crawlback (p->next->next));
	}
    }
  if (p->type == 'r')
    return;
  free (p->x.a);
}


void
test_locus_like (nr_fmt * nr, double *param0, double *param1, long df, long locus, world_fmt * world,
	   long *maxwhich, long maxnum, boolean withhead, char *this_string)
{
  char *teststat, temp[LRATIO_STRINGS];
  double like1, like0, testval, chiprob;
  int length;
  char tmp[100];
  long i, part1len = 1, part2len = 1;
  long elem, w = 0, z = 0, pop;
  double normd = 0;
  long *which;
  double *values, *saveparam0, tparam;
  which = (long *) calloc (1, sizeof (long) * (world->numpop2 + 1));
  values = (double *) calloc (1, sizeof (double) * (world->numpop2 + 1));
  saveparam0 = (double *) malloc (sizeof (double) * (world->numpop2 + 1));
  memcpy (saveparam0, param0, sizeof (double) * (world->numpop2 + 1));
  nr->skiploci = world->data->skiploci;
  elem = nr->numpop2 = world->numpop2;
  nr->numpop = world->numpop;
  if (maxnum > 0)
    {
      for (i = 0; i < elem; i++)
	{
	  if (i != maxwhich[z])
	    {
	      which[w] = i;
	      values[w++] = param0[i];
	    }
	  else
	    {
	      if (maxnum > z + 1)
		z++;
	    }
	}
      broyden_driver (world->atl, world->loci,
		  world, world->cov[world->loci], world->plane[world->loci],
		      which, values, elem - maxnum, param0, &like0, &normd,PROFILE);
    }
  else
    {
      memcpy (nr->param, param0, sizeof (double) * elem);
      like0 = calc_like (nr, nr->param, nr->tl, nr->numg, locus);
    }
  memcpy (nr->param, param1, sizeof (double) * elem);
  like1 = calc_like (nr, nr->param, nr->tl, nr->numg, locus);
  testval = -2. * (like0 - like1);
  chiprob = probchi (df, testval);
  teststat = (char *) calloc (1, sizeof (char) * LRATIO_STRINGS);
  length = MAX (0, 5 - sprintf (tmp, "%i", ((int) param0[0])));
  sprintf (teststat, "Loc%3li:{%.*f", locus, length, param0[0]);
  for (i = 1; i < elem; i++)
    {
      if (i >= world->numpop)
	{
	  pop = (i - world->numpop) / (world->numpop - 1);
	  tparam = param0[i] * saveparam0[pop];
	}
      else
	tparam = param0[i];
      length = MAX (0, 5 - sprintf (tmp, "%i", ((int) tparam)));
      sprintf (temp, ",%.*f", length, tparam);
      strcat (teststat, temp);
    }
  sprintf (temp, "}=");
  strcat (teststat, temp);
  part1len = strlen (teststat);
  length = MAX (0, 5 - sprintf (tmp, "%i", ((int) param1[0])));
  sprintf (temp, "\n       {%.*f", length, param1[0]);
  strcat (teststat, temp);
  for (i = 1; i < elem; i++)
    {
      if (i >= world->numpop)
	{
	  pop = (i - world->numpop) / (world->numpop - 1);
	  tparam = param1[i] * param1[pop];
	}
      else
	tparam = param1[i];
      length = MAX (0, 5 - sprintf (tmp, "%i", ((int) tparam)));
      sprintf (temp, ",%.*f", length, tparam);
      strcat (teststat, temp);
    }
  sprintf (temp, "}");
  strcat (teststat, temp);
  part2len = strlen (teststat);

  if (withhead)
    {
      fprintf (world->outfile, "Likelihood ratio test\nNull-Hypothesis");
      part1len = MAX (part1len, part2len - part1len);
      fprintf (world->outfile, "%*.*s %-10.10s %-3.3s %-4.4s\n",
	   (int) part1len, (int) part1len, " ", "Test value", "DF", "Prob");
      for (i = 0; i < 79 /*part2len + 35 */ ; i++)
	fputc ('-', world->outfile);
      fprintf (world->outfile, "\n");
    }
  if (testval > 10000 && chiprob < EPSILON)
    sprintf (temp, " >10000.00");
  else
    sprintf (temp, "%10.2f", testval);
  fprintf (world->outfile, "%s %11.11s %3li %8.5f\n", teststat,
	   temp, df, chiprob);
  fprintf (world->outfile, "       [%s]\n", this_string);
  free (teststat);
  free (values);
  free (saveparam0);
  free (which);
}


void
test_loci_like (nr_fmt * nr, double *param0, double *param1, long df, long loci,
		world_fmt * world, long *maxwhich, long maxnum,
		boolean withhead, char *this_string)
{

  char *teststat, temp[LRATIO_STRINGS];
  double like1, like0, testval, chiprob;
  int length;
  char tmp[100];
  long i, g = world->atl[1].T, part1len = 1, part2len = 1;
  long elem, z = 0, w = 0, pop;
  double normd = 0;
  long *which;
  double *values, *saveparam0, tparam;
  which = (long *) calloc (1, sizeof (long) * (world->numpop2 + 1));
  values = (double *) calloc (1, sizeof (double) * (world->numpop2 + 1));
  saveparam0 = (double *) malloc (sizeof (double) * (world->numpop2 + 1));
  memcpy (saveparam0, param0, sizeof (double) * (world->numpop2 + 1));
  for (i = 1; i < loci + 1; i++)
    {
      if (g < world->atl[i].T)
	g = world->atl[i].T;
    }

  elem = world->options->gamma ? nr->numpop2 + 1 : nr->numpop2;
  nr->skiploci = world->data->skiploci;
  if (maxnum > 0)
    {
      for (i = 0; i < elem; i++)
	{
	  if (i != maxwhich[z])
	    {
	      which[w] = i;
	      values[w++] = param0[i];
	    }
	  else
	    {
	      if (maxnum > z + 1)
		z++;
	    }
	}
      broyden_driver (world->atl, world->loci,
		  world, world->cov[world->loci], world->plane[world->loci],
		      which, values, elem - maxnum, param0, &like0, &normd,PROFILE);
    }
  else
    {
      memcpy (nr->param, param0, sizeof (double) * elem);
      like0 = calc_loci_like (nr, world->atl, world->loci, world->options->gamma);
    }

  memcpy (nr->param, param1, sizeof (double) * elem);
  like1 = calc_loci_like (nr, world->atl, world->loci, world->options->gamma);
  testval = -2. * (like0 - like1);
  chiprob = probchi (df, testval);
  teststat = (char *) calloc (1, sizeof (char) * LRATIO_STRINGS);
  length = MAX (0, 5 - sprintf (tmp, "%i", ((int) param0[0])));
  sprintf (teststat, "All   :{%.*f", length, param0[0]);
  for (i = 1; i < elem; i++)
    {
      if (i >= world->numpop)
	{
	  pop = (i - world->numpop) / (world->numpop - 1);
	  tparam = param0[i] * param0[pop];
	}
      else
	tparam = param0[i];
      length = MAX (0, 5 - sprintf (tmp, "%i", ((int) tparam)));
      sprintf (temp, ",%.*f", length, tparam);
      strcat (teststat, temp);
    }
  sprintf (temp, "}=");
  strcat (teststat, temp);
  part1len = strlen (teststat);
  length = MAX (0, 5 - sprintf (tmp, "%i", ((int) param1[0])));
  sprintf (temp, "\n       {%.*f", length, param1[0]);
  strcat (teststat, temp);
  for (i = 1; i < elem; i++)
    {
      if (i >= world->numpop)
	{
	  pop = (i - world->numpop) / (world->numpop - 1);
	  tparam = param1[i] * param1[pop];
	}
      else
	tparam = param1[i];
      length = MAX (0, 5 - sprintf (tmp, "%i", ((int) tparam)));
      sprintf (temp, ",%.*f", length, tparam);
      strcat (teststat, temp);
    }
  sprintf (temp, "}");
  strcat (teststat, temp);
  part2len = strlen (teststat);

  if (withhead)
    {
      fprintf (world->outfile, "\n\n\nLikelihood ratio tests\n");
      part1len = MAX (part1len, part2len - part1len);
      fprintf (world->outfile, "%*.*s %-10.10s %-3.3s %-4.4s\n",
	   (int) part1len, (int) part1len, " ", "Test value", "Df", "Prob");
      for (i = 0; i < part1len + 35; i++)
	fputc ('-', world->outfile);
      fprintf (world->outfile, "\n");
    }

  if (testval > 10000 && chiprob < EPSILON)
    sprintf (temp, " >10000.00");
  else
    sprintf (temp, "%10.2f", testval);

  fprintf (world->outfile, "%s %11.11s %3li %8.5f\n", teststat,
	   temp, df, chiprob);
  fprintf (world->outfile, "       [%s]\n", this_string);
  free (teststat);
  free (values);
  free (saveparam0);
  free (which);

}

long
set_test_param (double *param, char *strp, world_fmt * world, long lrline, long locus, long *maxwhich, long *maxnum)
{
  long i = 0, z = 0, zz = 0, zzz = 0, df = 0;
  long offset = 0, limit = 0, pop;
  char *tmp, *ss;
  double *meanparam, mean;
  ss = (char *) calloc (1, sizeof (char) * LINESIZE);
  tmp = (char *) calloc (1, sizeof (char) * LINESIZE);
  *maxnum = 0;
  strcpy (ss, strp);
  if (world->loci - world->skipped > 1 && world->options->lratio->data[lrline].type == MEAN)
    meanparam = world->atl[world->loci + 1].param;
  else
    meanparam = world->atl[locus + 1].param;
  while (ss[zzz] != '\0')
    {
      tmp[i] = ss[zzz++];
      if (!(tmp[i] == ',' || tmp[i] == '\0' || tmp[i] == '\n' || tmp[i] == ';'))
	{
	  if (tmp[i] != ' ')
	    i++;
	}
      else
	{
	  tmp[i] = '\0';
	  i = 0;
	  switch (tolower (tmp[0]))
	    {
	    case 'x':
	      param[z] = meanparam[z];
	      maxwhich[(*maxnum)++] = z;
	      z++;
	      df++;
	      break;
	    case '*':
	      param[z] = meanparam[z];
	      z++;
	      break;
	    case 't':
	    case 'm':
	      zz = atol (tmp) - 1;
	      df++;
	      if (zz < 0)
		{
		  mean = 0.0;
		  offset = (z >= world->numpop) ? world->numpop : 0;
		  limit = (z >= world->numpop) ? world->numpop2 : world->numpop;
		  for (zz = offset; zz < limit; zz++)
		    mean += meanparam[zz];
		  mean /= limit - offset;
		  param[z] = mean;
		}
	      else
		{
		  param[z] = meanparam[zz];
		}
	      z++;
	      break;
	    default:
	      df++;
	      param[z] = MAX (atof (tmp), SMALLEST_THETA);
	      if (z >= world->numpop)
		{
		  pop = (z - world->numpop) / (world->numpop - 1);
		  param[z] /= param[pop];
		}
	      z++;
	      break;
	    }
	}
    }
  free(ss);
  free(tmp);
  return df;
}


void
print_CV (world_fmt * world)
{
  long i;
  long elem;
  char temp[100];
  fprintf (world->outfile, "\n\nVariance and coefficient of variance\n");
  if (world->loci - world->skipped > 1)
    {
      elem = world->numpop2 + (world->options->gamma ? 1 : 0);
      fprintf (world->outfile, "PARAM@ ");
      for (i = 0; i < elem; i++)
	{
	  fprintf (world->outfile, "%20.20g ", world->atl[world->loci + 1].param[i]);
	}
      fprintf (world->outfile, "\nVAR@   ");
      for (i = 0; i < elem; i++)
	{
	  fprintf (world->outfile, "%20.20g ", world->cov[world->loci][i][i]);
	}
      fprintf (world->outfile, "\nCV@    ");
      for (i = 0; i < elem; i++)
	{
	  strcpy (temp, "-");
	  if (world->cov[world->loci][i][i] >= 0)
	    {
	      sprintf (temp, "%20.20g", (sqrt (world->cov[world->loci][i][i])) /
		       world->atl[world->loci + 1].param[i]);
	    }
	  fprintf (world->outfile, "%-s ", temp);
	}
      fprintf (world->outfile, "\n");
    }
  else
    {				/* one locus */
      elem = world->numpop2;
      fprintf (world->outfile, "PARAM@ ");
      for (i = 0; i < elem; i++)
	{
	  fprintf (world->outfile, "%20.20g ", world->atl[1].param[i]);
	}
      fprintf (world->outfile, "\nVAR@   ");
      for (i = 0; i < elem; i++)
	{
	  fprintf (world->outfile, "%20.20g ", world->cov[0][i][i]);
	}
      fprintf (world->outfile, "\nCV@    ");
      for (i = 0; i < elem; i++)
	{
	  strcpy (temp, "-");
	  if (world->cov[0][i][i] >= 0)
	    {
	      sprintf (temp, "%20.20g", (sqrt (world->cov[0][i][i])) /
		       world->atl[1].param[i]);
	    }
	  fprintf (world->outfile, "%-s ", temp);
	}
      fprintf (world->outfile, "\n");
    }
}

void
print_param (FILE * file, double *param, long nn, char spacer[])
{
  long i, j, fpos;
  long tt = nn;
  long counter;
  fprintf (file, "%sMigration matrix with Theta values on diagonal\n%sParam=", spacer, spacer);
  for (i = 0; i < nn; i++)
    {
      fprintf (file, " ");
      counter = 0 ;
      for (j = 0; j < nn; j++)
	{
	  if (i != j)
	    fprintf (file, "% 7.5f", param[i] * param[tt++]);
	  else
	    fprintf (file, "% 7.5f", param[i]);
	  if (counter++> 10)
          {
              counter=0;
              fprintf (file, "\n%s         ", spacer);
          }
	}
      fpos = ftell (file);
      fseek (file, fpos - sizeof (char), SEEK_SET);
      fprintf (file, " \n%s      ", spacer);
    }
  fprintf (file, " \n");
}

/* print header
   ==============================================================================
   Titletext
   ==============================================================================
   Population     Loci  Ln(L)   Theta    4Nm
   [4Ne mu] xx     xx     xx     xx     xx     xx
   xx....
   -------------- ---- -------- -------- ----------------------------------------
 */
void
print_result_header (char *titletext, world_fmt * world)
{
  long p1, zz;

  char dline[] = "==============================================================================";
  char sline[] = "-------------- ---- -------- -------- ----------------------------------------";

  fprintf (world->outfile, "%s\n", dline);
  fprintf (world->outfile, "%s\n", titletext);
  fprintf (world->outfile, "%s\n", dline);
  fprintf (world->outfile, "Population [x] Loc.  Ln(L)   Theta    4Nm\n");
  fprintf (world->outfile, "                             [4Ne mu] ");
  zz = 38;
  for (p1 = 0; p1 < world->numpop; p1++)
    {
      zz = zz + 7;
      if (zz > 79)
	{
	  zz = 38;
	  fprintf (world->outfile, "\n                                      ");
	}
      fprintf (world->outfile, "%2li,x    ", p1 + 1);
    }
  fprintf (world->outfile, "\n%s\n", sline);
}

void
print_result_population (long pop, world_fmt * world)
{
  char popstring[LINESIZE];
  long skipped = 0, locus;
  if (world->options->readsum)
    {
      sprintf (popstring, "%2li: ", pop + 1);
    }
  else
    {
      sprintf (popstring, "%2li: %s", pop + 1, world->data->popnames[pop]);
    }
  fprintf (world->outfile, "%-14.14s ", popstring);
  for (locus = 1; locus < world->loci + 1; locus++)
    {
      if (world->data->skiploci[locus])
	{
	  skipped++;
	  continue;
	}
      fprintf (world->outfile, "%s%4li ", locus == 1 ? "" : "               ", locus);
      fprintf (world->outfile, "% 8.3f ", world->atl[locus].param_like);
      print_result_param (world->outfile, world->atl[locus].param, world->numpop, pop);
    }
  if (world->loci - skipped > 1)
    {
      fprintf (world->outfile, "                All ");
      fprintf (world->outfile, "% 8.3f ", world->atl[locus].param_like);
      print_result_param (world->outfile, world->atl[locus].param, world->numpop, pop);
    }
/*    fprintf(world->outfile,"%s\n",sline);     */
}


void
print_result_param (FILE * file, double *param, long numpop, long pop)
{
  long i;
  long linelen = 0;
  long offset = numpop + pop * (numpop - 1);
  double tmp=0;
  if (param[pop] <= SICK_VALUE)
    fprintf (file, "     -    ");
  else
    {
      if(param[pop]<0.0001 && param[pop]>0)
	fprintf (file, "%3.2e ", param[pop]);
      else
	fprintf (file, "%8.5f ", param[pop]);
    }
  for (i = offset; i < offset + numpop - 1; i++)
    {
      if(linelen > 4)
	{
	    fprintf (file, "\n                                      ");
	    linelen = 0;
	}
      linelen++;
      if (pop == i - offset)
	{
	    fprintf (file, "------- ");
	    linelen++;
	}
      if ((param[i] <= SICK_VALUE) || (param[pop] <= SICK_VALUE))
	fprintf (file, "    -    ");
      else
	{
	  tmp = param[pop] * param[i];
	  if(tmp <0.001 && tmp>0)
	    fprintf (file, "%3.2e ", tmp);
	  else
	    fprintf (file, "%7.4f ", tmp);
	}
    }
  if (pop == numpop - 1)
    fprintf (file, "-------");
  fprintf (file, "\n");
}

void
print_result_fst (long pop, world_fmt * world)
{

  char popstring[LINESIZE];
  long skipped = 0, locus;
  sprintf (popstring, "%2li: %s", pop + 1, world->data->popnames[pop]);
  fprintf (world->outfile, "%14.14s ", popstring);
  for (locus = 1; locus < world->loci + 1; locus++)
    {
      if (world->data->skiploci[locus])
	{
	  skipped++;
	  continue;
	}
      fprintf (world->outfile, "%s%4li ", locus == 1 ? "" : "               ", locus);
      fprintf (world->outfile, "   -    ");
      print_result_param (world->outfile, world->fstparam[locus - 1],
			  world->numpop, pop);
    }
  if (world->loci - skipped > 1)
    {
      fprintf (world->outfile, "                All ");
      fprintf (world->outfile, "   -    ");
      print_result_param (world->outfile, world->fstparam[locus - 1],
			  world->numpop, pop);
    }
/*    fprintf(world->outfile,"%s\n",sline);     */
}

void
print_simulation (world_fmt * world)
{
  long locus, skipped = 0;
  char spacer[] = " ";
  for (locus = 1; locus < world->loci + 1; locus++)
    {
      if (world->data->skiploci[locus])
	{
	  skipped++;
	  continue;
	}
      print_param (world->outfile, world->atl[locus].param, world->numpop, spacer);
    }
  if (world->loci - skipped > 1)
    {
      print_param (world->outfile, world->atl[locus].param, world->numpop, spacer);
    }
}

void
print_mighist(world_fmt *world)
{
  long l,i,j;
  FILE *mighist = world->mighistfile;
  mighistloci_fmt *aa;
  if(world->options->mighist)
    {
      fprintf(mighist,"mighist={");
      for(l=0;l<world->loci;l++)
	{
	  aa = &world->mighistloci[l];
	  fprintf(mighist,"{");
	  for(j=0;j<aa->mighistnum;j++)
	    {
	      fprintf(mighist,"{%li,%f,{",aa->mighist[j].copies,
                      aa->mighist[j].weight);
	      for(i=0;i<aa->mighist[j].migeventsize;i++)
		{
		  fprintf(mighist,"{%f,%f,%f}",aa->mighist[j].migevents[i][0],
			  aa->mighist[j].migevents[i][1],
                          aa->mighist[j].migevents[i][2]);
		  if(i<aa->mighist[j].migeventsize-1)
		    fprintf(mighist,",\n");
		  else
		    fprintf(mighist,"}\n");
		}
	      if(j<aa->mighistnum-1)
		fprintf(mighist,"},");
	      else
		fprintf(mighist,"}}");	    
	    }
	  if(l<world->loci-1)
	    fprintf(mighist,",\n");
	  else
	    fprintf(mighist,"}\n");
	}
    }
}





