/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 M A X I M I Z E R      R O U T I N E S

 - calculates single locus ML
 - mulitlocus ML with constant mutation rate
 - multilocus ML with gamma deviated mutation rate 

  using Broyden-Fletcher-Goldfarb-Shanno method
 

 Peter Beerli 1996-1998, Seattle
 beerli@genetics.washington.edu

 $Id: combroyden2.c,v 1.46 2000/07/26 23:03:20 beerli Exp $

-------------------------------------------------------*/

#include "migration.h"
#include "broyden.h"
#include "joint-chains.h"
#include "world.h"
#include "integrate.h"
#include "derivatives.h"
#include "derivatives2.h"

#include "reporter.h"
#include "tools.h"

#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif


/* global variable CALC_LIKE() points to calc_locus_like() or 
   norm_constant()*/
double (*calc_like) (nr_fmt *, double *, long);


/* prototypes ------------------------------------------- */
/* public function used in main.c */
/*long estimateParameter(long locus, long rep, long chain, char type, 
		       world_fmt *world, long kind);
		      estimateParameter (rep, EARTH->G+1, EARTH,
					 EARTH->cov[locus], chain,
					 type, EARTH->plane[locus], 
					 SINGLELOCUS, EARTH->repkind);
*/
long estimateParameter (long rep, long G,
			world_fmt * world, double **dd, long chain, char type,
			char ***plane, long kind, long repkind);


/* calculates likelihood */
double calc_loci_like (helper_fmt *helper, 
				      boolean multilocus, boolean boolgamma);
/* first derivatives for ML with constant mutation rate */
void simple_loci_derivatives (nr_fmt * nr,
			      timearchive_fmt ** tyme, double *param, long loci);
/* progress-reporter */
void print_menu_finalestimate (boolean progress, long locus, long loci);
/* maximizer routine */
long broyden_driver (timearchive_fmt ** tyme, long loci,
		     world_fmt * world, double **covariance, char ***plane,
		     long *profiles, double *value, long profilenum,
		     double *profparam, double *proflike, double *profnorm,
		     long kind, long rep, long repkind);
/* start parameter for multilocus estimator, creates simple average */
void calc_meanparam (world_fmt * world, long n, long repstart, long repstop);
/* multilocus first derivatives with or without gamma distrib. mut. rate */
void combine_gradient (nr_fmt * nr, helper_fmt *helper,
		       timearchive_fmt ** tyme, long loci, boolean boolgamma);
/* this helps to use the simple single locus material */
void copy_and_clear_d (nr_fmt * nr, double *d);
void add_back_d (nr_fmt * nr, double *d);
/* rescales p35narameter with theta1 for gamma calculation */
void set_gamma_param (double *paramn, double *paramo, double theta,
		      nr_fmt * nr);
/* calculates norm, this creates a stopping criteria */
double norm (double *d, long size);
/* used in multilocus  line search */
double psilo (double lamda, helper_fmt * helper);
/* used in integral for gamma distr. mutation rate */
double phi (double t, void *b);

/* calculates new multilocus parameter values */
void calc_loci_param (nr_fmt * nr, double *param, double lamda,
		      boolean boolgamma);
/* used to reset the BFSG stuff */
void reset_work (double **work, long nnn);

/* linesearch of BFGS
   and helper functions for this linesearch */
double line_search (double *dv, double *gxv,
		    double (*func) (double lamda, nr_fmt * nr), nr_fmt * nr,
		    double oldL);

void replace_with (int mode, double *a, double *b, double *c, double m,
		   double *la, double *lb, double *lc, double ll);
double psil (double (*func) (double lamda, nr_fmt * nr),
	     double a, double b, nr_fmt * nr, double *gxv, double *dv,
	     double oldL, double *val1, double *val2);
double quadratic_lamda (double lamda, double a, double b, double c, double la,
			double lb, double lc);


void print_reset_big_param (FILE * file, double average, double maxtheta,
			    long pop);

long do_profiles (world_fmt * world, char ***plane, double *likes,
		  double *normd, long kind, long rep, long repkind);

void correct_too_big_values (world_fmt * world, nr_fmt * nr);

void which_calc_like(long repkind);

void set_replicates(world_fmt *world, long repkind, long rep, 
	       long *repstart, long *repstop);

void  prepare_broyden(long kind,long repstart,long repstop, world_fmt *world,  
		boolean *multilocus, long *Gmax);

long multilocus_gamma_adjust(boolean multilocus, boolean boolgamma, 
			     world_fmt *world, long repstart, long repstop);
void profile_setup(long profilenum, long *profiles, world_fmt * world, 
		   double *value, long *nnn) ;

void setup_parameter0(world_fmt *world, nr_fmt *nr, 
  long repkind, long repstart, long repstop, long loci, long kind,
		 boolean multilocus);
void report_broyden(double newL, double normd, long trials, 
		    boolean boolgamma, nr_fmt *nr);
boolean guard_broyden(double newL, double oldL, long trials, double normd,
		   double *normd20);

/* public functions---------------------------------------------- */
/* driver for maximizing single locus, multilocus functions      */
long
estimateParameter (long rep, long G,
		   world_fmt * world, double **dd, long chain, char type,
		   char ***plane, long kind, long repkind)
     //timearchive_fmt * tyme
{
  long trials = 0;
  double normd = 0.;
  double likes = 0.;

  //if(repkind!=SINGLECHAIN)
  //  loci = world->thislocus+1;
  if (kind == MULTILOCUS)
    {
      world->locus=world->loci;
      print_menu_finalestimate (world->options->progress,				world->loci, world->loci);
    }
  if (strchr (world->options->custm, '0')
      || strchr (world->options->custm, 'c'))
    {
      trials = do_profiles (world, plane, &likes, &normd, kind, rep, repkind);
    }
  else
    {
      trials = broyden_driver (world->atl, world->locus,
			       world, world->cov[world->locus],
			       plane, NULL, NULL, 0, NULL, NULL, &normd, kind,
			       rep, repkind);
    }

  if (kind == SINGLELOCUS && world->options->progress)
    {
      print_menu_chain (type, chain, 
			world->atl[rep][world->locus].T, world, rep);
      fprintf (stdout, "           Maximization cycles needed: %li\n", trials);
      fprintf (stdout, "           Norm of first derivatives:  %f\n", normd);
      if(world->options->logfile)
	{
	  fprintf (world->options->logfile, 
		   "           Maximization cycles needed: %li\n", trials);
	  fprintf (world->options->logfile, 
		   "           Norm of first derivatives:  %f\n", normd);
	}
    }
  if (kind == MULTILOCUS && world->options->plotnow && world->loci > 1)
    {
      if((world->options->replicate && repkind!=SINGLECHAIN) ||
	 (!world->options->replicate && repkind==SINGLECHAIN))
	create_loci_plot (world, world->plane[world->loci], world->atl,
			  world->loci);
    }
  world->trials = trials;
  world->normd = normd;
  return trials;
}


/*---------------------------------------------------------------
  Maximizer using Broyden-Fletcher-Shanno-Goldfarb method
 
  TYME  is pointer to the timeliststructure (which is actually WORLD->ATL)
  LOCI  actual locus
  WORLD contains all necessary structures
  PLANE returns the contour plots
  PROFILES specifies which parameters to profile (PARAM[PROFILE[i]])
  VALUE the values of the profiled parameters
  PROFILENUM number of parameters to profile
  PROFPARAM returns the parameters at the ML of param[profile]
  PROFLIKE the log(likelihood) at the profile maximum
  PROFNORM the norm of the first derivatives of the profile-maximizer
  KIND of estiamtion: multilocus or singlelocus estimates
  REP #replicates (multiple..) or specific replicate_num (for singlechain) 
  REPKIND singlechain or multiple long chains or multiple run estimates
 */
long
broyden_driver (timearchive_fmt ** tyme, long loci,
		world_fmt * world, double **covariance, char ***plane,
		long *profiles, double *value, long profilenum,
		double *profparam, double *proflike, double *profnorm,
		long kind, long rep, long repkind)
{

  long repstart;
  long repstop; 

  boolean multilocus = FALSE;
  boolean boolgamma = (loci != world->loci) ? FALSE : world->options->gamma;
  long i, ii, trials;
  long Gmax = 1;
  helper_fmt helper;
  double two = -2.; 
  nr_fmt *nr;
  long nn, nnn;
  double oldL, newL, lamda, normd, normd20 = 1e20;
  double *xv, *xv0, **hess, *delta, *dv, *gxv, *gxv0, *gama;
  double *param;
  long *indeks;
  boolean stop=FALSE;
  nr = (nr_fmt *) calloc (1, sizeof (nr_fmt) * 1);
  set_replicates(world,repkind, rep, &repstart,&repstop);
  which_calc_like(repkind);//decides which likelihood function to use.
  prepare_broyden(kind,repstart,repstop, world,  &multilocus, &Gmax);
/* calculcate multilocus mean parameter */
  nnn = nn = multilocus_gamma_adjust(multilocus,boolgamma, world,
				     repstart, repstop);
  profile_setup(profilenum, profiles,world, value, &nnn);  
  create_nr (nr, world, Gmax, profilenum, loci, repkind, repstart);
  nr->profiles = profiles;
  indeks = (long *) calloc (1, sizeof (long) * (nnn>0 ? nnn : 1));
  nr->indeks = indeks;
  setup_parameter0(world, nr, repkind, repstart, repstop, loci, kind,
		   multilocus);
/* BEGIN allocation of BFGS stuff */
  param = (double *) malloc (sizeof (double) * nr->partsize);
  hess = nr->dd = covariance;
  reset_hess (hess, nnn);
  delta = nr->delta;
  gama = nr->gdelta;
  dv = nr->dv;
  gxv = nr->d;
  gxv0 = nr->od;
  xv0 = nr->oparam;
  xv = nr->param;
/*END BFGS allocation stuff */

/* profiling: filling of index array */
  if ((profilenum > 0) && profilenum < nr->partsize)
    {
      for (i = 0, ii = 0; i < nr->partsize; i++)
	{
	  if (!find (i, profiles, profilenum))
	    indeks[ii++] = i;
	}
    }
/* filling in of values and first gradients for maximization cycle*/
  fill_helper(&helper, world, nr, multilocus, boolgamma);
  oldL = calc_loci_like (&helper, multilocus, boolgamma);
  combine_gradient (nr, &helper, tyme, loci, boolgamma);
  memcpy (gxv0, gxv, sizeof (double) * nnn);
  memcpy (xv0, xv, sizeof (double) * nn);
  memcpy (dv, gxv, sizeof (double) * nnn);
  memset (delta, 0, sizeof (double) * nnn);
/* Maximization cycle ---------------------------------------------*/
  for (trials = 0; trials < NTRIALS; trials++)
    {
#ifdef MAC
      eventloop ();
#endif
      newL = -DBL_MAX;
      //  memcpy (param, nr->param, sizeof (double) * nr->partsize);
      //      if(boolgamma)
      //	lamda = 1.;
      //else
	lamda = calc_line (&helper, -10., 0.1, 10., 
			 world->locus >= world->loci ? psilo : psi);
      //memcpy (nr->param, param, sizeof (double) * nr->partsize);
      while ((newL < oldL || isnan(newL)) && fabs (lamda) > 10. * DBL_EPSILON)
	{
	  calc_loci_param (nr, nr->oparam, lamda, boolgamma);
	  fill_helper_denom(&helper);
	  newL = calc_loci_like (&helper, multilocus, boolgamma);
	  lamda /= two;
	}
      normd = norm (gxv, nnn);
      if (world->options->verbose && newL>oldL)
	report_broyden(newL,normd,trials, boolgamma,nr);
      stop=guard_broyden(newL, oldL, trials, normd,&normd20);
      if (normd < EPSILON /*LOCI_NORM*/ || stop)	
	break; 	// stopping criteria
      else if (oldL >= newL)
	lamda = 0;
      else
	oldL = newL;
      /* reset sec deri. matrix if lamda goes to 0 and retry */
      if (fabs (lamda) <= DBL_EPSILON * 10.)
	{
	  reset_hess (hess, nnn);
	  memcpy (dv, gxv, sizeof (double) * nnn);
	  continue;
	}
      memcpy (gxv0, gxv, sizeof (double) * nnn);
      combine_gradient (nr, &helper,tyme, loci, boolgamma);
      if ((profilenum > 0) && profilenum < nr->partsize)
	{
	  for (i = 0; i < nnn; i++)
	    {
	      ii = indeks[i];
	      delta[i] = log (xv[ii]) - log (xv0[ii]);
	      gama[i] = gxv[i] - gxv0[i];
	    }
	}
      else
	{
	  for (i = 0; i < nnn; i++)
	    {
	      delta[i] = log (xv[i]) - log (xv0[i]);
	      gama[i] = gxv[i] - gxv0[i];
	    }
	}
      calc_hessian (hess, nnn, delta, gama);
      calc_dv (dv, hess, gxv, nnn);
      memcpy (xv0, xv, sizeof (double) * nn);
    }
/* end maximizer cycle ------------------------------------*/

  memcpy (world->param0, nr->param, sizeof (double) * nn);
  correct_too_big_values (world, nr);	// resets huge thetas to average

  if (world->options->verbose)
      fprintf (stdout,"\n");
  
  switch (kind)
    {
    case MULTILOCUS:
      tyme[0][loci].param = (double *) realloc (tyme[0][loci].param,
						 sizeof (double) * nn);
      tyme[0][loci].param_like = newL;
      tyme[0][loci].normd = normd;
      tyme[0][loci].trials = trials;
      memcpy (tyme[0][loci].param, world->param0, sizeof (double) * nn);
      *profnorm = normd;
      convergence_check (world, world->options->verbose);	//Gelman's R check
      break;
    case SINGLELOCUS:
      world->param_like = newL;
      if(repkind!=SINGLECHAIN)
	{
	  tyme[world->repstop][world->locus].param_like = newL;
	  tyme[world->repstop][world->locus].normd = normd;
	  tyme[world->repstop][world->locus].trials = trials;
	  memcpy (tyme[world->repstop][world->locus].param, world->param0, sizeof (double) * nn);
	}
      else
	{
	  tyme[world->rep][world->locus].param_like = newL;
	  tyme[world->rep][world->locus].normd = normd;
	  tyme[world->rep][world->locus].trials = trials;
	  memcpy (tyme[world->rep][world->locus].param, world->param0, sizeof (double) * nn);
	}
      convergence_check (world, world->options->verbose);	//Gelman's R check
      if (
	  (!world->options->gelman
	   && world->param_like < world->options->lcepsilon
	   && world->options->plotnow && !world->options->simulation)
	  || (world->options->plotnow && world->options->gelman
	      && world->gelmanmaxR < GELMAN_MYSTIC_VALUE))
	{
	  if((world->options->replicate && repkind!=SINGLECHAIN) ||
	     (!world->options->replicate && repkind==SINGLECHAIN))
	    create_locus_plot (world, plane, nr->atl, nr, nr->atl[rep][0].T);
	}
      *profnorm = normd;
      break;
    case PROFILE:
      memmove (profparam, world->param0, sizeof (double) * nr->partsize);
      *proflike = newL;
      *profnorm = normd;
    }
  free (indeks);

  if (world->options->verbose && world->repkind==SINGLECHAIN)
    {
      if (kind == SINGLELOCUS)
	print_contribution (nr, nr->atl, nr->atl[world->rep][world->locus].T);
    }
  destroy_nr (nr, world);
  return trials;
}

/* calculates the likelihood over all loci for the new parameter set */
double
calc_loci_like (helper_fmt *helper, boolean multilocus, boolean boolgamma)
{
  long locus;
  nr_fmt *nr = helper->nr;
  world_fmt *world = helper->nr->world;
  double *param0;
  double logres = 0;
  param0 = (double *) malloc (sizeof (double) * nr->partsize);
  memcpy (param0, nr->param, sizeof (double) * nr->partsize);
  if (multilocus)
    {
      if (!boolgamma)
	{
	  /* mutation rate is constant log(L(loci)) = Sum[log(L(locus))) */
	  for (locus = 0; locus < world->loci; locus++)
	    {
	      if (nr->skiploci[locus])
		continue;
	      nr->locilikes[locus] =  (*calc_like) (nr, nr->param, locus) ;
	      logres += nr->locilikes[locus];
	    }
	}
      else
	{
	  /* mutation rate is Gamma deviated 
	     log(L(loci) = Sum[Integral[gammafunc[] exp[log[L(locus}]],
	     0,INF]
	   */
	  for (locus = 0; locus < world->loci; locus++)
	    {
	      if (nr->skiploci[locus])
		continue;
	      helper->locus = locus;
	      helper->nr->locilikes[locus] = calc_single_gammalike(helper);
	      logres += helper->nr->locilikes[locus];
	    }
	}
    }
  else	       /*->singlelocus*/
    logres +=  (*calc_like) (nr, nr->param, nr->world->locus);

  memcpy (nr->param, param0, sizeof (double) * nr->partsize);
  free (param0);
  nr->llike = logres;
  return logres;
}

double
phi (double t, void *b)
{
  double ll, alpha, beta;
  long locus;
  double weight;
  helper_fmt *helper;
  nr_fmt *nr;
  double *param;
  timearchive_fmt **atl;
  helper = (helper_fmt *) b;
  locus = (long) helper->locus;
  weight = (double) helper->weight;
  atl = (timearchive_fmt **) helper->atl;
  nr = (nr_fmt *) helper->nr;
  param = (double *) calloc (1, sizeof (double) * nr->partsize);
  alpha = 1./nr->param[nr->numpop2];
  beta = nr->param[0] / alpha;
  set_gamma_param (param, nr->param, t, nr);
  /* likelihood calculation */
  ll =  (*calc_like) (nr, param, locus);
  //if (ll > helper->ll)
  //  helper->ll = ll;
  ll = exp (-t / beta + (alpha - 1.) * log (t) + ll + weight);
  free (param);
  return ll;
}


/* private functions---------------------------------------------- */
/* derivatives */
void
combine_gradient (nr_fmt * nr, helper_fmt * helper,
		  timearchive_fmt ** tyme, long loci, boolean boolgamma)
{
  long locus;//, iii;
  long nnn = nr->partsize - nr->profilenum;
  memset (nr->d, 0, sizeof (double) * nnn);
  if (loci != nr->world->loci)
    {
      simple_loci_derivatives (nr, tyme, nr->param, nr->world->locus);
      force_symmetric_d (nr->world->options->migration_model, nr, nnn);
      grad2loggrad (nr, nr->d, 1., nnn);
    }
  else
    {
      if (!boolgamma)
	{
	  for (locus = 0; locus < loci; locus++)
	    {
	      if (nr->skiploci[locus])
		continue;
	      simple_loci_derivatives (nr, tyme, nr->param, locus);
	    }
	  force_symmetric_d (nr->world->options->migration_model, nr, nnn);
	  grad2loggrad (nr, nr->d, 1., nnn);
	}
      else
	{
          //dt2(helper);
	  dt (nr, tyme, loci);
	  force_symmetric_d (nr->world->options->migration_model, nr, nnn);
	  nr->param[nr->numpop2] = 1. / nr->param[nr->numpop2];
	  grad2loggrad (nr, nr->d, 1., nnn);
	  nr->param[nr->numpop2] = 1. / nr->param[nr->numpop2];
	  //for(iii=0;iii<nr->numpop2+1;iii++)
	  //  printf("d[%li]=%10.10f ",iii,nr->d[iii]);
	  //printf("\n");
	}
    }
}

void
simple_loci_derivatives (nr_fmt * nr,
			 timearchive_fmt ** tyme, double *param, long locus)
{
  static double *lparam;
  static double *d;
  static boolean done = FALSE;

  long g,r, copies;
  double *apg0;
  if (!done)
    {
      d = (double *) calloc (1, sizeof (double) * nr->numpop2);
      lparam = (double *) calloc (1, sizeof (double) * nr->numpop2);
      done = TRUE;
    }
  for (g = 0; g < nr->numpop2; g++)
    lparam[g] = log (nr->param[g]);
  copy_and_clear_d (nr, d);
  nr->PGC = 0;
  nr->apg_max = -DBL_MAX;
  for(r=nr->repstart;r<nr->repstop;r++)
    {
      apg0 = nr->apg0[r][locus];
      for (g = 0; g < tyme[r][locus].T; g++)
	{
	  nr->apg[r][g] = probG (nr->param, lparam, &tyme[r][locus].tl[g], nr) - 
	    apg0[g];
	  if (nr->apg[r][g] > nr->apg_max)
	    nr->apg_max = nr->apg[r][g];
	}
    }
  for(r=nr->repstart;r<nr->repstop;r++)
    {
      for (g = 0; g < tyme[r][locus].T; g++)
	{
	  copies = (g>0) ? tyme[r][locus].tl[g].copies : tyme[r][locus].tl[g].copies -1;
	  nr->apg[r][g] -= nr->apg_max;
	  nr->PGC += copies * exp (nr->apg[r][g]);
	}
      nr->atl = tyme;
      //      nr->numg = tyme[r][locus].T;
    }
  gradient (nr->d, nr,locus);
  for (g = 0; g < nr->partsize - nr->profilenum; g++)
    nr->d[g] /= nr->PGC;
  add_back_d (nr, d);
}

void
set_gamma_param (double *paramn, double *paramo, double theta, nr_fmt * nr)
{
  long i;
  paramn[0] = theta;
  for (i = 1; i < nr->numpop; i++)
    {
      paramn[i] = paramo[i] * theta / paramo[0];
    }
  for (i = nr->numpop; i < nr->numpop2; i++)
    {
      paramn[i] = paramo[i] * paramo[0] / theta;
    }
  paramn[i] = paramo[i];
}


void
copy_and_clear_d (nr_fmt * nr, double *d)
{
  memcpy (d, nr->d, sizeof (double) * nr->numpop2);
  memset (nr->d, 0, sizeof (double) * nr->numpop2);
}

void
add_back_d (nr_fmt * nr, double *d)
{
  long pop;
  for (pop = 0; pop < nr->numpop2; pop++)
    {
      nr->d[pop] += d[pop];
    }
}


void
print_menu_finalestimate (boolean progress, long locus, long loci)
{
  static char nowstr[STRSIZE];
  if (progress)
    {
      get_time (nowstr, "%H:%M:%S");
      if (locus < loci)
	fprintf (stdout, "%s   Parameter estimation for locus %li\n",
		 nowstr, locus);
      else
	fprintf (stdout, "%s   Parameter estimation over all loci\n", nowstr);
    }
}

void
print_reset_big_param (FILE * file, double average, double maxtheta, long pop)
{
  static char nowstr[STRSIZE];
  get_time (nowstr, "%H:%M:%S");

  fprintf (file, "%s   Theta for population %li was reset to the average\n",
	   nowstr, pop);
  fprintf (file, "          of %f, it exceeded the maximum of %f \n", average,
	   maxtheta);
}

void
calc_meanparam (world_fmt * world, long n, long repstart, long repstop)
{
  long r,pop, locus, loci = world->loci;


  memset (world->param0, 0, sizeof (double) * n);
  for(r=repstart; r< repstop; r++)
    {
      for (locus = 0; locus < loci; locus++)
	{
	  for (pop = 0; pop < n; pop++)
	    {
	      if (!world->data->skiploci[locus])
		world->param0[pop] += world->atl[r][locus].param[pop];
	    }
	}
    }
  for (pop = 0; pop < n; pop++)
    {
      world->param0[pop] /= (loci - world->skipped) * (repstop-repstart);
    }
}


double
norm (double *d, long size)
{
  int i;
  double summ = 0.;
  for (i = 0; i < (int) size; i++)
    {
      summ += d[i] * d[i];
    }
  return sqrt (summ);
}


double
psilo (double lamda, helper_fmt * helper)
{
  double like;
  calc_loci_param (helper->nr, helper->nr->oparam, lamda, helper->nr->world->options->gamma);
  fill_helper_denom(helper);
  like =  calc_loci_like (helper, helper->multilocus, helper->boolgamma);
  return -like;
}

void
calc_loci_param (nr_fmt * nr, double *param, double lamda, boolean boolgamma)
{
  long i, ii, z = 0;
  double alpha;
  double elem = (boolgamma ? nr->numpop2 : nr->partsize);

  for (i = 0, ii = 0; i < elem - nr->profilenum; i++)
    {
      ii = (nr->profilenum > 0) ? nr->indeks[z++] : i;
      //if(boolgamma)
      //	nr->param[ii] =
      //	  param[ii] -lamda * nr->dv[i];
      //else
      nr->param[ii] =
	param[ii] * exp ((MAX (-100, MIN (-lamda * nr->dv[i], 100))));
    }
  if (boolgamma)
    {
      ii = (nr->profilenum > 0) ? nr->indeks[z] : i;
      alpha =  1./param[nr->numpop2];
      alpha = alpha * exp ((MAX (-100, MIN (-lamda * nr->dv[i], 100))));
      nr->param[nr->numpop2] = 1./alpha;
    }
  param_all_adjust (nr, boolgamma);
}


void
reset_work (double **work, long nnn)
{
  long i, j;
  for (i = nnn; i < nnn + 5; i++)
    {
      for (j = 0; j < nnn + 1; j++)
	work[i][j] = 0.0;
    }
}



#define LAMBDA_ 1
#define M_      0

/* line_search returns LAMBDA (how far to jump in the search direction)
   needed in the the multilocus broyden-fletcher-shanno maximizer

   DV    search direction
   GXV   first derivatives of function to minimize
   FUNC  the function to minimize
   NR    all additional variables needed to calculate FUNC
   OLDL  -log(likelihood) of FUNC with LAMBDA=0 (previously calculated)

   needs functions: psil, replace_with, quadratic_lamda
 */
double
line_search (double *dv, double *gxv,
	     double (*func) (double lamda, nr_fmt * nr), nr_fmt * nr,
	     double oldL)
{
  long trials = 0;
  double lamda_, ql;
  double a, b, c;

  double ll, la, lb, lc;
  double m = 1.0;
  a = 0.0;
  b = 0.000001;
  c = 0.9;
  la = -oldL;
  lb = (*func) (b, nr);
  lc = (*func) (c, nr);
  while (trials++ < NTRIALS)
    {
      /*      printf ("*%3li> %f %f %f / %f %f %f\n", trials, a, b, c, la, lb, lc); */
      lamda_ =
	(c * c * (lb - la) + b * b * (la - lc) +
	 a * a * (lc - lb)) / (2. * (b * la - c * la - a * lb + c * lb +
				     a * lc - b * lc));
      if ((lamda_ <= 0.0) || (lamda_ >= m))
	{
	  if ((a == m || b == m || c == m))
	    return m;
	  else
	    {
	      ll = (*func) (m, nr);
//      replace_with(M_,&a,&b,&c,m,&la,&lb,&lc,ll);
	      replace_with (LAMBDA_, &a, &b, &c, m, &la, &lb, &lc, ll);
	      continue;
	    }
	}
      ll = (*func) (lamda_, nr);
      ql = quadratic_lamda (lamda_, a, b, c, la, lb, lc);
      if ((fabs (ll - MIN3 (la, lb, lc)) <= BIGEPSILON)
	  || (fabs (ll - ql) <= BIGEPSILON))
	return lamda_;
      else
	{
	  /*  if(((a<b<c) || (c<b<a)) && (lb < MIN(la,lc)))
	     return lamda_;
	     if(((b<a<c) || (c<a<b)) && (la < MIN(lb,lc)))
	     return lamda_;
	     if(((a<c<b) || (b<c<a)) && (lc < MIN(la,lb)))
	     return lamda_; */
	  replace_with (LAMBDA_, &a, &b, &c, lamda_, &la, &lb, &lc, ll);
	  m = MAX3 (a, b, c);
	}
    }
  return lamda_;
}

void
replace_with (int mode, double *a, double *b, double *c, double m,
	      double *la, double *lb, double *lc, double ll)
{
  double ma, mb, mc;
  if (mode == LAMBDA_)
    {
      ma = *la;
      mb = *lb;
      mc = *lc;
    }
  else
    {
      ma = ll - *la;
      mb = ll - *lb;
      mc = ll - *lc;
    }
  if (ma > mb)
    {
      if (ma > mc)
	{
	  *a = m;
	  *la = ll;
	}
      else
	{
	  *c = m;
	  *lc = ll;
	}
    }
  else
    {
      if (mb > mc)
	{
	  *b = m;
	  *lb = ll;
	}
      else
	{
	  *c = m;
	  *lc = ll;
	}
    }
}

double
psil (double (*func) (double lamda, nr_fmt * nr),
      double a, double b, nr_fmt * nr, double *gxv, double *dv, double oldL,
      double *val1, double *val2)
{
  long i;
  long nn = nr->partsize;
  double value = 0.0;

  if (a == 0.0 && b == 0)
    {
      for (i = 0; i < nn; i++)
	{
	  value += dv[i] /* gxv[i] */ ;
	}
      *val1 = -oldL;
      *val2 = -oldL;
      return value;
    }
  else
    {
      if (a == 0)
	value = (((*val2) = (*func) (b, nr)) - ((*val1) = -oldL)) / b;
      else
	{
	  if (b == 0)
	    value = (((*val2) = -oldL) - ((*val1) = (*func) (a, nr))) / (-a);
	  else
	    value =
	      (((*val2) = (*func) (b, nr)) -
	       ((*val1) = (*func) (a, nr))) / (b - a);
	}
      return value;
    }
}

double
quadratic_lamda (double lamda,
		 double a, double b, double c, double la, double lb, double lc)
{
  double alpha, beta, gama;
  double aa, bb, cc;
  aa = a * a;
  bb = b * b;
  cc = c * c;
  alpha =
    ((c - b) * la + (a - c) * lb +
     (b - a) * lc) / ((b - a) * (c - a) * (c - b));
  beta =
    ((cc - bb) * la + (aa - cc) * lb +
     (bb - aa) * lc) / ((b - a) * (b - c) * (c - a));
  gama =
    (b * cc * la - bb * c * la + aa * c * lb - a * cc * lb - aa * b * lc +
     a * bb * lc) / ((b - a) * (c - a) * (c - b));


  return alpha * lamda * lamda + beta * lamda + gama;

}


long
do_profiles (world_fmt * world, char ***plane, double *likes, double *normd,
	     long kind, long rep, long repkind)
{
  long *profiles;
  double *values;
  char *p;
  long z = 0;
  long trials;
  long i = 0;
  long loci = world->locus;


  profiles = (long *) calloc (1, sizeof (long) * world->numpop2);
  values = (double *) calloc (1, sizeof (double) * world->numpop2);
  p = world->options->custm2;
  while (*p != '\0')
    {
      if (*p == '0' || *p == 'c')
	{
	  profiles[z] = i;
	  values[z] = world->param0[i];
	  z++;
	}
      p++;
      i++;
    }
  trials = broyden_driver (world->atl, loci,
			   world, world->cov[loci],
			   plane, profiles,
			   values, z, world->param0, likes, normd, 
			   kind, rep, repkind);
  free (profiles);
  free (values);

  return trials;
}

void
correct_too_big_values (world_fmt * world, nr_fmt * nr)
{
  long pop, i;
  long ssum, nsum;

  for (pop = 0; pop < world->numpop; pop++)
    {
      if (world->param0[pop] >= BIGGEST_THETA)
	{
	  ssum = 0;
	  nsum = 0;
	  for (i = 0; i < world->numpop; i++)
	    {
	      if (world->param0[i] <= BIGGEST_THETA)
		{
		  ssum += world->param0[i];
		  nsum++;
		}
	    }
	  if (nsum > 0 && ssum != 0.0 )
	    world->param0[pop] = ssum / (double) nsum;
	  else
	    {
	      if(ssum==0.0)
		world->param0[pop] = SMALLEST_THETA;
	      else
		world->param0[pop] = BIGGEST_THETA;
	    }
	  print_reset_big_param (world->outfile, world->param0[pop],
				 BIGGEST_THETA, pop);
	  if(world->options->writelog)
	    print_reset_big_param (world->options->logfile, world->param0[pop],
				   BIGGEST_THETA, pop);
	  print_reset_big_param (stdout, world->param0[pop], BIGGEST_THETA,
				 pop);
	}
    }

  for (pop = 0; pop < world->numpop2; pop++)
    {
      if (world->param0[pop] >= BIGGEST_MIGRATION)
	world->param0[pop] = BIGGEST_MIGRATION;
      if (world->param0[pop] < SMALLEST_MIGRATION)
	world->param0[pop] = SMALLEST_MIGRATION;
    }
}

void which_calc_like(long repkind)
{
  switch(repkind)
    {
    case SINGLECHAIN:
      calc_like = (double (*)(nr_fmt *,double *, long)) calc_locus_like;
      break;
    case MULTIPLECHAIN:
    case MULTIPLERUN:
      //calc_like = (double (*)(nr_fmt *,double *, long)) calc_locus_like;
	    calc_like = (double (*)(nr_fmt *,double *, long)) norm_constant;
      break;
    default:
      calc_like = (double (*)(nr_fmt *,double *, long)) calc_locus_like;
      break;
    }
}


void 
set_replicates(world_fmt *world, long repkind, long rep, 
	       long *repstart, long *repstop)
{
  if(world->options->replicate)
    {
      *repstart = (repkind == SINGLECHAIN) ? rep : 0;
      *repstop = (repkind == SINGLECHAIN) ? rep+1 : world->repstop;
    }
  else
    {
      *repstart = 0; 
      *repstop=1;
    }
}



void  
prepare_broyden(long kind,long repstart,long repstop, world_fmt *world,  
		boolean *multilocus, long *Gmax)
{
  long r, locus;
  if (kind == SINGLELOCUS || (kind == PROFILE && world->loci==1))
    {
      *multilocus = FALSE;
      for(r=repstart;r<repstop;r++)
	{
	  if (*Gmax < world->atl[r][world->locus].T)
	    *Gmax = world->atl[r][world->locus].T;
	}
    }
  else
    {
      *multilocus = TRUE;
      for (locus = 0; locus < world->loci; locus++)
	{
	  for(r=repstart;r<repstop;r++)
	    {
	      if (*Gmax < world->atl[r][locus].T)
		*Gmax = world->atl[r][locus].T;
	    }
	}
    }
}


long multilocus_gamma_adjust(boolean multilocus, boolean boolgamma, 
			     world_fmt *world, long repstart, long repstop)
{
  long nn;
  if (multilocus)
    {
      nn = boolgamma ? world->numpop2 + 1 : world->numpop2;
      calc_meanparam (world, world->numpop2,repstart,repstop);
      if (boolgamma)
	{
	  world->param0 = (double *)
	    realloc (world->param0, sizeof (double) * (nn>0?nn:1));
	  world->param0[world->numpop2] = 1./START_ALPHA;
	}
    }
  else
    nn = world->numpop2;
  return nn;
}

  /* profiler setup-------------------------- 
     needs the passed in values generated in profiles setup (profile.c)
  */
void 
profile_setup(long profilenum, long *profiles, world_fmt * world, 
		   double *value, long *nnn) 
{
  long i;
  if (profilenum > 0)
    {
      *nnn -= profilenum;
      for (i = 0; i < profilenum; i++)
	world->param0[profiles[i]] = value[i];
    }
}


void 
setup_parameter0(world_fmt *world, nr_fmt *nr, 
  long repkind, long repstart, long repstop, long loci, long kind,
  boolean multilocus)
{
  long locus, r;
      if (multilocus)
	{
	  for (locus = 0; locus < loci ; locus++)
	    {
	      if(repkind==SINGLECHAIN)
		{
		  for(r=repstart; r < repstop; r++)
		    create_apg0 (nr->apg0[r][locus], nr, &world->atl[r][locus]);
		}
	      else
		{
		  if(kind!=PROFILE)
		    {
		      for(r=repstart; r < repstop; r++)
			create_apg0 (nr->apg0[r][locus], nr, 
				     &world->atl[r][locus]);
		      interpolate_like(nr, locus);
		    }
		  for(r=repstart; r < repstop; r++)
		    create_multiapg0 (nr->apg0[r][locus], nr, r, locus);
		}
	    }
	}
      else //single locus
	{
	  if(repkind==SINGLECHAIN)
	    {
	      for(r=repstart; r < repstop; r++)
		create_apg0 (nr->apg0[r][world->locus], nr, &world->atl[r][world->locus]);
	    }
	  else
	    {
	      if(kind!=PROFILE)
		{
		  for(r=repstart; r < repstop; r++)
		    create_apg0 (nr->apg0[r][world->locus], nr, 
				 &world->atl[r][world->locus]);
		  interpolate_like(nr, world->locus);
		}
	      for(r=repstart; r < repstop; r++)
		create_multiapg0 (nr->apg0[r][world->locus], nr, r, 
				  world->locus);
	    }
	}
}

void report_broyden(double newL, double normd, long trials, 
				boolean boolgamma, nr_fmt *nr)
{
  if(boolgamma)
    fprintf (stdout, 
	     "%li> Log(L)=%f Norm=%f Alpha=%f Theta_1=%f\n", 
	     trials, newL, normd, 1./nr->param[nr->numpop2],nr->param[0]);
  else
    fprintf (stdout, 
	     "%li> Log(L)=%f Norm=%f Theta_1=%f\n", 
	     trials, newL, normd, nr->param[0]);
}

boolean guard_broyden(double newL, double oldL, long trials, double normd,
		   double *normd20)
{
  if ((trials + 1) % 20 == 0)
    {
      if (fabs (normd - *normd20) < EPSILON)
	return TRUE;
      *normd20 = normd;
    }
  return FALSE;
}
