/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm   
 ------------------------------------------------------- 
 O P T I O N S   R O U T I N E S 

 creates options structures,
 reads options from parmfile if present

 prints options,
 and finally helps to destroy itself.
 
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: options.c,v 1.53 2000/07/26 23:03:20 beerli Exp $
-------------------------------------------------------*/
//#include <stdio.h>
#include <time.h>
#include "migration.h"
#include "fst.h"
#include "tools.h"

#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif

/* parmfile parameter specifications and keywords */
#define LINESIZE 1024
#define NUMBOOL 20
#define BOOLTOKENS {"menu","interleaved","print-data","mixplot",\
  "moving-steps","freqs-from-data","usertree", \
  "autocorrelation", "simulation","plot", "weights",\
  "read-summary","write-summary","mig-histogram", "heating", "print-fst",\
  "distfile","geofile","gelman-convergence", "randomtree"}
#define NUMNUMBER 37
#define NUMBERTOKENS {"ttratio","short-chains",\
 "short-steps","short-inc","long-chains",\
 "long-steps", "long-inc", "theta", \
 "nmlength","random-seed","migration","mutation",\
 "datatype", "categories","rates","prob-rates", \
 "micro-max", "micro-threshold", "delimiter","burn-in",\
 "infile", "outfile", "mathfile", "title", \
 "long-chain-epsilon","print-tree","progress","l-ratio",\
 "fst-type","profile","custom-migration","sumfile","short-sample",\
 "long-sample", "replicate","cpu","logfile"};



/* prototypes ------------------------------------------- */
void create_options (option_fmt ** options);
void init_options (option_fmt * options);
void get_options (option_fmt * options);
void set_param (world_fmt * world, data_fmt * data, option_fmt * options,
		long locus);
void set_profile_options (option_fmt * options);
void print_menu_options (option_fmt * options, world_fmt * world);
void print_options (FILE * file, option_fmt * options, world_fmt * world);

void decide_plot (option_fmt * options, long chain, long chains, char type);
void destroy_options (option_fmt * options);
void save_parmfile (option_fmt * options);
/* private functions */
boolean booleancheck (option_fmt * options, char *var, char *value);
long boolcheck (char ch);
boolean numbercheck (option_fmt * options, char *var, char *value);
void reset_oneline (option_fmt * options, long position);
void read_theta (option_fmt * options);
void read_mig (option_fmt * options);
char skip_space (option_fmt * options);
void read_custom_migration (FILE * file, option_fmt * options, char *value,
			    long customnumpop);
void synchronize_param (world_fmt * world, option_fmt * options);
void specify_migration_type (option_fmt * options);
void fillup_custm (long len, world_fmt * world, option_fmt * options);
void set_partmean_mig (long **mmparam, double *param, char *custm2, long migm,
		       long numpop2);
long scan_m (char *custm2, long start, long stop);
void set_plot (option_fmt * options);
void set_plot_values (double **values, double plotrange[],
		      long intervals, short type);
void set_grid_param(world_fmt *world, long gridpoints);
void print_arbitrary_migration_table(FILE *file, world_fmt *world);
void print_distance_table(FILE *file, world_fmt *world);

/*======================================================*/
void
create_options (option_fmt ** options)
{
  (*options) = (option_fmt *) calloc (1, sizeof (option_fmt));
}

void
init_options (option_fmt * options)
{
  long i;
  unsigned long timeseed;
  /* General options --------------------------------------- */
  options->nmlength = DEFAULT_NMLENGTH;
  options->popnmlength = DEFAULT_POPNMLENGTH;
  options->allelenmlength = DEFAULT_ALLELENMLENGTH;
  options->custm = (char *) calloc (1, sizeof (char) * (1 + NUMPOP));
  //options->custm2 = (char *) calloc (1, sizeof (char) * (1 + NUMPOP));
  options->symn = 0;
  options->sym2n = 0;
  options->zeron = 0;
  options->constn = 0;
  options->mmn = 0;
  /* input/output options ---------------------------------- */
  options->menu = TRUE;
  options->progress = TRUE;
  options->verbose = FALSE;
  options->writelog=FALSE;
  options->printdata = FALSE;
  options->usertree = FALSE;
  options->randomtree = FALSE;
  options->treeprint = NONE;
  options->printfst = FALSE;
  options->fsttype = THETAVARIABLE;
  fst_type (options->fsttype);
  options->plot = FALSE;
  options->plotmethod = PLOTALL;	/* outfile and mathematica file */
  options->plotvar = PLOT4NM;
  options->plotscale = PLOTSCALELOG;
  options->plotrange[0] = PLANESTART;	/* start x axis */
  options->plotrange[1] = PLANEEND;	/*end x axis */
  options->plotrange[2] = PLANESTART;	/*start y axis */
  options->plotrange[3] = PLANEEND;	/* end y axis */
  options->plotintervals = PLANEINTERVALS;
  options->simulation = FALSE;
  options->movingsteps = FALSE;
  options->acceptfreq = 0.1;
  options->mighist = FALSE;
  options->mixplot = FALSE;
  strcpy (options->infilename, INFILE);
  strcpy (options->outfilename, OUTFILE);
  strcpy (options->logfilename, LOGFILE);
  strcpy (options->mathfilename, MATHFILE);
  strcpy (options->sumfilename, SUMFILE);
  strcpy (options->treefilename, TREEFILE);
  strcpy (options->utreefilename, UTREEFILE);
  strcpy (options->catfilename, CATFILE);
  strcpy (options->weightfilename, WEIGHTFILE);
  strcpy (options->mighistfilename, MIGHISTFILE);
  strcpy (options->distfilename, DISTFILE);
  strcpy (options->geofilename, GEOFILE);
  strcpy (options->bootfilename, BOOTFILE);
  strcpy (options->title, "\0");
  options->lratio = (lratio_fmt *) calloc (1, sizeof (lratio_fmt));
  options->lratio->alloccounter = 1;
  options->lratio->data =
    (lr_data_fmt *) calloc (1,
			    sizeof (lr_data_fmt) *
			    options->lratio->alloccounter);
  options->lratio->data[0].value =
    (char *) calloc (1, sizeof (char) * LINESIZE);
  options->profile = ALL;
  options->profilemethod = 'p';
  options->df = 1;
  options->qdprofile = FALSE;
  options->printprofsummary = TRUE;
  options->printprofile = TRUE;
  options->profileparamtype = options->plotvar;

  /* data options ------------------------------------------ */
  options->datatype = 's';
  options->migration_model = MATRIX;
  options->thetag = (double *) calloc (1, sizeof (double) * NUMPOP);
  options->mg = (double *) calloc (1, sizeof (double) * NUMPOP);
  options->gamma = FALSE;
  /* EP data */
  options->dlm = '\0';
  /* microsat data */
  options->micro_threshold = MICRO_THRESHOLD;
  options->micro_stepnum = MAX_MICROSTEPNUM;
  /*sequence data */
  options->interleaved = FALSE;
  options->ttratio = (double *) calloc (1, sizeof (double) * 2);
  options->ttratio[0] = 2.0;
  options->freqsfrom = TRUE;
  options->categs = ONECATEG;
  options->rate = (double *) calloc (1, sizeof (double));
  options->rate[0] = 1.0;
  options->rcategs = 1;
  options->rrate = (double *) calloc (1, sizeof (double));
  options->probcat = (double *) calloc (1, sizeof (double));
  options->autocorr = FALSE;
  options->rrate[0] = 1.0;
  options->probcat[0] = 1.0;

  options->probsum = 0.0;

  options->lambda = 1.0;
  options->weights = FALSE;
  /* random number options --------------------------------- */
  options->autoseed = AUTO;
  options->autoseed = AUTO;
  //#ifndef MAC
  timeseed = (unsigned long) time (NULL) / 4;
  //#else
  //timeseed = (unsigned long) clock () / 4;
  //#endif
  options->inseed = (long) timeseed + 1;
  /* mcmc options ------------------------------------------ */
  options->thetaguess = FST;
  options->migrguess = FST;
  for (i = 0; i < NUMPOP; i++)
    {
      options->thetag[i] = 1.0;
      options->mg[i] = 1.0;
      options->custm[i] = '*';
    }
  options->custm[i] = '\0';
  options->custm2 = (char *) calloc (i+2,sizeof (char));
  strncpy (options->custm2, options->custm,i);
  options->numthetag = options->nummg = 0;
  options->schains = 10;
  options->sincrement = 20;
  options->ssteps = 500;
  options->lchains = 3;
  options->lincrement = 20;
  options->lsteps = 5000;
  options->burn_in = BURNINPERIOD;
  options->heating = 0;		/* no heating */
  options->heat[0]=COLD;
  options->heat[1]=WARM;
  options->heat[2]=HOT;
  options->heat[3]=VERYHOT;
  options->lcepsilon = LONGCHAINEPSILON;
  options->gelman = FALSE;
  options->pluschain = PLUSCHAIN;
  options->replicate = FALSE;
  options->replicatenum = 0;
  options->gridpoints = 0;
  /* genealogy summary options----------------------------------- */
  options->readsum = FALSE;
  options->writesum = FALSE;
  /*threading over loci*/
  options->cpu=1;
}

void
get_options (option_fmt * options)
{
  long counter = 0;
  long position = 0;
  char varvalue[LINESIZE];
  char parmvar[LINESIZE];
  char input[LINESIZE];
  char *p, *tmp;
  FILE *mixfile;

  options->parmfile = fopen (PARMFILE, "r");
  if (options->parmfile)
    {
      counter = 0;
      position = ftell (options->parmfile);
      while (fgets (input, LINESIZE, options->parmfile) != NULL)
	{
	  counter++;
	  if ((input[0] == '#') || isspace ((int) input[0]) || input[0] == ';')
	    continue;
	  else
	    {
	      if (!(isalnum ((int) input[0]) || strchr ("{}", input[0])))
		{
		  usererror ("The parmfile contains an error on line %li\n",
			     counter);
		}
	    }
	  if ((p = strchr (input, '#')) != NULL)
	    *p = '\n';
	  if (!strncmp (input, "end", 3))
	    break;
	  tmp = strtok (input, "=");
	  if(tmp!=NULL)
	    strcpy (parmvar, tmp);
	  else
	    {
	      warning("error in parmfile with %s\n",input);
	      continue;
	    }
	  if (!strncmp (parmvar, "theta", 5))
	    {
	      reset_oneline (options, position);
	      read_theta (options);
	      position = ftell (options->parmfile);
	      continue;
	    }
	  if (!strncmp (parmvar, "migration", 5))
	    {
	      reset_oneline (options, position);
	      read_mig (options);
	      position = ftell (options->parmfile);
	      continue;
	    }
	  tmp = strtok (NULL, "\n");
	  if(tmp!=NULL)
	    strncpy (varvalue, tmp, LINESIZE - 1);
	  if (!booleancheck (options, parmvar, varvalue))
	    {
	      if (!numbercheck (options, parmvar, varvalue))
		{
		  warning ("Inappropiate entry in parmfile: %s ignored\n",
			   input);
		}
	    }
	  position = ftell (options->parmfile);
	}
    }
  if (options->mixplot)
    {
      mixfile = fopen ("mixfile", "w");
      fprintf (mixfile, " ");
      fclose (mixfile);
    }
}

void
print_menu_options (option_fmt * options, world_fmt * world)
{
  if (options->numpop > world->numpop)
    usererror ("Inconsistency between your Menu/Parmfile and your datafile\n \
Check the number of populations!\n");
  if (options->progress)
    {
      print_options (stdout, options, world);
      if(options->writelog)
	print_options (options->logfile, options, world);
    }
}


void
print_options (FILE * file, option_fmt * options, world_fmt * world)
{
  long i, j, tt;
  char mytext[LINESIZE];
  char seedgen[LINESIZE], spacer[LINESIZE];
  char paramtgen[LINESIZE], parammgen[LINESIZE];
  if (options->datatype != 'g')
    {
      switch ((short) options->autoseed)
	{
	case AUTO:
	  strcpy (seedgen, "with internal timer");
	  strcpy (spacer, "  ");
	  break;
	case NOAUTOSELF:
	  strcpy (seedgen, "from parmfile");
	  strcpy (spacer, "      ");
	  break;
	case NOAUTO:
	  strcpy (seedgen, "from seedfile");
	  strcpy (spacer, "      ");
	  break;
	default:
	  strcpy (seedgen, "ERROR");
	  strcpy (spacer, " ");
	  break;
	}
      switch (options->thetaguess)
	{
	case OWN:
	  strcpy (paramtgen, "from guessed values");
	  break;
	case FST:
	  strcpy (paramtgen, "from the FST-calculation");
	  break;
	  //	case PARAMGRID:
	  //	  strcpy (paramtgen, "GRID values around a center");
	  //	  break;	  
	case RANDOMESTIMATE:
	  strcpy (paramtgen, "RANDOM start value from N(mean,std)");
	  break;
	default:
	  strcpy (paramtgen, "ERROR");
	  break;
	}
      switch (options->migrguess)
	{
	case OWN:
	  strcpy (parammgen, "from guessed values");
	  break;
	case FST:
	  strcpy (parammgen, "from the FST-calculation");
	  break;
	  //	case PARAMGRID:
	  //	  strcpy (parammgen, "GRID values around a center");
	  //	  break;	  
	case RANDOMESTIMATE:
	  strcpy (parammgen, "RANDOM start value from N(mean,std)");
	  break;
	default:
	  strcpy (parammgen, "ERROR");
	  break;
	}
    }
  fprintf (file, "Options in use:\n");
  fprintf (file, "---------------\n");
  switch (options->datatype)
    {
    case 'a':
      fprintf (file, "Datatype: Allelic data\n");
      break;
    case 'b':
      fprintf (file, "Datatype: Microsatellite data [Brownian motion]\n");
      break;
    case 'm':
      fprintf (file, "Datatype: Microsatellite data [Stepwise mutation]\n");
      break;
    case 's':
      fprintf (file, "Datatype: DNA sequence data\n");
      break;
    case 'n':
      fprintf (file, "Datatype: Single nucleotide polymorphism data\n");
      break;
    case 'u':
      fprintf (file, "Datatype: Single nucleotide polymorphism data (PANEL)\n");
      break;
    case 'g':
      fprintf (file, "Datatype: Genealogy summary of an older run\n");
      break;
    }
  if (options->datatype != 'g')
    {
      fprintf (file, "Random number seed (%s)%s%20li\n",
	       seedgen, " ", options->saveseed);
      fprintf (file, "Start parameters:\n   Theta values were generated ");
      fprintf (file, " %s\n", paramtgen);
      if (options->thetaguess == OWN)
	{
	  fprintf (file, "   Theta = ");
	  for (i = 0; i < options->numthetag - 1; i++)
	    {
	      fprintf (file, "%.5f,", options->thetag[i]);
	    }
	  fprintf (file, "%.5f\n", options->thetag[i]);
	}
      fprintf (file, "   M values were generated %s\n", parammgen);
      if (options->migrguess == OWN)
	{
	  tt = 0;
	  fprintf (file, "   4Nm-matrix: ");
	  if (options->nummg == 1)
	    {
	      fprintf (file, "%5.2f [all are the same]\n", options->mg[tt++]);
	    }
	  else
	    {
	      for (i = 0; i < world->numpop; i++)
		{
		  for (j = 0; j < world->numpop; j++)
		    {
		      if (i != j)
			fprintf (file, "%5.2f ", options->mg[tt++]);
		      else
			fprintf (file, "----- ");
		      if (j > 10)
			fprintf (file, "\n                ");
		    }
		  fprintf (file, "\n               ");
		}
	      fprintf (file, "\n");
	    }
	}
    }
  print_arbitrary_migration_table(file, world);
  print_distance_table(file,world);
  fprintf (file, "Gamma-distributed mutation rate %s\n",
	   options->gamma ? "is used" : "is not used");
  if (options->datatype != 'g')
    {
      fprintf (file, "Markov chain settings:\n");
      fprintf (file, "   Short chains (short-chains):         %20li\n",
	       options->schains);
      fprintf (file, "      Trees sampled (short-inc*samples):%20li\n",
	       options->sincrement * options->ssteps);
      fprintf (file, "      Trees recorded (short-sample):    %20li\n",
	       options->ssteps);
      fprintf (file, "   Long chains (long-chains):           %20li\n",
	       options->lchains);
      fprintf (file, "      Trees sampled (long-inc*samples): %20li\n",
	       options->lincrement * options->lsteps);
      fprintf (file, "      Trees recorded (long-sample):     %20li\n",
	       options->lsteps);
      if(options->replicate)
	{
	  if(options->replicatenum==0)
	    fprintf (file, "   Averaging over long chains\n");
	  else
	    fprintf (file, "   Averaging over replicates:           %20li\n",
	       options->replicatenum);
	}
      if (options->heating > 0)
	{
	  //	  if (options->heating == 1)
	    //	    fprintf (file, "   Heating scheme active (U shaped acceptance)\n");
	    fprintf (file, "   Heating scheme active\n      (chains with temperature %5.2f %5.2f %5.2f %5.2f)\n",
		     options->heat[0],options->heat[1],options->heat[2],options->heat[3]);
	    // else
	    //fprintf (file, "   Heating scheme active (V shaped acceptance)\n");
	}
      if (options->movingsteps)
	{
	  fprintf (file, "   Percentage of new genealogies:       %20.2f\n",
		   (double) options->acceptfreq);
	}
      if (options->burn_in > 0)
	{
	  fprintf (file, "   Number of discard trees per chain:   %20li\n",
		   (long) options->burn_in);
	}
      if (options->lcepsilon < LONGCHAINEPSILON)
	{
	  fprintf (file, "   Parameter-likelihood epsilon:        %20.5f\n",
		   options->lcepsilon);
	}
    }
  fprintf (file, "Print options:\n");
  if (options->datatype != 'g')
    {
      fprintf (file, "   Data file: %46.46s\n", options->infilename);
      fprintf (file, "   Output file: %44.44s\n", options->outfilename);
      fprintf (file, "   Print data: %45.45s\n",
	       options->printdata ? "Yes" : "No");
      switch (options->treeprint)
	{
	case NONE:
	  fprintf (file, "   Print genealogies: %38.38s\n", "No");
	  break;
	case ALL:
	  fprintf (file, "   Print genealogies: %38.38s\n", "Yes, all");
	  break;
	case LASTCHAIN:
	  fprintf (file, "   Print genealogies: %38.38s\n",
		   "Yes, only those in last chain");
	  break;
	case BEST:
	  fprintf (file, "   Print genealogies: %38.38s\n",
		   "Yes, only the best");
	  break;
	}
    }
  if (options->plot)
    {
      switch (options->plotmethod)
	{
	case PLOTALL:
	  sprintf (mytext, "Yes, to outfile and %s", options->mathfilename);
	  break;
	default:
	  strcpy (mytext, "Yes, to outfile");
	  break;
	}
      fprintf (file, "   Plot data: %-46.46s\n", mytext);
      fprintf (file,
	       "              Parameter: %s, Scale: %s, Intervals: %li\n",
	       options->plotvar == PLOT4NM ? "{Theta, 4Nm}" : "{Theta, M}",
	       options->plotscale == PLOTSCALELOG ? "Log10" : "Standard",
	       options->plotintervals);
      fprintf (file, "              Ranges: X-%5.5s: %f - %f\n",
	       options->plotvar == PLOT4NM ? "4Nm" : "M",
	       options->plotrange[0], options->plotrange[1]);
      fprintf (file, "              Ranges: Y-%5.5s: %f - %f\n", "Theta",
	       options->plotrange[2], options->plotrange[3]);
    }
  else
    {
      fprintf (file, "   Plot data: %-46.46s\n", "No");
    }
  switch (options->profile)
    {
    case NONE:
      strcpy (mytext, "No");
      break;
    case ALL:
      strcpy (mytext, "Yes, tables and summary");
      break;
    case TABLES:
      strcpy (mytext, "Yes, tables");
      break;
    case SUMMARY:
      strcpy (mytext, "Yes, summary");
      break;
    }
  fprintf (file, "   Profile likelihood: %-36.36s\n", mytext);
  if (options->profile != NONE)
    {
      switch (options->profilemethod)
	{
	case 'p':
	  fprintf (file, "             Percentile method\n");
	  break;
	case 'q':
	  fprintf (file, "             Quick method\n");
	  break;
	case 'f':
	  fprintf (file, "             Fast method\n");
	  break;
	case 'd':
	  fprintf (file, "             Discrete method\n");
	  break;
	case 's':
	  fprintf (file, "             Spline method\n");
	  break;
	default:
	  fprintf (file, "             UNKOWN method????\n");
	  break;
	}
      fprintf (file, "             with df=%li and for Theta and %s\n\n\n\n",
	       options->df, options->profileparamtype ? "M=m/mu" : "4Nm");
    }
}


void
set_param (world_fmt * world, data_fmt * data, option_fmt * options, long locus)
{
  long i, ii, j, iitest=0;
  switch (options->thetaguess)
    {
    case RANDOMESTIMATE:
      if (world->numpop < options->numpop)
	{
	  usererror
	    ("There is a conflict between your menu/parmfile\nand your datafile: number of populations\nare not the same\n");
	}
      for (i = 0; i < world->numpop; i++)
	{
	  world->param0[i] = rannor (options->thetag[0], options->thetag[1]);
	  while (world->param0[i] < 0)
	    world->param0[i] = rannor (options->thetag[0], options->thetag[1]);
	}
      break;
      //    case PARAMGRID:
    case OWN:
      if (world->numpop < options->numpop)
	{
	  usererror
	    ("There is a conflict between your menu/parmfile\nand your datafile: number of populations\nare not the same\n");
	}
      for (i = 0; i < world->numpop; i++)
	{
	  if (i < options->numthetag - 1)
	    ii = i;
	  else
	    ii = options->numthetag - 1;
	  if (options->thetag[ii] == 0.0)
	    world->param0[i] = SMALLEST_THETA;
	  else
	    {
	      world->param0[i] = options->thetag[ii];
	    }
	}
      break;

    case FST:
    default:
      for (i = 0; i < world->numpop; i++)
	{
	  if (world->fstparam[locus][i] > SMALLEST_THETA)
	    {
	      if (world->fstparam[locus][i] > 100)
		world->param0[i] = 1.0;
	      else
		world->param0[i] = world->fstparam[locus][i];
	    }
	  else
	    {
	      if (world->options->datatype == 's' ||
		  world->options->datatype == 'n' ||
		  world->options->datatype == 'u')
		{
		  world->param0[i] = 0.01;
		}
	      else
		{
		  world->param0[i] = 1.0;
		}
	    }
	}
      break;
    }
  switch (options->migrguess)
    {
    case RANDOMESTIMATE:
      for (i = world->numpop; i < world->numpop2; i++)
	{
	  world->param0[i] = rannor (options->mg[0], options->mg[1]);
	  while (world->param0[i] < 0)
	    world->param0[i] = rannor (options->mg[0], options->mg[1]);

	}
      break;
      //    case PARAMGRID:
    case OWN:
      for (i = 0; i < world->numpop; i++)
	{
	  for (j = 0; j < world->numpop; j++)
	    {
	      if(i==j)
		continue;
	      if ((iitest=mm2m(j,i,world->numpop)-world->numpop) < options->nummg)
		ii = iitest;
	      else
		ii = options->nummg - 1;
	      if (options->geo)
		{
		  world->param0[mm2m(j,i,world->numpop)] =
		    1./world->data->ogeo[j][i] * options->mg[ii] / world->param0[i];
		}
	      else
		{
		  world->param0[mm2m(j,i,world->numpop)] =
		    options->mg[ii] / world->param0[i];
		}
	    }
	}
      break;
    case SLATKIN:
    case FST:
    default:
      for (i = world->numpop; i < world->numpop2; i++)
	{
	  if (world->fstparam[locus][i] > 0)
	    {
	      if (world->fstparam[locus][i] > 100)
		{
		  world->param0[i] =
		    1.0 / world->param0[(i - world->numpop) / (world->numpop)];
		  if (world->param0[i] > 10000)
		    world->param0[i] = 10000;
		}
	      else
		world->param0[i] = world->fstparam[locus][i];
	    }
	  else
	    {
	      world->param0[i] =
		1.0 / world->param0[(i - world->numpop) / (world->numpop)];
	      if (world->param0[i] > 10000)
		world->param0[i] = 10000;
	    }

	}
      break;
    }
  synchronize_param (world, options);
  //perhaps to come  
  //if(options->thetaguess==PARAMGRID && options->migrguess==PARAMGRID)
  //set_grid_param(world,options->gridpoints);
}

void set_grid_param(world_fmt *world, long gridpoints)
{
  static long which=0;
  static long z=0;

  static double level=0.1;
  static 
  double bottom = -2.302585093;// => level =  0.1
  double top = 2.302585093; // => level = 10
  double len = top - bottom;
  double diff = len / (gridpoints-1.);
  
  if(z>=gridpoints)
   {
     z =0;
     which++;
   }
  level = exp(bottom + z * diff);
  world->param0[which] = world->param0[which] * level;
  z++;

}

void
synchronize_param (world_fmt * world, option_fmt * options)
{
  char type;
  long i, j, z, zz = 0, len;
  long ns = 0, ss = 0, ss2 = 0, xs = 0, migm = 0;
  boolean allsymmig = FALSE;
  boolean allsamesize = FALSE;
  boolean partmig = FALSE;
  double summ;
  len = strlen (options->custm);
  options->custm2 = (char *) realloc (options->custm2,
				      sizeof (char) * (world->numpop2 + 2));
  if (len < world->numpop2)
    {
      fillup_custm (len, world, options);
    }
  options->mmn = migm = scan_m (options->custm2, world->numpop, world->numpop2);
  for (i = 0; i < world->numpop; i++)
    {
      for (j = 0; j < world->numpop; j++)
	{
	  if (!(allsymmig && allsamesize))
	    {
	      type = options->custm[i * world->numpop + j];
	      switch (type)
		{
		case '*':
		  xs++;
		  break;
		case 's':	// M is symmetric
		  if (i != j)
		    {
		      z = world->numpop + i * (world->numpop - 1) + j -
			((j > i) ? 1 : 0);
		      zz = world->numpop + j * (world->numpop - 1) + i -
			((i > j) ? 1 : 0);
		      options->symparam = (twin_fmt *)
			realloc (options->symparam,
				 sizeof (twin_fmt) * (ss + 2));
		      options->symparam[ss][0] = zz;
		      options->symparam[ss++][1] = z;
		      options->symn = ss;
		      summ = (world->param0[z] + world->param0[zz]) / 2.;
		      world->param0[zz] = world->param0[z] = summ;
		    }
		  break;
		case 'S':	// 4Nm is symmetric, not completely
		  // implemented yet, -> derivatives.c
		  if (i != j)
		    {
		      z = world->numpop + i * (world->numpop - 1) + j -
			((j > i) ? 1 : 0);
		      zz = world->numpop + j * (world->numpop - 1) + i -
			((i > j) ? 1 : 0);
		      options->sym2param = (quad_fmt *)
			realloc (options->sym2param,
				 sizeof (quad_fmt) * (ss2 + 2));
		      options->sym2param[ss2][0] = zz;
		      options->sym2param[ss2][1] = z;
		      options->sym2param[ss2][2] = i;
		      options->sym2param[ss2++][3] = j;
		      options->sym2n = ss2;
		      summ = (world->param0[z] * world->param0[i] +
			      world->param0[zz] * world->param0[j]) / 2.;
		      world->param0[z] = summ / world->param0[i];
		      world->param0[zz] = summ / world->param0[j];
		    }
		  break;
		case 'C': 
		case 'c':
		  if (i != j)
		    z = world->numpop + i *
		      (world->numpop - 1) + j - ((j > i) ? 1 : 0);
		  else
		    z = i;
		  if (z < world->numpop)
		    world->param0[z] = options->thetag[z];
		  else
		    world->param0[z] = options->mg[z - world->numpop]/options->thetag[i];
		  
		  options->constparam = (long *)
		    realloc (options->constparam, sizeof (long) * (ns + 2));
		  options->constparam[ns++] = z;
		  options->constn = ns;
		  break;
		case '0':
		  if (i != j)
		    z = world->numpop + i *
		      (world->numpop - 1) + j - ((j > i) ? 1 : 0);
		  else
		    z = i;
		  if (i != j)
		    world->param0[z] = 0;
		  else
		    {
		      if (z < world->numpop)
			world->param0[z] = options->thetag[z];
		      else
			error("Confused in setting zeroes in custom-migration?!");
		    }
		  options->zeroparam = (long *)
		    realloc (options->zeroparam, sizeof (long) * (ns + 2));
		  options->zeroparam[ns++] = z;
		  options->zeron = ns;
		  break;
		case 'm':
		  summ = 0;
		  if (i == j)
		    {
		      if (!allsamesize)
			{
			  allsamesize = TRUE;
			  for (z = 0; z < world->numpop; z++)
			    {
			      summ += world->param0[z];
			    }
			  summ /= world->numpop;
			  for (z = 0; z < world->numpop; z++)
			    world->param0[z] = summ;
			}
		    }
		  else
		    {
		      if (migm < world->numpop2 - world->numpop)
			{
			  partmig = TRUE;
			}
		      else
			{
			  if (!allsymmig)
			    {
			      allsymmig = TRUE;
			      for (z = world->numpop; z < world->numpop2; z++)
				{
				  summ += world->param0[z];
				}
			      summ /= (world->numpop2 - world->numpop);
			      for (z = world->numpop; z < world->numpop2; z++)
				{
				  world->param0[z] = summ;
				}
			    }
			}
		    }
		  break;
		default:
		  error ("no defaults allowed in synchronize_param()\n");
		}
	    }
	}
    }
  if (partmig)
    {
      set_partmean_mig (&(options->mmparam), world->param0, options->custm2,
			migm, world->numpop2);
    }
  //--------gamma stuff
  if (options->gamma)
    {
      options->custm2[world->numpop2]='*';
      options->custm2[world->numpop2+1]='\0';
    }
  //--------gamma stuff
  if (allsymmig)
    {
      options->migration_model = ISLAND_VARTHETA;
      if (allsamesize)
	options->migration_model = ISLAND;
      return;
    }
  else
    {
      if (allsamesize && ns == 0 && ss == 0)
	options->migration_model = MATRIX_SAMETHETA;
      else
	{
	  if (xs < world->numpop2)
	    options->migration_model = MATRIX_ARBITRARY;
	  else
	    options->migration_model = MATRIX;
	}
    }
}


long
scan_m (char *custm2, long start, long stop)
{
  long i, summ = 0;
  for (i = start; i < stop; i++)
    {
      if ('m' == custm2[i])
	summ++;
    }
  return summ;
}

void
set_partmean_mig (long **mmparam, double *param, char *custm2, long migm,
		  long numpop2)
{
  long i, z = 0;
  double summ = 0;
  double start = sqrt (numpop2);
  (*mmparam) = (long *) realloc ((*mmparam), sizeof (long) * (migm + 2));

  for (i = start; i < numpop2; i++)
    {
      if (custm2[i] == 'm')
	{
	  summ += param[i];
	  (*mmparam)[z++] = i;
	}
    }
  summ /= migm;
  for (i = start; i < numpop2; i++)
    {
      if (custm2[i] == 'm')
	param[i] = summ;
    }
}


void
destroy_options (option_fmt * options)
{
  free (options->thetag);
  free (options->mg);
  free (options->ttratio);
  free (options->rate);
  free (options->probcat);
  free (options->rrate);
  /*  free(options->lratio->data[0]); */
  free (options->lratio->data);
  free (options->lratio);
  free (options->custm);
  if (options->plot)
    {
      free (options->plotxvalues);
      free (options->plotyvalues);
    }
  if (options->zeron > 0)
    free (options->zeroparam);
  if (options->constn > 0)
    free (options->constparam);
  if (options->symn > 0)
    free (options->symparam);
  if (options->sym2n > 0)
    free (options->sym2param);
  free (options);
}

void
decide_plot (option_fmt * options, long chain, long chains, char type)
{
  if (options->plot && (chain >= chains - 1) && (type == 'l'))
    options->plotnow = TRUE;
  else
    options->plotnow = FALSE;
}

void
set_plot (option_fmt * options)
{
  long intervals = options->plotintervals;
  double prangex[2];
  double prangey[2];
  if (!options->plot)
    return;
  prangex[0] = options->plotrange[0];
  prangex[1] = options->plotrange[1];
  prangey[0] = options->plotrange[2];
  prangey[1] = options->plotrange[3];


  options->plotxvalues = (double *) calloc (1, sizeof (double) * intervals);
  options->plotyvalues = (double *) calloc (1, sizeof (double) * intervals);
  set_plot_values (&options->plotxvalues, prangex,
		   options->plotintervals, options->plotscale);
  set_plot_values (&options->plotyvalues, prangey,
		   options->plotintervals, options->plotscale);
}

void
set_plot_values (double **values, double plotrange[],
		 long intervals, short type)
{
  long i;
  double diff = 0;
  double logstart = 0;
  (*values)[0] = plotrange[0];
  (*values)[intervals - 1] = plotrange[1];
  if (type == PLOTSCALELOG)
    {
      logstart = log10 ((*values)[0]);
      diff = (log10 ((*values)[intervals - 1]) - logstart) /
	(double) (intervals - 1);
      for (i = 1; i < intervals - 1; i++)
	{
	  (*values)[i] = pow (10., (logstart + i * diff));
	}
    }
  else
    {
      diff =
	((*values)[intervals - 1] - (*values)[0]) / (double) (intervals - 1);
      for (i = 1; i < intervals - 1; i++)
	{
	  (*values)[i] = (*values)[i - 1] + diff;
	}
    }
}

void
save_parmfile (option_fmt * options)
{
  FILE *fp;
  long num, i, j, z;
  char nowstr[LINESIZE] = "----";
  fp = options->parmfile;
  if (fp)
    {
      fclose (fp);
      fp = fopen (PARMFILE, "w");
    }
  else
    fp = fopen (PARMFILE, "w");
  get_time (nowstr, "%c");
  fprintf (fp, "#########################################################\n");
  fprintf (fp, "# Parmfile for Migrate %s\n", MIGRATEVERSION);
  fprintf (fp, "# generated automatically on\n# %s\n", nowstr);
  fprintf (fp, "#\n# please report problems to Peter beerli\n");
  fprintf (fp, "#    email: beerli@genetics.washington.edu\n");
  fprintf (fp, "#    http://evolution.genetics.washington.edu/lamarc.html\n");
  fprintf (fp, "#########################################################\n");
  fprintf (fp, "# General options ---------------------------------------\n");
  fprintf (fp, "nmlength=%li\n", options->nmlength);
  //  fprintf(fp, "popnmlength=%li\n",options->popnmlength);
  //fprintf(fp, "allelenmlength=%li\n",options->allelenmlength);
  fprintf (fp, "# data options ------------------------------------------\n");
  switch (options->datatype)
    {
    case 'a':
      fprintf (fp, "datatype=AllelicData\n");
      break;
    case 'b':
      fprintf (fp, "datatype=BrownianMicrosatelliteData\n");
      break;
    case 'm':
      fprintf (fp, "datatype=MicrosatelliteData\n");
      break;
    case 's':
      fprintf (fp, "datatype=SequenceData\n");
      break;
    case 'n':
      fprintf (fp, "datatype=NucleotidePolymorphismData\n");
      break;
    case 'u':
      fprintf (fp, "datatype=UnlinkedSNPData\n");
      break;
    case 'g':
      fprintf (fp, "datatype=GenealogySummaryOlderRun\n");
      break;
    }
  switch (options->datatype)
    {
    case 'a':
      fprintf (fp, "# Electrophoretic data options------------\n");
      /*  if(options->dlm!='\0')
         fprintf(fp,"delimiter=%c\n",options->dlm); */
      break;
    case 'm':
      fprintf (fp, "# Microsatellite data options------------\n");
      fprintf (fp, "micro-threshold=%li\n", options->micro_threshold);
      fprintf (fp, "micro-max=%li\n", options->micro_stepnum);
      /* fall through is necessary */
    case 'b':
      /*        fprintf(fp,"delimiter=%c\n",options->dlm); */
      break;
    case 's':
    case 'n':
    case 'u':

      fprintf (fp, "# Sequence data options------------\n");
      fprintf (fp, "ttratio=%f ", options->ttratio[0]);
      i = 1;
      while (options->ttratio[i] != 0.0)
	fprintf (fp, "%f ", options->ttratio[i++]);
      fprintf (fp, "\n");
      if (options->freqsfrom)
	fprintf (fp, "freqs-from-data=YES\n");
      else
	fprintf (fp, "freqs-from-data=NO:%f,%f, %f, %f\n",
		 options->freqa, options->freqc, options->freqg,
		 options->freqt);
      fprintf (fp, "categories=%li\n", options->categs);
      fprintf (fp, "rates=%li:", options->rcategs);
      for (i = 0; i < options->rcategs; i++)
	fprintf (fp, "%f ", options->rrate[i]);
      fprintf (fp, "\n");
      fprintf (fp, "prob-rates=%li:", options->rcategs);
      for (i = 0; i < options->rcategs; i++)
	fprintf (fp, "%f ", options->probcat[i]);
      fprintf (fp, "\n");
      if (!options->autocorr)
	fprintf (fp, "autocorrelation=NO\n");
      else
	fprintf (fp, "autocorrelation=YES:%f\n", 1. / options->lambda);
      fprintf (fp, "weights=%s\n", options->weights ? "YES" : "NO");
      fprintf (fp, "interleaved=%s\n", options->interleaved ? "YES" : "NO");
      fprintf (fp, "usertree=%s\n", options->usertree ? "YES" : "NO");
      if (options->randomtree)
	fprintf (fp, "randomtree=%s\n", "YES");
      fprintf (fp, "distfile=%s\n", options->dist ? "YES" : "NO");
      break;
      //    defaults:
      //      break;
    }
  fprintf (fp, "# input/output options ----------------------------------\n");
  fprintf (fp, "menu=%s\n", options->menu ? "YES" : "NO ");
  /*fprintf(fp,"simulation=%s\n",options->simulation ? "YES" : "NO"); */
  fprintf (fp, "# input formats\n");
  fprintf (fp, "infile=%s\n", options->infilename);
  switch (options->autoseed)
    {
    case NOAUTO:
      fprintf (fp, "random-seed=SEEDFILE");
      break;
    case AUTO:
      fprintf (fp, "random-seed=AUTO #OWN:%li\n", options->inseed);
      break;
    case NOAUTOSELF:
      fprintf (fp, "random-seed=OWN:%li\n", options->inseed);
      break;
    }
  if (strlen (options->title) > 0)
    fprintf (fp, "title=%s\n", options->title);
  fprintf (fp, "# output formats\n");
  fprintf (fp, "progress=%s\n", options->progress ?
	   (options->verbose ? "VERBOSE" : "YES") : "NO ");
  fprintf (fp, "logfile=%s\n", options->writelog ?
	   options->logfilename : "NONE");
  fprintf (fp, "print-data=%s\n", options->printdata ? "YES" : "NO");
  fprintf (fp, "outfile=%s\n", options->outfilename);
  if (options->plot)
    {
      fprintf (fp, "plot=YES:%s:%s:", options->plotmethod == PLOTALL ?
	       "BOTH" : "OUTFILE",
               options->plotscale==PLOTSCALELOG ? "LOG" : "STD"
               );
      fprintf (fp, "{%f,%f,%f,%f}:",options->plotrange[0],
               options->plotrange[1], options->plotrange[2],
               options->plotrange[3]);
      fprintf (fp, "%1.1s%li\n", options->plotvar==PLOT4NM ? "N" : "M",
               options->plotintervals);
      
    }
  else
    fprintf (fp, "plot=NO\n");
  switch (options->profile)
    {
    case NONE:
      fprintf (fp, "profile=NONE\n");
      break;
    case ALL:
      fprintf (fp, "profile=ALL");
      break;
    case TABLES:
      fprintf (fp, "profile=TABLES");
      break;
    case SUMMARY:
      fprintf (fp, "profile=SUMMARY");
      break;
    }
  if (options->profile != NONE)
    {
      switch (options->profilemethod)
	{
	case 'p':
	  fprintf (fp, ":PRECISE\n");
	  break;
	case 's':
	  fprintf (fp, ":SPLINE\n");
	  break;
	case 'd':
	  fprintf (fp, ":DISCRETE\n");
	  break;
	case 'q':
	  fprintf (fp, ":QUICKANDDIRTY\n");
	  break;
	case 'f':
	  fprintf (fp, ":FAST\n");
	  break;
	}
    }

  switch (options->treeprint)
    {
    case NONE:
      fprintf (fp, "print-tree=NONE\n");
      break;
    case ALL:
      fprintf (fp, "print-tree=ALL\n");
      break;
    case BEST:
      fprintf (fp, "print-tree=BEST\n");
      break;
    case LASTCHAIN:
      fprintf (fp, "print-tree=LASTCHAIN\n");
      break;
    default:
      fprintf (fp, "print-tree=NONE\n");
      break;
    }
  fprintf (fp, "mathfile=%s\n", options->mathfilename);
  fprintf (fp, "write-summary=%s\n", options->writesum ? "YES" : "NO");
  if (options->writesum || options->readsum)
    fprintf (fp, "sumfile=%s\n", options->sumfilename);
  fprintf (fp, "# likelihood-ratio test\n");
  for (i = 0; i < options->lratio->counter; i++)
    {
      fprintf (fp, "l-ratio=%s:%s\n", options->lratio->data[i].type == MEAN ?
	       "MEAN" : "LOCUS", options->lratio->data[i].value);
    }
  fprintf (fp, "# parameter options ------------------------------------\n");
  if (options->thetaguess == FST)
    fprintf (fp, "theta=FST\n");
  else
    {
      fprintf (fp, "theta=Own:{");
      if (options->numthetag == 0)
	{
	  if (strchr ("snup", options->datatype))
	    fprintf (fp, "0.01}\n");
	  else
	    fprintf (fp, "1}\n");
	}
      for (i = 0; i < options->numthetag - 1; i++)
	fprintf (fp, "%f ", options->thetag[i]);
      fprintf (fp, "%f}\n", options->thetag[i]);
    }
  if (options->migrguess == FST)
    fprintf (fp, "migration=FST\n");
  else
    {
      switch (options->nummg)
	{
	case 0:
	  fprintf (fp, "migration=Own:{1}\n");
	  break;
	case 1:
	  fprintf (fp, "migration=Own:{%f}\n", options->mg[0]);
	  break;
	default:
	  fprintf (fp, "migration=Own:{ ");
	  z = 0;
	  num = (long) sqrt (options->nummg);
	  for (i = 0; i < num; i++)
	    {
	      for (j = 0; j < num; j++)
		{
		  if (i == j)
		    {
		      z++;
		      fprintf (fp, "- ");
		    }
		  else
		    fprintf (fp, "%f ", options->mg[z++]);
		}
	    }
	  fprintf (fp, "}\n");
	}
    }
  fprintf (fp, "mutation=%s\n", options->gamma ? "GAMMA" : "NOGAMMA");

  fprintf (fp, "fst-type=%s\n", options->fsttype ? "THETA" : "MIGRATION");
  fprintf (fp, "custom-migration={%s}\n", options->custm);
#ifdef TEST
  fprintf (fp, "geofile=%s\n", options->geo ? "YES" : "NO");
#endif
  fprintf (fp, "# search strategies ------------------------------------\n");
  fprintf (fp, "short-chains=%li\n", options->schains);
  fprintf (fp, "short-inc=%li\n", options->sincrement);
  fprintf (fp, "short-sample=%li\n", options->ssteps);
  fprintf (fp, "long-chains=%li\n", options->lchains);
  fprintf (fp, "long-inc=%li\n", options->lincrement);
  fprintf (fp, "long-sample=%li\n", options->lsteps);
  fprintf (fp, "#obscure options\n");
  fprintf (fp, "burn-in=%li\n", options->burn_in);
  fprintf (fp, "heating=%s", options->heating ? "YES" : "NO\n"); //
  if(options->heating)
    {
	  fprintf(fp,":{%f,%f,%f,%f}\n",options->heat[0],options->heat[1],
		  options->heat[2],options->heat[3]); 
    }
  //(options->heating == 1 ? ":SimpleMethod" :
  //			       ":ComplexMethod") : "");
  if (options->movingsteps)
    fprintf (fp, "moving-steps=YES:%f\n", options->acceptfreq);
  else
    fprintf (fp, "moving-steps=NO\n");
  fprintf (fp, "long-chain-epsilon=%f\n", options->lcepsilon);
  fprintf (fp, "gelman-convergence=%s\n", options->gelman ? "Yes" : "No");
  fprintf (fp, "replicate=%s:", options->replicate ? "Yes" : "No");
  if(options->replicatenum==0)
      fprintf (fp, "LastChains\n");
  else
      fprintf (fp, "%li\n",options->replicatenum);
  if(options->cpu>1)
    fprintf(fp,"cpu=%i\n",options->cpu);
  fprintf(fp, "end\n");
  fflush (fp);
  printf ("\n\n+++ Parmfile written to current directory +++\n\n");
}

/*private functions============================================= */

long
boolcheck (char ch)
{
  char c = uppercase (ch);
  if ((c == 'F') || (c == 'N'))
    return 0;
  else if ((c == 'T') || (c == 'Y'))
    return 1;
  else
    return -1;
}				/* boolcheck */

boolean
booleancheck (option_fmt * options, char *var, char *value)
{
  long i, check;
  char *booltokens[NUMBOOL] = BOOLTOKENS;
  char *tmp;
  long ltemp;
  check = boolcheck (value[0]);
  if (check == -1)
    return FALSE;
  i = 0;
  while (i < NUMBOOL && strcmp (var, booltokens[i]))
    i++;
  switch ((short) i)
    {
    case 0:			/*menu = <yes | no> */
      options->menu = (boolean) (check);
      break;
    case 1:			/*interleaved =<yes | no> */
      options->interleaved = (boolean) (check);
      break;
    case 2:			/*print-data = <yes | no> */
      options->printdata = (boolean) (check);
      break;
    case 3:			/* mixplot=<yes | no> */
      options->mixplot = (boolean) (check);
      break;
    case 4:			/* moving-steps = <yes | no> */
      options->movingsteps = (boolean) (check);
      if (options->movingsteps)
	{
	  strtok (value, ":");
	  tmp = strtok (NULL, " ,\n");
	  if(tmp!=NULL)
	    options->acceptfreq = atof ((char *) tmp);
	  else
	    options->acceptfreq = 0.1;
	}
      break;
    case 5:			/* freqs-from-data =  <yes | no> */
      options->freqsfrom = (boolean) (check);
      if (!options->freqsfrom)
	{
	  strtok (value, ":");
	  tmp = strtok (NULL, " ,");
	  if(tmp!=NULL)
	    options->freqa = atof ((char *) tmp);
	  tmp = strtok (NULL, " ,");
	  if(tmp!=NULL)
	    options->freqc = atof ((char *) tmp);
	  tmp = strtok (NULL, " ,");
	  if(tmp!=NULL)
	    options->freqg = atof ((char *) tmp);
	  tmp = strtok (NULL, " ,\n");
	  if(tmp!=NULL)
	    options->freqt = atof ((char *) tmp);
	}
      break;
    case 6:			/* usertree =  <yes | no> (only for sequence data) */
      options->usertree = (boolean) (check);
      break;
    case 7:
      {				/* autocorrelation=<YES:value | NO> */
	options->autocorr = (boolean) (check);
	if (options->autocorr)
	  {
	    strtok (value, ":");
	    tmp = strtok (NULL, " ;\n");
	    if(tmp!=NULL)
	      options->lambda = 1.0 / atof ((char *) tmp);
	  }
	break;
      }
    case 8:			/* simulation =  <yes | no> */
      options->simulation = (boolean) check;
      break;
    case 9:			/* plot =  <not | yes:<outfile | both><:std|:log><:{xs,xe,ys,ye}<:N|:M><#of_intervals>>> */
      options->plot = (boolean) check;
      if (options->plot)
	{
	  strtok (value, ":");
	  if (toupper (value[0]) == 'Y' || toupper (value[0]) == 'Y')
	    {
	      tmp = strtok (NULL, ":;\n");
	      if (tmp == NULL)
		options->plotmethod = PLOTALL;
	      else
		{
		  switch (lowercase (tmp[0]))
		    {
		    case 'o':
		      options->plotmethod = PLOTOUTFILE;
		      break;
		    case 'b':
		      options->plotmethod = PLOTALL;
		      break;
		    default:
		      options->plotmethod = PLOTALL;
		      break;
		    }
		  tmp = strtok (NULL, ":;\n");
		  if (tmp != NULL)
		    {
		      switch (lowercase (tmp[0]))
			{
			case 'l':
			  options->plotscale = PLOTSCALELOG;
			  break;
			case 's':
			  options->plotscale = PLOTSCALESTD;
			  break;
			default:
			  options->plotscale = PLOTSCALELOG;
			}
		      tmp = strtok (NULL, ":;\n");
		      if (4 !=
			  sscanf (tmp, "{%lf,%lf,%lf,%lf",
				  &options->plotrange[0],
				  &options->plotrange[1],
				  &options->plotrange[2],
				  &options->plotrange[3]))
			sscanf (tmp, "{%lf%lf%lf%lf", &options->plotrange[0],
				&options->plotrange[1],
				&options->plotrange[2], &options->plotrange[3]);
		      tmp = strtok (NULL, ":;\n");
		      if (tmp != NULL)
			{
			  switch (lowercase (tmp[0]))
			    {
			    case 'm':
			      options->plotvar = 1;
			      while (!isdigit (*tmp) && *tmp != '\0')
				tmp++;
			      if ((ltemp = strtol (tmp, (char **) NULL, 10)) >
				  0)
				options->plotintervals = ltemp;
			      break;
			    case 'n':
			    default:
			      options->plotvar = PLOT4NM;
			      while (!isdigit (*tmp) && *tmp != '\0')
				tmp++;
			      if ((ltemp = strtol (tmp, (char **) NULL, 10)) >
				  0)
				options->plotintervals = ltemp;

			      break;
			    }
			}
		    }
		}
	    }
	}
      break;
    case 10:			/* weights =  <yes | no> */
      options->weights = (boolean) check;
      break;
    case 11:			/* read-summary  <yes | no> */
      options->readsum = (boolean) check;
      options->datatype = 'g';
      break;
    case 12:			/* write-summary =  <yes | no> */
      options->writesum = (boolean) check;
      break;
    case 13:			/* mig-histogram=<yes | no> */
      options->mighist = (boolean) check;
      break;
    case 14:			/* heating=<yes  | yes:<simple | complex> | no> */
      options->heating = (short) check;
      if (options->heating == 1)
	{
	  strtok (value, ":\n");
	  tmp = strtok (NULL, ":\n");
	  if(tmp!=NULL)
	    {
	      sscanf(tmp,"{%lf,%lf,%lf,%lf}",&options->heat[0],&options->heat[1],
		     &options->heat[2],&options->heat[3]); 
	    }
	  //	  if (toupper (value[0]) == 'C')
	  // options->heating = 2;
	}
      break;
    case 15:			/* print-fst =  <yes | no> */
      options->printfst = (boolean) check;
      break;
    case 16:			/* distfile =  <yes | no> */
      options->dist = (boolean) check;
      break;
    case 17:			/* geofile =  <yes | no> */
      options->geo = (boolean) check;
      break;
    case 18:			/* gelman-convergence =  <yes | no> */
      options->gelman = (boolean) check;
      break;
    case 19:			/* randomtree start =  <yes | no> */
      options->randomtree = (boolean) check;
      break;
    default:
      return FALSE;
    }
  return TRUE;
}				/* booleancheck */

boolean
numbercheck (option_fmt * options, char *var, char *value)
{
  long i = 0, z;
  char *tmp, *temp, *keeptmp;
  char *numbertokens[NUMNUMBER] = NUMBERTOKENS;
  tmp = (char *) calloc (1, sizeof (char) * LINESIZE);
  keeptmp = tmp;
  while (i < NUMNUMBER && strcmp (var, numbertokens[i]))
    i++;
  switch ((short) i)
    {
    case 0:			/*ttratio = value */
      z = 0;
      temp = strtok (value, " ,;\n");
      while (temp != NULL)
	{
	  options->ttratio[z++] = atof (temp);
	  options->ttratio =
	    (double *) realloc (options->ttratio, sizeof (double) * (z + 1));
	  options->ttratio[z] = 0.0;
	  temp = strtok (NULL, " ,;\n");
	}
      break;
    case 1:			/*short-chains = value */
      options->schains = atol (value);
      break;
    case 2:			/*short-steps = value */
    case 32:			/* short-sample = value */
      options->ssteps = atol (value);
      break;
    case 3:			/*short-increment = value */
      options->sincrement = atol (value);
      break;
    case 4:			/*long-chains = value */
      options->lchains = atol (value);
      break;
    case 5:			/*long-steps = value */
    case 33:			/*long-sample = value */
      options->lsteps = atol (value);
      break;
    case 6:			/*long-increment = value */
      options->lincrement = atol (value);
      break;
    case 7:
      break;			/* theta: already handled in read_theta() */
    case 8:			/*nmlength = value */
      options->nmlength = strtol (value, (char **) NULL, 10);	//atoi (value);
      break;
    case 9:			/* seed = <Auto | seedfile | Own:value> */
      switch (value[0])
	{
	case 'A':
	case 'a':
	case '0':
	  options->autoseed = AUTO;
	  options->inseed = (long) time (0)/ 4 + 1;
	  break;
	case 'S':
	case 's':
	case '1':
	  options->autoseed = NOAUTO;
	  options->seedfile = fopen ("seedfile", "r");
	  if (options->seedfile == NULL)
	    {
	      usererror ("cannot find seedfile\n");
	    }
	  fscanf (options->seedfile, "%ld%*[^\n]", &options->inseed);
	  fclose (options->seedfile);
	  break;
	case 'O':
	case 'o':
	case '2':
	  options->autoseed = NOAUTOSELF;
	  strtok (value, ":");
	  tmp = strtok (NULL, " ;\n");
	  if(tmp!=NULL)
	    options->inseed = atol ((char *) tmp);
	  if (options->inseed > 0)
	    break;
	default:
	  options->autoseed = AUTO;
	  options->inseed = (long) time (0) / 4 + 1;
	  usererror ("Failure to read seed method, should be\n \
random-seed=auto or random-seed=seedfile or random-seed=own:value\nwhere value is a positive integer\nUsing AUTOMATIC seed=%li\n", options->inseed);
	  break;
	}
      break;
    case 10:
      break;			/*"migration" fake: this is already handled in read_migrate */
    case 11:			/*mutation= <auto=gamma | nogamma> */
      switch (value[0])
	{
	case 'A':		/*automatic */
	case 'a':
	case 'G':
	case 'g':
	  options->gamma = TRUE;
	  break;
	case 'N':		/*none, all loci have same mu */
	case 'n':
	  options->gamma = FALSE;
	  break;
	default:
	  break;
	}
      break;
    case 12:			/*datatype=<allele|microsatellite|brownian|sequence|genealogies> */
      switch (value[0])
	{
	case 'a':
	case 'A':
	  options->datatype = 'a';
	  break;
	case 'm':
	case 'M':
	  options->datatype = 'm';
	  break;
	case 'b':
	case 'B':
	  options->datatype = 'b';
	  break;
	case 's':
	case 'S':
	  options->datatype = 's';
	  break;
	case 'n':
	case 'N':
	  options->datatype = 'n';
	  break;
	case 'u':
	case 'U':
	  options->datatype = 'u';
	  break;
	case 'g':
	case 'G':
	  options->datatype = 'g';
	  options->readsum = TRUE;
	  break;
	default:
	  options->datatype = 's';
	  break;
	}
      break;
    case 13:			/* categories=<None | Two=Yes | value> */
      if (toupper (value[0] == 'N'))
	{
	  options->categs = ONECATEG;
	  break;
	}
      if ((toupper (value[0]) == 'Y') || (toupper (value[0] == 'T')))
	options->categs = MANYCATEGS;
      else
	options->categs = strtol (value, (char **) NULL, 10);	//atoi (value);  /* categories */
      /* needs to read auxilliary file catfile */
      break;
    case 14:			/*create rates=value:list of rates */
      strncpy (tmp, value, strcspn (value, ":"));
      if (strtol (tmp, (char **) NULL, 10) /*;atoi (tmp) */  > 1)
	{			/* rate categories */
	  options->rcategs = strtol (tmp, (char **) NULL, 10);
	  options->rrate = (double *) realloc (options->rrate, sizeof (double)
					       * (options->rcategs + 1));
	  temp = strtok (value, " :");
	  temp = strtok (NULL, " ,;\n");
	  z = 0;
	  while (temp != NULL)
	    {
	      if (z > options->rcategs)
		usererror ("check parmfile-option  rates, missing rate\n");
	      options->rrate[z++] = atof (temp);
	      temp = strtok (NULL, " ,;\n");
	    }
	}
      break;
    case 15:			/* probabilities for each rate category */
      strncpy (tmp, value, strcspn (value, ":"));
      if (strtol (tmp, (char **) NULL, 10) > 1)
	{			/* probabilities for each rate category */
	  options->rcategs = strtol (tmp, (char **) NULL, 10);
	  options->probcat =
	    (double *) realloc (options->probcat,
				sizeof (double) * (options->rcategs + 1));
	  temp = strtok (value, " :");
	  temp = strtok (NULL, " ,;\n");
	  z = 0;
	  while (temp != NULL)
	    {
	      if (z > options->rcategs)
		usererror
		  ("check parmfile prob-rates, missing rate probability\n");
	      options->probcat[z++] = atof (temp);
	      temp = strtok (NULL, " ,;\n");
	    }
	}
      break;
    case 16:			/*micro-stepmax */
      options->micro_stepnum = strtol (value, (char **) NULL, 10);

      break;
    case 17:			/*micro-threshold */
      options->micro_threshold = strtol (value, (char **) NULL, 10);
      break;
    case 18:			/*delimiter */
      options->dlm = value[0];
      break;
    case 19:			/*burn-in */
      options->burn_in = atol (value);
      break;
    case 20:			/*infilename */
      strcpy (options->infilename, value);
      break;
    case 21:			/*outfilename */
      strcpy (options->outfilename, value);
      break;
    case 22:			/*mathfilename */
      strcpy (options->mathfilename, value);
      break;
    case 23:			/*title */
      strncpy (options->title, value, 80);
      break;
    case 24:			/*long-chain-epsilon */
      options->lcepsilon = atof (value);
      break;
    case 25:			/* print tree options */
      switch (uppercase (value[0]))
	{
	case 'N':
	  options->treeprint = NONE;
	  break;
	case 'A':
	  options->treeprint = ALL;
	  break;
	case 'B':
	  options->treeprint = BEST;
	  break;
	case 'L':
	  options->treeprint = LASTCHAIN;
	  break;
	default:
	  options->treeprint = NONE;
	  break;
	}
      break;
    case 26:			/* progress: No, Yes, Verbose */
      switch (uppercase (value[0]))
	{
	case 'F':
	case 'N':
	  options->progress = FALSE;
	  options->verbose = FALSE;
	  break;
	case 'T':
	case 'Y':
	  options->progress = TRUE;
	  options->verbose = FALSE;
	  break;
	case 'V':
	  options->progress = TRUE;
	  options->verbose = TRUE;
	  break;
	}
      break;
    case 27:			/* l-ratio: <none | mean | locus>:val1,val2,val3,val4,val5 */
      switch (uppercase (value[0]))
	{
	case 'M':
	  options->lratio->data[options->lratio->counter].type = MEAN;
	  break;
	case 'L':
	  options->lratio->data[options->lratio->counter].type = LOCUS;
	  break;
	case 'N':
	default:
	  free (keeptmp);
	  return FALSE;
	}
      temp = strtok (value, ":");
      temp = strtok (NULL, "\n");
      if (temp != NULL)
	strcpy (options->lratio->data[options->lratio->counter].value, temp);
      if (options->lratio->counter + 1 == options->lratio->alloccounter)
	{
	  options->lratio->alloccounter += 2;
	  options->lratio->data =
	    (lr_data_fmt *) realloc (options->lratio->data,
				     sizeof (lr_data_fmt) *
				     options->lratio->alloccounter);
	  for (i = options->lratio->counter + 1;
	       i < options->lratio->alloccounter; i++)
	    {
	      options->lratio->data[i].elem = 0;
	      options->lratio->data[i].value =
		(char *) calloc (1, sizeof (char) * LINESIZE);
	    }
	}
      options->lratio->counter++;
      break;
    case 28:			/* fst-type: <Theta | Migration> */
      switch (uppercase (value[0]))
	{
	case 'T':
	  options->fsttype = 'T';
	  break;
	case 'M':
	default:
	  options->fsttype = 'M';
	  break;
	}
      fst_type (options->fsttype);
      break;
    case 29:			/*profile=<NO| NONE | YES | ALL | TABLES | SUMMARY>><: <FAST |  */
      switch (uppercase (value[0]))
	{
	case 'S':
	  options->profile = SUMMARY;
	  break;
	case 'Y':
	case 'A':
	  options->profile = ALL;
	  break;
	case 'N':
	  options->profile = NONE;
	  break;
	case 'T':
	  options->profile = TABLES;
	  break;
	default:		/*A */
	  options->profile = ALL;
	  break;
	}
      temp = strtok (value, ":;\n");
      temp = strtok (NULL, ":;\n");
      if (temp != NULL)
	{
	  switch (lowercase (temp[0]))
	    {
	    case 'p':		/*precise percentiles */
	      options->profilemethod = 'p';
	      break;
	    case 'd':		/*discrete steps see at start of file */
	      options->profilemethod = 'd';
	      break;
	      //case 's':		/*spline */
	      //options->profilemethod = 's';
	      //break;
	    case 'x':		/* x-rated */
	    case 'u':		/* uncorrelated */
	    case 'q':		/* quick and dirty */
	      options->profilemethod = 'q';
	      break;
	    case 'f':		/* quick and exact mixture */
	      options->profilemethod = 'f';
	      break;
	    default:
	      options->profilemethod = 'f';
	      options->printprofsummary = TRUE;
	      break;
	    }
	  temp = strtok (NULL, ":;\n");
	  if (temp != NULL)
	    {
	      switch (lowercase (temp[0]))
		{
		case 'm':
		  options->profileparamtype = 1;
		  break;
		default:
		  options->profileparamtype = PLOT4NM;
		}
	    }
	}
      set_profile_options (options);
      break;
    case 30:			/* custom-migration:<{> migration matrix and theta on
				   diagonal:
				   0 means not estimated,
				   x means estimated, s means symmetrically
				   estimated, m means all are the same  <}> */
      read_custom_migration (options->parmfile, options, value,
			     options->numpop);
      break;
    case 31:			/*sumfilename */
      strcpy (options->sumfilename, value);
      break;
      /*case 32 and case 33 are fallthroughs to 2 and 3 */
    case 34:                       /*replicate*/
      switch (uppercase (value[0]))
	{
	case 'T':
	case 'Y': options->replicate=TRUE;
	  temp = strtok (value, ":;\n");
	  if (temp != NULL)
	    {
	      temp = strtok (NULL, ":;\n");
	      if(uppercase(temp[0])=='L')
		options->replicatenum=0;
	      else
		options->replicatenum= 
		  strtol(temp,(char **) NULL, 10);
	    }
	  else
	    options->replicatenum=0;
	  break;
	default:
	  options->replicate=FALSE;
	  options->replicatenum = 0;
	}        
      break;
    case 35: /* cpu number*/
      options->cpu =  ATOI (value);
      break;
    case 36: /* do we write a logfile or not*/
      if(strcmp(value,"NONE"))
	{
	  options->writelog=TRUE;
	  strncpy(options->logfilename,value,255);
	}
      break; 
    default:
      free (keeptmp);
      return FALSE;
      
    }
   free (keeptmp);
  return TRUE;
}				/* numbercheck */

void
reset_oneline (option_fmt * options, long position)
{
  fseek (options->parmfile, position, SEEK_SET);
}


void
read_theta (option_fmt * options)
{
  //  char parmvar[LINESIZE];
  char tmp[LINESIZE];
  char varvalue[LINESIZE];
  char ch;

  long i = 0;

  while ((ch = getc (options->parmfile)) != '=')
    ;
  //  {
  //    parmvar[i++] = ch;
  //  }
  //i = 0;
  ch = getc (options->parmfile);
  while (!isspace ((int) ch) && ch != ':' && ch != '{')
    {
      varvalue[i++] = ch;
      ch = getc (options->parmfile);
    }
  switch (toupper(varvalue[0]))
    {
    case 'R':
      options->thetaguess = RANDOMESTIMATE;
      options->numthetag = 2;
      options->thetag = realloc (options->thetag, sizeof (double) * 3);
      fscanf (options->parmfile, "{%lf,%lf}\n", &options->thetag[0],
	      &options->thetag[1]);
      break;
    case 'F':
    case '_':
      options->thetaguess = FST;
      break;
      //case 'G':
    case 'O':
    case '0':
      //perhaps to come
      //      if(varvalue[0]=='G')
      //options->thetaguess = PARAMGRID;
      //else
      options->thetaguess = OWN;
      ch = skip_space (options);
      if (ch == '\0')
	return;
      if (ch == '{')
	{

	  while (ch != '}')
	    {
	      i = 0;
	      ch = skip_space (options);
	      if (ch == '\0')
		return;
	      while (ch != ' ' && ch != ',' && ch != '}')
		{
		  tmp[i++] = ch;
		  ch = getc (options->parmfile);
		}
	      tmp[i] = '\0';
	      options->thetag[options->numthetag] = atof (tmp);
	      options->numthetag += 1;
	      options->thetag = realloc (options->thetag,
					 sizeof (double) * (1 +
							    options->numthetag));

	    }
	}
      else
	{
	  i = 0;
	  tmp[i++] = ch;
	  while (!isspace (ch))
	    {
	      tmp[i++] = ch;
	      ch = getc (options->parmfile);
	    }
	  tmp[i] = '\0';
	  options->thetag[options->numthetag] = atof (tmp);
	  options->numthetag += 1;
	  options->thetag = realloc (options->thetag,
				     sizeof (double) * (1 +
							options->numthetag));
	}
      options->numpop = options->numthetag;
      if (options->numthetag == 0)
	{
	  warning ("You forgot to add your guess value:\n");
	  warning ("Theta=Own:{pop1,pop2, ...}\n");
	  warning ("or Theta=Own:guess_pop (same value for all)\n");
	}
      break;
    default:
      usererror
	("Failure to read start theta method, should be\ntheta=FST or theta=Own:x.x\n or theta=Own:{x.x, x.x , x.x, .....}");
    }

}


void
read_mig (option_fmt * options)
{
  //  char parmvar[LINESIZE];
  char tmp[LINESIZE];
  char varvalue[LINESIZE];
  char ch;
  long test = 0;
  long i = 0;
  /* 1st example:  1.0 (n-island model)
     2nd example: {1.0} (migration matrix model, all the same start values  
     3rd example: the dashes on the diagonal are NECESSARY, {} are facultativ
     -  1.0 0.1
     1.0  -  2.0
     0.9 1.2  -
     to specify real 0.0 you need to use the custom-migration settings.
     0.0 in the table will be change to SMALLES_MIGRATION
   */

  while ((ch = getc (options->parmfile)) != '=')
    {
      //      parmvar[i++] = ch;
    }
  i = 0;
  ch = getc (options->parmfile);
  while (!isspace ((int) ch) && ch != ':' && ch != '{')
    {
      varvalue[i++] = ch;
      ch = getc (options->parmfile);
    }
  switch (toupper(varvalue[0]))
    {
    case 'R':
      options->migrguess = RANDOMESTIMATE;
      options->nummg = 2;
      options->mg = realloc (options->mg, sizeof (double) * 3);
      fscanf (options->parmfile, "{%lf,%lf}\n", &options->mg[0],
	      &options->mg[1]);
      break;
    case 'F':
    case '_':
      options->migrguess = FST;
      break;
      //    case 'G':
    case 'O':
    case '0':
      //      if(varvalue[0]=='G')
      //       options->migrguess = PARAMGRID;
      //      else
	options->migrguess = OWN;
      ch = skip_space (options);
      if (ch == '\0')
	return;
      if (ch == '{')
	{
	  options->migration_model = MATRIX;
	  while (ch != '}')
	    {
	      ch = skip_space (options);
	      if ((ch == '\0') || (ch == '}'))
		return;
	      i = 0;
	      while (!isspace (ch) && ch != ',' && ch != '}')
		{
		  tmp[i++] = ch;
		  ch = getc (options->parmfile);
		}
	      tmp[i] = '\0';
	      if (strcmp (tmp, "-"))
		{
		  options->mg[options->nummg] = atof (tmp);
		  options->nummg += 1;
		  options->mg = realloc (options->mg,
					 sizeof (double) * (1 +
							    options->nummg));
		}
	      else
		{
		  test++;
		}
	    }
	  options->numpop = test;
	}
      else
	{
	  options->migration_model = ISLAND;
	  i = 0;
	  options->numpop = 1;
	  tmp[i++] = ch;
	  while (!isspace (ch))
	    {
	      tmp[i++] = ch;
	      ch = getc (options->parmfile);
	    }
	  options->mg[options->nummg] = atof (tmp);
	  options->nummg += 1;
	}
      if (options->nummg == 0)
	{
	  warning ("You forgot to add your guess value, use either:\n");
	  warning ("migration=FST\n");
	  warning ("migration=Own:{migration matrix, diagonal is -}\n");
	  usererror
	    ("migration=Own:{migration value}, all matrix elements have the same value\n");
	}
      break;
    default:
      usererror ("Failure to read start migration method\n");
    }

}

char
skip_space (option_fmt * options)
{
  char ch = getc (options->parmfile);
  while (isspace ((int) ch) || ch == ',')
    {
      ch = getc (options->parmfile);
    }
  if (isalpha (ch))
    {
      ungetc (ch, options->parmfile);
      ch = '\0';
    }
  return ch;
}

void
set_profile_options (option_fmt * options)
{
  switch (options->profile)
    {
    case NONE:
      options->printprofsummary = options->printprofile = FALSE;
      break;
    case ALL:
      options->printprofsummary = options->printprofile = TRUE;
      break;
    case TABLES:
      options->printprofsummary = FALSE;
      options->printprofile = TRUE;
      break;
    case SUMMARY:
      options->printprofsummary = TRUE;
      options->printprofile = FALSE;
      break;
    }
  if (options->profilemethod == 'd')
    options->printprofsummary = FALSE;
}



/* custom-migration:<{> migration matrix and theta on
   diagonal:
   0 means not estimated,
   x means estimated, s means symmetrically
   estimated, m means all are the same, 
   c means remains constant at start value <}>
   example: 
   {* * s
    * c *
    s 0 *}
*/
void
read_custom_migration (FILE * file, option_fmt * options,
		       char *value, long customnumpop)
{

  long zz = 0, z = 0;
  char ch = '\0';
  long lc, numpop, i, j, ii;
  long position = 0;

  if (customnumpop == 0)
    customnumpop = 1000000;
  else
    customnumpop *= customnumpop;

  z = 0;
  zz = 0;
  while (ch != '}' && zz < customnumpop)
    {
      ch = value[z];
      switch (ch)
	{
	case '}':
	case '{':
	case ' ':
	case '\t':
	  z++;
	  break;
	case '\0':
	case '\n':
	  z = 0;
	  if (file == stdin)
	    printf ("Enter the next value or list of values\n");
	  fgets (value, LINESIZE, file);
	  break;
	default:
	  options->custm =
	    (char *) realloc (options->custm, sizeof (char) * (zz + 2));
	  options->custm2 =
	    (char *) realloc (options->custm2, sizeof (char) * (zz + 2));
          switch(ch)
	    {
	    case 'S':
              options->custm[zz++] = ch;
	      break;
	    case 'M': //do we have code for this?
              options->custm[zz++] = ch;
	      break;
	    case 'x':
	    case 'X':
	      options->custm[zz++] = '*';
	      break;
            default:
	      options->custm[zz++] = tolower (ch);
	      break;
	    }
	  z++;
	}
    }
  options->custm[zz] = '\0';
  lc = strlen (options->custm);
  numpop = sqrt (lc);
  z = numpop;
  for (i = 0; i < numpop; i++)
    {
      for (j = 0; j < numpop; j++)
	{
	  ii = i * numpop + j;
	  if (i == j)
	    options->custm2[i] = options->custm[ii];
	  else
	    options->custm2[z++] = options->custm[ii];
	}
    }
  specify_migration_type (options);
  if (file != stdin)
    position = ftell (options->parmfile);
  while (file != stdin && !(strstr (value, "end") || strchr (value, '=')))
    {
      position = ftell (options->parmfile);
      fgets (value, LINESIZE, file);
    }
  if (file != stdin)
    reset_oneline (options, position);
}


void
specify_migration_type (option_fmt * options)
{
  long len = strlen (options->custm);
  long ms = 0, xs = 0, ns = 0, ss = 0, len2, i;
  char *p;
  p = options->custm;
  while (*p != '\0')
    {
      switch (*p)
	{
	case 'm':
	  ms++;
	  break;
	case 'x':
	case '*':
	  xs++;
	  break;
	case '0':
	  ns++;
	  break;
	case 'S':
	case 's':
	  ss++;
	  break;
	case 'c':
	  break;
	}
      p++;
    }
  if (ms >= len)
    {
      options->migration_model = ISLAND;
      return;
    }
  if (xs >= len)
    {
      options->migration_model = MATRIX;
      return;
    }
  if (ns >= len)
    {
      usererror ("Custom migration matrix was completely set to zero?!\n");
      return;
    }
  len2 = sqrt (len);
  if (ms == len2 && xs == len - len2)
    {
      for (i = 0; i < len2; i++)
	{
	  if (options->custm[i * len2 + i] != 'm')
	    {
	      options->migration_model = MATRIX;
	      return;
	    }
	}
      options->migration_model = MATRIX_SAMETHETA;
      return;
    }
  if (xs == len2 && ms == len - len2)
    {
      for (i = 0; i < len2; i++)
	{
	  if (options->custm[i * len2 + i] != '*')
	    {
	      options->migration_model = MATRIX;
	      return;
	    }
	}
      options->migration_model = ISLAND_VARTHETA;
      return;
    }
  options->migration_model = MATRIX_ARBITRARY;
}



void
fillup_custm (long len, world_fmt * world, option_fmt * options)
{
  long i, j, ii, z;
  char *tmp;
  tmp = (char *) calloc (1, sizeof (char) * (world->numpop2 + 20));
  options->custm = (char *) realloc (options->custm,
				     sizeof (char) * (world->numpop2 + 1));

  options->custm2 = (char *) realloc (options->custm2,
				      sizeof (char) * (world->numpop2 + 1));
  strncpy (tmp, options->custm,world->numpop2);
  z = world->numpop;
  for (i = 0; i < world->numpop; i++)
    {
      for (j = 0; j < world->numpop; j++)
	{
	  ii = i * world->numpop + j;
	  if (ii < len)
	    options->custm[ii] = tmp[ii];
	  else
	    options->custm[ii] = '*';
	  if (i == j)
	    options->custm2[i] = options->custm[ii];
	  else
	    options->custm2[z++] = options->custm[ii];
	}
    }
  options->custm[world->numpop2] = '\0';
  options->custm2[world->numpop2] = '\0';
  free (tmp);
}

void print_arbitrary_migration_table(FILE *file, world_fmt *world)
{
  long i;
  char mytext[LINESIZE];
  switch (world->options->migration_model)
    {
    case ISLAND:
      strcpy (mytext, "N-Island migration model");
      fprintf (file, "Migration model:\n   %-44.44s\n", mytext);
      break;
    case ISLAND_VARTHETA:
      strcpy (mytext, "N-Island migration model with variable Theta");
      fprintf (file, "Migration model:\n   %-44.44s\n", mytext);
      break;
    case MATRIX:
      strcpy (mytext, "Migration matrix model with variable Theta ");
      fprintf (file, "Migration model:\n   %-44.44s\n", mytext);
      break;
    case MATRIX_SAMETHETA:
      strcpy (mytext, "Migration matrix model with same Theta");
      fprintf (file, "Migration model:\n   %-44.44s\n", mytext);
      break;
    case MATRIX_ARBITRARY:
    default:
      strcpy (mytext, "Arbitrary migration matrix model");
      fprintf (file, "Migration model: %-44.44s\n", mytext);
      fprintf (file, "[Legend: m = average (either all theta or all M),]\n");
      fprintf (file,
	       "[s = symmetric M, S = symmetric 4Nm,\n 0 = zero, and not estimated,   ]\n");
      fprintf (file, "[* = free to vary, Thetas are on diagonal]\n");
      if (world->options->migration_model == MATRIX_ARBITRARY)
	{
	  for (i = 0; i < world->numpop2; i++)
	    {
	      if (i % world->numpop == 0)
		fprintf (file, "\n             ");
	      fprintf (file, "%c ", world->options->custm[i]);
	    }
	  fprintf (file, "\n\n");
	}
      break;
    }
}

void print_distance_table(FILE *file, world_fmt *world)
{
  long i,j;

  if (world->options->geo)
    {
      fprintf(file,"Geographic distance matrix between locations\n     ");
      for (i = 0; i < world->numpop; i++)
	{
	  for (j = 0; j < world->numpop; j++)
	    {
	      if(i==j)
		fprintf(file,"   -   ");
	      else
		fprintf (file, "%6.3f ", world->data->ogeo[i][j]);
	    }
	  fprintf (file, "\n      ");
	}
      fprintf (file, "\n");
    }
}







