/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 M A I N   P R O G R A M 

 drives the whole program MIGRATE.

 for options of the program: README and parmfile.distrib, 
                             options.c
 and of course the documentation at:
    http://evolution.genetics.washington.edu/lamarc/migrate.html
                                                                                                               
 Peter Beerli, started 1997, Seattle
 beerli@genetics.washington.edu
$Id: main.c,v 1.40 2000/07/26 23:03:20 beerli Exp $

-------------------------------------------------------*/
#include "migration.h"
#include "definitions.h"
#include "heating.h"
#include "world.h"
#include "data.h"
#include "options.h"
#include "mcmc.h"
#include "broyden.h"
#include "combroyden.h"
#include "menu.h"
#include "random.h"
#include "sequence.h"
#include "tree.h"
#include "tools.h"
#include "profile.h"

#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif

#define EARTH  universe[0]
#define VENUS  universe[1]
#define MERKUR universe[2]
#define SUN    universe[3]

/* handles a function call stub if there is no thread library*/
#ifndef PTHREADS
#ifndef tpool_t  
#define tpool_t char
#endif
#endif

char appl[8];
longer seed;


int setup_locus(long locus, world_fmt *world, option_fmt *options,
		data_fmt *data);

void condense_time(world_fmt *world, long *step, long *j, 
		   long *accepted, long *G, long *steps, 
		   long oldsteps, long rep);

long set_repkind(option_fmt *options);

void heating_init(world_fmt **venus,
		  world_fmt **merkur,world_fmt **sun,
		  data_fmt *data, option_fmt *options,
		  char *this_string);

void heating_prepare(world_fmt *world, world_fmt *sun,
		     world_fmt *merkur, world_fmt *venus,
		     option_fmt *options);

void heating_prepare2(world_fmt *world, world_fmt *sun,
		      world_fmt *merkur, world_fmt *venus);

long replicate_number(option_fmt *options, long chain, char type, long replicate);

void  combine_loci( char type, 
		   option_fmt * options,
		   world_fmt *world);

void run_sampler(option_fmt *options, 
		 data_fmt *data, 
		 world_fmt **universe,
		 long *outfilepos);

void run_locus(world_fmt **universe, option_fmt *options, 
	       data_fmt *data, tpool_t *heating_pool,
	       long maxreplicate, long locus, long *treefilepos);

void heated_swap(world_fmt **universe, option_fmt *options,
		 double *acc_vh, double *acc_h, double *acc_w, double *acc_c,
		 long *vh, long *h, long *w, long *jj, long *j,
		 long *hot, long *warm, long *cold);
  
void print_progress(option_fmt *options, world_fmt *world, 
		    long rep, long visited, long accepted);

void change_chaintype(long locus, char *type, world_fmt *world,
		      long *increment, long *oldsteps,long *chains,
		      option_fmt *options);
  
void prepare_next_chain(world_fmt **universe,
			  option_fmt *options,
			char type, long chain, 
			long *chains, long *pluschain, 
			long locus, long replicate);

void print_heating_progress(option_fmt *options, long steps,long increment,
			   double *acc_vh, double *acc_h, double *acc_w, double *acc_c,
			    long *hot, long *warm, long *cold);

void analyze_olddata(world_fmt *world, 
		    option_fmt * options, 
		    data_fmt *data, long *outfilepos);

void profile_tables(option_fmt *options, world_fmt *world);

void finish_mac(option_fmt *options,data_fmt *data);
  
int setup_locus(long locus, world_fmt *world, option_fmt *options,
		data_fmt *data);

void condense_time(world_fmt *world, long *step, long *j,
		   long *accepted, long *G, long *steps, long oldsteps, long rep);

long 
set_repkind(option_fmt *options);

void
heating_init(world_fmt **venus,
	     world_fmt **merkur,world_fmt **sun,
	     data_fmt *data, option_fmt *options,
	     char *this_string);

void 
heating_prepare(world_fmt *world, world_fmt *sun,
		world_fmt *merkur, world_fmt *venus,
		option_fmt *options);

void heating_prepare2(world_fmt *world, world_fmt *sun,
		      world_fmt *merkur, world_fmt *venus);

long 
replicate_number(option_fmt *options, long chain, 
		 char type, long replicate);


/*--------------------------------------------
   main program 
*/
int
main (int argc, char **argv)
{
  long outfilepos=0;
  char type='s';
  char *this_string;
  data_fmt *data;

  option_fmt *options;

  world_fmt **universe;

  universe=(world_fmt **) calloc(1,sizeof(world_fmt*)*HEATED_CHAIN_NUM);
  this_string = (char *) calloc (1, sizeof (char) * 256);
  strcpy (appl, "Migrate");
  /* try to catch and beautify some error messages -------------- */
  signalhandling (ON);
  /* create main data structures -------------------------------- */
  create_data (&data);
  strcpy(this_string,"earth");
  create_world (&(EARTH),this_string,1);
  create_options (&options);
  /* parmfile and menu ------------------------------------------ */
  init_options (options);
  get_options (options);
  print_menu_title (stdout, options);
  get_menu (options);
  
  /* initialization and data reading phase ---------------------- */
  init_files (EARTH, data, options);
  if(options->writelog)
    print_menu_title (options->logfile, options);
  EARTH->repkind = SINGLECHAIN; 
  if (!options->readsum)
    run_sampler(options,data,universe,&outfilepos);// MCMC sampling 
  else
    analyze_olddata(EARTH,options,data,&outfilepos);     // reanalyze sumfile
  combine_loci(type, options,EARTH);  // multilocus MLE
  /* printing of data ------------------------------------------- */
  print_list (EARTH);
  fflush (EARTH->outfile);
  /* profile tables --------------------------------------------- */
  profile_tables(options, EARTH);
  /* closing all files ------------------------------------------ */
  print_finish (EARTH, outfilepos);
#ifdef MAC
  finish_mac(options, data);
#endif
  exit_files (EARTH, data, options);
  return 0;
}

/* multiple loci combination ---------------------------------- */
void  combine_loci(char type, 
		   option_fmt * options,
		   world_fmt *world)
{
  if ((world->loci - world->skipped > 1) || (options->readsum))
    {
      world->repkind = set_repkind(options);
      estimateParameter (options->replicatenum, world->G+1, world,
			 world->cov[world->loci], 0 /*chain*/,
			 type, world->plane[world->loci], 
			 MULTILOCUS, world->repkind);      
    } 
}

void run_sampler(option_fmt *options, 
		 data_fmt *data, 
		 world_fmt **universe,
		 long *outfilepos)
{
  long maxreplicate;
  long treefilepos;
  long locus;
  char *this_string;

#ifdef PTHREADS
  tpool_t heating_pool;
#else
#ifndef tpool_t  
#define tpool_t char
#endif
#endif

  this_string = (char *) calloc(1024,sizeof(char));
  get_data (data->infile, data, options);
  getseed (options);
  init_world (EARTH, data, options);
  if(options->heating)
    heating_init(&VENUS,&MERKUR,&SUN,data,options,this_string);
  set_plot (options);
  /* calculate FST values */
  calc_simple_param (EARTH);
  /* report to screen */
  print_menu_options (options, EARTH);
  if (options->progress)
    print_data_summary (stdout, EARTH);
  /* print to outfile */
  *outfilepos = print_title (EARTH);
  print_options (EARTH->outfile, options, EARTH);
  print_data_summary (EARTH->outfile, EARTH);
  print_data (EARTH, data, options);
  if(options->lchains<2 && options->replicatenum==0)
    options->replicate = 0;
  maxreplicate = (options->replicate && options->replicatenum>0) ?
    options->replicatenum : 1;
#ifdef PTHREADS
  if(options->heating)
    tpool_init(&heating_pool,HEATED_THREAD_NUM, HEATED_CHAIN_NUM, 0);
#endif
  for (locus = 0; locus < data->loci; locus++)
    {
      EARTH->locus = locus;
      run_locus(universe,options,data, 
#ifdef PTHREADS
		&heating_pool, 
#else
		NULL,
#endif
		maxreplicate, locus, &treefilepos);
    }    
  /* save all genealogy-summaries --------------------------- */
  if (options->writesum)
    {
      write_savesum (EARTH);
    }
#ifdef PTHREADS
  if(options->heating)
    tpool_destroy(heating_pool,1);
#endif
  free(this_string);
}

void run_locus(world_fmt **universe, option_fmt *options, 
	       data_fmt *data, tpool_t *heating_pool,
	       long maxreplicate, long locus, long *treefilepos)
{
  long replicate;
  char type;
  long runs, rep, pluschain, increment, oldsteps, chains;
  long chain=0, hot,warm,cold,accepted,steps, step,i,j,jj;
  double acc_vh,acc_h,acc_w,acc_c;
  long vh,h,w;
  
  /* loop over independent replicates----------------------- */
  /* but make sure that we get single chain estimates, too   */
  EARTH->repkind = SINGLECHAIN; 
  for(replicate=0; replicate < maxreplicate; replicate++)
    {
      EARTH->replicate = replicate;
      if(setup_locus(locus,EARTH,options,data)==0)
	continue;
      if(options->heating)
	heating_prepare(EARTH, SUN, MERKUR, VENUS, options);
      type = 's';
      runs = 1;
      rep = 0;
      pluschain = options->pluschain;//see obscure options
      /* short and long chains ----------------------------- */
      set_bounds (&increment, &oldsteps, &chains, options, type);
      EARTH->increment = increment;
      while (runs-- >= 0)
	{
	  print_menu_chain (type, FIRSTCHAIN, 
			    oldsteps, EARTH, rep);
	  if (EARTH->options->treeprint == ALL)
	    print_tree (EARTH, 0, treefilepos);
	  EARTH->chains = chains;
	  EARTH->lsteps = log(oldsteps);
	  /* loop over chains---------------------------*/
	  for (chain = 0; chain < chains ||
		 (type == 'l' && chain >= chains
		  && EARTH->param_like > options->lcepsilon); 
	       chain++)
	    {
	      hot = warm = cold = 0;
	      rep = replicate_number(options, chain, type, replicate);
	      precalc_world (EARTH);
	      polish_world(EARTH);
	      burnin_chain (EARTH);	
	      EARTH->G = 0; 
	      accepted = 0; 
	      steps = oldsteps;
	      EARTH->maxdatallike = EARTH->likelihood[0];
	      if(options->heating)
		heating_prepare2(EARTH, SUN, MERKUR, VENUS);
	      
	      if(type=='l')
		{
		  if (chain >= chains - 1)
		    EARTH->in_last_chain = TRUE;
		}
	      /* steps for each chain ------------------------ */
	      for (step = 0; step < steps + 1; step++)
		{
		  j = 0;
		  EARTH->increment = increment;
		  /* hop over INCREMENT trees,sample last */
		  for (i = 0; i < increment; i++)
		    {
		      EARTH->actualinc = i;
		      // run clones of world at different 
		      // temperatures
		      if(options->heating)
			{
#ifdef PTHREADS
			  fill_tpool(*heating_pool,
				     universe,
				     HEATED_CHAIN_NUM);
			  tpool_synchronize(*heating_pool,HEATED_CHAIN_NUM);
			  
			  acc_vh += (vh = SUN->accept);
			  acc_h  += (h = MERKUR->accept);
			  acc_w  += (w = VENUS->accept);
			  jj      = EARTH->accept;
#else
			  acc_vh += (vh = metropolize(SUN,
						      SUN->G));
			  acc_h  += (h = metropolize(MERKUR,
						     MERKUR->G));
			  acc_w  += (w = metropolize(VENUS,
						     VENUS->G));
			  
			  jj = metropolize (EARTH, EARTH->G);
#endif
			  heated_swap(universe, options,
				      &acc_vh, &acc_h, &acc_w, &acc_c,
				      &vh, &h, &w, &jj, &j,
				      &hot, &warm, &cold);
			}
		      else
			j += metropolize (EARTH, EARTH->G);
		      if (EARTH->likelihood[EARTH->G] > EARTH->maxdatallike)
			{
			  EARTH->maxdatallike = EARTH->likelihood[EARTH->G];
			}
		    }
		  /*if(options->verbose && !options->heating)
		    {
		      printf("Cold:%5.5f\n",
			     EARTH->likelihood[EARTH->G]);
			     }*/
		  if(options->heating)
		    {  
		      /*if(options->verbose)
			{
			  printf("***next step\n");
			  }*/
		      advance_clone_like(VENUS, w, &VENUS->G);
		      advance_clone_like(MERKUR, h, &MERKUR->G);
		      advance_clone_like(SUN, vh, &SUN->G);
		    }
		  /* store condensed tree information
		     for estimation */	
		  condense_time(EARTH, &step, &j, &accepted, 
				&EARTH->G, &steps, 
				oldsteps, rep);
		  if(step>0)
		    advance_clone_like(EARTH, j, &EARTH->G);
		} // end steps
	      decide_plot (options, chain, chains, type);
	      memcpy (EARTH->param00, EARTH->param0,
		      sizeof (double) * EARTH->numpop2);
	      memcpy (EARTH->atl[rep][locus].param0, EARTH->param00,
		      sizeof (double) * EARTH->numpop2);
	      log_param0 (EARTH->atl[rep][locus].param0,
			  EARTH->atl[rep][locus].lparam0, EARTH->numpop2);
	      EARTH->repkind = SINGLECHAIN;
	      
	      // single chain,  single replicate
	      // estimateParameter(locus, rep, chain, type, 
	      //			EARTH, SINGLELOCUS);
	      estimateParameter (rep, EARTH->G+1, EARTH,
				 EARTH->cov[locus], chain,
				 type, EARTH->plane[locus], 
				 SINGLELOCUS, EARTH->repkind);
	      print_progress(options,EARTH, 
			     rep, steps*increment, accepted);
	      prepare_next_chain(universe,options,type,chain,
				 &chains, &pluschain, 
				 locus, replicate);
	      print_heating_progress(options,steps,increment,
				     &acc_vh,&acc_h,&acc_w,&acc_c,
				     &hot,&warm,&cold);
	    } //end chains
	  change_chaintype(locus, &type,EARTH, 
			   &increment, &oldsteps, &chains,
			   options);
	  /* evaluate multiple long chain estimates*/
	  if(runs<0 && options->replicate && options->replicatenum==0)
	    {
	      EARTH->repkind = MULTIPLECHAIN;
	      estimateParameter (rep, EARTH->G+1, EARTH,
				 EARTH->cov[locus], chain,
				 type, EARTH->plane[locus], 
				 SINGLELOCUS,EARTH->repkind);
	    }
	} //end runs 
    } //end replicates
  if(options->replicate && options->replicatenum>0)
    {
      EARTH->repkind = MULTIPLERUN;
      estimateParameter (options->replicatenum, EARTH->G+1, 
			 EARTH,
			 EARTH->cov[locus], chain,
			 type, EARTH->plane[locus], 
			 SINGLELOCUS, EARTH->repkind);
    }
}

void heated_swap(world_fmt **universe, option_fmt *options,
		 double *acc_vh, double *acc_h, double *acc_w, double *acc_c,
		 long *vh, long *h, long *w, long *jj, long *j,
		 long *hot, long *warm, long *cold)
{
  long sm=0,mv=0,vw=0;
  switch(RANDINT(0,2))
    {
    case 0:
      sm = chance_swap_tree(SUN,MERKUR);
      mv=vw=0;
      *vh += sm;
      *h += sm;
      *acc_vh += sm;
      *acc_h += sm;
      break;
    case 1:
      mv = chance_swap_tree(MERKUR,VENUS);
      sm=vw=0;
      *h += mv;
      *w += mv;
      *acc_h += mv;
      *acc_w += mv;
      break;
    case 2:
      vw = chance_swap_tree(VENUS,EARTH);
      *acc_w += vw;
      *w += vw;
      mv=sm=0;
      break;
    }
  *jj += vw;
  /*  if(options->verbose)
    {
      printf("   Boil%c%c%5.5f ",(int)(*vh)+' ',
	     sm==0?':':'*',
	     SUN->likelihood[SUN->G]); 
      printf("Hot %c%c%5.5f ",(int)(*h)+' ',
	     sm==0 && mv==0 ?':':'*',
	     MERKUR->likelihood[MERKUR->G]);
      printf("Warm%c%c%5.5f ",(int)(*w)+' ',
	     mv==0 && vw==0?':':'*',
	     VENUS->likelihood[VENUS->G]);
      printf("Cold%c%c%5.5f\n",(int)(*jj)+' ',
	     vw==0?':':'*',
	     EARTH->likelihood[EARTH->G]);
      
	     }*/
  if(options->progress && options->heating)
    {
      *hot += sm;
      *warm += mv;
      *cold += vw;
    }
  if(*jj>0)
    {
      (*j)++; (*acc_c)++;
    }
}
  
void print_progress(option_fmt *options, world_fmt *world, 
		    long rep, long visited, long accepted)
{
  if (options->progress)
    {
      print_menu_coalnodes (stdout, world, world->G+1,rep);
      print_menu_accratio (stdout, accepted, visited);
      if(options->writelog)
	{
	  print_menu_coalnodes (options->logfile, world, world->G+1,rep);
	  print_menu_accratio (options->logfile, accepted, visited);
	}
      if (!world->start && world->chains > 1
	  && options->verbose)
	print_gelmanr (world->gelmanmeanR, world->gelmanmaxR);
    }
}

void change_chaintype(long locus, char *type, world_fmt *world,
		      long *increment, long *oldsteps,long *chains,
		      option_fmt *options)
{
  if (*type == 's')
    {
      *type = 'l';
      create_treetimelist (world, &(world->treetimes), 
			   locus);
      set_bounds (increment, oldsteps, chains, 
		  options, *type);
      world->increment = *increment;
    }
}
  
void prepare_next_chain(world_fmt **universe,
			option_fmt *options,
			char type, long chain, 
			long *chains, long *pluschain, 
			long locus, long replicate)
{
  EARTH->likelihood[0] = EARTH->likelihood[EARTH->G];
  if(options->heating)
    {
      SUN->likelihood[0] = SUN->likelihood[SUN->G];
      MERKUR->likelihood[0] = MERKUR->likelihood[MERKUR->G];
      VENUS->likelihood[0] = VENUS->likelihood[VENUS->G];
    }
  EARTH->treetimes[0].copies = 0;
  if(type=='l')
    {
      if(options->replicate && options->replicatenum>0)
	EARTH->chainlikes[locus][replicate] = EARTH->param_like;
      else
	EARTH->chainlikes[locus][chain] = EARTH->param_like;
    }
  EARTH->start = FALSE;
  if (type == 'l')
    {
      if(chain < *chains + *pluschain)
	{
	  if ((EARTH->param_like > options->lcepsilon)
	      || (options->gelman &&
		  EARTH->gelmanmaxR > GELMAN_MYSTIC_VALUE))
	    {
	      (*chains)++;
	      (*pluschain)--;
	    }
	}
    }
}

void print_heating_progress(option_fmt *options, long steps,long increment,
			   double *acc_vh, double *acc_h, double *acc_w, double *acc_c,
			   long *hot, long *warm, long *cold)
{
  if(options->progress && options->heating)
    {
      *acc_vh /= increment * steps;
      *acc_h /= increment * steps;
      *acc_w /= increment * steps;
      *acc_c /= increment * steps;
      fprintf(stdout,"           Swapping between 4 temperatures.\n");
      fprintf(stdout,"           Temp Accept:Swaps between temperatures\n");
      fprintf(stdout,"           Boil:%4.2f :     |\n",*acc_vh);
      fprintf(stdout,"           Hot :%4.2f :%5li||\n",*acc_h,*hot);
      fprintf(stdout,"           Warm:%4.2f :%5li ||\n",*acc_w,*warm);
      fprintf(stdout,"           Cold:%4.2f :%5li  |\n",*acc_c,*cold);
      *hot= *warm= *cold = 0;
      *acc_vh = *acc_h = *acc_w = *acc_c = 0.0;
    }
}


void analyze_olddata(world_fmt *world, 
		    option_fmt * options, 
		    data_fmt *data, long *outfilepos)
{
  read_savesum (world, options, data);
  synchronize_param (world, options);
  if (options->plot)
    options->plotnow = TRUE;
  read_geofile(data,options,world->numpop);
  set_plot (options);
  *outfilepos = print_title (world);
  print_options (stdout, options, world);
  print_options (world->outfile, options, world);
}

void profile_tables(option_fmt *options, world_fmt *world)
{
  long i;
  if (options->profile != NONE)
    {
      PAGEFEEDWORLD;
      if (options->printprofsummary)
	allocate_profile_percentiles (world);
#ifdef HAVE_STRFTIME
      time (&world->starttime);
#endif
      for (i = 0; i < world->numpop2 + (world->options->gamma ? 1 : 0); i++)
	{
	  if (print_profile_likelihood_driver (i, world))
	    {
	      if ((i + 1) % 2 == 0)
		PAGEFEEDWORLD;
	      fflush (world->outfile);
	    }
	}
      if (options->printprofsummary)
	{
	  print_profile_percentile (world);
	  destroy_profile_percentiles (world);
	}
    }
}

void finish_mac(option_fmt *options,data_fmt *data)
{
#ifdef MAC
  fixmacfile (options->outfilename);
  if (options->plotmethod == PLOTALL)
    fixmacfile (options->mathfilename);
  if (options->treeprint)
    fixmacfile (options->treefilename);
  if (options->parmfile)
    fixmacfile ("parmfile");
  if (data->sumfile)
    fixmacfile (options->sumfilename);
#endif
}
  
int setup_locus(long locus, world_fmt *world, option_fmt *options,
		data_fmt *data)
{
  world->locus = locus;
  set_param (world, data, options, locus);
  print_menu_locus (world, locus);
  world->start = TRUE;
  buildtree (world, locus);
  if(world->replicate==0)
    {
      if (strchr (SNPTYPES, options->datatype))
	{
	  if (options->datatype == 'u')
	    data->seq->endsite *= (data->seq->addon + 1);
	  data->seq->endsite += data->seq->addon;
	}
    }
  if(!world->options->replicate || 
     (world->options->replicate && world->options->replicatenum==0) ||
     (world->options->replicate && world->replicate==world->options->replicatenum))
    {
      free_datapart (data, options, locus);
      if (data->skiploci[locus])
	{			/* skip loci with no tips */
	  cleanup_world (world, locus);
	  return 0;
	}
    }
  create_treetimelist (world, &world->treetimes, locus);
  fix_times (world);
  first_smooth (world, locus);
  world->likelihood[0] = treelikelihood (world);
  world->allikemax = -DBL_MAX;	/* for best tree option */
  return 1;
}

void condense_time(world_fmt *world, long *step, long *j,
		   long *accepted, long *G, long *steps, long oldsteps, long rep)
{
  world->rep = rep;
  if (*step == 0)
    {
      copy_time (world, world->treetimes,
		 FIRSTSTEP, 0, world->numpop,rep);
      *accepted += *j;
    }
  else
    {
      copy_time (world, world->treetimes, *G,
		 *G + (long) (*j > 0), world->numpop, rep);
      if (*step < *steps)
	{
	  //handled in advance_world  *G += (long) (*j > 0);
	  *accepted += *j;
	}
    }
  if (*step >= oldsteps - 1 && world->options->movingsteps)
    {
      if (*G+1 < world->options->acceptfreq * oldsteps)
	{
	  (*steps)++;
	}
    }
}


long 
set_repkind(option_fmt *options)
{
  if(options->replicate)
    {
      if(options->replicatenum==0)
	return MULTIPLECHAIN;	    
      else
	return MULTIPLERUN;	    
    }
  return SINGLECHAIN;
}

void
heating_init(world_fmt **venus,
	     world_fmt **merkur,world_fmt **sun,
	     data_fmt *data, option_fmt *options,
	     char *this_string)
{
  strcpy(this_string,"venus");
  create_world (venus,this_string,data->loci);
  init_world (*venus, data, options);
  strcpy(this_string,"merkur");
  create_world (merkur,this_string,data->loci);
  init_world (*merkur, data, options);
  strcpy(this_string,"sun");
  create_world (sun,this_string,data->loci);
  init_world (*sun, data, options);
}

void 
heating_prepare(world_fmt *world, world_fmt *sun,
		world_fmt *merkur, world_fmt *venus,
		option_fmt *options)
{
  world->heat[0]= options->heat[0];
  klone(world,sun,options->heat[3]);
  klone(world,merkur,options->heat[2]);
  klone(world,venus,options->heat[1]);
}

void heating_prepare2(world_fmt *world, world_fmt *sun,
		      world_fmt *merkur, world_fmt *venus)
{
  sun->G = merkur->G = venus->G = 0;
  clone_polish(world,sun);
  metropolize(sun,0);
  clone_polish(world,merkur);
  metropolize(merkur,0);
  clone_polish(world,venus);
  metropolize(venus,0);
}

long 
replicate_number(option_fmt *options, long chain, 
		 char type, long replicate)
{
  long rep=0;
  if(options->replicate && 
     options->replicatenum>0)
    rep = replicate;
  else
    {
      if(type=='l' && options->replicate)
	rep = chain;
      else
	rep = 0;
    }
  return rep;
}




  
