/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 H E L P E R     R O U T I N E S 
 
 some math stuff and 
 string and file manipulation routines
 

 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: tools.c,v 1.28 2000/07/26 23:58:54 beerli Exp $

-------------------------------------------------------*/
#include "migration.h"
#include "world.h"
#include "random.h"
#include "broyden.h"

#include <math.h>
#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif

/* prototypes ------------------------------------------- */
double lengthof (node * p);
node *crawlback (const node * theNode);
/*node *crawl(node * theNode); */
node *showtop (node * theNode);
void adjust_time (node * theNode, double tyme);
void insert_migr_node (world_fmt * world, node * up, node * down,
		       migr_table_fmt * migr_table, long *migr_table_counter);
void children (node * mother, node ** brother, node ** sister);
/* math tools */
double incompletegamma (double x, double alpha);
double polygamma (long n, double z);
void invert_matrix (double **a, long nsize);
boolean nrcheck (double **m, double **tm, double *v, long nrows, double *r1,
		 double *r2, boolean do_newton);
double rannor (double mean, double sd);
char lowercase (char c);
char uppercase (char c);
double sum (double *vector, long n);
void bootstrap (world_fmt * world);

void gamma_rates (double *rate, long categs, char *input);
void calc_gamma (double alpha, double *gama, long categs);

double mylgamma (double z);


/*filemanipulation */
void init_files (world_fmt * world, data_fmt * data, option_fmt * options);
void exit_files (world_fmt * world, data_fmt * data, option_fmt * options);
void openfile (FILE ** fp, char *filename, char *mode, const char *applic,
	       char *perm);
void read_savesum (world_fmt * world, option_fmt * options, data_fmt * data);
void write_savesum (world_fmt * world);

/* string manipulation */
void translate (char *text, char from, char to);
long count_words (char *text);

/* time reporting */
void get_time (char *nowstr, char ts[]);
/*printing aid */
void print_llike (double llike, char *strllike);

/* searching and finding*/
boolean find (long i, long *list, long listlen);

/* conversion between the parameter schemes*/

long mstart (long pop, long numpop);
long mend (long pop, long numpop);
long mmstart (long pop, long numpop);
long mmend (long pop, long numpop);
long mm2m (long frompop, long topop, long numpop);
void m2mm (long i, long numpop, long *frompop, long *topop);
long m2mml (long i, long numpop);
long m2mml2 (long i, long topop, long numpop);



/* private functions */
double alnorm (double x, int up);
void lu_decomp (double **m, long *indeks, long nrows);
void lu_substitution (double **m, long *indeks, double *v, long nrows);
double d1mach (long i);
long i1mach (long i);
int dpsifn (double *x, long *n, long kode, long m, double *ans, long *nz,
	    long *ierr);
double find_chi (long df, double prob);
double probchi (long df, double chi);
double chisquare (long df, double alpha);



/*FILEMANIPULATION======================================================= */
void
init_files (world_fmt * world, data_fmt * data, option_fmt * options)
{
  openfile (&data->infile, options->infilename, "r+", appl, NULL);
  openfile (&world->outfile, options->outfilename, "w+", appl, NULL);
  if(options->writelog)
    openfile (&options->logfile, options->logfilename, "w+", appl, NULL);
  if (options->usertree)
    openfile (&data->utreefile, options->utreefilename, "r+", appl, NULL);
  if (options->readsum)
    openfile (&data->sumfile, options->sumfilename, "r+", appl, NULL);
  if (options->writesum)
    openfile (&data->sumfile, options->sumfilename, "w+", appl, NULL);
  if (options->weights)
    openfile (&data->weightfile, options->weightfilename, "r+", appl, NULL);
  if (options->categs > 1)
    openfile (&data->catfile, options->catfilename, "r+", appl, NULL);
  if (options->treeprint > 0)
    openfile (&world->treefile, options->treefilename, "w+", appl, NULL);
  if (options->dist)
    openfile (&data->distfile, options->distfilename, "r+", appl, NULL);
  if (options->geo)
    openfile (&data->geofile, options->geofilename, "r+", appl, NULL);
  if (options->mighist)
    openfile (&world->mighistfile, options->mighistfilename, "w+", appl, NULL);
  if (options->plot)
    {
      switch (options->plotmethod)
	{
	case 0:
	  openfile (&world->mathfile, options->mathfilename, "w+", appl, NULL);
	  break;
	default:		/*e.g. 0 this create just the plots in outfile */
	  break;
	}
    }
}

void
exit_files (world_fmt * world, data_fmt * data, option_fmt * options)
{
  FClose (data->infile);
  FClose (world->outfile);
  if(options->writelog)
    FClose(options->logfile);
  if (options->weights)
    FClose (data->weightfile);
  if (options->categs > 1)
    FClose (data->catfile);
  if (options->treeprint)
    FClose (world->treefile);
  if (options->mighist)
    FClose (world->mighistfile);
  if (options->dist)
    FClose (data->distfile);
  if (options->geo)
    FClose (data->geofile);
  if (options->plot && options->plotmethod == PLOTALL)
    FClose (world->mathfile);
  if (options->writesum || options->readsum)
    FClose (data->sumfile);
}

/* string manipulation ================================== */
/* Converts any character from to character to in string text */
void
translate (char *text, char from, char to)
{
  int i, j, gap = 0;
  while (text[gap] == from)
    gap++;
  for (i = gap, j = 0; text[i] != '\0'; i++)
    {
      if (text[i] != from)
	{
	  text[j++] = text[i];
	}
      else
	{
	  if (text[i - 1] != from)
	    {
	      text[j++] = to;
	    }
	}
    }
  text[j] = '\0';
}

/*===============================================
  count words in a string delimited by delimiter
*/
long
count_words (char *text)
{
  long counts = 0;
  char *pt = text;
  while (isspace (*pt) && *pt != '\0')
    pt++;
  while (*pt != '\0')
    {
      while (!isspace (*pt) && *pt != '\0')
	pt++;
      while (isspace (*pt) && *pt != '\0')
	pt++;
      counts++;
    }
  return counts;
}


/*===============================================
 timer utility
 
 ts = "%c" -> time + full date (see man strftime)
      = "%H:%M:%S" -> time hours:minutes:seconds */

void
get_time (char *nowstr, char ts[])
{
#ifdef NOTIME_FUNC
  switch (strlen (ts))
    {
    case 2:
      strcpy (nowstr, " ");
      break;
    case 3:
      strcpy (nowstr, "  ");
      break;
    case 8:
      strcpy (nowstr, "        ");
      break;
    default:
      strcpy (nowstr, " ");
      break;
    }
#else
  time_t nowbin;
  struct tm *nowstruct;
  if (time (&nowbin) != (time_t) - 1)
    {
      nowstruct = localtime (&nowbin);
      strftime (nowstr, LINESIZE, ts, nowstruct);
    }
#endif
}

/*===============================================
 printer utility
 */
void
print_llike (double llike, char *strllike)
{
  if (fabs (llike) > 10e20)
    {
      sprintf (strllike, "%cInfinity ", llike < 0 ? '-' : ' ');
    }
  else
    sprintf (strllike, "%-10.5f", llike);
}

void
openfile (FILE ** fp, char *filename, char *mode, const char *application,
	  char *perm)
{
  int trials = 0;
  FILE *of = NULL;
  char file[100];
  strcpy (file, filename);
  while (trials++ < 10)
    {
      of = fopen (file, mode);
      if (of)
	break;
      else
	{
	  switch (*mode)
	    {
	    case 'r':
	      printf ("%s:  can't read %s\n", application, file);
	      file[0] = '\0';
	      while (file[0] == '\0')
		{
		  printf ("Please enter a new filename for reading>");
		  fgets (file, LINESIZE, stdin);
		}
	      break;
	    case 'w':
	      printf ("%s: can't write %s\n", application, file);
	      file[0] = '\0';
	      while (file[0] == '\0')
		{
		  printf ("Please enter a new filename for writing>");
		  fgets (file, LINESIZE, stdin);
		}
	      break;
	    }
	}
      file[strlen (file) - 1] = '\0';
    }
  if (trials >= 10)
    {
      printf ("You cannot your file either, so I stop\n\n");
      exit (0);
    }
  *fp = of;
  if (perm != NULL)
    strcpy (perm, file);
  strcpy (filename, file);

}

void
read_savesum (world_fmt * world, option_fmt * options, data_fmt * data)
{
  FILE *sumfile = data->sumfile;
  timearchive_fmt **ta;
  long nrep=0;
  long l, i, j, r;
  char input[1024];
  long hits;
  long tmp;
  long z,zz, jj;
  double tmpm;
  boolean  newstyle=TRUE;

  fgets (input, 1024, sumfile);
  if (!strncmp ("# begin genealogy-summary file of migrate", input, 41))
    {
      fgets(input,1024,sumfile);
      if(input[0]=='#')
	{
	  newstyle=TRUE;
	  fgets(input,1024,sumfile);
	}
      else
	newstyle=FALSE;
      hits = sscanf (input, "%li %li %li %li %li",
	      &world->loci, &world->numpop, &world->numpop2, &tmp,
		     &options->replicatenum);
      if(hits!=5)
	{
	  nrep=1;
	  hits = sscanf (input, "%li %li %li",
			 &world->loci, &world->numpop, &world->numpop2);
	  options->replicate = FALSE;
	  options->replicatenum = 0;
	}
      else
	{
	  nrep= options->replicatenum;
	  if(nrep==0)
	    nrep=1;
	  options->replicate = (boolean) tmp;
	}
      data->loci = world->loci;
      data->skiploci = (boolean *)
	realloc (data->skiploci, sizeof (boolean) * (data->loci + 1));
      memset (data->skiploci, 0, sizeof (boolean) * (data->loci + 1));
      init_world (world, data, options);
      ta = world->atl; 
      for (l = 0; l < world->loci; l++)
	for (r = 0; r < nrep; r++)
	{
	  fgets (input, 1024, sumfile);
	  fscanf (sumfile, "%li %li %li\n", &ta[r][l].T,
		  &ta[r][l].numpop, &ta[r][l].sumtips);
	  ta[r][l].allocT = 0.;
	  increase_timearchive (world, l, ta[r][l].T, world->numpop,r);
	  fscanf (sumfile, "%lg %lg %lg\n", &ta[r][l].param_like,
		  &ta[r][l].thb, &ta[r][l].alpha);
	  for (i = 0; i < world->atl[r][l].T; i++)
	    {
	      fscanf (sumfile, "%li ", &ta[r][l].tl[i].copies);
	      for (j = 0; j < world->numpop; j++)
		{
		  fscanf (sumfile, "%lf %lf %li\n",
			  &ta[r][l].tl[i].km[j], &ta[r][l].tl[i].kt[j],
			  &ta[r][l].tl[i].p[j]);
		}
	      if(newstyle)
		{
		  for (j = 0; j < world->numpop2; j++)
		    {
		      fscanf (sumfile, "%lf ", &ta[r][l].tl[i].mindex[j]);
		    }
		}
	      else
		{
		  z=0;
		  zz = world->numpop;
		  for (j = 0; j < world->numpop; j++)
		    {
		      for(jj=0; jj < world->numpop;jj++)
			{
			  fscanf (sumfile, "%lf ", &tmpm);
			  if(j==jj)
			    ta[r][l].tl[i].mindex[z++] = tmpm;
			  else
			    ta[r][l].tl[i].mindex[zz++] = tmpm;
			}
		    }
		}
	    }
	  for (i = 0; i < world->numpop2; i++)
	    {
	      fscanf (sumfile, "%lg %lg\n", &ta[r][l].param[i], &ta[r][l].param0[i]);
	    }
	  log_param0 (ta[r][l].param0, ta[r][l].lparam0, world->numpop2);
	  //for(i=0;i<ta[l].T;i++)
	  //  {
	  //    fscanf(sumfile,"%lg ",&ta[l].likelihood[i]);
	  //  }
	  //      fgets(input,1024,sumfile);
	  fscanf (sumfile, "%li %lg\n", &ta[r][l].trials, &ta[r][l].normd);
	}
    }
  else
    {
      error ("This is not a genealogy-summary file for MIGRATE\n");
    }
}

void
write_savesum (world_fmt * world)
{
  long r, repmax;
  FILE *sumfile = world->data->sumfile;
  timearchive_fmt **ta = world->atl;

  long l, i, j;
  if(world->options->replicate)
    {
      if(world->options->replicatenum==0)
	repmax = world->options->lchains;
      else
	repmax = world->options->replicatenum;
    }
  else
    repmax=1;
  fprintf (sumfile, "# begin genealogy-summary file of migrate %s ------\n#\n",
	   MIGRATEVERSION);
  fprintf (sumfile, "%li %li %li %li %li\n",
	   world->loci, world->numpop, world->numpop2,
	   (long) world->options->replicate, repmax);
  for (l = 0; l < world->loci; l++)
    {
      for(r=0;r<repmax;r++)
	{ 
	  fprintf (sumfile,
		   "%li %li ####### locus %li, replicate %li ################\n%li %li %li\n", l, r, l, r,
		   ta[r][l].T, ta[r][l].numpop, ta[r][l].sumtips);
	  fprintf (sumfile, "%20.20g %20.20g %20.20g\n", ta[r][l].param_like,
		   ta[r][l].thb, ta[r][l].alpha);
	  for (i = 0; i < ta[r][l].T; i++)
	    {
	      fprintf (sumfile, "%li ", ta[r][l].tl[i].copies);
	      for (j = 0; j < world->numpop; j++)
		{
		  fprintf (sumfile, "%20.20f %20.20f %li\n",
			   ta[r][l].tl[i].km[j], ta[r][l].tl[i].kt[j], 
			   ta[r][l].tl[i].p[j]);
		}
	      for (j = 0; j < world->numpop2; j++)
		{
		  fprintf (sumfile, "%f ", ta[r][l].tl[i].mindex[j]);
		  //debug               fprintf (sumfile, "%li ", ta[l].tl[i].l[j]);              
		}
	      fprintf (sumfile, "\n");
	    }
	  for (i = 0; i < world->numpop2; i++)
	    {
	      fprintf (sumfile, "%20.20e %20.20e\n", ta[r][l].param[i],
		       ta[r][l].param0[i]);
	    }
	  //  for(i=0;i<ta[l].T;i++)
	  //        {
	  //          fprintf(sumfile,"%20.20e ",ta[l].likelihood[i]);
	  //        }
	  //      fprintf(sumfile,"\n");
	  fprintf (sumfile, "%li %20.20e\n", ta[r][l].trials, ta[r][l].normd);
	}
    }
  fprintf (sumfile, "# end genealogy-summary file of migrate %s ------\n",
	   MIGRATEVERSION);
  fflush (sumfile);
}
  

/*=======================================================*/




/*--------------------------------
creates the length value in a node
*/
double
lengthof (node * p)
{
  if (p->type == 'm')
    error ("A migration node was feed into lengthof");
  return fabs (p->tyme - crawlback (p)->tyme);
}				/* length */


/*------------------------------------------------
Find the next non-migration node starting
with the theNode, returns to backnode which is not 
a migration, does NOT return always a top-node!
*/
node *
crawlback (const node * theNode)
{
  node *tmp = theNode->back;

  while (tmp->type == 'm')
    {
      tmp = tmp->next->back;
    }
  return tmp;
}

/*--------------------------------------------
returns the last migration node in a branch or 
the node if there is no migration node

node *crawl(node * theNode)
{
   node *otmp, *tmp = theNode->back;

   otmp = theNode;
   if (tmp == NULL)
	  return otmp;
   while (tmp->type == 'm') {
	  otmp = tmp->next;
	  tmp = tmp->next->back;
	  if (tmp == NULL)
		 return otmp;
   }
   return otmp;
}
*/


node *
showtop (node * theNode)
{
  if (theNode == NULL)
    return NULL;
  else
    {
      if (theNode->top)
	{
	  return theNode;
	}
      else
	{
	  if (theNode->next->top)
	    {
	      return theNode->next;
	    }
	  else
	    {
	      return theNode->next->next;
	    }
	}
    }

}

/* adjust the time in a node to time */
void
adjust_time (node * theNode, double tyme)
{
  switch (theNode->type)
    {
    case 'm':
      theNode->tyme = theNode->next->tyme = tyme;
      break;
    case 'i':
      theNode->tyme = theNode->next->tyme = theNode->next->next->tyme = tyme;
      break;
    case 'r':
    case 't':
      break;
    default:
      error ("Wrong node type");
      break;

    }
}

void
insert_migr_node (world_fmt * world, node * up, node * down,
		  migr_table_fmt * migr_table, long *migr_table_counter)
{
  long i, panic;
  node *tmp, *tmp2, *oldNode, *oldNode2, *theNode;
  if (!up->top)
    error ("up has to be a top-node");
  theNode = showtop (up)->back;
  if (*migr_table_counter > 0 && up->tyme > migr_table[0].time)
    error
      ("insert_migr_node: the first migration node has a wrong time for up");
  if (migr_table[(*migr_table_counter) - 1].from != down->actualpop)
    {
      error ("this should never happen -> wrong choice of nodes\n");
      (*migr_table_counter)--;
    }
  if (((*migr_table_counter) > 0)
      && (migr_table[(*migr_table_counter) - 1].from != down->actualpop))
    error ("problem catched in inser_migr_table");
  for (i = 0; i < (*migr_table_counter); i++)
    {
      tmp = (node *) calloc (1, sizeof (node));
      tmp2 = (node *) calloc (1, sizeof (node));
      oldNode = up;
      theNode = up->back;
      panic = 0;
      while (theNode->tyme < migr_table[i].time && panic++ < 10000)
	{
	  if (theNode->tip && theNode->type != 'r')
	    {
	      oldNode = theNode;
	      theNode = theNode->back;
	    }
	  else
	    {
	      oldNode = theNode->next;
	      theNode = theNode->next->back;
	    }
	}
      tmp->back = oldNode;
      oldNode->back = tmp;
      tmp->number = -999;
      tmp->nayme = NULL;
      tmp->tip = 0;
      tmp->top = 0;
      tmp->dirty = TRUE;
      tmp->id = world->unique_id++;
      tmp->tyme = migr_table[i].time;
      tmp->type = 'm';
      tmp->actualpop = migr_table[i].to;
      tmp->pop = migr_table[i].from;
      tmp2->tyme = migr_table[i].time;
      tmp2->type = 'm';
      tmp2->id = world->unique_id++;
      tmp2->actualpop = migr_table[i].to;
      tmp2->pop = migr_table[i].from;
      tmp->next = tmp2;
      tmp2->next = tmp;
      tmp2->top = 1;

      oldNode2 = down;
      theNode = down->back;
      while (theNode->tyme > migr_table[i].time)
	{
	  oldNode2 = theNode->next;
	  theNode = theNode->next->back;
	}
      tmp2->back = oldNode2;
      oldNode2->back = tmp2;
    }
}


void
children (node * mother, node ** brother, node ** sister)
{
  node *m;

  m = showtop (mother);

  if (m->type == 't')
    {
      error ("this is a tip, so there are no more child nodes\n");
    }
  else
    {
      (*brother) = crawlback (m->next);
      (*sister) = crawlback (m->next->next);
    }
}

  /*       Uses Lanczos-type approximation to ln(gamma) for z > 0. */
  /*       Reference: */
  /*            Lanczos, C. 'A precision approximation of the gamma */
  /*                    function', J. SIAM Numer. Anal., B, 1, 86-96, 1964. */
  /*       Accuracy: About 14 significant digits except for small regions */
  /*                 in the vicinity of 1 and 2. */
  /*       Programmer: Alan Miller */
  /*                   CSIRO Division of Mathematics & Statistics */
  /*       Latest revision - 17 April 1988 */
  /* translated and modified into C by Peter Beerli 1997 */
double
mylgamma (double z)
{
  double a[9] = { 0.9999999999995183, 676.5203681218835,
    -1259.139216722289, 771.3234287757674, -176.6150291498386,
    12.50734324009056, -0.1385710331296526, 9.934937113930748e-6,
    1.659470187408462e-7
  };
  double lnsqrt2pi = 0.9189385332046727;
  double result;
  long j;
  double tmp;
  if (z <= 0.)
    {
      return DBL_MAX;/*this will kill the receiving calculation */
    }
  result = 0.;
  tmp = z + 7.;
  for (j = 9; j >= 2; --j)
    {
      result += a[j - 1] / tmp;
      tmp -= 1.;
    }
  result += a[0];
  result = log (result) + lnsqrt2pi - (z + 6.5) + (z - 0.5) * log (z + 6.5);
  return result;
}				/* lgamma */

/* ALGORITHM AS239  APPL. STATIST. (1988) VOL. 37, NO. 3 
   Computation of the Incomplete Gamma Integral 
   Auxiliary functions required: lgamma() = logarithm of the gamma 
   function, and alnorm() = algorithm AS66 
   in Mathematica this is GammaRegularized[a,0,x] === Gamma[a,0,x]/Gamma[a]
 */
double
incompletegamma (double x, double alpha)
{
  double gama, d_1, d_2, d_3;
  /*static*/ double a, b, c, an, rn;
  /*static*/ double pn1, pn2, pn3, pn4, pn5, pn6, arg;

  gama = 0.;
  /*  Check that we have valid values for X and P */
  if (alpha <= 0. || x < 0.)
      error ("failed in imcompletegamma(): wrong alpha or x\n");
  if (x == 0.)
      return gama;

  /*  Use a normal approximation if P > PLIMIT */
  if (alpha > 1e3)
    {
      pn1 =
	sqrt (alpha) * 3. * (pow (x / alpha, (1. / 3.)) + 
			     1. / (alpha * 9.) - 1.);
      gama = alnorm (pn1, FALSE);
      return gama;
    }

  /*  If X is extremely large compared to P then set GAMMAD = 1 */
  if (x > 1e8)
    {
      gama = 1.;
      return gama;
    }

  if (x <= 1. || x < alpha)
    {
      /*  Use Pearson's series expansion. */
      /*  (Note that P is not large enough to force overflow in lgamma()). */
      arg = alpha * log (x) - x - LGAMMA (alpha + 1.);
      c = 1.;
      gama = 1.;
      a = alpha;
      while (c > 1e-14)
	{
	  a += 1.;
	  c = c * x / a;
	  gama += c;
	}
      arg += log (gama);
      gama = 0.;
      if (arg >= -88.)
	{
	  gama = exp (arg);
	}

    }
  else
    {
      /*  Use a continued fraction expansion */
      arg = alpha * log (x) - x - LGAMMA (alpha);
      a = 1. - alpha;
      b = a + x + 1.;
      c = 0.;
      pn1 = 1.;
      pn2 = x;
      pn3 = x + 1.;
      pn4 = x * b;
      gama = pn3 / pn4;
      for (;;)
	{
	  a += 1.;
	  b += 2.;
	  c += 1.;
	  an = a * c;
	  pn5 = b * pn3 - an * pn1;
	  pn6 = b * pn4 - an * pn2;
	  if (fabs (pn6) > 0.)
	    {
	      rn = pn5 / pn6;
	      /* Computing MIN */
	      d_2 = 1e-14, d_3 = rn * 1e-14;
	      if ((d_1 = gama - rn, fabs (d_1)) <= MIN (d_2, d_3))
		{
		  arg += log (gama);
		  gama = 1.;
		  if (arg >= -88.)
		    {
		      gama = 1. - exp (arg);
		    }
		  return gama;
		}
	      gama = rn;
	    }
	  pn1 = pn3;
	  pn2 = pn4;
	  pn3 = pn5;
	  pn4 = pn6;
	  if (fabs (pn5) >= 1e37)
	    {
	      /*  Re-scale terms in continued fraction if terms are large */
	      pn1 /= 1e37;
	      pn2 /= 1e37;
	      pn3 /= 1e37;
	      pn4 /= 1e37;
	    }
	}
    }
  return gama;
}				/* incompletegamma() */


/* calculation is replaced by the correct function in 
   polygamma.c (which is a translation of a fortran program by amos

   driver for the polygamma calculation */
double
polygamma (long n, double z)
{
  double ans;
  long nz, ierr;
  dpsifn (&z, &n, 1, 1, &ans, &nz, &ierr);
  if (n == 0)
    return -ans;
  else
    return ans;
}

/*-------------------------------------------------------*/
/* nrcheck subroutine (used in damped newton raphson proc */
/* syntax: nrcheck(matrix,inversematrix,ncols=nrows,returnval1,returnval2) */
/* mai 95 PB                                             */
boolean
nrcheck (double **m, double **tm, double *v, long nrows, double *r1,
	 double *r2, boolean do_newton)
{
  long i, j, k;
  double *tmp, *tmp2, tmp3 = 0.0, tmp4 = 0.0;
  tmp = (double *) calloc (1, sizeof (double) * nrows);
  tmp2 = (double *) calloc (1, sizeof (double) * nrows);
  /*first evaluate r1 */
  (*r1) = (*r2) = 0.0;
  for (i = 0; i < nrows; i++)
    {
      (*r1) += v[i] * v[i];
    }
  /*                                       T    */
  for (j = 0; j < nrows; j++)
    {				/* g . G */
      for (k = 0; k < nrows; k++)
	{
	  tmp[j] += v[k] * m[j][k];
	  tmp2[j] += v[k] * tm[j][k];
	}
    }
  /*                                       T        */
  for (i = 0; i < nrows; i++)
    {				/* g . G . g */
      (*r2) += tmp[i] * v[i];
      tmp3 += tmp2[i] * v[i];
    }
  tmp4 = log (fabs ((*r1)));
  tmp4 = tmp4 + tmp4 - log (fabs ((*r2)));
  tmp4 = ((*r2) < 0 ? -1 : 1) * exp (tmp4);
  free (tmp);
  if (do_newton && (tmp3 > (tmp4 > 0 ? tmp4 : 0)))
    {
      memcpy (v, tmp2, sizeof (double) * nrows);
      free (tmp2);
      return TRUE;
    }
  free (tmp2);
  return FALSE;
}


/*-------------------------------------------------------*/
/* Matrix inversion subroutine                           */
/* The passed matrix will be replaced by its inverse!!!!! */
/* Gauss-Jordan reduction -- invert matrix a in place,   */
/* overwriting previous contents of a.  On exit, matrix a */
/* contains the inverse.                                 */
void
invert_matrix (double **a, long nsize)
{
  long i, j;
  long *indeks;
  double *column, **result;
  indeks = (long *) malloc (sizeof (long) * nsize);
  column = (double *) malloc (sizeof (double) * nsize);
  result = (double **) malloc (sizeof (double *) * nsize);
  for (i = 0; i < nsize; i++)
    {
      result[i] = (double *) malloc (sizeof (double) * nsize);
    }
  lu_decomp (a, indeks, nsize);
  for (j = 0; j < nsize; j++)
    {
      memset (column, 0, sizeof (double) * nsize);
      column[j] = 1.0;
      lu_substitution (a, indeks, column, nsize);
      for (i = 0; i < nsize; i++)
	result[i][j] = column[i];
    }
  for (i = 0; i < nsize; i++)
    {
      memcpy (a[i], result[i], sizeof (double) * nsize);
      free (result[i]);
    }
  free (result);
  free (column);
  free (indeks);
}

/*=======================================================*/

/*-------------------------------------------------------*/
/* LU decomposition                                      */
/* after Dahlquist et al. 1974 and Press et al. 1988     */
/* the method's uses Crout's procedure and the pivoting  */
/* described in Press et al.                             */
/* Syntax: lu_decomp(matrix, indeks, nrows)               */
/* matrix will be destroyed and filled with the two      */
/* triangular matrices, indeks is the index vector for the */
/* pivoting and the row change in case of 0 pivot values */
/* nrows is the number of rows and columns in matrix     */
/* april 95 PB                                           */
void
lu_decomp (double **m, long *indeks, long nrows)
{
  long i, j, k, p, kmax = -1;
  double *max_row_vals, big, summ, pivot, bigt;
  max_row_vals = (double *) calloc (1, sizeof (double) * nrows);
  for (i = 0; i < nrows; i++)
    {
      big = 0.0;
      for (j = 0; j < nrows; j++)
	{
	  if ((bigt = fabs (m[i][j])) > big)
	    big = bigt;
	}
      max_row_vals[i] = 1.0 / big;
      if (big == 0.0)
	{
	  error ("Singular matrix detected in lu_decomp\n");
	}
    }
  for (i = 0; i < nrows; i++)
    {
      for (k = 0; k < i; k++)
	{			/* upper half of matrix */
	  summ = m[k][i];
	  for (p = 0; p < k; p++)
	    summ -= m[k][p] * m[p][i];
	  m[k][i] = summ;
	}
      big = 0.0;
      for (k = i; k < nrows; k++)
	{			/* lower half of matrix */
	  summ = m[k][i];
	  for (p = 0; p < i; p++)
	    summ -= m[k][p] * m[p][i];
	  m[k][i] = summ;
	  pivot = fabs (summ) /**max_row_vals[k]*/ ;
	  /*  fprintf(stdout,"i=%li,pivot=%f,big=%f\n",i,pivot,big); */
	  if (pivot >= big)
	    {
	      big = pivot;
	      kmax = k;
	    }
	}
      if (i != kmax)
	{
	  for (p = 0; p < nrows; p++)
	    {
	      pivot = m[kmax][p];
	      m[kmax][p] = m[i][p];
	      m[i][p] = pivot;
	    }
	  max_row_vals[kmax] = max_row_vals[i];
	}
      indeks[i] = kmax;
      if (m[i][i] == 0.0)
	m[i][i] = SMALL_VALUE;
      if (i != nrows - 1)
	{
	  pivot = 1. / m[i][i];
	  for (k = i + 1; k < nrows; k++)
	    m[k][i] *= pivot;
	}
    }
  free (max_row_vals);
}				/* end of lu_decomp */

/*-------------------------------------------------------*/
/* LU substitution                                       */
/* after Dahlquist et al. 1974 and Press et al. 1988     */
/* needs first the evaluation LU decomposition           */
/* Syntax: lu_substition(matrix, indeks, vector, nrows)   */
/* matrix = LU decomposed matrix, indeks = order of matrix */
/* vector = value vevtor, nrows = number of rows/columns */
/* april 95 PB                                           */
void
lu_substitution (double **m, long *indeks, double *v, long nrows)
{
  long i, j;
  double summ;
  for (i = 0; i < nrows; i++)
    {
      summ = v[indeks[i]];
      v[indeks[i]] = v[i];
      for (j = 0; j < i; j++)
	summ -= m[i][j] * v[j];
      v[i] = summ;
    }
  for (i = nrows - 1; i >= 0; i--)
    {
      summ = v[i];
      for (j = i + 1; j < nrows; j++)
	summ -= m[i][j] * v[j];
      v[i] = summ / m[i][i];
    }
}


/* Algorithm AS66 Applied Statistics (1973) vol22 no.3
   Evaluates the tail area of the standardised normal curve
   from x to infinity if upper is .true. or
   from minus infinity to x if upper is .false. */
double
alnorm (double x, int up)
{
  /* Initialized data */
  /* *** machine dependent constants ????????????? */
  /*static*/ double zero = 0.;
  /*static*/ double a1 = 5.75885480458;
  /*static*/ double a2 = 2.62433121679;
  /*static*/ double a3 = 5.92885724438;
  /*static*/ double b1 = -29.8213557807;
  /*static*/ double b2 = 48.6959930692;
  /*static*/ double c1 = -3.8052e-8;
  /*static*/ double c2 = 3.98064794e-4;
  /*static*/ double c3 = -.151679116635;
  /*static*/ double c4 = 4.8385912808;
  /*static*/ double c5 = .742380924027;
  /*static*/ double one = 1.;
  /*static*/ double c6 = 3.99019417011;
  /*static*/ double d1 = 1.00000615302;
  /*static*/ double d2 = 1.98615381364;
  /*static*/ double d3 = 5.29330324926;
  /*static*/ double d4 = -15.1508972451;
  /*static*/ double d5 = 30.789933034;
  /*static*/ double half = .5;
  /*static*/ double ltone = 7.;
  /*static*/ double utzero = 18.66;
  /*static*/ double con = 1.28;
  /*static*/ double p = .398942280444;
  /*static*/ double q = .39990348504;
  /*static*/ double r = .398942280385;

  /*static*/ double y, result;

  if (x < zero)
    {
      up = !up;
      x = -x;
    }
  if (x <= ltone || (up && x <= utzero))
    {
      y = half * x * x;
      if (x > con)
	{
	  result =
	    r * exp (-y) / (x + c1 +
			    d1 / (x + c2 +
				  d2 / (x + c3 +
					d3 / (x + c4 +
					      d4 / (x + c5 + d5 / (x + c6))))));
	  return ((!up) ? one - result : result);
	}
      result =
	half - x * (p - q * y / (y + a1 + b1 / (y + a2 + b2 / (y + a3))));
      return ((!up) ? one - result : result);
    }
  else
    {
      return ((!up) ? 1.0 : 0.);
    }
  /*fake */ return -99;
}				/* alnorm */

/* dpsifn.c -- translated by f2c (version 19950808).
   and hand-patched by Peter Beerli Seattle, 1996
   SUBROUTINE DPSIFN (X, N, KODE, M, ANS, NZ, IERR)

   C***BEGIN PROLOGUE  DPSIFN
   C***PURPOSE  Compute derivatives of the Psi function.
   C***LIBRARY   SLATEC
   C***CATEGORY  C7C
   C***TYPE      double PRECISION (PSIFN-S, DPSIFN-D)
   C***KEYWORDS  DERIVATIVES OF THE GAMMA FUNCTION, POLYGAMMA FUNCTION,
   C             PSI FUNCTION
   C***AUTHOR  Amos, D. E., (SNLA)
   C***DESCRIPTION
   C
   C         The following definitions are used in DPSIFN:
   C
   C      Definition 1
   C         PSI(X) = d/dx (ln(GAMMA(X)), the first derivative of
   C                  the log GAMMA function.
   C      Definition 2
   C                     K   K
   C         PSI(K,X) = d /dx (PSI(X)), the K-th derivative of PSI(X).
   C   ___________________________________________________________________
   C      DPSIFN computes a sequence of SCALED derivatives of
   C      the PSI function; i.e. for fixed X and M it computes
   C      the M-member sequence
   C
   C                    ((-1)**(K+1)/GAMMA(K+1))*PSI(K,X)
   C                       for K = N,...,N+M-1
   C
   C      where PSI(K,X) is as defined above.   For KODE=1, DPSIFN returns
   C      the scaled derivatives as described.  KODE=2 is operative only
   C      when K=0 and in that case DPSIFN returns -PSI(X) + LN(X).  That
   C      is, the logarithmic behavior for large X is removed when KODE=2
   C      and K=0.  When sums or differences of PSI functions are computed
   C      the logarithmic terms can be combined analytically and computed
   C      separately to help retain significant digits.
   C
   C         Note that CALL DPSIFN(X,0,1,1,ANS) results in
   C                   ANS = -PSI(X)
   C
   C     Input      X is double PRECISION
   C           X      - Argument, X .gt. 0.0D0
   C           N      - First member of the sequence, 0 .le. N .le. 100
   C                    N=0 gives ANS(1) = -PSI(X)       for KODE=1
   C                                       -PSI(X)+LN(X) for KODE=2
   C           KODE   - Selection parameter
   C                    KODE=1 returns scaled derivatives of the PSI
   C                    function.
   C                    KODE=2 returns scaled derivatives of the PSI
   C                    function EXCEPT when N=0. In this case,
   C                    ANS(1) = -PSI(X) + LN(X) is returned.
   C           M      - Number of members of the sequence, M.ge.1
   C
   C    Output     ANS is double PRECISION
   C           ANS    - A vector of length at least M whose first M
   C                    components contain the sequence of derivatives
   C                    scaled according to KODE.
   C           NZ     - Underflow flag
   C                    NZ.eq.0, A normal return
   C                    NZ.ne.0, Underflow, last NZ components of ANS are
   C                             set to zero, ANS(M-K+1)=0.0, K=1,...,NZ
   C           IERR   - Error flag
   C                    IERR=0, A normal return, computation completed
   C                    IERR=1, Input error,     no computation
   C                    IERR=2, Overflow,        X too small or N+M-1 too
   C                            large or both
   C                    IERR=3, Error,           N too large. Dimensioned
   C                            array TRMR(NMAX) is not large enough for N
   C
   C         The nominal computational accuracy is the maximum of unit
   C         roundoff (=D1MACH(4)) and 1.0D-18 since critical constants
   C         are given to only 18 digits.
   C
   C         PSIFN is the single precision version of DPSIFN.
   C
   C *Long Description:
   C
   C         The basic method of evaluation is the asymptotic expansion
   C         for large X.ge.XMIN followed by backward recursion on a two
   C         term recursion relation
   C
   C                  W(X+1) + X**(-N-1) = W(X).
   C
   C         This is supplemented by a series
   C
   C                  SUM( (X+K)**(-N-1) , K=0,1,2,... )
   C
   C         which converges rapidly for large N. Both XMIN and the
   C         number of terms of the series are calculated from the unit
   C         roundoff of the machine environment.
   C
   C***REFERENCES  Handbook of Mathematical Functions, National Bureau
   C                 of Standards Applied Mathematics Series 55, edited
   C                 by M. Abramowitz and I. A. Stegun, equations 6.3.5,
   C                 6.3.18, 6.4.6, 6.4.9 and 6.4.10, pp.258-260, 1964.
   C               D. E. Amos, A portable Fortran subroutine for
   C                 derivatives of the Psi function, Algorithm 610, ACM
   C                 Transactions on Mathematical Software 9, 4 (1983),
   C                 pp. 494-502.
   C***ROUTINES CALLED  D1MACH, I1MACH
   C***REVISION HISTORY  (YYMMDD)
   C   820601  DATE WRITTEN
   C   890531  Changed all specific intrinsics to generic.  (WRB)
   C   890911  Removed unnecessary intrinsics.  (WRB)
   C   891006  Cosmetic changes to prologue.  (WRB)
   C   891006  REVISION DATE from Version 3.2
   C   891214  Prologue converted to Version 4.0 format.  (BAB)
   C   920501  Reformatted the REFERENCES section.  (WRB)
   C***END PROLOGUE  DPSIFN


 */

/*static*/ long fifteen = 15;
/*static*/ long sixteen = 16;
/*static*/ long five = 5;
/*static*/ long four = 4;
/*static*/ long fourteen = 14;

double
d1mach (long i)
{
  switch (i)
    {
    case 1:
      return DBL_MIN;
    case 2:
      return DBL_MAX;
    case 3:
      return DBL_EPSILON / FLT_RADIX;
    case 4:
      return DBL_EPSILON;
    case 5:
      return log10 (FLT_RADIX);
    }
  usererror ("invalid argument: d1mach(%ld)\n", i);
  return 0;			/* for compilers that complain of missing return values */
}

long
i1mach (long i)
{
  switch (i)
    {
    case 1:
      return 5;			/* standard input */
    case 2:
      return 6;			/* standard output */
    case 3:
      return 7;			/* standard punch */
    case 4:
      return 0;			/* standard error */
    case 5:
      return 32;		/* bits per integer */
    case 6:
      return 1;			/* Fortran 77 value */
    case 7:
      return 2;			/* base for integers */
    case 8:
      return 31;		/* digits of integer base */
    case 9:
      return LONG_MAX;
    case 10:
      return FLT_RADIX;
    case 11:
      return FLT_MANT_DIG;
    case 12:
      return FLT_MIN_EXP;
    case 13:
      return FLT_MAX_EXP;
    case 14:
      return DBL_MANT_DIG;
    case 15:
      return DBL_MIN_EXP;
    case 16:
      return DBL_MAX_EXP;
    }
  usererror ("invalid argument: i1mach(%ld)\n", i);
  return 0;			/* for compilers that complain of missing return values */
}

int
dpsifn (double *x, long *n, long kode, long m, double *ans, long *nz,
	long *ierr)
{
  /* Initialized data */

  /*static*/ long nmax = 100;
  /*static*/ double b[22] = { 1., -.5, .166666666666666667,
    -.0333333333333333333, .0238095238095238095, -.0333333333333333333,
    .0757575757575757576, -.253113553113553114, 1.16666666666666667,
    -7.09215686274509804, 54.9711779448621554, -529.124242424242424,
    6192.1231884057971, -86580.2531135531136, 1425517.16666666667,
    -27298231.067816092, 601580873.900642368, -15116315767.0921569,
    429614643061.166667, -13711655205088.3328, 488332318973593.167,
    -19296579341940068.1
  };

  /* System generated locals */
  long i1, i2;
  double d1, d2;


  /* Local variables */
  /*static*/ double elim, xinc, xmin, tols, xdmy, yint, trmr[100], rxsq;
  /*static*/ long i__, j, k;
  /*static*/ double s, t, slope, xdmln, wdtol;
  /*static*/ double t1, t2;
  /*static*/ long fn;
  /*static*/ double ta;
  /*static*/ long mm, nn, np;
  /*static*/ double fx, tk;
  /*static*/ long mx, nx;
  /*static*/ double xm, tt, xq, den, arg, fln, r1m4, r1m5, eps, rln, tol, xln,
    trm[22], tss, tst;

  /* Parameter adjustments */
  --ans;

  /* Function Body */
/* ----------------------------------------------------------------------- */
/*             BERNOULLI NUMBERS */
/* ----------------------------------------------------------------------- */

/* ***FIRST EXECUTABLE STATEMENT  DPSIFN */
  *ierr = 0;
  *nz = 0;
  if (*x <= 0.)
    {
      *ierr = 1;
    }
  if (*n < 0)
    {
      *ierr = 1;
    }
  if (kode < 1 || kode > 2)
    {
      *ierr = 1;
    }
  if (m < 1)
    {
      *ierr = 1;
    }
  if (*ierr != 0)
    {
      return 0;
    }
  mm = m;
/* Computing MIN */
  i1 = -fifteen, i2 = sixteen;
  nx = MIN (-i1mach (fifteen), i1mach (sixteen));
  r1m5 = d1mach (five);
  r1m4 = d1mach (four) * .5;
  wdtol = MAX (r1m4, 5e-19);
/* ----------------------------------------------------------------------- */
/*     ELIM = APPROXIMATE EXPONENTIAL OVER AND UNDERFLOW LIMIT */
/* ----------------------------------------------------------------------- */
  elim = (nx * r1m5 - 3.) * 2.302;
  xln = log (*x);
L41:
  nn = *n + mm - 1;
  fn = nn;
  t = (fn + 1) * xln;
/* ----------------------------------------------------------------------- */
/*     OVERFLOW AND UNDERFLOW TEST FOR SMALL AND LARGE X */
/* ----------------------------------------------------------------------- */
  if (fabs (t) > elim)
    {
      goto L290;
    }
  if (*x < wdtol)
    {
      goto L260;
    }
/* ----------------------------------------------------------------------- */
/*     COMPUTE XMIN AND THE NUMBER OF TERMS OF THE SERIES, FLN+1 */
/* ----------------------------------------------------------------------- */
  rln = r1m5 * i1mach (fourteen);
  rln = MIN (rln, 18.06);
  fln = MAX (rln, 3.) - 3.;
  yint = fln * .4 + 3.5;
  slope = fln * (fln * 6.038e-4 + .008677) + .21;
  xm = yint + slope * fn;
  mx = (long) xm + 1;
  xmin = (double) mx;
  if (*n == 0)
    {
      goto L50;
    }
  xm = rln * -2.302 - MIN (0., xln);
  arg = xm / *n;
  arg = MIN (0., arg);
  eps = exp (arg);
  xm = 1. - eps;
  if (fabs (arg) < .001)
    {
      xm = -arg;
    }
  fln = *x * xm / eps;
  xm = xmin - *x;
  if (xm > 7. && fln < 15.)
    {
      goto L200;
    }
L50:
  xdmy = *x;
  xdmln = xln;
  xinc = 0.;
  if (*x >= xmin)
    {
      goto L60;
    }
  nx = (long) (*x);
  xinc = xmin - nx;
  xdmy = *x + xinc;
  xdmln = log (xdmy);
L60:
/* ----------------------------------------------------------------------- */
/*     GENERATE W(N+MM-1,X) BY THE ASYMPTOTIC EXPANSION */
/* ----------------------------------------------------------------------- */
  t = fn * xdmln;
  t1 = xdmln + xdmln;
  t2 = t + xdmln;
/* Computing MAX */
  d1 = fabs (t), d2 = fabs (t1), d1 = MAX (d1, d2), d2 = fabs (t2);
  tk = MAX (d1, d2);
  if (tk > elim)
    {
      goto L380;
    }
  tss = exp (-t);
  tt = .5 / xdmy;
  t1 = tt;
  tst = wdtol * tt;
  if (nn != 0)
    {
      t1 = tt + 1. / fn;
    }
  rxsq = 1. / (xdmy * xdmy);
  ta = rxsq * .5;
  t = (fn + 1) * ta;
  s = t * b[2];
  if (fabs (s) < tst)
    {
      goto L80;
    }
  tk = 2.;
  for (k = 4; k <= 22; ++k)
    {
      t = t * ((tk + fn + 1) / (tk + 1.)) * ((tk + fn) / (tk + 2.)) * rxsq;
      trm[k - 1] = t * b[k - 1];
      if ((d1 = trm[k - 1], fabs (d1)) < tst)
	{
	  goto L80;
	}
      s += trm[k - 1];
      tk += 2.;
/* L70: */
    }
L80:
  s = (s + t1) * tss;
  if (xinc == 0.)
    {
      goto L100;
    }
/* ----------------------------------------------------------------------- */
/*     BACKWARD RECUR FROM XDMY TO X */
/* ----------------------------------------------------------------------- */
  nx = (long) xinc;
  np = nn + 1;
  if (nx > nmax)
    {
      goto L390;
    }
  if (nn == 0)
    {
      goto L160;
    }
  xm = xinc - 1.;
  fx = *x + xm;
/* ----------------------------------------------------------------------- */
/*     THIS LOOP SHOULD NOT BE CHANGED. FX IS ACCURATE WHEN X IS SMALL */
/* ----------------------------------------------------------------------- */
  i1 = nx;
  for (i__ = 1; i__ <= i1; ++i__)
    {
      i2 = -np;
      trmr[i__ - 1] = pow (fx, i2);
      s += trmr[i__ - 1];
      xm += -1.;
      fx = *x + xm;
/* L90: */
    }
L100:
  ans[mm] = s;
  if (fn == 0)
    {
      goto L180;
    }
/* ----------------------------------------------------------------------- */
/*     GENERATE LOWER DERIVATIVES, J.LT.N+MM-1 */
/* ----------------------------------------------------------------------- */
  if (mm == 1)
    {
      return 0;
    }
  i1 = mm;
  for (j = 2; j <= i1; ++j)
    {
      --fn;
      tss *= xdmy;
      t1 = tt;
      if (fn != 0)
	{
	  t1 = tt + 1. / fn;
	}
      t = (fn + 1) * ta;
      s = t * b[2];
      if (fabs (s) < tst)
	{
	  goto L120;
	}
      tk = (double) (fn + 4);
      for (k = 4; k <= 22; ++k)
	{
	  trm[k - 1] = trm[k - 1] * (fn + 1) / tk;
	  if ((d1 = trm[k - 1], fabs (d1)) < tst)
	    {
	      goto L120;
	    }
	  s += trm[k - 1];
	  tk += 2.;
/* L110: */
	}
    L120:
      s = (s + t1) * tss;
      if (xinc == 0.)
	{
	  goto L140;
	}
      if (fn == 0)
	{
	  goto L160;
	}
      xm = xinc - 1.;
      fx = *x + xm;
      i2 = nx;
      for (i__ = 1; i__ <= i2; ++i__)
	{
	  trmr[i__ - 1] *= fx;
	  s += trmr[i__ - 1];
	  xm += -1.;
	  fx = *x + xm;
/* L130: */
	}
    L140:
      mx = mm - j + 1;
      ans[mx] = s;
      if (fn == 0)
	{
	  goto L180;
	}
/* L150: */
    }
  return 0;
/* ----------------------------------------------------------------------- */
/*     RECURSION FOR N = 0 */
/* ----------------------------------------------------------------------- */
L160:
  i1 = nx;
  for (i__ = 1; i__ <= i1; ++i__)
    {
      s += 1. / (*x + nx - i__);
/* L170: */
    }
L180:
  if (kode == 2)
    {
      goto L190;
    }
  ans[1] = s - xdmln;
  return 0;
L190:
  if (xdmy == *x)
    {
      return 0;
    }
  xq = xdmy / *x;
  ans[1] = s - log (xq);
  return 0;
/* ----------------------------------------------------------------------- */
/*     COMPUTE BY SERIES (X+K)**(-(N+1)) , K=0,1,2,... */
/* ----------------------------------------------------------------------- */
L200:
  nn = (long) fln + 1;
  np = *n + 1;
  t1 = (*n + 1) * xln;
  t = exp (-t1);
  s = t;
  den = *x;
  i1 = nn;
  for (i__ = 1; i__ <= i1; ++i__)
    {
      den += 1.;
      i2 = -np;
      trm[i__ - 1] = pow (den, i2);
      s += trm[i__ - 1];
/* L210: */
    }
  ans[1] = s;
  if (*n != 0)
    {
      goto L220;
    }
  if (kode == 2)
    {
      ans[1] = s + xln;
    }
L220:
  if (mm == 1)
    {
      return 0;
    }
/* ----------------------------------------------------------------------- */
/*     GENERATE HIGHER DERIVATIVES, J.GT.N */
/* ----------------------------------------------------------------------- */
  tol = wdtol / 5.;
  i1 = mm;
  for (j = 2; j <= i1; ++j)
    {
      t /= *x;
      s = t;
      tols = t * tol;
      den = *x;
      i2 = nn;
      for (i__ = 1; i__ <= i2; ++i__)
	{
	  den += 1.;
	  trm[i__ - 1] /= den;
	  s += trm[i__ - 1];
	  if (trm[i__ - 1] < tols)
	    {
	      goto L240;
	    }
/* L230: */
	}
    L240:
      ans[j] = s;
/* L250: */
    }
  return 0;
/* ----------------------------------------------------------------------- */
/*     SMALL X.LT.UNIT ROUND OFF */
/* ----------------------------------------------------------------------- */
L260:
  i1 = -(*n) - 1;
  ans[1] = pow (*x, i1);
  if (mm == 1)
    {
      goto L280;
    }
  k = 1;
  i1 = mm;
  for (i__ = 2; i__ <= i1; ++i__)
    {
      ans[k + 1] = ans[k] / *x;
      ++k;
/* L270: */
    }
L280:
  if (*n != 0)
    {
      return 0;
    }
  if (kode == 2)
    {
      ans[1] += xln;
    }
  return 0;
L290:
  if (t > 0.)
    {
      goto L380;
    }
  *nz = 0;
  *ierr = 2;
  return 0;
L380:
  ++(*nz);
  ans[mm] = 0.;
  --mm;
  if (mm == 0)
    {
      return 0;
    }
  goto L41;
L390:
  *nz = 0;
  *ierr = 3;
  return 0;
}				/* dpsifn_ */




double
rannor (double mean, double sd)
{
  double r1, r2;
  r1 = RANDUM ();
  r2 = RANDUM ();
  return sd * sqrt (-2. * log (r1)) * cos (TWOPI * r2) + mean;
}


char
lowercase (char c)
{
  return (char) tolower ((int) c);
}

char
uppercase (char c)
{
  return (char) toupper ((int) c);
}


double
find_chi (long df, double prob)
{

  double a, b, m;
  double xb = 200.0;
  double xa = 0.0;
  double xm = 5.;
  a = probchi (df, xa);
  m = probchi (df, xm);
  b = probchi (df, xb);
  while (fabs (m - prob) > EPSILON)
    {
      if (m < prob)
	{
	  b = m;
	  xb = xm;
	}
      else
	{
	  a = m;
	  xa = xm;
	}
      xm = (-(b * xa) + prob * xa + a * xb - prob * xb) / (a - b);	//(xa + xb)/2.;

      m = probchi (df, xm);
    }
  return xm;
}


double
probchi (long df, double chi)
{
  double prob;
  double v = ((double) df)/2.;
  if (chi > DBL_EPSILON)
    {
      //lg = exp (LGAMMA (v));
      prob = 1. - incompletegamma (chi / 2., v);
    }
  else
    prob = 1.0;
  //  printf("prob=%f v=%f chi=%f lg(v/2)=%f  ig(chi/2,v/2)=%f\n",
   //  prob,v,chi,lg, incompletegamma(chi/2.,v/2.));
   
  return prob;
}

double
chisquare (long df, double alpha)
{
  const double table05[] =
    { 3.84146, 5.99147, 7.81473, 9.48773, 11.0705, 12.5916 };
  const double table01[] =
    { 6.63490, 9.21034, 11.3449, 13.2767, 15.0863, 16.8119 };

  if (alpha == 0.05)
    return table05[df - 1];
  if (alpha == 0.01)
    return table01[df - 1];
  error ("Chi-distribution for any probability alpha is not implemented");
  return -1;
}

double
sum (double *vector, long n)
{
  long i;
  double summ = 0.0;
  for (i = 0; i < n; i++)
    summ += vector[i];
  return summ;
}

//==========================================
// searching and finding

boolean find (long i, long *list, long listlen)
{
  long j;
  for (j = 0; j < listlen; j++)
    {
      if (i == list[j])
	return TRUE;
    }
  return FALSE;
}

//====================================================
// conversion between the different parameter schemes
// returns the begining of  mig_.i
long
mstart (long pop, long numpop)
{
  return numpop + pop * numpop - pop;
}
// returns the end of  mig_.i
long
mend (long pop, long numpop)
{
  return numpop + pop * numpop - pop + numpop - 1;
}


long
mmstart (long pop, long numpop)
{
  return pop * (numpop);
}

long
mmend (long pop, long numpop)
{
  return pop * numpop + numpop;
}
// returns the location in a full matrix given the abbreviated matrix
long
mm2m (long frompop, long topop, long numpop)
{
  if (frompop == topop)
    return (frompop);
  if (frompop < topop)
    return numpop + topop * (numpop - 1) + frompop;
  else
    return numpop + topop * (numpop - 1) + (frompop - 1);
}

// calulates the i and j from a linear abbrev matrix
void
m2mm (long i, long numpop, long *frompop, long *topop)
{
  if (i < numpop)
    {
      *frompop = i;
      *topop = i;
      return;
    }
  else
    {
      (*topop) = (long) (i - numpop) / (numpop - 1);
      (*frompop) = i - numpop - (*topop) * (numpop - 1);
      if (*frompop >= *topop)
	*frompop += 1;
    }
}


long
m2mml (long i, long numpop)
{
  long topop, frompop;

  if (i < numpop)
    {
      return i * numpop + i;
    }
  else
    {
      topop = (long) (i - numpop) / (numpop - 1);
      frompop = i - numpop - (topop) * (numpop - 1);
      if (frompop >= topop)
	frompop += 1;
      return numpop * topop + frompop;
    }
}


long
mml2m (long pos, long numpop)
{
  long topop = 0, frompop = 0, i = 1;
  if (pos == 0)
    return 0;
  while (pos > numpop * (i++))
    topop++;
  frompop = pos - topop * numpop;
  return mm2m (frompop, topop, numpop);
}


long
m2mml2 (long i, long topop, long numpop)
{
  long frompop;

  if (i < numpop)
    {
      return i * numpop + i;
    }
  else
    {
      frompop = i - numpop - (topop) * (numpop - 1);
      if (frompop >= topop)
	frompop += 1;
      return numpop * topop + frompop;
    }
}


void
gamma_rates (double *rate, long categs, char *input)
{
  double alpha = DBL_MAX;
  double value;
  while (!isdigit (*input) && *input != '\0')
    input++;
  if ((value = strtod (input, (char **) NULL)) > 0)
    alpha = value;
  calc_gamma (alpha, rate, categs);
}
/* calculation of rate values following a gamma distribution for
   given probability values */
void
calc_gamma (double alpha, double *gama, long categs)
{
  long i, panic;
  double low, mid, high, xlow, xhigh, tmp, freq = 0, x = 10,
    elements = (double) categs;
  freq = -(0.5 / elements);	/*so we have midpoints instead of endpoints */
  for (i = 0; i < categs; i++)
    {
      low = 0;
      mid = incompletegamma (10., alpha);
      high = 1.;
      freq += 1. / (elements);
      if (freq < mid)
	{
	  high = mid;
	  xlow = 0;
	  xhigh = 10.;
	  x = 5.;
	}
      else
	{
	  low = mid;
	  xhigh = 1e10;
	  xlow = 10.;
	  x = 1e5;
	}
      panic = 0;
      while (panic++ < 1000 && fabs (low - high) > 0.0001 && x > 0.000000001)
	{
	  mid = incompletegamma (x, alpha);
	  if (freq < mid)
	    {
	      high = mid;
	      tmp = x;
	      x = (x + xlow) / 2.;
	      xhigh = tmp;
	    }
	  else
	    {
	      low = mid;
	      tmp = x;
	      x = (x + xhigh) / 2.;
	      xlow = tmp;
	    }
	}
      gama[i] = x / alpha;
//Debug
      fprintf (stderr, "  %li> %f\n", i, gama[i]);

      if (x >= 10e10)
	{
	  error ("calc_gamma(): x is too big");
	}
    }
}


