/* heating scheme with threads if available


part of migrate 
Peter Beerli 2000
$Id: heating.c,v 1.7 2000/10/13 22:56:41 beerli Exp $

*/
#include "migration.h"
#include "mcmc.h"

#if PTHREADS			/* ==================threaded version======== */

#include <pthread.h>
#include "heating.h"


void fill_tpool (tpool_t tpool, world_fmt ** universe, int universe_size);
void tpool_init (tpool_t * tpoolp, int num_worker_threads, int max_queue_size, int do_not_block_when_full);
void thread_metropolize (void *world);
int tpool_destroy (tpool_t tpool, int finish);
void tpool_thread (tpool_t tpool);
int tpool_synchronize (tpool_t tpool, int finish);


void
tpool_init (tpool_t * tpoolp, int num_worker_threads, int max_queue_size, int do_not_block_when_full)
{
  int i, rtn;
  tpool_t tpool;

  tpool = (tpool_t) malloc (sizeof (struct tpool));

  tpool->num_threads = num_worker_threads;
  tpool->max_queue_size = max_queue_size;
  tpool->do_not_block_when_full = do_not_block_when_full;
  tpool->threads = (pthread_t *) malloc (sizeof (pthread_t) * num_worker_threads);

  tpool->cur_queue_size = 0;
  tpool->queue_head = NULL;
  tpool->queue_tail = NULL;
  tpool->queue_closed = 0;
  tpool->shutdown = 0;
  tpool->done = 0;
  if ((rtn = pthread_mutex_init (&(tpool->random_lock), NULL)) != 0)
    error ("pthread_mutex_init for tpool->random_lock failed");
  if ((rtn = pthread_mutex_init (&(tpool->queue_lock), NULL)) != 0)
    error ("pthread_mutex_init for tpool->queue_lock failed");
  if ((rtn = pthread_cond_init (&(tpool->queue_done), NULL)) != 0)
    error ("pthread_cond_init for tpool->queue_done failed");
  if ((rtn = pthread_cond_init (&(tpool->queue_not_empty), NULL)) != 0)
    error ("pthread_cond_init for tpool->queue_not_empty failed");
  if ((rtn = pthread_cond_init (&(tpool->queue_not_full), NULL)) != 0)
    error ("pthread_cond_init for tpool->queue_not_full failed");
  if ((rtn = pthread_cond_init (&(tpool->queue_empty), NULL)) != 0)
    error ("pthread_cond_init for tpool->queue_empty failed");

  for (i = 0; i < tpool->num_threads; i++)
    {
      if ((rtn = pthread_create (&(tpool->threads[i]), NULL, (void *) tpool_thread, (void *) tpool)) != 0)
	error ("Thread pool creation failed with pthread_create");
    }
  *tpoolp = tpool;
}

int
tpool_add_work (tpool_t tpool, void *routine, void *arg)
{
  tpool_work_t *workp;
  pthread_mutex_lock (&tpool->queue_lock);

  if ((tpool->cur_queue_size == tpool->max_queue_size) && tpool->do_not_block_when_full)
    {
      pthread_mutex_unlock (&tpool->queue_lock);
      return -1;
    }
  while ((tpool->cur_queue_size == tpool->max_queue_size) && (!tpool->shutdown || tpool->queue_closed))
    {
      pthread_cond_wait (&tpool->queue_not_full, &tpool->queue_lock);
    }

  if (tpool->shutdown || tpool->queue_closed)
    {
      pthread_mutex_unlock (&tpool->queue_lock);
      return -1;
    }

  workp = (tpool_work_t *) malloc (sizeof (tpool_work_t));
  workp->routine = routine;
  workp->arg = arg;
  workp->next = NULL;
  if (tpool->cur_queue_size == 0)
    {
      tpool->queue_tail = tpool->queue_head = workp;
      pthread_cond_broadcast (&tpool->queue_not_empty);
    }
  else
    {
      (tpool->queue_tail)->next = workp;
      tpool->queue_tail = workp;
    }
  tpool->cur_queue_size++;
  pthread_mutex_unlock (&tpool->queue_lock);
  return 1;
}

void
fill_tpool (tpool_t tpool, world_fmt ** universe, int universe_size)
{
  int i;
  for (i = 0; i < universe_size; i++)
    {
      //     printf("\nqueued: %s ",universe[i]->name);
      if (!tpool_add_work (tpool, thread_metropolize, (void *) universe[i]))
	error ("Filling heating queue failed");
    }
}

void
tpool_thread (tpool_t tpool)
{
  tpool_work_t *myworkp;

  for (;;)
    {
      pthread_mutex_lock (&(tpool->queue_lock));
      while ((tpool->cur_queue_size == 0) && (!tpool->shutdown))
	{
	  pthread_cond_wait (&(tpool->queue_not_empty), &(tpool->queue_lock));
	}
      if (tpool->shutdown)
	{
	  pthread_mutex_unlock (&tpool->queue_lock);
	  pthread_exit (NULL);
	}
      myworkp = tpool->queue_head;
      tpool->cur_queue_size--;
      if (tpool->cur_queue_size == 0)
	tpool->queue_head = tpool->queue_tail = NULL;
      else
	tpool->queue_head = myworkp->next;

      if ((!tpool->do_not_block_when_full) && (tpool->cur_queue_size == (tpool->max_queue_size - 1)))
	pthread_cond_broadcast (&(tpool->queue_not_full));

      if (tpool->cur_queue_size == 0)
	pthread_cond_signal (&(tpool->queue_empty));

      pthread_mutex_unlock (&tpool->queue_lock);
      (*(myworkp->routine)) (myworkp->arg);
      pthread_mutex_lock (&tpool->queue_lock);
      tpool->done++;
      //      printf("-%i",tpool->done);
      pthread_cond_signal (&tpool->queue_done);
      pthread_mutex_unlock (&tpool->queue_lock);
      //free(myworkp);
    }
}

void
thread_metropolize (void *world)
{
  long i;
  world_fmt *thisworld = (world_fmt *) world;
  for (i = 0; i < thisworld->options->heating_interval; i++)
    {
      thisworld->accept = metropolize (thisworld, thisworld->G);
      //      printf("%li> %s \n",i,thisworld->name);
    }
  //  return NULL;
  //  pthread_exit(NULL);
}

int
tpool_synchronize (tpool_t tpool, int fullnumber)
{
  int rtn;
  //  printf("\nSync ");
  if ((rtn = pthread_mutex_lock (&(tpool->queue_lock))) != 0)
    error ("pthread_mutex_lock failed in tpool_synchronize()");
  //  printf("{%i} ",tpool->done);
  while (tpool->done < fullnumber)
    {
      //      printf("(%i [%i]) ",tpool->done,   tpool->cur_queue_size);
      if ((rtn = pthread_cond_wait (&(tpool->queue_done), &(tpool->queue_lock))) != 0)
	error ("pthread_cond_wait failed in tpool_synchronize()");
    }
  tpool->done = 0;

  if ((rtn = pthread_mutex_unlock (&(tpool->queue_lock))) != 0)
    error ("pthread_mutex_unlock failed in tpool_destroy()");
  return 0;
}

int
tpool_destroy (tpool_t tpool, int finish)
{
  int i, rtn;
  tpool_work_t *cur_nodep;

  if ((rtn = pthread_mutex_lock (&(tpool->queue_lock))) != 0)
    error ("pthread_mutex_lock failed in tpool_destroy()");

  if (tpool->queue_closed || tpool->shutdown)
    {
      if ((rtn = pthread_mutex_unlock (&(tpool->queue_lock))) != 0)
	error ("pthread_mutex_unlock failed in tpool_destroy()");
      return 0;
    }

  tpool->queue_closed = 1;

  if (finish == 1)
    {
      while (tpool->cur_queue_size != 0)
	{
	  if ((rtn = pthread_cond_wait (&(tpool->queue_empty), &(tpool->queue_lock))) != 0)
	    error ("pthread_cond_wait failed in tpool_destroy()");
	}
    }
  tpool->shutdown = 1;

  if ((rtn = pthread_mutex_unlock (&(tpool->queue_lock))) != 0)
    error ("pthread_mutex_unlock failed in tpool_destroy()");

  if ((rtn = pthread_cond_broadcast (&(tpool->queue_not_empty))) != 0)
    error ("pthread_cond_broadcast failed for queue_not_empty()");
  if ((rtn = pthread_cond_broadcast (&(tpool->queue_not_full))) != 0)
    error ("pthread_cond_broadcast failed for queue_not_full");

  for (i = 0; i < tpool->num_threads; i++)
    {
      if ((rtn = pthread_join (tpool->threads[i], NULL)) != 0)
	error ("pthread_join failed in tpool_destroy()");
    }

  free (tpool->threads);
  while (tpool->queue_head != NULL)
    {
      cur_nodep = tpool->queue_head->next;
      tpool->queue_head = tpool->queue_head->next;
      free (cur_nodep);
    }
  free (tpool);
  return 0;
}

#else /* ==================NOT threaded version======== */



#endif
