/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 M E N U   R O U T I N E S 

 presents the menu and its submenus.                                                                                                               
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 $Id: menu.c,v 1.45 2001/03/09 16:29:18 beerli Exp $

-------------------------------------------------------*/
#include "migration.h"
#include "sequence.h"
#include "fst.h"
#include "options.h"
#include "migrate_mpi.h"

#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif
/* prototypes ------------------------------------------- */
void print_menu_title (FILE * file, option_fmt * options);
void print_menu_accratio (FILE * file, long a, long b);
long print_title (world_fmt * world, option_fmt * options);
void get_menu (option_fmt * options);
/* private functions */
void menuData (option_fmt * options, char datatype[]);
void menuInput (option_fmt * options);
void menuParameters (option_fmt * options);
void menuStrategy (option_fmt * options);
void display_standard_mcmc(option_fmt *options);
void display_adaptive_mcmc(option_fmt *options);
void menuSequences (option_fmt * options);
void read_custom_menu_migration (option_fmt * options);
void read_custom_menu_lratio (option_fmt * options);
char *custom_migration_type (long type);
void read_heatvalues (option_fmt * options);
void menuRandom(double *param);
void	  start_tree_method(option_fmt *options);
void start_data_method(option_fmt *options);
void how_many_pop(long *numpop);
char dialog_lrt(long numpop, lratio_fmt *lratio);
char menuread_lrt_paramvalue(char *value, long *counter, long numpop2);

void
print_menu_title (FILE * file, option_fmt * options)
{
  char nowstr[LINESIZE];
  if (options->menu || options->progress)
    {
      fprintf (file, "  =============================================\n");
      fprintf (file, "  MIGRATION RATE AND POPULATION SIZE ESTIMATION\n");
      fprintf (file, "  using Markov Chain Monte Carlo simulation\n");
      fprintf (file, "  =============================================\n");
#ifdef MPI
      fprintf (file, "  compiled for a PARALLEL COMPUTER ARCHITECTURE\n");
#endif
      fprintf (file, "  Version %s\n", MIGRATEVERSION);
      get_time (nowstr, "  %c");
      if (nowstr[0] != '\0')
	fprintf (file, "  Program started at %s\n", nowstr);
      fprintf (file, "\n\n");
    }
}

void
print_menu_accratio (FILE * file, long a, long b)
{
  fprintf (file, "           Acceptance-ratio = %li/%li (%f)\n", a, b, (double) a / (double) b);
}

long
print_title (world_fmt * world, option_fmt * options)
{
  char nowstr[LINESIZE];
  long len = 45, i, filepos = -1;
  if (!world->options->simulation)
    {
      fprintf (world->outfile, "  ");
      if (options->title[0] != '\0')
	{
	  len = (long) MAX (strlen (options->title), 45);
	  for (i = 0; i < len; i++)
	    fprintf (world->outfile, "=");
	  fprintf (world->outfile, "\n  %s\n  ", options->title);
	}
      for (i = 0; i < len; i++)
	fprintf (world->outfile, "=");

      fprintf (world->outfile, "\n  MIGRATION RATE AND POPULATION SIZE ESTIMATION\n");
      fprintf (world->outfile, "  using Markov Chain Monte Carlo simulation\n  ");

      for (i = 0; i < len; i++)
	fprintf (world->outfile, "=");
      fprintf (world->outfile, "\n  Version %s\n", MIGRATEVERSION);
      get_time (nowstr, "%c");
      if (nowstr[0] != '\0')
	{
	  fprintf (world->outfile, "\n  Program started at %s\n", nowstr);
	  filepos = ftell (world->outfile);
	  /* DO NOT REMOVE the blanks in the following print statement,
	     they are overwritten at the program end with the timestamp
	   */
	  fprintf (world->outfile, "                                                   \n");
	}
      fprintf (world->outfile, "\n\n");
    }
  return filepos;
}


void
get_menu (option_fmt * options)
{
  char input[LINESIZE];
  char datatype[200];
  if (options->menu)
    {
      switch (options->datatype)
	{
	case 'a':
	  strcpy (datatype, "infinite allele model");
	  break;
	case 'm':
	  strcpy (datatype, "microsatellite model");
	  break;
	case 'b':
	  strcpy (datatype, "microsatellite model [Brownian motion]");
	  break;
	case 's':
	  strcpy (datatype, "DNA sequence model");
	  break;
	case 'n':
	  strcpy (datatype, "single nucleotide polymorphism model");
	  break;
	case 'u':
	  strcpy (datatype, "single nucleotide polymorphism model with panel");
	  break;
	case 'f':
	  strcpy (datatype, "ancestral state reconstruction method");
	  break;
	case 'g':
	  strcpy (datatype, "genealogy-summary");
	  break;
	default:
	  if (options->readsum)
	    {
	      options->datatype = 'g';
	      strcpy (datatype, "genealogy summary");
	    }
	  else
	    {
	      options->datatype = 's';
	      strcpy (datatype, "DNA sequence model");
	    }
	  break;
	}
      do
	{
	  printf ("  Settings for this run:\n");
	  printf ("  D       Data type\n           (currently set: %s)\n", datatype);
	  printf ("  I       Input/Output formats\n");
	  printf ("  P       Start values for the Parameters\n");
	  printf ("  S       Search strategy\n");
	  printf ("  W       Write a parmfile\n");
	  printf ("  Q       Quit the program\n");
	  printf ("\n\n");
	  printf ("  Are the settings correct?\n");
	  printf ("  (Type Y or the letter for one to change)\n");
	  fgets (input, LINESIZE, stdin);
	  switch (uppercase (input[0]))
	    {
	    case 'D':
	      menuData (options, datatype);
	      break;
	    case 'I':
	      menuInput (options);
	      break;
	    case 'P':
	      menuParameters (options);
	      break;
	    case 'S':
	      menuStrategy (options);
	      break;
	    case 'W':
	      save_parmfile (options);
	      break;
	    case 'Q':
	      exit (0);
	    default:
	      break;
	    }
	}
      while (uppercase (input[0]) != 'Y');
      if (options->usertree && !strchr(SEQUENCETYPES, options->datatype))
	options->usertree = FALSE;
      if (options->dist && !strchr(SEQUENCETYPES, options->datatype))
	options->dist = FALSE;	//prevents incompatability
      // of tree and distance designation, only for
      // sequence data the datalines and tree-tips can match
      // msats or EP data is ambiguos concerning the tree.
    }
  if (options->datatype == 'g')	//prevents overwriting sumfile
    options->writesum = FALSE;

#ifdef UEP
  if (options->uep)
    options->randomtree = TRUE;
#endif
}


/* private functions--------------------------------------------------- */
void
menuData (option_fmt * options, char datatype[])
{
  static boolean didchangecat, didchangercat;

  long z = 0;
  char input[LINESIZE];
  char starttree[LINESIZE];

  /* in case we have  already read a parmfile */
  if (options->categs > 1)
    didchangecat = TRUE;
  else
    didchangecat = FALSE;
  if (options->rcategs > 1)
    didchangercat = TRUE;
  else
    didchangercat = FALSE;

  do
    {
      switch (options->datatype)
	{
	case 'a':
	  strcpy (datatype, "infinite allele model");
	  break;
	case 'm':
	  strcpy (datatype, "microsatellite model [Stepwise mutation]");
	  break;
	case 'b':
	  strcpy (datatype, "microsatellite model [Brownian motion]");
	  break;
	case 's':
	  strcpy (datatype, "DNA sequence model");
	  break;
	case 'n':
	  strcpy (datatype, "single nucleotide polymorphism");
	  break;
	case 'u':
	  strcpy (datatype, "Panel-SNP (panel is population 1)");
	  break;
	case 'f':
	  strcpy (datatype, "ancestral state reconstruction method");
	  break;
	case 'g':
	  strcpy (datatype, "genealogy summary");
	  options->readsum = TRUE;
	  break;
	default:
	  if (options->readsum)
	    {
	      options->datatype = 'g';
	      strcpy (datatype, "genealogy summary");
	    }
	  else
	    {
	      options->datatype = 's';
	      strcpy (datatype, "DNA sequence model");
	    }
	  break;
	}
      if (options->usertree && strchr(SEQUENCETYPES, options->datatype))
	sprintf (starttree, "is supplied in %s", options->utreefilename);
      else
	{
	  if (options->dist && strchr(SEQUENCETYPES, options->datatype))
	    sprintf (starttree, "generates using %s", options->distfilename);
	  else
	    {
	      if (options->randomtree)
		strcpy (starttree, "has a random topology");
	      else
		strcpy (starttree, "is estimated using a UPGMA topology");
	    }
	}
      printf ("  DATATYPE AND DATA SPECIFIC OPTIONS\n\n");
      printf ("  1       change Datatype (currently: %s)\n", datatype);
      switch (options->datatype)
	{
	case 'a':
	  printf ("          (using a k-allele model which simulates an infinite allele model)\n");

	  break;
	case 'm':
	  printf ("          (using a stepwise mutation model)\n");
	  //	  printf ("  2       Size of the step matrix:      %li\n", options->micro_stepnum);
	  printf ("  2       Threshold value:              %li\n\n\n\n", options->micro_threshold);
	  break;
	case 'u':
	case 'n':
	case 's':
	case 'f':
	  printf ("  2       Transition/transversion ratio:");
	  z = 0;
	  while (options->ttratio[z] > 0.0)
	    printf ("%8.4f ", options->ttratio[z++]);
	  printf ("\n");
	  printf ("  3       Use empirical base frequencies?  %s\n", (options->freqsfrom ? "Yes" : "No"));
	  printf ("  4       One category of sites?");
	  if (options->categs == ONECATEG)
	    printf ("  One category\n");
	  else
	    printf ("  More than one  category of sites\n");
	  printf ("  5       One region of substitution rates?");
	  if (options->rcategs == 1)
	    printf ("  Yes\n");
	  else
	    {
	      printf ("  %ld categories of regions\n", options->rcategs);
	      printf ("  6       Rates at adjacent sites correlated?");
	      if (!options->autocorr)
		printf ("  No, they are independent\n");
	      else
		printf ("  Yes, mean block length =%6.1f\n", 1.0 / options->lambda);
	    }
	  printf ("  7       Sites weighted?  %s\n", (options->weights ? "Yes" : "No"));
	  printf ("  8       Input sequences interleaved?  %s\n", (options->interleaved ? "Yes" : "No, sequential"));
	  printf ("  9       Sequencing error rate? [0.0 = no error] %4.3f\n", options->seqerror);
	  printf (" 10       Slow but safer Data likelihood calculation  %s\n", options->fastlike ? "No" : "Yes");

	  break;
	case 'g':
	  printf ("           [Maximization, Likelihood test-ratio]\n");
	  break;
	}
      printf ("  0       Start genealogy %s\n", starttree);
      printf ("\n\n");
      printf ("  Are the settings correct?\n");
      printf ("  (Type Y or the number of the entry to change)\n");
      input[0] = '\0';
      fgets (input, LINESIZE, stdin);
      if(strlen(input)==0)
	continue;
      switch (atol(input))
	{
	case 0:       
	  if(uppercase(input[0])=='Y')
	    break;
	  start_tree_method(options); 
	  break;
	case 1: start_data_method(options); break;
	default: break;
	}
      if (options->datatype == 'm')
	{
	  switch (atol(input))
	    {
	      //case 2:		/*micro-stepmax */
	      //printf ("  What is the maximal size of the stepmatrix?\n");
	      //printf ("  [the bigger the longer the run and the more accurate the estimate]\n");
	      //fscanf (stdin, "%ld%*[^\n]", &options->micro_stepnum);
	      //break;
	    case 2:		/*micro-threshold */
	      printf ("  What is the threshold value?\n");
	      printf ("  E.g. if your allele is 24 and the threshold is 10\n");
	      printf ("  there is some probability that the allele 24 can\n");
	      printf ("  change to allele 14 (or 38), but there is a probability\n");
	      printf ("  of 0.0 (ZERO) to go to 13 (39),\n");
	      printf ("  if you choose this too small, than the program will fail\n");
	      printf ("  [the bigger the longer the run and the more\n accurate the estimate]\n");
	      fgets (input, LINESIZE, stdin);
	      options->micro_threshold = atof(input);
	      input[0]='\0';
	      break;
	    }
	}
      if (strchr(SEQUENCETYPES,options->datatype))
	{
	  switch (atol(input))
	    {
	    case 2:
	      initratio (options);
	      break;
	    case 3:
	      options->freqsfrom = !options->freqsfrom;
	      if (!options->freqsfrom)
		{
		  initfreqs (&options->freqa, &options->freqc, &options->freqg, &options->freqt);
		}
	      break;
	    case 4:
	      if (options->categs == ONECATEG)
		{
		  options->categs = MANYCATEGS;
		  didchangecat = TRUE;
		}
	      else
		{
		  options->categs = ONECATEG;
		  didchangecat = FALSE;
		}
	      break;
	    case 5:
	      if (didchangercat)
		{
		  options->autocorr = FALSE;
		  didchangercat = FALSE;
		  options->rcategs = ONECATEG;
		}
	      else
		{
		  printf ("\n  Regional rates:\n");
		  initcatn (&options->rcategs);
		  options->probcat = (double *) realloc (options->probcat, options->rcategs * sizeof (double));
		  options->rrate = (double *) realloc (options->rrate, options->rcategs * sizeof (double));
		  didchangercat = TRUE;
		  options->gammarates = initcategs (options->rcategs, options->rrate);
		  if (!options->gammarates)
		    initprobcat (options->rcategs, &options->probsum, options->probcat);
		  else
		    for (z = 0; z < options->rcategs; z++)
		      options->probcat[z] = 1. / options->rcategs;
		}
	      break;
	    case 6:
	      options->autocorr = !options->autocorr;
	      if (options->autocorr)
		initlambda (options);
	      break;
	    case 7:
	      options->weights = !options->weights;
	      break;
	    case 8:
	      options->interleaved = !options->interleaved;
	      break;
	    case 9:
	      fprintf (stdout, "Enter the sequencing error rate per site\n[Good values are 0.0 (=no error), or 0.01 (1%% error):\n");
	      fgets (input, LINESIZE, stdin);
	      options->seqerror = atof(input);
	      input[0]='\0';
	      break;
	    case 10:
	      options->fastlike = !options->fastlike;
	      break;
	    default:
	      break;
	    }
	  if (!didchangercat)
	    {
	      options->probcat = (double *) realloc (options->probcat, sizeof (double) * 2);
	      options->rrate = (double *) realloc (options->rrate, sizeof (double) * 2);
	      options->rrate[0] = 1.0;
	      options->probcat[0] = 1.0;
	    }
	  if (!didchangecat)
	    {
	      options->rate = (double *) realloc (options->rate, sizeof (double) * 2);
	      options->rate[0] = 1.0;
	    }
	}
      if (options->datatype != 'g')
	options->readsum = FALSE;
    }
  while (uppercase (input[0]) != 'Y');
}

void
menuInput (option_fmt * options)
{
  long len, test = 0;
  unsigned long timeseed;
  char input[LINESIZE];
  char answer;			//, ch;
  char *stringstep, *string2, *string3, *string4;
  char treestr[4][20] = { "None", "All", "Best", "Last chain" };
  char progress[3][8] = { "Verbose", "Yes", "No" };
  stringstep = (char *) malloc (sizeof (char) * 128);
  string2 = (char *) malloc (sizeof (char) * 128);
  string3 = (char *) malloc (sizeof (char) * 128);
  string4 = (char *) malloc (sizeof (char) * 128);

  do
    {
      printf ("  INPUT FORMATS\n  -------------\n");
      printf ("   1      Datafile name is %s\n", options->infilename);
      switch (options->autoseed)
	{
	case AUTO:
	  sprintf (stringstep, "Yes");
	  break;
	case NOAUTO:
	  sprintf (stringstep, "No, use seedfile");
	  break;
	case NOAUTOSELF:
	  sprintf (stringstep, "No, seed=%li ", options->inseed);
	  break;
	default:
	  options->autoseed = AUTO;
	  sprintf (stringstep, "Yes");
	  break;
	}
      printf ("   2      Use automatic seed for randomisation?  %s\n", stringstep);
      printf ("   3      Title of the analysis is");
      if ((len = (strlen (options->title))) < 24)
	{
	  if (len == 0)
	    printf (" <no title given>\n");
	  else
	    printf (" %s\n", options->title);
	}
      else
	printf ("            %s\n", options->title);
      if (options->readsum)
	{
	  printf ("   4       Summary of genealogies are read from %s\n", options->sumfilename);
	}
      printf ("\n  OUTPUT FORMATS\n  --------------\n");

      printf ("   5      Print indications of progress of run?  %s\n", options->progress ? (options->verbose ? progress[0] : progress[1]) : progress[2]);
      printf ("   6      Print the data?                        %s\n", options->printdata ? "Yes" : "No");

      printf ("   7      Outputfile name is %s\n", options->outfilename);

      if (options->plot)
	{
	  switch (options->plotmethod)
	    {
	    case PLOTALL:
	      strcpy (string2, "Yes, to outfile and mathfile");
	      break;
	    default:
	      strcpy (string2, "Yes, to outfile");
	      break;
	    }
	}
      else
	{
	  strcpy (string2, "No");
	}
      printf ("   8      Plot likelihood surface?               %s\n", string2);
      switch (options->profile)
	{
	case NONE:
	  strcpy (string3, "No");
	  break;
	case ALL:
	  strcpy (string3, "Yes, tables and summary");
	  break;
	case TABLES:
	  strcpy (string3, "Yes, tables");
	  break;
	case SUMMARY:
	  strcpy (string3, "Yes, summary");
	  break;
	}
      printf ("   9      Profile-likelihood?%24.24s\n", string3);
      if (options->profile != NONE)
	{
	  switch (options->profilemethod)
	    {
	    case 'p':
	      strcpy (string4, "[Percentiles using exact Bisection method]");
	      break;
	    case 's':
	      strcpy (string4, "[Percentiles using not so exact Spline method]");
	      break;
	    case 'd':
	      strcpy (string4, "[Evaluation at preset fractions of ML estimate]");
	      break;
	    case 'q':
	      strcpy (string4, "[Assuming that parameter are independent]");
	      break;
	    case 'f':
	      strcpy (string4, "[Mixture of 'p' and 'q']");
	      break;
	    }
	  printf ("%60.60s\n", string4);
	}
      printf ("  10      Likelihood-Ratio tests?                %s\n", options->lratio->counter > 0 ? " Yes" : "No");

      switch (options->treeprint)
	{
	case ALL:
	  printf ("  11      Print genealogies?                     %s\n", treestr[1]);
	  break;
	case BEST:
	  printf ("  11     Print genealogies?                     %s\n", treestr[2]);
	  break;
	case LASTCHAIN:
	  printf ("  11      Print genealogies?                     %s\n", treestr[3]);
	  break;
	case NONE:
	default:
	  printf ("  11      Print genealogies?                     %s\n", treestr[0]);
	  break;
	}
      printf ("  12      Plot coordinates are saved in %s\n", options->mathfilename);
      if (options->writesum || options->datatype == 'g')
	{
	  if (!options->readsum && options->writesum)
	    {
	      printf ("  13      Summary of genealogies are saved in %s\n", options->sumfilename);
	    }
	}
      else
	{
	  printf ("  13      Summary of genealogies will not be saved\n");
	}
      if (options->writelog)
	printf ("  14      Save logging information into %s\n", options->logfilename);
      else
	printf ("  14      Save logging information? No\n");
#ifdef UEP
      if (options->uep)
	{
	  printf ("  15      Read unique polymorphism? From file %s\n", options->uepfilename);
	  printf ("  16      Mutation rate for UEP is %f %s\n",
		  options->ueprate, " x mutation rate");
	}
      else
	printf ("  15      Read unique polymorphism?  No\n");
#endif
      printf ("\n\n  Are the settings correct?\n");
      printf ("  (type Y to go back to the main menu or the letter for the entry to change)\n");
      fgets (input, LINESIZE, stdin);
      test = atoi (input);
      switch (test)
	{
	case 1:
	  printf ("  What is the datafile name?\n[Default: %s]\n", INFILE);
	  fgets (input, LINESIZE, stdin);
	  if (input[0] == '\n')
	    strcpy (options->infilename, INFILE);
	  else
	    {
	      input[strlen (input) - 1] = '\0';
	      strcpy (options->infilename, input);
	    }
	  break;
	case 2:
	  do
	    {
	      printf ("  (A)utomatic or (S)eedfile or (O)wn\n");
	      printf ("  Start value for Random-generator seed\n");
	      fgets (input, LINESIZE, stdin);
	      switch (uppercase (input[0]))
		{
		case 'A':
		  options->autoseed = AUTO;
#ifndef MAC
		  timeseed = (unsigned long) time (NULL) / 4;
#else
		  timeseed = (unsigned long) clock () / 4;
#endif
		  options->inseed = (long) timeseed + 1;
		  break;
		case 'S':
		  options->seedfile = fopen (SEEDFILE, "r");
		  if (options->seedfile)
		    {
		      options->autoseed = NOAUTO;
		      fscanf (options->seedfile, "%ld%*[^\n]", &options->inseed);
		      fclose (options->seedfile);
		    }
		  else
		    printf ("\n\n  There is no seedfile present\n");
		  break;
		case 'O':
		  options->autoseed = NOAUTOSELF;
		  printf ("  Random number seed (best values are x/4 +1)?\n");
		  scanf ("%ld%*[^\n]", &options->inseed);
		  break;
		}
	    }
	  while (options->autoseed < AUTO || options->autoseed > NOAUTOSELF);
	  break;
	case 3:
	  printf ("  Enter a title? [max 80 Characters are used]\n");
	  fgets (input, LINESIZE, stdin);
	  if (input[0] == '\0')
	    options->title[0] = '\0';
	  else
	    strncpy (options->title, input, 80);
	  break;
	case 4:
	  printf (" What is the filename for the summary of genealogies\n[Default: %s]\n", SUMFILE);
	  fgets (input, LINESIZE, stdin);
	  if (input[0] == '\n')
	    strcpy (options->sumfilename, SUMFILE);
	  else
	    {
	      input[strlen (input) - 1] = '\0';
	      strcpy (options->sumfilename, input);
	    }
	  break;
	case 5:
	  printf ("  Progress report during the run? <YES | Verbose | No>\n");
	  fgets (input, LINESIZE, stdin);
	  switch (tolower (input[0]))
	    {
	    case 'n':
	      options->progress = FALSE;
	      options->verbose = FALSE;
	      break;
	    case 'v':
	      options->progress = TRUE;
	      options->verbose = TRUE;
	      break;
	    case '\0':
	    case 'y':
	    default:
	      options->progress = TRUE;
	      options->verbose = FALSE;
	      break;
	    }
	  input[0] = 'X';
	  break;
	case 6:
	  options->printdata = !options->printdata;
	  break;
	case 7:
	  printf ("  What is the output filename?\n[Default: %s]\n", OUTFILE);
	  fgets (input, LINESIZE, stdin);
	  if (input[0] == '\n')
	    strcpy (options->outfilename, OUTFILE);
	  else
	    {
	      input[strlen (input) - 1] = '\0';
	      strcpy (options->outfilename, input);
	    }
	  break;
	case 8:
	  options->plot = !options->plot;
	  if (options->plot)
	    {
	      do
		{
		  printf ("  Plot Likelihood surface:\n");
		  printf ("  (B)oth to outfile and mathfile, (O)utfile only, (N)o plot\n");
		  fgets (input, LINESIZE, stdin);
		}
	      while (strchr ("BON", (int) uppercase (input[0])) == NULL);
	      switch (uppercase (input[0]))
		{
		case 'B':
		  options->plotmethod = PLOTALL;
		  break;
		case 'O':
		  options->plotmethod = PLOTOUTFILE;
		  break;
		case 'N':
		  options->plot = FALSE;
		  break;
		}
	      if (options->plot)
		{
		  answer = 'N';
		  do
		    {
		      printf ("   Current plot settings are:\n");
		      printf ("              Parameter: %s, Scale: %s, Intervals: %li\n", options->plotvar == PLOT4NM ? "{Theta, 4Nm}" : "{Theta, M}", options->plotscale == PLOTSCALELOG ? "Log10" : "Standard", options->plotintervals);
		      printf ("              Ranges: X-%5.5s: %f - %f\n", options->plotvar == PLOT4NM ? "4Nm" : "M", options->plotrange[0], options->plotrange[1]);
		      printf ("              Ranges: Y-%5.5s: %f - %f\n", "Theta", options->plotrange[2], options->plotrange[3]);
		      printf (" Is this OK? [Yes or No]\n");
		      fgets (input, LINESIZE, stdin);
		      answer = uppercase (input[0]);
		      if (answer != 'Y')
			{
			  printf ("1    Use as X-axis: 4Nm  Y-axis: Theta\n");
			  printf ("2    Use as X-axis: M    Y-axis: Theta\n");
			  fgets (input, LINESIZE, stdin);
			  if (input[0] == '2')
			    options->plotvar = PLOTM;
			  else
			    options->plotvar = PLOT4NM;
			  printf ("1     Scale is standard \n");
			  printf ("2     Scale is in Log10\n");
			  fgets (input, LINESIZE, stdin);
			  if (input[0] == '2')
			    options->plotscale = PLOTSCALELOG;
			  else
			    options->plotscale = PLOTSCALESTD;
			  do
			    {
			      printf ("How many plot positions?\n");
			      printf ("[Default is 36, this changes only in the mathfile]\n");
			      fgets (input, LINESIZE, stdin);
			      if (strlen (input) > 1)
				options->plotintervals = ATOL (input);
			    }
			  while (options->plotintervals < 2);
			  printf ("Change the ranges of the axes\n");
			  printf ("              Ranges: X-%5.5s: %f - %f\n", options->plotvar == PLOT4NM ? "4Nm" : "M", options->plotrange[0], options->plotrange[1]);
			  printf ("              Ranges: Y-%5.5s: %f - %f\n", "Theta", options->plotrange[2], options->plotrange[3]);
			  printf ("Give lowest and highest value for X axis:\n");
			  fgets (input, LINESIZE, stdin);
			  if (strlen (input) > 1)
			    sscanf (input, "%lf%lf", &options->plotrange[0], &options->plotrange[1]);
			  printf ("Give lowest and highest value for Y axis:\n");
			  fgets (input, LINESIZE, stdin);
			  if (strlen (input) > 1)
			    sscanf (input, "%lf%lf", &options->plotrange[2], &options->plotrange[3]);
			}
		    }
		  while (answer != 'Y');
		}
	    }
	  input[0] = '\0';
	  break;
	case 9:
	  do
	    {
	      printf ("  Evaluate profile likelihoods:\n");
	      printf ("  (N)o, (A)all [Tables and Summary],\n  (T)ables, (S)ummary\n");
	      fgets (input, LINESIZE, stdin);
	    }
	  while (strchr ("NATS", (int) uppercase (input[0])) == NULL);
	  switch (uppercase (input[0]))
	    {
	    case 'N':
	      options->profile = NONE;
	      break;
	    case 'A':
	      options->profile = ALL;
	      break;
	    case 'T':
	      options->profile = TABLES;
	      break;
	    case 'S':
	      options->profile = SUMMARY;
	      break;
	    }
	  if (options->profile != NONE)
	    {
	      do
		{
		  printf ("  Method to evaluate the profiles?\n");
		  printf ("  (P)ercentiles: exact evaluation (slow)\n");
		  printf ("  (F)ast       : Assuming parameters are indepenent\n                + one complete profile-maximization.\n");

		  //printf
		  //  ("  (S)plines    : percentiles using splines\n                (fails sometimes)\n");
		  printf ("  (D)iscrete   : evaluation at (0.02, 0.05, 0.1, .. , 50)*ML estimate\n");
		  printf ("  (Q)uick      : Assuming parameters are indepenent\n");

		  fgets (input, LINESIZE, stdin);
		}
	      while (strchr ("pdqf", (int) lowercase (input[0])) == NULL);
	      switch (lowercase (input[0]))	// in strchr removed s
		{
		case 'p':
		  options->profilemethod = 'p';
		  break;
		case 's':
		  options->profilemethod = 's';
		  break;
		case 'd':
		  options->profilemethod = 'd';
		  break;
		case 'q':
		  options->profilemethod = 'q';
		  break;
		case 'f':
		  options->profilemethod = 'f';
		  break;
		default:
		  options->profilemethod = 's';
		  break;

		}
	    }
	  set_profile_options (options);
	  break;
	case 10:
	  fprintf (stdout, "  Likelihood-Ratio tests:\n");
	  fprintf (stdout, "  -----------------------\n");
	  read_custom_menu_lratio (options);
	  break;
	case 11:
	  do
	    {
	      printf ("  Print genealogies:\n");
	      printf ("  (N)one, (A)all [!], (B)est, (L)ast chain\n");
	      fgets (input, LINESIZE, stdin);
	    }
	  while (strchr ("NABL", (int) uppercase (input[0])) == NULL);
	  switch (uppercase (input[0]))
	    {
	    case 'N':
	      options->treeprint = NONE;
	      break;
	    case 'A':
	      options->treeprint = ALL;
	      break;
	    case 'B':
	      options->treeprint = BEST;
	      break;
	    case 'L':
	      options->treeprint = LASTCHAIN;
	      break;
	    default:
	      options->treeprint = NONE;
	      break;
	    }
	  break;
	case 12:
	  printf ("  What is the plot coordinate filename?\n[Default: %s]\n", MATHFILE);
	  fgets (input, LINESIZE, stdin);
	  if (input[0] == '\n')
	    strcpy (options->mathfilename, MATHFILE);
	  else
	    {
	      input[strlen (input) - 1] = '\0';
	      strcpy (options->mathfilename, input);
	    }
	  break;
	case 13:
	  printf (" Save genealogy summaries? [No]\n");
	  fgets (input, LINESIZE, stdin);
	  if (toupper (input[0]) == 'Y')
	    {
	      options->writesum = TRUE;
	      printf (" What is the filename for the summary of genealogies\n[Default: %s]\n", SUMFILE);
	      fgets (input, LINESIZE, stdin);
	      if (input[0] == '\n')
		strcpy (options->sumfilename, SUMFILE);
	      else
		{
		  input[strlen (input) - 1] = '\0';
		  strcpy (options->sumfilename, input);
		}
	    }
	  else
	    {
	      options->writesum = FALSE;
	    }
	  input[0] = 'X';
	  break;
	case 14:
	  printf (" Save logging information? [No]\n");
	  fgets (input, LINESIZE, stdin);
	  if (toupper (input[0]) == 'Y')
	    {
	      options->writelog = TRUE;
	      printf (" What is the filename for logging\n[Default: %s]\n", LOGFILE);
	      fgets (input, LINESIZE, stdin);
	      if (input[0] == '\n')
		strcpy (options->logfilename, LOGFILE);
	      else
		{
		  input[strlen (input) - 1] = '\0';
		  strcpy (options->logfilename, input);
		}
	    }
	  else
	    {
	      options->writelog = FALSE;
	    }
	  input[0] = 'X';
	  break;
#ifdef UEP
	case 15:
	  printf (" UEP estimation? [No]\n");
	  fgets (input, LINESIZE, stdin);
	  if (toupper (input[0]) == 'Y')
	    {
	      options->uep = TRUE;
	      printf (" What is the filename of the UEP data\n[Default: %s]\n", UEPFILE);
	      fgets (input, LINESIZE, stdin);
	      if (input[0] == '\n')
		strcpy (options->uepfilename, UEPFILE);
	      else
		{
		  input[strlen (input) - 1] = '\0';
		  strcpy (options->uepfilename, input);
		}
	    }
	  else
	    {
	      options->uep = FALSE;
	    }
	  input[0] = 'X';
	  break;
	case 16:
	  do
	    {
	      fprintf(stdout,"What is mutation rate for the UEP locus\n");
	      fprintf(stdout,"[expressed as a ratio of the point mutation rate\n");
	      fprintf(stdout, "Reasonable values are 0.0 =< x <=1.0]\n");
	      fgets(input,LINESIZE,stdin);
	      options->ueprate = atof(input);
	    } while (options->ueprate < 0.0);
	  break;
#endif
	default:
	  break;
	}
    }
  while (uppercase (input[0]) != 'Y');
  free (stringstep);
  free (string2);
  free (string3);
  free (string4);
}


void
menuParameters (option_fmt * options)
{
  char input[LINESIZE], custmexplain[LINESIZE];
  long i, j, tt, check = 0, numpop = 0;

  do
    {
      printf ("  START VALUES FOR PARAMETERS\n  ---------------------------\n");
      printf ("  1       Use a simple estimate of theta as start?");
      switch (options->thetaguess)
	{
	case FST:
	  printf ("\n             Estimate with FST (Fw/Fb) measure\n");
	  break;
	case RANDOMESTIMATE:
	  printf ("\n             Random Theta from Normal distribution");
	  printf ("\n             (mean=%f, std=%f)\n", options->thetag[0],options->thetag[1]);
	  break;
	default:
	  printf ("\n             No, initial Theta = {");
	  for (i = 0; i < options->numthetag - 1; i++)
	    {
	      printf ("%.5f,", options->thetag[i]);
	      if (((i + 1) % 3) == 0)
		printf ("\n                        ");
	    }
	  printf ("%.5f}\n", options->thetag[i]);
	}
      printf ("  2       Use a simple estimate of migration rate as start?");
      switch (options->migrguess)
	{
	case FST:
	  printf ("\n             Estimate with FST (Fw/Fb) measure\n");
	  break;
	case RANDOMESTIMATE:
	  printf ("\n             Random migration rates from Normal distribution");
	  printf ("\n             (mean=%f, std=%f)\n",
		  options->mg[0],options->mg[1]);
	  break;
	default:
	  printf ("\n             No, initial migration rates are given\n");


	}
      printf ("  3       Mutation rate is constant? %s\n", !options->gamma ? "Yes" : "No, varying");
      if (options->migrguess == FST || options->thetaguess == FST)
	{
	  printf ("\n\n  FST-CALCULATION (for start value)\n  ----------------------------------\n");
	  printf ("  4       %s\n", options->fsttype == 'T' ? "Variable Theta, M symmetric" : "Variable M, Theta is the same for all populations");
	  printf ("  5       Print FST table in %s: %s\n", options->outfilename, options->printfst ? "Yes" : "No");
	}
      strcpy (custmexplain, custom_migration_type (options->migration_model));

      printf ("\n\n  MIGRATION MODEL\n  ---------------\n");
      printf ("  6       Model is set to %s\n", custmexplain);
      printf ("  7       Geographic distance matrix: %s\n", options->geo ? "Yes" : "No");
      printf ("\n\n  Are the settings correct?\n");
      printf ("  (Type Y to go back to the main menu or the letter for an entry to change)\n");
      fgets (input, LINESIZE, stdin);
      switch (input[0])
	{
	case '1':
	  printf ("  Which method? (F)st or (O)wn or (R)andom value\n");
	  fgets (input, LINESIZE, stdin);
	  switch(uppercase (input[0]))
	    {
	    case 'R':
	      options->thetaguess = RANDOMESTIMATE;
	      options->thetag = (double *) realloc (options->thetag, 
						    sizeof (double) * 2);
	      menuRandom(options->thetag);
	      options->numthetag = 2;
	      break;
	    case 'F':
	      options->thetaguess = FST;
	      break;
	    case 'O':
	      options->thetaguess = OWN;
	      printf ("  Initial Theta estimate?\n");
	      if (numpop == 0)
		{
		  printf ("  How many populations?\n");
		  check = scanf ("%ld", &numpop);
		  scanf ("%*[^\n]");
		}
	      options->thetag = (double *) realloc (options->thetag, sizeof (double) * (numpop + 1));
	      
	      for (i = 0; i < numpop; i++)
		{
		  printf ("Population %3li> ", i);
		  check = scanf ("%lf", &options->thetag[i]);
		}
	      options->numthetag = i;
	      break;
	    default:
	      options->thetaguess = FST;
	      break;
	    }
       
	  break;
	case '2':
	  printf ("Which method?  (F)st or (O)wn or (R)andom value\n");
	  fgets (input, LINESIZE, stdin);
	  switch (uppercase (input[0]))
	    {
	    case 'R':
	      options->migrguess = RANDOMESTIMATE;
	      options->mg = (double *) realloc (options->mg, 
						    sizeof (double) * 2);
	      menuRandom(options->mg);
	      options->nummg = 2;
	      break;
	    case 'F':
	      options->migrguess = FST;
	      break;
	    case 'O':
	      options->migrguess = OWN;
	      printf ("  Initial migration rate estimate?\n[give 4Nm for diploid data\n 2Nm for haploid data\n Nm for mtDNA data]\n");
	      while (numpop == 0)
		{
		  printf ("  How many populations?\n");
		  check = scanf ("%ld", &numpop);
		}
	      tt = 0;
	      options->mg = (double *) realloc (options->mg, sizeof (double) * (numpop * numpop + 1));
	      for (i = 0; i < numpop; i++)
		{
		  for (j = 0; j < numpop; j++)
		    {
		      if (j == i)
			{
			  tt++;
			  printf ("Population %3li              > --\n", i + 1);
			  continue;
			}
		      printf ("From population %-3li to %-3li> ", j + 1, i + 1);
		      check = scanf ("%lf", &options->mg[tt++]);
		      scanf ("%*[^\n]");
		      if (check == 0)
			break;
		    }
		}
	      fgets (input, LINESIZE, stdin);
	      options->nummg = tt;
	      tt = 0;
	      printf ("\n     You typed in the following migration matrix\n\n     ");
	      for (i = 0; i < numpop; i++)
		{
		  for (j = 0; j < numpop; j++)
		    {
		      if (i != j)
			printf ("%5.2f ", options->mg[tt++]);
		      else
			{
			  tt++;
			  printf ("----- ");
			}
		    }
		  printf ("\n     ");
		}
	      printf ("\n     [Press <Return> to continue]\n");
	      printf ("\n\n");
	      getchar ();
	      break;
	    default:
	      options->migrguess = FST;
	      break;
	    }
	  break;
	case '3':
	  options->gamma = !options->gamma;
	  break;
	case '4':
	  printf ("Which FST calculation method?\n");
	  printf ("(T)heta can be different for each population\n");
	  printf ("   and migration rates are symmetric.\n");
	  printf ("   (Number of populations >= 2)\n");
	  printf ("(M)igration rate can be asymmetric\n");
	  printf ("   and Theta is the same for both populations\n");
	  printf ("   (Number of populations = 2)\n");
	  fgets (input, LINESIZE, stdin);
	  switch (uppercase (input[0]))
	    {
	    case 'M':
	      options->fsttype = 'M';
	      fst_type ('M');
	      break;
	    case 'T':
	      options->fsttype = 'T';
	      fst_type ('T');
	      break;
	    default:
	      options->fsttype = 'T';
	      fst_type ('T');
	      break;
	    }
	  break;
	case '5':
	  options->printfst = !options->printfst;
	  break;
	case '6':		/* fill in the custom migration matrix */
	  read_custom_menu_migration (options);
	  break;
	}
    }
  while (uppercase (input[0]) != 'Y');
}


void
menuStrategy (option_fmt * options)
{
  char input[LINESIZE];
  do
    {
      printf ("  SEARCH STRATEGY\n\n");
#ifdef BAYESUPDATE
      printf ("  0       Strategy: %23.23s\n", options->bayes_update ?
	      "Adaptive likelihood surface estimation" :
	      "Kuhner, Yamato, Felsenstein method");
#endif
      if(!options->bayes_update)
	{
	  display_standard_mcmc(options);
	}
      else
	{
	  display_adaptive_mcmc(options);
	}
      printf ("\n\n  Are the settings correct?\n");
      printf ("  (Type Y to go back to the main menu or the letter for an entry to change)\n");
      fgets (input, LINESIZE, stdin);
      switch (atoi (input))
	{
	case 1:
	  do
	    {
	      printf ("  How many Short Chains?\n");
	      fgets (input, LINESIZE, stdin);
	      options->schains = atoi (input);
	      if (options->schains < 0)
		printf ("  Must be non-negative\n");
	    }
	  while (options->schains < 0);
	  break;
	case 2:
	  do
	    {
	      printf ("  How many trees to skip?\n");
	      fgets (input, LINESIZE, stdin);
	      options->sincrement = atoi (input);
	      if (options->sincrement <= 0)
		printf ("  Must be positive\n");
	    }
	  while (options->sincrement <= 0);
	  break;
	case 3:
	  do
	    {
	      printf ("  How many trees to sample?\n");
	      fgets (input, LINESIZE, stdin);
	      options->ssteps = atoi (input);
	      if (options->ssteps <= 0)
		printf ("  Must be a positive integer\n");
	    }
	  while (options->ssteps <= 0);
	  break;
	case 4:
	  do
	    {
	      printf ("  How many Long Chains?\n");
	      fgets (input, LINESIZE, stdin);
	      options->lchains = atoi (input);
	      if (options->lchains < 0)
		printf ("  Must be non-negative\n");
	    }
	  while (options->lchains < 0);
	  break;
	case 5:
	  do
	    {
	      printf ("  How many trees to skip?\n");
	      fgets (input, LINESIZE, stdin);
	      options->lincrement = atoi (input);
	      if (options->lincrement <= 0)
		printf ("  Must be positive\n");
	    }
	  while (options->lincrement <= 0);
	  break;
	case 6:
	  do
	    {
	      printf ("  How many trees to sample?\n");
	      fgets (input, LINESIZE, stdin);
	      options->lsteps = atoi (input);
	      if (options->lsteps <= 0)
		printf ("  Must be a positive integer\n");
	    }
	  while (options->lsteps <= 0);
	  break;
	case 7:
	  do
	    {
	      printf ("  How many genealogies to discard?\n");
	      fgets (input, LINESIZE, stdin);
	      options->burn_in = atoi (input);
	      if (options->burn_in <= 0)
		printf ("  Must be a positive integer or zero (0)\n");
	    }
	  while (options->burn_in < 0);
	  break;
	case 8:
	  options->replicate = !options->replicate;
	  if (options->replicate)
	    {
	      do
		{
		  printf ("  Summarize over (L)ong chains?\n");
		  printf ("  or (M)ultiple runs ?\n");
		  printf ("  [Default is (L)]\n");
		  fgets (input, LINESIZE, stdin);
		  if (uppercase (input[0]) == 'M')
		    {
		      printf ("  How many independent runs\n");
		      fgets (input, LINESIZE, stdin);
		      options->replicatenum = ATOL (input);
		      if (options->lcepsilon < 0)
			printf ("  Enter a number >= 1\n");
		    }
		  else
		    options->replicatenum = 0;
		}
	      while (options->replicatenum < 0);
	    }
	  else
	    {
	      options->replicatenum = 0;
	    }
	  break;
	case 9:
	  printf ("  Heating scheme? < NO | YES | (p)arallele chains | #heating_intervals (=YES)>\n");
	  fgets (input, LINESIZE, stdin);
	  switch (tolower (input[0]))
	    {
	    case 'p':
	    case 'y':
	      options->heating = 1;
	      options->heating_interval = 1;
	      break;
	    case '\0':
	    case 'n':
	    default:
	      if (atol (input) > 0)
		{
		  options->heating_interval = atol (input);
		  options->heating = 1;
		}
	      else
		{
		  options->heating = 0;
		}
	      break;
	    }
	  if (options->heating > 0)
	    {
	      fprintf (stdout, "Enter the number of different \"heated\" chains.\nMinimum is 4\n");
	      fgets (input, LINESIZE, stdin);
	      options->heated_chains = atol (input);
	      if (options->heated_chains < 4)
		options->heated_chains = HEATED_CHAIN_NUM;
	      read_heatvalues (options);
	    }
	  break;
	case 10:
	  options->movingsteps = !options->movingsteps;
	  if (options->movingsteps)
	    {
	      do
		{
		  printf ("  How big should the fraction of new genealogies\n");
		  printf ("  of the originally proposed number of samples be?\n");
		  fgets (input, LINESIZE, stdin);
		  options->acceptfreq = atof (input);
		  if (options->acceptfreq < 0)
		    printf ("  Range should be between 0 - 1, and not %f\n", options->acceptfreq);
		}
	      while (options->acceptfreq < 0);
	    }
	  break;
	case 11:
	  do
	    {
	      printf ("  Parameter likelihood epsilon?\n[INF is Default]\n");
	      fgets (input, LINESIZE, stdin);
	      if(uppercase(input[0])=='I')
		options->lcepsilon =  LONGCHAINEPSILON;
	      else
		options->lcepsilon = atof (input);
	      if (options->lcepsilon <= 0)
		printf ("  Must be a positive value, be warned: too small values will run the program forever\n");
	    }
	  while (options->lcepsilon <= 0);
	  break;
	case 12:
	  options->gelman = !options->gelman;
	  break;
	default:
	  break;
	}
    }
  while (uppercase (input[0]) != 'Y');
}

void display_standard_mcmc(option_fmt *options)
{
  char temp[LINESIZE];
  printf ("  1       Number of short chains to run?                %6ld\n", options->schains);
  if (options->schains > 0)
    {
      printf ("  2       Short sampling increment?                     %6ld\n", options->sincrement);
      printf ("  3       Number of recorded genealogies in short chain?%6ld\n", options->ssteps);
    }
  printf ("  4       Number of long chains to run?                 %6ld\n", options->lchains);
  if (options->lchains > 0)
    {
      printf ("  5       Long sampling increment?                      %6ld\n", options->lincrement);
      printf ("  6       Number of recorded genealogies in long chain? %6ld\n", options->lsteps);
    }
  printf ("  7       Number of genealogies to discard at \n");
  printf ("          the beginning of each chain? [Burn-in]        %6ld\n", options->burn_in);
  if (!options->replicate)
    printf ("  8       Combine chains for estimates?  No\n");
  else
    {
      if (options->replicatenum == 0)
	printf ("  8       Combine chains for estimates?  Yes, long chains\n");
      else
	printf ("  8       Combine chains for estimates?  Yes, over %li runs\n", options->replicatenum);
    }
  //        printf ("  9       Tempering (Heating) during increment:   %-10s\n",        
  if (options->heating == 0)
    printf ("  9       Heating:                             No\n");
  else
    printf ("  9       Heating:      Yes (%3.3li parallel chains)\n", options->heated_chains);
  //(U shaped acceptance)"
  //                : "Yes (V shaped acceptance)");
  printf ("\n -------------------------------------------------------------\n");
  printf (" Obscure options (consult the documentation on these)\n\n");
  if (options->movingsteps)
    printf (" 10       Sample a fraction of %2.2f new genealogies? Yes\n", options->acceptfreq);
  else
    printf (" 10       Sample at least a fraction of new genealogies? No\n");
  if(options->lcepsilon< LONGCHAINEPSILON)
    sprintf(temp,"%-11.2f",options->lcepsilon);
  else
    sprintf(temp,"%13.13s","infinity");
  printf (" 11       Epsilon of parameter likelihood    %s\n", temp);
  printf (" 12       Use Gelman's convergence criterium?      %3s\n", options->gelman ? "Yes" : " No");
}

void display_adaptive_mcmc(option_fmt *options)
{
      printf ("  1       Sampling increment?                           %6ld\n", options->lincrement);
      printf ("  2       Number of recorded genealogies in chain?      %6ld\n", options->lsteps);
  printf ("  3       Number of genealogies to discard at \n");
  printf ("          the beginning of chain? [Burn-in]             %6ld\n", options->burn_in);
  if (options->heating == 0)
    printf ("  4       Heating: %-43.43s\n","No");
  else
    printf ("  4       Heating:                          Yes (%3li parallel chains)\n", options->heated_chains);
  
}


void display_lratio_menutext(void)
{
  printf ("  H0: the ML-estimates and your values are the same\n");
  printf ("  H1: the ML-estimates and your values are different\n");
  printf ("  You need to specify values for ALL parameters\n");
  printf ("  Beware! If you restrict the migration model some test\n"); 
  printf ( " will not work. This test assumes that your hypothesis\n");
  printf ("  is nested in in the hypothesis you test against.\n");
  printf ("  Currently you cannot specify < or > comparisons AND\n");
  printf ("  averages of migration rates are calculated and not\n");
  printf ("  averages  for number of migrants\n\n");
}

void display_lrt_syntax(void)
{
  printf ("  Specify values  as an {n x n} matrix\n");
  printf ("  Theta values are on the diagonal, migration rates are\n");
  printf ("  off-diagonal, newlines are allowed, and\n");
  printf ("  values must be separated by space or commas or tabs.\n");
  printf ("\n  Syntax:\n");
  printf ("      * = sames as the MLE, and not estimated again\n");
  printf ("      s = averages of MIGRATION RATES from i->j and j->i\n");
  printf ("      m = averages of THETAS or MIGRATION RATES [!!!]\n");
  printf ("      value = specify your own value\n\n");
  printf ("  [If you want to leave this editor, type \"quit\" or \"end\" and <return>\n");// To review the Syntax type \"syntax\"\n");
}

char ask_lratio_type(lr_data_fmt *data, long counter)
{
  char answer='\0';
  char input[LINESIZE];
  printf ("  Is the test against the Maximum likelihood estimates or\n");
  printf("   or against arbitrary values? [(M)LE, (A)rbitrary]\n");
  fgets (input, 1024, stdin);
  switch (uppercase (input[0]))
    {
    case 'M':
      data[counter].type = MLE;
      answer = 'M';
      break;
    case 'A':
      answer = 'A';
      data[counter].type = ARBITRARY;
      break;
    case 'Q':
    case 'E': answer = 'Q';
      break;
    case 'S': answer = 'S';
      break;
    default:
      data[counter].type = MLE;
      break;
    }
  return answer;
}

void
read_custom_menu_lratio (option_fmt * options)
{
  //  static long numpop = 0, numpop2 = 0;
  //  char input[LINESIZE];
  //  long i, z = 0, pointer;
  //  char *temp;
  //  char *valpointer;
  char answer='\0';
  //  boolean done=FALSE;
  lratio_fmt *lratio = options->lratio;
  display_lratio_menutext();
  display_lrt_syntax(); 
  while(answer != 'Q')
    {
      //      answer = ask_lratio_type(lratio->data, lratio->counter);
      answer = 'M';
      lratio->data[lratio->counter].type = MLE;
      switch(answer)
	{
	case 'S': display_lrt_syntax(); break;
	case 'Q': break;
	case 'A': 
	case 'M':
	  how_many_pop(&options->numthetag);
	  answer = dialog_lrt(options->numthetag, lratio);
	  break;
	}
    }
}

void how_many_pop(long *numpop)
{
  char input[LINESIZE];
  if (*numpop == 0)
    {
      do
	{
	  printf ("  How many populations?\n");
	  fgets (input, 1024, stdin);
	  *numpop = atoi (input);
	}
      while (*numpop <= 0 && *numpop < 100);
    }
  else
    printf ("  The data set contains %li populations\n", *numpop);
}

void check_lrt_allocation(lratio_fmt *lratio)
{
  long i;
  if (lratio->counter + 1 == lratio->alloccounter)
    {
      lratio->alloccounter += 2;
      lratio->data = 
	(lr_data_fmt *) realloc (lratio->data, sizeof (lr_data_fmt) * 
				 lratio->alloccounter);
      for (i = lratio->counter + 1; i < lratio->alloccounter; i++)
	{
	  lratio->data[i].elem = 0;
	  lratio->data[i].value1 = 
	    (char *) calloc (1, sizeof (char) * LINESIZE);
	  lratio->data[i].value2 = 
	    (char *) calloc (1, sizeof (char) * LINESIZE);
	}
    }
}

char menuread_lrt_paramvalue(char *value, long *counter, long numpop2)
{
  char *valptr = value;
  long pointer = 0;
  long z = 0;
  char *temp;
  char input[LINESIZE];
  while (z < numpop2)
    {
      fgets (input, LINESIZE, stdin);
      temp = strtok (input, ", \n\t");
      if (temp != NULL)
	{
	  if (strchr ("EQ", uppercase (temp[0])))
	    {
	      // (*counter)--;
	      return 'Q';
	    }
	}
      while (temp != NULL)
	{
	  sprintf (valptr + pointer, " %s,", temp);
	  z++;
	  pointer += strlen (temp) + 2;
	  temp = strtok (NULL, ", \n\t");
	}
    }
  return ' ';
}

char dialog_lrt(long numpop, lratio_fmt *lratio)
{
  //  char input[LINESIZE];
  //  long z=0;
  char answer=' ';
  //  char *valptr1, *valptr2;
  long numpop2 = numpop * numpop;
  check_lrt_allocation(lratio);
  printf ("  %li. Likelihood ratio test\n",lratio->counter);
  printf ("       Enter now the %li values for the FIRST parameter set:\n",
	  numpop2);
  lratio->data[lratio->counter].type = MLE;
  answer = menuread_lrt_paramvalue(lratio->data[lratio->counter].value1,
				 &lratio->counter, numpop2);
  if(answer=='Q')
    return answer;
  if(lratio->data[lratio->counter].type==MLE)
    {
      lratio->counter++;
      return answer;
    }
  printf ("       Enter now the values for the SECOND parameter set:\n");
  answer = menuread_lrt_paramvalue(lratio->data[lratio->counter].value1,
				 &lratio->counter, numpop2);
  if(answer=='Q')
    return answer;
  lratio->counter++;
  return answer;
}	    

void
read_custom_menu_migration (option_fmt * options)
{
  char input[LINESIZE];
  long z = 0, numpop=0, numpop2;
  printf ("  Specify the migration model as an {n x n} matrix\n");
  printf ("  Theta values are on the diagonal, migration rates are\n");
  printf ("  off-diagonal, spaces (\" \"), \"{\", \"}\", or newlines\n");
  printf ("  are allowed, but not necessary.\n");
  printf ("\n  Syntax:\n");
  printf ("      * = independent parameter\n");
  printf ("      s = symmetric migration rates (M=m/mu)\n");
//  printf ("      S = symmetric migration rates (4Nm) \n");
  printf ("      m = average, either ALL thetas and/or ALL migration rates M\n");
  //  printf ("      M = average, either ALL thetas and/or ALL migration rates\n");
  printf ("      0 = (zero) not estimated]\n");
  printf ("      c = (constant) not estimated, taken from start-parameter]\n");
  if (options->numthetag > 0)
    numpop = options->numthetag;
  else
    {
      if (options->nummg > 0)
	numpop = (long) ((1. + sqrt (1. + (double) options->nummg * 4.)) / 2.);
      else
	{
	  how_many_pop(&options->numthetag);
	  printf ("\n  You must give %li values\n", 
		  options->numthetag * options->numthetag);
	}
    }
  numpop2 = numpop * numpop;
  printf ("  Enter now the values:\n");
  while (z < numpop2)
    {
      fgets (input, 1024, stdin);
      read_custom_migration (stdin, options, input, numpop);
      z = (long) strlen (options->custm);
    }
}

char *
custom_migration_type (long type)
{
  switch (type)
    {
    case MATRIX:
      return "Full migration matrix model";
      break;
    case MATRIX_SYMMETRIC:
      return "Symmetric migration matrix model";
      break;
    case MATRIX_SAMETHETA:
      return "Full migration matrix model (same Theta)";
      break;
    case MATRIX_ARBITRARY:
      return "User specified migration matrix model";
      break;
    case ISLAND:
      return "N-population island model";
      break;
    case ISLAND_VARTHETA:
      return "N-population island model (variable Theta)";
      break;
    case STEPSTONE:
      return "Stepping stone model";
      break;
    case CONTINUUM:
      return "Continuum model";
      break;
    case NEIGHBOR:
      return "Isolation by distance model";
      break;
    default:
      return "Illegal migration model";
      break;
    }
}


void
read_heatvalues (option_fmt * options)
{
  long i, z;
  char *tmp;
  char input[LINESIZE];
  double diff = 0.;
  fprintf (stdout, " ");
  fprintf (stdout, "Enter %li \"temperatures\".\n", options->heated_chains);
  fprintf (stdout, "or give a range of values [linear increase:      1 - 10]\n");
  fprintf (stdout, "or give a range of values [exponential increase: 1 @ 10]\n");
  fprintf (stdout, "The coldest, which is the first, has to be 1\n");
  fprintf (stdout, "[For example: 1 1.5 3 6]\n");
  fgets (input, LINESIZE, stdin);
  if (strstr (input, "-"))
    {
      tmp = strstr (input, "-");
      options->heat[options->heated_chains - 1] = fabs (atof (tmp + 1));
      options->heat[0] = 1.0;
      diff = options->heat[options->heated_chains - 1] - options->heat[0];
      diff /= options->heated_chains - 1.;
      for (i = 1; i < options->heated_chains; i++)
	options->heat[i] = options->heat[i - 1] + diff;
    }
  else
    {
      if (strstr (input, "@"))
	{
	  tmp = strstr (input, "@");
	  options->heat[options->heated_chains - 1] = fabs (atof (tmp + 1));
	  options->heat[0] = 1.0;
	  diff = 2. * (options->heat[options->heated_chains - 1] - options->heat[0]);
	  diff /= pow (2.0, (double) options->heated_chains) - 2.;
	  for (i = 1; i < options->heated_chains; i++)
	    options->heat[i] = options->heat[i - 1] + pow (2.0, i - 1.) * diff;
	}
      else
	{
	  z = 0;
	  tmp = strtok (input, " ,");
	  if (tmp != NULL)
	    {
	      options->heat[z++] = atof (tmp);
	    }
	  for (;;)
	    {
	      tmp = strtok (NULL, " ,");
	      if (tmp != NULL)
		{
		  options->heat[z++] = atof (tmp);
		}
	      else
		{
		  if (z < options->heated_chains)
		    {
		      fprintf (stdout, "%li more values are needed\n", options->heated_chains - z);
		      fgets (input, LINESIZE, stdin);
		      tmp = strtok (input, " ,");
		      if (tmp != NULL)
			{
			  options->heat[z++] = atof (tmp);
			}
		    }
		  else
		    break;
		}
	    }
	}
    }
  fprintf (stdout, "Chain         Temperature\n");
  fprintf (stdout, "-------------------------\n");
  for (i = options->heated_chains - 1; i >= 0; --i)
    fprintf (stdout, "%4li             %f\n", i, options->heat[i]);
  fprintf (stdout, "\n\nIs this correct? [Yes or No]\n");
  fgets (input, LINESIZE, stdin);
  if ('y' != tolower (input[0]))
    read_heatvalues (options);
}

void menuRandom(double *param)
{
  long check;
  printf ("Specify a MEAN and a STANDARD DEVIATION\nThis will be used to generate a random value from a Normal distribution\n");
  check = scanf ("%lf%lf", &param[0], &param[1]);
}

void	  start_tree_method(option_fmt *options)
{
  char input[LINESIZE];
  char compstring[LINESIZE];
  printf ("  Start genealogy is created with\n");
  printf ("      (a)utomatic\n");
  if (strchr(SEQUENCETYPES, options->datatype))
    {
      printf ("      (u)sertree\n");
      printf ("      (d)istancematrix\n");
    }
  printf ("      (r)andom\n");
  strcpy(compstring,(strchr(SEQUENCETYPES,options->datatype) ? 
		     "daur" : "dar"));
  do
    {
      fgets (input, LINESIZE, stdin);
    }
  while (strchr (compstring, (int) (lowercase (input[0]))) == NULL);
  switch (lowercase (input[0]))
    {
    case 'u':
      options->usertree = TRUE;
      options->dist = FALSE;
      break;
    case 'd':
      options->usertree = FALSE;
      options->dist = TRUE;
      options->randomtree = FALSE;
      break;
    case 'a':
      options->usertree = FALSE;
      options->dist = FALSE;
      options->randomtree = FALSE;
      break;
    case 'r':
      options->usertree = FALSE;
      options->dist = FALSE;
      options->randomtree = TRUE;
      break;
    }
}

void start_data_method(option_fmt *options)
{
  char input[LINESIZE];
  do
    {
      printf ("  (a)llele model\n");
      printf ("  (m)icrosatellite model [Ladder model; exact but slow]\n");
      printf ("  (b)rownian microsatellite model [Brownian motion model]\n");
      printf ("  (s)equence model\n");
      
      printf ("  (n)ucleotide polymorphism (SNP)\n");
      printf ("  (u)nlinked nucleotide polymorphism (SNP) using a PANEL\n");
      printf ("  (g)enealogy summaries\n");
      fgets (input, LINESIZE, stdin);
    }
  while (strchr ("ambnsguf", (int) (lowercase (input[0]))) == NULL);
  options->datatype = input[0];
  if (!strchr(SEQUENCETYPES,options->datatype))
    options->usertree = FALSE;
}
