/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 M A X I M I Z E R      R O U T I N E S

 - calculates single locus ML
 - mulitlocus ML with constant mutation rate
 - multilocus ML with gamma deviated mutation rate 

  using Broyden-Fletcher-Goldfarb-Shanno method
 

 Peter Beerli 1996-1998, Seattle
 beerli@genetics.washington.edu

 $Id: combroyden2.c,v 1.65 2001/04/09 17:59:04 beerli Exp $

-------------------------------------------------------*/

#include "migration.h"
#include "broyden.h"
#include "joint-chains.h"
#include "world.h"
#include "integrate.h"
#ifndef LAGUERRE
#include "derivatives.h"
#include "derivatives2.h"
#endif
#include "migrate_mpi.h"
#include "reporter.h"
#include "tools.h"
#ifdef LAGUERRE
#include "laguerre.h"
#include "gammalike.h"
#endif
#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif

extern void debug_plot(helper_fmt *helper);
/* global variable CALC_LIKE() points to calc_locus_like() or 
   norm_constant()*/
double (*calc_like) (nr_fmt *, double *, double *, long);

/* prototypes ------------------------------------------- */
/* public function used in main.c */
long estimateParameter (long rep, long G, world_fmt * world, option_fmt *options, double **dd, long chain, char type, long kind, long repkind);

/* calculates likelihood */
double calc_loci_like (helper_fmt * helper, double *param, double *lparam);

/* first derivatives for ML with constant mutation rate */
void simple_loci_derivatives (double *d, nr_fmt * nr, 
timearchive_fmt ** tyme, long locus);
/* progress-reporter */
void print_menu_finalestimate (boolean progress, long locus, long loci);

/* start parameter for multilocus estimator, creates simple average */
void calc_meanparam (world_fmt * world, long n, long repstart, long repstop);
/* multilocus first derivatives with or without gamma distrib. mut. rate */
void combine_gradient (nr_fmt * nr, helper_fmt * helper, double *gxv);
/* this helps to use the simple single locus material */
void copy_and_clear_d (nr_fmt * nr);
void add_back_d (nr_fmt * nr);
/* rescales parameter with theta1 for gamma calculation */
void set_gamma_param (double *paramn, double *paramo, 
		 double *lparamn, double *lparamo, 
		 double theta, nr_fmt * nr);
/* calculates norm, this creates a stopping criteria */
double norm (double *d, long size);
/* used in multilocus  line search */
double psilo (double lamda, helper_fmt * helper);
/* used in integral for gamma distr. mutation rate */
double phi (double t, helper_fmt *b);

/* calculates new multilocus parameter values */
void calc_loci_param (nr_fmt *nr, double *lparam, double *olparam, 
		      double *dv, double lamda, long nnn);
/* used to reset the BFSG stuff */
void reset_work (double **work, long nnn);

/* linesearch of BFGS
   and helper functions for this linesearch */
double line_search (double *dv, double *gxv, double (*func) (double lamda, helper_fmt * helper), helper_fmt * helper, double oldL);

void replace_with (int mode, double *a, double *b, double *c, double m, double *la, double *lb, double *lc, double ll);
double psil (double (*func) (double lamda, helper_fmt * helper), double a, double b, 
helper_fmt * helper, double *gxv, double *dv, double oldL, double *val1, double *val2);
double quadratic_lamda (double lamda, double a, double b, double c, double la, double lb, double lc);


void print_reset_big_param (FILE * file, double average, double maxtheta, long pop);

long do_profiles (world_fmt * world, nr_fmt *nr, double *likes, double *normd, long kind, long rep, long repkind);

void correct_values (world_fmt * world, nr_fmt * nr);

void which_calc_like (long repkind);

void set_replicates (world_fmt * world, long repkind, long rep, long *repstart, long *repstop);

void prepare_broyden (long kind, world_fmt * world, boolean * multilocus);

long multilocus_gamma_adjust (boolean multilocus, boolean boolgamma, world_fmt * world, long repstart, long repstop);
void profile_setup (long profilenum, long *profiles, double *param, double *value, long *nnn);

void setup_parameter0 (world_fmt * world, nr_fmt * nr, long repkind, 
		       long repstart, long repstop, long loci, long kind, 
		       boolean multilocus);

void report_broyden (double newL, double normd, long trials, 
		     boolean boolgamma, double theta1, double alpha, FILE *logfile);
boolean guard_broyden (double newL, double oldL, long trials, double normd, 
		       double *normd20);

void fill_helper (helper_fmt * helper, double *param, double *lparam, 
		  world_fmt * world, nr_fmt * nr);

/* maximizer routine */
long maximize(double *param, world_fmt *world, nr_fmt *nr, long analystype, long repkind);

void set_both_delta(nr_fmt *nr, 
	       double *delta, double *den, double *deo, 
	       double *gama,  double *gxv, double *gxv0, 
	       long size);
void set_expxv(double *expa, double *a, long size);
void set_expparam(double *expa, double *a, long size);
void set_logparam(double *loga, double *a, long size);
void finish_broyden(double newL, double normd, long trials, world_fmt *world, nr_fmt *nr, double *expxv, long nn, 
		    long analystype, long repkind);
void fill_profile_index(nr_fmt *nr);

/* Functions ++++++++++++++++++++++++++++++++++++++++++++++++*/
/* public functions---------------------------------------------- */
/* driver for maximizing single locus, multilocus functions      */
long
estimateParameter (long rep, long G, world_fmt * world, option_fmt *options, double **dd, long chain, char type, long kind, long repkind)
     //timearchive_fmt * tyme
{
  long trials = 0;
  double normd = 0.;
  double likes = 0.;
  double *param;
#ifdef MPI
  long i;
#else
  long repstart, repstop;
#endif
  nr_fmt *nr;
  nr = (nr_fmt *) calloc(1,sizeof(nr_fmt));
  if (kind == MULTILOCUS)
    {
      world->locus = world->loci;
      print_menu_finalestimate (world->options->progress, world->loci, world->loci);
    }

  create_nr(nr,world, G, 0, world->locus, repkind, rep);
  doublevec1d(&param,nr->partsize+1);

#ifdef MPI
  if(myID==MASTER)
    {
      for(i=0; i < world->numpop2; ++i)
	world->param0[i] = 1.;
    }
#else
  if (kind == MULTILOCUS)
    {
      set_replicates (world, world->repkind, world->rep, &repstart, &repstop);
      calc_meanparam (world, world->numpop2, repstart, repstop);
    }
#endif

  memcpy(param, world->param0, sizeof(double) * nr->numpop2);
  if (strchr (world->options->custm2, '0') || 
      strchr (world->options->custm2, 'c'))
    {
      if(kind==SINGLELOCUS && world->options->custm2[world->numpop2] == 'c')
	trials =  maximize(param, world, nr, kind, repkind);
      else
	trials = do_profiles (world, nr, &likes, &normd, kind, rep, repkind);
    }
  else
      trials =  maximize(param, world, nr, kind, repkind);

  if (kind == SINGLELOCUS && world->options->progress)
    {
      print_menu_chain (type, chain, world->atl[rep][world->locus].T, 
			world, options, rep);
      fprintf (stdout, "           Maximization cycles needed: %li\n", trials);
      fprintf (stdout, "           Norm of first derivatives:  %f\n", nr->normd);
      if (world->options->logfile)
	{
	  fprintf (world->options->logfile, 
		   "           Maximization cycles needed: %li\n", trials);
	  fprintf (world->options->logfile, 
		   "           Norm of first derivatives:  %f\n", nr->normd);
	}
    }
  if (kind == MULTILOCUS && world->options->plotnow && world->loci > 1)
    {
      if ((world->options->replicate && repkind != SINGLECHAIN) || 
	  (!world->options->replicate && repkind == SINGLECHAIN))
	create_loci_plot (world, world->plane[world->loci], nr, world->loci);
    }
  world->trials = trials;
  world->normd = nr->normd;
  free(param);
  destroy_nr(nr,world);
  return trials;
}


/* calculates the likelihood over all loci for the new parameter set */
double
calc_loci_like (helper_fmt * helper, double *param, double *lparam)
{
#ifndef MPI
  long locus;
#endif
  nr_fmt *nr = helper->nr;
  world_fmt *world = helper->nr->world;
  double logres = 0;
  if (helper->multilocus)
    {
#ifdef MPI
      /* must be MASTER node!
	 mutation rate is constant log(L(loci)) = Sum[log(L(locus))) */
      //      printf("%i> before mpi_likelihood_master\n",myID);
      logres = mpi_likelihood_master(param, lparam, world, nr, 
				     helper, world->who);
#else
      if (!helper->boolgamma)
	{
	  for (locus = 0; locus < world->loci; locus++)
	    {
	      if (nr->skiploci[locus])
		continue;
	      nr->locilikes[locus] = (*calc_like) (nr, param, 
						   lparam, locus);
	      logres += nr->locilikes[locus];
	    }
	}
      else
	{
#ifdef LAGUERRE
	  logres = gamma_loci_like(nr, helper->weight, param, lparam);
#else
	  for (locus = 0; locus < world->loci; locus++)
	    {
	      if (nr->skiploci[locus])
		continue;
	      helper->locus = locus;
	      helper->nr->locilikes[locus] = calc_single_gammalike (helper);
	      logres += helper->nr->locilikes[locus];
	    }
#endif  /* NOT LAGUERRE */
	}
#endif  /* NOT MPI */
    }
  else				       /* ->singlelocus */
    logres = (*calc_like) (nr, param, lparam, nr->world->locus);
  nr->llike = logres;
  return nr->llike;
}
/*
double
phi (double t, helper_fmt *b)
{
  double ll, alpha, beta;
  long locus;
  double weight;
  helper_fmt *helper;
  nr_fmt *nr;
  timearchive_fmt **atl;
  helper = (helper_fmt *) b;
  locus = (long) helper->locus;
  weight = (double) helper->weight;
  atl = (timearchive_fmt **) helper->atl;
  nr = (nr_fmt *) helper->nr;

  alpha = nr->oparam[nr->numpop2];
  beta = nr->oparam[0] / alpha;
  set_gamma_param (nr->param, nr->oparam, nr->lparam, nr->olparam, t, nr);
  ll = (*calc_like) (nr, nr->param, nr->lparam, locus);
#ifdef LAGUERRE
  ll = -t / beta + (alpha - 1.) * log (t) + ll  + weight;
#else
  ll = exp(-t / beta + (alpha - 1.) * log (t) + ll  + weight);
#endif
  return ll;
}

*/
/* private functions---------------------------------------------- */
/* derivatives */
void
combine_gradient (nr_fmt * nr, helper_fmt * helper,
		  double *gxv)
{
#ifndef MPI
  long locus;			//, iii;
#endif
  timearchive_fmt **tyme=NULL;
  
  long nnn = nr->partsize - nr->profilenum;
  memset (gxv, 0, sizeof (double) * nnn);
  if (helper->analystype==SINGLELOCUS)
    {
      simple_loci_derivatives (gxv, nr, tyme, nr->world->locus);
      force_symmetric_d (gxv, nr->world->options->migration_model, nr, nnn);
      grad2loggrad (helper->expxv, nr->indeks, gxv, nnn, nr->profilenum);
    }
  else
    {
      if (!helper->boolgamma)
	{
#ifdef MPI
	  if(myID==MASTER)
	    {
	      mpi_gradient_master(nr, nr->world, nr->world->who);
	      memcpy(gxv,nr->d,sizeof(double)*nr->partsize);
	    }
	  else //SLAVE
	    {
	      error("not allowed to execute mpi_gradient_worker()\n");
	    }	
#else
	  for (locus = 0; locus < nr->world->loci; locus++)
	    {
	      if (nr->skiploci[locus])
		continue;
	      memset(nr->d,0,sizeof(double)*nnn);
	      simple_loci_derivatives (nr->d, nr, tyme, locus);
	      add_vector(gxv, nr->d, nnn);
	    }
#endif

	  force_symmetric_d (gxv, nr->world->options->migration_model, nr, nnn);
	  grad2loggrad (helper->expxv, nr->indeks, gxv, nnn, nr->profilenum);
	}
      else
	{
#ifdef LAGUERRE
	  //if(nr->normd<5.)
	  //    gamma_loci_difference(helper);
	    //else
	    gamma_loci_derivative(helper);
	    memcpy(gxv,nr->d,sizeof(double) * nr->partsize);
#else
	  dt (gxv, helper);
#endif
	  force_symmetric_d (gxv, nr->world->options->migration_model, 
			     nr, nnn);
	  grad2loggrad (helper->expxv, nr->indeks, gxv, nnn, nr->profilenum);
	}
    }
}

void
simple_loci_derivatives (double *d, nr_fmt * nr, timearchive_fmt ** tyme, long locus)
{
  long g;
  gradient (d, nr, locus);
  for (g = 0; g < nr->partsize - nr->profilenum; g++)
    d[g] /= nr->PGC[locus];
}

void
set_gamma_param (double *param, double *oparam, 
		 double *lparam, double *olparam, 
		 double theta, nr_fmt * nr)
{
  long i;
  double logtheta;
  param[0] = theta;
  lparam[0] = logtheta = log(theta);
  for (i = 1; i < nr->numpop; i++)
    {
      lparam[i] = olparam[i] + logtheta - olparam[0];
      param[i] = exp(lparam[i]);
    }
  for (i = nr->numpop; i < nr->numpop2; i++)
    {
      lparam[i] = olparam[i] + olparam[0] - logtheta;
      param[i] = exp(lparam[i]);
    }
  lparam[i] = olparam[i];
  param[i] = exp(olparam[i]);
}


void
copy_and_clear_d (nr_fmt * nr)
{
  memcpy (nr->od, nr->d, sizeof (double) * nr->numpop2);
  memset (nr->d, 0, sizeof (double) * nr->numpop2);
}

void
add_back_d (nr_fmt * nr)
{
  long pop;
  for (pop = 0; pop < nr->numpop2; pop++)
    {
      nr->d[pop] += nr->od[pop];
    }
}


void
print_menu_finalestimate (boolean progress, long locus, long loci)
{
  static char nowstr[STRSIZE];
  if (progress)
    {
      get_time (nowstr, "%H:%M:%S");
      if (locus < loci)
	fprintf (stdout, "%s   Parameter estimation for locus %li\n", nowstr, locus);
      else
	fprintf (stdout, "%s   Parameter estimation over all loci\n", nowstr);
    }
}

void
print_reset_big_param (FILE * file, double average, double maxtheta, long pop)
{
  static char nowstr[STRSIZE];
  get_time (nowstr, "%H:%M:%S");

  fprintf (file, "%s   Theta for population %li was reset to the average\n", nowstr, pop);
  fprintf (file, "          of %f, it exceeded the maximum of %f \n", average, maxtheta);
}

void
calc_meanparam (world_fmt * world, long n, long repstart, long repstop)
{
  long r, pop, locus, loci = world->loci;


  memset (world->param0, 0, sizeof (double) * n);
  for (r = repstart; r < repstop; r++)
    {
      for (locus = 0; locus < loci; locus++)
	{
	  for (pop = 0; pop < n; pop++)
	    {
	      if (!world->data->skiploci[locus])
		world->param0[pop] += world->atl[r][locus].param[pop];
	    }
	}
    }
 for (pop = 0; pop < n; pop++)
    {
      world->param0[pop] /= (loci - world->skipped) * (repstop - repstart);
    }
}


double
norm (double *d, long size)
{
  int i;
  double summ = 0.;
  for (i = 0; i < (int) size; i++)
    {
      summ += d[i] * d[i];
    }
  return sqrt (summ);
}


double
psilo (double lamda, helper_fmt * helper)
{
  double like;
  nr_fmt *nr = helper->nr;
  double *lparam;
  double *param;
  lparam = (double *) calloc(helper->nr->partsize, sizeof(double));
  param = (double *) calloc(helper->nr->partsize, sizeof(double));
  calc_loci_param (nr, lparam, helper->xv, helper->dv, lamda, 
		   nr->partsize);
  set_expparam(param, lparam, nr->partsize);
  fill_helper(helper, param, lparam, nr->world, nr);
  like = calc_loci_like (helper, param, lparam);
  //printf("%f ",like);
  free(param);
  free(lparam);
  return -like;
}

void
calc_loci_param (nr_fmt *nr, double *lparam, double *olparam, double *dv, 
		 double lamda, long nnn)
{
  long i, ii, z = 0;
  if(nr->profilenum==0)
    {
      for (i = 0; i < nnn; i++)
	  lparam[i] = (MAX (-30., 
			    MIN (olparam[i] -lamda * dv[i], 30.)));
    }
  else
    {
      for (i = 0, ii = 0; i < nnn - nr->profilenum; i++)
	{
	  ii = nr->indeks[z++];
	  lparam[ii] =  (MAX (-30., 
			      MIN (olparam[ii] -lamda * dv[i], 30.)));
	}
    }
  param_all_adjust (lparam, nr->world->options, nr->world->numpop);
#ifdef LAGUERRE
  if(nr->world->locus >= nr->world->loci && nr->world->options->gamma)
    {
      if(lparam[nr->numpop2]>9.903487553)
      	lparam[nr->numpop2]=9.903487553;
      initgammacat(nr->categs, exp(lparam[nr->numpop2]), exp(lparam[0]), 
		   nr->rate, nr->probcat);
    }
#endif

}


void
reset_work (double **work, long nnn)
{
  long i, j;
  for (i = nnn; i < nnn + 5; i++)
    {
      for (j = 0; j < nnn + 1; j++)
	work[i][j] = 0.0;
    }
}



#define LAMBDA_ 1
#define M_      0

/* line_search returns LAMBDA (how far to jump in the search direction)
   needed in the the multilocus broyden-fletcher-shanno maximizer

   DV    search direction
   GXV   first derivatives of function to minimize
   FUNC  the function to minimize
   NR    all additional variables needed to calculate FUNC
   OLDL  -log(likelihood) of FUNC with LAMBDA=0 (previously calculated)

   needs functions: psil, replace_with, quadratic_lamda
 */
double
line_search (double *dv, double *gxv, double (*func) (double lamda, 
helper_fmt * helper), helper_fmt * helper, double oldL)
{
  long trials = 0;
  double lamda_, ql;
  double a, b, c;

  double ll, la, lb, lc;
  double m = 1.0;
  a = 0.0;
  b = 0.000001;
  c = 0.9;
  la = -oldL;
  lb = (*func) (b, helper);
  lc = (*func) (c, helper);
  while (trials++ < NTRIALS)
    {
      /*      printf ("*%3li> %f %f %f / %f %f %f\n", trials, a, b, c, la, lb, lc); */
      lamda_ = (c * c * (lb - la) + b * b * (la - lc) + a * a * (lc - lb)) / (2. * (b * la - c * la - a * lb + c * lb + a * lc - b * lc));
      if ((lamda_ <= 0.0) || (lamda_ >= m))
	{
	  if ((a == m || b == m || c == m))
	    return m;
	  else
	    {
	      ll = (*func) (m, helper);
//      replace_with(M_,&a,&b,&c,m,&la,&lb,&lc,ll);
	      replace_with (LAMBDA_, &a, &b, &c, m, &la, &lb, &lc, ll);
	      continue;
	    }
	}
      ll = (*func) (lamda_, helper);
      ql = quadratic_lamda (lamda_, a, b, c, la, lb, lc);
      if ((fabs (ll - MIN3 (la, lb, lc)) <= BIGEPSILON) || (fabs (ll - ql) <= BIGEPSILON))
	return lamda_;
      else
	{
	  /*  if(((a<b<c) || (c<b<a)) && (lb < MIN(la,lc)))
	     return lamda_;
	     if(((b<a<c) || (c<a<b)) && (la < MIN(lb,lc)))
	     return lamda_;
	     if(((a<c<b) || (b<c<a)) && (lc < MIN(la,lb)))
	     return lamda_; */
	  replace_with (LAMBDA_, &a, &b, &c, lamda_, &la, &lb, &lc, ll);
	  m = MAX3 (a, b, c);
	}
    }
  return lamda_;
}

void
replace_with (int mode, double *a, double *b, double *c, double m, double *la, double *lb, double *lc, double ll)
{
  double ma, mb, mc;
  if (mode == LAMBDA_)
    {
      ma = *la;
      mb = *lb;
      mc = *lc;
    }
  else
    {
      ma = ll - *la;
      mb = ll - *lb;
      mc = ll - *lc;
    }
  if (ma > mb)
    {
      if (ma > mc)
	{
	  *a = m;
	  *la = ll;
	}
      else
	{
	  *c = m;
	  *lc = ll;
	}
    }
  else
    {
      if (mb > mc)
	{
	  *b = m;
	  *lb = ll;
	}
      else
	{
	  *c = m;
	  *lc = ll;
	}
    }
}

double
psil (double (*func) (double lamda, helper_fmt * helper), double a, double b, helper_fmt * helper, double *gxv, double *dv, double oldL, double *val1, double *val2)
{
  long i;
  long nn = helper->nr->partsize;
  double value = 0.0;

  if (a == 0.0 && b == 0)
    {
      for (i = 0; i < nn; i++)
	{
	  value += dv[i] /* gxv[i] */ ;
	}
      *val1 = -oldL;
      *val2 = -oldL;
      return value;
    }
  else
    {
      if (a == 0)
	value = (((*val2) = (*func) (b, helper)) - ((*val1) = -oldL)) / b;
      else
	{
	  if (b == 0)
	    value = (((*val2) = -oldL) - ((*val1) = (*func) (a, helper))) / (-a);
	  else
	    value = (((*val2) = (*func) (b, helper)) - ((*val1) = (*func) (a, helper))) / (b - a);
	}
      return value;
    }
}

double
quadratic_lamda (double lamda, double a, double b, double c, double la, double lb, double lc)
{
  double alpha, beta, gama;
  double aa, bb, cc;
  aa = a * a;
  bb = b * b;
  cc = c * c;
  alpha = ((c - b) * la + (a - c) * lb + (b - a) * lc) / ((b - a) * (c - a) * (c - b));
  beta = ((cc - bb) * la + (aa - cc) * lb + (bb - aa) * lc) / ((b - a) * (b - c) * (c - a));
  gama = (b * cc * la - bb * c * la + aa * c * lb - a * cc * lb - aa * b * lc + a * bb * lc) / ((b - a) * (c - a) * (c - b));


  return alpha * lamda * lamda + beta * lamda + gama;

}


long
do_profiles (world_fmt * world,  nr_fmt *nr, double *likes, 
	     double *normd, long kind, long rep, long repkind)
{
  char *p;
  long z = 0;
  long trials;
  long i = 0;
  p = world->options->custm2;
  while (*p != '\0')
    {
      if (*p == '0' || *p == 'c')
	{
	  nr->profiles[z] = i;
	  nr->values[z] = world->param0[i];
	  z++;
	}
      p++;
      i++;
    }
  nr->profilenum= z;
  trials = maximize(world->param0, world, nr, kind, repkind);
  return trials;
}

void
correct_values (world_fmt * world, nr_fmt * nr)
{
  long pop, i, ii, z;
  double ssum;
  long nsum;
  long elem = (world->options->gamma ? nr->numpop2 : nr->partsize);
  for (pop = 0; pop < world->numpop; pop++)
    {
      if (world->param0[pop] >= BIGGEST_THETA)
	{
	  ssum = 0;
	  nsum = 0;
	  for (i = 0; i < world->numpop; i++)
	    {
	      if (world->param0[i] <= BIGGEST_THETA)
		{
		  ssum += world->param0[i];
		  nsum++;
		}
	    }
	  if (nsum > 0 && ssum != 0.0)
	    world->param0[pop] = ssum / (double) nsum;
	  else
	    {
	      if (ssum == 0.0)
		world->param0[pop] = SMALLEST_THETA;
	      else
		world->param0[pop] = BIGGEST_THETA;
	    }
	  if (world->options->writelog)
	    print_reset_big_param (world->options->logfile, world->param0[pop], BIGGEST_THETA, pop);
	  print_reset_big_param (stdout, world->param0[pop], BIGGEST_THETA, pop);
	}
    }
  z=0;
  for (i = 0, ii = 0; i < elem - nr->profilenum; i++)
    {
      ii = (nr->profilenum > 0) ? nr->indeks[z++] : i;
      if (world->param0[ii] >= BIGGEST_MIGRATION)
	world->param0[ii] = BIGGEST_MIGRATION;
      if (world->param0[ii] < SMALLEST_MIGRATION)
	world->param0[ii] = SMALLEST_MIGRATION;
    }
}

void
which_calc_like (long repkind)
{
  switch (repkind)
    {
    case SINGLECHAIN:
      calc_like = (double (*)(nr_fmt *, double *, double *, long)) calc_locus_like;
      break;
    case MULTIPLECHAIN:
    case MULTIPLERUN:
      //calc_like = (double (*)(nr_fmt *,double *, long)) calc_locus_like;
      calc_like = (double (*)(nr_fmt *, double *, double *, long)) norm_constant;
      // break;
    default:
      calc_like = (double (*)(nr_fmt *, double *, double *, long)) calc_locus_like;
      break;
    }
}


void
set_replicates (world_fmt * world, long repkind, long rep, long *repstart, long *repstop)
{
  if (world->options->replicate)
    {
      *repstart = (repkind == SINGLECHAIN) ? rep : 0;
      *repstop = (repkind == SINGLECHAIN) ? rep + 1 : world->repstop;
    }
  else
    {
      *repstart = 0;
      *repstop = 1;
    }
}



void
prepare_broyden (long kind, world_fmt * world, boolean * multilocus)
{
  if (kind == SINGLELOCUS || (kind == PROFILE && world->loci == 1))
    {
      if (world->loci == 1)
	world->locus = 0;
      *multilocus = FALSE;
    }
  else
    {
      *multilocus = TRUE;
    }
}


long
multilocus_gamma_adjust (boolean multilocus, boolean boolgamma, world_fmt * world, long repstart, long repstop)
{
  long nn;
  if (multilocus)
    {
      nn = boolgamma ? world->numpop2 + 1 : world->numpop2;
      if (boolgamma)
	{
	  world->param0 = (double *) realloc (world->param0, sizeof (double) * (nn > 0 ? nn : 1));
	  world->param0[world->numpop2] = world->options->alphavalue;
	}
    }
  else
    nn = world->numpop2;
  return nn;
}

/* profiler setup-------------------------- 
   needs the passed in values generated in profiles setup (profile.c)
*/
void
profile_setup (long profilenum, long *profiles, double *param, double *value, long *nnn)
{ 
  long i;
  if (profilenum > 0)
    {
      *nnn -= profilenum;
      for (i = 0; i < profilenum; i++)
	param[profiles[i]] = log(value[i]);
    }
  if(*nnn==0)
    *nnn = 1;
}


void
setup_parameter0 (world_fmt * world, nr_fmt * nr, long repkind, long repstart, long repstop, long loci, long kind, boolean multilocus)
{
  long locus, r;
#ifdef MPI
  long ll;
  if(myID!=MASTER)
    {
#endif
      if (multilocus)
	{
#ifdef MPI
	  for (ll = 0; ll < locidone; ll++)
	    {
	      locus = world->locus = world->who[ll];
	      printf("%i> interpolation for locus %li\n", myID, locus);
#else
	  for (locus = 0; locus < loci; locus++)
	    {
#endif
	      if (repkind == SINGLECHAIN)
		{
		  for (r = repstart; r < repstop; r++)
		    create_apg0 (nr->apg0[r][locus], nr, &world->atl[r][locus]);
		}
	      else
		{
		  if (kind != PROFILE)
		    {
		      for (r = repstart; r < repstop; r++)
			create_apg0 (nr->apg0[r][locus], nr, &world->atl[r][locus]);
		      interpolate_like (nr, locus);
		    }
		  for (r = repstart; r < repstop; r++)
		    create_multiapg0 (nr->apg0[r][locus], nr, r, locus);
		}
	    }
	}
      else				//single locus
	{
	  if (repkind == SINGLECHAIN)
	    {
	      for (r = repstart; r < repstop; r++)
		create_apg0 (nr->apg0[r][world->locus], nr, &world->atl[r][world->locus]);
	    }
	  else
	    {
	      if (kind != PROFILE)
		{
		  for (r = repstart; r < repstop; r++)
		    create_apg0 (nr->apg0[r][world->locus], nr, &world->atl[r][world->locus]);
		  interpolate_like (nr, world->locus);
		}
	      for (r = repstart; r < repstop; r++)
		create_multiapg0 (nr->apg0[r][world->locus], nr, r, world->locus);
	    }
	}
#ifdef MPI
    }
#endif
}

void
report_broyden (double newL, double normd, long trials, 
		boolean boolgamma, double theta1, double alpha, FILE *logfile)
{
  if (boolgamma)
    fprintf (stdout, "%li> Log(L)=%f Norm=%f Alpha=%f Theta_1=%f\n", 
	     trials, newL, normd, alpha, theta1);
  else
    fprintf (stdout, "%li> Log(L)=%f Norm=%f Theta_1=%f\n", 
	     trials, newL, normd, theta1);
  if(logfile!=NULL)
    {
      if (boolgamma)
	fprintf (logfile, "%li> Log(L)=%f Norm=%f Alpha=%f Theta_1=%f\n", 
	     trials, newL, normd, alpha, theta1);
      else
	fprintf (logfile, "%li> Log(L)=%f Norm=%f Theta_1=%f\n", 
	     trials, newL, normd, theta1);
    }
}

boolean
guard_broyden (double newL, double oldL, long trials, double normd, double *normd20)
{
  if ((trials + 1) % 20 == 0)
    {
      if (fabs (normd - *normd20) < EPSILON)
	return TRUE;
      *normd20 = normd;
    }
  return FALSE;
}

/* for profiling: filling of index array */
void fill_profile_index(nr_fmt *nr)
{
  long i, ii;
  long profilenum = nr->profilenum;
  if ((profilenum > 0) && profilenum < nr->partsize)
    {
      for (i = 0, ii = 0; i < nr->partsize; i++)
	{
	  if (!find (i, nr->profiles, profilenum))
	    nr->indeks[ii++] = i;
	}
    }
}

// maximizer
// - MPI

// likelihood
// - MPI
// - Gamma
// - custom migration matrix aware
// - profile aware
// - lrt aware [special case of cumstom migration]

// maximizer sceleton
// analystype = (SINGLELOCUS, MULTILOCUS, PROFILE)
long
maximize(double *param, world_fmt *world, nr_fmt *nr, long analystype, long repkind)
{
  boolean stop;
  long repstart, repstop;
  //  long Gmax;
  double two = 2.;
  double newL;
  double oldL;
  double lamda;
  long trials;
  double normd=1000000;
  double normd20;
  worldoption_fmt * wopt = world->options;
  long nnn = nr->partsize; // keeps track of profile or standard 
  long nn; // keeps track of the full parameter array
  double ** hess;
  double *delta;
  double *gama;
  double *dv;
  double *gxv;
  double *gxv0;
  double *xv0;
  double *xv;
  double *expxv;
  helper_fmt helper;
  //  long reset_z = 0;
  helper.boolgamma = world->locus==world->loci ? wopt->gamma : FALSE;
  if(helper.boolgamma)
      param[nnn-1] = world->options->alphavalue;
  doublevec2d(&hess,nnn,nnn);
  reset_hess (hess, nnn);
  doublevec1d(&delta,nnn);
  doublevec1d(&gama,nnn);
  doublevec1d(&xv,nnn+1);
  helper.xv = xv;
  //  expxv = xv + nnn;
  //memcpy(expxv,param,sizeof(double)*nnn);
  doublevec1d(&gxv,nnn);
  doublevec1d(&gxv0,nnn);
  setdoublevec1d(&expxv, param, nnn+1);
  helper.expxv = expxv;
  set_logparam(xv,expxv,nnn);
  setdoublevec1d(&xv0,xv, nnn);
  helper.analystype = analystype;
  set_replicates (world, world->repkind, world->rep, &repstart, &repstop);
  nr->repstart = repstart;
  nr->repstop = repstop;
  nr->normd = normd;
  which_calc_like (world->repkind);
  prepare_broyden (analystype,world, &helper.multilocus);
  nnn = multilocus_gamma_adjust (helper.multilocus, helper.boolgamma, 
				 world, repstart, repstop);
  nn = nnn;
  profile_setup (nr->profilenum, nr->profiles, xv, nr->values, &nnn);
  fill_profile_index(nr);

  calc_loci_param (nr, xv, xv0, gxv, 0. , nnn);
  set_expparam(expxv,xv,nn);
  setup_parameter0 (world, nr, world->repkind, repstart, repstop, 
		    world->loci, analystype, helper.multilocus);
  fill_helper (&helper, expxv, xv, world, nr);
  oldL = calc_loci_like (&helper, expxv, xv);
  fill_helper (&helper, expxv, xv, world, nr);
  combine_gradient (nr, &helper, gxv);
  memcpy (gxv0, gxv, sizeof (double) * nnn);
  memcpy (xv0, xv, sizeof (double) * nn);
  setdoublevec1d(&dv,gxv, nnn);
  helper.dv = dv;
/* Maximization cycle ---------------------------------------------*/
  for (trials = 0; trials < NTRIALS; trials++)
    {
#ifdef MAC
      eventloop ();
#endif
      newL = -DBL_MAX;
      if(helper.boolgamma)
      	lamda = 10.;//calc_line (&helper, -10., 0.1, 10., 
      //   world->locus > 1 ? psilo : psi);//line_search(dv,gxv,analystype==MULTILOCUS ? psilo : psi,
      // nr,oldL);
      else
	lamda = calc_line (&helper, -10., 0.1, 10., 
			 world->locus > 1 ? psilo : psi);
      //       	printf("\n");
      //      printf("lamda=%f, %s\n",lamda,analystype==MULTILOCUS ? "psilo" : "psi");
      //reset_z++;
      while ((newL <= oldL || isnan (newL) || newL <= -DBL_MAX) && 
	     fabs (lamda) > 10. * DBL_EPSILON)
	{
	  calc_loci_param (nr, xv, xv0, dv, lamda, nn);
	  set_expparam(expxv,xv,nn);
	  fill_helper (&helper, expxv, xv, world, nr);
	  newL = calc_loci_like (&helper, expxv, xv);
	  lamda /= two; 
	}
      nr->normd = normd = norm (gxv, nnn);
#ifdef LAGUERRE
      if(nr->world->locus >= nr->world->loci && nr->world->options->gamma)
	{
	  initgammacat(nr->categs, expxv[nr->numpop2], expxv[0], 
		       nr->rate, nr->probcat);  
	  if(fabs(oldL-newL)<1e-9)
	    lamda=0.;
	}
#endif
      if (world->options->verbose && newL > oldL)
	report_broyden (newL, normd, trials, helper.boolgamma, 
			expxv[0], expxv[nr->numpop2], world->options->logfile);
      stop = guard_broyden (newL, oldL, trials, normd, &normd20);
      if (normd < EPSILON || stop)
	break;			// stopping criteria
      else if (oldL >= newL)
	lamda = 0;
      else
	oldL = newL;
      /* reset sec deri. matrix if lamda goes to 0 and retry */
      //      if(reset_z > 10)
      //{
      //  lamda=0;
      //  reset_z = 0;
      //}
      if (fabs (lamda) <= DBL_EPSILON * 10.)
	{
	  reset_hess (hess, nnn);
	  two = -two;
	  memcpy (dv, gxv, sizeof (double) * nnn);
	  continue;
	}
      memcpy (gxv0, gxv, sizeof (double) * nnn);
      //calc_loci_param (nr, xv, xv0, dv, lamda * two, nn);
      //set_expparam(expxv,xv,nn);
      fill_helper (&helper, expxv, xv, world, nr);
      combine_gradient (nr, &helper, gxv);
      set_both_delta(nr, delta, xv, xv0, gama, gxv, gxv0, nnn);
      calc_hessian (hess, nnn, delta, gama);
      calc_dv (dv, hess, gxv, nnn); 
      //memcpy (dv, gxv, sizeof (double) * nnn);
      memcpy (xv0, xv, sizeof (double) * nn);
    }
/* end maximizer cycle ------------------------------------*/
  finish_broyden(newL, normd, trials, world, nr, expxv, nn, analystype, repkind);
  memcpy(param, expxv,sizeof(double)*nn);
  free(hess[0]);
  free(hess);
  free(delta);
  free(gama);
  free(dv);
  free(gxv);
  free(gxv0);
  free(xv0);
  free(xv);
  free(expxv);

  return trials;
}   

void finish_broyden(double newL, double normd, long trials, 
		    world_fmt *world, nr_fmt *nr, double *expxv, long nn, 
		    long analystype, long repkind)
{ 
  long loci=world->loci;

  timearchive_fmt ** tyme = world->atl;
  memcpy (world->param0, expxv, sizeof (double) * nn);
  correct_values (world, nr);	// resets huge thetas to average

  if (world->options->verbose)
    fprintf (stdout, "\n");

  switch (analystype)
    {
    case MULTILOCUS:
      tyme[0][loci].param = (double *) realloc (tyme[0][loci].param, 
						sizeof (double) * nn);
      tyme[0][loci].param_like = newL;
      tyme[0][loci].normd = normd;
      tyme[0][loci].trials = trials;
      memmove (tyme[0][loci].param, expxv, sizeof (double) * nn);
      nr->normd = normd;
      convergence_check (world, world->options->verbose);
      break;
    case SINGLELOCUS:
      world->param_like = newL;
      if (repkind != SINGLECHAIN)
	{
	  tyme[world->repstop][world->locus].param_like = newL;
	  tyme[world->repstop][world->locus].normd = normd;
	  tyme[world->repstop][world->locus].trials = trials;
	  memmove (tyme[world->repstop][world->locus].param, expxv, 
		  sizeof (double) * nn);
	}
      else
	{
	  tyme[world->rep][world->locus].param_like = newL;
	  tyme[world->rep][world->locus].normd = normd;
	  tyme[world->rep][world->locus].trials = trials;
	  memmove (tyme[world->rep][world->locus].param, expxv, 
		  sizeof (double) * nn);
	}
      convergence_check (world, world->options->verbose);
      if(world->loci==1)
	{
	  if ((!world->options->gelman && 
	       world->param_like < world->options->lcepsilon && 
	       world->options->plotnow && !world->options->simulation) || 
	      (world->options->plotnow && world->options->gelman && 
	       world->gelmanmaxR < GELMAN_MYSTIC_VALUE))
	    {
	      if ((world->options->replicate && repkind != SINGLECHAIN) || 
		  (!world->options->replicate && repkind == SINGLECHAIN))
		create_locus_plot (world, world->plane[world->locus], nr, 
				   nr->atl[world->rep][0].T);
	    }
	}
      nr->normd = normd;
      break;
    case PROFILE:
      //      memmove (nr->profparam, world->param0, sizeof (double) * nr->partsize);
      nr->llike = newL;
      nr->normd = normd;
    }

  if (world->options->verbose && world->repkind == SINGLECHAIN)
    {
      if (analystype == SINGLELOCUS)
	print_contribution (nr, nr->atl, nr->atl[world->rep][world->locus].T);
    }
}


void set_both_delta(nr_fmt *nr, 
	       double *delta, double *den, double *deo, 
	       double *gama,  double *gxv, double *gxv0, 
	       long size)
{
  long i,ii;
  if ((nr->profilenum > 0) && nr->profilenum < nr->partsize)
    {
      for (i = 0; i < size; i++)
      {
	ii = nr->indeks[i];
	delta[i] =  den[ii] - deo[ii];
	gama[i] =  gxv[i] - gxv0[i];
      }
    }
  else
    {
      for (i = 0; i < size; i++)
      {
	delta[i] = den[i] - deo[i];
	gama[i] =  gxv[i] - gxv0[i];
      }
    }
}
  
void set_expparam(double *expa, double *a, long size)
{
  long i;
  for(i=0;i<size;i++)
    expa[i] = exp(a[i]);
}
  
void set_logparam(double *loga, double *a, long size)
{
  long i;
  for(i=0;i<size;i++)
    loga[i] = log(a[i]);
}


void
fill_helper (helper_fmt * helper, double *param, double *lparam,
	     world_fmt * world, nr_fmt * nr)
{
  double alpha = 0;
  double theta1 = param[0];
  double denom = 0;
  helper->boolgamma = world->locus < world->loci ? 
    FALSE : world->options->gamma;
  if (helper->boolgamma)
    {
      alpha = param[nr->numpop2];
      if(alpha<=0.0)
	error("no no! alpha=0\n");
      denom = LGAMMA (alpha) + log (theta1 / alpha) * alpha;
    }
  memcpy(nr->lparam,lparam,sizeof(double) * nr->partsize);
  memcpy(nr->param,param,sizeof(double) * nr->partsize);

  helper->nr = nr;
  helper->ll = -DBL_MAX;
  helper->weight = denom;
  helper->atl = world->atl;
}




