/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 S E Q U E N C E S   R O U T I N E S 

 used in menu.c

 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
<<<<<<< sequence.c
 $Id: sequence.c,v 1.37 2001/07/25 19:27:24 beerli Exp $
=======
 
Copyright 2001 Peter Beerli and Joseph Felsenstein
>>>>>>> 1.34

$Id: sequence.c,v 1.37 2001/07/25 19:27:24 beerli Exp $

-------------------------------------------------------*/

#include "migration.h"

#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif
/* prototypes ------------------------------------------- */
void make_sequences (world_fmt * world, option_fmt * options, data_fmt * data,
		     long locus);
void init_sequences (world_fmt * world, option_fmt * options, data_fmt * data,
		     long locus);
void init_sequences2 (world_fmt * world, seqmodel_fmt * seq, long locus);
void initratio (option_fmt * options);
void initfreqs (double *freqa, double *freqc, double *freqg, double *freqt);
void initcatn (long *categs);
boolean initcategs (long categs, double *rate, double *probcat);
void initprobcat (long categs, double *probsum, double *probcat);
void init_tbl (world_fmt * world, long locus);
void print_weights (FILE * outfile, world_fmt * world, option_fmt * options,
		    long locus);
void print_tbl (FILE * outfile, world_fmt * world, option_fmt * options,
		long locus);
double treelike_seq (world_fmt * world, long locus);
double treelike_snp (world_fmt * world, long locus);
/*private functions */
void getbasefreqs (option_fmt * options, seqmodel_fmt * seq, long locus);
void empiricalfreqs (world_fmt * world, option_fmt * options,
		     seqmodel_fmt * seq, long locus);
void makeweights (world_fmt * world, data_fmt * data, long locus);
void makevalues_seq (world_fmt * world, option_fmt * options, data_fmt * data,
		     long locus);
void make_invarsites (world_fmt * world, data_fmt * data, long locus);
void make_invarsites_unlinked (world_fmt * world, data_fmt * data,
			       long locus);
void sitecombine2 (world_fmt * world, data_fmt * data, long sites,
		   long locus);
void sitesort2 (world_fmt * world, data_fmt * data, long sites, long locus);
void sitescrunch2 (world_fmt * world, long sites, long i, long j, long locus);
void inputoptions (world_fmt * world, option_fmt * options, data_fmt * data,
		   long locus);
void inputweights (world_fmt * world, data_fmt * data, long chars);
void inputcategs (long a, long b, world_fmt * world, option_fmt * options,
		  data_fmt * data);
void initlambda (option_fmt * options);
void printweights (FILE * outfile, world_fmt * world, option_fmt * options,
		   short inc, long chars, short *weight, char *letters);
void print_seqfreqs (FILE * outfile, world_fmt * world, option_fmt * options);

void snp_invariants (contribarr invariants, long endsite, long rcategs,
		     seqmodel_fmt * seq, phenotype x1);

void makevalues_snp (world_fmt * world, option_fmt * options, data_fmt * data,
		     long locus);

void init_sequences_aliases (world_fmt * world, option_fmt * options,
			     data_fmt * data, long locus);


void check_basefreq (option_fmt * options);

extern void swap (void *, void *);

const short C = 1;
const short G = 2;
const short T = 3;


/* allocation things */
void
init_sequences (world_fmt * world, option_fmt * options, data_fmt * data,
		long locus)
{
  static boolean done = FALSE;
  long sites = world->data->seq->sites[locus];
  if (world->options->datatype == 'u')
    sites *= 5;
  if (!done)
    {
      done = TRUE;
      world->data->seq->alias = (long *) calloc (1, sizeof (long) * sites);
      world->data->seq->ally = (long *) calloc (1, sizeof (long) * sites);
      world->data->seq->aliasweight =
	(long *) calloc (1, sizeof (long) * sites);
      world->data->seq->location = (long *) calloc (1, sizeof (long) * sites);
      world->data->seq->category = (long *) calloc (1, sizeof (long) * sites);
      world->data->seq->weight = (short *) calloc (1, sizeof (short) * sites);
    }
  else
    {
      world->data->seq->alias =
	(long *) realloc (world->data->seq->alias, sizeof (long) * sites);
      world->data->seq->ally =
	(long *) realloc (world->data->seq->ally, sizeof (long) * sites);
      world->data->seq->aliasweight =
	(long *) realloc (world->data->seq->aliasweight,
			  sizeof (long) * sites);
      world->data->seq->location =
	(long *) realloc (world->data->seq->location, sizeof (long) * sites);
      world->data->seq->category =
	(long *) realloc (world->data->seq->category, sizeof (long) * sites);
      world->data->seq->weight =
	(short *) realloc (world->data->seq->weight, sizeof (short) * sites);
    }
}

void
init_sequences_aliases (world_fmt * world, option_fmt * options,
			data_fmt * data, long locus)
{
  inputoptions (world, options, data, locus);
  if (!options->freqsfrom)
    getbasefreqs (options, world->data->seq, locus);
  makeweights (world, data, locus);
}


/* menu material ----------------------------------------- */
void
initratio (option_fmt * options)
{
  long z = 0;
  char *tmp;
  char input[LINESIZE];
  printf
    ("Transition/transversion ratio?\nEnter a value for each locus, spaced by blanks or commas\n");
  FGETS (input, LINESIZE, stdin);
  tmp = strtok (input, " ,\n");
  while (tmp != NULL)
    {
      options->ttratio[z++] = atof (tmp);
      tmp = strtok (NULL, " ,;\n");
      options->ttratio =
	(double *) realloc (options->ttratio, sizeof (double) * (z + 1));
      options->ttratio[z] = 0.0;
    }
}

void
initfreqs (double *freqa, double *freqc, double *freqg, double *freqt)
{
  char input[LINESIZE];
  int scanned;
  double summ = 0;

  printf
    ("Base frequencies for A, C, G, T/U\n (use blanks to separate, if all are equal use a sign [=])?\n");
  for (;;)
    {
      FGETS (input, LINESIZE, stdin);
      if (input[0] == '=')
	{
	  scanned = 4;
	  *freqa = *freqc = *freqg = *freqt = 0.25;
	}
      else
	scanned =
	  sscanf (input, "%lf%lf%lf%lf%*[^\n]", freqa, freqc, freqg, freqt);
      if (scanned == 4)
	break;
      else
	printf ("Please enter exactly 4 values.\n");
    };
  // adjust frequencies to a total of 1
  summ = *freqa + *freqc + *freqg + *freqt;
  if (summ != 1.0)
    printf ("Frequency values were adjusted to add up to 1.0.\n");
  *freqa /= summ;
  *freqc /= summ;
  *freqg /= summ;
  *freqt /= summ;
  printf ("Nucleotide frequencies: A=%f, C=%f, G=%f, T=%f\n", *freqa, *freqc,
	  *freqg, *freqt);
}


void
initcatn (long *categs)
{				/* initialize category number */

  do
    {
      printf ("Number of categories (1-%d)?\n", MAXCATEGS);
      scanf ("%ld%*[^\n]", categs);
      getchar ();
    }
  while (*categs > MAXCATEGS || *categs < 1);
}


boolean
initcategs (long categs, double *rate, double *probcat)
{				/* initialize rate categories */
  long i;
  char input[LINESIZE];
  char rest[LINESIZE];
  int scanned;
  boolean done;

  for (;;)
    {
      printf
	("Either enter the Shape parameter alpha for Gamma deviated rates\n*OR* enter the rates for each category (use a space to separate)\n");
      FGETS (input, LINESIZE, stdin);
      done = TRUE;
      if (count_words (input) == 1)
	{
	  gamma_rates (rate, probcat, categs, input);
	  return TRUE;
	}
      for (i = 0; i < categs; i++)
	{
	  scanned = sscanf (input, "%lf %[^\n]", &rate[i], rest);
	  if ((scanned < 2 && i < (categs - 1))
	      || (scanned < 1 && i == (categs - 1)))
	    {
	      printf ("Please enter exactly %ld values.\n", categs);
	      done = FALSE;
	      break;
	    }
	  strcpy (input, rest);
	}
      if (done)
	break;
    }
  return FALSE;

}

void
initprobcat (long categs, double *probsum, double *probcat)
{
  long i;
  boolean done;
  char input[LINESIZE];
  char rest[LINESIZE];
  int scanned;

  do
    {
      printf ("Probability for each category?");
      printf (" (use a space to separate)\n");
      FGETS (input, LINESIZE, stdin);
      done = TRUE;
      for (i = 0; i < categs; i++)
	{
	  scanned = sscanf (input, "%lf %[^\n]", &probcat[i], rest);
	  if ((scanned < 2 && i < (categs - 1))
	      || (scanned < 1 && i == (categs - 1)))
	    {
	      done = FALSE;
	      printf ("Please enter exactly %ld values.\n", categs);
	      break;
	    }
	  strcpy (input, rest);
	}
      if (!done)
	continue;
      *probsum = 0.0;
      for (i = 0; i < categs; i++)
	*probsum += probcat[i];

      if (fabs (1.0 - (*probsum)) > 0.001)
	{
	  for (i = 0; i < categs; i++)
	    probcat[i] /= *probsum;
	  printf ("Probabilities were adjusted to add up to one\n");
	  for (i = 0; i < categs; i++)
	    printf ("  %li> %f\n", i + 1, probcat[i]);
	  printf ("\n\n");
	}
    }
  while (!done);
}


/*data read material ===================================== */

void
make_sequences (world_fmt * world, option_fmt * options, data_fmt * data,
		long locus)
{
  makevalues_seq (world, options, data, locus);
  if (options->freqsfrom)
    {
      empiricalfreqs (world, options, world->data->seq, locus);
      getbasefreqs (options, world->data->seq, locus);
    }
}

void
make_snp (world_fmt * world, option_fmt * options, data_fmt * data,
	  long locus)
{
  makevalues_snp (world, options, data, locus);
  /*if (options->freqsfrom)
     {
     empiricalfreqs (world, world->data->seq, locus); */
  options->freqsfrom = FALSE;
  getbasefreqs (options, world->data->seq, locus);
  /* }
   */
}

/* private functions================================== */
void
getbasefreqs (option_fmt * options, seqmodel_fmt * seq, long locus)
{
  long l;
  double aa, bb;
  if (locus == 0)
    seq->ttratio = options->ttratio[0];
  else
    {
      for (l = 1; l <= locus; l++)
	{
	  if (options->ttratio[l] == 0.0)
	    {
	      seq->ttratio = options->ttratio[l - 1];
	      break;
	    }
	  seq->ttratio = options->ttratio[l];
	}
      if (l > locus)
	seq->ttratio = options->ttratio[locus];
    }
  check_basefreq (options);

  seq->freqa = options->freqa;
  seq->freqc = options->freqc;
  seq->freqg = options->freqg;
  seq->freqt = options->freqt;
  seq->freqr = seq->freqa + seq->freqg;
  seq->freqy = seq->freqc + seq->freqt;
  seq->freqar = seq->freqa / seq->freqr;
  seq->freqcy = seq->freqc / seq->freqy;
  seq->freqgr = seq->freqg / seq->freqr;
  seq->freqty = seq->freqt / seq->freqy;
  aa =
    seq->ttratio * (seq->freqr) * (seq->freqy) - seq->freqa * seq->freqg -
    seq->freqc * seq->freqt;
  bb = seq->freqa * (seq->freqgr) + seq->freqc * (seq->freqty);
  seq->xi = aa / (aa + bb);
  seq->xv = 1.0 - seq->xi;
  if (seq->xi <= 0.0)
    {
      printf ("\n WARNING: This transition/transversion ratio\n");
      printf (" is impossible with these base frequencies!\n");
      seq->xi = 0.0;
      seq->xv = 1.0;
      seq->ttratio =
	(seq->freqa * seq->freqg +
	 seq->freqc * seq->freqt) / ((seq->freqr) * (seq->freqy));
      printf (" Transition/transversion parameter reset\n");
      printf ("  so transition/transversion ratio is %10.6f\n\n",
	      (seq->ttratio));
    }
  seq->fracchange =
    (seq->xi) * (2 * seq->freqa * (seq->freqgr) +
		 2 * seq->freqc * (seq->freqty)) + (seq->xv) * (1.0 -
								seq->freqa *
								seq->freqa -
								seq->freqc *
								seq->freqc -
								seq->freqg *
								seq->freqg -
								seq->freqt *
								seq->freqt);
}

/*===================================================*/

void
makeweights (world_fmt * world, data_fmt * data, long locus)
{
  /* make up weights vector to avoid duplicate computations */
  long i;
//  world->data->seq->endsite = 1;
  for (i = 0; i < world->data->seq->sites[locus]; i++)
    {
      world->data->seq->alias[i] = i + 1;
      world->data->seq->ally[i] = 0;
      world->data->seq->aliasweight[i] = world->data->seq->weight[i];
      world->data->seq->location[i] = 0;
    }
  sitesort2 (world, data, world->data->seq->sites[locus], locus);
  sitecombine2 (world, data, world->data->seq->sites[locus], locus);
  sitescrunch2 (world, world->data->seq->sites[locus], 1, 2, locus);
  for (i = 1; i <= world->data->seq->sites[locus]; i++)
    {
      if (world->data->seq->aliasweight[i - 1] > 0)
	world->data->seq->endsite = i;
    }
  for (i = 1; i <= world->data->seq->endsite; i++)
    {
      world->data->seq->location[world->data->seq->alias[i - 1] - 1] = i;
      world->data->seq->ally[world->data->seq->alias[i - 1] - 1] =
	world->data->seq->alias[i - 1];
    }
  init_sequences2 (world, world->data->seq, locus);
}				/* makeweights */

void
init_sequences2 (world_fmt * world, seqmodel_fmt * seq, long locus)
{
  if (locus == 0)
    world->contribution =
      (contribarr *) malloc (seq->endsite * sizeof (contribarr));
  else
    world->contribution =
      (contribarr *) realloc (world->contribution,
			      seq->endsite * sizeof (contribarr));
}


void
makevalues_seq (world_fmt * world, option_fmt * options, data_fmt * data,
		long locus)
{
  long i, ii, j, k, l, pop;
  long b;
  node **treenode = world->nodep;
  double seqerr = options->seqerror;
  double seqerr3 = seqerr / 3.;
  double seqerr23 = 2. * seqerr / 3.;
  for (k = 0; k < world->data->seq->endsite; k++)
    {
      j = world->data->seq->alias[k] - 1;
      i = -1;
      for (pop = 0; pop < world->numpop; pop++)
	{
	  for (ii = 0; ii < data->numind[pop][locus]; ii++)
	    {
	      i++;
	      if (!options->usertree)
		strcpy (treenode[i]->nayme, data->indnames[pop][ii]);
	      for (l = 0; l < world->options->rcategs; l++)
		{
		  for (b = 0; b < 4; b++)
		    treenode[i]->x.s[k][l][b] = 0.0 + seqerr3;
		  switch (data->yy[pop][ii][locus][0][j])
		    {
		    case 'A':
		      treenode[i]->x.s[k][l][0] = 1.0 - seqerr;
		      break;

		    case 'C':
		      treenode[i]->x.s[k][l][C] = 1.0 - seqerr;
		      break;

		    case 'G':
		      treenode[i]->x.s[k][l][G] = 1.0 - seqerr;
		      break;

		    case 'T':
		      treenode[i]->x.s[k][l][T] = 1.0 - seqerr;
		      break;

		    case 'U':
		      treenode[i]->x.s[k][l][T] = 1.0 - seqerr;
		      break;

		    case 'M':
		      treenode[i]->x.s[k][l][0] = 1.0 - seqerr;
		      treenode[i]->x.s[k][l][C] = 1.0 - seqerr;
		      break;

		    case 'R':
		      treenode[i]->x.s[k][l][0] = 1.0 - seqerr23;
		      treenode[i]->x.s[k][l][G] = 1.0 - seqerr23;
		      break;

		    case 'W':
		      treenode[i]->x.s[k][l][0] = 1.0 - seqerr23;
		      treenode[i]->x.s[k][l][T] = 1.0 - seqerr23;
		      break;

		    case 'S':
		      treenode[i]->x.s[k][l][C] = 1.0 - seqerr23;
		      treenode[i]->x.s[k][l][G] = 1.0 - seqerr23;
		      break;

		    case 'Y':
		      treenode[i]->x.s[k][l][C] = 1.0 - seqerr23;
		      treenode[i]->x.s[k][l][T] = 1.0 - seqerr23;
		      break;

		    case 'K':
		      treenode[i]->x.s[k][l][G] = 1.0 - seqerr23;
		      treenode[i]->x.s[k][l][T] = 1.0 - seqerr23;
		      break;

		    case 'B':
		      treenode[i]->x.s[k][l][0] = seqerr;
		      treenode[i]->x.s[k][l][C] = 1.0 - seqerr3;
		      treenode[i]->x.s[k][l][G] = 1.0 - seqerr3;
		      treenode[i]->x.s[k][l][T] = 1.0 - seqerr3;
		      break;

		    case 'D':
		      treenode[i]->x.s[k][l][0] = 1.0 - seqerr3;
		      treenode[i]->x.s[k][l][C] = seqerr;
		      treenode[i]->x.s[k][l][G] = 1.0 - seqerr3;
		      treenode[i]->x.s[k][l][T] = 1.0 - seqerr3;
		      break;

		    case 'H':
		      treenode[i]->x.s[k][l][0] = 1.0 - seqerr3;
		      treenode[i]->x.s[k][l][C] = 1.0 - seqerr3;
		      treenode[i]->x.s[k][l][G] = seqerr;
		      treenode[i]->x.s[k][l][T] = 1.0 - seqerr3;
		      break;

		    case 'V':
		      treenode[i]->x.s[k][l][0] = 1.0 - seqerr3;
		      treenode[i]->x.s[k][l][C] = 1.0 - seqerr3;
		      treenode[i]->x.s[k][l][G] = 1.0 - seqerr3;
		      treenode[i]->x.s[k][l][T] = seqerr;
		      break;

		    case 'N':
		      for (b = 0; b < 4; b++)
			treenode[i]->x.s[k][l][b] = 1.0;
		      break;

		    case 'X':
		      for (b = 0; b < 4; b++)
			treenode[i]->x.s[k][l][b] = 1.0;
		      break;

		    case '?':
		      for (b = 0; b < 4; b++)
			treenode[i]->x.s[k][l][b] = 1.0;
		      break;

		    case 'O':
		      for (b = 0; b < 4; b++)
			treenode[i]->x.s[k][l][b] = 1.0;
		      break;

		    case '-':
		      for (b = 0; b < 4; b++)
			treenode[i]->x.s[k][l][b] = 1.0;
		      break;
		    }
		}
	    }
	}
    }
}


void
makevalues_snp (world_fmt * world, option_fmt * options, data_fmt * data,
		long locus)
{
  long i, ii, j, k, l, pop;
  long b, f;
  node **treenode = world->nodep;
  f = world->data->seq->addon;
  for (k = 0; k < world->data->seq->endsite * (world->data->seq->addon + 1);
       k += (1 + world->data->seq->addon))
    {
      j = world->data->seq->alias[k / (world->data->seq->addon + 1)] - 1;
      i = -1;
      for (pop = 0; pop < world->numpop; pop++)
	{
	  for (ii = 0; ii < data->numind[pop][locus]; ii++)
	    {
	      i++;
	      if (!options->usertree)
		strcpy (treenode[i]->nayme, data->indnames[pop][ii]);
	      for (l = 0; l < world->options->rcategs; l++)
		{
		  if (pop == 0)
		    {
		      for (b = 0; b < 4; b++)
			{
			  // set panel data column to ?
			  treenode[i]->x.s[k][l][b] = 1.0;
			}
		      // reset all values for the snp columns
		      for (b = 1; b <= 4; b++)
			{
			  memset (treenode[i]->x.s[k + b][l], 0,
				  sizeof (double) * 4);
			  treenode[i]->x.s[k + b][l][b - 1] = 1.0;
			}
		      continue;
		    }
		  else
		    {
		      for (b = 0; b <= 4; b++)
			memset (treenode[i]->x.s[k + b][l], 0,
				sizeof (double) * 4);
		    }

		  switch (data->yy[pop][ii][locus][0][j])
		    {
		    case 'A':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      treenode[i]->x.s[k + 1][l][0] = 1.0;
		      treenode[i]->x.s[k + 2][l][0] = 1.0;
		      treenode[i]->x.s[k + 3][l][0] = 1.0;
		      treenode[i]->x.s[k + 4][l][0] = 1.0;
		      break;

		    case 'C':
		      treenode[i]->x.s[k][l][C] = 1.0;
		      treenode[i]->x.s[k + 1][l][C] = 1.0;
		      treenode[i]->x.s[k + 2][l][C] = 1.0;
		      treenode[i]->x.s[k + 3][l][C] = 1.0;
		      treenode[i]->x.s[k + 4][l][C] = 1.0;
		      break;

		    case 'G':
		      treenode[i]->x.s[k][l][G] = 1.0;
		      treenode[i]->x.s[k + 1][l][G] = 1.0;
		      treenode[i]->x.s[k + 2][l][G] = 1.0;
		      treenode[i]->x.s[k + 3][l][G] = 1.0;
		      treenode[i]->x.s[k + 4][l][G] = 1.0;
		      break;
		    case 'T':
		    case 'U':
		      treenode[i]->x.s[k][l][T] = 1.0;
		      treenode[i]->x.s[k + 1][l][T] = 1.0;
		      treenode[i]->x.s[k + 2][l][T] = 1.0;
		      treenode[i]->x.s[k + 3][l][T] = 1.0;
		      treenode[i]->x.s[k + 4][l][T] = 1.0;
		      break;

		    case 'M':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      treenode[i]->x.s[k][l][C] = 1.0;
		      treenode[i]->x.s[k + 1][l][0] = 1.0;
		      treenode[i]->x.s[k + 2][l][0] = 1.0;
		      treenode[i]->x.s[k + 3][l][0] = 1.0;
		      treenode[i]->x.s[k + 4][l][0] = 1.0;
		      treenode[i]->x.s[k + 1][l][C] = 1.0;
		      treenode[i]->x.s[k + 2][l][C] = 1.0;
		      treenode[i]->x.s[k + 3][l][C] = 1.0;
		      treenode[i]->x.s[k + 4][l][C] = 1.0;
		      break;

		    case 'R':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      treenode[i]->x.s[k][l][G] = 1.0;
		      treenode[i]->x.s[k + 1][l][0] = 1.0;
		      treenode[i]->x.s[k + 2][l][0] = 1.0;
		      treenode[i]->x.s[k + 3][l][0] = 1.0;
		      treenode[i]->x.s[k + 4][l][0] = 1.0;
		      treenode[i]->x.s[k + 1][l][G] = 1.0;
		      treenode[i]->x.s[k + 2][l][G] = 1.0;
		      treenode[i]->x.s[k + 3][l][G] = 1.0;
		      treenode[i]->x.s[k + 4][l][G] = 1.0;
		      break;

		    case 'W':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      treenode[i]->x.s[k][l][T] = 1.0;
		      treenode[i]->x.s[k + 1][l][0] = 1.0;
		      treenode[i]->x.s[k + 2][l][0] = 1.0;
		      treenode[i]->x.s[k + 3][l][0] = 1.0;
		      treenode[i]->x.s[k + 4][l][0] = 1.0;
		      treenode[i]->x.s[k + 1][l][T] = 1.0;
		      treenode[i]->x.s[k + 2][l][T] = 1.0;
		      treenode[i]->x.s[k + 3][l][T] = 1.0;
		      treenode[i]->x.s[k + 4][l][T] = 1.0;
		      break;

		    case 'S':
		      treenode[i]->x.s[k][l][C] = 1.0;
		      treenode[i]->x.s[k][l][G] = 1.0;
		      treenode[i]->x.s[k + 1][l][C] = 1.0;
		      treenode[i]->x.s[k + 2][l][C] = 1.0;
		      treenode[i]->x.s[k + 3][l][C] = 1.0;
		      treenode[i]->x.s[k + 4][l][C] = 1.0;
		      treenode[i]->x.s[k + 1][l][G] = 1.0;
		      treenode[i]->x.s[k + 2][l][G] = 1.0;
		      treenode[i]->x.s[k + 3][l][G] = 1.0;
		      treenode[i]->x.s[k + 4][l][G] = 1.0;
		      break;

		    case 'Y':
		      treenode[i]->x.s[k][l][C] = 1.0;
		      treenode[i]->x.s[k][l][T] = 1.0;
		      treenode[i]->x.s[k + 1][l][C] = 1.0;
		      treenode[i]->x.s[k + 2][l][C] = 1.0;
		      treenode[i]->x.s[k + 3][l][C] = 1.0;
		      treenode[i]->x.s[k + 4][l][C] = 1.0;
		      treenode[i]->x.s[k + 1][l][T] = 1.0;
		      treenode[i]->x.s[k + 2][l][T] = 1.0;
		      treenode[i]->x.s[k + 3][l][T] = 1.0;
		      treenode[i]->x.s[k + 4][l][T] = 1.0;
		      break;

		    case 'K':
		      treenode[i]->x.s[k][l][G] = 1.0;
		      treenode[i]->x.s[k][l][T] = 1.0;
		      treenode[i]->x.s[k + 1][l][G] = 1.0;
		      treenode[i]->x.s[k + 2][l][G] = 1.0;
		      treenode[i]->x.s[k + 3][l][G] = 1.0;
		      treenode[i]->x.s[k + 4][l][G] = 1.0;
		      treenode[i]->x.s[k + 1][l][T] = 1.0;
		      treenode[i]->x.s[k + 2][l][T] = 1.0;
		      treenode[i]->x.s[k + 3][l][T] = 1.0;
		      treenode[i]->x.s[k + 4][l][T] = 1.0;
		      break;

		    case 'B':
		      treenode[i]->x.s[k][l][C] = 1.0;
		      treenode[i]->x.s[k][l][G] = 1.0;
		      treenode[i]->x.s[k][l][T] = 1.0;
		      treenode[i]->x.s[k + 1][l][C] = 1.0;
		      treenode[i]->x.s[k + 2][l][C] = 1.0;
		      treenode[i]->x.s[k + 3][l][C] = 1.0;
		      treenode[i]->x.s[k + 4][l][C] = 1.0;
		      treenode[i]->x.s[k + 1][l][G] = 1.0;
		      treenode[i]->x.s[k + 2][l][G] = 1.0;
		      treenode[i]->x.s[k + 3][l][G] = 1.0;
		      treenode[i]->x.s[k + 4][l][G] = 1.0;
		      treenode[i]->x.s[k + 1][l][T] = 1.0;
		      treenode[i]->x.s[k + 2][l][T] = 1.0;
		      treenode[i]->x.s[k + 3][l][T] = 1.0;
		      treenode[i]->x.s[k + 4][l][T] = 1.0;
		      break;

		    case 'D':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      treenode[i]->x.s[k][l][G] = 1.0;
		      treenode[i]->x.s[k][l][T] = 1.0;
		      treenode[i]->x.s[k + 1][l][0] = 1.0;
		      treenode[i]->x.s[k + 2][l][0] = 1.0;
		      treenode[i]->x.s[k + 3][l][0] = 1.0;
		      treenode[i]->x.s[k + 4][l][0] = 1.0;
		      treenode[i]->x.s[k + 1][l][G] = 1.0;
		      treenode[i]->x.s[k + 2][l][G] = 1.0;
		      treenode[i]->x.s[k + 3][l][G] = 1.0;
		      treenode[i]->x.s[k + 4][l][G] = 1.0;
		      treenode[i]->x.s[k + 1][l][T] = 1.0;
		      treenode[i]->x.s[k + 2][l][T] = 1.0;
		      treenode[i]->x.s[k + 3][l][T] = 1.0;
		      treenode[i]->x.s[k + 4][l][T] = 1.0;
		      break;

		    case 'H':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      treenode[i]->x.s[k][l][C] = 1.0;
		      treenode[i]->x.s[k][l][T] = 1.0;
		      treenode[i]->x.s[k + 1][l][0] = 1.0;
		      treenode[i]->x.s[k + 2][l][0] = 1.0;
		      treenode[i]->x.s[k + 3][l][0] = 1.0;
		      treenode[i]->x.s[k + 4][l][0] = 1.0;
		      treenode[i]->x.s[k + 1][l][C] = 1.0;
		      treenode[i]->x.s[k + 2][l][C] = 1.0;
		      treenode[i]->x.s[k + 3][l][C] = 1.0;
		      treenode[i]->x.s[k + 4][l][C] = 1.0;
		      treenode[i]->x.s[k + 1][l][T] = 1.0;
		      treenode[i]->x.s[k + 2][l][T] = 1.0;
		      treenode[i]->x.s[k + 3][l][T] = 1.0;
		      treenode[i]->x.s[k + 4][l][T] = 1.0;
		      break;

		    case 'V':
		      treenode[i]->x.s[k][l][0] = 1.0;
		      treenode[i]->x.s[k][l][C] = 1.0;
		      treenode[i]->x.s[k][l][G] = 1.0;
		      treenode[i]->x.s[k + 1][l][0] = 1.0;
		      treenode[i]->x.s[k + 2][l][0] = 1.0;
		      treenode[i]->x.s[k + 3][l][0] = 1.0;
		      treenode[i]->x.s[k + 4][l][0] = 1.0;
		      treenode[i]->x.s[k + 1][l][C] = 1.0;
		      treenode[i]->x.s[k + 2][l][C] = 1.0;
		      treenode[i]->x.s[k + 3][l][C] = 1.0;
		      treenode[i]->x.s[k + 4][l][C] = 1.0;
		      treenode[i]->x.s[k + 1][l][G] = 1.0;
		      treenode[i]->x.s[k + 2][l][G] = 1.0;
		      treenode[i]->x.s[k + 3][l][G] = 1.0;
		      treenode[i]->x.s[k + 4][l][G] = 1.0;
		      break;

		    case 'N':
		    case 'X':
		    case '-':
		    case 'O':
		    case '?':
		      for (b = 0; b < 4; b++)
			{
			  treenode[i]->x.s[k][l][b] = 1.0;
			  treenode[i]->x.s[k + 1][l][b] = 1.0;
			  treenode[i]->x.s[k + 2][l][b] = 1.0;
			  treenode[i]->x.s[k + 3][l][b] = 1.0;
			  treenode[i]->x.s[k + 4][l][b] = 1.0;
			}
		      break;
		    default:
		      break;

		    }
		}
	    }
	}
    }
}


void
make_invarsites (world_fmt * world, data_fmt * data, long locus)
{
  long i, k, z, ii, pop, l;
  long endsite = world->data->seq->endsite;
  node **treenode = world->nodep;
  i = -1;
  for (pop = 0; pop < world->numpop; pop++)
    {
      for (ii = 0; ii < data->numind[pop][locus]; ii++)
	{
	  i++;
	  z = 0;
	  for (k = endsite; k < endsite + 4; k++)
	    {
	      for (l = 0; l < world->options->rcategs; l++)
		{
		  memset (treenode[i]->x.s[k][l], 0, sizeof (double) * 4);
		  treenode[i]->x.s[k][l][z] = 1.0;
		}
	      z++;
	    }
	}
    }
}

void
make_invarsites_unlinked (world_fmt * world, data_fmt * data, long locus)
{
  long i, k, z, ii, pop, l;
  long endsite = world->data->seq->endsite;
  node **treenode = world->nodep;
  i = -1;
  for (pop = 0; pop < world->numpop; pop++)
    {
      for (ii = 0; ii < data->numind[pop][locus]; ii++)
	{
	  i++;
	  z = 0;
	  if (pop == PANEL)
	    {
	      for (k = endsite * 5; k < endsite * 5 + 4; k++)
		{
		  for (l = 0; l < world->options->rcategs; l++)
		    {
		      memset (treenode[i]->x.s[k][l], 0, sizeof (double) * 4);
		      treenode[i]->x.s[k][l][z] = 1.0;
		    }
		  z++;
		}
	    }
	  else
	    {
	      for (k = endsite * 5; k < endsite * 5 + 4; k++)
		{
		  for (l = 0; l < world->options->rcategs; l++)
		    {
		      treenode[i]->x.s[k][l][0] = 1.0;	//putting ? for data
		      treenode[i]->x.s[k][l][1] = 1.0;
		      treenode[i]->x.s[k][l][2] = 1.0;
		      treenode[i]->x.s[k][l][3] = 1.0;
		    }
		}
	    }
	}
    }
}

void
sitesort2 (world_fmt * world, data_fmt * data, long sites, long locus)
{
  long gap, i, j, jj, jg, k, kk, itemp, pop, z = 0;
  boolean flip, tied, samewt;
  seqmodel_fmt *seq;
  long *tempsum, *temppop;
  tempsum = (long *) calloc (1, sizeof (long) * world->numpop);
  temppop = (long *) calloc (1, sizeof (long) * world->numpop);
  for (i = 0; i < world->numpop; i++)
    {
      if (data->numind[i][FLOC] > 0)
	{
	  temppop[z] = i;
	  if (z == 0)
	    tempsum[z] = data->numind[i][FLOC];
	  else
	    tempsum[z] = tempsum[z - 1] + data->numind[i][FLOC];
	  z++;
	}
    }
  seq = world->data->seq;
  gap = sites / 2;
  while (gap > 0)
    {
      for (i = gap + 1; i <= sites; i++)
	{
	  j = i - gap;
	  flip = TRUE;
	  while (j > 0 && flip)
	    {
	      jj = seq->alias[j - 1];
	      jg = seq->alias[j + gap - 1];
	      samewt = ((seq->weight[jj - 1] != 0)
			&& (seq->weight[jg - 1] != 0))
		|| ((seq->weight[jj - 1] == 0) && (seq->weight[jg - 1] == 0));
	      tied = samewt
		&& (seq->category[jj - 1] == seq->category[jg - 1]);
	      flip = ((!samewt) && (seq->weight[jj - 1] == 0)) || (samewt
								   &&
								   (seq->
								    category
								    [jj - 1] >
								    seq->
								    category
								    [jg -
								     1]));
	      k = 0;
	      pop = 0;
	      kk = -1;
	      while (k < world->sumtips && tied)
		{
		  if (k == tempsum[pop])
		    {
		      kk = 0;
		      pop++;
		    }
		  else
		    {
		      kk++;
		    }
		  flip =
		    (data->yy[temppop[pop]][kk][locus][0][jj - 1] >
		     data->yy[temppop[pop]][kk][locus][0][jg - 1]);
		  tied = (tied
			  && data->yy[temppop[pop]][kk][locus][0][jj - 1] ==
			  data->yy[temppop[pop]][kk][locus][0][jg - 1]);
		  k++;
		}
	      if (!flip)
		break;
	      itemp = seq->alias[j - 1];
	      seq->alias[j - 1] = seq->alias[j + gap - 1];
	      seq->alias[j + gap - 1] = itemp;
	      itemp = seq->aliasweight[j - 1];
	      seq->aliasweight[j - 1] = seq->aliasweight[j + gap - 1];
	      seq->aliasweight[j + gap - 1] = itemp;
	      j -= gap;
	    }
	}
      gap /= 2;
    }
  free (tempsum);
  free (temppop);
}				/* sitesort2 */


void
sitecombine2 (world_fmt * world, data_fmt * data, long sites, long locus)
{
  long i, j, k, kk, pop, z = 0;
  boolean tied, samewt;
  seqmodel_fmt *seq;
  long *tempsum, *temppop;
  tempsum = (long *) calloc (1, sizeof (long) * world->numpop);
  temppop = (long *) calloc (1, sizeof (long) * world->numpop);
  tempsum[0] = data->numind[0][FLOC];
  for (i = 0; i < world->numpop; i++)
    {
      if (data->numind[i][FLOC] > 0)
	{
	  temppop[z] = i;
	  if (z == 0)
	    tempsum[z] = data->numind[i][FLOC];
	  else
	    tempsum[z] = tempsum[z - 1] + data->numind[i][FLOC];
	  z++;
	}
    }

  seq = world->data->seq;
  i = 1;
  while (i < sites)
    {
      j = i + 1;
      tied = TRUE;
      while (j <= sites && tied)
	{
	  samewt = ((seq->aliasweight[i - 1] != 0)
		    && (seq->aliasweight[j - 1] != 0))
	    || ((seq->aliasweight[i - 1] == 0)
		&& (seq->aliasweight[j - 1] == 0));
	  tied = samewt
	    && (seq->category[seq->alias[i - 1] - 1] ==
		seq->category[seq->alias[j - 1] - 1]);
	  k = 0;
	  pop = 0;
	  kk = -1;
	  while (k < world->sumtips && tied)
	    {
	      if (k == tempsum[pop])
		{
		  kk = 0;
		  pop++;
		}
	      else
		{
		  kk++;
		}
	      tied = (tied
		      && data->yy[temppop[pop]][kk][locus][0][seq->
							      alias[i - 1] -
							      1] ==
		      data->yy[temppop[pop]][kk][locus][0][seq->alias[j - 1] -
							   1]);
	      k++;
	    }
	  if (!tied)
	    break;
	  seq->aliasweight[i - 1] += seq->aliasweight[j - 1];
	  seq->aliasweight[j - 1] = 0;
	  seq->ally[seq->alias[j - 1] - 1] = seq->alias[i - 1];
	  j++;
	}
      i = j;
    }
  free (temppop);
  free (tempsum);
}				/* sitecombine2 */


void
sitescrunch2 (world_fmt * world, long sites, long i, long j, long locus)
{
  /* move so positively weighted sites come first */
  /* used by dnainvar, dnaml, dnamlk, & restml */
  long itemp;
  boolean done, found;
  seqmodel_fmt *seq;
  seq = world->data->seq;
  done = FALSE;
  while (!done)
    {
      found = FALSE;
      if (seq->aliasweight[i - 1] > 0)
	i++;
      else
	{
	  if (j <= i)
	    j = i + 1;
	  if (j <= sites)
	    {
	      found = FALSE;
	      do
		{
		  found = (seq->aliasweight[j - 1] > 0);
		  j++;
		}
	      while (!(found || j > sites));
	      if (found)
		{
		  j--;
		  itemp = seq->alias[i - 1];
		  seq->alias[i - 1] = seq->alias[j - 1];
		  seq->alias[j - 1] = itemp;
		  itemp = seq->aliasweight[i - 1];
		  seq->aliasweight[i - 1] = seq->aliasweight[j - 1];
		  seq->aliasweight[j - 1] = itemp;
		}
	      else
		done = TRUE;
	    }
	  else
	    done = TRUE;
	}
      done = (done || i >= sites);
    }
}				/* sitescrunch2 */

void
inputoptions (world_fmt * world, option_fmt * options, data_fmt * data,
	      long locus)
{
  long i;
  long sites = world->data->seq->sites[locus];
  for (i = 0; i < sites; i++)
    world->data->seq->category[i] = 1;
  for (i = 0; i < sites; i++)
    world->data->seq->weight[i] = 1;
  if (options->weights)
    inputweights (world, data, sites);
  world->data->seq->weightsum = 0;
  for (i = 0; i < sites; i++)
    world->data->seq->weightsum += world->data->seq->weight[i];
  if (world->options->categs > 1)
    {
      inputcategs (0, sites, world, options, data);
    }
}				/* inputoptions */

void
inputweights (world_fmt * world, data_fmt * data, long chars)
{
  /* input the character weights, 0-9 and A-Z for weights 0 - 35 */
  char ch;
  long i;

  for (i = 0; i < chars; i++)
    {
      do
	{
	  ch = getc (data->weightfile);
	  if (ch == '\n')
	    ch = ' ';
	}
      while (ch == ' ');
      world->data->seq->weight[i] = 1;
      if (isdigit ((int) ch))
	world->data->seq->weight[i] = ch - '0';
      else if (isalpha ((int) ch))
	{
	  ch = uppercase (ch);
	  world->data->seq->weight[i] = (short) (ch - 'A' + 10);
	}
      else
	{
	  printf ("BAD WEIGHT CHARACTER: %c\n", ch);
	  exit (EXIT_FAILURE);
	}
    }
}				/* inputweights */

void
inputcategs (long a, long b, world_fmt * world, option_fmt * options,
	     data_fmt * data)
{
  /* input the categories, 1-9 */
  char ch;
  long i;
  ch = getc (data->catfile);
  while (ch == '#')
    {
      while (ch != '\n')
	ch = getc (data->catfile);
      ch = getc (data->catfile);
    }
  ungetc (ch, data->catfile);
  fscanf (data->catfile, "%ld", &options->categs);
  options->rate =
    (double *) realloc (options->rate, sizeof (double) * options->categs);
  for (i = 0; i < options->categs; i++)
    {
      fscanf (data->catfile, "%lf", &options->rate[i]);
    }

  for (i = a; i < b; i++)
    {
      do
	{
	  ch = getc (data->catfile);
	  if (ch == '\n')
	    ch = ' ';
	}
      while (ch == ' ');
      if ((ch >= '1') && (ch <= ('0' + world->options->categs)))
	world->data->seq->category[i] = ch - '0';
      else
	{
	  printf
	    ("BAD CATEGORY CHARACTER: %c -- CATEGORIES ARE CURRENTLY 1-%ld\n",
	     ch, world->options->categs);
	  exit (EXIT_FAILURE);
	}
    }
}				/* inputcategs */




void
empiricalfreqs (world_fmt * world, option_fmt * options, seqmodel_fmt * seq,
		long locus)
{
  /* Get empirical base frequencies from the data */
  long i, j, k;
  double summ, suma, sumc, sumg, sumt, w;
  long snps = world->options->datatype == 'u' ? 5 : 1;
  options->freqa = 0.25;
  options->freqc = 0.25;
  options->freqg = 0.25;
  options->freqt = 0.25;
  for (k = 1; k <= 8; k++)
    {
      suma = 0.0;
      sumc = 0.0;
      sumg = 0.0;
      sumt = 0.0;
      for (i = 0; i < world->sumtips; i++)
	{
	  for (j = 0; j < seq->endsite * snps; j += snps)
	    {
	      w = (double) seq->aliasweight[j / snps];
	      summ = (options->freqa) * world->nodep[i]->x.s[j][0][0];
	      summ += (options->freqc) * world->nodep[i]->x.s[j][0][C];
	      summ += (options->freqg) * world->nodep[i]->x.s[j][0][G];
	      summ += (options->freqt) * world->nodep[i]->x.s[j][0][T];
	      suma +=
		w * (options->freqa) * world->nodep[i]->x.s[j][0][0] / summ;
	      sumc +=
		w * (options->freqc) * world->nodep[i]->x.s[j][0][C] / summ;
	      sumg +=
		w * (options->freqg) * world->nodep[i]->x.s[j][0][G] / summ;
	      sumt +=
		w * (options->freqt) * world->nodep[i]->x.s[j][0][T] / summ;
	    }
	}
      summ = suma + sumc + sumg + sumt;
      options->freqa = suma / summ;
      options->freqc = sumc / summ;
      options->freqg = sumg / summ;
      options->freqt = sumt / summ;
    }
}				/* empiricalfreqs */


void
initlambda (option_fmt * options)
{
  while (options->lambda <= 1.0)
    {
      printf
	("Mean block length of sites having the same rate\n (needs to be greater than 1)?\n");
      scanf ("%lf%*[^\n]", &options->lambda);
      getchar ();
    }
  options->lambda = 1.0 / options->lambda;
}



void
init_tbl (world_fmt * world, long locus)
{
  /* Define a lookup table. Precompute values and print them out in tables */
  long i, j;
  double sumrates;
  long categs = world->options->categs;
  long rcategs = world->options->rcategs;
  world->tbl = (valrec ***) malloc (rcategs * sizeof (valrec **));
  for (i = 0; i < rcategs; i++)
    {
      world->tbl[i] = (valrec **) malloc (categs * sizeof (valrec *));
      for (j = 0; j < categs; j++)
	world->tbl[i][j] = (valrec *) malloc (sizeof (valrec));
    }
  for (i = 0; i < rcategs; i++)
    {
      for (j = 0; j < categs; j++)
	{
	  world->tbl[i][j]->rat =
	    world->options->rrate[i] * world->options->rate[j];
	  world->tbl[i][j]->ratxi =
	    world->tbl[i][j]->rat * world->data->seq->xi;
	  world->tbl[i][j]->ratxv =
	    world->tbl[i][j]->rat * world->data->seq->xv;
	}
    }
  sumrates = 0.0;
  for (i = 0; i < world->data->seq->endsite; i++)
    {
      for (j = 0; j < rcategs; j++)
	sumrates +=
	  world->data->seq->aliasweight[i] * world->options->probcat[j] *
	  world->tbl[j][world->data->seq->
			category[world->data->seq->alias[i] - 1] - 1]->rat;
    }
  sumrates /= (double) world->data->seq->sites[locus];
  for (i = 0; i < rcategs; i++)
    for (j = 0; j < categs; j++)
      {
	world->tbl[i][j]->rat /= sumrates;
	world->tbl[i][j]->ratxi /= sumrates;
	world->tbl[i][j]->ratxv /= sumrates;
      }
}				/* inittable */

void
print_weights (FILE * outfile, world_fmt * world, option_fmt * options,
	       long locus)
{
  if (options->weights)
    {
      if ((options->printdata) || (options->progress && outfile == stdout))
	{
	  printweights (outfile, world, options, 0,
			world->data->seq->sites[locus],
			world->data->seq->weight, "Sites");
	}
    }
}

void
print_tbl (FILE * outfile, world_fmt * world, option_fmt * options,
	   long locus)
{
  long i;

  option_fmt *opt;


  opt = options;
  if (opt->rcategs > 1)
    {
      fprintf (outfile, "\nRegion type     Rate of change    Probability\n");
      fprintf (outfile, "---------------------------------------------\n");
      for (i = 0; i < opt->rcategs; i++)
	fprintf (outfile, "%9ld%16.3f%17.3f\n", i + 1, opt->rrate[i],
		 opt->probcat[i]);
      putc ('\n', outfile);
      if (opt->autocorr)
	fprintf (outfile,
		 "Expected length of a patch of sites having the same rate = %8.3f\n",
		 1. / opt->lambda);
      putc ('\n', outfile);
    }
  if (opt->categs > 1)
    {
      fprintf (outfile, "Site category   Rate of change\n");
      fprintf (outfile, "------------------------------\n");
      for (i = 0; i < opt->categs; i++)
	fprintf (outfile, "%9ld%16.3f\n", i + 1, opt->rate[i]);
    }
  if ((opt->rcategs > 1) || (opt->categs > 1))
    fprintf (outfile, "\n");
}


void
printweights (FILE * outfile, world_fmt * world, option_fmt * options,
	      short inc, long chars, short *weight, char *letters)
{
  /* print out the weights of sites */
  long i, j;
  fprintf (outfile, "\n    %s are weighted as follows:\n", letters);
  for (i = 0; i < chars; i++)
    {
      if (i % 60 == 0)
	{
	  putc ('\n', outfile);
	  for (j = 1; j <= options->nmlength + 3; j++)
	    putc (' ', outfile);
	}
      fprintf (outfile, "%hd", weight[i + inc]);
      if ((i + 1) % 10 == 0 && (i + 1) % 60 != 0)
	putc (' ', outfile);
    }
  fprintf (outfile, "\n\n");
}				/* printweights */



void
print_seqfreqs (FILE * outfile, world_fmt * world, option_fmt * options)
{
  if (world->locus == 0 || outfile == stdout)
    {
      if (options->freqsfrom)
	fprintf (outfile, "\nEmpirical ");
      fprintf (outfile, "Base Frequencies\n");
      fprintf (outfile,
	       "------------------------------------------------------------\n");
      fprintf (outfile,
	       "Locus     Nucleotide                        Transition/\n");
      fprintf (outfile,
	       "          ------------------------------  Transversion ratio\n");
      fprintf (outfile, "          A       C       G       T(U)\n");
      fprintf (outfile,
	       "------------------------------------------------------------\n");
    }
  fprintf (outfile, "%4li      %6.4f  %6.4f  %6.4f  %6.4f    %10.5f\n",
	   world->locus + 1, options->freqa, options->freqc, options->freqg,
	   options->freqt, world->data->seq->ttratio);
  if (outfile == stdout)
    fprintf (outfile, "\n");

}

double
treelike_seq (world_fmt * world, long locus)
{
  contribarr tterm;
  contribarr like;
  contribarr nulike;
  contribarr clai;
  //  long size = sizeof(double) * world->options->rcategs;
  double summ, sum2, sumc, sumterm, lterm;
  long i, j, k, lai;
  double scale;
  node *p;
  sitelike *x1;
  worldoption_fmt *opt;
  seqmodel_fmt *seq;
  opt = world->options;
  seq = world->data->seq;
  p = crawlback (world->root->next);
  summ = 0.0;

  if (opt->rcategs == 1)
    {
      for (i = 0; i < seq->endsite; i++)
	{
	  x1 = &(p->x.s[i][0]);
	  scale = p->scale[i];
	  tterm[0] =
	    seq->freqa * (*x1)[0] + seq->freqc * (*x1)[1] +
	    seq->freqg * (*x1)[2] + seq->freqt * (*x1)[3];
	  summ += seq->aliasweight[i] * (log (tterm[0]) + scale);
	}
    }
  else
    {
      for (i = 0; i < seq->endsite; i++)
	{
	  scale = p->scale[i];
	  //k = seq->category[seq->alias[i] - 1] - 1;
	  for (j = 0; j < opt->rcategs; j++)
	    {
	      x1 = &(p->x.s[i][j]);
	      tterm[j] =
		seq->freqa * (*x1)[0] + seq->freqc * (*x1)[1] +
		seq->freqg * (*x1)[2] + seq->freqt * (*x1)[3];
	    }
	  sumterm = 0.0;
	  for (j = 0; j < opt->rcategs; j++)
	    sumterm += opt->probcat[j] * tterm[j];
	  lterm = log (sumterm) + scale;
	  for (j = 0; j < opt->rcategs; j++)
	    clai[j] = tterm[j] / sumterm;
	  swap (clai, world->contribution[i]);
	  //      memcpy (world->contribution[i], clai, size);
	  summ += seq->aliasweight[i] * lterm;
	}
      for (j = 0; j < opt->rcategs; j++)
	like[j] = 1.0;
      for (i = 0; i < seq->sites[locus]; i++)
	{
	  sumc = 0.0;
	  for (k = 0; k < opt->rcategs; k++)
	    sumc += opt->probcat[k] * like[k];
	  sumc *= opt->lambda;
	  if ((seq->ally[i] > 0) && (seq->location[seq->ally[i] - 1] > 0))
	    {
	      lai = seq->location[seq->ally[i] - 1];
	      swap (world->contribution[lai - 1], clai);
	      //memcpy (clai, world->contribution[lai - 1], size);
	      for (j = 0; j < opt->rcategs; j++)
		nulike[j] = ((1.0 - opt->lambda) * like[j] + sumc) * clai[j];
	    }
	  else
	    {
	      for (j = 0; j < opt->rcategs; j++)
		nulike[j] = ((1.0 - opt->lambda) * like[j] + sumc);
	    }
	  swap (nulike, like);
	  //memcpy (like, nulike, size);
	}
      sum2 = 0.0;
      for (i = 0; i < opt->rcategs; i++)
	sum2 += opt->probcat[i] * like[i];
      summ += log (sum2);
    }
  return summ;
}				/* treelikelihood */

void
snp_invariants (contribarr invariants, long endsite, long rcategs,
		seqmodel_fmt * seq, phenotype x1)
{
  long i, j;
  memset (invariants, 0, sizeof (contribarr));
  for (i = endsite - seq->addon; i < endsite - seq->addon + 4; i++)
    {
      for (j = 0; j < rcategs; j++)
	{
	  invariants[j] +=
	    (seq->freqa * x1[i][j][0] + seq->freqc * x1[i][j][1] +
	     seq->freqg * x1[i][j][2] + seq->freqt * x1[i][j][3]);
	}
    }
  for (j = 0; j < rcategs; j++)
    invariants[j] = 1. - invariants[j];
}

double
treelike_snp (world_fmt * world, long locus)
{
  worldoption_fmt *opt;
  seqmodel_fmt *seq;
  double scale;
  contribarr tterm, invariants, like, nulike;
  //  long size = sizeof(double) * world->options->rcategs;
  double summ, sum2, sumc, sumterm, lterm;
  long i, j, k, lai;

  node *p;
  sitelike *x1;
  opt = world->options;
  seq = world->data->seq;
  p = crawlback (world->root->next);
  summ = 0.0;
  /* snp invariants */
  snp_invariants (invariants, seq->endsite, opt->rcategs, seq, p->x.s);
  for (i = 0; i < seq->endsite - seq->addon; i++)
    {
      scale = p->scale[i];
      for (j = 0; j < opt->rcategs; j++)
	{
	  x1 = &(p->x.s[i][j]);
	  tterm[j] =
	    (seq->freqa * (*x1)[0] + seq->freqc * (*x1)[1] +
	     seq->freqg * (*x1)[2] + seq->freqt * (*x1)[3]) / invariants[j];
	}
      sumterm = 0.0;
      for (j = 0; j < opt->rcategs; j++)
	sumterm += opt->probcat[j] * tterm[j];
      lterm = log (sumterm) + scale;
      for (j = 0; j < opt->rcategs; j++)
	world->contribution[i][j] = tterm[j] / sumterm;

      summ += seq->aliasweight[i] * lterm;
    }				/* over endsite - 4[snp-invariants] */
  for (j = 0; j < opt->rcategs; j++)
    like[j] = 1.0;
  for (i = 0; i < seq->sites[locus]; i++)
    {
      sumc = 0.0;
      for (k = 0; k < opt->rcategs; k++)
	sumc += opt->probcat[k] * like[k];
      sumc *= opt->lambda;
      if ((seq->ally[i] > 0) && (seq->location[seq->ally[i] - 1] > 0))
	{
	  lai = seq->location[seq->ally[i] - 1];
	  for (j = 0; j < opt->rcategs; j++)
	    nulike[j] =
	      ((1.0 - opt->lambda) * like[j] +
	       sumc) * world->contribution[lai - 1][j];
	}
      else
	{
	  for (j = 0; j < opt->rcategs; j++)
	    nulike[j] = ((1.0 - opt->lambda) * like[j] + sumc);
	}
      swap (nulike, like);
      //memcpy (like, nulike, size);
    }
  sum2 = 0.0;
  for (i = 0; i < opt->rcategs; i++)
    sum2 += opt->probcat[i] * like[i];
  summ += log (sum2);
  return summ;
}				/* treelike_snp */


double
treelike_snp_unlinked (world_fmt * world, long locus)
{
  worldoption_fmt *opt;
  seqmodel_fmt *seq;
  double scale;
  contribarr tterm, invariants;
  double summ, datasum = 0, lterm, result = 0;
  long i, ii;

  node *p;
  sitelike *x1;
  opt = world->options;
  seq = world->data->seq;
  p = crawlback (world->root->next);
  summ = 0.0;
  /* snp invariants */
  snp_invariants (invariants, seq->endsite, opt->rcategs, seq, p->x.s);
  /* no rate categories used */
  for (i = 0; i < seq->endsite - seq->addon; i++)
    {
      ii = i / 5;
      x1 = &(p->x.s[i][0]);
      scale = p->scale[i];
      tterm[0] =
	(seq->freqa * (*x1)[0] + seq->freqc * (*x1)[1] +
	 seq->freqg * (*x1)[2] + seq->freqt * (*x1)[3]);
      if (i % 5 == 0)
	{
	  lterm = log (tterm[0]) + scale;
	  summ = 0;
	  datasum = seq->aliasweight[ii] * lterm;
	}
      else
	summ += pow (tterm[0], (double) seq->aliasweight[ii]);
      if (((i + 1) % 5) == 0 && i != 0)
	result +=
	  datasum + log ((1 - EXP (log (summ) - datasum)) / invariants[0]);
    }
  return result;
}				/* treelike_snp_unlinked */

void
check_basefreq (option_fmt * options)
{

  if (options->freqa == 0. || options->freqc == 0. || options->freqt == 0.
      || options->freqg == 0.)
    {
      options->freqa = 0.25;
      options->freqc = 0.25;
      options->freqg = 0.25;
      options->freqt = 0.25;
    }
}


void
copy_seq (world_fmt * original, world_fmt * kopie)
{
  long sites;
  seqmodel_fmt *kseq;
  seqmodel_fmt *oseq;
  kseq = kopie->data->seq;
  oseq = original->data->seq;
  sites = oseq->sites[original->locus];
  kseq->freqa = oseq->freqa;
  kseq->freqt = oseq->freqt;
  kseq->freqg = oseq->freqg;
  kseq->freqc = oseq->freqc;
  kseq->freqr = oseq->freqr;
  kseq->freqy = oseq->freqy;
  kseq->freqar = oseq->freqar;
  kseq->freqcy = oseq->freqcy;
  kseq->freqgr = oseq->freqgr;
  kseq->freqty = oseq->freqty;
  kseq->aa = oseq->aa;
  kseq->bb = oseq->bb;
  kseq->endsite = oseq->endsite;
  kseq->xi = oseq->xi;
  kseq->xv = oseq->xv;
  kseq->ttratio = oseq->ttratio;
  kseq->fracchange = oseq->fracchange;
  memcpy (kseq->sites, oseq->sites, sizeof (long) * original->loci);
  memcpy (kseq->alias, oseq->alias, sizeof (long) * sites);
  memcpy (kseq->ally, oseq->ally, sizeof (long) * sites);
  memcpy (kseq->category, oseq->category, sizeof (long) * sites);
  memcpy (kseq->weight, oseq->weight, sizeof (short) * sites);
  kseq->weightsum = oseq->weightsum;
  memcpy (kseq->aliasweight, oseq->aliasweight, sizeof (long) * sites);
  memcpy (kseq->location, oseq->location, sizeof (long) * sites);
  kseq->addon = oseq->addon;
}
