/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 P R O F I L E    L I K E L I H O O D    R O U T I N E S 

 Peter Beerli 1997, Seattle
 beerli@genetics.washington.edu
 
Copyright 2001 Peter Beerli and Joseph Felsenstein

$Id: profile.c,v 1.55 2001/09/07 23:56:13 beerli Exp $

-------------------------------------------------------*/

#include "migration.h"

#include "world.h"
#ifndef LAGUERRE
#include "derivatives.h"
#include "derivatives2.h"
#endif
#include "tools.h"
#include "broyden.h"
#include "combroyden.h"
#include "spline.h"
#include "joint-chains.h"
#include "migrate_mpi.h"

#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif
long find_profilelike (double testlike, long whichprob, double minparam,
		       double maxparam, double *param, double *llike,
		       long which, world_fmt * world, boolean QD, boolean QD2,
		       double *mlparam, double *normd, nr_fmt * nr);
void calc_profile_likelihood (char method, long which, double *likes,
			      double **param, world_fmt * world, long nn,
			      nr_fmt * nr);
void print_profile_likelihood (long which, world_fmt * world, long *gmaxptr);
void prepare_header (char *var, long which, world_fmt * world);
void print_profile_title (world_fmt * world);
void print_profile_table (char method, long which, char *var, double likes[],
			  double **param, world_fmt * world);
void print_profile_percentile (world_fmt * world);
void sprint_percentile_header (char * outfile, boolean first);
void allocate_profile_percentiles (world_fmt * world);
void destroy_profile_percentiles (world_fmt * world);
void print_menu_profile (long which, long nn);
double interpolate (double xl, double xc, double xh, double low,
		    double center, double high, double testlike);

long warp (long ii);

void prognose_profile_end (time_t starttime, long numpop2, long nn);


void profile_max_percentiles (long which, double *likes, double **param,
			      world_fmt * world, long nn, nr_fmt * nr);

void profile_max_spline (char method, long which, double *likes,
			 double **param, world_fmt * world, nr_fmt * nr,
			 long nn);

int calc_spline (double *param, double *like, long nn, long *constr,
		 double *diff, double *diff2, long *diagn, double *work,
		 long nwork);
void setup_spline (spline_fmt * spline);

void destroy_spline (spline_fmt * spline);

void prepare_spline_nodes (long n, long which, double **rawparam,
			   double *rawlikes, long *indeks, double *param,
			   double *likes);

void prepare_spline_first (double *param, double *likes);

void prepare_spline_last (long n, double *param, double *likes);

void calc_spline_diff (double *diff, double *param, double *likes, long n);

double recalc_profilelike (double testparam, double *param, double *like,
			   long nn, double *diff, double *diff2, double *work,
			   long nwork);


long set_df (long which, char *custm2, long numpop, long numpop2);


long set_profile_param (double *param, long which, long *allwhich,
			double xval, double *xvals, world_fmt * world);
void addinvar (double *param, long *fixed, long *allwhich, double *xvals,
	       long start, long len);
void sprint_nice_param (double parameter, double bottom, double top,
		       char * file);



#define GRIDSIZE 9
#define GRIDMIDDLE 5
#define GRID    {0.01,0.05,0.10,0.50,0.99,0.95,0.90,0.50,1.0}
#define SHOWGRID {0.005,0.025,0.05,0.25,0.995,0.975,0.95,0.75,0.50}
#define GRID2   {0.005,0.025,0.05,0.25,0.5, 0.75,0.95,0.975,0.995}
#define INDEX {0,1,2,3,8,7,6,5,4}
#define DEVIATE {0.02,0.10,0.20, 0.5, 50.,10., 5., 2., 1.}
//#define DEVIATE {0.002,0.010,0.020, 0.05, 5000.,1000., 500., 200., 1.}
#define ABSOLUTE {1e-100,1e100}

boolean
print_profile_likelihood_driver (long which, world_fmt * world, long *gmaxptr)
{
  static boolean onetheta = FALSE;
  static boolean onemig = FALSE;
  long i, j;
  //world->options->verbose = FALSE;

  if (which == world->numpop2)	//gamma deviated mutation rate
    print_profile_likelihood (which, world, gmaxptr);
  else
    {
      switch (world->options->custm2[which])
	{
	case 'C':
	case 'c':
	case '0':
	  return 0;
	case 'M':
	case 'm':
	  if (which < world->numpop && onetheta)
	    return 0;
	  if (which < world->numpop)
	    {
	      print_profile_likelihood (which, world, gmaxptr);
	      onetheta = TRUE;
	    }
	  if (which >= world->numpop && onemig)
	    return 0;
	  if (which >= world->numpop)
	    {
	      print_profile_likelihood (which, world, gmaxptr);
	      onemig = TRUE;
	    }
	  return 1;
	case 'S':
	case 's':
	  j = (which - world->numpop) % (world->numpop - 1);
	  i = (which - world->numpop) / (world->numpop - 1);
	  if (i <= j)
	    {
	      print_profile_likelihood (which, world, gmaxptr);
	      return 1;
	    }
	  return 0;
	default:
	  print_profile_likelihood (which, world, gmaxptr);
	}
    }
  return 1;
}

void
print_profile_likelihood (long which, world_fmt * world, long *gmaxptr)
{
  long i, ii;
  nr_fmt *nr;
  char var[LINESIZE];
  char method = world->options->profilemethod;
  double **param;
  double likes[GRIDSIZE];
  long elem = world->options->gamma ? world->numpop * world->numpop + 1 :
    world->numpop * world->numpop;
  nr = (nr_fmt *) calloc (1, sizeof (nr_fmt));
  create_nr (nr, world, *gmaxptr, which, world->loci,
	     world->repkind, world->rep);
  param = (double **) calloc (1, sizeof (double *) * GRIDSIZE);
  param[0] = (double *) calloc (1, sizeof (double) * GRIDSIZE * elem);
  for (i = 1; i < GRIDSIZE; i++)
    {
      param[i] = param[0] + i * elem;
    }
  if (world->options->progress)
    {
      print_menu_profile (which,
			  world->numpop2 + (long) world->options->gamma);
    }
  calc_profile_likelihood (method, which, likes, param, world, GRIDSIZE, nr);

  prepare_header (var, which, world);
  if (world->options->printprofile)
    {
	print_profile_table (method, which, var, likes, param, world);
    }
  if (world->options->printprofsummary)
    {
      strcpy (world->quantiles[which].name, var);
      for (ii = 0; ii < GRIDSIZE; ii++)
	{
	  i = warp (ii);
	  world->quantiles[which].param[i] = param[i][which];
	}
    }
  destroy_nr (nr, world);
  free (param[0]);
  free (param);
}

void
print_profile_table (char method, long which, char *var, double likes[],
		     double **param, world_fmt * world)
{
  const double probabilities[] = SHOWGRID;
  char star;
  char methodstring[LINESIZE];
  long i, ii, j, likei = 0;
  long elem =
    world->options->gamma ? world->numpop * world->numpop +
    1 : world->numpop * world->numpop;
	
//  FILE *outfile = world->outfile;
  char fp[LINESIZE * 10]; //large local string buffer
  char *buffer = world->buffer; // buffer for printing whole table
  char *temp, *var2, temp2[LINESIZE], likestr[LINESIZE], *paramstr;
  double likemax = -DBL_MAX;
  memset(buffer,0,sizeof(char)*strlen(buffer));
  var2 =
    (char *) calloc (1,
		     sizeof (char) * MAX (LINESIZE,
					  (long) (elem / 10. * LINESIZE)));
  paramstr =
    (char *) calloc (1,
		     sizeof (char) * MAX (LINESIZE,
					  (long) (elem / 10. * LINESIZE)));
  temp = var2;
  for (j = 0; j < elem; j++)
    {
      prepare_header (temp2, j, world);
      if (which == j)
	star = '*';
      else
	star = ' ';
      if (((j + 1) % 5) == 0)
	{
	  sprintf (temp, "\n                            %c%.7s%c     ", star,
		   temp2, star);
	  temp += 28;
	}
      else
	sprintf (temp, "%c%.7s%c    ", star, temp2, star);
      temp += 11;
    }
  switch (method)
    {
    case 'p':
      strcpy (methodstring,
	      "Parameters are evaluated at percentiles\nusing bisection method (slow, but exact).");
      break;
    case 's':
      strcpy (methodstring,
	      "Parameters are evaluated at percentiles\nusing cubic splines of profiled parameter\n(faster, but not so exact).");
      break;
    case 'd':
      strcpy (methodstring,
	      "Parameters are evaluated at pre-defined values\n");
      break;
    case 'q':
      strcpy (methodstring,
	      "Parameters are evaluated assuming complete independence\n");
      break;
    case 'f':
      strcpy (methodstring,
	      "Parameters are evaluated assuming complete independence\n and then once maximized at the found profiled parameter value");
      break;
    }
  sprintf (fp, "\n\nProfile likelihood for parameter %s\n", var);
  strcat(buffer,fp);
  sprintf (fp, "%s\n", methodstring);
  strcat(buffer,fp);
  sprint_line (fp, '-', 79, CONT);
  strcat(buffer,fp);
  if (method == 'd')
    sprintf (fp, "      Ln(L)     %7.7s     %s\n", var, var2);
  else
    sprintf (fp, "Per.  Ln(L)     %7.7s     %s\n", var, var2);
    strcat(buffer,fp);
  sprint_line (fp, '-', 79, START);
    strcat(buffer,fp);
  for (i = 0; i < GRIDSIZE; i++)
    {
      if (likemax < MIN (likes[i], DBL_MAX))
	{
	  likemax = likes[i];
	  likei = i;
	}
    }
  for (ii = 0; ii < GRIDSIZE; ii++)
    {
      i = warp (ii);
      if (likes[i] >= DBL_MAX - EPSILON)
	sprintf (likestr, "     -    ");
      else
	sprintf (likestr, "% 6.3f%c", likes[i], likei == i ? '*' : ' ');
      temp = paramstr;
      for (j = 0; j < elem; j++)
	{
	  if (((j + 1) % 5) == 0)
	    {
	      sprintf (temp, "\n                              ");
	      temp += 26;
	    }
	  if (param[i][j] <= 0)
	    sprintf (temp, "      -     ");
	  else
	    {
	      if (param[i][j] < 0.000001)
		sprintf (temp, " % 6.3e ", param[i][j]);
	      else
		sprintf (temp, "% 10.6f ", param[i][j]);
	    }

	  temp += 11;
	}
      if (param[i][which] <= 0)
	{
	  if (method == 'd')
	    sprintf (fp, "        -           -       %s\n", paramstr);
	  else
	    sprintf (fp, "%3.3f     -           -       %s\n",
		     probabilities[i], paramstr);
	  strcat(buffer,fp);
	}
      else
	{
	  if (method == 'd')
	    sprintf (fp, "    %8.8s % 10.6g %s\n", likestr,
		     param[i][which], paramstr);
	  else
	    {
	      if (probabilities[i] == 0.5)
		sprintf (fp, "MLE   %8.8s % 10.6g %s\n",
			 likestr, param[i][which], paramstr);
	      else
		sprintf (fp, "%4.3f %8.8s % 10.6g %s\n",
			 probabilities[i], likestr,
			 param[i][which], paramstr);
	    }
	  strcat(buffer,fp);
	}
    }
  sprint_line (fp, '-', 79, STOP);
    strcat(buffer,fp);
  sprintf (fp,
	   "- = not possible to evaluate, most likely value either 0.0 or Infinity\n   in the parameter direction, the likelihood surface is so flat\n    that the calculation of the percentile(s)  failed.\n");
	     strcat(buffer,fp);
  free (paramstr);
  free (var2);
}


void
print_profile_title (world_fmt * world)
{
  print_line (world->outfile, '=', 79, CONT);
  fprintf (world->outfile, "Profile likelihood tables\n");
  print_line (world->outfile, '=', 79, CONT);
}

long
warp (long ii)
{
  long i;
  if (ii < 4)
    i = ii;
  else
    {
      if (ii == 4)
	{
	  i = GRIDSIZE - 1;
	}
      else
	{
	  i = GRIDSIZE - ii + 3;
	}
    }
  return i;
}


void
print_profile_percentile (world_fmt * world)
{
  const double probabilities[] = SHOWGRID;	//debug GRID2
  long i, j, jj;
  boolean first = TRUE;
  boolean last = FALSE;
  char fp[LINESIZE];
  memset(world->buffer,0,sizeof(char)*strlen(world->buffer));
  sprintf (fp, "\n\n");
  strcat(world->buffer,fp);
  sprint_line (fp, '=', 79, CONT);
  strcat(world->buffer,fp);
  sprintf (fp,
	   "Summary of profile likelihood percentiles of all parameters\n");
  strcat(world->buffer,fp);	   
  sprint_line (fp, '=', 79, CONT);
  strcat(world->buffer,fp);
  sprintf(fp, "\n");
  strcat(world->buffer,fp);
  sprint_percentile_header (fp, first);
  strcat(world->buffer,fp);
  for (i = 0; i < world->numpop2 + (long) world->options->gamma; i++)
    {
      if (world->quantiles[i].name[0] == '\0')
	continue;		/* this variable was not estimated, so return */
      sprintf (fp, "%-10.10s  ", world->quantiles[i].name);
      strcat(world->buffer,fp);
      for (jj = 0; jj < GRIDSIZE; jj++)
	{
	  j = warp (jj);
	  if (probabilities[j] > 0.5)
	    continue;
	  if (world->quantiles[i].param[j] < 0)
	    sprintf (fp, "      -       ");
	  else
	    sprint_nice_param (world->quantiles[i].param[j], 0.000001,
			      999.99999, fp);
	    strcat(world->buffer,fp);
	}
      sprintf (fp, "\n");
      strcat(world->buffer,fp);
    }
  sprintf (fp, "\n\n");
  strcat(world->buffer,fp);
  sprint_percentile_header (fp, last);
  strcat(world->buffer,fp);
  for (i = 0; i < world->numpop2 + (long) world->options->gamma; i++)
    {
      if (world->quantiles[i].name[0] == '\0')
	continue;		/* this variable was not estimated, so return */
      sprintf (fp, "%-10.10s  ", world->quantiles[i].name);
      strcat(world->buffer,fp);
      for (jj = 0; jj < GRIDSIZE; jj++)
	{
	  j = warp (jj);
	  if (probabilities[j] < 0.5)
	    continue;
	  if (world->quantiles[i].param[j] < 0)
	    sprintf (fp, "      -       ");
	  else
	    sprint_nice_param (world->quantiles[i].param[j], 0.000001,
			      999.99999, fp);
	   strcat(world->buffer,fp);
	}
      sprintf (fp, "\n");
      strcat(world->buffer,fp);
    }

  sprintf (fp, "\n");
  strcat(world->buffer,fp);
  sprint_line (fp, '-', 80, CONT);
  strcat(world->buffer,fp);
  sprintf (fp,
	   "- = not possible to evaluate, most likely value either 0.0 or Infinity\n    in the parameter direction, the likelihood surface is so flat\n    that the percentiles cannot be calculated.\n");
    strcat(world->buffer,fp);
}

void
sprint_nice_param (double parameter, double bottom, double top, char * file)
{
  if (parameter > 0.000001 && parameter < 999.99999)
    sprintf (file, " %11.6f  ", parameter);
  else
    sprintf (file, " %11.6g ", parameter);
}

void
sprint_percentile_header (char * outfile, boolean first)
{
  const double probabilities[] = GRID2;
  long i;
  char fp[LINESIZE];
  if (first)
    sprintf (fp,
	     "Parameter                          Lower percentiles\n");
  else
    sprintf (fp,
	     "Parameter                          Upper percentiles\n");
  strcat(outfile,fp);		 
  sprintf (fp, "            ");
  strcat(outfile,fp);
  sprint_line (fp, '-', 68, CONT);
  strcat(outfile,fp);
  sprintf (fp, "            ");
  strcat(outfile,fp);
  for (i = 0; i < GRIDSIZE - 1; i++)
    {
      if (first)
	{
	  if (probabilities[i] >= 0.5)
	    break;
	}
      else
	{
	  if (probabilities[i] < 0.5)
	    continue;
	}
      if (probabilities[i] == 0.5)
	sprintf (fp, "    MLE       ");
      else
	sprintf (fp, "    %5.3f     ", probabilities[i]);
      strcat(outfile,fp);
    }
  if (probabilities[i] == 0.5)
    sprintf (fp, "    MLE\n");
  else
    sprintf (fp, "    %5.3f\n", probabilities[i]);
  strcat(outfile,fp);
  sprint_line (fp, '-', 80, CONT);
  strcat(outfile,fp);
}


void
prepare_header (char *var, long which, world_fmt * world)
{
  long i, j, from, to;
  if (which < world->numpop)
    {
      sprintf (var, "Theta_%li", from = which + 1);
    }
  else
    {
      if (world->numpop > 1)
	{
	  i = (which - world->numpop) / (long) (world->numpop - 1);
	  j = which - world->numpop - i * (world->numpop - 1);
	  to = i + 1;
	  from = (i <= j) ? j + 2 : j + 1;
	  if (world->options->profileparamtype == PLOT4NM)
	    sprintf (var, "4Nm_%li%li", from, to);
	  else
	    sprintf (var, "  M_%li%li", from, to);
	}
      if (which == world->numpop2)
	sprintf (var, " Alpha  ");
    }
}

void
calc_profile_likelihood (char method, long which, double *likes,
			 double **param, world_fmt * world, long nn,
			 nr_fmt * nr)
{
  long i, j;
  switch (method)
    {
    case 'p' /*percentiles */ :
    case 'q':			/*quick and dirty: no correlation between parametes */
    case 'f':			/*mixture of quick and precise */
      profile_max_percentiles (which, likes, param, world, nn, nr);
      break;
    case 's':			/* spline */
    case 'd' /*discrete */ :
      profile_max_spline (method, which, likes, param, world, nr, nn);
      break;
    }
  if (world->options->profileparamtype == PLOT4NM)
    {
      for (i = 0; i < GRIDSIZE; i++)
	{
	  for (j = world->numpop; j < world->numpop2; j++)
	    {
	      param[i][j] *=
		param[i][(long) (j - world->numpop) / (world->numpop - 1)];
	    }
	}
    }
  //else
  //  printf ("           Using M (m/mu) instead of 4Nm\n");
}


void
prognose_profile_end (time_t starttime, long numpop2, long nn)
{
#ifndef NOTIME_FUNC
  static long sumtime = 0;
  static long tally = 0;
  static char nowstr[LINESIZE];
  time_t endtime;
  time_t nowbin;
  struct tm *nowstruct;

  if (tally % nn == 0)
    {
      tally++;
      time (&endtime);
      sumtime = (endtime - starttime);
#ifdef MPI
      nowbin = starttime + (sumtime / tally * (nn * numpop2));
#else
      nowbin = starttime + (sumtime / tally * (nn * numpop2));
#endif
      nowstruct = localtime (&nowbin);

      strftime (nowstr, LINESIZE, "%R %B %e %Y", nowstruct);
      fprintf (stdout, "           Prognosed end of run: %s\n", nowstr);
    }
  else
    tally++;
#endif
}


void
profile_max_percentiles (long which, double *likes, double **param,
			 world_fmt * world, long nn, nr_fmt * nr)
{
  const double probabilities[] = GRID;

  long i, trials = 0;
  long df;
  boolean QD = FALSE, QD2 = FALSE;
  double prob, normd, maxlike, testlike, minparam = 0, maxparam = 0;
  double *mlparam;
  long rep =
    world->options->replicate ? (world->loci > 1 ? 0 : world->repstop) : 0;
  df =
    (world->options->df > 0) ? world->options->df : set_df (which,
							    world->options->
							    custm2,
							    world->numpop,
							    world->numpop2 +
							    (long) world->
							    options->gamma);

  /* QD assumes that there is NO correlation between parameters
     that is certainly wrong but sometimes close and much faster
   */
  if (world->options->profilemethod == 'q')
    QD = TRUE;
  else
    {
      if (world->options->profilemethod == 'f')
	{
	  QD = TRUE;
	  QD2 = TRUE;
	}
      else
	QD = FALSE;
    }
  /* the minimum */
  minparam = which < world->numpop ? SMALLEST_THETA : SMALLEST_MIGRATION;
  /* the maximum */
  if (world->loci > 1)
    {
      mlparam = world->atl[rep][world->loci].param;
      maxlike = world->atl[rep][world->loci].param_like;
    }
  else
    {
      mlparam = world->atl[rep][world->locus].param;
      maxlike = world->atl[rep][world->locus].param_like;
    }


  memcpy (param[nn - 1], mlparam,
	  sizeof (double) * (world->numpop2 + (long) world->options->gamma));
  likes[nn - 1] = maxlike;
  maxparam = param[nn - 1][which];

  for (i = 0; i < nn - 1; i++)
    {
      if (probabilities[i] > 0.5)
	prob = 1 - probabilities[i];
      else
	prob = probabilities[i];
      testlike = maxlike - 0.5 * find_chi (df, prob);
      if (i > 0)
	{
	  minparam = param[i - 1][which];
	  if (minparam < 0)
	    minparam = which < world->numpop ?
	      SMALLEST_THETA : SMALLEST_MIGRATION;
	}
#ifdef __MWERKS__
      eventloop ();
#endif
      memcpy (param[i], mlparam,
	      sizeof (double) * (world->numpop2 +
				 (long) world->options->gamma));
      trials =/* prob was testlike*/
	find_profilelike (prob, i, minparam, maxparam, param[i],
			  &likes[i], which, world, QD, QD2, mlparam, &normd,
			  nr);
      if (world->options->progress)
	prognose_profile_end (world->starttime, world->numpop2 +
			      (long) world->options->gamma, nn);
    }
}

void
profile_max_spline (char method, long which, double *likes, double **param,
		    world_fmt * world, nr_fmt * nr, long nn)
{

  double deviate[] = DEVIATE;
  long errc;
  long indeks[] = INDEX;
  const double probabilities[] = GRID;
  long i, trials = 0;
  long panic, df;
  double prob, normd, maxlike, testlike, minparam = 0, maxparam = 0;
  double *mlparam;
  double tmp, x, xx, xlow, xhigh, mid, low, high, value;
  long rep =
    world->options->replicate ? (world->loci > 1 ? 0 : world->repstop) : 0;
  spline_fmt *spline;

  /* set degree of freedom */
  df =
    (world->options->df > 0) ? world->options->df : set_df (which,
							    world->options->
							    custm2,
							    world->numpop,
							    world->numpop2 +
							    (long) world->
							    options->gamma);
  /* the minimum */
  minparam = which < world->numpop ? SMALLEST_THETA : SMALLEST_MIGRATION;
  /* the maximum */
  if (world->loci > 1)
    {
      mlparam = world->atl[rep][world->loci].param;
      maxlike = world->atl[rep][world->loci].param_like;
    }
  else
    {
      mlparam = world->atl[rep][world->locus].param;
      maxlike = world->atl[rep][world->locus].param_like;
    }
  memcpy (param[nn - 1], mlparam,
	  sizeof (double) * (world->numpop2 + (long) world->options->gamma));
  likes[nn - 1] = maxlike;
  maxparam = param[nn - 1][which];
  /* calculate likelihood at GRIDNODES */
  for (i = 0; i < nn - 1; i++)
    {
      if (which >= world->numpop && mlparam[which] < EPSILON)
	value = deviate[i];
      else
	value = deviate[i] * mlparam[which];
      nr->profilenum = set_profile_param (param[i], which, nr->profiles,
					  value, nr->values, world);
      //do_profiles(nr->world, nr, likes, &normd, PROFILE, 
      //          nr->world->rep, nr->world->repkind);
      maximize (param[i], world, nr, PROFILE, world->repkind);
      likes[i] = nr->llike;
      normd = nr->normd;
      memcpy (param[i], world->param0, sizeof (double) * nr->partsize);
      trials = 1;
      if (method != 's')
	prognose_profile_end (world->starttime,
			      world->numpop2 + (long) world->options->gamma,
			      nn);
    }
  if (method == 's')
    {
      spline = (spline_fmt *) calloc (1, sizeof (spline_fmt));
      setup_spline (spline);
      prepare_spline_nodes (nn, which, param, likes, indeks, spline->param,
			    spline->like);
      prepare_spline_first (spline->param, spline->like);
      prepare_spline_last (nn, spline->param, spline->like);
      calc_spline_diff (spline->diff, spline->param, spline->like, nn);
      errc =
	calc_spline (spline->param, spline->like, nn, spline->constr,
		     spline->diff, spline->diff2, spline->diagn, spline->work,
		     spline->nwork);
      for (i = 0; i < nn - 1; i++)
	{
	  if (probabilities[i] < 0.5)
	    {
	      xhigh = spline->param[(nn + 2) / 2];
	      high = spline->like[(nn + 2) / 2];
	      xlow = spline->param[0];
	      low = spline->like[0];
	      prob = probabilities[i];
	    }
	  else
	    {
	      xlow = spline->param[nn + 1];
	      low = spline->like[nn + 1];
	      xhigh = spline->param[(nn + 2) / 2];
	      high = spline->like[(nn + 2) / 2];
	      prob = 1. - probabilities[i];
	    }
	  testlike = maxlike - 0.5 * find_chi (df, prob);
	  x = (xhigh + xlow) / 2.;
	  panic = 0;
	  mid = testlike - 1000.;
	  while (panic++ < MAX_PROFILE_TRIALS
		 && fabs (mid - testlike) > BIGEPSILON)
	    {
	      mid =
		recalc_profilelike (x, spline->param, spline->like, nn,
				    spline->diff, spline->diff2, spline->work,
				    spline->nwork);
	      if (mid < low || mid > high)
		break;
	      if (testlike < mid)
		{
		  high = mid;
		  tmp = x;
		  x = (xlow + tmp) / 2;
		  xhigh = tmp;
		}
	      else
		{
		  low = mid;
		  tmp = x;
		  x = (tmp + xhigh) / 2;
		  xlow = tmp;
		}
	    }
	  xx = EXP (x);
	  nr->profilenum = set_profile_param (param[i], which, nr->profiles,
					      xx, nr->values, world);
	  do_profiles (nr->world, nr, likes, &normd, PROFILE,
		       nr->world->rep, nr->world->repkind);
	  //      maximize(param[i], world, nr, PROFILE, world->repkind);
	  likes[i] = nr->llike;
	  normd = nr->normd;
	  memcpy (param[i], world->param0, sizeof (double) * nr->partsize);
	  x = log (xx);
	  prognose_profile_end (world->starttime,
				world->numpop2 + (long) world->options->gamma,
				nn);
	}
      destroy_spline (spline);
    }
}

// setup spline: allocs all the necessary arrays and fill with
//               values when necessary.
void
setup_spline (spline_fmt * spline)
{
  const long degree = 3;
  long i;
  spline->ntab = 1;
  spline->nwork = (GRIDSIZE + 2) * 9 + 5 + degree * (degree + 11) / 2 + 9;
  spline->param = (double *) calloc (1, sizeof (double) * (GRIDSIZE + 2));
  spline->like = (double *) calloc (1, sizeof (double) * (GRIDSIZE + 2));
  spline->diff = (double *) calloc (1, sizeof (double) * (GRIDSIZE + 2));
  spline->diff2 = (double *) calloc (1, sizeof (double) * (GRIDSIZE + 2));
  spline->constr = (long *) calloc (1, (GRIDSIZE + 2) * sizeof (long));
  spline->diagn = (long *) calloc (1, (GRIDSIZE + 2) * sizeof (long));
  spline->work = (double *) calloc (1, spline->nwork * sizeof (double));

  for (i = 0; i < GRIDSIZE + 2; i++)
    spline->constr[i] = 3;
}

void
destroy_spline (spline_fmt * spline)
{
  free (spline->param);
  free (spline->like);
  free (spline->diff);
  free (spline->diff2);
  free (spline->constr);
  free (spline->diagn);
  free (spline->work);
}

/* calc_spline() calculates the spline function derivatives,
   and adds two additional
   points, these two points bracket the real data points:
   so that param[0]=0 and param[n+1] = 1000 * the last real param,
   likes[0] is set -DBL_MAX and likes[n+1] is set using a linear interpolation
   using the last two real data points.

   Attention the arrays param, likes, and yy need to have a size
   of  n+2 and not n
 */
int
calc_spline (double *param, double *like, long nn, long *constr, double *diff,
	     double *diff2, long *diagn, double *work, long nwork)
{
  /* call to spline.c routines */
  long degree = 3;
  long smoothness = 1;
  long opt = 414;		//convex, monotone, derivatives constraints
  double d0 = 0, dnp = 0, d20 = 0, d2np = 0;	//not used
  double eps = 0.0001;
  long kmax = 10;		//not used
  long maxstp = 10;		//not used
  long errc;
  dbvssc_ (param, like, &nn, &degree, &smoothness, &opt, &d0, &dnp, &d20,
	   &d2np, constr, &eps, NULL, NULL, NULL, NULL, &kmax, &maxstp, &errc,
	   diff, diff2, diagn, work, &nwork);
  /* end call to spline.c routines */
  return (long) errc;
}

void
prepare_spline_nodes (long n, long which, double **rawparam, double *rawlikes,
		      long *indeks, double *param, double *likes)
{
  long i;
  for (i = 0; i < n; i++)
    {
      param[i + 1] = log (rawparam[indeks[i]][which]);
      likes[i + 1] = rawlikes[indeks[i]];
    }
}

void
prepare_spline_first (double *param, double *likes)
{
  const long i = 2;
  const long i0 = 1;
  param[0] = log (SMALLEST_THETA);

  likes[0] =
    ((likes[i] - likes[i0]) / (param[i] - param[i0]) * param[0]) +
    (likes[i0] + param[i0] * (likes[i] + likes[i0]) / (param[i] - param[i0]));
}

void
prepare_spline_last (long n, double *param, double *likes)
{
  long n0 = n - 1;
  long n1 = n + 1;
  param[n1] = 100000. + param[n];
  likes[n1] =
    ((likes[n] - likes[n0]) / (param[n] - param[n0]) * param[n1]) +
    (likes[n0] + param[n0] * (likes[n] + likes[n0]) / (param[n] - param[n0]));
}


void
calc_spline_diff (double *diff, double *param, double *likes, long n)
{
  long i;
  diff[0] = ((likes[1] - likes[0]) / (param[1] - param[0]));
  diff[n] = ((likes[n] - likes[n + 1]) / (param[n] - param[n + 1]));
  for (i = 1; i < n; i++)
    {
      diff[i] =
	((likes[i + 1] - likes[i - 1]) / (param[i + 1] - param[i - 1]));
    }
  diff[GRIDMIDDLE] = 0.0;
}


/* recalc_profilelike uses the spline derivatives and returns a new 
   loglikelihood value,
   testparam is THE LOG(testparam) !
 */
double
recalc_profilelike (double testparam, double *param, double *like, long nn,
		    double *diff, double *diff2, double *work, long nwork)
{
  long degree = 3;
  long smoothness = 1;
  long errc;
  long sbopt = 2;
  long ntab = 1;
  long yy0 = 1;
  long yy1 = 1;
  long yy2 = 1;
  long erre;
  double tmp;

  double *y0tab, *y1tab, *y2tab;
  //double *gaga;
  //long i,gagatab=10;
  //gaga= (double*)calloc(1,sizeof(double)*10);
  y0tab = (double *) calloc (1, sizeof (double) * 10);
  y1tab = (double *) calloc (1, sizeof (double) * 10);
  y2tab = (double *) calloc (1, sizeof (double) * 10);
  //gaga[0]= -5;
  //for(i=1;i<gagatab;i++)
  //gaga[i] =  gaga[i-1] + 0.5;

  dbvsse_ (param, like, &nn, &degree, &smoothness,
	   //gaga, &gagatab,
	   &testparam, &ntab, &sbopt, &yy0, &yy1, &yy2, &errc, diff, diff2,
	   y0tab, y1tab, y2tab, &erre, work, &nwork);
  //for(i=0;i<gagatab;i++)
  //printf("%f %f\n",gaga[i],y0tab[i]);
  //exit(0);

  tmp = y0tab[0];
  free (y0tab);
  free (y1tab);
  free (y2tab);
  return tmp;

}


long
set_profile_param (double *param, long which, long *allwhich,
		   double xval, double *xvals, world_fmt * world)
{
  boolean has_fixed = FALSE;

  long z = 0, i = 0, zz = 0, zzz = 0;	//zz is the counter for # fixed elements
  long numpop = world->numpop;
  long numpop2 = world->numpop2;
  char *p;
  char *custm2 = world->options->custm2;
  long *fixed = NULL;
  long profnum = 0;
  // find all real zero parameters and adds them to
  // the list of parameters not to  maximize.
  // [inefficient, then this will be called several times
  //  and recalculated for nothing]
  if (strpbrk (world->options->custm, "0c"))
    {
      has_fixed = TRUE;
      fixed =
	(long *) calloc (1,
			 sizeof (long) * (numpop2 +
					  (long) world->options->gamma));
      p = world->options->custm2;
      while (*p != '\0')
	{
	  if ((*p == '0' || *p == 'c') && i != which)
	    {
	      fixed[zz] = i;
	      zz++;
	    }
	  p++;
	  i++;

	}
    }
  switch (custm2[which])
    {
    case '0':
      profnum = 0;
      break;
    case '*':
      param[which] = xval;
      allwhich[0] = which, xvals[0] = xval;
      if (has_fixed)
	{
	  zzz = 0;
	  addinvar (param, fixed, allwhich, xvals, 1, zz);
	  free (fixed);
	}
      profnum = 1 + zz;
      break;
    case 'S':
      z = 0;
      while (z < world->options->sym2n &&
	     which != world->options->sym2param[z][0] &&
	     which != world->options->sym2param[z][1])
	z++;
      param[world->options->sym2param[z][0]] = xval;
      param[world->options->sym2param[z][1]] = xval;
      if (which == world->options->sym2param[z][0])
	{
	  allwhich[0] = which;
	  allwhich[1] = world->options->sym2param[z][1];
	}
      else
	{
	  allwhich[0] = world->options->sym2param[z][0];
	  allwhich[1] = which;
	}
      xvals[0] = xval;
      xvals[1] = xval;
      if (has_fixed)
	{
	  addinvar (param, fixed, allwhich, xvals, 2, zz);
	  free (fixed);
	}
      profnum = 2 + zz;
      break;
    case 's':
      z = 0;
      while (z < world->options->symn &&
	     which != world->options->symparam[z][0] &&
	     which != world->options->symparam[z][1])
	z++;
      param[world->options->symparam[z][0]] = xval;
      param[world->options->symparam[z][1]] = xval;
      if (which == world->options->symparam[z][0])
	{
	  allwhich[0] = which;
	  allwhich[1] = world->options->symparam[z][1];
	}
      else
	{
	  allwhich[0] = world->options->symparam[z][0];
	  allwhich[1] = which;
	}
      xvals[0] = xval;
      xvals[1] = xval;
      if (has_fixed)
	{
	  addinvar (param, fixed, allwhich, xvals, 2, zz);
	  free (fixed);
	}
      profnum = 2 + zz;
      break;
    case 'm':
      if (which < numpop)
	{
	  for (i = 0; i < numpop; i++)
	    {
	      param[i] = xval;
	      xvals[i] = xval;
	      allwhich[i] = i;
	    }
	  if (has_fixed)
	    {
	      addinvar (param, fixed, allwhich, xvals, numpop, zz);
	      free (fixed);
	    }
	  profnum = numpop + zz;
	  break;
	}
      else
	{
	  for (i = numpop; i < numpop2 + (long) world->options->gamma; i++)
	    {
	      param[i] = xval;
	      xvals[i - numpop] = xval;
	      allwhich[i - numpop] = i;
	    }
	  profnum = numpop2 + (long) world->options->gamma - numpop;
	  break;
	}
    }
  return profnum;
}

// adds the invariants to the profiles list so that we do not
// evaluate them and run unnecessary cycles in the maximizer
// fixed = the index of the fixed parameters
// allwhich = the variables in the profile list
// start = last element filled element in allwhich
// len = length of the fixed array
// PB 06/30/99
void
addinvar (double *param, long *fixed, long *allwhich, double *xvals,
	  long start, long len)
{
  long i, z = 0;
  for (i = start; i < len + start; i++)
    {
      allwhich[i] = fixed[z];
      xvals[i] = param[fixed[z]];
      ++z;
    }

}

long
find_profilelike (double testlike, long whichprob, double minparam,
		  double maxparam, double *testparam, double *llike,
		  long which, world_fmt * world, boolean QD, boolean QD2,
		  double *mlparam, double *normd, nr_fmt * nr)
{
  const double probabilities[] = GRID;
#ifndef MPI
  long locus, r;
#else
#ifdef SLOWNET
//long locus, r;
#endif
#endif

  //  boolean test;
  double tmp;
  //  double likes=0;
  double xlow, low;
  double x, mid = -DBL_MAX;
  double xhigh, high;
  long panic = 0;
  helper_fmt helper;
  double *ltestparam;
  ltestparam = (double *) calloc (nr->numpop2 + 1, sizeof (double));
  *normd = -1;
  if (probabilities[whichprob] < 0.5)
    {
      xhigh = maxparam;
      xlow = whichprob > 0 ? minparam : SMALLEST_THETA / 10000.;
    }
  else
    {
      if (probabilities[whichprob] > 0.5)
	{
	  xhigh = maxparam;
	  if (maxparam < EPSILON)
	    xlow = probabilities[whichprob] >= 0.98 ? 1000000.0 : minparam;
	  else
	    xlow =
	      probabilities[whichprob] >= 0.98 ? maxparam * 10000 : minparam;
	}
      else
	{
	  xlow = minparam;
	  xhigh = maxparam;
	}
    }
  if (QD)
    {
/*#ifndef MPI
      for (locus = 0; locus < world->loci; locus++)
	{
	  if (world->repkind == SINGLECHAIN)
	    {
	      for (r = 0; r < world->repstop; r++)
		{
		  create_apg0 (nr->apg0[r][locus], nr, &world->atl[r][locus]);
		}
	    }
	  else
	    {
	     //interpolate_like(nr, locus);
	      for (r = nr->repstart; r < nr->repstop; r++)
		create_multiapg0 (nr->apg0[r][locus], nr, r, locus);
	    }
	}
#else
#ifdef SLOWNET
      for (locus = 0; locus < world->loci; locus++)
	{
	  if (world->repkind == SINGLECHAIN)
	    {
	      for (r = 0; r < world->repstop; r++)
		{
		  create_apg0 (nr->apg0[r][locus], nr, &world->atl[r][locus]);
		}
	    }
	  else
	    {
		exit(-1);
	     // interpolate_like(nr, locus);
	      //for (r = nr->repstart; r < nr->repstop; r++)
		//create_multiapg0 (nr->apg0[r][locus], nr, r, locus);
	    }
	}
#endif
#endif 
*/
      memcpy (testparam, mlparam, sizeof (double) *
	      (world->numpop2 + (long) world->options->gamma));
      nr->profilenum = set_profile_param (testparam, which, nr->profiles,
					  xlow, nr->values, world);
      testparam[which] = xlow;
      set_logparam (ltestparam, testparam, nr->partsize);
      fill_helper (&helper, testparam, ltestparam, nr->world, nr);
      low = CALCLIKE (&helper, testparam, ltestparam);
      memcpy (testparam, mlparam, sizeof (double) *
	      (world->numpop2 + (long) world->options->gamma));
      nr->profilenum = set_profile_param (testparam, which, nr->profiles,
					  xhigh, nr->values, world);
      testparam[which] = xhigh;
      set_logparam (ltestparam, testparam, nr->partsize);
      fill_helper (&helper, testparam, ltestparam, world, nr);
      high = CALCLIKE (&helper, testparam, ltestparam);
    }
  else
    {
      memcpy (testparam, mlparam, sizeof (double) * nr->partsize);
      nr->profilenum = set_profile_param (testparam, which, nr->profiles,
					  xlow, nr->values, world);
      maximize (testparam, world, nr, PROFILE, world->repkind);
      low = nr->llike;
      memcpy (testparam, mlparam, sizeof (double) * nr->partsize);
      nr->profilenum = set_profile_param (testparam, which, nr->profiles,
					  xhigh, nr->values, world);
      maximize (testparam, world, nr, PROFILE, world->repkind);
      high = nr->llike;
	  	      testlike = high - 0.5 * find_chi (1, testlike);
      
	  if(high< world->atl[0][world->loci].param_like)
		printf("%i> DARN: high=%f ml=%f xigh=%f mlparam[which]=%f testlike=%f\n", myID, high, world->atl[0][world->loci].param_like,xhigh,mlparam[which],testlike); 
    }
  panic = 0;
  x = (xhigh + xlow) / 2.;
  while (panic++ < MAX_PROFILE_TRIALS && fabs (mid - testlike) > BIGEPSILON)
    {
      if (QD)
	{
	  memcpy (testparam, mlparam, sizeof (double) * nr->partsize);
	  nr->profilenum = set_profile_param (testparam, which,
					      nr->profiles, x,
					      nr->values, world);
	  testparam[which] = x;
	  set_logparam (ltestparam, testparam, nr->partsize);
	  fill_helper (&helper, testparam, ltestparam, world, nr);
	  mid = CALCLIKE (&helper, testparam, ltestparam);
	}
      else
	{
	  memcpy (testparam, mlparam, sizeof (double) * nr->partsize);
	  nr->profilenum = set_profile_param (testparam, which, nr->profiles,
					      x, nr->values, world);
	  maximize (testparam, world, nr, PROFILE, world->repkind);
	  mid = *llike = nr->llike;
	  *normd = nr->normd;
	}
      if ((testlike < low) || (testlike > high))
	return MAX_PROFILE_TRIALS;
      if (testlike < mid)
	{
	  high = mid;
	  tmp = x;
	  x = (xlow + tmp) / 2;
	  xhigh = tmp;
	}
      else
	{
	  low = mid;
	  tmp = x;
	  x = (tmp + xhigh) / 2;
	  xlow = tmp;
	}
    }
  if (QD)
    {

      *llike = mid;
      if (QD2)
	{
	  x = testparam[which];
	  memcpy (testparam, mlparam,
		  sizeof (double) * (nr->numpop2 +
				     (long) world->options->gamma));
	  nr->profilenum =
	    set_profile_param (testparam, which, nr->profiles, x, nr->values,
			       world);
	  maximize (testparam, world, nr, PROFILE, world->repkind);
	  *llike = nr->llike;
	  *normd = nr->normd;
	  memcpy (testparam, world->param0, sizeof (double) * nr->partsize);
	}
    }
  free (ltestparam);
  return panic;
}


double
interpolate (double xl, double xc, double xh, double low, double center,
	     double high, double testlike)
{
  double xc2, xl2, xh2, a, b, c, x;

  xc2 = xc * xc;
  xl2 = xl * xl;
  xh2 = xh * xh;

  a =
    (-(high * xc) + low * xc + center * xh - low * xh - center * xl +
     high * xl) / ((-xc + xh) * (xh - xl) * (-xc + xl));

  b =
    (high * xc2 - low * xc2 - center * xh2 + low * xh2 + center * xl2 -
     high * xl2) / ((-xc + xh) * (xc - xl) * (-xh + xl));

  c =
    -((low * xc2 * xh - low * xc * xh2 - high * xc2 * xl + center * xh2 * xl +
       high * xc * xl2 - center * xh * xl2) / ((-xc + xh) * (xc - xl) * (xh -
									 xl)));


  x = (-b - sqrt (b * b - 4 * a * (c - testlike))) / (2 * a);

  return x;
}


void
allocate_profile_percentiles (world_fmt * world)
{
  long i;
  world->quantiles =
    (quantile_fmt *) calloc (1, sizeof (quantile_fmt) * (world->numpop2 + 1));
  for (i = 0; i < world->numpop2 + 1; i++)
    {
      world->quantiles[i].name = (char *) calloc (1, sizeof (char) * 20);
      world->quantiles[i].param =
	(double *) calloc (1, sizeof (double) * GRIDSIZE);
    }
}

void
destroy_profile_percentiles (world_fmt * world)
{
  long i;
  for (i = 0; i < world->numpop2 + (long) world->options->gamma; i++)
    {
      free (world->quantiles[i].name);
      free (world->quantiles[i].param);
    }
  free (world->quantiles);
}


void
print_menu_profile (long which, long nn)
{
  char nowstr[LINESIZE];
  get_time (nowstr, "%H:%M:%S");
  fprintf (stdout,
	   "%s   Calculating profile likelihoods for parameter %2li out of %2li\n",
	   nowstr, 1 + which, nn);
}


long
set_df (long which, char *custm2, long numpop, long numpop2)
{
  char ch;
  long i;
  long zeros = 0;
  long ms = 0;
  long ss = 0;

  for (i = 0; i < numpop2; i++)
    {
      ch = custm2[i];
      switch (ch)
	{
	case '0':
	  zeros++;
	  break;
	case 'm':
	  ms++;
	  break;
	case 's':
	  ss++;
	  break;
	}
    }
  if (numpop2 - zeros == ms)
    return 2;
  else
    {
      if (custm2[0] == 'm')
	return numpop2 - numpop + 1 - zeros - ss / 2;
      else
	return numpop2 - zeros - ss / 2;
    }
  return -1;
}
