/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 R A N D O M   G E N E R A T O R   R O U T I N E S 

 creates options structures,
 reads options from parmfile if present
 
 prints options,
 and finally helps to destroy itself.
                                                                                                               
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 
Copyright 2001 Peter Beerli and Joseph Felsenstein

$Id: random.c,v 1.25 2001/07/25 19:27:24 beerli Exp $
-------------------------------------------------------*/

#include "sighandler.h"
#include "migration.h"
#include "heating.h"
#include "random.h"
#include "migrate_mpi.h"
#ifdef SPRNG			/*NCSA random number generator library, slower than ours */
#define SIMPLE_SPRNG
#ifdef MPI
#ifndef USE_MPI
#define USE_MPI
#endif
#endif
#include "sprng.h"
#endif
#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif
#ifdef PTHREADS
extern tpool_t heating_pool;
#endif

/* prototypes ----------------------------------------- */
void getseed (option_fmt * options);
void swap_ptr (long **ptr, long **newptr);

double randum (void);
#ifdef PTHREADS
double randum_thread (void);
#endif

//=============================================
void
getseed (option_fmt * options)
{
#ifdef MPI
  long test;
#endif
#ifndef SPRNG
  long i;
#endif
  long timeseed;

  switch (options->autoseed)
    {
    case AUTO:
      timeseed = abs ((long) time (NULL));
      switch (timeseed % 4)
	{
	case 0:
	  ++timeseed;
	  break;
	case 1:
	  break;
	case 2:
	  ++timeseed;
	  break;
	case 3:
	  break;
	default:
	  error ("Problem with automatic random number seed assignment");
	}
      options->inseed = abs ((long) timeseed);
      break;
    case NOAUTO:
      break;
    case NOAUTOSELF:
      break;
    default:
      error ("Error: Seed value not defined");
      break;
    }
#ifndef SPRNG
#ifdef MPI
  MPI_Bcast (&options->inseed, 1, MPI_LONG, MASTER, comm_world);
  if (myID != MASTER)
    {
      for (i = 1; i <= 1000; i++)	/* clear the random number generator */
	RANDUM ();

      for (i = 0; i <= 2; i++)
	seed[i] = 0;
      i = 0;
      do
	{
	  seed[i] = options->inseed & 2047;
	  options->inseed /= 2048;
	  i++;
	}
      while (options->inseed != 0);

      for (i = 0; i < myID; i++)
	{
	  test = RANDINT (1, MAXLONG);
	  //printf("     %i %li> %li\n", myID, i, test); 
	}
      switch (test % 4)
	{
	case 0:
	  test++;
	  break;
	case 1:
	  break;
	case 2:
	  test--;
	  break;
	case 3:
	  test--;
	  test--;
	  break;
	}
      options->inseed = test;
    }
  printf ("%i> random number seed = %li\n", myID, options->inseed);
#endif
#endif
  options->saveseed = options->inseed;
#ifdef SPRNG
  init_sprng (options->inseed, SPRNG_DEFAULT);
#else /* if we do not have SPRNG */
  for (i = 1; i <= 1000; i++)	/* clear the random number generator */
    RANDUM ();

  for (i = 0; i <= 2; i++)
    seed[i] = 0;
  i = 0;
  do
    {
      seed[i] = options->inseed & 2047;
      options->inseed /= 2048;
      i++;
    }
  while (options->inseed != 0);
#endif /* not SPRNG */
}

void
swap_ptr (long **ptr, long **newptr)
{
  long *temp;
  temp = *ptr;
  *ptr = *newptr;
  *newptr = temp;
}

#ifdef PTHREADS
double
randum_thread (void)
/* thread save random number generator */
{
#ifdef SPRNG
  return sprng ();
#else
  double value;
  //longer newseed;
#ifdef PTHREADS
  if ((pthread_mutex_lock (&(heating_pool->random_lock))) != 0)
    error ("pthread_mutex_lock failed in random_thread()");
#endif
  newseed[0] = 1549 * seed[0];
  newseed[1] = newseed[0] / 2048;
  newseed[0] &= 2047;
  newseed[2] = newseed[1] / 2048;
  newseed[1] &= 2047;
  newseed[1] += 1549 * seed[1] + 812 * seed[0];
  newseed[2] += newseed[1] / 2048;
  newseed[1] &= 2047;
  newseed[2] += 1549 * seed[2] + 812 * seed[1];
  swap_ptr (&newseed, &seed);
  //  memcpy (seed, newseed, sizeof (longer));
  seed[2] &= 1023;
  value = (((seed[0] / 2048.0 + seed[1]) / 2048.0 + seed[2]) / 1024.0);
#ifdef PTHREADS
  if ((pthread_mutex_unlock (&(heating_pool->random_lock))) != 0)
    error ("pthread_mutex_unlock failed in randum_thread()");
#endif
  return value;
#endif /*SPRNG*/
}				/* randum */
#else
double
randum (void)
/* Non thread-safe random number generator (faster) */
{
#ifdef SPRNG
  return sprng ();
#else
  double value;
  //  longer newseed;
  newseed[0] = 1549 * seed[0];
  newseed[1] = newseed[0] / 2048;
  newseed[0] &= 2047;
  newseed[2] = newseed[1] / 2048;
  newseed[1] &= 2047;
  newseed[1] += 1549 * seed[1] + 812 * seed[0];
  newseed[2] += newseed[1] / 2048;
  newseed[1] &= 2047;
  newseed[2] += 1549 * seed[2] + 812 * seed[1];
  //memcpy (seed, newseed, sizeof (longer));
  swap_ptr (&newseed, &seed);
  seed[2] &= 1023;
  value = (((seed[0] / 2048.0 + seed[1]) / 2048.0 + seed[2]) / 1024.0);
  return value;
#endif /*SPRNG*/
}				/* randum */
#endif
