/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm   
 ------------------------------------------------------- 
 O P T I O N S   R O U T I N E S 

 creates options structures,
 reads options from parmfile if present

 prints options,
 and finally helps to destroy itself.
 
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 
Copyright 2001 Peter Beerli and Joseph Felsenstein

$Id: options.c,v 1.80 2001/09/07 23:56:13 beerli Exp $
-------------------------------------------------------*/
//#include <stdio.h>
#include <time.h>
#include "migration.h"
#include "fst.h"
#include "tools.h"
#include "migrate_mpi.h"
#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif

/* parmfile parameter specifications and keywords */
#define LINESIZE 1024
#define NUMBOOL 24
#define BOOLTOKENS {"menu","interleaved","print-data","mixplot",\
  "moving-steps","freqs-from-data","usertree", \
  "autocorrelation", "simulation","plot", "weights",\
  "read-summary","write-summary","mig-histogram", "heating", "print-fst",\
  "distfile","geofile","gelman-convergence", "randomtree",\
  "fast-likelihood", "aic-modeltest", "use-M", "use-Nm"}
#define NUMNUMBER 40
#define NUMBERTOKENS {"ttratio","short-chains",\
 "short-steps","short-inc","long-chains",\
 "long-steps", "long-inc", "theta", \
 "nmlength","random-seed","migration","mutation",\
 "datatype", "categories","rates","prob-rates", \
 "micro-max", "micro-threshold", "delimiter","burn-in",\
 "infile", "outfile", "mathfile", "title", \
 "long-chain-epsilon","print-tree","progress","l-ratio",\
 "fst-type","profile","custom-migration","sumfile","short-sample",\
 "long-sample", "replicate","cpu","logfile", "seqerror-rate","uepfile", \
 "uep-rates"};



/* prototypes ------------------------------------------- */
void set_usem_related (option_fmt * options);
void create_options (option_fmt ** options);
void init_options (option_fmt * options);
void get_options (option_fmt * options);
void set_param (world_fmt * world, data_fmt * data, option_fmt * options,
		long locus);
void set_profile_options (option_fmt * options);
void print_menu_options (world_fmt * world, option_fmt * options,
			 data_fmt * data);
void print_options (FILE * file, world_fmt * world, option_fmt * options,
		    data_fmt * data);
void decide_plot (worldoption_fmt * options, long chain, long chains,
		  char type);
void destroy_options (option_fmt * options);
long save_parmfile (option_fmt * options, world_fmt * world);
void read_options_master (option_fmt * options);
void read_options_worker (char **buffer, option_fmt * options);
/* private functions */
boolean booleancheck (option_fmt * options, char *var, char *value);
long boolcheck (char ch);
boolean numbercheck (option_fmt * options, char *var, char *value);
void reset_oneline (option_fmt * options, long position);
void read_theta (option_fmt * options);
void read_mig (option_fmt * options);
#ifdef MPI
void read_theta_worker (char **buffer, option_fmt * options);
void read_mig_worker (char **buffer, option_fmt * options);
char skip_sspace (char **buffer);
void
read_custom_migration_worker (char **buffer, option_fmt * options,
			      char *value, long customnumpop);

#endif
char skip_space (option_fmt * options);
void read_custom_migration (FILE * file, option_fmt * options, char *value,
			    long customnumpop);
void synchronize_param (world_fmt * world, option_fmt * options);
void resynchronize_param (world_fmt * world);
void specify_migration_type (option_fmt * options);
void fillup_custm (long len, world_fmt * world, option_fmt * options);
void set_partmean_mig (long **mmparam, double *param, char *custm2, long migm,
		       long numpop2);
long scan_m (char *custm2, long start, long stop);
void set_plot (option_fmt * options);
void set_plot_values (double **values, double plotrange[], long intervals,
		      short type);
void set_grid_param (world_fmt * world, long gridpoints);
void print_arbitrary_migration_table (FILE * file, world_fmt * world,
				      data_fmt * data);
void print_distance_table (FILE * file, world_fmt * world,
			   option_fmt * options, data_fmt * data);
long save_options_buffer (char **buffer, option_fmt * options,
			  world_fmt * world);

/*======================================================*/
void
create_options (option_fmt ** options)
{
  (*options) = (option_fmt *) calloc (1, sizeof (option_fmt));
}

void
init_options (option_fmt * options)
{
  long i;
  unsigned long timeseed;
  /* General options --------------------------------------- */
  options->nmlength = DEFAULT_NMLENGTH;
  options->popnmlength = DEFAULT_POPNMLENGTH;
  options->allelenmlength = DEFAULT_ALLELENMLENGTH;
  options->custm = (char *) calloc (1, sizeof (char) * 1000);	//needs fix
  //options->custm2 = (char *) calloc (1, sizeof (char) * (1 + NUMPOP));
  options->symn = 0;
  options->sym2n = 0;
  options->zeron = 0;
  options->constn = 0;
  options->mmn = 0;
  /* input/output options ---------------------------------- */
  options->menu = TRUE;
  options->progress = TRUE;
  options->verbose = FALSE;
  options->writelog = FALSE;
#ifdef UEP
  options->uep = FALSE;
  options->ueprate = 1.0;
  options->uepmu = 1.0;
  options->uepnu = 0.0;
  //  options->uep_last = FALSE;
#endif
  options->printdata = FALSE;
  options->usertree = FALSE;
  options->randomtree = FALSE;
  options->fastlike = TRUE;
  options->treeprint = NONE;
  options->printfst = FALSE;
  options->fsttype = THETAVARIABLE;
  options->usem = FALSE;
  options->migvar = PLOT4NM;
  fst_type (options->fsttype);
  options->plot = FALSE;
  options->plotmethod = PLOTALL;	/* outfile and mathematica file */
  options->plotvar = PLOT4NM;
  options->plotscale = PLOTSCALELOG;
  options->plotrange[0] = PLANESTART;	/* start x axis */
  options->plotrange[1] = PLANEEND;	/*end x axis */
  options->plotrange[2] = PLANESTART;	/*start y axis */
  options->plotrange[3] = PLANEEND;	/* end y axis */
  options->plotintervals = PLANEINTERVALS;
  options->simulation = FALSE;
  options->movingsteps = FALSE;
  options->acceptfreq = 0.1;
  options->mighist = FALSE;
  options->mixplot = FALSE;
  strcpy (options->infilename, INFILE);
  strcpy (options->outfilename, OUTFILE);
  strcpy (options->logfilename, LOGFILE);
  strcpy (options->mathfilename, MATHFILE);
  strcpy (options->sumfilename, SUMFILE);
  strcpy (options->treefilename, TREEFILE);
  strcpy (options->utreefilename, UTREEFILE);
  strcpy (options->catfilename, CATFILE);
  strcpy (options->weightfilename, WEIGHTFILE);
  strcpy (options->mighistfilename, MIGHISTFILE);
  strcpy (options->distfilename, DISTFILE);
  strcpy (options->geofilename, GEOFILE);
  strcpy (options->bootfilename, BOOTFILE);
  strcpy (options->aicfilename, AICFILE);
#ifdef UEP
  strcpy (options->uepfilename, UEPFILE);
#endif
  strcpy (options->title, "\0");
  options->lratio = (lratio_fmt *) calloc (1, sizeof (lratio_fmt));
  options->lratio->alloccounter = 1;
  options->lratio->data =
    (lr_data_fmt *) calloc (1,
			    sizeof (lr_data_fmt) *
			    options->lratio->alloccounter);
  options->lratio->data[0].value1 =
    (char *) calloc (1, sizeof (char) * LINESIZE);
  options->lratio->data[0].value2 =
    (char *) calloc (1, sizeof (char) * LINESIZE);
  options->aic = FALSE;
  options->fast_aic = FALSE;
  options->profile = ALL;
  options->profilemethod = 'p';
  options->df = 1;
  options->qdprofile = FALSE;
  options->printprofsummary = TRUE;
  options->printprofile = TRUE;
  if (options->usem)
    options->profileparamtype = options->plotvar = options->migvar = PLOTM;
  else
    options->profileparamtype = options->plotvar = options->migvar = PLOT4NM;
  /* data options ------------------------------------------ */
  options->datatype = 's';
  options->migration_model = MATRIX;
  options->thetag = (double *) calloc (1, sizeof (double) * NUMPOP);
  options->mg = (double *) calloc (1, sizeof (double) * NUMPOP);
  options->gamma = FALSE;
  options->alphavalue = START_ALPHA;
  /* EP data */
  options->dlm = '\0';
  /* microsat data */
  options->micro_threshold = MICRO_THRESHOLD;
  options->micro_stepnum = MAX_MICROSTEPNUM;
  /*sequence data */
  options->seqerror = 0.0;
  options->interleaved = FALSE;
  options->ttratio = (double *) calloc (1, sizeof (double) * 2);
  options->ttratio[0] = 2.0;
  options->freqsfrom = TRUE;
  options->categs = ONECATEG;
  options->rate = (double *) calloc (1, sizeof (double));
  options->rate[0] = 1.0;
  options->rcategs = 1;
  options->rrate = (double *) calloc (1, sizeof (double));
  options->probcat = (double *) calloc (1, sizeof (double));
  options->autocorr = FALSE;
  options->rrate[0] = 1.0;
  options->probcat[0] = 1.0;

  options->probsum = 0.0;

  options->lambda = 1.0;
  options->weights = FALSE;
  /* random number options --------------------------------- */
  options->autoseed = AUTO;
  options->autoseed = AUTO;
  //#ifndef MAC
  timeseed = (unsigned long) time (NULL) / 4;
  //#else
  //timeseed = (unsigned long) clock () / 4;
  //#endif
  options->inseed = (long) timeseed + 1;
  /* mcmc options ------------------------------------------ */
  options->thetaguess = FST;
  options->migrguess = FST;
  for (i = 0; i < NUMPOP; i++)
    {
      options->thetag[i] = 1.0;
      options->mg[i] = 1.0;
      options->custm[i] = '*';
    }
  options->custm[i] = '\0';
  options->custm2 = (char *) calloc (i + 2, sizeof (char));
  strncpy (options->custm2, options->custm, i);
  options->numthetag = options->nummg = 0;
  options->bayes_update = FALSE;
  options->schains = 10;
  options->sincrement = 20;
  options->ssteps = 500;
  options->lchains = 3;
  options->lincrement = 20;
  options->lsteps = 5000;
  options->burn_in = BURNINPERIOD;
  options->heating = 0;		/* no heating */
  options->heating_interval = 10;
  options->heat[0] = COLD;
  options->heat[1] = WARM;
  options->heat[2] = HOT;
  options->heat[3] = VERYHOT;
  options->heated_chains = HEATED_CHAIN_NUM;
  options->lcepsilon = LONGCHAINEPSILON;
  options->gelman = FALSE;
  options->pluschain = PLUSCHAIN;
  options->replicate = FALSE;
  options->replicatenum = 0;
  options->gridpoints = 0;
  /* genealogy summary options----------------------------------- */
  options->readsum = FALSE;
  options->writesum = FALSE;
  /*threading over loci */
  options->cpu = 1;
}

void
read_options_master (option_fmt * options)
{
  long counter = 0;
  long position = 0;
  char varvalue[LINESIZE];
  char parmvar[LINESIZE];
  char input[LINESIZE];
  char *p, *tmp;
  FILE *mixfile;

  options->parmfile = fopen (PARMFILE, "r");
  if (options->parmfile)
    {
      counter = 0;
      position = ftell (options->parmfile);
      while (FGETS (input, LINESIZE, options->parmfile) != NULL)
	{
	  counter++;
	  if ((input[0] == '#') || isspace ((int) input[0])
	      || input[0] == ';')
	    continue;
	  else
	    {
	      if (!(isalnum ((int) input[0]) || strchr ("{}", input[0])))
		{
		  usererror ("The parmfile contains an error on line %li\n",
			     counter);
		}
	    }
	  if ((p = strchr (input, '#')) != NULL)
	    *p = '\n';
	  if (!strncmp (input, "end", 3))
	    break;
	  tmp = strtok (input, "=");
	  if (tmp != NULL)
	    strcpy (parmvar, tmp);
	  else
	    {
	      fprintf(stderr,"WARNING: error in parmfile on line %li with %s\n", counter, input);
	      continue;
	    }
	  if (!strncmp (parmvar, "theta", 5))
	    {
	      reset_oneline (options, position);
	      read_theta (options);
	      position = ftell (options->parmfile);
	      continue;
	    }
	  if (!strncmp (parmvar, "migration", 5))
	    {
	      reset_oneline (options, position);
	      read_mig (options);
	      position = ftell (options->parmfile);
	      continue;
	    }
	  tmp = strtok (NULL, "\n");
	  if (tmp != NULL)
	    strncpy (varvalue, tmp, LINESIZE - 1);
	  if (!booleancheck (options, parmvar, varvalue))
	    {
	      if (!numbercheck (options, parmvar, varvalue))
		{
		  warning ("Inappropiate entry in parmfile: %s ignored\n",
			   input);
		}
	    }
	  position = ftell (options->parmfile);
	}
    }
  if (options->mixplot)
    {
      mixfile = fopen ("mixfile", "w");
      fprintf (mixfile, " ");
      fclose (mixfile);
    }
}

#ifdef MPI
void
read_options_worker (char **buffer, option_fmt * options)
{
  long counter = 0;
  char *position;
  char varvalue[LINESIZE];
  char parmvar[LINESIZE];
  char input[LINESIZE];
  char *p, *tmp;

  options->buffer = buffer;
  options->parmfile = NULL;
  if (strlen (*buffer) > 0)
    {
      counter = 0;
      position = *buffer;
      while (sgets (input, LINESIZE, buffer) != NULL)
	{
	  counter++;
	  if ((input[0] == '#') || isspace ((int) input[0])
	      || input[0] == ';')
	    continue;
	  else
	    {
	      if (!(isalnum ((int) input[0]) || strchr ("{}", input[0])))
		{
		  usererror ("The parmfile contains an error on line %li\n",
			     counter);
		}
	    }
	  if ((p = strchr (input, '#')) != NULL)
	    *p = '\n';
	  if (!strncmp (input, "end", 3))
	    break;
	  tmp = strtok (input, "=");
	  if (tmp != NULL)
	    strcpy (parmvar, tmp);
	  else
	    {
	      fprintf (stderr, "WARNING: error in parmfile on line %li with %s\n", counter, input);
	      continue;
	    }
	  if (!strncmp (parmvar, "theta", 5))
	    {
	      *buffer = position;
	      read_theta_worker (buffer, options);
	      position = *buffer;
	      continue;
	    }
	  if (!strncmp (parmvar, "migration", 5))
	    {
	      *buffer = position;
	      read_mig_worker (buffer, options);
	      position = *buffer;
	      continue;
	    }
	  tmp = strtok (NULL, "\n");
	  if (tmp != NULL)
	    strncpy (varvalue, tmp, LINESIZE - 1);
	  if (!booleancheck (options, parmvar, varvalue))
	    {
	      if (!numbercheck (options, parmvar, varvalue))
		{
		  warning ("Inappropiate entry in parmfile: %s ignored\n",
			   input);
		}
	    }
	  position = *buffer;
	}
    }
}
#endif

void
print_menu_options (world_fmt * world, option_fmt * options, data_fmt * data)
{
  if (options->numpop > world->numpop)
    usererror ("Inconsistency between your Menu/Parmfile and your datafile\n \
Check the number of populations!\n");
  if (options->progress)
    {
      print_options (stdout, world, options, data);
      if (options->writelog)
	print_options (options->logfile, world, options, data);
    }
}


void
print_options (FILE * file, world_fmt * world, option_fmt * options,
	       data_fmt * data)
{
  long i, j, tt;
  char mytext[LINESIZE];
  char seedgen[LINESIZE], spacer[LINESIZE];
  char paramtgen[LINESIZE], parammgen[LINESIZE];
  if (options->datatype != 'g')
    {
      switch ((short) options->autoseed)
	{
	case AUTO:
	  strcpy (seedgen, "with internal timer");
	  strcpy (spacer, "  ");
	  break;
	case NOAUTOSELF:
	  strcpy (seedgen, "from parmfile");
	  strcpy (spacer, "      ");
	  break;
	case NOAUTO:
	  strcpy (seedgen, "from seedfile");
	  strcpy (spacer, "      ");
	  break;
	default:
	  strcpy (seedgen, "ERROR");
	  strcpy (spacer, " ");
	  break;
	}
      switch (options->thetaguess)
	{
	case OWN:
	  strcpy (paramtgen, "from guessed values");
	  break;
	case FST:
	  strcpy (paramtgen, "from the FST-calculation");
	  break;
	  //    case PARAMGRID:
	  //      strcpy (paramtgen, "GRID values around a center");
	  //      break;          
	case RANDOMESTIMATE:
	  strcpy (paramtgen, "RANDOM start value from N(mean,std)");
	  break;
	default:
	  strcpy (paramtgen, "ERROR");
	  break;
	}
      switch (options->migrguess)
	{
	case OWN:
	  strcpy (parammgen, "from guessed values");
	  break;
	case FST:
	  strcpy (parammgen, "from the FST-calculation");
	  break;
	  //    case PARAMGRID:
	  //      strcpy (parammgen, "GRID values around a center");
	  //      break;          
	case RANDOMESTIMATE:
	  strcpy (parammgen, "RANDOM start value from N(mean,std)");
	  break;
	default:
	  strcpy (parammgen, "ERROR");
	  break;
	}
    }
  fprintf (file, "Options in use:\n");
  fprintf (file, "---------------\n");
  switch (options->datatype)
    {
    case 'a':
      fprintf (file, "Datatype: Allelic data\n");
      break;
    case 'b':
      fprintf (file, "Datatype: Microsatellite data [Brownian motion]\n");
      break;
    case 'm':
      fprintf (file, "Datatype: Microsatellite data [Stepwise mutation]\n");
      break;
    case 's':
      fprintf (file, "Datatype: DNA sequence data\n");
      break;
    case 'n':
      fprintf (file, "Datatype: Single nucleotide polymorphism data\n");
      break;
    case 'u':
      fprintf (file,
	       "Datatype: Single nucleotide polymorphism data (PANEL)\n");
      break;
    case 'f':
      fprintf (file, "Datatype: Ancestral state method\n");
      break;
    case 'g':
      fprintf (file, "Datatype: Genealogy summary of an older run\n");
      break;
    }
  if (options->datatype != 'g')
    {
      fprintf (file, "Random number seed (%s)%s%20li\n", seedgen, " ",
	       options->saveseed);
      fprintf (file, "Start parameters:\n   Theta values were generated ");
      fprintf (file, " %s\n", paramtgen);
      if (options->thetaguess == OWN)
	{
	  fprintf (file, "   Theta = ");
	  for (i = 0; i < options->numthetag - 1; i++)
	    {
	      fprintf (file, "%.5f,", options->thetag[i]);
	    }
	  fprintf (file, "%.5f\n", options->thetag[i]);
	}
      fprintf (file, "   M values were generated %s\n", parammgen);
      if (options->migrguess == OWN)
	{
	  tt = 0;
	  if (options->usem)
	    fprintf (file, "   M-matrix: ");
	  else
	    fprintf (file, "   4Nm-matrix: ");
	  if (options->nummg == 1)
	    {
	      fprintf (file, "%5.2f [all are the same]\n", options->mg[tt++]);
	    }
	  else
	    {
	      for (i = 0; i < world->numpop; i++)
		{
		  for (j = 0; j < world->numpop; j++)
		    {
		      if (i != j)
			fprintf (file, "%5.2f ", options->mg[tt++]);
		      else
			fprintf (file, "----- ");
		      if (j > 10)
			fprintf (file, "\n                ");
		    }
		  fprintf (file, "\n               ");
		}
	      fprintf (file, "\n");
	    }
	}
    }
  print_arbitrary_migration_table (file, world, data);
  print_distance_table (file, world, options, data);
  if (options->gamma)
    {
      fprintf (file, "Mutation rate among loci is Gamma-distributed\n");
      fprintf (file, "Initial scale parameter alpha = %f\n",
	       options->alphavalue);
      if (options->custm[world->numpop2] == 'c')
	fprintf (file, "and is constant [will not be estimated]\n");
    }
  else
    fprintf (file, "Mutation rate is constant %s\n", world->loci > 1 ?
	     "for all loci" : "");
#ifdef UEP
  if (options->uep)
    {
      fprintf (file, "0/1 polymorphism analysis, with 0/1 data in file:%s\n",
	       options->uepfilename);
      fprintf (file, "     with forward mutation rate %f*mu\n", options->uepmu);
      fprintf (file, "     with back    mutation rate %f*mu\n", options->uepnu);
    }
#endif
  if (options->datatype != 'g')
    {
      fprintf (file, "Markov chain settings:\n");
      fprintf (file, "   Short chains (short-chains):         %20li\n",
	       options->schains);
      fprintf (file, "      Trees sampled (short-inc*samples):%20li\n",
	       options->sincrement * options->ssteps);
      fprintf (file, "      Trees recorded (short-sample):    %20li\n",
	       options->ssteps);
      fprintf (file, "   Long chains (long-chains):           %20li\n",
	       options->lchains);
      fprintf (file, "      Trees sampled (long-inc*samples): %20li\n",
	       options->lincrement * options->lsteps);
      fprintf (file, "      Trees recorded (long-sample):     %20li\n",
	       options->lsteps);
      if (options->replicate)
	{
	  if (options->replicatenum == 0)
	    fprintf (file, "   Averaging over long chains\n");
	  else
	    fprintf (file, "   Averaging over replicates:           %20li\n",
		     options->replicatenum);
	}
      if (options->heating > 0)
	{
	  fprintf (file,
		   "   Heating scheme active\n      %li chains with temperature ",
		   options->heated_chains);
	  for (i = 0; i < options->heated_chains - 1; i++)
	    fprintf (file, "%5.2f,", options->heat[i]);
	  fprintf (file, "%5.2f\n      Swapping interval is %li\n",
		   options->heat[i], options->heating_interval);
	}
      if (options->movingsteps)
	{
	  fprintf (file, "   Percentage of new genealogies:       %20.2f\n",
		   (double) options->acceptfreq);
	}
      if (options->burn_in > 0)
	{
	  fprintf (file, "   Number of discard trees per chain:   %20li\n",
		   (long) options->burn_in);
	}
      if (options->lcepsilon < LONGCHAINEPSILON)
	{
	  fprintf (file, "   Parameter-likelihood epsilon:        %20.5f\n",
		   options->lcepsilon);
	}
    }
  fprintf (file, "Print options:\n");
  if (options->datatype != 'g')
    {
      fprintf (file, "   Data file: %46.46s\n", options->infilename);
      fprintf (file, "   Output file: %44.44s\n", options->outfilename);
      fprintf (file, "   Print data: %45.45s\n",
	       options->printdata ? "Yes" : "No");
      switch (options->treeprint)
	{
	case NONE:
	  fprintf (file, "   Print genealogies: %38.38s\n", "No");
	  break;
	case ALL:
	  fprintf (file, "   Print genealogies: %38.38s\n", "Yes, all");
	  break;
	case LASTCHAIN:
	  fprintf (file, "   Print genealogies: %38.38s\n",
		   "Yes, only those in last chain");
	  break;
	case BEST:
	  fprintf (file, "   Print genealogies: %38.38s\n",
		   "Yes, only the best");
	  break;
	}
    }
  if (options->plot)
    {
      switch (options->plotmethod)
	{
	case PLOTALL:
	  sprintf (mytext, "Yes, to outfile and %s", options->mathfilename);
	  break;
	default:
	  strcpy (mytext, "Yes, to outfile");
	  break;
	}
      fprintf (file, "   Plot data: %-46.46s\n", mytext);
      fprintf (file,
	       "              Parameter: %s, Scale: %s, Intervals: %li\n",
	       options->plotvar == PLOT4NM ? "{Theta, 4Nm}" : "{Theta, M}",
	       options->plotscale == PLOTSCALELOG ? "Log10" : "Standard",
	       options->plotintervals);
      fprintf (file, "              Ranges: X-%5.5s: %f - %f\n",
	       options->plotvar == PLOT4NM ? "4Nm" : "M",
	       options->plotrange[0], options->plotrange[1]);
      fprintf (file, "              Ranges: Y-%5.5s: %f - %f\n", "Theta",
	       options->plotrange[2], options->plotrange[3]);
    }
  else
    {
      fprintf (file, "   Plot data: %-46.46s\n", "No");
    }
  switch (options->profile)
    {
    case NONE:
      strcpy (mytext, "No");
      break;
    case ALL:
      strcpy (mytext, "Yes, tables and summary");
      break;
    case TABLES:
      strcpy (mytext, "Yes, tables");
      break;
    case SUMMARY:
      strcpy (mytext, "Yes, summary");
      break;
    }
  if (options->mighist)
    fprintf (file,
	     "   Histogram of the frequency of migration events        %3s\n",
	     options->mighist ? "Yes" : " No");
  fprintf (file, "   Time of migration events are save in file %15.15s\n",
	   "mighistfile");

  fprintf (file, "   Profile likelihood: %-36.36s\n", mytext);
  if (options->profile != NONE)
    {
      switch (options->profilemethod)
	{
	case 'p':
	  fprintf (file, "             Percentile method\n");
	  break;
	case 'q':
	  fprintf (file, "             Quick method\n");
	  break;
	case 'f':
	  fprintf (file, "             Fast method\n");
	  break;
	case 'd':
	  fprintf (file, "             Discrete method\n");
	  break;
	case 's':
	  fprintf (file, "             Spline method\n");
	  break;
	default:
	  fprintf (file, "             UNKOWN method????\n");
	  break;
	}
      fprintf (file, "             with df=%li and for Theta and %s\n\n\n\n",
	       options->df, options->profileparamtype ? "M=m/mu" : "4Nm");
    }
}


void
set_param (world_fmt * world, data_fmt * data, option_fmt * options,
	   long locus)
{
  long i, ii, j, iitest = 0;
  double tmp;
  switch (options->thetaguess)
    {
    case RANDOMESTIMATE:
      if (world->numpop < options->numpop)
	{
	  usererror
	    ("There is a conflict between your menu/parmfile\nand your datafile: number of populations\nare not the same\n");
	}
      for (i = 0; i < world->numpop; i++)
	{
	  world->param0[i] = rannor (options->thetag[0], options->thetag[1]);
	  while (world->param0[i] < 0)
	    world->param0[i] =
	      rannor (options->thetag[0], options->thetag[1]);
	}
      break;
      //    case PARAMGRID:
    case OWN:
      if (world->numpop < options->numpop)
	{
	  usererror
	    ("There is a conflict between your menu/parmfile\nand your datafile: number of populations\nare not the same\n");
	}
      for (i = 0; i < world->numpop; i++)
	{
	  if (i < options->numthetag - 1)
	    ii = i;
	  else
	    ii = options->numthetag - 1;
	  if (options->thetag[ii] == 0.0)
	    world->param0[i] = SMALLEST_THETA;
	  else
	    {
	      world->param0[i] = options->thetag[ii];
	    }
	}
      break;

    case FST:
    default:
      for (i = 0; i < world->numpop; i++)
	{
	  if (world->fstparam[locus][i] > SMALLEST_THETA)
	    {
	      if (world->fstparam[locus][i] > 100)
		world->param0[i] = 1.0;
	      else
		world->param0[i] = world->fstparam[locus][i];
	    }
	  else
	    {
	      if (strchr (SEQUENCETYPES, options->datatype))
		{
		  world->param0[i] = 0.01;
		}
	      else
		{
		  world->param0[i] = 1.0;
		}
	    }
	}
      break;
    }
  switch (options->migrguess)
    {
    case RANDOMESTIMATE:
      for (i = world->numpop; i < world->numpop2; i++)
	{
	  world->param0[i] = rannor (options->mg[0], options->mg[1]);
	  while (world->param0[i] < 0)
	    world->param0[i] = rannor (options->mg[0], options->mg[1]);

	}
      break;
      //    case PARAMGRID:
    case OWN:
      for (i = 0; i < world->numpop; i++)
	{
	  for (j = 0; j < world->numpop; j++)
	    {
	      if (i == j)
		continue;
	      if ((iitest =
		   mm2m (j, i,
			 world->numpop) - world->numpop) < options->nummg)
		ii = iitest;
	      else
		ii = options->nummg - 1;
	      if (options->usem)
		tmp = options->mg[ii];
	      else
		tmp = options->mg[ii] / world->param0[i];
	      if (options->geo)
		{
		  world->param0[mm2m (j, i, world->numpop)] =
		    1. / data->ogeo[j][i] * tmp;
		}
	      else
		{
		  world->param0[mm2m (j, i, world->numpop)] = tmp;
		}
	    }
	}
      break;
    case SLATKIN:
    case FST:
    default:
      for (i = world->numpop; i < world->numpop2; i++)
	{
	  if (world->fstparam[locus][i] > 0)
	    {
	      if (world->fstparam[locus][i] > 100)
		{
		  world->param0[i] =
		    1.0 / world->param0[(i - world->numpop) /
					(world->numpop)];
		  if (world->param0[i] > 10000)
		    world->param0[i] = 10000;
		}
	      else
		world->param0[i] = world->fstparam[locus][i];
	    }
	  else
	    {
	      world->param0[i] =
		1.0 / world->param0[(i - world->numpop) / (world->numpop)];
	      if (world->param0[i] > 10000)
		world->param0[i] = 10000;
	    }

	}
      break;
    }
  synchronize_param (world, options);
  //options->thetag = (double *) realloc (options->thetag, 
  //                            sizeof (double) * (1+world->numpop));
  //memcpy(options->thetag,world->param0, sizeof(double)*world->numpop);
  //options->numthetag = world->numpop;
  //options->mg = (double *) realloc (options->mg, 
  //                                    sizeof (double) * (1+world->numpop2));
  //memcpy(options->mg,world->param0+world->numpop,
  // sizeof(double)*world->numpop*(world->numpop-1));
  //options->nummg = world->numpop * ( world->numpop-1);
  if (options->gamma)
    world->options->alphavalue = options->alphavalue;
  //perhaps to come  
  //if(options->thetaguess==PARAMGRID && options->migrguess==PARAMGRID)
  //set_grid_param(world,options->gridpoints);
}

void
set_grid_param (world_fmt * world, long gridpoints)
{
  static long which = 0;
  static long z = 0;

  static double level = 0.1;
  static double bottom = -2.302585093;	// => level =  0.1
  double top = 2.302585093;	// => level = 10
  double len = top - bottom;
  double diff = len / (gridpoints - 1.);

  if (z >= gridpoints)
    {
      z = 0;
      which++;
    }
  level = EXP (bottom + z * diff);
  world->param0[which] = world->param0[which] * level;
  z++;

}

void
synchronize_param (world_fmt * world, option_fmt * options)
{
  char type;
  boolean found = FALSE;
  long i, z = 0, zz = 0, zzz = 0, len;
  long ii, jj;
  long ns = 0, ss = 0, ss2 = 0, xs = 0, migm = 0;
  boolean allsymmig = FALSE;
  boolean allsamesize = FALSE;
  boolean partmig = FALSE;
  double summ;
  double nsum = 0;
  len = (long) strlen (world->options->custm2);
  world->options->custm2 =
    (char *) realloc (world->options->custm2,
		      sizeof (char) * (world->numpop2 + 2));
  if (len < world->numpop2)
    {
      fillup_custm (len, world, options);
    }
  migm = scan_m (world->options->custm2, world->numpop, world->numpop2);
  world->options->mmn = migm;
  for (i = 0; i < world->numpop2; i++)
    {
      if (!(allsymmig && allsamesize))
	{
	  type = world->options->custm2[i];
	  switch (type)
	    {
	    case '*':
	      xs++;
	      break;
	    case 's':		// M is symmetric
	      if (i >= world->numpop)
		{
		  z = i;
		  m2mm (z, world->numpop, &ii, &jj);
		  zz = mm2m (jj, ii, world->numpop);
		  world->options->symparam = (twin_fmt *) realloc
		    (world->options->symparam, sizeof (twin_fmt) * (ss + 2));
		  world->options->symparam[ss][0] = zz;
		  world->options->symparam[ss++][1] = z;
		  world->options->symn = ss;
		  summ = (world->param0[z] + world->param0[zz]) / 2.;
		  world->param0[zz] = world->param0[z] = summ;
		}
	      break;
	    case 'S':		// 4Nm is symmetric, not completely
	      // implemented yet, -> derivatives.c
	      if (i >= world->numpop)
		{
		  z = i;
		  m2mm (z, world->numpop, &ii, &jj);
		  zz = mm2m (jj, ii, world->numpop);
		  zzz = 0;
		  found = FALSE;
		  while (zzz < ss2)
		    {
		      if (world->options->sym2param[zzz][1] == zz)
			found = TRUE;
		      zzz++;
		    }
		  if (found)
		    break;
		  world->options->sym2param = (quad_fmt *)
		    realloc (world->options->sym2param,
			     sizeof (quad_fmt) * (ss2 + 2));
		  world->options->sym2param[ss2][0] = zz;
		  world->options->sym2param[ss2][1] = z;
		  world->options->sym2param[ss2][2] = ii;
		  world->options->sym2param[ss2++][3] = jj;
		  world->options->sym2n = ss2;
		  summ = (world->param0[z] * world->param0[jj] +
			  world->param0[zz] * world->param0[ii]) / 2.;
		  world->param0[z] = summ / world->param0[jj];
		  world->param0[zz] = summ / world->param0[ii];
		}
	      break;
	    case 'C':
	    case 'c':
	      world->options->constparam = (long *) realloc
		(world->options->constparam, sizeof (long) * (ns + 2));
	      world->options->constparam[ns++] = i;
	      world->options->constn = ns;
	      break;
	    case '0':
	      z = i;
	      world->param0[z] = 0;
	      world->options->zeroparam = (long *) realloc
		(world->options->zeroparam, sizeof (long) * (ns + 2));
	      world->options->zeroparam[ns++] = i;
	      world->options->zeron = ns;
	      break;
	    case 'm':
	      summ = 0;
	      if (i < world->numpop)	/*theta */
		{
		  if (!allsamesize)
		    {
		      nsum = 0;
		      allsamesize = TRUE;
		      for (z = 0; z < world->numpop; z++)
			{
			  if (world->options->custm2[z] == 'm')
			    {
			      nsum++;
			      summ += world->param0[z];
			    }
			}
		      summ /= nsum;
		      for (z = 0; z < world->numpop; z++)
			if (world->options->custm2[z] == 'm')
			  world->param0[z] = summ;
		    }
		}
	      else		/* migration */
		{
		  summ = 0;
		  if (!partmig)
		    {
		      nsum = 0;
		      partmig = TRUE;
		      for (z = world->numpop; z < world->numpop2; z++)
			{
			  if (world->options->custm2[z] == 'm')
			    {
			      nsum++;
			      summ += world->param0[z];
			    }
			}
		      summ /= nsum;
		      for (z = world->numpop; z < world->numpop2; z++)
			if (world->options->custm2[z] == 'm')
			  world->param0[z] = summ;
		    }
		}
	      break;
	    default:
	      error ("no defaults allowed in synchronize_param()\n");
	    }
	}
    }
  //--------gamma stuff
  if (world->options->gamma)
    {
      if (world->options->custm2[world->numpop2] == 'c')
	{
	  world->options->constparam = (long *) realloc
	    (world->options->constparam, sizeof (long) * (ns + 2));
	  world->options->constparam[ns++] = i;
	  world->options->constn = ns;
	}
      else
	{
	  world->options->custm2[world->numpop2] = '*';
	  world->options->custm2[world->numpop2 + 1] = '\0';
	}
    }
}


void
resynchronize_param (world_fmt * world)
{
  char type;
  long i, j = 0, z = 0, zz = 0, len;
  long ii, jj;
  long ns = 0, ss = 0, ss2 = 0, xs = 0, migm = 0;
  boolean allsymmig = FALSE;
  boolean allsamesize = FALSE;
  boolean partmig = FALSE;
  double summ;
  double nsum = 0;
  len = (long) strlen (world->options->custm2);
  world->options->symn = 0;
  world->options->sym2n = 0;
  world->options->zeron = 0;
  world->options->constn = 0;
  world->options->mmn = 0;
  migm = scan_m (world->options->custm2, world->numpop, world->numpop2);
  world->options->mmn = migm;
  for (i = 0; i < world->numpop2; i++)
    {
      if (!(allsymmig && allsamesize))
	{
	  type = world->options->custm2[i];
	  switch (type)
	    {
	    case '*':
	      xs++;
	      break;
	    case 's':		// M is symmetric
	      if (i >= world->numpop)
		{
		  z = i;
		  m2mm (z, world->numpop, &ii, &jj);
		  zz = mm2m (jj, ii, world->numpop);
		  world->options->symparam = (twin_fmt *) realloc
		    (world->options->symparam, sizeof (twin_fmt) * (ss + 2));
		  world->options->symparam[ss][0] = zz;
		  world->options->symparam[ss++][1] = z;
		  world->options->symn = ss;
		  summ = (world->param0[z] + world->param0[zz]) / 2.;
		  world->param0[zz] = world->param0[z] = summ;
		}
	      break;
	    case 'S':		// 4Nm is symmetric, not completely
	      // implemented yet, -> derivatives.c
	      if (i >= world->numpop)
		{
		  z = i;
		  m2mm (z, world->numpop, &ii, &jj);
		  zz = mm2m (jj, ii, world->numpop);
		  world->options->sym2param = (quad_fmt *)
		    realloc (world->options->sym2param,
			     sizeof (quad_fmt) * (ss2 + 2));
		  world->options->sym2param[ss2][0] = zz;
		  world->options->sym2param[ss2][1] = z;
		  world->options->sym2param[ss2][2] = i;
		  world->options->sym2param[ss2++][3] = j;
		  world->options->sym2n = ss2;
		  summ = (world->param0[z] * world->param0[i] +
			  world->param0[zz] * world->param0[j]) / 2.;
		  world->param0[z] = summ / world->param0[i];
		  world->param0[zz] = summ / world->param0[j];
		}
	      break;
	    case 'C':
	    case 'c':
	      world->options->constparam = (long *) realloc
		(world->options->constparam, sizeof (long) * (ns + 2));
	      world->options->constparam[ns++] = i;
	      world->options->constn = ns;
	      break;
	    case '0':
	      z = i;
	      world->param0[z] = 0;
	      world->options->zeroparam = (long *) realloc
		(world->options->zeroparam, sizeof (long) * (ns + 2));
	      world->options->zeroparam[ns++] = i;
	      world->options->zeron = ns;
	      break;
	    case 'm':
	      summ = 0;
	      if (i < world->numpop)	/*theta */
		{
		  if (!allsamesize)
		    {
		      allsamesize = TRUE;
		      nsum = 0;
		      for (z = 0; z < world->numpop; z++)
			{
			  if (world->options->custm2[z] == 'm')
			    {
			      nsum++;
			      summ += world->param0[z];
			    }
			}
		      summ /= nsum;
		      for (z = 0; z < world->numpop; z++)
			if (world->options->custm2[z] == 'm')
			  world->param0[z] = summ;
		    }
		}
	      else		/* migration */
		{
		  summ = 0;
		  if (!partmig)
		    {
		      nsum = 0;
		      partmig = TRUE;
		      for (z = world->numpop; z < world->numpop2; z++)
			{
			  if (world->options->custm2[z] == 'm')
			    {
			      nsum++;
			      summ += world->param0[z];
			    }
			}
		      summ /= nsum;
		      for (z = world->numpop; z < world->numpop2; z++)
			if (world->options->custm2[z] == 'm')
			  world->param0[z] = summ;
		    }
		}
	      break;
	    default:
	      error ("no defaults allowed in resynchronize_param()\n");
	    }
	}
    }
  //--------gamma stuff
  if (world->options->gamma)
    {
      if (world->options->custm2[world->numpop2] == 'c')
	{
	  world->options->constparam = (long *) realloc
	    (world->options->constparam, sizeof (long) * (ns + 2));
	  world->options->constparam[ns++] = i;
	  world->options->constn = ns;
	}
      else
	{
	  world->options->custm2[world->numpop2] = '*';
	  world->options->custm2[world->numpop2 + 1] = '\0';
	}
    }
}


long
scan_m (char *custm2, long start, long stop)
{
  long i, summ = 0;
  for (i = start; i < stop; i++)
    {
      if ('m' == custm2[i])
	summ++;
    }
  return summ;
}

void
set_partmean_mig (long **mmparam, double *param, char *custm2, long migm,
		  long numpop2)
{
  long i, z = 0;
  double summ = 0;
  long start = (long) sqrt ((double) numpop2);
  (*mmparam) = (long *) realloc ((*mmparam), sizeof (long) * (migm + 2));

  for (i = start; i < numpop2; i++)
    {
      if (custm2[i] == 'm')
	{
	  summ += param[i];
	  (*mmparam)[z++] = i;
	}
    }
  summ /= migm;
  for (i = start; i < numpop2; i++)
    {
      if (custm2[i] == 'm')
	param[i] = summ;
    }
}


void
destroy_options (option_fmt * options)
{
  free (options->thetag);
  free (options->mg);
  free (options->ttratio);
  free (options->rate);
  free (options->probcat);
  free (options->rrate);
  while (--options->lratio->counter >= 0)
    {
      free (options->lratio->data[options->lratio->counter].value1);
      free (options->lratio->data[options->lratio->counter].value2);
    }
  free (options->lratio->data);
  free (options->lratio);
  free (options->custm);
  if (options->plot)
    {
      free (options->plotxvalues);
      free (options->plotyvalues);
    }
  if (options->zeron > 0)
    free (options->zeroparam);
  if (options->constn > 0)
    free (options->constparam);
  if (options->symn > 0)
    free (options->symparam);
  if (options->sym2n > 0)
    free (options->sym2param);
  free (options);
}

void
decide_plot (worldoption_fmt * options, long chain, long chains, char type)
{
  if (options->plot && (chain >= chains - 1) && (type == 'l'))
    options->plotnow = TRUE;
  else
    options->plotnow = FALSE;
}

void
set_plot (option_fmt * options)
{
  long intervals = options->plotintervals;
  double prangex[2];
  double prangey[2];
  if (!options->plot)
    return;
  prangex[0] = options->plotrange[0];
  prangex[1] = options->plotrange[1];
  prangey[0] = options->plotrange[2];
  prangey[1] = options->plotrange[3];


  options->plotxvalues = (double *) calloc (1, sizeof (double) * intervals);
  options->plotyvalues = (double *) calloc (1, sizeof (double) * intervals);
  set_plot_values (&options->plotxvalues, prangex, options->plotintervals,
		   options->plotscale);
  set_plot_values (&options->plotyvalues, prangey, options->plotintervals,
		   options->plotscale);
}

void
set_plot_values (double **values, double plotrange[], long intervals,
		 short type)
{
  long i;
  double diff = 0;
  double logstart = 0;
  (*values)[0] = plotrange[0];
  (*values)[intervals - 1] = plotrange[1];
  if (type == PLOTSCALELOG)
    {
      logstart = log10 ((*values)[0]);
      diff =
	(log10 ((*values)[intervals - 1]) - logstart) / (double) (intervals -
								  1);
      for (i = 1; i < intervals - 1; i++)
	{
	  (*values)[i] = pow (10., (logstart + i * diff));
	}
    }
  else
    {
      diff =
	((*values)[intervals - 1] - (*values)[0]) / (double) (intervals - 1);
      for (i = 1; i < intervals - 1; i++)
	{
	  (*values)[i] = (*values)[i - 1] + diff;
	}
    }
}

long
save_parmfile (option_fmt * options, world_fmt * world)
{
  FILE *fp;
  long bufsize;
  char *sbuffer;
  char *buffer;
  sbuffer = (char *) calloc (MAXBUFSIZE, sizeof (char));
  buffer = sbuffer;
  fp = options->parmfile;
  if (fp)
    {
      fclose (fp);
      fp = fopen (PARMFILE, "w");
    }
  else
    fp = fopen (PARMFILE, "w");
  bufsize = save_options_buffer (&buffer, options, world);
  fprintf (fp, "%s", buffer);
  fflush (fp);
  printf ("\n\n+++ Parmfile written to current directory +++\n\n");
  free (sbuffer);
  return bufsize;
}


long
save_options_buffer (char **buffer, option_fmt * options, world_fmt * world)
{
  long num, i, j, z;
  char nowstr[LINESIZE] = "----";
  char fp[LINESIZE];
  long bufsize = 0;
  get_time (nowstr, "%c");
  sprintf (fp, "#########################################################\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
  sprintf (fp, "# Parmfile for Migrate %s\n", MIGRATEVERSION);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "# generated automatically on\n# %s\n", nowstr);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "#\n# please report problems to Peter beerli\n");
  bufsize += strlen(fp)+1; 
  *buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
  strcat (*buffer, fp);

  sprintf (fp, "#    email: beerli@genetics.washington.edu\n");
  bufsize += strlen(fp)+1; 
  *buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
  strcat (*buffer, fp);

  sprintf (fp, "#    http://evolution.genetics.washington.edu/lamarc.html\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "#########################################################\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "# General options ---------------------------------------\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "nmlength=%li\n", options->nmlength);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "# data options ------------------------------------------\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  switch (options->datatype)
    {
    case 'a':
      sprintf (fp, "datatype=AllelicData\n");
      break;
    case 'b':
      sprintf (fp, "datatype=BrownianMicrosatelliteData\n");
      break;
    case 'm':
      sprintf (fp, "datatype=MicrosatelliteData\n");
      break;
    case 's':
      sprintf (fp, "datatype=SequenceData\n");
      break;
    case 'n':
      sprintf (fp, "datatype=NucleotidePolymorphismData\n");
      break;
    case 'u':
      sprintf (fp, "datatype=UnlinkedSNPData\n");
      break;
    case 'f':
      sprintf (fp, "datatype=F-Ancestral state method\n");
      break;
    case 'g':
      sprintf (fp, "datatype=GenealogySummaryOlderRun\n");
      break;
    default:
      error ("the parmfile-writer contains an error");
    }
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
  switch (options->datatype)
    {
    case 'a':
      sprintf (fp, "# Electrophoretic data options------------\n");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
      break;
    case 'm':
      sprintf (fp, "# Microsatellite data options------------\n");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

      sprintf (fp, "micro-threshold=%li\n", options->micro_threshold);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
      //
      //sprintf (fp, "micro-max=%li\n", options->micro_stepnum);
      //strcat(buffer, fp); 
      break;
    case 'b':
      break;
    case 's':
    case 'n':
    case 'u':
    case 'f':
      sprintf (fp, "# Sequence data options------------\n");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
      sprintf (fp, "ttratio=%f ", options->ttratio[0]);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
      i = 1;
      while (options->ttratio[i] != 0.0)
	{
	  sprintf (fp, "%f ", options->ttratio[i++]);
	  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	}
      sprintf (fp, "\n");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

      if (options->freqsfrom)
	{
	  sprintf (fp, "freqs-from-data=YES\n");
	}
      else
	{
	  sprintf (fp, "freqs-from-data=NO:%f,%f, %f, %f\n", options->freqa,
		   options->freqc, options->freqg, options->freqt);
	}
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

      if (options->seqerror > 0.)
	{
	  sprintf (fp, "seqerror-rate=%f\n", options->seqerror);
	  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	}

      sprintf (fp, "categories=%li\nrates=%li:", options->categs,
	       options->rcategs);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
      for (i = 0; i < options->rcategs; i++)
	{
	  sprintf (fp, "%f ", options->rrate[i]);
	  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	}
      sprintf (fp, "\n");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
      sprintf (fp, "prob-rates=%li:", options->rcategs);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
      for (i = 0; i < options->rcategs; i++)
	{
	  sprintf (fp, "%f ", options->probcat[i]);
	  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	}
      sprintf (fp, "\n");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

      if (!options->autocorr)
	{
	  sprintf (fp, "autocorrelation=NO\n");
	  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	}
      else
	{
	  sprintf (fp, "autocorrelation=YES:%f\n", 1. / options->lambda);
	  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	}
      sprintf (fp, "weights=%s\n", options->weights ? "YES" : "NO");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

      sprintf (fp, "interleaved=%s\n", options->interleaved ? "YES" : "NO");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

      sprintf (fp, "fast-likelihood=%s\n", options->fastlike ? "YES" : "NO");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

      sprintf (fp, "usertree=%s\n", options->usertree ? "YES" : "NO");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

      if (options->randomtree)
	{
	  sprintf (fp, "randomtree=%s\n", "YES");
	  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	}
      sprintf (fp, "distfile=%s\n", options->dist ? "YES" : "NO");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
      break;
      //    defaults:
      //      break;
    }
#ifdef UEP
  sprintf (fp, "uepfile=%s\n", options->uep ? options->uepfilename : "NONE");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
  if (options->uep)
    {
      sprintf (fp, "uep-rates=%f:%f\n", options->uepmu,options->uepnu);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
    }
#endif
  sprintf (fp, "# input/output options ----------------------------------\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "menu=%s\n", options->menu ? "YES" : "NO ");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "# input formats\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "infile=%s\n", options->infilename);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  switch (options->autoseed)
    {
    case NOAUTO:
      sprintf (fp, "random-seed=SEEDFILE");
      break;
    case AUTO:
      sprintf (fp, "random-seed=AUTO #OWN:%li\n", options->inseed);
      break;
    case NOAUTOSELF:
      sprintf (fp, "random-seed=OWN:%li\n", options->inseed);
      break;
    default:
      error ("error in wrinting parmfile start seed method unknown");
    }
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
  if (strlen (options->title) > 0)
    {
      sprintf (fp, "title=%s\n", options->title);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
    }
  sprintf (fp, "# output formats\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "progress=%s\n",
	   options->progress ? (options->
				verbose ? "VERBOSE" : "YES") : "NO ");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "logfile=%s\n",
	   options->writelog ? options->logfilename : "NONE");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "print-data=%s\n", options->printdata ? "YES" : "NO");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "outfile=%s\n", options->outfilename);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
  if (options->usem)
    {
      sprintf (fp, "use-M=%s\n", options->usem ? "YES" : "NO");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
    }
  if (options->plot)
    {
      sprintf (fp, "plot=YES:%s:%s:",
	       options->plotmethod == PLOTALL ? "BOTH" : "OUTFILE",
	       options->plotscale == PLOTSCALELOG ? "LOG" : "STD");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

      sprintf (fp, "{%f,%f,%f,%f}:", options->plotrange[0],
	       options->plotrange[1], options->plotrange[2],
	       options->plotrange[3]);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

      sprintf (fp, "%1.1s%li\n", options->plotvar == PLOT4NM ? "N" : "M",
	       options->plotintervals);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
    }
  else
    {
      sprintf (fp, "plot=NO\n");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
    }
  switch (options->profile)
    {
    case ALL:
      sprintf (fp, "profile=ALL");
      break;
    case TABLES:
      sprintf (fp, "profile=TABLES");
      break;
    case SUMMARY:
      sprintf (fp, "profile=SUMMARY");
      break;
    case NONE:
    default:
      sprintf (fp, "profile=NONE\n");
      break;

    }
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
  if (options->profile != NONE)
    {
      switch (options->profilemethod)
	{
	case 's':
	  sprintf (fp, ":SPLINE\n");
	  break;
	case 'd':
	  sprintf (fp, ":DISCRETE\n");
	  break;
	case 'q':
	  sprintf (fp, ":QUICKANDDIRTY\n");
	  break;
	case 'f':
	  sprintf (fp, ":FAST\n");
	  break;
	case 'p':
	default:
	  sprintf (fp, ":PRECISE\n");
	  break;
	}
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
    }

  switch (options->treeprint)
    {
    case NONE:
      sprintf (fp, "print-tree=NONE\n");
      break;
    case ALL:
      sprintf (fp, "print-tree=ALL\n");
      break;
    case BEST:
      sprintf (fp, "print-tree=BEST\n");
      break;
    case LASTCHAIN:
      sprintf (fp, "print-tree=LASTCHAIN\n");
      break;
    default:
      sprintf (fp, "print-tree=NONE\n");
      break;
    }
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "mathfile=%s\n", options->mathfilename);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "write-summary=%s\n", options->writesum ? "YES" : "NO");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  if (options->writesum || options->readsum)
    {
      sprintf (fp, "sumfile=%s\n", options->sumfilename);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
    }
  sprintf (fp, "# likelihood-ratio test and AIC\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  for (i = 0; i < options->lratio->counter; i++)
    {
      if (options->lratio->data[i].type == MLE)
	sprintf (fp, "l-ratio=%s:%s\n", "MLE",
		 options->lratio->data[i].value1);
      else
	sprintf (fp, "l-ratio=%s:%s:%s\n", "ARBITRARY",
		 options->lratio->data[i].value1,
		 options->lratio->data[i].value2);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
    }
  if (options->aic)
    {
      if (options->fast_aic)
	sprintf (fp, "aic-modeltest=YES:FAST\n");
      else
	sprintf (fp, "aic-modeltest=YES\n");
    }
  else
    sprintf (fp, "aic-modeltest=NO\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
  sprintf (fp, "# parameter options ------------------------------------\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
  if (options->thetaguess == FST)
    {
      sprintf (fp, "theta=FST\n");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
    }
  else
    {
      sprintf (fp, "theta=Own:{");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
      if (options->numthetag == 0)
	{
	  if (strchr ("snupf", options->datatype))
	    {
	      sprintf (fp, "0.01}\n");
	      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	    }
	  else
	    {
	      sprintf (fp, "1}\n");
	      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	    }
	}
      for (i = 0; i < options->numthetag - 1; i++)
	{
	  sprintf (fp, "%f ", options->thetag[i]);
	  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	}
      sprintf (fp, "%f}\n", options->thetag[i]);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
    }
  if (options->migrguess == FST)
    {
      sprintf (fp, "migration=FST\n");
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
    }
  else
    {
      switch (options->nummg)
	{
	case 0:
	  sprintf (fp, "migration=Own:{1}\n");
	  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	  break;
	case 1:
	  sprintf (fp, "migration=Own:{%f}\n", options->mg[0]);
	  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	  break;
	default:
	  sprintf (fp, "migration=Own:{ ");
	  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	  z = 0;
	  num = (long) (1. + sqrt (4. * (double) options->nummg + 1.) / 2.);
	  for (i = 0; i < num; i++)
	    {
	      for (j = 0; j < num; j++)
		{
		  if (i == j)
		    {
		      sprintf (fp, "- ");
		      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
		    }
		  else
		    {
		      sprintf (fp, "%f ", options->mg[z++]);
		      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
		    }
		}
	    }
	  sprintf (fp, "}\n");
	  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	}
    }
  if (options->gamma)
    sprintf (fp, "mutation=%s:%f\n", "GAMMA", options->alphavalue);
  else
    sprintf (fp, "mutation=%s\n", "NOGAMMA");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "fst-type=%s\n", options->fsttype ? "THETA" : "MIGRATION");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "custom-migration={%s}\n", options->custm);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "geofile=%s\n", options->geo ? "YES" : "NO");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "# search strategies ------------------------------------\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "short-chains=%li\n", options->schains);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "short-inc=%li\n", options->sincrement);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "short-sample=%li\n", options->ssteps);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "long-chains=%li\n", options->lchains);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "long-inc=%li\n", options->lincrement);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "long-sample=%li\n", options->lsteps);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "#obscure options\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "burn-in=%li\n", options->burn_in);
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "mig-histogram=%s\n", options->mighist ? "YES" : "NO");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "heating=%s", options->heating ? "YES" : "NO\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  if (options->heating)
    {
      sprintf (fp, ":%li:{%f,", options->heating_interval, options->heat[0]);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

      for (i = 1; i < options->heated_chains - 1; i++)
	{
	  sprintf (fp, "%f,", options->heat[i]);
	  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
	}
      sprintf (fp, "%f}\n", options->heat[i]);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
    }
  //(options->heating == 1 ? ":SimpleMethod" :
  //                           ":ComplexMethod") : "");
  if (options->movingsteps)
    {
      sprintf (fp, "moving-steps=YES:%f\n", options->acceptfreq);
    }
  else
    {
      sprintf (fp, "moving-steps=NO\n");
    }
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
  if (options->lcepsilon < LONGCHAINEPSILON)
    sprintf (fp, "long-chain-epsilon=%f\n", options->lcepsilon);
  else
    sprintf (fp, "long-chain-epsilon=INFINITY\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "gelman-convergence=%s\n", options->gelman ? "Yes" : "No");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  sprintf (fp, "replicate=%s:", options->replicate ? "Yes" : "No");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  if (options->replicatenum == 0)
    {
      sprintf (fp, "LastChains\n");
    }
  else
    {
      sprintf (fp, "%li\n", options->replicatenum);
    }
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);

  if (options->cpu > 1)
    {
      sprintf (fp, "cpu=%i\n", options->cpu);
      bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
    }
  sprintf (fp, "end\n");
  bufsize += strlen(fp)+1; 
*buffer = (char *) realloc(*buffer,bufsize * sizeof(char));
strcat (*buffer, fp);
  return bufsize;
}

/*private functions============================================= */

long
boolcheck (char ch)
{
  char c = uppercase (ch);
  if ((c == 'F') || (c == 'N'))
    return 0;
  else if ((c == 'T') || (c == 'Y'))
    return 1;
  else
    return -1;
}				/* boolcheck */

boolean
booleancheck (option_fmt * options, char *var, char *value)
{
  long i, check, z = 0;
  char *booltokens[NUMBOOL] = BOOLTOKENS;
  char *tmp;
  long ltemp;
  check = boolcheck (value[0]);
  if (check == -1)
    return FALSE;
  i = 0;
  while (i < NUMBOOL && strcmp (var, booltokens[i]))
    i++;
  switch ((short) i)
    {
    case 0:			/*menu = <yes | no> */
      options->menu = (boolean) (check);
      break;
    case 1:			/*interleaved =<yes | no> */
      options->interleaved = (boolean) (check);
      break;
    case 2:			/*print-data = <yes | no> */
      options->printdata = (boolean) (check);
      break;
    case 3:			/* mixplot=<yes | no> */
      options->mixplot = (boolean) (check);
      break;
    case 4:			/* moving-steps = <yes | no> */
      options->movingsteps = (boolean) (check);
      if (options->movingsteps)
	{
	  strtok (value, ":");
	  tmp = strtok (NULL, " ,\n");
	  if (tmp != NULL)
	    options->acceptfreq = atof ((char *) tmp);
	  else
	    options->acceptfreq = 0.1;
	}
      break;
    case 5:			/* freqs-from-data =  <yes | no> */
      options->freqsfrom = (boolean) (check);
      if (!options->freqsfrom)
	{
	  strtok (value, ":");
	  tmp = strtok (NULL, " ,");
	  if (tmp != NULL)
	    options->freqa = atof ((char *) tmp);
	  tmp = strtok (NULL, " ,");
	  if (tmp != NULL)
	    options->freqc = atof ((char *) tmp);
	  tmp = strtok (NULL, " ,");
	  if (tmp != NULL)
	    options->freqg = atof ((char *) tmp);
	  tmp = strtok (NULL, " ,\n");
	  if (tmp != NULL)
	    options->freqt = atof ((char *) tmp);
	}
      break;
    case 6:			/* usertree =  <yes | no> (only for sequence data) */
      options->usertree = (boolean) (check);
      break;
    case 7:
      {				/* autocorrelation=<YES:value | NO> */
	options->autocorr = (boolean) (check);
	if (options->autocorr)
	  {
	    strtok (value, ":");
	    tmp = strtok (NULL, " ;\n");
	    if (tmp != NULL)
	      options->lambda = 1.0 / atof ((char *) tmp);
	  }
	break;
      }
    case 8:			/* simulation =  <yes | no> */
      options->simulation = (boolean) check;
      break;
    case 9:			/* plot =  <not | yes:<outfile | both><:std|:log><:{xs,xe,ys,ye}<:N|:M><#of_intervals>>> */
      options->plot = (boolean) check;
      if (options->plot)
	{
	  strtok (value, ":");
	  if (toupper (value[0]) == 'Y' || toupper (value[0]) == 'Y')
	    {
	      tmp = strtok (NULL, ":;\n");
	      if (tmp == NULL)
		options->plotmethod = PLOTALL;
	      else
		{
		  switch (lowercase (tmp[0]))
		    {
		    case 'o':
		      options->plotmethod = PLOTOUTFILE;
		      break;
		    case 'b':
		      options->plotmethod = PLOTALL;
		      break;
		    default:
		      options->plotmethod = PLOTALL;
		      break;
		    }
		  tmp = strtok (NULL, ":;\n");
		  if (tmp != NULL)
		    {
		      switch (lowercase (tmp[0]))
			{
			case 'l':
			  options->plotscale = PLOTSCALELOG;
			  break;
			case 's':
			  options->plotscale = PLOTSCALESTD;
			  break;
			default:
			  options->plotscale = PLOTSCALELOG;
			}
		      tmp = strtok (NULL, ":;\n");
		      if (4 !=
			  sscanf (tmp, "{%lf,%lf,%lf,%lf",
				  &options->plotrange[0],
				  &options->plotrange[1],
				  &options->plotrange[2],
				  &options->plotrange[3]))
			sscanf (tmp, "{%lf%lf%lf%lf", &options->plotrange[0],
				&options->plotrange[1],
				&options->plotrange[2],
				&options->plotrange[3]);
		      tmp = strtok (NULL, ":;\n");
		      if (tmp != NULL)
			{
			  switch (lowercase (tmp[0]))
			    {
			    case 'm':
			      options->plotvar = 1;
			      while (!isdigit (*tmp) && *tmp != '\0')
				tmp++;
			      if ((ltemp =
				   strtol (tmp, (char **) NULL, 10)) > 0)
				options->plotintervals = ltemp;
			      break;
			    case 'n':
			    default:
			      options->plotvar = PLOT4NM;
			      while (!isdigit (*tmp) && *tmp != '\0')
				tmp++;
			      if ((ltemp =
				   strtol (tmp, (char **) NULL, 10)) > 0)
				options->plotintervals = ltemp;

			      break;
			    }
			}
		    }
		}
	    }
	}
      break;
    case 10:			/* weights =  <yes | no> */
      options->weights = (boolean) check;
      break;
    case 11:			/* read-summary  <yes | no> */
      options->readsum = (boolean) check;
      options->datatype = 'g';
      break;
    case 12:			/* write-summary =  <yes | no> */
      options->writesum = (boolean) check;
      break;
    case 13:			/* mig-histogram=<yes | no> */
      options->mighist = (boolean) check;
      break;
    case 14:			/* heating=<yes  | yes:<simple | complex> | no> */
      options->heating = (short) check;
      if (options->heating == 1)
	{
	  strtok (value, ":\n");
	  tmp = strtok (NULL, ":");
	  if (tmp != NULL)
	    {
	      options->heating_interval = atol (tmp);
	      tmp = strtok (NULL, "{,");
	      if (tmp != NULL)
		{
		  z = 0;
		  while (1)
		    {
		      options->heat[z++] = atof (tmp);
		      tmp = strtok (NULL, ", :\n");
		      if (tmp == NULL || z >= 20)
			break;
		    }
		  options->heated_chains = z;
		}
	    }
	}
      break;
    case 15:			/* print-fst =  <yes | no> */
      options->printfst = (boolean) check;
      break;
    case 16:			/* distfile =  <yes | no> */
      options->dist = (boolean) check;
      break;
    case 17:			/* geofile =  <yes | no> */
      options->geo = (boolean) check;
      break;
    case 18:			/* gelman-convergence =  <yes | no> */
      options->gelman = (boolean) check;
      break;
    case 19:			/* randomtree start =  <yes | no> */
      options->randomtree = (boolean) check;
      break;
    case 20:			/* fast like calculator =  <yes | no> */
      options->fastlike = (boolean) check;
      break;
    case 21:			/*aic-modeltest=yes/no */
      options->aic = (boolean) check;
      if (options->aic)
	{
	  strtok (value, ":\n");
	  tmp = strtok (NULL, ":");
	  if (tmp != NULL)
	    {
	      if (toupper (tmp[0]) == 'F')
		{
		  options->fast_aic = TRUE;
		  tmp = strtok (NULL, ":");
		  if (tmp != NULL)
		    {
		      options->aicmod = atof (tmp);
		    }
		  else
		    options->aicmod = 2.;
		}
	      else
		options->aicmod = atof (tmp);
	    }
	  else
	    options->aicmod = 2.;
	}
      else
	options->aicmod = 2.;
      break;
    case 22:			//use-M=true/false
      options->usem = (boolean) check;
      set_usem_related (options);
      break;
    case 23:			//alternative to above is use-4Nm=false/true
      options->usem = !(boolean) check;
      set_usem_related (options);
      break;
    default:
      return FALSE;
    }
  return TRUE;
}				/* booleancheck */

void
set_usem_related (option_fmt * options)
{
  if (options->usem)
    {
      options->plotvar = PLOTM;
      options->migvar = PLOTM;
      options->profileparamtype = PLOTM;

    }
  else
    {
      options->plotvar = PLOT4NM;
      options->migvar = PLOT4NM;
      options->profileparamtype = PLOT4NM;
    }

}

boolean
numbercheck (option_fmt * options, char *var, char *value)
{
  long i = 0, z, cc = 0;
  char *tmp, *temp, *temp2, *keeptmp;
  char *numbertokens[NUMNUMBER] = NUMBERTOKENS;
  tmp = (char *) calloc (1, sizeof (char) * LINESIZE);
  keeptmp = tmp;
  while (i < NUMNUMBER && strcmp (var, numbertokens[i]))
    i++;
  switch ((short) i)
    {
    case 0:			/*ttratio = value */
      z = 0;
      temp = strtok (value, " ,;\n\0");
      while (temp != NULL)
	{
	  options->ttratio[z++] = atof (temp);
	  options->ttratio =
	    (double *) realloc (options->ttratio, sizeof (double) * (z + 1));
	  options->ttratio[z] = 0.0;

	  temp = strtok (NULL, " ,;\n\0");
	}
      break;
    case 1:			/*short-chains = value */
      options->schains = atol (value);
      break;
    case 2:			/*short-steps = value */
    case 32:			/* short-sample = value */
      options->ssteps = atol (value);
      break;
    case 3:			/*short-increment = value */
      options->sincrement = atol (value);
      break;
    case 4:			/*long-chains = value */
      options->lchains = atol (value);
      break;
    case 5:			/*long-steps = value */
    case 33:			/*long-sample = value */
      options->lsteps = atol (value);
      break;
    case 6:			/*long-increment = value */
      options->lincrement = atol (value);
      break;
    case 7:
      break;			/* theta: already handled in read_theta() */
    case 8:			/*nmlength = value */
      options->nmlength = strtol (value, (char **) NULL, 10);	//atoi (value);
      break;
    case 9:			/* seed = <Auto | seedfile | Own:value> */
      switch (value[0])
	{
	case 'A':
	case 'a':
	case '0':
	  options->autoseed = AUTO;
	  options->inseed = (long) time (0) / 4 + 1;
	  break;
	case 'S':
	case 's':
	case '1':
	  options->autoseed = NOAUTO;
	  options->seedfile = fopen ("seedfile", "r");
	  if (options->seedfile == NULL)
	    {
	      usererror ("cannot find seedfile\n");
	    }
	  fscanf (options->seedfile, "%ld%*[^\n]", &options->inseed);
	  fclose (options->seedfile);
	  break;
	case 'O':
	case 'o':
	case '2':
	  options->autoseed = NOAUTOSELF;
	  strtok (value, ":");
	  tmp = strtok (NULL, " ;\n");
	  if (tmp != NULL)
	    options->inseed = atol ((char *) tmp);
	  if (options->inseed > 0)
	    break;
	default:
	  options->autoseed = AUTO;
	  options->inseed = (long) time (0) / 4 + 1;
	  usererror ("Failure to read seed method, should be\n \
random-seed=auto or random-seed=seedfile or random-seed=own:value\nwhere value is a positive integer\nUsing AUTOMATIC seed=%li\n", options->inseed);
	  break;
	}
      break;
    case 10:
      break;			/*"migration" fake: this is already handled in read_migrate */
    case 11:			/*mutation= <auto=gamma | nogamma> */
      switch (value[0])
	{
	case 'A':		/*automatic */
	case 'a':
	case 'G':
	case 'g':
	  options->gamma = TRUE;
	  temp = strtok (value, " :");
	  temp = strtok (NULL, " ,;\n");
	  if (temp != NULL)
	    options->alphavalue = atof (temp);
	  else
	    options->alphavalue = START_ALPHA;
	  break;
	case 'N':		/*none, all loci have same mu */
	case 'n':
	  options->gamma = FALSE;
	  break;
	default:
	  break;
	}
      break;
    case 12:			/*datatype=<allele|microsatellite|brownian|sequence|f-ancestral states|genealogies> */
      switch (value[0])
	{
	case 'a':
	case 'A':
	  options->datatype = 'a';
	  break;
	case 'm':
	case 'M':
	  options->datatype = 'm';
	  break;
	case 'b':
	case 'B':
	  options->datatype = 'b';
	  break;
	case 's':
	case 'S':
	  options->datatype = 's';
	  break;
	case 'n':
	case 'N':
	  options->datatype = 'n';
	  break;
	case 'u':
	case 'U':
	  options->datatype = 'u';
	  break;
	case 'f':
	case 'F':
	  options->datatype = 'f';
	  break;
	case 'g':
	case 'G':
	  options->datatype = 'g';
	  options->readsum = TRUE;
	  break;
	default:
	  options->datatype = 's';
	  break;
	}
      break;
    case 13:			/* categories=<None | Two=Yes | value> */
      if (toupper (value[0] == 'N'))
	{
	  options->categs = ONECATEG;
	  break;
	}
      if ((toupper (value[0]) == 'Y') || (toupper (value[0] == 'T')))
	options->categs = MANYCATEGS;
      else
	options->categs = strtol (value, (char **) NULL, 10);

      /* needs to read auxilliary file catfile */
      break;
    case 14:			/*create rates=value:list of rates */
      strncpy (tmp, value, strcspn (value, ":"));
      if (strtol (tmp, (char **) NULL, 10) /*;atoi (tmp) */  > 1)
	{			/* rate categories */
	  options->rcategs = strtol (tmp, (char **) NULL, 10);
	  options->rrate =
	    (double *) realloc (options->rrate,
				sizeof (double) * (options->rcategs + 1));
	  temp = strtok (value, " :");
	  temp = strtok (NULL, " ,;\n");
	  z = 0;
	  while (temp != NULL)
	    {
	      if (z > options->rcategs)
		usererror ("check parmfile-option  rates, missing rate\n");
	      options->rrate[z++] = atof (temp);
	      temp = strtok (NULL, " ,;\n");
	    }
	}
      break;
    case 15:			/* probabilities for each rate category */
      strncpy (tmp, value, strcspn (value, ":"));
      if (strtol (tmp, (char **) NULL, 10) > 1)
	{			/* probabilities for each rate category */
	  options->rcategs = strtol (tmp, (char **) NULL, 10);
	  options->probcat =
	    (double *) realloc (options->probcat,
				sizeof (double) * (options->rcategs + 1));
	  temp = strtok (value, " :");
	  temp = strtok (NULL, " ,;\n");
	  z = 0;
	  while (temp != NULL)
	    {
	      if (z > options->rcategs)
		usererror
		  ("check parmfile prob-rates, missing rate probability\n");
	      options->probcat[z++] = atof (temp);
	      temp = strtok (NULL, " ,;\n");
	    }
	}
      break;
    case 16:			/*micro-stepmax */
      // options->micro_stepnum = strtol (value, (char **) NULL, 10);

      break;
    case 17:			/*micro-threshold */
      options->micro_threshold = strtol (value, (char **) NULL, 10);
      break;
    case 18:			/*delimiter */
      options->dlm = value[0];
      break;
    case 19:			/*burn-in */
      options->burn_in = atol (value);
      break;
    case 20:			/*infilename */
      strcpy (options->infilename, value);
      break;
    case 21:			/*outfilename */
      strcpy (options->outfilename, value);
      break;
    case 22:			/*mathfilename */
      strcpy (options->mathfilename, value);
      break;
    case 23:			/*title */
      strncpy (options->title, value, 80);
      break;
    case 24:			/*long-chain-epsilon */
      options->lcepsilon = atof (value);
      if (options->lcepsilon <= 0)
	options->lcepsilon = LONGCHAINEPSILON;
      break;
    case 25:			/* print tree options */
      switch (uppercase (value[0]))
	{
	case 'N':
	  options->treeprint = NONE;
	  break;
	case 'A':
	  options->treeprint = ALL;
	  break;
	case 'B':
	  options->treeprint = BEST;
	  break;
	case 'L':
	  options->treeprint = LASTCHAIN;
	  break;
	default:
	  options->treeprint = NONE;
	  break;
	}
      break;
    case 26:			/* progress: No, Yes, Verbose */
      switch (uppercase (value[0]))
	{
	case 'F':
	case 'N':
	  options->progress = FALSE;
	  options->verbose = FALSE;
	  break;
	case 'T':
	case 'Y':
	  options->progress = TRUE;
	  options->verbose = FALSE;
	  break;
	case 'V':
	  options->progress = TRUE;
	  options->verbose = TRUE;
	  break;
	}
      break;
    case 27:			/* l-ratio: <none | mle | arbitrary>:val1,val2,val3,val4,val5 */
      cc = options->lratio->counter;
      switch (uppercase (value[0]))
	{
	case 'M':
	  options->lratio->data[cc].type = MLE;
	  break;
	case 'A':
	  options->lratio->data[cc].type = ARBITRARY;
	  break;
	case 'N':
	default:
	  free (keeptmp);
	  return FALSE;
	}
      temp = strtok (value, ":");
      temp = strtok (NULL, "\n");
      if (temp != NULL)
	{
	  temp2 = strchr (temp, ':');
	  if (temp2 != NULL)
	    {
	      strcpy (options->lratio->data[cc].value2, temp2);
	      *temp2 = '\0';
	      strcpy (options->lratio->data[cc].value1, temp);
	    }
	  else
	    strcpy (options->lratio->data[cc].value1, temp);
	}
      if (cc + 1 == options->lratio->alloccounter)
	{
	  options->lratio->alloccounter += 2;
	  options->lratio->data =
	    (lr_data_fmt *) realloc (options->lratio->data,
				     sizeof (lr_data_fmt) *
				     options->lratio->alloccounter);
	  for (i = cc + 1; i < options->lratio->alloccounter; i++)
	    {
	      options->lratio->data[i].elem = 0;
	      options->lratio->data[i].value1 =
		(char *) calloc (1, sizeof (char) * LINESIZE);
	      options->lratio->data[i].elem = 0;
	      options->lratio->data[i].value2 =
		(char *) calloc (1, sizeof (char) * LINESIZE);
	    }
	}
      cc = ++options->lratio->counter;
      break;
    case 28:			/* fst-type: <Theta | Migration> */
      switch (uppercase (value[0]))
	{
	case 'T':
	  options->fsttype = 'T';
	  break;
	case 'M':
	default:
	  options->fsttype = 'M';
	  break;
	}
      fst_type (options->fsttype);
      break;
    case 29:			/*profile=<NO| NONE | YES | ALL | TABLES | SUMMARY>><: <FAST |  */
      switch (uppercase (value[0]))
	{
	case 'S':
	  options->profile = SUMMARY;
	  break;
	case 'Y':
	case 'A':
	  options->profile = ALL;
	  break;
	case 'N':
	  options->profile = NONE;
	  break;
	case 'T':
	  options->profile = TABLES;
	  break;
	default:		/*A */
	  options->profile = ALL;
	  break;
	}
      temp = strtok (value, ":;\n");
      temp = strtok (NULL, ":;\n");
      if (temp != NULL)
	{
	  switch (lowercase (temp[0]))
	    {
	    case 'p':		/*precise percentiles */
	      options->profilemethod = 'p';
	      break;
	    case 'd':		/*discrete steps see at start of file */
	      options->profilemethod = 'd';
	      break;
	      //case 's':              
	      //options->profilemethod = 's';
	      //break;
	    case 'x':		/* x-rated */
	    case 'u':		/* uncorrelated */
	    case 'q':		/* quick and dirty */
	      options->profilemethod = 'q';
	      break;
	    case 'f':		/* quick and exact mixture */
	      options->profilemethod = 'f';
	      break;
	    default:
	      options->profilemethod = 'f';
	      options->printprofsummary = TRUE;
	      break;
	    }
	  temp = strtok (NULL, ":;\n");
	  if (temp != NULL)
	    {
	      switch (lowercase (temp[0]))
		{
		case 'm':
		  options->profileparamtype = 1;
		  break;
		default:
		  options->profileparamtype = PLOT4NM;
		}
	    }
	}
      set_profile_options (options);
      break;
    case 30:			/* custom-migration:<{> migration matrix and theta on
				   diagonal:
				   0 means not estimated,
				   x means estimated, s means symmetrically
				   estimated, m means all are the same  <}> */
      if (myID == MASTER)
	read_custom_migration (options->parmfile, options, value,
			       options->numpop);
#ifdef MPI
      else
	read_custom_migration_worker (options->buffer, options, value,
				      options->numpop);
#endif
      break;
    case 31:			/*sumfilename */
      strcpy (options->sumfilename, value);
      break;
      /*case 32 and case 33 are fallthroughs to 2 and 3 */
    case 34:			/*replicate */
      switch (uppercase (value[0]))
	{
	case 'T':
	case 'Y':
	  options->replicate = TRUE;
	  temp = strtok (value, ":;\n");
	  if (temp != NULL)
	    {
	      temp = strtok (NULL, ":;\n");
	      if (uppercase (temp[0]) == 'L')
		options->replicatenum = 0;
	      else
		options->replicatenum = strtol (temp, (char **) NULL, 10);
	    }
	  else
	    options->replicatenum = 0;
	  break;
	default:
	  options->replicate = FALSE;
	  options->replicatenum = 0;
	}
      break;
    case 35:			/* cpu number */
      options->cpu = (short) ATOI (value);
      break;
    case 36:			/* do we write a logfile or not */
      if (strcmp (value, "NONE"))
	{
	  options->writelog = TRUE;
	  strncpy (options->logfilename, value, 255);
	}
      break;
    case 37:			/* sequencing error */
      options->seqerror = atof (value);
      if (options->seqerror < 0.0)
	{
	  warning
	    ("Sequencing error was misspecified in parmfile, reset to 0.0\n");
	  options->seqerror = 0.0;
	}
      break;
#ifdef UEP
    case 38:			/* do we have a uep file or not */
      if (strcmp (value, "NONE"))
	{
	  options->uep = TRUE;
	  strncpy (options->uepfilename, value, 255);
	}
      break;
    case 39:			/* do we have uep-rates */
      temp = strtok (value, ":;\n");
      if (temp != NULL)
	{
	  options->uepmu = atof (temp);
	  temp = strtok (NULL, ":;\n");
	  if (temp != NULL)
	    {
	      options->uepnu = atof (temp);
	    }
	}
      break;
#endif
    default:
      free (keeptmp);
      return FALSE;

    }
  free (keeptmp);

  return TRUE;
}				/* numbercheck */

void
reset_oneline (option_fmt * options, long position)
{
  fseek (options->parmfile, position, SEEK_SET);
}


void
read_theta (option_fmt * options)
{
  //  char parmvar[LINESIZE];
  char tmp[LINESIZE];
  char varvalue[LINESIZE];
  char ch;

  long i = 0;

  while ((ch = getc (options->parmfile)) != '=')
    ;
  //  {
  //    parmvar[i++] = ch;
  //  }
  //i = 0;
  ch = getc (options->parmfile);
  while (!isspace ((int) ch) && ch != ':' && ch != '{')
    {
      varvalue[i++] = ch;
      ch = getc (options->parmfile);
    }
  switch (toupper (varvalue[0]))
    {
    case 'R':
      options->thetaguess = RANDOMESTIMATE;
      options->numthetag = 2;
      options->thetag =
	(double *) realloc (options->thetag, sizeof (double) * 3);
      fscanf (options->parmfile, "{%lf,%lf}\n", &options->thetag[0],
	      &options->thetag[1]);
      break;
    case 'F':
    case '_':
      options->thetaguess = FST;
      break;
      //case 'G':
    case 'O':
    case '0':
      //perhaps to come
      //      if(varvalue[0]=='G')
      //options->thetaguess = PARAMGRID;
      //else
      options->thetaguess = OWN;
      ch = skip_space (options);
      if (ch == '\0')
	return;
      if (ch == '{')
	{

	  while (ch != '}')
	    {
	      i = 0;
	      ch = skip_space (options);
	      if (ch == '\0')
		return;
	      while (ch != ' ' && ch != ',' && ch != '}')
		{
		  tmp[i++] = ch;
		  ch = getc (options->parmfile);
		}
	      tmp[i] = '\0';
	      options->thetag[options->numthetag] = atof (tmp);
	      options->numthetag += 1;
	      options->thetag =
		(double *) realloc (options->thetag,
				    sizeof (double) * (1 +
						       options->numthetag));

	    }
	}
      else
	{
	  i = 0;
	  tmp[i++] = ch;
	  while (!isspace (ch))
	    {
	      tmp[i++] = ch;
	      ch = getc (options->parmfile);
	    }
	  tmp[i] = '\0';
	  options->thetag[options->numthetag] = atof (tmp);
	  options->numthetag += 1;
	  options->thetag =
	    (double *) realloc (options->thetag,
				sizeof (double) * (1 + options->numthetag));
	}
      options->numpop = options->numthetag;
      if (options->numthetag == 0)
	{
	  warning ("You forgot to add your guess value:\n");
	  warning ("Theta=Own:{pop1,pop2, ...}\n");
	  warning ("or Theta=Own:guess_pop (same value for all)\n");
	}
      break;
    default:
      usererror
	("Failure to read start theta method, should be\ntheta=FST or theta=Own:x.x\n or theta=Own:{x.x, x.x , x.x, .....}");
    }

}


void
read_mig (option_fmt * options)
{
  //  char parmvar[LINESIZE];
  char tmp[LINESIZE];
  char varvalue[LINESIZE];
  char ch;
  long test = 0;
  long i = 0;
  /* 1st example:  1.0 (n-island model)
     2nd example: {1.0} (migration matrix model, all the same start values  
     3rd example: the dashes on the diagonal are NECESSARY, {} are facultativ
     -  1.0 0.1
     1.0  -  2.0
     0.9 1.2  -
     to specify real 0.0 you need to use the custom-migration settings.
     0.0 in the table will be change to SMALLES_MIGRATION
   */

  while ((ch = getc (options->parmfile)) != '=')
    {
      //      parmvar[i++] = ch;
    }
  i = 0;
  ch = getc (options->parmfile);
  while (!isspace ((int) ch) && ch != ':' && ch != '{')
    {
      varvalue[i++] = ch;
      ch = getc (options->parmfile);
    }
  switch (toupper (varvalue[0]))
    {
    case 'R':
      options->migrguess = RANDOMESTIMATE;
      options->nummg = 2;
      options->mg = (double *) realloc (options->mg, sizeof (double) * 3);
      fscanf (options->parmfile, "{%lf,%lf}\n", &options->mg[0],
	      &options->mg[1]);
      break;
    case 'F':
    case '_':
      options->migrguess = FST;
      break;
      //    case 'G':
    case 'O':
    case '0':
      //      if(varvalue[0]=='G')
      //       options->migrguess = PARAMGRID;
      //      else
      options->migrguess = OWN;
      ch = skip_space (options);
      if (ch == '\0')
	return;
      if (ch == '{')
	{
	  options->migration_model = MATRIX;
	  while (ch != '}')
	    {
	      ch = skip_space (options);
	      if ((ch == '\0') || (ch == '}'))
		return;
	      i = 0;
	      while (!isspace (ch) && ch != ',' && ch != '}')
		{
		  tmp[i++] = ch;
		  ch = getc (options->parmfile);
		}
	      tmp[i] = '\0';
	      if (strcmp (tmp, "-"))
		{
		  options->mg[options->nummg] = atof (tmp);
		  options->nummg += 1;
		  options->mg =
		    (double *) realloc (options->mg,
					sizeof (double) * (1 +
							   options->nummg));
		}
	      else
		{
		  test++;
		}
	    }
	  options->numpop = test;
	}
      else
	{
	  options->migration_model = ISLAND;
	  i = 0;
	  options->numpop = 1;
	  tmp[i++] = ch;
	  while (!isspace (ch))
	    {
	      tmp[i++] = ch;
	      ch = getc (options->parmfile);
	    }
	  options->mg[options->nummg] = atof (tmp);
	  options->nummg += 1;
	}
      if (options->nummg == 0)
	{
	  warning ("You forgot to add your guess value, use either:\n");
	  warning ("migration=FST\n");
	  warning ("migration=Own:{migration matrix, diagonal is -}\n");
	  usererror
	    ("migration=Own:{migration value}, all matrix elements have the same value\n");
	}
      break;
    default:
      usererror ("Failure to read start migration method\n");
    }

}

#ifdef MPI
void
read_theta_worker (char **buffer, option_fmt * options)
{
  char tmp[LINESIZE];
  char varvalue[LINESIZE];
  char ch;

  long i = 0;

  while ((ch = sgetc (buffer)) != '=')
    ;
  ch = sgetc (buffer);
  while (!isspace ((int) ch) && ch != ':' && ch != '{')
    {
      varvalue[i++] = ch;
      ch = sgetc (buffer);
    }
  switch (toupper (varvalue[0]))
    {
    case 'R':
      options->thetaguess = RANDOMESTIMATE;
      options->numthetag = 2;
      options->thetag =
	(double *) realloc (options->thetag, sizeof (double) * 3);
      sgets (tmp, LINESIZE, buffer);
      sscanf (tmp, "{%lf,%lf}\n", &options->thetag[0], &options->thetag[1]);
      break;
    case 'F':
    case '_':
      options->thetaguess = FST;
      break;
      //case 'G':
    case 'O':
    case '0':
      //perhaps to come
      //      if(varvalue[0]=='G')
      //options->thetaguess = PARAMGRID;
      //else
      options->thetaguess = OWN;
      ch = skip_sspace (buffer);
      if (ch == '\0')
	return;
      if (ch == '{')
	{

	  while (ch != '}')
	    {
	      i = 0;
	      ch = skip_sspace (buffer);
	      if (ch == '\0')
		return;
	      while (ch != ' ' && ch != ',' && ch != '}')
		{
		  tmp[i++] = ch;
		  ch = sgetc (buffer);
		}
	      tmp[i] = '\0';
	      options->thetag[options->numthetag] = atof (tmp);
	      options->numthetag += 1;
	      options->thetag =
		(double *) realloc (options->thetag,
				    sizeof (double) * (1 +
						       options->numthetag));

	    }
	}
      else
	{
	  i = 0;
	  tmp[i++] = ch;
	  while (!isspace (ch))
	    {
	      tmp[i++] = ch;
	      ch = sgetc (buffer);
	    }
	  tmp[i] = '\0';
	  options->thetag[options->numthetag] = atof (tmp);
	  options->numthetag += 1;
	  options->thetag =
	    (double *) realloc (options->thetag,
				sizeof (double) * (1 + options->numthetag));
	}
      options->numpop = options->numthetag;
      if (options->numthetag == 0)
	{
	  warning ("You forgot to add your guess value:\n");
	  warning ("Theta=Own:{pop1,pop2, ...}\n");
	  warning ("or Theta=Own:guess_pop (same value for all)\n");
	}
      break;
    default:
      usererror
	("Failure to read start theta method, should be\ntheta=FST or theta=Own:x.x\n or theta=Own:{x.x, x.x , x.x, .....}");
    }

}


void
read_mig_worker (char **buffer, option_fmt * options)
{
  char input[LINESIZE];
  char tmp[LINESIZE];
  char varvalue[LINESIZE];
  char ch;
  long test = 0;
  long i = 0;
  /* 1st example:  1.0 (n-island model)
     2nd example: {1.0} (migration matrix model, all the same start values  
     3rd example: the dashes on the diagonal are NECESSARY, {} are facultativ
     -  1.0 0.1
     1.0  -  2.0
     0.9 1.2  -
     to specify real 0.0 you need to use the custom-migration settings.
     0.0 in the table will be change to SMALLES_MIGRATION
   */

  while ((ch = sgetc (buffer)) != '=')
    {
      //      parmvar[i++] = ch;
    }
  i = 0;
  ch = sgetc (buffer);
  while (!isspace ((int) ch) && ch != ':' && ch != '{')
    {
      varvalue[i++] = ch;
      ch = sgetc (buffer);
    }
  switch (toupper (varvalue[0]))
    {
    case 'R':
      options->migrguess = RANDOMESTIMATE;
      options->nummg = 2;
      options->mg = (double *) realloc (options->mg, sizeof (double) * 3);
      sgets (input, LINESIZE, buffer);
      sscanf (input, "{%lf,%lf}\n", &options->mg[0], &options->mg[1]);
      break;
    case 'F':
    case '_':
      options->migrguess = FST;
      break;
      //    case 'G':
    case 'O':
    case '0':
      //      if(varvalue[0]=='G')
      //       options->migrguess = PARAMGRID;
      //      else
      options->migrguess = OWN;
      ch = skip_sspace (buffer);
      if (ch == '\0')
	return;
      if (ch == '{')
	{
	  options->migration_model = MATRIX;
	  while (ch != '}')
	    {
	      ch = skip_sspace (buffer);
	      if ((ch == '\0') || (ch == '}'))
		return;
	      i = 0;
	      while (!isspace (ch) && ch != ',' && ch != '}')
		{
		  tmp[i++] = ch;
		  ch = sgetc (buffer);
		}
	      tmp[i] = '\0';
	      if (strcmp (tmp, "-"))
		{
		  options->mg[options->nummg] = atof (tmp);
		  options->nummg += 1;
		  options->mg =
		    (double *) realloc (options->mg,
					sizeof (double) * (1 +
							   options->nummg));
		}
	      else
		{
		  test++;
		}
	    }
	  options->numpop = test;
	}
      else
	{
	  options->migration_model = ISLAND;
	  i = 0;
	  options->numpop = 1;
	  tmp[i++] = ch;
	  while (!isspace (ch))
	    {
	      tmp[i++] = ch;
	      ch = sgetc (buffer);
	    }
	  options->mg[options->nummg] = atof (tmp);
	  options->nummg += 1;
	}
      if (options->nummg == 0)
	{
	  warning ("You forgot to add your guess value, use either:\n");
	  warning ("migration=FST\n");
	  warning ("migration=Own:{migration matrix, diagonal is -}\n");
	  usererror
	    ("migration=Own:{migration value}, all matrix elements have the same value\n");
	}
      break;
    default:
      usererror ("Failure to read start migration method\n");
    }

}
#endif

char
skip_space (option_fmt * options)
{
  char ch = getc (options->parmfile);
  while (isspace ((int) ch) || ch == ',')
    {
      ch = getc (options->parmfile);
    }
  if (isalpha (ch))
    {
      ungetc (ch, options->parmfile);
      ch = '\0';
    }
  return ch;
}

#ifdef MPI
char
skip_sspace (char **buffer)
{
  char ch = sgetc (buffer);
  while (isspace ((int) ch) || ch == ',')
    {
      ch = sgetc (buffer);
      if (isalpha (ch))
	return ch;
    }
  return ch;
}
#endif

void
set_profile_options (option_fmt * options)
{
  switch (options->profile)
    {
    case NONE:
      options->printprofsummary = options->printprofile = FALSE;
      break;
    case ALL:
      options->printprofsummary = options->printprofile = TRUE;
      break;
    case TABLES:
      options->printprofsummary = FALSE;
      options->printprofile = TRUE;
      break;
    case SUMMARY:
      options->printprofsummary = TRUE;
      options->printprofile = FALSE;
      break;
    }
  if (options->profilemethod == 'd')
    options->printprofsummary = FALSE;
}



/* custom-migration:<{> migration matrix and theta on
   diagonal:
   0 means not estimated,
   x means estimated, s means symmetrically
   estimated, m means all are the same, 
   c means remains constant at start value <}>
   example: 
   {* * s
    * c *
    s 0 *}
*/
void
read_custom_migration (FILE * file, option_fmt * options, char *value,
		       long customnumpop)
{

  long zz = 0, z = 0;
  char ch = '\0';
  long lc, numpop, i, j, ii;
  long position = 0;

  if (customnumpop == 0)
    customnumpop = 1000000;
  else
    customnumpop *= customnumpop;

  z = 0;
  zz = 0;
  while (ch != '}' && zz < customnumpop + options->gamma)
    {
      ch = value[z];
      switch (ch)
	{
	case '}':
	case '{':
	case ' ':
	case '\t':
	  z++;
	  break;
	case '\0':
	case '\n':
	  z = 0;
	  if (file == stdin)
	    printf ("Enter the next value or list of values\n");
	  FGETS (value, LINESIZE, file);
	  break;
	default:
	  options->custm =
	    (char *) realloc (options->custm, sizeof (char) * (zz + 2));
	  options->custm2 =
	    (char *) realloc (options->custm2, sizeof (char) * (zz + 2));
	  switch (ch)
	    {
	    case 'S':
	      options->custm[zz++] = ch;
	      break;
	    case 'M':		//do we have code for this?
	      options->custm[zz++] = ch;
	      break;
	    case 'x':
	    case 'X':
	      options->custm[zz++] = '*';
	      break;
	    case 'c':
	    case 'C':
	      options->custm[zz++] = 'c';
	      break;
	    default:
	      options->custm[zz++] = tolower (ch);
	      break;
	    }
	  z++;
	}
    }
  options->custm[zz] = '\0';
  lc = (long) strlen (options->custm);
  numpop = (long) sqrt ((double) lc);
  z = numpop;
  for (i = 0; i < numpop; i++)
    {
      for (j = 0; j < numpop; j++)
	{
	  ii = i * numpop + j;
	  if (i == j)
	    options->custm2[i] = options->custm[ii];
	  else
	    options->custm2[z++] = options->custm[ii];
	}
    }
  options->custm2[z] = '\0';
  specify_migration_type (options);
  if (file != stdin)
    position = ftell (options->parmfile);
  while (file != stdin && !(strstr (value, "end") || strchr (value, '=')))
    {
      position = ftell (options->parmfile);
      FGETS (value, LINESIZE, file);
    }
  if (file != stdin)
    reset_oneline (options, position);
}

#ifdef MPI
void
read_custom_migration_worker (char **buffer, option_fmt * options,
			      char *value, long customnumpop)
{

  long zz = 0, z = 0;
  char ch = '\0';
  long lc, numpop, i, j, ii;
  char *position;

  if (customnumpop == 0)
    customnumpop = 1000000;
  else
    customnumpop *= customnumpop;

  z = 0;
  zz = 0;
  while (ch != '}' && zz < customnumpop)
    {
      ch = value[z];
      switch (ch)
	{
	case '}':
	case '{':
	case ' ':
	case '\t':
	  z++;
	  break;
	case '\0':
	case '\n':
	  z = 0;
	  sgets (value, LINESIZE, buffer);
	  break;
	default:
	  options->custm =
	    (char *) realloc (options->custm, sizeof (char) * (zz + 2));
	  options->custm2 =
	    (char *) realloc (options->custm2, sizeof (char) * (zz + 2));
	  switch (ch)
	    {
	    case 'S':
	      options->custm[zz++] = ch;
	      break;
	    case 'M':		//do we have code for this?
	      options->custm[zz++] = ch;
	      break;
	    case 'x':
	    case 'X':
	      options->custm[zz++] = '*';
	      break;
	    default:
	      options->custm[zz++] = tolower (ch);
	      break;
	    }
	  z++;
	}
    }
  options->custm[zz] = '\0';
  lc = (long) strlen (options->custm);
  numpop = (long) sqrt ((double) lc);
  z = numpop;
  for (i = 0; i < numpop; i++)
    {
      for (j = 0; j < numpop; j++)
	{
	  ii = i * numpop + j;
	  if (i == j)
	    options->custm2[i] = options->custm[ii];
	  else
	    options->custm2[z++] = options->custm[ii];
	}
    }
  options->custm2[z] = '\0';
  specify_migration_type (options);
  position = *buffer;
  while (!(strstr (value, "end") || strchr (value, '=')))
    {
      position = *buffer;
      sgets (value, LINESIZE, buffer);
    }
  *buffer = position;;
}
#endif /*MPI*/
  void
specify_migration_type (option_fmt * options)
{
  long len = (long) strlen (options->custm);
  long ms = 0, xs = 0, ns = 0, ss = 0, len2, i;
  char *p;
  p = options->custm;
  while (*p != '\0')
    {
      switch (*p)
	{
	case 'm':
	  ms++;
	  break;
	case 'x':
	case '*':
	  xs++;
	  break;
	case '0':
	  ns++;
	  break;
	case 'S':
	case 's':
	  ss++;
	  break;
	case 'c':
	  break;
	}
      p++;
    }
  if (ms >= len)
    {
      options->migration_model = ISLAND;
      return;
    }
  if (xs >= len)
    {
      options->migration_model = MATRIX;
      return;
    }
  if (ns >= len)
    {
      usererror ("Custom migration matrix was completely set to zero?!\n");
      return;
    }
  len2 = (long) sqrt ((double) len);
  if (ms == len2 && xs == len - len2)
    {
      for (i = 0; i < len2; i++)
	{
	  if (options->custm[i * len2 + i] != 'm')
	    {
	      options->migration_model = MATRIX;
	      return;
	    }
	}
      options->migration_model = MATRIX_SAMETHETA;
      return;
    }
  if (xs == len2 && ms == len - len2)
    {
      for (i = 0; i < len2; i++)
	{
	  if (options->custm[i * len2 + i] != '*')
	    {
	      options->migration_model = MATRIX;
	      return;
	    }
	}
      options->migration_model = ISLAND_VARTHETA;
      return;
    }
  options->migration_model = MATRIX_ARBITRARY;
}



void
fillup_custm (long len, world_fmt * world, option_fmt * options)
{
  long i, j, ii, z;
  char *tmp;
  len = strlen (options->custm);
  tmp = (char *) calloc (1, sizeof (char) * (world->numpop2 + 2));
  options->custm =
    (char *) realloc (options->custm, sizeof (char) * (world->numpop2 + 2));

  options->custm2 =
    (char *) realloc (options->custm2, sizeof (char) * (world->numpop2 + 2));
  strncpy (tmp, options->custm, world->numpop2 + options->gamma);
  z = world->numpop;
  for (i = 0; i < world->numpop; i++)
    {
      for (j = 0; j < world->numpop; j++)
	{
	  ii = i * world->numpop + j;
	  if (ii < len)
	    options->custm[ii] = tmp[ii];
	  else
	    options->custm[ii] = '*';
	  if (i == j)
	    options->custm2[i] = options->custm[ii];
	  else
	    options->custm2[z++] = options->custm[ii];
	}
    }
  if (options->gamma)
    {
      if (len <= world->numpop2)
	{
	  options->custm[world->numpop2] = '*';
	  options->custm2[world->numpop2] = '*';
	}
      else
	{
	  options->custm[world->numpop2] = tmp[world->numpop2];
	  options->custm2[world->numpop2] = tmp[world->numpop2];
	}
      options->custm[world->numpop2 + 1] = '\0';
      options->custm2[world->numpop2 + 1] = '\0';
    }
  else
    {
      options->custm[world->numpop2] = '\0';
      options->custm2[world->numpop2] = '\0';
    }
  strcpy (world->options->custm, options->custm);
  strcpy (world->options->custm2, options->custm2);
  free (tmp);
}

void
print_arbitrary_migration_table (FILE * file, world_fmt * world,
				 data_fmt * data)
{
  long i;
  char mytext[LINESIZE];
  switch (world->options->migration_model)
    {
    case ISLAND:
      strcpy (mytext, "N-Island migration model");
      fprintf (file, "Migration model:\n   %-44.44s\n", mytext);
      break;
    case ISLAND_VARTHETA:
      strcpy (mytext, "N-Island migration model with variable Theta");
      fprintf (file, "Migration model:\n   %-44.44s\n", mytext);
      break;
    case MATRIX:
      strcpy (mytext, "Migration matrix model with variable Theta ");
      fprintf (file, "Migration model:\n   %-44.44s\n", mytext);
      break;
    case MATRIX_SAMETHETA:
      strcpy (mytext, "Migration matrix model with same Theta");
      fprintf (file, "Migration model:\n   %-44.44s\n", mytext);
      break;
    case MATRIX_ARBITRARY:
    default:
      strcpy (mytext, "Arbitrary migration matrix model");
      fprintf (file, "Migration model: %-44.44s\n", mytext);
      fprintf (file,
	       "[Legend: m = average (average over a group of Thetas or M]\n");
      fprintf (file,
	       "[s = symmetric M, S = symmetric 4Nm,\n 0 = zero, and not estimated,   ]\n");
      fprintf (file, "[* = free to vary, Thetas are on diagonal]\n");
      if (world->options->migration_model == MATRIX_ARBITRARY)
	{
	  for (i = 0; i < world->numpop2; i++)
	    {
	      if (i % world->numpop == 0)
		fprintf (file, "\n%10.10s     ",
			 data->popnames[i / world->numpop]);
	      fprintf (file, "%c ", world->options->custm[i]);
	    }
	  fprintf (file, "\n\n");
	}
      break;
    }
}

void
print_distance_table (FILE * file, world_fmt * world, option_fmt * options,
		      data_fmt * data)
{
  long i, j;

  if (options->geo)
    {
      fprintf (file,
	       "   Geographic distance matrix between locations\n      ");
      for (i = 0; i < world->numpop; i++)
	{
	  fprintf (file, "%-10.10s     ", data->popnames[i]);
	  for (j = 0; j < world->numpop; j++)
	    {
	      if (i == j)
		fprintf (file, "   -   ");
	      else
		fprintf (file, "%6.3f ", data->ogeo[i][j]);
	    }
	  fprintf (file, "\n      ");
	}
      fprintf (file, "\n");
    }
}
