/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 P A R A M E T E R E S T I M A T I O N   R O U T I N E S 
 
 estimates parameter for each locus
 using a Broyden minimization
 
 
 Peter Beerli 1997, Seattle
 beerli@genetics.washington.edu
 
 Copyright 2001 Peter Beerli and Joseph Felsenstein
 
 $Id: broyden.c,v 1.59 2001/09/07 23:56:13 beerli Exp $
-------------------------------------------------------*/
extern double norm (double *d, long size);

#include "migration.h"
#include "world.h"
#include "random.h"
#include "combroyden.h"
#include "joint-chains.h"
#include "migrate_mpi.h"
#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif



/* prototypes ----------------------------------------- */
double absmaxvec (double *v, long n);
void create_nr (nr_fmt * nr, world_fmt * world, long G, long profilenum,
                long thislocus, long repkind, long rep);
void reset_hess (double **hess, long n);
void destroy_nr (nr_fmt * nr, world_fmt * world);
double calc_locus_like (nr_fmt * nr, double *param, double *lparam,
                        long locus);
void param_all_adjust (double *xv, nr_fmt *nr); //worldoption_fmt * wopt, long numpop);
void gradient (double *d, nr_fmt * nr, long locus);
double probG (double *param, double *lparam, tarchive_fmt * tl, nr_fmt * nr,
              long locus);
double probG2 (double *param, double *lparam, double *sm, double *kt,
               double *km, double *p, double *mindex, int *msta, int *me,
               long numpop);
void probG3 (double *apg, double *apg0r, timearchive_fmt * tyme, long numpop,
             double *apgmax, double *param, double *lparam, double *sm);
double probG4 (double *fullparam, double *data, long numpop, long numpop2);
double logprob_noevent (world_fmt * world, long interval);
double sum_migprob (world_fmt * world, long pop, long interval);
double sum_mig (double *param, long msta, long msto);
void calc_cov (double **dd, double *d, double *param, long n);
boolean is_singular (double **dd, long n);
void print_contribution (nr_fmt * nr, timearchive_fmt ** atl, long G);

void calc_dv (double *dv, double **hess, double *gxv, long n);
double calc_line (helper_fmt * helper, double a, double b, double c,
                  double (*psi) (double lamda, helper_fmt * helper));
void calc_hessian (double **hess, long n, double *delta, double *gama);
double psi (double lamda, helper_fmt * helper, double *param, double *lparam);
void grad2loggrad (double *param, long *indeks, double *d, long nn,
                   long profilenum);
void log_param0 (double *param, double *lparam, long nn);
void copies2lcopies (timearchive_fmt * atl);
void create_apg0 (double *apg0, nr_fmt * nr, timearchive_fmt * tyme,
                  long locus);


void force_sametheta (double *param, worldoption_fmt * wopt, long numpop);
void force_samemigration (double *param, worldoption_fmt * wopt, long numpop);
void calc_same_d (double *grad, nr_fmt * nr, long nn, long start, long stop);
void calc_symmetric_d (double *grad, nr_fmt * nr, long nn, long start,
                       long stop);
void force_symmetric_d (double *gxv, long model, nr_fmt * nr, long nn);
void check_symmetric_d (double *gxv, nr_fmt * nr, long nn);
void check_matrix_arbitrary (double *param, worldoption_fmt * wopt,
                             long numpop, long which_profile);
void print_menu_contribution (FILE * file, long contribution[]);
void alloc_apg (double ****apg, long repstop, long loci, long G);


void quadratic_constants (double *xguess,
                          double low, double mid, double high,
                          double xlow, double xmid, double xhigh);
void symmetric_other (long i, long numpop, long *other, long *otherpop);
double ln_copies (long n);

double normal_func_ok(double *param, double *param0, long numpop2);
double normal_func_no(double *param, double *param0, long numpop2);
double normal_func_gradient_ok(double p1, double p0);
double normal_func_gradient_no(double p1, double p0);

void unset_penalizer_function(boolean inprofiles);

double (*normal_func)(double *, double *, long);
double (*normal_func_gradient)(double, double);

/* Functions ++++++++++++++++++++++++++++++++++++++++++++++++*/
void
calc_dv (double *dv, double **hess, double *gxv, long n)
{
  long i, j;
  memset (dv, 0, sizeof (double) * n);
  for (i = 0; i < n; i++)
    {
      for (j = 0; j < n; j++)
        {
          dv[i] += hess[i][j] * gxv[j];
        }
    }
}

// line searcher
// finds the maximum in a direction
// this should be replaced with something more efficient.
#define PP 0.61803399
#define QQ 0.38196601
#define MOVE3(a,b,c,d) (a)=(b);(b)=(c);(c)=(d)




double
calc_line_new (helper_fmt * helper, double a, double b, double c,
               double (*fpsi) (double lamda, helper_fmt * helper))
{
  double xhigh = c;
  double xlow = a;
  double low, high, mid, xmid;
  double xguess;
  long panic = 0;
  low = (*fpsi) (xlow, helper);
  high = (*fpsi) (xhigh, helper);
  xmid = (xlow + xhigh) / 2.;
  while (panic++ < 10000 && fabs (xhigh - xlow) > EPSILON)
    {
      mid = (*fpsi) (xmid, helper);
      quadratic_constants (&xguess, low, mid, high, xlow, xmid, xhigh);
      if (isnan (xguess))
        return 1.;
      if (xguess < xmid)
        {
          xhigh = xmid;
          high = mid;
        }
      else
        {
          xlow = xmid;
          low = mid;
        }
      xmid = xguess;
    }
  return xlow;
}

void
quadratic_constants (double *xguess,
                     double low, double mid, double high,
                     double xlow, double xmid, double xhigh)
{
  double xhigh2 = xhigh * xhigh;
  double xlow2 = xlow * xlow;
  double xmid2 = xmid * xmid;

  double a = -((-(xmid * low) + xhigh * low + xlow * mid - xhigh * mid -
                xlow * high + xmid * high) /
               ((xlow - xmid) * (xlow * xmid - xlow * xhigh - xmid * xhigh +
                                 xhigh2)));
  double b =
    -((xmid2 * low - xhigh2 * low - xlow2 * mid + xhigh2 * mid +
       xlow2 * high - xmid2 * high) / ((xlow - xmid) * (xlow * xmid -
                                       xlow * xhigh -
                                       xmid * xhigh +
                                       xhigh2)));
  //    double c = -((-(xmid2*xhigh*low) + xmid*xhigh2*low +
  //    xlow2*xhigh*mid - xlow*xhigh2*mid -
  //    xlow2*xmid*high + xlow*xmid2*high)/
  //  (xlow2*xmid - xlow*xmid2 - xlow2*xhigh +
  //   xmid2*xhigh + xlow*xhigh2 - xmid*xhigh2));
  *xguess = -b / (2. * a);
  // printf("quadr:{%f %f}  {%f %f} {%f %f}\n", xlow, low, *xguess, a * (*xguess * *xguess) + b * *xguess + c, xhigh, high);
}



double
calc_line (helper_fmt * helper, double a, double b, double c,
           double (*fpsi) (double lamda, helper_fmt * helper))
{
  /* a < b < c AND psia > psib < psic */
  double d, psib, psic;
  d = c;
  if (fabs (c - b) > fabs (b - a))
    {
      c = b + QQ * (c - b);
    }
  else
    {
      c = b;
      b = b - QQ * (b - a);
    }
  psib = (*fpsi) (b, helper);
  psic = (*fpsi) (c, helper);
  while (fabs (d - a) > EPSILON * (fabs (b) + fabs (c)))
    {
      if (psic < psib)
        {
          MOVE3 (a, b, c, PP * b + QQ * d);
          psib = psic;
          psic = (*fpsi) (c, helper);
        }
      else
        {
          MOVE3 (d, c, b, PP * c + QQ * a);
          psic = psib;
          psib = (*fpsi) (b, helper);
        }
      //      printf("b=%f(%f
      //), c=%f(%f) {%f, %f}\n",b,psib,c,psic, helper->nr->param[0],
      //     helper->nr->param[helper->nr->partsize-1]);
    }
  if (psib < psic)
    {
      return b;
    }
  else
    {
      return c;
    }
}

double
psi (double lamda, helper_fmt * helper, double *param, double *lparam)
{
  double like;
  calc_loci_param (helper->nr, helper->nr->lparam, helper->xv,
                   helper->dv, lamda, helper->nr->partsize);
  set_expparam (helper->nr->param, helper->nr->lparam, helper->nr->partsize);
  fill_helper (helper, helper->nr->param, helper->nr->lparam,
               helper->nr->world, helper->nr);
  like = CALCLIKE (helper, helper->nr->param, helper->nr->lparam);
  return -like;
}


void
calc_hessian (double **hess, long n, double *delta, double *gama)
{
  double **dd, *temp, t;
  long i, j, k;
  double dtg;
  temp = (double *) calloc (n, sizeof (double));
  dd = (double **) calloc (n, sizeof (double *));
  dd[0] = (double *) calloc (n * n, sizeof (double));
  dtg = delta[0] * gama[0];
  for (i = 1; i < n; i++)
    {
      dd[i] = dd[0] + n * i;
      dtg += delta[i] * gama[i];
    }
  if (dtg != 0.0)
    dtg = 1. / dtg;
  else
    {
      reset_hess (hess, n);
      free (temp);
      free (dd[0]);
      free (dd);
      return;
    }
  for (i = 0; i < n; i++)
    {
      for (j = 0; j < n; j++)
        {
          temp[i] += gama[j] * hess[j][i];
        }
    }
  t = 0.0;
  for (i = 0; i < n; i++)
    t += temp[i] * gama[i];
  t = (1.0 + t * dtg) * dtg;
  for (i = 0; i < n; i++)
    {
      for (j = 0; j < n; j++)
        {
          for (k = 0; k < n; k++)
            {
              dd[i][j] += delta[i] * gama[k] * hess[k][j];
            }
        }
    }
  for (i = 0; i < n; i++)
    {
      temp[i] = 0.0;
      for (j = 0; j < n; j++)
        {
          temp[i] += hess[i][j] * gama[j];
        }
    }
  for (i = 0; i < n; i++)
    {
      for (j = 0; j < n; j++)
        {
          dd[i][j] += temp[i] * delta[j];
          dd[i][j] *= dtg;
          hess[i][j] += delta[i] * delta[j] * t - dd[i][j];
        }
    }
  free (temp);
  free (dd[0]);
  free (dd);
}

double
absmaxvec (double *v, long n)
{
  long i;
  double max = fabs (v[0]);
  for (i = 1; i < n; i++)
    {
      if (max < fabs (v[i]))
        max = fabs (v[i]);
    }
  return max;
}

void
create_nr (nr_fmt * nr, world_fmt * world, long G,
           long profilenum, long thislocus, long repkind, long rep)
{
  long i;
  nr->numpop = world->numpop;
  nr->numpop2 = world->numpop2;
  nr->skiploci = world->data->skiploci;
  nr->world = world;
  nr->mstart = world->mstart;
  nr->mend = world->mend;
  nr->repkind = repkind;
  nr->repstart = (repkind == SINGLECHAIN) ? rep : 0;
  nr->repstop = (repkind == SINGLECHAIN) ? rep + 1 : world->repstop;
  nr->profilenum = profilenum;
  nr->apg_max = (double *) calloc (world->loci + 1, sizeof (double));
  nr->PGC = (double *) calloc (world->loci + 1, sizeof (double));
  if (/*myID == MASTER && */ world->options->gamma /* && world->locus >= world->loci*/)
    {
      nr->categs = GAMMA_INTERVALS;
      nr->rate = (double *) calloc (nr->categs, sizeof (double));
      nr->probcat = (double *) calloc (nr->categs, sizeof (double));
      nr->partsize = world->numpop2 + 1;
    }
  else
    {
      if (!world->options->gamma && world->locus >= world->loci)
        {
          nr->categs = 0;
          nr->rate = NULL;
          nr->partsize = world->numpop2;
        }
      else
        {
          nr->categs = 0;
          nr->rate = NULL;
          nr->partsize = world->numpop2;
        }
    }
  //  nr->numg = G;
  nr->atl = world->atl;
  nr->parts = (double *) calloc (nr->partsize, sizeof (double));
  nr->d = (double *) calloc (nr->partsize, sizeof (double));
  nr->od = (double *) calloc (nr->partsize, sizeof (double));
  nr->dv = (double *) calloc (nr->partsize, sizeof (double));
  nr->delta = (double *) malloc (nr->partsize * sizeof (double));
  nr->gdelta = (double *) calloc (nr->partsize, sizeof (double));
  nr->param = (double *) calloc (2 * (nr->partsize + 1), sizeof (double));
  nr->lparam = (double *) calloc (2 * (nr->partsize + 1), sizeof (double));
  //  nr->lparam = nr->param + nr->partsize+1;
  memcpy (nr->param, world->param0, nr->partsize * sizeof (double));
  for (i = 0; i < nr->partsize; ++i)
    {
      if (nr->param[i] > 0.0)
        nr->lparam[i] = log (nr->param[i]);
      else
        nr->lparam[i] = -DBL_MAX;
    }
  nr->datalike = (double *) malloc ((G+1) * sizeof (double));
  nr->locilikes = (double *) malloc ((world->loci + 1) * sizeof (double));
  nr->profiles = (long *) calloc (world->numpop2 + 1, sizeof (long));
  nr->values = (double *) calloc (world->numpop2 + 1, sizeof (double));
  nr->indeks = (long *) calloc (nr->partsize, sizeof (long));
  // allocate earlier in world: alloc_apg (&nr->apg0, world->repstop, world->loci, G);
  // allicate earlier in world:  alloc_apg (&nr->apg, world->repstop, world->loci, G);
  nr->apg = world->apg;
  nr->apg0 = world->apg0;
}

void
alloc_apg (double ****apg, long repstop, long loci, long G)
{
  long j, i;
  (*apg) = (double ***) calloc (repstop, sizeof (double **));
  for (j = 0; j < repstop; j++)
    {
      (*apg)[j] = (double **) calloc (loci + 1, sizeof (double *));
      (*apg)[j][0] =
        (double *) calloc ((1 + G) * (loci + 1), sizeof (double));
      for (i = 1; i < loci + 1; i++)
        {
          (*apg)[j][i] = (*apg)[j][0] + i * (G + 1);
        }
    }
}

void
reuse_nr (nr_fmt * nr, world_fmt * world, long G, long profilenum,
          long thislocus, long repkind, long rep)
{
  long i;
  nr->repkind = repkind;
  nr->repstart = (repkind == SINGLECHAIN) ? rep : 0;
  nr->repstop = (repkind == SINGLECHAIN) ? rep + 1 : world->repstop;
  nr->profilenum = profilenum;
  if (world->options->gamma && world->locus >= world->loci)
    {
      nr->categs = GAMMA_INTERVALS;
      nr->rate = (double *) realloc (nr->rate, nr->categs * sizeof (double));
      nr->probcat =
        (double *) realloc (nr->probcat, nr->categs * sizeof (double));
      nr->partsize = world->numpop2 + 1;
    }
  memset (nr->parts, 0, nr->partsize * sizeof (double));
  memset (nr->d, 0, nr->partsize * sizeof (double));
  memset (nr->od, 0, nr->partsize * sizeof (double));
  memset (nr->dv, 0, nr->partsize * sizeof (double));
  memset (nr->delta, 0, nr->partsize * sizeof (double));
  memset (nr->gdelta, 0, nr->partsize * sizeof (double));
  memcpy (nr->param, world->param0, nr->partsize * sizeof (double));
  for (i = 0; i < nr->partsize; ++i)
    {
      if (nr->param[i] > 0.0)
        nr->lparam[i] = log (nr->param[i]);
      else
        nr->lparam[i] = -DBL_MAX;
    }
  nr->datalike = (double *) realloc (nr->datalike, G * sizeof (double));
  memset (nr->datalike, 0, G * sizeof (double));
  memset (nr->locilikes, 0, (world->loci + 1) * sizeof (double));
  /*
    for (j = world->repstop - 1; j >= 0; j--)
      {
        free (nr->apg0[j][0]);
        free (nr->apg[j][0]);
        free (nr->apg0[j]);
        free (nr->apg[j]);
      }
    free (nr->apg0);
    free (nr->apg);
    alloc_apg (&nr->apg0, world->repstop, world->loci, G);
    alloc_apg (&nr->apg, world->repstop, world->loci, G);*/
}

void
reset_hess (double **hess, long n)
{
  long pop;
  //  printf("resetting hessian\n");
  memset (hess[0], 0, sizeof (double) * n * n);
  for (pop = 1; pop < n; pop++)
    {
      hess[pop] = hess[0] + pop * n;
      hess[pop][pop] = 1.0;
    }
  hess[0][0] = 1.0;
}

void
destroy_nr (nr_fmt * nr, world_fmt * world)
{
  free (nr->parts);
  free (nr->d);
  free (nr->od);
  free (nr->dv);
  free (nr->delta);
  free (nr->gdelta);
  free (nr->param);
  free (nr->lparam);
  free (nr->locilikes);
  free (nr->datalike);
  free (nr->apg_max);
  free (nr->PGC);
  /* for (j = world->repstop - 1; j >= 0; j--)
     {
     free (nr->apg0[j][0]);
     free (nr->apg0[j]);
     free (nr->apg[j][0]);
     free (nr->apg[j]);
     }
     free (nr->apg0);
     free (nr->apg); */
  free (nr->profiles);
  free (nr->values);
  free (nr->indeks);
  if (nr->categs != 0)
    {
      free (nr->rate);
      free (nr->probcat);
    }
  free (nr);
}

double
calc_locus_like (nr_fmt * nr, double *param, double *lparam, long locus)
{
  long g, r, j, copies;
  const long numpop = nr->world->numpop;
  const long numpop2 = nr->world->numpop2;
  int msta, msto;
  double gsum = 0;
  double ***apg0;
  double apgmax = -DBL_MAX;
  //  double ***apgall = nr->apg;
  double *apg;
  double *apg0r;
  timearchive_fmt tyme;
  tarchive_fmt *tl;
  double *geo = nr->world->data->geo;
  double *lgeo = nr->world->data->lgeo;
  double *locallparam;
  double *localparam;
  double *sm;
  double mu_rate = nr->world->options->mu_rates[locus];
  double lmu_rate = nr->world->options->lmu_rates[locus];
  //  double temp;
  locallparam = (double *) calloc ((numpop2 + numpop + numpop),
                                   sizeof (double));
  localparam = locallparam + numpop2;

  sm = localparam + numpop;
  memcpy (localparam, param, sizeof (double) * numpop);
  memcpy (locallparam, lparam, sizeof (double) * numpop2);

  nr->PGC[locus] = 0.0;
  apg0 = nr->apg0;

  for (r = 0; r < nr->numpop; r++)
    {
      locallparam[r] = LOG2 - (locallparam[r] + lmu_rate);
      localparam[r] = 1. / (localparam[r] * mu_rate);
      msta = nr->mstart[r];
      msto = nr->mend[r];
      for (j = msta; j < msto; j++)
        {
          sm[r] += geo[j] * param[j] / mu_rate;
          locallparam[j] += lgeo[j] - lmu_rate;
        }
    }
  gsum = 0;
  for (r = nr->repstart; r < nr->repstop; r++)
    {
      apg = nr->apg[r][locus];
      apg0r = apg0[r][locus];
      tyme = nr->atl[r][locus];
      //first element
      tl = &(tyme.tl[0]);
      copies = tl->copies - 1;
      gsum += copies;
      if (copies > 0)
        {
          apg[0] =
            tl->lcopies + probG4 (locallparam, tl->data, numpop,
                                  numpop2) - apg0r[0]
            + (*normal_func)(param, tyme.param0, numpop2)
            ;
        }
      else
        apg[0] = -DBL_MAX;
      if (apgmax < apg[0])
        apgmax = apg[0];
      //other elements
      for (g = 1; g < tyme.T; g++)
        {
          tl = &(tyme.tl[g]);
          copies = tl->copies;
          gsum += copies;
          apg[g] =
            tl->lcopies + probG4 (locallparam, tl->data, numpop,
                                  numpop2) - apg0r[g]
            + (*normal_func)(param, tyme.param0, numpop2);
          if (apg[g] > apgmax)
            apgmax = apg[g];
        }
    }				// end replicates
  for (r = nr->repstart; r < nr->repstop; r++)
    {
      apg = nr->apg[r][locus];
      apg0r = apg0[r][locus];
      tyme = nr->atl[r][locus];
      // first element
      tl = &(tyme.tl[0]);
      // first element
      apg[0] -= apgmax;
      nr->PGC[locus] += EXP (apg[0]);
      // all other elements
      for (g = 1; g < tyme.T; g++)
        {
          apg[g] -= apgmax;
          if (apg[g] > -40.)
            nr->PGC[locus] += EXP (apg[g]);
        }
    }				// replicates
  nr->apg_max[locus] = apgmax;
  nr->llike = apgmax + log (nr->PGC[locus]) - log (gsum);
  free (locallparam);
  return nr->llike;
}


double
ln_copies (long n)
{
  switch (n)
    {
    case 0:
      return -DBL_MAX;
    case 1:
      return 0.;
    case 2:
      return 0.69314718055994530942;
    case 3:
      return 1.0986122886681096914;
    case 4:
      return 1.3862943611198906188;
    case 5:
      return 1.6094379124341003746;
    case 6:
      return 1.7917594692280550008;
    case 7:
      return 1.9459101490553133051;
    case 8:
      return 2.0794415416798359283;
    case 9:
      return 2.1972245773362193828;
    case 10:
      return 2.3025850929940456840;
    case 11:
      return 2.3978952727983705441;
    case 12:
      return 2.4849066497880003102;
    case 13:
      return 2.5649493574615367361;
    case 14:
      return 2.6390573296152586145;
    case 15:
      return 2.7080502011022100660;
    case 16:
      return 2.7725887222397812377;
    case 17:
      return 2.8332133440562160802;
    case 18:
      return 2.8903717578961646922;
    case 19:
      return 2.9444389791664404600;
    case 20:
      return 2.9957322735539909934;
    default:
      return log ((double) n);
    }
}

void
force_sametheta (double *param, worldoption_fmt * wopt, long numpop)
{
  long i, n = 0;
  double summ = 0, logsumm;
  for (i = 0; i < numpop; i++)
    {
      if (wopt->custm2[i] == 'm')
        {

          summ += EXP (param[i]);
          n++;
        }
    }
  summ /= n;
  logsumm = log (summ);
  for (i = 0; i < numpop; i++)
    {
      if (wopt->custm2[i] == 'm')
        param[i] = logsumm;
    }
}

void
force_samemigration (double *param, worldoption_fmt * wopt, long numpop)
{
  long i;
  double summ = 0, logsumm;
  long numpop2 = numpop * numpop;
  long n = 0;

  for (i = numpop; i < numpop2; i++)
    {
      if (wopt->custm2[i] == 'm')
        {
          summ += EXP (param[i]);
          n++;
        }
    }
  summ /= n;
  logsumm = log (summ);
  for (i = numpop; i < numpop2; i++)
    {
      if (wopt->custm2[i] == 'm')
        param[i] = logsumm;
    }
}

void
param_all_adjust (double *xv, nr_fmt *nr)
{
  double *param = xv;
  worldoption_fmt * wopt= nr->world->options;
  long numpop = nr->world->numpop;
  //	long numpop2 = nr->world->numpop2;
  //===================
  double which_profile= -1;
  char *custm = NULL;
  long zeron = 0;
  long constn = 0;
  long symn = 0;
  long sym2n = 0;
  long from, to;
  twin_fmt *syms = NULL;
  quad_fmt *sym2s = NULL;
  long *zeros = NULL;
  long *consts = NULL;
  long z = 0, i;
  char *p;
  double mm, lmm;
  custm = wopt->custm2;
  zeron = wopt->zeron;
  constn = wopt->constn;
  symn = wopt->symn;
  sym2n = wopt->sym2n;
  if (symn > 0)
    syms = wopt->symparam;
  if (sym2n > 0)
    sym2s = wopt->sym2param;
  if (zeron > 0)
    zeros = wopt->zeroparam;
  if (constn > 0)
    consts = wopt->constparam;
  p = custm;
  z = strcspn(p, "m");
  if (z > 0 && z < numpop)
    force_sametheta (param, wopt, numpop);
  p = custm + numpop;
  z = strcspn(p, "m");
  if (z > 0 && z < numpop*(numpop-1))
    force_samemigration (param, wopt, numpop);
  for (i = 0; i < zeron; i++)
    param[zeros[i]] = -30.;	//-DBL_MAX;
  for (i = 0; i < constn; i++)
    {
      if (consts[i] < numpop)
        param[consts[i]] = log (wopt->thetag[consts[i]]);
      if (consts[i] >= numpop)
        {
          if (consts[i] >= numpop * numpop)
            param[consts[i]] = log (wopt->alphavalue);
          else
            {
              m2mm (consts[i], numpop, &from, &to);
              param[consts[i]] = log (wopt->mg[to*(numpop-1)+from]);
            }
        }
    }

  // this is so weird because of the profiled parameters which should not change of course
  for (i = 0; i < symn; i++)
    {
      if(nr->profilenum>0)
        {
          for(z=0;z<nr->profilenum;z++)
            {
              which_profile = nr->profiles[z];
              if(which_profile == syms[i][0])
                {
                  param[syms[i][1]] = param[syms[i][0]];
                  break;
                }
              else
                {
                  if(which_profile == syms[i][1])
                    {
                      param[syms[i][0]] = param[syms[i][1]];
                      break;
                    }
                  else
                    {
                      mm = (EXP (param[syms[i][0]]) + EXP (param[syms[i][1]])) / 2.;
                      param[syms[i][0]] = param[syms[i][1]] = log (mm);
                      break;
                    }
                }
            }
        }
      else
        {
          mm = (EXP (param[syms[i][0]]) + EXP (param[syms[i][1]])) / 2.;
          param[syms[i][0]] = param[syms[i][1]] = log (mm);
        }
    }
  for (i = 0; i < sym2n; i++)
    {
      if(nr->profilenum>0)
        {
          for(z=0;z<nr->profilenum;z++)
            {
              which_profile = nr->profiles[z];
              if(which_profile == sym2s[i][0])
                {
                  //printf(".1.\n");
                  mm = param[sym2s[i][0]] + param[sym2s[i][2]];
                  param[sym2s[i][1]] = mm - param[sym2s[i][3]];

                  break;
                }
              else
                {
                  if(which_profile == sym2s[i][1])
                    {
                      //printf(".2.\n");
                      mm = param[sym2s[i][1]] + param[sym2s[i][3]];
                      param[sym2s[i][0]] = mm - param[sym2s[i][2]];
                      break;
                    }
                  else
                    {
                      mm = (EXP (param[sym2s[i][0]] + param[sym2s[i][2]]) +
                            EXP (param[sym2s[i][1]] + param[sym2s[i][3]])) / 2.;
                      param[sym2s[i][0]] = (lmm = log (mm)) - param[sym2s[i][2]];
                      param[sym2s[i][1]] = lmm - param[sym2s[i][3]];
                      break;
                    }
                }
            }
        }
      else
        {
          mm = (EXP (param[sym2s[i][0]] + param[sym2s[i][2]]) +
                EXP (param[sym2s[i][1]] + param[sym2s[i][3]])) / 2.;
          param[sym2s[i][0]] = (lmm = log (mm)) - param[sym2s[i][2]];
          param[sym2s[i][1]] = lmm - param[sym2s[i][3]];
        }
    }
}
//======================
/*  switch (wopt->migration_model)
    {
    case MATRIX:
      break;
    case MATRIX_ARBITRARY:
      check_matrix_arbitrary (xv, wopt, numpop, which_profile);
      break;
    case MATRIX_SAMETHETA:
      force_sametheta (xv, wopt, numpop);
      break;
    case ISLAND:
      force_sametheta (xv, wopt, numpop);
      force_samemigration (xv, wopt, numpop);
      break;
    case ISLAND_VARTHETA:
      force_samemigration (xv, wopt, numpop);
      break;
    }
}
*/

// checks and corrects the migration matrix according to custom migration matrix
// which_profile is necessary to avoid accidental changes in the variables that need profiling
void
check_matrix_arbitrary (double *param, worldoption_fmt * wopt, long numpop, long which_profile)
{
  char *custm = NULL;
  long zeron = 0;
  long constn = 0;
  long symn = 0;
  long sym2n = 0;
  long from, to;
  //   boolean done = FALSE;
  twin_fmt *syms = NULL;
  quad_fmt *sym2s = NULL;
  long *zeros = NULL;
  long *consts = NULL;
  long z = 0, i;
  long count;
  char *p;
  double mm, lmm;
  //     long locus = 0;
  custm = wopt->custm2;
  zeron = wopt->zeron;
  constn = wopt->constn;
  symn = wopt->symn;
  sym2n = wopt->sym2n;
  if (symn > 0)
    syms = wopt->symparam;
  if (sym2n > 0)
    sym2s = wopt->sym2param;
  if (zeron > 0)
    zeros = wopt->zeroparam;
  if (constn > 0)
    consts = wopt->constparam;
  p = custm;
  z = 0;
  for (count = 0; count < numpop; count++)
    {
      if (*p == 'm')
        z++;
      p++;
    }
  if (z > 0)
    force_sametheta (param, wopt, numpop);
  p = custm + numpop;
  z = 0;
  for (count = numpop; count < numpop * numpop; count++)
    {
      if (*p == 'm')
        z++;
      p++;
    }
  if (z > 0)
    force_samemigration (param, wopt, numpop);
  for (i = 0; i < zeron; i++)
    {
      param[zeros[i]] = -30.;	//-DBL_MAX;
    }

  for (i = 0; i < constn; i++)
    {
      if (consts[i] < numpop)
        param[consts[i]] = log (wopt->thetag[consts[i]]);
      if (consts[i] >= numpop)
        {
          if (consts[i] >= numpop * numpop)
            param[consts[i]] = log (wopt->alphavalue);
          else
            {
              m2mm (consts[i], numpop, &from, &to);
              param[consts[i]] = log (wopt->mg[to*(numpop-1)+from]);
            }
        }
    }

  for (i = 0; i < symn; i++)
    {
      if(which_profile == syms[i][0])
        param[syms[i][1]] = param[syms[i][0]];
      else
        {
          if(which_profile == syms[i][1])
            param[syms[i][0]] = param[syms[i][1]];
          else
            {
              mm = (EXP (param[syms[i][0]]) + EXP (param[syms[i][1]])) / 2.;
              param[syms[i][0]] = param[syms[i][1]] = log (mm);
            }
        }
    }
  for (i = 0; i < sym2n; i++)
    {
      if(which_profile == sym2s[i][0])
        mm = EXP (param[sym2s[i][0]] + param[sym2s[i][2]]);
      else
        {
          if(which_profile == sym2s[i][1])
            mm = EXP (param[sym2s[i][0]] + param[sym2s[i][2]]);
          else
            {
              mm = (EXP (param[sym2s[i][0]] + param[sym2s[i][2]]) +
                    EXP (param[sym2s[i][1]] + param[sym2s[i][3]])) / 2.;
            }
        }
      param[sym2s[i][0]] = (lmm = log (mm)) - param[sym2s[i][2]];
      param[sym2s[i][1]] = lmm - param[sym2s[i][3]];
    }
}

void
gradient (double *d, nr_fmt * nr, long locus)
{
  boolean found=FALSE;
  double tk1,m1,th1,th2,l1,nm1;
  //double nm2, l2, m2, tk2;
  long z, r;
  long other , otherpop;
  //	double tsq1;
  long g, i, ii, pop, msta, msto;
  long numpop = nr->numpop;
  long nn = nr->partsize - nr->profilenum;
  double expapg, *thetas, *m;
  double *geo = nr->world->data->geo;
  tarchive_fmt *tl;
  double tet;
  double mu_rate = nr->world->options->mu_rates[locus];
  //  long model = nr->world->options->migration_model;
  memset (d, 0, sizeof (double) * nr->numpop2);
  thetas = nr->param;
  m = nr->param + numpop;
  for (r = nr->repstart; r < nr->repstop; r++)
    {
      for (g = 0; g < nr->world->atl[r][locus].T; g++)
        {
          if (nr->apg[r][locus][g] < -40.)
            continue;
          tl = nr->world->atl[r][locus].tl;
          expapg = EXP (nr->apg[r][locus][g]);
          for (pop = 0; pop < numpop; pop++)
            {
              tet = thetas[pop];
              nr->parts[pop] = -tl[g].p[pop] / tet
                               + tl[g].kt[pop] / (tet * tet * mu_rate)
                               - (*normal_func_gradient)(tet, nr->world->atl[r][locus].param0[pop]);
              msta = nr->mstart[pop];
              msto = nr->mend[pop];
              z = 0;
              found=FALSE;
              for (i = msta; i < msto; i++)
                {
                  if (nr->param[i] > 0.)
                    {
                      if(nr->world->options->custm2[i]=='S')
                        {
                          found =find (i, nr->profiles, nr->profilenum);
                          if (!found)
                            nr->parts[i] = ((tl[g].mindex[i] / (nr->param[i]))
                                            - geo[i] * tl[g].km[pop] / mu_rate)
                                           -(*normal_func_gradient)(nr->param[i],nr->world->atl[r][locus].param0[i]);

                          symmetric_other(i, numpop, &other, &otherpop);
                          tk1 = geo[i]* tl[g].km[pop] / mu_rate;
                          //	  tk2 = geo[other] * tl[g].km[otherpop] / mu_rate;
                          l1 = tl[g].mindex[i];
                          //	  l2 = tl[g].mindex[other];
                          m1 = nr->param[i];
                          //	  m2 = nr->param[other];
                          th1 = nr->param[pop];
                          th2 = nr->param[otherpop];
                          nm1 = m1 * th1;
                          //	  nm2 = m2 * th2;
                          //	  tsq1 = nr->param[pop] * nr->param[pop];
                          //	 if (found)
                          //	nr->parts[pop]  = tk1 * nm1/th1 - l1/th1
                          //	-tl[g].p[pop] / th1
                          //+ tl[g].kt[pop] / (th1 * th1 * mu_rate);
                          // else
                          nr->parts[pop]  += tk1 * m1/th1 - l1/th1;
                          //if (find (i, nr->profiles, nr->profilenum))
                          //  nr->parts[otherpop] -= tk2 * m2 / th2 - l2/th2;
                          //		  done=TRUE;
                          //		}
                        }
                      else
                        nr->parts[i] = ((tl[g].mindex[i] / (nr->param[i]))
                                        - geo[i] * tl[g].km[pop] / mu_rate)
                                       - (*normal_func_gradient)(nr->param[i], nr->world->atl[r][locus].param0[i]);

                    }
                }
            }
          z = 0;
          for (i = 0; i < nn; i++)
            {
              ii = (nr->profilenum > 0) ? nr->indeks[z++] : i;
              d[i] += expapg * nr->parts[ii];
            }
        }
    }
}

/* calculates the derivatives for the different
   migration models
 */
void
force_symmetric_d (double *gxv, long model, nr_fmt * nr, long nn)
{
  switch (model)
    {
    case MATRIX:
      break;
    case MATRIX_ARBITRARY:
      check_symmetric_d (gxv, nr, nn);
      break;
    case MATRIX_SAMETHETA:
      calc_symmetric_d (gxv, nr, nn, 0, nr->numpop);
      break;
    case ISLAND:
      calc_symmetric_d (gxv, nr, nn, 0, nr->numpop);
      calc_symmetric_d (gxv, nr, nn, nr->numpop, nr->numpop2);
      break;
    case ISLAND_VARTHETA:
      calc_symmetric_d (gxv, nr, nn, nr->numpop, nr->numpop2);
      break;
    }
}

void
calc_same_d (double *grad, nr_fmt * nr, long nn, long start, long stop)
{
  long i, ii;
  long z = 0;
  double dt = 0;
  long nsum = 0;
  char *custm = nr->world->options->custm2;
  if (nr->profilenum == 0)
    {
      for (i = start; i < stop; i++)
        {
          if (custm[i] == 'm')
            {
              dt += grad[i];
              nsum++;
            }
        }
      dt /= nsum;
      z = 0;
      for (i = start; i < stop; i++)
        {
          if (custm[i] == 'm')
            grad[i] = dt;
        }
    }
  else
    {
      for (i = 0; i < nn; i++)
        {
          ii = (nr->profilenum > 0) ? nr->indeks[z++] : i;
          if (ii >= start && ii < stop && custm[ii] == 'm')
            {
              dt += grad[i];
              nsum++;
            }
        }
      dt /= nsum;
      z = 0;
      for (i = 0; i < nn; i++)
        {
          ii = (nr->profilenum > 0) ? nr->indeks[z++] : i;
          if (ii >= start && ii < stop && custm[ii] == 'm')
            grad[i] = dt;
        }
    }
}

void
calc_symmetric_d (double *grad, nr_fmt * nr, long nn, long start, long stop)
{
  long i, ii;
  long z = 0;
  double dt = 0;
  if (nr->profilenum == 0)
    {
      for (i = start; i < stop; i++)
        {
          dt += grad[i];
        }
      dt /= (stop - start);
      z = 0;
      for (i = start; i < stop; i++)
        {
          grad[i] = dt;
        }
    }
  else
    {
      for (i = 0; i < nn; i++)
        {
          ii = (nr->profilenum > 0) ? nr->indeks[z++] : i;
          if (ii >= start && ii < stop)
            dt += grad[i];
        }
      dt /= (stop - start);
      z = 0;
      for (i = 0; i < nn; i++)
        {
          ii = (nr->profilenum > 0) ? nr->indeks[z++] : i;
          if (ii >= start && ii < stop)
            grad[i] = dt;
        }
    }
}

void
check_symmetric_d (double *grad, nr_fmt * nr, long nn)
{
  char *custm = NULL;
  long zeron = 0;
  long constn = 0;
  long symn = 0;
  long sym2n = 0;
  long mmn = 0;
  long numpop = 0;
  /*static boolean done = FALSE; */
  twin_fmt *syms = NULL;
  quad_fmt *sym2s = NULL;
  long *zeros = NULL;
  long *consts = NULL;
  long *mms = NULL;
  long count;
  long z = 0, zz = 0, i;
  char *p;
  double mm, sq1, sq2;


  custm = nr->world->options->custm2;
  zeron = nr->world->options->zeron;
  constn = nr->world->options->constn;
  symn = nr->world->options->symn;
  sym2n = nr->world->options->sym2n;
  mmn = nr->world->options->mmn;
  if (symn > 0)
    syms = nr->world->options->symparam;
  if (sym2n > 0)
    sym2s = nr->world->options->sym2param;
  if (zeron > 0)
    zeros = nr->world->options->zeroparam;
  if (constn > 0)
    consts = nr->world->options->constparam;
  if (mmn > 0)
    mms = nr->world->options->mmparam;
  p = custm;
  custm = nr->world->options->custm2;
  numpop = nr->numpop;
  p = custm;
  z = 0;
  for (count = 0; count < numpop; count++)
    {
      if (*p == 'm')
        z++;
      p++;
    }
  if (z > 0)
    calc_same_d (grad, nr, nn, 0, nr->numpop);
  p = custm + nr->numpop;
  z = 0;
  for (count = numpop; count < numpop * numpop; count++)
    {
      if (*p == 'm')
        z++;
      p++;
    }
  if (z > 0)
    calc_same_d (grad, nr, nn, nr->numpop, nr->numpop2);
  /* if there are symmetric migration rates M go here*/
  for (i = 0; i < symn; i++)
    {
      if (nr->profilenum > 0)
        {
          z = 0;
          while (nr->indeks[z] != syms[i][0] && z++ < nr->partsize)
            ;
          zz = 0;
          while (nr->indeks[zz] != syms[i][1] && zz++ < nr->partsize)
            ;
          if (z < nr->partsize)
            mm = grad[z];
          else
            mm = 0;
          if (zz < nr->partsize)
            mm += grad[zz];
          //else
          //  mm += 0;
          grad[z] = grad[zz] = mm;
        }
      else
        {
          mm = grad[syms[i][0]] + grad[syms[i][1]];
          grad[syms[i][0]] = grad[syms[i][1]] = mm;
        }
    }
  /* if there are symmetric migration rates 4Nm */
  if (nr->profilenum > 0)
    {
      for (i = 0; i < sym2n; i++)
        {
          sq1 = nr->param[sym2s[i][2]] ;//* nr->param[sym2s[i][2]]);
          sq2 = nr->param[sym2s[i][3]];// * nr->param[sym2s[i][3]]);
          z = 0;
          while (nr->indeks[z] != sym2s[i][0] && z++ < nr->partsize)
            ;
          zz = 0;
          while (nr->indeks[zz] != sym2s[i][1] && zz++ < nr->partsize)
            ;
          if (z < nr->partsize)
            mm = grad[z] / sq1;
          else
            mm = 0;
          if (zz < nr->partsize)
            mm += grad[zz] / sq2;
          //else
          //  mm += 0;
          //  mm /= 2.;
          grad[z] = mm * sq1;
          grad[zz] = mm * sq2;
        }
    }
  else
    {
      for (i = 0; i < sym2n; i++)
        {
          sq1 = nr->param[sym2s[i][2]];// * nr->param[sym2s[i][2]];
          sq2 = nr->param[sym2s[i][3]];//* nr->param[sym2s[i][3]];
          mm = (grad[sym2s[i][0]] / sq1 + grad[sym2s[i][1]] / sq2);
          grad[sym2s[i][0]] = mm * sq1;
          grad[sym2s[i][1]] = mm * sq2;
        }
    }
}

void
symmetric_other (long i, long numpop, long *other, long *otherpop)
{
  long pop;
  m2mm (i, numpop, otherpop, &pop);
  *other = mm2m (pop, *otherpop, numpop);
}


/* derivatives to log derivatives */
void
grad2loggrad (double *param, long *indeks, double *d, long nn,
              long profilenum)
{
  long i, ii, z = 0;
  for (i = 0; i < nn; i++)
    {
      ii = (profilenum > 0) ? indeks[z++] : i;
      d[i] = -param[ii] * d[i];
      /*the minus is for minimizing  the function -L
         instead of maximizing L
      */
    }
}

double
probG (double *param, double *lparam, tarchive_fmt * tl, nr_fmt * nr,
       long locus)
{
  const int numpop = (int) nr->numpop;
  int i, j, msta, msto;
  double result = 0., sm;
  double *geo = nr->world->data->geo;
  double *lgeo = nr->world->data->lgeo;
  double mu_rate = nr->world->options->mu_rates[locus];
  double lmu_rate = nr->world->options->lmu_rates[locus];
  for (i = 0; i < numpop; i++)
    {
      if (lparam[i] <= -DBL_MAX)
        return DBL_MAX;
      result += tl->p[i] * (LOG2 - (lparam[i] + lmu_rate));
      msta = nr->mstart[i];
      msto = nr->mend[i];
      sm = 0.0;
      for (j = msta; j < msto; j++)
        {
          if (param[j] > 0.0)
            {
              result += tl->mindex[j] * (lgeo[j] + lparam[j] - lmu_rate);
              sm += geo[j] * param[j] / mu_rate;
            }
        }
      result += -tl->km[i] * sm - tl->kt[i] / (param[i] * mu_rate);
    }
  return result;
}

double
probG4 (double *fullparam, double *data, long numpop, long numpop2)
{
  int i;
  double result = 0.;
  // fullparam is a linearized
  //     log(param) [size: numpop * numpop] [Log(2/theta), Log(M)]
  //     param [thetas are 1/param] [size: numpop * numpop]
  //     sum(migparam) [size: numpop]
  // data is linearized
  //     p      [size: numpop]
  //     mindex [sizes: numpop * (numpop -1)]
  //     kt     [size: numpop]
  //     km     [size: numpop]
  //
  //  calculation: p * log(2/theta) + mindex * log(M)
  for (i = 0; i < numpop2; i++)
    {
      if (fullparam[i] > -DBL_MAX)
        result += fullparam[i] * data[i];
    }
  // - kt * 1/theta - km * sm
  for (i = numpop2; i < numpop2 + 2 * numpop; i++)
    {
      result -= fullparam[i] * data[i];
    }
  return result;
}

void
probG3 (double *apg, double *apg0r, timearchive_fmt * tyme,
        long numpop, double *apgmax,
        double *param, double *lparam, double *sm)
{
  long g, g1, g2, g3, i, j;
  long msta, me;
  double temp, temp1, temp2;
  double result1, result2, result3, result4;
  double tparam;
  tarchive_fmt *tl1;
  tarchive_fmt *tl2;
  tarchive_fmt *tl3;
  tarchive_fmt *tl4;
  long remaind = tyme->T % 4;

  if (remaind > 0)
    {
      for (g = 0; g < remaind; g++)
        {
          tl1 = &(tyme->tl[g]);
          result1 = 0.;
          for (i = 0; i < numpop; i++)
            {
              msta = numpop + i * (numpop - 1);
              me = msta + numpop - 1;
              temp = (LOG2 - lparam[i]);
              result1 += tl1->p[i] * temp;
              for (j = msta; j < me; j++)
                {
                  if (param[j] > 0.0)
                    {
                      result1 += tl1->mindex[j] * lparam[j];
                    }
                }
              result1 += -tl1->km[i] * sm[i] - tl1->kt[i] / param[i];
            }
          apg[g] = result1 - apg0r[g];
          if (apg[g] > *apgmax)
            *apgmax = apg[g];
        }
    }
  for (g = remaind; g < tyme->T; g += 4)
    {
      g1 = g + 1;
      g2 = g1 + 1;
      g3 = g2 + 1;
      tl1 = &(tyme->tl[g]);
      tl2 = &(tyme->tl[g1]);
      tl3 = &(tyme->tl[g2]);
      tl4 = &(tyme->tl[g3]);
      result1 = 0.;
      result2 = 0.;
      result3 = 0.;
      result4 = 0.;
      for (i = 0; i < numpop; i++)
        {
          tparam = 1. / param[i];
          msta = numpop + i * (numpop - 1);
          me = msta + numpop - 1;
          temp = (LOG2 - lparam[i]);
          result1 += tl1->p[i] * temp;
          result2 += tl2->p[i] * temp;
          result3 += tl3->p[i] * temp;
          result4 += tl4->p[i] * temp;
          for (j = msta; j < me; j++)
            {
              if (param[j] > 0.0)
                {
                  result1 += tl1->mindex[j] * lparam[j];
                  result2 += tl2->mindex[j] * lparam[j];
                  result3 += tl3->mindex[j] * lparam[j];
                  result4 += tl4->mindex[j] * lparam[j];
                }
            }
          result1 += -tl1->km[i] * sm[i] - tl1->kt[i] * tparam;
          result2 += -tl2->km[i] * sm[i] - tl2->kt[i] * tparam;
          result3 += -tl3->km[i] * sm[i] - tl3->kt[i] * tparam;
          result4 += -tl4->km[i] * sm[i] - tl4->kt[i] * tparam;
        }
      apg[g] = result1 - apg0r[g];
      apg[g1] = result2 - apg0r[g1];
      apg[g2] = result3 - apg0r[g2];
      apg[g3] = result4 - apg0r[g3];
      temp1 = MAX (apg[g], apg[g1]);
      temp2 = MAX (apg[g2], apg[g3]);
      if ((temp = MAX (temp1, temp2)) > *apgmax)
        *apgmax = temp;
    }
}

double
probG2 (double *param, double *lparam, double *sm,
        double *kt, double *km, double *p, double *mindex,
        int *msta, int *me, long numpop)
{
  int i, j;
  double result = 0.;
  for (i = 0; i < numpop; i++)
    {
      if (lparam[i] <= -DBL_MAX)
        return DBL_MAX;
      //      result += p[i] * (LOG2 - lparam[i]);
      for (j = msta[i]; j < me[i]; j++)
        {
          if (param[j] > 0.0)
            result += mindex[j] * lparam[j];
        }
      result += p[i] * lparam[i] - km[i] * sm[i] - kt[i] * param[i];
      //      result += -km[i] * sm[i] - kt[i] / param[i];
    }
  return result;
}


boolean
is_singular (double **dd, long n)
{
  long i, j;
  double temp;
  boolean singular = FALSE;
  for (i = 0; i < n; i++)
    {
      temp = 0.0;
      for (j = 0; j < n; j++)
        {
          temp += dd[i][j];
        }
      if (temp == 0.0)
        {
          singular = TRUE;
          break;
        }
    }
  for (i = 0; i < n; i++)
    {
      temp = 0.0;
      for (j = 0; j < n; j++)
        {
          temp += dd[i][j];
        }
      if (temp == 0.0)
        {
          singular = TRUE;
          break;
        }
    }
  return singular;
}


void
calc_cov (double **dd, double *d, double *param, long n)
{
  long i, j;
  if (!is_singular (dd, n))
    invert_matrix (dd, n);
  else
    {
      reset_hess (dd, n);
      return;
    }
  for (i = 0; i < n; i++)
    {
      for (j = 0; j < i; j++)
        {
          dd[i][j] /= (param[i] * param[j]);
          dd[j][i] = dd[i][j];
        }
      dd[i][i] = (dd[i][i] - param[i] * d[i]) / (param[i] * param[i]);
    }
  if (!is_singular (dd, n))
    invert_matrix (dd, n);
  else
    reset_hess (dd, n);
}


void
print_contribution (nr_fmt * nr, timearchive_fmt ** atl, long G)
{
  long g, i, r;
  double temp = 0, temp2 = 0, maxtemp;
  FILE *mixfile = NULL;
  long events = 0;
  long contribution[11];
  long locus = nr->world->locus;
  tarchive_fmt *tyme;
  for (g = 0; g < 11; g++)
    contribution[g] = 0;
  if (nr->world->options->mixplot)
    {
      mixfile = fopen ("mixfile", "a");
      fprintf (mixfile, "\n\n");
    }
  maxtemp = -DBL_MAX;
  for (r = nr->repstart; r < nr->repstop; r++)
    {
      for (g = 0; g < G; g++)
        {
          tyme = atl[r][nr->world->locus].tl;
          temp = nr->apg[r][locus][g] + nr->apg0[r][locus][g];
          temp2 = temp + nr->world->likelihood[g];
          if (temp2 > maxtemp)
            maxtemp = temp2;
          if (nr->world->options->mixplot)
            {
              events = 0;
              for (i = nr->world->numpop; i < nr->world->numpop2; i++)
                events += (long) tyme[g].mindex[i];
              for (i = 0; i < tyme[g].copies; i++)
                fprintf (mixfile, "%li %f %f ", events, temp,
                         nr->world->likelihood[g]);
            }
          temp2 -= maxtemp;
          if (temp2 > -20)
            {
              contribution[9 - (long) (fabs (temp2) / 2)] +=
                (g > 0) ? tyme[g].copies : tyme[g].copies - 1;
            }
          contribution[10] += (g > 0) ? tyme[g].copies : tyme[g].copies - 1;
        }
    }
  if (nr->world->options->progress)
    {
      print_menu_contribution (stdout, contribution);
      if (nr->world->options->writelog)
        print_menu_contribution (nr->world->options->logfile, contribution);
    }
  if (nr->world->options->mixplot)
    fclose (mixfile);
}

void
print_menu_contribution (FILE * file, long contribution[])
{
  long g;
  fprintf (file, "           log(P(g|Param) * P(data|g)\n");
  fprintf (file, "                            -20 to ");
  for (g = -18; g <= 0; g += 2)
    {
      fprintf (file, "%4li ", g);
    }
  fprintf (file, "  All\n");
  fprintf (file, "           Counts                  ");
  for (g = 0; g < 10; g++)
    {
      fprintf (file, "%4li ", contribution[g]);
    }
  fprintf (file, "%5li\n", contribution[10]);
}

/* calculates log(parameters)*/
void
log_param0 (double *param, double *lparam, long nn)
{
  long i;
  for (i = 0; i < nn; i++)
    {
      if (param[i] > 0)
        lparam[i] = log (param[i]);
      else
        lparam[i] = -30;	//-DBL_MAX;
    }
}


void
copies2lcopies (timearchive_fmt * atl)
{
  long g;
  if (atl->tl[0].copies > 1)
    atl->tl[0].lcopies = ln_copies (atl->tl[0].copies - 1);
  else
    atl->tl[0].lcopies = -DBL_MAX;
  for (g = 1; g < atl->T; g++)
    {
      atl->tl[g].lcopies = ln_copies (atl->tl[g].copies);
    }
}

void
create_apg0 (double *apg0, nr_fmt * nr, timearchive_fmt * tyme, long locus)
{
  long g;
  long copies;
  /* Prob(G|Param0) */
//  fprintf(stdout,"first 4 theta0: %f %f %f %f \n",tyme->param0[0], tyme->param0[1], tyme->param0[2], tyme->param0[3]);
		  for (g = 0; g < tyme->T; g++)
    {
      if (g > 0)
        copies = tyme->tl[g].copies;
      else
        copies = tyme->tl[g].copies - 1;
      if (copies == 0)
        apg0[g] = -DBL_MAX;
      else
        apg0[g] =
          probG (tyme->param0, tyme->lparam0, &tyme->tl[g], nr, locus);
    }
	//	  fprintf(stdout,"first 4 apg0: %f %f %f %f \n",apg0[0], apg0[1], apg0[2], apg0[3]);

}

double
logprob_noevent (world_fmt * world, long interval)
{
  long pop, k;
  double result = 0.0;
  for (pop = 0; pop < world->numpop; pop++)
    {
      k = world->treetimes[0].tl[interval].lineages[pop];
      result +=
        (k * (k - 1) / world->param0[pop]) + sum_migprob (world, pop,
            interval);
    }
  return result;
}

double
sum_migprob (world_fmt * world, long pop, long interval)
{
  long i;
  double result = 0.0;
  long *lineages = world->treetimes[0].tl[interval].lineages;
  long msta = world->mstart[pop];
  long msto = world->mend[pop];
  for (i = msta; i < msto; i++)
    {
      result += world->param0[i];
    }
  return result * lineages[pop];
}

double
sum_mig (double *param, long msta, long msto)
{
  long i;
  double result = 0.0;
  for (i = msta; i < msto; i++)
    {
      result += param[i];
    }
  return result;
}

// penalizes if theta is too far from theta0
// std22 = std*std*2
double normal_func_ok(double *param, double *param0, long numpop2)
{
  long i;
  double p0;
  double result=0.0;
  double diff;
  for(i=0; i<numpop2;i++)
    {
      diff = param[i]-(p0=param0[i]);
      result += -(diff*diff/(2. * p0 *p0)) - 0.91893853320467274178 - log(p0);//(STD22))+ INVTWOSQRTPILOGSTD;
    }
  return result;
}

double normal_func_no(double *param, double *param0, long numpop2)
{
  return 0.0;
}

double normal_func_gradient_ok(double p1, double p0)
{

  return  (p1-p0)/(p0*p0);//STD2;
}

double normal_func_gradient_no(double p1, double p0)
{
  return 0.0;
}

void unset_penalizer_function(boolean inprofiles)
{
  if(inprofiles)
    {
      normal_func = (double (*)(double *, double *, long)) normal_func_no;
      normal_func_gradient = (double (*)(double , double)) normal_func_gradient_no;
    }
  else
    {
      normal_func = (double (*)(double *, double *,  long)) normal_func_ok;
      normal_func_gradient = (double (*)(double, double)) normal_func_gradient_ok;
    }
}

