/*------------------------------------------------------
 Maximum likelihood estimation
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm
 -------------------------------------------------------
 W O R L D   R O U T I N E S
 
 creates tree structures
 
 prints results,
 and finally helps to destroy itself.
 
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 
Copyright 1996-2002 Peter Beerli and Joseph Felsenstein
  
        This software is distributed free of charge for non-commercial use
        and is copyrighted. Of course, we do not guarantee that the software
        works and are not responsible for any damage you may cause or have.
 
 
$Id: world.c,v 1.101 2001/09/07 23:56:13 beerli Exp $
-------------------------------------------------------*/
#define SICK_VALUE    -1
#include "migration.h"
#include "mcmc.h"

#include "fst.h"
#include "random.h"
#include "tools.h"
#include "bayes.h"
#include "broyden.h"
#include "combroyden.h"
#include "lrt.h"
#include "laguerre.h"
#include "options.h"
#include "tree.h"
#ifndef LAGUERRE
#include "derivatives2.h"
#endif
#include "sequence.h"
#include "sort.h"
#include "migrate_mpi.h"
#include "mig-histogram.h"
#ifdef UEP
#include "uep.h"
#endif
#ifdef ALTIVEC
#include "altivec.h"
#endif

#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif

#define PLANESIZE 36
#define PLANEBIGTICKS 6
// #define PLANEBIGTICKVALUES {-3, -2, -1, 0, 1, 2}
#define PLANEBIGTICKVALUES {1,2,3,4,5,6}
/*
 * #define PLANETICKS   {0.001, 0.0013895, 0.0019307, 0.0026827, 0.00372759,
 * \ 0.00517947, 0.00719686, 0.01, 0.013895, 0.019307, \ 0.026827, 0.0372759,
 * 0.0517947, 0.0719686, 0.1, \ 0.13895, 0.19307, 0.26827, 0.372759,
 * 0.517947, \ 0.719686, 1., 1.3895, 1.9307, 2.6827, 3.72759, \ 5.17947,
 * 7.19686, 10., 13.895, 19.307, 26.827, 37.2759, \ 51.7947, 71.9686, 100.}
 */
#define CONTOURLEVELS 8
#define CONTOURS_LOCUS {0.0,-3.35670/2., -9.48773/2., -13.2767/2.,\
                        0.0,-3.35670/2., -9.48773/2., -13.2767/2.}
#define CONTOURS_LOCI  {0.0,-4.35146/2., -11.0705/2., -15.0863/2.,\
                        0.0,-4.35146/2., -11.0705/2., -15.0863/2.}

/* prototypes ------------------------------------------- */
void create_world (world_fmt ** world, char *name, long loci);
void init_world (world_fmt * world, data_fmt * data, option_fmt * options);
void calc_simple_param (world_fmt * world, data_fmt * data);
void print_menu_locus (world_fmt * world, long locus);
void print_menu_chain (char type, long chain, long steps, world_fmt * world,
                       option_fmt * options, long rep);
void set_bounds (long *increment, long *steps, long *chain,
                 const option_fmt * options, const char type);
void burnin_chain (world_fmt * world);
void print_list (world_fmt ** universe, option_fmt * options, data_fmt * data);
void plot_surface (world_fmt * world, option_fmt * options,
                   data_fmt * data, char ****plane, long x);
void plot_surface_header2 (FILE * outfile, long locus, char plotvar[]);
void plot_surface2 (FILE * outfile, long x, long locus, long numpop,
                    char ****plane, plotmax_fmt ** plotmax,
                    char **popnames, char plotvar[], long migrmodel);
void print_alpha_curve (world_fmt * world, timearchive_fmt ** atl, long *gmaxptr);
void create_loci_plot (world_fmt * world, char ***plane,
                       nr_fmt * nr, long loci);
void create_locus_plot (world_fmt * world, char ***plane,
                        nr_fmt * nr, long G);
void cleanup_world (world_fmt * world, long locus);
void free_universe (world_fmt ** worlds, long numworlds);
void free_world(world_fmt *world);
void precalc_world (world_fmt * world);
extern double prob_tree (world_fmt * world, timelist_fmt * tyme);
extern void set_tree_dirty (node * p);
/* private functions */
void create_timearchive (timearchive_fmt *** atl, long loci,
                         long samples, long numpop, long replicates);
void create_plotplane (world_fmt * world);
void create_cov (world_fmt * world);
void print_menu_equilib (worldoption_fmt * options);
void print_menu_heating (worldoption_fmt * options);
void print_finish (world_fmt * world, long filepos);
void copy_time (world_fmt * world, timelist_fmt * ltl, long from,
                long to, long np, long rep);
void archive_timelist (tarchive_fmt * atl, vtlist * tl, long T,
                       long np, world_fmt * world);
void copy_timelist (tarchive_fmt * from, tarchive_fmt * to, long np);
long whichp (long from, long to, long pop);
/* long whichl (long from, long to, long pop); */
/* long calc_T(timearchive_fmt * old, long steps); */
void print_results (world_fmt ** universe, option_fmt * options, data_fmt * data);
void print_fst (world_fmt * world, option_fmt * options,
                data_fmt * data, double **fstparam);
void prepare_print_nu (double nu, char *str);
void prepare_print_nm (double nm, double nmu, char *strllike);
void print_menu_coalnodes (FILE * file, world_fmt * world, long G, long rep);
void print_menu_createplot (void);
void calc_loci_plane (world_fmt * world, nr_fmt * nr, double ***pl,
                      long loci, double **contours);
void calc_locus_plane (world_fmt * world, nr_fmt * nr, long G,
                       double ***pl, double **contours);
void fill_plotplane (world_fmt * world, char **plane, double **pl,
                     double *contours);
void print_mathematica (world_fmt * world, double ***plane, long x, long y);
void print_cov (world_fmt * world, long numpop, long loci, double ***cov);
void print_cov_table (FILE * outfile, long locus, world_fmt * world,
                      double *corr, long addvar);
void free_seqx (node * p, long sites);
void free_x (node * p);
void increase_timearchive (world_fmt * world, long locus,
                           long sample, long numpop, long rep);
void print_CV (world_fmt * world);
void print_param (FILE * file, boolean usem, world_fmt *world, long nn,
                  char spacer[]);
void set_contours (double **contours, long df, long numpop);
void print_result_header (char *titletext, world_fmt * world);
void print_result_population (long pop, world_fmt * world,
                              option_fmt * options, data_fmt * data);
void print_result_param (FILE * file, double *param, long numpop, long pop,
                         boolean usem);
void print_result_fst (long pop, world_fmt * world, data_fmt * data);
void prognose_time (char *nowstr, world_fmt * world,
                    option_fmt * options, long steps, char *spacer);
void klone (world_fmt * original, world_fmt * kopie,
            option_fmt * options, data_fmt * data, double temperature);
void klone_part (world_fmt * original, world_fmt * kopie,
                 option_fmt * options, data_fmt * data, double temperature);
void clone_polish (world_fmt * original, world_fmt * kopie);
long chance_swap_tree (world_fmt * tthis, world_fmt * that);
void advance_clone_like (world_fmt * world, long accepted, long *j);
void polish_world (world_fmt * world);
void fill_worldoptions (worldoption_fmt * wopt,
                        option_fmt * options, long numpop);
void fill_worlddata (worlddata_fmt * wdata, world_fmt * world,
                     data_fmt * data, long numpop, boolean readsum);
void print_replicate(world_fmt *world, long maxrep, long rep, long locus);
#ifdef LONGSUM
void print_fluctuate_header(world_fmt *world);
void setup_fluctuate(world_fmt *world, option_fmt * options);
void print_popstring(long pop, world_fmt *world, option_fmt *options, data_fmt *data);
void print_fluctuate(world_fmt **universe, option_fmt *options, data_fmt *data);
#endif /*LONGSUM*/

extern void   unset_penalizer_function(boolean inprofiles);

/* Functions ++++++++++++++++++++++++++++++++++++++++++++++++ */
void
create_world (world_fmt ** world, char *name, long loci)
{
  (*world) = (world_fmt *) calloc (1, sizeof (world_fmt));
  (*world)->treetimes = (timelist_fmt *) calloc (1, sizeof (timelist_fmt));
  (*world)->param_like = -DBL_MAX;
  (*world)->name = (char *) calloc (11, sizeof (char));
  strncpy ((*world)->name, name, 10);
}

void
fill_worldoptions (worldoption_fmt * wopt, option_fmt * options, long numpop)
{
  long i;
  long numpop2 = numpop * numpop;
  wopt->gamma = options->gamma;
  wopt->alphavalue = options->alphavalue;
  wopt->murates = options->murates;
  wopt->mu_rates = (double *) calloc (options->muloci, sizeof (double));
  wopt->lmu_rates = (double *) calloc (options->muloci, sizeof (double));
  if (options->murates)
    {
      memcpy (wopt->mu_rates, options->mu_rates,
              sizeof (double) * options->muloci);
      for (i = 0; i < options->muloci; i++)
        {
          wopt->lmu_rates[i] = log (options->mu_rates[i]);
        }
    }
  else
    {
      for (i = 0; i < options->muloci; i++)
        {
          wopt->mu_rates[i] = 1.;
        }
    }
  wopt->migration_model = options->migration_model;
  wopt->custm = (char *) calloc (numpop2 + 2, sizeof (char));
  strcpy (wopt->custm, options->custm);
  wopt->custm2 = (char *) calloc (numpop2 + 2, sizeof (char));
  strcpy (wopt->custm2, options->custm2);
  wopt->thetag = (double *) calloc (numpop2 + 2, sizeof (double));
  memcpy (wopt->thetag, options->thetag,
          sizeof (double) * options->numthetag);
  wopt->mg = (double *) calloc (numpop2 + 2, sizeof (double));
  memcpy (wopt->mg, options->mg, sizeof (double) * options->nummg);
  wopt->zeron = options->zeron;
  wopt->zeroparam = (long *) calloc (1 + wopt->zeron, sizeof (long));
  memcpy (wopt->zeroparam, options->zeroparam, sizeof (long) * wopt->zeron);
  wopt->constn = options->constn;
  wopt->constparam = (long *) calloc (1 + wopt->constn, sizeof (long));
  memcpy (wopt->constparam, options->constparam,
          sizeof (long) * wopt->constn);
  wopt->symn = options->symn;
  wopt->symparam = (twin_fmt *) calloc (1 + wopt->symn, sizeof (twin_fmt));
  memcpy (wopt->symparam, options->symparam, sizeof (twin_fmt) * wopt->symn);
  wopt->sym2n = options->sym2n;
  wopt->sym2param = (quad_fmt *) calloc (1 + wopt->sym2n, sizeof (quad_fmt));
  memcpy (wopt->sym2param, options->sym2param,
          sizeof (quad_fmt) * wopt->sym2n);
  wopt->mmn = options->mmn;
  wopt->mmparam = (long *) calloc (1 + wopt->mmn, sizeof (long));
  memcpy (wopt->mmparam, options->mmparam, sizeof (long) * wopt->mmn);
  wopt->mixplot = options->mixplot;
  wopt->progress = options->progress;
  wopt->writelog = options->writelog;
  wopt->logfile = options->logfile;
  wopt->plotnow = options->plotnow;
  wopt->verbose = options->verbose;
  wopt->replicate = options->replicate;
  wopt->gelman = options->gelman;
  wopt->lcepsilon = options->lcepsilon;
  wopt->simulation = options->simulation;
  wopt->datatype = options->datatype;
  wopt->lsteps = options->lsteps;
  wopt->loglsteps = log (options->lsteps);
  wopt->treeprint = options->treeprint;
  wopt->movingsteps = options->movingsteps;
  wopt->acceptfreq = options->acceptfreq;
  wopt->rcategs = options->rcategs;
  wopt->categs = options->categs;
  wopt->heating = options->heating;
  wopt->heated_chains = options->heated_chains;
  wopt->heating_interval = options->heating_interval;
  wopt->adaptiveheat = options->adaptiveheat;
  wopt->profilemethod = options->profilemethod;
  wopt->printprofile = options->printprofile;
  wopt->printprofsummary = options->printprofsummary;
  wopt->profileparamtype = options->profileparamtype;
  wopt->df = options->df;
  wopt->lchains = options->lchains;
  wopt->replicatenum = options->replicatenum;
  wopt->micro_threshold = options->micro_threshold;
  wopt->micro_stepnum = options->micro_stepnum;
  wopt->rrate = (double *) calloc (wopt->rcategs, sizeof (double));
  memcpy (wopt->rrate, options->rrate, sizeof (double) * wopt->rcategs);
  wopt->rate = (double *) calloc (wopt->categs, sizeof (double));
  memcpy (wopt->rate, options->rate, sizeof (double) * wopt->categs);
  wopt->probcat = (double *) calloc (wopt->rcategs, sizeof (double));
  memcpy (wopt->probcat, options->probcat, sizeof (double) * wopt->rcategs);
  wopt->fastlike = options->fastlike;
  wopt->pluschain = options->pluschain;
  wopt->mighist = options->mighist;
  wopt->burn_in = options->burn_in;
  wopt->usem = options->usem;
  if ((wopt->plot = options->plot))
    {
      wopt->plotmethod = options->plotmethod;
      wopt->plotintervals = options->plotintervals;
      memcpy (wopt->plotrange, options->plotrange, sizeof (double) * 4);
      wopt->plotscale = options->plotscale;
      wopt->plotvar = options->plotvar;
      wopt->plotxvalues =
        (double *) calloc (wopt->plotintervals, sizeof (double));
      wopt->plotyvalues =
        (double *) calloc (wopt->plotintervals, sizeof (double));
      memcpy (wopt->plotxvalues, options->plotxvalues,
              sizeof (double) * wopt->plotintervals);
      memcpy (wopt->plotyvalues, options->plotyvalues,
              sizeof (double) * wopt->plotintervals);
    }
  wopt->lratio = options->lratio;
  //non - threaded !
  wopt->fast_aic = options->fast_aic;
  wopt->aic = options->aic;
  wopt->aicmod = options->aicmod;
  wopt->aicfile = options->aicfile;
  strcpy(wopt->aictype, options->aictype);
  wopt->lambda = options->lambda;
#ifdef BAYESUPDATE

  wopt->bayes_infer = options->bayes_infer;
#endif
#ifdef UEP

  wopt->uep = options->uep;
  wopt->ueprate = options->ueprate;
  wopt->uepmu = options->uepmu;
  wopt->uepnu = options->uepnu;
  wopt->uepfreq0 = options->uepfreq0;
  wopt->uepfreq1 = options->uepfreq1;
#endif
}

void
fill_worlddata (worlddata_fmt * wdata, world_fmt * world,
                data_fmt * data, long numpop, boolean readsum)
{
  long numpop2 = numpop * numpop;
  wdata->skiploci =
    (boolean *) calloc (1, sizeof (boolean) * (data->loci + 1));
  wdata->geo = (double *) calloc (1, sizeof (double) * numpop2);
  memcpy (wdata->geo, data->geo, sizeof (double) * numpop2);
  wdata->lgeo = (double *) calloc (1, sizeof (double) * numpop2);
  memcpy (wdata->lgeo, data->lgeo, sizeof (double) * numpop2);
  if (!readsum)
    {
      wdata->maxalleles = (long *) calloc (data->loci, sizeof (long));
      memcpy (wdata->maxalleles, data->maxalleles,
              sizeof (long) * data->loci);
      wdata->seq->sites = (long *) calloc (data->loci, sizeof (long));
      memcpy (wdata->seq->sites, data->seq->sites,
              sizeof (long) * data->loci);
      wdata->seq->addon = data->seq->addon;
      wdata->sumfile = data->sumfile;
#ifdef UEP

      wdata->uepsites = data->uepsites;
#endif

    }
}


void
init_world (world_fmt * world, data_fmt * data, option_fmt * options)
{
  long locus, i, rep;
  long custmlen = 0;
  long addition;
  world->options = (worldoption_fmt *) calloc (1, sizeof (worldoption_fmt));
  world->buffer = (char *) calloc (LINESIZE, sizeof (char));
#ifdef MPI

  world->who = (int *) calloc (data->loci, sizeof (int));
#ifdef SLOWNET

  world->profilewho =
    (int *) calloc (data->numpop * data->numpop + 2, sizeof (int));
#endif
#endif

  world->data = (worlddata_fmt *) calloc (1, sizeof (worlddata_fmt));
  world->data->seq = (seqmodel_fmt *) calloc (1, sizeof (seqmodel_fmt));
  options->muloci = data->loci;
  fill_worldoptions (world->options, options, data->numpop);
  fill_worlddata (world->data, world, data, data->numpop, options->readsum);
  if (!options->readsum)
    {
      world->loci = data->loci;
      world->skipped = 0;
      world->numpop = data->numpop;
      world->numpop2 = world->numpop * world->numpop;
      if (world->numpop == 1)
        options->plot = FALSE;
    }
#ifdef BAYESUPDATE
  if(options->bayes_infer)
    {
      world->bayes = calloc(1, sizeof(bayes_fmt));
      bayes_init(world->bayes,world->numpop2);
      bayes_fill(world, options);
    }
#endif

  custmlen = (long) strlen (options->custm);
  fillup_custm (custmlen, world, options);
  world->mig0list = (double *) calloc (world->numpop, sizeof (double));
  //for precalc_world
  world->migproblist = (double **) calloc (world->numpop, sizeof (double *));
  //for precalc_world
  for (i = 0; i < world->numpop; ++i)
    {
      world->migproblist[i] =
        (double *) calloc (world->numpop, sizeof (double));
    }
  world->design0list = (long *) calloc (world->numpop, sizeof (long));
  //for precalc_world
  rep = world->repstop =
          world->options->replicate ? (world->options->replicatenum >
                                       0 ? world->options->
                                       replicatenum : world->options->lchains) : 1;
  create_timearchive (&(world->atl), world->loci, SAMPLETREE_GUESS,
                      world->numpop, rep);
  if (world->numpop > 1)
    create_plotplane (world);
  create_cov (world);
  addition = 1;
#ifdef LONGSUM

  addition = 1 + world->numpop * 3;
#endif /*LONGSUM*/

  world->likelihood = (double *) calloc (1, sizeof (double) * SAMPLETREE_GUESS);
  world->lineages = (long *) calloc (world->numpop, sizeof (long));
  world->param0 = (double *) calloc (world->numpop2 + addition, sizeof (double));
  world->param00 = (double *) calloc (world->numpop2 +addition, sizeof (double));
  world->heat = (double *) calloc (1, sizeof (double) * 1
                                   /* realloc in heating_ratio() */ );
  world->mstart = (int *) calloc (world->numpop, sizeof (int));
  world->mend = (int *) calloc (world->numpop, sizeof (int));
  for (i = 0; i < world->numpop; i++)
    {
      world->mstart[i] = (short) mstart (i, world->numpop);
      world->mend[i] = (short) mend (i, world->numpop);
    }
  switch (options->datatype)
    {
    case 's':
    case 'n':
    case 'u':
    case 'a':
    case 'f':
      world->fstparam =
        (double **) calloc (1, sizeof (double *) * (world->loci + 1));
      for (locus = 0; locus < world->loci + 1; locus++)
        world->fstparam[locus] =
          (double *) calloc (1, sizeof (double) * world->numpop2 * 2);
      break;
    case 'm':
      world->options->steps =
        (double **) calloc (1,
                            sizeof (double *) *
                            world->options->micro_stepnum);
      for (locus = 0; locus < world->options->micro_stepnum; locus++)
        {
          world->options->steps[locus] =
            (double *) calloc (1,
                               sizeof (double) *
                               world->options->micro_stepnum);
        }
      /* do not break here */
    case 'b':
      world->fstparam =
        (double **) calloc (1, sizeof (double *) * (world->loci + 1));
      for (locus = 0; locus < world->loci + 1; locus++)
        world->fstparam[locus] =
          (double *) calloc (1, sizeof (double) * world->numpop2 * 2);
      break;
    }
  /*
   * chainlikelihood - for over last chains or replicate combining
   * estimator
   */
  world->chainlikes = (double **) calloc (world->loci + 1, sizeof (double *));
  world->chainlikes[0] =
    (double *) calloc ((world->loci + 1) *
                       (rep + world->options->pluschain), sizeof (double));
  for (i = 1; i < world->loci + 1; i++)
    world->chainlikes[i] =
      world->chainlikes[0] + (rep + world->options->pluschain) * i;
  //defined earlier:world->repstop = rep;
  world->rep = 0;
  world->lsteps = 0;
  /* timer -- how long will the program run? */
  time (&world->starttime);
  world->treestotal =
    world->loci * (options->schains * options->sincrement *
                   options->ssteps +
                   options->lchains * options->lincrement *
                   options->lsteps + (options->lchains +
                                      options->schains) * options->burn_in);
  if (world->options->replicatenum > 0)
    world->treestotal *= world->options->replicatenum;
  //migration histogram
  setup_mighist (world, options);
#ifdef UEP
  //setup UEP
  setup_uep (world, options);
#endif

#ifdef LONGSUM
  // allows fro three rates through time at GIVEN times [needs relaxation of this of course]
  world->flucrates = (double *) calloc (world->numpop * 6, sizeof (double));
  world->lflucrates = (double *) calloc (world->numpop * 6, sizeof (double));
  setup_fluctuate(world,options);
#endif /*LONGSUM*/
  // nr related structures
  alloc_apg (&world->apg0, world->repstop, world->loci,
             world->options->lsteps);
  alloc_apg (&world->apg, world->repstop, world->loci,
             world->options->lsteps);

  /* tree and treetimes are not yet allocated */
}

#ifdef LONGSUM
void setup_fluctuate(world_fmt *world, option_fmt * options)
{
  long i,ii;
  long numpop = world->numpop;
  long numpop3 = 3 * numpop;
  long numpop6 = 6 * numpop;
  if (options->fluctuate)
    {
      world->options->fluctuate = options->fluctuate;
      for (i = 0, ii=0 ; i < numpop6; i+=2, ii++)
        {
          //even elements are rates, (odd are times)
          world->flucrates[ii] = options->flucrates[i];
          world->lflucrates[ii] = log (options->flucrates[i]);
          world->flucrates[ii+numpop3] = options->flucrates[i+1];
        }
    }
  else
    {
      for (i = 0; i < numpop3; i++)
        {
          world->flucrates[i] = 1.; //all have the same rate
          world->lflucrates[i] = 0.;
          world->flucrates[i+numpop3] = DBL_MAX; //all times are large
        }
    }
}
#endif /*LONGSUM*/


void
calc_simple_param (world_fmt * world, data_fmt * data)
{
  switch (world->options->datatype)
    {
    case 'a':
    case 's':
    case 'n':
    case 'b':
    case 'm':
    case 'u':
    case 'f':
      calc_fst (world, data);
      break;
    default:
      break;
    }
  //param_all_adjust(world->param0, nr???not present here??????);//world->options, world->numpop);
}

void
set_bounds (long *increment, long *steps, long *chains,
            const option_fmt * options, const char type)
{
  switch (type)
    {
    case 's':
      *increment = options->sincrement;
      *steps = options->ssteps;
      *chains = options->schains;
      break;
    case 'l':
      *increment = options->lincrement;
      *steps = options->lsteps;
      *chains = options->lchains;
      break;
    default:
      error ("Wrong chain type\n");
      break;
    }
}

void
burnin_chain (world_fmt * world)
{
  long heating;
  long step, treeprint;
  if (world->options->burn_in == 0)
    return;
  heating = world->options->heating;
  world->options->heating = FALSE;
  print_menu_equilib (world->options);
  treeprint = world->options->treeprint;
  world->options->treeprint = NONE;
  for (step = 0; step < world->options->burn_in; step++)
    {
#ifdef BAYESUPDATE
      if(world->options->bayes_infer)
        world->bayesaccept = bayes_update (world);
#endif

      tree_update (world, 0);
    }
  world->options->heating = heating;
  world->options->treeprint = treeprint;
}

/*
void
copy_atl (world_fmt * world, long rep, timearchive_fmt * old, timearchive_fmt * new,
   long steps, long locus)
{
  long j;
  long numpop = world->numpop;
 
  new->T =  old->T;
  increase_timearchive (world, locus, steps, numpop, rep);
  new->param_like = world->param_like;
  new->trials = world->trials;
  new->normd = world->normd;
  new->sumtips = world->sumtips;
  new->numpop = world->numpop;
  memcpy (new->param, world->param0, sizeof (double) * world->numpop2);
  memcpy (new->param0, old->param0, sizeof (double) * world->numpop2);
  memcpy (new->lparam0, old->lparam0, sizeof (double) * world->numpop2);
  memcpy (new->likelihood, world->likelihood, sizeof (double) * steps);
  for (j = 0; j < new->T; j++)
    {
      new->tl[j].copies = old->tl[j].copies;
      copy_timelist (&old->tl[j], &new->tl[j], numpop);
    }
}
*/
#ifndef EARTH
#define EARTH universe[0]
#endif
void
print_list (world_fmt ** universe, option_fmt * options, data_fmt * data)
{
#ifdef MPI
  long maxreplicate;
#endif

#ifdef MPI
  //printf ("%i> entering mpi_results_table\n", myID);
  maxreplicate = (options->replicate ?
                  ((options->replicatenum > 0) ?
                   options->replicatenum  : options->lchains ) : 1);
  mpi_results_master (MIGMPI_RESULT, EARTH, maxreplicate,
                      unpack_result_buffer);
#endif

  print_results (universe, options, data);
#ifdef LONGSUM

  print_fluctuate(universe,options,data);
#endif /*LONGSUM*/

  if (!options->readsum)
    {
      if (options->printfst)
        print_fst (EARTH, options, data, EARTH->fstparam);
    }
  if (EARTH->options->plot)
    {
      plot_surface (EARTH, options, data, EARTH->plane,
                    EARTH->options->plotintervals + 2);
    }
}

void
set_ticks (double **ticks, double *plotrange, short type)
{
  long i;
  double diff;
  (*ticks)[0] = plotrange[0];
  (*ticks)[PLANEBIGTICKS - 1] = plotrange[1];
  if (type == PLOTSCALELOG)
    {
      diff =
        (log10 (plotrange[1]) - log10 (plotrange[0])) / (PLANEBIGTICKS - 1.);
      for (i = 1; i < PLANEBIGTICKS - 1; i++)
        {
          (*ticks)[i] = pow (10., (log10 ((*ticks)[0]) + i * diff));
        }
    }
  else
    {
      diff = (plotrange[1] - plotrange[0]) / (PLANEBIGTICKS - 1.);
      for (i = 1; i < PLANEBIGTICKS - 1; i++)
        {
          (*ticks)[i] = (*ticks)[0] + i * diff;
        }
    }
}

void
plot_surface (world_fmt * world, option_fmt * options,
              data_fmt * data, char ****plane, long x)
{
  long locus;
  long loci = world->loci;
  FILE *outfile = world->outfile;
  double *ticks;
  double prangex[2];
  double prangey[2];
  ticks = (double *) calloc (1, sizeof (double) * PLANEBIGTICKS);
  prangex[0] = world->options->plotrange[0];
  prangex[1] = world->options->plotrange[1];
  prangey[0] = world->options->plotrange[2];
  prangey[1] = world->options->plotrange[3];
  if (world->options->progress)
    fprintf (stdout, "           Plotting the likelihood surfaces\n");
  if (world->options->writelog)
    fprintf (world->options->logfile,
             "           Plotting the likelihood surfaces\n");
  PAGEFEED;
  fprintf (outfile,
           "Ln-Likelihood surfaces for each of the %3li populations\n",
           world->numpop);
  fprintf (outfile,
           "-------------------------------------------------------\n\n");
  fprintf (outfile, "Legend:\n");
  fprintf (outfile, "   X = Maximum likelihood\n");
  fprintf (outfile, "   * = in approximative 50%% confidence limit\n");
  fprintf (outfile, "   + = in approximative 95%% confidence limit\n");
  fprintf (outfile, "   - = in approximative 99%% confidence limit\n");
  set_ticks (&ticks, prangex, world->options->plotscale);
  fprintf (outfile, "X-tickmarks are (1) %f, (2) %f, (3) %f\n",
           ticks[0], ticks[1], ticks[2]);
  fprintf (outfile, "                (4) %f, (5) %f, (6) %f\n",
           ticks[3], ticks[4], ticks[5]);
  set_ticks (&ticks, prangey, world->options->plotscale);
  fprintf (outfile, "Y-tickmarks are (1) %f, (2) %f, (3) %f\n",
           ticks[0], ticks[1], ticks[2]);
  fprintf (outfile, "                (4) %f, (5) %f, (6) %f\n",
           ticks[3], ticks[4], ticks[5]);
  /* change: show only over all */
  if (!options->readsum)
    {
      if (world->loci == 1)
        {
          /*
           * for (locus = 0; locus < loci; locus++) { if
           * (world->data->skiploci[locus]) { continue; }
           * fprintf(outfile, "\n\nLocus %li\n", locus + 1);
           */

          plot_surface_header2 (outfile, 0,
                                world->options->plotvar ==
                                PLOT4NM ? "4Nm" : "M");
          plot_surface2 (outfile, PLANESIZE + 2, 0, world->numpop, plane,
                         world->plotmax, data->popnames,
                         world->options->plotvar == PLOT4NM ? "4Nm" : "M",
                         world->options->migration_model);
        }
    }
  if ((loci - world->skipped > 1) || (options->readsum))
    {
      fprintf (outfile, "\n%s\n\n",
               (options->readsum) ? ((loci - world->skipped >
                                      1) ? "Over all loci" : "Over locus 1") :
                   "Over all loci");
      locus = loci;
      plot_surface_header2 (outfile, locus,
                            world->options->plotvar == PLOT4NM ? "4Nm" : "M");
      plot_surface2 (outfile, PLANESIZE + 2, locus, world->numpop, plane,
                     world->plotmax, data->popnames,
                     world->options->plotvar == PLOT4NM ? "4Nm" : "M",
                     world->options->migration_model);
    }
}
void
plot_surface_header2 (FILE * outfile, long locus, char plotvar[])
{
  fprintf (outfile,
           "    x-axis= %3.3s [4Nm = effective population size * migration rate = Theta * M\n                  M   = migration rate / mutation rate = m/mu],\n[but see manual for mtDNA or haploid data]\n",
           plotvar);
  fprintf (outfile, "    y-axis = Theta,\n    units = see above\n");
}

void
plot_surface2 (FILE * outfile, long x, long locus, long numpop,
               char ****plane, plotmax_fmt ** plotmax,
               char **popnames, char plotvar[], long migrmodel)
{
  long pop, i;
  if (migrmodel == ISLAND)
    {
      fprintf (outfile, "N-ISLAND MODEL: For each Population\n");
      fprintf (outfile,
               "  **Average** immigration: %s=%f, Theta=%f, log likelihood=%f\n",
               plotvar, plotmax[locus][0].x1, plotmax[locus][0].y1,
               plotmax[locus][0].l1);
      fprintf (outfile, "[Remember: the maximum values are from a grid]\n");
      fprintf (outfile, "\n            Mean(Immigration)  \n\n");
      for (i = x; i >= 0; i--)
        fprintf (outfile, "%46.46s\n", plane[locus][0][i]);
      fprintf (outfile, "%46.46s\n", plane[locus][0][x]);
      PAGEFEED;
    }
  else
    {
      for (pop = 0; pop < numpop; pop++)
        {
          if (popnames == NULL)
            fprintf (outfile, "Population %li\n", pop + 1);
          else
            fprintf (outfile, "Population %li: %s\n", pop + 1, popnames[pop]);
          fprintf (outfile,
                   "  **Average** immigration: %s=%f, Theta=%f, log likelihood=%f\n",
                   plotvar, plotmax[locus][pop].x1, plotmax[locus][pop].y1,
                   plotmax[locus][pop].l1);
          fprintf (outfile,
                   "  **Average** emigration: %s=%f, Theta=%f, log likelihood=%f\n",
                   plotvar, plotmax[locus][pop].x2, plotmax[locus][pop].y2,
                   plotmax[locus][pop].l2);
          fprintf (outfile,
                   "[Remember: the maximum values are from a grid]\n");
          fprintf (outfile,
                   "\n           Mean(Immigration)                       Mean(Emigration)\n\n");
          for (i = x; i >= 0; i--)
            fprintf (outfile, "%s\n", plane[locus][pop][i]);
          fprintf (outfile, "%s\n", plane[locus][pop][x]);
          PAGEFEED;
        }
    }
}



void
print_alpha_curve (world_fmt * world, timearchive_fmt ** atl, long *gmaxptr)
{
  const double alphas[19] =
    {
      0.001, 0.002, 0.005,
      0.01, 0.02, 0.05,
      0.1, 0.2, 0.5,
      1., 2., 5.,
      10., 20., 50.,
      100., 200., 500., 1000.
    };
  long nnn;
  double adiff, amin = 0, amax;
  long g, a, loci = world->loci;
  double likes[21];
  char confidence[21];
  nr_fmt *nr;
  FILE *outfile = world->outfile;
  double **contours;
  double *testparam, *ltestparam;
  long repstart, repstop;
  helper_fmt helper;
  double alpha = atl[0][loci].param[world->numpop2];
  //
  double normd= DBL_MAX;
  long kind = PROFILE;
  long repkind = world->repkind;
  long rep=0;
  char save_a = world->options->custm2[world->numpop2];
  world->options->custm2[world->numpop2] = 'c';
  helper.multilocus = TRUE;
  testparam = (double *) calloc (world->numpop2 + 1, sizeof (double));
  ltestparam = (double *) calloc (world->numpop2 + 1, sizeof (double));
  if (world->options->gamma && loci - world->skipped > 1)
    {
      contours = (double **) calloc (1, sizeof (double *) * 1);
      contours[0] =
        (double *) calloc (1, sizeof (double) * 1 * CONTOURLEVELS);
      set_contours (contours, world->numpop2, 1);
      for (g = 0; g < CONTOURLEVELS; g++)
        {
          contours[0][g] += atl[0][loci].param_like;
        }
      if (world->options->progress)
        fprintf (stdout, "           Printing Alpha x Ln(likelihood)\n");
      if (world->options->writelog)
        fprintf (world->options->logfile,
                 "           Printing Alpha x Ln(likelihood)\n");

      nr = (nr_fmt *) calloc (1, sizeof (nr_fmt));
      create_nr (nr, world, *gmaxptr, world->numpop2, world->loci,
                 world->repkind, world->rep);
      nnn = nr->partsize;
      nr->skiploci = world->data->skiploci;
      set_replicates (world, world->repkind, world->rep, &repstart, &repstop);

      nr->repstart = repstart;
      nr->repstop = repstop;
      memcpy (testparam, atl[0][loci].param, sizeof (double) *
              (world->numpop2 + 1));
      set_logparam (ltestparam, testparam, world->numpop2 + 1);
      SETUPPARAM0 (world, nr, world->repkind, repstart, repstop,
                   world->loci, PROFILE, helper.multilocus);
      //   initgammacat (nr->categs, testparam[nr->numpop2], testparam[0],
      //   nr->rate, nr->probcat);
      //   fill_helper (&helper, testparam, ltestparam, world, nr);
      //   likes[19] = CALCLIKE (&helper, testparam, ltestparam);
      memcpy(world->param0,testparam, (world->numpop2+1)*sizeof(double));
      do_profiles (world, nr, &likes[19], &normd, kind, rep, repkind);
      likes[19] = nr->llike;
      if (alpha < alphas[0])
        amin = 1e-10;
      for (a = 0; a < 19; a++)
        {
          if (alpha > alphas[a])
            amin = alphas[a];
          else
            break;
        }
      if (a < 19)
        amax = alphas[a];
      else
        amax = alpha + 10;
      adiff = (amax - amin) / 19.;
      for (a = 0; a < 19; a++)
        {
          memcpy (testparam, atl[0][loci].param, sizeof (double) *
                  (world->numpop2 + 1));
          testparam[nr->numpop2] = alphas[a];
          //   set_logparam (ltestparam, testparam, world->numpop2 + 1);
          //   initgammacat (nr->categs, testparam[nr->numpop2], testparam[0],
          //   nr->rate, nr->probcat);
          //   fill_helper (&helper, testparam, ltestparam, world, nr);
          //   likes[a] = CALCLIKE (&helper, testparam, ltestparam);
          memcpy(world->param0,testparam, (world->numpop2+1)*sizeof(double));
          do_profiles (world, nr, &likes[a], &normd, kind, rep, repkind);
          likes[a] = nr->llike;
        }
      for (a = 0; a < 19; a++)
        {
          if (likes[a] < contours[0][1])
            {
              if (likes[a] < contours[0][2])
                {
                  if (likes[a] < contours[0][3])
                    {
                      confidence[a] = ' ';
                    }
                  else
                    {
                      confidence[a] = '-';
                    }
                }
              else
                {
                  confidence[a] = '+';
                }
            }
          else
            {
              if (likes[a] < contours[0][0])
                confidence[a] = '*';
              else
                {
                  confidence[a] = 'X';
                }
            }
        }
      fprintf (outfile, "Ln-Likelihood curve for the shape\n");
      fprintf (outfile, "parameter Alpha    [1/(CV(mu)^2)]\n");
      fprintf (outfile, "-------------------------------------\n\n");
      fprintf (outfile, "Legend:\n");
      fprintf (outfile, "   X = Maximum likelihood\n");
      fprintf (outfile, "   * = in approximative 50%% confidence limit\n");
      fprintf (outfile, "   + = in approximative 95%% confidence limit\n\n");
      fprintf (outfile, "   - = in approximative 99%% confidence limit\n\n");
      fprintf (outfile,
               "   Alpha              Ln(Likelihood)   Confidence limit\n");
      fprintf (outfile,
               "-------------------------------------------------------\n");
      if (alphas[0] > alpha)
        {
          fprintf (outfile, "%11.3g % 20.5g            X\n", alpha,
                   likes[20]);
        }
      for (a = 0; a < 18; a++)
        {
          if (alphas[a] < alpha && alphas[a + 1] > alpha)
            {
              fprintf (outfile, "% 11.3f     % 20.5g            %c\n",
                       alphas[a], likes[a], confidence[a]);
              fprintf (outfile, "%11.3g     % 20.5g            X\n",
                       alpha, likes[19]);
            }
          else
            fprintf (outfile, "% 11.3f     % 20.5g            %c\n",
                     alphas[a], likes[a], confidence[a]);
        }
      if (alphas[a] < alpha)
        {
          fprintf (outfile, "% 11.3f     % 20.5g            %c\n",
                   alphas[a], likes[a], confidence[a]);
          fprintf (outfile, "%11.3g     % 20.5g            X\n",
                   alpha, likes[19]);
        }
      else
        fprintf (outfile, "% 11.3f     % 20.5g            %c\n",
                 alphas[a], likes[a], confidence[a]);
      destroy_nr (nr, world);
    }
  world->options->custm2[world->numpop2] = save_a;
  free (testparam);
  free (ltestparam);
}

void
create_loci_plot (world_fmt * world, char ***plane, nr_fmt * nr, long loci)
{
  long intervals = world->options->plotintervals;
  long pop, i;
  double ***pl;
  double **contours;
  unset_penalizer_function(TRUE); //disables the penalizing of big differences of param and param0
  contours = (double **) calloc (1, sizeof (double *) * world->numpop);
  contours[0] =
    (double *) calloc (1, sizeof (double) * world->numpop * CONTOURLEVELS);
  for (pop = 1; pop < world->numpop; pop++)
    {
      contours[pop] = contours[0] + pop * CONTOURLEVELS;
    }
  set_contours (contours, world->numpop2, world->numpop);
  if (world->options->progress)
    {
      if (world->options->verbose)
        fprintf (stdout, "\n");
      print_menu_createplot ();
    }
  nr->skiploci = world->data->skiploci;
  pl = (double ***) calloc (1, sizeof (double **) * world->numpop);
  for (pop = 0; pop < world->numpop; pop++)
    {
      pl[pop] = (double **) calloc (1, sizeof (double *) * intervals);
      pl[pop][0] =
        (double *) calloc (1, sizeof (double) * 2 * intervals * intervals);
      for (i = 1; i < intervals; i++)
        {
          pl[pop][i] = pl[pop][0] + i * 2 * intervals;
        }
    }
  calc_loci_plane (world, nr, pl, loci, contours);
  for (pop = 0; pop < world->numpop; pop++)
    {
      fill_plotplane (world, plane[pop], pl[pop], contours[pop]);
    }
  print_mathematica (world, pl, intervals, intervals);
  for (i = 0; i < world->numpop; i++)
    {
      free (pl[i][0]);
      free (pl[i]);
    }
  free (pl);
  free (contours[0]);
  free (contours);
  unset_penalizer_function(FALSE); //reenables the penalizing of big differences of param and param0
}


void
create_locus_plot (world_fmt * world, char ***plane, nr_fmt * nr, long G)
{
  long intervals = world->options->plotintervals;
  long i, pop;
  double ***pl;
  double **contours;
  if (world->options->progress)
    {
      if (world->options->verbose)
        fprintf (stdout, "\n");
      print_menu_createplot ();
    }
  unset_penalizer_function(TRUE); //disables the penalizing of big differences of param and param0
  pl = (double ***) calloc (1, sizeof (double **) * world->numpop);
  contours = (double **) calloc (1, sizeof (double *) * world->numpop);
  contours[0] =
    (double *) calloc (1, sizeof (double) * CONTOURLEVELS * world->numpop);
  for (pop = 0; pop < world->numpop; pop++)
    {
      pl[pop] = (double **) calloc (1, sizeof (double *) * intervals);
      pl[pop][0] =
        (double *) calloc (1, sizeof (double) * 2 * intervals * intervals);
      for (i = 1; i < intervals; i++)
        {
          pl[pop][i] = pl[pop][0] + i * 2 * intervals;
        }
    }
  for (pop = 1; pop < world->numpop; pop++)
    {
      contours[pop] = contours[0] + pop * CONTOURLEVELS;
    }
  set_contours (contours, nr->numpop2, nr->numpop);
  calc_locus_plane (world, nr, G, pl, contours);
  for (pop = 0; pop < world->numpop; pop++)
    {

      fill_plotplane (world, plane[pop], pl[pop], contours[pop]);
    }
  print_mathematica (world, pl, intervals, intervals);
  for (i = 0; i < world->numpop; i++)
    {
      free (pl[i][0]);
      free (pl[i]);
    }
  free (contours[0]);
  free (contours);
  free (pl);
  unset_penalizer_function(FALSE); //reenables the penalizing of big differences of param and param0
}

void
set_contours (double **contours, long df, long numpop)
{
  long pop, i;
  contours[0][0] = contours[0][4] = 0.0;
  contours[0][1] = contours[0][5] = -0.5 * find_chi (df, 0.5);
  contours[0][2] = contours[0][6] = -0.5 * find_chi (df, 0.05);
  contours[0][3] = contours[0][7] = -0.5 * find_chi (df, 0.01);
  for (pop = 1; pop < numpop; pop++)
    {
      for (i = 0; i < CONTOURLEVELS; i++)
        {
          contours[pop][i] = contours[0][i];
        }
    }
}

/*
 * void cleanup_world (world_fmt * world, long locus) { //  if (locus >= 1)
 * //  { if (strchr (SEQUENCETYPES, world->options->datatype)) free_seqx
 * (world->root, world->data->seq->endsite); else free_x (world->root);
 * free_tree (world->root); free (world->nodep); //   } }
 */

void
free_universe (world_fmt ** worlds, long numworlds)
{
  long i;
	for(i=0;i<numworlds; i++)
		free_world(worlds[i]);
	free (worlds);
}


void
free_world(world_fmt *world)
  {
	long i,j,z;
	if (world->options->plot)
    {
      for (i = 0; i < world->loci + 1; i++)
        {
          for (j = 0; j < world->numpop; j++)
            free (world->plane[i][j]);
          free (world->plane[i]);
        }

      free (world->plane);
    }
  for (j = 0; j < world->repstop; j++)
    {
      for (i = 0; i < world->loci + 2; i++)
        {
          for (z = 0; z < world->atl[j][i].allocT; z++)
            {
              free (world->atl[j][i].tl[z].data);
              //free(world->atl[j][i].tl[z].p);
              //free(world->atl[j][i].tl[z].mindex);
              //free(world->atl[j][i].tl[z].kt);
              //free(world->atl[j][i].tl[z].km);
#ifdef ALTIVEC
              free(world->atl[j][i].tl[z].vdata);
#endif
            }
          free (world->atl[j][i].tl);
          free (world->atl[j][i].parameters);
        }
    }
  free (world->atl[0]);
  free (world->atl);
  if (strchr (SEQUENCETYPES, world->options->datatype))
    {
      if (world->tbl != NULL)
        {
          for (i = 0; i < world->options->rcategs; i++)
            {
              for (j = 0; j < world->options->categs; j++)
                free (world->tbl[i][j]);
              free (world->tbl[i]);
            }
        }
    }
#ifdef BAYESUPDATE
  if(world->options->bayes_infer)
    bayes_free(world);
#endif

  free (world->data->skiploci);
  free (world->data->geo);
  free (world->data->lgeo);
  free (world->data);
  free (world->options->zeroparam);
  free (world->options->constparam);
  free (world->options->sym2param);
  free (world->options->custm);
  free (world->options->custm2);
  free (world->options->mu_rates);
  free (world->options->lmu_rates);
#ifdef LONGSUM

  free(world->flucrates);
  free(world->lflucrates);
#endif /*LONGSUM*/

  free (world->options);
  free (world->nodep);
  free (world->likelihood);
  free (world->param0);
}

/* printing stuff =========================================== */
void
print_menu_locus (world_fmt * world, long locus)
{
  if (world->options->progress)
    {
      if (world->replicate > 0)
        fprintf (stdout, "\n\nLocus %-3li: Replicate %li\n", locus + 1,
                 world->replicate + 1);
      else
        fprintf (stdout, "\n\nLocus %-3li:\n", locus + 1);
      if (world->options->writelog)
        {
          if (world->replicate > 0)
            fprintf (world->options->logfile,
                     "\n\nLocus %-3li: Replicate %li\n", locus + 1,
                     world->replicate + 1);
          else
            fprintf (world->options->logfile, "\n\nLocus %-3li:\n",
                     locus + 1);
        }
    }
}

void
print_menu_chain (char type, long chain, long steps,
                  world_fmt * world, option_fmt * options, long rep)
{
  long i, j, k;
  char strllike[LINESIZE];
  char nowstr[LINESIZE];
  char spacer[] = "           ";
  double value;
  double mini, maxi;
  double summ = 0;
  FILE *logg = NULL;
  boolean writelog = world->options->writelog;
  if (world->options->progress)
    {
      if (writelog)
        logg = world->options->logfile;
      print_llike (world->param_like, strllike);
      get_time (nowstr, "%H:%M:%S");
      if (chain == FIRSTCHAIN)
        {
          if (writelog)
            {
              fprintf (logg, "%s   Locus %li: Start conditions:\n", nowstr,
                       world->locus + 1);
              print_param (logg, world->options->usem, world,
                           world->numpop, spacer);
              fprintf (logg, "%sStart-tree-log(L)=%f\n", spacer,
                       world->likelihood[0]);

            }
          fprintf (stdout, "%s   Locus %li: Start conditions:\n", nowstr,
                   world->locus + 1);
          print_param (stdout, world->options->usem, world,
                       world->numpop, spacer);
          fprintf (stdout, "%sStart-tree-log(L)=%f\n", spacer,
                   world->likelihood[0]);
          if (writelog)
            {
              if (world->repkind != SINGLECHAIN)
                fprintf (logg, "%s   Multiple chain estimates:\n", nowstr);
              else
                fprintf (logg, "%s   Locus: %li: %*s chain %3li:\n", nowstr,
                         world->locus + 1, type == 's' ? 5 : 4,
                         type == 's' ? "Short" : "Long", chain + 1);
              fprintf (logg,
                       "%sln L(Param)=ln (P(G|Param)/P(G|Param0))=%s,\n",
                       spacer, strllike);
              if (world->repkind == SINGLECHAIN)
                fprintf (logg,
                         "%sStart-tree-ln(L)=ln P(D|G)=%f\n",
                         spacer, world->likelihood[0]);
              print_param (logg, world->options->usem, world,
                           world->numpop, spacer);
            }
          prognose_time (nowstr, world, options, steps, spacer);
        }
      else
        {
          if (world->repkind != SINGLECHAIN)
            fprintf (stdout, "%s   Multiple chain estimates:\n", nowstr);
          else
            fprintf (stdout, "%s   Locus %li: %*s chain %3li:\n", nowstr,
                     world->locus + 1, type == 's' ? 5 : 4,
                     type == 's' ? "Short" : "Long", chain + 1);
          prognose_time (nowstr, world, options, steps, spacer);

          fprintf (stdout,
                   "%sln L(Param)=ln (P(G|Param)/P(G|Param0)):%15.15s\n",
                   spacer, strllike);
          if (world->repkind == SINGLECHAIN)
            fprintf (stdout,
                     "%sStart-tree-ln(L)=ln P(D|G):             %.5f\n",
                     spacer, world->likelihood[0]);
          print_param (stdout, world->options->usem, world,
                       world->numpop, spacer);
          if (writelog)
            {
              if (world->repkind != SINGLECHAIN)
                fprintf (logg, "%s   Multiple chain estimates:\n", nowstr);
              else
                fprintf (logg, "%s   Locus %li: %*s chain %3li:\n", nowstr,
                         world->locus + 1, type == 's' ? 5 : 4,
                         type == 's' ? "Short" : "Long", chain + 1);
              fprintf (logg,
                       "%sln L(Param)=ln (P(G|Param)/P(G|Param0))=%s,\n",
                       spacer, strllike);
              if (world->repkind == SINGLECHAIN)
                fprintf (logg,
                         "%sStart-tree-ln(L)=ln P(D|G)=%f\n",
                         spacer, world->likelihood[0]);
              print_param (logg, world->options->usem, world,
                           world->numpop, spacer);
            }
          if (world->options->verbose && world->repkind == SINGLECHAIN)
            {
              mini = DBL_MAX;
              maxi = -DBL_MAX;
              for (i = 0; i < steps; i++)
                {
                  value = world->likelihood[i];
                  if (mini > value)
                    mini = value;
                  if (maxi < value)
                    maxi = value;
                }
              fprintf (stdout, "           Sampled tree-log(L)={%f .. %f},\n",
                       mini, maxi);
              fprintf (stdout, "           Best of all visited trees =%f\n",
                       world->maxdatallike);
              /* report migration events */
              fprintf (stdout,
                       "           Average migration events per genealogy (x-axis=to, y=from)\n           ");
              if (writelog)
                {
                  fprintf (logg,
                           "           Sampled tree-log(L)={%f .. %f},\n",
                           mini, maxi);
                  fprintf (logg, "           Best of all visited trees =%f\n",
                           world->maxdatallike);
                  /* report migration events */
                  fprintf (logg,
                           "           Average migration events per genealogy (x-axis=to, y=from)\n           ");
                }
              for (i = 0; i < world->numpop; i++)
                {
                  for (j = 0; j < world->numpop; j++)
                    {
                      if (i != j)
                        {
                          summ = 0;
                          for (k = 0; k < world->atl[rep][world->locus].T;
                               k++)
                            summ +=
                              world->atl[rep][world->locus].tl[k].
                              mindex[mm2m (j, i, world->numpop)] *
                              world->atl[rep][world->locus].tl[k].copies;
                          fprintf (stdout, "%6.2f ",
                                   ((double) summ) / ((double) steps));
                          if (writelog)
                            fprintf (logg, "%6.2f ",
                                     ((double) summ) / ((double) steps));
                        }
                      else
                        {
                          fprintf (stdout, "------- ");
                          if (writelog)
                            fprintf (logg, "------- ");
                        }
                    }
                  fprintf (stdout, "\n           ");
                  if (writelog)
                    fprintf (logg, "\n           ");
                }
              fprintf (stdout, "\n");
              if (writelog)
                fprintf (logg, "\n");
            }
          //        prognose_time (nowstr, world, options, steps, spacer);
        }
    }
}

void
prognose_time (char *nowstr, world_fmt * world,
               option_fmt * options, long steps, char *spacer)
{
#ifndef NOTIME_FUNC
  time_t nowbin;
  time_t proposetime;
  struct tm *nowstruct;
  long increment =
    (steps ==
     options->sincrement) ? options->sincrement : options->lincrement;
  if (world->treesdone == 0)
    {
      strcpy (nowstr, "indeterminate");
      world->treesdone += steps * increment + world->options->burn_in;
    }
  else
    {
      if (steps != 0)
        world->treesdone += steps * increment + world->options->burn_in;
#ifdef MPI

      proposetime =
        (long) (((double)
                 ((double) time (&nowbin) -
                  (double) world->starttime)) /
                ((double) world->treesdone) * world->treestotal) / (numcpu -
                    1);
#else

      proposetime =
        (long) (((double)
                 ((double) time (&nowbin) -
                  (double) world->starttime)) /
                ((double) world->treesdone) * world->treestotal);
#endif

      if (nowbin != (time_t) - 1)
        {
          nowbin = world->starttime + proposetime;
          nowstruct = localtime (&nowbin);
          strftime (nowstr, LINESIZE, "%R %B %e %Y", nowstruct);
        }
      else
        {
          strcpy (nowstr, "indeterminate");
        }
    }
  fprintf (stdout, "%sPrognosed end of sampling is    %s\n", spacer, nowstr);
  if (world->options->writelog)
    fprintf (world->options->logfile,
             "%sPrognosed end of sampling is %s\n", spacer, nowstr);
#endif
}

void
copy_time (world_fmt * world, timelist_fmt * ltl, long from,
           long to, long np, long rep)
{
  long T;
  tarchive_fmt *tl;
  if (from == -1)
    {
      (*ltl).copies = 1;
      T = (*ltl).T - 1;
      increase_timearchive (world, world->locus, 1, np, world->rep);
      tl = world->atl[rep][world->locus].tl;
      tl[0].copies = (*ltl).copies;
      archive_timelist (tl, (*ltl).tl, T, np, world);
      if (world->in_last_chain && world->options->mighist)
        {
          increase_mighist(&world->mighistloci[world->locus]);
          world->mighistloci[world->locus].mighist[world->
              mighistloci[world->locus].
              mighistnum].copies = 1;
          world->mighistloci[world->locus].mighistnum++;
        }
      return;
    }
  if (from == to)
    {
      (*ltl).copies += 1;
      tl = world->atl[rep][world->locus].tl;
      tl[from].copies = (*ltl).copies;
      if (world->in_last_chain && world->options->mighist)
        {
          world->mighistloci[world->locus].mighist[world->
              mighistloci[world->locus].
              mighistnum].copies++;
        }
    }
  else
    {
      T = (*ltl).T - 1;
      increase_timearchive (world, world->locus, to + 1, np, world->rep);
      tl = world->atl[rep][world->locus].tl;
      tl[to].copies = (*ltl).copies = 1;
      archive_timelist (&(tl[to]), (*ltl).tl, T, np, world);
      if (world->in_last_chain && world->options->mighist)
        {
          increase_mighist(&world->mighistloci[world->locus]);
          world->mighistloci[world->locus].mighist[world->
              mighistloci[world->locus].
              mighistnum].copies = 1;
          world->mighistloci[world->locus].mighistnum++;
        }
    }
}

void
precalc_world (world_fmt * world)
{
  long numpop = world->numpop;
  long pop;
  long msta;
  long msto;
  long i, j;
  memset (world->mig0list, 0, sizeof (double) * world->numpop);
  if (numpop > 1)
    {
      for (i = 0; i < numpop; ++i)
        memset (world->migproblist[i], 0,
                sizeof (double) * (world->numpop - 1));
    }
  memset (world->design0list, 0, sizeof (long) * world->numpop);
  for (pop = 0; pop < numpop; pop++)
    {
      msta = world->mstart[pop];
      msto = world->mend[pop];
      for (i = msta, j = 0; i < msto; i++, j++)
        {
          world->mig0list[pop] += world->data->geo[i] * world->param0[i];
          world->migproblist[pop][j] = world->data->geo[i] * world->param0[i];
          world->design0list[pop] += world->options->custm2[i] == '0' ? 1 : 0;
        }
      for (j = 1; j < numpop - 1; ++j)
        world->migproblist[pop][j] += world->migproblist[pop][j - 1];
    }
  for (pop = 0; pop < numpop; pop++)
    {
      for (j = 0; j < numpop - 1; ++j)
        {
          if (world->migproblist[pop][numpop - 2] != 0)
            world->migproblist[pop][j] /= world->migproblist[pop][numpop - 2];
          else
            world->migproblist[pop][j] = 0.;
        }
    }
}



/* private functions========================================== */
/*
 * --------------------------------------------------------- creates memory
 * for archive of timelists for each locus
 */
void
create_timearchive (timearchive_fmt *** atl, long loci,
                    long samples, long numpop, long replicates)
{
  long h, i, j;
  long numpop2 = numpop * numpop;
#ifdef ALTIVEC

  long size = numpop2 + numpop + numpop;
  size -= (size % 4) - 4;
  size /= 4;
#endif /*ALTIVEC*/

  (*atl) =
    (timearchive_fmt **) calloc (replicates + 1, sizeof (timearchive_fmt *));
  (*atl)[0] =
    (timearchive_fmt *) calloc ((replicates + 1) * (2 + loci),
                                sizeof (timearchive_fmt));
  for (h = 1; h < replicates + 1; h++)
    (*atl)[h] = (*atl)[0] + h * (2 + loci);
  for (h = 0; h < replicates + 1; h++)
    {
      for (i = 0; i < loci + 2; i++)
        {
#ifdef ALTIVEC
          //on ALTIVEC processors we want ot shift some data into streamlined
          // vectors so that they can be processed in calc_locus_like at
          // max speed
          //alllog(copies) padded so that they are allows multyplies of 4
          // because the calc_locus_function will stuff 4 vectors at the same time into the
          // altivec unit to fill the pipeline.
          //(*atl)[h][i].lcopiesvec = (FloatVec *) calloc(samples + samples % 4, sizeof(FloatVec));

          // contains all the compressed treesummaries
          //size = (2 * numpop + numpop2) * (samples + (samples % 4));
          //(*atl)[h][i].data = (FloatVec *) calloc(size, sizeof(FloatVec)); //all tl[].data
#endif /*ALTIVEC*/
          (*atl)[h][i].parameters =
            (double *) calloc (4 * (numpop2 + 1), sizeof (double));
          //    (double *) calloc (3 * numpop2 + samples + 1, sizeof (double));
          (*atl)[h][i].param = (*atl)[h][i].parameters;
          (*atl)[h][i].param0 = (*atl)[h][i].parameters + numpop2 + 1;
          (*atl)[h][i].lparam0 = (*atl)[h][i].param0 + numpop2 + 1;
          (*atl)[h][i].likelihood = (*atl)[h][i].lparam0 + numpop2 + 1;
          (*atl)[h][i].tl =
            (tarchive_fmt *) calloc (samples, sizeof (tarchive_fmt));
          (*atl)[h][i].T = (*atl)[h][i].allocT = samples;
          for (j = 0; j < samples; j++)
            {
              (*atl)[h][i].tl[j].data =
                (double *) calloc (2 * numpop + numpop2, sizeof (double));
#ifdef ALTIVEC

              (*atl)[h][i].tl[j].vdata = (FloatVec *) calloc (size, sizeof (FloatVec));
#endif

              (*atl)[h][i].tl[j].point = (*atl)[h][i].tl[j].data;
              (*atl)[h][i].tl[j].wait = &(*atl)[h][i].tl[j].data[numpop2];
              (*atl)[h][i].tl[j].kt = (*atl)[h][i].tl[j].wait;
              (*atl)[h][i].tl[j].km = &(*atl)[h][i].tl[j].wait[numpop];
              (*atl)[h][i].tl[j].p = (*atl)[h][i].tl[j].point;
              (*atl)[h][i].tl[j].mindex = (*atl)[h][i].tl[j].point; // the first numpop element are not used
              // this is really ugly and should be changed
#ifdef LONGSUM

              (*atl)[h][i].tl[j].longsum = (longsum_fmt *) calloc(1,sizeof(longsum_fmt));
#endif /*LONGSUM*/

            }
        }
    }
}

void
increase_timearchive (world_fmt * world, long locus, long sample,
                      long numpop, long rep)
{
  long i = locus, j, oldT = 0, size;
  long numpop2 = world->numpop2;
  long datasize = numpop2 + world->numpop + world->numpop;
#ifdef ALTIVEC

  long vsize = datasize;
  vsize -= (vsize % 4) - 4;
  vsize /= 4;
#endif /*ALTIVEC*/
  // needs to be a multiple of 4 because of vdot_product;
  //datasize += datasize % 4 ;

  if (sample >= world->atl[rep][i].allocT)
    {
      oldT = world->atl[rep][i].allocT;
      world->atl[rep][i].allocT =
        MAX (sample + 1,
             (long) (world->atl[rep][i].allocT +
                     world->atl[rep][i].allocT / 4.));
      size = world->atl[rep][i].allocT;
      world->atl[rep][i].tl =
        (tarchive_fmt *) realloc (world->atl[rep][i].tl,
                                  (1 + size) * sizeof (tarchive_fmt));
      //#ifdef ALTIVEC
      // world->atl[rep][i].lcopiesvec = (FloatVec *) realloc(world->atl[rep][i].lcopiesvec, (size + size % 4) *
      //sizeof(FloatVec));
      // contains all the compressed treesummaries
      //  sizevec = (2 * numpop + numpop2) * (size + (size % 4));
      //  world->atl[rep][i].data = (FloatVec *) realloc(world->atl[rep][i].data, sizevec * sizeof(FloatVec));
      //#endif /*ALTIVEC*/
      for (j = oldT; j < world->atl[rep][i].allocT; j++)
        {
          world->atl[rep][i].tl[j].data = (double *) calloc (datasize, sizeof (double));
#ifdef ALTIVEC

          world->atl[rep][i].tl[j].vdata = (FloatVec *) calloc (vsize, sizeof (FloatVec));
#endif

          world->atl[rep][i].tl[j].point = world->atl[rep][i].tl[j].data;
          world->atl[rep][i].tl[j].wait = &world->atl[rep][i].tl[j].data[numpop2];
          world->atl[rep][i].tl[j].kt = world->atl[rep][i].tl[j].wait;
          world->atl[rep][i].tl[j].km = &world->atl[rep][i].tl[j].wait[numpop];
          world->atl[rep][i].tl[j].p = world->atl[rep][i].tl[j].point;
          world->atl[rep][i].tl[j].mindex = world->atl[rep][i].tl[j].point; //ugly:the first numpop elem are not used
#ifdef LONGSUM

          world->atl[rep][i].tl[j].longsum = (longsum_fmt *) calloc(1,sizeof(longsum_fmt));
#endif /*LONGSUM*/

        }
      world->atl[rep][i].T = sample;
    }
  else
    {
      world->atl[rep][i].T = sample;
    }
}

void
create_plotplane (world_fmt * world)
{
  long pop, locus, i;
  long plotsize = world->options->plotintervals;
  world->plane =
    (char ****) calloc (1, sizeof (char ***) * (world->loci + 1));
  world->plotmax =
    (plotmax_fmt **) calloc (1, sizeof (plotmax_fmt *) * (world->loci + 1));
  world->plotmax[0] =
    (plotmax_fmt *) calloc (1,
                            sizeof (plotmax_fmt) * (world->loci +
                                                    1) * world->numpop);
  for (pop = 1; pop < world->loci + 1; pop++)
    {
      world->plotmax[pop] = world->plotmax[0] + pop * (world->numpop);
    }
  for (locus = 0; locus < world->loci + 1; locus++)
    {
      world->plane[locus] =
        (char ***) calloc (1, sizeof (char **) * world->numpop);
      for (pop = 0; pop < world->numpop; pop++)
        {
          world->plane[locus][pop] =
            (char **) calloc (1, sizeof (char *) * (plotsize + 3));
          for (i = 0; i < plotsize + 3; i++)
            {
              world->plane[locus][pop][i] =
                (char *) calloc (1, sizeof (char) * plotsize + plotsize + 20);
            }
        }
    }
}

void
create_cov (world_fmt * world)
{
  long locus, i;
  world->cov =
    (double ***) calloc (1, sizeof (double **) * (world->loci + 1));
  for (locus = 0; locus < world->loci + 1; locus++)
    {
      world->cov[locus] =
        (double **) calloc (1, sizeof (double *) * (world->numpop2 + 1));
      world->cov[locus][0] =
        (double *) calloc (1,
                           sizeof (double) * (world->numpop2 +
                                              1) * (world->numpop2 + 1));
      for (i = 1; i < world->numpop2 + 1; i++)
        {
          world->cov[locus][i] =
            world->cov[locus][0] + i * (world->numpop2 + 1);
        }
    }
}

void
print_menu_equilib (worldoption_fmt * options)
{
  char nowstr[LINESIZE];
  if (options->progress)
    {
      get_time (nowstr, "%H:%M:%S");
      fprintf (stdout,
               "%s   Equilibrate tree (first %li trees are not used)\n",
               nowstr, options->burn_in);
      if (options->writelog)
        fprintf (options->logfile,
                 "%s   Equilibrate tree (first %li trees are not used)\n",
                 nowstr, options->burn_in);
    }
}


void
print_finish (world_fmt * world, long filepos)
{
  char nowstr[LINESIZE];
  if (world->options->progress)
    {
      get_time (nowstr, "%H:%M:%S");
      fprintf (stdout, "%s   Program finished\n", nowstr);
      if (world->options->writelog)
        fprintf (world->options->logfile, "%s   Program finished\n", nowstr);
    }
  get_time (nowstr, "%c");
  if (nowstr[0] != '\0')
    {
      if (filepos > 0)
        fseek (world->outfile, filepos, SEEK_SET);
      fprintf (world->outfile, "         finished at %s\n", nowstr);
    }
}

void
archive_timelist (tarchive_fmt * atl, vtlist * tl, long T, long np,
                  world_fmt * world)
{
  long j, i;
  double t;
  double line;
#ifdef LONGSUM

  longsum_fmt *longsum;
#endif /*LONGSUM*/

  mighist_fmt *aa = NULL;
  for (i = 0; i < np; i++)
    {
      line = (double) tl[0].lineages[i];
      atl->km[i] = line * tl[0].age;
      atl->kt[i] = line * (line - 1) * tl[0].age;
      atl->p[i] = (double) whichp (tl[0].from, tl[0].to, i);
    }
  memset (atl->mindex + np, 0, sizeof (double) * np * (np - 1));
  if (tl[0].from != tl[0].to)
    atl->mindex[mm2m (tl[0].from, tl[0].to, np)] += 1;
#ifdef LONGSUM

  atl->longsumlen = T;
  atl->longsum = (longsum_fmt *) realloc(atl->longsum,sizeof(longsum_fmt)*T);
  atl->longsum[0].lineages = (long *) calloc(T,sizeof(long)*np);
  atl->longsum[0].lineages2 = (long *) calloc(T,sizeof(long)*np);
  longsum = atl->longsum;
  memcpy(longsum[0].lineages,tl[0].lineages,sizeof(long)*np);
  memcpy(longsum[0].lineages2,tl[0].lineages,sizeof(long)*np);
  for(i=0;i<np;i++)
    longsum[0].lineages2[i] *= longsum[0].lineages2[i] - 1;
  longsum[0].fromto = mm2m(tl[0].from, tl[0].to,np);
  longsum[0].to = tl[0].to;
  longsum[0].eventtime = tl[0].age;
  longsum[0].interval = tl[0].age;
  longsum[0].eventtype = tl[0].from != tl[0].to ? 'm' : 'c';
#endif /*LONGSUM*/

  if (world->in_last_chain && world->options->mighist)
    {
      aa = &(world->mighistloci[world->locus].mighist[world->mighistloci[world->locus].mighistnum]);
      //  if(aa->migevents==NULL)
      // {
      //   aa->allocsize = T;
      //   aa->migevents = calloc(aa->allocsize, sizeof (migevent_fmt));
      // }
      //else
      // {
      //   aa->allocsize += T;
      //   aa->migevents=(migevent_fmt *) realloc(aa->migevents,sizeof(migevent_fmt)*aa->allocsize);
      //   //wrong for replicates   aa->migeventsize = 0;
      if (tl[0].from != tl[0].to)
        {
          if(aa->allocsize <= aa->migeventsize + 1 )
            {
              aa->allocsize += DEFAULTALLOCSIZE ;
              aa->migevents = realloc(aa->migevents, sizeof(migevent_fmt) * aa->allocsize);
            }
          //    printf("(%li) migeventsize=%li, allocsize=%li\n",world->mighistloci[world->locus].mighistnum,
          //    aa->migeventsize,aa->allocsize);
          aa->migevents[aa->migeventsize][0] = tl[0].age;
          aa->migevents[aa->migeventsize][1] = (double) tl[0].from;
          aa->migevents[aa->migeventsize][2] = (double) tl[0].to;
          aa->migeventsize++;
        }
    }
  for (j = 1; j < T; j++)
    {
      t = tl[j].age - tl[j - 1].age;
      for (i = 0; i < np; i++)
        {
          line = (double) tl[j].lineages[i];
          atl->km[i] += line * t;
          atl->kt[i] += line * (line - 1.) * t;
          atl->p[i] += whichp (tl[j].from, tl[j].to, i);
        }
      if (tl[j].from != tl[j].to)
        atl->mindex[mm2m (tl[j].from, tl[j].to, np)] += 1;
#ifdef LONGSUM

      atl->longsum[j].lineages = (long *) calloc(1,sizeof(long)*np);
      atl->longsum[j].lineages2 = (long *) calloc(1,sizeof(long)*np);
      memcpy(longsum[j].lineages,tl[j].lineages,sizeof(long)*np);
      memcpy(longsum[j].lineages2,tl[j].lineages,sizeof(long)*np);
      for(i=0;i<np;i++)
        longsum[j].lineages2[i] *= longsum[j].lineages2[i] - 1;

      longsum[j].fromto = mm2m(tl[j].from,tl[j].to,np);
      longsum[j].to = tl[j].to;
      longsum[j].eventtime = tl[j].age;
      longsum[j].interval = t;
      longsum[j].eventtype = tl[j].from != tl[j].to ? 'm' : 'c';
#endif /*LONGSUM*/

      if (world->in_last_chain && world->options->mighist)
        {
          if(tl[j].from != tl[j].to)
            {
              if(aa->allocsize <= aa->migeventsize + 1 )
                {
                  aa->allocsize += DEFAULTALLOCSIZE ;
                  aa->migevents = realloc(aa->migevents, sizeof(migevent_fmt) * aa->allocsize);
                }
              // printf("(%li) %li> migeventsize=%li, allocsize=%li\n",world->mighistloci[world->locus].mighistnum, j,aa->migeventsize,aa->allocsize);
              aa->migevents[aa->migeventsize][0] = tl[j].age;
              aa->migevents[aa->migeventsize][1] = (double) tl[j].from;
              aa->migevents[aa->migeventsize][2] = (double) tl[j].to;
              aa->migeventsize++;
            }
        }
    }
  if(!world->in_last_chain)
    {
      for (j = world->numpop; j < world->numpop2; j++)
        if ((atl->mindex[j] < 1.0) && (world->param0[j] > 0))
          atl->mindex[j] = MINMIGSUMSTAT;
    }
#ifdef ALTIVEC
  load_double_floatvec(atl->vdata,atl->data,world->numpop2+2*world->numpop);
#endif
}


void
copy_timelist (tarchive_fmt * from, tarchive_fmt * to, long np)
{
  memcpy (to->data, from->data, sizeof (double) * (3 * np + np * (np - 1)));
  //memcpy(to->p, from->p, sizeof(long) * np);
  //memcpy(to->mindex, from->mindex, sizeof(double) * np * (np-1));
  //memcpy(to->km, from->km, sizeof(double) * np);
  //memcpy(to->kt, from->kt, sizeof(double) * np);
}

long
whichp (long from, long to, long pop)
{
  if (from == to)
    {
      if (from == pop)
        return 1;
    }
  return 0;
}
#ifdef LONGSUM
void print_fluctuate_header(world_fmt *world)
{
  fprintf(world->outfile,"\n\n\n");
  fprintf(world->outfile,"============================================================\n");
  fprintf (world->outfile, "Population size Rate changes in the past [today's rate is 1] \n");
  fprintf(world->outfile,"============================================================\n\n");
  fprintf(world->outfile,"Population      Loc.  Rate_2     Time_2       Rate_3     Time_3\n");
  fprintf(world->outfile,"--------------- ----  --------------------    -------------------- \n");
}

void print_fluctuate_results(world_fmt * world, long locus, long rep,
                             long pop)
{
  fprintf(world->outfile," %10.7f %10.7f   %10.7f %10.7f\n",
          world->atl[rep][locus].param[world->numpop2 + pop * 3+1],
          world->flucrates[world->numpop * 3 + pop * 3+1],
          world->atl[rep][locus].param[world->numpop2 + pop * 3+2],
          world->flucrates[world->numpop * 3 + pop * 3+2]);
}

void print_fluctuate(world_fmt **universe, option_fmt *options, data_fmt *data)
{
  long skipped = 0;
  long pop;
  long locus;
  world_fmt *world=EARTH;
  long maxrep = world->options->replicate ?
                (world->options->replicatenum > 0 ?
                 world->options->replicatenum + 1 : world->options->lchains + 1) : 1;
  long rep;
  print_fluctuate_header(world);
  for(pop=0; pop < world->numpop; pop++)
    {
      print_popstring(pop, world, options, data);
      for (locus = 0; locus < world->loci; locus++)
        {
          if (world->data->skiploci[locus])
            {
              skipped++;
              continue;
            }
          for(rep=0; rep < maxrep; rep++)
            {
              print_replicate(world, maxrep, rep,locus);
              print_fluctuate_results(world, locus, rep, pop);
            }
        }
      if (world->loci - skipped > 1)
        {
          fprintf (world->outfile, "                All ");
          print_fluctuate_results(world, world->loci, 0, pop);
        }

    }
  fprintf(world->outfile,"\n");
}
#endif /*LONGSUM*/

void
print_results (world_fmt ** universe, option_fmt * options, data_fmt * data)
{
  long pop;
  FILE *outfile;
  world_fmt *world=EARTH;
  worldoption_fmt *wopt = world->options;
  char sch[10], lch[10], cva[50];
  long rep = wopt->replicate ? world->repstop : 0;
  outfile = world->outfile;
  if (options->schains == 1)
    strcpy (sch, "chain");
  else
    strcpy (sch, "chains");
  if (options->lchains == 1)
    strcpy (lch, "chain");
  else
    strcpy (lch, "chains");
  fprintf (outfile, "\n\n");
  PAGEFEED;
  print_result_header ("MCMC estimates ", world);
  for (pop = 0; pop < world->numpop; pop++)
    {
      print_result_population (pop, world, options, data);
    }
  fprintf (outfile,
           "\nComments:\n There were %li short %s (%li used trees out of ",
           options->schains, sch, options->ssteps);
  fprintf (outfile, "sampled %li)\n", options->sincrement * options->ssteps);
  fprintf (outfile,
           "  and %li long %s (%li used trees out of sampled %li)\n",
           options->lchains, lch, options->lsteps,
           options->lincrement * options->lsteps);
  if (wopt->heating)
    {
      if(options->heatingadaptive)
        {
          fprintf (outfile,
                   "  Adaptive heating with %li chains was active\n  Average last chain temp: 1.0",
                   options->heated_chains);
          for(pop=1;pop<options->heated_chains;pop++)
            fprintf(outfile, ", %f", universe[pop]->averageheat);
        }
      else
        fprintf (outfile, "  Static heating with %li chains was active\n",options->heated_chains);
    }
  if (options->gamma)
    {
      if (world->atl[rep][world->loci].param[world->numpop2] < 10e-9)
        strcpy (cva, "0");
      else
        sprintf (cva, "%f",
                 sqrt (1. /
                       world->atl[rep][world->loci].param[world->numpop2]));
      fprintf (outfile,
               "With shape parameter Alpha=%g ([1/CV(mu)]^2; CV(mu)=%s)\n",
               world->atl[rep][world->loci].param[world->numpop2],
               cva);
    }
  if (world->options->replicate)
    {
      if (world->repkind == MULTIPLECHAIN)
        fprintf (outfile, "  COMBINATION OF ALL LONG CHAINS\n");
      else
        fprintf (outfile, "  COMBINATION OF %li MULTIPLE RUNS)\n",
                 world->options->replicatenum);
    }
  if (world->atl[rep][world->loci].normd > LOCI_NORM)
    fprintf (outfile,
             "  [Last maximization needed %li cycles of maximal %i,\n  Norm(first derivatives)=%f (Normal stopping criteria is < %f)]\n\n\n",
             world->atl[rep][world->loci].trials,
             NTRIALS, world->atl[rep][world->loci].normd, LOCI_NORM);
}

void
print_fst (world_fmt * world, option_fmt * options, data_fmt * data,
           double **fstparam)
{
  long pop;
  long loci = world->loci;
  FILE *outfile = world->outfile;
  if (loci < 40)
    {
      PAGEFEED;
    }
  if (options->fsttype == 'T')
    print_result_header
    ("FST estimates (Thetas are variable)\n   [Only used as start values for MCMC run]",
     world);
  else
    print_result_header
    ("FST estimates (Migration rates are variable)\n  [Only used as start values for MCMC run]",
     world);
  for (pop = 0; pop < world->numpop; pop++)
    {
      print_result_fst (pop, world, data);
    }
  fprintf (world->outfile, "\nComments:\n");
  fprintf (world->outfile, "(-) can not be estimated\n");
  fprintf (world->outfile, "(0/0 or x/0) Divisions by zero\n");
  fprintf (world->outfile,
           "Maynard Smith, J. 1970. Population size, polymorphism, and the rate of\n");
  fprintf (world->outfile,
           "    non-Darwinian evolution. American Naturalist 104:231-237\n");
  fprintf (world->outfile,
           "Nei, M., and M. Feldman 1972. Identity of genes by descent within and\n");
  fprintf (world->outfile,
           "    between populations under mutation and migration pressures.\n");
  fprintf (world->outfile, "    Theoretical Population Biology 3: 460-465\n");
}

void
prepare_print_nu (double nu, char *str)
{
  if (nu <= -999)
    sprintf (str, "-");
  else
    sprintf (str, "% 12.6f", nu);
}

void
prepare_print_nm (double nm, double nmu, char *strllike)
{
  if ((fabs (nmu) > 10e-20) && (fabs (nm) > 10e-20))
    sprintf (strllike, "% 10.5f", nm * nmu);
  else
    {
      if ((fabs (nmu) < 10e-20) && (fabs (nm) < 10e-20))
        {
          sprintf (strllike, "0/0");
        }
      else
        {
          if ((fabs (nmu) < 10e-20) && (fabs (nm) > 10e-20))
            sprintf (strllike, "% 10.5f/0", nm);
          else
            {
              if ((fabs (nmu) > 10e-20) && (fabs (nm) < 10e-20))
                sprintf (strllike, "0");
            }
        }
    }
}

void
print_menu_coalnodes (FILE * file, world_fmt * world, long G, long rep)
{
  long g, pop, minp = world->sumtips, maxp = 0;
  char ss[10];
  long **contribution;
  long nodenum = world->sumtips;
  tarchive_fmt *tl = world->atl[rep][world->locus].tl;
  if (world->options->verbose)
    {
      contribution = (long **) calloc (1, sizeof (long *) * nodenum);
      contribution[0] =
        (long *) calloc (1, sizeof (long) * nodenum * world->numpop);
      for (pop = 1; pop < nodenum; pop++)
        {
          contribution[pop] = contribution[0] + pop * world->numpop;
        }
      for (g = 0; g < G; g++)
        {
          for (pop = 0; pop < world->numpop; pop++)
            {
              contribution[pop][(long) tl[g].p[pop]] += tl[g].copies;
            }
        }
      for (g = 0; g < nodenum; g++)
        {
          for (pop = 0; pop < world->numpop; pop++)
            {
              if (maxp < g && contribution[pop][g] > 0)
                maxp = g;
              if (minp > g && contribution[pop][g] > 0)
                minp = g;
            }
        }
      fprintf (file, "           Coalescent nodes: ");
      for (g = minp; g < maxp + 1; g++)
        {
          fprintf (file, "%2li ", g);
        }
      fprintf (file, "\n");
      for (pop = 0; pop < world->numpop; pop++)
        {
          fprintf (file, "             population %3li: ", pop);
          for (g = minp; g < maxp + 1; g++)
            {
              if (contribution[pop][g] == 0)
                {
                  strcpy (ss, "-");
                }
              else
                {
                  if (contribution[pop][g] >= 100)
                    strcpy (ss, "*");
                  else
                    sprintf (ss, "%-6li", contribution[pop][g]);
                }
              fprintf (file, "%2.2s ", ss);
            }
          fprintf (file, "\n");
        }
      free (contribution[0]);
      free (contribution);
    }
}


void
print_menu_createplot (void)
{
  char nowstr[LINESIZE];
  get_time (nowstr, "%H:%M:%S");
  fprintf (stdout, "%s   Creating data for plots\n", nowstr);
}


void
calc_loci_plane (world_fmt * world, nr_fmt * nr, double ***pl,
                 long loci, double **contours)
{
  long intervals = world->options->plotintervals;
  long i, ii, j, jj, k, kk, m, offset, pop;
  double max1 = 0;
  double max2 = 0;
  double value;
  double *xvalues = world->options->plotxvalues;
  double *yvalues = world->options->plotyvalues;
  plotmax_fmt *plotmax;
  double mllike = world->atl[0][loci].param_like;
  double *param, *lparam, *sparam, *slparam;
  long plusgamma = world->options->gamma ? 1 : 0;
  long nnn = nr->partsize;
  long combined_size = 2 * (nr->numpop2 + 1);
  helper_fmt helper;
  param = (double *) calloc (combined_size, sizeof (double));
  lparam = &param[nr->numpop2 + 1];
  sparam = (double *) calloc (combined_size, sizeof (double));
  slparam = &sparam[nr->numpop2 + 1];
  memcpy (param, world->atl[0][loci].param, sizeof (double) * nr->numpop2);
  if (world->options->gamma)
    {
      plusgamma = 1;
      param[nr->numpop2] = world->atl[0][loci].param[nr->numpop2];
    }
  set_logparam (lparam, param, nnn);
  fill_helper (&helper, param, lparam, world, nr);
  memcpy (sparam, param, sizeof (double) * combined_size);
  for (pop = 0; pop < nr->numpop; pop++)
    //over all populations == == == =
    {
      if (pop > 0 && nr->world->options->migration_model == ISLAND)
        break;
      plotmax = &nr->world->plotmax[loci][pop];
      //memcpy(nr->oparam, param, sizeof(double) * (nr->numpop2));
      //set_logparam(nr->olparam, nr->oparam, nnn);
      offset = nr->mstart[pop];
      for (i = 0; i < intervals; i++)
        //grid over theta == == == == == == ==
        {
          param[pop] = yvalues[i];
          if (nr->world->options->migration_model == ISLAND)
            {
              for (ii = 0; ii < nr->numpop; ii++)
                param[ii] = yvalues[i];
            }
          for (j = 0; j < intervals; j++)
            //grid over 4 Nm or M == == ==
            {
              //immigration(left plot)
              if (nr->world->options->plotvar == PLOT4NM)
                value = xvalues[j] / yvalues[i];
              else
                value = xvalues[j];
              param[offset] = value;
              if (nr->world->options->migration_model == ISLAND)
                {
                  for (ii = nr->numpop; ii < nr->numpop2; ii++)
                    param[ii] = value;
                }
#ifdef __MWERKS__
              eventloop ();
#endif

              set_logparam (lparam, param, nnn);
              fill_helper (&helper, param, lparam, world, nr);
              CALCLIKE (&helper, param, lparam);
              pl[pop][i][j] = EXP (nr->llike - mllike);
              for (k = offset + 1; k < offset + nr->numpop - 1; k++)
                {
                  param[k - 1] = sparam[k - 1];
                  param[k] = value;
                  if (nr->world->options->migration_model == ISLAND)
                    {
                      for (ii = nr->numpop; ii < nr->numpop2; ii++)
                        param[ii] = value;
                    }
                  set_logparam (lparam, param, nnn);
                  fill_helper (&helper, param, lparam, world, nr);
                  CALCLIKE (&helper, param, lparam);
                  pl[pop][i][j] += EXP (nr->llike - mllike);
                }
              param[k - 1] = param[k - 1];
              if (max1 < pl[pop][i][j])
                {
                  max1 = pl[pop][i][j];
                  plotmax->l1 = log (pl[pop][i][j]) + mllike;
                  plotmax->x1 = xvalues[j];
                  plotmax->y1 = yvalues[i];
                }
              if (pl[pop][i][j] == 0)
                pl[pop][i][j] = -DBL_MAX;
              else
                pl[pop][i][j] = log (pl[pop][i][j]) + mllike;
            }
          memcpy (param, sparam, sizeof (double) * combined_size);
          //emmigration(right plot) only if not n - island == == == == == == ==
          if (nr->world->options->migration_model != ISLAND)
            {
              param[pop] = yvalues[i];
              for (j = 0; j < intervals; j++)
                //over all 4 Nm or M == == ==
                {
                  jj = j + intervals;
                  if (nr->world->options->plotvar == PLOT4NM)
                    value = xvalues[j] / yvalues[i];
                  else
                    value = xvalues[j];
                  k = pop;
                  for (m = 0; m < nr->numpop; m++)
                    {
                      if (k != m)
                        {
                          kk = mm2m (k, m, nr->numpop);
                          param[kk] = value;
                          set_logparam (lparam, param, nnn);
                          fill_helper (&helper, param, lparam, world, nr);
                          CALCLIKE (&helper, param, lparam);
                          if (nr->llike > -DBL_MAX)
                            pl[pop][i][jj] += EXP (nr->llike - mllike);
                          param[kk] = sparam[kk];
                        }
                    }
                  if (max2 < pl[pop][i][jj])
                    {
                      max2 = pl[pop][i][jj];
                      plotmax->l2 = log (pl[pop][i][jj]) + mllike;
                      plotmax->x2 = xvalues[j];
                      plotmax->y2 = yvalues[i];
                    }
                  if (pl[pop][i][jj] == 0)
                    pl[pop][i][jj] = -DBL_MAX;
                  else
                    pl[pop][i][jj] = log (pl[pop][i][jj]) + mllike;
                }
            }
        }
      max1 = log (max1) + mllike;
      max2 = log (max2) + mllike;
      contours[pop][0] += max1;
      contours[pop][1] += max1;
      contours[pop][2] += max1;
      contours[pop][3] += max1;
      contours[pop][4] += max2;
      contours[pop][5] += max2;
      contours[pop][6] += max2;
      contours[pop][7] += max2;
      max1 = max2 = 0;
    }
  free (param);
  free (sparam);
}

void
calc_locus_plane (world_fmt * world, nr_fmt * nr, long G,
                  double ***pl, double **contours)
{
  long intervals = world->options->plotintervals;
  long locus;
  long i, ii, j, jj, k, kk, m, offset, pop;
  double max1 = 0;
  double max2 = 0;
  double value;
  double *xvalues = world->options->plotxvalues;
  double *yvalues = world->options->plotyvalues;
  plotmax_fmt *plotmax;
  double mllike = world->param_like;
  double *param, *lparam, *sparam, *slparam;
  long combined_size = 2 * (nr->numpop2 + 1);
  long nnn = nr->partsize;
  long rep = world->options->replicate ? world->repstop : world->repstop - 1;
  helper_fmt helper;
  which_calc_like (world->repkind);
  param = (double *) calloc (combined_size, sizeof (double));
  lparam = &param[nr->numpop2 + 1];
  sparam = (double *) calloc (combined_size, sizeof (double));
  slparam = &sparam[nr->numpop2 + 1];
  if (world->options->datatype == 'g')
    locus = 1;
  else
    locus = world->locus;
  memcpy (param, world->atl[rep][locus].param, sizeof (double) * nr->numpop2);
  set_logparam (lparam, param, nnn);
  fill_helper (&helper, param, lparam, world, nr);
  memcpy (sparam, param, sizeof (double) * combined_size);
  /*
   * population pop -> row of2 pictures with left: immigration into pop
   * right emmigration out of pop immigration is calculated as the summ
   * of all loglikelihoods of m_{k,pop}, emigration is the summ of
   * m_{pop,{k,m}} which is moving over {intervals}.
   */
  for (pop = 0; pop < nr->numpop; pop++)
    {
      if (nr->world->options->migration_model == ISLAND && pop > 0)
        break;
      plotmax = &nr->world->plotmax[nr->world->locus][pop];
      //memcpy(nr->oparam, param, sizeof(double) * nr->numpop2);
      offset = nr->mstart[pop];
      for (i = 0; i < intervals; i++)
        {
          param[pop] = yvalues[i];
          if (nr->world->options->migration_model == ISLAND)
            {
              for (ii = 0; ii < nr->numpop; ii++)
                param[ii] = yvalues[i];
            }
          for (j = 0; j < intervals; j++)
            {
              if (nr->world->options->plotvar == PLOT4NM)
                value = xvalues[j] / yvalues[i];
              else
                value = xvalues[j];
              param[offset] = value;
              if (nr->world->options->migration_model == ISLAND)
                {
                  for (ii = nr->numpop; ii < nr->numpop2; ii++)
                    param[ii] = value;
                }
              set_logparam (lparam, param, nnn);
              fill_helper (&helper, param, lparam, world, nr);
              nr->llike = CALCLIKE (&helper, param, lparam);
              pl[pop][i][j] = EXP (nr->llike - mllike);
              if (nr->world->options->migration_model != ISLAND)
                {
                  for (k = offset + 1; k < offset + nr->numpop - 1; k++)
                    {
                      param[k - 1] = sparam[k - 1];
                      param[k] = value;
                      lparam[k] = log (value);
                      fill_helper (&helper, param, lparam, world, nr);
                      nr->llike = CALCLIKE (&helper, param, lparam);
                      pl[pop][i][j] += EXP (nr->llike - mllike);
                    }
                }
              else
                {
                  //pl[pop][i][j] += pl[pop][i][j];
                  k = offset + 1;
                }
              param[k - 1] = sparam[k - 1];
              if (max1 < pl[pop][i][j])
                {
                  max1 = pl[pop][i][j];
                  plotmax->l1 = log (pl[pop][i][j]) + mllike;
                  plotmax->x1 = xvalues[j];
                  plotmax->y1 = yvalues[i];
                }
              if (pl[pop][i][j] == 0)
                pl[pop][i][j] = -DBL_MAX;
              else
                pl[pop][i][j] = log (pl[pop][i][j]) + mllike;
            }   //end of j intervals
        }   //end of i intervals
      memcpy (param, sparam, sizeof (double) * combined_size);
      if (nr->world->options->migration_model != ISLAND)
        {
          for (i = 0; i < intervals; i++)
            {
              param[pop] = yvalues[i];
              if (nr->world->options->migration_model == ISLAND)
                {
                  for (ii = 0; ii < nr->numpop; ii++)
                    param[ii] = yvalues[i];
                }
              for (j = 0; j < intervals; j++)
                {
                  jj = j + intervals;
                  if (nr->world->options->plotvar == PLOT4NM)
                    value = xvalues[j] / yvalues[i];
                  else
                    value = xvalues[j];
                  pl[pop][i][jj] = 0.;
                  k = pop;
                  for (m = 0; m < nr->numpop; m++)
                    {
                      if (k != m)
                        {
                          kk = mm2m (k, m, nr->numpop);
                          param[kk] = value;
                          lparam[kk] = log (value);
                          fill_helper (&helper, param, lparam, world, nr);
                          nr->llike = CALCLIKE (&helper, param, lparam);
                          //(boolean) world->options->gamma);
                          pl[pop][i][jj] += EXP (nr->llike - mllike);
                          param[kk] = sparam[kk];
                        }
                    }
                  if (max2 < pl[pop][i][jj])
                    {
                      max2 = pl[pop][i][jj];
                      plotmax->l2 = log (pl[pop][i][jj]) + mllike;
                      plotmax->x2 = xvalues[j];
                      plotmax->y2 = yvalues[i];
                    }
                  if (pl[pop][i][jj] == 0)
                    pl[pop][i][jj] = -DBL_MAX;
                  else
                    pl[pop][i][jj] = log (pl[pop][i][jj]) + mllike;
                }  //end of j interval
            }   //end of i interval
        }
      max1 = log (max1) + mllike;
      max2 = log (max2) + mllike;
      contours[pop][0] += max1;
      contours[pop][1] += max1;
      contours[pop][2] += max1;
      contours[pop][3] += max1;
      contours[pop][4] += max2;
      contours[pop][5] += max2;
      contours[pop][6] += max2;
      contours[pop][7] += max2;
      max1 = max2 = 0;
    }
  free (param);
  free (sparam);
}

void
fill_plotplane (world_fmt * world, char **plane, double **pl,
                double *contours)
{
  long i, j, ii, jj, z, zz = 0;
  char line[100];
  long myval[PLANEBIGTICKS] = PLANEBIGTICKVALUES;
  long intervals = MAX (world->options->plotintervals, PLANESIZE);
  double delta, deltahalf;
  delta = (double) intervals / PLANESIZE;
  deltahalf = delta / 2.;
  for (i = 0; i < PLANESIZE; i++)
    {
      if (i % 7)
        line[i] = '-';
      else
        line[i] = '+';
    }
  line[i] = '\0';
  sprintf (plane[0], "     +%s+    +%s+   ", line, line);
  for (i = 0; i < PLANESIZE; i++)
    {
      memset (plane[i + 1], ' ', sizeof (char) * (2 * PLANESIZE + 20));
      plane[i + 1][2 * PLANESIZE + 19] = '\0';
      if (!((i) % 7))
        {
          sprintf (plane[i + 1], "  %2.0li +", myval[zz++]);
          if (myval[zz - 1] == 0)
            plane[i + 1][3] = '0';
        }
      else
        plane[i + 1][5] = '|';
      ii = (long) (delta * i + deltahalf);
      for (j = 0; j < PLANESIZE; j++)
        {
          jj = (long) (delta * j + deltahalf);
          if (pl[ii][jj] < contours[1])
            {
              if (pl[ii][jj] < contours[2])
                {
                  if (pl[ii][jj] < contours[3])
                    {
                      plane[i + 1][j + 6] = ' ';
                    }
                  else
                    {
                      plane[i + 1][j + 6] = '-';
                    }
                }
              else
                {
                  plane[i + 1][j + 6] = '+';
                }
            }
          else
            {
              if (pl[ii][jj] < (contours[0] - EPSILON))
                {
                  plane[i + 1][j + 6] = '*';
                }
              else
                {
                  plane[i + 1][j + 6] = 'X';
                }
            }
        }
      if ((i) % 7)
        {
          plane[i + 1][j + 6] = '|';
          plane[i + 1][j + 11] = '|';
        }
      else
        {
          plane[i + 1][j + 6] = '+';
          plane[i + 1][j + 11] = '+';
        }
      for (j = PLANESIZE + 12; j < 2 * PLANESIZE + 12; j++)
        {
          z = (long) (delta * (j - 12) + deltahalf);
          if (pl[ii][z] < contours[5])
            {
              if (pl[ii][z] < contours[6])
                {
                  if (pl[ii][z] < contours[7])
                    {
                      plane[i + 1][j] = ' ';
                    }
                  else
                    {
                      plane[i + 1][j] = '-';
                    }
                }
              else
                {
                  plane[i + 1][j] = '+';
                }
            }
          else
            {
              if (pl[ii][z] < (contours[4] - EPSILON))
                {
                  plane[i + 1][j] = '*';
                }
              else
                {
                  plane[i + 1][j] = 'X';
                }
            }
        }
      if ((i) % 7)
        {
          plane[i + 1][j] = '|';
          plane[i + 1][j + 1] = '\0';
        }
      else
        {
          plane[i + 1][j] = '+';
          plane[i + 1][j + 1] = '\0';
        }
    }
  sprintf (plane[PLANESIZE + 1], "     +%s+    +%s+", line, line);
  sprintf (plane[PLANESIZE + 2],
           "      1      2      3      4      5      6      1      2      3      4      5      6");
}

void
print_mathematica (world_fmt * world, double ***plane, long x, long y)
{
  long i, j, pop;
  FILE *mathfile = world->mathfile;
  if (world->options->plot == 1 && world->options->plotmethod == PLOTALL)
    {
      for (pop = 0; pop < world->numpop; pop++)
        {
          for (i = 0; i < x; i++)
            {
              for (j = 0; j < y - 1; j++)
                {
                  fprintf (mathfile, "%20.20g ", plane[pop][i][j]);
                }
              fprintf (mathfile, "%20.20g\n", plane[pop][i][j]);
            }
          if (world->options->migration_model == ISLAND)
            break;
          for (i = 0; i < x; i++)
            {
              for (j = y; j < 2 * y - 1; j++)
                {
                  fprintf (mathfile, "%20.20g ", plane[pop][i][j]);
                }
              fprintf (mathfile, "%20.20g\n", plane[pop][i][j]);
            }
        }
    }
  fflush (mathfile);
}

void
print_cov (world_fmt * world, long numpop, long loci, double ***cov)
{
  FILE *outfile = world->outfile;
  long locus, skipped = 0;
  double *corr;
  corr = (double *) calloc (1, sizeof (double) * (world->numpop2 + 1));
  fprintf (outfile, "\n\n");
  fprintf (outfile,
           "--------------------------------------------------------------------\n");
  fprintf (outfile, "MCMC estimation:\n");
  fprintf (outfile,
           "Covariance matrix(*)                         Correlation matrix\n");
  fprintf (outfile,
           "-------------------------------------------  -----------------------\n");
  for (locus = 0; locus < world->loci; locus++)
    {
      if (world->data->skiploci[locus])
        {
          skipped++;
          continue;
        }
      fprintf (outfile, "Locus %li:\n", locus + 1);
      print_cov_table (outfile, locus, world, corr, 0);
      fprintf (outfile, "\n");
    }
  fprintf (outfile, "\n\n");
  if (world->loci - skipped > 1)
    {
      if (world->options->gamma)
        {
          fprintf (outfile, "Over all loci\n");
          fprintf (outfile,
                   "------------------------------------------------------  ");
          fprintf (outfile, "----------------------------\n");
          print_cov_table (outfile, locus, world, corr, 1);
        }
      else
        {
          fprintf (outfile, "Over all loci\n");
          fprintf (outfile,
                   "-------------------------------------------  -----------------------\n");
          print_cov_table (outfile, locus, world, corr, 0);
        }
    }
  free (corr);
}

void
print_cov_table (FILE * outfile, long locus, world_fmt * world,
                 double *corr, long addvar)
{
  long i, j;
  double denom, temp1, temp2;
  for (i = 0; i < world->numpop2 + addvar; i++)
    {
      for (j = 0; j < world->numpop2 + addvar; j++)
        {
          temp1 = fabs (world->cov[locus][i][i]);
          if (temp1 < DBL_EPSILON)
            temp1 = DBL_EPSILON;
          temp2 = fabs (world->cov[locus][j][j]);
          if (temp2 < DBL_EPSILON)
            temp2 = DBL_EPSILON;
          denom = 0.5 * (log (temp1) + log (temp2));
          if ((temp1 = EXP (denom)) == 0.0)
            corr[j] = 9.99;
          else
            corr[j] = world->cov[locus][i][j] / EXP (denom);
        }
      for (j = 0; j < world->numpop2 + addvar; j++)
        {
          //#if TESTING
          //          fprintf (outfile, "% 20.10f ", world->cov[locus][i][j]); /* was 10.4 */
          //#else

          fprintf (outfile, "% 10.4f ", world->cov[locus][i][j]);
          //#endif

        }
      for (j = 0; j < world->numpop2 + addvar; j++)
        {
          if (corr[j] < 9.)
            fprintf (outfile, "% 4.2f ", corr[j]);
          else
            fprintf (outfile, "  --  ");
        }
      fprintf (outfile, "\n");
    }
}


void
free_seqx (node * p, long sites)
{
  long j;
  if (p->type != 't')
    {
      if (p->next->back != NULL)
        {
          free_seqx (crawlback (p->next), sites);
        }
      if (p->next->next->back != NULL)
        {
          free_seqx (crawlback (p->next->next), sites);
        }
    }
  if (p->type == 'r')
    return;
  for (j = 0; j < sites; j++)
    {
      free (p->x.s[j]);
    }
}

void
free_x (node * p)
{
  if (p->type != 't')
    {
      if (p->next->back != NULL)
        {
          free_x (crawlback (p->next));
        }
      if (p->next->next->back != NULL)
        {
          free_x (crawlback (p->next->next));
        }
    }
  if (p->type == 'r')
    return;
  free (p->x.a);
}


void
print_CV (world_fmt * world)
{
  long i;
  long elem;
  char temp[100];
  fprintf (world->outfile, "\n\nVariance and coefficient of variance\n");
  if (world->loci - world->skipped > 1)
    {
      elem = world->numpop2 + (world->options->gamma ? 1 : 0);
      fprintf (world->outfile, "PARAM@ ");
      for (i = 0; i < elem; i++)
        {
          fprintf (world->outfile, "%20.20g ",
                   world->atl[0][world->loci].param[i]);
        }
      fprintf (world->outfile, "\nVAR@   ");
      for (i = 0; i < elem; i++)
        {
          fprintf (world->outfile, "%20.20g ", world->cov[world->loci][i][i]);
        }
      fprintf (world->outfile, "\nCV@    ");
      for (i = 0; i < elem; i++)
        {
          strcpy (temp, "-");
          if (world->cov[world->loci][i][i] >= 0)
            {
              sprintf (temp, "%20.20g",
                       (sqrt (world->cov[world->loci][i][i])) /
                       world->atl[0][world->loci].param[i]);
            }
          fprintf (world->outfile, "%-s ", temp);
        }
      fprintf (world->outfile, "\n");
    }
  else
    {    /* one locus */
      elem = world->numpop2;
      fprintf (world->outfile, "PARAM@ ");
      for (i = 0; i < elem; i++)
        {
          fprintf (world->outfile, "%20.20g ", world->atl[0][1].param[i]);
        }
      fprintf (world->outfile, "\nVAR@   ");
      for (i = 0; i < elem; i++)
        {
          fprintf (world->outfile, "%20.20g ", world->cov[0][i][i]);
        }
      fprintf (world->outfile, "\nCV@    ");
      for (i = 0; i < elem; i++)
        {
          strcpy (temp, "-");
          if (world->cov[0][i][i] >= 0)
            {
              sprintf (temp, "%20.20g",
                       (sqrt (world->cov[0][i][i])) /
                       world->atl[0][1].param[i]);
            }
          fprintf (world->outfile, "%-s ", temp);
        }
      fprintf (world->outfile, "\n");
    }
}


void
print_param (FILE * file, boolean usem, world_fmt *world, long nn, char spacer[])
{
  long i, j;   //, fpos;
  long tt = nn;
  long counter;
  double *param = world->param0;
  fprintf (file, "%sPop. Theta    %s\n", spacer, usem ? "M" : "Theta*M");
  for (i = 0; i < nn; i++)
    {
      //fprintf (file, " ");
      counter = 0;
      fprintf (file, "%s%3li % 7.5f", spacer, i + 1, param[i]);
      for (j = 0; j < nn; j++)
        {
          if (i != j)
            {
              if (usem)
                fprintf (file, "% 7.1f", param[tt++]);
              else
                fprintf (file, "% 7.5f", param[i] * param[tt++]);
            }
          else
            fprintf (file, " ------");
          if (counter++ > 10)
            {
              counter = 0;
              fprintf (file, "\n%s         ", spacer);
            }
        }
      //fpos = ftell (file);
      //fseek (file, fpos - sizeof (char), SEEK_SET);
      fprintf (file, " \n"); //, spacer);
    }
  fprintf (file, " \n");
#ifdef LONGSUM

  fprintf(file,"\nRates (at specific times):\n");
  for(i=0;i<nn; i++ )
    {
      j = i*3; //because we use only 3 different rates per pop
      fprintf(file,"Pop %li: %g (%g) %g (%g) %g (%g)\n", i+1,
              world->flucrates[j],world->flucrates[j+nn*3],
              world->flucrates[j+1], world->flucrates[j+1+nn*3],
              world->flucrates[j+2], world->flucrates[j+2+nn*3]);
    }
#endif
}

void
print_param_old (FILE * file, boolean usem, double *param, long nn,
                 char spacer[])
{
  long i, j, fpos;
  long tt = nn;
  long counter;
  fprintf (file,
           "%sMigration matrix with Theta values on diagonal\n%sParam=",
           spacer, spacer);
  for (i = 0; i < nn; i++)
    {
      fprintf (file, " ");
      counter = 0;
      for (j = 0; j < nn; j++)
        {
          if (i != j)
            fprintf (file, "% 7.5f", (usem ? 1.0 : param[i]) * param[tt++]);
          else
            fprintf (file, "% 7.5f", param[i]);
          if (counter++ > 10)
            {
              counter = 0;
              fprintf (file, "\n%s         ", spacer);
            }
        }
      fpos = ftell (file);
      fseek (file, fpos - sizeof (char), SEEK_SET);
      fprintf (file, " \n%s      ", spacer);
    }
  fprintf (file, " \n");
}

/*
 * print header
 * ===========================================================================
 * === Titletext
 * ===========================================================================
 * === Population     Loci  Ln(L)   Theta    4Nm [4Ne mu] xx     xx     xx
 * xx     xx     xx xx.... -------------- ---- -------- --------
 * ----------------------------------------
 */
void
print_result_header (char *titletext, world_fmt * world)
{
  long p1, zz;
  char dline[] =
    "==============================================================================";
  char sline[] =
    "-------------- ---- -------- -------- ----------------------------------------";
  fprintf (world->outfile, "%s\n", dline);
  fprintf (world->outfile, "%s\n", titletext);
  fprintf (world->outfile, "%s\n", dline);
  if (!world->options->usem)
    {
      fprintf (world->outfile,
               "Population [x] Loc.  Ln(L)   Theta    4Nm [x=receiving population]\n");
      fprintf (world->outfile, "                             [4Ne mu] ");
    }
  else
    {
      fprintf (world->outfile,
               "Population [x] Loc.  Ln(L)   Theta    M [m/mu] [x=receiving population]  \n");
      fprintf (world->outfile, "                             [4Ne mu]   ");
      //              fprintf (world->outfile, "                             [4Ne mu]   [m/mu]");
    }
  zz = 38;
  for (p1 = 0; p1 < world->numpop; p1++)
    {
      zz = zz + 7;
      if (zz > 79)
        {
          zz = 38;
          fprintf (world->outfile,
                   "\n                                      ");
        }
      fprintf (world->outfile, "%2li,x    ", p1 + 1);
    }
  fprintf (world->outfile, "\n%s\n", sline);
}

void print_popstring(long pop, world_fmt *world, option_fmt *options, data_fmt *data)
{
  char popstring[LINESIZE];
  if (options->readsum)
    {
      sprintf (popstring, "%2li: ", pop + 1);
    }
  else
    {
      sprintf (popstring, "%2li: %s", pop + 1, data->popnames[pop]);
    }
  fprintf (world->outfile, "%-14.14s ", popstring);
}

void print_replicate(world_fmt *world, long maxrep, long rep, long locus)
{
  char repstring[LINESIZE];
  sprintf (repstring, "%2li", rep + 1);
  fprintf (world->outfile, "%s%2li%2s ", locus == 0
           && rep == 0 ? "" : "               ", locus + 1,
           maxrep > 1 ? (rep ==
                         maxrep - 1 ? " A" : repstring) : "  ");
}

void
print_result_population (long pop, world_fmt * world,
                         option_fmt * options, data_fmt * data)
{
  long skipped = 0, locus;
  long maxrep = world->options->replicate ?
                (world->options->replicatenum > 0 ?
                 world->options->replicatenum + 1 : world->options->lchains + 1) : 1;
  long rep;
  print_popstring(pop, world, options, data);
  for (locus = 0; locus < world->loci; locus++)
    {
      if (world->data->skiploci[locus])
        {
          skipped++;
          continue;
        }
      for (rep = 0; rep < maxrep; rep++)
        {
          print_replicate(world, maxrep, rep, locus);
          fprintf (world->outfile, "% 8.3f ",
                   world->atl[rep][locus].param_like);
          print_result_param (world->outfile, world->atl[rep][locus].param,
                              world->numpop, pop, world->options->usem);
        }
    }
  if (world->loci - skipped > 1)
    {
      fprintf (world->outfile, "                All ");
      //locus is exactly world->loci
      // re is always one because we have only replication of single locus chains
      fprintf (world->outfile, "% 8.3f ", world->atl[0][locus].param_like);
      print_result_param (world->outfile, world->atl[0][locus].param,
                          world->numpop, pop, world->options->usem);
    }
  /* fprintf(world->outfile,"%s\n",sline);     */
}


void
print_result_param (FILE * file, double *param, long numpop, long pop,
                    boolean usem)
{
  long i;
  long linelen = 0;
  long msta = mstart (pop, numpop);
  long msto = mend (pop, numpop);
  double tmp = 0;
  if (param[pop] <= SICK_VALUE)
    fprintf (file, "     -    ");
  else
    {
      if (param[pop] < 0.0001 && param[pop] > 0)
        fprintf (file, "%3.2e ", param[pop]);
      else
        fprintf (file, "%8.5f ", param[pop]);
    }
  for (i = msta; i < msto; i++)
    {
      if (pop == i - msta)
        {
          fprintf (file, "------- ");
          linelen++;
        }
      if (linelen > 4)
        {
          fprintf (file, "\n                                      ");
          linelen = 0;
        }
      linelen++;
      if ((param[i] <= SICK_VALUE) || (param[pop] <= SICK_VALUE))
        fprintf (file, "    -    ");
      else
        {
          if (usem)
            tmp = param[i];
          else
            tmp = param[pop] * param[i];
          if (tmp < 0.001 && tmp > 0)
            fprintf (file, "%3.2e ", tmp);
          else
            fprintf (file, "%7.4f ", tmp);
        }
    }
  if (pop == numpop - 1)
    fprintf (file, "-------");
  fprintf (file, "\n");
}

void
print_result_fst (long pop, world_fmt * world, data_fmt * data)
{

  char popstring[LINESIZE];
  long skipped = 0, locus;
  sprintf (popstring, "%2li: %s", pop + 1, data->popnames[pop]);
  fprintf (world->outfile, "%14.14s ", popstring);
  for (locus = 0; locus < world->loci; locus++)
    {
      if (world->data->skiploci[locus])
        {
          skipped++;
          continue;
        }
      fprintf (world->outfile, "%s%4li ",
               locus == 1 ? "" : "               ", locus);
      fprintf (world->outfile, "   -    ");
      print_result_param (world->outfile, world->fstparam[locus - 1],
                          world->numpop, pop, world->options->usem);
    }
  if (world->loci - skipped > 1)
    {
      fprintf (world->outfile, "                All ");
      fprintf (world->outfile, "   -    ");
      print_result_param (world->outfile, world->fstparam[locus - 1],
                          world->numpop, pop, world->options->usem);
    }
  /* fprintf(world->outfile,"%s\n",sline);     */
}

void
klone (world_fmt * original, world_fmt * kopie,
       option_fmt * options, data_fmt * data, double temperature)
{
  long i, j;
  kopie->repkind = SINGLECHAIN;
  kopie->loci = original->loci;
  kopie->skipped = original->skipped;
  kopie->locus = original->locus;
  kopie->numpop = original->numpop;
  kopie->numpop2 = original->numpop2;
  kopie->sumtips = original->sumtips;
  memcpy (kopie->mstart, original->mstart, sizeof (int) * original->numpop);
  memcpy (kopie->mend, original->mend, sizeof (int) * original->numpop);
  kopie->atl = NULL;
  fill_worldoptions (kopie->options, options, original->numpop);
  switch (options->datatype)
    {
    case 'm':
      for (i = 0; i < options->micro_stepnum; i++)
        {
          for (j = 0; j < options->micro_stepnum; j++)
            {
              kopie->options->steps[i][j] = original->options->steps[i][j];
            }
        }
      break;
    case 'a':
      kopie->data->freq = original->data->freq;
      kopie->data->freqlast = original->data->freqlast;
    }
  fill_worlddata (kopie->data, original, data, original->numpop,
                  options->readsum);
  if (strchr (SEQUENCETYPES, original->options->datatype))
    {
      init_sequences2 (kopie, original->data->seq, original->locus);
      init_sequences (kopie, options, data, original->locus);
      copy_seq (original, kopie);
    }
  else
    kopie->data->seq->endsite = original->data->seq->endsite;
  /* mighistloci not copied */
  kopie->options->datatype = original->options->datatype;
  copy_tree (original, kopie);
  if (strchr (SEQUENCETYPES, original->options->datatype))
    init_tbl (kopie, kopie->locus);
  memcpy (kopie->param0, original->param0,
          sizeof (double) * original->numpop2);
  memcpy (kopie->param00, original->param00,
          sizeof (double) * original->numpop2);
  /* fstparam not copied */
  memcpy (kopie->lineages, original->lineages,
          sizeof (long) * original->numpop);
  create_treetimelist (kopie, &kopie->treetimes, kopie->locus);
  memcpy (kopie->mig0list, original->mig0list,
          sizeof (double) * original->numpop);
  for (i = 0; i < original->numpop; ++i)
    {
      memcpy (kopie->migproblist[i], original->migproblist[i],
              sizeof (double) * (original->numpop - 1));
    }
  memcpy (kopie->design0list, original->design0list,
          sizeof (long) * original->numpop);
#ifdef UEP

  if (original->options->uep)
    {
      for (j = 0; j < original->data->uepsites; ++j)
        {
          memcpy (kopie->ueplike[j], original->ueplike[j],
                  sizeof (double) * original->numpop);
          memcpy (kopie->ueptime[j].populations,
                  original->ueptime[j].populations,
                  sizeof (long) * original->ueptime[j].size);
          memcpy (kopie->ueptime[j].ueptime,
                  original->ueptime[j].ueptime,
                  sizeof (double) * original->ueptime[j].size);
          kopie->ueptime[j].size = original->ueptime[j].size;
        }
    }
  if (original->options->uep)
    for (j = 0; j < original->loci; j++)
      memcpy (kopie->uepanc[j],
              original->uepanc[j],
              sizeof (long) * original->data->uepsites *
              original->numpop * 2);
#endif
  /* here would come some things that are filled later */
  kopie->heat[0] = 1. / temperature;
  kopie->starttime = original->starttime;
  kopie->treesdone = original->treesdone;
  kopie->treestotal = original->treestotal;
  kopie->chains = original->chains;
  kopie->start = original->start;
  set_tree_dirty (kopie->root);
  kopie->likelihood[0] = treelikelihood (kopie);
}

void
klone_part (world_fmt * original, world_fmt * kopie,
            option_fmt * options, data_fmt * data, double temperature)
{
  long i;
#ifdef UEP

  long j;
#endif

  kopie->repkind = SINGLECHAIN;
  kopie->loci = original->loci;
  kopie->skipped = original->skipped;
  kopie->locus = original->locus;
  kopie->numpop = original->numpop;
  kopie->numpop2 = original->numpop2;
  kopie->sumtips = original->sumtips;
  memcpy (kopie->param0, original->param0,
          sizeof (double) * original->numpop2);
  memcpy (kopie->param00, original->param00,
          sizeof (double) * original->numpop2);
  memcpy (kopie->mig0list, original->mig0list,
          sizeof (double) * original->numpop);
  for (i = 0; i < original->numpop; ++i)
    {
      memcpy (kopie->migproblist[i], original->migproblist[i],
              sizeof (double) * (original->numpop - 1));
    }
  memcpy (kopie->design0list, original->design0list,
          sizeof (long) * original->numpop);
#ifdef UEP

  if (original->options->uep)
    {
      for (j = 0; j < original->data->uepsites; ++j)
        {
          memcpy (kopie->ueplike[j], original->ueplike[j],
                  sizeof (double) * original->numpop);
          memcpy (kopie->ueptime[j].populations,
                  original->ueptime[j].populations,
                  sizeof (long) * original->ueptime[j].size);
          memcpy (kopie->ueptime[j].ueptime,
                  original->ueptime[j].ueptime,
                  sizeof (double) * original->ueptime[j].size);
          kopie->ueptime[j].size = original->ueptime[j].size;
        }
    }
  if (original->options->uep)
    for (j = 0; j < original->loci; j++)
      memcpy (kopie->uepanc[j],
              original->uepanc[j],
              sizeof (long) * original->data->uepsites *
              original->numpop * 2);
#endif
}

void
clone_polish (world_fmt * original, world_fmt * kopie)
{
  long i;
  kopie->increment = original->increment;
  kopie->chains = original->chains;
  kopie->lsteps = original->lsteps;
  kopie->numlike = original->numlike;
  memcpy (kopie->mig0list, original->mig0list,
          sizeof (double) * original->numpop);
  for (i = 0; i < original->numpop; ++i)
    {
      memcpy (kopie->migproblist[i], original->migproblist[i],
              sizeof (double) * (original->numpop - 1));
    }
  memcpy (kopie->design0list, original->design0list,
          sizeof (long) * original->numpop);
  polish_world (kopie);
  if (kopie->likelihood[0] == 0)
    kopie->likelihood[0] = original->likelihood[0];
}

long
chance_swap_tree (world_fmt * tthis, world_fmt * that)
{
  double templike;
  double a, b, rr, quot;
  double ha = tthis->heat[0];
  double hb = that->heat[0];
#ifdef BAYESUPDATE

  double tempparam;
#endif
#ifdef UEP

  double **tempuep;
  ueptime_fmt *tempueptime;
  long *tempanc;
  double treelen;
#endif

  timelist_fmt *templist;
  //vtlist * temptl;
#ifdef BAYESUPDATE

  if(tthis->options->bayes_infer)
    {
      a = tthis->likelihood[tthis->numlike - 1] + tthis->bayes->oldval;
      b = that->likelihood[that->numlike - 1] + that->bayes->oldval;
    }
  else
    {
      a = tthis->likelihood[tthis->numlike - 1];
      b = that->likelihood[that->numlike - 1];
    }
#else
  a = tthis->likelihood[tthis->numlike - 1];
  b = that->likelihood[that->numlike - 1];
#endif

  if (tthis == that)
    error ("mcmcmc swap error: original and target chain are the same");
  rr = log (RANDUM ());
  quot = a * hb + b * ha - a * ha - b * hb;
  if (rr < quot)
    {
#ifdef BAYESUPDATE

      if (tthis->options->bayes_infer)
        {
          tempparam = tthis->param0[tthis->bayes->paramnum];
          tthis->param0[tthis->bayes->paramnum] = that->param0[tthis->bayes->paramnum];
          that->param0[tthis->bayes->paramnum] = tempparam;
          swap_tree (tthis, that);
        }
      else
        swap_tree (tthis, that);
#else

      swap_tree (tthis, that);
#endif

      //printf("%s swapped with %s\n", tthis->name, that->name);
      if (tthis->root == that->root)
        error ("mcmcmc swap error: roots are identical?!");
      templike = tthis->likelihood[tthis->numlike - 1];
      tthis->likelihood[tthis->numlike - 1] =
        that->likelihood[that->numlike - 1];
      that->likelihood[that->numlike - 1] = templike;
      templist = tthis->treetimes;
      tthis->treetimes = that->treetimes;
      that->treetimes = templist;
      if (tthis->treetimes == that->treetimes)
        error ("this is that treetimes, shreek!");
#ifdef UEP

      if (tthis->options->uep)
        {
          tempuep = tthis->ueplike;
          tthis->ueplike = that->ueplike;
          that->ueplike = tempuep;
          tempueptime = tthis->ueptime;
          tthis->ueptime = that->ueptime;
          that->ueptime = tempueptime;
          templike = tthis->ueplikelihood;
          tthis->ueplikelihood = that->ueplikelihood;
          tthis->ueplikelihood = templike;
          tempanc = tthis->oldrootuep;
          tthis->oldrootuep = that->oldrootuep;
          tthis->oldrootuep = tempanc;
          treelen = tthis->treelen;
          tthis->treelen = that->treelen;
          that->treelen = treelen;
        }
#endif
      return 1;
    }
  return 0;
}


void
advance_clone_like (world_fmt * world, long accepted, long *j)
{
  if (accepted > 0)
    {
      world->numlike = 1 + *j + (accepted > 0 ? 1 : 0);
      *j += 1;
      world->likelihood =
        (double *) realloc (world->likelihood,
                            sizeof (double) * world->numlike);
      world->likelihood[*j] = world->likelihood[*j - 1];
    }
}

void
polish_world (world_fmt * world)
{
  double tmp = world->likelihood[0];
  world->numlike = 1;
  //world->atl[rep][0].allocT;
  world->likelihood = (double *) realloc (world->likelihood, sizeof (double));
  //*(world->atl[rep][0].allocT));
  memset (world->likelihood, 0, sizeof (double));
  //*(world->atl[rep][0].allocT));
  world->likelihood[0] = tmp;
#ifdef UEP

  if (world->options->uep)
    {
      world->treelen = 0.0;
      calc_treelength (world->root->next->back, &world->treelen);
      update_uep (world->root->next->back, world);
      check_uep_root (world->root->next->back, world);
    }
#endif
}
