/*------------------------------------------------------
Maximum likelihood estimation
of migration rate  and effectice population size
using a Metropolis-Hastings Monte Carlo algorithm
-------------------------------------------------------
 M A I N   P R O G R A M
 
 drives the whole program MIGRATE.
 
 for options of the program: README and parmfile.distrib,
 options.c
and of course the documentation at:
        http://evolution.genetics.washington.edu/lamarc/migrate.html
 
 Peter Beerli, started 1997, Seattle
 beerli@genetics.washington.edu
 
 Copyright 2001 Peter Beerli and Joseph Felsenstein
 
 This software is distributed free of charge for non-commercial use
 and is copyrighted. Of course, we do not guarantee that the software
 works and are not responsible for any damage you may cause or have.
 
$Id: main.c,v 1.81 2002/06/20 06:25:35 beerli Exp $
-------------------------------------------------------*/
#include "migration.h"
#include "definitions.h"
#include "heating.h"
#include "bayes.h"
#include "world.h"
#include "data.h"
#include "options.h"
#include "mcmc.h"
#include "broyden.h"
#include "combroyden.h"
#include "menu.h"
#include "random.h"
#include "sequence.h"
#include "tree.h"
#include "tools.h"
#include "profile.h"
#include "aic.h"
#include "lrt.h"
#ifdef UEP
#include "uep.h"
#endif
#include "migrate_mpi.h"
#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif
//use this if you have nice() and need to be nice with others
#ifdef HAVE_NICE
#include <unistd.h>
#endif
//
#define EARTH  universe[0]
#define VENUS  universe[1]
#define MERKUR universe[2]
#define SUN    universe[3]

/* GLOBAL VARIABLES +++++++++++++++++++++++++++++++++++++ */
int             myID;  /* used if MPI is running otherwise == 0       */
//int           allID;
int             color;  /* either 1 or MPI_UNDEFINED */
int             numcpu;  /* used if MPI is running otherwise == 1     */
int             locidone; /* used in MPI workers */
#ifdef MPI
MPI_Comm        comm_world;
#endif
#ifdef SLOWNET
int             profiledone; /* used in MPI workers in profiles on slow networks */
#endif
#ifdef PTHREADS
tpool_t         heating_pool;
#else
#ifndef tpool_t
#define tpool_t char
#endif
tpool_t         heating_pool;
#endif

char            appl[20];
long           *seed;
long           *newseed;

int
setup_locus(long locus, world_fmt * world, option_fmt * options,
            data_fmt * data);

void
condense_time(world_fmt * world, long *step, long *j,
              long *accepted, long *G, long *steps, long oldsteps,
              long rep);

long            set_repkind(option_fmt * options);

void
heating_init(world_fmt ** universe, int usize, data_fmt * data,
             option_fmt * options, char *this_string);

void
heating_prepare(world_fmt ** universe, int usize,
                option_fmt * options, data_fmt * data, long rep);

void            heating_prepare2(world_fmt ** universe, int usize);

long
replicate_number(option_fmt * options, long chain, char type,
                 long replicate);

void
combine_loci(char type, option_fmt * options, world_fmt * world,
             long *Gmax);

void
run_sampler(option_fmt * options, data_fmt * data,
            world_fmt ** universe, int usize, long *outfilepos,
            long *Gmax);

void
run_loci(world_fmt ** universe, int usize, option_fmt * options,
         data_fmt * data, tpool_t * heating_pool, long maxreplicate,
         long *treefilepos, long *Gmax);

void
run_locus(world_fmt ** universe, int usize, option_fmt * options,
          data_fmt * data, tpool_t * heating_pool, long maxreplicate,
          long locus, long *treefilepos, long *Gmax);

void
heated_swap(world_fmt ** universe, worldoption_fmt * options,
            double *acc_vh, double *acc_h, double *acc_w,
            double *acc_c, long *vh, long *h, long *w, long *jj,
            long *j, long *hot, long *warm, long *cold);

void
print_progress(worldoption_fmt * options, world_fmt * world,
               long rep, long visited, long accepted);

void
change_chaintype(long locus, char *type, world_fmt * world,
                 long *increment, long *oldsteps, long *chains,
                 option_fmt * options);

void
prepare_next_chain(world_fmt ** universe, worldoption_fmt * options,
                   char type, long chain, long *chains,
                   long *pluschain, long locus, long replicate);

void
print_heating_progress(world_fmt ** universe,
                       worldoption_fmt * options,
                       long steps, long increment, double *acc_vh,
                       double *acc_h, double *acc_w, double *acc_c,
                       long *hot, long *warm, long *cold);

long
analyze_olddata(world_fmt * world, option_fmt * options,
                data_fmt * data, long *outfilepos);

void            profile_tables(option_fmt * options, world_fmt * world, long *gmaxptr);

void            finish_mac(option_fmt * options, data_fmt * data);

int
setup_locus(long locus, world_fmt * world, option_fmt * options,
            data_fmt * data);

void
condense_time(world_fmt * world, long *step, long *j,
              long *accepted, long *G, long *steps,
              long oldsteps, long rep);

long            set_repkind(option_fmt * options);

void            heating_prepare2(world_fmt ** universe, int usize);

long
replicate_number(option_fmt * options, long chain, char type,
                 long replicate);

void
print_heating_progress2(FILE * file, worldoption_fmt * options,
                        world_fmt ** universe, double acc_vh,
                        double acc_h, double acc_w, double acc_c,
                        long hot, long warm, long cold);


void            get_treedata(world_fmt * world, option_fmt * options);
void            get_mighistdata(world_fmt * world, option_fmt * options);

extern void unset_penalizer_function(boolean inprofiles);

/* Functions ++++++++++++++++++++++++++++++++++++++++++++++++ */
/*--------------------------------------------
main program
*/
int
main(int argc, char **argv)
{
    char           *this_string;
    char            type = 's';
    data_fmt       *data;
    int             usize = HEATED_CHAIN_NUM;
    long            Gmax = 0;
    long            outfilepos = 0;
    option_fmt     *options;
    world_fmt     **universe;
#ifdef HAVE_NICE
    nice(10);  //nice value arbitrarily set to 10, 5 is still nice, 0 is standard
#endif
#ifdef MPI
    // parallel version
    MPI_Init(&argc, &argv);
    comm_world = MPI_COMM_WORLD;
    MPI_Comm_size(comm_world, &numcpu);
    MPI_Comm_rank(comm_world, &myID);
    MPI_Barrier(comm_world);
#ifdef SLOWNET
    // slow network parallel version
    profiledone = 0;
    which_calc_like(SINGLELOCUS);
#endif
#else
    //scalar version of migrate
    myID = MASTER;
    numcpu = 1;
#endif
    //random number seed global inits
    seed = malloc(sizeof(long) * 3);
    newseed = malloc(sizeof(long) * 3);
    //
    universe = (world_fmt **) calloc(HEATED_CHAIN_NUM, sizeof(world_fmt *));
    this_string = (char *) calloc(1, sizeof(char) * 256);
    strcpy(appl, "Migrate");

    /* try to catch and beautify some error messages -------------- */
    signalhandling(ON);

    /* create main data structures -------------------------------- */
    create_data(&data);
    strcpy(this_string, "earth");
    create_world(&(EARTH), this_string, 1);
    create_options(&options);

    /* parmfile and menu ------------------------------------------ */
    init_options(options);
    if(argc>1) //if user want to use a different parmfile than the default
        strncpy(options->parmfilename, argv[1], strlen(argv[1])+1);

#ifdef MPI

    if (myID == MASTER)
    {
#endif
        read_options_master(options);
        print_menu_title(stdout, options);
        get_menu(options);
#ifdef BAYESUPDATE

        if (options->bayes_infer)
            options->schains = 0;
#endif
#ifdef MPI

        broadcast_options_master(options);
    }
    else
    {
        broadcast_options_worker(options);
        options->writelog = FALSE;
        options->menu = FALSE;
    }
#endif
    /* initialization and data reading phase ---------------------- */
    usize = MAX(1, options->heated_chains);
    universe = (world_fmt **) realloc(universe, usize * sizeof(world_fmt *));
    if (myID == MASTER)
        init_files(EARTH, data, options);
    else
    {
        init_files(EARTH, data, options);
    }
    if (options->writelog)
        print_menu_title(options->logfile, options);
    EARTH->repkind = SINGLECHAIN;
    unset_penalizer_function(FALSE); // install penalizer for far jumps in MCMC
    if (!options->readsum)
    {
        //MCMC sampling
        run_sampler(options, data, universe, usize, &outfilepos, &Gmax);
    }
    else
    {
        //reanalyze sumfile
        Gmax = analyze_olddata(EARTH, options, data, &outfilepos);
    }
    //with MPI:the workers stay here much longer than the Master
    unset_penalizer_function(TRUE);
    combine_loci(type, options, EARTH, &Gmax);
    //multilocus MLE
    // gather the treedata(1) into sumfile or from worker nodes
    get_treedata(EARTH, options);
    /* printing of data ------------------------------------------- */
#ifdef MPI

    if (myID == MASTER)
    {
#endif
        print_list(universe, options, data);
        if (EARTH->options->gamma)
            print_alpha_curve(EARTH, EARTH->atl, &Gmax);
        //print_cov(EARTH, EARTH->numpop, EARTH->loci, EARTH->cov);
        fflush(EARTH->outfile);
#ifdef MPI

    }
#endif
    if (options->aic || (EARTH->options->lratio->counter > 0))
    {
#ifdef MPI
        if (myID == MASTER)
        {
#endif
            if (EARTH->options->lratio->counter > 0)
                print_lratio_test(EARTH, &Gmax);

            if (options->aic)
            {
                akaike_information(EARTH, &Gmax);
            }
#ifdef MPI

        }
#endif

    }
    if (myID == MASTER)
    {
#ifdef MPI
        get_mighistdata(EARTH, options);
#endif

        print_mighist(EARTH);
    }
    /* print UEP probabilities ------------------------------------ */
#ifdef UEP
    if (options->uep)
        analyze_uep(EARTH);
#endif
    /* profile tables --------------------------------------------- */

    unset_penalizer_function(TRUE); //now we calculate profile and shut down
    // the penalizing of far jumps in the maximizer function

#ifdef SLOWNET
    //if there are many loci but only few parameters to estimate
    // this will be still slow and therefore saturating the network
    // with parameter requests may still be the better solution
    // for some - compile using make mpi instead make mpis
    which_calc_like(PROFILE);
    if (myID == MASTER)
    {
        mpi_send_stop(EARTH);
        //stop workers
    }
#ifdef PRIVATE
    //    master_gridder(world, &Gmax);
#endif

    profile_tables(options, EARTH, &Gmax);
#else
#ifdef MPI   /* WORKS */

    if (myID == MASTER)
    {
#endif
#ifdef PRIVATE
        //       master_gridder(EARTH, &Gmax);
#endif

        profile_tables(options, EARTH, &Gmax);
#ifdef MPI

    }
#endif
#endif    /* SLOWNET */
    if (myID == MASTER)
    {
        print_finish(EARTH, outfilepos);
        /*
         * closing all files
         * ------------------------------------------
         */
#ifdef MAC

        finish_mac(options, data);
#endif

        exit_files(EARTH, data, options);
    }
#ifdef MPI
#ifndef SLOWNET
    mpi_send_stop(EARTH);
    //stop workers
    // printf("%i> at barrier in slownet-main before finalize\n", myID);
    MPI_Barrier(comm_world);
#endif

    MPI_Finalize();
#endif

    //free_universe(universe, usize);
    return 0;
}    /* main end */

/* ============================================================= */
/* multiple loci combination ---------------------------------- */
void
combine_loci(char type, option_fmt * options, world_fmt * world, long *Gmax)
{
#ifdef MPI
    long            maxreplicate;
#endif

    long            kind = MULTILOCUS;
    if (options->readsum)
    {
        if (world->loci == 1)
        {
            kind = SINGLELOCUS;
            //return;
        }
    }
#if MPI

    if ((world->loci - world->skipped > 0) || (options->readsum))
#else

    if ((world->loci - world->skipped > 1) || (options->readsum))
#endif

    {
        if (options->gamma)
            world->param0[world->numpop2] = world->options->alphavalue;
        world->repkind = set_repkind(options);
#ifdef MPI

        if (myID == MASTER)
        {
            mpi_gmax_master(world, Gmax);
#endif

            decide_plot(world->options, world->options->lchains,
                        world->options->lchains, 'l');
            estimateParameter(options->replicatenum, *Gmax, world, options,
                              world->cov[world->loci], 0 /* chain */ ,
                              type, kind, world->repkind);
#ifdef MPI

        }
        else
        {
            //printf("%i> before gmax worker\n", myID);
            mpi_gmax_worker(world);
            //printf("%i> start worker postlike calculations\n", myID);
            mpi_maximize_worker(world, options->replicatenum);
        }
#endif
    }
#ifdef MPI
    else
    {
        if (myID != MASTER)
        {
            maxreplicate = (options->replicate
                            && options->replicatenum >
                            0) ? options->replicatenum : 1;

            mpi_results_worker(MIGMPI_SUMFILE, world, maxreplicate, pack_result_buffer);
            //mpi_maximize_worker(world, options->replicatenum);
        }
    }
#endif
}

void
run_sampler(option_fmt * options, data_fmt * data, world_fmt ** universe,
            int usize, long *outfilepos, long *Gmax)
{
    long            i;
    long            maxreplicate;
    long            treefilepos;
    char           *this_string;
#ifdef MPI
    //int          *ranks;
    //MPI_Group worldgroup, locigroup;
    if (myID < data->loci + 1)
        color = 1;
    else
        color = MPI_UNDEFINED;
#endif

    this_string = (char *) calloc(1024, sizeof(char));
#ifdef MPI

    if (myID == MASTER)
    {
#endif
        get_data(data->infile, data, options);
#ifdef MPI

        printf("%i > finished data reading\n", myID);
        broadcast_data_master(data, options);
        if (numcpu == 1)
        {
            error
            ("This program was compiled to use a parallel computer\n and you tried to run it on only a single node.\nThis will not work because it uses a \n\"single_master-many_worker\" architecture \nand needs at least TWO nodes\n");
        }
    }
    else
    {
        //get_data(data->infile, data, options);
        broadcast_data_worker(data, options);
    }
    //MPI_Comm_dup(comm_world, &comm_all_worlds);
    //MPI_Comm_rank(comm_world, &myID);
#endif
    set_plot(options);
    init_world(EARTH, data, options);
#ifdef PTHREADS

    tpool_init(&heating_pool, usize, usize, 0);
#endif

    if (options->heating)
        //first in universe is the cold chain[EARTH]
        heating_init(universe, usize, data, options, this_string);
    getseed(options);
    /* calculate FST values */
    calc_simple_param(EARTH, data);
    /* report to screen */
    if (myID == MASTER)
    {
        print_menu_options(EARTH, options, data);
        if (options->progress)
            print_data_summary(stdout, EARTH, options, data);
        /* print to outfile */
        *outfilepos = print_title(EARTH, options);
        print_options(EARTH->outfile, EARTH, options, data);
        print_data_summary(EARTH->outfile, EARTH, options, data);
        print_data(EARTH, options, data);
    }
    if (options->lchains < 2 && options->replicatenum == 0)
        options->replicate = 0;
    maxreplicate = (options->replicate
                    && options->replicatenum > 0) ? options->replicatenum : 1;
#ifdef MPI

    if (myID == MASTER)
    {
        mpi_runloci_master(data->loci, EARTH->who);
    }
    else
    {
        mpi_runloci_worker(universe, usize, options, data,
                           &heating_pool, maxreplicate, &treefilepos, Gmax);
    }
#else
    run_loci(universe, usize, options, data,
             &heating_pool, maxreplicate, &treefilepos, Gmax);
#endif

#ifdef PTHREADS

    tpool_destroy(heating_pool, 1);
#endif
#ifdef MPI

    if (myID != MASTER)
    {
#endif
        if (options->heating)
        {
            for (i = 0; i < options->heated_chains; i++)
            {
                free_tree(universe[i]->root, universe[i]);
                free_timevector(universe[i]->treetimes);
            }
        }
        else
        {
            free_tree(EARTH->root, EARTH);
            free_timevector(EARTH->treetimes);
        }
#ifdef MPI

    }
#endif
    free(this_string);
}

void
run_loci(world_fmt ** universe, int usize, option_fmt * options,
         data_fmt * data, tpool_t * heating_pool, long maxreplicate,
         long *treefilepos, long *Gmax)
{
    long            locus;
    for (locus = 0; locus < data->loci; locus++)
    {
        run_locus(universe, usize, options, data,
                  heating_pool, maxreplicate, locus, treefilepos, Gmax);
    }
}

void
run_locus(world_fmt ** universe, int usize, option_fmt * options,
          data_fmt * data, tpool_t * heating_pool, long maxreplicate,
          long locus, long *treefilepos, long *Gmax)
{
    long            kind;
    long            replicate;
    char            type;
    long            runs, rep, pluschain, increment, oldsteps, chains;
    long            chain = 0, hot, warm, cold, accepted, steps, step,
                            i, ii, j, jj;
    double          acc_vh = 0.0, acc_h = 0.0, acc_w = 0.0, acc_c = 0.0;
    long            vh = 0, h = 0, w = 0;
    /* loop over independent replicates----------------------- */
    /* but make sure that we get single chain estimates, too   */
    EARTH->locus = locus;
    EARTH->repkind = SINGLECHAIN;
    for (replicate = 0; replicate < maxreplicate; replicate++)
    {
        EARTH->replicate = replicate;
        if (setup_locus(locus, EARTH, options, data) == 0)
            continue;
        if (options->heating)
            heating_prepare(universe, usize, options, data, replicate);
        type = 's';
        runs = 1;
        rep = 0;
        pluschain = options->pluschain;
        //see obscure options
        /* short and long chains ----------------------------- */
        set_bounds(&increment, &oldsteps, &chains, options, type);
        EARTH->increment = increment;
        while (runs-- >= 0)
        {
            if (myID == MASTER)
            {
                print_menu_chain(type, FIRSTCHAIN, oldsteps, EARTH,
                                 options, rep);
                if (EARTH->options->treeprint == ALL)
                    print_tree(EARTH, 0, treefilepos);
            }
            EARTH->chains = chains;
            /* loop over chains--------------------------- */
            for (chain = 0;
                    chain < chains || (type == 'l' && chain >= chains
                                       && EARTH->param_like >
                                       options->lcepsilon); chain++)
            {
                hot = warm = cold = 0;
                rep = replicate_number(options, chain, type, replicate);
                precalc_world(EARTH);
                polish_world(EARTH);
                burnin_chain(EARTH);
                EARTH->G = 0;
                accepted = 0;
                steps = oldsteps;
                EARTH->maxdatallike = EARTH->likelihood[0];
                if (options->heating)
                    heating_prepare2(universe, usize);
                if (type == 'l')
                {
                    if (chain >= chains - 1)
                    {
                        EARTH->in_last_chain = TRUE;
                        unset_penalizer_function(TRUE);
                    }
                }
                else
                {
                    EARTH->in_last_chain = FALSE;
                    unset_penalizer_function(FALSE);
                }
                /*
                 * steps for each chain
                 * ------------------------
                 */
                for (step = 0; step < steps + 1; step++)
                {
                    j = 0;
                    EARTH->increment = increment;
                    /*
                     * hop over INCREMENT trees,sample
                     * last
                     */
                    for (i = 0; i < increment; i++)
                    {
                        EARTH->actualinc = i;
                        //runclones of world at different
                        // temperatures
                        if (options->heating)
                        {
#ifdef PTHREADS
                            fill_tpool(*heating_pool, universe, usize);
                            tpool_synchronize(*heating_pool, usize);
                            acc_vh += (vh = SUN->accept);
                            acc_h += (h = MERKUR->accept);
                            acc_w += (w = VENUS->accept);
                            jj = EARTH->accept;
#else

                            for (ii = HEATED_CHAIN_NUM;
                                    ii < options->heated_chains; ii++)
                            {
#ifdef BAYESUPDATE
                                if (universe[ii]->options->bayes_infer)
                                    universe[ii]->bayesaccept = bayes_update(universe[ii]);
#endif

                                universe[ii]->accept =
                                    tree_update(universe[ii], universe[ii]->G);
                            }
#ifdef BAYESUPDATE
                            if (EARTH->options->bayes_infer)
                            {
                                SUN->bayesaccept = bayes_update(SUN);
                                MERKUR->bayesaccept = bayes_update(MERKUR);
                                VENUS->bayesaccept = bayes_update(VENUS);
                                EARTH->bayesaccept = bayes_update(EARTH);
                            }
#endif
                            acc_vh += (vh = tree_update(SUN, SUN->G));
                            acc_h += (h = tree_update(MERKUR, MERKUR->G));
                            acc_w += (w = tree_update(VENUS, VENUS->G));

                            jj = tree_update(EARTH, EARTH->G);
#endif
#ifdef UEP

                            if (options->uep && EARTH->in_last_chain)
                                update_uepanc(EARTH);
#endif

                            heated_swap(universe, EARTH->options, &acc_vh,
                                        &acc_h, &acc_w, &acc_c, &vh, &h, &w,
                                        &jj, &j, &hot, &warm, &cold);
                            if(options->adaptiveheat)
                                adjust_temperatures(universe, options->heated_chains, i/*ncrement*/+step*increment, steps*increment);

                        }
                        else
                        {
#ifdef BAYESUPDATE
                            if (EARTH->options->bayes_infer)
                                EARTH->bayesaccept = bayes_update(EARTH);
#endif

                            j += tree_update(EARTH, EARTH->G);
#ifdef UEP

                            if (options->uep && EARTH->in_last_chain)
                                update_uepanc(EARTH);
#endif

                        }
                        if (EARTH->likelihood[EARTH->G] > EARTH->maxdatallike)
                        {
                            EARTH->maxdatallike = EARTH->likelihood[EARTH->G];
                        }
                    }
                    if (options->heating)
                    {
                        advance_clone_like(VENUS, w, &VENUS->G);
                        advance_clone_like(MERKUR, h, &MERKUR->G);
                        advance_clone_like(SUN, vh, &SUN->G);
                        for (ii = HEATED_CHAIN_NUM;
                                ii < EARTH->options->heated_chains; ++ii)
                            advance_clone_like(universe[ii],
                                               universe[ii]->accept,
                                               &(universe[ii])->G);
                    }
                    /*
                     * store condensed tree information
                     * for estimation
                     */
#ifdef UEP
                    store_uep(EARTH);
#endif

                    condense_time(EARTH, &step, &j, &accepted, &EARTH->G,
                                  &steps, oldsteps, rep);
                    if (step > 0)
                        advance_clone_like(EARTH, j, &EARTH->G);
                } //end steps
                decide_plot(EARTH->options, chain, chains, type);
                memcpy(EARTH->param00, EARTH->param0,
                       sizeof(double) * EARTH->numpop2);
                memcpy(EARTH->atl[rep][locus].param0, EARTH->param00,
                       sizeof(double) * EARTH->numpop2);
                log_param0(EARTH->atl[rep][locus].param0,
                           EARTH->atl[rep][locus].lparam0, EARTH->numpop2);
                copies2lcopies(&EARTH->atl[rep][locus]);
                EARTH->repkind = SINGLECHAIN;
                //single chain, single replicate
                kind = SINGLELOCUS;
                *Gmax = (EARTH->G > *Gmax) ? EARTH->G + 1 : (*Gmax);
#ifdef BAYESUPDATE

                if (options->bayes_infer)
                    bayes_stat(EARTH, EARTH->numpop2);
                else
#endif

                    estimateParameter(rep, *Gmax, EARTH, options,
                                      EARTH->cov[locus], chain, type, kind,
                                      EARTH->repkind);
                print_progress(EARTH->options, EARTH, rep, steps * increment,
                               accepted);
                prepare_next_chain(universe, EARTH->options, type, chain,
                                   &chains, &pluschain, locus, replicate);
                print_heating_progress(universe, EARTH->options, steps,
                                       increment, &acc_vh, &acc_h, &acc_w,
                                       &acc_c, &hot, &warm, &cold);
            } //end chains
            change_chaintype(locus, &type, EARTH, &increment, &oldsteps,
                             &chains, options);
            /* evaluate multiple long chain estimates */
            if (runs < 0 && options->replicate && options->replicatenum == 0)
            {
                EARTH->repkind = MULTIPLECHAIN;
                estimateParameter(rep, *Gmax, EARTH, options,
                                  EARTH->cov[locus], chain, type,
                                  SINGLELOCUS, EARTH->repkind);
            }
        } //end runs
    } //end replicates
#ifdef UEP
    if (options->uep)
        show_uep_store(EARTH);
#endif

    if (options->replicate && options->replicatenum > 0)
    {
        EARTH->repkind = MULTIPLERUN;
#ifdef BAYESUPDATE

        if (options->bayes_infer)
        {
            bayes_stat(EARTH, EARTH->numpop2);
            return;
        }
#endif
        estimateParameter(options->replicatenum, *Gmax, EARTH, options,
                          EARTH->cov[locus], chain, type,
                          SINGLELOCUS, EARTH->repkind);
    }
}

/* save all genealogy-summaries --------------------------- */
void
get_treedata(world_fmt * world, option_fmt * options)
{
#ifdef MPI
    long            maxreplicate = (options->replicate
                                    && options->replicatenum >
                                    0) ? options->replicatenum : 1;
#endif

    if (myID == MASTER)
    {
        if (options->writesum)
        {
#ifdef MPI
            //get all sumfile data from the workers using unpack_result_buffer()
            mpi_results_master(MIGMPI_SUMFILE, world, maxreplicate, unpack_sumfile_buffer);
#endif

            write_savesum(world);
        }
#ifdef SLOWNET
        else
        {
            printf("%i> about to receive the sumfile data\n", myID);
            mpi_results_master(MIGMPI_SUMFILE, world, maxreplicate, unpack_sumfile_buffer);
            printf("%i> all sumfile data received and unpacked\n", myID);
        }
#endif

    }
}

/* get all mig-histogram data --------------------------- */
void
get_mighistdata(world_fmt * world, option_fmt * options)
{
#ifdef MPI
    long            maxreplicate = (options->replicate
                                    && options->replicatenum >
                                    0) ? options->replicatenum : 1;
#endif

    if (myID == MASTER)
    {
        if (options->mighist)
        {
#ifdef MPI
            //get all mighist data from the workers using unpack_mighist_buffer()
            mpi_results_master(MIGMPI_MIGHIST, world, maxreplicate,
                               unpack_mighist_buffer);
#endif

        }
    }
}

void
heated_swap(world_fmt ** universe, worldoption_fmt * options, double *acc_vh,
            double *acc_h, double *acc_w, double *acc_c, long *vh, long *h,
            long *w, long *jj, long *j, long *hot, long *warm, long *cold)
{
    long            sm = 0, mv = 0, vw = 0;
    long            r;
    switch (r = RANDINT(0, options->heated_chains - 2))
    {
    case 2:
        MERKUR->treeswapcount += (sm = chance_swap_tree(SUN, MERKUR));
        mv = vw = 0;
        *vh += sm;
        *h += sm;
        *acc_vh += sm;
        *acc_h += sm;
        break;
    case 1:
        VENUS->treeswapcount += (mv = chance_swap_tree(MERKUR, VENUS));
        sm = vw = 0;
        *h += mv;
        *w += mv;
        *acc_h += mv;
        *acc_w += mv;
        break;
    case 0:
        EARTH->treeswapcount += (vw = chance_swap_tree(VENUS, EARTH));
        *acc_w += vw;
        *w += vw;
        mv = sm = 0;
        break;
    default:
        universe[r]->treeswapcount += chance_swap_tree(universe[r+1], universe[r]);
        sm = vw = mv = 0;
    }
    *jj += vw;
    /*
     * if(options->verbose) { printf("   Boil%c%c%5.5f ",(int)(*vh)+' ',
     * sm==0?':':'*', SUN->likelihood[SUN->G]); printf("Hot %c%c%5.5f
     * ",(int)(*h)+' ', sm==0 && mv==0 ?':':'*',
     * MERKUR->likelihood[MERKUR->G]); printf("Warm%c%c%5.5f
     * ",(int)(*w)+' ', mv==0 && vw==0?':':'*',
     * VENUS->likelihood[VENUS->G]);
     * printf("Cold%c%c%5.5f\n",(int)(*jj)+' ', vw==0?':':'*',
     * EARTH->likelihood[EARTH->G]);
     * 
     * } */
    if (options->progress && options->heating)
    {
        *hot += sm;
        *warm += mv;
        *cold += vw;
    }
    if (*jj > 0)
    {
        (*j)++;
        (*acc_c)++;
    }
}

void
print_progress(worldoption_fmt * options, world_fmt * world, long rep,
               long visited, long accepted)
{
    if (options->progress)
    {
        print_menu_coalnodes(stdout, world, world->G + 1, rep);
        print_menu_accratio(stdout, accepted, visited);
        if (options->writelog)
        {
            print_menu_coalnodes(options->logfile, world, world->G + 1, rep);
            print_menu_accratio(options->logfile, accepted, visited);
        }
        if (!world->start && world->chains > 1 && options->verbose)
            print_gelmanr(world->gelmanmeanR, world->gelmanmaxR);
    }
}

void
change_chaintype(long locus, char *type, world_fmt * world, long *increment,
                 long *oldsteps, long *chains, option_fmt * options)
{
    if (*type == 's')
    {
        *type = 'l';
        create_treetimelist(world, &(world->treetimes), locus);
        set_bounds(increment, oldsteps, chains, options, *type);
        world->increment = *increment;
    }
}

void
prepare_next_chain(world_fmt ** universe, worldoption_fmt * options,
                   char type, long chain, long *chains, long *pluschain,
                   long locus, long replicate)
{
    long            i;
    EARTH->likelihood[0] = EARTH->likelihood[EARTH->G];
    if (options->heating)
    {
        for (i = 1; i < options->heated_chains; i++)
            universe[i]->likelihood[0] = universe[i]->likelihood[universe[i]->G];
    }
    EARTH->treetimes[0].copies = 0;
    if (type == 'l')
    {
        if (options->replicate && options->replicatenum > 0)
            EARTH->chainlikes[locus][replicate] = EARTH->param_like;
        else
            EARTH->chainlikes[locus][chain] = EARTH->param_like;
    }
    EARTH->start = FALSE;
    if (type == 'l')
    {
        if (chain < *chains + *pluschain)
        {
            if ((EARTH->param_like > options->lcepsilon)
                    || (options->gelman && EARTH->gelmanmaxR > GELMAN_MYSTIC_VALUE))
            {
                (*chains)++;
                (*pluschain)--;
            }
        }
    }
}

void
print_heating_progress(world_fmt ** universe, worldoption_fmt * options,
                       long steps, long increment, double *acc_vh,
                       double *acc_h, double *acc_w, double *acc_c,
                       long *hot, long *warm, long *cold)
{
    if (options->heating)
    {
        *acc_vh /= increment * steps;
        *acc_h /= increment * steps;
        *acc_w /= increment * steps;
        *acc_c /= increment * steps;
        if (options->progress)
            print_heating_progress2(stdout, options, universe, *acc_vh, *acc_h,
                                    *acc_w, *acc_c, *hot, *warm, *cold);
        if (options->writelog)
            print_heating_progress2(options->logfile, options, universe, *acc_vh,
                                    *acc_h, *acc_w, *acc_c, *hot, *warm, *cold);
        *hot = *warm = *cold = 0;
        *acc_vh = *acc_h = *acc_w = *acc_c = 0.0;
    }
}

void
print_heating_progress2(FILE * file, worldoption_fmt * options,
                        world_fmt ** universe, double acc_vh, double acc_h,
                        double acc_w, double acc_c, long hot, long warm,
                        long cold)
{
    fprintf(file, "           Swapping between %li temperatures.\n",
            options->heated_chains);
    if (options->heated_chains > 4)
        fprintf(file, "               Only the 4 coldest chains are shown\n");
    fprintf(file,
            "           Temperature | Accepted |  Swaps between temperatures\n");
    fprintf(file, "           %10.4f  |   %4.2f   |          |\n",
            universe[3]->averageheat, acc_vh);
    fprintf(file, "           %10.4f  |   %4.2f   |     %5li||\n",
            universe[2]->averageheat, acc_h, hot);
    fprintf(file, "           %10.4f  |   %4.2f   |     %5li ||\n",
            universe[1]->averageheat, acc_w, warm);
    fprintf(file, "           %10.4f  |   %4.2f   |     %5li  |\n", universe[0]->averageheat,
            acc_c, cold);
}

long
analyze_olddata(world_fmt * world,
                option_fmt * options, data_fmt * data, long *outfilepos)
{
#ifndef MPI
    long            locus;
#else

    long            listofthings[6];
#endif

    long            Gmax = 0;
    if (myID == MASTER)
    {
        unset_penalizer_function(TRUE);
        Gmax = read_savesum(world, options, data);
#ifdef MPI

        MPI_Bcast(data->geo, world->numpop2, MPI_DOUBLE, MASTER, comm_world);
        MPI_Bcast(data->lgeo, world->numpop2, MPI_DOUBLE, MASTER, comm_world);
    }
    else
    {
        MPI_Bcast(listofthings, 6, MPI_LONG, MASTER, comm_world);
        //      printf("listofthings received\n");
        //    fflush(stdout);
        world->loci = listofthings[0];
        world->numpop = listofthings[1];
        world->numpop2 = listofthings[2];
        options->replicate = (boolean) listofthings[3];
        options->replicatenum = listofthings[4];
        Gmax = listofthings[5];
        data->numpop = world->numpop;
        data->geo = (double *) calloc(1, sizeof(double) * world->numpop2);
        data->lgeo = (double *) calloc(1, sizeof(double) * world->numpop2);
        MPI_Bcast(data->geo, world->numpop2, MPI_DOUBLE, MASTER, comm_world);
        MPI_Bcast(data->lgeo, world->numpop2, MPI_DOUBLE, MASTER, comm_world);
        options->muloci = data->loci = world->loci;
        init_world(world, data, options);
#endif

    }
#ifdef MPI
    mpi_broadcast_results(world, world->loci,
                          pack_sumfile_buffer,
                          unpack_sumfile_buffer);
#endif

    synchronize_param(world, options);
    if (options->plot)
    {
        options->plotnow = TRUE;
        world->options->plotnow = TRUE;
    }
    world->locus = 0;
    if (myID == MASTER)
    {
        *outfilepos = print_title(world, options);
        print_options(stdout, world, options, data);
        print_options(world->outfile, world, options, data);
#ifdef MPI

        mpi_runloci_master(world->loci, world->who);
#else

        for (locus = 0; locus < world->loci; locus++)
        {
            if (options->replicate && options->replicatenum > 0)
            {
                world->locus = locus;
                world->repkind = MULTIPLERUN;
                estimateParameter(options->replicatenum, Gmax, world, options,
                                  world->cov[locus], 1, 'l',
                                  SINGLELOCUS, world->repkind);
            }
        }
#endif

    }
#ifdef MPI
    else
    {
        assignloci_worker(world);
    }
#endif
    return Gmax;
}

void
profile_tables(option_fmt * options, world_fmt * world, long *gmaxptr)
{
    long len;
#ifndef SLOWNET

    long            i;
#endif

    if (options->profile != NONE)
    {
        world->buffer =
            (char *) realloc(world->buffer, LINESIZE * 100 * sizeof(char));
        if (options->printprofsummary)
            allocate_profile_percentiles(world);
#ifdef HAVE_STRFTIME

        time(&world->starttime);
#endif

        len = world->numpop2 + (world->options->gamma ? 1 : 0);
#ifdef LONGSUM

        len += world->numpop*3;
#endif /*LONGSUM*/
#ifdef SLOWNET

        if (!options->readsum)
            mpi_broadcast_results(world, world->loci,
                                  pack_sumfile_buffer,
                                  unpack_sumfile_buffer);
        MPI_Barrier(comm_world);
        mpi_broadcast_results(world, world->loci + ((world->loci == 1) ? 0 : 1),
                              pack_result_buffer,
                              unpack_result_buffer);
        if (myID == MASTER)
        {
            print_profile_title(world);
            mpi_profiles_master(world, len, world->profilewho);
        }
        else
            mpi_profiles_worker(world, gmaxptr);
#else

        print_profile_title(world);

        for (i = 0; i < len; i++)
        {
            print_profile_likelihood_driver(i, world, gmaxptr);
            fprintf(world->outfile, "%s\n\n", world->buffer);
            world->buffer[0] = '\0';
        }
#endif
        if (myID == MASTER && options->printprofsummary)
        {
            print_profile_percentile(world);
            fprintf(world->outfile, "%s\n\n", world->buffer);
            world->buffer[0] = '\0';
            destroy_profile_percentiles(world);
        }
    }
    //free(world->buffer);
}

void
finish_mac(option_fmt * options, data_fmt * data)
{
#ifdef MAC
    fixmacfile(options->outfilename);
    if (options->plotmethod == PLOTALL)
        fixmacfile(options->mathfilename);
    if (options->treeprint)
        fixmacfile(options->treefilename);
    if (options->parmfile)
        fixmacfile("parmfile");
    if (data->sumfile)
        fixmacfile(options->sumfilename);
#endif
}

int
setup_locus(long locus, world_fmt * world, option_fmt * options,
            data_fmt * data)
{
    world->locus = locus;
    set_param(world, data, options, locus);
    print_menu_locus(world, locus);
    world->start = TRUE;
    buildtree(world, options, data, locus);
    if (world->replicate == 0)
    {
        if (strchr(SNPTYPES, options->datatype))
        {
            if (options->datatype == 'u')
                world->data->seq->endsite *= (data->seq->addon + 1);
            world->data->seq->endsite += data->seq->addon;
        }
    }
    if (!options->replicate
            || (options->replicate && options->replicatenum == 0)
            || (options->replicate && world->replicate == options->replicatenum))
    {
        //free_datapart(data, options, locus);
        if (data->skiploci[locus])
        { /* skip loci with no tips */
            //cleanup_world(world, locus);
            return 0;
        }
    }
    create_treetimelist(world, &world->treetimes, locus);
    fix_times(world, options);
    first_smooth(world, locus);
#ifdef UEP

    if (options->uep)
    {
        world->treelen = 0.0;
        if (options->ueprate > 0.0)
            calc_treelength(world->root->next->back, &world->treelen);
    }
#endif
    world->likelihood[0] = treelikelihood(world);
#ifdef UEP

    if (options->uep)
    {
        //world->ueplikelihood = ueplikelihood(world);
        //world->likelihood[0] += world->ueplikelihood;
        //printf("first: wl=%f wuk=%f\n", world->likelihood[0],
        //world->ueplikelihood);
    }
#endif
    world->allikemax = -DBL_MAX; /* for best tree option */
    return 1;
}

void
condense_time(world_fmt * world, long *step, long *j, long *accepted,
              long *G, long *steps, long oldsteps, long rep)
{

#ifdef BAYESUPDATE
    if (world->in_last_chain && world->options->bayes_infer && *accepted)
        bayes_save(world);
#endif

    world->rep = rep;
    if (*step == 0)
    {
        copy_time(world, world->treetimes, FIRSTSTEP, 0, world->numpop, rep);
        *accepted += *j;
    }
    else
    {
        copy_time(world, world->treetimes, *G, *G + (long) (*j > 0),
                  world->numpop, rep);
        if (*step < *steps)
        {
            //handled in advance_world * G += (long) (*j > 0);
            *accepted += *j;
        }
    }
    if (*step >= oldsteps - 1 && world->options->movingsteps)
    {
        if (*G + 1 < world->options->acceptfreq * oldsteps)
        {
            (*steps)++;
        }
    }
}


long
set_repkind(option_fmt * options)
{
    if (options->replicate)
    {
        if (options->replicatenum == 0)
            return MULTIPLECHAIN;
        else
            return MULTIPLERUN;
    }
    return SINGLECHAIN;
}

void
heating_init(world_fmt ** universe, int usize, data_fmt * data,
             option_fmt * options, char *this_string)
{
    long            chain;
    char            tmp[20];
    for (chain = 1; chain < usize; ++chain)
    {
        sprintf(tmp, "%10.5f", options->heat[chain]);
        strncpy(this_string, tmp, 10);
        create_world(&universe[chain], this_string, data->loci);
        init_world(universe[chain], data, options);
    }
}

void
heating_prepare(world_fmt ** universe, int usize,
                option_fmt * options, data_fmt * data, long rep)
{
    long            chain;
    EARTH->heat[0] = options->heat[0];
    if (rep == 0)
    {
        for (chain = 1; chain < usize; ++chain)
            klone(EARTH, universe[chain], options, data, options->heat[chain]);
    }
    else
    {
        for (chain = 1; chain < usize; ++chain)
            klone_part(EARTH, universe[chain], options, data, options->heat[chain]);
    }
}

void
heating_prepare2(world_fmt ** universe, int usize)
{
    long            chain;
    universe[0]->averageheat = 1.0;
    for (chain = 1; chain < usize; ++chain)
    {
        universe[chain]->G = 0;
        universe[chain]->averageheat = universe[chain]->options->adaptiveheat ?
                                       0.0 :
                                       1./universe[chain]->heat[0];
    }
    for (chain = 1; chain < usize; ++chain)
    {
        clone_polish(EARTH, universe[chain]);
#ifdef BAYESUPDATE

        if (universe[chain]->options->bayes_infer)
            universe[chain]->bayesaccept = bayes_update(universe[chain]);
#endif

        tree_update(universe[chain], 0);
    }
}

long
replicate_number(option_fmt * options, long chain, char type, long replicate)
{
    long            rep = 0;
    if (options->replicate && options->replicatenum > 0)
        rep = replicate;
    else
    {
        if (type == 'l' && options->replicate)
            rep = chain;
        else
            rep = 0;
    }
    return rep;
}
