/*------------------------------------------------------
 Maximum likelihood estimation 
 of migration rate  and effective population size
 using a Metropolis-Hastings Monte Carlo algorithm                            
 -------------------------------------------------------                        
 M C M C   R O U T I N E S 
 
 Markov Monte Carlo stuff: treechange, acceptance
 
 
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 
Copyright 2001 Peter Beerli and Joseph Felsenstein
 
 This software is distributed free of charge for non-commercial use
 and is copyrighted. Of course, we do not guarantee that the software
 works and are not responsible for any damage you may cause or have.
 
$Id: mcmc1.c,v 1.61 2002/06/20 06:25:35 beerli Exp $
-------------------------------------------------------*/
#include "migration.h"
#include "random.h"
#include "tree.h"
#include "mcmc2.h"
#ifdef UEP
#include "uep.h"
#endif

#include "bayes.h"

#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif

#define MIGRATION_AIR (boolean) 1
#define MIGRATION_IN_TREE (boolean) 0
#define NO_MIGR_NODES 0
#define WITH_MIGR_NODES 1

/* prototypes ------------------------------------------- */
// metropolize over trees
long tree_update (world_fmt * world, long g);
/* private functions */
void new_localtimelist (timelist_fmt ** ntl, timelist_fmt * otl, long numpop);
void new_proposal (proposal_fmt ** proposal, timelist_fmt * tl,
                   world_fmt * world);
void chooseOrigin (proposal_fmt * proposal);
void construct_localtimelist (timelist_fmt * timevector,
                              proposal_fmt * proposal);
void traverseAllNodes (node * theNode, node *** nodelist, long *node_elem,
                       long *oldnode_elem, int include_migration);
void add_locallineages (timelist_fmt * timevector, proposal_fmt * proposal);
void chooseTarget (proposal_fmt * proposal, timelist_fmt * timevector,
                   node ** bordernodes, long *bordernum);
void findbordernodes (node * theNode, proposal_fmt * proposal, long pop,
                      node ** bordernodes, long *bordernum, vtlist ** tyme,
                      long gte);
void free_proposal (proposal_fmt * proposal);
void free_timevector (timelist_fmt * timevector);
long remove_doublettes (timelist_fmt * timevector, node ** ptr);
long xor (node ** ptrl1, node ** ptrl2);
long rmigrcount (proposal_fmt * proposal);

int migrate_old (proposal_fmt * proposal, node * up,
                 long *old_migr_table_counter, boolean air);
int migrate (proposal_fmt * proposal, node * up);
int migrateb (proposal_fmt * proposal, node * up);

int pre_population (proposal_fmt * proposal, vtlist * ltime, long gte,
                    long *slider);

boolean acceptlike (world_fmt * world, proposal_fmt * proposal, long g,
                    timelist_fmt * tyme);
double eventtime (proposal_fmt * proposal, long pop, vtlist * tentry,
                  char *event);
node *showsister (node * theNode);
void count_migrations (node * p, long *count);
long migration_from (long to, proposal_fmt * proposal);

double prob_tree (world_fmt * world, timelist_fmt * tyme);

/* Functions ++++++++++++++++++++++++++++++++++++++++++++++++*/
void
set_tree_dirty (node * p)
{
    switch (p->type)
    {
    case 'm':
        set_dirty (p);
        set_tree_dirty (p->next->back);
        break;
    case 't':
        break;
    case 'i':
        set_dirty (p);
        set_tree_dirty (p->next->back);
        set_tree_dirty (p->next->next->back);
        break;
    case 'r':
        set_dirty (p);
        set_tree_dirty (p->next->back);
        break;
    }
}

/*=======================================================*/
long
tree_update (world_fmt * world, long g)
{    /*return 1 if tree was acceped 0 otherwise */
    static long treefilepos; /* write position in the treefile */
    boolean coalesced;
    //  boolean test;
    char event;
    long slider;
    long bordernum;
    long actualpop = -99, zz;
    double endtime, nexttime, age;
#ifdef UEP

    boolean uepsuccess = FALSE;
#endif

    proposal_fmt *proposal; /*scratchpad  in which all involved nodes
               are recorded and help arrays, e.g. migration arrays,
               are stored */
    timelist_fmt *timevector = NULL; /* local timelist */
    vtlist *tentry = NULL; /*pointer into timeslice */

    /* ---------------------------------------------------------
       initialize local timelist and construct residual timelist 
       find the start node and snip of the branch and node below */
#ifdef __MWERKS__

    eventloop ();
#endif

    new_localtimelist (&timevector, &world->treetimes[0], world->numpop);
    new_proposal (&proposal, &world->treetimes[0], world);
    chooseOrigin (proposal);
    construct_localtimelist (timevector, proposal);
    tentry = &(*timevector).tl[0];
    if (proposal->origin->tyme == 0.0)
    {
        age = 0.0;
    }
    else
    {
        age = proposal->origin->tyme;
        zz = 0;
        while (tentry->age < age && zz < (*timevector).T)
        {
            tentry = &(*timevector).tl[zz];
            zz++;
        }
    }
    nexttime = tentry->age;
    if ((*timevector).T > 1)
        endtime = (*timevector).tl[(*timevector).T - 2].age;
    else
        endtime = 0.0;
    proposal->time = age;
    coalesced = FALSE;
    /*------------------------------------
      main loop: sliding down the tree  */
    slider = 0;
    while (nexttime <= endtime)
    {
        actualpop =
            (proposal->migr_table_counter >
             0) ? proposal->migr_table[proposal->migr_table_counter -
                                       1].from : proposal->origin->pop;
        proposal->time = age + eventtime (proposal, actualpop, tentry, &event);
        //      fprintf(stderr,"age=%f event=%c\n",proposal->time,event);
        if (proposal->time < nexttime)
        {
            if (event == 'm')
            {
                if (!migrate (proposal, proposal->origin))
                {
                    free_proposal (proposal);
                    free_timevector (timevector);
                    return 0;
                }
                age = proposal->time;
                continue;
            }
            else
            {   /*coalesce */
                chooseTarget (proposal, timevector, proposal->bordernodes,
                              &bordernum);
                pretendcoalesce1p (proposal);
#ifdef UEP

                uepsuccess = is_success_pseudo_uep (proposal);
#endif

                coalesced = TRUE;
                break;
            }
        }   /*end if proposal->time < nextime */
        age = nexttime;
        tentry = &(*timevector).tl[(tentry->slice) + 1]; /*next entry in timelist */
        nexttime = tentry->age;
    }
    if (!coalesced)
    {
        if (!pre_population
                (proposal, (*timevector).tl, (*timevector).T - 1, &slider))
        {
            free_proposal (proposal);
            free_timevector (timevector);
            return 0;
        }
        pretendcoalesce1p (proposal);
#ifdef UEP

        uepsuccess = is_success_pseudo_uep (proposal);
#endif

    }
    if (
#ifdef UEP
        ((!world->options->uep && !uepsuccess)
         || (world->options->uep && uepsuccess)) &&
#endif
        (acceptlike (world, proposal, g, timevector)))
    {
        if (proposal->time > world->root->tyme)
        {   /*saveguard */
            world->root->tyme += proposal->time;
        }
        coalesce1p (proposal);
#ifdef UEP

        world->likelihood[g] = treelikelihood (world);
        if (world->options->uep)
        {
            update_uep (world->root->next->back, world);
            check_uep_root (world->root->next->back, world);
            world->treelen = 0.0;
            calc_treelength (world->root->next->back, &world->treelen);
            world->ueplikelihood = ueplikelihood (world);
            world->likelihood[g] = /*world->ueplikelihood +*/ world->likelihood[g];
        }
#else
        world->likelihood[g] = treelikelihood (world);
#endif
        /* create a new timelist */
        construct_tymelist (world, &world->treetimes[0]);
        if (world->options->treeprint != NONE)
            print_tree (world, g, &treefilepos);
        world->migration_counts = 0;
        /* report the number of migration on the tree */
        count_migrations (world->root->next->back, &world->migration_counts);
        free_proposal (proposal);
        free_timevector (timevector);
        return 1;   /* new tree accepted */
    }
    free_proposal (proposal);
    free_timevector (timevector);
    return 0;   /* not accepted */

}



/*=======================================================*/
void
new_localtimelist (timelist_fmt ** ntl, timelist_fmt * otl, long numpop)
{
    long i;
    (*ntl) = (timelist_fmt *) calloc (1, sizeof (timelist_fmt));
    (*ntl)->tl = (vtlist *) malloc ((*otl).allocT * sizeof (vtlist));
    (*ntl)->allocT = otl->allocT;
    (*ntl)->T = otl->T;
    memcpy ((*ntl)->tl, otl->tl, otl->allocT * sizeof (vtlist));
    for (i = 0; i < (*ntl)->allocT; i++)
    {
        (*ntl)->tl[i].lineages = (long *) calloc (numpop, sizeof (long));
        memcpy ((*ntl)->tl[i].lineages, otl->tl[i].lineages,
                numpop * sizeof (long));
    }
}

void
new_proposal (proposal_fmt ** proposal, timelist_fmt * tl, world_fmt * world)
{
    long j;
    long mal = world->data->maxalleles[world->locus];
    (*proposal) = (proposal_fmt *) calloc (1, sizeof (proposal_fmt));
    (*proposal)->listsize = ((*tl).allocT + (*tl).T + 5);

    (*proposal)->aboveorigin =
        (node **) calloc (1, sizeof (node *) * (*proposal)->listsize);
    (*proposal)->bordernodes =
        (node **) calloc (1, sizeof (node *) * (*proposal)->listsize);
    (*proposal)->world = world;
    (*proposal)->datatype = world->options->datatype;
    (*proposal)->sumtips = world->sumtips;
    (*proposal)->numpop = world->numpop;
    (*proposal)->endsite = world->data->seq->endsite;
    (*proposal)->fracchange = world->data->seq->fracchange;
    (*proposal)->param0 = world->param0;
    (*proposal)->root = world->root;
    (*proposal)->migration_model = world->options->migration_model;
    (*proposal)->line_f =
        (node **) calloc (1, sizeof (node *) * (*proposal)->sumtips);
    (*proposal)->line_t =
        (node **) calloc (1, sizeof (node *) * (*proposal)->sumtips);
    (*proposal)->mf = (double *) calloc ((*proposal)->endsite, sizeof (double));
    (*proposal)->mt = (double *) calloc ((*proposal)->endsite, sizeof (double));
    if (strchr (SEQUENCETYPES, (*proposal)->datatype))
    {
        (*proposal)->xf.s =
            (phenotype) malloc (world->data->seq->endsite * sizeof (ratelike *));
        (*proposal)->xt.s =
            (phenotype) malloc (world->data->seq->endsite * sizeof (ratelike *));
        for (j = 0; j < (*proposal)->endsite; j++)
        {
            (*proposal)->xf.s[j] =
                (ratelike) malloc (world->options->rcategs * sizeof (sitelike));
            (*proposal)->xt.s[j] =
                (ratelike) malloc (world->options->rcategs * sizeof (sitelike));
        }
    }
    else
    {
        (*proposal)->xf.a = (double *) calloc (1, sizeof (double) * mal);
        (*proposal)->xt.a = (double *) calloc (1, sizeof (double) * mal);
    }
    (*proposal)->old_migr_table_counter = 4 * (*proposal)->sumtips /* 100 */ ;
    (*proposal)->old_migr_table_counter2 = 4 * (*proposal)->sumtips /* 100 */ ;
    (*proposal)->migr_table =
        (migr_table_fmt *) calloc (1,
                                   sizeof (migr_table_fmt) *
                                   (*proposal)->old_migr_table_counter);
    (*proposal)->migr_table2 =
        (migr_table_fmt *) calloc ((*proposal)->old_migr_table_counter2,
                                   sizeof (migr_table_fmt));
    (*proposal)->migr_table_counter = 0;
    (*proposal)->migr_table_counter2 = 0;
    // precalculated values
    (*proposal)->mig0list = world->mig0list;
    (*proposal)->design0list = world->design0list;
#ifdef UEP

    if (world->options->uep)
    {
        (*proposal)->ueplike =
            (double **) calloc (world->data->uepsites, sizeof (double *));
        (*proposal)->ueplike[0] =
            (double *) calloc (world->numpop * world->data->uepsites,
                               sizeof (double));
        for (j = 1; j < world->data->uepsites; ++j)
            (*proposal)->ueplike[j] = (*proposal)->ueplike[0] + j * world->numpop;

        (*proposal)->ut.s = (pair *) calloc (world->data->uepsites, sizeof (pair));
        (*proposal)->uf.s = (pair *) calloc (world->data->uepsites, sizeof (pair));
        (*proposal)->umt = (double *) calloc (world->data->uepsites, sizeof (double));
        (*proposal)->umf = (double *) calloc (world->data->uepsites, sizeof (double));
    }
#endif
}


void
jumblenodes (node ** s, long n)
{
    node **temp;

    long i, rr, tn = n;

    temp = (node **) calloc (1, sizeof (node *) * n);
    memcpy (temp, s, sizeof (node *) * n);
    for (i = 0; i < n && tn > 0; i++)
    {
        s[i] = temp[rr = RANDINT (0, tn - 1)];
        temp[rr] = temp[tn - 1];
        tn--;
    }
    free (temp);
}


void
chooseOrigin (proposal_fmt * proposal)
{
    long elem = 0, oldelem = (proposal->sumtips * 2);
    node *tmp, **goal;
    goal = (node **) calloc (1, sizeof (node *) * oldelem);
    traverseAllNodes (crawlback (proposal->root->next), &goal, &elem, &oldelem,
                      NO_MIGR_NODES);
    tmp = goal[RANDINT (0, elem - 2)];
    proposal->origin = tmp;
    if (proposal->origin != showtop (crawlback (proposal->root->next)))
    {
        proposal->oback = showtop (crawlback (proposal->origin));
        proposal->osister = showsister (proposal->origin);
        if (proposal->oback != showtop (crawlback (proposal->root->next)))
        {
            proposal->ocousin = showsister (proposal->oback);
        }
        else
        {
            proposal->ocousin = NULL;
        }
    }
    if (proposal->origin == NULL)
        error ("Designation of origin for branch removal failed");
    free (goal);
}

void
construct_localtimelist (timelist_fmt * timevector, proposal_fmt * proposal)
{
    long z = 0;
    long oz = proposal->listsize;
    timevector->numpop = proposal->numpop;
    traverseAllNodes (crawlback (proposal->origin)->back,
                      &proposal->aboveorigin, &z, &oz, WITH_MIGR_NODES);
    proposal->aboveorigin[z++] = proposal->oback;
    z = remove_doublettes (timevector, proposal->aboveorigin);
    qsort ((void *) (*timevector).tl, (*timevector).T, sizeof (vtlist), agecmp);
    (*timevector).T -= z;
    if ((*timevector).tl[(*timevector).T - 1].eventnode->type != 'r')
    {
        error ("Root not at the end of timelist\n");
    }
    timeslices (&timevector);
    add_locallineages (timevector, proposal);
}

/*----------------------------------------------------------------------------
finds all nodes in a tree starting at the root node and crawling up 
to the tips in a recursive fashion, writing nodeptrs in the nodelist vector
the flag include_migration is 1 if we want to touch the migration nodes too,
otherwise =0 -> jump over the migration nodes. for convenience we define the 
the macros NO_MIGR_NODES=0 and WITH_MIGR_NODES=1 in the treesetup.h file
PB 1995
 */
void
traverseAllNodes (node * theNode, node *** nodelist, long *node_elem,
                  long *oldnode_elem, int include_migration)
{
    long elem;
    if (include_migration == NO_MIGR_NODES)
    {
        if (theNode->type != 't')
        {
            if (crawlback (theNode->next) != NULL)
                traverseAllNodes (crawlback (theNode->next), nodelist, node_elem,
                                  oldnode_elem, NO_MIGR_NODES);
            if (theNode->type != 'm' && crawlback (theNode->next->next) != NULL)
                traverseAllNodes (crawlback (theNode->next->next), nodelist,
                                  node_elem, oldnode_elem, NO_MIGR_NODES);
            if ((*node_elem) == (*oldnode_elem - 1))
            {
                elem = *oldnode_elem = ((*oldnode_elem) + (*oldnode_elem) / 2);
                (*nodelist) =
                    (node **) realloc ((*nodelist), sizeof (node *) * elem);
                memset ((*nodelist) + (*oldnode_elem), 0,
                        sizeof (node *) * (elem - (*oldnode_elem)));
                *oldnode_elem = elem;
            }
            (*nodelist)[(*node_elem)++] = theNode;
            if (theNode->type == 'm')
            {
                error ("Migration node encountered?! and died!");
            }
        }
        else
        {
            if ((*node_elem) == (*oldnode_elem - 1))
            {
                elem = *oldnode_elem = ((*oldnode_elem) + (*oldnode_elem) / 2);
                (*nodelist) =
                    (node **) realloc ((*nodelist), sizeof (node *) * elem);
                memset ((*nodelist) + (*oldnode_elem), 0,
                        sizeof (node *) * (elem - (*oldnode_elem)));
                *oldnode_elem = elem;
            }
            (*nodelist)[(*node_elem)] = theNode;
            (*node_elem) += 1;
        }
    }
    else
    {
        if (theNode->type != 't')
        {
            if (theNode->next->back != NULL)
                traverseAllNodes (theNode->next->back, nodelist, node_elem,
                                  oldnode_elem, WITH_MIGR_NODES);
            if (theNode->type != 'm' && theNode->next->next->back != NULL)
                traverseAllNodes (theNode->next->next->back, nodelist, node_elem,
                                  oldnode_elem, WITH_MIGR_NODES);
            if ((*node_elem) == (*oldnode_elem - 1))
            {
                elem = *oldnode_elem = ((*oldnode_elem) + (*oldnode_elem) / 2);
                (*nodelist) =
                    (node **) realloc ((*nodelist), sizeof (node *) * elem);
                memset ((*nodelist) + (*oldnode_elem), 0,
                        sizeof (node *) * (elem - (*oldnode_elem)));
                *oldnode_elem = elem;
            }
            (*nodelist)[(*node_elem)++] = theNode;
        }
        else
        {
            if ((*node_elem) == (*oldnode_elem - 1))
            {
                elem = *oldnode_elem = ((*oldnode_elem) + (*oldnode_elem) / 2);
                (*nodelist) =
                    (node **) realloc ((*nodelist), sizeof (node *) * elem);
                memset ((*nodelist) + (*oldnode_elem), 0,
                        sizeof (node *) * (elem - (*oldnode_elem)));
                *oldnode_elem = elem;
            }
            (*nodelist)[(*node_elem)++] = theNode;
        }
    }
}

/* prepares to remove elements of a timelist by setting eventnode to NULL
   and age to Infinity, so a simple sort will push them over the edge */
long
remove_doublettes (timelist_fmt * timevector, node ** ptr)
{
    long i = 0, j = 0, slot = 0;
    /* assumes that there is an NULL element at the end */
    for (i = 0; i < (*timevector).T; j = 0, i++)
    {
        while (((*timevector).tl[i].eventnode != ptr[j]) && (ptr[j] != NULL))
            j++;
        if (ptr[j] != NULL)
        {
            slot++;
            (*timevector).tl[i].age = DBL_MAX;
        }
    }
    return slot;
}

void
add_locallineages (timelist_fmt * timevector, proposal_fmt * proposal)
{
    long pop = -99, numpop = proposal->world->numpop;
    if (timevector->T <= 0)
        error ("Help: timelist contains 0 elements");
    for (pop = 0; pop < numpop; pop++)
        timevector->tl[timevector->T - 1].lineages[pop] = 0;
    if (timevector->T == 1)
        timevector->tl[0].lineages[proposal->osister->actualpop] += 1;
    else
        timevector->tl[timevector->T -
                       1].lineages[timevector->tl[timevector->T - 2].from] += 1;
    add_partlineages (numpop, &timevector);

}


/* replaces nodepointers in list 1 with NULL if they are present in list 2
   returns the first NULL slot in the array.
 */
long
xor (node ** ptrl1, node ** ptrl2)
{
    long i = 0, j = 0, slot = -1;
    /* assumes that there is an NULL element at the end */
    for (i = 0; ptrl1[i] != NULL; j = 0, i++)
    {
        while ((ptrl1[i] != ptrl2[j]) && (ptrl2[j] != NULL))
            j++;
        if (ptrl2[j] != NULL)
        {
            if (slot == -1)
                slot = i;
            ptrl1[i] = NULL;
        }
    }
    return slot;
}

/* migrate() fills the PROPOSAL->MIGRATION_TABLE in the tree or
   PROPOSAL->MIGRATION_TABLE2 when at the bottom of the tree
 
   PROPOSAL proposal-scratchpad
   UP       node above (younger)
   MIGR_TABLE_COUNTER migration array counter, 
   will increase by one during execution
   AIR      if true standard execution, if false updating the last 
   lineage in the residual tree.
 */
int
migrate (proposal_fmt * proposal, node * up)
{
    long i = proposal->migr_table_counter;
    migr_table_fmt *array = proposal->migr_table;
    if (i > MIGRATION_LIMIT * proposal->numpop)
    {
        return 0;
    }
    if (i > 0)
        array[i].to = array[i - 1].from;
    else
        array[i].to = up->pop;
    array[i].from = migration_from (array[i].to, proposal);
    //  printf("i: %3li %li %li\n",i,proposal->migr_table[i].from,proposal->migr_table[i].to);
    array[i++].time = proposal->time;
    if (i >= proposal->old_migr_table_counter)
    {
        proposal->old_migr_table_counter += 10;
        proposal->migr_table =
            (migr_table_fmt *) realloc (proposal->migr_table,
                                        sizeof (migr_table_fmt) *
                                        (proposal->old_migr_table_counter));
    }
    proposal->migr_table_counter = i;
    return 1;
}

int
migrateb (proposal_fmt * proposal, node * up)
{
    long i = proposal->migr_table_counter2;
    migr_table_fmt *array = proposal->migr_table2;
    if (i > MIGRATION_LIMIT * proposal->numpop)
    {
        return 0;
    }
    if (i > 0)
        array[i].to = array[i - 1].from;
    else
        array[i].to = up->pop;
    array[i].from = migration_from (array[i].to, proposal);
    //  printf("t: %3li %li %li\n",i,proposal->migr_table2[i].from,proposal->migr_table2[i].to);
    array[i].time = proposal->time;
    i++;
    if (i >= proposal->old_migr_table_counter2)
    {
        proposal->old_migr_table_counter2 += 10;
        proposal->migr_table2 =
            (migr_table_fmt *) realloc (proposal->migr_table2,
                                        sizeof (migr_table_fmt) *
                                        (proposal->old_migr_table_counter2));
    }
    proposal->migr_table_counter2 = i;
    return 1;
}

int
migrate_old (proposal_fmt * proposal, node * up, long *old_migr_table_counter,
             boolean air)
{
    migr_table_fmt *array;
    long i;
    if (air)
    {
        array = proposal->migr_table;
        i = proposal->migr_table_counter;
    }
    else
    {
        array = proposal->migr_table2;
        i = proposal->migr_table_counter2;
    }
    if (i > MIGRATION_LIMIT * proposal->numpop)
    {
        //      fprintf (stdout, "migration limit reached\n");
        return 0;
    }
    switch (proposal->migration_model)
    {
    case ISLAND:
    case ISLAND_VARTHETA:
    case MATRIX:
    case MATRIX_SAMETHETA:
    case MATRIX_ARBITRARY:
        if (i > 0)
            array[i].to = array[i - 1].from;
        else
            array[i].to = up->pop;
        array[i].from = migration_from (array[i].to, proposal);
        //      printf("i: %3li %li %li\n",i,proposal->migr_table[i].from,proposal->migr_table[i].to);
        //      printf("b: %3li %li %li\n",i,proposal->migr_table2[i].from,proposal->migr_table2[i].to);
        break;
    case CONTINUUM:
    case STEPSTONE:
        error ("not yet implemented\n");
        break;
    default:
        break;
    }
    array[i++].time = proposal->time;
    if (i >= (*old_migr_table_counter))
    {
        (*old_migr_table_counter) += 10;
        if (air)
        {
            proposal->migr_table =
                (migr_table_fmt *) realloc (proposal->migr_table,
                                            sizeof (migr_table_fmt) *
                                            (*old_migr_table_counter));
            array = proposal->migr_table;
        }
        else
        {
            proposal->migr_table2 =
                (migr_table_fmt *) realloc (proposal->migr_table2,
                                            sizeof (migr_table_fmt) *
                                            (*old_migr_table_counter));
            array = proposal->migr_table;
        }
    }
    if (air)
    {
        proposal->migr_table_counter = i;
    }
    else
    {
        proposal->migr_table_counter2 = i;
    }
    return 1;
}

/* migration_from() returns the FROM population when there was a migration
   TO        population to migrate to
   PROPOSAL  proposal-scratchpad
 */
long
migration_from_old (long to, proposal_fmt * proposal)
{
    long j, ii, msta, msto;
    double *geo = proposal->world->data->geo;
    double *r, rr = RANDUM ();
    r = (double *) calloc (1, sizeof (double) * proposal->numpop);
    msta = mstart (to, proposal->numpop);
    msto = mend (to, proposal->numpop);
    r[0] = proposal->param0[msta] * geo[msta];
    for (j = 1, ii = msta + 1; ii < msto; j++, ii++)
    {
        r[j] = r[j - 1] + geo[ii] * proposal->param0[ii];
    }
    ii = 0;
    while (rr > r[ii] / r[j - 1])
    {
        ii++;
    }
    free (r);
    if (ii < to)
        return ii;
    else
        return ++ii;
}

long
migration_from (long to, proposal_fmt * proposal)
{
    long ii = 0;
    double *r = proposal->world->migproblist[to];
    double rr = RANDUM ();
    while (rr > r[ii])
    {
        ii++;
    }
    if (ii < to)
        return ii;
    else
        return ++ii;
}

void
chooseTarget (proposal_fmt * proposal, timelist_fmt * timevector,
              node ** bordernodes, long *bordernum)
{
    long actualpop = -99;
    node *rb = crawlback (proposal->root->next);
    *bordernum = 0;
    proposal->target = NULL;
    proposal->realtarget = NULL;
    if (proposal->migr_table_counter == 0)
        actualpop = proposal->origin->pop;
    else
        actualpop = proposal->migr_table[proposal->migr_table_counter - 1].from;
    if (rb->tyme < proposal->time)
    {
        error ("Wrong Time for action in chooseTarget()\n");
    }
    findbordernodes (rb, proposal, actualpop, bordernodes, bordernum,
                     &(*timevector).tl, (*timevector).T);
    if (*bordernum > 0)
    {
        proposal->target = bordernodes[RANDINT (0, (*bordernum) - 1)];
        if (proposal->target != rb)
        {
            proposal->tsister = showsister (proposal->target);
            proposal->realtsister = crawlback (proposal->tsister)->back;
        }
        else
            proposal->tsister = NULL;
        proposal->realtarget = proposal->target;
        if (proposal->target->type == 'm')
            proposal->target = crawlback (showtop (proposal->target)->next);
    }
    else
    {
        proposal->target = NULL;
        proposal->tsister = NULL;
        proposal->realtsister = NULL;
        proposal->realtarget = NULL;
    }
}

void
findbordernodes (node * theNode, proposal_fmt * proposal, long pop,
                 node ** bordernodes, long *bordernum, vtlist ** tyme,
                 long gte)
{
    node *tmp, *back;
    if (theNode == proposal->oback)
    {
        tmp = showtop (crawlback (proposal->osister)->back);
        back = showtop (proposal->oback->back);
    }
    else
    {
        tmp = showtop (theNode);
        back = showtop (theNode->back);
    }
    if (pop == tmp->pop && pop == back->actualpop && tmp->tyme < proposal->time
            && back->tyme > proposal->time)
    {
        bordernodes[(*bordernum)++] = tmp;
        return;
    }
    else
    {
        if (back->tyme < proposal->time)
            return;
        if (tmp->type != 't')
        {
            if (tmp->next->back != NULL)
                findbordernodes (tmp->next->back, proposal, pop, bordernodes,
                                 bordernum, tyme, gte);
            if (tmp->type != 'm' && tmp->next->next->back != NULL)
                findbordernodes (tmp->next->next->back, proposal, pop,
                                 bordernodes, bordernum, tyme, gte);
        }
    }
}

/*
   boolean
   same_pop(node * up, double tyme, long pop)
   {
   node *oldnn = showtop(up->back);
   node *nn = up;
   while (nn->tyme < tyme) {
   oldnn = nn;
   nn = showtop(nn->back);
   }
   if (oldnn->pop == pop && nn->actualpop == pop)
   return TRUE;
   else
   return FALSE;
   }
 */


/* -----------------------------------------------------------------------
   simulates two lineages at once, if we are extending below the last node */
int
pre_population (proposal_fmt * proposal, vtlist * ltime, long gte,
                long *slider)
{
    boolean coalesced = FALSE;
    boolean choice = FALSE;
    long pop1 = -99, pop2 = -98;
    //  long msta1=0, msto1=0;
    //  long msta2=0, msto2=0;
    double age1, denom, rr, r0, r1, horizon, mm, mm2;
    if (gte > 0)
        proposal->realtarget = ltime[gte - 1].eventnode;
    else
        proposal->realtarget = ltime[0].eventnode->next->back; //?????gte
    if (proposal->realtarget == proposal->oback)
    {
        proposal->realtarget = crawlback (proposal->osister)->back;
    }
    if (proposal->realtarget->type == 'm')
    {
        proposal->target = crawlback (proposal->realtarget->next);
        if (proposal->target == proposal->oback)
        {
            proposal->target = proposal->osister;
        }
    }
    else
    {
        proposal->target = proposal->realtarget;
    }
    proposal->tsister = NULL;
    pop2 = proposal->realtarget->pop;
    pop1 =
        proposal->migr_table_counter >
        0 ? proposal->migr_table[proposal->migr_table_counter -
                                 1].from : proposal->origin->pop;
    age1 =
        MAX (proposal->realtarget->tyme,
             proposal->migr_table_counter >
             0 ? proposal->migr_table[proposal->migr_table_counter -
                                      1].time : proposal->origin->tyme);
    horizon = MAX (proposal->oback->tyme, age1);
    while (age1 < horizon)
    {
        mm = proposal->mig0list[pop1];
        //mm = 0.0;
        //msta1 = mstart(pop1,proposal->numpop);
        //msto1 = mend(pop1,proposal->numpop);
        //      for (i = msta1; i < msto1; i++)
        //{
        //mm += proposal->param0[i];
        //}
        if (pop1 == pop2)
        {
            denom = mm + (2. / proposal->param0[pop1]);
            proposal->time = age1 - LOG (RANDUM ()) / denom;
            age1 = proposal->time;
            if (age1 < horizon)
            {
                rr = RANDUM ();
                r0 = (2. / proposal->param0[pop1]) / denom;
                if (rr < r0)
                {
                    return 1;
                }
            }
        }
        else
        {
            denom = mm;
            proposal->time = age1 - LOG (RANDUM ()) / denom;
            age1 = proposal->time;
        }
        if (age1 < horizon)
        {
            if (!migrate (proposal, proposal->origin))
                //                      &proposal->old_migr_table_counter, MIGRATION_AIR))
            {
                return 0;
            }
            pop1 =
                proposal->migr_table_counter >
                0 ? proposal->migr_table[proposal->migr_table_counter -
                                         1].from : proposal->origin->pop;
        }
    }
    age1 = horizon;
    while (!coalesced)
    {
        //mm = mm2 = 0;
        //msta1 = mstart(pop1,proposal->numpop);
        //msto1 = mend(pop1,proposal->numpop);
        //msta2 = mstart(pop2,proposal->numpop);
        //msto2 = mend(pop2,proposal->numpop);
        //for (i = msta1; i < msto1; i++)
        //{
        //mm += proposal->param0[i];
        //}
        mm = proposal->mig0list[pop1];
        mm2 = proposal->mig0list[pop2];
        if (pop1 == pop2)
        {
            denom = 2. * mm + (2. / proposal->param0[pop1]);
            proposal->time = age1 - LOG (RANDUM ()) / denom;
            age1 = proposal->time;
            rr = RANDUM ();
            r0 = ((2. / proposal->param0[pop1]) / denom);
            r1 = r0 + mm / denom;
            if (rr < r0)
            {
                return 1;
            }
            else
            {
                if (rr < r1)
                {
                    choice = TRUE;
                }
                else
                {
                    choice = FALSE;
                }
            }
        }
        else
        {   /*pop1 not equal pop2 */
            //for (i = msta2; i < msta2; i++)
            //{
            //mm2 += proposal->param0[i];
            //}
            denom = mm + mm2;
            proposal->time = age1 - LOG (RANDUM ()) / denom;
            age1 = proposal->time;
            if (RANDUM () < (mm / denom))
            {
                choice = TRUE;
            }
            else
            {
                choice = FALSE;
            }
        }
        if (choice)
        {
            if (!migrate (proposal, proposal->origin))
                //                      &proposal->old_migr_table_counter, MIGRATION_AIR))
            {
                return 0;  /* migration limit reached */
            }
            pop1 =
                proposal->migr_table_counter >
                0 ? proposal->migr_table[proposal->migr_table_counter -
                                         1].from : proposal->origin->pop;
        }
        else
        {
            if (!migrateb (proposal, proposal->realtarget))
                //                      &proposal->old_migr_table_counter2,
                //                      MIGRATION_IN_TREE))
            {
                return 0;  /* migration limit reached */
            }
            pop2 =
                proposal->migr_table_counter2 >
                0 ? proposal->migr_table2[proposal->migr_table_counter2 -
                                          1].from : proposal->realtarget->pop;
        }
    }
    error ("Reached the end of function without coalescing");
    return -1;   /*makes the compiler happy */
}

void
free_proposal (proposal_fmt * proposal)
{
    long j;
    free (proposal->aboveorigin);
    free (proposal->bordernodes);
    free (proposal->line_f);
    free (proposal->line_t);
    free (proposal->mf);
    free (proposal->mt);
    if (strchr (SEQUENCETYPES, proposal->datatype))
    {
        for (j = 0; j < proposal->endsite; j++)
        {
            free (proposal->xt.s[j]);
            free (proposal->xf.s[j]);
        }
        free (proposal->xf.s);
        free (proposal->xt.s);
    }
    else
    {
        free (proposal->xf.a);
        free (proposal->xt.a);
    }
    free (proposal->migr_table);
    free (proposal->migr_table2);
#ifdef UEP

    if (proposal->world->options->uep)
    {
        free (proposal->ueplike[0]);
        free (proposal->ueplike);
        free (proposal->uf.s);
        free (proposal->ut.s);
        free (proposal->umf);
        free (proposal->umt);

    }
#endif
    free (proposal);
}

void
free_timevector (timelist_fmt * timevector)
{
    long i;
    for (i = 0; i < timevector->allocT; i++)
    {
        free (timevector->tl[i].lineages);
    }
    free (timevector->tl);
    free (timevector);
}

/*----------------------------------------------------------*
 * rejection/acceptance of the new tree according to the likelihood
 * and an acceptance ratio which is higher the better the
 * likelihood values are (-> Metropolis)
 */
boolean
acceptlike (world_fmt * world, proposal_fmt * proposal, long g,
            timelist_fmt * tyme)
{
#ifdef UEP
    node *first;
#endif

    boolean report = TRUE;
    long oldg = -1;
    double *zz;
    double rr, expo;
    long rm;
    long rmc = rmigrcount (proposal);
    zz = world->heat;
    if ((rm =
                proposal->migr_table_counter + proposal->migr_table_counter2 +
                world->migration_counts - rmc) > MIGRATION_LIMIT * world->numpop)
    {
        if (report || g != oldg)
        {
            warning ("migration limit (%li) exceeded: %li\n",
                     MIGRATION_LIMIT * world->numpop, rm);
            warning
            ("results may be underestimating migration rates for this chain\n");
            report = FALSE;
            oldg = g;
        }
        return FALSE;
    }
#ifdef UEP
    if (world->options->uep)
    {
        first =
            first_uep2 (proposal->world->root->next->back,
                        proposal->world->root->next->back,
                        proposal->world->data->uepsites);
        proposal->firstuep =
            first_uep (first, proposal->world->root,
                       proposal->world->data->uepsites);
        proposal->ueplikelihood = pseudo_ueplikelihood (world, proposal);
        proposal->likelihood = pseudotreelikelihood (world, proposal);
        //     printf("%s> pl=%f pul=%f\n",proposal->world->name,proposal->likelihood,proposal->ueplikelihood);
        // proposal->likelihood += proposal->ueplikelihood;
    }
    else
    {
        proposal->likelihood = pseudotreelikelihood (world, proposal);
    }
#else
    proposal->likelihood = pseudotreelikelihood (world, proposal);
#endif

    if (world->likelihood[g] < proposal->likelihood)
    {
        return TRUE;
    }
    if (!world->options->heating)
    {
        expo = proposal->likelihood - world->likelihood[g];
        rr = LOG (RANDUM ());
        if (rr < expo)
            return TRUE;
    }
    else
    {
        expo = (proposal->likelihood - world->likelihood[g]) * world->heat[0];
        rr = LOG (RANDUM ());
        if (rr < expo)
            return TRUE;
    }
    //printf("debug: always true\n");
    //return TRUE;
    return FALSE;
}

long
rmigrcount (proposal_fmt * proposal)
{
    node *p;
    long count = 0;
    for (p = proposal->origin; p != proposal->oback; p = showtop (p->back))
    {
        if (p->type == 'm')
            count++;
    }
    return count;
}

double
eventtime (proposal_fmt * proposal, long pop, vtlist * tentry, char *event)
{
    static boolean mig_force = TRUE;
    double interval, lines, denom;
    //double rate = proposal->world->options->mu_rates[proposal->world->locus];
    double mm = proposal->mig0list[pop];
    long design0 = proposal->design0list[pop];
    lines = 2.0 * tentry->lineages[pop];
    denom = mm + (lines / proposal->param0[pop]);
    interval = (-(LOG (RANDUM ())) / denom); // * rate;
    if (lines > 0)
    {
        if ((RANDUM ()) < (mm / denom))
        {
            *event = 'm';
            return interval;
        }
        else
        {
            /* if migration parameter x is 0
               then we will insert a migration in population x
               with probability so that 4Nm=0.1*theta
             */
            if ((mm <= SMALLEST_MIGRATION))
            {
                if (design0 < proposal->numpop - 1)
                {
                    if (RANDUM () < 0.1)
                    {
                        if (mig_force)
                        {
                            warning ("Migration forced\n");
                            warning
                            ("results may be overestimating migration rates\n");
                            warning ("for this chain\n");
                            mig_force = FALSE;
                        }
                        *event = 'm';
                        return interval;
                    }
                }
            }
            *event = 'c';
            return interval;
        }
    }
    else
    {
        *event = 'm';
        return interval;
    }
}

/*--------------------------------------------------------*
 * showsister() 
 * find the sisternode, by going down the branch and up on 
 * the other side again, neglecting the migration nodes.
 */
node *
showsister (node * theNode)
{
    node *tmp = crawlback (theNode);

    if (tmp->next->top)
    {
        return crawlback (tmp->next->next);
    }
    else
    {
        if (tmp->next->next->top)
        {
            return crawlback (tmp->next);
        }
        else
        {
            error ("error in treestructure, cannot find sisternode\n");
        }
    }
    return NULL;
}

void
count_migrations (node * p, long *count)
{
    if (p->type != 't')
    {
        if (p->type == 'm')
        {
            *count += 1;
            count_migrations (p->next->back, count);
        }
        else
        {
            count_migrations (p->next->back, count);
            count_migrations (p->next->next->back, count);
        }
    }
}

double
prob_tree (world_fmt * world, timelist_fmt * tyme)
{
    long j, pop;
    double mm, cc, ss = 0;

    tyme->tl[0].interval = tyme->tl[0].age;
    for (j = 1; j < tyme->T; j++)
    {
        tyme->tl[j].interval = tyme->tl[j].age - tyme->tl[j - 1].age;
    }
    for (j = 0; j < tyme->T - 1; j++)
    {
        mm = cc = 0.0;
        for (pop = 0; pop < world->numpop; pop++)
        {
            mm += tyme->tl[j].lineages[pop] * world->mig0list[pop];
            cc +=
                tyme->tl[j].lineages[pop] * (tyme->tl[j].lineages[pop] -
                                             1) / world->param0[pop];
        }
        ss += -(tyme->tl[j].interval) * (mm + cc);
        if (tyme->tl[j].from == tyme->tl[j].to)
            ss += LOG2 - LOG (world->param0[tyme->tl[j].from]);
        else
            ss += LOG (world->param0[tyme->tl[j].from]);
    }
    return ss;
}
