/*------------------------------------------------------
 Maximum likelihood estimation
 of migration rate  and effectice population size
 using a Metropolis-Hastings Monte Carlo algorithm
 -------------------------------------------------------
 T R E E B U I L D I N G   R O U T I N E S
 
 Peter Beerli 1996, Seattle
 beerli@genetics.washington.edu
 
Copyright 2001 Peter Beerli and Joseph Felsenstein
 
  This software is distributed free of charge for non-commercial use
 and is copyrighted. Of course, we do not guarantee that the software
 works and are not responsible for any damage you may cause or have.
 
 
$Id: tree.c,v 1.82 2002/12/22 04:50:47 beerli Exp $
 
-------------------------------------------------------*/


#include "migration.h"
#include "random.h"
#include "data.h"
#include "sequence.h"
#include "world.h"
#include "tools.h"
#include "migrate_mpi.h"
#ifdef UEP
#include "uep.h"
#endif
/*
 * this is defined in broyden.c, but this is the only function I need from
 * there
 */
extern double logprob_noevent (world_fmt * world, long interval);
//debug
// extern FILE *startfile;

#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif

#define NOTIPS 0
#define WITHTIPS 1



/* prototypes ------------------------------------------- */
void buildtree (world_fmt * world, option_fmt * options, data_fmt * data,
                long locus);
void create_treetimelist (world_fmt * world, timelist_fmt ** ltl, long locus);
void fix_times (world_fmt * world, option_fmt * options);
void first_smooth (world_fmt * world, long locus);
void set_dirty (node * p);
void construct_tymelist (world_fmt * world, timelist_fmt * timevector);
void timeslices (timelist_fmt ** timevector);
void add_partlineages (long numpop, timelist_fmt ** timevector);
double treelikelihood (world_fmt * world);
double pseudotreelikelihood (world_fmt * world, proposal_fmt * proposal);
double treelike_anc (world_fmt * world, long locus);
void set_pop (node * theNode, long pop, long actualpop);
void pseudonuview (proposal_fmt * proposal, xarray_fmt xx1, double *lx1,
                   double v1, xarray_fmt xx2, double *lx2, double v2);
void ltov (node * p);
void treeout (FILE * treefile, node * joint, node * p, long s);
void print_tree (world_fmt * world, long g, long *filepos);
/* private functions------------------------------------- */
void allocate_tree (world_fmt * world, option_fmt * options, data_fmt * data,
                    long locus);
/* allocations of nodes */
void allocatetips (world_fmt * world, option_fmt * options, data_fmt * data,
                   long locus);
void allocateinterior (world_fmt * world, data_fmt * data, long locus);
void allocatepoproot (world_fmt * world, data_fmt * data, long locus);
void allocate_tip (world_fmt * world, option_fmt * options, node ** p,
                   long pop, long locus, long a);
void alloc_seqx (world_fmt * world, node * theNode);
/* first tree material (upgma, distance) */
void set_tree (world_fmt * world, option_fmt * options, data_fmt * data,
               long locus);
void distance_EP (char **data, long tips, double **m);
void distance_micro (char **data, long tips, double **m);
void distance_sequence (data_fmt * data, long locus, long tips, long sites,
                        long nmlength, double **m);
void distance_allele (world_fmt * world, option_fmt * options, long locus,
                      long tips, double **distm);
void constrain_distance_zeromig (double **m, data_fmt * data, long locus,
                                 long tips, char *custm);

void makevalues (world_fmt * world, option_fmt * options, data_fmt * data,
                 long locus);
void make_alleles (world_fmt * world, option_fmt * options, data_fmt * data, long locus);
void make_microsatellites (world_fmt * world, option_fmt * options, data_fmt * data, long locus);
void make_microbrownian (world_fmt * world, option_fmt *options, data_fmt * data, long locus);
void upgma (world_fmt * world, double **x, long tips, node ** nodep);
void set_top (world_fmt * world, node * p, long pop, long locus);
void set_v (node * p);
void calc_sancost (double **cost, long numpop);

short findAllele (data_fmt * data, char s[], long locus);
void free_treetimes (world_fmt * world, long size);
void traverseNodes (node * theNode, timelist_fmt ** timevector, long *slice);
void increase_timelist (timelist_fmt ** timevector);
void allocate_lineages (vtlist * tl, long allocT, long offset, long numpop);
void add_lineages (long numpop, timelist_fmt ** timevector);
void smooth (const node * root, node * p, world_fmt * world,
             const long locus);
void which_nuview (char datatype, boolean fastlike);
void nuview_allele (node * mother, world_fmt * world, const long locus);
void nuview_micro (node * mother, world_fmt * world, const long locus);
void nuview_brownian (node * mother, world_fmt * world, const long locus);
void nuview_sequence (node * mother, world_fmt * world, const long locus);
void nuview_sequence_slow (node * mother, world_fmt * world,
                           const long locus);
void nuview_ancestral (node * mother, world_fmt * world, const long locus);
void adjustroot (node * r);
double pseudo_tl_seq (phenotype xx1, phenotype xx2, double v1, double v2,
                      proposal_fmt * proposal, world_fmt * world);

double pseudo_tl_snp (phenotype xx1, phenotype xx2, double v1, double v2,
                      proposal_fmt * proposal, world_fmt * world);

double pseudo_tl_snp_unlinked (phenotype xx1, phenotype xx2, double v1,
                               double v2, proposal_fmt * proposal,
                               world_fmt * world);
double pseudo_tl_anc (phenotype xx1, phenotype xx2, double v1, double v2,
                      proposal_fmt * proposal, world_fmt * world);
void pseudonu_allele (proposal_fmt * proposal, double **xx1, double *lx1,
                      double v1, double *xx2, double lx2, double v2);
void pseudonu_micro (proposal_fmt * proposal, double **xx1, double *lx1,
                     double v1, double *xx2, double lx2, double v2);
void pseudonu_brownian (proposal_fmt * proposal, double **xx1, double *lx1,
                        double v1, double *xx2, double lx2, double v2);
void pseudonu_seq (proposal_fmt * proposal, phenotype xxx1, double v1,
                   phenotype xxx2, double v2);
void pseudonu_seq_slow (proposal_fmt * proposal, phenotype xxx1, double *sxx1,
                        double v1, phenotype xxx2, double *sxx2, double v2);
void pseudonu_anc (proposal_fmt * proposal, phenotype xxx1, double v1,
                   phenotype xxx2, double v2);
void calculate_steps (world_fmt * world);
double logfac (long n);
inline double prob_micro (double t, long diff, world_fmt * world);

void treereader (world_fmt * world, data_fmt * data);
void length_to_times (node * p);
void treeread (FILE * file, node ** pp, node * q);
char processlength (FILE * file, node ** p);
node *allocate_nodelet (long num, char type);
void find_tips (node * p, node ** nodelist, long *z);
node *add_migration (node * p, long from, long to, double utime);
node *create_interior_node (node ** q);
node *create_root_node (node ** q);
node *create_tip_node (FILE * file, node ** q, char *ch);
char processbracket (FILE * file, node ** p);
void set_tree_pop (node * p, long *pop);
void allocate_x (node * p, world_fmt * world, char datatype,
                 boolean withtips);
long find_firstpop (node * p);

void sankoff (world_fmt * world);
double minimum (double *vec1, double *vec2, long n);
void santraverse (node * theNode, double **cost, long numpop);
long ranbest (double *array, long tie, double best, long n);
void jumble (long *s, long n);
long number_genomes (char datatype);

/* copy whole tree */
void copy_tree (world_fmt * original, world_fmt * kopie);
node *copy_node (world_fmt * original, node * o, world_fmt * kopie,
                 node * last);
void copy_node_content (world_fmt * original, world_fmt * kopie, node * o,
                        node * t);
void swap_tree (world_fmt * this, world_fmt * that);
void swap (void *a, void *b);

void free_tree (node * p, world_fmt * world);
void free_nodelet (node * p, long num, world_fmt * world);
void free_nodedata (node * p, world_fmt * world);

/* global variable NUVIEW points to function nuview_datatype() */
static void (*nuview) (node *, world_fmt *, long);


/* ======================================================= */
/*
 * Creates a start-genealogy using a coalescence approach
 *
 * - set NUVIEW according to datatype (this should go somewhere else perhaps) -
 * initializes tree structure - fills tree with data - set_tree():
 * upgma-tree, adjust for times, sankoff() for migration events
 */
void
buildtree (world_fmt * world, option_fmt * options, data_fmt * data,
           long locus)
{
    long pop;
    //  long genomes = number_genomes (options->datatype);
    if(world->replicate > 0)  // the first replicate has no tree, but the others
        // have already setup up everything, so we purge .
    {
        free_tree(world->root, world);
    }
    world->sumtips = 0;
    world->migration_counts = 0;
    for (pop = 0; pop < world->numpop; pop++)
        //    {
        //       if (world->replicate > 0)
        //          data->numind[pop][locus] /= genomes;
        world->sumtips += data->numalleles[pop][locus];
    //    }
    which_nuview (options->datatype, options->fastlike);
    switch (options->datatype)
    {
    case 's':
    case 'n':
    case 'u':
    case 'f':
        init_sequences (world, options, data, locus);
        init_sequences_aliases (world, options, data, locus);
        break;
    case 'b':
        world->data->seq->endsite = 1;
        data->freq = -10000000000000.;
        break;
    case 'm':
        world->data->seq->endsite = 1;
        /* world->data->freq = 1. / world->options->micro_stepnum; */
        break;
    case 'a':
        world->data->seq->endsite = 1;
        world->data->freq = 1. / (data->maxalleles[locus]);
        world->data->freqlast = 1. - 1. / (data->maxalleles[locus]);
    }

    if (options->usertree)
    {
        treereader (world, data);
        makevalues (world, options, data, locus);//insert values into tips
    }
    else
    {
        allocate_tree (world, options, data, locus); //allocate nodep
        makevalues (world, options, data, locus); //allocate/insert data into tips
        allocateinterior(world,data,locus); //allocate nodep guts
        allocatepoproot (world, data, locus); //allocate bottom parts
        if (world->data->skiploci[locus])
            return;
    }
    if (strchr (SEQUENCETYPES, world->options->datatype))
    {
        init_tbl (world, locus);
        if (myID == MASTER && world->replicate == 0)
        {
            print_seqfreqs (world->outfile, world, options);
            print_tbl (world->outfile, world, options, locus);
            print_weights (world->outfile, world, options, locus);
        }
        if (world->options->progress)
        {
            if (myID == MASTER && world->replicate == 0)
            {
                print_seqfreqs (stdout, world, options);
                print_tbl (stdout, world, options, locus);
                print_weights (stdout, world, options, locus);
                if (world->options->writelog)
                {
                    print_seqfreqs (world->options->logfile, world, options);
                    print_tbl (world->options->logfile, world, options, locus);
                    print_weights (world->options->logfile, world, options,
                                   locus);
                }
            }
        }
    }
    // }
    set_tree (world, options, data, locus);
    if (world->options->datatype == 'b')
        world->data->maxalleles[locus] = XBROWN_SIZE;
}

/*
 * creates the timelist which represents all time intervals on a tree. The
 * timelist is an array of pointers and not a linked list.
 *
 * - allocates memory using an arbitrary value this will be later adjusted if a
 * longer list is needed - construct
 */
void
create_treetimelist (world_fmt * world, timelist_fmt ** ltl, long locus)
{
    if ((*ltl)->tl == NULL)
    {
        (*ltl)->tl = (vtlist *) calloc (1, TIMELIST_GUESS * sizeof (vtlist));
        allocate_lineages ((*ltl)->tl, TIMELIST_GUESS, 0, world->numpop);
        (*ltl)->allocT = TIMELIST_GUESS;
    }
    (*ltl)->copies = 0;
    construct_tymelist (world, (*ltl));
}


void
allocate_lineages (vtlist * tl, long allocT, long offset, long numpop)
{
    long i;
    for (i = offset; i < allocT; i++)
    {
        tl[i].lineages = (long *) calloc (1, sizeof (long) * numpop);
    }
}

/*
 * start first pass trhough the tree to calculate the tree-likleihood
 */
void
first_smooth (world_fmt * world, long locus)
{
    smooth (world->root->next, crawlback (world->root->next), world, locus);
}

/*
 * Marks a node, so that TREELIKELIHOOD() will recalulated values in node
 */
void
set_dirty (node * p)
{
    p->dirty = TRUE;
}


/*
 * timlist constructor parts
 */
void
timeslices (timelist_fmt ** timevector)
{
    long z;
    for (z = 0; z < (*timevector)->T; z++)
    {
        (*timevector)->tl[z].from = (*timevector)->tl[z].eventnode->pop;
        (*timevector)->tl[z].to = (*timevector)->tl[z].eventnode->actualpop;
        (*timevector)->tl[z].slice = z;
        // Debug
        //	  if((*timevector)->tl[z].eventnode->tyme != (*timevector)->tl[z].age)
        //		{
        //		  FPRINTF(stderr,"ups ");
        //		}
        //	  FPRINTF(stderr,"%3li> eventtyme=%f age=%f\n",z,(*timevector)->tl[z].eventnode->tyme,(*timevector)->tl[z].age);
    }
}

void
add_partlineages (long numpop, timelist_fmt ** timevector)
{
    long i, pop;
    for (i = (*timevector)->T - 2; i >= 0; i--)
    {
        for (pop = 0; pop < numpop; pop++)
            (*timevector)->tl[i].lineages[pop] =
                (*timevector)->tl[i + 1].lineages[pop];
        if ((*timevector)->tl[i].from == (*timevector)->tl[i].to)
        {
            (*timevector)->tl[i].lineages[(*timevector)->tl[i].from] += 1;
        }
        else
        {
            (*timevector)->tl[i].lineages[(*timevector)->tl[i].from] -= 1;
            if ((*timevector)->tl[i].lineages[(*timevector)->tl[i].from] < 0)
            {
                error ("Error in add_partlineages");
            }
            (*timevector)->tl[i].lineages[(*timevector)->tl[i].to] += 1;
        }
    }
}

/*
   calculates the tree-likelihood according to datatype a, m, b, s,u
 */
double
treelikelihood (world_fmt * world)
{
    long a;
    double term = 0.0;
    node *nn = crawlback (world->root->next);
    set_dirty (nn);
    smooth (world->root->next, crawlback (world->root->next), world,
            world->locus);
    adjustroot (world->root);
    switch (world->options->datatype)
    {
    case 's':
        term = treelike_seq (world, world->locus);
        break;
    case 'n':
        term = treelike_snp (world, world->locus);
        break;
    case 'u':
        term = treelike_snp_unlinked (world, world->locus);
        break;
    case 'a':
        for (a = 0; a < world->data->maxalleles[world->locus] - 1; a++)
            term += (world->data->freq * nn->x.a[a]);
        term += (world->data->freqlast * nn->x.a[a]);
        term = (term != 0.0) ? log (term) + nn->scale[0] : -DBL_MAX;
        break;
    case 'm':
        for (a = 0; a < world->data->maxalleles[world->locus]; a++)
            term += nn->x.a[a];
        term = term != 0.0 ? log (term) + nn->scale[0] : -DBL_MAX;
        break;
    case 'b':
        term = nn->x.a[2];
        break;
    case 'f':
        term = treelike_anc (world, world->locus);
        break;
    }
#ifdef UEP
    if (world->options->uep)
        ueplikelihood (world);
#endif

    return term;
}

/*
 * calculates tree-likelihood using only arrays DOES NOT CHANGE ARRAYS IN THE
 * TREE
 */
double
pseudotreelikelihood (world_fmt * world, proposal_fmt * proposal)
{
    long a, locus = world->locus;
    /* freq is not different between pop */
    double term = 0.0;
    switch (world->options->datatype)
    {
    case 's':
        term = pseudo_tl_seq (proposal->xf.s, proposal->xt.s, proposal->v,
                              proposal->vs, proposal, world);
        break;
    case 'n':
        term = pseudo_tl_snp (proposal->xf.s, proposal->xt.s, proposal->v,
                              proposal->vs, proposal, world);
        break;
    case 'u':
        term = pseudo_tl_snp_unlinked (proposal->xf.s, proposal->xt.s,
                                       proposal->v, proposal->vs, proposal,
                                       world);
        break;
    case 'a':
        for (a = 0; a < world->data->maxalleles[locus] - 1; a++)
            term += (world->data->freq * proposal->xf.a[a]);
        term += (world->data->freqlast * proposal->xf.a[a]);
        if (term == 0.0)
            term = -DBL_MAX;
        else
            term = (log (term) + proposal->mf[0]);
        break;
    case 'b':
        term = proposal->xf.a[2];
        break;
    case 'm':
        for (a = 0; a < world->data->maxalleles[locus]; a++)
        {
            term += proposal->xf.a[a];
        }
        if (term == 0.0)
            term = -DBL_MAX;
        else
            term = (log (term) + proposal->mf[0]);
        break;
    case 'f':
        term = pseudo_tl_anc (proposal->xf.s, proposal->xt.s, proposal->v,
                              proposal->vs, proposal, world);
        break;
    default:
        term = -DBL_MAX;
    }
#ifdef UEP
    if (proposal->world->options->uep)
        term += pseudo_tl_uep (&(proposal->uf), &proposal->ut, proposal->v,
                               proposal->vs, proposal, world);
#endif

    return term;
}


/*
 * Calculates the sub-likelihoods but does not change the arrays in the tree,
 * it uses the passed arrays and overwrites the xx1 array DOES NOT CHANGE THE
 * TREE
 */
void
pseudonuview (proposal_fmt * proposal, xarray_fmt xx1, double *lx1, double v1,
              xarray_fmt xx2, double *lx2, double v2)
{
    switch (proposal->datatype)
    {
    case 'a':
        pseudonu_allele (proposal, &xx1.a, &(lx1[0]), v1, xx2.a, lx2[0], v2);
        break;
    case 'b':
        pseudonu_brownian (proposal, &xx1.a, &(lx1[0]), v1, xx2.a, lx2[0], v2);
        break;
    case 'm':
        pseudonu_micro (proposal, &xx1.a, &(lx1[0]), v1, xx2.a, lx2[0], v2);
        break;
    case 's':
    case 'n':
    case 'u':
        if (proposal->world->options->fastlike)
            pseudonu_seq (proposal, xx1.s, v1, xx2.s, v2);
        else
            pseudonu_seq_slow (proposal, xx1.s, lx1, v1, xx2.s, lx2, v2);
        break;
    case 'f':
        pseudonu_anc (proposal, xx1.s, v1, xx2.s, v2);
        break;
    }
#ifdef  UEP
    if (proposal->world->options->uep)
    {
        pseudonu_twostate (proposal, &proposal->uf, proposal->umf, v1,
                           &proposal->ut, proposal->umt, v2);
    }
#endif
}


void
pseudonu_allele (proposal_fmt * proposal, double **xx1, double *lx1,
                 double v1, double *xx2, double lx2, double v2)
{
    long a, aa, locus = proposal->world->locus; /* allele counters */
    long mal = proposal->world->data->maxalleles[locus]; /* maxalleles */
    double freq = proposal->world->data->freq;
    double freqlast = proposal->world->data->freqlast;
    double w1 = 0.0, w2 = 0.0; /* time variables */
    double pija1, pija2;  /* summary of probabilities */
    double x3m = -DBL_MAX;
    v1 = 1 - EXP (-v1);
    v2 = 1 - EXP (-v2);
    if (v1 >= 1.)
    {
        w1 = 0.0;
        v1 = 1.0;
    }
    else
    {
        w1 = 1.0 - (v1);
    }
    if (v2 >= 1.)
    {
        w2 = 0.0;
        v2 = 1.0;
    }
    else
    {
        w2 = 1.0 - v2;
    }
    for (aa = 0; aa < mal; aa++)
    {
        pija1 = pija2 = 0.0;
        for (a = 0; a < mal - 1; a++)
        {
            pija1 += ((aa == a) * w1 + v1 * freq) * (*xx1)[a];
            pija2 += ((aa == a) * w2 + v2 * freq) * xx2[a];
        }
        pija1 += ((aa == a) * w1 + v1 * freqlast) * (*xx1)[a];
        pija2 += ((aa == a) * w2 + v2 * freqlast) * xx2[a];
        (*xx1)[aa] = pija1 * pija2;
        if ((*xx1)[aa] > x3m)
            x3m = (*xx1)[aa];
    }

    for (aa = 0; aa < mal; aa++)
    {
        (*xx1)[aa] /= x3m;
    }
    *lx1 += log (x3m) + lx2;
}

void
pseudonu_micro (proposal_fmt * proposal, double **xx1, double *lx1, double v1,
                double *xx2, double lx2, double v2)
{
    long a, s, diff, locus = proposal->world->locus; /* allele counters */
    long smax = proposal->world->data->maxalleles[locus];
    long margin = proposal->world->options->micro_threshold;
    double pija1s, pija2s, vv1, vv2;
    double x3m = -DBL_MAX;
    world_fmt *world = proposal->world;
    vv1 = v1;
    vv2 = v2;
    for (s = 0; s < smax; s++)
    {
        pija1s = pija2s = 0.0;
        for (a = MAX (0, s - margin); a < s + margin && a < smax; a++)
        {
            diff = labs (s - a);
            if ((*xx1)[a] > 0)
            {
                pija1s += prob_micro (vv1, diff, world) * (*xx1)[a];
            }
            if (xx2[a] > 0)
            {
                pija2s += prob_micro (vv2, diff, world) * xx2[a];
            }
        }
        (*xx1)[s] = pija1s * pija2s;
        if ((*xx1)[s] > x3m)
            x3m = (*xx1)[s];
    }
    for (s = 0; s < smax; s++)
    {
        (*xx1)[s] /= x3m;
    }
    *lx1 += log (x3m) + lx2;
}

void
pseudonu_brownian (proposal_fmt * proposal, double **xx1, double *lx1,
                   double v1, double *xx2, double lx2, double v2)
{
    double vtot, c12;
    double mean1, mean2, mean, vv1, vv2, f1, f2, diff;
    mean1 = (*xx1)[0];
    mean2 = xx2[0];

    vv1 = v1 + (*xx1)[1];
    vv2 = v2 + xx2[1];
    vtot = vv1 + vv2;
    if (vtot > 0.0)
        f1 = vv2 / vtot;
    else
        f1 = 0.5;
    f2 = 1.0 - f1;
    mean = f1 * mean1 + f2 * mean2;
    vtot = vv1 + vv2;
    diff = mean1 - mean2;
    c12 = diff * diff / vtot;
    (*xx1)[2] =
        (*xx1)[2] + xx2[2] + MIN (0, -0.5 * (log (vtot) + c12) + LOG2PIHALF);
    (*xx1)[1] = vv1 * f1;
    (*xx1)[0] = mean;
}

/*
   adjust the variables POP and ACTUALPOP in interior nodes
 */
void
set_pop (node * theNode, long pop, long actualpop)
{
    switch (theNode->type)
    {
    case 'm':
        theNode->pop = theNode->next->pop = pop;
        theNode->actualpop = theNode->next->actualpop = actualpop;
        break;
    case 'i':
    case 'r':
        theNode->pop = theNode->next->pop = theNode->next->next->pop = pop;
        theNode->actualpop = theNode->next->actualpop = actualpop;
        theNode->next->next->actualpop = actualpop;
        break;
    case 't':
        if (theNode->pop != pop)
            error ("Population designation scrambled");
        break;
    default:
        error ("Undefined node type?!");
        break;
    }
}



/*
 * ======================================================= local functions
 */

void
allocate_tree(world_fmt * world, option_fmt * options, data_fmt * data,
              long locus)
{
    long nodenum = 0, pop, numpop = world->numpop;
    for (pop = 0; pop < numpop; pop++)
    {
        nodenum += data->numalleles[pop][locus] * 2;
    }
    world->nodep = (node **) malloc (nodenum * sizeof (node *));
}

void
allocatetips (world_fmt * world, option_fmt * options,
              data_fmt * data, long locus)
{
    long a, pop;
    long zz = 0;
    for (pop=0; pop < world->numpop; pop++)
    {
        for(a=0; a < data->numalleles[pop][locus]; a++)
        {
            allocate_tip (world, options, &world->nodep[zz], pop, locus, zz);
            zz++;
        }
    }
}

void
allocateinterior (world_fmt * world, data_fmt * data, long locus)
{
    node *p;
    long i;
    long mini = 0, maxi = 0;
    long numpop = world->numpop;
    for (i = 0; i < numpop; i++)
    {
        mini += data->numalleles[i][locus];
        maxi += data->numalleles[i][locus] *  2;
    }
    for (i = mini; i < maxi; i++)
    {
        p = allocate_nodelet (3, 'i');
        p->top = TRUE;
        p->scale =
            (double *) calloc (world->data->seq->endsite, sizeof (double));
        p->s = (double *) calloc (1, sizeof (double) * world->numpop);
        world->nodep[i] = p;
    }
}

node *
allocate_nodelet (long num, char type)
{
    static long unique_id = 0;
    boolean isfirst = TRUE;
    long j, temp;
    node *p, *q = NULL, *pfirst = NULL;
    temp = unique_id;
    for (j = 0; j < num; j++)
    {
        p = (node *) malloc (sizeof (node));
        p->tip = FALSE;
        p->number = temp;
        p->pop = -1;
        p->actualpop = -1;
        p->type = type;
        p->id = unique_id++;
        p->top = FALSE;
        p->dirty = TRUE;
        p->next = q;
        p->s = NULL;
        p->x.s = NULL;
        p->x.a = NULL;
#ifdef UEP

        p->uep = NULL;
        p->ux.s = NULL;
        p->ux.a = NULL;
#endif
        //p->scale[0] = 0.0;
        p->back = NULL;
        p->nayme = NULL;
        p->v = 0.0;
        p->tyme = 0.0;
        p->length = 0.0;
        if (isfirst)
        {
            isfirst = FALSE;
            pfirst = p;
        }
        q = p;
    }
    pfirst->next = q;
    return q;
}

void
allocatepoproot (world_fmt * world, data_fmt * data, long locus)
{
    long i;
    node *p, *q, *qq;
    long nodenum = 0;  /* arbitrarily to the first */
    q = NULL;
    p = allocate_nodelet (3, 'i');
    p->top = TRUE;
    for (i = 0; i < world->numpop; i++)
        nodenum += data->numalleles[i][locus];
    p->x.a = (double *) calloc (1, nodenum * sizeof (double));
    p->top = TRUE;
    qq = p;
    q = NULL;
    p = allocate_nodelet (3, 'r');
    p->top = TRUE;
    p->scale = (double *) calloc (world->data->seq->endsite, sizeof (double));
    p->next->back = qq;
    qq->back = p->next;
    world->root = p;
}


void
allocate_tip (world_fmt * world, option_fmt * options, node ** p, long pop,
              long locus, long a)
{

    long i;
    if(options->usertree)
        return; //do nothing because the tip is alread allocated
    (*p) = allocate_nodelet (1, 't');
    (*p)->tip = TRUE;
    (*p)->top = TRUE;
    (*p)->scale =
        (double *) calloc (world->data->seq->endsite, sizeof (double));
    (*p)->pop = (*p)->actualpop = pop;
    (*p)->s = (double *) calloc (1, sizeof (double) * world->numpop);
    for (i = 0; i < world->numpop; i++)
        (*p)->s[i] = DBL_MAX;
    (*p)->s[pop] = 0;
    if (strchr (SEQUENCETYPES, world->options->datatype))
    {

        (*p)->nayme =
            (char *) calloc (1, sizeof (char) * (options->nmlength + 1));
        alloc_seqx (world, (*p));
    }
    else
    {

        (*p)->x.a =
            (double *) calloc (1,
                               world->data->maxalleles[locus] * sizeof (double));
        (*p)->nayme =
            (char *) calloc (1, sizeof (char) * (DEFAULT_ALLELENMLENGTH + 1));
    }
#ifdef UEP
    if (world->options->uep)
    {
        (*p)->uep = (int *) calloc (world->data->uepsites, sizeof (int));
        (*p)->ux.s = (pair *) calloc (world->data->uepsites, sizeof (pair));
    }
#endif
}

void
makevalues (world_fmt * world, option_fmt * options, data_fmt * data,
            long locus)
{
    switch (world->options->datatype)
    {
    case 'a':
        make_alleles (world, options, data, locus);
        break;
    case 'b':
        make_microbrownian (world, options, data, locus);
        break;
    case 'm':
        make_microsatellites (world, options, data, locus);
        break;
    case 's':
    case 'f':
        allocatetips (world, options, data, locus);
        make_sequences (world, options, data, locus);
        break;
    case 'n':
        allocatetips (world, options, data, locus);
        make_sequences (world, options, data, locus);
        make_invarsites (world, data, locus);
        break;
    case 'u':
        allocatetips (world, options, data, locus);
        make_snp (world, options, data, locus);
        make_invarsites_unlinked (world, data, locus);
        break;
    default:
        usererror ("Oh yes, it would be nice if there were more\n \
                   possible datatypes than just an\n \
                   allele model, microsatellite model,\n sequence model, or \
                   single nucleotide polymorphism model.\n \
                   But there are currently no others, so the programs stops\n\n");
        break;
    }
#ifdef UEP
    if (world->options->uep)
    {
        make_uep_values (world, data, locus);
    }
#endif
}

/*
   creates the branchlength and adds migration nodes to the
   start tree
   - creates rough genetic distance for upgma
   - upgma
   - find branches where we need to insert migrations
   - insert migrations
   - adjust time of all nodes using the coalescent with migration
 */
void
set_tree (world_fmt * world, option_fmt * options, data_fmt * data,
          long locus)
{
    long pop, tips = world->sumtips;

    double **distm;
    node **topnodes;
    topnodes = (node **) calloc (1, sizeof (node *) * tips);
    distm = (double **) calloc (1, sizeof (double *) * tips);
    distm[0] = (double *) calloc (1, sizeof (double) * tips * tips);
    for (pop = 1; pop < tips; pop++)
        distm[pop] = distm[0] + pop * tips;
    if (!options->randomtree)
    {
        /* create a crude distance matrix according to the datatype */
        switch (world->options->datatype)
        {
        case 'a':
        case 'b':
        case 'm':
            distance_allele (world, options, locus, tips, distm);
            break;
        case 's':
        case 'n':
        case 'u':
        case 'f':
            distance_sequence (data, locus, tips,
                               world->data->seq->sites[locus],
                               options->nmlength, distm);
            break;
        }
        constrain_distance_zeromig (distm, data, locus, tips,
                                    world->options->custm);
    }
#ifdef UEP
    else
    {
        if (options->uep)
            constrain_distance_uep (data->uep, world->data->uepsites, distm,
                                    tips);
    }
#endif
    upgma (world, distm, tips, world->nodep);
    free (distm[0]);
    free (distm);
    world->root->tyme = world->root->next->tyme =
                            world->root->next->next->tyme = world->root->next->back->tyme + 10000.;
    /* orient the tree up-down, set the length and v */
    set_top (world, world->root->next->back, pop, locus);
    set_v (world->root->next->back);
    /*
     * insert migration nodes into the tree using the Slatkin and
     * Maddison approach (Fitch parsimony)
     */
    memcpy (topnodes, world->nodep, sizeof (node *) * tips);
    //zzz = 0;
    allocate_x (world->root, world, world->options->datatype, NOTIPS);
#ifdef UEP

    if (world->options->uep)
    {
        //      allocate_uep (world->root, world, world->options->datatype, NOTIPS);
        update_uep (world->root->next->back, world);
        check_uep_root (world->root->next->back, world);
    }
#endif
    sankoff (world);
    free (topnodes);
}    /* set_tree */

void
constrain_distance_zeromig (double **m, data_fmt * data, long locus,
                            long tips, char *custm)
{
    long pop, ind, j, i = 0;
    long *pops;
    pops = calloc (tips, sizeof (long));
    for (pop = 0; pop < data->numpop; pop++)
    {
        for (ind = 0; ind < data->numalleles[pop][locus]; ind++)
            pops[i++] = pop;
    }
    for (i = 0; i < tips; i++)
    {
        for (j = 0; j < i; j++)
        {
            if (custm[pops[i] + pops[i] * pops[j]] == '0')
                m[i][j] = m[j][i] = 1000;
        }
    }
    free(pops);
}

void
distance_EP (char **data, long tips, double **m)
{
    long i, j;
    for (i = 0; i < tips; i++)
    {
        for (j = 0; j < i; j++)
        {
            if (!strcmp (data[i], data[j]))
                m[i][j] = m[j][i] = fabs (rannor (1., 0.1));
            else
                m[i][j] = m[j][i] = fabs (rannor (0., 0.1));
        }
    }
}

void
distance_micro (char **data, long tips, double **m)
{
    long i, j;
    for (i = 0; i < tips; i++)
    {
        for (j = 0; j < i; j++)
        {
            m[i][j] = m[j][i] = pow (atof (data[i]) - atof (data[j]), 2.);
            m[i][j] = m[j][i] = fabs (rannor (m[i][j], 0.1));
        }
    }
}

#if(0)
/* improvementsto come @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ */

void
inittable (world_fmt * world)
{
    /* Define a lookup table. Precompute values and store in a table */
    long i;

    for (i = 0; i < categs; i++)
    {
        tbl[i].rat = rate[i];
        tbl[i].ratxv = rate[i] * xv;
    }
}    /* inittable */



void
make_ml_distance (char *x1, char *x2, long endsites, long m, long n,
                  double *v)
{
    /* compute ml distance */
    const double conda[4] =
        {
            1, 0, 0, 0
        };
    const double condc[4] =
        {
            0, 1, 0, 0
        };
    const double condg[4] =
        {
            0, 0, 1, 0
        };
    const double condt[4] =
        {
            0, 0, 0, 1
        };
    const double condo[4] =
        {
            1, 1, 1, 1
        };

    long i, j, k, l, it, numerator, denominator, num1, num2, idx;
    double summ, sum1, sum2, sumyr, lz, aa, bb, cc, vv, p1,
    p2, p3, q1, q2, q3, tt, delta, slope, xx1freqa,
    xx1freqc, xx1freqg, xx1freqt;
    double *prod, *prod2, *prod3;
    bases b;
    node *p, *q;
    sitelike xx1, xx2;

    double basefreq1[4], basefreq2[4];

    prod = (double *) malloc (sites * sizeof (double));
    prod2 = (double *) malloc (sites * sizeof (double));
    prod3 = (double *) malloc (sites * sizeof (double));
    for (i = 0; i < endsite; i++)
    {
        switch (uppercase (dat[i]))
        {
        case 'A':
            xx1 = conda;
            break;
        case 'C':
            xx1 = condc;
            break;
        case 'G':
            xx1 = condg;
            break;
        case 'T':
            xx1 = condt;
            break;
        default:
            xx1 = condo;
            break;
        }
        switch (uppercase (dat[j]))
        {
        case 'A':
            xx2 = conda;
            break;
        case 'C':
            xx2 = condc;
            break;
        case 'G':
            xx2 = condg;
            break;
        case 'T':
            xx2 = condt;
            break;
        default:
            xx2 = condo;
            break;
        }
        xx1freqa = xx1[0] * seq->freqa;
        xx1freqc = xx1[C] * seq->freqc;
        xx1freqg = xx1[G] * seq->freqg;
        xx1freqt = xx1[T] * seq->freqt;
        sum1 = xx1freqa + xx1freqc + xx1freqg + xx1freqt;
        sum2 =
            freqa * xx2[0] + freqc * xx2[C] + freqg * xx2[G] + freqt * xx2[T];
        prod[i] = sum1 * sum2;
        prod2[i] =
            (xx1freqa + xx1freqg) * (xx2[0] * freqar + xx2[G] * freqgr) +
            (xx1freqc + xx1freqt) * (xx2[C] * freqcy + xx2[T] * freqty);
        prod3[i] =
            xx1freqa * xx2[0] + xx1freqc * xx2[C] + xx1freqg * xx2[G] +
            xx1freqt * xx2[T];
    }

    tt = 0.1;
    delta = 0.1;
    it = 1;
    while (it < iterationsd && fabs (delta) > 0.00002)
    {
        slope = 0.0;
        if (tt > 0.0)
        {
            lz = -tt;
            for (i = 0; i < categs; i++)
            {
                tbl[i].z1 = EXP (tbl[i].ratxv * lz);
                tbl[i].y1 = 1.0 - tbl[i].z1;
                tbl[i].z1zz = EXP (tbl[i].rat * lz);
                tbl[i].z1yy = tbl[i].z1 - tbl[i].z1zz;
                tbl[i].z1xv = tbl[i].z1 * xv;
            }
            for (i = 0; i < endsite; i++)
            {
                idx = category[alias[i] - 1];
                cc = prod[i];
                bb = prod2[i];
                aa = prod3[i];
                slope +=
                    weightrat[i] * (tbl[idx - 1].z1zz * (bb - aa) +
                                    tbl[idx - 1].z1xv * (cc -
                                                         bb)) / (aa * tbl[idx -
                                                                          1].z1zz
                                                                 + bb * tbl[idx -
                                                                            1].
                                                                 z1yy +
                                                                 cc * tbl[idx -
                                                                          1].y1);
            }
        }
        if (slope < 0.0)
            delta = fabs (delta) / -2.0;
        else
            delta = fabs (delta);
        tt += delta;
        it++;
    }
    vv = tt * fracchange;
    free (prod);
    free (prod2);
    free (prod3);
    *v = vv;
}    /* makev */




void
makedists (world_fmt * world, option_fmt * options)
{
    /* compute distance matrix */
    long i, j;
    double v;

    inittable ();
    for (i = 0; i < endsite; i++)
        weightrat[i] = weight[i] * rate[category[alias[i] - 1] - 1];
    for (i = 0; i < world->numpop; i++)
        d[i][i] = 0.0;
    for (i = 1; i < world->numpop; i++)
    {
        for (j = i + 1; j <= spp; j++)
        {
            makev (i, j, &v);
            d[i - 1][j - 1] = v;
            d[j - 1][i - 1] = v;
        }
    }
}    /* makedists */


/* @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ */
#endif
/* calculate pairwise distances using allele disimilarity */
void
distance_allele (world_fmt * world, option_fmt * options, long locus,
                 long tips, double **distm)
{
    char **mdata;
    long pop;

    mdata = (char **) calloc (1, sizeof (char *) * (tips + 1));
    for (pop = 0; pop < tips; pop++)
    {
        mdata[pop] =
            (char *) calloc (1, sizeof (char) * options->allelenmlength);
        strcpy (mdata[pop], world->nodep[pop]->nayme);
    }
    if (world->options->datatype == 'a')
        distance_EP (mdata, tips, distm);
    else
        distance_micro (mdata, tips, distm);
    for (pop = 0; pop < tips; pop++)
    {
        free (mdata[pop]);
    }
    free (mdata);
}

/* calculate  pairwise distances using sequence similarity */
void
distance_sequence (data_fmt * data, long locus, long tips, long sites,
                   long nmlength, double **m)
{
    long i = 0, j, z, pop, ind;
    char **dat;
    if (data->distfile != NULL)
    {
        read_distance_fromfile (data->distfile, tips, nmlength, m);
    }
    else
    {
        dat = (char **) malloc (sizeof (char *) * tips);
        for (pop = 0; pop < data->numpop; pop++)
        {
            for (ind = 0; ind < data->numalleles[pop][locus]; ind++)
            {
                dat[i++] = data->yy[pop][ind][locus][0];
            }
        }
        if (i != tips)
        {
            error ("Mistake in distance_sequence() tips is not equal sum(i)\n");
        }
        for (i = 0; i < tips; i++)
        {

            for (j = i + 1; j < tips; j++)
            {
                //to come m[i][j] = m[j][i] = make_ml_distance(dat[i], dat[j], i, j);

                for (z = 0; z < sites; z++)
                {

                    if (dat[i][z] != dat[j][z])
                    {
                        //m[i][j] = m[j][i] += fabs(rannor(1.0, 0.1));
                        m[i][j] = m[j][i] += 1.0;
                    }
                }
            }
        }
        free (dat);
    }
}



void
fix_times (world_fmt * world, option_fmt * options)
{
    long k;
    double age = 0;
    if (!options->usertree)
    {
        age = 1. / logprob_noevent (world, 0);
        world->treetimes[0].tl[0].age = age;
        adjust_time (world->treetimes[0].tl[0].eventnode, age);
        for (k = 1; k < world->treetimes[0].T - 1; k++)
        {
            age += 1. / logprob_noevent (world, k);
            world->treetimes[0].tl[k].age = age;
            adjust_time (world->treetimes[0].tl[k].eventnode, age);
        }
        world->treetimes[0].tl[k].age = age + 10000.; /* this is the root */
        adjust_time (world->treetimes[0].tl[k].eventnode, age);
    }
    set_v (world->root->next->back);
}

/*
 * creates a UPGMA tree: x     = distance matrix which will be destroyed
 * through the process, tips  = # of sequences/alleles, nodep = treenodes
 * have to be allocated for ALL nodes
 *
 * This code is stripped neighbor-joining code out of phylip v3.6. Only the
 * upgma option is present.
 */
void
upgma (world_fmt * world, double **x, long tips, node ** nodep)
{
    long nc, nextnode, mini = -900, minj = -900, i, j, ia, iaa, ja, jaa, zz = 1;
    double total, tmin, bi, bj, /* ti, tj, */ da;
    double *av;
    long *oc;
    node **cluster;
    long *enterorder;

    /* First initialization */
    enterorder = (long *) calloc (tips, sizeof (long));
    for (ia = 0; ia < tips; ia++)
        enterorder[ia] = ia;
    jumble (enterorder, tips);
    nextnode = tips;
    av = (double *) calloc (1, tips * sizeof (double));
    oc = (long *) malloc (tips * 10 * sizeof (long));
    cluster = (node **) calloc (1, tips * sizeof (node *));
    for (i = 0; i < tips; i++)
        oc[i] = 1;
    for (i = 0; i < tips; i++)
        cluster[i] = nodep[i];
    /* Enter the main cycle */
    for (nc = 0; nc < tips - 1; nc++)
    {
        tmin = 99999.0;
        /* Compute sij and minimize */
        for (jaa = 1; jaa < tips; jaa++)
        {
            ja = enterorder[jaa];
            if (cluster[ja] != NULL)
            {
                for (iaa = 0; iaa < jaa; iaa++)
                {
                    ia = enterorder[iaa];
                    if (cluster[ia] != NULL)
                    {
                        total = x[ia][ja];
                        if (total < tmin)
                        {
                            tmin = total;
                            mini = ia;
                            minj = ja;
                        }
                    }
                }
            }
        }   /* compute lengths and print */
        bi = x[mini][minj] / 2.0 - av[mini];
        bj = x[mini][minj] / 2.0 - av[minj];
        av[mini] += bi;
        /* ti = (bi<1) ? log(1.- bi) : 0.0 - at[mini]; */
        /* tj = (bj<1) ? log(1.- bj) : 0.0 - at[minj]; */
        /* at[mini] -= ti; */
        nodep[nextnode]->next->back = cluster[mini];
        cluster[mini]->back = nodep[nextnode]->next;
        nodep[nextnode]->next->next->back = cluster[minj];
        cluster[minj]->back = nodep[nextnode]->next->next;
        cluster[mini]->back->v = cluster[mini]->v = bi;
        cluster[minj]->back->v = cluster[minj]->v = bj;
        /* cluster[mini]->back->tyme = cluster[mini]->tyme = ti; */
        /* cluster[minj]->back->tyme = cluster[minj]->tyme = tj; */
        cluster[mini] = nodep[nextnode];
        adjust_time (nodep[nextnode], (double) zz++);

        cluster[minj] = NULL;
        nextnode++;
        /* re-initialization */
        for (j = 0; j < tips; j++)
        {
            if (cluster[j] != NULL)
            {
                da =
                    (x[mini][j] * oc[mini] + x[minj][j] * oc[minj]) / (oc[mini] +
                            oc[minj]);
                x[mini][j] = x[j][mini] = da;
            }
        }
        for (j = 0; j < tips; j++)
        {
            x[minj][j] = x[j][minj] = 0.0;
        }
        oc[mini] += oc[minj];
    }
    /* the last cycle */
    for (i = 0; i < tips; i++)
    {
        if (cluster[i] != NULL)
            break;
    }
    world->root->next->back = cluster[i];
    cluster[i]->back = world->root->next;
    free (av);
    free (oc);
    free (cluster);
    free (enterorder);
}

void
set_top (world_fmt * world, node * p, long pop, long locus)
{


    if (p->tip)
    {
        p->top = TRUE;
        p->tyme = 0.0;
        return;
    }
    p->top = TRUE;
    p->next->top = FALSE;
    if (p->type != 'm')
    {
        p->next->next->top = FALSE;
    }
    set_top (world, p->next->back, pop, locus);
    if (p->type != 'm')
    {
        set_top (world, p->next->next->back, pop, locus);
    }
    if (p->id == crawlback (world->root->next)->id)
    {
        p->back->top = FALSE;
        p->back->tyme = ROOTLENGTH;
    }
}    /* set_top */

void
set_v (node * p)
{
    if (p->tip)
    {
        p->v = p->length = lengthof (p);
        return;
    }
    ltov (p);
    set_v (crawlback (p->next));
    set_v (crawlback (p->next->next));
}    /* set_v */

void
ltov (node * p)
{
    p->v = lengthof (p);
}    /* ltov */

/*
 * cost matrix COST is for the ISLAND and MATRIX model all 1 with a 0
 * diagonal, this will be changable through options, but perhaps this is not
 * so important
 */
void
calc_sancost (double **cost, long numpop)
{
    long i, j;
    for (i = 0; i < numpop; i++)
    {
        for (j = 0; j < numpop; j++)
        {
            if (i != j)
                cost[i][j] = 1.0;
            else
                cost[i][i] = 0.0;
        }
    }
}

void
sankoff (world_fmt * world)
{
    double **cost;
    long i;
    cost = (double **) malloc (sizeof (double *) * world->numpop);
    cost[0] =
        (double *) malloc (sizeof (double) * world->numpop * world->numpop);
    for (i = 1; i < world->numpop; i++)
    {
        cost[i] = cost[0] + world->numpop * i;
    }
    calc_sancost (cost, world->numpop);
    santraverse (crawlback (world->root->next), cost, world->numpop);
    free (cost[0]);
    free (cost);
}

void
jumble (long *s, long n)
{
    long *temp, i, rr, tn = n;

    temp = (long *) calloc (1, sizeof (long) * n);
    memcpy (temp, s, sizeof (long) * n);
    for (i = 0; i < n && tn > 0; i++)
    {
        s[i] = temp[rr = RANDINT (0, tn - 1)];
        temp[rr] = temp[tn - 1];
        tn--;
    }
    free (temp);
}

void
santraverse (node * theNode, double **cost, long numpop)
{
    long i, ii, tie, which = 0;
    node *p = NULL, *q = NULL, *tmp = NULL, *left = NULL, *right = NULL;
    double best;
    long *poplist;
    poplist = (long *) calloc (1, sizeof (long) * numpop);
    if (!theNode->tip)
    {
        if (RANDUM () > 0.5)
        {
            left = theNode->next;
            right = theNode->next->next;
        }
        else
        {
            left = theNode->next->next;
            right = theNode->next;
        }
        if (left->back != NULL)
        {
            santraverse (p = crawlback (left), cost, numpop);
        }
        if (right->back != NULL)
        {
            santraverse (q = crawlback (right), cost, numpop);
        }
        best = DBL_MAX;
        tie = 0;
        for (i = 0; i < numpop; i++)
            poplist[i] = i;
        jumble (poplist, numpop);
        for (ii = 0; ii < numpop; ii++)
        {
            i = poplist[ii];
            theNode->s[i] =
                minimum (cost[i], p->s, numpop) + minimum (cost[i], q->s, numpop);
            if (theNode->s[i] < best)
            {
                best = theNode->s[i];
                which = i;
                tie = 0;
            }
            else
            {
                if (theNode->s[i] == best)
                {
                    tie++;
                    which = i;
                }
            }
        }
        if (tie != 0)
        {
            theNode->pop = theNode->actualpop = which =
                                                    ranbest (theNode->s, tie, best, numpop);
            theNode->s[which] -= SANKOFF_DELTA;
        }
        else
        {
            theNode->pop = theNode->actualpop = which;
        }
        if (p->actualpop != which)
        {
            tmp =
                add_migration (p, theNode->actualpop, p->actualpop,
                               (theNode->tyme - p->tyme) / 2.);
            left->back = tmp;
            tmp->back = left;
            //debug FPRINTF(startfile, "%li %li\n", theNode->actualpop, p->actualpop);
        }
        if (q->actualpop != which)
        {
            tmp =
                add_migration (q, theNode->actualpop, q->actualpop,
                               (theNode->tyme - q->tyme) / 2.);
            right->back = tmp;
            tmp->back = right;
            //debug FPRINTF(startfile, "%li %li\n", theNode->actualpop, q->actualpop);
        }
    }
    free (poplist);
}

/* returns minimum for sankoff routine */
double
minimum (double *vec1, double *vec2, long n)
{
    long j;
    double summ, min = DBL_MAX;
    for (j = 0; j < n; j++)
    {
        if (vec2[j] < DBL_MAX)
        {
            if ((summ = vec1[j] + vec2[j]) < min)
                min = summ;
        }
    }
    return min;
}

long
ranbest (double *array, long tie, double best, long n)
{
    long i;
    long which = RANDINT (0, tie);
    for (i = 0; i < n; i++)
    {
        if (fabs (array[i] - best) < EPSILON)
        {
            if (which == 0)
                return i;
            else
                which--;
        }
    }
    return -2;
}


void
alloc_seqx (world_fmt * world, node * theNode)
{
    long j;
    long endsite = world->data->seq->endsite;
    if (strchr ("u", world->options->datatype))
        endsite =
            endsite * (world->data->seq->addon + 1) + world->data->seq->addon;
    if (strchr ("n", world->options->datatype))
        endsite += world->data->seq->addon;
    theNode->x.s = (phenotype) calloc (endsite, sizeof (ratelike *));
    for (j = 0; j < endsite; j++)
        theNode->x.s[j] =
            (ratelike) calloc (world->options->rcategs, sizeof (sitelike));
}

void
make_alleles (world_fmt * world, option_fmt * options, data_fmt * data, long locus)
{
    long pop, ind;
    long zz=0;
	long zpop;
    long iu;
    char a1[DEFAULT_ALLELENMLENGTH];
    char a2[DEFAULT_ALLELENMLENGTH];
    node **nodelist = world->nodep;
    for (pop = 0; pop < world->numpop; pop++)
    {
		zpop=0;
        for (ind = 0; ind < data->numind[pop][locus]; ind++)
        {
            strcpy (a1, data->yy[pop][ind][locus][0]);
            strcpy (a2, data->yy[pop][ind][locus][1]);
            if (strcmp (a1, "?"))
            {
                allocate_tip(world,options, &world->nodep[zz],pop,locus,zz);
                strcpy (nodelist[zz]->nayme, a1);
                nodelist[zz++]->x.a[findAllele (data, a1, locus)] = 1.0;
					zpop++;
            }
            else
              {
                if(options->include_unknown)
                  {
                    fprintf(stdout,"?found\n");
                    allocate_tip(world,options, &world->nodep[zz],pop,locus,zz);
                    strcpy (nodelist[zz]->nayme, a1);
                    for(iu=0;iu<data->maxalleles[locus]; iu++)
                        nodelist[zz]->x.a[iu] = 1.0;
                    zz++;
                    zpop++;
                  }                    
              }
            if (strcmp (a2, "?"))
            {
                allocate_tip(world,options, &world->nodep[zz],pop,locus,zz);
                strcpy (nodelist[zz]->nayme, a2);
                nodelist[zz++]->x.a[findAllele (data, a2,
                                                locus)] = 1.0;
				zpop++;
            }
            else
              {
                if(options->include_unknown)
                  {
                    allocate_tip(world,options, &world->nodep[zz],pop,locus,zz);
                    strcpy (nodelist[zz]->nayme, a1);
                    for(iu=0;iu<data->maxalleles[locus]; iu++)
                        nodelist[zz]->x.a[iu] = 1.0;
                    zz++;
                    zpop++;
                  }
              }
        }
		data->numalleles[pop][locus] = zpop;
    }
    world->sumtips = zz;
    if (world->sumtips==0)
    {
        world->data->skiploci[locus] = TRUE;
        world->skipped += 1;
    }
}

void
find_minmax_msat_allele (world_fmt * world, data_fmt * data, long locus,
                         long *smallest, long *biggest)
{
    long pop, ind, tmp;
    *biggest = 0;
    *smallest = LONG_MAX;
    for (pop = 0; pop < world->numpop; pop++)
    {
        for (ind = 0; ind < data->numind[pop][locus]; ind++)
        {
            if (data->yy[pop][ind][locus][0][0] != '?')
            {
                if ((tmp = atoi (data->yy[pop][ind][locus][0])) > *biggest)
                    *biggest = tmp;
                if (tmp < *smallest)
                    *smallest = tmp;
            }
            if (data->yy[pop][ind][locus][1][0] != '?')
            {
                if ((tmp = atoi (data->yy[pop][ind][locus][1])) > *biggest)
                    *biggest = tmp;
                if (tmp < *smallest)
                    *smallest = tmp;
            }
        }
    }
}

void
make_microsatellites (world_fmt * world, option_fmt *options, data_fmt * data, long locus)
{
    long pop, ind;//, tmp = 0;//, tips = 0, unknownsum = 0;
    long zz=0;
	long zpop;
    long iu;
    long smallest = 0;
    long biggest = 0;
    long smax = world->data->maxalleles[locus];
    node **nodelist = world->nodep;
    char a1[DEFAULT_ALLELENMLENGTH];
    char a2[DEFAULT_ALLELENMLENGTH];
    calculate_steps (world);
    find_minmax_msat_allele (world, data, locus, &smallest, &biggest);
    smax = biggest - smallest + 2 * world->options->micro_threshold;
    world->data->maxalleles[locus] = data->maxalleles[locus] = smax;
    //we start with the smallest - margin and go up to biggest + margin
    smallest = smallest - 1 - world->options->micro_threshold;
    if (smallest < 0)
        smallest = 0;
    for (pop = 0; pop < world->numpop; pop++)
    {
		zpop=0;
        for (ind = 0; ind < data->numind[pop][locus]; ind++)
        {
            strcpy (a1, data->yy[pop][ind][locus][0]);
            strcpy (a2, data->yy[pop][ind][locus][1]);
            if (strcmp (a1, "?"))
            {
                allocate_tip(world,options, &world->nodep[zz],pop,locus,zz);
                nodelist[zz]->x.a =
                    realloc (nodelist[zz]->x.a, sizeof (double) * smax);
                strcpy (nodelist[zz]->nayme, a1);
                nodelist[zz++]->x.a[atoi (a1) - smallest] = 1.0;
				zpop++;
            }
            else
              {
                if(options->include_unknown)
                  {
                    allocate_tip(world,options, &world->nodep[zz],pop,locus,zz);
                    strcpy (nodelist[zz]->nayme, a1);
                    for(iu=0;iu<data->maxalleles[locus]; iu++)
                        nodelist[zz]->x.a[iu] = 1.0;
                    zz++;
                    zpop++;
                  }
              }
            
            if (strcmp (a2, "?"))
            {
                allocate_tip(world,options, &world->nodep[zz],pop,locus,zz);
                nodelist[zz]->x.a =
                    realloc (nodelist[zz]->x.a,
                             sizeof (double) * smax);
                strcpy (nodelist[zz]->nayme, a2);
                nodelist[zz++]->x.a[atoi (a2) - smallest] = 1.0;
				zpop++;
            }
            else
              {
                if(options->include_unknown)
                  {
                    allocate_tip(world,options, &world->nodep[zz],pop,locus,zz);
                    strcpy (nodelist[zz]->nayme, a1);
                    for(iu=0;iu<data->maxalleles[locus]; iu++)
                        nodelist[zz]->x.a[iu] = 1.0;
                    zz++;
                    zpop++;
                  }
              }            
        }
		data->numalleles[pop][locus] = zpop;
    }
    world->sumtips = zz;
    if (world->sumtips==0)
    {
        world->data->skiploci[locus] = TRUE;
        world->skipped += 1;
    }
}

void
make_microbrownian (world_fmt * world, option_fmt * options, data_fmt * data, long locus)
{
    long pop, ind;
    char a1[DEFAULT_ALLELENMLENGTH];
    char a2[DEFAULT_ALLELENMLENGTH];
    node **nodelist = world->nodep;
    long zz=0;
    long zpop;
    long iu;
    long btotal;
    double bsum;
    for (pop = 0; pop < world->numpop; pop++)
    {
        btotal=0;
        bsum=0.;
        for (ind = 0; ind < data->numind[pop][locus]; ind++)
          {
            strcpy (a1, data->yy[pop][ind][locus][0]);
            strcpy (a2, data->yy[pop][ind][locus][1]);
            if (strcmp (a1, "?"))
              {
                btotal++;
                bsum += atof (a1);
              }
            if (strcmp (a2, "?"))
              {
                btotal++;
                bsum += atof (a2);
              }
          }
        bsum /= btotal;

        zpop = 0;
        for (ind = 0; ind < data->numind[pop][locus]; ind++)
        {
            strcpy (a1, data->yy[pop][ind][locus][0]);
            strcpy (a2, data->yy[pop][ind][locus][1]);
            if (strcmp (a1, "?"))
            {
                allocate_tip(world,options, &world->nodep[zz],pop,locus,zz);
                strcpy (nodelist[zz]->nayme, a1);
                nodelist[zz++]->x.a[0] = atof (a1);
                zpop++;
            }
            else
              {
                if(options->include_unknown)
                  {
                    allocate_tip(world,options, &world->nodep[zz],pop,locus,zz);
                    strcpy (nodelist[zz]->nayme, a1);
                    nodelist[zz++]->x.a[0] = bsum; //average over other values
                    // this might not be very sensible at all.
                    zpop++;
                  }
              }
            if (strcmp (a2, "?"))
            {
                allocate_tip(world,options, &world->nodep[zz],pop,locus,zz);
                strcpy (nodelist[zz]->nayme, a2);
                nodelist[zz++]->x.a[0] = atof (a2);
                zpop++;
            }
            else
              {
                if(options->include_unknown)
                  {
                    allocate_tip(world,options, &world->nodep[zz],pop,locus,zz);
                    strcpy (nodelist[zz]->nayme, a1);
                    nodelist[zz++]->x.a[0] = bsum;
                    zpop++;
                  }
              }
            
            data->numalleles[pop][locus] = zpop;
        }
    }
    world->sumtips = zz;
    if (world->sumtips==0)
    {
        world->data->skiploci[locus] = TRUE;
        world->skipped += 1;
    }
}


short
findAllele (data_fmt * data, char s[], long locus)
{
    short found = 0;
    while ((strcmp (s, data->allele[locus][found])
            && data->allele[locus][found][0] != '\0'))
        found++;
    return found;
}

/*---------------------------------------------
free_treetimes frees the ptr_array of timeslist
*/
void
free_treetimes (world_fmt * world, long size)
{
    while (size >= 0)
        free (world->treetimes[size--].tl);
}

void
construct_tymelist (world_fmt * world, timelist_fmt * timevector)
{
    long z = 0;
    /*
     * timevector->tl[0].age = world->root->tyme;
     * timevector->tl[0].eventnode = world->root;
     */
    timevector->numpop = world->numpop;
    traverseNodes (world->root, &timevector, &z);
    timevector->T = z;
    qsort ((void *) timevector->tl, timevector->T, sizeof (vtlist), agecmp);
    if ((*timevector).tl[(*timevector).T - 1].eventnode->type != 'r')
    {
        z = 0;
        while ((*timevector).tl[z].eventnode->type != 'r')
            z++;

        (*timevector).tl[z].eventnode->tyme =
            (*timevector).tl[z].eventnode->next->tyme =
                (*timevector).tl[z].eventnode->next->next->tyme =
                    (*timevector).tl[(*timevector).T - 1].eventnode->tyme + 10000.;
        (*timevector).tl[z].age = (*timevector).tl[z].eventnode->tyme;
        qsort ((void *) timevector->tl, timevector->T, sizeof (vtlist), agecmp);
        //printf("construct_tymelist root moved\n");
    }
    timeslices (&timevector);
    add_lineages (world->numpop, &timevector);
}

/*
 * traverse the tree and writes node-information into the real timelist also
 * takes care that the size of timelist is increased accordingly
 */
void
traverseNodes (node * theNode, timelist_fmt ** timevector, long *slice)
{
    if (theNode->type != 't')
    {
        if (theNode->next->back != NULL)
            traverseNodes (theNode->next->back, timevector, slice);
        if (theNode->type != 'm' && theNode->next->next->back != NULL)
            traverseNodes (theNode->next->next->back, timevector, slice);

        if (theNode->top)
        {
            /*
             * Here we are on the save side if we increase the
             * timelist so never a fence-write can happen
             */
            if (*slice >= (*timevector)->allocT)
            {
                increase_timelist (timevector);
            }
            /*
             * (*timevector)->tl[*slice].pop =
             * theNode->actualpop;
             */
            (*timevector)->tl[*slice].age = theNode->tyme;
            (*timevector)->tl[*slice].eventnode = theNode;
            (*slice) += 1;
        }
    }
}

void
increase_timelist (timelist_fmt ** timevector)
{
    (*timevector)->oldT = (*timevector)->allocT;
    (*timevector)->allocT += (*timevector)->allocT / 4; /* increase timelist by
                * 25% */
    (*timevector)->tl =
        (vtlist *) realloc ((*timevector)->tl,
                            sizeof (vtlist) * ((*timevector)->allocT + 1));
    memset ((*timevector)->tl + (*timevector)->oldT, 0,
            ((*timevector)->allocT - (*timevector)->oldT) * sizeof (vtlist));
    allocate_lineages ((*timevector)->tl, (*timevector)->allocT,
                       (*timevector)->oldT, (*timevector)->numpop);
}

void
add_lineages (long numpop, timelist_fmt ** timevector)
{
    long pop;
    for (pop = 0; pop < numpop; pop++)
        (*timevector)->tl[(*timevector)->T - 1].lineages[pop] = 0;

    (*timevector)->tl[(*timevector)->T -
                      1].lineages[(*timevector)->tl[(*timevector)->T -
                                                    2].from] += 1;
    add_partlineages (numpop, timevector);
}

void
smooth (const node * root, node * p, world_fmt * world, const long locus)
{
    /* static */ long panic;
    /* only changed lineages are considered */
    if (!p->dirty)
        return;

    if (p == (crawlback (root)))
        panic = 0;
    if (p->type == 'm')
        error ("MIGRATION NODE IN SMOOTH FOUND, PLEASE REPORT!!!!\n");
    if (p->type == 'i')
    {
        smooth (root, crawlback (p->next), world, locus);
        smooth (root, crawlback (p->next->next), world, locus);
        (*nuview) (p, world, locus);
#ifdef UEP

        if(world->options->uep)
            twostate_nuview (p, world, locus);
#endif

        p->dirty = FALSE;
    }
}    /* smooth */

void
which_nuview (char datatype, boolean fastlike)
{
    switch (datatype)
    {
    case 'a':
        nuview = (void (*)(node *, world_fmt *, long)) nuview_allele;
        break;
    case 'b':
        nuview = (void (*)(node *, world_fmt *, long)) nuview_brownian;
        break;
    case 'm':
        nuview = (void (*)(node *, world_fmt *, long)) nuview_micro;
        break;
    case 's':
    case 'n':
    case 'u':
        if (fastlike)
            nuview = (void (*)(node *, world_fmt *, long)) nuview_sequence;
        else
            nuview = (void (*)(node *, world_fmt *, long)) nuview_sequence_slow;
        break;
    case 'f':   /* fitch, reconstruction of ancestral state
                         * method */
        nuview = (void (*)(node *, world_fmt *, long)) nuview_ancestral;
    }
}

void
nuview_allele (node * mother, world_fmt * world, const long locus)
{
    node *d1 = NULL, *d2 = NULL;
    long a, aa;
    long mal = world->data->maxalleles[locus];
    double w1, w2, v1, v2;
    double pija1, pija2, lx1, lx2;
    double test = 0.0;
    double freq = world->data->freq;
    double freql = world->data->freqlast;
    double x3m = -DBL_MAX;
    double *xx1, *xx2;
    double *xx3;
    children (mother, &d1, &d2);
    xx1 = d1->x.a;
    xx2 = d2->x.a;
    xx3 = mother->x.a;
    lx1 = d1->scale[0];
    lx2 = d2->scale[0];
    v1 = 1 - EXP (-d1->v);
    v2 = 1 - EXP (-d2->v);
    if (v1 >= 1.)
    {
        w1 = 0.0;
        v1 = 1.0;
    }
    else
    {
        w1 = 1.0 - v1;
    }
    if (v2 >= 1.)
    {
        w2 = 0.0;
        v2 = 1.0;
    }
    else
    {
        w2 = 1.0 - v2;
    }
    for (aa = 0; aa < mal; aa++)
    {
        pija1 = pija2 = 0.0;
        for (a = 0; a < mal - 1; a++)
        {
            pija1 += ((aa == a) * w1 + v1 * freq) * xx1[a];
            pija2 += ((aa == a) * w2 + v2 * freq) * xx2[a];
        }
        pija1 += ((aa == a) * w1 + v1 * freql) * xx1[a];
        pija2 += ((aa == a) * w2 + v2 * freql) * xx2[a];
        xx3[aa] = pija1 * pija2;
        test += xx3[aa];
        if (xx3[aa] > x3m)
            x3m = xx3[aa];

    }
    if (test <= 0.0)
        error ("xx3 is 0 or garbage!");
    for (aa = 0; aa < mal; aa++)
    {
        xx3[aa] /= x3m;
    }
    mother->scale[0] = log (x3m) + lx2 + lx1;
}

void
nuview_brownian (node * mother, world_fmt * world, const long locus)
{
    node *d1 = NULL, *d2 = NULL;
    double xx1, xx2, c12, diff;
    double mean1, mean2, mean, v1, v2, vtot, f1, f2;
    children (mother, &d1, &d2);
    mean1 = d1->x.a[0];
    mean2 = d2->x.a[0];
    xx1 = d1->x.a[2];
    xx2 = d2->x.a[2];

    v1 = d1->v + d1->x.a[1]; /* di->v == length of branch time(n1) -
             * time(n2) */
    v2 = d2->v + d2->x.a[1]; /* x.a[1] contains the deltav */
    vtot = v1 + v2;
    if (vtot > 0.0)
        f1 = v2 / vtot;
    else
        f1 = 0.5;
    f2 = 1.0 - f1;
    mean = f1 * mean1 + f2 * mean2;
    diff = mean1 - mean2;
    c12 = diff * diff / vtot;
    mother->x.a[2] =
        xx1 + xx2 + MIN (0.0, -0.5 * (log (vtot) + c12) + LOG2PIHALF);
    /*
     * printf("L=%f , L1=%f, L2=%f, log(vtot=%f)=%f,
     * c12=%f\n",mother->x.a[2], xx1, xx2,vtot,log(vtot),c12);
     */
    mother->x.a[1] = v1 * f1;
    mother->x.a[0] = mean;

}


void
nuview_micro (node * mother, world_fmt * world, const long locus)
{
    node *d1 = NULL, *d2 = NULL;
    long s, a, diff;
    long margin = world->options->micro_threshold;
    double vv1, vv2, lx1, lx2;
    double x3m = -DBL_MAX;
    double pija1s, pija2s;
    double *xx1 = NULL, *xx2 = NULL;

    long smax = world->data->maxalleles[locus];
    double *xx3 = NULL;

    children (mother, &d1, &d2);
    vv1 = d1->v;
    vv2 = d2->v;
    xx1 = d1->x.a;
    xx2 = d2->x.a;
    xx3 = mother->x.a;
    lx1 = d1->scale[0];
    lx2 = d2->scale[0];
    for (s = 0; s < smax; s++)
    {
        pija1s = pija2s = 0.0;
        for (a = MAX (0, s - margin); a < s + margin && a < smax; a++)
        {
            diff = labs (s - a);
            if (xx1[a] > 0)
            {
                pija1s += prob_micro (vv1, diff, world) * xx1[a];
            }
            if (xx2[a] > 0)
            {
                pija2s += prob_micro (vv2, diff, world) * xx2[a];
            }
        }
        xx3[s] = pija1s * pija2s;
        if (xx3[s] > x3m)
            x3m = xx3[s];
    }
    if (x3m == 0.0)
    {
        mother->scale[0] = -DBL_MAX;
    }
    else
    {
        for (s = 0; s < smax; s++)
        {
            xx3[s] /= x3m;
        }
        mother->scale[0] = log (x3m) + lx1 + lx2;
    }
}


void
calculate_steps (world_fmt * world)
{
    long k, diff;
    const long stepnum = world->options->micro_threshold;
    double **steps = world->options->steps;

    for (diff = 0; diff < stepnum; diff++)
    {
        for (k = 0; k < stepnum; k++)
        {
	  steps[diff][diff] = logfac(k + diff) + logfac (k);
        }
    }
}

inline double
prob_micro (double t, long diff, world_fmt * world)
{
    double **steps = world->options->steps;
    long stepnum = world->options->micro_threshold;
    long k;
    double summ = 0.0, oldsum = 0.0;
    double logt = log (0.5 * t);
    if (diff >= stepnum)
        return summ;
    for (k = 0; k < stepnum; k++)
    {
        summ += EXP (-t + logt * (2. * k + diff) - steps[diff][k]);
        if (fabs (oldsum - summ) < DBL_EPSILON)
            break;
        oldsum = summ;
    }
//    if (summ >= 1.0)
//        printf ("step prob = %f\n", summ);
    return summ;
}

// version with steps += 2
// does it run differently than the version with steps+= 1 ???
// decide on one!!!! pb dec 06 02
inline double
alternative_prob_micro (double t, long diff, world_fmt * world)
{
    double **steps = world->options->steps;
    long stepnum = world->options->micro_threshold;
    long k;
    double summ = 0.0, oldsum = 0.0;
    double logt = log (t);
    if (diff >= stepnum)
        return summ;
    for (k = diff; k < diff + stepnum; k += 2)
    {
        summ += EXP (-t + logt * k  - steps[diff][k - diff]);
        if (fabs (oldsum - summ) < DBL_EPSILON)
            break;
        oldsum = summ;
    }
//    if (summ >= 1.0)
//        printf ("step prob = %f\n", summ);
    return summ;
}

void
nuview_sequence (node * mother, world_fmt * world, const long locus)
{
    long i, j, k;
    double lw1, lw2, yy1, yy2, ww1zz1, vv1zz1, ww2zz2, vv2zz2, vzsumr1,
    vzsumr2, vzsumy1, vzsumy2, sum1, sum2, sumr1, sumr2,
    sumy1, sumy2, ww1, ww2, zz1, zz2;
    node *q, *r;
    sitelike *xx1, *xx2, *xx3;
    long rcategs = world->options->rcategs;
    long categs = world->options->categs;
    tbl_fmt tbl = world->tbl;
    double freqa, freqc, freqg, freqt, freqr, freqy;
    seqmodel_fmt *seq;
    valrec *tbl00;
    valrec *tblij;
    valrec *tbljk;
    seq = world->data->seq;
    freqa = seq->freqa;
    freqc = seq->freqc;
    freqg = seq->freqg;
    freqt = seq->freqt;
    freqr = seq->freqr;
    freqy = seq->freqy;
    q = crawlback (mother->next);
    r = crawlback (mother->next->next);
    lw1 = -q->v / seq->fracchange;
    if (rcategs == 1 && categs == 1)
    {
        tbl00 = tbl[0][0];
        ww1 = EXP (tbl00->ratxi * lw1);
        zz1 = EXP (tbl00->ratxv * lw1);
        ww1zz1 = ww1 * zz1;
        vv1zz1 = (1.0 - ww1) * zz1;
        lw2 = -r->v / seq->fracchange;
        ww2 = EXP (tbl00->ratxi * lw2);
        zz2 = EXP (tbl00->ratxv * lw2);
        ww2zz2 = ww2 * zz2;
        vv2zz2 = (1.0 - ww2) * zz2;
        yy1 = 1.0 - zz1;
        yy2 = 1.0 - zz2;
        for (i = 0; i < seq->endsite; i++)
        {
            xx1 = &(q->x.s[i][0]);
            xx2 = &(r->x.s[i][0]);
            xx3 = &(mother->x.s[i][0]);
            sum1 =
                yy1 * (seq->freqa * (*xx1)[0] + seq->freqc * (*xx1)[1] +
                       seq->freqg * (*xx1)[2] + seq->freqt * (*xx1)[3]);
            sum2 =
                yy2 * (seq->freqa * (*xx2)[0] + seq->freqc * (*xx2)[1] +
                       seq->freqg * (*xx2)[2] + seq->freqt * (*xx2)[3]);
            sumr1 = seq->freqar * (*xx1)[0] + seq->freqgr * (*xx1)[2];
            sumr2 = seq->freqar * (*xx2)[0] + seq->freqgr * (*xx2)[2];
            sumy1 = seq->freqcy * (*xx1)[1] + seq->freqty * (*xx1)[3];
            sumy2 = seq->freqcy * (*xx2)[1] + seq->freqty * (*xx2)[3];
            vzsumr1 = vv1zz1 * sumr1;
            vzsumr2 = vv2zz2 * sumr2;
            vzsumy1 = vv1zz1 * sumy1;
            vzsumy2 = vv2zz2 * sumy2;
            (*xx3)[0] =
                (sum1 + ww1zz1 * (*xx1)[0] + vzsumr1) * (sum2 +
                        ww2zz2 * (*xx2)[0] +
                        vzsumr2);
            (*xx3)[1] =
                (sum1 + ww1zz1 * (*xx1)[1] + vzsumy1) * (sum2 +
                        ww2zz2 * (*xx2)[1] +
                        vzsumy2);
            (*xx3)[2] =
                (sum1 + ww1zz1 * (*xx1)[2] + vzsumr1) * (sum2 +
                        ww2zz2 * (*xx2)[2] +
                        vzsumr2);
            (*xx3)[3] =
                (sum1 + ww1zz1 * (*xx1)[3] + vzsumy1) * (sum2 +
                        ww2zz2 * (*xx2)[3] +
                        vzsumy2);
            //memcpy(mother->x.s[i][0], xx3[0], sizeof(sitelike));
        }
    }
    else
    {
        for (i = 0; i < rcategs; i++)
            for (j = 0; j < categs; j++)
            {
                tblij = tbl[i][j];
                tblij->ww1 = EXP (tblij->ratxi * lw1);
                tblij->zz1 = EXP (tblij->ratxv * lw1);
                tblij->ww1zz1 = tblij->ww1 * tblij->zz1;
                tblij->vv1zz1 = (1.0 - tblij->ww1) * tblij->zz1;
            }
        lw2 = -r->v / seq->fracchange;
        for (i = 0; i < rcategs; i++)
            for (j = 0; j < categs; j++)
            {
                tblij = tbl[i][j];
                tblij->ww2 = EXP (tblij->ratxi * lw2);
                tblij->zz2 = EXP (tblij->ratxv * lw2);
                tblij->ww2zz2 = tblij->ww2 * tblij->zz2;
                tblij->vv2zz2 = (1.0 - tblij->ww2) * tblij->zz2;
            }
        for (i = 0; i < seq->endsite; i++)
        {
            k = seq->category[seq->alias[i] - 1] - 1;
            for (j = 0; j < rcategs; j++)
            {
                tbljk = tbl[j][k];
                ww1zz1 = tbljk->ww1zz1;
                vv1zz1 = tbljk->vv1zz1;
                yy1 = 1.0 - tbljk->zz1;
                ww2zz2 = tbljk->ww2zz2;
                vv2zz2 = tbljk->vv2zz2;
                yy2 = 1.0 - tbljk->zz2;
                xx1 = &(q->x.s[i][j]);
                xx2 = &(r->x.s[i][j]);
                xx3 = &(mother->x.s[i][j]);
                sum1 =
                    yy1 * (seq->freqa * (*xx1)[0] + seq->freqc * (*xx1)[1] +
                           seq->freqg * (*xx1)[2] + seq->freqt * (*xx1)[3]);
                sum2 =
                    yy2 * (seq->freqa * (*xx2)[0] + seq->freqc * (*xx2)[1] +
                           seq->freqg * (*xx2)[2] + seq->freqt * (*xx2)[3]);
                sumr1 = seq->freqar * (*xx1)[0] + seq->freqgr * (*xx1)[2];
                sumr2 = seq->freqar * (*xx2)[0] + seq->freqgr * (*xx2)[2];
                sumy1 = seq->freqcy * (*xx1)[1] + seq->freqty * (*xx1)[3];
                sumy2 = seq->freqcy * (*xx2)[1] + seq->freqty * (*xx2)[3];
                vzsumr1 = vv1zz1 * sumr1;
                vzsumr2 = vv2zz2 * sumr2;
                vzsumy1 = vv1zz1 * sumy1;
                vzsumy2 = vv2zz2 * sumy2;
                (*xx3)[0] =
                    (sum1 + ww1zz1 * (*xx1)[0] + vzsumr1) * (sum2 +
                            ww2zz2 * (*xx2)[0] +
                            vzsumr2);
                (*xx3)[1] =
                    (sum1 + ww1zz1 * (*xx1)[1] + vzsumy1) * (sum2 +
                            ww2zz2 * (*xx2)[1] +
                            vzsumy2);
                (*xx3)[2] =
                    (sum1 + ww1zz1 * (*xx1)[2] + vzsumr1) * (sum2 +
                            ww2zz2 * (*xx2)[2] +
                            vzsumr2);
                (*xx3)[3] =
                    (sum1 + ww1zz1 * (*xx1)[3] + vzsumy1) * (sum2 +
                            ww2zz2 * (*xx2)[3] +
                            vzsumy2);
                //memcpy(mother->x.s[i][j], xx3[j], sizeof(sitelike));
            }
        }
    }
}    /* nuview */

void
nuview_sequence_slow (node * mother, world_fmt * world, const long locus)
{
    static long count = 0;
    long i, j, k;
    double lw1, lw2, yy1, yy2, ww1zz1, vv1zz1, ww2zz2, vv2zz2, vzsumr1,
    vzsumr2, vzsumy1, vzsumy2, sum1, sum2, sumr1, sumr2,
    sumy1, sumy2, ww1, ww2, zz1, zz2;
    double *sxx1 = NULL, *sxx2 = NULL;
    double sxx3m, tempsxx3m;
    node *q, *r;
    sitelike *xx1, *xx2, *xx3;
    long rcategs = world->options->rcategs;
    long categs = world->options->categs;
    tbl_fmt tbl = world->tbl;
    double freqa, freqc, freqg, freqt, freqr, freqy;
    seqmodel_fmt *seq;
    valrec *tbl00;
    valrec *tblij;
    valrec *tbljk;
    seq = world->data->seq;
    freqa = seq->freqa;
    freqc = seq->freqc;
    freqg = seq->freqg;
    freqt = seq->freqt;
    freqr = seq->freqr;
    freqy = seq->freqy;
    q = crawlback (mother->next);
    r = crawlback (mother->next->next);
    lw1 = -q->v / seq->fracchange;
    sxx1 = q->scale;
    sxx2 = r->scale;
    tbl00 = tbl[0][0];
    if (rcategs == 1 && categs == 1)
    {
        ww1 = EXP (tbl00->ratxi * lw1);
        zz1 = EXP (tbl00->ratxv * lw1);
        ww1zz1 = ww1 * zz1;
        vv1zz1 = (1.0 - ww1) * zz1;
        lw2 = -r->v / seq->fracchange;
        ww2 = EXP (tbl00->ratxi * lw2);
        zz2 = EXP (tbl00->ratxv * lw2);
        ww2zz2 = ww2 * zz2;
        vv2zz2 = (1.0 - ww2) * zz2;
        yy1 = 1.0 - zz1;
        yy2 = 1.0 - zz2;
        for (i = 0; i < seq->endsite; i++)
        {
            xx1 = &(q->x.s[i][0]);
            xx2 = &(r->x.s[i][0]);
            xx3 = &(mother->x.s[i][0]);
            sum1 =
                yy1 * (freqa * (*xx1)[0] + freqc * (*xx1)[1] + freqg * (*xx1)[2] +
                       freqt * (*xx1)[3]);
            sum2 =
                yy2 * (freqa * (*xx2)[0] + freqc * (*xx2)[1] + freqg * (*xx2)[2] +
                       freqt * (*xx2)[3]);
            sumr1 = seq->freqar * (*xx1)[0] + seq->freqgr * (*xx1)[2];
            sumr2 = seq->freqar * (*xx2)[0] + seq->freqgr * (*xx2)[2];
            sumy1 = seq->freqcy * (*xx1)[1] + seq->freqty * (*xx1)[3];
            sumy2 = seq->freqcy * (*xx2)[1] + seq->freqty * (*xx2)[3];
            vzsumr1 = vv1zz1 * sumr1;
            vzsumr2 = vv2zz2 * sumr2;
            vzsumy1 = vv1zz1 * sumy1;
            vzsumy2 = vv2zz2 * sumy2;
            (*xx3)[0] =
                (sum1 + ww1zz1 * (*xx1)[0] + vzsumr1) * (sum2 +
                        ww2zz2 * (*xx2)[0] +
                        vzsumr2);
            (*xx3)[1] =
                (sum1 + ww1zz1 * (*xx1)[1] + vzsumy1) * (sum2 +
                        ww2zz2 * (*xx2)[1] +
                        vzsumy2);
            (*xx3)[2] =
                (sum1 + ww1zz1 * (*xx1)[2] + vzsumr1) * (sum2 +
                        ww2zz2 * (*xx2)[2] +
                        vzsumr2);
            (*xx3)[3] =
                (sum1 + ww1zz1 * (*xx1)[3] + vzsumy1) * (sum2 +
                        ww2zz2 * (*xx2)[3] +
                        vzsumy2);
            mother->scale[i] = sxx1[i] + sxx2[i];
        }
        count++;
        if (count == SCALEINTERVAL)
        {
            count = 0;
            for (i = 0; i < seq->endsite; i++)
            {
                xx3 = &(mother->x.s[i][0]);
                sxx3m = MAX ((*xx3)[0], (*xx3)[1]);
                sxx3m = MAX (sxx3m, (*xx3)[2]);
                sxx3m = MAX (sxx3m, (*xx3)[3]);
                (*xx3)[0] /= sxx3m, (*xx3)[1] /= sxx3m, (*xx3)[2] /= sxx3m,
                                                 (*xx3)[3] /= sxx3m;
                mother->scale[i] += log (sxx3m);
            }
        }
    }
    else
    {
        for (i = 0; i < rcategs; i++)
            for (j = 0; j < categs; j++)
            {
                tblij = tbl[i][j];
                tblij->ww1 = EXP (tblij->ratxi * lw1);
                tblij->zz1 = EXP (tblij->ratxv * lw1);
                tblij->ww1zz1 = tblij->ww1 * tblij->zz1;
                tblij->vv1zz1 = (1.0 - tblij->ww1) * tblij->zz1;
            }
        lw2 = -r->v / seq->fracchange;
        for (i = 0; i < rcategs; i++)
            for (j = 0; j < categs; j++)
            {
                tblij = tbl[i][j];
                tblij->ww2 = EXP (tblij->ratxi * lw2);
                tblij->zz2 = EXP (tblij->ratxv * lw2);
                tblij->ww2zz2 = tblij->ww2 * tblij->zz2;
                tblij->vv2zz2 = (1.0 - tblij->ww2) * tblij->zz2;
            }
        for (i = 0; i < seq->endsite; i++)
        {
            k = seq->category[seq->alias[i] - 1] - 1;
            for (j = 0; j < rcategs; j++)
            {
                tbljk = tbl[j][k];
                ww1zz1 = tbljk->ww1zz1;
                vv1zz1 = tbljk->vv1zz1;
                yy1 = 1.0 - tbljk->zz1;
                ww2zz2 = tbljk->ww2zz2;
                vv2zz2 = tbljk->vv2zz2;
                yy2 = 1.0 - tbljk->zz2;
                xx1 = &(q->x.s[i][j]);
                xx2 = &(r->x.s[i][j]);
                xx3 = &(mother->x.s[i][j]);
                sum1 =
                    yy1 * (seq->freqa * (*xx1)[0] + seq->freqc * (*xx1)[1] +
                           seq->freqg * (*xx1)[2] + seq->freqt * (*xx1)[3]);
                sum2 =
                    yy2 * (seq->freqa * (*xx2)[0] + seq->freqc * (*xx2)[1] +
                           seq->freqg * (*xx2)[2] + seq->freqt * (*xx2)[3]);
                sumr1 = seq->freqar * (*xx1)[0] + seq->freqgr * (*xx1)[2];
                sumr2 = seq->freqar * (*xx2)[0] + seq->freqgr * (*xx2)[2];
                sumy1 = seq->freqcy * (*xx1)[1] + seq->freqty * (*xx1)[3];
                sumy2 = seq->freqcy * (*xx2)[1] + seq->freqty * (*xx2)[3];
                vzsumr1 = vv1zz1 * sumr1;
                vzsumr2 = vv2zz2 * sumr2;
                vzsumy1 = vv1zz1 * sumy1;
                vzsumy2 = vv2zz2 * sumy2;
                (*xx3)[0] =
                    (sum1 + ww1zz1 * (*xx1)[0] + vzsumr1) * (sum2 +
                            ww2zz2 * (*xx2)[0] +
                            vzsumr2);
                (*xx3)[1] =
                    (sum1 + ww1zz1 * (*xx1)[1] + vzsumy1) * (sum2 +
                            ww2zz2 * (*xx2)[1] +
                            vzsumy2);
                (*xx3)[2] =
                    (sum1 + ww1zz1 * (*xx1)[2] + vzsumr1) * (sum2 +
                            ww2zz2 * (*xx2)[2] +
                            vzsumr2);
                (*xx3)[3] =
                    (sum1 + ww1zz1 * (*xx1)[3] + vzsumy1) * (sum2 +
                            ww2zz2 * (*xx2)[3] +
                            vzsumy2);
            }
            mother->scale[i] = sxx1[i] + sxx2[i];
        }
        count++;
        if (count == SCALEINTERVAL)
        {
            count = 0;
            for (i = 0; i < seq->endsite; i++)
            {
                sxx3m = -DBL_MAX;
                for (j = 0; j < rcategs; j++)
                {
                    xx3 = &(mother->x.s[i][j]);
                    tempsxx3m = MAX ((*xx3)[0], (*xx3)[1]);
                    tempsxx3m = MAX (sxx3m, (*xx3)[2]);
                    tempsxx3m = MAX (sxx3m, (*xx3)[3]);
                    if (tempsxx3m > sxx3m)
                        sxx3m = tempsxx3m;
                }
                for (j = 0; j < rcategs; j++)
                {
                    xx3 = &(mother->x.s[i][j]);
                    (*xx3)[0] /= sxx3m, (*xx3)[1] /= sxx3m, (*xx3)[2] /= sxx3m,
                                                     (*xx3)[3] /= sxx3m;
                }
                mother->scale[i] += log (sxx3m);
            }
        }
    }
}    /* nuview */

void
nuview_ancestral (node * mother, world_fmt * world, const long locus)
{
    long i;
    double lw1, lw2, ratio1, yy1, yy2, sum1, sum2;
    node *q, *r;
    sitelike *xx1, *xx2;
    double freqa, freqc, freqg, freqt;
    seqmodel_fmt *seq;
    seq = world->data->seq;
    freqa = seq->freqa;
    freqc = seq->freqc;
    freqg = seq->freqg;
    freqt = seq->freqt;
    q = crawlback (mother->next);
    r = crawlback (mother->next->next);
    lw1 = -q->v / seq->fracchange;
    lw2 = -r->v / seq->fracchange;
    ratio1 = lw1 / (lw1 + lw2);
    yy1 = (1. - ratio1);
    yy2 = ratio1;
    //printf("%f ", q->tyme);
    for (i = 0; i < seq->endsite; i++)
        //printf("(%f %f %f %f)", q->x.s[i][0][0], q->x.s[i][0][1], q->x.s[i][0][2], q->x.s[i][0][3]);
        //printf("\n%f ", r->tyme);
        //for (i = 0; i < seq->endsite; i++)
        //printf("(%f %f %f %f)", r->x.s[i][0][0], r->x.s[i][0][1], r->x.s[i][0][2], r->x.s[i][0][3]);
        //printf("\n");
        for (i = 0; i < seq->endsite; i++)
        {
            xx1 = &(q->x.s[i][0]);
            xx2 = &(r->x.s[i][0]);
            sum1 =
                yy1 * (seq->freqa * (*xx1)[0] + seq->freqc * (*xx1)[1] +
                       seq->freqg * (*xx1)[2] + seq->freqt * (*xx1)[3]);
            sum2 =
                yy2 * (seq->freqa * (*xx2)[0] + seq->freqc * (*xx2)[1] +
                       seq->freqg * (*xx2)[2] + seq->freqt * (*xx2)[3]);
            if (sum1 == sum2)
                sum1 += RANDUM () > 0.5 ? -1. : 1.;
            if (sum1 > sum2)
                memcpy (mother->x.s[i][0], xx1, sizeof (sitelike));
            else
                memcpy (mother->x.s[i][0], xx2, sizeof (sitelike));
        }
}    /* nuview_ancestral */

void
adjustroot (node * r)
{
    r->next->tyme = r->tyme;
    r->next->length = r->length;
    r->next->v = r->v;
    r->next->next->tyme = r->tyme;
    r->next->next->length = r->length;
    r->next->next->v = r->v;
}

void
pseudonu_seq (proposal_fmt * proposal, phenotype xxx1,
              double v1, phenotype xxx2, double v2)
//pseudonu_seq(proposal_fmt * proposal, phenotype xxx1, double v1, phenotype xxx2, double v2)
{
    long i, j, k;
    double lw1, lw2, yy1, yy2, ww1zz1, vv1zz1, ww2zz2, vv2zz2, vzsumr1,
    vzsumr2, vzsumy1, vzsumy2, sum1, sum2, sumr1, sumr2,
    sumy1, sumy2, ww1, ww2, zz1, zz2;
    double freqa, freqc, freqg, freqt, freqar, freqgr, freqcy, freqty;
    seqmodel_fmt *seq;
    valrec *tbl00;
    valrec *tblij;
    valrec *tbljk;
    sitelike *xx1, *xx2;
    long rcategs = proposal->world->options->rcategs;
    long categs = proposal->world->options->categs;
    tbl_fmt tbl = proposal->world->tbl;
    seq = proposal->world->data->seq;
    freqa = seq->freqa;
    freqc = seq->freqc;
    freqg = seq->freqg;
    freqt = seq->freqt;
    freqar = seq->freqar;
    freqgr = seq->freqgr;
    freqcy = seq->freqcy;
    freqty = seq->freqty;
    lw1 = -v1 / seq->fracchange;
    if (rcategs == 1 && categs == 1)
    {
        tbl00 = tbl[0][0];
        ww1 = EXP (tbl00->ratxi * lw1);
        zz1 = EXP (tbl00->ratxv * lw1);
        ww1zz1 = ww1 * zz1;
        vv1zz1 = (1.0 - ww1) * zz1;
        lw2 = -v2 / seq->fracchange;
        ww2 = EXP (tbl00->ratxi * lw2);
        zz2 = EXP (tbl00->ratxv * lw2);
        ww2zz2 = ww2 * zz2;
        vv2zz2 = (1.0 - ww2) * zz2;
        yy1 = 1.0 - zz1;
        yy2 = 1.0 - zz2;
        for (i = 0; i < seq->endsite; i++)
        {
            xx1 = &(xxx1[i][0]);
            xx2 = &(xxx2[i][0]);

            sum1 =
                yy1 * (freqa * (*xx1)[0] + freqc * (*xx1)[1] + freqg * (*xx1)[2] +
                       freqt * (*xx1)[3]);
            sum2 =
                yy2 * (freqa * (*xx2)[0] + freqc * (*xx2)[1] + freqg * (*xx2)[2] +
                       freqt * (*xx2)[3]);
            sumr1 = freqar * (*xx1)[0] + freqgr * (*xx1)[2];
            sumr2 = freqar * (*xx2)[0] + freqgr * (*xx2)[2];
            sumy1 = freqcy * (*xx1)[1] + freqty * (*xx1)[3];
            sumy2 = freqcy * (*xx2)[1] + freqty * (*xx2)[3];
            vzsumr1 = vv1zz1 * sumr1;
            vzsumr2 = vv2zz2 * sumr2;
            vzsumy1 = vv1zz1 * sumy1;
            vzsumy2 = vv2zz2 * sumy2;
            (*xx1)[0] =
                (sum1 + ww1zz1 * (*xx1)[0] + vzsumr1) * (sum2 +
                        ww2zz2 * (*xx2)[0] +
                        vzsumr2);
            (*xx1)[1] =
                (sum1 + ww1zz1 * (*xx1)[1] + vzsumy1) * (sum2 +
                        ww2zz2 * (*xx2)[1] +
                        vzsumy2);
            (*xx1)[2] =
                (sum1 + ww1zz1 * (*xx1)[2] + vzsumr1) * (sum2 +
                        ww2zz2 * (*xx2)[2] +
                        vzsumr2);
            (*xx1)[3] =
                (sum1 + ww1zz1 * (*xx1)[3] + vzsumy1) * (sum2 +
                        ww2zz2 * (*xx2)[3] +
                        vzsumy2);
            //memcpy(xxx1[i][0], xx3[0], sizeof(sitelike));
        }
    }
    else
    {
        for (i = 0; i < rcategs; i++)
            for (j = 0; j < categs; j++)
            {
                tblij = tbl[i][j];
                tblij->ww1 = EXP (tblij->ratxi * lw1);
                tblij->zz1 = EXP (tblij->ratxv * lw1);
                tblij->ww1zz1 = tblij->ww1 * tblij->zz1;
                tblij->vv1zz1 = (1.0 - tblij->ww1) * tblij->zz1;
            }
        lw2 = -v2 / seq->fracchange;
        for (i = 0; i < rcategs; i++)
            for (j = 0; j < categs; j++)
            {
                tblij = tbl[i][j];
                tblij->ww2 = EXP (tblij->ratxi * lw2);
                tblij->zz2 = EXP (tblij->ratxv * lw2);
                tblij->ww2zz2 = tblij->ww2 * tblij->zz2;
                tblij->vv2zz2 = (1.0 - tblij->ww2) * tblij->zz2;
            }
        for (i = 0; i < seq->endsite; i++)
        {
            k = seq->category[seq->alias[i] - 1] - 1;
            for (j = 0; j < rcategs; j++)
            {
                tbljk = tbl[j][k];
                ww1zz1 = tbljk->ww1zz1;
                vv1zz1 = tbljk->vv1zz1;
                yy1 = 1.0 - tbljk->zz1;
                ww2zz2 = tbljk->ww2zz2;
                vv2zz2 = tbljk->vv2zz2;
                yy2 = 1.0 - tbljk->zz2;
                xx1 = &(xxx1[i][j]);
                xx2 = &(xxx2[i][j]);

                sum1 =
                    yy1 * (freqa * (*xx1)[0] + freqc * (*xx1)[1] +
                           freqg * (*xx1)[2] + freqt * (*xx1)[3]);
                sum2 =
                    yy2 * (freqa * (*xx2)[0] + freqc * (*xx2)[1] +
                           freqg * (*xx2)[2] + freqt * (*xx2)[3]);
                sumr1 = freqar * (*xx1)[0] + freqgr * (*xx1)[2];
                sumr2 = freqar * (*xx2)[0] + freqgr * (*xx2)[2];
                sumy1 = freqcy * (*xx1)[1] + freqty * (*xx1)[3];
                sumy2 = freqcy * (*xx2)[1] + freqty * (*xx2)[3];
                vzsumr1 = vv1zz1 * sumr1;
                vzsumr2 = vv2zz2 * sumr2;
                vzsumy1 = vv1zz1 * sumy1;
                vzsumy2 = vv2zz2 * sumy2;
                /* xx3[j][0] */
                (*xx1)[0] =
                    (sum1 + ww1zz1 * (*xx1)[0] + vzsumr1) * (sum2 +
                            ww2zz2 * (*xx2)[0] +
                            vzsumr2);
                /* xx3[j][1] */
                (*xx1)[1] =
                    (sum1 + ww1zz1 * (*xx1)[1] + vzsumy1) * (sum2 +
                            ww2zz2 * (*xx2)[1] +
                            vzsumy2);
                /* xx3[j][2] */
                (*xx1)[2] =
                    (sum1 + ww1zz1 * (*xx1)[2] + vzsumr1) * (sum2 +
                            ww2zz2 * (*xx2)[2] +
                            vzsumr2);
                /* xx3[j][3] */
                (*xx1)[3] =
                    (sum1 + ww1zz1 * (*xx1)[3] + vzsumy1) * (sum2 +
                            ww2zz2 * (*xx2)[3] +
                            vzsumy2);
                //memcpy(xxx1[i][j], xx3[j], sizeof(sitelike));
            }
        }
    }
}    /* pseudonu_seq */

void
pseudonu_seq_slow (proposal_fmt * proposal, phenotype xxx1, double *sxx1,
                   double v1, phenotype xxx2, double *sxx2, double v2)
{
    long i, j, k;
    long count = 0;
    double lw1, lw2, yy1, yy2, ww1zz1, vv1zz1, ww2zz2, vv2zz2, vzsumr1,
    vzsumr2, vzsumy1, vzsumy2, sum1, sum2, sumr1, sumr2,
    sumy1, sumy2, ww1, ww2, zz1, zz2;
    double freqa, freqc, freqg, freqt, freqar, freqgr, freqcy, freqty;
    seqmodel_fmt *seq;
    double sxx3m, tempsxx3m;
    sitelike *xx1, *xx2;
    valrec *tblij;
    valrec *tbljk;
    valrec *tbl00;
    long rcategs = proposal->world->options->rcategs;
    long categs = proposal->world->options->categs;
    tbl_fmt tbl = proposal->world->tbl;
    seq = proposal->world->data->seq;
    freqa = seq->freqa;
    freqc = seq->freqc;
    freqg = seq->freqg;
    freqt = seq->freqt;
    freqar = seq->freqar;
    freqgr = seq->freqgr;
    freqcy = seq->freqcy;
    freqty = seq->freqty;
    lw1 = -v1 / seq->fracchange;
    if (rcategs == 1 && categs == 1)
    {
        tbl00 = tbl[0][0];
        ww1 = EXP (tbl00->ratxi * lw1);
        zz1 = EXP (tbl00->ratxv * lw1);
        ww1zz1 = ww1 * zz1;
        vv1zz1 = (1.0 - ww1) * zz1;
        lw2 = -v2 / seq->fracchange;
        ww2 = EXP (tbl00->ratxi * lw2);
        zz2 = EXP (tbl00->ratxv * lw2);
        ww2zz2 = ww2 * zz2;
        vv2zz2 = (1.0 - ww2) * zz2;
        yy1 = 1.0 - zz1;
        yy2 = 1.0 - zz2;
        for (i = 0; i < seq->endsite; i++)
        {
            xx1 = &(xxx1[i][0]);
            xx2 = &(xxx2[i][0]);
            //xx3 = &(xxx1[i][0]);
            sum1 =
                yy1 * (freqa * (*xx1)[0] + freqc * (*xx1)[1] + freqg * (*xx1)[2] +
                       freqt * (*xx1)[3]);
            sum2 =
                yy2 * (freqa * (*xx2)[0] + freqc * (*xx2)[1] + freqg * (*xx2)[2] +
                       freqt * (*xx2)[3]);
            sumr1 = freqar * (*xx1)[0] + freqgr * (*xx1)[2];
            sumr2 = freqar * (*xx2)[0] + freqgr * (*xx2)[2];
            sumy1 = freqcy * (*xx1)[1] + freqty * (*xx1)[3];
            sumy2 = freqcy * (*xx2)[1] + freqty * (*xx2)[3];
            vzsumr1 = vv1zz1 * sumr1;
            vzsumr2 = vv2zz2 * sumr2;
            vzsumy1 = vv1zz1 * sumy1;
            vzsumy2 = vv2zz2 * sumy2;
            (*xx1)[0] =
                (sum1 + ww1zz1 * (*xx1)[0] + vzsumr1) * (sum2 +
                        ww2zz2 * (*xx2)[0] +
                        vzsumr2);
            (*xx1)[1] =
                (sum1 + ww1zz1 * (*xx1)[1] + vzsumy1) * (sum2 +
                        ww2zz2 * (*xx2)[1] +
                        vzsumy2);
            (*xx1)[2] =
                (sum1 + ww1zz1 * (*xx1)[2] + vzsumr1) * (sum2 +
                        ww2zz2 * (*xx2)[2] +
                        vzsumr2);
            (*xx1)[3] =
                (sum1 + ww1zz1 * (*xx1)[3] + vzsumy1) * (sum2 +
                        ww2zz2 * (*xx2)[3] +
                        vzsumy2);
            sxx1[i] += sxx2[i];
        }
        count++;
        if (count == SCALEINTERVAL)
        {
            count = 0;
            for (i = 0; i < seq->endsite; i++)
            {
                xx1 = &(xxx1[i][0]);
                sxx3m = MAX ((*xx1)[0], (*xx1)[1]);
                sxx3m = MAX (sxx3m, (*xx1)[2]);
                sxx3m = MAX (sxx3m, (*xx1)[3]);
                (*xx1)[0] /= sxx3m, (*xx1)[1] /= sxx3m;
                (*xx1)[2] /= sxx3m, (*xx1)[3] /= sxx3m;
                sxx1[i] += log (sxx3m);
            }
        }
    }
    else
    {
        lw2 = -v2 / seq->fracchange;
        for (i = 0; i < rcategs; i++)
            for (j = 0; j < categs; j++)
            {
                tblij = tbl[i][j];
                tblij->ww1 = EXP (tblij->ratxi * lw1);
                tblij->zz1 = EXP (tblij->ratxv * lw1);
                tblij->ww1zz1 = tblij->ww1 * tblij->zz1;
                tblij->vv1zz1 = (1.0 - tblij->ww1) * tblij->zz1;
                tblij->ww2 = EXP (tblij->ratxi * lw2);
                tblij->zz2 = EXP (tblij->ratxv * lw2);
                tblij->ww2zz2 = tblij->ww2 * tblij->zz2;
                tblij->vv2zz2 = (1.0 - tblij->ww2) * tblij->zz2;
            }
        for (i = 0; i < seq->endsite; i++)
        {
            k = seq->category[seq->alias[i] - 1] - 1;
            for (j = 0; j < rcategs; j++)
            {
                tbljk = tbl[j][k];
                ww1zz1 = tbljk->ww1zz1;
                vv1zz1 = tbljk->vv1zz1;
                yy1 = 1.0 - tbljk->zz1;
                ww2zz2 = tbljk->ww2zz2;
                vv2zz2 = tbljk->vv2zz2;
                yy2 = 1.0 - tbljk->zz2;
                xx1 = &(xxx1[i][j]);
                xx2 = &(xxx2[i][j]);
                sum1 =
                    yy1 * (freqa * (*xx1)[0] + freqc * (*xx1)[1] +
                           freqg * (*xx1)[2] + freqt * (*xx1)[3]);
                sum2 =
                    yy2 * (freqa * (*xx2)[0] + freqc * (*xx2)[1] +
                           freqg * (*xx2)[2] + freqt * (*xx2)[3]);
                sumr1 = freqar * (*xx1)[0] + freqgr * (*xx1)[2];
                sumr2 = freqar * (*xx2)[0] + freqgr * (*xx2)[2];
                sumy1 = freqcy * (*xx1)[1] + freqty * (*xx1)[3];
                sumy2 = freqcy * (*xx2)[1] + freqty * (*xx2)[3];
                vzsumr1 = vv1zz1 * sumr1;
                vzsumr2 = vv2zz2 * sumr2;
                vzsumy1 = vv1zz1 * sumy1;
                vzsumy2 = vv2zz2 * sumy2;
                (*xx1)[0] =
                    (sum1 + ww1zz1 * (*xx1)[0] + vzsumr1) * (sum2 +
                            ww2zz2 * (*xx2)[0] +
                            vzsumr2);
                (*xx1)[1] =
                    (sum1 + ww1zz1 * (*xx1)[1] + vzsumy1) * (sum2 +
                            ww2zz2 * (*xx2)[1] +
                            vzsumy2);
                (*xx1)[2] =
                    (sum1 + ww1zz1 * (*xx1)[2] + vzsumr1) * (sum2 +
                            ww2zz2 * (*xx2)[2] +
                            vzsumr2);
                (*xx1)[3] =
                    (sum1 + ww1zz1 * (*xx1)[3] + vzsumy1) * (sum2 +
                            ww2zz2 * (*xx2)[3] +
                            vzsumy2);
            }
            sxx1[i] += sxx2[i];
        }
        count++;
        if (count == SCALEINTERVAL)
        {
            count = 0;
            for (i = 0; i < seq->endsite; i++)
            {
                sxx3m = -DBL_MAX;
                for (j = 0; j < rcategs; j++)
                {
                    xx1 = &(xxx1[i][j]);
                    tempsxx3m = MAX ((*xx1)[0], (*xx1)[1]);
                    tempsxx3m = MAX (sxx3m, (*xx1)[2]);
                    tempsxx3m = MAX (sxx3m, (*xx1)[3]);
                    if (tempsxx3m > sxx3m)
                        sxx3m = tempsxx3m;
                }
                for (j = 0; j < rcategs; j++)
                {
                    xx1 = &(xxx1[i][j]);
                    (*xx1)[0] /= sxx3m, (*xx1)[1] /= sxx3m;
                    (*xx1)[2] /= sxx3m, (*xx1)[3] /= sxx3m;
                }
                sxx1[i] += log (sxx3m) + sxx2[i];
            }
        }
    }
}    /* pseudonu_seq */

double
pseudo_tl_seq (phenotype xx1, phenotype xx2, double v1, double v2,
               proposal_fmt * proposal, world_fmt * world)
{
    contribarr tterm;
    contribarr clai;
    contribarr like;
    contribarr nulike;
    //long          size = sizeof(double) * world->options->rcategs;
    double summ, sum2, sumc;
    double sumterm = 0.0, lterm;
    long i, j, k, lai;
    sitelike *x1;
    worldoption_fmt *opt;
    double freqa, freqc, freqg, freqt, freqr, freqy;
    seqmodel_fmt *seq;
    seq = proposal->world->data->seq;
    freqa = seq->freqa;
    freqc = seq->freqc;
    freqg = seq->freqg;
    freqt = seq->freqt;
    freqr = seq->freqr;
    freqy = seq->freqy;

    opt = world->options;
    seq = world->data->seq;
    summ = 0.0;
    /*
     * y = v1 / seq->fracchange; lz = -y;
     */
    if (opt->rcategs == 1 && opt->categs == 1)
    {
        for (i = 0; i < seq->endsite; i++)
        {
            x1 = &(xx1[i][0]);
            tterm[0] =
                freqa * (*x1)[0] + freqc * (*x1)[1] + freqg * (*x1)[2] +
                freqt * (*x1)[3];
            summ += seq->aliasweight[i] * (log (tterm[0]) + proposal->mf[i]);
        }
        return summ;
    }
    else
    {
        for (i = 0; i < seq->endsite; i++)
        {
            //k = seq->category[seq->alias[i] - 1] - 1;
            for (j = 0; j < opt->rcategs; j++)
            {
                x1 = &(xx1[i][j]);
                (tterm[j] =
                     freqa * (*x1)[0] + freqc * (*x1)[1] + freqg * (*x1)[2] +
                     freqt * (*x1)[3]);
                //if (tterm[j] == 0.0)
                //error("Tree incompatible with data\n");
            }
            sumterm = 0.0;
            for (j = 0; j < opt->rcategs; j++)
                sumterm += opt->probcat[j] * tterm[j];
            lterm = log (sumterm) + proposal->mf[i];
            for (j = 0; j < opt->rcategs; j++)
                clai[j] = tterm[j] / sumterm;
            swap (clai, world->contribution[i]);
            //memcpy(world->contribution[i], clai, size);
            summ += seq->aliasweight[i] * lterm;
        }
        for (j = 0; j < opt->rcategs; j++)
            like[j] = 1.0;
        for (i = 0; i < seq->sites[world->locus]; i++)
        {
            sumc = 0.0;
            for (k = 0; k < opt->rcategs; k++)
                sumc += opt->probcat[k] * like[k];
            sumc *= opt->lambda;
            if ((seq->ally[i] > 0) && (seq->location[seq->ally[i] - 1] > 0))
            {
                lai = seq->location[seq->ally[i] - 1];
                swap (world->contribution[lai - 1], clai);
                //memcpy(clai, world->contribution[lai - 1], size);
                for (j = 0; j < opt->rcategs; j++)
                    nulike[j] = ((1.0 - opt->lambda) * like[j] + sumc) * clai[j];
            }
            else
            {
                for (j = 0; j < opt->rcategs; j++)
                    nulike[j] = ((1.0 - opt->lambda) * like[j] + sumc);
            }
            swap (nulike, like);
            //memcpy(like, nulike, size);
        }
        sum2 = 0.0;
        for (i = 0; i < opt->rcategs; i++)
            sum2 += opt->probcat[i] * like[i];
        summ += log (sum2);
        return summ;
    }
}

double
pseudo_tl_snp (phenotype xx1, phenotype xx2, double v1, double v2,
               proposal_fmt * proposal, world_fmt * world)
{
    contribarr tterm, invariants;
    contribarr like, nulike, clai;
    //long          size = sizeof(double) * world->options->rcategs;
    double summ, sum2, sumc, sumterm, lterm;
    long i, j, k, lai;
    sitelike *x1;
    worldoption_fmt *opt;
    double freqa, freqc, freqg, freqt, freqr, freqy;
    seqmodel_fmt *seq;
    seq = proposal->world->data->seq;
    freqa = seq->freqa;
    freqc = seq->freqc;
    freqg = seq->freqg;
    freqt = seq->freqt;
    freqr = seq->freqr;
    freqy = seq->freqy;

    opt = world->options;
    seq = world->data->seq;
    summ = 0.0;
    memset (invariants, 0, sizeof (contribarr));
    snp_invariants (invariants, seq->endsite, opt->rcategs, seq, xx1);
    if (opt->rcategs == 1 && opt->categs == 1)
    {
        for (i = 0; i < seq->endsite - seq->addon; i++)
        {
            x1 = &(xx1[i][0]);
            tterm[0] =
                (seq->freqa * (*x1)[0] + seq->freqc * (*x1)[1] +
                 seq->freqg * (*x1)[2] + seq->freqt * (*x1)[3]) / invariants[0];
            if (tterm[0] == 0.0)
                error ("Tree incompatible with data\n");
            lterm = log (tterm[0]) + proposal->mf[i];
            summ += seq->aliasweight[i] * lterm;
        }
        like[0] = 1.0;
        for (i = 0; i < seq->sites[world->locus]; i++)
        {
            sumc = opt->lambda * like[0];
            nulike[0] = ((1.0 - opt->lambda) * like[0] + sumc);
            //memcpy(like, nulike, size);
            swap (nulike, like);
        }
        summ += log (like[0]);
        return summ;
    }
    else
    {
        for (i = 0; i < seq->endsite - 4; i++)
        {
            k = seq->category[seq->alias[i] - 1] - 1;
            for (j = 0; j < opt->rcategs; j++)
            {
                x1 = &(xx1[i][j]);
                tterm[j] =
                    (seq->freqa * (*x1)[0] + seq->freqc * (*x1)[1] +
                     seq->freqg * (*x1)[2] +
                     seq->freqt * (*x1)[3]) / invariants[j];
                //if (tterm[j] == 0.0)
                //error("Tree incompatible with data\n");
            }
            sumterm = 0.0;
            for (j = 0; j < opt->rcategs; j++)
                sumterm += opt->probcat[j] * tterm[j];
            lterm = log (sumterm) + proposal->mf[i];
            for (j = 0; j < opt->rcategs; j++)
                clai[j] = tterm[j] / sumterm;
            //memcpy(world->contribution[i], clai, size);
            swap (clai, world->contribution[i]);
            summ += seq->aliasweight[i] * lterm;
        }
        for (j = 0; j < opt->rcategs; j++)
            like[j] = 1.0;
        for (i = 0; i < seq->sites[world->locus]; i++)
        {
            sumc = 0.0;
            for (k = 0; k < opt->rcategs; k++)
                sumc += opt->probcat[k] * like[k];
            sumc *= opt->lambda;
            if ((seq->ally[i] > 0) && (seq->location[seq->ally[i] - 1] > 0))
            {
                lai = seq->location[seq->ally[i] - 1];
                swap (world->contribution[lai - 1], clai);
                //memcpy(clai, world->contribution[lai - 1], size);
                for (j = 0; j < opt->rcategs; j++)
                    nulike[j] = ((1.0 - opt->lambda) * like[j] + sumc) * clai[j];
            }
            else
            {
                for (j = 0; j < opt->rcategs; j++)
                    nulike[j] = ((1.0 - opt->lambda) * like[j] + sumc);
            }
            swap (nulike, like);
            //memcpy(like, nulike, size);
        }
        sum2 = 0.0;
        for (i = 0; i < opt->rcategs; i++)
            sum2 += opt->probcat[i] * like[i];
        summ += log (sum2);
        return summ;
    }
}

double
pseudo_tl_snp_unlinked (phenotype xx1, phenotype xx2, double v1, double v2,
                        proposal_fmt * proposal, world_fmt * world)
{
    contribarr tterm, invariants;
    double summ, datasum = 0, lterm, result = 0;
    long i;
    sitelike *x1;
    worldoption_fmt *opt;
    double freqa, freqc, freqg, freqt, freqr, freqy;
    seqmodel_fmt *seq;
    seq = proposal->world->data->seq;
    freqa = seq->freqa;
    freqc = seq->freqc;
    freqg = seq->freqg;
    freqt = seq->freqt;
    freqr = seq->freqr;
    freqy = seq->freqy;

    opt = world->options;
    seq = world->data->seq;
    summ = 0.0;
    memset (invariants, 0, sizeof (contribarr));
    snp_invariants (invariants, seq->endsite, opt->rcategs, seq, xx1);
    for (i = 0; i < seq->endsite - seq->addon; i++)
    {
        x1 = &(xx1[i][0]);
        tterm[0] =
            (seq->freqa * (*x1)[0] + seq->freqc * (*x1)[1] +
             seq->freqg * (*x1)[2] + seq->freqt * (*x1)[3]);
        if (tterm[0] == 0.0)
            error ("Tree incompatible with data\n");

        if (i % 5 == 0)
        {
            lterm = log (tterm[0]) + proposal->mf[i];
            summ = 0;
            datasum = seq->aliasweight[i / 5] * lterm;
        }
        else
            summ += pow (tterm[0], (double) seq->aliasweight[i / 5]);
        if (((i + 1) % 5) == 0 && i != 0)
            result +=
                datasum + log ((1 - EXP (log (summ) - datasum)) / invariants[0]);
    }
    //EXP (sum) is the prob(xa | g)
    //              EXP (datasum) is prob(? a | g)
    //              panelsum = invariants is prob(x ? |g)
    // (datasum - sum) / invariants
    // ++some small number business
    return result;
}


void
pseudonu_anc (proposal_fmt * proposal, phenotype xxx1, double v1,
              phenotype xxx2, double v2)
{
    long i;
    double lw1, lw2, ratio1, yy1, yy2, sum1, sum2;
    double freqa, freqc, freqg, freqt;
    seqmodel_fmt *seq;
    sitelike *xx1, *xx2;
    seq = proposal->world->data->seq;
    freqa = seq->freqa;
    freqc = seq->freqc;
    freqg = seq->freqg;
    freqt = seq->freqt;
    lw1 = -v1 / seq->fracchange;
    lw2 = -v2 / seq->fracchange;
    ratio1 = lw1 / (lw1 + lw2);
    yy1 = (1. - ratio1);
    yy2 = ratio1;
    //for (i = 0; i < seq->endsite; i++)
    //printf("(%f %f %f %f)", xxx1[i][0][0], xxx1[i][0][1], xxx1[i][0][2], xxx1[i][0][3]);
    //printf("\n");
    //for (i = 0; i < seq->endsite; i++)
    //printf("(%f %f %f %f)", xxx2[i][0][0], xxx2[i][0][1], xxx2[i][0][2], xxx2[i][0][3]);
    //printf("\n");
    for (i = 0; i < seq->endsite; i++)
    {
        xx1 = &(xxx1[i][0]);
        xx2 = &(xxx2[i][0]);
        sum1 =
            yy1 * (freqa * (*xx1)[0] + freqc * (*xx1)[1] + freqg * (*xx1)[2] +
                   freqt * (*xx1)[3]);
        sum2 =
            yy2 * (freqa * (*xx2)[0] + freqc * (*xx2)[1] + freqg * (*xx2)[2] +
                   freqt * (*xx2)[3]);
        if (sum1 == sum2)
            sum1 += RANDUM () > 0.5 ? -1. : 1.;
        if (sum1 > sum2)
            memcpy (xxx1[i][0], *xx1, sizeof (sitelike));
        else
            memcpy (xxx1[i][0], *xx2, sizeof (sitelike));
    }
}    /* pseudo_nu_anc */

double
pseudo_tl_anc (phenotype xx1, phenotype xx2, double v1, double v2,
               proposal_fmt * proposal, world_fmt * world)
{
    contribarr tterm;
    double summ;
    long i;
    sitelike *x1;

    double freqa, freqc, freqg, freqt;
    seqmodel_fmt *seq;
    seq = proposal->world->data->seq;
    freqa = seq->freqa;
    freqc = seq->freqc;
    freqg = seq->freqg;
    freqt = seq->freqt;
    seq = world->data->seq;
    summ = 0.0;
    for (i = 0; i < seq->endsite; i++)
    {
        x1 = &(xx1[i][0]);
        tterm[0] =
            freqa * (*x1)[0] + freqc * (*x1)[1] + freqg * (*x1)[2] +
            freqt * (*x1)[3];
        summ += seq->aliasweight[i] * log (tterm[0]);
        //printf("pseudo %3li> %f %f \n", i, tterm[0], summ);
    }
    return summ;
}


double
treelike_anc (world_fmt * world, long locus)
{
    contribarr tterm;

    double summ;
    long i;
    node *p;
    sitelike *x1;
    seqmodel_fmt *seq;
    seq = world->data->seq;
    p = crawlback (world->root->next);
    summ = 0.0;
    for (i = 0; i < seq->endsite; i++)
    {
        x1 = &(p->x.s[i][0]);
        tterm[0] =
            seq->freqa * (*x1)[0] + seq->freqc * (*x1)[1] +
            seq->freqg * (*x1)[2] + seq->freqt * (*x1)[3];
        summ += seq->aliasweight[i] * log (tterm[0]);
        //printf("real  %3li> %f %f \n", i, tterm[0], summ);
    }
    return summ;
}    /* treelike_anc */


void
treeout (FILE * file, node * joint, node * p, long s)
{
    /* write out file with representation of final tree */
    static long col = 0;
    long w;
    double x;
    char migstring[30];
    if (p->type == 't')
    {
        translate (p->nayme, ' ', '_');
#ifdef UEP

        if(p->uep!=NULL)
            FPRINTF (file, "%s [%c]", p->nayme, p->uep[0]);
        else
            FPRINTF (file, "%s ", p->nayme);
#else

        FPRINTF (file, "%s ", p->nayme);
#endif

        col += strlen (p->nayme);
    }
    else
    {
        putc ('(', file);
        col++;
        treeout (file, joint, crawlback (p->next), s);
        putc (',', file);
        col++;
        if (col > 80)
        {
            putc ('\n', file);
            col = 0;
        }
        treeout (file, joint, crawlback (p->next->next), s);
        putc (')', file);
        col++;
    }
    if (p == joint)
    {
        x = 0.0;
    }
    else
    {
        x = crawlback (p)->tyme - p->tyme;
    }
    if (x > 0.0)
    {
        w = (long) (0.4343 * log (x));
    }
    else
    {
        if (x == 0.0)
            w = 0;
        else
            w = (long) (0.4343 * log (-x)) + 1;
    }
    if (w < 0)
        w = 0;
#ifdef UEP

    if(p->uep!=NULL)
        FPRINTF (file, ":%*.10f [%c]", (int) (w + 7), x, p->uep[0]);
    else
        FPRINTF (file, ":%*.10f ", (int) (w + 7), x);
#else

    FPRINTF (file, ":%*.10f ", (int) (w + 7), x);
#endif

    col += w + 8;
    if (col > 80)
    {
        putc ('\n', file);
        col = 0;
    }
    if (p != joint)
    {
        p = showtop (p->back);
        while (p->type == 'm')
        {
            sprintf (migstring, " [&M %li %li:%g]", p->pop, p->actualpop,
                     p->tyme - showtop (p->next->back)->tyme);
            FPRINTF (file, "%s", migstring);
            col += strlen (migstring) + 1;
            if (col > 80)
            {
                putc ('\n', file);
                col = 0;
            }
            p = showtop (p->back);
        }
    }
    else
    {
        FPRINTF (file, ";\n");
        col = 0;
    }
}    /* treeout */


void
print_tree (world_fmt * world, long g, long *filepos)
{
    switch (world->options->treeprint)
    {
    case BEST:
        if (world->likelihood[g] > world->allikemax)
        {
            if (world->allikemax == -DBL_MAX)
            {
                *filepos = ftell (world->treefile);
            }
            else
            {
                fseek (world->treefile, *filepos, SEEK_SET);
            }
            world->allikemax = world->likelihood[g];
            FPRINTF (world->treefile,
                     "\n[& Comment: Locus %li, best log likelihood = %f]\n",
                     world->locus + 1, world->likelihood[g]);
            treeout (world->treefile, crawlback (world->root->next),
                     crawlback (world->root->next), 0);
        }
        break;
    case ALL:
        /*
         * FPRINTF (world->treefile, "%g %g %g\n", crawlback
         * (world->root->next)->tyme, crawlback (crawlback
         * (world->root->next)->next)->tyme, crawlback
         * (crawlback
         * (world->root->next)->next->next)->tyme);
         */
        FPRINTF (world->treefile,
                 "\n[& Comment: Locus %li, log likelihood = %f]\n",
                 world->locus + 1, world->likelihood[g]);
        treeout (world->treefile, crawlback (world->root->next),
                 crawlback (world->root->next), 0);
        break;
    case LASTCHAIN:
        if (world->in_last_chain)
        {
            FPRINTF (world->treefile,
                     "\n[& Comment: Locus %li, log likelihood = %f]\n",
                     world->locus + 1, world->likelihood[g]);
            treeout (world->treefile, crawlback (world->root->next),
                     crawlback (world->root->next), 0);
        }
        break;
    case NONE:
        break;
    default:
        break;
    }

}


void
treereader (world_fmt * world, data_fmt * data)
{
    /*
     * read a migration tree from the usertree and set up nodes
     * and pointers
     */

    node **nodelist;
    char *nayme;
    char *temp, *temp2;
    long pop, w, zz, z = 0, zzz = 0;
    world->nodep = (node **) calloc (1, world->sumtips * sizeof (node *));
    temp = (char *) malloc (LINESIZE * sizeof (char));
    temp2 = (char *) malloc (LINESIZE * sizeof (char));
    treeread (data->utreefile, &(world->root), NULL);
    length_to_times (world->root->next->back);
    nodelist = (node **) calloc (1, sizeof (node *) * (world->sumtips + 1));
    pop = find_firstpop (world->root);
    set_tree_pop (world->root, &pop);
    allocate_x (world->root, world, world->options->datatype, WITHTIPS);
    find_tips (world->root, nodelist, &z);
    for (pop = 0; pop < world->numpop; pop++)
    {
        for (w = 0; w < data->numind[pop][world->locus]; w++)
        {
            strcpy (temp2, data->indnames[pop][w]);
            temp2[strcspn (temp2, " ")] = '\0';
            sprintf (temp, "%li%s", pop, temp2);
            for (zz = 0; zz < z; zz++)
            {
                nayme = nodelist[zz]->nayme;
                if (!strcmp (temp, nayme) || !strcmp (temp2, nayme))
                {
                    world->nodep[zzz++] = nodelist[zz];
                    break;
                }
            }
        }
    }
    free (nodelist);
    free (temp);
    free (temp2);
}


char
processlength (FILE * file, node ** p)
{
    char ch;
    long digit, ordzero;
    double valyew, divisor;
    boolean pointread, minusread;

    ordzero = '0';
    pointread = FALSE;
    minusread = FALSE;
    valyew = 0.0;
    divisor = 1.0;
    ch = getc (file);
    digit = ch - ordzero;
    while (((unsigned long) digit <= 9) | (ch == '.') || (ch == '-'))
    {
        if (ch == '.')
            pointread = TRUE;
        else if (ch == '-')
            minusread = TRUE;
        else
        {
            valyew = valyew * 10.0 + digit;
            if (pointread)
                divisor *= 10.0;
        }
        ch = getc (file);
        digit = ch - ordzero;
    }
    if (!minusread)
        (*p)->length = valyew / divisor;
    else
        (*p)->length = 0.0;
    return ch;
}

void
treeread (FILE * file, node ** pp, node * q)
{
    node *p;
    char ch = getc (file);
    while (ch != ';')
    {
        switch (ch)
        {
        case '(':
            p = create_interior_node (&q);
            q = p->next;
            ch = getc (file);
            break;
        case ',':
            q = q->next;
            if (q->top)
            {
                usererror ("Multifurcation handling not yet installed");
            }
            ch = getc (file);
            break;
        case ')':
            p = showtop (q);
            q = p->back;
            ch = getc (file);
            break;
        case ' ':
        case '\n':
        case '\t':
            ch = getc (file);
            break;
        case ':':
            ch = processlength (file, &p);
            break;
        case '[':
            ch = processbracket (file, &p);
            q->back = p;
            p->back = q;
            break;
        default:
            p = create_tip_node (file, &q, &ch);
            break;
        }
    }
    p->length = 10000.;
    (*pp) = showtop (p->back);
    fscanf (file, "%*[^\n]");
    getc (file);
}

void
length_to_times (node * p)
{
    node *q;
    if (p->type != 't')
    {
        length_to_times ((p)->next->back);
        if ((p)->type == 'i')
            length_to_times ((p)->next->next->back);
    }
    q = showtop ((p)->back);
    q->tyme = q->next->tyme = q->next->next->tyme = (p)->tyme + (p)->length;
}

void
find_tips (node * p, node ** nodelist, long *z)
{
    if (p->type == 't')
    {
        nodelist[(*z)++] = p;
    }
    else
    {
        if (p->next->back != NULL)
            find_tips (crawlback (p->next), nodelist, z);
        if (p->next->next->back != NULL)
            find_tips (crawlback (p->next->next), nodelist, z);
    }
}

long
find_firstpop (node * p)
{
    static boolean found = FALSE;
    static long pop = -1;
    if (p->type == 'm')
    {
        found = TRUE;
        pop = p->pop;
    }
    else
    {
        if (p->next->back != NULL)
        {
            find_firstpop (p->next->back);
            if (found)
                return pop;
        }
        if (p->next->next->back != NULL)
            find_firstpop (p->next->next->back);
    }
    return pop;
}

/* touches only coalescent nodes! migration nodes are already set */
void
set_tree_pop (node * p, long *pop)
{
    if (p->type != 'r')
    {

        (*pop) =
            (showtop (p->back)->actualpop !=
             *pop) ? showtop (p->back)->actualpop : *pop;
    }
    p->actualpop = p->pop = *pop;
    if (p->type != 't')
    {
        if (p->next->back != NULL)
        {
            set_tree_pop (crawlback (p->next), pop);
        }
        if (p->type != 'm' && p->next->next->back != NULL)
        {
            set_tree_pop (crawlback (p->next->next), pop);
        }
    }
}


node *
create_interior_node (node ** q)
{
    node *p;
    p = allocate_nodelet (3, 'i');
    p->top = TRUE;
    p->back = *q;
    if ((*q) == NULL)
        create_root_node (&p);
    else
        (*q)->back = p;
    return p;
}

node *
create_root_node (node ** q)
{
    node *p;
    p = allocate_nodelet (3, 'r');
    p->top = TRUE;
    p->next->back = *q;
    (*q)->back = p->next;
    return p;
}


node *
create_tip_node (FILE * file, node ** q, char *ch)
{
    node *p;
    char c;
    char *nayme;
    long nl;
    long i = 1;
    nayme = (char *) calloc (1, sizeof (char) * 20);
    nayme[0] = (*ch);
    while (strchr ("[):;,\t\n\r", (int) (c = getc (file))) == NULL)
        nayme[i++] = c;
    nayme[i] = '\0';
    p = allocate_nodelet (1, 't');
    nl = (long) strlen (nayme);
    p->nayme = (char *) calloc (1, sizeof (char) * (nl + 1));
    p->top = TRUE;
    p->tip = TRUE;
    strcpy (p->nayme, nayme);
    p->back = *q;
    (*q)->back = p;
    free (nayme);
    (*ch) = c;
    return p;
}

char
processbracket (FILE * file, node ** p)
{
    long pop1, pop2;
    double utime;
    char c;
    c = getc (file);
    if (c == '&')
    {
        c = getc (file);
        switch (c)
        {
        case 'M':
            fscanf (file, "%li %li:%lf", &pop1, &pop2, &utime);
            c = getc (file);
            (*p) = add_migration (*p, pop1, pop2, utime);
            break;
        default:
            while (c != ']')
                c = getc (file);
            break;
        }
    }
    else
    {
        while (c != ']')
            c = getc (file);
    }
    c = getc (file);
    return c;
}


node *
add_migration (node * p, long from, long to, double utime)
{
    node *tmp;
    tmp = allocate_nodelet (2, 'm');
    tmp->top = TRUE;
    tmp->next->back = p;
    p->back = tmp->next;
    tmp->length = p->length - utime;
    p->length = utime;
    tmp->tyme = p->tyme + utime;
    tmp->pop = tmp->next->pop = from;
    tmp->actualpop = tmp->next->actualpop = to;
    return tmp;
}

void
allocate_x (node * p, world_fmt * world, char datatype, boolean withtips)
{
    if (p->type != 't')
    {
        if (p->next->back != NULL)
            allocate_x (crawlback (p->next), world, datatype, withtips);
        if (p->next->next->back != NULL)
            allocate_x (crawlback (p->next->next), world, datatype, withtips);
        if (strchr (SEQUENCETYPES, world->options->datatype))
        {
            alloc_seqx (world, p);
        }
        else
        {
            if (strchr (SEQUENCETYPES, world->options->datatype))
                p->x.a = (double *) calloc (1, world->sumtips * sizeof (double));
            else
                p->x.a =
                    (double *) calloc (1,
                                       MAX (world->sumtips,
                                            world->data->maxalleles[world->locus]) *
                                       sizeof (double));
        }
#ifdef UEP
        if(world->options->uep)
        {
            p->uep = (int *) calloc (world->data->uepsites, sizeof (int));
            p->ux.s = (pair *) calloc (world->data->uepsites, sizeof (pair));
        }
#endif

    }
    else
    {
        if (withtips)
        {
            if (strchr (SEQUENCETYPES, world->options->datatype))
            {
                alloc_seqx (world, p);
            }
            else
            {
                p->x.a =
                    (double *) calloc (1,
                                       world->data->maxalleles[world->locus] *
                                       sizeof (double));
            }
        }
        //#ifdef UEP
        //      if(world->options->uep)
        // {
        //   p->uep = (long *) calloc (world->data->uepsites, sizeof (long));
        //   p->ux.s = (pair *) calloc (world->data->uepsites, sizeof (pair));
        // }
        //#endif

    }
}

long
number_genomes (char datatype)
{
    switch (datatype)
    {
    case 'a':
    case 'b':
    case 'm':
        return 2;
    case 's':
    case 'n':
    case 'u':
    case 'f':
        return 1;
    default:
        error ("Wrong data type");
    }
    return 0;

}


void
copy_tree (world_fmt * original, world_fmt * kopie)
{
    kopie->root = copy_node (original, original->root, kopie, NULL);
}

node *
copy_node (world_fmt * original, node * o, world_fmt * kopie, node * last)
{
    static long z = 0;

    node *t = NULL, *t2, *t3;
    if (o == NULL)
        return NULL;
    if (!o->top)
        error ("copy_tree messed up");

    switch (o->type)
    {
    case 'r':
        z = 0;
    case 'i':
        t = (node *) calloc (1, sizeof (node));
        t2 = (node *) calloc (1, sizeof (node));
        t3 = (node *) calloc (1, sizeof (node));
        t->next = t2;
        t2->next = t3;
        t3->next = t;
        copy_node_content (original, kopie, o, t);
        copy_node_content (original, kopie, o->next, t2);
        copy_node_content (original, kopie, o->next->next, t3);
        if (o->next->back != NULL)
            t2->back = copy_node (original, o->next->back, kopie, t2);
        if (o->next->next->back != NULL)
            t3->back = copy_node (original, o->next->next->back, kopie, t3);
        t->back = last;
        break;
    case 'm':
        t = (node *) calloc (1, sizeof (node));
        t2 = (node *) calloc (1, sizeof (node));
        t->next = t2;
        t2->next = t;
        copy_node_content (original, kopie, o, t);
        copy_node_content (original, kopie, o->next, t2);
        t2->back = copy_node (original, o->next->back, kopie, t2);
        t->back = last;
        break;
    case 't':
        t = (node *) calloc (1, sizeof (node));
        //kopie->nodep[z++] = t;
        t->next = t;
        copy_node_content (original, kopie, o, t);
        t->back = last;
        break;
    }
    return t;
}

void
copy_node_content (world_fmt * original, world_fmt * kopie, node * o,
                   node * t)
{
    long i, j;
    long endsite = original->data->seq->endsite;
    long rcategs = original->options->rcategs;
    t->type = o->type;
    t->number = o->number;
    t->pop = o->pop;
    t->actualpop = o->actualpop;
    t->id = o->id;
    t->top = o->top;
    t->dirty = o->dirty;
    t->v = o->v;
    t->tyme = o->tyme;
    t->length = o->length;
    //t->xcoord = o->xcoord;
    //t->ycoord = o->ycoord;
    //t->ymin = o->ymin;
    //t->ymax = o->ymax;


    if (t->top && t->type != 'm')
    {
        t->scale = (double *) calloc (endsite, sizeof (double));
        memcpy (t->scale, o->scale, sizeof (double) * endsite);
#ifdef UEP

        if (original->options->uep)
        {
            t->uep = (int *) calloc (original->data->uepsites, sizeof (int));
            memcpy (t->uep, o->uep, sizeof (long) * original->data->uepsites);
            t->ux.s = (pair *) calloc (original->data->uepsites, sizeof (pair));
            for (i = 0; i < original->data->uepsites; i++)
            {
                memcpy (t->ux.s[i], o->ux.s[i], sizeof (pair));
            }
        }
#endif
        if (strchr (SEQUENCETYPES, original->options->datatype))
        {
            alloc_seqx (kopie, t);
            memcpy (t->scale, o->scale, sizeof (double) * endsite);
            for (i = 0; i < endsite; i++)
            {
                for (j = 0; j < rcategs; j++)
                    memcpy (t->x.s[i][j], o->x.s[i][j], sizeof (sitelike));
            }
        }
        else
        {
            t->x.a =
                (double *) calloc (1,
                                   original->data->maxalleles[original->locus] *
                                   sizeof (double));
            memcpy (t->x.a, o->x.a,
                    sizeof (double) *
                    original->data->maxalleles[original->locus]);
        }
    }
    //if (o->s != NULL)
    ///{
    //t->s = (double *) calloc(1, sizeof(double) * original->numpop);
    //memcpy] (t->s, o->s, sizeof(double) * original->numpop);
    //
    //      }
    //
    else
        t->s = NULL;
    if (o->nayme != NULL)
    {
        t->nayme = (char *) calloc (20, sizeof (char));
        strncpy (t->nayme, o->nayme, 10);
    }
    else
        t->nayme = NULL;
}

void
swap_tree (world_fmt * this, world_fmt * that)
{
    node *tmp;
    tmp = this->root;
    this->root = that->root;
    that->root = tmp;
}

void
calc_treelength (node * p, double *treelen)
{
    node *pn, *pnn;
    switch (p->type)
    {
    case 't':
        break;
    case 'm':
        error ("yelp\n");
        break;
    case 'i':
        pn = crawlback (p->next);
        calc_treelength (pn, treelen);
        pnn = crawlback (p->next->next);
        calc_treelength (pnn, treelen);
        break;
    default:
        error ("default reached");
    }
    pn = showtop (crawlback (p));
    if (pn->type != 'r')
        *treelen += pn->tyme - p->tyme;
}

double
calc_pseudotreelength (proposal_fmt * proposal, double treelen)
{
    double len = 0.0;
    double ot = proposal->origin->tyme;
    double obt = proposal->oback->tyme;
    double tt = proposal->target->tyme;
    double rt = proposal->world->root->next->back->tyme;
    //target is not root
    if (proposal->target != proposal->world->root)
    {
        //oback is not root
        if (proposal->oback != proposal->world->root)
        {
            len = treelen - (obt - ot) + (proposal->time - ot);
            //printf("pseudo_treelen: ob!=r t!=r %f\n", len);
        }
        else
        {
            //oback is root
            len = treelen - (obt - ot) - (rt - tt) +
                  (proposal->time - ot) + (proposal->time - tt);
            //printf("pseudo_treelen: ob=r t!=r %f\n", len);
        }
    }
    else
        //target is root
    {
        //oback is not root
        if (proposal->oback != proposal->world->root)
        {
            len = treelen - (obt - ot) + (proposal->time - ot) - (rt - tt) +
                  (proposal->time - tt);
            //printf("pseudo_treelen: ob!=r t=r %f\n", len);
        }
        else
        {
            //oback is root
            len = treelen - (obt - ot) - (obt - tt) + (proposal->time - ot)
                  + (proposal->time - tt);
            //printf("pseudo_treelen: ob=r t=r %f\n", len);
        }
    }
    return len;
}

void
swap (void *a, void *b)
{
    void *t;
    t = a;
    a = b;
    b = t;
}


void
free_tree (node * p, world_fmt * world)
{
    if (p != NULL)
    {
        if (p->type != 't')
        {
            if (p->next->back != NULL)
            {
                free_tree (p->next->back, world);
            }
            if (p->type != 'm' && p->next->next->back != NULL)
            {
                free_tree (p->next->next->back, world);
            }
        }
        else
        {
            free (p->nayme);
        }
        switch (p->type)
        {
        case 'm':
            free_nodelet (p, 2, world);
            break;
        case 't':
            free_nodelet (p, 1, world);
            break;
        default:
            free_nodelet (p, 3, world);
            break;
        }
    }
}

void
free_nodelet (node * p, long num, world_fmt * world)
{
    long i;
    node *q;
    switch ((short) num)
    {
    case 3:
        free (p->next->next);
    case 2:
        free (p->next);
    case 1:
        if (p->type != 'm')
            free_nodedata (p, world);
        free (p);
        break;
    default:
        for (i = 0; i < num; i++)
        {
            q = p->next;
            free (p);
            p = q;
        }
    }
}

void
free_nodedata (node * p, world_fmt * world)
{
    long endsite, j;
    if (strchr (SEQUENCETYPES, world->options->datatype))
    {
        endsite = world->data->seq->endsite;
        //        if (strchr ("u", world->options->datatype))
        //        endsite = endsite * (world->data->seq->addon + 1) +
        //              world->data->seq->addon;
        //     if (strchr ("n", world->options->datatype))
        //     endsite += world->data->seq->addon;
        for (j = 0; j < endsite; j++)
            free (p->x.s[j]);
        free (p->x.s);
        if (p->s != NULL)
            free (p->s);
        free (p->scale);
    }
    else
    {
        free (p->x.a);
    }
#ifdef UEP
    if (world->options->uep)
        free (p->uep);
#endif

}
