/*------------------------------------------------------
Maximum likelihood estimation
of migration rate  and effectice population size
using a Metropolis-Hastings Monte Carlo algorithm
-------------------------------------------------------
    Bayesian   R O U T I N E S

    Peter Beerli 2003, Seattle
    beerli@csit.fsu.edu

    Copyright 2003-2004 Peter Beerli, Tallahassee

    This software is distributed free of charge for non-commercial use
    and is copyrighted. Of course, we do not guarantee that the software
    works and are not responsible for any damage you may cause or have.


 $Id: bayes.c,v 1.31 2005/01/21 16:48:32 beerli Exp $
    */
/*! \file bayes.c 

bayes.c contains functions to handle the Bayesian implementation of migrate
it is triggered through the options->bayes_infer=TRUE.

*/

#include "bayes.h"
#include "random.h"
#include "tools.h"
#include "sighandler.h"
#include "world.h"

#ifdef MPI
#include "migrate_mpi.h"
#else
extern int myID;
#endif

#ifdef BAYESUPDATE

extern void reprecalc_world(world_fmt *world, long that);


boolean bayes_accept (MYREAL newval, MYREAL oldval, MYREAL heat, MYREAL hastingsratio);
void bayes_print_accept(FILE * file,  world_fmt *world);


MYREAL propose_uni_newparam (MYREAL param, MYREAL mean,
                             MYREAL delta, MYREAL minparam, MYREAL maxparam, MYREAL r);
MYREAL propose_exp_newparam (MYREAL param,MYREAL mean,
                             MYREAL delta, MYREAL minparam, MYREAL maxparam, MYREAL r);
MYREAL propose_expb_newparam (MYREAL param,
                              MYREAL mean, MYREAL delta, MYREAL minparam, MYREAL maxparam, MYREAL r);
MYREAL propose_expa_newparam (MYREAL param,
                              MYREAL mean, MYREAL delta, MYREAL minparam, MYREAL maxparam, MYREAL r);

MYREAL (*propose_newparam) (MYREAL, MYREAL, MYREAL, MYREAL, MYREAL, MYREAL);
MYREAL (*hastings_ratio) (MYREAL , MYREAL, MYREAL, MYREAL, bayes_fmt *, long );

long bayes_update (world_fmt * world);
MYREAL hastings_ratio_uni(MYREAL newparam, MYREAL oldparam, MYREAL delta, MYREAL r, bayes_fmt * bayes, long whichparam);
MYREAL hastings_ratio_exp(MYREAL newparam, MYREAL oldparam, MYREAL delta, MYREAL r, bayes_fmt * bayes, long whichparam);
MYREAL hastings_ratio_expb(MYREAL newparam, MYREAL oldparam, MYREAL delta, MYREAL r, bayes_fmt * bayes, long whichparam);
MYREAL hastings_ratio_expa(MYREAL newparam, MYREAL oldparam, MYREAL delta, MYREAL r, bayes_fmt * bayes, long whichparam);

MYINLINE MYREAL probWait(long *lines,MYREAL *locallparam, long numpop, long numpop2);

void calculate_credibility_interval(world_fmt * world, long locus);

void calc_hpd_credibility(bayes_fmt *bayes,long locus, long numparam);

void bayes_combine_loci(world_fmt * world);
void print_locus_histogram_header(FILE *bayesfile, MYREAL *deltas, char *custm2, long numparam);
void print_locus_histogram(FILE *bayesfile, bayes_fmt * bayes, long locus, long numparam);
void print_loci_histogram(FILE *bayesfile, bayes_fmt * bayes, long locus, long numparam);

void bayes_set_param(MYREAL *param, MYREAL newparam, long which, char *custm2, long numpop);

boolean adjust_bayes_min_max(world_fmt* world, MYREAL **mini, MYREAL **maxi, MYREAL **adjmaxi);

void bayes_progress(world_fmt *world);


/// \brief Decide which prior distribution to use

/// Decide which prior distribution to use: the functionpointer propose_newparam will hold 
/// either the Exponential prior distribution or a Uniform prior distribution
/// each prior distribution ahs its own specific hastings ratio that will be calculated in the
/// function ptr hastings_ratio
void
which_prior (int kind)
{
    switch (kind)
    {
		case UNIFORMPRIOR:
			propose_newparam = (MYREAL (*) (MYREAL, MYREAL, MYREAL, MYREAL, MYREAL, MYREAL )) propose_uni_newparam;
			hastings_ratio = (MYREAL (*) (MYREAL, MYREAL, MYREAL, MYREAL, bayes_fmt *, long)) hastings_ratio_uni;
			break;
		case EXPPRIOR:  
			propose_newparam = (MYREAL (*) (MYREAL, MYREAL, MYREAL, MYREAL, MYREAL, MYREAL)) propose_exp_newparam;
			hastings_ratio = (MYREAL (*) (MYREAL, MYREAL, MYREAL, MYREAL, bayes_fmt *, long )) hastings_ratio_exp;
            break;
		case EXPPRIORB:
			propose_newparam = (MYREAL (*) (MYREAL, MYREAL, MYREAL, MYREAL, MYREAL, MYREAL)) propose_expb_newparam;
			hastings_ratio = (MYREAL (*) (MYREAL, MYREAL, MYREAL, MYREAL, bayes_fmt *, long)) hastings_ratio_expb;
            break;
		case ADAPTIVEPRIOR:  
			propose_newparam = (MYREAL (*) (MYREAL, MYREAL, MYREAL, MYREAL, MYREAL, MYREAL)) propose_expa_newparam;
			hastings_ratio = (MYREAL (*) (MYREAL, MYREAL, MYREAL, MYREAL, bayes_fmt *, long )) hastings_ratio_expa;
            break;
    }
}


/// \brief Calculate Prob(g|param) from world->treetimes
///
/// Calculate Prob(g|param) from world->treetimes
MYREAL probg_treetimes(world_fmt *world)
{
    const long numpop = world->numpop;
    const long numpop2 = world->numpop2;
	const long locus = world->locus;
	
	vtlist *tl = world->treetimes->tl;
	vtlist *tli = &tl[0];
	vtlist *tli1;
	
	long i;
	long r, j;
    long tlif = tli->from;
	long tlit = tli->to;
    int msta, msto;
    
    MYREAL deltatime = tl[0].age;
    MYREAL sumprob = 0.;
    MYREAL eventprob=0.;
	
	MYREAL *geo = world->data->geo;
    MYREAL *lgeo = world->data->lgeo;
	MYREAL *param0 = world->param0;
    MYREAL *locallparam;
    MYREAL *localparam;
    MYREAL *sm;
    MYREAL mu_rate = world->options->mu_rates[locus];
    MYREAL lmu_rate = world->options->lmu_rates[locus];
    
#ifdef LONGSUM	
    MYREAL *rates;
    MYREAL *lrates;
    MYREAL *rtimes;
#endif /*LONGSUM*/
	
#ifndef LONGSUM	
    locallparam = (MYREAL *) mycalloc ((numpop2 + numpop + numpop), sizeof (MYREAL));
#else /*LONGSUM*/	
    locallparam = (MYREAL *) mycalloc ((numpop2 + numpop + numpop + 9 * numpop), sizeof (MYREAL));
    rates = locallparam + numpop2 + numpop + numpop;
    lrates = rates + 3 * numpop;
    rtimes = rates +  6 * numpop;
    // rates and lrates=log(rates) hold the multipliers for Theta in the the timeslices
    memcpy (rates, param+nr->partsize - 3 * numpop, sizeof(MYREAL) * 3 * numpop);
    memcpy (lrates, lparam+nr->partsize - 3 * numpop, sizeof(MYREAL) * 3 * numpop);
    // rtimes hold the ages at the bottom of a rate timeslice 
    memcpy (rtimes, nr->world->flucrates + 3 * numpop, sizeof(MYREAL) * 3 * numpop);
#endif /*LONGSUM*/
    
	// pointers into the localparam vector
    localparam = locallparam + numpop2;
    sm = localparam + numpop;
	
	// fill localparam vector with data
    memcpy (localparam, param0, sizeof (MYREAL) * numpop);	
    for (r = 0; r < numpop; r++)
    {
        locallparam[r] = LOG2 - (log(param0[r]) + lmu_rate);
        localparam[r] = -1. / (localparam[r] * mu_rate); // minus, so that we can loop over all in probG4
        msta = world->mstart[r];
        msto = world->mend[r];
        for (j = msta; j < msto; j++)
        {
            if (param0[j] > 0.0)
            {
                sm[r] -= geo[j] * param0[j] / mu_rate; //minus, so that we can loop over all in probG4
                locallparam[j] = log(param0[j]) + lgeo[j] - lmu_rate;
            }
        }
    }
    
	eventprob = (tlif==tlit) ? (locallparam[tlif]) : locallparam[mm2m(tlif,tlit, numpop)];
	sumprob = deltatime * probWait(tli->lineages, locallparam, numpop, numpop2) + eventprob;
	
    for(i=1; i<world->treetimes->T-1;i++)
	{
		tli = &tl[i];
		tli1 = &tl[i-1];
		tlif = tli->from;
		tlit = tli->to;
		
        deltatime = (tli->age - tli1->age);
		
        eventprob = ((tlif==tlit) ? (locallparam[tlif]) : locallparam[mm2m(tlif,tlit, numpop)]);
#ifndef LONGSUM        
		sumprob += deltatime * probWait(tli->lineages, locallparam, numpop, numpop2) + eventprob;
#else
		error("Bayesian method of multiple timeslices is not yet implemented");
#endif
    }
    free (locallparam);
    return sumprob;
}

///
/// calculates Prob(t | param)
MYINLINE 
MYREAL probWait(long *lines, MYREAL *locallparam, long numpop, long numpop2)
{
	MYREAL *invtheta = locallparam + numpop2;
	MYREAL *msum = invtheta + numpop;
    
    long j;
    MYREAL line;
	MYREAL probm = 0.;
	MYREAL probth = 0.;
    for(j=0; j < numpop; j++)
    {
        line = lines[j];
		probm += line * msum[j];
        probth += line *(line-1.) * invtheta[j];
    }
    return probth + probm;
}

///
/// do we accept parameter update
boolean
bayes_accept (MYREAL newval, MYREAL oldval, MYREAL heat, MYREAL hastingsratio)
{
    MYREAL diff = (newval - oldval) * heat + hastingsratio;// heat = 1/temperature
    if (diff > 0.0)
        return TRUE;
    if (log (RANDUM ()) < diff)
        return TRUE;
    else
        return FALSE;
}

///
/// Uniform flat prior, coding based on Rasmus Nielsen, veryfied with mathematica
/// the correlation among values is dependent on delta
/// \TODO delta needs to be tunable in the menu, because there might not be no good
/// default delta.
MYREAL
propose_uni_newparam (MYREAL param, MYREAL mean, MYREAL delta, MYREAL minparam, MYREAL maxparam, MYREAL r)
{
    MYREAL np;
    MYREAL rr = 2. * r;
    
    if(rr > 1.0)
    {
        np = param + (rr - 1.) * delta;
        if (np > maxparam)
            return 2. * maxparam - np;
    }
    else
    {
        np = param -  rr * delta;
        if (np < minparam)
            return 2. * minparam - np;
    }        
    return np;
}

///
/// Hastings ratio calculator for uniform distribution
MYREAL hastings_ratio_uni(MYREAL newparam, MYREAL oldparam, MYREAL delta, MYREAL r, bayes_fmt * bayes, long whichparam)
{
    return 0.;
}

/// \brief Exponential prior distribution

/// Exponential prior using a uniform distribution centered around the old value
/// to allow only smaller changes to the parameters the range for the uniform is
/// arbitrarliy fixed.
///
/// We use the function
///   p[x] = Integrate[Exp[-u/mean]/mean,{u,a,x}]/K
/// K is a constant to integrate to 1 and it is
/// $$
/// -e^{-b/mean} + e^{-a/mean}
/// $$
/// where a is the lower bound and b is an upper bound,
/// The solution to find theta_new is to solve the above for x.
/// and it is
/// $$
/// - m log(-(e^{-\frac{a}{mean}}(r-1))  + 
///                  e^{-\frac{b}{mean}} r)
/// $$
///
MYREAL
propose_exp_newparam (MYREAL param, MYREAL mean,
					  MYREAL delta, MYREAL minparam, MYREAL maxparam, MYREAL r)
{
    MYREAL np = 0.;
    np = - mean * log(-(EXP(-minparam/mean)*(r-1)) + EXP(-maxparam/mean)*r);
    return np;
}

///
/// Hastings ratio calculator for exponential distribution
MYREAL hastings_ratio_exp(MYREAL newparam, MYREAL oldparam, MYREAL delta, MYREAL r, bayes_fmt * bayes, long whichparam)
{
   return 0.; 
}


///
/// uses an exponential prior with boundaries minparm and maxparm and uses also a window around the old parameter
/// the window is of arbitrary size
/// (see propose_exp_newparam() for details)
MYREAL
propose_expb_newparam (MYREAL param, MYREAL mean, MYREAL delta, MYREAL minparam, MYREAL maxparam, MYREAL r)
{
    MYREAL a = MAX(param - delta, 0.0);
    MYREAL b = param + delta;
    MYREAL np = - mean * log(-(EXP(-a/mean)*(r-1)) + EXP(-b/mean)*r);
    while(np < minparam || np > maxparam)
    {
        if(np < minparam)
        {
            np = np + fabs(minparam - np);
        }
        else
        {
            np = maxparam - fabs(maxparam - np);
        }
    }
    return np;
}

///
/// Hastings ratio calculator for exponential distribution
MYREAL hastings_ratio_expb(MYREAL newparam, MYREAL oldparam, MYREAL delta, MYREAL r, bayes_fmt * bayes, long whichparam)
{
    return 0. ;
}

///
/// Use the parameter from the last trial and use bounds that are 
/// multiplicatives lower bound a = mean / delta, upper bound b = mean * delta
/// this results in  
/// $$
/// x = - mean log(-(r-1) e^-1/delta + r e^-delta)
/// $$
/// the mean is the old parameter value
MYREAL
propose_expa_newparam (MYREAL param, MYREAL mean, MYREAL delta, MYREAL minparam, MYREAL maxparam, MYREAL r)
{
    MYREAL np = - (param * log(-(r-1) * EXP(-1./delta) + r * EXP(-delta)));
    while(np < minparam || np > maxparam)
    {
        if(np < minparam)
        {
            np = np + fabs(minparam - np);
        }
        else
        {
            np = maxparam - fabs(maxparam - np);
        }
    }
    return np;
}

///
/// Hastings ratio calculator for exponential distribution with multiplicator
/// using the old parameter as mean value
/// simply the jacobian from x->param, matrix if derivatives [should be seocnd right?]
/// [check this with the green paper] 
/// first derivatives 
MYREAL hastings_ratio_expa(MYREAL newparam, MYREAL oldparam, MYREAL delta, MYREAL r, bayes_fmt * bayes, long whichparam)
{
    MYREAL val= - (log(-(r-1) * EXP(-1./delta) + r * EXP(-delta)));
    return log(val);
}

///
/// Bayesian parameter update, this function is called in every step
/// but does consider changes only every updateratio count.
long
bayes_update (world_fmt * world)
{
    static unsigned long count=0;
    long        ba=0;
    long        which;  
    MYREAL      *oldparam;
    MYREAL      oldval;
    MYREAL      newparam;
    MYREAL      mean;
    MYREAL      newval;
    MYREAL      r;
    MYREAL      hastingsratio;
    bayes_fmt   *bayes = world->bayes;
    boolean     success=FALSE;
    const boolean   log = (myID==MASTER && world->options->progress && world->heat == 1.0);
	const MYREAL      updateratio = world->options->updateratio;

    //
    // progress reporter nested here so that we do need to ask whether we run in Bayes mode or not
    count++;
    if(log && (count % 100000) == 0)
    {
        bayes_progress(world);
    }    
    // 
    //


	if(RANDUM() < updateratio) 
    {
		return -1L; // update the tree
	}
    // update a parameter
    which = world->bayes->paramnum = RANDINT(0,world->numpop2-1);//DEBUG
    // try to change a parameter if the parameter is not part of the 
    // connection matrix and set zero or another constant value
    while(strchr("0c",bayes->custm2[which]))
    {
        which = bayes->paramnum = RANDINT(0,world->numpop2-1);
    }
    // savecopy the old parameters
    // we use a memcopy because the cumstom migration matrix can be filled with m and s and S
    // this will change multiple parameters at the same time
    doublevec1d(&oldparam,world->numpop2);
    memcpy(oldparam, world->param0,sizeof(MYREAL) * world->numpop2);
    
    // calculate the probability for the old parameter set
    // this is done from scratch because the tree might have changed in the last step
    // oldval is using a Hastings ratio of oldparam/mean [DEBUG]
    mean = bayes->priormean[which];
    oldval = probg_treetimes(world) - oldparam[which]/mean;
    r = RANDUM();
    
    // draw a new parmeter from the prior distribution
    newparam = (*propose_newparam) (world->param0[which],
                                    mean,
                                    bayes->delta[which],
                                    bayes->minparam[which],
                                    bayes->maxparam[which], r);
    // set new parameters and allow for custom migration matrix
    bayes_set_param(world->param0,newparam,which,world->options->custm2, world->numpop);
    reprecalc_world(world, which);
    newval =probg_treetimes(world) - newparam/mean;
    // calculate the hasting ratio [needs adjustment for custom migration sS and mM]
    hastingsratio = (*hastings_ratio)(oldparam[which], newparam, bayes->delta[which], r, bayes, which);
    
    
    //Acceptance of rejection of the new value
    success = bayes_accept(newval, oldval,world->heat, hastingsratio);
    if(success)
    {
        bayes->oldval = newval;
        ba = 1;
        bayes->accept[which] += ba;
    }
    else
    {
        memcpy(world->param0, oldparam, sizeof(MYREAL) * world->numpop2);
        reprecalc_world(world, which);
        bayes->oldval = oldval;
        ba = 0;
    }
    bayes->trials[which] += 1;
    free(oldparam);
    return ba;
}

///
/// fill bayes record in arry for histograms and bayesfile
/// set parameter meaning according to option settings
void bayes_save_parameter(world_fmt *world, long pnum)
{
    long i;
    long frompop;
    long topop;
    bayes_fmt * bayes = world->bayes;
    bayes->params[pnum][0] = bayes->oldval + world->likelihood[world->G];
    if(world->options->usem)
    {
        memcpy(&bayes->params[pnum][1], world->param0,sizeof(MYREAL)*world->numpop2);
    }
    else
    {
        memcpy(&bayes->params[pnum][1], world->param0,sizeof(MYREAL)*world->numpop);
        for(i=world->numpop; i < world->numpop2; i++)
        {
            m2mm(i,world->numpop,&frompop,&topop);
            bayes->params[pnum][1 + i] = world->param0[i] * world->param0[topop];
        }
    }
}

///
/// Save the Bayesian results for printout into bayesfile
void bayes_save(world_fmt *world)
{
    long i;
    long pnum = world->bayes->numparams;
    long allocparams = world->bayes->allocparams;
    bayes_fmt * bayes = world->bayes;
        
    bayes_save_parameter(world, pnum);
    
    pnum++;
    if(pnum>=allocparams)
    {
        allocparams += 1000; 
        world->bayes->params = (MYREAL **) myrealloc((void *) world->bayes->params,sizeof(MYREAL*)*allocparams);
        for(i=pnum;i<allocparams;i++)
            world->bayes->params[i] = (MYREAL *) mycalloc(world->numpop2+1,sizeof(MYREAL));
    }
    bayes->numparams = pnum;
    bayes->allocparams = allocparams;
}

///
/// Initialize the Bayesian framwork
void bayes_init(bayes_fmt *bayes, long size)
{
    bayes->oldval = -DBL_MAX;
    bayes->allocparams = 1;
    bayes->numparams = 0;
    bayes->paramnum = 0;
    // datastore for several variables
    bayes->datastore = (MYREAL *) mycalloc(7 * size + 2,sizeof(MYREAL));
    // pointers into datastore
    bayes->priormean = bayes->datastore;
    bayes->delta = bayes->priormean + size;
    bayes->minparam = bayes->delta + size;
    bayes->maxparam = bayes->minparam + size;
    bayes->meanparam = bayes->maxparam + size;
    bayes->datastore2 = (long *) mycalloc(2 * size + 2,sizeof(long));
    bayes->accept = bayes->datastore2;
    bayes->trials = bayes->accept + size + 1;
    // records for all bayes derived values
    bayes->params = (MYREAL **) mycalloc(1,sizeof(MYREAL *));
    bayes->params[0] = (MYREAL *) mycalloc(size+1,sizeof(MYREAL));
}


/// initialize the Bayes histogram structure, adds an additional element for the
/// summary over loci.
void bayes_init_histogram(world_fmt * world, option_fmt * options)
{
    bayes_fmt *bayes = world->bayes;
    long loc;
    long np = world->numpop2;
    long pa;
    bayeshistogram_fmt *hist;
    bayes->histogram = (bayeshistogram_fmt *) mycalloc(world->loci + 1,sizeof(bayeshistogram_fmt));
    bayes->deltahist = (MYREAL *) mycalloc(world->numpop2,sizeof(MYREAL));
    for(loc=0; loc <= world->loci; loc++)
    {
        hist = &(bayes->histogram[loc]);
        hist->bins = (long *) mycalloc(world->numpop2,sizeof(long));
        
        hist->results = NULL; //calloc(hist->binsum, sizeof(MYREAL));   // contains histogram, size is bins*numparam
                              // on a per parameter basis
                              // structure has a data storage vectors and the following are all pointers into it
        hist->numparam = np;    // number of parameters
        hist->datastore = (MYREAL *) mycalloc(11 * np, sizeof(MYREAL)); // data storage, size is numparam*11
                                                                        // pointers into data storage
        hist->minima = hist->datastore;    // contains minimal values for each parameter
        hist->maxima = hist->datastore + np;    // contains maximal values for each parameter
        hist->adjmaxima = hist->datastore + 2*np;// holds maxima values used in histogram [are smaller than maxima]
            hist->cred50l  = hist->datastore + 3*np;    // holds 50%-credibility margins (<all lower values>, 
            hist->cred50u = hist->datastore + 4*np;   //<all high values>)
            hist->cred95l = hist->datastore + 5*np;    // holds 95%-credibility margins (<all lower values>)
            hist->cred95u = hist->datastore + 6*np;   //<all high values>)
            hist->modes = hist->datastore + 7*np;    // holds 95%-credibility margins (<all lower values>, <all high values>)
            hist->medians = hist->datastore + 8*np;
            hist->means = hist->datastore + 9*np;            
            hist->stds = hist->datastore + 10*np;            
            
            for(pa=0; pa < world->numpop; pa++)
            {
                if(!strchr("c", world->options->custm2[pa]))
                {
                    hist->bins[pa] = options->bayespriortheta->bins;        
                    hist->binsum += options->bayespriortheta->bins;
                    hist->minima[pa] = HUGE;
                }
            }
            for(pa=world->numpop; pa < world->numpop2; pa++)
            {
                if(!strchr("0c", world->options->custm2[pa]))
                {
                    hist->bins[pa] = options->bayespriorm->bins;        
                    hist->binsum += options->bayespriorm->bins;
                    hist->minima[pa] = HUGE;
                }
            }
            
    }
}

///
/// selects the specific set of prior parameters according to the prior options setting
/// for each parameter with array_count i
MYINLINE  void select_prior_param(int selector, long i, bayes_fmt *bayes, const prior_fmt *prior)       
{
    switch(selector)
    {
        case ADAPTIVEPRIOR:
        case EXPPRIORB:
        case EXPPRIOR:
            bayes->priormean[i] = prior->mean; // fill mean for the call to bayes_epxb_newparam
            bayes->delta[i] =  prior->delta ; //(prior->min + prior->max)/(20.); // 1/10 of the max span
            break;
        case UNIFORMPRIOR:
            bayes->priormean[i] = prior->mean; // fill mean for the call to bayes_epxb_newparam
            bayes->delta[i] =  prior->delta; //(prior->max - prior->min)/(10.); // 1/10 of the max span
            break;
        default:
            error("Problems with the specification of the prior distribution");
            break;
    }
    bayes->minparam[i] = prior->min;
    bayes->maxparam[i] = prior->max;
    bayes->meanparam[i] = prior->mean; 
    bayes->deltahist[i] = (prior->max - prior->min)/ prior->bins;
}

/// fill the Bayesian framework with values
void bayes_fill(world_fmt *world, option_fmt *options)
{
    long i;
    
    bayes_fmt * bayes = world->bayes;
	which_prior(options->bayesprior);
    
    for(i=0; i< world->numpop;i++)
    {
        select_prior_param(options->bayesprior, i, bayes, options->bayespriortheta);
    }
    for(i=world->numpop; i< world->numpop2;i++)
    {        
        select_prior_param(options->bayesprior, i, bayes, options->bayespriorm);
    }
    // attach to the custm migration matrix
    bayes->custm2 = world->options->custm2;
}

void bayes_reset(world_fmt * world)
{
    if(world->options->bayes_infer)
    {
        world->bayes->numparams = 0; // each locus start a new set overwriting the old, allocparam is not reset
    }
}

/// free the Bayesian framework
void bayes_free(world_fmt *world)
{
    long i;
    free(world->bayes->datastore);
    for(i=world->bayes->allocparams; i > 0; i--)
        free(world->bayes->params[i]);
    free(world->bayes->params);
    //   free(world->bayes);
}


/// calculate the Bayesian statistics and prints the statistics
void bayes_stat(world_fmt *world)
{
    bayes_fmt * bayes = world->bayes;
    bayeshistogram_fmt *hist;
    long locus;
    long frompop=0;
    long topop=0;
    long j;
    long size;
    long numpop2 = world->numpop2;
    // for single locus data one is not calculating the overall values
    long lozi = world->loci > 1 ? world->loci : 0;
#ifdef LONGSUM
    long addon = (world->fluctuate ? world->numpop * 3 : 0);
#else
    long addon = 0;
#endif
    
    char st[6];
    char stemp[9];
    
    
    addon += world->options->gamma ? 1 : 0;
    size = numpop2 + addon;
    
    // print raw histogram data into the bayesfile
    print_locus_histogram_header(world->bayesfile, bayes->deltahist, bayes->custm2, world->numpop2);
    for(locus=0; locus < world->loci; locus++)
    {
        print_locus_histogram(world->bayesfile, bayes, locus, world->numpop2);
    }
    if(world->loci>1)
    {
        bayes_combine_loci(world);
        print_loci_histogram(world->bayesfile, bayes, world->loci, world->numpop2);
    }
    FPRINTF(world->outfile,"\n\n\nBayesian estimates\n");
    FPRINTF(world->outfile,"==================\n\n");
    FPRINTF(world->outfile,"Locus Parameter        2.5%%      25.0%%   median    75.0%%   97.5%%     mode     mean\n");
    FPRINTF(world->outfile,"-----------------------------------------------------------------------------------\n");
    for(locus=0; locus <= lozi; locus++)
    {
        hist = &bayes->histogram[locus];
        if(locus == world->loci)
            strcpy(st,"  All ");
        else
            sprintf(st,"%5li ",locus + 1);
        
        for(j=0; j< size; j++)
        {

            if(strchr("0c", world->options->custm2[j]))
                continue;

            if(j < world->numpop)
            {
                FPRINTF(world->outfile,"%5s ", st);
                FPRINTF(world->outfile,"Theta_%-3li      ",j+1);
                FPRINTF(world->outfile, "%8.5f %8.5f %8.5f %8.5f %8.5f %8.5f %8.5f\n",
                        hist->cred95l[j], hist->cred50l[j], hist->medians[j], 
                        hist->cred50u[j], hist->cred95u[j],hist->modes[j], hist->means[j]);
            }
            else
            {
                m2mm(j,world->numpop,&frompop, &topop);
                if(world->options->usem)
                    sprintf(stemp,"M_%li->%li", frompop+1, topop+1);
                else
                    sprintf(stemp,"Theta_%li*M_%li->%li", topop+1, frompop+1, topop+1);

                FPRINTF(world->outfile,"%5s ", st);
                FPRINTF(world->outfile, "%-15.15s%8.2f %8.2f %8.2f %8.2f %8.2f %8.2f %8.2f\n",stemp,
                        hist->cred95l[j], hist->cred50l[j], hist->medians[j],
                        hist->cred50u[j], hist->cred95u[j],hist->modes[j], hist->means[j]);
            }     
        }
    }
    // print out the acceptance ratios for every parameter and the tree
    if(world->options->progress)
    {
        // final acceptance ratios for the Bayesian run
        bayes_print_accept(stdout,world);
    }
    bayes_print_accept(world->outfile,world);
    
}


///
/// print out the acceptance ratios for all the different Bayesian updates
void
bayes_print_accept(FILE * file,  world_fmt *world)
{
    long j;         //used to loop over all parameters
    long topop;     // indicator into the parameter vector, specifying originating population 
    long frompop;   // receiving population
    char *stempo;   // string variable holding print-string 
    char *stemp;    // pointer to string, seems to be need to don't get MYREAL free warnings
    long trials=0;
    // on macs in conjunction with sprintf?
    long tc = world->numpop2; //this ignores alpha among multiple loci
    bayes_fmt *bayes = world->bayes; 
    
    stempo = (char *) mycalloc(LINESIZE,sizeof(char));
    stemp = stempo;

    FPRINTF(file,"\n\n\nAcceptance ratios for all parameters and the genealogies\n");
    FPRINTF(file,"---------------------------------------------------------------------\n\n");
    FPRINTF(file,"Parameter           Accepted changes               Ratio\n");

    // population sizes
    for(j=0; j < world->numpop; j++)
    {
        if(!strchr("0c", bayes->custm2[j]))
        {
            if((trials=bayes->trials[j])>0)
            {
                FPRINTF(file,"Theta_%-3li             %8li/%-8li         %8.5f\n", j+1, bayes->accept[j],
                        trials, (MYREAL) bayes->accept[j]/trials);
            }
        }
    }
    // migration rates
    for(j=world->numpop; j < world->numpop2; j++)
    {
        if(!strchr("0c", bayes->custm2[j]))
        {
            if((trials=bayes->trials[j])>0)
            {
                m2mm (j, world->numpop, &frompop, &topop);
                sprintf(stemp, "M_%li->%li", frompop+1, topop+1);
                FPRINTF(file, "%-9.9s             %8li/%-8li         %8.5f\n", stemp, bayes->accept[j], 
                        trials, (MYREAL) bayes->accept[j]/trials);
                memset(stemp,0,sizeof(char)*(LINESIZE-1));        
            }
        }
    }
    // accepted trees
    if((trials=bayes->trials[tc])>0)
    {
        FPRINTF(file, "Genealogies           %8li/%-8li         %8.5f\n", bayes->accept[tc], (long)
            ((bayes->trials[tc]) * (1. - world->options->updateratio)), 
            (MYREAL) bayes->accept[tc]/((bayes->trials[tc]) * (1. - world->options->updateratio)));
    }
    free(stempo);
}


///
/// print out the acceptance ratios for all the different Bayesian updates
void
bayes_progress(world_fmt *world)
{
    // printing machinery
    char *buffer;
    long bufsize=0;
    const boolean writelog = world->options->writelog;
    const boolean progress = world->options->progress;
    char spacer[]="";
    //
    char nowstr[LINESIZE]; // will hold time of day
    long j;         //used to loop over all parameters
    long topop;     // indicator into the parameter vector, specifying originating population 
    long frompop;   // receiving population
    char *stempo;   // string variable holding print-string 
    char *stemp;    // pointer to string, seems to be need to don't get MYREAL free warnings
                    // on macs in conjunction with sprintf?
    long tc = world->numpop2; //this ignores alpha among multiple loci
    bayes_fmt *bayes = world->bayes; 

    buffer = (char *) mycalloc(MAXBUFSIZE,sizeof(char));

    stempo = (char *) mycalloc(LINESIZE,sizeof(char));
    stemp = stempo;
        
    get_time (nowstr, "%H:%M:%S");
    bufsize += sprintf(buffer+bufsize,"\n%s   [NODE:%i, Locus: %li] ", nowstr,myID, world->locus + 1);
    prognose_time (nowstr, world, /*world->options->lincr*/ 1, bayes->trials[tc], spacer, TRUE);
    bufsize += sprintf(buffer+bufsize,"(prognosed end of run is %s [%f done])\n", nowstr, (MYREAL) world->treesdone/world->treestotal);

    for(j=0; j < world->numpop; j++)
    {
        if(!strchr("0c", bayes->custm2[j]))
        {
            bufsize += sprintf(buffer+bufsize, "           Theta_%-3li      acc-ratio=%10.5f  value=%10.5f\n", 
                    j+1, (MYREAL) bayes->accept[j]/bayes->trials[j],world->param0[j]);
        }
    }
    // migration rates
    for(j=world->numpop; j < world->numpop2; j++)
    {
        if(!strchr("0c", bayes->custm2[j]))
        {
            m2mm (j, world->numpop, &frompop, &topop);
            sprintf(stemp, "M_%li->%li", frompop+1, topop+1);
            bufsize += sprintf(buffer+bufsize, "           %-9.9s      acc-ratio=%10.5f  value=%10.5f\n", 
                    stemp, (MYREAL) bayes->accept[j]/bayes->trials[j], world->param0[j]);
            memset(stemp,0,sizeof(char)*(LINESIZE-1));        
        }
    }
    // accepted trees
    bufsize += sprintf(buffer+bufsize, "           Genealogies    acc-ratio=%10.5f\n",  
            (MYREAL) bayes->accept[tc] *((bayes->trials[tc]) * (1. - world->options->updateratio)));
    free(stempo);
    if(progress)
    {
        FPRINTF(stdout,"%s",buffer);
    }
    if(writelog)
    {
        FPRINTF(world->options->logfile,"%s",buffer);
    }
    free(buffer);
}




/// adjusts the allocations for the histogram bins for the current locus 
void adjust_bayes_bins(world_fmt * world, long locus)
{
    long pa;
    bayeshistogram_fmt *hist = &(world->bayes->histogram[locus]);
    hist->binsum = 0;
    for(pa=0; pa < world->numpop2; pa++)
    {
        // if custom migration matrix is set to zero, zero the bins else set it
        if(strchr("0c",world->options->custm2[pa]))
        {
            hist->bins[pa] = 0;
        }
        else
        {
            hist->bins[pa] = (long) (0.5 + (hist->maxima[pa] - hist->minima[pa]) / world->bayes->deltahist[pa]) ;
            //fprintf(stdout,"%i> LOCUS: bins[%li]=%li = (%f - %f) /%f \n",myID,pa,hist->bins[pa],hist->maxima[pa], hist->minima[pa], world->bayes->deltahist[pa]);
            hist->binsum += hist->bins[pa];
        }
    }
    // allocate the found number of bins for the histogram
    world->bayes->histogram[locus].results = (MYREAL *) mycalloc(hist->binsum, sizeof(MYREAL));
    world->bayes->histogram[locus].set95 = (char *) mycalloc(hist->binsum* 2 + 2, sizeof(char));
    world->bayes->histogram[locus].set50 = world->bayes->histogram[locus].set95 + hist->binsum + 1;
    memset(hist->results, 0 , sizeof(MYREAL) * (hist->binsum));    
}

/// construct bayes histogram: make a at least BAYESNUMBIN slices through the min-max range for each locus
/// while calculating the histogram calculate also the mean and standard deviation.
/// This can be done in the same loop, but is somehwat opaque.
void construct_locus_histogram(world_fmt *world, long locus, MYREAL *mini, MYREAL *maxi, MYREAL **results)
{
    long bin;
    long pa;
    long pa0;
    long i;
    MYREAL delta;
    long total;
    long numbin=0 ;
    MYREAL themean;
    MYREAL thestd;
    MYREAL value;
    MYREAL ** data = world->bayes->params;
    MYREAL *means = world->bayes->histogram[locus].means;
    MYREAL *stds = world->bayes->histogram[locus].stds;
    for(pa=1, pa0=0; pa <= world->numpop2; pa++, pa0++)
    {
        // if custom migration matrix is set to zero
        // continue
        if(strchr("0c",world->options->custm2[pa0]))
            continue;
        
        total = 0;
        themean = 0.0;
        thestd = 0.0;
        //        means[pa0] = 0.0;
        //        stds[pa0] = 0.0;
        for(i=0;i < world->bayes->numparams; i++)
        {
            delta = world->bayes->deltahist[pa0];
            value = data[i][pa];
            //            means[pa0] += value;
            //            stds[pa0] += value * value;
            themean += value;
            thestd += value * value;
            if(value < mini[pa0])
            {
                warning("data is smaller than minimum in function construct_locus_histogram()\n");
                bin=0;
            }
            else
            {
                if(value > maxi[pa0])
                {
                    warning("data is larger than maximum in function construct_locus_histogram()\n");
                    bin = world->bayes->histogram[locus].bins[pa0] - 1;
                }
                else
                {
                    bin = (long) (-0.5 + (value - mini[pa0])/delta);
                    if(bin<0)
                        bin=0;
                }
            }
            if((bin) > world->bayes->histogram[locus].bins[pa0])
                warning("DARN failed in construct_locus_histogram()\n");
            (*results)[numbin + bin] += 1.;
            total += 1;
        }
        //        for(bin=0; bin < world->bayes->histogram[locus].bins[pa0] ; bin++)
        //        {
        //            (*results)[numbin + bin] /= (MYREAL) total;
        //        }
        numbin += world->bayes->histogram[locus].bins[pa0];
        if(locus!=world->loci)
        {
            stds[pa0] = (themean * themean - thestd) / (total-1);
            means[pa0] = themean / total;
        }
    }
}

///
/// adjust min and max so that the min and max are in sync with delta from the first locus
/// this allows combination of different histograms without storing all raw data
boolean adjust_bayes_min_max(world_fmt* world, MYREAL **mini, MYREAL **maxi, MYREAL **adjmaxi)
{
    MYREAL *allmini = world->bayes->histogram[world->loci].minima;
    MYREAL *allmaxi = world->bayes->histogram[world->loci].maxima;
    MYREAL delta;
    long pa;
    for(pa=0; pa < world->numpop2; pa++)
    {
        // if custom migration matrix is set to zero
        // continue
        if(strchr("0c", world->options->custm2[pa]))
            continue;
        
        
        delta = world->bayes->deltahist[pa];
        //fprintf(stdout,"%i> old:(%f-%f) loci:[%f-%f] delta: %f locus: %li", myID, (*mini)[pa], (*maxi)[pa], allmini[pa],allmaxi[pa], delta, world->locus);
        if((*mini)[pa] < allmini[pa])
        {
            //            (*mini)[pa] = allmini[pa] ;//MPI changes, does this work for single cpu?  - delta * ceilf((allmini[pa]- (*mini)[pa])/delta);
            allmini[pa] = (*mini)[pa];
        }
        else
        {
            (*mini)[pa] = allmini[pa] ;
        }
        
        if((*maxi)[pa] > allmaxi[pa])
        {
            //            (*maxi)[pa] = allmaxi[pa] + delta * ceilf(((*maxi)[pa]-allmaxi[pa])/delta);
            allmaxi[pa] = (*maxi)[pa];
        }
        else
        {
            (*maxi)[pa] = allmaxi[pa];
        }
        //fprintf(stdout,"new:(%f-%f)\n", (*mini)[pa], (*maxi)[pa]);
    }
    // 
    //    (*mini)[0] = allmini[0];
    //    (*maxi)[0] = allmaxi[0];
    return TRUE;  // always dirty and needs reallocation of results
}


/// find minimum and maximum values for each parameters 
void find_bayes_min_max(world_fmt* world, MYREAL **mini, MYREAL **maxi, MYREAL **adjmaxi)
{
    long pa;
    long i;
    long changed_min;
    long changed_max;
    MYREAL value;

    MYREAL delta;
    MYREAL minivalue;
    MYREAL maxivalue;

    for(pa=0; pa < world->numpop2; pa++)
    {
        // if custom migration matrix is set to zero
        // continue
        if(strchr("0c", world->bayes->custm2[pa]))
            continue;
        
        minivalue = HUGE;
        maxivalue = 0.;
        changed_min = 0;
        changed_max = 0;
        //#ifdef MPI
        delta = world->bayes->deltahist[pa];
        //#endif
        for(i=0; i < world->bayes->numparams; i++)
        {
                value = world->bayes->params[i][pa+1];
                if( minivalue > value)
                {
                    minivalue = value;
                    changed_min++;
                }
                if(maxivalue < value)
                {
                    maxivalue = value;
                    changed_max++;
                }
        }
        if(maxivalue-minivalue<EPSILON)
        {
#ifdef MPI
            maxivalue += BAYESNUMBIN * delta;
#else
            maxivalue += BAYESNUMBIN * EPSILON;        
#endif
        }
        if(changed_max==0)
            maxivalue = world->bayes->maxparam[pa];
        
        if(changed_min==0)
            minivalue = world->bayes->minparam[pa];
        
        
        // adjust the mini and maxi so that deltahist will fit in
        (*mini)[pa] =  delta * floor(minivalue / delta);
        (*maxi)[pa] =  delta * ceil(maxivalue / delta);
//        fprintf(stdout,"find_bayes_min_max() @@@@@ %f - %f @@@@@ [global: %f - %f]\n",(*mini)[pa],(*maxi)[pa],
//                world->bayes->histogram[world->loci].minima[pa],world->bayes->histogram[world->loci].maxima[pa]);
    }
}


///
/// prints a comment header, using shell script comments for the output of the raw histogram data for bayesdata
///
/// # Raw data for the histogram of the posterior probabilities for all parameters and loci\n
/// # produced by the program migrate-n VERSIONNUMBER (www.gs.washington.edu/lamarc/migrate.hml)\n
/// # written by Peter Beerli 2004, Tallahassee, if you have problems email to beerli@csit.fsu.edu\n
/// #
/// # The HPC values are indicators whether the parametervalue is in the highest-posterior credibility set,\n
/// # a 0 means it is outside and a 1 means the value is inside the credibility set.\n
/// #
/// # --------------------------------------------------------------\n
/// # Locus Parameter 50%HPC  95%HPC parameter-value count frequency\n 
/// # --------------------------------------------------------------\n
/// #
void print_locus_histogram_header(FILE *bayesfile, MYREAL *deltas, char *custm2, long numparam)
{
    long pa;
    fprintf(bayesfile, "# Raw data for the histogram of the posterior probabilities for all parameters and loci\n");
    fprintf(bayesfile, "# produced by the program migrate-n VERSIONNUMBER (www.gs.washington.edu/lamarc/migrate.hml)\n");
    fprintf(bayesfile, "# written by Peter Beerli 2004-2005, Tallahassee, if you have problems email to beerli@csit.fsu.edu\n");
    fprintf(bayesfile, "#\n");
    fprintf(bayesfile, "# The HPC values are indicators whether the parametervalue is in the highest-posterior credibility set,\n");
    fprintf(bayesfile, "# a 0 means it is outside and a 1 means the value is inside the credibility set.\n");
    fprintf(bayesfile, "#\n");
    fprintf(bayesfile, "#\n");
    fprintf(bayesfile, "# Delta for Theta and M ");
    for(pa=0;pa<numparam; pa++)
    {
        fprintf(bayesfile,"%f ", custm2[pa]!='0' ? deltas[pa] : -99);
    }
    fprintf(bayesfile, "\n# ---------------------------------------------------------------------------------\n");
    fprintf(bayesfile, "# Locus Parameter 50%%HPC  95%%HPC parameter-value count frequency cummulative_freq\n");
    fprintf(bayesfile, "# ---------------------------------------------------------------------------------\n");
}        

///
/// print the locus data for a histogram to the file bayesfile the data has a header starting with # so that
/// other programs easiliy can remove ot for processing. the function calculates the total of all bins and
/// then is printing locusnumber parameternumber 95%HPC 50%HPC bincounts frequency for all bins and parameters
/// the HPC columns are 0 and 1, where 0 means no in the credibiliity set.
/// The header is printed in print_locus_histogram_header(bayesfile)
void print_locus_histogram(FILE *bayesfile, bayes_fmt * bayes, long locus, long numparam)
{
    long bin;
    long pa;
    long numbins = 0;
    MYREAL delta; 
    MYREAL value; 
    MYREAL total=0. ;
    MYREAL freq=0.;
    MYREAL sumfreq=0.;
    
    long *bins = bayes->histogram[locus].bins;
    MYREAL *results = bayes->histogram[locus].results;
    MYREAL *mini = bayes->histogram[locus].minima;
    MYREAL *maxi = bayes->histogram[locus].maxima;
    char *set50 = bayes->histogram[locus].set50;
    char *set95 = bayes->histogram[locus].set95;
    
    for(pa=0; pa < numparam; pa++)
    {
        // if custom migration matrix is set to zero
        // continue
        if(strchr("0c",bayes->custm2[pa]))
            continue;
        
        
        //fprintf(stdout,"########### (* locus=%li, parameter %li: *) bins=%li; min=%f; max=%f\n",locus+1, pa+1, bins[pa], mini[pa], maxi[pa]);
        delta = (maxi[pa] - mini[pa])/bins[pa];
        value = mini[pa] + delta/2.;
        total = 0. ;
        for(bin=0;bin<bins[pa];bin++)
        {
            total += results[numbins + bin];
        }
        sumfreq = 0.;
        for(bin=0;bin<bins[pa];bin++)
        {
            freq = results[numbins + bin]/total;
            sumfreq += freq;
            fprintf(bayesfile,"%li %li %c %c %f %li %f %f\n", locus+1, pa+1, set50[numbins+bin], set95[numbins+bin], value, 
                    (long) results[numbins + bin],freq,sumfreq);
            value += delta;
        }
        fprintf(bayesfile,"\n");
        numbins += bins[pa];
    }
}


///
/// print the loci data for a histogram to the file bayesfile the data has a header starting with # so that
/// other programs easiliy can remove ot for processing. the function calculates the total of all bins and
/// then is printing locusnumber parameternumber 95%HPC 50%HPC bincounts frequency for all bins and parameters
/// the HPC columns are 0 and 1, where 0 means no in the credibiliity set.
/// The header is printed in print_locus_histogram_header(bayesfile)
void print_loci_histogram(FILE *bayesfile, bayes_fmt * bayes, long locus, long numparam)
{
    long bin;
    long pa;
    long numbins = 0;
    MYREAL delta; 
    MYREAL value; 
    MYREAL total=0. ;
    MYREAL freq=0. ;
    MYREAL sumfreq=0. ;
    long *bins = bayes->histogram[locus].bins;
    MYREAL *results = bayes->histogram[locus].results;
    MYREAL *mini = bayes->histogram[locus].minima;
    MYREAL *maxi = bayes->histogram[locus].maxima;
    char *set50 = bayes->histogram[locus].set50;
    char *set95 = bayes->histogram[locus].set95;
    
    for(pa=0; pa < numparam; pa++)
    {
        // if custom migration matrix is set to zero
        // continue
        if(strchr("0c",bayes->custm2[pa]))
            continue;
        
        
        delta = (maxi[pa] - mini[pa])/bins[pa];
        value = mini[pa] + delta/2.;
        sumfreq =0.;
        for(bin=0;bin<bins[pa];bin++)
        {
            freq = results[numbins + bin];
            sumfreq += freq;
            total = bayes->trials[pa];
            fprintf(bayesfile,"%li %li %c %c %f %li %f %f\n", locus+1, pa+1, set50[numbins+bin], set95[numbins+bin], value, 
                    (long) (results[numbins + bin] * total) , freq, sumfreq);
            value += delta;
        }
        fprintf(bayesfile,"\n");
        numbins += bins[pa];
    }
}


void print_bayes_credibility(FILE *file, MYREAL *cred, MYREAL *results, long numpop)
{
    long pa;
    long numpop2 = numpop * numpop;
    for(pa=0; pa < numpop; pa++)
        fprintf(file,"#Theta lower=%f upper=%f\n",cred[pa], cred[pa+ numpop2]);
    for(pa=numpop; pa < numpop2; pa++)
        fprintf(file,"#Migration lower=%f upper=%f\n",cred[pa], cred[pa+ numpop2]);
}


///
/// finds the mode of the results histogram
/// fills the bayes->histogram[locus].modes
void     find_bayes_mode(bayes_fmt *bayes, long locus, long numparam)
{
    
    long bin;
    long numbin=0;
    long pa;
    long *bins = bayes->histogram[locus].bins;
    
    MYREAL tmp;
    MYREAL *modes = bayes->histogram[locus].modes;
    MYREAL *mini = bayes->histogram[locus].minima;
    MYREAL *results = bayes->histogram[locus].results;
    
    for(pa=0; pa < numparam; pa++)
    {
        // if custom migration matrix is set to zero
        // continue
        if(strchr("0c",bayes->custm2[pa]))
            continue;
        
        tmp = 0. ;
        for(bin=0;bin<bins[pa]; bin++)
        {
            if (tmp < results[numbin + bin])
            {
                tmp = results[numbin+bin];
                modes[pa] = bin * bayes->deltahist[pa] + mini[pa];
            }
        }
        numbin += bins[pa];
    }
}

///
/// smooth the histogram 
/// a window of size 2*el + 1 is averaged and replaces the central value
/// the array with the values is prepended with el zeros and also appended with el zeros,
/// and the output replaces the input array,
void bayes_smooth(MYREAL *x, long xelem, long el, MYREAL delta)
{
    MYREAL xsum;
    MYREAL *xx;
    long i, j;
    long el2 = 2 * el;
    xx = (MYREAL *) mycalloc(2 * el + xelem,sizeof(MYREAL));
    memcpy(xx+el, x, sizeof(MYREAL) * xelem);
    for(i=0; i< xelem; i++)
    {
        xsum = 0.;
        for(j=i; j< i+ el2; j++)
            xsum += xx[j];
        xsum /= el2;
        x[i] = xsum;
    }
    free(xx);
}

///
/// find the credibility set by using the Highest Posterior Probability Density(HPD) and the standard point 
/// descriptors. the statistics are filled into the statistics part of the bayes structure (datastore)
void calc_hpd_credibility(bayes_fmt *bayes,long locus, long numparam)
{
    const MYREAL alpha95 = 0.95;
    const MYREAL alpha50 = 0.50;
    
    long li;
    long pa;
    long numbins=0;
    long locmedian;
    
    pair *parts; // pairs of doubles
    long csum = 0;
    
    MYREAL biggestcolumn;
    MYREAL total;
    MYREAL cdf;
    MYREAL cutoff95;
    MYREAL cutoff50;
    MYREAL delta;
    MYREAL tmp;
    
    MYREAL * mini;
    MYREAL * maxi;
    MYREAL *results;
    long *bins;
    MYREAL *modes;
    MYREAL *medians;
    MYREAL *cred50;
    MYREAL *cred95;
    char *set50;
    char *set95;
    
    bins = bayes->histogram[locus].bins;
    modes = bayes->histogram[locus].modes;
    medians = bayes->histogram[locus].medians;
    mini =  bayes->histogram[locus].minima;
    maxi =  bayes->histogram[locus].maxima;
    cred50 =  bayes->histogram[locus].cred50l;
    cred95 =  bayes->histogram[locus].cred95l;
    set50 =  bayes->histogram[locus].set50;
    set95 =  bayes->histogram[locus].set95;
    results =  bayes->histogram[locus].results;

    for(pa=0; pa < numparam; pa++)
    {
        if(csum<bins[pa])
            csum = bins[pa];
    }
    parts = (pair *) mycalloc(csum, sizeof(pair));
    
    for(pa=0; pa < numparam; pa++)
    {
        // if custom migration matrix is set to zero
        // continue
        if(strchr("0c",bayes->custm2[pa]))
            continue;
        
        delta = (maxi[pa] - mini[pa]) / bins[pa];

        parts = memset(parts,0,csum*sizeof(pair));
        
        //
        // calculate the total and then the average
        total = 0.;
        biggestcolumn = 0.;
        locmedian = 0;
        //
        // smooth the results before we calculate means etc
        bayes_smooth(results+numbins,bins[pa], 5, delta);
        
        for(li=0;li<bins[pa]; li++)
        {
            tmp = results[numbins + li];
            parts[li][1] = tmp;
            total += tmp;
            parts[li][0] = delta/2 + mini[pa] + li * delta;
            if(tmp > biggestcolumn)
            {
                biggestcolumn = tmp;
                locmedian = li;
            }
        }
        //
        // mode is the value with largest column
        modes[pa] = parts[locmedian][0];
        //
        // median is at 0.5 of the cumulative probability distribution
        li = 0;
        tmp = 0;
        while(tmp < total/2  && li < bins[pa]-1)
        {
            tmp += parts[li][1];
            li++;
        }
        medians[pa] = parts[li][0];
        //
        // sort parts using the histogram y-axis for easy finding of quantiles
        paired_qsort2(parts, bins[pa]);
        
        // find HPD crediblity intervals for 50% and 95% credibility intervals
        // for 50% intervals, starting from the highest value (mode) and moving down
        // until the cumulative sum / total is >=0.5
        cdf = 0.;        
        li = bins[pa]-1;
        while(cdf < alpha50 && li>=0)
        {
            cdf += parts[li][1] /total;
            li--;
        }
        cutoff50 = parts[li][1]; // anything lower than this will be in the 50% credibility set
                                 // or 95% interval
        while(cdf < alpha95 && li>=0)
        {
            cdf += parts[li][1] /total;
            li--;
        }
        cutoff95 = parts[li][1]; // anything lower than this will be in the 95% credibility set
        
        // fill the credibility sets (are printed into Bayesfile
        for(li=numbins;li<numbins + bins[pa]; li++)
        {
            if(results[li] < cutoff95)
            {
                set50[li] = '0';
                set95[li] = '0';
            }
            else
            {
                set95[li] = '1';
                if(results[li] < cutoff50)
                    set50[li] = '0';
                else
                    set50[li] = '1';
            }
        }
        // fill the innermost 50% levels, smooth over adjacent bins
        // 
        // start at the modus and go left
        li = numbins + locmedian;
        while (set50[li] == '1' && li > numbins)
            --li ;
        cred50[pa] = mini[pa] + (li-numbins) * delta + delta/2.;
        while (set95[li] == '1' && li > numbins)
            --li ;
        cred95[pa] = mini[pa] + (li-numbins) * delta + delta/2.;
        
        // start at the modus and go right
        li = numbins + locmedian;
        while (set50[li] == '1' && li < numbins + bins[pa])
            ++li ;
        cred50[pa + numparam] = maxi[pa] - (bins[pa] - li + numbins) * delta - delta/2.;
        while (set95[li] == '1' && li < numbins + bins[pa])
            ++li ;        
        cred95[pa + numparam] = maxi[pa] - (bins[pa] - li + numbins) * delta - delta/2.;
        
        numbins += bins[pa];
    }    
    free(parts);
}

///
/// combines over loci
/// this function has the potential to be distributed over nodes by number of parameters
void bayes_combine_loci(world_fmt * world)
{
    long locus;
    long pa;
    long sourcebin;
    long targetbin;
    long sourcesumbin;
    long targetsumbin;
    MYREAL count;
    MYREAL sumprob;
    long bin;
    MYREAL maxval;
    MYREAL *maxvala;
    MYREAL total;
    //
    // records whether a cell in the all-loci histogram was filled or not
    char *touched;
    //
    // source is the locus histogram, already filled in by the locus workers or then by the locus loop
    bayeshistogram_fmt * source;
    //
    // target is the summary over all loci
    bayeshistogram_fmt * target = &world->bayes->histogram[world->loci];
    MYREAL *results;
    MYREAL *minima;
    MYREAL *maxima;
    long *bins;
    //
    // allocation of space for maximal values for parameters and set to some large negative number (logs!)   
    doublevec1d(&maxvala, world->numpop2);
    for(pa=0;pa<world->numpop2; pa++)
        maxvala[pa] = -DBL_MAX;
    //
    // set the all-loci minima and maxima
    for(locus=0;locus<world->loci; locus++)
        adjust_bayes_min_max(world,&world->bayes->histogram[locus].minima,  &world->bayes->histogram[locus].maxima, &world->bayes->histogram[locus].maxima);
    //
    // allocation of alloci-histogram table into results, and for the 50% and 95% credibility sets
    adjust_bayes_bins(world, world->loci);
    bins = target->bins;
    results = target->results;
    minima = target->minima;
    maxima = target->maxima;
    //
    // records whether a cell in the all-loci histogram was filled or not
    touched = calloc(world->bayes->histogram[world->loci].binsum + 1, sizeof(char));
    //
    // combine previous results into target->results
    // over all loci
    for(locus=0;locus<world->loci; locus++)
    {
        sourcesumbin = 0;
        targetsumbin = 0;
        source = &world->bayes->histogram[locus];
        // over all parameters
        for(pa=0; pa < world->numpop2; pa++)
        {
            // if custom migration matrix for this parameter is set to zero continue
            if(strchr("0c", world->bayes->custm2[pa]))
                continue;
            
            // placeholder for the maximum value of a parameter
            maxval = maxvala[pa] ; 
            //
            // number of bins (is this needed and not already adjusted?
            bins[pa] = (long) ((maxima[pa] - minima[pa])/ world->bayes->deltahist[pa]);
            //fprintf(stdout,"%i> LOCI: bins[%li]=%li = (%f - %f) / %f \n",myID,pa,target->bins[pa],target->maxima[pa], target->minima[pa], world->bayes->deltahist[pa]);
            // where were the maxima and minima set? %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            
            // find the entry point for source into the target array
            // this should always work fine without reminder, but rounding error might produce errors
            // therefore the 0.5 addition and flooring
            targetbin = (long) (0.5 + (source->minima[pa] - minima[pa]) / world->bayes->deltahist[pa]);
            
            for(sourcebin=0; sourcebin < source->bins[pa]; sourcebin++)
            {
                if(source->results[sourcesumbin + sourcebin] > 0.)
                {
                    // frequencies are normalized per locus and need adjustment for all loci
                    count = source->results[sourcesumbin + sourcebin];
                    results[targetsumbin + targetbin + sourcebin] += log(count);
                    touched[targetsumbin + targetbin+ sourcebin] = '1';
                    if(maxval < results[targetsumbin + targetbin + sourcebin])
                        maxval = results[targetsumbin + targetbin + sourcebin];
                }
                else
                {
                    results[targetsumbin + targetbin + sourcebin] += -100;
                }
            }
            sourcesumbin += source->bins[pa];
            targetsumbin += target->bins[pa];
            maxvala[pa] = maxval;
        }
    }
    //
    // adjust the total and use the overflow safeguard
    targetsumbin = 0;
    for(pa = 0; pa < world->numpop2; pa++)
    {
        // if custom migration matrix for this parameter is set to zero continue
        if(strchr("0c", world->bayes->custm2[pa]))
            continue;
        
        
        target->means[pa] = 0.;
        sumprob = 0.;
//        fprintf(stdout,"== start summary \nprobability   running mean of parameter value\n");
        for(bin=targetsumbin; bin < targetsumbin + bins[pa]; bin++)
        {
            if(touched[bin]=='1')
            {
                results[bin] = EXP(results[bin] - maxvala[pa]);
                sumprob += results[bin];
            }
            else
            {
                target->results[bin] = 0.;
            }
        }
        total = 0;
        for(bin=targetsumbin; bin < targetsumbin + target->bins[pa]; bin++)
        {
            if(touched[bin]=='1')
            {
                results[bin] /= sumprob;
                total +=  results[bin];
                target->means[pa] += results[bin] * (minima[pa] +  world->bayes->deltahist[pa]/2. 
                                                             + world->bayes->deltahist[pa] * (bin-targetsumbin));
            }
            //fprintf(stdout,"%20.20f %20.20f %f\n", results[bin], total, target->means[pa]);
        }
        //fprintf(stdout,"== end summary \n");
        targetsumbin += bins[pa];
    }
    free (touched);
    free(maxvala);
    
    // calculate the credibility intervals
    calc_hpd_credibility(world->bayes, world->loci, world->numpop2);
    
}

///
/// combines over loci
/// this function has the potential to be distributed over nodes by number of parameters
void old_bayes_combine_loci(world_fmt * world)
{
    long locus;
    long pa;
    long sourcebin;
    long targetbin;
    long sourcesumbin;
    long targetsumbin;
    MYREAL count;
    MYREAL sumprob;
    long bin;
    MYREAL maxval;
    MYREAL *maxvala;
    MYREAL total;
    
    // records whether a cell in the all-loci histogram was filled or not
    char *touched = calloc(world->bayes->histogram[world->loci].binsum + 1, sizeof(char));
    // source is the locus hostirgram, already filled in by the locus workers or then by the locus loop
    bayeshistogram_fmt * source;
    // target is the summary over all loci
    bayeshistogram_fmt * target = &world->bayes->histogram[world->loci];
    // allocation of space for maximal values for parameters and set to some large negative number (logs)   
    doublevec1d(&maxvala, world->numpop2);
    for(pa=0;pa<world->numpop2; pa++)
        maxvala[pa] = -DBL_MAX;
    // allocation of histogram table into results, and for the 50% and 95% credibility sets
    target->results = (MYREAL *) mycalloc(world->bayes->histogram[world->loci].binsum + 1, sizeof(MYREAL));
    target->set95 = (char *) mycalloc(2*(world->bayes->histogram[world->loci].binsum + 1) + 2, sizeof(char));
    target->set50 = target->set95 + world->bayes->histogram[world->loci].binsum + 1 + 1;
    //
    // combine previous results into target->results
    // over all loci
    for(locus=0;locus<world->loci; locus++)
    {
        sourcesumbin = 0;
        targetsumbin = 0;
        source = &world->bayes->histogram[locus];
        // over all parameters
        for(pa=0; pa < world->numpop2; pa++)
        {
            // if custom migration matrix for this parameter is set to zero continue
            if(strchr("0c", world->bayes->custm2[pa]))
                continue;
            
            // placeholder for the maximum value of a parameter
            maxval = maxvala[pa] ; 

            // where were the maxima and minima set? %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            target->bins[pa] = (long) ((target->maxima[pa] - target->minima[pa])/ world->bayes->deltahist[pa]);
      //      fprintf(stdout,"%i> LOCI: bins[%li]=%li = (%f - %f) / %f \n",myID,pa,target->bins[pa],target->maxima[pa], target->minima[pa], world->bayes->deltahist[pa]);
            // where were the maxima and minima set? %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            
            // find the entry point for source into the target array
            // this should always work fine without reminder, but rounding error might produce errors
            // therefore the 0.5 addition and flooring
            targetbin = (long) (0.5 + (source->minima[pa] - target->minima[pa]) / world->bayes->deltahist[pa]);
            
            for(sourcebin=0; sourcebin < source->bins[pa]; sourcebin++)
            {
                if(source->results[sourcesumbin + sourcebin]>0)
                {
                    // frequencies are normalized per locus and need adjustment for all loci
                    count = source->results[sourcesumbin + sourcebin];
                    target->results[targetsumbin + targetbin+ sourcebin] += log(count);
                    touched[targetsumbin + targetbin+ sourcebin] = '1';
                    if(maxval < target->results[targetsumbin + targetbin + sourcebin])
                        maxval = target->results[targetsumbin + targetbin + sourcebin];
                }
                else
                {
                    target->results[targetsumbin + targetbin+ sourcebin] += -100;
                }
            }
            sourcesumbin += source->bins[pa];
            targetsumbin += target->bins[pa];
            maxvala[pa] = maxval;
        }
    }
    //
    // adjust the total and undo the overflow safeguard
    targetsumbin = 0;
    for(pa = 0; pa < world->numpop2; pa++)
    {
        target->means[pa] = 0.;
        sumprob = 0.;
    //    fprintf(stdout,"== start summary \nprobability   running mean of parameter value\n");
        for(bin=targetsumbin; bin < targetsumbin + target->bins[pa]; bin++)
        {
            if(touched[bin]=='1')
            {
                target->results[bin] = EXP(target->results[bin] - maxvala[pa]);
                sumprob += target->results[bin];
            }
            else
            {
                target->results[bin] = 0.;
            }
        }
        total = 0;
        for(bin=targetsumbin; bin < targetsumbin + target->bins[pa]; bin++)
        {
            if(touched[bin]=='1')
            {
                target->results[bin] /= sumprob;
                total +=  target->results[bin];
                target->means[pa] += target->results[bin] * (target->minima[pa] +  world->bayes->deltahist[pa]/2. 
                                                             + world->bayes->deltahist[pa] * (bin-targetsumbin));
            }
    //        fprintf(stdout,"%20.20f %20.20f %f\n", target->results[bin], total, target->means[pa]);
        }
    //    fprintf(stdout,"== end summary \n");
        targetsumbin += target->bins[pa];
    }
    free (touched);
    free(maxvala);
    
    // calculate the credibility intervals
    calc_hpd_credibility(world->bayes, world->loci, world->numpop2);
    
}



/// sets the delta for the first locus, this is a guide for all other loci
/// see adjust_bayes_min_max()
void set_bayes_first_delta(world_fmt * world, MYREAL * mini, MYREAL *maxi)
{
    long pa;
    for(pa=0; pa < world->numpop2; pa++)
    {
        // if custom migration matrix is set to zero
        // continue
        if(strchr("0c", world->options->custm2[pa]))
            continue;
        
        world->bayes->deltahist[pa] = (maxi[pa] - mini[pa])/ world->bayes->histogram[0].bins[pa];
    }
}

/// calculates the HPC credibility set and also calculates the posterior singlelocus-histogram
void calculate_credibility_interval(world_fmt * world, long locus)
{
    MYREAL *mini;
    MYREAL *maxi;
    MYREAL *adjmaxi;
    
    mini =  world->bayes->histogram[locus].minima;
    maxi =  world->bayes->histogram[locus].maxima;
    adjmaxi =  world->bayes->histogram[locus].adjmaxima;
 
    // find min and max of locus
    find_bayes_min_max(world, &mini, &maxi, &adjmaxi);
    
    // adjust the bin number and allocate the histogram parts
    adjust_bayes_bins(world, locus);
    
     // construct histogram
    construct_locus_histogram(world, locus, mini, maxi, &world->bayes->histogram[locus].results);

    // calc_credibility
    calc_hpd_credibility(world->bayes, locus, world->numpop2);
}

///
/// adjust the parameters so that the new set is consistent with the custom migration matrix
void bayes_set_param(MYREAL *param, MYREAL newparam, long which, char *custm2, long numpop)
{
    char    migtype = custm2[which];
    long    frompop;
    long    topop;
    long    i;
    long    limit;
    MYREAL  nmig;
    //  check custm matrix and then decide
    switch(migtype)
    {
        case 'C':
        case 'c':
            break;
        case 's':
            m2mm (which, numpop, &frompop, &topop);
            param[mm2m(frompop,topop,numpop)] = newparam; // makes the 
            param[mm2m(topop,frompop,numpop)] = newparam; // two parameter equal
            break;
        case 'S':
            m2mm (which, numpop, &frompop, &topop);
            param[mm2m(frompop,topop,numpop)] = newparam; // makes the 
            param[mm2m(topop,frompop,numpop)] = param[mm2m(frompop,topop,numpop)] * param[topop] / param[frompop]; // two parameter equal
            break;
        case 'm':
            if(which<numpop)
                limit = numpop;
            else
                limit = numpop*numpop;
            for(i=0;i<limit; ++i)
            {
                if(custm2[i]=='m')
                    param[i] = newparam;
            }
            break;
        case 'M':
            if(which<numpop)
            {
                for(i=0;i<numpop; ++i)
                {
                    if(strchr("Mm",custm2[i]))
                        param[i] = newparam;
                }
            }
            else
            {
                m2mm (which, numpop, &frompop, &topop);
                nmig = param[topop] * newparam;
                for(i=numpop;i<numpop*numpop; ++i)
                {
                    if(custm2[i]=='M')
                    {
                        m2mm (i, numpop, &frompop, &topop);
                        param[i] = nmig / param[frompop]; // makes the 
                    }
                }
            }
            break;
     case '*':
     default:
         param[which] = newparam;
         break;
    }
}

#endif /*bayesupdate*/
