/*! \file main.c */
/*! \mainpage M I G R A T E
 *  \section intro Introduction
 *  Migrate is a program to estimate population-genetic parameters from genetic data such as
 *  electrophoretic marker data, microsatellite data, DNA or RNA sequence data, and single 
 *  nucleotide polymorphisms. Migrate uses the concepts of Maximum likelihood and Bayesian
 *  inference to infer parameters based on coalescence theory. For more information related
 *  to the use of the program or the theory behind check out 
 *  http://evolution.gs.washington.edu/lamarc/migratedoc/
 *  
 *  \section install Installation
 *  to install the program simply do
 *  \subsection step1 configure
 *  \subsection step2 make
 *  \subsection step3 sudo make install
 *
 *  \section maintainer Maintainer and Copyrights 
 *   Peter Beerli,
 *   beerli@csit.fsu.edu
 *
 *   Copyright 1997-2002 Peter Beerli and Joseph Felsenstein, Seattle WA\n
 *   Copyright 2003 Peter Beerli, Tallahassee FL\n
 *
 *   This software is distributed free of charge for non-commercial use
 *   and is copyrighted. Of course, I do not guarantee that the software
 *   works and am not responsible for any damage you may cause or have.
 *
 */
/* $Id: main.c,v 1.113 2005/01/21 16:48:35 beerli Exp $
 * $Log: main.c,v $
 * Revision 1.113  2005/01/21 16:48:35  beerli
 * fixing issues with the windows .net-studio development system
 *
 * Revision 1.112  2004/11/16 02:54:30  beerli
 * added more prior distribution EXP with windoing and EXP with windowing around
 * old parameter, included a hastingsratio function, update to the menu was also needed
 * the changes in tree.c is misterious. [I hate cvs as there is no easy way in this editor to
 * check what was changed there]
 *
 * Revision 1.111  2004/10/24 20:53:18  beerli
 * changes to bayes parts (it should work now for mpi and restrictions of migration matrix
 * for constant and zero parameters it will not work yet for symmetric parameters but this will
 * come soon easily by changing only the proposal mechanism.
 *
 * Revision 1.110  2004/08/28 19:59:32  beerli
 * added doucmentation for doxygen (still very far from complete added changes to the bayesian
 * part (should work now but major tests not done yet, chekcing multinode replication scheme
 *
 * Revision 1.109  2004/06/06 21:50:11  beerli
 * fix of parallel migrate for parallel replication scheme, several minor ripple effects
 * hopefully removed.
 *
 * Revision 1.108  2004/05/11 18:22:38  beerli
 * preliminary fix for LRT printing bug with many populations,
 * start to work on replication in migratre_mpi.c, unclear why reporter.c shows
 * up in the changed files.
 *
 * Revision 1.107  2004/04/19 15:27:12  beerli
 * doxygen changes,
 *
 * Revision 1.106  2004/03/07 20:43:13  beerli
 * changes to bayes part to allow multiple loci and correct histogram, not working yet
 * a memory bug is bothering, I need to check on a linux box
 *
 * Revision 1.105  2004/02/26 22:46:39  beerli
 * changes to writing parmfile-buffer() extensive documentation added and some options changed too
 * especially all input files should have now options that are option=YES:filename or option=NO
 * and changed settings for use-M=YES is now default, and changes to usertree option
 * it combines now all USERTREE, DISTANCE and RANDOM, the old options still exist but are
 * printing a warning that they are obsolete.
 * Bayesian analysis should work now? fixed a bug in reprecalc_world().
 *
 * Revision 1.104  2004/02/09 16:12:17  beerli
 * working on bayes implementation and adding documentation that is doxygen compliant
 *
 * Revision 1.103  2004/02/03 16:33:15  beerli
 * changed copyright notices and cleaned out this_string from main.c and consolidated
 * functions between tree.c and broyden.c (two functions were only used in tree.c but
 * were located in broyden.c
 *
 * Revision 1.102  2004/02/02 04:09:19  beerli
 * *** empty log message ***
 *
 *
 * 
 */

#include "migration.h"
#include "sighandler.h"
#include "heating.h"
#include "bayes.h"
#include "world.h"
#include "data.h"
#include "options.h"
#include "mcmc.h"
#include "broyden.h"
#include "combroyden.h"
#include "menu.h"
#include "random.h"
#include "sequence.h"
#include "tree.h"
#include "tools.h"
#include "profile.h"
#include "aic.h"
#include "lrt.h"
#include "migrate_mpi.h"

#include <stdio.h>

#ifdef UEP
#include "uep.h"
#endif
#ifdef DMALLOC_FUNC_CHECK
#include <dmalloc.h>
#endif
//use this if you have nice() and need to be nice with others
#ifdef HAVE_NICE
#include <unistd.h>
#endif

/* Definitions for MCMCMC -- heated chains
* definitions of the first four heated chains, they are ordered from cold to hot */
#define EARTH  universe[0]	/*!< cold chain has always a temperature=1            */
#define VENUS  universe[1]	/*!< second coldest chain                             */
#define MERKUR universe[2]	/*!< thrid coldest chain                              */
#define SUN    universe[3]	/*!< fourth coldest chain                             */

/* GLOBAL VARIABLES ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
int myID;			/*!< myID=0 for single-cpu program and master in MPI program, worker nodes myID > 0 */
int myRepID;			/*!< myID=0 for single-cpu program and master in MPI program, worker nodes myID > 0 */
int color;			/*!< either 1 or MPI_UNDEFINED, in anticipation of more complicated MPI schemes    */
int numcpu;			/*!< used if MPI is running otherwise == 1                                         */
int locidone;			/*!< used in MPI workers                                                           */


#ifdef MPI
filedb_fmt filedb[20];
long filenum;

MPI_Comm comm_world;		/*!< parallel environment that contains knowledge about all workers and master */
MPI_Comm comm_workers;		/*!< parallel environment that contains knowledge about all workers */
MPI_Group worker_group;
MPI_Group world_group;
#endif
#ifdef SLOWNET
int profiledone;		/*!< used in MPI workers in profiles on slow networks    */
#endif
#ifdef PTHREADS
tpool_t heating_pool;		/*!< when compiled with PTHREADS then holds all threads */
#else
#ifndef tpool_t
#define tpool_t char
#endif
tpool_t heating_pool;
#endif

long *seed;			/*!< contains the seed of the random number */
long *newseed;			/*!< contains the new random seed */

int
setup_locus (long locus, world_fmt * world, option_fmt * options,
	     data_fmt * data);

void
condense_time (world_fmt * world, long *step, long *j,
	       MYREAL * accepted, long *G, long *steps, long oldsteps,
	       long rep);

long set_repkind (option_fmt * options);

void
heating_init (world_fmt ** universe, int usize, data_fmt * data,
	      option_fmt * options);

void
heating_prepare (world_fmt ** universe, int usize,
		 option_fmt * options, data_fmt * data, long rep);

void heating_prepare2 (world_fmt ** universe, int usize);

long
replicate_number (option_fmt * options, long chain, char type,
		  long replicate);

void
combine_loci_standard (char type, option_fmt * options, world_fmt * world,
		       long *Gmax);
void
combine_loci_mpi (char type, option_fmt * options, world_fmt * world,
		  long *Gmax);

void set_penalizer (long chain, long chains, char type,
		    world_fmt ** universe);

void
run_sampler (option_fmt * options, data_fmt * data,
	     world_fmt ** universe, int usize, long *outfilepos, long *Gmax);

void run_replicate (long locus,
		    long replicate,
		    world_fmt ** universe,
		    option_fmt * options,
		    data_fmt * data,
		    tpool_t * localheating_pool,
		    int usize, long *treefilepos, long *Gmax);


void
run_locus (world_fmt ** universe, int usize, option_fmt * options,
	   data_fmt * data, tpool_t * localheating_pool, long maxreplicate,
	   long locus, long *treefilepos, long *Gmax);

void
run_loci (world_fmt ** universe, int usize, option_fmt * options,
	  data_fmt * data, tpool_t * localheating_pool, long maxreplicate,
	  long *treefilepos, long *Gmax);

void run_one_update (world_fmt * world);

void run_updates (world_fmt ** universe,
		  int usize, option_fmt * options,
		  tpool_t * localheating_pool, long inc, long increment,
		  long step, long steps);

void heated_swap (world_fmt ** universe, worldoption_fmt * options);

void
change_chaintype (long locus, char *type, world_fmt * world,
		  long *increment, long *oldsteps, long *chains,
		  option_fmt * options);

void
prepare_next_chain (world_fmt ** universe, worldoption_fmt * options,
		    char type, long chain, long *chains,
		    long *pluschain, long locus, long replicate);

void
print_heating_progress (world_fmt ** universe,
			worldoption_fmt * options, long stepinc);

long
analyze_olddata (world_fmt * world, option_fmt * options,
		 data_fmt * data, long *outfilepos);

void profile_tables (option_fmt * options, world_fmt * world, long *gmaxptr);

void finish_mac (option_fmt * options, data_fmt * data);

int
setup_locus (long locus, world_fmt * world, option_fmt * options,
	     data_fmt * data);

long set_repkind (option_fmt * options);

void heating_prepare2 (world_fmt ** universe, int usize);

long replicate_number (option_fmt * options, long chain, char type,
		       long replicate);

void print_heating_progress2 (FILE * file, worldoption_fmt * options,
			      world_fmt ** universe);


void get_bayeshist (world_fmt * world, option_fmt * options);
void get_treedata (world_fmt * world, option_fmt * options);
void get_mighistdata (world_fmt * world, option_fmt * options);

void change_longsum_times (world_fmt * world);

extern void unset_penalizer_function (boolean inprofiles);


///
/// the program migrate calculates migration rates and population sizes
/// from genetic data, it allows for a wide variety of data types and many options
/// \callgraph
int
main (int argc, char **argv)
{
  char type = 's';
  data_fmt *data;
  int usize = HEATED_CHAIN_NUM;
  long Gmax = 0;
  long outfilepos = 0;
  option_fmt *options;
  world_fmt **universe;
#ifdef MPI
  int server;
#endif

#ifdef HAVE_NICE
  nice (10);			//nice value arbitrarily set to 10, 5 is still nice, 0 is standard
#endif
#ifdef MPI
  // parallel version
  filenum = 0;			//setting up the filename database so that worker node can write back to master
  MPI_Init (&argc, &argv);
  comm_world = MPI_COMM_WORLD;
  MPI_Comm_size (comm_world, &numcpu);
  MPI_Comm_rank (comm_world, &myID);

  MPI_Comm_group (comm_world, &world_group);
  server = MASTER;		//server ID
  MPI_Group_excl (world_group, 1, &server, &worker_group);
  MPI_Comm_create (comm_world, worker_group, &comm_workers);
  if (myID != MASTER)
    MPI_Comm_rank (comm_workers, &myRepID);
  MYMPIBARRIER (comm_world);
#ifdef SLOWNET
  // slow network parallel version
  profiledone = 0;
  which_calc_like (SINGLELOCUS);
#endif
#else
  //scalar version of migrate
  myID = MASTER;
  numcpu = 1;
#endif
  seed = (long *) mymalloc (sizeof (long) * 3);
  newseed = (long *) mymalloc (sizeof (long) * 3);
  //
  universe = (world_fmt **) mycalloc (HEATED_CHAIN_NUM, sizeof (world_fmt *));

  /* try to catch and beautify some error messages -------------- */
  signalhandling (ON);

  /* create main data structures -------------------------------- */
  create_data (&data);
  create_world (&(EARTH), 1L);
  create_options (&options);

  /* parmfile and menu ------------------------------------------ */
  init_options (options);
  if (argc > 1)			// user uses another parmfile
    strncpy (options->parmfilename, argv[1], (long) strlen (argv[1]) + 1);

#ifdef MPI
  if (myID == MASTER)
    {
#endif
      read_options_master (options);
      print_menu_title (stdout, options);
      get_menu (options);
#ifdef BAYESUPDATE
      if (options->bayes_infer)
	options->schains = 0;
#endif
#ifdef MPI
      broadcast_options_master (options);
    }
  else
    {
      broadcast_options_worker (options);
      //options->writelog = FALSE;
      options->menu = FALSE;
    }
#endif
  /* initialization and data reading phase ---------------------- */
  usize = MAX (1, options->heated_chains);
  universe =
    (world_fmt **) myrealloc (universe, usize * sizeof (world_fmt *));
  if (myID == MASTER)
    {
      init_files (EARTH, data, options);
    }
  else
    {
      init_files (EARTH, data, options);
    }
  if (options->writelog && myID == MASTER)
    {
      print_menu_title (options->logfile, options);
    }
  EARTH->repkind = SINGLECHAIN;
  unset_penalizer_function (FALSE);	// install penalizer for far jumps in MCMC
  /* sampling phase ------------------------------------------------ */
  if (!options->readsum)
    {
      //MCMC sampling
      run_sampler (options, data, universe, usize, &outfilepos, &Gmax);
    }
  else
    {
      //reanalyze sumfile
      Gmax = analyze_olddata (EARTH, options, data, &outfilepos);
    }
  unset_penalizer_function (TRUE);
  //with MPI:the workers stay here much longer than the Master
#ifdef MPI
  combine_loci_mpi (type, options, EARTH, &Gmax);
#else
  combine_loci_standard (type, options, EARTH, &Gmax);
#endif
#ifdef BAYESUPDATE
  if (options->bayes_infer)
    {
      get_bayeshist (EARTH, options);
    }
  else
#endif
    {
      // gather the treedata(1) into sumfile or from worker nodes
      get_treedata (EARTH, options);
    }
  /* printing of data ------------------------------------------- */
#ifdef MPI
  if (myID == MASTER)
    {
#endif

#ifdef BAYESUPDATE
      if (options->bayes_infer)
	bayes_stat (EARTH);
      else
#endif
	print_list (universe, options, data);
      if (EARTH->options->gamma)
	print_alpha_curve (EARTH, EARTH->atl, &Gmax);
      //print_cov(EARTH, EARTH->numpop, EARTH->loci, EARTH->cov);
      fflush (EARTH->outfile);
#ifdef MPI

    }
#endif
  if (options->aic || (EARTH->options->lratio->counter > 0))
    {
#ifdef MPI
      if (myID == MASTER)
	{
#endif
	  if (EARTH->options->lratio->counter > 0)
	    print_lratio_test (EARTH, &Gmax);

	  if (options->aic)
	    {
	      akaike_information (EARTH, &Gmax);
	    }
#ifdef MPI

	}
#endif

    }
  if (myID == MASTER)
    {
#ifdef MPI
      get_mighistdata (EARTH, options);
#endif

      print_mighist (EARTH);
    }
  /* print UEP probabilities ------------------------------------ */
#ifdef UEP
  if (options->uep)
    analyze_uep (EARTH);
#endif
  /* profile tables --------------------------------------------- */

  unset_penalizer_function (TRUE);	//now we calculate profile and shut down
  // the penalizing of far jumps in the maximizer function

#ifdef SLOWNET
  //if there are many loci but only few parameters to estimate
  // this will be still slow and therefore saturating the network
  // with parameter requests may still be the better solution
  // for some - compile using make mpi instead make mpis
  which_calc_like (PROFILE);
  if (myID == MASTER)
    {
      // release the workers from the mpi_maximize_worker() function.
      // the workers will stop woring on locus-likelihoods and advance to
      //
      mpi_send_stop (EARTH);

    }
#ifdef PRIVATE
  //    master_gridder(world, &Gmax);
#endif

#ifdef BAYESUPDATE
  if (!options->bayes_infer)
#endif
    profile_tables (options, EARTH, &Gmax);
#else
#ifdef MPI			/* WORKS */

  if (myID == MASTER)
    {
#endif
#ifdef PRIVATE
      //       master_gridder(EARTH, &Gmax);
#endif
/*---------
#ifdef BAYESUPDATE
        if(!options->bayes_infer)
#endif
        profile_tables(options, EARTH, &Gmax);
#ifdef MPI
        
    }
#endif
#endif   
        if (myID == MASTER)
        {
            print_finish(EARTH, outfilepos);
            // closing all files ------------------------------------------------
#ifdef MAC
            
            finish_mac(options, data);
#endif
            
            exit_files(EARTH, data, options);
        }
#ifdef MPI
#ifndef SLOWNET
        mpi_send_stop(EARTH);
        //stop workers
        // printf("%i> at barrier in slownet-main before finalize\n", myID);
        
#endif
        MYMPIBARRIER(comm_world);
        MPI_Finalize();
#endif    
        //free_universe(universe, usize);
        return 0;
        
        ---------------*/

#ifdef BAYESUPDATE
      if (!options->bayes_infer)
#endif
	profile_tables (options, EARTH, &Gmax);
#ifdef MPI

    }
#endif
#endif /* SLOWNET */
  if (myID == MASTER)
    {
      print_finish (EARTH, outfilepos);
// closing all files ------------------------------------------------
#ifdef MAC

      finish_mac (options, data);
#endif

      exit_files (EARTH, data, options);
    }
#ifdef MPI
#ifndef SLOWNET
  mpi_send_stop (EARTH);
  //stop workers
  // printf("%i> at barrier in slownet-main before finalize\n", myID);

#endif
  MYMPIBARRIER (comm_world);
  MPI_Finalize ();
#endif
  //free_universe(universe, usize);
  return 0;
}				/* main end */

///
/// Calculates the maximum likelihood estimates from the trees that were gathered for each locus
/// \callgraph
void
combine_loci_standard (char type, option_fmt * options, world_fmt * world,
		       long *Gmax)
{
  long kind = MULTILOCUS;
  if (options->readsum)
    {
      if (world->loci == 1)
	kind = SINGLELOCUS;
    }
  if ((world->loci - world->skipped > 1) || (options->readsum))
    {
      if (options->gamma)
	world->param0[world->numpop2] = world->options->alphavalue;
      world->repkind = set_repkind (options);
      decide_plot (world->options, world->options->lchains,
		   world->options->lchains, 'l');
#ifdef LONGSUM
      change_longsum_times (world);	//multilocus
#endif /*LONGSUM*/
      (void) estimateParameter (options->replicatenum, *Gmax, world, options,
				world->cov[world->loci], 0 /* chain */ ,
				type, kind, world->repkind);
    }
}

///
/// Calculates the maximum likelihood estimates from the trees that were gathered for each locus
/// \callgraph
void
combine_loci_mpi (char type, option_fmt * options, world_fmt * world,
		  long *Gmax)
{
#ifdef MPI
  long kind = MULTILOCUS;
  if (options->readsum)
    {
      if (world->loci == 1)
	kind = SINGLELOCUS;
    }
  if ((world->loci - world->skipped > 0) || (options->readsum))
    {
      if (options->gamma)
	world->param0[world->numpop2] = world->options->alphavalue;
      world->repkind = set_repkind (options);
      if (myID == MASTER)
	{
	  mpi_gmax_master (world, Gmax);
	  mpi_startparam_master (world);
	  decide_plot (world->options, world->options->lchains,
		       world->options->lchains, 'l');
#ifdef LONGSUM
	  change_longsum_times (world);	//multilocus
#endif	   /*LONGSUM*/
	    // broadcast Gmax to all workers, the worker pendant is in mpi_gmax_worker
	    MYMPIBCAST (Gmax, 1, MPI_LONG, MASTER, comm_world);
	  (void) estimateParameter (options->replicatenum, *Gmax, world,
				    options, world->cov[world->loci],
				    0 /* chain */ ,
				    type, kind, world->repkind);
	}
      else
	{
	  //            printf("%i> before gmax worker\n", myID);
	  mpi_gmax_worker (world);
	  mpi_startparam_worker (world);
	  //            printf("%i> start worker postlike calculations\n", myID);
	  mpi_maximize_worker (world, MULTILOCUS, options->replicatenum);	// receive broadcast for Gmax
	}
    }
  //   else
  //   {
  //       if (myID != MASTER)
  //       {
  //           maxreplicate = (options->replicate
  //                           && options->replicatenum >
  //                           0) ? options->replicatenum : 1;
  //           fprintf(stdout,"about to pack result buffer and send too");
  //           mpi_results_worker(MIGMPI_SUMFILE, world, maxreplicate, pack_result_buffer);
  //           //mpi_maximize_worker(world, options->replicatenum);
  //       }
  //   }
#endif
}


///
/// runs the MCMC sampling 
void
run_sampler (option_fmt * options, data_fmt * data, world_fmt ** universe,
	     int usize, long *outfilepos, long *Gmax)
{
  long i;
  long maxreplicate;
  long treefilepos;
  char *this_string;
#ifdef MPI
//#ifdef MPIREPLICANT
  MPI_Status status;		// contains status used in MPI_Recv
  MPI_Request *irequests;	// contains requests generated in MPI_Isend
  MPI_Status *istatus;		// conatins stata from MPI_Wait_any
  long minnodes;
  long sent;			// counter for the array irequests of the MPI_Isend requests
//#endif
  long *twolongs;		// message sent to workers contains locus and replicate number
  //int          *ranks;
  //MPI_Group worldgroup, locigroup;
  if (myID < data->loci + 1)
    color = 1;
  else
    color = MPI_UNDEFINED;

  twolongs = (long *) calloc (TWO, sizeof (long));
#endif

  this_string = (char *) mycalloc (1024, sizeof (char));
#ifdef MPI

  if (myID == MASTER)
    {
#endif
      get_data (data->infile, data, options);
#ifdef MPI

//        printf("%i > finished data reading\n", myID);
      broadcast_data_master (data, options);
      if (numcpu == 1)
	{
	  error
	    ("This program was compiled to use a parallel computer\n and you tried to run it on only a single node.\nThis will not work because it uses a \n\"single_master-many_worker\" architecture \nand needs at least TWO nodes\n");
	}
    }
  else
    {
      broadcast_data_worker (data, options);
    }
  //MPI_Comm_dup(comm_world, &comm_all_worlds);
  //MPI_Comm_rank(comm_world, &myID);
#endif
  set_plot (options);
  init_world (EARTH, data, options);
#ifdef PTHREADS
  tpool_init (&heating_pool, usize, usize, 0);
#endif

  if (options->heating)
    //first in universe is the cold chain[EARTH]
    heating_init (universe, usize, data, options);
  getseed (options);
//#ifdef MPI
//    MYMPIBARRIER(comm_world);
//#endif
  /* calculate FST values */
  calc_simple_param (EARTH, data);
  /* report to screen */
  if (myID == MASTER)
    {
      print_menu_options (EARTH, options, data);
      if (options->progress)
	print_data_summary (stdout, EARTH, options, data);
      /* print to outfile */
      *outfilepos = print_title (EARTH, options);
      print_options (EARTH->outfile, EARTH, options, data);
      print_data_summary (EARTH->outfile, EARTH, options, data);
      print_data (EARTH, options, data);
    }
  if (options->lchains < 2 && options->replicatenum == 0)
    options->replicate = 0;
  maxreplicate = (options->replicate
		  && options->replicatenum > 0) ? options->replicatenum : 1;
#ifdef MPI
  minnodes = MAX (0, numcpu - data->loci - 1);
  irequests = (MPI_Request *) mycalloc (minnodes + 1, sizeof (MPI_Request));
  istatus = (MPI_Status *) mycalloc (minnodes + 1, sizeof (MPI_Status));

  if (myID == MASTER)
    {
      mpi_runloci_master (data->loci, EARTH->who, EARTH);
#ifndef MPIREPLICANT
      sent =
	mpi_send_stop_mcmc_worker (numcpu, data->loci, &comm_world, irequests,
				   istatus, myID);
#endif
    }
  else
    {
      mpi_runloci_worker (universe, usize, options, data,
			  &heating_pool, maxreplicate, &treefilepos, Gmax);
#ifdef MPIREPLICANT
      // the first worker will always be here waiting 
      if (myID == FIRSTWORKER)
	{
	  MYMPIRECV (twolongs, TWO, MPI_LONG, MASTER, 0, comm_world, &status);
	  sent =
	    mpi_send_stop_mcmc_worker (numcpu, data->loci, &comm_workers,
				       irequests, istatus, myID);
	}
#endif /*MPIREPLICANT*/
    }
  free (istatus);
  free (irequests);
#else
  run_loci (universe, usize, options, data,
	    &heating_pool, maxreplicate, &treefilepos, Gmax);
#endif /*MPI*/
#ifdef PTHREADS
    tpool_destroy (heating_pool, 1);
#endif
#ifdef MPI

  if (myID != MASTER)
    {
#endif
      if (options->heating)
	{
	  for (i = 0; i < options->heated_chains; i++)
	    {
	      // free_tree(universe[i]->root, universe[i]);
	      free_timevector (universe[i]->treetimes);
	    }
	}
      else
	{
	  // free_tree(EARTH->root, EARTH);
	  free_timevector (EARTH->treetimes);
	}
#ifdef MPI

    }
  free (twolongs);
#endif
}

/// generate samples of genealogies for all loci
void
run_loci (world_fmt ** universe, int usize, option_fmt * options,
	  data_fmt * data, tpool_t * localheating_pool, long maxreplicate,
	  long *treefilepos, long *Gmax)
{
  long locus;
  //  long            i;
  for (locus = 0; locus < data->loci; locus++)
    {
      run_locus (universe, usize, options, data,
		 localheating_pool, maxreplicate, locus, treefilepos, Gmax);
    }
}


/// save all genealogy summaries
void
get_treedata (world_fmt * world, option_fmt * options)
{
#ifdef MPI
  long maxreplicate = (options->replicate
		       && options->replicatenum >
		       0) ? options->replicatenum : 1;
#endif

  if (myID == MASTER)
    {
      if (options->writesum)
	{
#ifdef MPI
	  //get all sumfile data from the workers using unpack_sumfile_buffer()
	  mpi_results_master (MIGMPI_SUMFILE, world, maxreplicate,
			      unpack_sumfile_buffer);
#endif

	  write_savesum (world);
	}
#ifdef SLOWNET
      else
	{
	  //      printf("%i> about to receive the sumfile data\n", myID);
	  mpi_results_master (MIGMPI_SUMFILE, world, maxreplicate,
			      unpack_sumfile_buffer);
	  //      printf("%i> all sumfile data received and unpacked\n", myID);
	}
#endif

    }
}

///
/// save all genealogy summaries
#ifdef BAYESUPDATE
void
get_bayeshist (world_fmt * world, option_fmt * options)
{
#ifdef MPI
  long maxreplicate = (options->replicate
		       && options->replicatenum >
		       0) ? options->replicatenum : 1;

  if (myID == MASTER)
    {
      mpi_results_master (MIGMPI_BAYESHIST, world, maxreplicate,
			  unpack_bayes_buffer);
    }
#endif
}
#endif

/// get migration event time data
void
get_mighistdata (world_fmt * world, option_fmt * options)
{
#ifdef MPI
  long maxreplicate = (options->replicate
		       && options->replicatenum >
		       0) ? options->replicatenum : 1;

  if (myID == MASTER)
    {
      if (options->mighist)
	{
	  //get all mighist data from the workers using unpack_mighist_buffer()
	  mpi_results_master (MIGMPI_MIGHIST, world, maxreplicate,
			      unpack_mighist_buffer);

	}
    }
#endif

}

/// swap the tree pointer between chains with different temperatures
void
heated_swap (world_fmt ** universe, worldoption_fmt * options)
{
  long sm = 0, mv = 0, vw = 0;
  long r;
  switch (r = RANDINT (0, options->heated_chains - 2))
    {
    case 2:
      sm = chance_swap_tree (SUN, MERKUR);
      MERKUR->swapped += sm;
      break;
    case 1:
      VENUS->swapped += (mv = chance_swap_tree (MERKUR, VENUS));
      break;
    case 0:
      EARTH->swapped += (vw = chance_swap_tree (VENUS, EARTH));
      break;
    default:
      universe[r]->swapped += chance_swap_tree (universe[r + 1], universe[r]);
    }
}


/// change the type of the chain form short to long and reset the treelist
void
change_chaintype (long locus, char *type, world_fmt * world, long *increment,
		  long *oldsteps, long *chains, option_fmt * options)
{
  if (*type == 's')
    {
      *type = 'l';
      create_treetimelist (world, &(world->treetimes), locus);
      set_bounds (increment, oldsteps, chains, options, *type);
      world->increment = *increment;
    }
}

/// prepare the next chain
void
prepare_next_chain (world_fmt ** universe, worldoption_fmt * options,
		    char type, long chain, long *chains, long *pluschain,
		    long locus, long replicate)
{
  long i;
  EARTH->likelihood[0] = EARTH->likelihood[EARTH->G];
  if (options->heating)
    {
      for (i = 1; i < options->heated_chains; i++)
	universe[i]->likelihood[0] = universe[i]->likelihood[universe[i]->G];
    }
  EARTH->treetimes[0].copies = 0;
  if (type == 'l')
    {
      if (options->replicate && options->replicatenum > 0)
	EARTH->chainlikes[locus][replicate] = EARTH->param_like;
      else
	EARTH->chainlikes[locus][chain] = EARTH->param_like;
    }
  EARTH->start = FALSE;
  if (type == 'l')
    {
      if (chain < *chains + *pluschain)
	{
	  if ((EARTH->param_like > options->lcepsilon)
	      || (options->gelman && EARTH->gelmanmaxR > GELMAN_MYSTIC_VALUE))
	    {
	      (*chains)++;
	      (*pluschain)--;
	    }
	}
    }
}

/// print progress about heated chains
void
print_heating_progress (world_fmt ** universe, worldoption_fmt * options,
			long stepinc)
{

  if (options->heating)
    {
      universe[0]->accept_freq /= stepinc;
      universe[1]->accept_freq /= stepinc;
      universe[2]->accept_freq /= stepinc;
      universe[3]->accept_freq /= stepinc;
      if (options->progress)
	print_heating_progress2 (stdout, options, universe);
      if (options->writelog)
	print_heating_progress2 (options->logfile, options, universe);
      if (options->heated_chains > 4)
	universe[3]->swapped = 0;
      universe[2]->swapped = 0;
      universe[1]->swapped = 0;
      universe[0]->swapped = 0;

      universe[0]->accept = 0L;
      universe[1]->accept = 0L;
      universe[2]->accept = 0L;
      universe[3]->accept = 0L;

      universe[0]->accept_freq = 0.;
      universe[1]->accept_freq = 0.;
      universe[2]->accept_freq = 0.;
      universe[3]->accept_freq = 0.;
    }
}


/// sub-function of print_heating
void
print_heating_progress2 (FILE * file, worldoption_fmt * options,
			 world_fmt ** universe)
{
  FPRINTF (file, "           Swapping between %li temperatures.\n",
	   options->heated_chains);
  if (options->heated_chains > 4)
    FPRINTF (file, "               Only the 4 coldest chains are shown\n");
  FPRINTF (file,
	   "           Temperature | Accepted |  Swaps between temperatures\n");
  if (options->heated_chains > 4)
    FPRINTF (file, "           %10.4f  |   %4.2f   |    %5li||\n",
	     universe[3]->averageheat, universe[3]->accept_freq,
	     universe[3]->swapped);
  else
    FPRINTF (file, "           %10.4f  |   %4.2f   |          |\n",
	     universe[3]->averageheat, universe[3]->accept_freq);
  FPRINTF (file, "           %10.4f  |   %4.2f   |    %5li ||\n",
	   universe[2]->averageheat, universe[2]->accept_freq,
	   universe[2]->swapped);
  FPRINTF (file, "           %10.4f  |   %4.2f   |    %5li  ||\n",
	   universe[1]->averageheat, universe[1]->accept_freq,
	   universe[1]->swapped);
  FPRINTF (file, "           %10.4f  |   %4.2f   |    %5li   |\n",
	   universe[0]->averageheat, universe[0]->accept_freq,
	   universe[0]->swapped);
}


/// analyze old sumfile data
long
analyze_olddata (world_fmt * world,
		 option_fmt * options, data_fmt * data, long *outfilepos)
{
#ifndef MPI
  long locus;
#else

  long listofthings[6];
#endif

  long Gmax = 0;
  long pop;
  options->heating = FALSE;
  if (myID == MASTER)
    {
      unset_penalizer_function (TRUE);
      Gmax = read_savesum (world, options, data);

      data->popnames = (char **) mymalloc (sizeof (char *) * world->numpop);
      for (pop = 0; pop < world->numpop; pop++)
	{
	  data->popnames[pop] =
	    (char *) mycalloc (1, sizeof (char) * LINESIZE);
	  MYSNPRINTF (data->popnames[pop], LINESIZE, "Pop_%li ", pop + 1);
	}

#ifdef MPI

      MYMPIBCAST (data->geo, world->numpop2, MPI_DOUBLE, MASTER, comm_world);
      MYMPIBCAST (data->lgeo, world->numpop2, MPI_DOUBLE, MASTER, comm_world);
    }
  else
    {
      MYMPIBCAST (listofthings, 6, MPI_LONG, MASTER, comm_world);
      //      printf("listofthings received\n");
      //    fflush(stdout);
      world->loci = listofthings[0];
      world->numpop = listofthings[1];
      world->numpop2 = listofthings[2];
      options->replicate = (boolean) listofthings[3];
      options->replicatenum = listofthings[4];
      Gmax = listofthings[5];
      data->numpop = world->numpop;
      data->geo = (MYREAL *) mycalloc (1, sizeof (MYREAL) * world->numpop2);
      data->lgeo = (MYREAL *) mycalloc (1, sizeof (MYREAL) * world->numpop2);

      MYMPIBCAST (data->geo, world->numpop2, MPI_DOUBLE, MASTER, comm_world);
      MYMPIBCAST (data->lgeo, world->numpop2, MPI_DOUBLE, MASTER, comm_world);
      options->muloci = data->loci = world->loci;
      init_world (world, data, options);
#endif

    }
#ifdef MPI
  mpi_broadcast_results (world, world->loci,
			 pack_sumfile_buffer, unpack_sumfile_buffer);
#endif

  synchronize_param (world, options);
  if (options->plot)
    {
      options->plotnow = TRUE;
      world->options->plotnow = TRUE;
    }
  world->locus = 0;
  if (myID == MASTER)
    {
      *outfilepos = print_title (world, options);
      print_options (stdout, world, options, data);
      print_options (world->outfile, world, options, data);
#ifdef MPI

      mpi_runloci_master (world->loci, world->who, world);
#else

      for (locus = 0; locus < world->loci; locus++)
	{
	  if (options->replicate && options->replicatenum > 0)
	    {
	      world->locus = locus;
	      world->repkind = MULTIPLERUN;
#ifdef LONGSUM
	      change_longsum_times (world);	//multi run replicates
#endif	       /*LONGSUM*/
	      (void) estimateParameter (options->replicatenum, Gmax, world,
					options, world->cov[locus], 1, 'l',
					SINGLELOCUS, world->repkind);
	    }
	}
#endif

    }
#ifdef MPI
  else
    {
      assignloci_worker (world);
    }
#endif
  return Gmax;
}

/// generate profile likelihood tables
void
profile_tables (option_fmt * options, world_fmt * world, long *gmaxptr)
{
  long len;
#ifndef SLOWNET

  long i;
#endif

  if (options->profile != NONE)
    {
      world->buffer =
	(char *) myrealloc (world->buffer, SUPERLINESIZE * sizeof (char));
      if (options->printprofsummary)
	allocate_profile_percentiles (world);
#ifdef HAVE_STRFTIME

      time (&world->starttime);
#endif

      len = world->numpop2 + (world->options->gamma ? 1 : 0);
#ifdef LONGSUM

      len += world->numpop * 3;
#endif /*LONGSUM*/
#ifdef SLOWNET
	if (!options->readsum)
	mpi_broadcast_results (world, world->loci,
			       pack_sumfile_buffer, unpack_sumfile_buffer);
      mpi_broadcast_results (world,
			     world->loci + ((world->loci == 1) ? 0 : 1),
			     pack_result_buffer, unpack_result_buffer);
      //fprintf(stdout,"%i >>>>>>>>> wait for all at barrier in profiles_tables [main.c:1053]",myID);
      MYMPIBARRIER (comm_world);
      if (myID == MASTER)
	{
	  print_profile_title (world);
	  mpi_profiles_master (world, len, world->profilewho);
	}
      else
	mpi_profiles_worker (world, gmaxptr);
#else
	print_profile_title (world);

      for (i = 0; i < len; i++)
	{
	  (void) print_profile_likelihood_driver (i, world, gmaxptr);
	  FPRINTF (world->outfile, "%s\n\n", world->buffer);
	  world->buffer[0] = '\0';
	}
#endif
      if (myID == MASTER && options->printprofsummary)
	{
	  print_profile_percentile (world);
	  // print profile summary
	  FPRINTF (world->outfile, "%s\n\n", world->buffer);
	  world->buffer[0] = '\0';
	  destroy_profile_percentiles (world);
	}
    }
  //free(world->buffer);
}

/// finish up mac files \todo remove this function
void
finish_mac (option_fmt * options, data_fmt * data)
{
#ifdef MAC
#if 0
  fixmacfile (options->outfilename);
  if (options->plotmethod == PLOTALL)
    fixmacfile (options->mathfilename);
  if (options->treeprint)
    fixmacfile (options->treefilename);
  if (options->parmfile)
    fixmacfile ("parmfile");
  if (data->sumfile)
    fixmacfile (options->sumfilename);
#endif
#endif
}

/// \brief setup a locus
/// 
/// Set up a the structures for a locus. Run-Parameters are copied into the 
/// major structure WORLD from options and data. a first genealogy is build
/// adjusted to the starting parameters (changing times of nodes) and a first
/// conditional likelihood is calculated for all nodes, also the tree-parallele
/// time structure is generated.
/// Several run controllers are checked: replicate, datatype
/// \retvalue 0 { failed to setup structure for locus because there is no data }
/// \retvalue 1 { succeeded to setup locus-structure } 
int
setup_locus (long locus, world_fmt * world, option_fmt * options,
	     data_fmt * data)
{
  world->locus = locus;
  set_param (world, data, options, locus);
  if (world->options->progress)
    print_menu_locus (stdout, world, locus);
  if (world->options->writelog)
    print_menu_locus (world->options->logfile, world, locus);
  world->start = TRUE;
  buildtree (world, options, data, locus);
  if (world->replicate == 0)
    {
      if (strchr (SNPTYPES, options->datatype))
	{
	  if (options->datatype == 'u')
	    world->data->seq->endsite *= (data->seq->addon + 1);
	  world->data->seq->endsite += data->seq->addon;
	}
    }
  if (!options->replicate
      || (options->replicate && options->replicatenum == 0)
      || (options->replicate && world->replicate == options->replicatenum))
    {
      if (data->skiploci[locus])	/* skip loci with no tips */
	return 0;
    }
  create_treetimelist (world, &world->treetimes, locus);
  fix_times (world, options);
  first_smooth (world, locus);
  world->likelihood[0] = treelikelihood (world);
  world->allikemax = -DBL_MAX;	/* for best tree option */
#ifdef UEP
  if (options->uep)
    {
      world->treelen = 0.0;
      if (options->ueprate > 0.0)
	calc_treelength (world->root->next->back, &world->treelen);
      //world->ueplikelihood = ueplikelihood(world);
      //world->likelihood[0] += world->ueplikelihood;
    }
#endif
  return 1;
}

/// condense a single genealogy into a sufficient statistic for the maximization phase
void
condense_time (world_fmt * world, long *step, long *j, MYREAL * accepted,
	       long *G, long *steps, long oldsteps, long rep)
{

#ifdef BAYESUPDATE
  if (world->in_last_chain && world->options->bayes_infer	/*&& 
										     (((1+world->bayes->count) % ((long) world->options->updateratio)) == 0) *//**accepted*/
      )
    bayes_save (world);
#endif

  world->rep = rep;
  if (*step == 0)
    {
      copy_time (world, world->treetimes, FIRSTSTEP, 0, world->numpop, rep);
      *accepted += *j;
    }
  else
    {
      copy_time (world, world->treetimes, *G, *G + (long) (*j > 0),
		 world->numpop, rep);
      if (*step < *steps)
	{
	  //handled in advance_world * G += (long) (*j > 0);
	  *accepted += *j;
	}
    }
  if (*step >= oldsteps - 1 && world->options->movingsteps)
    {
      if (((MYREAL) (*G + 1)) < world->options->acceptfreq * oldsteps)
	{
	  (*steps)++;
	}
    }
}


/// set type of replication scheme
long
set_repkind (option_fmt * options)
{
  if (options->replicate)
    {
      if (options->replicatenum == 0)
	return MULTIPLECHAIN;
      else
	return MULTIPLERUN;
    }
  return SINGLECHAIN;
}

/// intialize heating scheme
void
heating_init (world_fmt ** universe, int usize, data_fmt * data,
	      option_fmt * options)
{
  long chain;
  char tmp[20];
  for (chain = 1; chain < usize; ++chain)
    {
      MYSNPRINTF (tmp, 20, "%10.5f", options->heat[chain]);
      create_world (&universe[chain], data->loci);
      init_world (universe[chain], data, options);
    }
}

/// prepare for heating scheme: Step I
void
heating_prepare (world_fmt ** universe, int usize,
		 option_fmt * options, data_fmt * data, long rep)
{
  long chain;
  EARTH->heat = options->heat[0];
  for (chain = 1; chain < usize; ++chain)
    klone (EARTH, universe[chain], options, data, options->heat[chain]);
}

/// prepare for heating scheme: Step I [this function failed in the parallel run, unused for now, replaced by above]
void
heating_prepare_old (world_fmt ** universe, int usize,
		     option_fmt * options, data_fmt * data, long rep)
{
  long chain;
  EARTH->heat = options->heat[0];
  if (rep == 0)
    {
      for (chain = 1; chain < usize; ++chain)
	klone (EARTH, universe[chain], options, data, options->heat[chain]);
    }
  else
    {
      for (chain = 1; chain < usize; ++chain)
	klone_part (EARTH, universe[chain], options, data,
		    options->heat[chain]);
    }
}

/// prepare for heating scheme: Step II
void
heating_prepare2 (world_fmt ** universe, int usize)
{
  long chain;
  universe[0]->averageheat = 1.0;
  for (chain = 1; chain < usize; ++chain)
    {
      universe[chain]->G = 0;
      universe[chain]->averageheat = universe[chain]->options->adaptiveheat ?
	0.0 : 1. / universe[chain]->heat;
    }
  for (chain = 1; chain < usize; ++chain)
    {
      clone_polish (EARTH, universe[chain]);
/* not needed ?
#ifdef BAYESUPDATE
        if (universe[chain]->options->bayes_infer)
		{
            universe[chain]->bayesaccept = bayes_update(universe[chain]);
			if(universe[chain]->bayesaccept == -1) //either do tree updates or parameter updates
			{
				(void) tree_update(universe[chain], 0);
			}
		}
		else
		{
			(void) tree_update(universe[chain], 0);
		}
#else
		(void) tree_update(universe[chain], 0);
#endif
		*/
    }
}

/// \brief generates replicate number
///
/// generates replicate number
/// \rtval 0 no replicate
/// \rtval rep number of rpelicates
long
replicate_number (option_fmt * options, long chain, char type, long replicate)
{
  long rep = 0;
  if (options->replicate && options->replicatenum > 0)
    rep = replicate;
  else
    {
      if (type == 'l' && options->replicate)
	rep = chain;
      else
	rep = 0;
    }
  return rep;
}

/// generate genealogies for a single locus
/// \callgraph
void
run_locus (world_fmt ** universe, int usize, option_fmt * options,
	   data_fmt * data, tpool_t * localheating_pool, long maxreplicate,
	   long locus, long *treefilepos, long *Gmax)
{
  long i;
  long replicate;
  for (replicate = 0; replicate < maxreplicate; replicate++)
    run_replicate (locus, replicate, universe, options, data,
		   localheating_pool, usize, treefilepos, Gmax);

#ifdef UEP
  if (options->uep)
    show_uep_store (EARTH);
#endif
  if (options->replicate && options->replicatenum > 0)
    {
      EARTH->repkind = MULTIPLERUN;
#ifdef BAYESUPDATE
      if (options->bayes_infer)
	{
	  calculate_credibility_interval (EARTH, locus);
	  return;
	}
#endif
#ifdef LONGSUM
      change_longsum_times (EARTH);
#endif /*LONGSUM*/
	// over multiple replicates if present or single locus
	(void) estimateParameter (options->replicatenum, *Gmax, EARTH,
				  options, EARTH->cov[locus],
				  options->lchains, /*type */ 'l',
				  SINGLELOCUS, EARTH->repkind);
    }
  else
    {
#ifdef BAYESUPDATE
      if (options->bayes_infer)
	{
	  calculate_credibility_interval (EARTH, locus);
//            return;
	}
#endif
    }
  //=======
  if (options->heating)
    {
      for (i = 0; i < options->heated_chains; i++)
	{
	  free_tree (universe[i]->root, universe[i]);
	  //   free_timevector(universe[i]->treetimes);
	}
    }
  else
    {
      free_tree (EARTH->root, EARTH);
      //  free_timevector(EARTH->treetimes);
    }
  //=======
#ifdef BAYESUPDATE
  bayes_reset (universe[0]);
#endif

}

/// \brief updates the tree and records acceptance
///
/// updates the tree and records acceptance
/// when in bayesian mode change between changing the tree and the parameters
void
run_one_update (world_fmt * world)
{
#ifdef BAYESUPDATE
  long count = 0;
  if (world->options->bayes_infer)
    {
      world->bayesaccept = bayes_update (world);
      if (world->bayesaccept == -1)	//either do tree updates or parameter updates
	{
	  world->accept += (count = tree_update (world, world->G));
	  world->bayes->accept[world->numpop2] += count;	//permanent recorder for the tree acceptance, 
	  // for some obscure reason the world->accept gets lost, most likely every cycle;
	  world->bayes->trials[world->numpop2] += 1;	// counter for trees tried 
	}
    }
  else
    {
      world->accept += tree_update (world, world->G);
    }
#else
  world->accept += tree_update (world, world->G);
#endif
}

/// \brief updates all trees, controls updates and heating
///
/// controls updates and temperatures (threaded or unthreaded)
/// updates the tree and/or parameters
/// \callgraph
void
run_updates (world_fmt ** universe,
	     int usize,
	     option_fmt * options,
	     tpool_t * localheating_pool,
	     long inc, long increment, long step, long steps)
{
#ifndef  PTHREADS
  long ii;
#endif
  if (options->heating)
    {
#ifdef PTHREADS			/*using threads and running on an SMP machine */
      fill_tpool (*localheating_pool, universe, usize);
      tpool_synchronize (*localheating_pool, usize);
#else /*heating but not using threads */
      for (ii = 0; ii < options->heated_chains; ii++)
	{
	  run_one_update (universe[ii]);
	}
#endif /* end of not-using threads */
      heated_swap (universe, EARTH->options);
      if (options->adaptiveheat)
	adjust_temperatures (universe, options->heated_chains,
			     inc /*rement */  + step * increment,
			     steps * increment);
    }
  else				/* no heating */
    {
      run_one_update (EARTH);
    }
#ifdef UEP
  if (options->uep && EARTH->in_last_chain)
    update_uepanc (EARTH);
#endif
}

/// generates trees over the interval increments
void
run_increments (world_fmt ** universe,
		long usize,
		option_fmt * options,
		tpool_t * localheating_pool,
		long increment, long step, long steps)
{
  long i;
  for (i = 0; i < increment; i++)
    {
      EARTH->actualinc = i;
      run_updates (universe, usize, options,
		   localheating_pool, i, increment, step, steps);

      if (EARTH->likelihood[EARTH->G] > EARTH->maxdatallike)
	EARTH->maxdatallike = EARTH->likelihood[EARTH->G];
    }
}

/// \brief run a chain
/// run a chain
void
run_steps (world_fmt ** universe,
	   long usize,
	   option_fmt * options,
	   tpool_t * localheating_pool, long increment, long steps, long rep)
{
  long step;
  long oldsteps = steps;
  long ii;
  for (step = 0; step < steps + 1; step++)
    {
      EARTH->increment = increment;
      run_increments (universe, usize, options, localheating_pool,
		      increment, step, steps);
#ifdef UEP
      store_uep (EARTH);
#endif
      condense_time (EARTH, &step, &EARTH->accept,
		     &EARTH->accept_freq, &EARTH->G, &steps, oldsteps, rep);
      if (step > 0)
	{
	  advance_clone_like (EARTH, EARTH->accept, &EARTH->G);
	  EARTH->accept = 0L;
	}
      // if we do heating
      if (EARTH->options->heating)
	{
	  for (ii = 1; ii < EARTH->options->heated_chains; ++ii)
	    {
	      universe[ii]->accept_freq += universe[ii]->accept;
	      advance_clone_like (universe[ii],
				  universe[ii]->accept, &(universe[ii])->G);
	      universe[ii]->accept = 0L;
	    }
	}
    }
}

/// run all chains for short and then for long
void
run_chains (world_fmt ** universe,
	    long usize,
	    option_fmt * options,
	    tpool_t * localheating_pool,
	    long replicate,
	    long chains,
	    char type,
	    long increment, long oldsteps, long *treefilepos, long *Gmax)
{
  long steps;
  long chain;
  long rep;
  long locus = EARTH->locus;
  long kind;
  long pluschain = 0;
  for (chain = 0;
       chain < chains || (type == 'l' && chain >= chains
			  && EARTH->param_like > options->lcepsilon); chain++)
    {
      if (options->heating)
	MERKUR->swapped = VENUS->swapped = EARTH->swapped = 0;
      rep = replicate_number (options, chain, type, replicate);
      precalc_world (EARTH);
      polish_world (EARTH);
      burnin_chain (EARTH);
      EARTH->G = 0;
      EARTH->accept_freq = 0.;
      steps = oldsteps;
      EARTH->maxdatallike = EARTH->likelihood[0];
      if (options->heating)
	heating_prepare2 (universe, usize);

      set_penalizer (chain, chains, type, universe);
      // run steps, run increments, run updaters 
      run_steps (universe, usize, options, localheating_pool,
		 increment, steps, rep);

      decide_plot (EARTH->options, chain, chains, type);
      // prepare for parameter estimation, precompute and copy
      memcpy (EARTH->param00, EARTH->param0,
	      sizeof (MYREAL) * EARTH->numpop2);
      memcpy (EARTH->atl[rep][locus].param0, EARTH->param00,
	      sizeof (MYREAL) * EARTH->numpop2);
      log_param0 (EARTH->atl[rep][locus].param0,
		  EARTH->atl[rep][locus].lparam0, EARTH->numpop2);
      copies2lcopies (&EARTH->atl[rep][locus]);
      // start parameter estimation for 
      // single chain, single replicate
      EARTH->repkind = SINGLECHAIN;
      kind = SINGLELOCUS;
      *Gmax = (EARTH->G > *Gmax) ? EARTH->G + 1 : (*Gmax);
#ifdef LONGSUM
      change_longsum_times (EARTH);
#endif /*LONGSUM*/
      (void) estimateParameter (rep, *Gmax, EARTH, options,
				EARTH->cov[locus], chain, type, kind,
				EARTH->repkind);
      if (EARTH->options->heating)
	print_heating_progress (universe, EARTH->options, increment * steps);
      else
	print_progress (EARTH->options, EARTH, rep,
			increment * steps, (long) EARTH->accept_freq);
      // cleanup and prepare next
      prepare_next_chain (universe, EARTH->options, type, chain,
			  &chains, &pluschain, locus, replicate);
    }				//end chains
}

/// set penalizing distance for jumps in the maximizer on or off
void
set_penalizer (long chain, long chains, char type, world_fmt ** universe)
{
  if (type == 'l')
    {
      if (chain >= chains - 1)
	{
	  EARTH->in_last_chain = TRUE;
	  unset_penalizer_function (TRUE);
	}
    }
  else
    {
      EARTH->in_last_chain = FALSE;
      unset_penalizer_function (FALSE);
    }
}


/// \brief run a replicate
///
/// run a replicate
/// \callgraph
void
run_replicate (long locus,
	       long replicate,
	       world_fmt ** universe,
	       option_fmt * options,
	       data_fmt * data,
	       tpool_t * localheating_pool,
	       int usize, long *treefilepos, long *Gmax)
{
  long chains = 0;
  char type = 's';
  long pluschain;
  long runs;
  long increment = 1;
  long steps;
  long oldsteps = 0;
  /* loop over independent replicates----------------------- */
  /* but make sure that we get single chain estimates, too   */
  EARTH->locus = locus;
  EARTH->repkind = SINGLECHAIN;
  EARTH->replicate = replicate;
  if (setup_locus (locus, EARTH, options, data) == 0)
    return;
  if (options->heating)
    heating_prepare (universe, usize, options, data, replicate);
  type = 's';
  runs = 1;
  // rep = 0;
  pluschain = options->pluschain;
  //see obscure options
  /* short and long chains ----------------------------- */
  set_bounds (&increment, &oldsteps, &chains, options, type);
  steps = oldsteps;
  EARTH->increment = increment;
  while (runs-- >= 0)
    {
      if (myID == MASTER && type == 's')
	{
	  print_menu_chain (type, FIRSTCHAIN, oldsteps, EARTH,
			    options, replicate);
	  if (EARTH->options->treeprint == ALL)
	    print_tree (EARTH, 0, treefilepos);
	}
      EARTH->chains = chains;
      /* loop over chains--------------------------- */
      run_chains (universe, usize, options, localheating_pool, replicate, chains, type, increment, oldsteps, treefilepos, Gmax);	//PB 020203
      change_chaintype (locus, &type, EARTH, &increment, &oldsteps,
			&chains, options);
      /* evaluate multiple long chain estimates */
      if (runs < 0 && options->replicate && options->replicatenum == 0)
	{
	  EARTH->repkind = MULTIPLECHAIN;
#ifdef LONGSUM
	  change_longsum_times (EARTH);
#endif	   /*LONGSUM*/
	  (void) estimateParameter (replicate, *Gmax, EARTH, options,
				    EARTH->cov[locus], chains, type,
				    SINGLELOCUS, EARTH->repkind);
	}
    }				//end runs
}

#ifdef LONGSUM
/// put timepoints into the total tree to allow for bottlenecks and growth 
long
find_chaincutoffs (MYREAL * treetime, world_fmt * world, long locus, long r)
{

  long j;
  long T;
  MYREAL totaltime = 0.;
  long copies = 0;
  tarchive_fmt *atl;
  atl = world->atl[r][locus].tl;
  for (j = 0; j < world->atl[r][locus].T; j++)
    {
      T = atl[j].longsumlen - 1;
      copies += atl[j].copies;
      totaltime = atl[j].longsum[T].eventtime / 3.;
      treetime[0] += totaltime;
      treetime[1] += totaltime + totaltime;
      treetime[2] += totaltime + totaltime + totaltime;
    }
  return copies;
}

/// find the cutoffs for the locus, averaging over multiple replicates
void
find_loci_cutoffs (MYREAL * treetime, world_fmt * world)
{
  long r;
  long locus;
  long count = 0;
  long repstart = 0;
  long repstop = 1;
  memset (treetime, 0, sizeof (MYREAL) * 3);
  set_replicates (world, world->repkind, world->rep, &repstart, &repstop);

  for (locus = 0; locus < world->loci; locus++)
    {
      for (r = repstart; r < repstop; r++)
	{
	  count += find_chaincutoffs (treetime, world, locus, r);
	}
    }
  treetime[0] /= (MYREAL) count;
  treetime[1] /= (MYREAL) count;
  treetime[2] /= (MYREAL) count;
}


/// find the cutoffs avaergin over all replicates 
void
find_replicate_cutoffs (MYREAL * treetime, world_fmt * world)
{
  long r;
  long count = 0;
  long repstart = 0;
  long repstop = 1;
  memset (treetime, 0, sizeof (MYREAL) * 3);
  set_replicates (world, world->repkind, world->rep, &repstart, &repstop);

  for (r = repstart; r < repstop; r++)
    {
      count += find_chaincutoffs (treetime, world, world->locus, r);
    }
  treetime[0] /= (MYREAL) count;
  treetime[1] /= (MYREAL) count;
  treetime[2] /= (MYREAL) count;
}


/// change the cutoff times
void
change_longsum_times (world_fmt * world)
{
  long i;
  long numpop3 = 3 * world->numpop;
  long numpop6 = 2 * numpop3;
  MYREAL *treetime;
  treetime = (MYREAL *) mycalloc (3, sizeof (MYREAL));	// we have 3 classes
  if (world->locus < world->loci)
    find_replicate_cutoffs (treetime, world);
  else
    find_loci_cutoffs (treetime, world);
  for (i = numpop3; i < numpop6; i += 3)
    {
      world->flucrates[i] = treetime[0];
      world->flucrates[i + 1] = treetime[1];
      world->flucrates[i + 2] = treetime[2];
    }
  free (treetime);
}

#endif /*LONGSUM*/
