/*
 *  pretty.c
 *  migrate-n
 *
 *  Created by Peter Beerli on 7/25/05.
 *  Copyright 2005 Peter Beerli. All rights reserved.
 *
 */
#ifdef PRETTY
#include "pretty.h"
#include "data.h"
#include "mig-histogram.h"
// test
#include <stdarg.h>

#define LINESTRETCH 15

extern pdf_doc doc;
extern pdf_page page;
extern pdf_contents canvas;
extern int page_counter;
extern char pdf_pagetitle[LINESIZE];
extern float page_height;
extern float page_width;

int pdf_print_header(char *title);
void pdf_print_contents_at(float x, float y, char *title);
void pdf_draw_line(float xs, float ys, float xe, float ye);
void pdf_migrate_logo(float x, float y, float stretch);
void pdf_migrate_logo_lines(float x, float y, float stretch);
float pdf_advance(float *page_height);
void pdf_print_seqdata (float *page_height, float margin, world_fmt * world, data_fmt * data, option_fmt * options);
void pdf_printf_next(float x, float *y, char string[], ...);
void pdf_printf_ralign(float rx, float y, char string[], ...);
void pdf_printf(float x, float y, char string[], ...);


///
/// Draw a simple line from x0,y0 to x0/y0, this function does not change
/// thickness or line type
void pdf_draw_line(float xs, float ys, float xe, float ye)
{
    pdf_contents_move_to(canvas, xs, ys);
    pdf_contents_line_to(canvas, xe, ye);
    pdf_contents_stroke(canvas);
}

///
/// draw a rectangle [from hary examples]
void draw_rect(pdf_contents canvas, float x, float y, const char* label)
{
    pdf_contents_set_font_and_size(canvas, "Helvetica", 10);
    pdf_contents_begin_text(canvas);
    pdf_contents_move_text_pos(canvas, x, y - 10);
    pdf_contents_show_text(canvas, label);
    pdf_contents_end_text(canvas);
    
    pdf_contents_rectangle(canvas, x, y - 40, 220, 25);
}

#define HORIZONTAL 0
#define VERTICAL   1 
///
/// plot a single tick with label at location x/y.
/// \param xs  x coordinate in true paper coordinates (points)
/// \param xy  y coordinate in true paper coordinates (points)
/// \param orientation either HORIZONTAL or VERTICAL, on HORIZONTAL=0 is checked
/// \param ticklength length of a the tick to draw
/// \param value this is a float value to put into the label
/// \param digits number of decimal digits
void   pdf_draw_tick(float xs, float ys, int orientation, float ticklength, float value, int digits)
{
    float w;
    float h;
    float tl = ticklength;
    char *title;
    title = (char *) mycalloc(100,sizeof(char));
    sprintf(title,"%.*f",digits, value);
    w = pdf_contents_get_text_width(canvas, title, NULL, NULL);
    h = pdf_contents_get_font_size(canvas);
    if(orientation==HORIZONTAL)
      {
        pdf_print_contents_at(xs-w/2, ys-tl-h, title);
        pdf_draw_line(xs,ys, xs , ys - tl);
      }
    else
      {
        pdf_print_contents_at(xs-tl-w, ys-h/2, title);
        pdf_draw_line(xs,ys, xs-tl , ys);
      }
    free(title);
}

/// \TODO implement a pretty print
/// find pretty values for tick labels
/// using heuristics
float prettytick(float value, int searchdirection)
{
    return value;
}

///
/// create a vertical or horizontal axis
/// \param up is either HORIZONTAL=1 or VERTICAL
/// \param xmin  start value
/// \param xmax  end value
/// \param ticks number of ticks
/// \param lx    x paper coordinate to start axis
/// \param ly    y paper coordinate to start axis
/// \param length  length on paper in paper coordinate system
void  pdf_create_axes(int up, float xmin, float xmax, long ticks, float lx, float ly, float length)
{
    long i;
    int digits = 3;
    float value = 0.0;
    float plotdelta = length/ticks;
    float realdelta = (xmax - xmin)/ticks;
    
    // pretty printing of labels
    // find pretty lowest value that is either the same as xmin or then higher
    // newxmin = prettytick(xmin,SEARCHUP);
    // find pretty upper value 
    // newnexmin = prettytick(xmax,SEARCHDOWN);
    // 
    if(up!=HORIZONTAL)
      {
        pdf_draw_line(lx,ly, lx ,ly + length);
        for(i=0; i<ticks; i++)
          {
            value = xmin+i*realdelta;
            digits = 3;
            pdf_draw_tick(lx, ly+i*plotdelta, VERTICAL, 3, value, digits);
          }
      }
    else
      {
        pdf_draw_line(lx,ly, lx + length, ly);
        for(i=0; i<ticks; i++)
          {
            value = xmin+i*realdelta;
            digits = 3;
            pdf_draw_tick(lx+i*plotdelta, ly, HORIZONTAL, 3, value, digits);
          }
      }
}

void findmoments(MYREAL *vals, long n, float *p01, long *pos01, float *p99, long *pos99, float *mm, long *posmm)
{
    long i;
    float val;
    float total = 0. ;
    *mm = - (float) (LONG_MAX);
    *p01 = 0.;
    *p99 = 0.;
    *pos01 = 0;
    *pos99 = 0;
    *posmm = 0;
    for(i=0;i<n; i++)
      {
        val = (float) vals[i];
        total += val;
        if( *mm < val)
          {
            *mm = val;
            *posmm = i;
          }
      }
    val = 0.;
    for(i=0;i<n; i++)
      {
        val += (float) vals[i]/total;
        if(val < 0.01)
          {
            *p01 = (float) vals[i];
            *pos01 = i;
          }
        if(val < 0.99)
          {
            *p99 = (float) vals[i];
            *pos99 = i;
          }
      }
}
void findminmax(MYREAL *vals, long n, float *p00, long *pos00, float *p100, long *pos100)
{
    long i;
    float val;
    float total = 0. ;
    *p00 = 0.;
    *p100 = 0.;
    *pos00 = 0;
    *pos100 = 0;
    for(i=0;i<n; i++)
      {
        val = (float) vals[i];
        total += val;
      }
    val = 0.;
    for(i=0;i<n; i++)
      {
        val += (float) vals[i]/total;
        if(val < 0.000001)
          {
            *p00 = (float) vals[i];
            *pos00 = i;
          }
        if(val < 0.999999)
          {
            *p100 = (float) vals[i];
            *pos100 = i;
          }
      }
}

///
/// create a histogram from bincounts at a specific location and width and height
/// If binmax is -9999 then the maximum is reset to the 99% percentile.
/// If binmax is -999 then the maximum is reset to the 100% percentile
void pdf_histogram(MYREAL *binvals, char *set50, char *set95, long bins, float bindelta, float binmin, float binmax, float lx, float ly, float width, float height)
{
    long i;
    float total;
    float binvalsmax;
    long binvalspos;
    float p99;
    long pos99;
    float p100;
    long pos100;
    float p00;
    long pos00;
    float p01;
    long pos01;
    float x = 0.;
    float delta = width / bins;
    long numbins = bins;
    
    findmoments(binvals,bins,&p01,&pos01, &p99,
                &pos99, &binvalsmax, &binvalspos);
    
    total = 0. ;
    for(i=0;i<bins;i++)
      {
        total += binvals[i];
      }
    
    if(binmax < -9000)
      {
        binmax = binmin + pos99 * bindelta;
        delta = width/pos99;
        numbins = pos99;
      }
    else
      {
        if(binmax < -900)
          {
            findminmax(binvals,bins,&p00,&pos00, &p100,
                       &pos100);
            binmax = binmin + pos100 * bindelta;
            delta = width/pos100;
            numbins = pos100;
          }
      }
    pdf_create_axes(VERTICAL, 0, binvalsmax/total, 5, lx, ly, height);
    pdf_create_axes(HORIZONTAL, binmin , binmax, 5, lx, ly, width);
    pdf_contents_set_line_width(canvas, delta);
    
    x = delta/1.95;
    for(i=0;i<numbins;i++)
      {
        if(set50[i] == '1')
            pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0.1, 0.1, 0.1));
        else
          {
            if(set95[i] == '1')
                pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0.4, 0.4, 0.4));
            else
                pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0.8, 0.8, 0.8));
          }
        pdf_draw_line(lx+x,ly+0,lx+x,ly + binvals[i]/binvalsmax * height);
        //fprintf(stderr,"%f %f %f\n",lx+x,ly+ binvals[i]/binvalsmax * height, binvals[i]);
        x += delta;
      }
    pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0, 0, 0));
    pdf_contents_set_line_width(canvas, 1);
}

int pdf_init()
{
    pdf_type1_fontdef font1_def;
    pdf_type1_fontdef font2_def;
    pdf_type1_fontdef font3_def;
    pdf_type1_fontdef font4_def;
    
    /* Create a new PDF document. */
    doc = pdf_doc_new();
    pdf_doc_new_doc(doc);
    
    /* Add Helvetica Font. */
    font1_def = pdf_create_type1_fontdef(PDF_FONT_HELVETICA);
    pdf_doc_add_type1font(doc, font1_def, NULL, NULL);
    /* Add Helvetica-Oblique Font. */
    font2_def = pdf_create_type1_fontdef(PDF_FONT_HELVETICA_OBLIQUE);
    pdf_doc_add_type1font(doc, font2_def, NULL, NULL);
    /* Add Symbol Font. */
    font3_def = pdf_create_type1_fontdef(PDF_FONT_SYMBOL);
    pdf_doc_add_type1font(doc, font3_def, NULL, NULL);
    /* Add Courier Font. */
    font4_def = pdf_create_type1_fontdef(PDF_FONT_COURIRE);
    pdf_doc_add_type1font(doc, font4_def, NULL, NULL);
    return 0;
}

///
/// generate a new PDF page with a border rectangle and with page numbers and
/// title in top right corner and impressum at bottom left
int pdf_new_page(char *title)
{
    char stemp[LINESIZE];
    if(strlen(title)>1)
        strncpy(pdf_pagetitle,title,80);
    /* Add a page to the document. */
    page = pdf_doc_add_page(doc);
    page_counter += 1;
    /* Get the canvas object of the page. */
    canvas = pdf_page_get_canvas(page);
    
    /*Set current font to "Helvetica" and set the size to 11. */
    pdf_contents_set_font_and_size(canvas, "Helvetica", 11);
    
    pdf_contents_set_line_width(canvas, 1);
    /* draw page rectangle 50pt from border*/
    pdf_contents_rectangle(canvas, 50, 50, 
                           pdf_contents_get_width(canvas) - 100, 
                           pdf_contents_get_height(canvas) - 110);
    pdf_contents_stroke(canvas);
    /* print the title of the analysis*/
    pdf_print_header(pdf_pagetitle);
    /* print the impressum at the bottome*/
    sprintf(stemp,"Migrate %s: (http://popgen.csit.fsu.edu)",MIGRATEVERSION);
    pdf_contents_set_font_and_size(canvas, "Helvetica", 6);
    pdf_print_contents_at(50, 42, stemp);
    pdf_contents_set_font_and_size(canvas, "Helvetica", 11);
    return 0;
}


///
/// print title on every page with page number
int pdf_print_header(char *title)
{
    float w;
    float page_height;
    float page_width;
    char *fulltitle;
    
    fulltitle = (char*) mycalloc(255,sizeof(char));
    /* Print the title of the page (with positioning center). */
    sprintf(fulltitle,"%s -- %i",title,page_counter);
    //printf("%s\n",fulltitle);
    w = pdf_contents_get_text_width(canvas, fulltitle, NULL, NULL);
    /* Start to print text. */
    pdf_contents_begin_text(canvas);    
    /* Move the position of the text to center */
    page_height = pdf_contents_get_height(canvas);
    page_width = pdf_contents_get_width(canvas);
    pdf_contents_move_text_pos(canvas, (page_width - w - 50),
                               page_height - 50);
    
    /* Print title with pagenumber to rightadjusted */
    pdf_contents_show_text(canvas, fulltitle);    
    /* Finish to print text. */
    pdf_contents_end_text(canvas);
    
    free(fulltitle);
    
    return 0;
}

void pdf_print_contents_at(float x, float y, char *title)
{
    pdf_contents_begin_text(canvas);
    pdf_contents_move_text_pos(canvas, x, y);
    pdf_contents_show_text(canvas, title);
    pdf_contents_end_text(canvas);
    //fprintf(stderr,"(%f, %f) %s\n",x,y,title);
}

///
/// print title page
int pdf_print_title(char *title, float *orig_page_height, float *left_margin)
{
    float w;
    float page_height;
    float page_width;
    char newtitle[LINESIZE];
    /* Move the position of the text to center */
    page_height = pdf_contents_get_height(canvas);
    page_width = pdf_contents_get_width(canvas);
    /* Print the title of the page (with positioning center). */
    pdf_contents_set_font_and_size(canvas, "Helvetica-Oblique", 24);

    MYSNPRINTF(newtitle,(long) (page_width-110)/10, "%s",title);
    w = pdf_contents_get_text_width(canvas, newtitle, NULL, NULL);

    /* Start to print text. */
    pdf_contents_begin_text(canvas);    
    /* Print title centered */
    pdf_print_contents_at((page_width - w)/2, page_height - 100, newtitle);
    pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0, 0, 0));
    pdf_draw_line(50, page_height - 126, page_width-50, page_height - 126);
    pdf_contents_set_font_and_size(canvas, "Helvetica", 12);
    pdf_print_contents_at(55, page_height - 150, "MIGRATION RATE AND POPULATION SIZE ESTIMATION");  
    pdf_print_contents_at(55, page_height - 165, "using a Markov chain Monte Carlo approach");    
    pdf_printf(55, page_height - 180, "Migrate-n version %s [compare ASCII outputfile with this PDF file!!]",MIGRATEVERSION);    
    pdf_draw_line(50, page_height - 195, page_width-50, page_height - 195);
    
    // print migrate logo
    pdf_contents_set_line_width(canvas, 1);
    pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0, 0, 0));
    pdf_contents_set_rgb_fill(canvas, PdfRGBColor(0.99, 0, 0));
    pdf_migrate_logo(-1400, -1415, 100.0);
    pdf_contents_set_rgb_fill(canvas, PdfRGBColor(0, 0.1, 0.9));
    pdf_migrate_logo(-1364, -1415, 100.0);
    pdf_contents_set_line_width(canvas, 4);
    pdf_migrate_logo_lines(-1400, -1415, 100.0);
    /* Finish to print text. */
    
    *orig_page_height = page_height - 200;
    *left_margin = 55;
    return 0;
}

int pdf_write_file(option_fmt *options)
{
    pdf_doc_write_to_file(doc, options->pdfoutfilename);
    return 0;
}

///
/// print elements of Bayesian table for character variables
float pdf_print_line_element(float lx, float ly, float offset, char *title)
{
    float w=0;
    w = pdf_contents_get_text_width(canvas, title, NULL, NULL);
    if(offset>=0)
        pdf_print_contents_at(lx+offset-w, ly, title);
    else
        pdf_print_contents_at(lx-offset, ly, title);
    return w;
}

///
/// print elements of Bayesian table for float variables
float pdf_print_line_element2(float lx, float ly, float offset, float value, int fmt1, int fmt2)
{
    float w=0;
    char title[100];
    sprintf(title,"%*.*f",fmt1,fmt2,value);
    w = pdf_contents_get_text_width(canvas, title, NULL, NULL);
    if(offset>=0)
        pdf_print_contents_at(lx+offset-w, ly, title);
    else
        pdf_print_contents_at(lx-offset, ly, title);
    return w;
}


void pdf_print_line_theta(float lx, float ly, float offset, long j)
{
    char tempstring[100];
    char * thetatitle="Q";
    if(j < 0)
        sprintf(tempstring," ");
    else
        sprintf(tempstring,"%li",j+1);
    pdf_contents_set_font_and_size(canvas, "Symbol", 11);
    pdf_print_contents_at(lx-offset,ly,thetatitle);
    pdf_contents_set_font_and_size(canvas, "Symbol", 8);
    pdf_print_contents_at(lx-offset+10,ly-4,tempstring);
    pdf_contents_set_font_and_size(canvas, "Helvetica", 11);
}

///
/// pretty print the M values
void  pdf_print_line_mig(char *migtitle, float lx, float page_height, float offset, long frompop, long topop)
{
  float w;
    char tostring[100];
    char tempstring[100];
    if(topop < 0)
        sprintf(tostring,"+");
    else
        sprintf(tostring,"%li",topop+1);
    sprintf(tempstring,"%li->%s",frompop+1,tostring);
    pdf_contents_set_font_and_size(canvas, "Helvetica", 11);
    pdf_print_contents_at(lx-offset, page_height,migtitle);
    w = pdf_contents_get_text_width(canvas, migtitle, NULL, NULL);
    pdf_contents_set_font_and_size(canvas, "Helvetica", 8);
    pdf_print_contents_at(lx-offset+w,page_height-4,tempstring);
    pdf_contents_set_font_and_size(canvas, "Helvetica", 11);
}


///
/// Print Bayesian table header
/// and report the width of the text in the headerline
void pdf_print_bayestableheader(float page_height, float left_margin, float right_margin, float *offset)
{
    float lx = left_margin;
    float ly = page_height;
    
    pdf_print_line_element(lx, ly, offset[0], "Locus");
    pdf_print_line_element(lx, ly, offset[1], "Parameter");
    pdf_print_line_element(lx, ly, offset[2], "2.5%");
    pdf_print_line_element(lx, ly, offset[3], "25.0%");
    pdf_print_line_element(lx, ly, offset[4], "Mode");
    pdf_print_line_element(lx, ly, offset[5], "75.0%");
    pdf_print_line_element(lx, ly, offset[6], "97.5%");
    pdf_print_line_element(lx, ly, offset[7], "Median");
    pdf_print_line_element(lx, ly, offset[8], "Mean");
}

// Parameter        2.5%%      25.0%%   median    75.0%%   97.5%%     mode     mean\n"

///
/// print Bayesian table
void pdf_print_bayestable(world_fmt *world)
{
    int fmt1;
    int fmt2;
    
    long j;
    long size = world->numpop2;
    float lx;
    float *offset;
    bayes_fmt * bayes = world->bayes;
    bayeshistogram_fmt * hist;
    
    float page_height = pdf_contents_get_height(canvas);
    float page_width = pdf_contents_get_width(canvas);
    
    float left_margin = 60;
    float right_margin = page_width - 60;
    
    float w;
    
    char st[100];
    char title[100] = "Bayesian Analysis: Posterior distribution table";
    long frompop;
    long topop;
    long locus;
    long lozi = world->loci > 1 ? world->loci : 0;
    
    //column to to right-align the table columns
    offset = (float *) mycalloc(8,sizeof(float));
    offset[0] = -1; //left align
    offset[1] = -40;
    offset[2] = 135;
    offset[3] = 191;
    offset[4] = 247;
    offset[5] = 303;
    offset[6] = 359;
    offset[7] = 415;
    offset[8] = 471;
    pdf_new_page("");
    pdf_contents_set_font_and_size(canvas, "Helvetica-Oblique", 16);
    w = pdf_contents_get_text_width(canvas, title, NULL, NULL);
    /* Start to print text. */ 
    pdf_print_contents_at((page_width - w)/2, page_height - 100, title);
    pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0, 0, 0));
    pdf_draw_line(50, page_height - 126, page_width-50, page_height - 126);
    // reset font for table text
    // prepare coordinates for printint table
    pdf_contents_set_font_and_size(canvas, "Helvetica", 11);
    page_height -= 126;
    lx = left_margin;
    page_height -= 20;
    pdf_draw_line(left_margin, page_height, right_margin, page_height);
    pdf_advance(&page_height);
    pdf_print_bayestableheader(page_height, left_margin, right_margin, offset);
    pdf_advance(&page_height);
    pdf_draw_line(left_margin, page_height, right_margin, page_height);
    pdf_advance(&page_height);    
    for(locus=0; locus <= lozi; locus++)
      {
        hist = &bayes->histogram[locus];
        if(locus == world->loci)
            strcpy(st,"  All ");
        else
            sprintf(st,"%5li ",locus + 1);
        
        for(j=0; j< size; j++)
          {
            
            if(strchr("0c", world->options->custm2[j]))
                continue;
            pdf_print_line_element(lx, page_height, offset[0], st);
            if(j < world->numpop)
              {
                pdf_print_line_theta(lx, page_height, offset[1], j);
                fmt1 = 8;
                fmt2 = 5;
              }
            else
              {
                m2mm(j,world->numpop,&frompop, &topop);
                if(world->options->usem)
                  {
                    pdf_print_line_mig("M", lx, page_height, offset[1], frompop, topop);
                    fmt1 = 8;
                    if (strchr (SEQUENCETYPES, world->options->datatype))
                        fmt2 = 1;
                    else
                        fmt2 = 3;
                  }
                else
                  {
                    pdf_print_line_mig("xNm", lx, page_height, offset[1], frompop, topop);
                    fmt1 = 8;
                    fmt2 = 5;
                  }
              }
            // reset the to helvetica from symbol or small helvetica
            pdf_contents_set_font_and_size(canvas, "Helvetica", 11);
            pdf_print_line_element2(lx, page_height, offset[2], hist->cred95l[j],fmt1,fmt2);
            pdf_print_line_element2(lx, page_height, offset[3], hist->cred50l[j],fmt1,fmt2);
            pdf_print_line_element2(lx, page_height, offset[4], hist->modes[j],fmt1,fmt2);
            pdf_print_line_element2(lx, page_height, offset[5], hist->cred50u[j],fmt1,fmt2);
            pdf_print_line_element2(lx, page_height, offset[6], hist->cred95u[j],fmt1,fmt2);
            pdf_print_line_element2(lx, page_height, offset[7], hist->medians[j],fmt1,fmt2);
            pdf_print_line_element2(lx, page_height, offset[8], hist->means[j],fmt1,fmt2);
            pdf_advance(&page_height);
            if(page_height < 64)
              {
                pdf_new_page("");
                page_height = pdf_contents_get_height(canvas);
                page_width = pdf_contents_get_width(canvas);
                page_height -= 50;
                lx = left_margin;
                page_height -= 20;
                pdf_draw_line(left_margin, page_height, right_margin, page_height);
                page_height -= 20;
                pdf_print_bayestableheader(page_height, left_margin, right_margin, offset);
                page_height -= 20;
                pdf_draw_line(left_margin, page_height, right_margin, page_height);
                pdf_advance(&page_height);
              }
          }
        pdf_draw_line(left_margin, page_height, right_margin, page_height);
        pdf_advance(&page_height);
        
      }
    
    free(offset);
}

///
/// print profile tables

///
/// plot bayesian histograms

///
/// plot 2D likelihood surfaces

///
/// plot migration time histograms, assumes that the ascii_printer has already filled the 
/// plotfield table when used with boolean PRECALC (=True)
void
pdf_mig_histogram(histogram_fmt ** histogram,
                  plotfield_fmt ** plotfield, long loci, long numparams,
                  long bins, long *sum, MYREAL ***migtable,  boolean precalc, world_fmt *world)
{
    float left_margin = 55;
    float page_height;
    float page_width;
    float lx;
    float ly;
    long loc, i, j, z, zz;
    
    char *thetatitle = "Q";
    char *migt1 = "M";
    char *migt2 = "xNm";
    char *migtitle = world->options->usem ? migt1 : migt2;
    long msub = world->options->usem ?  -17 : -5;
    
    char tempstring[20];
    long frompop;
    long topop;
    
    MYREAL biggest = 0.;
    MYREAL *binning;
    MYREAL *binvec;
    MYREAL tempmin = DBL_MAX;
    MYREAL tempmax = -DBL_MAX;
    MYREAL begin = DBL_MAX;
    MYREAL end = -DBL_MAX;
    MYREAL delta;
    
    MYREAL mtime;
    char *set50;
    char *set95;
    long weight;
    char title[] = "Migration event histogram";
    float w;
    
    // add a new page so that we can print at least four histograms
    pdf_new_page("");
    pdf_contents_set_font_and_size(canvas, "Helvetica-Oblique", 16);
    w = pdf_contents_get_text_width(canvas, title, NULL, NULL);
    
    /* Start to print text. */ 
    page_height = pdf_contents_get_height(canvas);
    page_width = pdf_contents_get_width(canvas);
    pdf_print_contents_at((page_width - w)/2, page_height - 100, title);
    pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0, 0, 0));
    pdf_draw_line(50, page_height - 126, page_width-50, page_height - 126);
    pdf_contents_set_font_and_size(canvas, "Helvetica", 10);
    page_height -= 126;
    lx = 100;
    ly = page_height - 190;
    
    
    set50 = (char *) mycalloc(bins+1, sizeof(char));
    set95 = (char *) mycalloc(bins+1, sizeof(char));
    memset (set50, '0', bins);
    memset (set95, '0', bins);
    
    binning = (MYREAL *) mycalloc (bins+1, sizeof (MYREAL));
    binvec = (MYREAL *) mycalloc (bins+1, sizeof (MYREAL));
    
    for (loc = 0; loc < loci; loc++)
      {
        for (i = (world->options->mighist_all ? 0 : world->numpop); i < numparams; i++)
          {
            if (migtable[loc][i][2] == NOAVERAGE)
              {
                plotfield[loc][i].print = FALSE;
                continue;
              }
            minmax (&histogram[loc][i], &tempmin, &tempmax);
            if (tempmin < begin)
                begin = tempmin;
            if (tempmax > end)
                end = tempmax;
          }
      }
    delta = (end - begin) / bins;
    binning[0] = begin + 0.5 * delta;
    for (i = 1; i < bins; i++)
        binning[i] = delta + binning[i - 1];
    if(!precalc)
      {
        for (loc = 0; loc < loci; loc++)
          {
            
            for (i = (world->options->mighist_all ? 0 : world->numpop); i < numparams; i++)
              {
                if (migtable[loc][i][2] == NOAVERAGE)
                    continue;
                memset (binvec, 0, sizeof (MYREAL) * (bins+1));
                for (j = 0; j < histogram[loc][i].count; j++)
                  {
                    mtime = histogram[loc][i].time[j];
                    weight = histogram[loc][i].weight[j];
                    z = 0;
                    while (mtime > binning[z] && z < bins)
                        z++;
                    binvec[z] += weight;
                  }
                biggest = 0.;
                for (j = 0; j < bins; j++)
                  {
                    plotfield[loc][i].y[j] = (long) binvec[j];
                    plotfield[loc][i].yfreq[j] = binvec[j] = binvec[j] / sum[loc];
                    if (biggest < binvec[j])
                        biggest = binvec[j];
                  }
                for (j = 0; j < bins; j++)
                  {
                    for (zz = 0;
                         zz <
                         (long) (binvec[j] * plotfield[loc][i].ysize / biggest);
                         zz++)
                        plotfield[loc][i].data[j][zz] = '+';
                    plotfield[loc][i].data[j][zz] = '\0';
                  }
              }
          }
      }
    for (loc = 0; loc < loci; loc++)
      {
        if (loc == (loci - 1))
          {
            
            if (loci > 1)
                pdf_printf_next(left_margin, &page_height,"Over all loci");
            else
                pdf_printf_next(left_margin, &page_height,"Locus %li\n",loc + 1);
          }
        else
            pdf_printf_next(left_margin, &page_height,"Locus %li\n",loc + 1);
        
        for (i = (world->options->mighist_all ? 0 : world->numpop); i < numparams; i++)
          {
            if (plotfield[loc][i].print)
              {
                m2mm(i, world->numpop, &frompop, &topop);
                if(frompop==topop)
                  {
                    sprintf(tempstring,"%li",frompop+1);
                    pdf_contents_set_font_and_size(canvas, "Symbol", 12);
                    pdf_print_contents_at(lx-30,ly+125,thetatitle);
                    pdf_contents_set_font_and_size(canvas, "Symbol", 9);
                    pdf_print_contents_at(lx-20,ly+121,tempstring);
                  }
                else
                  {
                    sprintf(tempstring,"%li->%li",frompop+1,topop+1);
                    pdf_contents_set_font_and_size(canvas, "Helvetica", 12);
                    pdf_print_contents_at(lx-30,ly+125,migtitle);
                    pdf_contents_set_font_and_size(canvas, "Helvetica", 9);
                    pdf_print_contents_at(lx+msub,ly+121,tempstring);
                  }
                pdf_contents_set_font_and_size(canvas, "Helvetica", 10);
                pdf_histogram(plotfield[loc][i].yfreq, set50, set95, bins, delta, binning[0], -999, lx, ly, page_width - 55 - lx, 116);
                ly -= 190;
                page_height -= 190;
                if(page_height-50 < 190)
                  {
                    pdf_new_page("");
                    page_height = pdf_contents_get_height(canvas) - 10;
                    ly = page_height -190;
                  }
              }
          }
      }
}

void
pdf_print_mighist_table (world_fmt * world, MYREAL ***migtable,
                         long *total)
{
    float lx;
    float ly;
    float page_height;
    float page_width;
    float w;
    float left_margin = 55;
    float right_margin = 55;
    const     float offset[] = {-1, -40, 135, 191, 247, 303, 359, 415, 471};
    const     float roffset[] = {5, 50, 135, 191, 247, 303, 359, 415, 471};
    long loc, p1;
    long loci1 = world->loci == 1 ? 1 : world->loci + 1;
    char title[LINESIZE];
    sprintf (title, "Summary of %s events", 
             world->options->mighist_all ? "coalescence and migration" : "migration");
    pdf_new_page("");
    page_height = pdf_contents_get_height(canvas);
    page_width = pdf_contents_get_width(canvas);
    pdf_contents_set_font_and_size(canvas, "Helvetica-Oblique", 16);
    w = pdf_contents_get_text_width(canvas, title, NULL, NULL);
    /* Start to print text. */ 
    pdf_print_contents_at((page_width - w)/2, page_height - 100, title);
    pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0, 0, 0));
    pdf_draw_line(50, page_height - 126, page_width-50, page_height - 126);
    // reset font for table text
    // prepare coordinates for printint table
    pdf_contents_set_font_and_size(canvas, "Helvetica", 11);
    page_height -= 126;
    lx = 55;
    page_height -= 20;
    pdf_draw_line(left_margin, page_height, right_margin, page_height);
    pdf_advance(&page_height);
    
    for (loc = 0; loc < world->loci; loc++)
        total[world->loci] += total[loc];
    
    for (loc = 0; loc < loci1; loc++)
      {    /* Each locus + Summary */
        if (loc != world->loci)
            sprintf(title, "Locus %li", loc + 1);
        else
            sprintf(title, "Over all loci");
        pdf_print_contents_at(lx,page_height,title);
        ly = pdf_advance(&page_height);
        // table header
        pdf_draw_line(left_margin+5, page_height, right_margin+5, page_height);
        pdf_print_contents_at(lx+offset[0],page_height,title);
        ly = pdf_advance(&page_height);
        pdf_print_line_element(lx, ly, offset[0], "Population");
        pdf_print_line_element(lx, ly, offset[2], "Time");
        pdf_print_line_element(lx, ly, offset[5], "Frequency");
        ly = pdf_advance(&page_height);
        pdf_draw_line(left_margin+5+offset[2], page_height, left_margin+5+offset[4], page_height);
        ly = pdf_advance(&page_height);
        pdf_print_line_element(lx, ly, offset[0], "From");
        pdf_print_line_element(lx, ly, offset[1], "To");
        pdf_print_line_element(lx, ly, offset[2], "Average");
        pdf_print_line_element(lx, ly, offset[3], "Median");
        pdf_print_line_element(lx, ly, offset[4], "SE");
        ly = pdf_advance(&page_height);
        pdf_draw_line(left_margin+5+offset[2], page_height, left_margin+5+offset[4], page_height);
        ly = pdf_advance(&page_height);
        
        for (p1 = (world->options->mighist_all ? 0 : world->numpop); p1 < world->numpop2; p1++)
          {
            if (migtable[loc][p1][2] == NOAVERAGE)
              {
                pdf_printf(lx, ly, "%4li %4li    No migration event encountered\n",
                           (long) migtable[loc][p1][0] + 1,
                           (long) migtable[loc][p1][1] + 1);
              }
            else
              {
                pdf_printf_ralign(lx + roffset[0], ly,"%4li", (long) migtable[loc][p1][0] + 1);
                pdf_printf_ralign(lx + roffset[1], ly,"%4li", (long) migtable[loc][p1][1] + 1);
                pdf_printf_ralign(lx + roffset[2], ly,"%3.5f",migtable[loc][p1][2]);
                pdf_printf_ralign(lx + roffset[2], ly,"%3.5f",migtable[loc][p1][3]);
                pdf_printf_ralign(lx + roffset[2], ly,"%3.5f",migtable[loc][p1][4]);
                pdf_printf_ralign(lx + roffset[2], ly,"%3.5f",migtable[loc][p1][5] / total[loc]);
              }
            ly = pdf_advance(&page_height);            
          }
        pdf_draw_line(left_margin+5+offset[2], page_height, left_margin+5+offset[4], page_height);
      }
}


///
/// fill and stroke bezier curve
void pdf_fill_stroke(float x1, float y1, float x2, float y2, float x3, float y3, float x4, float y4)
{
    //  pdf_contents_set_line_width(canvas, 1.5);
    pdf_contents_move_to(canvas, x1, y1);
    pdf_contents_curve_to(canvas, x2, y2, x3, y3, x4, y4);
    pdf_contents_fill_stroke(canvas);
}

///
/// plot migrate logo
void pdf_migrate_logo(float x, float y, float stretch)
{
    pdf_contents_move_to(canvas, x + stretch * 18.705982, y + stretch * 21.16717);
    pdf_contents_curve_to(canvas, x + stretch * 18.705982, 
                          y + stretch * 21.234596, x + stretch * 18.779981,y + stretch * 21.289254,
                          x + stretch * 18.871266, y + stretch * 21.289254);
    pdf_contents_curve_to(canvas, x + stretch * 18.962551, 
                          y + stretch * 21.289254, x + stretch * 19.03655, y + stretch * 21.234596, 
                          x + stretch * 19.03655, y + stretch * 21.167171);
    pdf_contents_curve_to(canvas, x + stretch * 19.03655, 
                          y + stretch *  21.099745, x + stretch * 18.962551, y + stretch * 21.045087, 
                          x + stretch * 18.871266, y + stretch * 21.045087);
    pdf_contents_curve_to(canvas, x + stretch * 18.779981, 
                          y + stretch * 21.045087, x + stretch * 18.705982, y + stretch * 21.099745, 
                          x + stretch * 18.705982, y + stretch * 21.167171);
    pdf_contents_fill_stroke(canvas);
}

void pdf_migrate_logo_lines(float x, float y, float stretch)
{
    pdf_contents_move_to(canvas, x + stretch * 18.773599, y + stretch * 21.177612);
    pdf_contents_line_to(canvas, x + stretch * 18.773599, y + stretch * 20.997156);
    pdf_contents_line_to(canvas, x + stretch * 18.882535, y + stretch * 20.997156);
    pdf_contents_line_to(canvas, x + stretch * 18.882861, y + stretch * 21.177612);
    pdf_contents_stroke(canvas);
    pdf_contents_move_to(canvas, x + stretch * 18.979539,  y + stretch * 21.177612);
    pdf_contents_line_to(canvas, x + stretch * 18.980202,  y + stretch * 20.948318);
    pdf_contents_line_to(canvas, x + stretch * 19.104166,  y + stretch * 20.948318);
    pdf_contents_line_to(canvas, x + stretch * 19.10341 ,  y + stretch * 21.177612);
    pdf_contents_stroke(canvas);
    pdf_contents_move_to(canvas, x + stretch * 19.213683,  y + stretch * 21.177612);
    pdf_contents_line_to(canvas, x + stretch * 19.213103,  y + stretch * 20.891974);
    pdf_contents_line_to(canvas, x + stretch * 19.045865,  y + stretch * 20.891974);
    pdf_contents_line_to(canvas, x + stretch * 19.045865,  y + stretch * 20.948318);
    pdf_contents_stroke(canvas);
    pdf_contents_move_to(canvas, x + stretch * 19.318285,   y + stretch * 21.177612);
    pdf_contents_line_to(canvas, x + stretch * 19.318285,   y + stretch * 20.809329);
    pdf_contents_line_to(canvas, x + stretch * 19.132266,   y + stretch * 20.809329);
    pdf_contents_line_to(canvas, x + stretch * 19.132521,   y + stretch * 20.890561);
    pdf_contents_stroke(canvas);
    pdf_contents_move_to(canvas, x + stretch * 19.228543,  y + stretch * 20.645554);
    pdf_contents_line_to(canvas, x + stretch * 19.229199,  y + stretch * 20.808985);
    pdf_contents_stroke(canvas);
    pdf_contents_move_to(canvas, x + stretch * 18.829904,  y + stretch * 20.647076);
    pdf_contents_line_to(canvas, x + stretch * 18.829904,  y + stretch * 20.996422);
    pdf_contents_stroke(canvas);
    
}

void pdf_loci_histogram(world_fmt *world)
{
    char *thetatitle = "Q";
    char *migt1 = "M";
    char *migt2 = "xNm";
    char *migtitle = world->options->usem ? migt1 : migt2;
    long msub = world->options->usem ?  -17 : -5;
    //float *temp;
    float lx;
    float ly;
    long locus = world->loci > 1 ? world->loci : 0;
    long numparam = world->numpop2;
    //    long bin;
    long pa;
    long numbins = 0;
    float p01, p99, mm, thetamin, migmin, thetamax, migmax;
    long pos01, pos99, posmm;
    long p99bins = 0;
    MYREAL delta; 
    float theta;
    float mig;
    //    MYREAL value; 
    //    MYREAL total=0. ;
    //    MYREAL freq=0. ;
    //    MYREAL sumfreq=0. ;
    bayes_fmt * bayes = world->bayes;
    long *bins = bayes->histogram[locus].bins;
    MYREAL *results = bayes->histogram[locus].results;
    MYREAL *mini = bayes->histogram[locus].minima;
    MYREAL *maxi = bayes->histogram[locus].maxima;
    char *set50 = bayes->histogram[locus].set50;
    char *set95 = bayes->histogram[locus].set95;
    //temp = (float *) mycalloc(1,sizeof(float));
    float page_height = pdf_contents_get_height(canvas);
    float page_width = pdf_contents_get_width(canvas);
    float w;
    char title[100] = "Bayesian Analysis: Posterior distribution over all loci";
    char tempstring[20];
    long frompop;
    long topop;
    // add a new page so that we can print at least four histograms
    pdf_new_page("");
    pdf_contents_set_font_and_size(canvas, "Helvetica-Oblique", 16);
    w = pdf_contents_get_text_width(canvas, title, NULL, NULL);
    
    /* Start to print text. */ 
    pdf_print_contents_at((page_width - w)/2, page_height - 100, title);
    pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0, 0, 0));
    pdf_draw_line(50, page_height - 126, page_width-50, page_height - 126);
    pdf_contents_set_font_and_size(canvas, "Helvetica", 10);
    page_height -= 126;
    lx = 100;
    ly = page_height - 190;
    // set all parameter print maxima to the same value for theta and M
    // this should be menu-parameter controlled and take values
    // [all,p99,p99max]
    thetamin = HUGE;
    thetamax = 0.;
    numbins=0;
    for(pa=0; pa < world->numpop; pa++)
      {
        findmoments(&results[numbins],bins[pa],&p01,&pos01,&p99,&pos99,&mm,&posmm);
        if(mini[pa] < thetamin)
            thetamin = mini[pa];
        theta = pos99 * (maxi[pa]-mini[pa])/bins[pa];
        if(theta>thetamax)
            thetamax=theta;
        numbins += bins[pa];
      }
    migmin = HUGE;
    migmax = 0. ;
    numbins = 0;
    for(pa=world->numpop; pa < numparam; pa++)
      {
        // if custom migration matrix is set to zero
        // continue
        if(strchr("0c",bayes->custm2[pa]))
            continue;
        findmoments(&results[numbins],bins[pa],&p01,&pos01,&p99,&pos99,&mm,&posmm);
        mig = pos99 * (maxi[pa]-mini[pa])/bins[pa];
        if(mini[pa] < migmin)
            migmin = mini[pa];
        if(mig>migmax)
            migmax=mig;
        numbins += bins[pa];
      }
    numbins = 0;
    for(pa=0; pa < numparam; pa++)
      {
        // if custom migration matrix is set to zero
        // continue
        if(strchr("0c",bayes->custm2[pa]))
            continue;
        delta = (maxi[pa] - mini[pa])/bins[pa];
        m2mm(pa, world->numpop, &frompop, &topop);
        if(frompop==topop)
          {
            sprintf(tempstring,"%li",frompop+1);
            pdf_contents_set_font_and_size(canvas, "Symbol", 12);
            pdf_print_contents_at(lx-30,ly+125,thetatitle);
            pdf_contents_set_font_and_size(canvas, "Symbol", 9);
            pdf_print_contents_at(lx-20,ly+121,tempstring);
          }
        else
          {
            sprintf(tempstring,"%li->%li",frompop+1,topop+1);
            pdf_contents_set_font_and_size(canvas, "Helvetica", 12);
            pdf_print_contents_at(lx-30,ly+125,migtitle);
            pdf_contents_set_font_and_size(canvas, "Helvetica", 9);
            pdf_print_contents_at(lx+msub,ly+121,tempstring);
          }
        pdf_contents_set_font_and_size(canvas, "Helvetica", 10);
        switch(bayes->prettyhist)
          {
            case PRETTY_MAX:
                pdf_histogram(&results[numbins],&set50[numbins], &set95[numbins], 
                              bins[pa],(float) delta, (float) mini[pa], (float) maxi[pa],lx,ly,187,116);
                break;
            case PRETTY_P99:
                pdf_histogram(&results[numbins],&set50[numbins], &set95[numbins], 
                              bins[pa],(float) delta, (float) mini[pa], -9999,lx,ly,187,116);
                break;
            case PRETTY_P100:
                pdf_histogram(&results[numbins],&set50[numbins], &set95[numbins], 
                              bins[pa],(float) delta, (float) mini[pa], -999,lx,ly,187,116);
                break;
            case PRETTY_P99MAX:
            default:
                if(pa < world->numpop)
                  {
                    p99bins = (long)((thetamax-thetamin)/(float)delta);
                    pdf_histogram(&results[numbins],&set50[numbins], &set95[numbins], 
                                  p99bins,(float) delta, thetamin, thetamax,
                                  lx,ly,187,116);
                  }
                else
                  {
                    p99bins = (long) ((migmax - migmin)/(float)delta);
                    pdf_histogram(&results[numbins],&set50[numbins], &set95[numbins], 
                                  p99bins,(float) delta, migmin, migmax,
                                  lx,ly,187,116);
                  }
                break;
          }
        if(pa % 2 == 0)
          {
            lx = 350;
            ly = page_height - 190;
          }
        else
          {
            lx = 100;
            page_height -= 190;
            if(page_height-50 < 190)
              {
                pdf_new_page("");
                page_height = pdf_contents_get_height(canvas) - 10 ;
              }
            ly = page_height - 190;
            
          }
        //
        //if(pa % 2 == 0)
        //	{
        //	  pdf_new_page("");
        //	  page_height = pdf_contents_get_height(canvas) - 10 ; //headroom for parameter text
        //	}
        //ly = page_height - 190;
        numbins += bins[pa];
      }
}

///
/// prints right-adjusted text, based on margin-width x and stay on the line 
void pdf_printf_right(float x, float y, char string[],...)
{
  //    float     page_width = pdf_contents_get_width(canvas);
    float w;
    char message[LINESIZE];
	char fp[LINESIZE];
	va_list args;
    va_start (args, string);
    vsprintf (message, string, args);
    va_end (args);	
    sprintf(fp,"%s",message);
    w = pdf_contents_get_text_width(canvas, fp, NULL, NULL);
    pdf_print_contents_at(/*page_width-*/x-w, y, fp);
}

///
/// advance a line and check whether we are at the end of the page, if yes then add a new page
float pdf_advance(float *page_height)
{
    *page_height -= LINESTRETCH;
    if(*page_height < 55)
      {
        pdf_new_page("");
        *page_height = pdf_contents_get_height(canvas);
        *page_height -= 75;
      }
    return *page_height;
}

///
/// prints right-adjusted text, based on margin-width x and jumps to next line 
void pdf_printf_right_next(float x, float *y, char string[],...)
{
    float     page_width = pdf_contents_get_width(canvas);
    float w;
    char message[LINESIZE];
	char fp[LINESIZE];
	va_list args;
    va_start (args, string);
    vsprintf (message, string, args);
    va_end (args);	
    sprintf(fp,"%s",message);
    w = pdf_contents_get_text_width(canvas, fp, NULL, NULL);
    pdf_print_contents_at(page_width-x-w, *y, fp);
    pdf_advance(y);
}

///
/// prints right-aligned mutable text. Like vprintf
void pdf_printf_ralign(float rx, float y, char string[], ...)
{
    float w;
    char message[LINESIZE];
	char fp[LINESIZE];
	va_list args;
    va_start (args, string);
    vsprintf (message, string, args);
    va_end (args);	
	sprintf(fp,"%s",message);
    w = pdf_contents_get_text_width(canvas, fp, NULL, NULL);
    pdf_print_contents_at(rx-w, y, fp);
}



///
/// prints left-aligned mutable text. like vprintf
void pdf_printf(float x, float y, char string[], ...)
{
    char message[LINESIZE];
	char fp[LINESIZE];
	va_list args;
    va_start (args, string);
    vsprintf (message, string, args);
    va_end (args);	
	sprintf(fp,"%s",message);
    pdf_print_contents_at(x, y, fp);
}


///
/// prints mutable text and advances to next line.
void pdf_printf_next(float x, float *y, char string[], ...)
{
    char message[LINESIZE];
	char fp[LINESIZE];
	va_list args;
    va_start (args, string);
    vsprintf (message, string, args);
    va_end (args);	
	sprintf(fp,"%s",message);
    pdf_print_contents_at(x, *y, fp);
    pdf_advance(y);
}

void    pdf_print_connection_table (float * page_height, world_fmt * world, data_fmt * data)
{
  const float offset = 20;
  long i;
  long j;
  long z;
  float right_margin = pdf_contents_get_width(canvas) - 2*55 - offset;
  float left_margin = 55;
  float left_margin2 = 60;
  float lx = left_margin + 90;
  pdf_advance(page_height);		
  pdf_printf_next(left_margin, page_height,"Connection type matrix:");
  pdf_printf_next (left_margin2, page_height, "where m = average (average over a group of Thetas or M,");
  pdf_printf_next (left_margin2, page_height, "s = symmetric M, S = symmetric 4Nm,\n 0 = zero, and not estimated,");
  pdf_printf_next (left_margin2, page_height, "* = free to vary, Thetas are on diagonal\n");
  pdf_advance(page_height);		
  pdf_printf (left_margin, *page_height, "Population");
  for (i = 0; i < world->numpop; i++)
    {
      pdf_printf (lx, *page_height, "%3li", i);
      lx += offset;
      if(lx > right_margin)
	{
	  lx = left_margin + 90;
	  pdf_advance(page_height);		                
	}
    }
  lx = left_margin + 90;
  for (i = 0; i < world->numpop; i++)
    {
      lx = left_margin + 90;
      pdf_advance(page_height);		
      pdf_printf (left_margin, *page_height, "%3li %-15.15s",i,  data->popnames[i]);
      for (j = 0; j < world->numpop; j++)
	{
	  z = mm2m(j,i,world->numpop);
	  pdf_printf(lx, *page_height, " %c ", world->options->custm2[z]);
	  lx += offset;
	  if(lx > right_margin)
	    {
	      lx = left_margin + 90;
	      pdf_advance(page_height);		                
	    }
	}
    }
  pdf_advance(page_height);		
  pdf_advance(page_height);		
}

void    pdf_print_distance_table (float * page_height, world_fmt * world, option_fmt * options, data_fmt * data)
{
  const float offset = 60;
  long i;
  long j;
  long z;
  float right_margin = pdf_contents_get_width(canvas) - 2*55 - offset;
  float left_margin = 55;
  float lx=left_margin + 70;

  if(!options->geo)
    return;
  
  pdf_advance(page_height);		
  pdf_printf_next(left_margin, page_height,"Distance among populations:");
  pdf_advance(page_height);		
  pdf_printf (left_margin, *page_height, "Population");
  for (i = 0; i < world->numpop; i++)
    {
      pdf_printf (lx, *page_height, "%3li", i);
      lx += offset;
      if(lx > right_margin)
	{
	  lx = left_margin + 70;
	  pdf_advance(page_height);		                
	}
    }
  lx = left_margin + 70;
  for (i = 0; i < world->numpop; i++)
    {
      lx = left_margin + 70;
      pdf_advance(page_height);		
      pdf_printf (left_margin, *page_height, "%3li %-15.15s",i,  data->popnames[i]);
      for (j = 0; j < world->numpop; j++)
	{
	  z = mm2m(j,i,world->numpop);
	  pdf_printf(lx, *page_height, " %10.4f ", world->options->custm2[z]);
	  lx += offset;
	  if(lx > right_margin)
	    {
	      lx = left_margin + 70;
	      pdf_advance(page_height);		                
	    }
	}
    }
  pdf_advance(page_height);		
  pdf_advance(page_height);		
}



void pdf_print_options(world_fmt * world, option_fmt *options, data_fmt * data, float *orig_page_height, float *orig_left_margin)
{
    
    float page_width;
    float page_height = *orig_page_height;
    float left_margin = *orig_left_margin;
    float right_margin = 300. ;
    float w;
    char *title = "Options";
    long i, j, tt, ii;
    char mytext[LINESIZE];
    char seedgen[LINESIZE], spacer[LINESIZE];
    char paramtgen[LINESIZE], parammgen[LINESIZE];
    if (options->datatype != 'g')
      {
        switch ((short) options->autoseed)
          {
            case AUTO:
                strcpy (seedgen, "with internal timer");
                strcpy (spacer, "  ");
                break;
            case NOAUTOSELF:
                strcpy (seedgen, "from parmfile");
                strcpy (spacer, "      ");
                break;
            case NOAUTO:
                strcpy (seedgen, "from seedfile");
                strcpy (spacer, "      ");
                break;
            default:
                strcpy (seedgen, "ERROR");
                strcpy (spacer, " ");
                break;
          }
        switch (options->thetaguess)
          {
            case OWN:
                strcpy (paramtgen, "from guessed values");
                break;
            case FST:
                strcpy (paramtgen, "from the FST-calculation");
                break;
            case RANDOMESTIMATE:
                strcpy (paramtgen, "RANDOM start value from N(mean,std)");
                break;
            default:
                strcpy (paramtgen, "ERROR");
                break;
          }
        switch (options->migrguess)
          {
            case OWN:
                strcpy (parammgen, "from guessed values");
                break;
            case FST:
                strcpy (parammgen, "from the FST-calculation");
                break;
            case RANDOMESTIMATE:
                strcpy (parammgen, "RANDOM start value from N(mean,std)");
                break;
            default:
                strcpy (parammgen, "ERROR");
                break;
          }
      }
    pdf_contents_set_font_and_size(canvas, "Helvetica-Oblique", 18);    
    w = pdf_contents_get_text_width(canvas, title, NULL, NULL);
    page_width = pdf_contents_get_width(canvas);
    right_margin = page_width - 55;
    pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0, 0, 0));
    pdf_contents_set_rgb_fill(canvas, PdfRGBColor(0, 0, 0));
    pdf_contents_set_line_width(canvas, 1);
    pdf_print_contents_at((page_width - w)/2, page_height, title);
    page_height -= 20;
    pdf_draw_line(50, page_height, page_width-50, page_height);
    page_height -= 20;
    pdf_contents_set_font_and_size(canvas, "Helvetica", 11);    
    
    switch (options->datatype)
      {
        case 'a':
            pdf_print_contents_at(left_margin, page_height,"Datatype:");
            pdf_printf_right_next(left_margin, &page_height,"Allelic data");
            pdf_print_contents_at(left_margin, page_height,"Missing data:");
            pdf_printf_right_next(left_margin, &page_height,"%s\n",options->include_unknown ? "included" : "not included");
            break;
        case 'b':
            pdf_print_contents_at(left_margin, page_height,"Datatype:");
            pdf_printf_right_next(left_margin, &page_height,"Microsatellite data [Brownian motion]\n");
            pdf_print_contents_at(left_margin, page_height,"Missing data:");
            pdf_printf_right_next(left_margin, &page_height,"%s\n",options->include_unknown ? "included" : "not included");
            break;
        case 'm':
            pdf_print_contents_at(left_margin, page_height,"Datatype:");
            pdf_printf_right_next(left_margin, &page_height,"Microsatellite data [Stepwise mutation]\n");
            pdf_print_contents_at(left_margin, page_height,"Missing data:");
            pdf_printf_right_next(left_margin, &page_height,"%s\n",options->include_unknown ? "included" : "not included");
            break;
        case 's':
            pdf_print_contents_at(left_margin, page_height,"Datatype:");
            pdf_printf_right_next(left_margin, &page_height,"DNA sequence data\n");
            break;
        case 'n':
            pdf_print_contents_at(left_margin, page_height,"Datatype:");
            pdf_printf_right_next(left_margin, &page_height,"Single nucleotide polymorphism data\n");
            break;
        case 'u':
            pdf_print_contents_at(left_margin, page_height,"Datatype:");
            pdf_printf_right_next(left_margin, &page_height,"Single nucleotide polymorphism data (PANEL)\n");
            break;
        case 'f':
            pdf_print_contents_at(left_margin, page_height,"Datatype:");
            pdf_printf_right_next(left_margin, &page_height,"Ancestral state method\n");
            break;
        case 'g':
            pdf_print_contents_at(left_margin, page_height,"Datatype:");
            pdf_printf_right_next(left_margin, &page_height,"Genealogy summary of an older run\n");
            break;
      }
    if (options->datatype != 'g')
      {
        pdf_print_contents_at(left_margin, page_height,"Random number seed:");
        pdf_printf_right_next(left_margin, &page_height,"(%s)%s%20li", seedgen, " ",
                              options->saveseed);
        pdf_printf_next(left_margin, &page_height,"Start parameters:");
        pdf_advance(&page_height);
        pdf_print_contents_at(left_margin, page_height,"Theta values were generated");
        pdf_printf_right_next(left_margin, &page_height," %s", paramtgen);
        if (options->thetaguess == OWN)
          {
            ii = 0;
            pdf_print_contents_at(left_margin+10, page_height,"Theta = ");
            for (i = 0; i < options->numthetag - 1; i++)
              {
                sprintf(mytext,"%.5f", options->thetag[i]);
                if(i % 5 == 4)
                  {
                    ii = 0;
                    pdf_printf_next(left_margin + 100 + ii * 60, &page_height,"%s", mytext);
                  }
                else
                  {
                    pdf_print_contents_at(left_margin + 100 + ii * 60, page_height,mytext);
                  }
                ii++;
              }
            sprintf(mytext," %.5f, ",options->thetag[i]);
            pdf_printf_next(left_margin + 100 + ii * 60, &page_height,"%.5f", mytext);
          }
        if(options->usem)
            pdf_print_contents_at(left_margin, page_height,"M values were generated");
        else
            pdf_print_contents_at(left_margin, page_height,"xNm values were generated");
        pdf_printf_right_next(left_margin, &page_height," %s", parammgen);
        if (options->migrguess == OWN)
          {
            tt = 0;
            if (options->usem)
                pdf_print_contents_at(left_margin+10, page_height,"M-matrix:");
            else
                pdf_print_contents_at(left_margin+10, page_height,"xNm-matrix:");
            if (options->nummg == 1)
              {
                pdf_printf_right_next(left_margin, &page_height,"%5.2f [all are the same]\n", options->mg[tt++]);
              }
            else
              {
                for (i = 0; i < world->numpop; i++)
                  {
                    ii = 0;
                    for (j = 0; j < world->numpop; j++)
                      {
                        if (i != j)
                          {
                            if(i % 5 == 4)
                              {
                                ii = 0;
                                sprintf(mytext," %5.*f, ",options->usem ? 1 : 5, options->mg[tt++]);
                                pdf_printf_next(left_margin+100 + ii * 60, &page_height,"%s", mytext);
                              }
                            else
                              {
                                sprintf(mytext," %5.*f, ",options->usem ? 1 : 5, options->mg[tt++]);
                                pdf_print_contents_at(left_margin + 100 + ii * 60, page_height,mytext);
                              }
                          }
                        else
                          {
                            if(i % 5 == 4)
                              {
                                pdf_print_contents_at(left_margin + 100 + ii * 60, page_height,"-----");
                              }
                            else
                              {
                                pdf_print_contents_at(left_margin + 100 + ii * 60, page_height,"-----");
                              }
                          }
                        ii++;
                      }
                    pdf_advance(&page_height);
                  }
                pdf_advance(&page_height);
              }
          }
      }
    pdf_print_connection_table (&page_height, world, data);
    pdf_print_distance_table (&page_height, world, options, data);
    if (options->gamma)
      {
        pdf_print_contents_at(left_margin, page_height,"Mutation rate among loci:");
        pdf_printf_right_next(left_margin, &page_height,"from a Gamma distribution\n");
        pdf_printf_right_next(left_margin, &page_height,"Initial scale parameter alpha = %f\n",
                              options->alphavalue);
        if (options->custm[world->numpop2] == 'c')
          {
            pdf_printf_next(left_margin + 300, &page_height,"and is constant [will not be estimated]\n");
          }
      }
    else
      {
        pdf_print_contents_at(left_margin, page_height,"Mutation rate among loci:");
        if (options->murates && world->loci > 1)
          {
            pdf_printf_right_next(left_margin, &page_height,"Varying\n");
            pdf_print_contents_at(left_margin + 10, page_height,"Rates per locus: ");
            ii=0;
            for (i = 0; i < world->loci - 1; i++)
              {
                sprintf(mytext,"%.5f, ", options->mu_rates[i]);
                if (i % 6 == 5)
                  {
                    ii=0;
                    pdf_print_contents_at(left_margin + 100 + ii * 60, page_height,mytext);
                    pdf_advance(&page_height);
                  }
                else
                  {
                    pdf_print_contents_at(left_margin + 100 + ii * 60, page_height,mytext);
                  }
                ii++;
              }
            pdf_printf_next(left_margin + 100 + ii * 60, &page_height,"%.5f\n", options->mu_rates[i]);
          }
        else
            pdf_printf_right_next(left_margin, &page_height,"Mutation rate is constant %s\n", world->loci > 1 ?
                                  "for all loci" : "");
      }
#ifdef UEP
    if (options->uep)
      {
        pdf_printf_next(left_margin, &page_height,"0/1 polymorphism analysis, with 0/1 data in file:");
        pdf_printf_right_next(left_margin, &page_height,"%s\n", options->uepfilename);
        pdf_printf_right_next(left_margin, &page_height,"with forward mutation rate %f*mu\n",
                              options->uepmu);
        pdf_printf_right_next(left_margin, &page_height,"with back mutation rate %f*mu\n",
                              options->uepnu);
        pdf_printf_right_next(left_margin, &page_height,"with base frequencies \"0\"=%f and \"1\"=%f\n",
                              options->uepfreq0,options->uepfreq1);
      }
#endif
    pdf_advance(&page_height);
    
    if(options->bayes_infer)
      {
        pdf_print_contents_at(left_margin, page_height,"Analysis strategy:");
        pdf_printf_right_next(left_margin, &page_height,"Bayesian inference");
        pdf_print_contents_at(left_margin + 10, page_height,"Used prior distribution for parameter:");
        switch(options->bayesprior)
          {
            case EXPPRIOR:
                pdf_printf_right_next(left_margin, &page_height,"EXPONENTIAL\n");
                pdf_printf_right_next(left_margin, &page_height,"Minimum, mean, and maximum values:\n");
                pdf_printf_right_next(left_margin, &page_height,"Theta: %f, %f, %f\n", 
                                      options->bayespriortheta->min, 
                                      options->bayespriortheta->mean, 
                                      options->bayespriortheta->max);
                if(options->usem)
                    pdf_printf_right_next(left_margin, &page_height,"M: %f, %f, %f\n",
                                          options->bayespriorm->min,
                                          options->bayespriorm->mean, 
                                          options->bayespriorm->max);
                else
                    pdf_printf_right_next(left_margin, &page_height,"xNm: %f, %f, %f\n",
                                          options->bayespriorm->min,
                                          options->bayespriorm->mean, 
                                          options->bayespriorm->max);
                break;
            case WEXPPRIOR:
                pdf_printf_right_next(left_margin, &page_height,"EXPONENTIAL\n");
                pdf_printf_right_next(left_margin, &page_height,"Minimum, mean, maximum values and window (delta) value:\n");
                pdf_printf_right_next(left_margin, &page_height,"    Theta: %f, %f, %f, %f\n", 
                                      options->bayespriortheta->min, 
                                      options->bayespriortheta->mean, 
                                      options->bayespriortheta->max,
                                      options->bayespriortheta->delta);
                if(options->usem)
                    pdf_printf_right_next(left_margin, &page_height,"   M:     %f, %f, %f, %f\n",
                                          options->bayespriorm->min,
                                          options->bayespriorm->mean, 
                                          options->bayespriorm->max,
                                          options->bayespriorm->delta);
                else
                    pdf_printf_right_next(left_margin, &page_height," xNm:     %f, %f, %f, %f\n",
                                          options->bayespriorm->min,
                                          options->bayespriorm->mean, 
                                          options->bayespriorm->max,
                                          options->bayespriorm->delta);
                break;
            case MULTPRIOR:
                pdf_printf_right_next(left_margin, &page_height,"MULTIPLIER\n");
                pdf_printf_right_next(left_margin, &page_height,"Minimum, maximum values and delta multiplier:\n");
                pdf_printf_right_next(left_margin, &page_height,"Theta:  %f, %f, %f\n", 
                                      options->bayespriortheta->min, 
                                      options->bayespriortheta->max,
                                      options->bayespriortheta->delta);
                if(options->usem)
		  pdf_printf_right_next(left_margin, &page_height,"M: %f, %f, %f\n",
					options->bayespriorm->min,
					options->bayespriorm->max,
					options->bayespriorm->delta);
		else
		  pdf_printf_right_next(left_margin, &page_height,"xNm:%f, %f, %f\n",
					options->bayespriorm->min,
					options->bayespriorm->max,
					options->bayespriorm->delta);
                break;
		  
                break;
            case UNIFORMPRIOR:
            default:
                pdf_printf_right_next(left_margin, &page_height,"UNIFORM\n");
                pdf_print_contents_at(left_margin, page_height,"Minimum and maximum values:");
                pdf_printf_right_next(left_margin, &page_height,"Theta:%f - %f\n",options->bayespriortheta->min,options->bayespriortheta->max);
                if(options->usem)
		  pdf_printf_right_next(left_margin, &page_height,"M:   %f - %f\n",options->bayespriorm->min,options->bayespriorm->max);
		else
		  pdf_printf_right_next(left_margin, &page_height,"xNm: %f - %f\n",options->bayespriorm->min,options->bayespriorm->max);
                break;
          }
        pdf_printf(left_margin, page_height,"Maximal number of bins for posterior distribution histogram:");
        pdf_print_contents_at(left_margin + 400, page_height,"Theta:");
        pdf_printf_right_next(left_margin, &page_height,"%li\n", options->bayespriortheta->bins);
	if(options->usem)
	  pdf_print_contents_at(left_margin + 400, page_height,"M:  ");
	else
	  pdf_print_contents_at(left_margin + 400, page_height,"xNm:");
        pdf_printf_right_next(left_margin, &page_height,"%li\n", options->bayespriorm->bins);
      }
    else
      {
        pdf_print_contents_at(left_margin, page_height,"Analysis strategy is");
        pdf_printf_right_next(left_margin, &page_height,"Maximum likelihood\n");
      }
    
    pdf_advance(&page_height);
    
    if (options->datatype != 'g')
      {
        pdf_print_contents_at(left_margin, page_height,"Markov chain settings:");
        pdf_print_contents_at(left_margin + 300, page_height,"Short chain");
        pdf_printf_right_next(left_margin, &page_height, "Long chain");
        pdf_print_contents_at(left_margin, page_height,"Number of chains");
        pdf_printf_ralign(left_margin + 340, page_height,"%20li", options->schains);
        pdf_printf_right_next(left_margin+10, &page_height,"%20li", options->lchains);
        
        pdf_print_contents_at(left_margin + 10, page_height,"Recorded genealogies [a]");
        pdf_printf_ralign(left_margin + 340, page_height,"%20li", options->ssteps);
        pdf_printf_right_next(left_margin + 10, &page_height,"%20li", options->lsteps);
        
        pdf_print_contents_at(left_margin + 10, page_height,"Increment (record every x genealogy [b]");
        pdf_printf_ralign(left_margin + 340, page_height,"%20li", options->sincrement);
        pdf_printf_right_next(left_margin+10, &page_height,"%20li", options->lincrement);
        if(options->bayes_infer)
          {
            pdf_print_contents_at(left_margin + 10, page_height,"Number of concurrent chains (replicates) [c]");
            pdf_printf_ralign(left_margin + 340, page_height," ");
            pdf_printf_right_next(left_margin + 10, &page_height,"%20li", options->replicate ? 
                                  options->replicatenum : 1);
            pdf_print_contents_at(left_margin + 10, page_height,"Visited (sampled) genealogies [a*b*c]");
            pdf_printf_ralign(left_margin + 340, page_height,"%20li", 
                              options->ssteps * options->sincrement*(options->replicate ? options->replicatenum : 1));
            pdf_printf_right_next(left_margin + 10, &page_height,"%20li", options->lsteps * options->lincrement*(options->replicate ? options->replicatenum : 1));
          } 
        else
          {
            pdf_print_contents_at(left_margin + 10, page_height,"Visited (sampled) genealogies [a*b]");
            pdf_printf_ralign(left_margin + 340, page_height,"%20li", options->ssteps * options->sincrement);
            pdf_printf_right_next(left_margin + 10, &page_height,"%20li", options->lsteps * options->lincrement);
            
          }
        if (options->burn_in > 0)
          {
            pdf_print_contents_at(left_margin + 10, page_height,"Number of discard trees per chain (burn-in)");
            pdf_printf(left_margin + 320, page_height,"%li",  (long) options->burn_in);
            pdf_printf_right_next(left_margin + 10, &page_height,"%li", (long) options->burn_in);        
          }
        if (options->movingsteps)
          {
            pdf_print_contents_at(left_margin + 10, page_height,"Forcing percentage of new genealogies");
            //    pdf_printf(left_margin + 360, page_height,"%5.2f",   (MYREAL) options->acceptfreq);
            pdf_printf_right_next(left_margin + 10, &page_height,"%5.2f",   (MYREAL) options->acceptfreq);
          }
        if (options->lcepsilon < LONGCHAINEPSILON)
          {
            pdf_print_contents_at(left_margin + 10, page_height,"Forcing parameter-likelihood improvement");
            //    pdf_printf(left_margin + 360, page_height,"%20.5f",   (MYREAL) options->lcepsilon);
            pdf_printf_right_next(left_margin + 10, &page_height,"%20.5f",   (MYREAL) options->lcepsilon);
          }
        
        pdf_advance(&page_height);
        
        if(options->replicate || options->heating > 0)
            pdf_printf_next(left_margin, &page_height,"Multiple Markov chains:");
        
        if (options->replicate && !options->bayes_infer)
          {
            pdf_print_contents_at(left_margin + 10, page_height,"Averaging over replicates");
            if (options->replicatenum == 0)
                pdf_printf_right_next(left_margin, &page_height,"Long chains only");
            else
                pdf_printf_right_next(left_margin, &page_height,"Over indepedent %li replicates", options->replicatenum);
          }
        
        if (options->heating > 0)
          {
            pdf_printf(left_margin+10, page_height,"%s heating scheme", options->adaptiveheat ? "Adaptive" : "Static");
            pdf_printf_right_next(left_margin, &page_height, "%li chains with %s temperatures",
                                  options->heated_chains, options->adaptiveheat ? "start" : "" );
            for (i = options->heated_chains - 1; i >= 0; i--)
                pdf_printf_right(right_margin - i * 50, page_height,"%5.2f ", options->heat[i]);
            pdf_advance(&page_height);
            
            pdf_printf_right_next(left_margin, &page_height,"Swapping interval is %li\n",
                                  options->heating_interval);
          }        
      }
    
    pdf_advance(&page_height);
    pdf_printf_next(left_margin, &page_height,"Print options:\n");
    if (options->datatype != 'g')
      {
        pdf_print_contents_at(left_margin + 10, page_height,"Data file:");
        pdf_printf_right_next(left_margin, &page_height,"%s", options->infilename);
        pdf_print_contents_at(left_margin + 10, page_height,"Output file:");
        pdf_printf_right_next(left_margin, &page_height,"%s", options->outfilename);
        if(options->writesum)
          {
            pdf_print_contents_at(left_margin + 10, page_height,"Summary of genealogies for further run:");
            pdf_printf_right_next(left_margin, &page_height,"%s", options->sumfilename);
          }
        if(options->writelog)
          {
            pdf_print_contents_at(left_margin + 10, page_height,"Log file:");
            pdf_printf_right_next(left_margin, &page_height,"%s", options->logfilename);
          }                              
        
        if(options->bayes_infer)
          {
            pdf_print_contents_at(left_margin + 10, page_height,"Posterior distribution raw histogram file:");
            pdf_printf_right_next(left_margin, &page_height,"%s\n", options->bayesfilename);
          }
        
        pdf_print_contents_at(left_margin + 10, page_height,"Print data:");
        pdf_printf_right_next(left_margin, &page_height,"%45.45s", options->printdata ? "Yes" : "No");
        
        pdf_printf(left_margin + 10, page_height,"Print genealogies [only some for some data type]:");
        switch (options->treeprint)
          {
            case NONE:
                pdf_printf_right_next(left_margin, &page_height,"None");
                break;
            case ALL:
                pdf_printf_right_next(left_margin, &page_height,"Yes, all");
                break;
            case LASTCHAIN:
                pdf_printf_right_next(left_margin, &page_height,"Yes, only those in last chain");
                break;
            case BEST:
                pdf_printf_right_next(left_margin, &page_height,"Yes, only the best");
                break;
          }
        if (options->mighist)
          {
            pdf_print_contents_at(left_margin + 10, page_height,"Histogram of the frequency of migration events");
            pdf_printf_right_next(left_margin, &page_height,"%s", options->mighistfilename);
          }        
      }
    else
      {
        pdf_print_contents_at(left_margin + 10, page_height,"Data file:");
        pdf_printf_right_next(left_margin, &page_height,"%s", options->infilename);
        pdf_print_contents_at(left_margin + 10, page_height,"Output file:");
        pdf_printf_right_next(left_margin, &page_height,"%s", options->outfilename);        
      }
    pdf_print_contents_at(left_margin + 10, page_height,"Plot log(likelihood) surface:");
    if (options->plot)
      {
        switch (options->plotmethod)
          {
            case PLOTALL:
                sprintf (mytext, "Yes, to outfile and %s", options->mathfilename);
                break;
            default:
                strcpy (mytext, "Yes, to outfile");
                break;
          }
        pdf_printf_right_next(left_margin, &page_height,"%s\n", mytext);
        pdf_printf_right_next(left_margin, &page_height,
                              "Parameter: %s, Scale: %s, Intervals: %li\n",
                              options->plotvar == PLOT4NM ? "{Theta, 4Nm}" : "{Theta, M}",
                              options->plotscale == PLOTSCALELOG ? "Log10" : "Standard",
                              options->plotintervals);
        pdf_printf_right_next(left_margin, &page_height,"Ranges: X-%5.5s: %f - %f\n",
                              options->plotvar == PLOT4NM ? "4Nm" : "M",
                              options->plotrange[0], options->plotrange[1]);
        pdf_printf_right_next(left_margin, &page_height,"Ranges: Y-%5.5s: %f - %f\n", "Theta",
                              options->plotrange[2], options->plotrange[3]);
      }
    else
      {
        pdf_printf_right_next(left_margin, &page_height,"No");
      }
    switch (options->profile)
      {
        case NONE:
            strcpy (mytext, "No");
            break;
        case ALL:
            strcpy (mytext, "Yes, tables and summary");
            break;
        case TABLES:
            strcpy (mytext, "Yes, tables");
            break;
        case SUMMARY:
            strcpy (mytext, "Yes, summary");
            break;
      }
    pdf_printf(left_margin+10, page_height,"Profile likelihood:");
    pdf_printf_right_next(left_margin, &page_height,"%s", mytext);
    
    if (options->profile != NONE)
      {
        switch (options->profilemethod)
          {
            case 'p':
                pdf_printf_right_next(left_margin, &page_height,"Percentile method\n");
                break;
            case 'q':
                pdf_printf_right_next(left_margin, &page_height,"Quick method\n");
                break;
            case 'f':
                pdf_printf_right_next(left_margin, &page_height,"Fast method\n");
                break;
            case 'd':
                pdf_printf_right_next(left_margin, &page_height,"Discrete method\n");
                break;
            case 's':
                pdf_printf_right_next(left_margin, &page_height,"Spline method\n");
                break;
            default:
                pdf_printf_right_next(left_margin, &page_height,"UNKOWN method????\n");
                break;
          }
        pdf_printf_right_next(left_margin, &page_height,"with df=%li and for Theta and %s",
                              options->df, options->profileparamtype ? "M=m/mu" : "4Nm");
      }
    *orig_page_height = page_height;
    *orig_left_margin = left_margin;
}


void pdf_print_data_summary(world_fmt * world, option_fmt *options, data_fmt * data, 
                            float *orig_page_height, float *orig_left_margin)
{
    long locus;
    long pop;
    long numind;
    long nummiss;
    char dstring[LINESIZE];
    long *total;
    long *totalmiss;
    char *title = "Data summary";
    float w;
    float page_height = *orig_page_height;
    float left_margin = *orig_left_margin;
    float page_width;
    
    
    float col1 = left_margin + 300;
    float col2 = left_margin + 320;
    float col3 = left_margin + 420;
    
    
    total = (long *) mycalloc(data->loci,sizeof(long));
    totalmiss = (long *) mycalloc(data->loci,sizeof(long));
    // setup new page and title
    pdf_new_page("");
    pdf_contents_set_font_and_size(canvas, "Helvetica-Oblique", 18);    
    w = pdf_contents_get_text_width(canvas, title, NULL, NULL);
    page_height = pdf_contents_get_height(canvas) - 75;
    page_width = pdf_contents_get_width(canvas);
    page_height -= 20;
    pdf_print_contents_at((page_width - w)/2, page_height, title);
    pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0, 0, 0));
    page_height -= 20;
    pdf_draw_line(50, page_height, page_width-50, page_height);
    page_height -= 20;
    pdf_contents_set_font_and_size(canvas, "Helvetica", 11);    
    switch (options->datatype)
      {
        case 'a':
            strcpy (dstring, "Allelic data");
            break;
        case 'f':
        case 's':
            strcpy (dstring, "Sequence data");
            break;
        case 'b':
        case 'm':
            strcpy (dstring, "Microsatellite data");
            break;
        case 'n':
        case 'u':
            strcpy (dstring, "SNP data");
            break;
        default:
            strcpy (dstring, "Unknown data [ERROR]");
            break;
      }
    pdf_print_contents_at(left_margin, page_height,"Datatype:");
    pdf_printf_right_next(left_margin, &page_height,"%s", dstring);
    pdf_print_contents_at(left_margin, page_height,"Number of loci:");
    pdf_printf_right_next(left_margin, &page_height,"%li", data->loci);
    pdf_advance(&page_height);
    
    pdf_print_contents_at(left_margin, page_height,"Population");
    pdf_print_contents_at(col1, page_height,"Locus");
    pdf_print_contents_at(col3, page_height, "Gene copies");    
    pdf_advance(&page_height);
    if (!strchr (SEQUENCETYPES, options->datatype))
      {
        pdf_print_contents_at(col3, page_height,"data");
        pdf_printf_right_next(left_margin, &page_height,"(missing)");
      }
    
    for (pop = 0; pop < world->numpop; pop++)
      {
        if (!strchr (SEQUENCETYPES, options->datatype))
          {
            nummiss = find_missing(data,pop,0);
            numind = data->numalleles[pop][0] - nummiss;
            pdf_printf(left_margin, page_height,"%li %s",pop+1, data->popnames[pop]);
            pdf_printf_ralign(col2, page_height,"1");
            pdf_printf_ralign(col3, page_height,"%li",numind);
            pdf_printf_right_next(left_margin+10, &page_height,"(%li)", nummiss);
          }
        else
          {
            nummiss = 0;
            numind = data->numind[pop][0];
            pdf_printf(left_margin, page_height,"%li %s",pop+1, data->popnames[pop]);
            pdf_printf_ralign(col2, page_height,"1");
            pdf_printf_ralign(col3, page_height,"%li", numind);
            pdf_advance(&page_height);
          }
        total[0] += numind;
        totalmiss[0] += nummiss;
        
        for(locus=1; locus< data->loci; locus++)
          {
            if (!strchr (SEQUENCETYPES, options->datatype))
              {
                nummiss = find_missing(data,pop,locus);
                numind = data->numalleles[pop][locus] - nummiss;
                pdf_printf_ralign(col2, page_height,"%li",locus+1);
                pdf_printf_ralign(col3, page_height,"%li",numind);
                pdf_printf_right_next(left_margin+10, &page_height,"(%li)", nummiss);
              }
            else
              {
                nummiss=0;
                numind = data->numind[pop][locus];
                pdf_printf_ralign(col2, page_height,"%li",locus+1);
                pdf_printf_ralign(col3, page_height,"%li",numind);
                pdf_advance(&page_height);
              }
            total[locus] += numind;
            totalmiss[locus] += nummiss;
          }
      }
    pdf_printf(left_margin, page_height,"Total of all populations");
    pdf_printf_ralign(col2, page_height,"1");
    pdf_printf_ralign(col3, page_height,"%li",total[0]);
    if (!strchr (SEQUENCETYPES, options->datatype))
      {
        pdf_printf_right_next(left_margin+10, &page_height,"(%li)", totalmiss[0]);
        for(locus=1; locus< data->loci; locus++)
          {
            pdf_printf_ralign(col2, page_height,"%li",locus+1);
            pdf_printf_ralign(col3, page_height,"%li",total[locus]);
            pdf_printf_right_next(left_margin+10, &page_height,"(%li)", totalmiss[locus]);
          }
      }
    else
      {
        pdf_advance(&page_height);
        for(locus=1; locus< data->loci; locus++)
          {
            pdf_printf_ralign(col2, page_height,"%li",locus+1);
            pdf_printf_ralign(col3, page_height,"%li",total[locus]);
            pdf_advance(&page_height);
          }
      }    
    free(total);
    free(totalmiss);
    *orig_page_height = page_height;
    *orig_left_margin = left_margin;
}

///
/// \param[in] *fmt format string identical to a fomrat string in printf()
/// \reval int returns the number of elements by counting the %
int count_elements(char *fmt)
{
    int element;
    int count = 0;
    for(element=0; element < (int) strlen(fmt); element++)
      {
        if(fmt[element]=='%')
            count++;
      }
    return count;
}


///
/// Print  a line in a table in the PDF file at a specific height on the page and with a given width
/// \param *page_height gets modified by this function, page_height specifies the Y coordinate
/// \param *width contains a list of floating points numbers that specify the columns, negative numbers
/// mean left-adjusted and positive numbers are right-adjusted
/// \param *fmt format string identical to printf (number % needs to match the element in width)
/// \param ...  parameters to print
void pdf_print_table(float *page_height, float *width, char *fmt, ...)
{
    boolean start=FALSE;
    boolean stop=FALSE;
    char fmtval;
    char this_fmt[LINESIZE];
    long ival;
    va_list ap;
    double dval;
    long fmti=0;
    char cval, *sval;
    float y = *page_height;
    long ii=0;
    boolean left_aligned=FALSE;
    float offset = 55;
    va_start(ap, fmt);
    while (*fmt)
      {
        if(*fmt == '%')
          {
            if(width[ii] <= EPSILON)
              {
                left_aligned = TRUE;
                offset = 55 - width[ii++];
              }
            else
              {
                left_aligned = FALSE;
                offset = 55 + width[ii++];
              }
            
            start = TRUE;
            *fmt++;
            this_fmt[0] = '%';
            fmti = 1;
            continue;
          }
        else
          {   
            switch(*fmt) 
              {
                case 'c':                       /* string */
                    stop = TRUE;
                    fmtval = 'c';
                    cval = va_arg(ap, int);
                    this_fmt[fmti++] = fmtval;
                    this_fmt[fmti] = '\0';
                    if(left_aligned)
                        pdf_printf(offset, y,this_fmt, cval);
                    else
                        pdf_printf_ralign(offset, y,this_fmt, cval);
                    break;
                case 's':                       /* string */
                    stop = TRUE;
                    fmtval = 's';
                    sval = va_arg(ap, char *);
                    this_fmt[fmti++] = fmtval;
                    this_fmt[fmti] = '\0';              
                    if(left_aligned)
                        pdf_printf(offset, y,this_fmt, sval);
                    else
                        pdf_printf_ralign(offset, y,this_fmt, sval);
                    break;
                case 'i':                       /* int */
                    stop = TRUE;
                    fmtval = 'i';
                    ival = va_arg(ap, int);
                    this_fmt[fmti++] = fmtval;
                    this_fmt[fmti] = '\0';   
                    if(left_aligned)
                        pdf_printf(offset, y,this_fmt, ival);
                    else
                        pdf_printf_ralign(offset, y,this_fmt, ival);
                    break;
                case 'f':                       /* char */
                    stop = TRUE;
                    fmtval = 'f';
                    dval = va_arg(ap, double);
                    this_fmt[fmti++] = fmtval;
                    this_fmt[fmti] = '\0';                     
                    if(left_aligned)
                        pdf_printf(offset, y,this_fmt, dval);
                    else
                        pdf_printf_ralign(offset, y,this_fmt, dval);
                    break;
                case '\n':
                    pdf_advance(&y);
                    break;
              }
            if(start)
              {
                if(!stop)
                  {
                    if(strchr("1234567890.", *fmt)!=NULL)
                        this_fmt[fmti++] = *fmt;
                    if(*fmt == '-')
                        left_aligned = TRUE;
                  }
                else
                  {
                    start = FALSE;
                    stop = FALSE;
                  }
              }
            *fmt++;
          }
        va_end(ap);
      }
}


void pdf_print_allelelegend(float *column_width, float *page_height, long loci)
{
    long locus;
    char stemp[LINESIZE];
    sprintf(stemp, "%-s", (loci == 1 ? "locus" : "loci "));
    
    pdf_printf(column_width[0], *page_height, "Indiv.");
    for(locus=1; locus < loci+1; locus++)
      {
        pdf_printf(column_width[locus], *page_height, "%li",locus);        
        if(column_width[locus+1]<column_width[locus])
          {
            pdf_advance(page_height);
          }
      }
}

#define NOTFIRSTDATAPAGE (boolean) 0
#define FIRSTDATAPAGE (boolean) 1
///
/// print data header for allelic or sequence data
void
pdf_print_dataheader (boolean first, char *title, float *page_height, float *column_width, long pop, world_fmt * world,
                      option_fmt * options, data_fmt * data)
{
    
    float w;
    float page_width;
    float left_margin = 55;
    
    // setup new page and title
    if(first)
      {
        pdf_new_page("");
        pdf_contents_set_font_and_size(canvas, "Helvetica-Oblique", 18);    
        w = pdf_contents_get_text_width(canvas, title, NULL, NULL);
        *page_height = pdf_contents_get_height(canvas) - 75;
        page_width = pdf_contents_get_width(canvas);
        *page_height -= 20;
        pdf_print_contents_at((page_width - w)/2, *page_height, title);
        pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0, 0, 0));
        *page_height -= 20;
        pdf_draw_line(50, *page_height, page_width-50, *page_height);
        *page_height -= 20;
        pdf_contents_set_font_and_size(canvas, "Helvetica", 11);    
      }
    else
      {
        pdf_advance(page_height);
        page_width = pdf_contents_get_width(canvas);
        pdf_draw_line(50, *page_height, page_width-50, *page_height);
        pdf_advance(page_height);    
      }
    pdf_printf_next(left_margin, page_height,"\n%-s", data->popnames[pop]);
    pdf_draw_line(50, *page_height, page_width-50, *page_height);
    pdf_advance(page_height);   
    if (!strchr (SEQUENCETYPES, options->datatype))
        pdf_print_allelelegend(column_width, page_height, world->loci);
    //    else
    //    pdf_print_seqlegend(left_margin, page_height, world->loci);
    
    pdf_advance(page_height);            
    pdf_draw_line(50, *page_height, page_width-50, *page_height);
    pdf_advance(page_height);   
}



///
/// print the allele typ data in diploid format 
/// \param page_width page width
/// \param page_height is manipulated and is always at the actual y coordinate on the page
/// \param[in] margin is the margin that is used, either left or right margin
/// \param[in] world contains all main data structures
/// \param[in] data  contains all the biological data structures
/// \param[in] options cintains all option structures
void pdf_print_alleledata (float *page_height, float margin, world_fmt * world, data_fmt * data, option_fmt * options)
{
    
    long pop, ind, locus;
    char stemp[LINESIZE];
    float w;
    float * column_width = (float *) mycalloc(world->loci+2, sizeof(float));
    float pixel;
    float total;
    float oldpage_height;
    float page_width = pdf_contents_get_width(canvas);
    
    // calculate column width for indvidual name and each locus
    // we use a cummulative sum to set the x-coord just right without further
    // calculations
    for (locus = 0; locus < data->loci; locus++)
      {
        for (pop = 0; pop < data->numpop; pop++)
          {
            for (ind = 0; ind < data->numind[pop][0]; ind++)
              {
                w = 1. + (float) (strlen (data->yy[pop][ind][locus][0]) +
                                  strlen (data->yy[pop][ind][locus][1]));
                if(column_width[locus+2] < w)
                    column_width[locus+2] = w;
              }
          }
      }
    // calculate columnwidth for in page units for individual name and locus column
    column_width[1] = 1. + (float) sprintf(stemp, "%-*.*s", (int) options->nmlength,
                                           (int) options->nmlength, data->indnames[0][0]);
    w = pdf_contents_get_text_width(canvas, stemp, NULL, NULL);
    pixel = w / options->nmlength;
    column_width[0] = margin;
    column_width[1] = margin + w;
    total = margin + w ;
    for (locus = 0; locus < data->loci; locus++)
      {
        column_width[locus+2] *= pixel;
        total += column_width[locus+2];
        if(total > (page_width - 55))
          {
            total = margin;
            column_width[locus+2] = total;
          }
        else
          {
            column_width[locus+2] = total;
          }
      }
    
    for (pop = 0; pop < data->numpop; pop++)
      {
        if(pop==0)
            pdf_print_dataheader (FIRSTDATAPAGE, "Allelic data", page_height, column_width, pop, world, options, data);
        else
            pdf_print_dataheader (NOTFIRSTDATAPAGE, "Allelic data",page_height, column_width, pop, world, options, data);
        
        for (ind = 0; ind < data->numind[pop][0]; ind++)
          {
            pdf_printf(column_width[0], *page_height, "%-*.*s", (int) options->nmlength,
                       (int) options->nmlength, data->indnames[pop][ind]);
            for (locus = 0; locus < data->loci; locus++)
              {
                pdf_printf(column_width[locus+1], *page_height, " %s.%-s",
                           data->yy[pop][ind][locus][0],
                           data->yy[pop][ind][locus][1]);
                if(column_width[locus+2]<column_width[locus+1])
                  {
                    pdf_advance(page_height);
                  }
              }
            oldpage_height = *page_height;
            pdf_advance(page_height);
            if(oldpage_height < *page_height)
                pdf_print_dataheader (NOTFIRSTDATAPAGE, "Allelic data", page_height, column_width, pop, world, options, data);
            
          }
      }
    free(column_width);
}




void
pdf_print_data (world_fmt * world, option_fmt * options, data_fmt * data, float *orig_page_height, float *orig_left_margin)
{
    if (options->printdata)
      {
        switch (options->datatype)
          {
            case 'a':
            case 'b':
            case 'm':
                pdf_print_alleledata (orig_page_height,*orig_left_margin, world, data, options);
                break;
            case 's':
            case 'n':
            case 'u':
            case 'f':
                
                pdf_print_seqdata (orig_page_height, *orig_left_margin, world, data, options);
                break;
          }
      }
}



void pdf_print_sequence(float left_margin, float right_margin, float *page_height, data_fmt *data, long locus, long pop, long ind)
{
    long site;
    float w = 0.;
    float wtot = 0.;
    char stemp[LINESIZE];
    w = 60.;
    for(site=0; site < data->seq->sites[locus]; site+=10)
      {
        sprintf(stemp,"%-10.10s", &data->yy[pop][ind][locus][0][site]);
        pdf_contents_set_font_and_size(canvas, "Courier", 9);
        pdf_print_contents_at(left_margin + wtot, *page_height, stemp);
        wtot += w;
        if((left_margin + wtot + w) > right_margin)
          {
            wtot = 0;
            pdf_advance(page_height);
          }            
      }
    pdf_contents_set_font_and_size(canvas, "Helvetica", 11);
}

void pdf_print_seqdata (float *page_height, float margin, world_fmt * world, data_fmt * data, option_fmt * options)
{
    
    long pop, ind, locus;
    char stemp[LINESIZE];
    float w;
    float * column_width = (float *) mycalloc(world->loci+2, sizeof(float));
    float pixel;
    float oldpage_height;
    float page_width = pdf_contents_get_width(canvas);
    
    // calculate columnwidth for in page units for individual name and locus column
    column_width[1] = 1. + (float) sprintf(stemp, "%-*.*s", (int) options->nmlength,
                                           (int) options->nmlength, data->indnames[0][0]);
    w = pdf_contents_get_text_width(canvas, stemp, NULL, NULL);
    pixel = w / options->nmlength;
    column_width[0] = margin;
    column_width[1] = margin + w;
    column_width[3] = margin + w + 10;
    
    for (pop = 0; pop < data->numpop; pop++)
      {
        if(pop==0)
            pdf_print_dataheader (FIRSTDATAPAGE, "Sequence data", page_height, column_width, pop, world, options, data);
        else
            pdf_print_dataheader (NOTFIRSTDATAPAGE,  "Sequence data", page_height, column_width, pop, world, options, data);
        for (ind = 0; ind < data->numind[pop][0]; ind++)
          {
            pdf_printf(column_width[0], *page_height, "%-*.*s", (int) options->nmlength,
                       (int) options->nmlength, data->indnames[pop][ind]);
            for (locus = 0; locus < data->loci; locus++)
              {
                pdf_printf(column_width[1], *page_height,"%li",locus+1);
                pdf_print_sequence(column_width[1]+10, page_width - 55, page_height, data, locus, pop, ind);
                pdf_advance(page_height);
              }
            oldpage_height = *page_height;
            pdf_advance(page_height);
            pdf_advance(page_height);
            if(oldpage_height < *page_height)
                pdf_print_dataheader (NOTFIRSTDATAPAGE,  "Sequence data", page_height, column_width, pop, world, options, data);            
          }
      }
    free(column_width);
}

void format_helper(MYREAL * matrix, long len, int *fmt1, int *fmt2)
{
    long i;
    long count = 0;
    long maxcount = 0;
    MYREAL val;
    
    for(i=0;i<len; i++)
      {
        val = matrix[i];
        if(val>0.)
            count = (long)(log10(val)) + 1; 
        if(count>maxcount)
            maxcount = count;     
      }
    *fmt1 = 7;
    if(maxcount<2)
      {
        *fmt2 = 5;
      }
    else
      {
        *fmt2 = 2;
      }
}


///
/// print matrix line for various uses
/// - print linear shortened migration matrix in n x n matrix with - as diagonals
/// alowing for some pretty print using a format_helper() the columns are right-aligned
/// based on the width.
void pdf_print_matrix_line(long whichline, float *page_height, float left_margin, float width, long cols,long rows,MYREAL * matrix,boolean ismig)
{
    long i=0;
    long col, row = whichline;
    float offset;
    float page_width = pdf_contents_get_width(canvas);
    int fmt1=5, fmt2=3;
    long pad = ((ismig)?cols:0);
    format_helper(matrix+pad, cols * rows - pad ,&fmt1, &fmt2);
    offset = left_margin;
    if(ismig)
        i = cols;
    for(col = 0; col < cols; col++)
      {
        if(offset > page_width - 55)
          {
            offset = left_margin;
            pdf_advance(page_height);
          }
        if(ismig && row==col)
          {
            pdf_printf_ralign(offset+width - 15, *page_height, "-");
          }
        else
          {
            pdf_printf_ralign(offset + width - 5, *page_height, "%*.*f", fmt1, fmt2, matrix[i]);
            i++;
          }
        offset += width;
      }
    pdf_advance(page_height);
}




///
/// Printing MCMC table for PDF
/// \param *page_height  
void pdf_print_mcmctable(float *page_height, float page_width, world_fmt *world, data_fmt *data, option_fmt *options)
{
    char stemp[LINESIZE];
    char stemp2[LINESIZE];
    float width[6] = {0, -90, 145, 190, -240, 260};
    char *title = "Maximum likelihood estimates";
    float w;
    long pop;
    long frompop;
    long topop;
    long skipped = 0, locus;
    long maxrep = world->options->replicate ?
        (world->options->replicatenum > 0 ?
         world->options->replicatenum + 1 : world->options->lchains + 1) : 1;
    long rep;
    long j;
    float offset;
    
    // setup new page and title
    pdf_new_page("");
    pdf_contents_set_font_and_size(canvas, "Helvetica-Oblique", 18);    
    w = pdf_contents_get_text_width(canvas, title, NULL, NULL);
    *page_height = pdf_contents_get_height(canvas) - 75;
    page_width = pdf_contents_get_width(canvas);
    *page_height -= 20;
    pdf_print_contents_at((page_width - w)/2, *page_height, title);
    pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0, 0, 0));
    *page_height -= 20;
    pdf_draw_line(50, *page_height, page_width-50, *page_height);
    *page_height -= 20;
    pdf_contents_set_font_and_size(canvas, "Helvetica", 11);    
    
    pdf_draw_line(50, *page_height, page_width-50, *page_height);
    pdf_advance(page_height);
    
    pdf_print_table(page_height, width, "%s %s %s\n", 
                    "Population [x]", "Loc.", "Ln(L)");
    pdf_print_line_theta(fabs(width[3])+45, *page_height, 0, -1);
    offset = width[5];
    for(j=0; j < world->numpop; j++)
      {
        if(offset > page_width - 55)
          {
            offset = width[5];
            pdf_advance(page_height);
          }
        m2mm(j,world->numpop,&frompop, &topop);
        if(world->options->usem)
            pdf_print_line_mig("M", offset, *page_height, 0. , frompop, -1);
        else
            pdf_print_line_mig("xNm", offset, *page_height, 0. , frompop, -1);
        offset += 60.;
      }
    pdf_advance(page_height);
    
    pdf_print_contents_at(width[5], *page_height,"[+=receiving population]" );
    pdf_advance(page_height);
    
    pdf_draw_line(50, *page_height, page_width-50, *page_height);
    pdf_advance(page_height);
    for(pop = 0 ; pop < world->numpop ; pop++)
      {
        sprintf(stemp,"%li: %12.12s", pop+1, data->popnames[pop]);
        pdf_print_table(page_height, width,"%s", stemp);
        for (locus = 0; locus < world->loci; locus++)
          {
            if (world->data->skiploci[locus])
              {
                skipped++;
                continue;
              }
            
            for (rep = 0; rep < maxrep; rep++)
              {
                sprintf(stemp,"%2li", rep + 1);
                sprintf(stemp2,"%4li%2s ",  locus + 1,
                        maxrep > 1 ? (rep ==
                                      maxrep - 1 ? " A" : stemp2) : "  ");
                pdf_print_table(page_height, width,"%s %s %6.3f %8.4f", 
                                " ", stemp2,
                                world->atl[rep][locus].param_like, 
                                world->atl[rep][locus].param[pop]);
                pdf_print_matrix_line(pop, page_height, width[5], 60.,
                                      world->numpop,world->numpop,
                                      world->atl[rep][locus].param+pop*(world->numpop-1),TRUE);
              }
          }
        if (world->loci - skipped > 1)
          {
            //FPRINTF (world->outfile, "                All ");
            pdf_print_table(page_height, width,"%s %s %6.3f %8.4f", 
                            " ", "All",
                            world->atl[0][locus].param_like, 
                            world->atl[0][locus].param[pop]);
            pdf_print_matrix_line(pop, page_height, width[5], 60.,
                                  world->numpop,world->numpop,
                                  world->atl[0][locus].param+pop*(world->numpop-1),TRUE);
          }
      }
}


#if 0
///
/// Printing PROFILE table for PDF
/// \param *page_height start y coordinate
/// \param page_width   width of paper
/// \param world        master parameter structure
/// \param data         data structures
/// \param options      options
///
void pdf_print_profiletable(float *page_height, float page_width, world_fmt *world, data_fmt *data, option_fmt *options)
{
    char stemp[LINESIZE];
    char stemp2[LINESIZE];
    float width[6] = {0, -90, 145, 190, -240, 260};
    char *title = "Profile likelihood tables";
    float w;
    long pop;
    long frompop;
    long topop;
    long skipped = 0, locus;
    long maxrep = world->options->replicate ?
        (world->options->replicatenum > 0 ?
         world->options->replicatenum + 1 : world->options->lchains + 1) : 1;
    long rep;
    long j;
    long lozi = world->loci > 1 ? world->loci+1 : 1; 
    float offset;
    
    // setup new page and title
    pdf_new_page("");
    pdf_contents_set_font_and_size(canvas, "Helvetica-Oblique", 18);    
    w = pdf_contents_get_text_width(canvas, title, NULL, NULL);
    *page_height = pdf_contents_get_height(canvas) - 75;
    page_width = pdf_contents_get_width(canvas);
    *page_height -= 20;
    pdf_print_contents_at((page_width - w)/2, *page_height, title);
    pdf_contents_set_rgb_stroke(canvas, PdfRGBColor(0, 0, 0));
    *page_height -= 20;
    pdf_draw_line(50, *page_height, page_width-50, *page_height);
    *page_height -= 20;
    pdf_contents_set_font_and_size(canvas, "Helvetica", 11);    
    
    // pdf_print_profile_tables();
    
}

void pdf_print_profile_header()
{
    const long numpop = world->numpop;
    //column to to right-align the table columns
    float offset[] = {-1, -40, 135, 191, 247, 303, 359, 415, 471};
    long numoffset = 9;
    long off=0;
    
    pdf_print_contents_at(lx, ly, "Per.");
    
    
    for(pop=0; pop < numpop; pop++,off++)
      {
        if(off>numoffset-1)
          {
            off=2;
          }
        pdf_print_line_theta(lx, ly,offset[off], pop);
      }
}

void pdf_print_profile_line()
{
    pdf_print_matrix_line()
}


void pdf_print_profile_table(long whichparam,float *page_height, float page_width, world_fmt *world, data_fmt *data, option_fmt *options)
{
    long l; 
    pdf_print_profile_header();
    for(l = 0; l < 9; l++)    
        pdf_print_profile_line(l,whichparam *page_height, page_width, world);
    
}
#endif
#endif /*end of PRETTY*/













