/*------------------------------------------------------
Bayesian inference 
of migration rate  and effectice population size
using a Metropolis-Hastings Monte Carlo algorithm
-------------------------------------------------------
    Prior distributions   R O U T I N E S

send questions concerning this software to:
Peter Beerli
beerli@fsu.edu

Copyright 2013 Peter Beerli 

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
of the Software, and to permit persons to whom the Software is furnished to do
so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies
or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 $Id$
    */
/*! \file priors.c 

priors.c supplies prior distributions
*/
#include "migration.h"
#include "random.h"
#include "definitions.h"
#include "tools.h" 
#include "sighandler.h"

#ifdef HAVE_LGAMMA
#define LGAMMA lgamma
#else
#define LGAMMA mylgamma
#endif

extern int myID;

MYREAL cdf_gamma(MYREAL a, MYREAL b, MYREAL x);
MYREAL trunc_gamma_rand(MYREAL alpha, MYREAL beta, MYREAL lower, MYREAL upper);

///
/// Normal distribution
/// this generates only one random number, but could be easily changed to supply two numbers
MYREAL normal_rand(MYREAL mean, MYREAL std)
{
  MYREAL u = RANDUM();
  MYREAL v = RANDUM();
  return sqrt(-2 * log(u)) * cos(TWOPI*v);
  //return sqrt(-2 * log(u)) * sin(TWOPI*v);
}

// for prior_fmt
float trunc_random_normal(float *p)
{
  float r = (float) normal_rand(p[2],p[3]);
  while (r < p[0] || r > p[1])
    r = normal_rand(p[2],p[3]);
  return r;
}

// for prior_fmt
float cdf_normal(float x, float *p)
{
  return 0.5 * (1.0 + erf((x-p[2])/(sqrt(2.0) * p[3])));
}
float trunc_cdf_normal(float x, float *p)
{
  float aa =  cdf_normal((p[0]-p[2])/p[3],p);
  float Z = cdf_normal((p[1]-p[2])/p[3],p) - aa;
  return (cdf_normal(x,p) - aa)/ Z;
}

// uniform for prior_fmt
float trunc_random_uni(float *p)
{
  MYREAL u = RANDUM();
  return p[0] + (p[1]-p[0]) * u;
}

// uniform for prior_fmt
float trunc_cdf_uni(float x, float *p)
{
  //x==(y-p[0])/(p[1]-p[0])
  return x * (p[1] - p[0]) + p[0];
}

// uniform for prior_fmt
float trunc_random_exp(float *p)
{
  MYREAL u = RANDUM();
  float r = -log(u) * p[2];
    while(r < p[0] || r > p[1])
      {
	u = RANDUM();
	r = -log(u) * p[2];
      }
    return r;
}

// uniform for prior_fmt
float trunc_cdf_exp(float x, float *p)
{
  float lambda = 1.0/p[2];
  //float cdfx = 1.0 - exp(-lambda * x);
  float cdfa = 1.0 - exp(-lambda * p[0]);
  float cdfb = 1.0 - exp(-lambda * p[1]);
  //return cdfx/(cdfb-cdfa);
  return -log(-x * (cdfb-cdfa) + 1.0)/lambda;
}

float trunc_random_gamma(float *p)
{
  MYREAL alpha = p[2];
  MYREAL beta = p[3];
  return  (float) trunc_gamma_rand(alpha,beta,p[0],p[1]);
}

double bisection(int n, double a, double b, double tol, float p[], float x) {
  MYREAL alpha = p[2];
  MYREAL beta = p[3];
  if (beta < SMALLEPSILON)
    {
      beta = SMALLEPSILON;
    }
  MYREAL Z = cdf_gamma(alpha, beta, p[1]) -cdf_gamma(alpha, beta, p[0]);
  float fa = (float) cdf_gamma(alpha, beta, a)/Z - x;
  //float fb = (float) cdf_gamma(alpha, beta, b)/Z - x;
  float     c = (a + b) / 2.0;
  float fc;
  int count = 0;
  
  while (((b - a) > tol) && (count++ < n)) 
    {
      c = (a + b) / 2.0;
      fc = (float) cdf_gamma(alpha, beta, c)/Z - x;
      if (fc < 0.0) 
	{
	  if (fa < 0.0) 
	    {
	      a = c;
	      fa = fc;
	    } 
	  else 
	    {
	      b = c;
	      //fb = fc;
	    }
	} 
      else 
	{
	  if (fa >= 0.0) 
	    {
	      a = c;
	      fa = fc;
	    } 
	  else 
	    {
	      b = c;
	      //fb = fc;
	    }
	}
    }
  return c;
}


float trunc_cdf_gamma(float x, float *p)
{
  return  bisection(100, p[0],p[1], 0.0001, p, x); 
}

///
// Gamma distribution
//
// gamma deviated random number 
// from Marsaglia G., Tsang, W. W. (2000) A simple method for generating Gamma variables. 
// ACM Transactions on Mathematical Software Vol 26. No. 3: 363-372
// requirements are a 
//   - normal random number generator
//   - uniform random number generator
//
// [see Python program to test implementation] 
//
// returns a random number for shape parameter alpha>=1 and scale parameter beta=1
MYREAL __gamma_rand(MYREAL alpha)
{
  const MYREAL d = alpha - 1./3.;
  const MYREAL c = 1./sqrt(9. * d);
  MYREAL v,x,xx, u;
  while(1)
    {
      x = normal_rand(0.,1.);
      v = 1. + c * x;
      while(v<=0.0)
	{
	  x = normal_rand(0.,1.);
	  v = 1. + c * x;
	}
      v  = v * v * v;
      u  = RANDUM();
      xx = x * x;
      if (u < 1.0-0.0331*xx*xx)
	return d*v;
      if (log(u) < 0.5*xx + d * (1.0 - v + log(v)))
	return d*v;
    }
}

/// returns a random number from a gamma distribution with shape parameter alpha and scale parameter beta
MYREAL gamma_rand(MYREAL alpha, MYREAL beta)
{
  MYREAL aa;
  if (alpha<1.0)
    {
      aa = 1.0 + alpha;
      return __gamma_rand(aa)*pow(RANDUM(),(1.0/alpha)) * beta;
    }
  else
    {
      return __gamma_rand(alpha) * beta;
    }
}

/// returns a random number from a truncated gamma distribution
MYREAL trunc_gamma_rand(MYREAL alpha, MYREAL beta, MYREAL lower, MYREAL upper)
{
  MYREAL x;
  while(1)
    {
      x = gamma_rand(alpha,beta);
      if (lower < x)
	{
	  if (x < upper)
	    return x;
	}
    }
}

MYREAL logpdf_gamma(MYREAL a, MYREAL b, MYREAL x)
{
  if (a>0.0 && b> 0.0)
    return -(x/b) - a*log(b) + (-1 + a)*log(x) - LGAMMA(a);
  else 
    return -HUGE;
}

MYREAL cdf_gamma(MYREAL a, MYREAL b, MYREAL x)
{
  //Gamma[a, 0, xmax/b]/Gamma[a]
  return incompletegamma(x/b,a);
}

MYREAL logpdf_truncgamma(MYREAL a, MYREAL b, MYREAL  xmin, MYREAL xmax, MYREAL x)
{
  if((x > xmin) && (x <= xmax))
    return logpdf_gamma(a,b,x)-log(cdf_gamma(a,b,xmax)-cdf_gamma(a,b,xmin));
  else 
    return -HUGE;
}


MYREAL mean_truncgamma(MYREAL alpha, MYREAL beta, MYREAL lower, MYREAL upper)
{
  MYREAL n1, n2, d1, d2;
  MYREAL m;
  MYREAL nom;
  MYREAL denom ;
  //1>  beta*(Gamma(1+alpha,lower/beta,upper/beta)/Gamma(alpha,lower/beta,upper/beta)
  //2> (beta*(Gamma(1 + alpha,upper/beta) - Gamma(1 + alpha,lower/beta)))/
  //   (Gamma(alpha,upper/beta) - Gamma(alpha,lower/beta));
  // 1 and 2 should be the same
  // 
  n1 = logincompletegamma(upper/beta,1+alpha);
  n2 =  logincompletegamma(lower/beta,1+alpha);
  d1 = logincompletegamma(upper/beta,alpha);
  d2 =  logincompletegamma(lower/beta,alpha);
  nom = (1.0 - exp(n2-n1));
  denom = exp(d1-n1) - exp(d2-n1);
  if ((denom == 0.0) && (nom == 0.0))
    return beta * alpha;
  else
    m = beta * alpha * nom / denom;
  return m;
}


MYREAL find_beta_truncgamma(MYREAL mean, MYREAL alpha, MYREAL lower, MYREAL upper)
{
  /* from wikipedia:
     INPUT: Function f, endpoint values a, b, tolerance TOL, maximum iterations NMAX
     CONDITIONS: a < b, either f(a) < 0 and f(b) > 0 or f(a) > 0 and f(b) < 0
     OUTPUT: value which differs from a root of f(x)=0 by less than TOL
     N ← 1
     While N ≤ NMAX { limit iterations to prevent infinite loop
     c ← (a + b)/2 new midpoint
     If (f(c) = 0 or (b – a)/2 < TOL then { solution found
     Output(c)
     Stop
     }
     N ← N + 1 increment step counter
     If sign(f(c)) = sign(f(a)) then a ← c else b ← c new interval
     }
     Output("Method failed.") max number of steps exceeded  
  */
  MYREAL tolerance = EPSILON;
  MYREAL nmax = 1000;
  long n=1;
  MYREAL beta = (upper+lower)* 0.5; /*=c in readme*/
  MYREAL fbeta;
  MYREAL fa;
  MYREAL a = 0.0;  //was lower
  MYREAL b = upper*2.0; // was upper
  //return mean/alpha;
  if (a==0.0)
    a += SMALLEPSILON;
  while (n<= nmax)
    {
      beta = (a+b)/2.0;
      fbeta = mean_truncgamma(alpha,beta,lower,upper) - mean;    
      if(fbeta  == 0.0 || (b-a)/2. < tolerance)
	{
	  //printf("beta=%f\n",beta);
	  return beta;
	}
      n += 1;
      fa = mean_truncgamma(alpha,a,lower,upper) - mean;    
      if(((fbeta<0.0) && (fa<0.0)) || ((fbeta>0.0) && (fa>0.0)))
	a = beta;
      else
	b = beta;
    }
  warning("limit reached without achieving necessary accuracy");
  //printf("beta=%f --------------\n",beta);
  return beta;
}
	

///
/// Gamma prior retrieve a new value from a truncated gamma between lower and upper
/// currently does not use the old parameter, but this 
MYREAL
propose_gamma_newparam (MYREAL param, long which, bayes_fmt * bayes, MYREAL *r)
{
  MYREAL rr;
  // test with delta
  // const MYREAL delta = bayes->delta[which];
  // const MYREAL l = param - delta;
  // const MYREAL u = param + delta;
  // const MYREAL lower = l < bayes->minparam[which] ? bayes->minparam[which] : l;
  // const MYREAL upper = u > bayes->maxparam[which] ? bayes->maxparam[which] : u;
  const MYREAL lower = bayes->minparam[which];
  const MYREAL upper = bayes->maxparam[which];
  //const MYREAL mean = bayes->meanparam[which];
  MYREAL alpha = bayes->alphaparam[which];
  MYREAL beta = bayes->betaparam[which];
  //rr = trunc_gamma_rand(alpha, beta, param-delta, param+delta);
  rr = trunc_gamma_rand(alpha, beta, lower, upper);
  return rr;
}



///
/// Hastings ratio calculator for gamma distribution
/// P(new -> old)    P(old)
/// ------------- = -------
/// P(old -> new)    P(new)
/// cancels with log_prior_gamma -> 0.0
MYREAL hastings_ratio_gamma(MYREAL newparam, MYREAL oldparam, MYREAL delta, MYREAL r, bayes_fmt * bayes, long whichparam)
{
  if((newparam > bayes->maxparam[whichparam]) || (newparam < bayes->minparam[whichparam]))
    return -HUGE;
  else
    return 0.;
}


///
/// Log Prior gamma distribution ratios between old and new parameter:
/// cancels with hastings ratio
MYREAL log_prior_ratio_gamma(MYREAL newparam, 
			   MYREAL oldparam, 
			   bayes_fmt * bayes, 
			   long which)
{
  if((newparam > bayes->maxparam[which]) || (newparam < bayes->minparam[which]))
    return -HUGE;
  else
    return 0.;
}

///
/// Gamma prior distribution for theta or migration rate used in heating acceptance
/// uses pdf_truncgamma(a,b,min,max,x)
MYREAL log_prior_gamma(world_fmt *world, long numparam)
{
  //  long frompop;
  //long topop;
  MYREAL p0;
  long numpop = world->numpop;
  long start = ((numparam <= numpop || numpop==1) ? 0 : numpop);
  long stop = ((start == numpop) ? world->numpop2 : numpop);
  long i;
  MYREAL * param0 = world->param0;
  bayes_fmt * bayes = world->bayes;
  MYREAL a;
  MYREAL b;
  MYREAL val=0.0;
  for(i = start; i < stop; i++)
    {
      if(!strchr("0c", world->options->custm2[i]))
	{
	  //@@if(i>=numpop)//@@ && !world->options->usem)
	  //@@  {
#ifndef PRIORTEST 
	      //@@     m2mm(i,numpop,&frompop,&topop);
#endif
	      //@@p0 = param0[i] * param0[topop];
	  //@@ }
	  //@@else
	  //@@{
	      p0 = param0[i];
	      //@@  }

	  if((p0 > bayes->maxparam[i]) || (p0 < bayes->minparam[i]))
	    return -HUGE;
	  else
	    {
	      a = bayes->alphaparam[i];
	      b = bayes->betaparam[i];
	      val += logpdf_truncgamma(a,b,bayes->minparam[i],bayes->maxparam[i],p0);
	      //old!! val += -p0 * ib + a * log(ib) + (a - 1.) * log(p0) - LGAMMA(a) ;
	    }
	}
    }
  return val;
}

///
/// 
MYREAL log_prior_gamma1(world_fmt *world, long numparam, MYREAL val)
{
  bayes_fmt * bayes = world->bayes;
  MYREAL retval;
  MYREAL a = bayes->alphaparam[numparam];
  MYREAL b = bayes->betaparam[numparam]; 
  retval =  logpdf_truncgamma(a,b,bayes->minparam[numparam],bayes->maxparam[numparam],val);
  return retval;
}



void set_option_prior(prior_fmt **p, int type, MYREAL mini, MYREAL maxi, MYREAL mean, long bins)
{
  MYREAL b = 10.;
  //(*p) = calloc(1,sizeof(prior_fmt));
  (*p)->next = NULL;
  (*p)->number = -1;
  (*p)->type = type;
  switch(type)
    {
    case THETAPRIOR:
      strcpy((*p)->ptypename,"THETA"); break;
    case MIGPRIOR:
      strcpy((*p)->ptypename,"MIG"); break;
    case SPECIESTIMEPRIOR:
      strcpy((*p)->ptypename,"SPLIT"); break;
    case SPECIESSTDPRIOR:
      strcpy((*p)->ptypename,"SPLITSTD"); break;
    case RATEPRIOR:
      strcpy((*p)->ptypename,"RATE"); break;
    default:
      error("unknown prior in check_bayes_priors()");
	break;
    }
  (*p)->delta = (maxi-mini)/b;
  (*p)->min = mini;
  (*p)->mean = mean;
  (*p)->max = maxi;
  (*p)->bins = bins;

  //  p->delta = (p->max - p->min)/b;
  (*p)->alpha = -1.0;
}

prior_fmt * copy_option_prior(prior_fmt *pmodel, option_fmt *options, MYREAL ratemin)
{
  prior_fmt * p = (prior_fmt *) calloc(1,sizeof(prior_fmt));
  p->next = NULL;
  p->kind = pmodel->kind;
  p->type = pmodel->type;
  strcpy(p->ptypename, pmodel->ptypename);
  p->number = pmodel->number;
  p->min = pmodel->min+ratemin;
  p->mean = pmodel->mean;
  if (p->mean < p->min)
    p->mean = p->min;
  p->std = pmodel->std;
  p->max = pmodel->max;
  p->delta = pmodel->delta;
  p->alpha = pmodel->alpha;
  p->beta = pmodel->beta;
  p->bins = pmodel->bins;
  return p;
}


prior_fmt * insert_prior(prior_fmt *p, prior_fmt **plist,long *z,option_fmt * options)
{
  prior_fmt *q;
  if (plist[*z]->type == p->type)
    {
      q = copy_option_prior(plist[*z],options,0.0);
      p->next = q;
      p = q;
      (*z)++;
    }
  else
    {
      q = copy_option_prior(p,options,0.0);
      p->next = q;
      p = q;
    }
  return p;
}

prior_fmt * insert_prior_old(option_fmt *options, long index, prior_fmt *p, long type)
{
  prior_fmt *q, *tmp;
  if (p==NULL)
    return NULL;
  if ((p->next==NULL) || (p->next != NULL &&  p->next->type != type))
    {
      q = copy_option_prior(p, options,0.0);
      tmp = p->next;
      p->next = q;
      q->next = tmp;
      p = q;
    }
  else
    return p->next;
  return p;
}

long count_d(char *s)
{
  long count=0;
  char *t = s;
  while (*t != '\0')
    {
      if (*t == 'd' || *t == 'D')
	count++;
      t++;
    }
  return count;
}

/// checks the settings of the number of long an short chain for bayes options and resets useless settings
void check_bayes_priors(option_fmt *options, data_fmt *data, world_fmt *world)
{
  const long numpop = world->numpop;
  const long numpop2 = numpop * numpop;
  const int  has_mu = (int) options->bayesmurates;
  const long a = world->species_model_size * 2;
  const long np = numpop2 + has_mu + a;
  MYREAL ratemin = 0.0;
  long       i;
  long       j=0;
  long       pos;
  float      beta=0.0;
  longpair   *typelist; 
  long oldnp;
  // puts the original PRIOR_SIZE priorlist aside, this gets freed at the end of this function
  options->bayes_priors_num =   options->bayes_priors_num < PRIOR_SIZE ? PRIOR_SIZE : options->bayes_priors_num;
  prior_fmt  **plist = (prior_fmt **) calloc(options->bayes_priors_num+1, sizeof(prior_fmt *));
  typelist = (longpair *) calloc(np,sizeof(longpair));
  plist[0] = options->bayes_priors[0];
  prior_fmt *p = options->bayes_priors[0];
  for (i=1; i < options->bayes_priors_num; i++)
    {
      prior_fmt * q = p->next;
      plist[i] = q;
      p = q;
    }
  oldnp = options->bayes_priors_num;
  // resets the key priorlist to be filled with copies of elements from plist
  options->bayes_priors = (prior_fmt **) calloc(np,sizeof(prior_fmt *));
  // list of types
  for(j=0;j<np;j++)
    {
      typelist[j][1] = -1;
      if(j<numpop)
	{
	  typelist[j][0] = THETAPRIOR;
	  ratemin = SMALLEPSILON;
	}
      else 
	{
	  if (j < numpop2)
	    {
	      typelist[j][0] = MIGPRIOR;
	      ratemin = 0.0;
	    }
	  else
	    {
	      if (j==numpop2 && has_mu)
		{
		  typelist[j][0] = RATEPRIOR;
		  ratemin = SMALLEPSILON;
		}
	      else
		{
		  if((j-world->numpop2) % 2 == 0)
		    {
		      typelist[j][0] = SPECIESTIMEPRIOR;
		      ratemin = 0.0;
		    }
		  else
		    {
		      typelist[j][0] = SPECIESSTDPRIOR;
		      ratemin = SMALLEPSILON;
		    }
		}
	    }
	}
    }
  for (i=0; i < options->bayes_priors_num; i++)
    {
      for(j=i;j<np;j++)
	{
	  
	  if(typelist[j][0] == plist[i]->type && typelist[j][1] == -1)
	    {
#ifdef DEBUG
	      printf("%i> typelist[%li][1]=%li\n",myID, j,i);
#endif
	      typelist[j][1] = i;
	      break;
	    }
	}
    }
  pos = 0;
  for (j=0;j<np;j++)
    {
      if (typelist[j][1] != -1)
	{
	  pos = typelist[j][1];
	}
      else
	{	  
	  long jj = j;
	  while(jj>0)
	    {
	      if(typelist[j][0] != typelist[jj][1])
		jj--;
	      else
		{
		  pos = typelist[jj][1];
		  break;
		}
	    }	   
	}
      options->bayes_priors[j] = copy_option_prior(plist[pos],options, ratemin);
      if(j>0)
	{
	  options->bayes_priors[j-1]->next = options->bayes_priors[j];
	  if (options->bayes_priors[j]->bins <= 0)
	    options->bayes_priors[j]->bins = options->bayes_priors[j-1]->bins;
	}
    }
  options->bayes_priors_num = np;
  for(j=0;j<np;j++)
    {
      options->bayes_priors[j]->v[0]= options->bayes_priors[j]->min;
      options->bayes_priors[j]->v[1]= options->bayes_priors[j]->max;
      switch(options->bayes_priors[j]->kind)
	{
	case EXPPRIOR:
	case WEXPPRIOR:
	  options->bayes_priors[j]->v[2]= options->bayes_priors[j]->mean;
	  options->bayes_priors[j]->v[3]= options->bayes_priors[j]->std;
	  options->bayes_priors[j]->random = trunc_random_exp; 
	  options->bayes_priors[j]->cdf = trunc_cdf_exp;
	  break;
	case GAMMAPRIOR:
	  options->bayes_priors[j]->v[2]= options->bayes_priors[j]->alpha;
	  if (options->bayes_priors[j]->beta < SMALLEPSILON)
	    {
	      beta = find_beta_truncgamma(options->bayes_priors[j]->mean, options->bayes_priors[j]->alpha, 
				   options->bayes_priors[j]->min, options->bayes_priors[j]->max);
	      options->bayes_priors[j]->beta = beta;
	    }
	  options->bayes_priors[j]->v[3]= options->bayes_priors[j]->beta;
	  options->bayes_priors[j]->random = trunc_random_gamma; 
	  options->bayes_priors[j]->cdf = trunc_cdf_gamma;
	  break;
	case NORMALPRIOR:
	  options->bayes_priors[j]->v[2]= options->bayes_priors[j]->mean;
	  options->bayes_priors[j]->v[3]= options->bayes_priors[j]->std;
	  options->bayes_priors[j]->random = trunc_random_normal; 
	  options->bayes_priors[j]->cdf = trunc_cdf_normal;
	  break;
	case UNIFORMPRIOR:
	default:
	  options->bayes_priors[j]->v[2]= options->bayes_priors[j]->mean;
	  options->bayes_priors[j]->v[3]= options->bayes_priors[j]->std;
	  options->bayes_priors[j]->random = trunc_random_uni; 
	  options->bayes_priors[j]->cdf = trunc_cdf_uni;
	  break;
	}
    }
  for (i=oldnp-1; i>=0; i--)
    {
      myfree(plist[i]);
    }
  myfree(plist);
  myfree(typelist);
}


prior_fmt * get_prior_list(prior_fmt **list, int type)
{
  prior_fmt * p = list[0];
  prior_fmt *oldp = p;
  
  while(p->type != type)
    {
      p = p->next;
      if (p==NULL)
	return NULL;
    }
  while(p->type == type)
    {
      oldp = p;
      p = p->next;
      if (p==NULL)
	break;
    }
  return oldp;
}

// test section
// compile using this:
// gcc -o priortest -g priors.c random.c sighandler.c tools.c -DPRIORTEST -DMERSENNE_TWISTER -DNOJPEGLIB -DMEXP=19937
// then call by  
// priortest alpha beta
// it will print 10000 numbers
// the mean and standard deviation
//
#ifdef PRIORTEST
#include <stdio.h>
#include "sighandler.h"

char * generator;
int myID;
long *seed;

 int main(long argc, char **argv)
 {
   long i;
   MYREAL xx;
   MYREAL pxx;
   MYREAL a;
   MYREAL b;
   MYREAL mean=0.0;
   MYREAL var=0.0;
   
   world_fmt *world;
   world = calloc(1,sizeof(world_fmt));
   world->bayes = calloc(1,sizeof(bayes_fmt));
   world->bayes->maxparam = calloc(1,sizeof(MYREAL));
   world->bayes->minparam = calloc(1,sizeof(MYREAL));
   world->bayes->alphaparam = calloc(1,sizeof(MYREAL));
   world->bayes->betaparam = calloc(1,sizeof(MYREAL));
   world->bayes->meanparam = calloc(1,sizeof(MYREAL));
   world->numpop=1;

   generator = (char *) mycalloc (1,sizeof(char) * 80);

   init_gen_rand(123789);

   a = atof(argv[1]);
   b = atof(argv[2]);
   world->bayes->alphaparam[0] = a;
   world->bayes->meanparam[0] = a*b; 
   world->bayes->minparam[0] = 1.0; 
   world->bayes->maxparam[0] = 50.0; 
   world->bayes->betaparam[0] = find_beta_truncgamma(a*b, a, world->bayes->minparam[0],world->bayes->maxparam[0]);
   printf("Truncated gamma distribution with alpha=%f, beta=%f, lower=%f, upper=%f\n",
	  a,world->bayes->betaparam[0],world->bayes->minparam[0],world->bayes->maxparam[0]);
   b = world->bayes->betaparam[0];
   for (i=0;i<10000;i++)
     {
       xx=trunc_gamma_rand(a,b,world->bayes->minparam[0], world->bayes->maxparam[0]);
       pxx = log_prior_gamma1(world, 0, xx);
       mean += (xx - mean)/(i+1);
       var  += (xx - mean) * (xx-mean);
       printf("%f %f\n",xx, pxx);
     }
   printf("results of random gamma truncated 0 and 500, using a=%f, b=%f\n",a,b);
   printf("Mean=%f Standard deviation = %f\n",mean, sqrt(var/10000));
   printf("Expected=%f expected standard deviation = %f\n",a*b, sqrt(a)*b);
   MYREAL v = (world->bayes->maxparam[0]-world->bayes->minparam[0]) * 0.2;
   MYREAL fx = cdf_gamma(a,b,0.5);
   MYREAL fx2 = logpdf_gamma(a,b,v);
   MYREAL fx3 = logpdf_truncgamma(a,b,world->bayes->minparam[0],world->bayes->maxparam[0],v);
   printf("v=%f CDF(%f,%f,%f)=%f PDF(%f,%f,%f)=%f TPDF(%f,%f,%f,%f,%f)=%f\n",v,a,b,0.5,fx,a,b,v,fx2,a,b,
	  world->bayes->minparam[0],world->bayes->maxparam[0],v,fx3);
 }


float CDF(prior_fmt *prior, float p)
{
  float x = prior->cdf(prior->values);
  return x;
}

float random_prior(prior_fmt *prior)
{
  float x = prior->random(prior->values);
  return x;
}

#endif
