/*------------------------------------------------------
 inference of population parameters
 using a Metropolis-Hastings Monte Carlo algorithm
 -------------------------------------------------------
 speciation (fusion and fission routines)
 
 Peter Beerli 2013, Seattle
 beerli@fsu.edu
 
 Copyright 2013 Peter Beerli
 
 Permission is hereby granted, free of charge, to any person obtaining
 a copy of this software and associated documentation files (the
 "Software"), to deal in the Software without restriction, including
 without limitation the rights to use, copy, modify, merge, publish,
 distribute, sublicense, and/or sell copies of the Software, and to
 permit persons to whom the Software is furnished to do so, subject
 to the following conditions:
 
 The above copyright notice and this permission notice shall be
 included in all copies or substantial portions of the Software.
 
 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 
$Id:$
*/
/* \file speciate.c 
speciation tools
*/
#ifndef __SPECIATE__
#define __SPECIATE__
#include "migration.h"
#include "bayes.h"
#include "tools.h"
#include "tree.h"
#include "random.h"
#include "uep.h"
#include "random.h"
#include "sighandler.h"
#include "migrate_mpi.h"
#include "assignment.h"
#include "mcmc.h"
#include "mcmc2.h"
#include "pretty.h"

extern profuncptr *propose_new;

#ifdef AVX_not_working
#include <immintrin.h>
void myerf(double *x, double *result)
{
  __m256d v1 = _mm256_loadu_pd(x);
  __m256d yy = _mm256_erf_pd(v1);
  _mm256_storeu_pd(result,yy);
}
#endif



// Abramovitz and Stegun 7.1.26
// gain on Mac seems about 1/3
double myerf(double x)
{
  const double p = 0.3275911;
  const double a1= 0.254829592;
  const double a2= -0.284496736;
  const double a3=1.421413741;
  const double a4 = -1.453152027;
  const double a5 = 1.061405429;
  const double t = 1.0 / (1.0 + p*x);
  const double t2 = t * t;
  const double t3 = t * t2;
  const double t4 = t2 * t2;
  const double t5 = t2 * t3;
  return 1.0 - (a1*t + a2*t2 + a3*t3 + a4*t4 + a5*t5)*exp(-(x*x));
}
double myerfc(double x)
{
  const double p = 0.3275911;
  const double a1= 0.254829592;
  const double a2= -0.284496736;
  const double a3=1.421413741;
  const double a4 = -1.453152027;
  const double a5 = 1.061405429;
  const double t = 1.0 / (1.0 + p*x);
  const double t2 = t * t;
  const double t3 = t * t2;
  const double t4 = t2 * t2;
  const double t5 = t2 * t3;
  return (a1*t + a2*t2 + a3*t3 + a4*t4 + a5*t5)*exp(-(x*x));
}


double myerf1(double x)
{
  const double p = 0.47047;
  const double a1= 0.3480242;
  const double a2= -0.0958798;
  const double a3=0.7478556;
  double t = 1.0 / (1.0 + p*x);
  double t2 = t * t;
  double t3 = t * t2;
  return 1.0 - (a1*t + a2*t2 + a3*t3)*exp(-(x*x));
}

double myerfc1(double x)
{
  const double p = 0.47047;
  const double a1= 0.3480242;
  const double a2= -0.0958798;
  const double a3=0.7478556;
  double t = 1.0 / (1.0 + p*x);
  double t2 = t * t;
  double t3 = t * t2;
  return (a1*t + a2*t2 + a3*t3)*exp(-(x*x));
}


#ifdef FASTERF
#define ERF myerf
#define ERFC myerfc
#else
#define ERF erf
#define ERFC erfc
#endif
MYREAL eventtime_single(world_fmt *world, long pop, long timeslice, long *lineages, double age, char * event, long *to, long *from);

int beyond_last_node(proposal_fmt* proposal, vtlist *tentry, long gte, long *slider);

long init_speciesvector(world_fmt * world)
{
  const long numpop = world->numpop;
  const long numpop2 = world->numpop2;
  long i;
  long ssize = 0;
  char * custm2 = world->options->custm2;
  if (world->has_speciation)
    {
      for (i=numpop; i<numpop2; i++)
	{
	  if(custm2[i]=='d')
	    {
	      ssize++;
	    }
	  if(custm2[i]=='D')
	    {
	      ssize++;
	    }
	}
#ifdef DEBUG
      printf("%i> speciation events: %li %s\n",myID, ssize, custm2);
#endif
      world->species_model = (species_fmt *) calloc(ssize+1,sizeof(species_fmt));
      world->species_model_size = ssize;
    }
  return ssize;
}

// init speciation vector
//
void fill_speciesvector(world_fmt*world, option_fmt * options)
{
  const long numpop = world->numpop;
  const long numpop2 = world->numpop2;
  long i;
  long from;
  long to;
  long ssize = 0;
  char * custm2 = world->options->custm2;
  species_fmt *s = world->species_model;
  bayeshistogram_fmt *hist;
  long z;
  long zz;
  long z3;
  if (world->has_speciation)
    {
      z = numpop2+world->bayes->mu;
      zz = numpop2+world->bayes->mu;
      z3 = numpop2+world->bayes->mu;
      for (i=numpop; i<numpop2; i++)
	{
	  if(custm2[i]=='d' || custm2[i] == 'D')
	    {	    
	      //1229 setting custm2 to the correct migration setting!
	      custm2[i] = (custm2[i]=='D' ? '*' : '0');
	      m2mm(i,numpop,&from,&to);
	      s[ssize].id = ssize;
	      s[ssize].from = from;
	      s[ssize].to   = to;
	      s[ssize].mu   = options->bayes_priors[z3]->mean;
	      s[ssize].min  = options->bayes_priors[z3]->min;
	      s[ssize].max  = options->bayes_priors[z3++]->max;
	      s[ssize].sigma= options->bayes_priors[z3]->mean;
	      s[ssize].sigmamin  = options->bayes_priors[z3]->min;
	      s[ssize].sigmamax  =  options->bayes_priors[z3++]->max;
	      s[ssize].allocsize = HUNDRED;
	      s[ssize].data = (float*) mycalloc(s[ssize].allocsize,sizeof(float));
	      world->bayes->minparam[zz]=s[ssize].min;
	      world->bayes->maxparam[zz++]=s[ssize].max;
	      world->bayes->minparam[zz]=s[ssize].sigmamin;
	      world->bayes->maxparam[zz++]=s[ssize].sigmamax;
	      if (world->cold)
		{
		  hist = &(world->bayes->histogram[world->locus]);
		  hist->bins[z] = options->bayes_priors[z]->bins; 
		  hist->minima[z] = s[ssize].min;
		  hist->maxima[z++] = s[ssize].max;
		  //sigma
		  hist->bins[z] = options->bayes_priors[z]->bins;  
		  hist->minima[z] = s[ssize].sigmamin;
		  hist->maxima[z++] = s[ssize].sigmamax;	       
		}
	      ssize += 1;
	    }
	}
    }
}

void species_datarecorder(world_fmt *world)
{
  species_fmt * adb = world->species_model;
  species_fmt * db;
  long s ;
  long i;
  // records the mu and sigma
  // for posterior distributions 
  for (i=0;i<world->species_model_size;i++)
    {
      db = &(adb[i]);
      s = db->size;
      if (s+3 >= db->allocsize)
	{
	  db->allocsize += HUNDRED;
	  db->data = (float *) myrealloc(db->data,sizeof(float)*db->allocsize);
	}
      // this is always one single locus
      //
      db->data[s++] = db->mu;
      db->data[s++] = db->sigma;
      //printf("#EXPERIMENT: %f %f ",db->mu,db->sigma);
      //for(i=0;i<world->numpop2+ world->bayes->mu + 2* world->species_model_size;i++)
      //	printf("%f ",world->param0[i]);
      //printf("\n");
      db->size = s;
    }
}

#if defined(MPI) && !defined(PARALIO) /* */
void print_species_record(float *temp, long *z, world_fmt *world)
{
  long i;
  //long locus = world->locus;
  if (world->has_speciation)
    {
      for (i=0; i<world->species_model_size;i++)
	{
	  temp[(*z)++] = world->species_model[i].mu; 
	  temp[(*z)++] = world->species_model[i].sigma; 
	}
    }
}
#else /*not MPI or MPI & PARALIO*/
void  print_species_record(char *temp, long *c, world_fmt * world)
{
  //long locus = world->locus;
  long i;
  if (world->has_speciation)
    {
      for (i=0; i<world->species_model_size;i++)
	{
	  *c += sprintf(temp+ *c,"\t%f", world->species_model[i].mu); 
	  *c += sprintf(temp+ *c,"\t%f",world->species_model[i].sigma); 
	} 
    }
}
#endif


void read_species_record(long masternumbinall, long locus, MYREAL *params, long *n, MYREAL *oldmeans, MYREAL *lowerbound, MYREAL *upperbound, MYREAL *delta, world_fmt *world, char **inptr)
{
  const long numpop2 = world->numpop2;
  bayes_fmt *bayes = world->bayes;
  long j0;
  long j;
  long np = numpop2 + bayes->mu + world->species_model_size * 2 ;
  bayeshistogram_fmt *hist = &(world->bayes->histogram[locus]);
  long numbinsall = masternumbinall;
  long numbins;
  long bin;

  for(j0=numpop2+bayes->mu;j0 < numpop2+world->species_model_size*2; j0++)
    {
      if(shortcut(j0,bayes,&j))
	{
	  continue;
	}
      params[j+2] =  atof(strsep(inptr,"\t"));
      n[j] += 1;
      oldmeans[j] = hist->means[j];
      hist->means[j] += (params[j+2] - hist->means[j]) / n[j];
      numbinsall += hist->bins[j];
      numbins = numbinsall - hist->bins[j];
      
      if (params[j+2]>upperbound[j])
	{
	  warning("above upper bound: %f\n",params[j+2]);
	  continue;
	}
      bin = (long) ((params[j+2]-lowerbound[j]) / delta[j]);
      hist->minima[j0] = lowerbound[j0];
      hist->maxima[j0] = upperbound[j0];
      hist->results[numbins + bin] += 1.;
      bayes->histtotal[locus * np + j0] += 1;
    } 
}

void check_speciateparameters(species_fmt *s)
{
  if (s->mu < 0.0)
    {
      s->mu = 0.1;
    }
  if (s->sigma < 0.0)
    {
      s->sigma= 1.0;
    }
}

species_fmt * get_species_model(long to, species_fmt * s , long ssize)
{
  const long siz = ssize;
  long i;
  for ( i=0 ; i<siz; i++ )
    {
      if (to == s[i].to)
	return &(s[i]);
    }
  return NULL;
}

// for bayesian probwait and eventtime()
// this assumes that the speciation time follows a normal distribution
// with mean mu and standard deviation sigma, but mu and sigma
// are drawn from a prior 
MYREAL log_prob_wait_speciate_old(MYREAL t0, MYREAL t1, MYREAL mu, MYREAL sigma, species_fmt *s)
{
  const MYREAL d = (sqrt(2.0) * sigma);
  MYREAL x0 = (mu - t0)/d;
  MYREAL x1 = (mu - t1)/d;
  MYREAL e0 = 1.0 + ERF(x0);
  MYREAL e1 = 1.0 + ERF(x1);
  if (e1 == 0.0)
    return -HUGE;
  else
    return -log(e0/e1);
}

// mathematica integral over t0 to t1 for truncated normal between 0 and b1 for mu and sigma
// returns -integral(hazardfunction(truncated(normal(mu,sigma),{0,b1}))
// -Log[Erf[(a - b1)/(Sqrt[2] b)] - Erf[(a - tau0)/(Sqrt[2] b)]] + 
//     Log[Erf[(a - b1)/(Sqrt[2] b)] - Erf[(a - tau1)/(Sqrt[2] b)]]
MYREAL log_prob_wait_speciate_truncold(MYREAL t0, MYREAL t1, MYREAL mu, MYREAL sigma, species_fmt *s)
{
  const MYREAL qsd = 1./(sqrt(2.0) * sigma);
  const MYREAL b1 = s->max;
  if (0.0 > t0)
    return -HUGE;
  if(t1 > b1)
    return -HUGE;
  MYREAL eb1 = ERF((mu - b1) * qsd);
  MYREAL x0 = (mu - t0) * qsd;
  MYREAL x1 = (mu - t1) * qsd;
  MYREAL e0 = eb1 - ERF(x0);
  MYREAL e1 = eb1 - ERF(x1);
  if (e0 == 0.0)
    return -HUGE;
  else
    return log(e1/e0);
}
MYREAL log_prob_wait_speciate(MYREAL t0, MYREAL t1, MYREAL mu, MYREAL sigma, species_fmt *s)
{
  const MYREAL qsd = 1./(sqrt(2.0) * sigma);
  const MYREAL b1 = s->max;
  if (0.0 > t0)
    return -HUGE;
  if(t1 > b1)
    return -HUGE;
  MYREAL eb1 = ERF((b1 - mu) * qsd);
  MYREAL x0 = (mu - t0) * qsd;
  MYREAL x1 = (mu - t1) * qsd;
  MYREAL e0 = eb1 + ERF(x0);
  MYREAL e1 = eb1 + ERF(x1);
  if ((e0 == 0.0) || (e1 == 0.0))
    return -HUGE;
  else
    return log(e1) - log(e0);
}

MYREAL log_point_prob_speciate(MYREAL t1, MYREAL mu, MYREAL sigma, species_fmt * s)
{
  const MYREAL b1 = s->max;
  const MYREAL qsd = 1./(sqrt(2.0) * sigma);
  const MYREAL mut = (mu - t1);
  const MYREAL mut2 = mut*mut;
  const MYREAL var = sigma*sigma;
  // using a truncated normal (0..b1)
  // old
  // -((-a + x)^2/(2 b^2)) - Log[b] + 1/2 (-Log[2] - Log[\[Pi]]) - 
  //    Log[-(1/2) Erfc[a/(Sqrt[2] b)] + 1/2 Erfc[(a - b1)/(Sqrt[2] b)]]
  // new
  //-((mu - t)^2/(2 sigma^2)) + 1/2 (Log[2] - Log[\[Pi]]) - Log[sigma] - 
  //Log[Erfc[(-b1 + mu)/(Sqrt[2] sigma)] - 
  //    Erfc[(mu - t)/(Sqrt[2] sigma)]]
  MYREAL prob =  -(mut2/(2 * var)) - log(sigma) + LOG2MINUSPIHALF 
    - log(ERFC((mu - b1)*qsd) - ERFC(mu*qsd));
  return prob;
}

node * set_type2(world_fmt *world, node *p, node *q, char *custm2)
{
  long topop = p->actualpop;
  long frompop = q->actualpop;
  //long numpop = world->numpop;
  node *tmp=NULL;
  //char *custm2 = world->options->custm2;
  char type = 'm';
  species_fmt *s = NULL; 
  s = get_species_model(topop, world->species_model, world->species_model_size); 
  if (s!=NULL)
    {
      type = 'd';
      tmp = add_migration (world, p, type, q->actualpop, p->actualpop,
			   (MYREAL) RANDDOUBLE(0.0, q->tyme - p->tyme));
      while (frompop != s->from)
	{  
	  s = get_species_model(s->from, world->species_model, world->species_model_size);
	  if (s==NULL)
	    break;
	  tmp = add_migration (world, tmp, type, q->actualpop, tmp->actualpop,
			       (MYREAL) RANDDOUBLE(0.0, q->tyme - tmp->tyme));
	}
    }
  return tmp;
}

MYREAL set_type(world_fmt *world, long topop, long frompop, char *custm2, long numpop)
{
  char type = 'm';
  //long i = mm2m(frompop, topop, numpop);
  species_fmt *s = NULL; 
  s = get_species_model(topop, world->species_model, world->species_model_size); 
  if (s!=NULL)
    {
      type = 'd';
    }
  //1229if (custm2[i]=='d' || custm2[i]=='D')
  //1229   type = 'd';
  return type;
}


MYREAL time_to_speciate(world_fmt *world, long pop, MYREAL t0, char *event, long *to, long *from)
{
  long ssize = world->species_model_size;
  species_fmt *s = get_species_model(pop, world->species_model, ssize);
  MYREAL r;
  MYREAL tmax;
  MYREAL tmin = t0;
  MYREAL t1 ;
  MYREAL lambda ;
  MYREAL interval;
  long paramindex;
  MYREAL mu;
  MYREAL sigma;
  //boolean done=FALSE;
  long counter = 0;
  //printf("random=%f\n",UNIF_RANDUM());
  if(s==NULL)
    return HUGE;
  paramindex = world->numpop2 + world->bayes->mu + 2 * s->id;
  mu = world->param0[paramindex];
  sigma = world->param0[paramindex+1];
  tmax = s->max;
  if (tmax < tmin)
    {
      *event = 'd';
      *from = s->from;
      *to = s->to;//should be the same as pop
      return 1e-10;
    }
  t1 = (tmax+tmin)/2.;
  //check_speciateparameters(s);
  r = LOG(UNIF_RANDUM ());
  lambda = log_prob_wait_speciate(t0, t1 , mu, sigma, s) - r;
  while (fabs(lambda)>EPSILON && counter <= 1000 && (tmax-tmin)> EPSILON)
    { 
      counter += 1;
      if (lambda < 0.0)
	{
	  tmax = t1;
	}
      else
	{
	  tmin = t1;
	}
      t1 = (tmin+tmax)/2.0;
      //

      //printf("@@ %i> %li t0=%f t1=%f r=%f lambda=%f\n",myID,counter, t0,t1,r,lambda);
      lambda = log_prob_wait_speciate(t0, t1 , mu,sigma,s)-r;
    }  
  //printf("%i> D %li t0=%f t1=%f r=%f lambda=%f\n",myID,counter, t0,t1,r,lambda);

  if (counter>=1000)
    t1=HUGE;

  interval =  t1-t0;  
  *event = 'd';
  *from = s->from;
  *to = s->to;//should be the same as pop
  return interval;

}

MYREAL
time_to_coalmig (world_fmt * world, long pop, long timeslice, long *lineages, char * event,  long * to, long * from)
{
    MYREAL  interval;
    MYREAL  lines;
    MYREAL  denom;
    MYREAL  invdenom;
    MYREAL  inheritance = world->options->inheritance_scalars[world->locus];
    MYREAL  rate = world->options->mu_rates[world->locus];
    MYREAL  invrate = 1./ rate;
    long    addition=0;
    long    timepop = (world->numpop2+addition)*timeslice + pop;
    MYREAL  mm         = world->mig0list[pop] * invrate ;
    MYREAL  timethetarate = world->timek[timepop] * world->param0[pop];
    MYREAL  *skyparam = world->timek+(world->numpop2+addition)*timeslice;
    lines    = 2.0 * lineages[pop];
    // param0 could be partitioned over time so that we we have param0[timeslice][pop]
    // makeing proposal for particular timeslices instead of all, one could have equal weights
    // for all timeslices or then weigh them using lineages? or an arbitrary prior.
    // we also need a vector for times to establish the size of the timeslices
    if(!world->options->skyline)
      denom    = mm + (lines * (1.0 / (inheritance*rate*timethetarate)));
    else
      {
	long msta = world->mstart[pop];
        long msto = world->mend[pop];
	mm = 0.0;
	int i;
        for (i = msta; i < msto; i++)
	  {
	    if(skyparam[i]<=0.0)
	      {
		warning("skyparam[%li]=%f rate=%f\n",i, skyparam[i],rate);
		error("mismatch with skyparams\n");
	      }
	    else
	      mm += world->data->geo[i] * world->param0[i] * skyparam[i]/rate;
	  }
	denom    = mm + (lines * (1.0 / (inheritance*rate*timethetarate)));
      }
    invdenom = 1.0 / denom;
    interval =  (-(LOG (/*nu = */UNIF_RANDUM ())) * invdenom) ;
    if(interval < 0.0 || isnan(interval))
      {
	error("abort in eventtime()");
      }
    if (lines > 0.0)
    {
        if ((UNIF_RANDUM ()) < (mm * invdenom))
        {
	  *to = pop;
	  *event = 'm';
	  long ii=0;
	  MYREAL *r = world->migproblist[pop];
	  MYREAL rr = UNIF_RANDUM ();
	  while (rr >  r[ii] && ii < world->numpop-1)
	    {
	      ii++;
	    }
	  if (ii < pop)
	    *from = ii;
	  else
	    *from =  ++ii;
	  return interval;
        }
        else
	  {
            *event = 'c';
	    *to = pop;
	    *from = pop;
            return interval;
        }
    }
    else
    {
        //      printf("mcmc1.c 1653 pop = %li, lines = %li\n", pop, tentry->lineages[pop]);
        *event = 'm';
	long ii=0;
	MYREAL *r = world->migproblist[pop];
	MYREAL rr = UNIF_RANDUM ();
	while (rr >  r[ii] && ii < world->numpop-1)
	  {
	    ii++;
	  }
	*to = pop;
	if (ii < pop)
	  *from = ii;
	else
	  *from =  ++ii;
	return interval;
    }
}



MYREAL time_to_coalescence(world_fmt * world, long pop, long timeslice, long *lineages, char * event, long * to)
{
  world_fmt *w = world;
  long  lines;
  MYREAL  rate = w->options->mu_rates[w->locus];
  //long    timeslice;
  long    addition=0;
  long    timepop;
  MYREAL  inheritance = w->options->inheritance_scalars[w->locus];
  MYREAL  timethetarate;
  MYREAL denom=0.0;
  MYREAL invdenom;
  MYREAL interval=HUGE;
  MYREAL r;
  //timeslice=tentry->timeslice;
  timepop = (w->numpop2+addition)*timeslice + pop;;
  timethetarate = w->timek[timepop] * w->param0[pop];

  lines    = 2 * lineages[pop];
  if (lines == 0)
    {
      interval=HUGE;
    }
  else
    {
      denom    = lines * (1.0 / (inheritance*rate*timethetarate));
      invdenom = 1.0 / denom;
      r = UNIF_RANDUM ();
      interval =  (-(LOG (r)) * invdenom) ;
    }
  if (interval<0.0)
    warning("%i> x=%f in timecoal [denom=%f lines=%li r=%f inh=%f rate=%f timethetarate=%f]\n",myID, 
	    interval,denom,lines,r,inheritance,rate,timethetarate);
  *event = 'c';
  *to = pop;
  return interval;
}

MYREAL time_to_migration(world_fmt *world, long pop, long timeslice, long * lineages, char * event, long * to, long * from)
{
  long    addition=0;
  long    numpop = world->numpop;
  MYREAL  *skyparam = world->timek+(world->numpop2+addition)*timeslice;
  char    *custm2 = world->options->custm2;
  MYREAL  rate = world->options->mu_rates[world->locus];
  //MYREAL  invrate = 1./ rate;
  //long    timepop = (proposal->world->numpop2+addition)*timeslice + pop;
  //MYREAL  inheritance = proposal->world->options->inheritance_scalars[proposal->world->locus];
  //MYREAL  timethetarate = proposal->world->timek[timepop] * proposal->param0[pop];
  //MYREAL denom;
  MYREAL invdenom;
  long msta = world->mstart[pop];
  long msto = world->mend[pop];
  MYREAL mm = 0.0;
  int i;
  MYREAL interval=0.0;
  MYREAL the_eventtime=HUGE;
  //MYREAL eventtime;
  char the_event=' ';
  //char myevent=' ';
  long tox = *to;
  long fromx = *from;
  long the_to = *to;
  long the_from = *from;
  for (i = msta; i < msto; i++)
    { 
      switch (custm2[i])
	{
	case 'd':
	  error("should not go here");
	  break;
	case '0':
	  continue;
	default: /*migration with *, m, M, s, S, or c */
	  if(skyparam[i]<=0.0)
	    {
	      warning("skyparam[%li]=%f rate=%f\n",i, skyparam[i],rate);
	      error("mismatch with skyparams\n");
	    }
	  mm =  world->data->geo[i] * world->param0[i] * skyparam[i]/rate;
	  invdenom = 1.0 / (mm);
	  interval =  (-(LOG (UNIF_RANDUM ())) * invdenom) ;
	  if (the_eventtime > interval)
	    {
	      the_eventtime = interval;
	      the_event = 'm';
	      m2mm(i,numpop,&fromx,&tox);
	      the_to = tox;
	      the_from = fromx;
	    }
	}
    }
  *to = the_to;
  *from = the_from;
  *event = the_event;
#ifdef DEBUG
  //if (the_event == 'd')
  //  printf("%i@ %li <- %li: @time:%f (%f)\n",myID, the_to, the_from, tentry->age + interval, interval);
#endif
  return the_eventtime;
}

///
/// standard time evaluator for a single line, the time_to_speciate() is also called
/// when the last two lines in the tree need to be worked on jointly
MYREAL time_to_speciation(world_fmt * world, long pop, double age, char * event, long * to, long * from)
{
  MYREAL the_eventtime=HUGE;
  //char the_event;
  char myevent=' ';
  long tox = *to;
  long fromx = *from;
  //long the_to;
  //long the_from;

  MYREAL interval= -1;
  while(interval < 0.0)
    interval = time_to_speciate(world,pop,age, &myevent, &tox, &fromx);
  if (the_eventtime > interval)
    {
      the_eventtime = interval;
      *event = myevent;
      *to = tox;
      *from = fromx;
    }
  return the_eventtime;
}


void keep_min_eventtime(MYREAL *the_eventtime, MYREAL eventtime, char *the_event, char myevent,
			long *to, long tox, long *from, long fromx)
{
  if (eventtime < *the_eventtime)
    {
      *the_eventtime = eventtime;
      *the_event = myevent;
      *to = tox;
      *from = fromx;
    }
}

MYREAL eventtime_single(world_fmt *world, long pop, long timeslice, long *lineages, double age, char * event, long *to, long *from)
{
  char the_event = ' ';
  MYREAL the_eventtime=HUGE;
  MYREAL eventtime;
  char myevent=' ';
  long fromx = *from;
  long tox = *to;
  //eventtime = time_to_coalmig(world, pop, timeslice, lineages, &myevent,&tox, &fromx);
  eventtime = time_to_coalescence(world, pop, timeslice, lineages, &myevent,&tox);
  keep_min_eventtime(&the_eventtime, eventtime, &the_event, myevent, to, tox, from, fromx);
  eventtime = time_to_migration(world, pop, timeslice, lineages, &myevent, &tox, &fromx);
  keep_min_eventtime(&the_eventtime, eventtime, &the_event, myevent, to, tox, from, fromx);
  eventtime = time_to_speciation(world, pop, age, &myevent, &tox, &fromx);
  keep_min_eventtime(&the_eventtime, eventtime, &the_event, myevent, to, tox, from, fromx);
  *event = the_event;
  if (*event == 'm' && *from == *to)
    error("FAIL");
  return the_eventtime;
}


long speciation_from(long to, proposal_fmt * proposal)
{
  species_fmt * s = proposal->world->species_model;
  const long siz = proposal->world->species_model_size;
  long i;
  for ( i=0 ; i<siz; i++ )
    {
      if (to == s[i].to)
	return s[i].from;
    }
  error("there should be a species 'from'");
  return -1;
}



void loopcleanup(boolean assign, world_fmt * world, proposal_fmt *proposal, long oldpop, timelist_fmt *timevector)
{
  if(assign)
    {
      reassign_individual(world,proposal->origin,oldpop);
    }
  free_timevector (timevector);
#ifndef TESTING2
  free_masterproposal (proposal);
#endif
}

/*
return 1 if tree was accepted, 0 otherwise 
assign is set for assigning individuals, caller is responsible to reset the 
origin back to the original state when fail
*/
long
newtree_update (world_fmt * world, long g, boolean assign)
{    
    boolean coalesced;
    char event;
    long slider;
    long bordernum;
    long actualpop = -99, zz;
    long skyz=0;
    MYREAL endtime, nexttime, age;
    long newpop;
    long oldpop;
    long from = -1;
    long to = -1;
    proposal_fmt *proposal=NULL; 
    timelist_fmt *timevector = NULL; /* local timelist */
    vtlist *tentry = NULL; /*pointer into timeslice */
    MYREAL x;
    if (assign && world->unassignednum<2)
      return 0;

    new_localtimelist (&timevector, &world->treetimes[0], world->numpop);
    new_proposal (&proposal, &world->treetimes[0], world);

    if(assign)
      {
	chooseUnassigned (proposal);
	oldpop = proposal->origin->actualpop;
	newpop = oldpop;
	if (world->numpop>1)
	  {
	    while (newpop == oldpop)
	      {
		newpop = RANDINT(0,world->numpop-1);
	      }
	  }
	reassign_individual(world,proposal->origin,newpop);
      }
    else
      {
	chooseOrigin (proposal);
	oldpop = proposal->origin->actualpop;
      }

    if(world->options->bayes_infer)
    {
      //negative value to flag for no-change later
      world->bayes->starttime = -proposal->origin->tyme;    
    }

    construct_localtimelist (timevector, proposal);
    tentry = &(*timevector).tl[0];
    age = proposal->origin->tyme;
    if(age<0.0)
      {
	error("Abort time was negative");
      }
    zz = 0;
    // finding timeslice int the timelist with age
    while ((tentry->age < age || tentry->age - age < SMALLEPSILON)&& zz < (*timevector).T)
    {
        tentry = &(*timevector).tl[zz];
        zz++;
    }
    zz--;
 
    // adjusting the timeinterval for the skylineplots
    skyz=world->timeelements-1;
    while (skyz > 0 && tentry->age <  proposal->world->times[skyz])
      skyz-- ;
    skyz++;
    nexttime =  proposal->world->times[skyz];
    tentry->timeslice=skyz-1;
   
    if (tentry->age < nexttime)
      nexttime = tentry->age;

    if ((*timevector).T > 1)
        endtime = (*timevector).tl[(*timevector).T - 2].age;
    else
        endtime = 0.0;

    proposal->time = age;
    coalesced = FALSE;
    /*------------------------------------
     main loop: sliding down the tree  */
    slider = 0;
    //printf("-@---------------------------------------------------\n");
    while (nexttime <= endtime)
    {
        actualpop =
        (proposal->migr_table_counter >
         0) ? proposal->migr_table[proposal->migr_table_counter -
                                   1].from : proposal->origin->pop;

	x = eventtime_single(world, actualpop, tentry->timeslice, tentry->lineages, tentry->age, &event, &to, &from);
	int count=0;
	while(x < 0.0 && count++ < 100)
	  {
	    x = eventtime_single(world, actualpop, tentry->timeslice, tentry->lineages, tentry->age, &event, &to, &from);
	    printf("%i> event=%c to=%li from=%li x=%f age+x=%f nexttime%f\n",myID,event,to, from, x, age+x,nexttime);
	  }
	//printf("%i> event=%c to=%li from=%li x=%f age+x=%f nexttime%f\n",myID,event,to, from, x, age+x,nexttime);
	if (x<0.0)
	  error("smaller than 0");
	proposal->time = age + x;
	if(proposal->time < 0.0 || isnan(proposal->time))
	  {
	    error("abort");
	  }
        if(proposal->time < proposal->origin->tyme)
        {
	  fprintf(stdout,"%i> Proposal failed because of unordered entry in time list, abort this sample\nProposed time=%f origin time=%f nexttime=%f, deltatime=%f\n", myID, proposal->time, proposal->origin->tyme, nexttime, age);
            // we end up here when the migration events exceed the upper limit
	  loopcleanup(assign, world, proposal, oldpop, timevector);
	  return 0;
        }
        if (proposal->time < nexttime)
        {
            if (event == 'm' || event == 'd')
            {
	      if (!migrate (proposal, proposal->origin, event))
                {
		  // we end up here when the migration events exceed the upper limit
		  loopcleanup(assign, world, proposal, oldpop, timevector);
		  return 0;
                }
                age = proposal->time;
		//printf("%c: to=%li from=%li time=%f\n", event,
		//       proposal->migr_table[proposal->migr_table_counter - 1].to,
		//       proposal->migr_table[proposal->migr_table_counter - 1].from,
		//       proposal->migr_table[proposal->migr_table_counter - 1].time);
                continue;
            }
            else
            {   /*coalesce */
	      //printf("%c: pop=%li time=%f\n", event, actualpop, proposal->time); 
	      //printf("**\n");
                chooseTarget (proposal, timevector, proposal->bordernodes,
                              &bordernum);
                if(bordernum == 0)
                {
		  if (!migrate (proposal, proposal->origin, event))
                    {
		      // we end up here when the migration events exceed the upper limit
		      loopcleanup(assign, world, proposal, oldpop, timevector);
		      return 0;
                    }
                    age = proposal->time;
                    continue;
                }
                pretendcoalesce1p (proposal);
                coalesced = TRUE;
                break;
            }
        }   /*end if proposal->time < nextime */
        age = nexttime;
        zz++;
        if(zz >= timevector->T)
        {
            break;
        }
        tentry = &(*timevector).tl[zz]; /*next entry in timelist */
	// skyparam
	if (tentry->age > proposal->world->times[skyz])
	  {
	    zz--;
	    tentry = &(*timevector).tl[zz]; /*next entry in timelist */
	    tentry->timeslice = skyz;
	    nexttime =  proposal->world->times[skyz];
	    skyz++;
	  }
	else
	  {
	    nexttime = tentry->age;
	    tentry->timeslice = skyz-1;
	  }
    }
    if (!coalesced)
    {
        if (!beyond_last_node(proposal, (*timevector).tl, (*timevector).T - 1, &slider))
        {
	  loopcleanup(assign, world, proposal, oldpop, timevector);
	  return 0;
        }
        pretendcoalesce1p (proposal);
    }
    if (acceptlike (world, proposal, g, timevector))
      {
        if (proposal->time > world->root->tyme)
        {   /*saveguard */
            world->root->tyme += proposal->time;
        }
        coalesce1p (proposal);
	if(world->cold && assign)
	  {
	    record_assignment(world->locus, world);
	  }
        if(world->options->bayes_infer)
        {
            world->bayes->starttime = -world->bayes->starttime;
            world->bayes->stoptime = proposal->time;
            world->treelen = 0.0;
            calc_treelength (world->root->next->back, &world->treelen);
        }
        //
        world->likelihood[g] = treelikelihood (world);
        /* create a new timelist */
        construct_tymelist (world, &world->treetimes[0]);
        world->migration_counts = 0;
        /* report the number of migration on the tree */
        count_migrations (world->root->next->back, &world->migration_counts);
        free_timevector (timevector);
#ifndef TESTING2
            free_masterproposal (proposal);
#endif
        return 1;   /* new tree accepted */
      }
    else
      {
        // record the time interval that was used for the lineage.
        if(world->options->bayes_infer)
	  {
            world->bayes->stoptime = -proposal->time;
	  }
	loopcleanup(assign, world, proposal, oldpop, timevector);
	return 0;
      }
}

void set_things(MYREAL t1, MYREAL t2, char e1, char e2, long to1, long from1, long to2, long from2, 
		MYREAL *time, char *event, long *to, long *from)
{
  if (t1 < t2)
    {
      *time = t1;
      *event = e1;
      *to = to1;
      *from = from1;
    }
  else
    {
      *time = t2;
      *event = e2;
      *to = to2;
      *from = from2;
    }
}

// calculates the time=age and event for each lineage (of the final 2) 
// in beyond_....()
void calc_time_per_line(proposal_fmt * proposal, boolean same,MYREAL *time, char *event, long *to, long *from)
{
  long pop1 = *to;
  MYREAL age1 = *time;
  MYREAL r0, r1;
  long froms, tos;
  char events;
  MYREAL time_coal, time_mig, time_spec;
  world_fmt * w = proposal->world;
  MYREAL  rate = w->options->mu_rates[w->locus];
  //MYREAL  invrate = 1./rate; 
  MYREAL  *skyparam;
  //MYREAL denom;
  MYREAL invdenom;
  long msta = w->mstart[pop1];
  long msto = w->mend[pop1];
  long t;
  long pop2;
  long i;
  MYREAL mm;

  long    timeslice;
  long    addition=0;
  long    timepop;
  MYREAL  inheritance = w->options->inheritance_scalars[w->locus];
  MYREAL  timethetarate;

  long skyz=w->timeelements-1;
  while (skyz > 0 && *time < w->times[skyz])
      skyz-- ;
  skyz++;
  timeslice=skyz-1;
  timepop = (w->numpop2+addition)*timeslice + pop1;
  timethetarate = w->timek[timepop] * proposal->param0[pop1];
  skyparam= w->timek+(w->numpop2+addition)*timeslice;
  *time = HUGE;
  if (same) //both lines are in the same population --> coalescence is possible
    {
      r0 = LOG(UNIF_RANDUM());
      time_coal = -r0 * (inheritance*rate*timethetarate) * 0.5;
      // this is the first 'comparison': setting 
      *event = 'c';
      *time = age1 + time_coal;
      *to = pop1;
      *from = pop1;
    }
  else
    {
      //time_coal = HUGE;
      *time = HUGE;
    }
  // generate now times for each possible migration j->pop1, and also speciation pattern i->pop1
  for (i = msta; i < msto; i++)
    {
      // if the combintion j->pop1 is not a speciation type event
      //1229if (w->options->custm2[i] != 'd')
      //1229{
	  mm = 1.0;//initialize static analyzer
	  if(skyparam[i]<=0.0)
	    {
	      warning("skyparam[%li]=%f rate=%f\n",i, skyparam[i],rate);
	      error("mismatch with skyparams\n");
	    }
	  else
	    mm = proposal->world->data->geo[i] * proposal->world->param0[i] * skyparam[i]/rate;
	  if (mm>0.0)
	    invdenom = 1.0 / (mm);
	  else
	    invdenom = HUGE;
	  r1 = LOG(UNIF_RANDUM());
	  time_mig  = -r1 * invdenom;
	  m2mm(i,w->numpop,&pop2,&t);
	  set_things(*time, age1 + time_mig, *event, 'm', *to, *from, pop1, pop2 , time, event, to, from);
	  //1229}
    }
  // assumes no population fusion only splitting
  // revisit this!
  time_spec = time_to_speciate(w, pop1, age1, &events, &tos, &froms);
  set_things(*time, age1 + time_spec, *event, 'd', *to, *from, tos, froms, time, event, to, from);
}




// simulating beyond the last node in the tree
// replaces pre_population in mcmc1.c
//
int beyond_last_node(proposal_fmt* proposal, vtlist *tentry, long gte, long *slider)
{
  long pop1, pop2;
  MYREAL age1, horizon;
  char event;
  MYREAL time;
  long to, from;
  long pmc1, pmc2;
  //world_fmt * w = proposal->world;
  //boolean not_in_ancestor1;
  //boolean not_in_ancestor2;
  boolean dangling;
  MYREAL time1, time2;
  char event1, event2;
  long to1, to2, from1, from2;
  //some standard stuff copied from pre_population
  if (gte > 0)
    proposal->realtarget = tentry[gte - 1].eventnode;
  else
    proposal->realtarget = tentry[0].eventnode->next->back; //?????gte
  if (proposal->realtarget == proposal->oback)
    {
      proposal->realtarget = crawlback (proposal->osister)->back;
    }
  if (proposal->realtarget->type == 'm' || proposal->realtarget->type == 'd')
    {
      proposal->target = crawlback (proposal->realtarget->next);
      if (proposal->target == proposal->oback)
        {
	  proposal->target = proposal->osister;
        }
    }
  else
    {
      proposal->target = proposal->realtarget;
    }
  proposal->tsister = NULL;
  pop2 = proposal->realtarget->pop;
  pop1 = proposal->migr_table_counter > 0 ? proposal->migr_table[proposal->migr_table_counter -
								 1].from : proposal->origin->pop;
  age1 = MAX (proposal->realtarget->tyme,
	      proposal->migr_table_counter >
	      0 ? proposal->migr_table[proposal->migr_table_counter -
				       1].time : proposal->origin->tyme);
  horizon = MAX (proposal->oback->tyme, age1);

  //the residual tree is only one lineages after the last node, and the simulated line
  //can be in the same population, or different populations, it also could be that the
  // lineages have not crossed into the ancestor yet.
  // one population
  event = ' ';
  //if(get_species_model(pop1, w->species_model, w->species_model_size)!=NULL)
  //  {
  //    not_in_ancestor1 = TRUE;
  //  }
  //if(get_species_model(pop2, w->species_model, w->species_model_size)!=NULL)
   // {
	//  not_in_ancestor2 = TRUE;
   // }
  // start
  // pop1 is the actual population at proposal->time in the dangling lineage
  // pop2 is the last lineage in the residual tree 
  time = age1;
  proposal->time = time;
  // after the last node in the residual tree but before the last node in the 
  // original tree
  while (time < horizon)
    {
      time = proposal->time;
      pop2 = proposal->realtarget->pop;
      pop1 = proposal->migr_table_counter > 0 ? proposal->migr_table[proposal->migr_table_counter -
								     1].from : proposal->origin->pop;
      // calculate only for the dangling lineage, gets the timeinterval (not the full age)
      to = pop1;
      calc_time_per_line(proposal, pop1 == pop2, &time, &event, &to, &from);
      // start proposal->time pop1, pop2
      // proposal time        to/from pop2
      if (time >= horizon)
        break;

      proposal->time = time;
      if (event == 'c')
	{
	  //printf("@%c: pop=%li time=%f\n", event, to , proposal->time); 
	  return 1;
	}
      if (!migrate (proposal, proposal->origin, event))
	{
	  return 0; // too many migrations! abort this trial
	}
    }
  proposal->time =  horizon;
  to1 = pop1;
  to2 = pop2;
  from1 = -1;
  from2 = -1;
  event = '@';
  // went past last node information in the old tree
  while (event != 'c')
    {
      time1 = time2 = proposal->time;
      pmc1 = proposal->migr_table_counter;
      pop1 = pmc1 > 0 ? proposal->migr_table[pmc1 - 1].from : proposal->origin->pop;
      pmc2 = proposal->migr_table_counter2;
      pop2 = pmc2 > 0 ? proposal->migr_table2[pmc2 - 1].from : proposal->realtarget->pop;      
      to1 = pop1;
      to2 = pop2;
      calc_time_per_line(proposal, pop1 == pop2, &time1, &event1, &to1, &from1);
      calc_time_per_line(proposal, pop1 == pop2, &time2, &event2, &to2, &from2);
      //printf("%i> _both() interval line1: (%f,%c) line 2: (%f, %c) \n",myID,time1 - proposal->time, event1,time2 - proposal->time, event2);
      if (time1 < time2)
	{
	  dangling = TRUE;
	  time = time1;
	  event = event1;
	  to = to1;
	  from = from1;
	}
      else
	{
	  dangling = FALSE;
	  time = time2;
	  event = event2;
	  to = to2;
	  from = from2;
	}
      proposal->time = time;
      if (event == 'c')
	{
	  //printf("@@%c: to=%li from=%li time=%f\n", event, to, from, proposal->time); 
	  return 1;
	}
      if(dangling)
	{
	  if (!migrate (proposal, proposal->origin, event))
	    {
	      return 0;  /* migration limit reached */
	    }
	  //pmc1 = proposal->migr_table_counter;
	  //pop1 = pmc1 > 0 ? proposal->migr_table[pmc1 - 1].from : proposal->origin->pop;
        }
      else
        {
	  if (!migrateb (proposal, proposal->realtarget, event))
            {
	      return 0;  /* migration limit reached */
            }
	  //pmc2 = proposal->migr_table_counter2;
	  //pop2 = pmc2 > 0 ? proposal->migr_table2[pmc2 - 1].from : proposal->realtarget->pop;
	}
    }
  return 0;
}

long get_species_record(world_fmt * world,long which)
{
  long f,t;
  m2mm(which,world->numpop,&f,&t);
  species_fmt * s = get_species_model(t, world->species_model, world->species_model_size);
  return s->id;
}


long propose_new_spec_mu(world_fmt * world, long which, boolean *is_mu, MYREAL *newparam)
{
  species_fmt *s;
  MYREAL r;
  long x = which-world->numpop2-world->bayes->mu;
  long remainder = x % 2; // even are the mu and odd are the sigma [0,1,2,3,....]
  x = (long) x/2; //[0,1,2,3,4,5,6,7,8,9]==>[0,0,1,1,2,2,3,3,4,4]
  s = &world->species_model[x];
  r = RANDUM();

  if (remainder==0) // pick mu or sigma
    {
      *is_mu = TRUE;
      do{
	*newparam = (*propose_new[which])(world->param0[which],which,world->bayes,&r);
	check_min_max_param(newparam,s->min,s->max);
      } while (*newparam <= s->min && *newparam >= s->max);
      s->mu = *newparam;
    }
  else
    {
      *is_mu = FALSE;
      do{
	*newparam = (*propose_new[which])(world->param0[which],which,world->bayes,&r);
	check_min_max_param(newparam,s->sigmamin,s->sigmamax);
      } while (*newparam <= s->sigmamin && *newparam >= s->sigmamax);
      s->sigma = *newparam;
    }
#ifdef DEBUG
  //  printf("%i> propose_newmu_sigma: %li <- %li: specmu=%f (param[%li]=%f) sigma=%f mumin=%f mumax=%f\n",myID, s->to, s->from, s->mu,which,world->param0[which], s->sigma, s->min, s->max);
#endif
  return s->id;
}


MYREAL wait_event_species(world_fmt *world, vtlist *tli, MYREAL t0, MYREAL t1, boolean waitonly, MYREAL *eventprob)
{
  const long numpop = world->numpop;
  species_fmt * s;
  long ssize = world->species_model_size;
  long i;
  long npp = world->numpop2 + world->bayes->mu;
  MYREAL specw = 0.0;
  long *lineages = tli->lineages;
  node *eventnode = tli->eventnode;
  long tlit = tli->to;
  MYREAL mu;
  MYREAL sigma;
  //*eventprob = 0.0;
  if (!world->has_speciation)
    return 0.0;
  for (i=0;i<numpop;i++)
    {
      s = get_species_model(i, world->species_model, ssize);
      if (s==NULL)
	continue;
      else
	{
	  // log_prob_wait_speciate already returns negative
	  mu = world->param0[npp + 2* s->id];
	  sigma = world->param0[npp + 2* s->id + 1];
	  //if (mu<EPSILON)
	  //  warning("%i> heat=%f speciation mu is zero\n",myID, world->heat);
	  specw += lineages[s->to] * log_prob_wait_speciate(t0,t1,mu,sigma,s);
	  if (!waitonly && eventnode->type == 'd' && tlit==i)
	    *eventprob = log_point_prob_speciate(t1,mu,sigma,s); 
	}
    }
  return specw;
}

// for skyline plots calculate waiting for speciation event
MYREAL wait_D(long pop, MYREAL t0, MYREAL t1, long *lineages, world_fmt *world)
{
  //const long numpop = world->numpop;
  species_fmt * s;
  long ssize = world->species_model_size;
  long npp = world->numpop2 + world->bayes->mu;
  MYREAL specw = 0.0;
  MYREAL mu;
  MYREAL sigma;
  if (!world->has_speciation)
    return 0.0;
  s = get_species_model(pop, world->species_model, ssize);
  if (s==NULL)
    return 0.0;
  else
    {
      mu = world->param0[npp + 2* s->id];
      sigma = world->param0[npp + 2* s->id + 1];
      specw = lineages[s->to] * log_prob_wait_speciate(t0,t1,mu,sigma,s);
    }
  return specw;
}


void set_first_speciestree(node *mrca, world_fmt *world)
{
  long ssize = world->species_model_size;
  long i, j;
  long from;
  long the_from;
  long to;
  boolean found=FALSE;
  char type;
  node *p,*q;
  node *tmp;
  the_from = world->species_model[0].from;
  for (i=0;i<ssize;i++)
    {
      from = world->species_model[i].from;
      for (j=0;j<ssize;j++)
	{
	  to = world->species_model[i].to;
	  if (from == to)
	    {
	      found = TRUE;
	      break;
	    }
	  found = FALSE;
	}
      if (found)
	the_from = from;
    }
  mrca->pop = the_from;
  mrca->actualpop = the_from;
  p = mrca->next->back;
  q = mrca->next->next->back;
  type = set_type(world, p->actualpop,mrca->actualpop, world->options->custm2, world->numpop);
  tmp = add_migration (world, p, type, mrca->actualpop, p->actualpop,
		       (MYREAL) RANDDOUBLE(0.0, mrca->tyme - p->tyme));
  mrca->next->back = tmp;
  tmp->back = mrca->next;
  type = set_type(world, q->actualpop,mrca->actualpop, world->options->custm2, world->numpop);
  tmp = add_migration (world, q, type, mrca->actualpop, q->actualpop,
		       (MYREAL) RANDDOUBLE(0.0, mrca->tyme - q->tyme));
  mrca->next->next->back = tmp;
  tmp->back = mrca->next->next;
}

// assembles the histogram from the sampled parameter values in db->data
// there is a similar function that reads from the bayesallfile mdimfile 
void construct_species_hist(species_fmt *db, world_fmt *world, long locus, 
			    long npa, long pai, long offset, long numbin, 
			    MYREAL *mini, MYREAL *maxi, float **results,
			    float *total, float *themean, float *thestd)
{
  bayes_fmt *bayes = world->bayes;
  //char *custm2 = world->options->custm2;
  long      j, j0;
  long      i;
  long      bin;
  long      nb=0;

  MYREAL    delta;
  MYREAL    value;
  float    *values = db->data+offset;
  //long    floorindex;
  long size = db->size;
  long halfsize = size / 2;
  float *p = (float *) mycalloc(halfsize,sizeof(float));
  long z = 0;
  for(j0=0;j0<pai;j0++)
    {
      if(shortcut(j0,bayes,&j))//1229 || custm2[j0]=='d')
	{
	  continue;
	}
      if (j0>=world->numpop2)
	j = j0;
      nb += bayes->histogram[locus].bins[j];
    }
  //delta = bayes->deltahist[pai];
  for(i=0;i < size; i+=2)
    {
      value = values[i];
      p[z] = (float) value;
      z++;
    }
  qsort(p, halfsize, sizeof(float), floatcmp);
  for(i=0;i < halfsize; i++)
    {
      delta = bayes->deltahist[pai];
      value = p[i];
      *themean += value;
      *thestd += value * value;
      if(value < mini[pai])
	{
	  warning("%i> TOO SMALL value=%f, mini[%li]=%f\n",myID,value,pai,mini[pai]);
	}
      if(value > maxi[pai])
	{
	  warning("%i> TOO LARGE value=%f, maxi[%li]=%f\n",myID,value,pai,maxi[pai]);
	  bin = bayes->histogram[locus].bins[pai] - 1;
	}
      else
	{
	  bin = (long) ((value - mini[pai])/delta);
	  if(bin<0)
	    bin=0;
	}
      if((bin) > bayes->histogram[locus].bins[pai])
	{
	  warning("%i> value not counted for histogram: bin=%li > histbins=%li\n", myID,bin, bayes->histogram[locus].bins[pai]);
	  *total +=1;
	} 
      (*results)[nb+bin] += 1.;
      *total += 1;
    }
  //
  // adjusting the integral =1
  const float ttt = (float)(1.0/(*total));
  for(bin=0;bin < bayes->histogram[locus].bins[pai]; bin++)
    {
      (*results)[nb + bin] *= ttt;
    }
  myfree(p);
}


void construct_locusspecies_histogram(world_fmt *world, long locus, MYREAL *mini, MYREAL *maxi, float **results)
{
  bayes_fmt *bayes = world->bayes;
  species_fmt * adb = world->species_model;
  species_fmt * db;
  //long s ;
  long i;
  long j0,j;
  long np = world->numpop2 + bayes->mu;
  long npa = np + 2 * world->species_model_size;
  long pa = np;
  long pai;
  float themean;
  float thestd;
  float total;
  long pai2;
  float themean2;
  float thestd2;
  float total2;
  long numbin = 0;
  for(j0=0; j0 < np; j0++)
    {
      if(shortcut(j0,world->bayes,&j))
	{
	  continue;
	}
      if(j0 == world->numpop2)
	j=world->numpop2;
      //1229else
      //1229{
	  //1229if(world->options->custm2[j]=='d')
	  //1229  continue;
      //1229}
      numbin += bayes->histogram[locus].bins[j];
    }    
  pai = pa;
  for (i=0;i<world->species_model_size;i++)
    {
      db = &(adb[i]);
      //s = db->size;
      themean = 0.0;
      thestd = 0.0;
      total = 0;
      construct_species_hist(db,world,locus, npa, pai, 0,numbin, mini, maxi, 
			     results, &total,&themean,&thestd);
      world->bayes->histogram[locus].means[pai] = themean/total;
      world->bayes->histogram[locus].stds[pai]  = thestd / total;
      world->bayes->histtotal[locus*npa+pai] = (MYREAL) total;
      pai2= pai+1;
      themean2 = 0.0;
      thestd2 = 0.0;
      total2 = 0;
      numbin += bayes->histogram[locus].bins[pai];
      pai = pai2+1;

      construct_species_hist(db,world,locus, npa, pai2, 1, numbin, mini, maxi, 
			     results, &total2,&themean2,&thestd2);
      world->bayes->histogram[locus].means[pai2] = themean2/total2;
      world->bayes->histogram[locus].stds[pai2]  = thestd2 / total2;
      world->bayes->histtotal[locus*npa+pai2] = (MYREAL) total2;
      numbin += bayes->histogram[locus].bins[pai2];
    }
}


/*void adjust_priorsfor species
@@@@@@
      if(world->options->custm2[i]=='d')
	{
	  long f;
	  long t;
	  m2mm(i,world->numpop,&f,&t);
	  s = get_species_model(t, world->species_model, world->species_model_size);
	  bayes->minparam[i] = s->min;
	  bayes->maxparam[i] = s->max;
	  bayes->meanparam[i] = s->mu;
	  if(options->prior->alpha[i]
	  bayes->deltahist[z++] = (s->max - s->min)/ options->bayespriorm[world->numpop].bins;//use M number bins TODO FIX DEBUG
	  bayes->deltahist[z++] = (s->sigmamax - s->sigmamin)/ options->bayespriorm[world->numpop].bins;//use M number bins TODO FIX DEBUG
	}
*/

#endif




